"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setupMultitenantRoutes = setupMultitenantRoutes;

var _configSchema = require("@osd/config-schema");

var _htmlEntities = require("html-entities");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
function setupMultitenantRoutes(router, sessionStroageFactory, securityClient) {
  const PREFIX = '/api/v1';
  const entities = new _htmlEntities.AllHtmlEntities();
  /**
   * Updates selected tenant.
   */

  router.post({
    path: `${PREFIX}/multitenancy/tenant`,
    validate: {
      body: _configSchema.schema.object({
        username: _configSchema.schema.string(),
        tenant: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const tenant = request.body.tenant;
    const cookie = await sessionStroageFactory.asScoped(request).get();

    if (!cookie) {
      return response.badRequest({
        body: 'Invalid cookie'
      });
    }

    cookie.tenant = tenant;
    sessionStroageFactory.asScoped(request).set(cookie);
    return response.ok({
      body: entities.encode(tenant)
    });
  });
  /**
   * Gets current selected tenant from session.
   */

  router.get({
    path: `${PREFIX}/multitenancy/tenant`,
    validate: false
  }, async (context, request, response) => {
    const cookie = await sessionStroageFactory.asScoped(request).get();

    if (!cookie) {
      return response.badRequest({
        body: 'Invalid cookie.'
      });
    }

    return response.ok({
      body: entities.encode(cookie.tenant)
    });
  });
  /**
   * Gets multitenant info of current user.
   *
   * Sample response of this API:
   * {
   *   "user_name": "admin",
   *   "not_fail_on_forbidden_enabled": false,
   *   "opensearch_dashboards_mt_enabled": true,
   *   "opensearch_dashboards_index": ".opensearch_dashboards",
   *   "opensearch_dashboards_server_user": "opensearch_dashboardsserver"
   * }
   */

  router.get({
    path: `${PREFIX}/multitenancy/info`,
    validate: false
  }, async (context, request, response) => {
    try {
      const esResponse = await securityClient.getMultitenancyInfo(request);
      return response.ok({
        body: esResponse,
        headers: {
          'content-type': 'application/json'
        }
      });
    } catch (error) {
      return response.internalError({
        body: error.message
      });
    }
  });
  router.post({
    // FIXME: Seems this is not being used, confirm and delete if not used anymore
    path: `${PREFIX}/multitenancy/migrate/{tenantindex}`,
    validate: {
      params: _configSchema.schema.object({
        tenantindex: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        force: _configSchema.schema.literal('true')
      })
    }
  }, async (context, request, response) => {
    return response.ok(); // TODO: implement tenant index migration logic
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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