"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerParaRoute = registerParaRoute;

var _configSchema = require("@osd/config-schema");

var _notebooks = require("../../../common/constants/notebooks");

var _notebooks2 = _interopRequireDefault(require("../../adaptors/notebooks"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
function registerParaRoute(router) {
  /* --> Updates the input content in a paragraph
   * --> Runs the paragraph
   * --> Fetches the updated Paragraph (with new input content)
   */
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph/update/run/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.string(),
        paragraphInput: _configSchema.schema.string(),
        paragraphType: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const runResponse = await _notebooks2.default.updateRunFetchParagraph(context.observability_plugin.observabilityClient, request, _notebooks.wreckOptions);
      return response.ok({
        body: runResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  /* --> Updates the input content in a paragraph
   * --> Fetches the updated Paragraph (with new input content)
   */

  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.string(),
        paragraphInput: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const saveResponse = await _notebooks2.default.updateFetchParagraph(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: saveResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  /* --> Adds a new paragraph
   * --> Fetches the added Paragraph
   */

  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphIndex: _configSchema.schema.number(),
        paragraphInput: _configSchema.schema.string(),
        inputType: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const addResponse = await _notebooks2.default.addFetchNewParagraph(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: addResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  /* --> Update paragraphs in backend with paragraphs passed in
   * --> Fetches the added Paragraph
   */

  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/set_paragraphs/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphs: _configSchema.schema.arrayOf(_configSchema.schema.object({
          output: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({}, {
            unknowns: 'allow'
          }))),
          input: _configSchema.schema.object({
            inputText: _configSchema.schema.string(),
            inputType: _configSchema.schema.string()
          }),
          dateCreated: _configSchema.schema.string(),
          dateModified: _configSchema.schema.string(),
          id: _configSchema.schema.string()
        }))
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const updateNotebook = {
        paragraphs: request.body.paragraphs,
        dateModified: new Date().toISOString()
      };
      const updateResponse = await _notebooks2.default.updateNote(opensearchNotebooksClient, request.body.noteId, updateNotebook);
      return response.ok({
        body: updateResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  /* --> Deletes a paragraph
   * --> Fetches the all other Paragraphs as a list
   * --> Deletes all paragraphs if paragraphId is not provided
   */

  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph`,
    validate: {
      query: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    const params = {
      noteId: request.query.noteId,
      paragraphId: request.query.paragraphId
    };

    try {
      const deleteResponse = await _notebooks2.default.deleteFetchParagraphs(opensearchNotebooksClient, params, _notebooks.wreckOptions);
      return response.ok({
        body: deleteResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  /* --> Clears output for all the paragraphs
   * --> Fetches the all Paragraphs as a list (with cleared outputs)
   */

  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph/clearall/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const clearParaResponse = await _notebooks2.default.clearAllFetchParagraphs(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: clearParaResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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