package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.36.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.36.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````7!@``#0```"TD0````0`!30`(``)
M`"@`'``;``$``'"8C```F(P``)B,``!@````8`````0````$````!@```#0`
M```T````-````"`!```@`0``!`````0````#````5`$``%0!``!4`0``&0``
M`!D````$`````0````$```````````````````#\C```_(P```4```````$`
M`0```+"-``"PC0$`L(T!`)`"``"D`@``!@```````0`"````N(T``+B-`0"X
MC0$`&`$``!@!```&````!`````0```!P`0``<`$``'`!``!$````1`````0`
M```$````4>5T9```````````````````````````!@```!````!2Y71DL(T`
M`+"-`0"PC0$`4`(``%`"```$`````0```"]L:6(O;&0M;&EN=7@M87)M:&8N
M<V\N,P`````$````%`````,```!'3E4`8RKX+,@=@LS:L?5RBU@I_[Z&3#8$
M````$`````$```!'3E4```````,````"`````````"X```!-````"`````@`
M```!D!`$)4`$@"B```6@AE"$@`$```LH!(PX@!)!B@``"$T`````````3P``
M`%$`````````4@````````!3````5`````````!5````5@```%<```!8````
M`````%H``````````````%L`````````7`````````````````````````!=
M````7@```%\```!@`````````&$``````````````&,`````````9````&4`
M``````````````````!F````9P````````!H```````````````NQ9NV:W^:
M?`PZEPL5C"7,0_K7AZ/@^TE#1=7LB0=PT]EQ6!R_VE<;`:\TZ'ZIG)O/WD<M
MP^4_+-5DU'PS8MOMX:<""EGOEWEOQ=\?:PE#UG94/)0A_^0.N^.2?*U+X\"S
M`8MO:QJ>4.-'Q@%FK\QF^Z%L8```````````````````````````N!(`````
M```#``L```````"0`0```````P`7`-4"`````````````!(````_`0``````
M```````2````7`(`````````````$@```(X"`````````````!(```"`````
M```````````2````L```````````````$@````D#`````````````!(```!\
M`0`````````````2`````0``````````````(````+$!`````````````!(`
M``"G```````````````2````4P``````````````$@````,"````````````
M`!(```"[```````````````2````?@(`````````````$@```!``````````
M`````"`````?`0`````````````2````$@(`````````````$@```$0!````
M`````````!(```"L`0`````````````2````S@(`````````````$@```"P`
M`````````````"````"'`@`````````````2````CP$`````````````$@``
M``P!`````````````!8```"7```````````````2````;P``````````````
M$@```&L!`````````````!(```#8```````````````2````/`(`````````
M````$@```/4!`````````````!(```!W`@`````````````2````[@,`````
M````````$@```,D``````````````!(```#=`@`````````````1````[@$`
M````````````$@```/<"`````````````!$````I`@`````````````2````
M-@(`````````````$@```.<``````````````!(````@`@`````````````2
M````]0``````````````$@```$,"`````````````!(```!C````````````
M```2````E@(`````````````$@```%4!`````````````!(```"]`@``````
M```````2````V0$`````````````$@```/P!`````````````!(````P`@``
M```````````2````A@(`````````````$@```+\!`````````````!(```#E
M`@`````````````2````W`(`````````````(0```"H!`````````````!(`
M``"X`0`````````````2````20$`````````````$@```+0"````````````
M`!(```#;`@`````````````1````<`(`````````````$0```$8`````````
M`````!$```!2`@`````````````2````8@(`````````````$@````H"````
M`````````!(```",```````````````2````2@(`````````````$@```%P!
M`````````````!(```#'`0`````````````2````(`,`````````````$@``
M`!D"`````````````!(```#G`0`````````````2````KP(`````````````
M$@```)X"`````````````!(```"=`0`````````````B````;@,``'PB```$
M````$@`-`-0!``"H%0``M`(``!(`#0`4!````)`!```````@`!<`8`0``#Q'
M``!4````$@`-`"T$``"01P``&`$``!(`#0"K!```\"8``$0%```2``T`>P0`
M`$"0`0``````$``7`*$$```\D`$`!````!$`%P!4!```0)`!```````0`!@`
M2`0``%20`0``````$``8`$D$``!4D`$``````!``&`#<`P``_"P``"0````2
M``T`;@0``)0N```P"```$@`-`#@$```4(0``5````!(`#0",`P``Q#8``-0"
M```2``T`&`0``%P8````````$@`-`+`#``!H(0``%`$``!(`#0!`!```5)`!
M```````0`!@`]P,``)@Y``!L`0``$@`-`!($````D`$``````!``%P`?!```
M0)`!```````0`!@`E`0``(`B``"0`@``$@`-`&D$``!4D`$``````!``&``#
M!```,$H```0````1``\`H@,``#0L``#(````$@`-`-(#``"H2```;`$``!(`
M#0!_`P``$"4``#`````2``T`Q@,``$`E``"P`0``$@`-`(($```@+0``=`$`
M`!(`#0``7U]G;6]N7W-T87)T7U\`7TE435]D97)E9VES=&5R5$U#;&]N951A
M8FQE`%])5$U?<F5G:7-T97)434-L;VYE5&%B;&4`4$Q?9&]?=6YD=6UP`%!E
M<FQ?9W9?9F5T8VAP=@!097)L7VUG7W-E=`!097)L7V%T9F]R:U]L;V-K`%!E
M<FQ?9V5T7W-V`%!E<FQ?;F5W6%,`8F]O=%]$>6YA3&]A9&5R`'!E<FQ?<G5N
M`'!E<FQ?<&%R<V4`4&5R;%]S=E]S971P=@!097)L7W-V7W-E='!V;@!P97)L
M7V-O;G-T<G5C=`!P97)L7V1E<W1R=6-T`%!E<FQ?8W)O86M?;65M;W)Y7W=R
M87``4$Q?8W5R<F5N=%]C;VYT97AT`'!E<FQ?86QL;V,`4&5R;%]C<F]A:U]N
M;V-O;G1E>'0`<&5R;%]F<F5E`%!E<FQ?<V%F97-Y<VUA;&QO8P!097)L7W-Y
M<U]I;FET,P!097)L7VYE=UA37V9L86=S`%!E<FQ?871F;W)K7W5N;&]C:P!0
M97)L7W-Y<U]T97)M`%]?8WAA7V9I;F%L:7IE`')E860`<W1R8VAR`&=E='!I
M9`!M96UM;W9E`%]?;&EB8U]S=&%R=%]M86EN`%]?9G!R:6YT9E]C:&L`<W1R
M8VUP`&]P96XV-`!U;FQI;FL`<W1R=&]K`&UE;7-E=`!S=')N8VUP`&=E='5I
M9`!M96UC:'(`8VQO<V5D:7(`<W1R<W1R`')M9&ER`&-L;W-E`'-T<FQE;@!S
M=')D=7``<F5A;&QO8P!R96%D9&ER-C0`86)O<G0`7U]S<')I;G1F7V-H:P!S
M=&1E<G(`;65M8W!Y`'-T<G)C:'(`;'-T870V-`!O<&5N9&ER`&QS965K-C0`
M7U]E<G)N;U]L;V-A=&EO;@!E>&ET`')E861L:6YK`%]?<W1A8VM?8VAK7V9A
M:6P`86-C97-S`&UK9&ER`%]?96YV:7)O;@!?7W)E9VES=&5R7V%T9F]R:P!?
M7W-T86-K7V-H:U]G=6%R9`!?7V%E86)I7W5N=VEN9%]C<'!?<'(P`%]?865A
M8FE?=6YW:6YD7V-P<%]P<C$`;&EB<&5R;"YS;P!L:6)C+G-O+C8`;&0M;&EN
M=7@M87)M:&8N<V\N,P!L:6)G8V-?<RYS;RXQ`'!A<E]C=7)R96YT7V5X96,`
M<&%R7V)A<V5N86UE`%A37TEN=&5R;F%L<U]005)?0D]/5`!P87)?96YV7V-L
M96%N`'!A<E]C=7)R96YT7V5X96-?<')O8P!P87)?9&ER;F%M90!S:&%?9FEN
M86P`9V5T7W5S97)N86UE7V9R;VU?9V5T<'=U:60`<&%R7V-L96%N=7``7TE/
M7W-T9&EN7W5S960`7U]D871A7W-T87)T`%]?8G-S7W-T87)T7U\`<VAA7W5P
M9&%T90!X<U]I;FET`%]?96YD7U\`7U]B<W-?96YD7U\`7U]B<W-?<W1A<G0`
M<VAA7VEN:70`7V5N9`!P87)?;6MT;7!D:7(`7V5D871A`'!A<E]S971U<%]L
M:6)P871H`'!A<E]F:6YD<')O9P!M>5]P87)?<&P`<&%R7VEN:71?96YV`$=,
M24)#7S(N-`!'0T-?,RXU`$=,24)#7S(N,S0`1TQ)0D-?,BXS,P`O=7-R+VQI
M8B]P97)L-2\U+C,V+V-O<F5?<&5R;"]#3U)%```````````"``$``@`"``$`
M`0`#``$``0`"``$``0`"``$``@`!``$``@`"``(``@`!``0``0`!``$``0`!
M``$``@`"``(``@`!``(``@`%``(``@`!``(``0`"``$``@`"``(``@`"``(`
M!``"``(``@`!``(``0`"``(``@`!``(``@`"``$``@`!``8``P`"``(``@`"
M``(``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M`0`!``$``0`!``$``0`!``$``0!,`P``$````"`````4:6D-```%`+@$````
M`````0`!`&`#```0````(````%4F>0L```,`P@0````````!``,`0@,``!``
M````````M)&6!@``!@#*!```$````+.1E@8```0`U00``!`````4:6D-```"
M`+@$````````L(T!`!<```"TC0$`%P```/R/`0`7````!)`!`!<````,D`$`
M%P```!"0`0`7````&)`!`!<````<D`$`%P```""0`0`7````))`!`!<````H
MD`$`%P```#"0`0`7````-)`!`!<````\D`$`%P```,R/`0`5"@``T(\!`!4+
M``#4CP$`%1(``-B/`0`5&```W(\!`!,;``#@CP$`%1P``.2/`0`5'0``Z(\!
M`!4E``#LCP$`%2<``/"/`0`5/@``](\!`!4_``#XCP$`%4P``-R.`0`6`P``
MX(X!`!8$``#DC@$`%@4``.B.`0`6!@``[(X!`!8'``#PC@$`%@@``/2.`0`6
M"P``^(X!`!8,``#\C@$`%@T```"/`0`6#@``!(\!`!8/```(CP$`%A````R/
M`0`6$0``$(\!`!83```4CP$`%A0``!B/`0`6%0``'(\!`!86```@CP$`%A<`
M`"2/`0`6&0``*(\!`!8:```LCP$`%AX``#"/`0`6'P``-(\!`!8@```XCP$`
M%B$``#R/`0`6(@``0(\!`!8C``!$CP$`%B0``$B/`0`6)@``3(\!`!8H``!0
MCP$`%BD``%2/`0`6*@``6(\!`!8K``!<CP$`%BP``&"/`0`6+0``9(\!`!8N
M``!HCP$`%B\``&R/`0`6,```<(\!`!8Q``!TCP$`%C(``'B/`0`6,P``?(\!
M`!8T``"`CP$`%C4``(2/`0`6-@``B(\!`!8W``",CP$`%CD``)"/`0`6.@``
ME(\!`!8[``"8CP$`%CP``)R/`0`60```H(\!`!9!``"DCP$`%D(``*B/`0`6
M0P``K(\!`!9$``"PCP$`%D4``+2/`0`61@``N(\!`!9(``"\CP$`%DD``,"/
M`0`62@``Q(\!`!9+``#(CP$`%DP```A`+>EW`0#K"("]Z`3@+>4$X)_E#N"/
MX`CPON7\>P$``,:/XA?*C.+\^[SE`,:/XA?*C.+T^[SE`,:/XA?*C.+L^[SE
M`,:/XA?*C.+D^[SE`,:/XA?*C.+<^[SE`,:/XA?*C.+4^[SE`,:/XA?*C.+,
M^[SE`,:/XA?*C.+$^[SE`,:/XA?*C.*\^[SE`,:/XA?*C.*T^[SE`,:/XA?*
MC.*L^[SE`,:/XA?*C.*D^[SE`,:/XA?*C.*<^[SE`,:/XA?*C.*4^[SE`,:/
MXA?*C.*,^[SE`,:/XA?*C.*$^[SE`,:/XA?*C.)\^[SE`,:/XA?*C.)T^[SE
M`,:/XA?*C.)L^[SE`,:/XA?*C.)D^[SE`,:/XA?*C.)<^[SE`,:/XA?*C.)4
M^[SE`,:/XA?*C.),^[SE`,:/XA?*C.)$^[SE`,:/XA?*C.(\^[SE`,:/XA?*
MC.(T^[SE`,:/XA?*C.(L^[SE`,:/XA?*C.(D^[SE`,:/XA?*C.(<^[SE`,:/
MXA?*C.(4^[SE`,:/XA?*C.(,^[SE`,:/XA?*C.($^[SE`,:/XA?*C.+\^KSE
M`,:/XA?*C.+T^KSE`,:/XA?*C.+L^KSE`,:/XA?*C.+D^KSE`,:/XA?*C.+<
M^KSE`,:/XA?*C.+4^KSE`,:/XA?*C.+,^KSE`,:/XA?*C.+$^KSE`,:/XA?*
MC.*\^KSE`,:/XA?*C.*T^KSE`,:/XA?*C.*L^KSE`,:/XA?*C.*D^KSE`,:/
MXA?*C.*<^KSE`,:/XA?*C.*4^KSE`,:/XA?*C.*,^KSE`,:/XA?*C.*$^KSE
M`,:/XA?*C.)\^KSE`,:/XA?*C.)T^KSE`,:/XA?*C.)L^KSE`,:/XA?*C.)D
M^KSE`,:/XA?*C.)<^KSE`,:/XA?*C.)4^KSE`,:/XA?*C.),^KSE`,:/XA?*
MC.)$^KSE`,:/XA?*C.(\^KSE`,:/XA?*C.(T^KSE`,:/XA?*C.(L^KSE`,:/
MXA?*C.(D^KSE<$`MZ0'*H.,,P$W@R`^,Y4S"G^4HT$WB2#*?Y0S`C^!$0I_E
M'`"-Y1P`C>(8$(WE&!"-XA0@C>44((WB!$"/X`,PG.<`,)/E)#"-Y0`PH./5
M___K%"*?Y10RG^4"()3G#""-Y0(0H.$#,)3G$#"-Y0,`H.'\#`#K^#&?Y0,P
ME.<`0-/E``!4XU$```IP+QWNY#&?Y1Q0G>4$`(7B`U"%X@,PG^<'`7#C`R"2
MYYXSTN4#,(/CGC/"Y6,``(H``:#AI/__Z[0AG^480)WE`#"@X0(@C^"H$9_E
M`1"/X`0`@N4`()3E`""#Y9@AG^4<`)WE`B"/X`P@@^6,(9_E`0!0XP`0D>4"
M((_@"!"#Y00@@^5.``#:!!"$X@S`@^(`083@!."1Y`3@K.4!`%3A^___&@,`
M@.)489_E`!"@XP`0C>4%(*#A!F"/X``1@^=`$9_E#`"6Y0$0C^`#___K`$!0
MXB@```HL49_E(&"-X@50C^`,`)7E1___ZQP!G^4&$*#A``"/X*<``.L``%#C
M(0``"@@QG^4#,(_@#`"3Y>;^_^L;___K^""?Y;@PG^4"((_@`S"2YP`@D^4D
M,)WE`B`SX``PH.,E```:!`"@X2C0C>)P@+WH_/[_Z\@@G^4``%#C`B"/X`P`
M@N4:```*#O__Z[0PG^5P+QWN`S"?YP,PDN='1\/EH?__Z@P`EN7?_O_K`$"@
MX=+__^H(`)7E``!0XP$```IQ"`#KV/__ZGP`G^4&$*#A``"/X'H``.L``%#C
M"`"%Y='__PKU___J%___ZP0`H..V___J`0"@XU[__^LA___K!'D!`!P!``#D
M>`$`_````!0!```D`0``C'D!`,1Y`0"T>0$`(#<```@W``!0>0$`9`D``"AY
M`0!8-````'D!`'1W`0"T>`$`-'@!`$`S````L*#C`."@XP00G>0-(*#A!"`M
MY00`+>4<H)_E&#"/X@.@BN``,*#C!#`MY0P`G^4``)KG,O__ZY7^_^LX=@$`
M+`$``!0PG^44()_E`S"/X`(@D^<``%+C'O\O`9C^_^H@=@$```$``"P`G^4L
M,)_E``"/X"@@G^4#,(_@``!3X0(@C^`>_R\!&#"?Y0,PDN<``%/C'O\O`1/_
M+^%L=P$`9'<!`.QU`0`$`0``.`"?Y3@PG^4``(_@-""?Y0,PC^``,$/@`B"/
MX*,?H.%#$8'@P1"PX1[_+P$8,)_E`S"2YP``4^,>_R\!$_\OX2AW`0`@=P$`
MJ'4!``@!``!8,)_E6""?Y0,PC^`"((_@`##3Y0``4^,>_R\1$$`MZ0'*H.,,
MP$W@^`^,Y30PG^4#,)+G``!3XP(```HH,)_E`P"?Y_[^_^O'___K'#"?Y0$@
MH.,#,(_@`"##Y1"`O>C8=@$`9'4!`"@!``!D=@$`C'8!`,W__^KP1RWI`<J@
MXPS`3>#@#XSER#"?Y0"`4.(#,(_@*P``"KP@G^4",)/G`)"3Y0``6>,H```*
M`%#8Y0%PH.$``%7C/0!5$P%0H!,`4*`#$0``&@!`F>4``%3C#```"@E@H.$$
M`*#A!2"@X0@0H.&[_O_K``!0XP(``!H%,-3G/0!3XPL```H$0+;E``!4X_/_
M_QH$`*#A\(>]Z`A0H.$!,/7E``!3XST`4Q/[__\:"%!%X.;__^H)8$;@`5"%
MX@5`A.!&8:#A!`"@X0!@A^7PA[WH"$"@X>W__^H)0*#AZ___ZN!T`0`8`0``
M$$`MZ0'*H.,,P$W@B`^,Y8P@G^5PT$WBB#"?Y0T0H.$"((_@`$"@X0,PDN<`
M,)/E;#"-Y0`PH.-S_O_K``!0XP``H!,$```:$#"=Y0\Z`^(*"E/C`0E3$PH`
M``I(()_E0#"?Y0(@C^`#,)+G`""3Y6PPG>4"(#/@`#"@XP<``!IPT(WB$("]
MZ`(0H.,$`*#A%?[_ZQ`/;^&@`J#A[O__ZDW^_^OP<P$`'`$``*QS`0#P0"WI
M`<J@XPS`3>!P#XSE7"&?Y7S03>)8,9_E`%!0X@(@C^`#,)+G`#"3Y70PC>4`
M,*#C`@``"@`PU>4``%/C"@``&C`AG^4H,9_E`B"/X`,PDN<`()/E=#"=Y0(@
M,^``,*#C00``&GS0C>+P@+WHQ_W_ZP!`4.+Q__\*!`"@X4K^_^L``%#C*@``
M"A,PT.438(#B+@!3XP(``!H!,-;E``!3X_3__PH3,-#E+@!3XR4```H%`*#A
M[OW_ZP!PH.$&`*#AZ_W_ZP``A^`"`(#B$O[_ZZ0PG^4`<*#A`1"@XP,PC^``
M(.#C!&"-Y0!0C>4Q_O_K"!"-X@<`H.'1_?_K`0!PXP<`H.$#```*&#"=Y0\Z
M`^(!"5/C$@``"MC]_^L'`*#AOOW_ZP0`H.$?_O_K``!0X]3__QH$`*#AZ/W_
MZP4`H.$!_O_KO?__Z@$PUN4N`%/CUO__&@(PUN4``%/CQ?__"M+__^JD___K
MZ___ZNW]_^M`<P$`'`$``!1S`0!T+@``\$$MZ0'*H.,,P$W@``",Y2``#.4!
M<*#A`!&?Y0`QG^4!VDWB`1"/X`C03>(`8*#A`0J-X@0`@.($@(WB`S"1Y_\O
M`.,($*#A`#"3Y0`P@.4`,*#CS`"?Y0``C^#O_?_K`$!0X@<``-H(4(WB`3!$
MX@/`A>`$(%SE``!2XQ<``!H``%/C%```&IPPG^5P#QWN!R"@X080H.$#,)_G
M`P"0YW+]_^N$()_E`1J-XG`PG^4$$('B`B"/X`,PDN<`()/E`#"1Y0(@,^``
M,*#C$@``&@':C>((T(WB\(&]Z`-`H.$$(*#A+Q"@XP@`H.'C_?_K``!0X^/_
M_PHT`)_E<,\=[@0PH.$((*#A!A"@X0``G^<``)SG@OW_Z^'__^JA_?_KN'$!
M`!P!``"`+0``4'(!`"AQ`0#<<0$`\$<MZ0'*H.,,P$W@V`^,Y10BG^4(T$WB
M$#*?Y0%`H.$"((_@`%"@X02"G^4#,)+G"("/X``PD^4$,(WE`#"@XP`PT>4]
M`%/C`4"!`@0`H.%9_?_K#1"@X0!@H.$%`*#AT?[_ZP"04.)````*4OW_ZP``
M5N$W``":N#&?Y0!PG>4#@)CG!W&@X0`PU>4``%/C/0!3$P$PH!,`,*`#!0``
M"@4PH.$!(//E``!2XST`4A/[__\:!3!#X`(`AN(`8)CE`P"`X&;]_^L``%#C
M!P"&YQ4```H`,)CE!0"@X0<0D^<!,-#D`##!Y0``4^,]`%,3!@``"@$@@>(!
M,-#D`A"@X0$PPN0]`%/C``!3$_G__QH!(*#A/3"@XP`PP>4!,-3D`3#BY0``
M4^/[__\:""&?Y?@PG^4"((_@`S"2YP`@D^4$,)WE`B`SX``PH.,U```:"-"-
MXO"'O>@!D$GB`3#4Y`$PZ>4``%/C^___&N[__^K`,)_E`X"8YP!PF.4`,)?E
M``!3XR0```H',*#A!""SY0D0H.$!D(GB``!2X_K__QH#$('B`1&@X9`PG^4#
M,(_@`""3Y0``4N,/```:`0"@X0$@H.,`((/E)?W_ZP"@4.+5__\*!Q"@X0EQ
MH.$'(*#A,?W_ZP"@B.4`D(WE`9")X@`PH.,),8KGHO__Z@<`H.%'_?_K`*"@
MX0``4.,``(CEQ?__"@EQH.'R___J`Y"@X0@0H./?___J$/W_ZX1P`0`<`0``
M='`!`!@!``!P;P$`;'`!`!!`+>D!RJ#C#,!-X.@/C.5P$)_E<$\=[FP@G^40
MT$WB:`"?Y0(@C^!D,)_E`1"?YP,PC^``()+G`0"4YU00G^4,((WE`1"/X"']
M_^M($)_E2#"?Y0#`H.-$()_E`S"/X`$0G^<"((_@`0"4YS00G^4$P(WE`1"/
MX``PC>6W_/_K$-"-XA"`O>A`;P$`/&X!`!`!``!$:P``*"H```QO`0#L+```
M\!4``!0J```X`)_E"-!-XC0@G^5PSQWN,!"?Y0(@C^`L,)_E``"?YP,PC^`!
M()+G(!"?Y0``G.<!$(_@!""-Y0C0C>+Y_/_JI&X!`*!M`0`0`0``J&H``)`I
M``!P0"WI`<J@XPS`3>```(SE`<I,X@``C.4H``SEW""?Y0+:3>+8,)_E&-!-
MX@(@C^`"&HWB%!"!X@Q`C>(!:HWB`S"2YQ!@AN(`,)/E`#"!Y0`PH./,_/_K
MJ!"?Y0$@`>.D,)_E`1"/X/``S>$#,(_@`1"@XP0`H.'/_/_K!`"@X0$JH.,&
M$*#AQ?S_ZP!`4.(8``"Z`0"$XJ#\_^L`4%#B%```"@0@H.$&$*#A=_S_ZP`P
MH.,$,,7G4""?Y0(:C>(\,)_E%!"!X@(@C^`#,)+G`""3Y0`PD>4"(#/@`#"@
MXP,``!H%`*#A`MJ-XAC0C>)P@+WHC/S_ZP!0H./M___J-&T!`!P!``!(*0``
M-"D``)AL`0"Y___J\$\MZ0'*H.,,P$W@``",Y:@`#.5$(I_E0MU-XD`RG^4$
MT$WB`B"/X`!@H.$T`I_E0<V-XA"`C>(\P(SB`S"2YP%`H.$``(_@"!"@X0`P
MD^4`,(SE`#"@X[S]_^LO$*#C`%"@X08`H.$/_/_K``!0XQ(```KP`9_E!A"@
MX0``C^#'_O_K!@"@X>`AG^5!'8WBS#&?Y3P0@>("((_@`S"2YP`@D^4`,)'E
M`B`SX``PH.-I```:0MV-X@30C>+PC[WH!`"@X4;\_^ND$9_E`1"/X%7\_^L`
M0%#BY?__"I21G^6`H(WB`##4Y01P2N()D(_@``!3XSL```H``%7C!```"@00
MH.$%`*#A>OS_ZP``4.,K```*!`"@X1#\_^L!,$#B`S"$X`"PH.$#`%3A`#"$
M,``0H#,!`(0R`P``.@@``.H#`%#A`!##Y0(```H!('/E+P!2X_G__PH$`*#A
M__O_ZP"PH.$&`*#A_/O_ZPL`@.`!`(#B`0I0X[W__RKX`)_E"3"@X0$JH.,!
M$*#C``"/X`!`C>5!`(WI!P"@X4#\_^L($*#A!P"@X>#[_^L``%#C`P``&F`P
M&N4/.@/B`@E3XQ4```JT$)_E``"@XP$0C^`5_/_K`$!0XJ7__PH`,-3E``!3
MX\/__QH``%7C%@``"@`PU>4N`%/C`P``"H!`G^4!L*#C!$"/X-7__^H!,-7E
M``!3X^K__PKW___J`1"@XP<`H.&^^__K``!0X^3__QI0`)_E!Q"@X0``C^!8
M_O_K!P"@X>;[_^N.___J.$"?Y0&PH.,$0(_@P?__ZNS[_^LD;`$`'`$``%@H
M```L*```L&L!`.PG``#0)P``."<``.PF``"4)@``<"8``$0F```00"WI`<J@
MXPS`3>#X#XSE+Q"@XP!`H.&.^__K``!0XP%`@!($`*#A$$"]Z,C[_^KP0"WI
M`<J@XPS`3>```(SE(``,Y7@AG^4!VDWB=#&?Y0S03>("((_@`1J-X@00@>(`
M4%#B`S"2YP`PD^4`,('E`#"@XS$```H`,-7E``!3XRX```J1^__K`2"`X@!`
MH.$!"E+C.@``B@1PC>(%$*#A!P"@X8_[_^L!,$3B`S"'X`,`5^$)```J!`"'
MX`4@C>(`$*#C`@``Z@(`4.$`$,#E`@``"@$P<.4O`%/C^?__"B\0H.,'`*#A
M6OO_ZP``4.,E```*!P!0X0`PH(,`,,"%!P"@X9+[_^N\()_E`1J-XK`PG^4$
M$('B`B"/X`,PDN<`()/E`#"1Y0(@,^``,*#C(@``&@':C>(,T(WB\("]Z(@@
MG^4!&HWB>#"?Y000@>("((_@`S"2YP`@D^4`,)'E`B`SX``PH.,4```:8`"?
MY0``C^`!VHWB#-"-XO!`O>AS^__J``"@X][__^I$()_E`1J-XBPPG^4$$('B
M`B"/X`,PDN<`()/E`#"1Y0(@,^``,*#C(`"?!0``CP#L__\*;_O_ZV1I`0`<
M`0``H&@!`&AH`0"8)```'&@!`%`D``!P0"WI`<J@XPS`3>#H#XSEQ"2?Y0C0
M3>+`-)_E#4"@X0(@C^"X9)_EN%2?Y09@C^`#,)+G!5"/X``PD^4$,(WE`#"@
MXP00H.$&`*#AH_S_ZP``4.,.```*C#2?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E
M`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J5&2?Y09@C^`$
M$*#A!@"@X8W\_^L``%#C#@``"C0TG^4``)WE`S"5YP`0D^4`,8'@!""3Y0`A
M@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___Z@!DG^4&8(_@!!"@
MX08`H.%W_/_K``!0XPX```K<,Y_E``"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G
M``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^JL8Y_E!F"/X`00H.$&
M`*#A8?S_ZP``4.,.```*A#.?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``
M4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J6&.?Y09@C^`$$*#A!@"@
MX4O\_^L``%#C#@``"BPSG^4``)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C
M\?__"@0P@^($()/E!""#Y```4N/[__\:Z___Z@1CG^4&8(_@!!"@X08`H.$U
M_/_K``!0XPX```K4,I_E``"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'_
M_PH$,(/B!""3Y00@@^0``%+C^___&NO__^JP8I_E!F"/X`00H.$&`*#A'_S_
MZP``4.,.```*?#*?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*
M!#"#X@0@D^4$((/D``!2X_O__QKK___J7&*?Y09@C^`$$*#A!@"@X0G\_^L`
M`%#C#@``"B0RG^4``)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P
M@^($()/E!""#Y```4N/[__\:Z___Z@ABG^4&8(_@!!"@X08`H.'S^__K``!0
MXPX```K,,9_E``"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B
M!""3Y00@@^0``%+C^___&NO__^JT89_E!F"/X`00H.$&`*#AW?O_ZP``4.,.
M```*=#&?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@
MD^4$((/D``!2X_O__QKK___J8&&?Y09@C^`$$*#A!@"@X<?[_^L``%#C#@``
M"APQG^4``)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E
M!""#Y```4N/[__\:Z___Z@P!G^4$$*#A``"/X++[_^L`$%#B`@``"O@`G^4`
M`(_@POS_Z_``G^4$$*#A``"/X*G[_^L`$%#B`@``"MP`G^4``(_@N?S_Z]0`
MG^4$$*#A``"/X*#[_^L`$%#B$@``"L``G^4``(_@L/S_Z[@@G^5D,)_E`B"/
MX`,PDN<`()/E!#"=Y0(@,^``,*#C$```&I@0G^68`)_E`1"/X```C^`(T(WB
M<$"]Z*#\_^J$`)_E!!"@X0``C^"'^__K`!!0XNG__PIP`)_E``"/X)?\_^OE
M___J,?K_Z[AG`0`<`0``+"0``*1G`0`8`0``S",``'PC```P(P``X"(``)@B
M``#<(0``]"$``*@A``!<(0``B"```+0@``"((```I"```*0@``"@(```Q!\`
M`&!C`0!\(```X!\``$P@``#0'P``!.`MY0'*H.,,P$W@\`^,Y9@PG^4,T$WB
ME!"?Y90@G^4#,(_@`1"/X```D^4"()'G`0!PXP`@DN4$((WE`""@XPH```IP
M()_E:#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@XQ```!H,T(WB!/"=Y$@`
MG^4-$*#A``"/X$;[_^L``%#C!```"@``T.4P`%#C``!0$P$`H!,``*`#(#"?
MY0,PC^```(/EY?__ZNOY_^NL8P$`<&(!`!P!``!(8@$`X!X``"QC`0`00"WI
M`<J@XPS`3>#X#XSEF_G_Z[OY_^L``%#C``"0%1"`O>CP0"WI`<J@XPS`3>#0
M#XSE."&?Y1S03>(T,9_E$!"-X@(@C^`L89_E`%"@X09@C^`#,)+G!@"@X0`P
MD^44,(WE`#"@XQG[_^L`0%#B`@``"@`PU.4``%/C#@``&O@@G^7L,)_E`B"/
MX`,PDN<`()/E%#"=Y0(@,^``,*#C,@``&M@`G^4%$*#A``"/X!S0C>+P0+WH
M&OS_Z@40H.&9^?_K``!0XPH```JT()_EH#"?Y0(@C^`#,)+G`""3Y10PG>4"
M(#/@`#"@XQ\``!H<T(WB\("]Z`4`H.%X^?_K`'"@X00`H.%U^?_K``"'X`(`
M@.*<^?_K;#"?Y6P@G^4!$*#C`S"/X`A`C>4"((_@`%"-Y00@C>4`(.#C`$"@
MX;CY_^M(()_E*#"?Y0(@C^`#,)+G`""3Y10PG>4"(#/@`#"@XP00H`$&`*`!
MS___"HCY_^N(80$`'`$``,`>``!$80$`9!X``/A@`0`<'0``$!T``(!@`0#P
M3RWI`,"@XP+IH.,,P$W@#N!,X`'*3.(``(SE#@!<X?O__QH@`0[E;!>?Y6PG
MG^4"V4WB`1"/X&0WG^7\T$WB`'"@X0,PC^#X8(WB`B"1YP()C>)TH$;B$."#
MX@`@DN7T((#E`""@X_0`@.('`)/I*)"#X@<`BNA<P$;B:$!&X@\`ON@/`*SH
M&(>?Y>!01N((@(_@`P">Z`,`C.@'`)GH!P"$Z``'G^4%$*#A``"/X*'Z_^L`
M0%#B`@``"@`PU.4``%/C"@$`&@?Y_^LG^?_K``!0X^D```H`H)#E``!:X^8`
M``H*`*#A%_G_ZX``H.$!`(#B/OG_ZP`PVN4`0*#A`)"@X0``4^,+```*G+:?
MY0NPC^`$`*#A`#"-Y0$0H.,+,*#A`"#@XP)`A.)7^?_K`3#ZY0``4^/U__\:
M<`:?Y5R@1N(``(_@`@``Z@0`NN4``%#C#0``"@40H.%V^O_K`$!0XOC__PH`
M,-3E``!3X_7__PJJ^O_K``!0X_+__PH$`*#A$?G_ZP!`4.(.```:=*!&X@1`
MFN0``%3C"@``"@`PU.4``%/CL@``"@0`H.&;^O_K``!0X_7__PH$`*#A`OG_
MZP!`4.+Q__\*!`"@X=WX_^L`H*#A"0"@X=KX_^L`H(K@`:N*X@2@BN(*`*#A
M__C_Z[PUG^4`L*#A`1"@XP,PC^`(,(WEK#6?Y0`@X.,,D(WE`S"/X`0PC>6<
M-9_E`$"-Y0,PC^`8^?_K!QV@XPL`H.&"^/_K`0!PXP,``!HM^?_K`!"0Y1$`
M4>.=```:!1"@X0L`H.'T^/_K`)!0XA@```I8%9_E`#"7Y50EG^4!`)CG`B"/
MX`"PC>4!$*#C``"0Y>#X_^L`@*#C."6?Y0(9C>($-9_E]!"!X@(@C^`#,)+G
M`""3Y0`PD>4"(#/@`#"@XS<!`!H(`*#A`MF-XOS0C>+PC[WHT#`6Y0\Z`^(!
M"5/CXO__&LA`%N6"^/_K``!4X=[__QK0,!;E4S#HYP<-4^/:__\:"@"@X;SX
M_^L`@*#AQ`2?Y040H.$``(_@"?K_ZP`04.*.```*``"7Y3'\_^L`H%#BB@``
M"@H`H.&%^/_K`$"@X0,`4.,,``":C!2?Y00`0.(``(K@`1"/X./X_^L``%#C
M!0``&@0`5./9```*!$"*X`4P5.4O`%/CU0``"EQ$G^4$0(_@`#"4Y0$`<^/"
M```*``!3XW,```I$%)_E1`2?Y0$0C^!`=)_E``"/X#QDG^7X^O_K.%2?Y:KX
M_^LT-)_E"`"-Y0$0H.,#,(_@##"-Y20TG^4(`*#A`+"-Y0`@X.,#,(_@!#"-
MY1`TG^4'<(_@!F"/X`50C^`#,(_@I?C_ZP\``.J^^/_K`#"0Y1$`4^,1```:
MD_C_ZP&0B>(`P*#A!S"@X00@H.$!$*#C"`"@X1!@C>4,D(WE(!"-Z0"PC>64
M^/_K!QV@XP@`H.'^]__K`$"@X0$`<./I__\*"P"@X2;X_^N8`Y_E"!"@X0``
MC^#(^O_K"`"@X87^_^N$___J`T"@X5+__^IH0$;B!`"4Y```4.,I```*!1"@
MX:CY_^L`,%#B^/__"@`PT^4``%/C]?__"D?X_^L`H%#B\O__"B/X_^N``*#A
M`0"`X@K__^KT(I_E`#"7Y0(`F.<$$(WE`1"@XP"PC>48(Y_E``"0Y0(@C^!&
M^/_K9/__ZF3^_^L$(Y_EG#*?Y0(9C>("((_@]!"!X@,PDN<`()/E`#"1Y0(@
M,^``,*#CG```&@0`H.$"V8WB_-"-XO!/O>@E^/_JR**?Y0T`H.,K^/_K4S"@
MXP!`H.$*H(_@`)"@X>S^_^H`H)?E<O__Z@H`H.$`$*#C!OC_ZP!`4.*&__\*
M`##@XQ$@X.,"<*#C!%!&X@!PC>46^/_K!B"@XP40H.$$`*#AV??_ZP0P5N4`
M`%/C`@``&@,P5N5#`%/C5```"@`PH.,`(*#C`#"-Y00`H.$`,*#C!OC_Z[4$
M`.L`<*#A`@``Z@40H.$'`*#AQ00`ZP(IH.,%$*#A!`"@X</W_^L`(%#B]O__
MR@B2G^4$`*#A,*!&XN3W_^M%4$;B!Q"@X0F0C^`*0*#A,7!&XD0`1N+Z!`#K
M`2#UY00`H.$`((WE"3"@X0$0H.,`(.#C$OC_ZP)`A.('`%7A]?__&K@AG^4!
M$*#C"*"-Y0@`H.$"((_@`+"-Y00@C>4`(*#CG#&?Y0@@1N68(9_E`S"/X`(@
MC^`,((WE`"#@X__W_^MO___J@`&?Y040H.$``(_@)?G_ZP``4.,``(0%J___
M"@`PT.4``%/C,`!3$P$PH!,`,*`#`#"$Y2[__^H$0)?E``!4XR;__PH$`*#A
MFO?_ZP,`4.,B__^:,!&?Y00`0.(``(3@`1"/X/GW_^L``%#C!*"@`1K__^H"
M,%;E00!3XZ?__QH!,%;E0P!3XZ3__QH`,-;E2`!3XZ'__QH!,-;E10!3XY[_
M_QH`,.#C!`"@X0!PC>4Y(.#CIO?_ZP40H.$I(*#C!`"@X6GW_^O`,)_EP""?
MY0$0H.,#,(_@##"-Y;0PG^4"((_@"`"@X00@C>4#,(_@"%"-Y0"PC>4`(.#C
MOO?_ZR[__^J8]__K`&`!`!P!```H80$`I%\!`-P;``"`'````!P``*0;``"<
M&@``=!L``"`!``!T&P``F%T!`#`;``#X&@``]%T!`-P9``!8&0``D!H``#0;
M``#$&```7!L``-@8``!T&@``)!@``,08```P6P$`/!@``'P7``!0%@``V!<`
M`)P8``!<%@``6!<``+07```\%0``U!8``/!`+>D!RJ#C#,!-X.`/C.5<$I_E
M#-!-XE@BG^5P3QWN`1"/X%`RG^50`I_E#6"@X0(@D><``(_@!A"@X0`@DN4$
M((WE`""@XP,PG^<#4)3G9##5Y64PQ>6K^/_K``!0XP(```H`,-#E,0!3XV8`
M``KM^__K"#*?Y0,PG^<#4)3G`!*?Y04`H.$#,*#C`2"@XP$0C^#Y]O_K`%!0
MX@P```KD,9_E#""5Y0,PG^<`$)+EV"&?Y0,`E.<"((_@]?;_ZPPPE>4`$)/E
M"#"1Y0$%$^-9```:N!&?Y0,PH.,!(*#C`1"?YP$`E.>H$9_E`1"/X.+V_^L`
M4%#B"P``"I@!G^4&$*#A``"/X'_X_^L`(%#B00``"H0QG^4,$)7E`S"?YP`0
MD>4#`)3GVO;_ZW`QG^4!(*#C;!&?Y6QQG^4!$(_@`S"?YP=PC^`#4)3G!0"@
MX;OV_^M4(9_E`!"@X04`H.$"((_@R_;_ZT0QG^5$`9_E`""@XP80H.$``(_@
M`S"?YP,PE.=E(,/E7_C_ZP!`H.$``%#C"`"'Y1(```H'':#C!`"@X9KV_^L!
M((#B`#"@X1$`4.,!`%(3)0``BO@@G^6P,)_E`B"/X`,PDN<`()/E!#"=Y0(@
M,^``,*#C&P``&@S0C>+P@+WH!`"7Y7?]_^L`0*#A``!0XP@`A^7F__\:N`"?
MY0``C^`%]__K`3#0Y0``4^.9__\*E/__ZJ`PG^4,()7E`S"/X```DN4$,)/E
M`!"3Y?OX_^N[___JA#"?Y0,PG^<#`)3GU_;_ZZ#__^K>]O_K<!"?Y00@H.%L
M`)_E`1"/X```C^`$$)'E`!"1Y>OV_^OD5P$`'`$``,A8`0!X%```F%@!`,@5
M``!H6`$`I!4``#!8`0!X%0``8!,``/17`0#05P$`+!4``#18`0`@%0``E%<!
M`.`2``!`5@$`A!0``%17`0#05@$`&%<!`$@4``#P0"WI`<J@XPS`3>#@#XSE
M."&?Y0S03>(T,9_E`%"@X0(@C^`L<9_E!W"/X`,PDN<`,)/E!#"-Y0`PH./9
M^O_K+Q"@XP!`H.%@]O_K``!0XP%`@!($`*#AF_;_ZP`PE^4`0%7B`&"@X0%`
MH!,!`'/C)@``"@``4^,`0*`#`4`$$@``5.,*```:S""?Y<`PG^4"((_@`S"2
MYP`@D^4$,)WE`B`SX``PH.,G```:#-"-XO"`O>@`,-7E``!3X_'__PJ8$)_E
M!""@XP8`H.$!$(_@L/;_ZP``4./J__\:@""?Y6PPG^4"((_@`S"2YP`@D^4$
M,)WE`B`SX``PH.,2```:!0"@X0S0C>+P0+WH+_C_ZE``G^4-$*#A``"/X,7W
M_^L``%#C``"'!=;__PH`,-#E``!3XS``4Q,!,*`3`#"@`P-`!.``,(?ES/__
MZFKV_^L050$`'`$``$!6`0"@5`$`U!$``$Q4`0#<$````````/!/+>D!RJ#C
M#,!-X(`.C.403)_E5]]-X@SLG^40,(WB!$"/X`,0H.$<((#B7,"`X@[@E.<`
MX)[E5.&-Y0#@H..-"F+TX`#P\P(`7.'="D'T^O__&@9+W>T(.]WM$1Z-X@H;
MW>T```#JLA%B\@@KT^TT((/B``O3[0@P@^(!`%/ACU=B]+`!0O.U`4#SM`%`
M\[-!8_*Q,6'RH!A@\C``X?.P`6'R#@O#[>[__QH@0)#IF3D'XX(Z1>,,0)#E
M$,"=Y0!@D.4$$"[@$'"0Y0/`C.`%$`'@99&@X0?`C.`$$"'@%""=Y>;-C.`!
MP(S@"1`NX`,@@N`&$`'@9H&@X00@@N`8H)WE#A`AX`(0@>`(("G@`Z"*X`P@
M`N`.H(K@"2`BX`H@@N`<H)WE[!V!X&S!H.$#H(K@":"*X`R0*.`!D`G@X2V"
MX`B0*>!A$:#A"I")X""@G>7BG8G@`Z"*X`B@BN`!@"S@`H`(X&(AH.$,@"C@
M`K`AX`J`B.`DH)WE";`+X.F-B.`#H(K@#*"*X&G!H.$!D"O@#+`BX`J0B>`H
MH)WE"+`+X.B=B>`#H(K@`:"*X&@1H.$"@"O@`;`LX`J`B.`LH)WE";`+X.F-
MB.`#H(K@`J"*X&DAH.$,D"O@`K`AX`J0B>`PH)WE"+`+X.B=B>`#H(K@#*"*
MX&C!H.$!@"O@#+`BX`J`B.`TH)WE";`+X.F-B.`#H(K@`:"*X&D1H.$"D"O@
M`;`LX`J0B>`XH)WE"+`+X.B=B>`#H(K@`J"*X&@AH.$,@"O@`K`AX`J`B.`\
MH)WE";`+X.F-B.`#H(K@#*"*X&G!H.$!D"O@#+`BX`J0B>!`H)WE"+`+X.B=
MB>`#H(K@`:"*X&@1H.$"@"O@`;`LX`J`B.!$H)WE";`+X.F-B.`#H(K@`J"*
MX&DAH.$,D"O@`K`AX`J0B>!(H)WE"+`+X.B=B>`#H(K@#*"*X&C!H.$!@"O@
M#+`BX`J`B.!,H)WE";`+X.F-B.`#H(K@`:"*X&D1H.$"D"O@`;`LX`J0B>!0
MH)WE"+`+X.B=B>`#H(K@`J"*X&@AH.$,@"O@"H"(X%2@G>7IC8C@`Z"*X`S`
MBN`"H"'@":`*X&F1H.$!H"K@#*"*X%C`G>7HK8K@`\",X`'`C.`)$"+@"!`!
MX&B!H.$"$"'@#!"!X%S`G>7J'8'@`S",X`C`*>`*P`S@`C"#X`G`+.!@()WE
M`\",X*$[#N/9/D;C:J&@X0,@@N#AS8S@"9""X&0@G>4#((+@""""X`J`*.`!
M@"C@"(")X&&1H.%H$)WE[(V(X`,0@>`*$('@":`JX`R@*N!LP:#A"B""X.BM
M@N!L()WE`R""X`D@@N`,D"G@")`IX&B!H.$)$('@ZIV!X'`0G>4#$('@#!"!
MX`C`+.`*P"S@:J&@X0P@@N#IS8+@=""=Y0,@@N`(((+@"H`HX`F`*.!ID:#A
M"!"!X.R-@>!X$)WE`Q"!X`H0@>`)H"K@#*`JX&S!H.$*((+@Z*V"X'P@G>4#
M((+@"2""X`R0*>`(D"G@:(&@X0D0@>#JG8'@@!"=Y0,0@>`,$('@",`LX`K`
M+.!JH:#A#"""X.G-@N"$()WE`R""X`@@@N`*@"C@"8`HX&F1H.$($('@[(V!
MX(@0G>4#$('@"A"!X`F@*N`,H"K@;,&@X0H@@N#HK8+@C""=Y0,@@N!JL:#A
M"2""X`R0*>`(D"G@:(&@X0D0@>#JG8'@D!"=Y0,0@>`,$('@",`LX`K`+.`,
M((+@Z:V"X)0@G>4#((+@""""X`N`*.`)P"C@:9&@X0P0@>"8P)WEZHV!X`D0
M*^`#P(S@"A`AX`O`C."<L)WE:J&@X0$@@N`#L(O@Z!V"X`FPB^`*D"G@""`I
MX&B!H.$"P(S@H""=Y>&=C.`#((+@"B""X`B@*N`!P"K@I*"=Y6$1H.$,P(O@
M`Z"*X.G-C.`(H(K@`8`HX`F`*.!ID:#A""""X+"PG>7LC8+@J""=Y0,@@N`!
M((+@"1`AX`P0(>!LP:#A`:"*X`P0*>#HK8K@"!`AX`$@@N!H@:#AZAV"X*P@
MG>4#((+@"#`LX`HP(^`)((+@`R""X-P\"^,;/TCC:I&@X0.PB^`)H(CA#+"+
MX+3`G>7A+8+@`:`*X`/`C.!A$:#A",",X`F`".`(H(KA`8")X0N@BN`"@`C@
M`;`)X.*MBN`+@(CAO+"=Y0R`B."XP)WE8B&@X0.PB^`#P(S@`;"+X`F0C.`"
MP('A"L`,X`(0`>`!P(SAZHV(X`G`C.#`D)WE:J&@X>C-C.`*$(+A`Y")X`*0
MB>`($`'@"B`"X&B!H.$"$('A""`*X`NP@>`($(KA#!`!X.R]B^`"$('A;,&@
MX0D0@>#$D)WE#""(X>L=@>`#D(G@"R`"X`J0B>#(H)WE:[&@X0.@BN`(H(K@
M#(`(X`@@@N%A@:#A"9""X`L@C.'AG8G@`2`"X,P0G>4#$('@#!"!X`O`#.`,
MP(+A"""+X0K`C.`)(`+@"*`+X.G-C.`*((+AT*"=Y6F1H.$!$(+@`Z"*X`D@
MB.$+H(K@U+"=Y0P@`N#L'8'@`["+X&S!H.$(L(O@"8`(X`@@@N%A@:#A"J""
MX`P@B>'AK8K@`2`"X-@0G>4#$('@"1"!X`R0">`)D(+A""",X0N0B>`*(`+@
M"+`,X.J=B>`+((+AX+"=Y0$0@N#<()WE:J&@X0.PB^`#((+@"+"+X`S`@N`*
M((CA"2`"X`J`".#I'8'@"(""X6F1H.$,@(C@"2"*X6'!H.'AC8C@`2`"X.00
MG>4#$('@"A"!X`F@"N`*H(+A#"")X0N@BN`((`+@#+`)X.BMBN`+((+A[+"=
MY0$0@N#H()WE:(&@X0.PB^`#((+@#+"+X`F0@N`(((SA"B`"X`C`#.#J'8'@
M#"""X6JAH.$)D(+@"B"(X6'!H.'AG8G@`2`"X/`0G>4#$('@"!"!X`J`".`(
M@(+A#""*X0N`B.`)(`+@#+`*X.F-B.`+((+A^+"=Y0$@@N#T$)WE:9&@X0.P
MB^`#$('@#+"+X`J@@>`)$(SA"!`!X`G`#.`,P('AZ"V"X`K`C.#\H)WE:(&@
MX>+-C.`($(GA`S"*X`DP@^`"$`'@")`)X&(AH.$"H(CA"1"!X0L0@>`,H`K@
M`K`(X.P=@>!LD:#A"Z"*X0#!G>4#H(K@UC$,XV(Z3.,#P(S@X:V*X`B`C.`$
MP9WE8;&@X0/`C.`"P(S@"2`BX`$@(N`($9WE`H"(X`L@*>#JC8C@"B`BX`,0
M@>!JH:#A"1"!X`+`C.`*D"O@Z,V,X`B0*>`,(9WE"1"!X&B!H.'LG8'@$!&=
MY0,@@N`#$('@"R""X`H0@>`(H"K@#*`JX&S!H.$*((+@Z:V"X!0AG>4#((+@
M""""X`R`*.`)@"C@:9&@X0@0@>#JC8'@&!&=Y0,0@>`,$('@"<`LX`K`+.!J
MH:#A#"""X.C-@N`<(9WE`R""X`D@@N`*D"G@")`IX&B!H.$)$('@[)V!X"`1
MG>4#$('@"A"!X`B@*N`,H"K@;,&@X0H@@N#IK8+@)"&=Y0,@@N`(((+@#(`H
MX`F`*.!ID:#A"!"!X.J-@>`H$9WE`Q"!X`P0@>`)P"S@"L`LX&JAH.$,((+@
MZ,V"X"PAG>4#((+@"2""X`J0*>`(D"G@:(&@X0D0@>#LG8'@,!&=Y0,0@>`*
M$('@"*`JX`R@*N!LP:#A"B""X.FM@N`T(9WE`R""X`@@@N`,@"C@"8`HX&F1
MH.$($('@ZHV!X#@1G>4#$('@#!"!X`G`+.`*P"S@:J&@X0P@@N#HS8+@/"&=
MY0,@@N`)((+@"I`IX`B0*>!H@:#A"1"!X$"1G>7L'8'@`Y")X`J0B>`(H"K@
M#*`JX&S!H.$*((+@1*&=Y>$M@N`#H(K@"*"*X`R`*.`!@"C@81&@X0B0B>!(
M@9WEXIV)X`.`B.`,@(C@`<`LX`+`+.!B(:#A#,"*X`)PA^#IS8S@$'"`Y6SA
MCN`(X(WE3.&=Y0,PCN`&,(/@`3"#X`(0(>`)$"'@:9&@X0$0B.`)("+@[!V!
MX`P@(N`",(/@"4"$X.$]@^`,0(WE`#"-Y0%0A>`\()_E!%"-Y=\*;?0"((_@
M*#"?Y8\*0/0#,)+G`""3Y50QG>4"(#/@`#"@XP$``!I7WXWB\(^]Z%CS_^N@
M4P$`'`$``,A'`0`00"WI`<J@XPS`3>#X#XSE8`"@XTSS_^L'"]_M""O?[0D[
MW^T`(*#C!`O`[1@@@.5<((#EWRI`]!"`O>CPX=+#``````$C16>)J\WO_MRZ
MF'94,A#P1RWI`<J@XPS`3>#@#XSE%#"0Y0)@H.&"(:#A`%"@X0(PD^`8()#E
M%#"`Y0%`H.%<`)#E`2""(J8N@N`<@(7B``!0XQ@@A>4@```:/P!6XRX``-I`
MD$;B*6.@X0%PAN('<X3@!!"@X0@PH.%`0(3B`."1Y1`0@>(,P!'E$#"#X@@`
M$>4$(!'E!`!1X1#@`^4,P`/E"``#Y00@`^7S__\:`4"@X04`H.&R_/_K!P!4
MX>O__QH&8TG@!B"@X0<0H.$(`*#AY_+_ZUQ@A>7PA[WH0'!@X@``B.`&`%?A
M!G"@H0<@H.'?\O_K7#"5Y0,PA^!<,(7E0`!3X_"'O1@%`*#A!V!&X`=`A.":
M_/_KSO__Z@1PH.'G___J<$`MZ0'*H.,"BRWM#,!-X.@/C.4!0*#A%!"1Y0!0
MH.$<8(3B,3^_YA00E.48`)3ET1'EYS`OO^88*T/L`2"!XG\PX.,X`%+C`@"&
MX`$PQN=!``#:0"!BX@`0H..1\O_K!`"@X7W\_^LL,(3B4`#`\@\*1O0/"D/T
M/#"$X@\*0_1,,(3B#P=#]%0PA.($`*#A#X<#]''\_^L#,-3E`##%Y00`H.&R
M,-3A`3#%Y0`PE.4C-*#A`C#%Y0`PE.4#,,7E!S#4Y00PQ>6V,-3A!3#%Y00P
ME.4C-*#A!C#%Y00PE.4',,7E"S#4Y0@PQ>6Z,-3A"3#%Y0@PE.4C-*#A"C#%
MY0@PE.4+,,7E#S#4Y0PPQ>6^,-3A#3#%Y0PPE.4C-*#A#C#%Y0PPE.4/,,7E
M$S#4Y1`PQ>6R,=3A$3#%Y1`PE.4C-*#A$C#%Y1`PE.43,,7E`HN]['!`O>AA
M\O_J."!BX@`0H.-/\O_KQO__Z@@PG^4#,(_@`#"3Y:WR_^KD10$`"$`MZ0B`
MO>@!``(`/0```$`````+````0````!````!`````)@```$`````U````0```
M``,```!`````!@```$`````4````0`````X```!`````#0```$`````E````
M0````!(```!`````*````$`````"````0````!@```!`````"0```$`````*
M````0```````````````)7,O)7,````O<')O8R]S96QF+V5X90``1'EN84QO
M861E<CHZ8F]O=%]$>6YA3&]A9&5R`$EN=&5R;F%L<SHZ4$%2.CI"3T]4````
M`"]P<F]C+R5I+R5S`&5X90`N````4$%27U1%35``````4$%27U!23T=.04U%
M`````#H````E<R5S)7,``"\```!015),-4Q)0@````!015),3$E"`%!%4DPU
M3U!4`````%!%4DQ)3P``4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$`%!!
M4E]#3$5!3@```%!!4E]$14)51P```%!!4E]#04-(10```%!!4E]'3$]"04Q?
M1$5"54<`````4$%27T=,3T)!3%]435!$25(```!005)?5$U01$E2``!005)?
M1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#3$5!3@`````Q````3$1?3$E"4D%2
M65]0051(`%-94U1%30``)3`R>``````E<R5S)7,E<P````!P87(M`````"5S
M.B!C<F5A=&EO;B!O9B!P<FEV871E('-U8F1I<F5C=&]R>2`E<R!F86EL960@
M*&5R<FYO/25I*0H``"5S.B!P<FEV871E('-U8F1I<F5C=&]R>2`E<R!I<R!U
M;G-A9F4@*'!L96%S92!R96UO=F4@:70@86YD(')E=')Y('EO=7(@;W!E<F%T
M:6]N*0H``%!!5$@`````<&%R;``````N<&%R`````"5S)7-C86-H92TE<R5S
M```E<R5S=&5M<"TE=25S````)7,E<W1E;7`M)74M)74E<P`````8````<&5R
M;``````P````4$%2.CI086-K97(Z.E9%4E-)3TX`````,2XP-38```!5;F%B
M;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R>0`````E<SH@8W)E871I;VX@
M;V8@<')I=F%T92!C86-H92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N
M;STE:2D*`````"UE```M+0``55-%4@````!54T523D%-10````!414U01$E2
M`%1-4``O=&UP`````&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@
M)'LQ*R(D0")])PII9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE
M;&P*<&%C:V%G92!?7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F
M9G-E=%]S:7IE+"`D0T%#2$5?;6%R:V5R+"`D8V%C:&5?;F%M95]S:7IE*3L*
M0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2+G!M7&XB.PHD1DE,15]O9F9S
M971?<VEZ92`](#0[("`@(R!P86-K*").(BD*)&-A8VAE7VUA<FME<B`](")<
M,$-!0TA%(CL*)&-A8VAE7VYA;65?<VEZ92`](#0P.PI]"G-U8B!F:6YD7W!A
M<E]M86=I8PI["FUY("@D9F@I(#T@0%\["FUY("1C:'5N:U]S:7IE(#T@-C0@
M*B`Q,#(T.PIM>2`D8G5F.PIM>2`D<VEZ92`]("US("1F:#L*;7D@)'!O<R`]
M("1S:7IE("T@)'-I>F4@)2`D8VAU;FM?<VEZ93L@("`@("`C($Y/5$4Z("1P
M;W,@:7,@82!M=6QT:7!L92!O9B`D8VAU;FM?<VEZ90IW:&EL92`H)'!O<R`^
M/2`P*2!["G-E96L@)&9H+"`D<&]S+"`P.PIR96%D("1F:"P@)&)U9BP@)&-H
M=6YK7W-I>F4@*R!L96YG=&@H)%!!4E]-04=)0RD["FEF("@H;7D@)&D@/2!R
M:6YD97@H)&)U9BP@)%!!4E]-04=)0RDI(#X](#`I('L*)'!O<R`K/2`D:3L*
M<F5T=7)N("1P;W,["GT*)'!O<R`M/2`D8VAU;FM?<VEZ93L*?0IR971U<FX@
M+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL92P@)4UO
M9'5L94-A8VAE*3L*14Y$('L@:68@*"1%3E9[4$%27T-,14%.?2D@>PIR97%U
M:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ0F%S96YA;64["G)E<75I
M<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I<B`]($9I;&4Z.D)A<V5N86UE.CID
M:7)N86UE*"1P87)?=&5M<"D["F]U=',H<7%;4F5M;W9I;F<@9FEL97,@:6X@
M(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![("@@
M+60@*2`_(')M9&ER(#H@=6YL:6YK('TL("1P87)?=&5M<"D["G)M9&ER("1P
M87)?=&5M<#L*:68@*"UD("1P87)?=&5M<"`F)B`D7D\@;F4@)TU35VEN,S(G
M*2!["FUY("1T;7`@/2!N97<@1FEL93HZ5&5M<"@*5$5-4$Q!5$4@/3X@)W1M
M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1T;W!D
M:7(I+`I3549&25@@/3X@)RYC;60G+`I53DQ)3DL@/3X@,"P**3L*;7D@)&9I
M;&5N86UE(#T@)'1M<"T^9FEL96YA;64["G!R:6YT("1T;7`@/#PB+BXN(CL*
M>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@+7)F("<D<&%R7W1E
M;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@73L@=&AE;@IB<F5A:PIF:0IS
M;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G)&9I;&5N86UE)PHN
M+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P,"P@)&9I;&5N86UE.PIM>2`D8VUD
M(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL(#(^)C$@)B(["G-Y<W1E;2@D8VUD
M*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F
M;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI.PI]"GT@?0I"14=)3B!["DEN=&5R
M;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE9"`F26YT97)N86QS.CI005(Z
M.D)/3U0["F5V86P@>PI?<&%R7VEN:71?96YV*"D["FUY("1Q=6EE="`]("$D
M14Y6>U!!4E]$14)51WT["FUY("5S>7,@/2`H"G!A=&A?<V5P("`@(#T^("@D
M7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I+`I?97AE("`@("`@("`]/B`H
M)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE>&4G(#H@)R<I
M+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>35-7:6Y\3U,R+R`_("=<7"<@
M.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D["F]U=',H<7%;7"1P<F]G;F%M
M92`]("(D<')O9VYA;64B72D["E]S971?<&%R7W1E;7`H*3L*;W5T<RAQ<5M<
M)$5.5GM005)?5$5-4'T@/2`B)$5.5GM005)?5$5-4'TB72D["FUY("@D<W1A
M<G1?<&]S+"`D9&%T85]P;W,I.PI-04=)0SH@>PIL;V-A;"`D4TE'>U]?5T%2
M3E]??2`]('-U8B![?3L*=6YL97-S("AO<&5N(%]&2"P@)SPZ<F%W)RP@)'!R
M;V=N86UE*2!["F]U=',H<7%;0V%N)W0@<F5A9"!F<F]M(&9I;&4@(B1P<F]G
M;F%M92)=*3L@(",@9&]N)W0@=7-E("0A(&AE<F4@87,@:70@<F5Q=6ER97,@
M17)R;F\N<&T*;&%S="!-04=)0SL*?0IM>2`D;6%G:6-?<&]S(#T@9FEN9%]P
M87)?;6%G:6,H*E]&2"D["FEF("@D;6%G:6-?<&]S(#P@,"D@>PIO=71S*'%Q
M6T-A;B=T(&9I;F0@;6%G:6,@<W1R:6YG("(D4$%27TU!1TE#(B!I;B!F:6QE
M("(D<')O9VYA;64B72D["FQA<W0@34%'24,["GT*;W5T<R@B1F]U;F0@4$%2
M(&UA9VEC(&%T('!O<VET:6]N("1M86=I8U]P;W,B*3L*<V5E:R!?1D@L("1M
M86=I8U]P;W,@+2`D1DE,15]O9F9S971?<VEZ92P@,#L*<F5A9"!?1D@L("1B
M=68L("1&24Q%7V]F9G-E=%]S:7IE.PIM>2`D;V9F<V5T(#T@=6YP86-K*").
M(BP@)&)U9BD["F]U=',H(D]F9G-E="!F<F]M('-T87)T(&]F($9)3$5S(&ES
M("1O9F9S970B*3L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`D1DE,15]O9F9S
M971?<VEZ92`M("1O9F9S970L(#`["B1D871A7W!O<R`]('1E;&P@7T9(.PIM
M>2`E<F5Q=6ER95]L:7-T.PIR96%D(%]&2"P@)&)U9BP@-#L@("`@("`@("`@
M("`@("`@("`@("`@("`@("`C(')E860@=&AE(&9I<G-T(")&24Q%(@IW:&EL
M92`H)&)U9B!E<2`B1DE,12(I('L*<F5A9"!?1D@L("1B=68L(#0["G)E860@
M7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*;7D@)&9U;&QN86UE(#T@
M)&)U9CL*;W5T<RAQ<5M5;G!A8VMI;F<@1DE,12`B)&9U;&QN86UE(BXN+ETI
M.PIM>2`D8W)C(#T@*"`D9G5L;&YA;64@/7X@<WQ>*%MA+69<9%U[.'TI+WQ\
M("D@/R`D,2`Z('5N9&5F.PIM>2`H)&)A<V5N86UE+"`D97AT*2`]("@D8G5F
M(#U^(&U\*#\Z+BHO*3\H+BHI*%PN+BHI?"D["G)E860@7T9(+"`D8G5F+"`T
M.PIR96%D(%]&2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD["FEF("AD969I
M;F5D*"1E>'0I(&%N9"`D97AT("%^("]<+B@_.G!M?'!L?&EX?&%L*20O:2D@
M>PIM>2`D9FEL96YA;64@/2!?<V%V95]A<R@B)&-R8R1E>'0B+"`D8G5F+"`P
M-S4U*3L*)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9G5L;&YA;65](#T@)&9I
M;&5N86UE.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F:6QE;F%M97T@/2`D
M9G5L;&YA;64["GT*96QS:68@*"`D9G5L;&YA;64@/7X@;7Q>+S]S:&QI8B]\
M(&%N9"!D969I;F5D("1%3E9[4$%27U1%35!]("D@>PIM>2`D9FEL96YA;64@
M/2!?<V%V95]A<R@B)&)A<V5N86UE)&5X="(L("1B=68L(#`W-34I.PIO=71S
M*")32$Q)0CH@)&9I;&5N86UE7&XB*3L*?0IE;'-E('L*)$UO9'5L94-A8VAE
M>R1F=6QL;F%M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M
M92`]/B`D9G5L;&YA;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK
M.PI]"G)E860@7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*
M;7D@*"1S96QF+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U
M;&4@;W(@(21M;V1U;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U
M;&5].PIR971U<FX@=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES
M='LD;6]D=6QE?3L*)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M
M;V1U;&4B.PII9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ
M1FEL93HZ;F5W*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@
M;W(@9&EE(")#86XG="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN
M;6]D92@I.PHD9F@M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P
M+"`P*3L*<F5T=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A
M=F5?87,H(B1I;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O
M+3Y[9FEL97T@/2`D9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D
M9FEL96YA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A
M73L*<F5T=7)N("1F:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C
M86XG="!F:6YD(&UO9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E
M(%A33&]A9&5R.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ
M2&5A=GD["G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?
M:6YI=%]D>6YA;&]A9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM
M>2`D9FEL96YA;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2![
M"G5N;&5S<R`H)$E.0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$
M4$%.+RD@>PII9B`H)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER
M92`D9FEL96YA;64["GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I
M;&5N86UE(#U^("]S:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R
M97%U:7)E7VQI<W1[)&9I;&5N86UE?3L*?0IU;FQE<W,@*"1B=68@97$@(E!+
M7#`P,UPP,#0B*2!["F]U=',H<7%;3F\@>FEP(&9O=6YD(&%F=&5R($9)3$4@
M<V5C=&EO;B!I;B!F:6QE("(D<')O9VYA;64B72D["FQA<W0@34%'24,@.PI]
M"B1S=&%R=%]P;W,@/2`H=&5L;"!?1D@I("T@-#L@("`@("`@("`@("`@("`@
M(R!S=&%R="!O9B!Z:7`*?0IM>2!`<&%R7V%R9W,["FUY("@D;W5T+"`D8G5N
M9&QE+"`D;&]G9F@L("1C86-H95]N86UE*3L*9&5L971E("1%3E9[4$%27T%0
M4%]2155317T[(",@<V%N:71I>F4@*%)%55-%(&UA>2!B92!A('-E8W5R:71Y
M('!R;V)L96TI"FEF("@A)'-T87)T7W!O<R!O<B`H)$%21U9;,%T@97$@)RTM
M<&%R+6]P=&EO;G,G("8F('-H:69T*2D@>PIM>2`E9&ES=%]C;60@/2!Q=R@*
M<"`@(&)L:6)?=&]?<&%R"FD@("!I;G-T86QL7W!A<@IU("`@=6YI;G-T86QL
M7W!A<@IS("`@<VEG;E]P87(*=B`@('9E<FEF>5]P87(**3L*:68@*$!!4D=6
M(&%N9"`D05)'5ELP72!E<2`G+2UR975S92<I('L*<VAI9G0@0$%21U8["B1%
M3E9[4$%27T%04%]2155317T@/2!S:&EF="!`05)'5CL*?0IE;'-E('L@(R!N
M;W)M86P@<&%R;"!B96AA=FEO=7(*;7D@0&%D9%]T;U]I;F,["G=H:6QE("A`
M05)'5BD@>PHD05)'5ELP72`]?B`O7BTH6T%)34]"3&)Q<&EU<U1V72DH+BHI
M+R!O<B!L87-T.PII9B`H)#$@97$@)TDG*2!["G!U<V@@0&%D9%]T;U]I;F,L
M("0R.PI]"F5L<VEF("@D,2!E<2`G32<I('L*979A;"`B=7-E("0R(CL*?0IE
M;'-I9B`H)#$@97$@)T$G*2!["G5N<VAI9G0@0'!A<E]A<F=S+"`D,CL*?0IE
M;'-I9B`H)#$@97$@)T\G*2!["B1O=70@/2`D,CL*?0IE;'-I9B`H)#$@97$@
M)V(G*2!["B1B=6YD;&4@/2`G<VET92<["GT*96QS:68@*"0Q(&5Q("=")RD@
M>PHD8G5N9&QE(#T@)V%L;"<["GT*96QS:68@*"0Q(&5Q("=Q)RD@>PHD<75I
M970@/2`Q.PI]"F5L<VEF("@D,2!E<2`G3"<I('L*;W!E;B`D;&]G9F@L("(^
M/B(L("0R(&]R(&1I92!Q<5M#86XG="!O<&5N(&QO9R!F:6QE("(D,B(Z("0A
M73L*?0IE;'-I9B`H)#$@97$@)U0G*2!["B1C86-H95]N86UE(#T@)#(["GT*
M<VAI9G0H0$%21U8I.PII9B`H;7D@)&-M9"`]("1D:7-T7V-M9'LD,7TI('L*
M9&5L971E("1%3E9[)U!!4E]414U0)WT["FEN:71?:6YC*"D["G)E<75I<F4@
M4$%2.CI$:7-T.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@I('5N;&5S<R!`05)'
M5CL*)GLB4$%2.CI$:7-T.CHD8VUD(GTH)%\I(&9O<B!`05)'5CL*97AI=#L*
M?0I]"G5N<VAI9G0@0$E.0RP@0&%D9%]T;U]I;F,["GT*?0II9B`H)&]U="D@
M>PI["G)E<75I<F4@24\Z.D9I;&4["G)E<75I<F4@07)C:&EV93HZ6FEP.PIR
M97%U:7)E($1I9V5S=#HZ4TA!.PI]"FUY("1P87(@/2!S:&EF="A`05)'5BD[
M"FUY("1Z:7`["FEF("AD969I;F5D("1P87(I('L*;W!E;B!M>2`D9F@L("<\
M.G)A=R<L("1P87(@;W(@9&EE('%Q6T-A;B=T(&9I;F0@<&%R(&9I;&4@(B1P
M87(B.B`D(5T["F)L97-S*"1F:"P@)TE/.CI&:6QE)RD["B1Z:7`@/2!!<F-H
M:79E.CI::7`M/FYE=SL**"`D>FEP+3YR96%D1G)O;49I;&5(86YD;&4H)&9H
M+"`D<&%R*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"D@*0IO<B!D:64@<7%;
M17)R;W(@<F5A9&EN9R!Z:7`@87)C:&EV92`B)'!A<B)=.PI]"FUY("5M971A
M7W!A<B`](&1O('L*:68@*"1Z:7`@86YD(&UY("1M971A(#T@)'II<"T^8V]N
M=&5N=',H)TU%5$$N>6UL)RDI('L*)&UE=&$@/7X@<R\N*EYP87(Z)"\O;7,[
M"B1M971A(#U^(',O7EQ3+BHO+VUS.PHD;65T82`]?B`O7B`@*%M>.ETK*3H@
M*"XK*20O;6<["GT*?3L*:68@*&1E9FEN960@)'!A<BD@>PIO<&5N(&UY("1P
M:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@<F5A9"!P87(@9FEL
M92`B)'!A<B(Z("0A73L*;7D@)&)U9CL*<F5A9"`D<&@L("1B=68L(#0["F1I
M92!Q<5LB)'!A<B(@:7,@;F]T(&$@<&%R(&9I;&5=('5N;&5S<R`D8G5F(&5Q
M(")02UPP,#-<,#`T(CL*8VQO<V4@)'!H.PI]"D-R96%T95!A=&@H)&]U="D@
M.PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W*`HD;W5T+`I)3SHZ1FEL93HZ3U]#
M4D5!5"@I('P@24\Z.D9I;&4Z.D]?5U)/3DQ9*"D@?"!)3SHZ1FEL93HZ3U]4
M4E5.0R@I+`HP-S<W+`HI(&]R(&1I92!Q<5M#86XG="!C<F5A=&4@9FEL92`B
M)&]U="(Z("0A73L*)&9H+3YB:6YM;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM
M>2`D;&]A9&5R.PII9B`H9&5F:6YE9"`D9&%T85]P;W,I('L*<F5A9"!?1D@L
M("1L;V%D97(L("1D871A7W!O<SL*?2!E;'-E('L*;&]C86P@)"\@/2!U;F1E
M9CL*)&QO861E<B`](#Q?1D@^.PI]"FEF("@A)$5.5GM005)?5D520D%424U]
M(&%N9"`D;&]A9&5R(#U^("]>*#\Z(R%\7$!R96TI+RD@>PIR97%U:7)E(%!!
M4CHZ1FEL=&5R.CI0;V13=')I<#L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YA
M<'!L>2A<)&QO861E<BP@)#`I.PI]"FEF("@D;65T85]P87)[8VQE86Y]*2![
M"FUY("1P87)?8VQE86X@/2`B/3$B.PIM>2`D<&%S<U]P87)?8VQE86X@/2!U
M8R`B7U]P87-S7W!A<E]C;&5A;E]?(CL*)&QO861E<B`]?B!S>R@_/#TD>W!A
M<W-?<&%R7V-L96%N?2DH("LI?0I[)'!A<E]C;&5A;B`N("@B("(@>"`H;&5N
M9W1H*"0Q*2`M(&QE;F=T:"@D<&%R7V-L96%N*2DI?65G.PI]"B1F:"T^<')I
M;G0H)&QO861E<BD*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@;&]A9&5R('1O
M("(D;W5T(CH@)"%=.PII9B`H)&)U;F1L92D@>PIR97%U:7)E(%!!4CHZ2&5A
M=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*:6YI=%]I;F,H
M*3L*<F5Q=6ER95]M;V1U;&5S*"D["FUY($!I;F,@/2!G<F5P('L@(2]"4T10
M04XO('T*9W)E<"!["B@D8G5N9&QE(&YE("=S:71E)RD@;W(**"1?(&YE("1#
M;VYF:6<Z.D-O;F9I9WMA<F-H;&EB97AP?2!A;F0*)%\@;F4@)$-O;F9I9SHZ
M0V]N9FEG>W!R:79L:6)E>'!]*3L*?2!`24Y#.PIS?"\K?"]\9RP@<WPO)'Q\
M(&9O<F5A8V@@0&EN8SL*;7D@)69I;&5S.PHD9FEL97-[)%]]*RL@9F]R($!$
M>6YA3&]A9&5R.CID;%]S:&%R961?;V)J96-T<RP@=F%L=65S("5)3D,["FUY
M("1L:6)?97AT(#T@)$-O;F9I9SHZ0V]N9FEG>VQI8E]E>'1].R`@("`@("`@
M(",@6%A8(&QI8E]E>'0@=G,@9&QE>'0@/PIM>2`E=W)I='1E;CL*9F]R96%C
M:"!M>2`D:V5Y("AS;W)T(&ME>7,@)69I;&5S*2!["FUY("@D9FEL92P@)&YA
M;64I.PII9B`H9&5F:6YE9"AM>2`D9F,@/2`D4$%2.CI(96%V>3HZ1G5L;$-A
M8VAE>R1K97E]*2D@>PHH)&9I;&4L("1N86UE*2`]("@D:V5Y+"`D9F,I.PI]
M"F5L<V4@>PIF;W)E86-H(&UY("1D:7(@*$!I;F,I('L*:68@*"1K97D@/7X@
M;7Q>7%$D9&ER7$4O*"XJ*21\:2D@>PHH)&9I;&4L("1N86UE*2`]("@D:V5Y
M+"`D,2D["FQA<W0["GT*:68@*"1K97D@/7X@;7Q>+VQO861E<B];7B]=*R\H
M+BHI)'PI('L*:68@*&UY("1R968@/2`D36]D=6QE0V%C:&5[)#%]*2!["B@D
M9FEL92P@)&YA;64I(#T@*"1R968L("0Q*3L*;&%S=#L*?0II9B`H+68@(B1D
M:7(O)#$B*2!["B@D9FEL92P@)&YA;64I(#T@*"(D9&ER+R0Q(BP@)#$I.PIL
M87-T.PI]"GT*?0I]"FYE>'0@=6YL97-S(&1E9FEN960@)&YA;64["FYE>'0@
M:68@)'=R:71T96Y[)&YA;65]*RL["FYE>'0@:68@(7)E9B@D9FEL92D@86YD
M("1F:6QE(#U^("]<+EQ1)&QI8E]E>'1<120O:3L*;W5T<RAS<')I;G1F*'%Q
M6U!A8VMI;F<@1DE,12`B)7,B+BXN72P@<F5F("1F:6QE(#\@)&9I;&4M/GMN
M86UE?2`Z("1F:6QE*2D["FUY("1C;VYT96YT.PII9B`H<F5F*"1F:6QE*2D@
M>PHD8V]N=&5N="`]("1F:6QE+3Y[8G5F?3L*?0IE;'-E('L*;&]C86P@)"\@
M/2!U;F1E9CL*;W!E;B!M>2`D=&@L("<\.G)A=R<L("1F:6QE(&]R(&1I92!Q
M<5M#86XG="!R96%D("(D9FEL92(Z("0A73L*)&-O;G1E;G0@/2`\)'1H/CL*
M8VQO<V4@)'1H.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD8V]N
M=&5N="P@(CQE;6)E9&1E9#XO)&YA;64B*0II9B`A)$5.5GM005)?5D520D%4
M24U](&%N9"`D;F%M92`]?B`O7"XH/SIP;7QI>'QA;"DD+VD["E!!4CHZ1FEL
M=&5R.CI0871C:$-O;G1E;G0M/FYE=RT^87!P;'DH7"1C;VYT96YT+"`D9FEL
M92P@)&YA;64I.PI]"B1F:"T^<')I;G0H(D9)3$4B+`IP86-K*"=.)RP@;&5N
M9W1H*"1N86UE*2`K(#DI+`IS<')I;G1F*"(E,#AX+R5S(BP@07)C:&EV93HZ
M6FEP.CIC;VUP=71E0U)#,S(H)&-O;G1E;G0I+"`D;F%M92DL"G!A8VLH)TXG
M+"!L96YG=&@H)&-O;G1E;G0I*2P*)&-O;G1E;G0I"F]R(&1I92!Q<5M%<G)O
M<B!W<FET:6YG(&5M8F5D9&5D($9)3$4@=&\@(B1O=70B.B`D(5T["F]U=',H
M<7%;5W)I='1E;B!A<R`B)&YA;64B72D["GT*?0II9B`H)'II<"D@>PHD>FEP
M+3YW<FET951O1FEL94AA;F1L92@D9F@I(#T]($%R8VAI=F4Z.EII<#HZ05I?
M3TL*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@>FEP('!A<G0@;V8@(B1O=70B
M73L*?0HD8V%C:&5?;F%M92`]('-U8G-T<B`D8V%C:&5?;F%M92P@,"P@)&-A
M8VAE7VYA;65?<VEZ93L*:68@*"$D8V%C:&5?;F%M92!A;F0@;7D@)&UT:6UE
M(#T@*'-T870H)&]U="DI6SE=*2!["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^
M;F5W*#$I.PIO<&5N(&UY("1T:"P@(CPZ<F%W(BP@)&]U=#L*)&-T>"T^861D
M9FEL92@D=&@I.PIC;&]S92`D=&@["B1C86-H95]N86UE(#T@)&-T>"T^:&5X
M9&EG97-T.PI]"B1C86-H95]N86UE("X](")<,"(@>"`H)&-A8VAE7VYA;65?
M<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M92D["B1C86-H95]N86UE("X]("1C
M86-H95]M87)K97(["FUY("1O9F9S970@/2`D9F@M/G1E;&P@*R!L96YG=&@H
M)&-A8VAE7VYA;64I("T@;&5N9W1H*"1L;V%D97(I.PHD9F@M/G!R:6YT*"1C
M86-H95]N86UE+`IP86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I"B8F
M("1F:"T^8VQO<V4*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@=')A:6QE<B!O
M9B`B)&]U="(Z("0A73L*8VAM;V0@,#<U-2P@)&]U=#L*97AI=#L*?0I["FQA
M<W0@=6YL97-S(&1E9FEN960@)'-T87)T7W!O<SL*7V9I>%]P<F]G;F%M92@I
M.PIR97%U:7)E(%!!4CL*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I
M.PI["G)E<75I<F4@1FEL93HZ1FEN9#L*<F5Q=6ER92!!<F-H:79E.CI::7`[
M"GT*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=SL@("`@("`@("`@("`@("`@("`@
M("`@("`@("`@(",@07)C:&EV93HZ6FEP(&]P97)A=&5S(&]N(&%N($E/.CI(
M86YD;&4*)&9H+3YF9&]P96XH9FEL96YO*%]&2"DL("=R)RD@;W(@9&EE('%Q
M6V9D;W!E;B@I(&9A:6QE9#H@)"%=.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK
M4VEZ92@M<R!?1D@I.PIM>2`D>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@D
M>FEP+3YR96%D1G)O;49I;&5(86YD;&4H)&9H+"`D<')O9VYA;64I(#T]($%R
M8VAI=F4Z.EII<#HZ05I?3TLH*2D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@
M>FEP(&%R8VAI=F4@(B1P<F]G;F%M92)=.PI!<F-H:79E.CI::7`Z.G-E=$-H
M=6YK4VEZ92@V-"`J(#$P,C0I.PIP=7-H($!005(Z.DQI8D-A8VAE+"`D>FEP
M.PHD4$%2.CI,:6)#86-H97LD<')O9VYA;65](#T@)'II<#L*;W5T<R@B17AT
M<F%C=&EN9R!Z:7`N+BXB*3L*:68@*&1E9FEN960@)$5.5GM005)?5$5-4'TI
M('L@(R!S:&]U;&0@8F4@<V5T(&%T('1H:7,@<&]I;G0A"F9O<F5A8V@@;7D@
M)&UE;6)E<B`H("1Z:7`M/FUE;6)E<G,@*2!["FYE>'0@:68@)&UE;6)E<BT^
M:7-$:7)E8W1O<GD["FUY("1M96UB97)?;F%M92`]("1M96UB97(M/F9I;&5.
M86UE.PIN97AT('5N;&5S<R`D;65M8F5R7VYA;64@/7X@;7L*7@HO/W-H;&EB
M+PHH/SHD0V]N9FEG.CI#;VYF:6=[=F5R<VEO;GTO*3\**#\Z)$-O;F9I9SHZ
M0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R86-T
M7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A=&9I
M;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@)&1E
M<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I>F4H
M*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E(&ET
M(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L<V4@>PIO
M=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T7VYA
M;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E<W1?
M;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;17)R
M;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE(ET["F-H
M;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*?0I]
M"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I92`\
M/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@72!;
M("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A;2YP
M;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.5GM0
M05)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'5BD[
M"GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I<F4@
M1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D97AT*2`]
M($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I*3L*
M<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A=&@I
M('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O<@I]
M"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER92!$
M>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T.PIR
M97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R<#L*
M<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER92!%
M>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@1F-N
M=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C.PIR
M97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)3SHZ
M2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S.CI:
M;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(
M03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!0
M05(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR97%U
M:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T<FEB
M=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E(%=I
M;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L('L@
M<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E<75I
M<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I<F4@
M4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R97%U
M:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I;F5D
M("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\I
M('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P871H
M("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%35!$
M25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*;F5X
M="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW("1P
M871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I9"D@
M/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN960H
M)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z.DQO
M9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N86UE
M(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).04U%
M?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"FUY("1S=&UP9&ER(#T@
M(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R;F%M
M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER.PII9B`H
M)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["GT*
M96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@9&EE
M('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM>2`D;6%G
M:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS965K("1F
M:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H*"1C
M86-H95]M87)K97(I+"`P.PIM>2`D8G5F.PIR96%D("1F:"P@)&)U9BP@;&5N
M9W1H*"1C86-H95]M87)K97(I.PII9B`H)&)U9B!E<2`D8V%C:&5?;6%R:V5R
M*2!["G-E96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*
M+2!L96YG=&@H)&-A8VAE7VUA<FME<BD*+2`D8V%C:&5?;F%M95]S:7IE+"`P
M.PIR96%D("1F:"P@)&)U9BP@)&-A8VAE7VYA;65?<VEZ93L*)&)U9B`]?B!S
M+UPP+R]G.PHD8V%C:&5?9&ER(#T@(F-A8VAE+21B=68B.PI]"GT*8VQO<V4@
M)&9H.PI]"FEF("@A)&-A8VAE7V1I<BD@>PHD8V%C:&5?9&ER(#T@(G1E;7`M
M)"0B.PHD14Y6>U!!4E]#3$5!3GT@/2`Q.PI]"B1S=&UP9&ER("X]("(D<WES
M>U]D96QI;7TD8V%C:&5?9&ER(CL*;6MD:7(@)'-T;7!D:7(L(#`W-34["B1%
M3E9[4$%27U1%35!](#T@)'-T;7!D:7(["FQA<W0["GT*)'!A<E]T96UP(#T@
M)#$@:68@)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H
M+BLI+SL*?0IS=6(@7W-A=F5?87,@>PIM>2`H)&YA;64L("1C;VYT96YT<RP@
M)&UO9&4I(#T@0%\["FUY("1F=6QL;F%M92`]("(D<&%R7W1E;7`O)&YA;64B
M.PIU;FQE<W,@*"UE("1F=6QL;F%M92D@>PIM>2`D=&5M<&YA;64@/2`B)&9U
M;&QN86UE+B0D(CL*;7D@)&9H.PHH;W!E;B`D9F@L("<^.G)A=R<L("1T96UP
M;F%M92D*)B8@*'!R:6YT("1F:"`D8V]N=&5N=',I"B8F("AC;&]S92`D9F@I
M"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG("(D=&5M<&YA;64B.B`D(5T["F-H
M;6]D("1M;V1E+"`D=&5M<&YA;64@:68@9&5F:6YE9"`D;6]D93L*<F5N86UE
M*"1T96UP;F%M92P@)&9U;&QN86UE*2!O<B!U;FQI;FLH)'1E;7!N86UE*3L*
M?0IR971U<FX@)&9U;&QN86UE.PI]"G-U8B!?<V5T7W!R;V=N86UE('L*:68@
M*&1E9FEN960@)$5.5GM005)?4%)/1TY!345](&%N9"`D14Y6>U!!4E]04D]'
M3D%-17T@/7X@+R@N*RDO*2!["B1P<F]G;F%M92`]("0Q.PI]"B1P<F]G;F%M
M92!\?#T@)#`["FEF("@D14Y6>U!!4E]414U0?2!A;F0@:6YD97@H)'!R;V=N
M86UE+"`D14Y6>U!!4E]414U0?2D@/CT@,"D@>PHD<')O9VYA;64@/2!S=6)S
M='(H)'!R;V=N86UE+"!R:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI
M("L@,2D["GT*:68@*"$D14Y6>U!!4E]04D]'3D%-17T@;W(@:6YD97@H)'!R
M;V=N86UE+"`D<WES>U]D96QI;7TI(#X](#`I('L*:68@*&]P96X@;7D@)&9H
M+"`G/"<L("1P<F]G;F%M92D@>PIR971U<FX@:68@+7,@)&9H.PI]"FEF("@M
M<R`B)'!R;V=N86UE)'-Y<WM?97AE?2(I('L*)'!R;V=N86UE("X]("1S>7-[
M7V5X97T["G)E='5R;CL*?0I]"F9O<F5A8V@@;7D@)&1I<B`H<W!L:70@+UQ1
M)'-Y<WMP871H7W-E<'U<12\L("1%3E9[4$%42'TI('L*;F5X="!I9B!E>&ES
M=',@)$5.5GM005)?5$5-4'T@86YD("1D:7(@97$@)$5.5GM005)?5$5-4'T[
M"B1D:7(@/7X@<R]<421S>7-[7V1E;&EM?5Q%)"\O.PHH*"1P<F]G;F%M92`]
M("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(I+"!L87-T
M*0II9B`M<R`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M921S>7-[7V5X97TB
M.PHH*"1P<F]G;F%M92`]("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE(BDL
M(&QA<W0I"FEF("US("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE(CL*?0I]
M"G-U8B!?9FEX7W!R;V=N86UE('L*)#`@/2`D<')O9VYA;64@?'P]("1%3E9[
M4$%27U!23T=.04U%?3L*:68@*&EN9&5X*"1P<F]G;F%M92P@)'-Y<WM?9&5L
M:6U]*2`\(#`I('L*)'!R;V=N86UE(#T@(BXD<WES>U]D96QI;7TD<')O9VYA
M;64B.PI]"FUY("1P=V0@/2`H9&5F:6YE9"`F0W=D.CIG971C=V0I(#\@0W=D
M.CIG971C=V0H*0HZ("@H9&5F:6YE9"`F5VEN,S(Z.D=E=$-W9"D@/R!7:6XS
M,CHZ1V5T0W=D*"D@.B!@<'=D8"D["F-H;VUP*"1P=V0I.PHD<')O9VYA;64@
M/7X@<R]>*#\]7"Y<+C]<421S>7-[7V1E;&EM?5Q%*2\D<'=D)'-Y<WM?9&5L
M:6U]+SL*)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE.PI]"G-U8B!?
M<&%R7VEN:71?96YV('L*:68@*"`D14Y6>U!!4E])3DE424%,25I%1'TK*R`]
M/2`Q("D@>PIR971U<FX["GT@96QS92!["B1%3E9[4$%27TE.251)04Q)6D5$
M?2`](#(["GT*9F]R("AQ=R@@4U!!5TY%1"!414U0($-,14%.($1%0E5'($-!
M0TA%(%!23T=.04U%("D@*2!["F1E;&5T92`D14Y6>R)005)?)%\B?3L*?0IF
M;W(@*'%W+R!435!$25(@5$5-4"!#3$5!3B!$14)51R`O*2!["B1%3E9[(E!!
M4E\D7R)](#T@)$5.5GLB4$%27T=,3T)!3%\D7R)](&EF(&5X:7-T<R`D14Y6
M>R)005)?1TQ/0D%,7R1?(GT["GT*:68@*"1%3E9[4$%27U1%35!]*2!["F1E
M;&5T92`D14Y6>U!!4E]#3$5!3GT["GT*?0IS=6(@;W5T<R!["G)E='5R;B!I
M9B`D<75I970["FEF("@D;&]G9F@I('L*<')I;G0@)&QO9V9H(")`7UQN(CL*
M?0IE;'-E('L*<')I;G0@(D!?7&XB.PI]"GT*<W5B(&EN:71?:6YC('L*<F5Q
M=6ER92!#;VYF:6<["G!U<V@@0$E.0RP@9W)E<"!D969I;F5D+"!M87`@)$-O
M;F9I9SHZ0V]N9FEG>R1??2P@<7<H"F%R8VAL:6)E>'`@<')I=FQI8F5X<"!S
M:71E87)C:&5X<"!S:71E;&EB97AP"G9E;F1O<F%R8VAE>'`@=F5N9&]R;&EB
M97AP"BD["GT*<&%C:V%G92!M86EN.PIR97%U:7)E(%!!4CL*=6YS:&EF="!`
M24Y#+"!<)E!!4CHZ9FEN9%]P87(["E!!4BT^:6UP;W)T*$!P87)?87)G<RD[
M"F1I92!Q<5MP87(N<&PZ($-A;B=T(&]P96X@<&5R;"!S8W)I<'0@(B1P<F]G
M;F%M92(Z($YO('-U8V@@9FEL92!O<B!D:7)E8W1O<GE<;ET*=6YL97-S("UE
M("1P<F]G;F%M93L*9&\@)'!R;V=N86UE.PI#3U)%.CIE>&ET*"0Q*2!I9B`H
M)$`@/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)$`@:68@)$`["GT[
M"B0Z.E]?15)23U(@/2`D0"!I9B`D0#L*?0I#3U)%.CIE>&ET*"0Q*2!I9B`H
M)#HZ7U]%4E)/4B`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D.CI?
M7T524D]2(&EF("0Z.E]?15)23U(["C$["E]?14Y$7U\*`````'!E<FQX<VDN
M8P```#T```!`````"P```$`````0````0````"8```!`````-0```$`````#
M````0`````8```!`````%````$`````.````0`````T```!`````)0```$``
M```2````0````"@```!``````@```$`````8````0`````D```!`````"@``
M`$```````````````(&R`8&PL*P'`````+ZR`8&PL*\_`````!")_W^PJ@F`
MO(O_?P$```#`CO]_L*L>@#R0_W_,__]_;)'_?P$```"HD_]_L*@#@$R4_W\!
M````+*#_?["PJ(!(H/]_`0```+2A_W^H__]_W*G_?["K`H"HK/]_`0``````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``#,&0``6!D```$````W`P```0```$(#```!````3`,```$```!@`P``#P``
M`.`$```,````N!(```T````H2@``&0```+"-`0`;````!````!H```"TC0$`
M'`````0```#U_O]OM`$```4```"P"0``!@```!`#```*`````P4```L````0
M````%0`````````#````T(X!``(```#@`0``%````!$````7````V!```!$`
M```($```$@```-`````3````"````!@`````````^___;P$```C^__]OB`\`
M`/___V\#````\/__;[0.``#Z__]O#@``````````````````````````````
M``````````````````````````"XC0$```````````#$$@``Q!(``,02``#$
M$@``Q!(``,02``#$$@``Q!(``,02``#$$@``Q!(``,02``#$$@``Q!(``,02
M``#$$@``Q!(``,02``#$$@``Q!(``,02``#$$@``Q!(``,02``#$$@``Q!(`
M`,02``#$$@``Q!(``,02``#$$@``Q!(``,02``#$$@``Q!(``,02``#$$@``
MQ!(``,02``#$$@``Q!(``,02``#$$@``Q!(``,02``#$$@``Q!(``,02``#$
M$@``Q!(``,02``#$$@``Q!(``,02``#$$@``Q!(``,02``#$$@``Q!(``,02
M````````````````````````````````````````````````````````````
M``````"H%0````````20`0#_____Y$T``"!+````````X$L``.1+``#830``
M*$L``.!-````````Q$T``,Q-````````[$T``$=#0SH@*$=.52D@,3(N,2XP
M`$=#0SH@*&-R;W-S=&]O;"U.1R`Q+C(U+C`N,CA?8C@X9#,S."D@,3(N,2XP
M`$$T````865A8FD``2H````%-RU!``8*!T$(`0D""@,,`1($$P$4`14!%P,8
M`1D!&@(<`2(!`"YS:'-T<G1A8@`N:6YT97)P`"YN;W1E+F=N=2YB=6EL9"UI
M9``N;F]T92Y!0DDM=&%G`"YG;G4N:&%S:``N9'EN<WEM`"YD>6YS='(`+F=N
M=2YV97)S:6]N`"YG;G4N=F5R<VEO;E]R`"YR96PN9'EN`"YR96PN<&QT`"YI
M;FET`"YT97AT`"YF:6YI`"YR;V1A=&$`+D%232YE>'1A8@`N05)-+F5X:61X
M`"YE:%]F<F%M90`N:6YI=%]A<G)A>0`N9FEN:5]A<G)A>0`N9'EN86UI8P`N
M9V]T`"YD871A`"YB<W,`+F-O;6UE;G0`+D%232YA='1R:6)U=&5S````````
M```````````````````````````````````````````````+`````0````(`
M``!4`0``5`$``!D```````````````$`````````$P````<````"````<`$`
M`'`!```D```````````````$`````````"8````'`````@```)0!``"4`0``
M(```````````````!``````````T````]O__;P(```"T`0``M`$``%P!```%
M``````````0````$````/@````L````"````$`,``!`#``"@!@``!@````,`
M```$````$````$8````#`````@```+`)``"P"0```P4``````````````0``
M``````!.````____;P(```"T#@``M`X``-0````%``````````(````"````
M6P```/[__V\"````B`\``(@/``"`````!@````,````$`````````&H````)
M`````@````@0```($```T`````4`````````!`````@```!S````"0```$(`
M``#8$```V!```.`!```%````%@````0````(````?`````$````&````N!(`
M`+@2```,```````````````$`````````'<````!````!@```,02``#$$@``
MY`(`````````````!`````0```""`````0````8```"H%0``J!4``(`T````
M``````````@`````````B`````$````&````*$H``"A*```(````````````
M```$`````````(X````!`````@```#!*```P2@``4$(`````````````!```
M``````"6`````0````(```"`C```@(P``!@```````````````0`````````
MH0````$``'""````F(P``)B,``!@````#0`````````$`````````*P````!
M`````@```/B,``#XC```!```````````````!`````````"V````#@````,`
M``"PC0$`L(T```0```````````````0````$````P@````\````#````M(T!
M`+2-```$```````````````$````!````,X````&`````P```+B-`0"XC0``
M&`$```8`````````!`````@```#7`````0````,```#0C@$`T(X``#`!````
M``````````0````$````W`````$````#`````)`!``"0``!`````````````
M```$`````````.(````(`````P```$"0`0!`D```%```````````````!```
M``````#G`````0```#``````````0)```#\```````````````$````!````
M\`````,``'```````````'^0```U```````````````!``````````$````#
C``````````````"TD`````$``````````````0``````````
