package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.36.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.36.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@(!`0````````````,`MP`!````P!X```````!``````````+#!````
M`````````$``.``)`$``&@`9``8````$````0`````````!``````````$``
M````````^`$```````#X`0````````@``````````P````0````X`@``````
M`#@"````````.`(````````;`````````!L``````````0`````````!````
M!0```````````````````````````````````&"N````````8*X`````````
M``$```````$````&````6+L```````!8NP$``````%B[`0``````*`4`````
M``!0!0```````````0```````@````8```!HNP```````&B[`0``````:+L!
M```````@`@```````"`"````````"``````````$````!````%0"````````
M5`(```````!4`@```````$0`````````1``````````$`````````%#E=&0$
M````X*8```````#@I@```````."F````````_`````````#\``````````0`
M````````4>5T9`8`````````````````````````````````````````````
M````````````$`````````!2Y71D!````%B[````````6+L!``````!8NP$`
M`````*@$````````J`0````````!`````````"]L:6(O;&0M;&EN=7@M86%R
M8V@V-"YS;RXQ```$````%`````,```!'3E4`Q`5:.!\&/H7ZR/+53*ST);*=
M_W($````$`````$```!'3E4```````,````'`````````"X```!*````!```
M``@````A$!``!<`$A"@"$`6`AD"$@@D`!`D@!(B(@!!!N@`""$H`````````
M3````$X`````````3P````````!0````40````````!2````4P```%0```!5
M`````````%<``````````````%@`````````60``````````````````````
M``!:````6P```%P```!=`````````%X``````````````&``````````80``
M`&(```````````````````!C````9`````````!E```````````````NQ9NV
M:W^:?`PZEPL5C"7,0_K7AZ/@^TE#1=7LB0=PT]EQ6!R_VE<;`:\TZ'ZIG)O/
MWD<MP^4_+-5DU'PS8MOMX:<""EGOEWEOQ=\?:PE#UG94/)0A_^0.N^.2?*U+
MX\"S`8MO:QJ>4.-'Q@%FK\QF^Z%L8```````````````````````````````
M`````````````P`+`,`;`````````````````````````P`6``#``0``````
M```````````H`0``$@````````````````````````#>`0``$@``````````
M``````````````"`````$@````````````````````````"P````$@``````
M``````````````````!_`@``$@````````````````````````#E`0``$@``
M``````````````````````!E`0``$@`````````````````````````!````
M(`````````````````````````"=`@``$@````````````````````````"A
M`0``$@````````````````````````"=`P``$@``````````````````````
M``"G````$@````````````````````````#@`@``$0``````````````````
M``````!3````$@````````````````````````!@`@``$@``````````````
M``````````"[````$@`````````````````````````S`@``$@``````````
M``````````````"8`@``$@````````````````````````!G`@``$@``````
M```````````````````[`@``$@````````````````````````"&`0``(@``
M```````````````````````0````(``````````````````````````2`@``
M$@`````````````````````````(`0``$@`````````````````````````L
M````(``````````````````````````)`@``$@``````````````````````
M``#S`0``$@````````````````````````!P`@``$@``````````````````
M``````"W`@``$@````````````````````````!9`@``$0``````````````
M``````````!%`@``$@````````````````````````!X`0``$@``````````
M``````````````#U````%@````````````````````````"7````$@``````
M``````````````````!O````$@`````````````````````````9`@``$@``
M``````````````````````!4`0``$@````````````````````````#8````
M$@````````````````````````#[`0``$@````````````````````````"'
M`@``$@````````````````````````#.`@``$@``````````````````````
M``"5`0``$@````````````````````````#"`0``$@``````````````````
M``````#)````$@````````````````````````"F`@``$@``````````````
M``````````#G````$@`````````````````````````^`0``$@``````````
M``````````````!C````$@````````````````````````!O`@``$@``````
M``````````````````#0`0``$@````````````````````````!W`@``$@``
M```````````````````````L`@``$@`````````````````````````3`0``
M$@````````````````````````":`0``$@`````````````````````````R
M`0``$@````````````````````````"^`@``$@``````````````````````
M```"`@``$@````````````````````````!&````$0``````````````````
M```````M`0``$@````````````````````````",````$@``````````````
M``````````!%`0``$@````````````````````````"H`0``$@``````````
M```````````````?`@``$@````````````````````````"P`0``$@``````
M``````````````````#$`@``$0````````````````````````#7`0``$@``
M``````````````````````#%`@``(0`````````````````````````E`@``
M$@````````````````````````!+`@``$@````````````````````````#&
M`@``$0````````````````````````#L`0``$@``````````````````````
M```=`P``$@`-`-`H````````!`````````"]`0``$@`-`$`<````````;`(`
M``````##`P``(``6``#``0`````````````````/!```$@`-`'!A````````
M2`````````#<`P``$@`-`,!A````````'`$```````!:!```$@`-`(`M````
M````S`0````````J!```$``6`(#``0````````````````!0!```$0`6`'C`
M`0``````"``````````#!```$``7`(#``0````````````````#W`P``$``7
M`*C``0````````````````#X`P``$``7`*C``0````````````````"+`P``
M$@`-`/`R````````,``````````=!```$@`-`(`T````````:`@```````#G
M`P``$@`-`(`G````````-``````````[`P``$@`-`/`\````````;`(`````
M``#'`P``$@`-`,`>````````-`````````!?`P``$@`-`+0G````````%`$`
M``````#O`P``$``7`*C``0````````````````"F`P``$@`-`&`_````````
M4`$```````#!`P``$``6``#``0````````````````#.`P``$``7`(#``0``
M``````````````!#!```$@`-`-0H````````N`(````````8!```$``7`*C`
M`0````````````````"R`P``$0`/`/!D````````!`````````!1`P``$@`-
M`%`R````````F`````````"!`P``$@`-`.!B````````V`$````````N`P``
M$@`-`)`K````````/`````````!U`P``$@`-`-`K````````J`$````````Q
M!```$@`-`"`S````````8`$`````````7U]G;6]N7W-T87)T7U\`7TE435]D
M97)E9VES=&5R5$U#;&]N951A8FQE`%])5$U?<F5G:7-T97)434-L;VYE5&%B
M;&4`4$Q?9&]?=6YD=6UP`%!E<FQ?9W9?9F5T8VAP=@!097)L7VUG7W-E=`!0
M97)L7V%T9F]R:U]L;V-K`%!E<FQ?9V5T7W-V`%!E<FQ?;F5W6%,`8F]O=%]$
M>6YA3&]A9&5R`'!E<FQ?<G5N`'!E<FQ?<&%R<V4`4&5R;%]S=E]S971P=@!0
M97)L7W-V7W-E='!V;@!P97)L7V-O;G-T<G5C=`!P97)L7V1E<W1R=6-T`%!,
M7V-U<G)E;G1?8V]N=&5X=`!P97)L7V%L;&]C`%!E<FQ?8W)O86M?;F]C;VYT
M97AT`'!E<FQ?9G)E90!097)L7W-A9F5S>7-M86QL;V,`4&5R;%]S>7-?:6YI
M=#,`4&5R;%]N97=84U]F;&%G<P!097)L7V%T9F]R:U]U;FQO8VL`4&5R;%]S
M>7-?=&5R;0!?7V-X85]F:6YA;&EZ90!R96%D`'-T<F-H<@!G971P:60`;65M
M;6]V90!?7VQI8F-?<W1A<G1?;6%I;@!?7V9P<FEN=&9?8VAK`'-T<F-M<`!O
M<&5N-C0`=6YL:6YK`'-T<G1O:P!M96US970`<W1R;F-M<`!G971U:60`;65M
M8VAR`&-L;W-E9&ER`'-T<G-T<@!R;61I<@!C;&]S90!S=')L96X`<W1R9'5P
M`')E86QL;V,`<F5A9&1I<C8T`&%B;W)T`%]?<W!R:6YT9E]C:&L`<W1D97)R
M`&UE;6-P>0!S=')R8VAR`&QS=&%T-C0`;W!E;F1I<@!L<V5E:S8T`%]?97)R
M;F]?;&]C871I;VX`97AI=`!R96%D;&EN:P!?7W-T86-K7V-H:U]F86EL`&%C
M8V5S<P!M:V1I<@!?7V5N=FER;VX`7U]R96=I<W1E<E]A=&9O<FL`7U]S=&%C
M:U]C:&M?9W5A<F0`;&EB<&5R;"YS;P!L:6)C+G-O+C8`;&0M;&EN=7@M86%R
M8V@V-"YS;RXQ`'!A<E]C=7)R96YT7V5X96,`<&%R7V)A<V5N86UE`%A37TEN
M=&5R;F%L<U]005)?0D]/5`!P87)?96YV7V-L96%N`'!A<E]C=7)R96YT7V5X
M96-?<')O8P!P87)?9&ER;F%M90!S:&%?9FEN86P`9V5T7W5S97)N86UE7V9R
M;VU?9V5T<'=U:60`<&%R7V-L96%N=7``7TE/7W-T9&EN7W5S960`7U]D871A
M7W-T87)T`%]?8G-S7W-T87)T7U\`<VAA7W5P9&%T90!X<U]I;FET`%]?96YD
M7U\`7U]B<W-?96YD7U\`7U]B<W-?<W1A<G0`<VAA7VEN:70`7V5N9`!P87)?
M;6MT;7!D:7(`7V5D871A`'!A<E]S971U<%]L:6)P871H`'!A<E]F:6YD<')O
M9P!M>5]P87)?<&P`<&%R7VEN:71?96YV`$=,24)#7S(N,3<`1TQ)0D-?,BXS
M-`!'3$E"0U\R+C,S`"]U<W(O;&EB+W!E<FPU+S4N,S8O8V]R95]P97)L+T-/
M4D4```````````$``@`!``$``@`"``$``0`"``(``@`!``,``0`"``$``@`"
M``(``@`"``$``@`!``$``@`"``0``@`"``(``0`!``$``0`"``$``0`"``(`
M`@`"``(``0`"``$``@`!``0``@`"``(``0`"``$``@`"``$``@`!``$``@`"
M``4``@`"``(``@`"``(``@`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M`0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``````````0`!``<#```0
M````(````)>1E@8```,`9P0````````!``,`_0(``!``````````M)&6!@``
M!0!R!```$````+.1E@8```0`?00``!````"7D98&```"`&<$````````6+L!
M```````#!````````-`?````````8+L!```````#!````````(`?````````
M.+X!```````#!````````.!B````````X+X!```````#!````````,!A````
M````2+\!```````#!````````,!D````````6+\!```````#!````````'!A
M````````\+\!```````#!````````$`<````````",`!```````#!```````
M``C``0``````&,`!```````#!````````*!H````````(,`!```````#!```
M`````&!E````````,,`!```````#!````````%!F````````.,`!```````#
M!````````(!H````````0,`!```````#!````````(AH````````2,`!````
M```#!````````)!H````````4,`!```````#!````````)AH````````8,`!
M```````#!````````&AH````````:,`!```````#!````````'!H````````
M>,`!```````#!````````*AH````````P+T!```````!!````P``````````
M````R+T!```````!!```!```````````````T+T!```````!!```!0``````
M````````V+T!```````!!```!@``````````````X+T!```````!!```!P``
M````````````Z+T!```````!!```"```````````````\+T!```````!!```
M"0``````````````^+T!```````!!```"@```````````````+X!```````!
M!```"P``````````````"+X!```````!!```#```````````````$+X!````
M```!!```#0``````````````&+X!```````!!```#@``````````````(+X!
M```````!!```#P``````````````*+X!```````!!```$```````````````
M,+X!```````!!```$0``````````````0+X!```````!!```$@``````````
M````2+X!```````!!```$P``````````````4+X!```````!!```%```````
M````````6+X!```````!!```%0``````````````8+X!```````!!```%@``
M````````````:+X!```````!!```%P``````````````<+X!```````!!```
M&```````````````>+X!```````!!```&0``````````````@+X!```````!
M!```&@``````````````B+X!```````!!```&P``````````````D+X!````
M```!!```'```````````````F+X!```````!!```'0``````````````H+X!
M```````!!```'@``````````````J+X!```````!!```'P``````````````
ML+X!```````!!```(```````````````N+X!```````!!```(@``````````
M````P+X!```````&!```(P``````````````R+X!```````!!```)```````
M````````T+X!```````!!```)0``````````````V+X!```````!!```)@``
M````````````Z+X!```````!!```)P``````````````\+X!```````!!```
M*```````````````^+X!```````!!```*0```````````````+\!```````!
M!```*@``````````````"+\!```````!!```*P``````````````$+\!````
M```!!```+```````````````&+\!```````!!```+0``````````````(+\!
M```````!!```+@``````````````*+\!```````!!```+P``````````````
M,+\!```````!!```,```````````````.+\!```````!!```,0``````````
M````0+\!```````!!```,@``````````````4+\!```````!!```,P``````
M````````8+\!```````!!```-```````````````:+\!```````!!```-0``
M````````````<+\!```````!!```-@``````````````>+\!```````!!```
M-P``````````````@+\!```````!!```.```````````````B+\!```````!
M!```.0``````````````D+\!```````!!```.@``````````````F+\!````
M```!!```.P``````````````H+\!```````!!```/```````````````J+\!
M```````!!```/0``````````````L+\!```````!!```/@``````````````
MN+\!```````!!```/P``````````````P+\!```````!!```0```````````
M````R+\!```````!!```00``````````````T+\!```````!!```1```````
M````````V+\!```````!!```1@``````````````X+\!```````!!```1P``
M````````````Z+\!```````!!```2```````````````^+\!```````!!```
M20``````````````H+T!```````"!```"@``````````````J+T!```````"
M!```(0``````````````L+T!```````"!```0@``````````````'R`#U?U[
MOZG]`P"1R@``E/U[P:C``U_6``````````#P>[^IT```T!'.1OD08C:1(`(?
MUA\@`]4?(`/5'R`#U=```-`1TD;Y$((VD2`"'];0``#0$=9&^1"B-I$@`A_6
MT```T!':1OD0PC:1(`(?U@````````````````````#]>[JIPP``T&,01_G]
M`P"1\U,!J?5;`JGBAP.IXN,`D>$#`9'@3P"YX#,!D60`0/GD+P#Y!`"`TL,`
M`-!CW$?Y8``_UL(``-!"^$;YP```T`!H1_GA`P*JPP``T&.D1_E@`#_6P```
MT`#01_D``$`Y@`H`-,```-``8$?Y0=`[U=,``/`B:&#X]$]`N4$H4CF`$@`1
ME`X`$2$$`#)!*!(YP0``T"'$1_D`?'V3(``_UN,#`*KF(T#YP```\&(B`I'E
M3T"Y!#Q`^2$``+#'`$#Y(6`AD2```+``@"&19P0`J4,$`/ED``&IOP0`<>T*
M`%0A`(#2'R`#U2`,`!'$>&'X(00`D638(/B_``%K;/__5*4,`!%U(@*1?]@E
M^,4``-"E[$;YX@,4*J`.0/D!``"P!`"`TB$`')&@`#_6]`,`*@`&`#1U(@*1
M]E,!D<$``-`AF$?YH`Y`^2``/]8@``"PX0,6J@"`%Y&.``"48`4`M',B`I'!
M``#0(>!&^6`.0/D@`#_6P```T`!<1_D``#_6P```T``01_GB+T#Y`0!`^4(`
M`>L!`(#2P04`5/5;0JG@`Q0J\U-!J?U[QJC``U_6P```T`!`1_G3``#P```_
MUF(B`I%`#`#YP`,`M,$``-`A>$?Y(``_UL```-``8$?Y0=`[U2)H8/A?W"@Y
MHO__%Z`.0/G!``#0(0Q'^2``/];T`P`JS/__%Z`*0/E@``"T0`@`E-/__Q?A
M`Q:J(```L`"@%9%;``"4H`H`^:#Y_[0X"`"4R___%X4`@%*R__\7P0``T"$H
M1_D@`(!2(``_UL```-``E$?Y```_UA\@`]4?(`/5'R`#U1\@`]4?(`/5'R`#
MU1T`@-(>`(#2Y0,`JN$#0/GB(P"1Y@,`D<```-``^$?Y`P"`T@0`@-)-__^7
M2/__E\```-``_$;Y0```M$#__Q?``U_6'R`#U1\@`]7```#P```"D<$``/`A
M``*1/P``Z\```%3!``#0(3A'^6$``+3P`P&J``(?UL`#7];```#P```"D<$`
M`/`A``*1(0``RR+\?]-!#(&+(?Q!D\$``+3"``#00D1'^6(``+3P`P*J``(?
MUL`#7]8?(`/5_7N^J?T#`)'S"P#YTP``\&`"0CE``0`UP0``T"$T1_F!``"T
MP```\``$0/D@`#_6V/__ER``@%)@`@(Y\PM`^?U[PJC``U_6'R`#U1\@`]7<
M__\7'R`#U1\@`]4?(`/5_7N[J?T#`)'S4P&I(`<`M/=C`ZGW`P"J0(`,$`#T
M1_GY(P#Y&0!`^9D&`+3U6P*I]@,!JA4`@%+@`D`Y'_0`<0080'H!`P!4,P-`
M^?,!`+38``"P&$]'^;5^0)/T`QFJX@,5JN$#%ZK@`Q.J``,_UH```#5@:G4X
M'_0`<2`"`%23CD#X\_[_M?5;0JGW8T.I^2-`^>`#$ZKS4T&I_7O%J,`#7];U
M`Q>JH!Y`.!_T`'$$&$!ZH?__5+4"%TOC__\7@`(9R[4&`)%S`A6+`/Q#D\`"
M`+G@`Q.J\U-!J?5;0JGW8T.I^2-`^?U[Q:C``U_6$P"`TNC__Q?W8T.I$P"`
MTODC0/GD__\7'R`#U1\@`]7]>[6IP@``L$(01_G]`P"1\PL`^>&C`)'S`P"J
M0P!`^>-7`/D#`(#2P@``L$*H1_E``#_6(`(`->$[0+D"`)12(0P4$C\00'$D
M$$)ZH`$`5,$``+`A$$?YXU=`^2(`0/EC``+K`@"`TL$!`%3S"T#Y_7O+J,`#
M7]8``(!2]?__%\(``+!"5$?YX`,3JD$`@%)``#_6'P``<>`7GQKM__\7P```
ML`"41_D``#_6'R`#U<$``+`A$$?Y_7NSJ?T#`)$B`$#YXF<`^0(`@-+```"T
M\U,!J?,#`*H!`$`Y80$`-?-30:G```"P`!!'^>)G0/D!`$#Y0@`!ZP$`@-)A
M"P!4_7O-J,`#7];!``"P(;1'^2``/];T`P"J0/[_M/5;`JG6``"PUC)'^>`#
M%*K``C_60`<`M`%,0#D53`"1/[@`<8$``%2A!D`Y`?__-!\@`]4`3$`Y'[@`
M<6`'`%3W8P.IX`,3JM@``+`8[T?Y``,_UO<#`*K@`Q6J``,_UL$``+`AG$?Y
MX`(`"P`(`!$`?$"3(``_UL8``+#&\$?YY0,5J@(`@)+U`P"J(0"`4N0#$ZHC
M``"08^`3D<``/];"``"P0E!'^>`#%:KA(P&10``_UA\$`#&@``!4X%M`N0`,
M%!(?$$!QP`,`5,$``+`AY$;YX`,5JB``/];!``"P(=1'^>`#%:H@`#_6]V-#
MJ>`#%*K``C_6`/G_M<$``+`A2$?YX`,4JB``/];!``"P(6Q'^>`#$ZH@`#_6
M\U-!J?5;0JFI__\7H`9`.1^X`'&!^/]4H`I`.:#V_S3!__\7X`,5JI3__Y?E
M__\7P```L`"41_GS4P&I]5L"J?=C`ZD``#_6#`B"TO]C+,O#``"P8Q!'^>+_
M@=+]>P"I_0,`D?-3`:GS`P"J]`,!JO5;`JGUXP"1X0,5JF``0/G@'PCY``"`
MTL,``+!C`$?Y(```D```%)%@`#_6'P``<<T``%0$!`!1X@,`*J#*9#A``P`U
MY`(`-<```+``8$?Y0]`[U>(#%*KA`Q.J8&A@^,,``+!C($?Y8``_UL```+``
M$$?YXA](^0$`0/E"``'K`0"`TH$#`%3]>T"I#`B"TO-30:GU6T*I_V,LB\`#
M7];B`P0J'R`#U<,``+!CS$?Y5GQ`D^(#%JK@`Q6JX06`4F``/]8@_/^TP```
ML`!@1_E$T#O5XP,6JN(#%:KA`Q.J@&A@^,0``+"$D$?Y@``_UM___Q?```"P
M`)1'^0``/]8?(`/5'R`#U?U[N:G]`P"1\U,!J?0#`:KS`P"J]5L"J?=C`ZGY
M:P2I(`!`.<$``+`A$$?Y'_0`<=<``+#W[D?Y(@!`^>(W`/D"`(#2E!:4FN`#
M%*K@`C_6]@,`JN&3`9'@`Q.JN?[_EV`(`+3U`P"JX`(_UM\"`.LI!P!4]V>`
MN=@``+`8]T?YV@``L%J?1_GW\GW38`)`.1_T`'$$&$!ZP`P`5.0#$ZJ#'$`X
M?_0`<6080'JA__]4A``3RX1\0),5`T#YP`H`D0``!(M``S_6H&HW^*`"`+0!
M`T#YX`,3JB-H=_@!%$`X80``.3_T`'$D&$!ZP```5`(40#AB'``X7_0`<408
M0'J!__]4H0>`4@``@-)A%``X@FI@.&)H(#@`!`"1HO__-<```+``$$?YXC=`
M^0$`0/E"``'K`0"`TF$(`%3S4T&I]5M"J?=C0ZGY:T2I_7O'J,`#7]8!`(#2
M@FIA.*)J(3@A!`"1HO__->[__Q?8``"P&/='^?LK`/D;`T#Y8`-`^<`%`+3@
M`QNJ%0"`4A\@`]4!C$#XX@,5*K4&`!&A__^U0`P`$0!\?9/!``#0(HA`N8("
M`#7:``"P6I]'^2(`@%(BB`"Y0`,_UOD#`*I`!`"TPP``L&/@1_FW?GV3X0,;
MJN(#%ZI@`#_6&0,`^:`&`!'U9P"Y/]L@^/LK0/FD__\7P@``L$(D1_GA`P"J
MX`,;JD``/]8``P#Y^0,`JL`!`+3:``"P6I]'^;=^?9/P__\7!`"`TJ'__Q<`
M`H#2%0"`4MK__Q?```"P`)1'^?LK`/D``#_6^RM`^;+__Q<?(`/5_7N^J4,`
M`)`A``"0_0,`D?-3`:E4T#O5TP``L'-B1_G"``"P0F1'^6.`&I'$``"PA-A'
M^2%`%)&`:G/X@``_UH!J<_@D``"0\U-!J81`(9']>\*HXP,$JM```+`0=D?Y
M`@``L"$``)!"P#.1(<`4D04`@%(``A_6'R`#U1\@`]7```"P`&!'^430.]7"
M``"P0F1'^4,``)#0``"P$-I'^6.`&I&`:&#X(0``D"%`%)$``A_6#`J$TO]C
M+,O```"P`!!'^0$(@M+]>P"I_0,`D?-3`:GSXP"1]1,`^?5C(8L!`$#YX2<0
M^0$`@-+```"P``1'^0``/];D`P`JQ@``L,;P1_DB`(+2(0"`4N`#$ZHE``"0
M(P``D*4@%9%C0!61P``_UL,``+!C`$?YX`,3JN$#%:H"`(+28``_UB`$^#?!
M``"P(9Q'^?,#`*H`!``1`'Q`DR``/];T`P"J(`,`M,,``+!C&$?Y<WY`D^$#
M%:KB`Q.J8``_UI]J,SC```"P`!!'^>(G4/D!`$#Y0@`!ZP$`@-(!`0!4_7M`
MJ>`#%*KS4T&I#`J$TO430/G_8RR+P`-?UL```+``E$?Y```_UA0`@-+N__\7
M'R`#U1\@`]6Y__\7#"*"TO]C+,O"``"P0A!'^?U[`*G]`P"1\U,!J?0#`*KU
M6P*I]0,!JN$3`I%``$#YX(<(^0``@-(@``"0`*`5D;/]_Y?S`P"JP@``L$+`
M1_G@`Q2JX06`4D``/]9@`@"TX0,4JB```)``X!61V/[_E^`#%*K!``"P(1!'
M^>.'2/DB`$#Y8P`"ZP(`@-*A$`!4_7M`J0PB@M+S4T&I]5M"J?]C+(O``U_6
MP0``L"&X1_GW8P.IX`,5JM<``+#W]D;Y^W,%J38``)#A/P#Y(``_UL$B%I'@
M`C_6_`,`JL`-`+3```"P`%!'^?EK!*GX(P21V@``L%KS1_G@-P#YU0``L+7N
M1_G```"P`+!'^>`[`/D?(`/5@`-`.6`'`#33``"TXCM`^>$#'*K@`Q.J0``_
MUH`%`#3@`QRJH`(_U@($`-'[`Q6J@@,"B_D#`*J?`P+KHP``5`H``!1?]!\X
MGP,"ZX```%1``$`Y'[P`<6#__U3@`QRJH`(_UOD#`*K@`Q2J8`,_U@``&8L`
M!`"1'_P_\2@#`%0"`(+2(0"`4N0#'*KF`Q2JX`,8JB4``)`C``"0I4`6D6-@
M%I%``S_6XC=`^>$C`I'@`QBJ0``_UJ```#7@FT"Y``P4$A\@0''@`@!4P2(6
MD0``@-+@`C_6_`,`JB#Y_[7W8T.I^6M$J?MS1:F8__\7TP,`M&`"0#D?N`!Q
MX```5"0``)`Y`(#2VP``L'OO1_F<@!61U?__%V`&0#F`_?\T^/__%\(``+!"
M5$?YX`,8JB$`@%)``#_6H/S_->$#&*H@``"0`.`5D5S^_Y?A/T#YX`,8JB``
M/];W8T.I^6M$J?MS1:E___\7)```D#D`@-+;``"P>^]'^9R`%9&[__\7]V-#
MJ?MS1:EQ__\7P```L`"41_GW8P.I^6L$J?MS!:D``#_6'R`#U?U[OJGA!8!2
MP@``L$(L1_G]`P"1\PL`^?,#`*I``#_6'P``\6`&@)KS"T#Y_7O"J-```+`0
MND?Y``(?UA\@`]4,"(+2_V,LR\$``+`A$$?Y_7L`J?T#`)$B`$#YXA\(^0(`
M@-(`!P"T]1,`^?4#`*H!`$`Y808`-,$``+`A[$?Y\U,!J2``/]8"!`"1\P,`
MJE\$0/&H!P!4PP``L&,81_GTXP"1X0,5JN`#%*I@`#_68`8`T8`"`(L?`!3K
MJ```5`<``!0?]!\X'P`4ZX```%0!`$`Y/[P`<6#__U3"``"P0BQ'^>`#%*KA
M!8!20``_U@`%`+0?`!3K2`0`5,$``+`AN$?YX`,4JB``/];!``"P(1!'^>,?
M2/DB`$#Y8P`"ZP(`@-(A!0!4_7M`J0P(@M+S4T&I]1-`^?]C+(O``U_6]1-`
M^<```+``$$?YXA](^0$`0/E"``'K`0"`TL$#`%30``"P$+I'^2```)``@!61
M_7M`J0P(@M+_8RR+``(?UA\``#G>__\7``"`TN#__Q?```"P`!!'^>(?2/D!
M`$#Y0@`!ZP$`@-(!`0!4\U-!J2```)#U$T#Y`(`5D=```+`0ND?YZ___%\``
M`+``E$?Y```_UL```+``E$?Y\U,!J?43`/D``#_6'R`#U1\@`]7]>[JIP```
ML``01_G]`P"1\U,!J30``)#S4P&1]5L"J92"%I'W8P.I^2,`^=4``+"U]D?Y
M`0!`^>$O`/D!`(#2X0,3JN`#%*J&_/^7X`$`M*("0/GC5X"Y0`P#BP$$0/E!
M>"/XX?[_M`$(0/D!!`#Y`"``D:'__[7A`Q.JX`,4JGC\_Y=@_O^UU0``L+7V
M1_DT``"0E,(6D>$#$ZK@`Q2J</S_E^`!`+2B`D#YXU>`N4`,`XL!!$#Y07@C
M^.'^_[0!"$#Y`00`^0`@`)&A__^UX0,3JN`#%*IB_/^78/[_M=4``+"U]D?Y
M-```D)3B%I'A`Q.JX`,4JEK\_Y?@`0"TH@)`^>-7@+E`#`.+`01`^4%X(_CA
M_O^T`0A`^0$$`/D`(`"1H?__M>$#$ZK@`Q2J3/S_EV#^_[75``"PM?9'^30`
M`)"4(A>1X0,3JN`#%*I$_/^7X`$`M*("0/GC5X"Y0`P#BP$$0/E!>"/XX?[_
MM`$(0/D!!`#Y`"``D:'__[7A`Q.JX`,4JC;\_Y=@_O^UU@``L-;V1_DT``"0
ME4(7D>$#$ZK@`Q6J+OS_E^`!`+3"`D#YXU>`N4`,`XL!!$#Y07@C^.'^_[0!
M"$#Y`00`^0`@`)&A__^UX0,3JN`#%:H@_/^78/[_M=8``+#6]D?Y-0``D+6"
M%Y'A`Q.JX`,5JAC\_Y?@`0"TP@)`^>-7@+E`#`.+`01`^4%X(_CA_O^T`0A`
M^0$$`/D`(`"1H?__M>$#$ZK@`Q6J"OS_EV#^_[77``"P]_9'^34``)"VHA61
MX0,3JN`#%JH"_/^7X`$`M.("0/GC5X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!
M!`#Y`"``D:'__[7A`Q.JX`,6JO3[_Y=@_O^UV```D!CW1_D6``#PU\(7D>$#
M$ZK@`Q>J[/O_E^`!`+0"`T#YXU>`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y`00`
M^0`@`)&A__^UX0,3JN`#%ZK>^_^78/[_M=D``)`Y]T?Y%P``\/@"&)'A`Q.J
MX`,8JM;[_Y?@`0"T(@-`^>-7@+E`#`.+`01`^4%X(_CA_O^T`0A`^0$$`/D`
M(`"1H?__M>$#$ZK@`QBJR/O_EV#^_[79``"0.?='^1@``/`80QB1X0,3JN`#
M&*K`^_^7X`$`M"(#0/GC5X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!!`#Y`"``
MD:'__[7A`Q.JX`,8JK+[_Y=@_O^UV0``D#GW1_D8``#P&.,5D>$#$ZK@`QBJ
MJOO_E^`!`+0B`T#YXU>`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y`00`^0`@`)&A
M__^UX0,3JN`#&*J<^_^78/[_M>$#$ZH```#P`(`8D9?[_Y?A`P"J8```M.`"
M&)'#_/^7X0,3J@```/``X!B1C_O_E^$#`*J```"T````\`!`&9&Z_/^7X0,3
MJ@```/``@!F1AOO_E^$#`*I@`@"TH*(5D;+\_Y?```"0`!!'^>(O0/D!`$#Y
M0@`!ZP$`@-)A`@!4]5M"J8!"%Y'S4T&I`0``\/=C0ZDA(!J1^2-`^?U[QJBB
M_/\7X0,3J@```/``P!F1;OO_E^$#`*I@_?^TP,(7D9K\_Y?H__\7P```D`"4
M1_D``#_6'R`#U?U[O:G]`P"1P0``D"$01_GS"P#YTP``L"(`0/GB%P#Y`@"`
MTF`20+D?!``Q8`$`5,$``)`A$$?YXQ=`^2(`0/EC``+K`@"`TB$"`%3S"T#Y
M_7O#J,`#7];ADP"1````\`#`%Y%+^_^7X0,`J@``@%*A``"T(`!`.1_``'$$
M&$!ZX`>?&F`2`+GJ__\7P```D`"41_D``#_6'R`#U1\@`]7```"0`'Q'^?U[
MOZG]`P"1```_UL$``)`A"$?Y(``_UD```+0``$#Y_7O!J,`#7];]>[NIP0``
MD"$01_G]`P"1\U,!J?0#`*KU6P*I%0``\+9"&I$B`$#YXB<`^0(`@-+@`Q:J
MX1,!D2+[_Y>```"T`0!`.?,#`*K!`0`UP```D``01_GB)T#Y`0!`^4(``>L!
M`(#280<`5.$#%*J@0AJ1\U-!J?5;0JG]>\6H0?S_%\(``)!"/$?YX0,4JD``
M/]:``0"TP```D``01_GB)T#Y`0!`^4(``>L!`(#2(04`5/-30:GU6T*I_7O%
MJ,`#7];W&P#YX`,4JM<``)#W[D?YX`(_UO4#`*K@`Q.JX`(_UL$``)`AG$?Y
MH`(`BP`(`)$@`#_6QP``D.?P1_GF`Q.J`@"`DB$`@%+D`Q2J\P,`J@4``/`#
M``#PI2`6D6-@%I'@`#_6P```D``01_GB)T#Y`0!`^4(``>L!`(#2P0``5/<;
M0/GA`Q.JX`,6JLG__Q?W&P#YP```D`"41_D``#_6##B0TO]C+,O!``"0(1!'
M^?U[`:G]0P"1^W,&J=L``+!B0P"10X``D?-3`JE&G$"I]5L#J?8#`Y%$%$6I
M]V,$J?<S`I'@.P#Y8`1`K6((P#T#`I#2XV,CBR``0/E@U`#Y``"`TN$#%ZI#
M#$#Y````\$(P0/D`H!61P`8`K>8?":GC4P#YY)<*J>)?`/GB.X`]M/K_EX``
M`+0!`$`Y\P,`JD$G`#7```"0`'Q'^?EK!:G@/P#Y```_UL$``)`A"$?Y(``_
MUJ`2`+09`$#Y>1(`M-H``)!:[T?YX`,9JD`#/]8`^'_3``0`D=4``)"UGD?Y
MH`(_UOP#`*HD`T`Y\P,`JN0F`#34``"0E/)'^1@``/`8HQJ1'R`#U>`#'*KC
M`QBJ`@"`DB$`@%*``C_6G`L`D20?0#@D__\U````\-8B`)$`0!F1`P``%,"&
M0/C@`0"TX0,7JH/Z_Y?X`P"J8/__M`$`0#DA__\TPOK_E^#^_S3!``"0(;A'
M^>`#&*H@`#_6^0,`JN`!`+76``"0UKI'^?A#`I$9`T#Y60$`M"`#0#F`"P`T
MX`,9JK+Z_Y=@"@`U&0=`^1@C`)$Y__^U'R`#U>`#&:I``S_6]@,`JN`#$ZI`
M`S_6U@(`"]82$!'8``"0&,M'^=Q^0)/@`QRJH`(_U@$``/#G`Q.J)4`6D>0#
M&:H&``#P`P``\,;`&I%CX!J1`@"`DO8#`*HA`(!2@`(_UN`#%JH!.(!2``,_
MUA\$`#'A``!4P```D`"`1_D``#_6!0!`N;]$`'%A&`!4P@``D$*H1_GAPP.1
MX`,6JD``/];S`P`JH```->`#0;D`#!02'Q!`<8`)`%3A.T#Y`@``\,```)``
M6$?YY`,6JB,`0/E"(!R1``!`^2$`@%+%``"0I8Q'^:``/]85`(#2P```D``0
M1_D!`I#2XF,ABT/40/D!`$#Y8P`!ZP$`@-(!+`!4_7M!J>`#%:KS4T*I##B0
MTO5;0ZGW8T2I^6M%J?MS1JG_8RR+P`-?UN`#&:H8(P"1P`(_UOD#`*I@]/^T
MK?__%QD`@-*K__\7X%=`^?2C`I'3``"0<[I'^4`!`+0?(`/5X0,7J@OZ_Y=@
M``"T`0!`.0$"`#6`!D#YE"(`D2#__[75``"0M9Y'^:`!@-+:``"06N]'^:`"
M/];\`P"J!0``\/,#`*JY@!J19`J`4F+__Q=@`C_6^0,`JN#]_[3:``"06N]'
M^4`#/]8`^'_3``0`D5+__Q?@/T#Y^0M!N0``/]8_`P!K(?;_5.`#0;D`(``2
M'P`'<:'U_U3@`QRJH`(_U@$``/#U`P"J((`=D>$#%ZK@^?^7X0,`JB`<`+3@
M.T#Y``!`^1C\_Y?\`P"J@!L`M.`#'*I``S_6X@,`JA\,`'%-`@!4V0``D#FS
M1_D`?$"3`!``T0$``/"``P"+(:`=D>(_`/D@`S_6``$`->(_0/E?$`!QH!D`
M5$(4`%&`RV(X'[P`<2`9`%1@$T"Y'P0`,4`,`%2@#0`TVP``D'L'1_D!``#P
M````\"$@&I$`P!>1Z/K_EP```/`70!:1&@``\!D``/!@`S_6.6,>D>8#`"KE
M`Q>J1T,AD>0#%JK@`Q6J`P``\`(`@))C(!Z1(0"`4H`"/];<``"0G(-'^1(`
M`!2``S_6``!`N1]$`'%A`@!48`,_UG,&`!%!0R&1X0,`^>8#`"KG`Q,JY0,7
MJN0#%JKC`QFJX`,5J@(`@)(A`(!2@`(_UN`#%:H!.(!2``,_UA\$`#%@_?]4
MP0``D"'41_G@`Q:J(``_UN$#%:H```#P`*`5D;?Z_Y?@`Q6J5?[_EU?__Q?A
M.T#YY`,6JL```)``6$?Y`@``\",`0/E"(!N1``!`^2$`@%+&``"0QHQ'^<``
M/]9)__\71O[_E\```)``$$?Y`@*0TN)C(HM#U$#Y`0!`^6,``>L!`(#2814`
M5/U[0:G@`Q.J\U-"J0PXD-+U6T.I]V-$J?MS1JG_8RR+T```D!"Z1_D``A_6
MU```D)3R1_G5_O\7````\.$#%ZH`P!>16_G_ET`.`+0``$`Y'\``<0080'K@
M!Y\:8!,`N<#R_S4?(`/5P@``D$+H1_G@`QRJ`0"`4D``/];W`P`JP/'_--@`
M`)`8\T;Y0@"`4MD``)`YBT?Y(0*`DO/C!I$``S_6X0,3JN`#%RK"`(#2(`,_
MUN#C1CF````UX.=&.1\,`7&@"@!4`@"`4@$`@-+@`Q<J``,_UL```)``K$?Y
MV@``D%IS1_D``#_6^`,`J@4``!3B`P`JX0,3JN`#&*I``S_6X0,3JN`#%RH"
M`)#2(`,_UA\``''L_O]4P0``D"'D1_G@`Q<J\\,%D?HC!I'W`Q.J^P,:JB``
M/]99HP"1P@``D$(<1_GA`QBJ&```\!BC&I'@`Q.J0``_UA\@`]7D%D`XX`,;
MJN,#&*I["P"1`@"`DB$`@%*``C_6?P,9ZP'__U3_PP8Y````\.8#&JH%0!:1
MY`,6JN`#%:H'``#P`P``\.=`(9%CX!V1`@"`DB$`@%*``C_6</__%^`[0/D<
M`$#Y(___%^`[0/D"!$#YPN;_M.`#`JKB.P#Y0`,_UA\,`/$IYO]4XCM`^0`0
M`-$!``#P(<`=D4```(L@`S_6'P``<>([0/F<$X*:)___%W\3`+F5__\7X.M&
M.1\$`7%!]?]4X.]&.1\,`7'A]/]4X/-&.1\@`7&!]/]4X/=&.1\4`7$A]/]4
MX`,7*D(`@%(A!X"2``,_UN$#$ZK@`Q<J(@6`TB`#/]8```#PY@,3J@5`%I'D
M`Q:JX`,5J@<``/`#``#PYT`AD6/@'9$"`("2(0"`4H`"/]8X__\7P```D`"4
M1_D``#_6P```D`"41_GY:P6I```_UA\@`]4?(`/5_7NZJ0```/``0!>1_0,`
MD?-3`:E4T#O5TP``D'-B1_GU6P*I]E,!D>$#%JKW8P.I^2,`^<(``)!"$$?Y
ME6IS^$,`0/GC+P#Y`P"`TJ+"0CFBQ@(YI_C_EX```+0!`$`Y/\0`<<`,`%0*
M_/^7E6IS^.`#%:K8``"0&!='^0$``/!C`(!2(<`>D2(`@%(``S_6]0,`JJ`)
M`+2A"D#Y`@``\(!J<_A"X!Z1(0!`^=D``)`Y(T?Y(`,_UJ`*0/D!`$#Y(`Q`
MN0`+L#>`:G/X`0``\&,`@%(A`!^1(@"`4M<``+```S_6]0,`JF`!`+3A`Q:J
M````\`#@%9%^^/^7X@,`JD`(`+2A"D#Y@&IS^"$`0/D@`S_6PP``D&/H1ODB
M`(!2E6IS^`$``/`A(!^1X`,5JF``/]8"``#PX0,`JD*`'Y'@`Q6J(`,_UH)J
M<_CT(@*1X0,6J@```/``H!617\0".63X_Y>`"@#Y\P,`JD`#`+3"``"00LA'
M^0$X@%+@`Q.J0``_U@$$`!$_!`!Q!(A1>J$%`%3```"0`!!'^>(O0/D!`$#Y
M0@`!ZP$`@-)A!`!4\U-!J?5;0JGW8T.I^2-`^?U[QJC``U_6V0``D#DC1_F^
M__\7@`9`^6[]_Y>`"@#Y\P,`JF#\_[7!``"0(;Q'^0```/``H!^1(``_U@`$
M0#F`\_\TF?__%^`B`I&B"D#Y`01`^4``0/DA`$#Y&?G_E[W__Q>`:G/XP@``
MD$*@1_E``#_6I?__%\```)``E$?Y```_UO<B`I'C`P`JQ```D(2\1_D```#P
MX09`^>(#$ZH`0""1(0!`^8``/]8?(`/5_7N\J<$``)`A$$?Y_0,`D?-3`:GS
M`P"J]5L"J=4``+`B`$#YXA\`^0(`@-(1^_^7]`,`JL(``)!"+$?YX06`4D``
M/]8?``#QP0``D"&X1_F`!H":(``_UO8#`*JA$D"Y?P(`\?0'GQH_!``QP`0`
M5#\``''@!Y\:E`(`"I0!`#7```"0`!!'^>(?0/D!`$#Y0@`!ZP$`@-)A!0!4
M\U-!J?5;0JG]>\2HP`-?UF`"0#F`_O\THP``\&-,1_D!``#0X`,6JB'`&I&"
M`(#28``_UH#]_S6@``#P`!!'^>(?0/D!`$#Y0@`!ZP$`@-+!`@!4]5M"J>`#
M$ZKS4T&I_7O$J%#X_Q<```#0X=,`D0#`%Y'<]_^70`$`M`$`0#D_``!QX`>?
M&C_``''A!Y\:```!"J`2`+D4`!0*T?__%[\2`+G0__\7H```\`"41_D``#_6
M``0`%!\@`]49`$#YT_?_%P``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`/^#"M$!X`"1HP``T&,01_GB0P"1_7L`J03@`9']`P"14>0`;_#E`&]E`$#Y
MY4\!^04`@-*'Y`!OXP,"JB,`0&TB!,$\8SB`#D$$8&]`!%AO8P@@+D($2&\F
M5%A/`!PQ3F2D("\A5$A/PQPP3@`<HDXA'"=.`!RC3@4<H4YDC)],GP`!Z\']
M_U3P'T&M0J`)D>8/0JWAXP&1Y1-#K>$CP#WQY0!O'R`#U2*`V3PA0`"1(`#?
M/``<(FX`'"-N`!PP;O`<IT[''*9.9ARC3@)404\`!&%OHQRE3H4<I$XD'*%.
M0!R@3@$<,4XA@(`\/P`"Z\']_U0'*$"I)#./T@@D0:E$4*ORY3]!J2-TG=+B
M_%O3[/Q"TT(4!ZJ,>0>J(0$(RB/;K?(A``J*(0`)RJ4`!(LB0"*+0?U"TPL0
M0/DA>`JJ+7Q`DD(`!8NE`0C*0@`+"^9#0JFE``>*I0`(RN\!!(M._%O3I0`)
MB\X5`JHA``S*I0`/BR$``HJE``X+3_Q"TR$`#<KB>0*JQ@`$BZ[\6],A``B+
MSA4%JB$`!HN&`0+*(0`."XQ]0)+&``6*KOQ"T\8`#,H0`@2+Q@`-BR_\6]/-
M>06J[Q4!JNX;0/E%``W*Q@`0BT)\0)+&``\+I0`!BJ4``LHP_$+3S@$$BZ4`
M#(O/_%O3#'H!JN\5!JJE``Z+I0`/"Z$!#,K/_$+3(0`&BN9Y!JJM?4"2[K]#
MJ2$`#<HA``*+L/Q"T\(!!(NN_%O3(0`"BX(!!LI"``6*C'U`DLX5!:I"``S*
M+@`."P5Z!:KA`02+0@`-BT(``8O!``7*T'Q`DB$`#HHA`!#*S?U;TR$`#(NM
M%0ZJ[[-$J4T`#0O&_4+3SG@.JJ9\0)+B`02+K_U;TR$``HNB``[*[Q4-JD(`
M#8HO``\+I?U"TT(`!LJ!`02+I7@-JD(`$(M"``&+[/U;T\$!!<KP_4+3C!4/
MJB$`#XH/>@^J3``,"^W#1:G.?4"2(0`.RB$`!HN&_5O3QA0,JJ(!!(L0`@2+
M(0`"BZ(`#\HM``8+I7Q`DD(`#(J&_4+30@`%RL9X#*JA_5O30@`.BT(`$(LL
M%`VJL/U"T^$!!LHA``V*#7H-JN[#1JE,``P+[WU`DB$`#\HA``6+A?U;TZ44
M#*K"`02+$`($BR$``HO"``W*+@`%"\9\0))"``R*A?U"TT(`!LJE>`RJP?U;
MTT(`#XM"`!"++!0.JM#]0M.A`07*(0`.B@YZ#JKOPT>I3``,"ZU]0)(A``W*
M(0`&BX;]6]/&%`RJX@$$BQ`"!(LA``*+H@`.RB\`!@NE?$"20@`,BH;]0M-"
M``7*QG@,JN']6]-"``V+0@`0BRP4#ZKP_4+3P0$&RB$`#XH/>@^J[<-(J4P`
M#`O.?4"2(0`.RB$`!8N%_5O3I10,JJ(!!(L0`@2+(0`"B\(`#\HM``4+A?U"
MTZ5X#*I"``R*H?U;T\9\0)(L%`VJX0$%RB$`#8KO?4"20@`&RB$`#\I"``Z+
M(0`&B^Z;2:E"`!"+3``,"[#]0M,->@VJD/U"T\(!!(N._5O3(0`"B\X5#*JB
M``W*+@`."Z5\0))"``R*0@`%R@QZ#*K!``2+0@`/B\;]6]-"``&+[U=`^:$!
M#,K&%`ZJK7U`DM(``@LA``Z*(0`-RN0!!(LA``6+T/U"TT7^6],.>@ZJ[UM`
M^2$`!(NE%!*J1D`FB[```0N$`0[*A``&2N(!`XN$``V+!OY;TTW^0M.$``*+
MXE]`^;)Y$JK&%!"J)4`EB\\`!`O!`1+*)0`%2D$``XNB0"R+#/Y"T^7]6].-
M>1"J0@`!BZ44#ZKA8T#YL0`""X9`)HM$`@W*A``&2NS]0M,F``.+@4`NBR3^
M6]./>0^J(0`&BX04$:KF9T#YD``!"T5`)8NB`0_*10`%2B[^0M/1>1&JK$`R
MB\(``XL%_EO3YFM`^8(!`HNE%!"J)$`DB^$!$<JN``(+)``$2@S^0M/!``.+
MAD`MBY!Y$*K!``&+YF]`^<3]6]-%0"6+A!0.JB("$,J-``$+0@`%2L8``XM"
M0"^+S/U"TXYY#JI"``:+YG-`^:7]6].E%`VJ)$`DBP$"#LJO``(+(0`$2JS]
M0M.->0VJQ@`#BR%`,8OD_5O3(0`&BT5`)8OF=T#YP@$-RH04#ZI%``5*D0`!
M"ZQ`,(OP_4+3P@`#B^9[0/D/>@^J)?Y;TR1`)(NA`0_*@@$"BZ44$:HD``1*
ML``""\$``XLL_D+3AD`NBXQY$:K!``&+YG]`^45`)8L$_EO3X@$,RH04$*I"
M``5*C@`!"\8``XM"0"V+#?Y"T_&#0/E"``:+IGD0JB1`)(O%_5O3@0$&RJ44
M#JHA``1*K0`""S`"`XLA0"^+S_U"TR$`$(ON>0ZJ\(=`^:3]6].$%`VJ14`E
MB[']0M/"``[*CP`!"RUZ#:I%``5*`@(#B_"+0/DD0"2+P0$-RJQ`+(LD``1*
MY?U;TZ44#ZJ&0":+\?U"TP$"`XO!``&+@@$"BR9Z#ZJL``(+14`EBZ(!!LI"
M``5*A/U;TT)`+HN$%`RJ\+M1J9']0M./``$++'H,JB1`)(O!``S*Y?U;TR$`
M!$KQ_4+3I10/JA`"`XLO>@^J0@`0BR%`+8OMQU*IL``""\X!`XM%0"6+!/Y;
MTR$`#HN$%!"J@@$/RHX``0M"``5*0D`FBZ4!`XL-_D+310`%B\;]6].B>1"J
MQA0.JB1`)(OA`0+*S0`%"R0`!$HA`@.+A$`LB]']0M/PGT#YA``!BZS]6],A
M>@ZJC!4-JJ9`)HM%``'*C@$$"Z4`!DH0`@.+I4`OBZ_]0M/O>0VJI0`0B_"C
M0/G&_5O3QA0.JHQ`+(LD``_*S0`%"XP`#$K1_4+3!`(#BX)!(HNP_5O30@`$
MBR1Z#JJF0":+[*=`^1`6#:KE`03*0@`0"Z4`!DJN_4+3H4`AB\5Y#:J&`0.+
M3OQ"TXP`!:I#_%O38Q0"JHP!`HK">0*J(0`&B^T[5:DA``,+HP`$"H:;E]*,
M`0.J9N.Q\H-!+XLO_$+3K`$&BRW\6].M%0&J8P`,BZP``JIC``T+C`$!BDT`
M!0J,`0VJX7D!JLX!!HN$022+;?Q;TTP``:JM%0.JA``.BX0`#0N,`0.*+0`"
M"F[\0M.,`0VJC?Q;TX5!)8O,>0.J[SM6J:,5!*HM``RJ[P$&B\X!!HNE``^+
MC_Q"TZ,``PNE`02*C0$!"J4`#:IM_%O3I4`BB^)Y!*JM%0.JA`$"JJ4`#HN$
M``.*I0`-"TT`#`J$``VJ;?Q"TX1`(8NA>0.J[S=7J:[\6]/.%06J0P`!JF,`
M!8KO`0:+K0$&BX0`#XNO_$+3A``."RX``@IC``ZJY7D%JF-`+(N._%O3SA4$
MJBP`!:IC``V+C`$$BNW#0/EC``X+K@`!"H_\0M.,`0ZJY'D$JH)!(HMN_%O3
MK`$&B\X5`ZKMQT#Y0@`,BZP`!*I"``X+C`$#BHX`!0J,`0ZJ;_Q"T^-Y`ZJM
M`0:+@4$ABT[\6]/.%0*JC``#JB$`#8N,`0**(0`."VX`!`J,`0ZJ3OQ"T\)Y
M`JJ%026+[3M9J2_\0M.L`0:++?Q;TZT5`:JE``R+;``"JJ4`#0N,`0&*30`#
M"HP!#:KA>0&JS@$&BX1!)(NM_%O33``!JJT5!:J$``Z+A``-"XP!!8HM``(*
MKOQ"TXP!#:J-_%O3@T$CB\QY!:KO.UJIK14$JB4`#*JE``2*[P$&B\X!!HMC
M``^+C_Q"TV,`#0N-`0$*I0`-JN1Y!*JB0"*+;?Q;TZT5`ZJ%`02J0@`.BZ4`
M`XI"``T+C0`,"J4`#:IM_$+3H4`ABZ5Y`ZKO-UNI3OQ;T\X5`JJ#``6J8P`"
MBN\!!HNM`0:+(0`/BZ\`!`HA``X+8P`/JD[\0M-L0"R+PGD"JB/\6].,`0V+
M8Q0!JJT``JHO_$+3@P$#"ZP!`8KA>0&J30`%"NX_7*F,`0VJA$$DBVW\6].M
M%0.JS`$&BRX``@J$``R+3``!JH0`#0N,`0.*C`$.JFW\0M.M>0.J[P$&BX5!
M)8N._%O3[.M`^:,`#XO.%02J)0`-JF,`#@NE``2*K@$!"H_\0M.E``ZJ;OQ;
MTZ)`(HN%`0:+Y'D$JLX5`ZI"``6+I0$$JD(`#@MN_$+3I0`#BL-Y`ZKLNUVI
MCP`-"J4`#ZI/_$+3I4`ABT'\6],A%`*JC`$&BZ4`#(N,``.JI0`!"XP!`HIA
M``0*XGD"JHP!`:I0``,*@4$MB\T!!HNL_%O3(0`-B^_W0/EM``*JC!4%JJT!
M!8J.`0$+K0$0JJ1!)(NP_$+3[0$&B\;]6]/O^T#YA``-B\84#JH->@6JT0`$
M"RQ`+(M!``W*Q3J8TB$`#$I%3+GR(4`CB^P!!8O0_4+3(_Y;T^__0/D.>@ZJ
M8Q01JB$`#(ML``$+AD`FBZ0!#LHC0".+A``&2N8!!8N"0"*++_Y"TX3]6]/Q
M>1&J0@`&BX04#*KF`T'YD@`""\$!$<I$0"2+(P`#2L$`!8MF0"V+C?U"TT/^
M6].P>0RJQ@`!BV,4$JKA!T'Y;P`&"R("$,I,_D+30@`$2B0`!8M"0"Z+X?U;
MTXUY$JI$``2+X@M!^2$4#ZHN``0+PT`CBP8"#<KL_4+3PP`#2D8`!8MC0#&+
MPOU;TX]Y#ZIC``:+Y@]!^4(4#JI1``,+@4`ABZ0!#\IB0"*+@0`!2L0`!8LL
M0#"+T/U"TR'^6],.>@ZJYA-!^80!!(LA%!&JXP$.RC``!`MB``)*+/Y"T\,`
M!8M&0"V+`OY;TXQY$:K#``.+YA=!^4(4$*I-``,+@4`AB\0!#,H1_D+3@0`!
M2L8`!8LD0"^+H?U;T^\;0?F$``:+(10-JB9Z$*IB0"*+,``$"X,!!LKO`06+
M8@`"2K']0M-#0"Z+`OY;T^X?0?DM>@VJ8P`/BT(4$*I/``,+@4`AB\0`#<IB
M0"*+@0`!2L0!!8LL0"R+#OY"T^']6]/0>1"JA`$$BR$4#ZKL(T'Y,0`$"Z,!
M$,J!0"&+8@`"2H,!!8M&0":+[/U"TR+^6]./>0^JPP`#BT(4$:KF)T'Y3@`#
M"P0"#\IB0"*+@0`!2L8`!8LD0"V++?Y"T\']6].$``:+["M!^:9Y$:HA%`ZJ
MXP$&RBT`!`MB``)*C`$%BT-`,(O0_4+3HOU;TPYZ#JIC``R+["]!^4(4#:I1
M``,+@4`AB\0`#LIB0"*+@0`!2H0!!8LL0"^+K_U"TR'^6]/M>0VJA`$$BR$4
M$:KL,T'Y,``$"\,!#<HO_D+38@`"2HP!!8M"0":+`_Y;TT8`#(MC%!"JXC=!
M^>QY$:J!0"&+;P`&"Z0!#,K#0".+A``!2D$`!8N$0"Z+XOU;TP[^0M.$``&+
M\3M!^<%Y$*I"%`^JA@$!RDX`!`O#``-*,`(%BV9`+8OQ_4+3P_U;T^T_0?EC
M%`ZJ+WH/JL8`$(MP``8+@D`BBR0`#\K&0".+A``"2J(!!8N,0"R+S?U"TP3^
M6].,`0*+XD-!^804$*JN>0ZJ@P`,"^T!#LKG0"^+K0$&2D8`!8NA02&+8OQ;
MTT(4`ZHA``:+[4=!^2$``@L"_D+3AD$DBT)X$*HD_%O3S`$"RH04`:J,`09*
MI@$%BV7\0M/&``>+HW@#JH0`#(NG``"PYQ!'^2$`"@M"``D+Q0$+"X0`!@MC
M``@+!`0`J0,(`:D%$`#YXD]!^>$`0/E"``'K`0"`TH$``%3]>T"I_X,*D<`#
M7]:@``"P`)1'^0``/]:A``"P(9Q'^?U[OZD`$(#2_0,`D2``/]8?&`#Y(@``
MD"$``)`?>`"Y0;#!/2(``)`BK,$]0+3!/0($`*T`"(`]_7O!J,`#7]8?(`/5
M'R`#U?U[NZE"?$"3_0,`D?-3`:D4X`"1\P,!JO5;`JGU`P"J]@,"JO=C`ZGY
M(P#Y!8Q"J0!X0+FD#`(+OP`$ZV.4@YIB=$*+I(H"J>`#`#6X``"P&!M'^=_^
M`''M!0!4U@(!4=E^!E,W!P`1]VIZTW<"%XM@!D"MX`,5JH`&`*U@!D&M<P(!
MD8`&`:V:^_^7?P(7ZP'__U36&AE+X0,7JN`#%*K"?D"3``,_UK9Z`+GS4T&I
M]5M"J?=C0ZGY(T#Y_7O%J,`#7]87"(!2]P(`2_\"%FN`PB"+N```L!@;1_GW
MTI8:^7Y`D^(#&:H``S_6H'I`N>`"``N@>@"Y'P`!<8']_U36`A=+<P(9B^`#
M%:IY^_^7T?__%_<#$ZK@__\7'R`#U?U[O*GB#X`2_0,`D?-3`:GT`P"J\P,!
MJO5;`JDVX`"1Z!L`_3440/DH&$#]H"(#4P$$`!'"RB`XP,(ABS_@`''-"P!4
MHP``L&/\1_D""(!20@`!2P$`@%)"?$"38``_UN`#$ZI;^_^7?S8`^0`$`$_`
M`@"MP`J`/04%<'\`!6A_!`5X?Z$"9Y[@`Q.J(P1H?R($<'^@!`-N(01X?X`$
M!6X`!0=N8`0);D`$"VX@!`UNH!X/3F`Z`/U&^_^780)`^>`#$ZJP``"P$-9'
M^2'\6-.!`@`Y80)`^2'\4-.!!@`Y80)`^2'\2-.!"@`Y80)`^8$.`#EA!D#Y
M(?Q8TX$2`#EA!D#Y(?Q0TX$6`#EA!D#Y(?Q(TX$:`#EA!D#Y@1X`.6$*0/DA
M_%C3@2(`.6$*0/DA_%#3@28`.6$*0/DA_$C3@2H`.6$*0/F!+@`Y80Y`^2'\
M6-.!,@`Y80Y`^2'\4-.!-@`Y80Y`^2'\2-.!.@`Y80Y`^8$^`#EA$D#Y(?Q8
MTX%"`#EA$D#Y(?Q0TX%&`#EA$D#Y(?Q(TX%*`#EA$D#Y@4X`.?-30:GU6T*I
MZ!M`_?U[Q*@``A_6HP``L&/\1_D"!X!20@`!2P$`@%)"?$"38``_UJG__Q<?
M(`/5'R`#U:,``-"P``"P$(9'^6,$0/D``A_6'R`#U?U[OZG]`P"1_7O!J,`#
M7]8```````````$``@``````)7,O)7,````O<')O8R]S96QF+V5X90``1'EN
M84QO861E<CHZ8F]O=%]$>6YA3&]A9&5R``````!);G1E<FYA;',Z.E!!4CHZ
M0D]/5`````!E>&4``````"]P<F]C+R5I+R5S```````N`````````%!!4E]4
M14U0``````````!005)?4%)/1TY!344`````.@`````````O`````````"5S
M)7,E<P``4$523#5,24(``````````%!%4DQ,24(`4$523#5/4%0`````````
M`%!%4DQ)3P``4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$``````!005)?
M0TQ%04X`````````4$%27T1%0E5'`````````%!!4E]#04-(10````````!0
M05)?1TQ/0D%,7T1%0E5'``````````!005)?1TQ/0D%,7U1-4$1)4@``````
M``!005)?5$U01$E2````````4$%27T=,3T)!3%]414U0`%!!4E]'3$]"04Q?
M0TQ%04X``````````#$`````````3$1?3$E"4D%265]0051(`%-94U1%30``
M)3`R>`````!P87(M`````"5S)7,E<R5S```````````E<SH@8W)E871I;VX@
M;V8@<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*
M````````)7,Z('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&ES('5N<V%F92`H
M<&QE87-E(')E;6]V92!I="!A;F0@<F5T<GD@>6]U<B!O<&5R871I;VXI"@``
M`````%!!5$@`````<&%R;``````N<&%R`````"5S)7-C86-H92TE<R5S```E
M<R5S=&5M<"TE=25S````)7,E<W1E;7`M)74M)74E<P``````````&```````
M``!P97)L`````#``````````4$%2.CI086-K97(Z.E9%4E-)3TX`````,2XP
M-38```!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R>0``````````
M)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B9&ER96-T;W)Y("5S
M(&9A:6QE9"`H97)R;F\])6DI"@``````````+64````````M+0```````%53
M15(`````55-%4DY!344``````````%1-4$1)4@``5$5-4$1)4@!414U0````
M`%1-4```````+W1M<`````!E=F%L("=E>&5C("]U<W(O8FEN+W!E<FP@("U3
M("0P("1[,2LB)$`B?2<*:68@,#L@(R!N;W0@<G5N;FEN9R!U;F1E<B!S;VUE
M('-H96QL"G!A8VMA9V4@7U]P87)?<&P["FUY("@D4$%27TU!1TE#+"`D1DE,
M15]O9F9S971?<VEZ92P@)$-!0TA%7VUA<FME<BP@)&-A8VAE7VYA;65?<VEZ
M92D["D)%1TE.('L*)%!!4E]-04=)0R`](")<;E!!4BYP;5QN(CL*)$9)3$5?
M;V9F<V5T7W-I>F4@/2`T.R`@(",@<&%C:R@B3B(I"B1C86-H95]M87)K97(@
M/2`B7#!#04-(12(["B1C86-H95]N86UE7W-I>F4@/2`T,#L*?0IS=6(@9FEN
M9%]P87)?;6%G:6,*>PIM>2`H)&9H*2`]($!?.PIM>2`D8VAU;FM?<VEZ92`]
M(#8T("H@,3`R-#L*;7D@)&)U9CL*;7D@)'-I>F4@/2`M<R`D9F@["FUY("1P
M;W,@/2`D<VEZ92`M("1S:7IE("4@)&-H=6YK7W-I>F4[("`@("`@(R!.3U1%
M.B`D<&]S(&ES(&$@;75L=&EP;&4@;V8@)&-H=6YK7W-I>F4*=VAI;&4@*"1P
M;W,@/CT@,"D@>PIS965K("1F:"P@)'!O<RP@,#L*<F5A9"`D9F@L("1B=68L
M("1C:'5N:U]S:7IE("L@;&5N9W1H*"1005)?34%'24,I.PII9B`H*&UY("1I
M(#T@<FEN9&5X*"1B=68L("1005)?34%'24,I*2`^/2`P*2!["B1P;W,@*ST@
M)&D["G)E='5R;B`D<&]S.PI]"B1P;W,@+3T@)&-H=6YK7W-I>F4["GT*<F5T
M=7)N("TQ.PI]"FUY("@D<&%R7W1E;7`L("1P<F]G;F%M92P@0'1M<&9I;&4L
M("5-;V1U;&5#86-H92D["D5.1"![(&EF("@D14Y6>U!!4E]#3$5!3GTI('L*
M<F5Q=6ER92!&:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIR
M97%U:7)E($9I;&4Z.E-P96,["FUY("1T;W!D:7(@/2!&:6QE.CI"87-E;F%M
M93HZ9&ER;F%M92@D<&%R7W1E;7`I.PIO=71S*'%Q6U)E;6]V:6YG(&9I;&5S
M(&EN("(D<&%R7W1E;7`B72D["D9I;&4Z.D9I;F0Z.F9I;F1D97!T:"AS=6(@
M>R`H("UD("D@/R!R;61I<B`Z('5N;&EN:R!]+"`D<&%R7W1E;7`I.PIR;61I
M<B`D<&%R7W1E;7`["FEF("@M9"`D<&%R7W1E;7`@)B8@)%Y/(&YE("=-4U=I
M;C,R)RD@>PIM>2`D=&UP(#T@;F5W($9I;&4Z.E1E;7`H"E1%35!,051%(#T^
M("=T;7!86%A86"<L"D1)4B`]/B!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D
M=&]P9&ER*2P*4U5&1DE8(#T^("<N8VUD)RP*54Y,24Y+(#T^(#`L"BD["FUY
M("1F:6QE;F%M92`]("1T;7`M/F9I;&5N86UE.PIP<FEN="`D=&UP(#P\(BXN
M+B(["G@],3L@=VAI;&4@6R!<)'@@+6QT(#$P(%T[(&1O"G)M("UR9B`G)'!A
M<E]T96UP)PII9B!;(%PA("UD("<D<&%R7W1E;7`G(%T[('1H96X*8G)E86L*
M9FD*<VQE97`@,0IX/6!E>'!R(%PD>"`K(#%@"F1O;F4*<FT@)R1F:6QE;F%M
M92<*+BXN"F-L;W-E("1T;7`["F-H;6]D(#`W,#`L("1F:6QE;F%M93L*;7D@
M)&-M9"`]("(D9FEL96YA;64@/B]D978O;G5L;"`R/B8Q("8B.PIS>7-T96TH
M)&-M9"D["F]U=',H<7%;4W!A=VYE9"!B86-K9W)O=6YD('!R;V-E<W,@=&\@
M<&5R9F]R;2!C;&5A;G5P.B`D9FEL96YA;65=*3L*?0I]('T*0D5'24X@>PI)
M;G1E<FYA;',Z.E!!4CHZ0D]/5"@I(&EF(&1E9FEN960@)DEN=&5R;F%L<SHZ
M4$%2.CI"3T]4.PIE=F%L('L*7W!A<E]I;FET7V5N=B@I.PIM>2`D<75I970@
M/2`A)$5.5GM005)?1$5"54=].PIM>2`E<WES(#T@*`IP871H7W-E<"`@("`]
M/B`H)%Y/(#U^("]>35-7:6XO(#\@)SLG(#H@)SHG*2P*7V5X92`@("`@("`@
M/3X@*"1>3R`]?B`O7B@_.DU35VEN?$]3,GQC>6=W:6XI+R`_("<N97AE)R`Z
M("<G*2P*7V1E;&EM("`@("`@/3X@*"1>3R`]?B`O7DU35VEN?$]3,B\@/R`G
M7%PG(#H@)R\G*2P**3L*7W-E=%]P<F]G;F%M92@I.PIO=71S*'%Q6UPD<')O
M9VYA;64@/2`B)'!R;V=N86UE(ETI.PI?<V5T7W!A<E]T96UP*"D["F]U=',H
M<7%;7"1%3E9[4$%27U1%35!](#T@(B1%3E9[4$%27U1%35!](ETI.PIM>2`H
M)'-T87)T7W!O<RP@)&1A=&%?<&]S*3L*34%'24,Z('L*;&]C86P@)%-)1WM?
M7U=!4DY?7WT@/2!S=6(@>WT["G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L
M("1P<F]G;F%M92D@>PIO=71S*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D
M<')O9VYA;64B72D[("`C(&1O;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I
M<F5S($5R<FYO+G!M"FQA<W0@34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I
M;F1?<&%R7VUA9VEC*"I?1D@I.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T
M<RAQ<5M#86XG="!F:6YD(&UA9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@
M9FEL92`B)'!R;V=N86UE(ETI.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD
M(%!!4B!M86=I8R!A="!P;W-I=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(
M+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(
M+"`D8G5F+"`D1DE,15]O9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C
M:R@B3B(L("1B=68I.PIO=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%
M<R!I<R`D;V9F<V5T(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?
M;V9F<V5T7W-I>F4@+2`D;V9F<V5T+"`P.PHD9&%T85]P;W,@/2!T96QL(%]&
M2#L*;7D@)7)E<75I<F5?;&ES=#L*<F5A9"!?1D@L("1B=68L(#0[("`@("`@
M("`@("`@("`@("`@("`@("`@("`@(R!R96%D('1H92!F:7)S="`B1DE,12(*
M=VAI;&4@*"1B=68@97$@(D9)3$4B*2!["G)E860@7T9(+"`D8G5F+"`T.PIR
M96%D(%]&2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD["FUY("1F=6QL;F%M
M92`]("1B=68["F]U=',H<7%;56YP86-K:6YG($9)3$4@(B1F=6QL;F%M92(N
M+BY=*3L*;7D@)&-R8R`]("@@)&9U;&QN86UE(#U^('-\7BA;82UF7&1=>SA]
M*2]\?"`I(#\@)#$@.B!U;F1E9CL*;7D@*"1B87-E;F%M92P@)&5X="D@/2`H
M)&)U9B`]?B!M?"@_.BXJ+RD_*"XJ*2A<+BXJ*7PI.PIR96%D(%]&2"P@)&)U
M9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PII9B`H
M9&5F:6YE9"@D97AT*2!A;F0@)&5X="`A?B`O7"XH/SIP;7QP;'QI>'QA;"DD
M+VDI('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1C<F,D97AT(BP@)&)U
M9BP@,#<U-2D["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9U;&QN86UE?2`]
M("1F:6QE;F%M93L*)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9FEL96YA;65]
M(#T@)&9U;&QN86UE.PI]"F5L<VEF("@@)&9U;&QN86UE(#U^(&U\7B\_<VAL
M:6(O?"!A;F0@9&5F:6YE9"`D14Y6>U!!4E]414U0?2`I('L*;7D@)&9I;&5N
M86UE(#T@7W-A=F5?87,H(B1B87-E;F%M921E>'0B+"`D8G5F+"`P-S4U*3L*
M;W5T<R@B4TA,24(Z("1F:6QE;F%M95QN(BD["GT*96QS92!["B1-;V1U;&5#
M86-H97LD9G5L;&YA;65](#T@>PIB=68@/3X@)&)U9BP*8W)C(#T^("1C<F,L
M"FYA;64@/3X@)&9U;&QN86UE+`I].PHD<F5Q=6ER95]L:7-T>R1F=6QL;F%M
M97TK*SL*?0IR96%D(%]&2"P@)&)U9BP@-#L*?0IL;V-A;"!`24Y#(#T@*'-U
M8B!["FUY("@D<V5L9BP@)&UO9'5L92D@/2!`7SL*<F5T=7)N(&EF(')E9B`D
M;6]D=6QE(&]R("$D;6]D=6QE.PIM>2`D:6YF;R`]("1-;V1U;&5#86-H97LD
M;6]D=6QE?3L*<F5T=7)N('5N;&5S<R`D:6YF;SL*9&5L971E("1R97%U:7)E
M7VQI<W1[)&UO9'5L97T["B1)3D-[)&UO9'5L97T@/2`B+VQO861E<B\D:6YF
M;R\D;6]D=6QE(CL*:68@*"1%3E9[4$%27T-,14%.?2!A;F0@9&5F:6YE9"@F
M24\Z.D9I;&4Z.FYE=RDI('L*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=U]T;7!F
M:6QE(&]R(&1I92`B0V%N)W0@8W)E871E('1E;7`@9FEL93H@)"$B.PHD9F@M
M/F)I;FUO9&4H*3L*)&9H+3YP<FEN="@D:6YF;RT^>V)U9GTI.PHD9F@M/G-E
M96LH,"P@,"D["G)E='5R;B`D9F@["GT*96QS92!["FUY("1F:6QE;F%M92`]
M(%]S879E7V%S*"(D:6YF;RT^>V-R8WTN<&TB+"`D:6YF;RT^>V)U9GTI.PHD
M:6YF;RT^>V9I;&5](#T@)&9I;&5N86UE.PIO<&5N(&UY("1F:"P@)SPZ<F%W
M)RP@)&9I;&5N86UE(&]R(&1I92!Q<5M#86XG="!R96%D("(D9FEL96YA;64B
M.B`D(5T["G)E='5R;B`D9F@["GT*9&EE(")";V]T<W1R87!P:6YG(&9A:6QE
M9#H@8V%N)W0@9FEN9"!M;V1U;&4@)&UO9'5L92$B.PI]+"!`24Y#*3L*<F5Q
M=6ER92!84TQO861E<CL*<F5Q=6ER92!005(Z.DAE879Y.PIR97%U:7)E($-A
M<G`Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R.CI(96%V>3L*4$%2.CI(96%V
M>3HZ7VEN:71?9'EN86QO861E<B@I.PIR97%U:7)E($E/.CI&:6QE.PIW:&EL
M92`H;7D@)&9I;&5N86UE(#T@*'-O<G0@:V5Y<R`E<F5Q=6ER95]L:7-T*5LP
M72D@>PIU;FQE<W,@*"1)3D-[)&9I;&5N86UE?2!O<B`D9FEL96YA;64@/7X@
M+T)31%!!3B\I('L*:68@*"1F:6QE;F%M92`]?B`O7"YP;6,_)"]I*2!["G)E
M<75I<F4@)&9I;&5N86UE.PI]"F5L<V4@>PID;R`D9FEL96YA;64@=6YL97-S
M("1F:6QE;F%M92`]?B`O<VET96-U<W1O;6EZ95PN<&PD+SL*?0I]"F1E;&5T
M92`D<F5Q=6ER95]L:7-T>R1F:6QE;F%M97T["GT*=6YL97-S("@D8G5F(&5Q
M(")02UPP,#-<,#`T(BD@>PIO=71S*'%Q6TYO('II<"!F;W5N9"!A9G1E<B!&
M24Q%('-E8W1I;VX@:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL87-T($U!1TE#
M(#L*?0HD<W1A<G1?<&]S(#T@*'1E;&P@7T9(*2`M(#0[("`@("`@("`@("`@
M("`@(",@<W1A<G0@;V8@>FEP"GT*;7D@0'!A<E]A<F=S.PIM>2`H)&]U="P@
M)&)U;F1L92P@)&QO9V9H+"`D8V%C:&5?;F%M92D["F1E;&5T92`D14Y6>U!!
M4E]!4%!?4D554T5].R`C('-A;FET:7IE("A2155312!M87D@8F4@82!S96-U
M<FET>2!P<F]B;&5M*0II9B`H(21S=&%R=%]P;W,@;W(@*"1!4D=66S!=(&5Q
M("<M+7!A<BUO<'1I;VYS)R`F)B!S:&EF="DI('L*;7D@)61I<W1?8VUD(#T@
M<7<H"G`@("!B;&EB7W1O7W!A<@II("`@:6YS=&%L;%]P87(*=2`@('5N:6YS
M=&%L;%]P87(*<R`@('-I9VY?<&%R"G8@("!V97)I9GE?<&%R"BD["FEF("A`
M05)'5B!A;F0@)$%21U9;,%T@97$@)RTM<F5U<V4G*2!["G-H:69T($!!4D=6
M.PHD14Y6>U!!4E]!4%!?4D554T5](#T@<VAI9G0@0$%21U8["GT*96QS92![
M(",@;F]R;6%L('!A<FP@8F5H879I;W5R"FUY($!A9&1?=&]?:6YC.PIW:&EL
M92`H0$%21U8I('L*)$%21U9;,%T@/7X@+UXM*%M!24U/0DQB<7!I=7-4=ETI
M*"XJ*2\@;W(@;&%S=#L*:68@*"0Q(&5Q("=))RD@>PIP=7-H($!A9&1?=&]?
M:6YC+"`D,CL*?0IE;'-I9B`H)#$@97$@)TTG*2!["F5V86P@(G5S92`D,B([
M"GT*96QS:68@*"0Q(&5Q("=!)RD@>PIU;G-H:69T($!P87)?87)G<RP@)#([
M"GT*96QS:68@*"0Q(&5Q("=/)RD@>PHD;W5T(#T@)#(["GT*96QS:68@*"0Q
M(&5Q("=B)RD@>PHD8G5N9&QE(#T@)W-I=&4G.PI]"F5L<VEF("@D,2!E<2`G
M0B<I('L*)&)U;F1L92`]("=A;&PG.PI]"F5L<VEF("@D,2!E<2`G<2<I('L*
M)'%U:65T(#T@,3L*?0IE;'-I9B`H)#$@97$@)TPG*2!["F]P96X@)&QO9V9H
M+"`B/CXB+"`D,B!O<B!D:64@<7%;0V%N)W0@;W!E;B!L;V<@9FEL92`B)#(B
M.B`D(5T["GT*96QS:68@*"0Q(&5Q("=4)RD@>PHD8V%C:&5?;F%M92`]("0R
M.PI]"G-H:69T*$!!4D=6*3L*:68@*&UY("1C;60@/2`D9&ES=%]C;61[)#%]
M*2!["F1E;&5T92`D14Y6>R=005)?5$5-4"=].PII;FET7VEN8R@I.PIR97%U
M:7)E(%!!4CHZ1&ES=#L*)GLB4$%2.CI$:7-T.CHD8VUD(GTH*2!U;FQE<W,@
M0$%21U8["B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"1?*2!F;W(@0$%21U8["F5X
M:70["GT*?0IU;G-H:69T($!)3D,L($!A9&1?=&]?:6YC.PI]"GT*:68@*"1O
M=70I('L*>PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($%R8VAI=F4Z.EII
M<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*?0IM>2`D<&%R(#T@<VAI9G0H0$%2
M1U8I.PIM>2`D>FEP.PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)&9H
M+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!F:6YD('!A<B!F:6QE
M("(D<&%R(CH@)"%=.PIB;&5S<R@D9F@L("=)3SHZ1FEL92<I.PHD>FEP(#T@
M07)C:&EV93HZ6FEP+3YN97<["B@@)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE
M*"1F:"P@)'!A<BD@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I("D*;W(@9&EE
M('%Q6T5R<F]R(')E861I;F<@>FEP(&%R8VAI=F4@(B1P87(B73L*?0IM>2`E
M;65T85]P87(@/2!D;R!["FEF("@D>FEP(&%N9"!M>2`D;65T82`]("1Z:7`M
M/F-O;G1E;G1S*"=-151!+GEM;"<I*2!["B1M971A(#U^(',O+BI><&%R.B0O
M+VUS.PHD;65T82`]?B!S+UY<4RXJ+R]M<SL*)&UE=&$@/7X@+UX@("A;7CI=
M*RDZ("@N*RDD+VUG.PI]"GT["FEF("AD969I;F5D("1P87(I('L*;W!E;B!M
M>2`D<&@L("<\.G)A=R<L("1P87(@;W(@9&EE('%Q6T-A;B=T(')E860@<&%R
M(&9I;&4@(B1P87(B.B`D(5T["FUY("1B=68["G)E860@)'!H+"`D8G5F+"`T
M.PID:64@<7%;(B1P87(B(&ES(&YO="!A('!A<B!F:6QE72!U;FQE<W,@)&)U
M9B!E<2`B4$M<,#`S7#`P-"(["F-L;W-E("1P:#L*?0I#<F5A=&50871H*"1O
M=70I(#L*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=R@*)&]U="P*24\Z.D9I;&4Z
M.D]?0U)%050H*2!\($E/.CI&:6QE.CI/7U=23TY,62@I('P@24\Z.D9I;&4Z
M.D]?5%)53D,H*2P*,#<W-RP**2!O<B!D:64@<7%;0V%N)W0@8W)E871E(&9I
M;&4@(B1O=70B.B`D(5T["B1F:"T^8FEN;6]D92@I.PIS965K(%]&2"P@,"P@
M,#L*;7D@)&QO861E<CL*:68@*&1E9FEN960@)&1A=&%?<&]S*2!["G)E860@
M7T9(+"`D;&]A9&5R+"`D9&%T85]P;W,["GT@96QS92!["FQO8V%L("0O(#T@
M=6YD968["B1L;V%D97(@/2`\7T9(/CL*?0II9B`H(21%3E9[4$%27U9%4D)!
M5$E-?2!A;F0@)&QO861E<B`]?B`O7B@_.B,A?%Q`<F5M*2\I('L*<F5Q=6ER
M92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["E!!4CHZ1FEL=&5R.CI0;V13=')I
M<"T^87!P;'DH7"1L;V%D97(L("0P*3L*?0II9B`H)&UE=&%?<&%R>V-L96%N
M?2D@>PIM>2`D<&%R7V-L96%N(#T@(CTQ(CL*;7D@)'!A<W-?<&%R7V-L96%N
M(#T@=6,@(E]?<&%S<U]P87)?8VQE86Y?7R(["B1L;V%D97(@/7X@<WLH/SP]
M)'MP87-S7W!A<E]C;&5A;GTI*"`K*7T*>R1P87)?8VQE86X@+B`H(B`B('@@
M*&QE;F=T:"@D,2D@+2!L96YG=&@H)'!A<E]C;&5A;BDI*7UE9SL*?0HD9F@M
M/G!R:6YT*"1L;V%D97(I"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG(&QO861E
M<B!T;R`B)&]U="(Z("0A73L*:68@*"1B=6YD;&4I('L*<F5Q=6ER92!005(Z
M.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["FEN:71?
M:6YC*"D["G)E<75I<F5?;6]D=6QE<R@I.PIM>2!`:6YC(#T@9W)E<"![("$O
M0E-$4$%.+R!]"F=R97`@>PHH)&)U;F1L92!N92`G<VET92<I(&]R"B@D7R!N
M92`D0V]N9FEG.CI#;VYF:6=[87)C:&QI8F5X<'T@86YD"B1?(&YE("1#;VYF
M:6<Z.D-O;F9I9WMP<FEV;&EB97AP?2D["GT@0$E.0SL*<WPO*WPO?&<L('-\
M+R1\?"!F;W)E86-H($!I;F,["FUY("5F:6QE<SL*)&9I;&5S>R1??2LK(&9O
M<B!`1'EN84QO861E<CHZ9&Q?<VAA<F5D7V]B:F5C=',L('9A;'5E<R`E24Y#
M.PIM>2`D;&EB7V5X="`]("1#;VYF:6<Z.D-O;F9I9WML:6)?97AT?3L@("`@
M("`@("`C(%A86"!L:6)?97AT('9S(&1L97AT(#\*;7D@)7=R:71T96X["F9O
M<F5A8V@@;7D@)&ME>2`H<V]R="!K97ES("5F:6QE<RD@>PIM>2`H)&9I;&4L
M("1N86UE*3L*:68@*&1E9FEN960H;7D@)&9C(#T@)%!!4CHZ2&5A=GDZ.D9U
M;&Q#86-H97LD:V5Y?2DI('L**"1F:6QE+"`D;F%M92D@/2`H)&ME>2P@)&9C
M*3L*?0IE;'-E('L*9F]R96%C:"!M>2`D9&ER("A`:6YC*2!["FEF("@D:V5Y
M(#U^(&U\7EQ1)&1I<EQ%+R@N*BDD?&DI('L**"1F:6QE+"`D;F%M92D@/2`H
M)&ME>2P@)#$I.PIL87-T.PI]"FEF("@D:V5Y(#U^(&U\7B]L;V%D97(O6UXO
M72LO*"XJ*21\*2!["FEF("AM>2`D<F5F(#T@)$UO9'5L94-A8VAE>R0Q?2D@
M>PHH)&9I;&4L("1N86UE*2`]("@D<F5F+"`D,2D["FQA<W0["GT*:68@*"UF
M("(D9&ER+R0Q(BD@>PHH)&9I;&4L("1N86UE*2`]("@B)&1I<B\D,2(L("0Q
M*3L*;&%S=#L*?0I]"GT*?0IN97AT('5N;&5S<R!D969I;F5D("1N86UE.PIN
M97AT(&EF("1W<FET=&5N>R1N86UE?2LK.PIN97AT(&EF("%R968H)&9I;&4I
M(&%N9"`D9FEL92`]?B`O7"Y<421L:6)?97AT7$4D+VD["F]U=',H<W!R:6YT
M9BAQ<5M086-K:6YG($9)3$4@(B5S(BXN+ETL(')E9B`D9FEL92`_("1F:6QE
M+3Y[;F%M97T@.B`D9FEL92DI.PIM>2`D8V]N=&5N=#L*:68@*')E9B@D9FEL
M92DI('L*)&-O;G1E;G0@/2`D9FEL92T^>V)U9GT["GT*96QS92!["FQO8V%L
M("0O(#T@=6YD968["F]P96X@;7D@)'1H+"`G/#IR87<G+"`D9FEL92!O<B!D
M:64@<7%;0V%N)W0@<F5A9"`B)&9I;&4B.B`D(5T["B1C;VYT96YT(#T@/"1T
M:#X["F-L;W-E("1T:#L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YA<'!L>2A<
M)&-O;G1E;G0L("(\96UB961D960^+R1N86UE(BD*:68@(21%3E9[4$%27U9%
M4D)!5$E-?2!A;F0@)&YA;64@/7X@+UPN*#\Z<&U\:7A\86PI)"]I.PI005(Z
M.D9I;'1E<CHZ4&%T8VA#;VYT96YT+3YN97<M/F%P<&QY*%PD8V]N=&5N="P@
M)&9I;&4L("1N86UE*3L*?0HD9F@M/G!R:6YT*")&24Q%(BP*<&%C:R@G3B<L
M(&QE;F=T:"@D;F%M92D@*R`Y*2P*<W!R:6YT9B@B)3`X>"\E<R(L($%R8VAI
M=F4Z.EII<#HZ8V]M<'5T94-20S,R*"1C;VYT96YT*2P@)&YA;64I+`IP86-K
M*"=.)RP@;&5N9W1H*"1C;VYT96YT*2DL"B1C;VYT96YT*0IO<B!D:64@<7%;
M17)R;W(@=W)I=&EN9R!E;6)E9&1E9"!&24Q%('1O("(D;W5T(CH@)"%=.PIO
M=71S*'%Q6U=R:71T96X@87,@(B1N86UE(ETI.PI]"GT*:68@*"1Z:7`I('L*
M)'II<"T^=W)I=&54;T9I;&5(86YD;&4H)&9H*2`]/2!!<F-H:79E.CI::7`Z
M.D%:7T]+"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('II<"!P87)T(&]F("(D
M;W5T(ET["GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE7VYA;64L(#`L
M("1C86-H95]N86UE7W-I>F4["FEF("@A)&-A8VAE7VYA;64@86YD(&UY("1M
M=&EM92`]("AS=&%T*"1O=70I*5LY72D@>PIM>2`D8W1X(#T@1&EG97-T.CI3
M2$$M/FYE=R@Q*3L*;W!E;B!M>2`D=&@L("(\.G)A=R(L("1O=70["B1C='@M
M/F%D9&9I;&4H)'1H*3L*8VQO<V4@)'1H.PHD8V%C:&5?;F%M92`]("1C='@M
M/FAE>&1I9V5S=#L*?0HD8V%C:&5?;F%M92`N/2`B7#`B('@@*"1C86-H95]N
M86UE7W-I>F4@+2!L96YG=&@@)&-A8VAE7VYA;64I.PHD8V%C:&5?;F%M92`N
M/2`D8V%C:&5?;6%R:V5R.PIM>2`D;V9F<V5T(#T@)&9H+3YT96QL("L@;&5N
M9W1H*"1C86-H95]N86UE*2`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP<FEN
M="@D8V%C:&5?;F%M92P*<&%C:R@G3B<L("1O9F9S970I+`HD4$%27TU!1TE#
M*0HF)B`D9F@M/F-L;W-E"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R86EL
M97(@;V8@(B1O=70B.B`D(5T["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*
M>PIL87-T('5N;&5S<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA
M;64H*3L*<F5Q=6ER92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D
M97(H*3L*>PIR97%U:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ
M6FEP.PI]"FUY("1F:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@
M("`@("`@("`@("`@("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)
M3SHZ2&%N9&QE"B1F:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I
M92!Q<5MF9&]P96XH*2!F86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#
M:'5N:U-I>F4H+7,@7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W
M.PHH)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]
M/2!!<F-H:79E.CI::7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D
M:6YG('II<"!A<F-H:79E("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS
M971#:'5N:U-I>F4H-C0@*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@
M)'II<#L*)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U=',H
M(D5X=')A8W1I;F<@>FEP+BXN(BD["FEF("AD969I;F5D("1%3E9[4$%27U1%
M35!]*2![(",@<VAO=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H
M(&UY("1M96UB97(@*"`D>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB
M97(M/FES1&ER96-T;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF
M:6QE3F%M93L*;F5X="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S
M:&QI8B\**#\Z)$-O;F9I9SHZ0V]N9FEG>W9E<G-I;VY]+RD_"B@_.B1#;VYF
M:6<Z.D-O;F9I9WMA<F-H;F%M97TO*3\**%M>+UTK*0HD"GUX.PIM>2`D97AT
M<F%C=%]N86UE(#T@)#$["FUY("1D97-T7VYA;64@/2!&:6QE.CI3<&5C+3YC
M871F:6QE*"1%3E9[4$%27U1%35!]+"`D97AT<F%C=%]N86UE*3L*:68@*"UF
M("1D97-T7VYA;64@)B8@+7,@7R`]/2`D;65M8F5R+3YU;F-O;7!R97-S9613
M:7IE*"DI('L*;W5T<RAQ<5M3:VEP<&EN9R`B)&UE;6)E<E]N86UE(B!S:6YC
M92!I="!A;')E861Y(&5X:7-T<R!A="`B)&1E<W1?;F%M92)=*3L*?2!E;'-E
M('L*;W5T<RAQ<5M%>'1R86-T:6YG("(D;65M8F5R7VYA;64B('1O("(D9&5S
M=%]N86UE(BXN+ETI.PHH)&UE;6)E<BT^97AT<F%C=%1O1FEL94YA;65D*"1D
M97-T7VYA;64I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2D*;W(@9&EE('%Q
M6T5R<F]R(&5X=')A8W1I;F<@>FEP(&UE;6)E<B!T;R`B)&1E<W1?;F%M92)=
M.PIC:&UO9"@P-34U+"`D9&5S=%]N86UE*2!I9B`D7D\@97$@(FAP=7@B.PI]
M"GT*?0I]"G5N;&5S<R`H)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2D@>PID
M:64@/#P@(BXB('5N;&5S<R!`05)'5CL*57-A9V4Z("0P(%L@+4%L:6(N<&%R
M(%T@6R`M261I<B!=(%L@+4UM;V1U;&4@72!;('-R8RYP87(@72!;('!R;V=R
M86TN<&P@70HD,"!;("U"?"UB(%T@6RU/;W5T9FEL95T@<W)C+G!A<@HN"B1%
M3E9[4$%27U!23T=.04U%?2`]("1P<F]G;F%M92`]("0P(#T@<VAI9G0H0$%2
M1U8I.PI]"G-U8B!#<F5A=&50871H('L*;7D@*"1N86UE*2`]($!?.PIR97%U
M:7)E($9I;&4Z.D)A<V5N86UE.PIM>2`H)&)A<V5N86UE+"`D<&%T:"P@)&5X
M="D@/2!&:6QE.CI"87-E;F%M93HZ9FEL97!A<G-E*"1N86UE+"`H)UPN+BHG
M*2D["G)E<75I<F4@1FEL93HZ4&%T:#L*1FEL93HZ4&%T:#HZ;6MP871H*"1P
M871H*2!U;FQE<W,H+64@)'!A=&@I.R`C(&UK<&%T:"!D:65S('=I=&@@97)R
M;W(*?0IS=6(@<F5Q=6ER95]M;V1U;&5S('L*<F5Q=6ER92!L:6(["G)E<75I
M<F4@1'EN84QO861E<CL*<F5Q=6ER92!I;G1E9V5R.PIR97%U:7)E('-T<FEC
M=#L*<F5Q=6ER92!W87)N:6YG<SL*<F5Q=6ER92!V87)S.PIR97%U:7)E($-A
M<G`["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17)R;F\["G)E<75I
M<F4@17AP;W)T97(Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R.PIR97%U:7)E
M($9C;G1L.PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ4W!E
M8SL*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!#;VYF:6<["G)E<75I<F4@
M24\Z.DAA;F1L93L*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER92!#;VUP<F5S
M<SHZ6FQI8CL*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T
M.CI32$$["G)E<75I<F4@4$%2.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I
M<F4@4$%2.CI$:7-T.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*
M<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT.PIR97%U:7)E(&%T
M=')I8G5T97,["F5V86P@>R!R97%U:7)E($-W9"!].PIE=F%L('L@<F5Q=6ER
M92!7:6XS,B!].PIE=F%L('L@<F5Q=6ER92!38V%L87(Z.E5T:6P@?3L*979A
M;"![(')E<75I<F4@07)C:&EV93HZ56YZ:7`Z.D)U<G-T('T["F5V86P@>R!R
M97%U:7)E(%1I93HZ2&%S:#HZ3F%M961#87!T=7)E('T["F5V86P@>R!R97%U
M:7)E(%!E<FQ)3SL@<F5Q=6ER92!097)L24\Z.G-C86QA<B!].PIE=F%L('L@
M<F5Q=6ER92!U=&8X('T["GT*<W5B(%]S971?<&%R7W1E;7`@>PII9B`H9&5F
M:6YE9"`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N
M*RDO*2!["B1P87)?=&5M<"`]("0Q.PIR971U<FX["GT*9F]R96%C:"!M>2`D
M<&%T:"`H"BAM87`@)$5.5GLD7WTL('%W*"!005)?5$U01$E2(%1-4$1)4B!4
M14U01$E2(%1%35`@5$U0("DI+`IQ=R@@0SI<7%1%35`@+W1M<"`N("D**2![
M"FYE>'0@=6YL97-S(&1E9FEN960@)'!A=&@@86YD("UD("1P871H(&%N9"`M
M=R`D<&%T:#L*;7D@)'5S97)N86UE.PIM>2`D<'=U:60["F5V86P@>R@D<'=U
M:60I(#T@9V5T<'=U:60H)#XI(&EF(&1E9FEN960@)#X[?3L*:68@*"!D969I
M;F5D*"97:6XS,CHZ3&]G:6Y.86UE*2`I('L*)'5S97)N86UE(#T@)E=I;C,R
M.CI,;V=I;DYA;64["GT*96QS:68@*&1E9FEN960@)'!W=6ED*2!["B1U<V5R
M;F%M92`]("1P=W5I9#L*?0IE;'-E('L*)'5S97)N86UE(#T@)$5.5GM54T52
M3D%-17T@?'P@)$5.5GM54T52?2!\?"`G4UE35$5-)SL*?0IM>2`D<W1M<&1I
M<B`]("(D<&%T:"1S>7-[7V1E;&EM?7!A<BTB+G5N<&%C:R@B2"HB+"`D=7-E
M<FYA;64I.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*;7D@)&-A8VAE7V1I<CL*
M:68@*"1%3E9[4$%27T-,14%.?2D@>PHD8V%C:&5?9&ER(#T@(G1E;7`M)"0B
M.PI]"F5L<V4@>PIO<&5N(&UY("1F:"P@(CPZ<F%W(BP@)'!R;V=N86UE(&]R
M(&1I92!Q<5M#86XG="!R96%D("(D<')O9VYA;64B.B`D(5T["FEF("@H;7D@
M)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"1F:"DI(#X](#`I('L*<V5E
M:R`D9F@L("1M86=I8U]P;W,*+2`D1DE,15]O9F9S971?<VEZ90HM(&QE;F=T
M:"@D8V%C:&5?;6%R:V5R*2P@,#L*;7D@)&)U9CL*<F5A9"`D9F@L("1B=68L
M(&QE;F=T:"@D8V%C:&5?;6%R:V5R*3L*:68@*"1B=68@97$@)&-A8VAE7VUA
M<FME<BD@>PIS965K("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S
M:7IE"BT@;&5N9W1H*"1C86-H95]M87)K97(I"BT@)&-A8VAE7VYA;65?<VEZ
M92P@,#L*<F5A9"`D9F@L("1B=68L("1C86-H95]N86UE7W-I>F4["B1B=68@
M/7X@<R]<,"\O9SL*)&-A8VAE7V1I<B`](")C86-H92TD8G5F(CL*?0I]"F-L
M;W-E("1F:#L*?0II9B`H(21C86-H95]D:7(I('L*)&-A8VAE7V1I<B`](")T
M96UP+20D(CL*)$5.5GM005)?0TQ%04Y](#T@,3L*?0HD<W1M<&1I<B`N/2`B
M)'-Y<WM?9&5L:6U])&-A8VAE7V1I<B(["FUK9&ER("1S=&UP9&ER+"`P-S4U
M.PHD14Y6>U!!4E]414U0?2`]("1S=&UP9&ER.PIL87-T.PI]"B1P87)?=&5M
M<"`]("0Q(&EF("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]
M?B`O*"XK*2\["GT*<W5B(%]S879E7V%S('L*;7D@*"1N86UE+"`D8V]N=&5N
M=',L("1M;V1E*2`]($!?.PIM>2`D9G5L;&YA;64@/2`B)'!A<E]T96UP+R1N
M86UE(CL*=6YL97-S("@M92`D9G5L;&YA;64I('L*;7D@)'1E;7!N86UE(#T@
M(B1F=6QL;F%M92XD)"(["FUY("1F:#L**&]P96X@)&9H+"`G/CIR87<G+"`D
M=&5M<&YA;64I"B8F("AP<FEN="`D9F@@)&-O;G1E;G1S*0HF)B`H8VQO<V4@
M)&9H*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R`B)'1E;7!N86UE(CH@)"%=
M.PIC:&UO9"`D;6]D92P@)'1E;7!N86UE(&EF(&1E9FEN960@)&UO9&4["G)E
M;F%M92@D=&5M<&YA;64L("1F=6QL;F%M92D@;W(@=6YL:6YK*"1T96UP;F%M
M92D["GT*<F5T=7)N("1F=6QL;F%M93L*?0IS=6(@7W-E=%]P<F]G;F%M92![
M"FEF("AD969I;F5D("1%3E9[4$%27U!23T=.04U%?2!A;F0@)$5.5GM005)?
M4%)/1TY!345](#U^("\H+BLI+RD@>PHD<')O9VYA;64@/2`D,3L*?0HD<')O
M9VYA;64@?'P]("0P.PII9B`H)$5.5GM005)?5$5-4'T@86YD(&EN9&5X*"1P
M<F]G;F%M92P@)$5.5GM005)?5$5-4'TI(#X](#`I('L*)'!R;V=N86UE(#T@
M<W5B<W1R*"1P<F]G;F%M92P@<FEN9&5X*"1P<F]G;F%M92P@)'-Y<WM?9&5L
M:6U]*2`K(#$I.PI]"FEF("@A)$5.5GM005)?4%)/1TY!345](&]R(&EN9&5X
M*"1P<F]G;F%M92P@)'-Y<WM?9&5L:6U]*2`^/2`P*2!["FEF("AO<&5N(&UY
M("1F:"P@)SPG+"`D<')O9VYA;64I('L*<F5T=7)N(&EF("US("1F:#L*?0II
M9B`H+7,@(B1P<F]G;F%M921S>7-[7V5X97TB*2!["B1P<F]G;F%M92`N/2`D
M<WES>U]E>&5].PIR971U<FX["GT*?0IF;W)E86-H(&UY("1D:7(@*'-P;&ET
M("]<421S>7-[<&%T:%]S97!]7$4O+"`D14Y6>U!!5$A]*2!["FYE>'0@:68@
M97AI<W1S("1%3E9[4$%27U1%35!](&%N9"`D9&ER(&5Q("1%3E9[4$%27U1%
M35!].PHD9&ER(#U^(',O7%$D<WES>U]D96QI;7U<120O+SL**"@D<')O9VYA
M;64@/2`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M921S>7-[7V5X97TB*2P@
M;&%S="D*:68@+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64D<WES>U]E
M>&5](CL**"@D<')O9VYA;64@/2`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M
M92(I+"!L87-T*0II9B`M<R`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M92([
M"GT*?0IS=6(@7V9I>%]P<F]G;F%M92!["B0P(#T@)'!R;V=N86UE('Q\/2`D
M14Y6>U!!4E]04D]'3D%-17T["FEF("AI;F1E>"@D<')O9VYA;64L("1S>7-[
M7V1E;&EM?2D@/"`P*2!["B1P<F]G;F%M92`]("(N)'-Y<WM?9&5L:6U])'!R
M;V=N86UE(CL*?0IM>2`D<'=D(#T@*&1E9FEN960@)D-W9#HZ9V5T8W=D*2`_
M($-W9#HZ9V5T8W=D*"D*.B`H*&1E9FEN960@)E=I;C,R.CI'971#=V0I(#\@
M5VEN,S(Z.D=E=$-W9"@I(#H@8'!W9&`I.PIC:&]M<"@D<'=D*3L*)'!R;V=N
M86UE(#U^(',O7B@_/5PN7"X_7%$D<WES>U]D96QI;7U<12DO)'!W9"1S>7-[
M7V1E;&EM?2\["B1%3E9[4$%27U!23T=.04U%?2`]("1P<F]G;F%M93L*?0IS
M=6(@7W!A<E]I;FET7V5N=B!["FEF("@@)$5.5GM005)?24Y)5$E!3$E:141]
M*RL@/3T@,2`I('L*<F5T=7)N.PI](&5L<V4@>PHD14Y6>U!!4E])3DE424%,
M25I%1'T@/2`R.PI]"F9O<B`H<7<H(%-005=.140@5$5-4"!#3$5!3B!$14)5
M1R!#04-(12!04D]'3D%-12`I("D@>PID96QE=&4@)$5.5GLB4$%27R1?(GT[
M"GT*9F]R("AQ=R\@5$U01$E2(%1%35`@0TQ%04X@1$5"54<@+RD@>PHD14Y6
M>R)005)?)%\B?2`]("1%3E9[(E!!4E]'3$]"04Q?)%\B?2!I9B!E>&ES=',@
M)$5.5GLB4$%27T=,3T)!3%\D7R)].PI]"FEF("@D14Y6>U!!4E]414U0?2D@
M>PID96QE=&4@)$5.5GM005)?0TQ%04Y].PI]"GT*<W5B(&]U=',@>PIR971U
M<FX@:68@)'%U:65T.PII9B`H)&QO9V9H*2!["G!R:6YT("1L;V=F:"`B0%]<
M;B(["GT*96QS92!["G!R:6YT(")`7UQN(CL*?0I]"G-U8B!I;FET7VEN8R![
M"G)E<75I<F4@0V]N9FEG.PIP=7-H($!)3D,L(&=R97`@9&5F:6YE9"P@;6%P
M("1#;VYF:6<Z.D-O;F9I9WLD7WTL('%W*`IA<F-H;&EB97AP('!R:79L:6)E
M>'`@<VET96%R8VAE>'`@<VET96QI8F5X<`IV96YD;W)A<F-H97AP('9E;F1O
M<FQI8F5X<`HI.PI]"G!A8VMA9V4@;6%I;CL*<F5Q=6ER92!005(["G5N<VAI
M9G0@0$E.0RP@7"9005(Z.F9I;F1?<&%R.PI005(M/FEM<&]R="A`<&%R7V%R
M9W,I.PID:64@<7%;<&%R+G!L.B!#86XG="!O<&5N('!E<FP@<V-R:7!T("(D
M<')O9VYA;64B.B!.;R!S=6-H(&9I;&4@;W(@9&ER96-T;W)Y7&Y="G5N;&5S
M<R`M92`D<')O9VYA;64["F1O("1P<F]G;F%M93L*0T]213HZ97AI="@D,2D@
M:68@*"1`(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("1`(&EF("1`
M.PI].PHD.CI?7T524D]2(#T@)$`@:68@)$`["GT*0T]213HZ97AI="@D,2D@
M:68@*"0Z.E]?15)23U(@/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@
M)#HZ7U]%4E)/4B!I9B`D.CI?7T524D]2.PHQ.PI?7T5.1%]?"@````!P97)L
M>'-I+F,``````````2-%9P````")J\WO`````/[<NI@`````=E0R$`````#P
MX=+#`````````````````1L#._P````>````8'7__WP&``#@=___%`$``#!X
M__\H`0``8'C__SP!``"@>/__4`$``/!X__]T`0```'G__X@!```0>O___`$`
M`,!Z__\D`@``D'S__X@"``#`??__P`(``""`__\0`P``H(#__S0#``#4@/__
M2`,``/"!__]\`P``]('__Y`#``"PA/__%`0``/"$__\X!```H(;__Z@$``!P
MB___Y`0``!",__\(!0``0(S__R@%``"@C?__;`4``!"6__\$!@``@)C__T`&
M``#0J?__L`8``)"Z___4!@``X+K___0&````O/__+`<``."]__]H!P``````
M`!```````````7I2``1X'@$;#!\`$````!@```#$=O__-`````!!!QX0````
M+`````!W__\P`````````!````!`````''?__SP`````````(````%0```!(
M=___2`````!!#B"=!)X#0I,"3M[=TPX`````$````'@```!T=___!```````
M``!P````C````'!W__\(`0```$$.4)T*G@E"DPB4!T*8`Y<$1)D"0Y8%E097
MUM5!V-=!V4/>W=/4#@!!#E"3")0'E0:6!9<$F`.9`IT*G@E/UM5!V-=!V4'>
MW=/4#@!!#E"3")0'G0J>"4*7!)@#F0)!V-="V20``````0``#'C__ZP`````
M00ZP`9T6GA5$DQ18"M[=TPX`00L```!@````*`$``)1X___0`0```$,.T`&=
M&IX91I07DQA$U--(WMT.`$$.T`&3&)07G1J>&4:6%9464)@3EQ1KV-=,"M33
M0=;500M&EQ28$T/3U-76U]A#E!>3&$&6%94609@3EQ0`-````(P!````>O__
M*`$```!"#L`@1)V(!)Z'!$*3A@24A01#E80$EH,$9@K=WM76T]0.`$$+``!,
M````Q`$``/AZ__]<`@```$$.<)T.G@U"DPR4"TF5"I8)EPB8!YD&F@4"2`K>
MW=G:U]C5UM/4#@!!"TF;!&$*VT$+3-M"FP1#VT.;!$+;`"`````4`@``"'W_
M_W@`````00X@G02>`T23`I0!4-[=T]0.`!`````X`@``9'W__S0`````````
M,````$P"``"$??__%`$```!"#M!`1)V*")Z)"$*3B`B4APA"E88(=0K=WM73
MU`X`00L``!````"``@``;'[__P0`````````@````)0"``!<?O__N`(```!"
M#I`B0YVB!)ZA!$*3H`24GP1"E9X$EIT$8`K=WM76T]0.`$$+0YB;!)><!$2<
MEP2;F`1*FID$F9H$`D,*V-=!VME!W-M!"UL*V-=!VME!W-M!"T;9VD'8UT'<
MVT28FP27G`1!FID$F9H$09R7!)N8!```(````!@#``"4@/__/`````!!#B"=
M!)X#19,"1M[=TPX`````;````#P#``"P@/__J`$```!"#L`@0YV(!)Z'!$:5
MA`1&E(4$DX8$;-330=5!W=X.`$$.P""5A`2=B`2>AP1!U4[=W@X`00[`().&
M!)2%!)6$!)V(!)Z'!$P*U--"U40+0]/4U4.4A023A@1!E80$`#@```"L`P``
M\('__\P$````00Y@G0R>"T23"I0)1I4(E@>7!I@%F00#&P$*WMW9U]C5UM/4
M#@!!"P```"````#H`P``A(;__Y@`````00XPG0:>!423!%`*WMW3#@!!"QP`
M```,!````(?__S``````0PX0G0*>`4C>W0X`````0````"P$```0A___8`$`
M``!!#E"="IX)1),(E`="E0:6!5@*WMW5UM/4#@!!"T\*WMW5UM/4#@!!"T&7
M!&#71)<$``"4````<`0``"R(__]H"````$(.P(,"0YVV()ZU($*;K""<JR!&
MD[0@E+,@E;(@EK$@0Y>P()BO(%R:K2"9KB`"CPK:V4+=WMO<U]C5UM/4#@!!
M"P*DV=I2W=[;W-?8U=;3U`X`0P[`@P*3M""4LR"5LB"6L2"7L""8KR"9KB":
MK2";K""<JR"=MB">M2`"G]G:0YJM()FN(#@````(!0``!)#__VP"````00Y@
MG0R>"T23"I0)1)4(E@=$EP:8!9D$`F0*WMW9U]C5UM/4#@!!"P```#@```!$
M!0``.)+__U`!````00Y`G0B>!T23!I0%0I4$E@-C"M[=U=;3U`X`00M5"M[=
MU=;3U`X`00L``#````"`!0``W&[__VP"````00Y@G0R>"T63"I0)E0B6!P)I
M"M[=U=;3U`X`00L````````@````M`4``!BC___`$````$$.H`5&G52>4P,E
M!`K=W@X`00L<````V`4``+2S__](`````$,.$)T"G@%.WMT.`````#0```#X
M!0``Y+/__QP!````00Y0G0J>"4.3")0'0Y4&E@5$EP28`YD"90K>W=G7V-76
MT]0.`$$+.````#`&``#,M/__V`$```!!#D"=")X'0Y,&E`5#E026`T(%2`("
M9`K>W=76T]0&2`X`00L`````````$````&P&``!PMO__%```````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````#0'P```````(`?
M`````````0````````#R`@````````$`````````_0(````````!````````
M``<#````````#P````````"(!`````````P`````````P!L````````-````
M`````-1D````````&0````````!8NP$``````!L`````````"``````````:
M`````````&"[`0``````'``````````(`````````/7^_V\`````F`(`````
M```%`````````*`-````````!@````````#X`P````````H`````````JP0`
M```````+`````````!@`````````%0````````````````````,`````````
MB+T!```````"`````````$@`````````%``````````'`````````!<`````
M````>!L````````'`````````(`3````````"`````````#X!P````````D`
M````````&``````````8````````````````````^___;P`````!```(````
M`/[__V\`````(!,```````#___]O``````(`````````\/__;P````!,$@``
M`````/G__V\`````$@``````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````.`;````````X!L`
M``````#@&P```````&B[`0``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````X&(`````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````P&$`````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````#`9```````````````````<&$`````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````!`'```
M```````````````````````````(P`$``````/____\`````H&@```````!@
M90``````````````````4&8```````"`:````````(AH````````D&@`````
M``"8:```````````````````:&@```````!P:```````````````````J&@`
M``````!'0T,Z("A'3E4I(#$R+C$N,`!'0T,Z("AC<F]S<W1O;VPM3D<@,2XR
M-2XP+C(X7V(X.&0S,S@I(#$R+C$N,```+G-H<W1R=&%B`"YI;G1E<G``+FYO
M=&4N9VYU+F)U:6QD+6ED`"YN;W1E+D%"22UT86<`+F=N=2YH87-H`"YD>6YS
M>6T`+F1Y;G-T<@`N9VYU+G9E<G-I;VX`+F=N=2YV97)S:6]N7W(`+G)E;&$N
M9'EN`"YR96QA+G!L=``N:6YI=``N=&5X=``N9FEN:0`N<F]D871A`"YE:%]F
M<F%M95]H9'(`+F5H7V9R86UE`"YI;FET7V%R<F%Y`"YF:6YI7V%R<F%Y`"YD
M>6YA;6EC`"YG;W0`+F1A=&$`+F)S<P`N8V]M;65N=```````````````````
M````````````````````````````````````````````````````````````
M``````````````````L````!`````@`````````X`@```````#@"````````
M&P````````````````````$````````````````````3````!P````(`````
M````5`(```````!4`@```````"0````````````````````$````````````
M````````)@````<````"`````````'@"````````>`(````````@````````
M````````````!````````````````````#0```#V__]O`@````````"8`@``
M`````)@"````````7`$````````%``````````@````````````````````^
M````"P````(`````````^`,```````#X`P```````*@)````````!@````,`
M```(`````````!@`````````1@````,````"`````````*`-````````H`T`
M``````"K!````````````````````0```````````````````$X```#___]O
M`@````````!,$@```````$P2````````S@`````````%``````````(`````
M`````@````````!;````_O__;P(`````````(!,````````@$P```````&``
M````````!@````(````(````````````````````:@````0````"````````
M`(`3````````@!,```````#X!P````````4`````````"``````````8````
M`````'0````$````0@````````!X&P```````'@;````````2``````````%
M````%0````@`````````&`````````!^`````0````8`````````P!L`````
M``#`&P```````!@````````````````````$````````````````````>0``
M``$````&`````````.`;````````X!L```````!0````````````````````
M$````````````````````(0````!````!@````````!`'````````$`<````
M````E$@``````````````````$````````````````````"*`````0````8`
M````````U&0```````#49````````!0````````````````````$````````
M````````````D`````$````"`````````/!D````````\&0```````#P00``
M````````````````$````````````````````)@````!`````@````````#@
MI@```````."F````````_`````````````````````0`````````````````
M``"F`````0````(`````````X*<```````#@IP```````(`&````````````
M```````(````````````````````L`````X````#`````````%B[`0``````
M6+L````````(````````````````````"``````````(`````````+P````/
M`````P````````!@NP$``````&"[````````"`````````````````````@`
M````````"`````````#(````!@````,`````````:+L!``````!HNP``````
M`"`"````````!@`````````(`````````!``````````T0````$````#````
M`````(B]`0``````B+T```````!X`@``````````````````"``````````(
M`````````-8````!`````P``````````P`$```````#`````````@```````
M``````````````@```````````````````#<````"`````,`````````@,`!
M``````"`P````````"@````````````````````(````````````````````
MX0````$````P````````````````````@,`````````_````````````````
M`````0`````````!``````````$````#`````````````````````````+_`
F````````Z@````````````````````$`````````````````````
