sap.ui.define([
	"jquery.sap.global",
	"sap/base/util/UriParameters",
	"sap/base/Log",
	"sap/ui/core/Component",
	"sap/ui/core/UIComponent",
	"sap/ui/model/odata/ODataModel",
	"sap/ui/model/odata/v2/ODataModel",
	"sap/ui/model/odata/v4/ODataModel",
	"sap/ui/model/json/JSONModel",
	"sap/ui/model/xml/XMLModel",
	"sap/ui/model/resource/ResourceModel",
	"sap/ui/test/v2models/parent/CustomModel"
], function(jQuery, UriParameters, Log, Component) {

	"use strict";
	/*global sinon, QUnit*/

	// Re-assigning the 'XMLHttpRequest' property on the window in this strange way prevents Safari 12/13 (or WebKit)
	// from wrongly optimizing access. As the sinon fake server is only used in some parts of this test module Safari
	// might wrongly optimize the access (e.g. within jQuery) to override the fake server which fails those tests.
	window.XMLHttpRequest = window["XML" + "HttpRequest"];


	var Helper = {
		spyModels: function() {
			this.modelSpy = {
				odata: sinon.spy(sap.ui.model.odata, "ODataModel"),
				odataV2: sinon.spy(sap.ui.model.odata.v2, "ODataModel"),
				odataV4: sinon.spy(sap.ui.model.odata.v4, "ODataModel"),
				json: sinon.spy(sap.ui.model.json, "JSONModel"),
				xml: sinon.spy(sap.ui.model.xml, "XMLModel"),
				resource: sinon.spy(sap.ui.model.resource, "ResourceModel"),
				custom: sinon.spy(sap.ui.test.v2models.parent, "CustomModel")
			};
		},
		restoreModels: function() {
			if (this.modelSpy) {
				for (var sName in this.modelSpy) {
					if (this.modelSpy[sName] && this.modelSpy[sName].restore) {
						this.modelSpy[sName].restore();
					}
				}
				this.modelSpy = null;
			}
		},
		stubGetUriParameters: function(mMockParams) {
			var oGetParameterStub = sinon.stub();

			oGetParameterStub.withArgs('sap-client').returns(mMockParams && mMockParams.sapClient || 'foo');
			oGetParameterStub.withArgs('sap-server').returns(mMockParams && mMockParams.sapServer || 'bar');
			oGetParameterStub.withArgs('sap-system').returns(mMockParams && mMockParams.sapSystem);

			if (mMockParams && mMockParams["preload-component-models"]) {
				oGetParameterStub.withArgs('sap-ui-xx-preload-component-models').returns('true');
			}

			this.oGetUriParametersStub = sinon.stub(UriParameters, 'fromQuery').returns({
				get: oGetParameterStub
			});

			var sSAPLanguage = sap.ui.getCore().getConfiguration().getSAPLogonLanguage();

			this.oConfigurationStub = sinon.stub(sap.ui.getCore().getConfiguration(), 'getSAPParam');
			this.oConfigurationStub.withArgs('sap-language').returns(mMockParams && mMockParams.sapLanguage || sSAPLanguage);
			this.oConfigurationStub.withArgs('sap-client').returns(mMockParams && mMockParams.sapClient || 'foo');
			this.oConfigurationStub.withArgs('sap-server').returns(mMockParams && mMockParams.sapServer || 'bar');
			this.oConfigurationStub.withArgs('sap-system').returns(mMockParams && mMockParams.sapSystem);
		},
		restoreGetUriParameters: function() {
			if (this.oGetUriParametersStub && this.oGetUriParametersStub.restore) {
				this.oGetUriParametersStub.restore();
				this.oGetUriParametersStub = null;
			}
			if (this.oConfigurationStub && this.oConfigurationStub.restore) {
				this.oConfigurationStub.restore();
				this.oConfigurationStub = null;
			}
		}
	};
	// Binds all the helper functions to the test instance
	var bindHelper = function() {
		for (var method in Helper) {
			if (Helper.hasOwnProperty(method)) {
				this[method] = Helper[method].bind(this);
			}
		}
	};


	QUnit.module("ui5:// URL resolution for local annotations", {
		before: function() {
			// preload any used libraries / modules to avoid sync requests
			return sap.ui.getCore().loadLibraries([
					"sap.ui.layout", "sap.ui.unified", "sap.m"
			]).then(function() {
				return new Promise(function(resolve, reject) {
					sap.ui.require([
						"sap/m/Label",
						"sap/ui/core/CustomData",
						"sap/ui/core/mvc/XMLView",
						"sap/ui/core/routing/Router",
						"sap/ui/model/odata/ODataAnnotations"
					], function() {
						resolve();
					}, reject);
				});
			});
		},
		beforeEach: function() {
			bindHelper.call(this);

			this.spyModels();
			this.oLogSpy = sinon.spy(Log, "error");

			sap.ui.loader.config({
				paths: {
					"path/to/odata/service": "https://remote.system:9000/odata/service",
					"sap/ui/test/v2models/ui5urls": "test-resources/sap/ui/core/qunit/component/testdata/v2models/ui5Urls",
					"another/name/space": "test-resources/sap/ui/core/qunit/component/testdata/v2models/ui5Urls/another/name/space",
					"cool.name.space": "test-resources/sap/ui/core/qunit/component/testdata/v2models/ui5Urls/cool/name/space"
				}
			});
		},
		afterEach: function() {
			this.restoreModels();
			this.oLogSpy.restore();

			this.oComponent.destroy();

			this.restoreGetUriParameters();

			// To keep reusing the same component for async and sync path tests,
			// we need to unload the Component and remove the leftovers from the ComponentMetadata.
			// This way all tests start fresh and actually load the Component again.
			sap.ui.loader._.unloadResources('sap/ui/test/v2models/ui5urls/Component.js', true, true, true);
			delete sap.ui.test.v2models.ui5Urls.Component.getMetadata()._oManifest;

			// remove the previous path-configs/resource-roots
			sap.ui.loader.config({
				paths: {
					"cool.name.space": null,
					"this/is/a/resourceRoot": null
				}
			});
		}
	});

	function fnAssert(assert) {
		// resource roots now defined after component creation
		assert.equal(
			sap.ui.require.toUrl("this/is/a/resourceRoot"),
			"test-resources/sap/ui/core/qunit/component/testdata/v2models/ui5Urls/resourceRoots/subfolder",
			"Resource-roots is now defined."
		);

		// sap.ui.model.odata.ODataModel
		sinon.assert.callCount(this.modelSpy.odataV2, 2);

		// model: "ODataModel"
		sinon.assert.calledWithExactly(this.modelSpy.odataV2.getCall(0), {
			serviceUrl: 'https://remote.system:9000/odata/service?sap-client=foo&sap-server=bar',
			metadataUrlParams: {"sap-language": "EN"},
			annotationURI: [
				'/path/to/odata/annotations/1?sap-language=EN&sap-client=foo',
				sap.ui.loader._.resolveURL('test-resources/sap/ui/core/qunit/component/testdata/v2models/ui5Urls/annotations/2?sap-language=EN&sap-client=foo'),
				sap.ui.loader._.resolveURL('test-resources/sap/ui/core/qunit/component/testdata/v2models/ui5Urls/another/name/space/annotations/3?sap-language=EN&sap-client=foo'),
				sap.ui.loader._.resolveURL('test-resources/sap/ui/core/qunit/component/testdata/v2models/ui5Urls/cool/name/space/annotations/4.xml?sap-language=EN&sap-client=foo'),
				sap.ui.loader._.resolveURL('resources/unkown.name.space/annotations/5.xml?sap-language=EN&sap-client=foo'),
				sap.ui.loader._.resolveURL('resources/another/unkown/name/space/annotations/6.xml?sap-language=EN&sap-client=foo'),
				sap.ui.loader._.resolveURL('test-resources/sap/ui/core/qunit/component/testdata/v2models/ui5Urls/resourceRoots/subfolder/annotations/file7.xml?sap-language=EN&sap-client=foo')
			],
			useBatch: false,
			refreshAfterChange: false
		});

		// model: "OtherODataModel"
		sinon.assert.calledWithExactly(this.modelSpy.odataV2.getCall(1), {
			serviceUrl: 'https://remote.system:9000/odata/service',
			useBatch: true,
			refreshAfterChange: true
		});
	}

	QUnit.test("SYNC: Resolve annotation urls; Manifest first;", function(assert) {
		// stub uri parameters with sap-client and sap-server
		this.stubGetUriParameters();

		assert.equal(sap.ui.require.toUrl("this/is/a/resourceRoot"), "resources/this/is/a/resourceRoot", "Resource-roots not defined yet.");

		this.oComponent = sap.ui.component({
			name: "sap.ui.test.v2models.ui5urls",
			manifest: true,
			async: false
		});

		fnAssert.call(this, assert);
	});

	QUnit.test("SYNC: Resolve annotation urls; Manifest last;", function(assert) {
		// stub uri parameters with sap-client and sap-server
		this.stubGetUriParameters();

		assert.equal(sap.ui.require.toUrl("this/is/a/resourceRoot"), "resources/this/is/a/resourceRoot", "Resource-roots not defined yet.");

		this.oComponent = sap.ui.component({
			name: "sap.ui.test.v2models.ui5urls"
		});

		fnAssert.call(this, assert);
	});
});