/*
 * Copyright (c) 2021, the SerenityOS developers.
 * Copyright (c) 2021, Arne Elster <arne@elster.li>
 *
 * SPDX-License-Identifier: BSD-2-Clause
 */

#pragma once

#include <AK/Array.h>
#include <AK/BitStream.h>
#include <AK/Span.h>

namespace Audio::MP3::Tables::Huffman {

template<typename Symbol>
struct HuffmanEntry {
    using SymbolType = Symbol;
    unsigned code;
    size_t code_length;
    Symbol symbol;
};

template<typename Symbol>
struct HuffmanNode {
    int left;
    int right;
    unsigned code;
    size_t code_length;
    Symbol symbol;
    bool is_leaf() const { return left == -1 && right == -1; }
};

template<typename Symbol, size_t Size>
using HuffmanEntries = Array<Symbol, Size>;

template<typename Symbol, size_t Size>
using HuffmanNodes = Array<HuffmanNode<Symbol>, Size>;

template<typename Symbol, size_t Size>
consteval size_t length_of_longest_huffman_code(HuffmanEntries<Symbol, Size> const& table)
{
    size_t max = 0;
    for (size_t i = 0; i < table.size(); i++) {
        if (table[i].code_length > max)
            max = table[i].code_length;
    }
    return max;
}

template<typename Symbol, typename NodeArrayType>
consteval int make_huffman_tree_internal(auto const& table, NodeArrayType&& nodes)
{
    size_t allocation_count = 1;

    nodes = {};
    nodes[0].left = nodes[0].right = -1;

    for (size_t i = 0; i < table.size(); i++) {
        HuffmanEntry<Symbol> const& entry = table[i];
        int tree_pointer = 0;

        for (size_t j = 0; j < entry.code_length; j++) {
            bool const bit = (entry.code >> (entry.code_length - j - 1)) & 1;
            bool const end_of_code = j == entry.code_length - 1;

            int& target_index = bit ? nodes[tree_pointer].left : nodes[tree_pointer].right;

            if (target_index != -1) {
                if (end_of_code)
                    return -1;
                tree_pointer = target_index;
            } else {
                tree_pointer = target_index = allocation_count++;
                nodes[target_index].left = nodes[target_index].right = -1;
                nodes[target_index].code_length = j + 1;
                if (end_of_code) {
                    nodes[target_index].symbol = entry.symbol;
                    nodes[target_index].code = entry.code;
                }
            }
        }
    }

    return allocation_count;
}

template<auto table>
consteval auto make_huffman_tree()
{
    using Symbol = typename decltype(table)::ValueType::SymbolType;
    constexpr size_t size_worst_case = 1 << (length_of_longest_huffman_code(table) + 1);
    constexpr int size_fitted = make_huffman_tree_internal<Symbol>(table, HuffmanNodes<Symbol, size_worst_case>());
    static_assert(size_fitted >= 0, "Huffman tree is ill-formed");

    HuffmanNodes<Symbol, size_fitted> nodes;
    make_huffman_tree_internal<Symbol>(table, nodes);

    return nodes;
}

template<typename T>
struct HuffmanDecodeResult {
    size_t bits_read;
    Optional<HuffmanNode<T>> code;
};

template<typename T>
HuffmanDecodeResult<T> huffman_decode(InputBitStream& bitstream, Span<HuffmanNode<T> const> tree, size_t max_bits_to_read)
{
    HuffmanNode<T> const* node = &tree[0];
    size_t bits_read = 0;

    while (!node->is_leaf() && !bitstream.has_any_error() && max_bits_to_read-- > 0) {
        bool const direction = bitstream.read_bit_big_endian();
        ++bits_read;
        if (direction) {
            if (node->left == -1)
                return {};
            node = &tree[node->left];
        } else {
            if (node->right == -1)
                return {};
            node = &tree[node->right];
        }
    }

    if (!node->is_leaf())
        return { bits_read, {} };
    return { bits_read, *node };
}

struct HuffmanXY {
    int x;
    int y;
};

struct HuffmanVWXY {
    int v;
    int w;
    int x;
    int y;
};

struct HuffmanEntryXY : public HuffmanEntry<HuffmanXY> {
    constexpr HuffmanEntryXY(int x, int y, size_t code_length, unsigned code)
    {
        symbol.x = x;
        symbol.y = y;
        this->code_length = code_length;
        this->code = code;
    }
};

struct HuffmanEntryVWXY : public HuffmanEntry<HuffmanVWXY> {
    constexpr HuffmanEntryVWXY(int v, int w, int x, int y, size_t code_length, unsigned code)
    {
        symbol.v = v;
        symbol.w = w;
        symbol.x = x;
        symbol.y = y;
        this->code_length = code_length;
        this->code = code;
    }
};

// All data taken from ISO/IEC 11172-3 (Table B.7)
constexpr HuffmanEntries<HuffmanEntryVWXY, 16> TableA { {
    { 0, 0, 0, 0, 1, 0b1 },
    { 0, 0, 0, 1, 4, 0b0101 },
    { 0, 0, 1, 0, 4, 0b0100 },
    { 0, 0, 1, 1, 5, 0b00101 },
    { 0, 1, 0, 0, 4, 0b0110 },
    { 0, 1, 0, 1, 6, 0b000101 },
    { 0, 1, 1, 0, 5, 0b00100 },
    { 0, 1, 1, 1, 6, 0b000100 },
    { 1, 0, 0, 0, 4, 0b0111 },
    { 1, 0, 0, 1, 5, 0b00011 },
    { 1, 0, 1, 0, 5, 0b00110 },
    { 1, 0, 1, 1, 6, 0b000000 },
    { 1, 1, 0, 0, 5, 0b00111 },
    { 1, 1, 0, 1, 6, 0b000010 },
    { 1, 1, 1, 0, 6, 0b000011 },
    { 1, 1, 1, 1, 6, 0b000001 },
} };

constexpr HuffmanEntries<HuffmanEntryVWXY, 16> TableB { {
    { 0, 0, 0, 0, 4, 0b1111 },
    { 0, 0, 0, 1, 4, 0b1110 },
    { 0, 0, 1, 0, 4, 0b1101 },
    { 0, 0, 1, 1, 4, 0b1100 },
    { 0, 1, 0, 0, 4, 0b1011 },
    { 0, 1, 0, 1, 4, 0b1010 },
    { 0, 1, 1, 0, 4, 0b1001 },
    { 0, 1, 1, 1, 4, 0b1000 },
    { 1, 0, 0, 0, 4, 0b0111 },
    { 1, 0, 0, 1, 4, 0b0110 },
    { 1, 0, 1, 0, 4, 0b0101 },
    { 1, 0, 1, 1, 4, 0b0100 },
    { 1, 1, 0, 0, 4, 0b0011 },
    { 1, 1, 0, 1, 4, 0b0010 },
    { 1, 1, 1, 0, 4, 0b0001 },
    { 1, 1, 1, 1, 4, 0b0000 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 1> Table0 { {
    { 0, 0, 0, 0b0 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 4> Table1 { {
    { 0, 0, 1, 0b1 },
    { 0, 1, 3, 0b001 },
    { 1, 0, 2, 0b01 },
    { 1, 1, 3, 0b000 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 9> Table2 { {
    { 0, 0, 1, 0b1 },
    { 0, 1, 3, 0b010 },
    { 0, 2, 6, 0b000001 },
    { 1, 0, 3, 0b011 },
    { 1, 1, 3, 0b001 },
    { 1, 2, 5, 0b00001 },
    { 2, 0, 5, 0b00011 },
    { 2, 1, 5, 0b00010 },
    { 2, 2, 6, 0b000000 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 9> Table3 { {
    { 0, 0, 2, 0b11 },
    { 0, 1, 2, 0b10 },
    { 0, 2, 6, 0b000001 },
    { 1, 0, 3, 0b001 },
    { 1, 1, 2, 0b01 },
    { 1, 2, 5, 0b00001 },
    { 2, 0, 5, 0b00011 },
    { 2, 1, 5, 0b00010 },
    { 2, 2, 6, 0b000000 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 16> Table5 { {
    { 0, 0, 1, 0b1 },
    { 0, 1, 3, 0b010 },
    { 0, 2, 6, 0b000110 },
    { 0, 3, 7, 0b0000101 },
    { 1, 0, 3, 0b011 },
    { 1, 1, 3, 0b001 },
    { 1, 2, 6, 0b000100 },
    { 1, 3, 7, 0b0000100 },
    { 2, 0, 6, 0b000111 },
    { 2, 1, 6, 0b000101 },
    { 2, 2, 7, 0b0000111 },
    { 2, 3, 8, 0b00000001 },
    { 3, 0, 7, 0b0000110 },
    { 3, 1, 6, 0b000001 },
    { 3, 2, 7, 0b0000001 },
    { 3, 3, 8, 0b00000000 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 16> Table6 { {
    { 0, 0, 3, 0b111 },
    { 0, 1, 3, 0b011 },
    { 0, 2, 5, 0b00101 },
    { 0, 3, 7, 0b0000001 },
    { 1, 0, 3, 0b110 },
    { 1, 1, 2, 0b10 },
    { 1, 2, 4, 0b0011 },
    { 1, 3, 5, 0b00010 },
    { 2, 0, 4, 0b0101 },
    { 2, 1, 4, 0b0100 },
    { 2, 2, 5, 0b00100 },
    { 2, 3, 6, 0b000001 },
    { 3, 0, 6, 0b000011 },
    { 3, 1, 5, 0b00011 },
    { 3, 2, 6, 0b000010 },
    { 3, 3, 7, 0b0000000 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 36> Table7 { {
    { 0, 0, 1, 0b1 },
    { 0, 1, 3, 0b010 },
    { 0, 2, 6, 0b001010 },
    { 0, 3, 8, 0b00010011 },
    { 0, 4, 8, 0b00010000 },
    { 0, 5, 9, 0b000001010 },
    { 1, 0, 3, 0b011 },
    { 1, 1, 4, 0b0011 },
    { 1, 2, 6, 0b000111 },
    { 1, 3, 7, 0b0001010 },
    { 1, 4, 7, 0b0000101 },
    { 1, 5, 8, 0b00000011 },
    { 2, 0, 6, 0b001011 },
    { 2, 1, 5, 0b00100 },
    { 2, 2, 7, 0b0001101 },
    { 2, 3, 8, 0b00010001 },
    { 2, 4, 8, 0b00001000 },
    { 2, 5, 9, 0b000000100 },
    { 3, 0, 7, 0b0001100 },
    { 3, 1, 7, 0b0001011 },
    { 3, 2, 8, 0b00010010 },
    { 3, 3, 9, 0b000001111 },
    { 3, 4, 9, 0b000001011 },
    { 3, 5, 9, 0b000000010 },
    { 4, 0, 7, 0b0000111 },
    { 4, 1, 7, 0b0000110 },
    { 4, 2, 8, 0b00001001 },
    { 4, 3, 9, 0b000001110 },
    { 4, 4, 9, 0b000000011 },
    { 4, 5, 10, 0b0000000001 },
    { 5, 0, 8, 0b00000110 },
    { 5, 1, 8, 0b00000100 },
    { 5, 2, 9, 0b000000101 },
    { 5, 3, 10, 0b0000000011 },
    { 5, 4, 10, 0b0000000010 },
    { 5, 5, 10, 0b0000000000 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 36> Table8 { {
    { 0, 0, 2, 0b11 },
    { 0, 1, 3, 0b100 },
    { 0, 2, 6, 0b000110 },
    { 0, 3, 8, 0b00010010 },
    { 0, 4, 8, 0b00001100 },
    { 0, 5, 9, 0b000000101 },
    { 1, 0, 3, 0b101 },
    { 1, 1, 2, 0b01 },
    { 1, 2, 4, 0b0010 },
    { 1, 3, 8, 0b00010000 },
    { 1, 4, 8, 0b00001001 },
    { 1, 5, 8, 0b00000011 },
    { 2, 0, 6, 0b000111 },
    { 2, 1, 4, 0b0011 },
    { 2, 2, 6, 0b000101 },
    { 2, 3, 8, 0b00001110 },
    { 2, 4, 8, 0b00000111 },
    { 2, 5, 9, 0b000000011 },
    { 3, 0, 8, 0b00010011 },
    { 3, 1, 8, 0b00010001 },
    { 3, 2, 8, 0b00001111 },
    { 3, 3, 9, 0b000001101 },
    { 3, 4, 9, 0b000001010 },
    { 3, 5, 10, 0b0000000100 },
    { 4, 0, 8, 0b00001101 },
    { 4, 1, 7, 0b0000101 },
    { 4, 2, 8, 0b00001000 },
    { 4, 3, 9, 0b000001011 },
    { 4, 4, 10, 0b0000000101 },
    { 4, 5, 10, 0b0000000001 },
    { 5, 0, 9, 0b000001100 },
    { 5, 1, 8, 0b00000100 },
    { 5, 2, 9, 0b000000100 },
    { 5, 3, 9, 0b000000001 },
    { 5, 4, 11, 0b00000000001 },
    { 5, 5, 11, 0b00000000000 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 36> Table9 { {
    { 0, 0, 3, 0b111 },
    { 0, 1, 3, 0b101 },
    { 0, 2, 5, 0b01001 },
    { 0, 3, 6, 0b001110 },
    { 0, 4, 8, 0b00001111 },
    { 0, 5, 9, 0b000000111 },
    { 1, 0, 3, 0b110 },
    { 1, 1, 3, 0b100 },
    { 1, 2, 4, 0b0101 },
    { 1, 3, 5, 0b00101 },
    { 1, 4, 6, 0b000110 },
    { 1, 5, 8, 0b00000111 },
    { 2, 0, 4, 0b0111 },
    { 2, 1, 4, 0b0110 },
    { 2, 2, 5, 0b01000 },
    { 2, 3, 6, 0b001000 },
    { 2, 4, 7, 0b0001000 },
    { 2, 5, 8, 0b00000101 },
    { 3, 0, 6, 0b001111 },
    { 3, 1, 5, 0b00110 },
    { 3, 2, 6, 0b001001 },
    { 3, 3, 7, 0b0001010 },
    { 3, 4, 7, 0b0000101 },
    { 3, 5, 8, 0b00000001 },
    { 4, 0, 7, 0b0001011 },
    { 4, 1, 6, 0b000111 },
    { 4, 2, 7, 0b0001001 },
    { 4, 3, 7, 0b0000110 },
    { 4, 4, 8, 0b00000100 },
    { 4, 5, 9, 0b000000001 },
    { 5, 0, 8, 0b00001110 },
    { 5, 1, 7, 0b0000100 },
    { 5, 2, 8, 0b00000110 },
    { 5, 3, 8, 0b00000010 },
    { 5, 4, 9, 0b000000110 },
    { 5, 5, 9, 0b000000000 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 64> Table10 { {
    { 0, 0, 1, 0b1 },
    { 0, 1, 3, 0b010 },
    { 0, 2, 6, 0b001010 },
    { 0, 3, 8, 0b00010111 },
    { 0, 4, 9, 0b000100011 },
    { 0, 5, 9, 0b000011110 },
    { 0, 6, 9, 0b000001100 },
    { 0, 7, 10, 0b0000010001 },
    { 1, 0, 3, 0b011 },
    { 1, 1, 4, 0b0011 },
    { 1, 2, 6, 0b001000 },
    { 1, 3, 7, 0b0001100 },
    { 1, 4, 8, 0b00010010 },
    { 1, 5, 9, 0b000010101 },
    { 1, 6, 8, 0b00001100 },
    { 1, 7, 8, 0b00000111 },
    { 2, 0, 6, 0b001011 },
    { 2, 1, 6, 0b001001 },
    { 2, 2, 7, 0b0001111 },
    { 2, 3, 8, 0b00010101 },
    { 2, 4, 9, 0b000100000 },
    { 2, 5, 10, 0b0000101000 },
    { 2, 6, 9, 0b000010011 },
    { 2, 7, 9, 0b000000110 },
    { 3, 0, 7, 0b0001110 },
    { 3, 1, 7, 0b0001101 },
    { 3, 2, 8, 0b00010110 },
    { 3, 3, 9, 0b000100010 },
    { 3, 4, 10, 0b0000101110 },
    { 3, 5, 10, 0b0000010111 },
    { 3, 6, 9, 0b000010010 },
    { 3, 7, 10, 0b0000000111 },
    { 4, 0, 8, 0b00010100 },
    { 4, 1, 8, 0b00010011 },
    { 4, 2, 9, 0b000100001 },
    { 4, 3, 10, 0b0000101111 },
    { 4, 4, 10, 0b0000011011 },
    { 4, 5, 10, 0b0000010110 },
    { 4, 6, 10, 0b0000001001 },
    { 4, 7, 10, 0b0000000011 },
    { 5, 0, 9, 0b000011111 },
    { 5, 1, 9, 0b000010110 },
    { 5, 2, 10, 0b0000101001 },
    { 5, 3, 10, 0b0000011010 },
    { 5, 4, 11, 0b00000010101 },
    { 5, 5, 11, 0b00000010100 },
    { 5, 6, 10, 0b0000000101 },
    { 5, 7, 11, 0b00000000011 },
    { 6, 0, 8, 0b00001110 },
    { 6, 1, 8, 0b00001101 },
    { 6, 2, 9, 0b000001010 },
    { 6, 3, 10, 0b0000001011 },
    { 6, 4, 10, 0b0000010000 },
    { 6, 5, 10, 0b0000000110 },
    { 6, 6, 11, 0b00000000101 },
    { 6, 7, 11, 0b00000000001 },
    { 7, 0, 9, 0b000001001 },
    { 7, 1, 8, 0b00001000 },
    { 7, 2, 9, 0b000000111 },
    { 7, 3, 10, 0b0000001000 },
    { 7, 4, 10, 0b0000000100 },
    { 7, 5, 11, 0b00000000100 },
    { 7, 6, 11, 0b00000000010 },
    { 7, 7, 11, 0b00000000000 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 64> Table11 { {
    { 0, 0, 2, 0b11 },
    { 0, 1, 3, 0b100 },
    { 0, 2, 5, 0b01010 },
    { 0, 3, 7, 0b0011000 },
    { 0, 4, 8, 0b00100010 },
    { 0, 5, 9, 0b000100001 },
    { 0, 6, 8, 0b00010101 },
    { 0, 7, 9, 0b000001111 },
    { 1, 0, 3, 0b101 },
    { 1, 1, 3, 0b011 },
    { 1, 2, 4, 0b0100 },
    { 1, 3, 6, 0b001010 },
    { 1, 4, 8, 0b00100000 },
    { 1, 5, 8, 0b00010001 },
    { 1, 6, 7, 0b0001011 },
    { 1, 7, 8, 0b00001010 },
    { 2, 0, 5, 0b01011 },
    { 2, 1, 5, 0b00111 },
    { 2, 2, 6, 0b001101 },
    { 2, 3, 7, 0b0010010 },
    { 2, 4, 8, 0b00011110 },
    { 2, 5, 9, 0b000011111 },
    { 2, 6, 8, 0b00010100 },
    { 2, 7, 8, 0b00000101 },
    { 3, 0, 7, 0b0011001 },
    { 3, 1, 6, 0b001011 },
    { 3, 2, 7, 0b0010011 },
    { 3, 3, 9, 0b000111011 },
    { 3, 4, 8, 0b00011011 },
    { 3, 5, 10, 0b0000010010 },
    { 3, 6, 8, 0b00001100 },
    { 3, 7, 9, 0b000000101 },
    { 4, 0, 8, 0b00100011 },
    { 4, 1, 8, 0b00100001 },
    { 4, 2, 8, 0b00011111 },
    { 4, 3, 9, 0b000111010 },
    { 4, 4, 9, 0b000011110 },
    { 4, 5, 10, 0b0000010000 },
    { 4, 6, 9, 0b000000111 },
    { 4, 7, 10, 0b0000000101 },
    { 5, 0, 8, 0b00011100 },
    { 5, 1, 8, 0b00011010 },
    { 5, 2, 9, 0b000100000 },
    { 5, 3, 10, 0b0000010011 },
    { 5, 4, 10, 0b0000010001 },
    { 5, 5, 11, 0b00000001111 },
    { 5, 6, 10, 0b0000001000 },
    { 5, 7, 11, 0b00000001110 },
    { 6, 0, 8, 0b00001110 },
    { 6, 1, 7, 0b0001100 },
    { 6, 2, 7, 0b0001001 },
    { 6, 3, 8, 0b00001101 },
    { 6, 4, 9, 0b000001110 },
    { 6, 5, 10, 0b0000001001 },
    { 6, 6, 10, 0b0000000100 },
    { 6, 7, 10, 0b0000000001 },
    { 7, 0, 8, 0b00001011 },
    { 7, 1, 7, 0b0000100 },
    { 7, 2, 8, 0b00000110 },
    { 7, 3, 9, 0b000000110 },
    { 7, 4, 10, 0b0000000110 },
    { 7, 5, 10, 0b0000000011 },
    { 7, 6, 10, 0b0000000010 },
    { 7, 7, 10, 0b0000000000 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 64> Table12 { {
    { 0, 0, 4, 0b1001 },
    { 0, 1, 3, 0b110 },
    { 0, 2, 5, 0b10000 },
    { 0, 3, 7, 0b0100001 },
    { 0, 4, 8, 0b00101001 },
    { 0, 5, 9, 0b000100111 },
    { 0, 6, 9, 0b000100110 },
    { 0, 7, 9, 0b000011010 },
    { 1, 0, 3, 0b111 },
    { 1, 1, 3, 0b101 },
    { 1, 2, 4, 0b0110 },
    { 1, 3, 5, 0b01001 },
    { 1, 4, 7, 0b0010111 },
    { 1, 5, 7, 0b0010000 },
    { 1, 6, 8, 0b00011010 },
    { 1, 7, 8, 0b00001011 },
    { 2, 0, 5, 0b10001 },
    { 2, 1, 4, 0b0111 },
    { 2, 2, 5, 0b01011 },
    { 2, 3, 6, 0b001110 },
    { 2, 4, 7, 0b0010101 },
    { 2, 5, 8, 0b00011110 },
    { 2, 6, 7, 0b0001010 },
    { 2, 7, 8, 0b00000111 },
    { 3, 0, 6, 0b010001 },
    { 3, 1, 5, 0b01010 },
    { 3, 2, 6, 0b001111 },
    { 3, 3, 6, 0b001100 },
    { 3, 4, 7, 0b0010010 },
    { 3, 5, 8, 0b00011100 },
    { 3, 6, 8, 0b00001110 },
    { 3, 7, 8, 0b00000101 },
    { 4, 0, 7, 0b0100000 },
    { 4, 1, 6, 0b001101 },
    { 4, 2, 7, 0b0010110 },
    { 4, 3, 7, 0b0010011 },
    { 4, 4, 8, 0b00010010 },
    { 4, 5, 8, 0b00010000 },
    { 4, 6, 8, 0b00001001 },
    { 4, 7, 9, 0b000000101 },
    { 5, 0, 8, 0b00101000 },
    { 5, 1, 7, 0b0010001 },
    { 5, 2, 8, 0b00011111 },
    { 5, 3, 8, 0b00011101 },
    { 5, 4, 8, 0b00010001 },
    { 5, 5, 9, 0b000001101 },
    { 5, 6, 8, 0b00000100 },
    { 5, 7, 9, 0b000000010 },
    { 6, 0, 8, 0b00011011 },
    { 6, 1, 7, 0b0001100 },
    { 6, 2, 7, 0b0001011 },
    { 6, 3, 8, 0b00001111 },
    { 6, 4, 8, 0b00001010 },
    { 6, 5, 9, 0b000000111 },
    { 6, 6, 9, 0b000000100 },
    { 6, 7, 10, 0b0000000001 },
    { 7, 0, 9, 0b000011011 },
    { 7, 1, 8, 0b00001100 },
    { 7, 2, 8, 0b00001000 },
    { 7, 3, 9, 0b000001100 },
    { 7, 4, 9, 0b000000110 },
    { 7, 5, 9, 0b000000011 },
    { 7, 6, 9, 0b000000001 },
    { 7, 7, 10, 0b0000000000 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 256> Table13 { {
    { 0, 0, 1, 0b1 },
    { 0, 1, 4, 0b0101 },
    { 0, 2, 6, 0b001110 },
    { 0, 3, 7, 0b0010101 },
    { 0, 4, 8, 0b00100010 },
    { 0, 5, 9, 0b000110011 },
    { 0, 6, 9, 0b000101110 },
    { 0, 7, 10, 0b0001000111 },
    { 0, 8, 9, 0b000101010 },
    { 0, 9, 10, 0b0000110100 },
    { 0, 10, 11, 0b00001000100 },
    { 0, 11, 11, 0b00000110100 },
    { 0, 12, 12, 0b000001000011 },
    { 0, 13, 12, 0b000000101100 },
    { 0, 14, 13, 0b0000000101011 },
    { 0, 15, 13, 0b0000000010011 },
    { 1, 0, 3, 0b011 },
    { 1, 1, 4, 0b0100 },
    { 1, 2, 6, 0b001100 },
    { 1, 3, 7, 0b0010011 },
    { 1, 4, 8, 0b00011111 },
    { 1, 5, 8, 0b00011010 },
    { 1, 6, 9, 0b000101100 },
    { 1, 7, 9, 0b000100001 },
    { 1, 8, 9, 0b000011111 },
    { 1, 9, 9, 0b000011000 },
    { 1, 10, 10, 0b0000100000 },
    { 1, 11, 10, 0b0000011000 },
    { 1, 12, 11, 0b00000011111 },
    { 1, 13, 12, 0b000000100011 },
    { 1, 14, 12, 0b000000010110 },
    { 1, 15, 12, 0b000000001110 },
    { 2, 0, 6, 0b001111 },
    { 2, 1, 6, 0b001101 },
    { 2, 2, 7, 0b0010111 },
    { 2, 3, 8, 0b00100100 },
    { 2, 4, 9, 0b000111011 },
    { 2, 5, 9, 0b000110001 },
    { 2, 6, 10, 0b0001001101 },
    { 2, 7, 10, 0b0001000001 },
    { 2, 8, 9, 0b000011101 },
    { 2, 9, 10, 0b0000101000 },
    { 2, 10, 10, 0b0000011110 },
    { 2, 11, 11, 0b00000101000 },
    { 2, 12, 11, 0b00000011011 },
    { 2, 13, 12, 0b000000100001 },
    { 2, 14, 13, 0b0000000101010 },
    { 2, 15, 13, 0b0000000010000 },
    { 3, 0, 7, 0b0010110 },
    { 3, 1, 7, 0b0010100 },
    { 3, 2, 8, 0b00100101 },
    { 3, 3, 9, 0b000111101 },
    { 3, 4, 9, 0b000111000 },
    { 3, 5, 10, 0b0001001111 },
    { 3, 6, 10, 0b0001001001 },
    { 3, 7, 10, 0b0001000000 },
    { 3, 8, 10, 0b0000101011 },
    { 3, 9, 11, 0b00001001100 },
    { 3, 10, 11, 0b00000111000 },
    { 3, 11, 11, 0b00000100101 },
    { 3, 12, 11, 0b00000011010 },
    { 3, 13, 12, 0b000000011111 },
    { 3, 14, 13, 0b0000000011001 },
    { 3, 15, 13, 0b0000000001110 },
    { 4, 0, 8, 0b00100011 },
    { 4, 1, 7, 0b0010000 },
    { 4, 2, 9, 0b000111100 },
    { 4, 3, 9, 0b000111001 },
    { 4, 4, 10, 0b0001100001 },
    { 4, 5, 10, 0b0001001011 },
    { 4, 6, 11, 0b00001110010 },
    { 4, 7, 11, 0b00001011011 },
    { 4, 8, 10, 0b0000110110 },
    { 4, 9, 11, 0b00001001001 },
    { 4, 10, 11, 0b00000110111 },
    { 4, 11, 12, 0b000000101001 },
    { 4, 12, 12, 0b000000110000 },
    { 4, 13, 13, 0b0000000110101 },
    { 4, 14, 13, 0b0000000010111 },
    { 4, 15, 14, 0b00000000011000 },
    { 5, 0, 9, 0b000111010 },
    { 5, 1, 8, 0b00011011 },
    { 5, 2, 9, 0b000110010 },
    { 5, 3, 10, 0b0001100000 },
    { 5, 4, 10, 0b0001001100 },
    { 5, 5, 10, 0b0001000110 },
    { 5, 6, 11, 0b00001011101 },
    { 5, 7, 11, 0b00001010100 },
    { 5, 8, 11, 0b00001001101 },
    { 5, 9, 11, 0b00000111010 },
    { 5, 10, 12, 0b000001001111 },
    { 5, 11, 11, 0b00000011101 },
    { 5, 12, 13, 0b0000001001010 },
    { 5, 13, 13, 0b0000000110001 },
    { 5, 14, 14, 0b00000000101001 },
    { 5, 15, 14, 0b00000000010001 },
    { 6, 0, 9, 0b000101111 },
    { 6, 1, 9, 0b000101101 },
    { 6, 2, 10, 0b0001001110 },
    { 6, 3, 10, 0b0001001010 },
    { 6, 4, 11, 0b00001110011 },
    { 6, 5, 11, 0b00001011110 },
    { 6, 6, 11, 0b00001011010 },
    { 6, 7, 11, 0b00001001111 },
    { 6, 8, 11, 0b00001000101 },
    { 6, 9, 12, 0b000001010011 },
    { 6, 10, 12, 0b000001000111 },
    { 6, 11, 12, 0b000000110010 },
    { 6, 12, 13, 0b0000000111011 },
    { 6, 13, 13, 0b0000000100110 },
    { 6, 14, 14, 0b00000000100100 },
    { 6, 15, 14, 0b00000000001111 },
    { 7, 0, 10, 0b0001001000 },
    { 7, 1, 9, 0b000100010 },
    { 7, 2, 10, 0b0000111000 },
    { 7, 3, 11, 0b00001011111 },
    { 7, 4, 11, 0b00001011100 },
    { 7, 5, 11, 0b00001010101 },
    { 7, 6, 12, 0b000001011011 },
    { 7, 7, 12, 0b000001011010 },
    { 7, 8, 12, 0b000001010110 },
    { 7, 9, 12, 0b000001001001 },
    { 7, 10, 13, 0b0000001001101 },
    { 7, 11, 13, 0b0000001000001 },
    { 7, 12, 13, 0b0000000110011 },
    { 7, 13, 14, 0b00000000101100 },
    { 7, 14, 16, 0b0000000000101011 },
    { 7, 15, 16, 0b0000000000101010 },
    { 8, 0, 9, 0b000101011 },
    { 8, 1, 8, 0b00010100 },
    { 8, 2, 9, 0b000011110 },
    { 8, 3, 10, 0b0000101100 },
    { 8, 4, 10, 0b0000110111 },
    { 8, 5, 11, 0b00001001110 },
    { 8, 6, 11, 0b00001001000 },
    { 8, 7, 12, 0b000001010111 },
    { 8, 8, 12, 0b000001001110 },
    { 8, 9, 12, 0b000000111101 },
    { 8, 10, 12, 0b000000101110 },
    { 8, 11, 13, 0b0000000110110 },
    { 8, 12, 13, 0b0000000100101 },
    { 8, 13, 14, 0b00000000011110 },
    { 8, 14, 15, 0b000000000010100 },
    { 8, 15, 15, 0b000000000010000 },
    { 9, 0, 10, 0b0000110101 },
    { 9, 1, 9, 0b000011001 },
    { 9, 2, 10, 0b0000101001 },
    { 9, 3, 10, 0b0000100101 },
    { 9, 4, 11, 0b00000101100 },
    { 9, 5, 11, 0b00000111011 },
    { 9, 6, 11, 0b00000110110 },
    { 9, 7, 13, 0b0000001010001 },
    { 9, 8, 12, 0b000001000010 },
    { 9, 9, 13, 0b0000001001100 },
    { 9, 10, 13, 0b0000000111001 },
    { 9, 11, 14, 0b00000000110110 },
    { 9, 12, 14, 0b00000000100101 },
    { 9, 13, 14, 0b00000000010010 },
    { 9, 14, 16, 0b0000000000100111 },
    { 9, 15, 15, 0b000000000001011 },
    { 10, 0, 10, 0b0000100011 },
    { 10, 1, 10, 0b0000100001 },
    { 10, 2, 10, 0b0000011111 },
    { 10, 3, 11, 0b00000111001 },
    { 10, 4, 11, 0b00000101010 },
    { 10, 5, 12, 0b000001010010 },
    { 10, 6, 12, 0b000001001000 },
    { 10, 7, 13, 0b0000001010000 },
    { 10, 8, 12, 0b000000101111 },
    { 10, 9, 13, 0b0000000111010 },
    { 10, 10, 14, 0b00000000110111 },
    { 10, 11, 13, 0b0000000010101 },
    { 10, 12, 14, 0b00000000010110 },
    { 10, 13, 15, 0b000000000011010 },
    { 10, 14, 16, 0b0000000000100110 },
    { 10, 15, 17, 0b00000000000010110 },
    { 11, 0, 11, 0b00000110101 },
    { 11, 1, 10, 0b0000011001 },
    { 11, 2, 10, 0b0000010111 },
    { 11, 3, 11, 0b00000100110 },
    { 11, 4, 12, 0b000001000110 },
    { 11, 5, 12, 0b000000111100 },
    { 11, 6, 12, 0b000000110011 },
    { 11, 7, 12, 0b000000100100 },
    { 11, 8, 13, 0b0000000110111 },
    { 11, 9, 13, 0b0000000011010 },
    { 11, 10, 13, 0b0000000100010 },
    { 11, 11, 14, 0b00000000010111 },
    { 11, 12, 15, 0b000000000011011 },
    { 11, 13, 15, 0b000000000001110 },
    { 11, 14, 15, 0b000000000001001 },
    { 11, 15, 16, 0b0000000000000111 },
    { 12, 0, 11, 0b00000100010 },
    { 12, 1, 11, 0b00000100000 },
    { 12, 2, 11, 0b00000011100 },
    { 12, 3, 12, 0b000000100111 },
    { 12, 4, 12, 0b000000110001 },
    { 12, 5, 13, 0b0000001001011 },
    { 12, 6, 12, 0b000000011110 },
    { 12, 7, 13, 0b0000000110100 },
    { 12, 8, 14, 0b00000000110000 },
    { 12, 9, 14, 0b00000000101000 },
    { 12, 10, 15, 0b000000000110100 },
    { 12, 11, 15, 0b000000000011100 },
    { 12, 12, 15, 0b000000000010010 },
    { 12, 13, 16, 0b0000000000010001 },
    { 12, 14, 16, 0b0000000000001001 },
    { 12, 15, 16, 0b0000000000000101 },
    { 13, 0, 12, 0b000000101101 },
    { 13, 1, 11, 0b00000010101 },
    { 13, 2, 12, 0b000000100010 },
    { 13, 3, 13, 0b0000001000000 },
    { 13, 4, 13, 0b0000000111000 },
    { 13, 5, 13, 0b0000000110010 },
    { 13, 6, 14, 0b00000000110001 },
    { 13, 7, 14, 0b00000000101101 },
    { 13, 8, 14, 0b00000000011111 },
    { 13, 9, 14, 0b00000000010011 },
    { 13, 10, 14, 0b00000000001100 },
    { 13, 11, 15, 0b000000000001111 },
    { 13, 12, 16, 0b0000000000001010 },
    { 13, 13, 15, 0b000000000000111 },
    { 13, 14, 16, 0b0000000000000110 },
    { 13, 15, 16, 0b0000000000000011 },
    { 14, 0, 13, 0b0000000110000 },
    { 14, 1, 12, 0b000000010111 },
    { 14, 2, 12, 0b000000010100 },
    { 14, 3, 13, 0b0000000100111 },
    { 14, 4, 13, 0b0000000100100 },
    { 14, 5, 13, 0b0000000100011 },
    { 14, 6, 15, 0b000000000110101 },
    { 14, 7, 14, 0b00000000010101 },
    { 14, 8, 14, 0b00000000010000 },
    { 14, 9, 17, 0b00000000000010111 },
    { 14, 10, 15, 0b000000000001101 },
    { 14, 11, 15, 0b000000000001010 },
    { 14, 12, 15, 0b000000000000110 },
    { 14, 13, 17, 0b00000000000000001 },
    { 14, 14, 16, 0b0000000000000100 },
    { 14, 15, 16, 0b0000000000000010 },
    { 15, 0, 12, 0b000000010000 },
    { 15, 1, 12, 0b000000001111 },
    { 15, 2, 13, 0b0000000010001 },
    { 15, 3, 14, 0b00000000011011 },
    { 15, 4, 14, 0b00000000011001 },
    { 15, 5, 14, 0b00000000010100 },
    { 15, 6, 15, 0b000000000011101 },
    { 15, 7, 14, 0b00000000001011 },
    { 15, 8, 15, 0b000000000010001 },
    { 15, 9, 15, 0b000000000001100 },
    { 15, 10, 16, 0b0000000000010000 },
    { 15, 11, 16, 0b0000000000001000 },
    { 15, 12, 19, 0b0000000000000000001 },
    { 15, 13, 18, 0b000000000000000001 },
    { 15, 14, 19, 0b0000000000000000000 },
    { 15, 15, 16, 0b0000000000000001 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 256> Table15 { {
    { 0, 0, 3, 0b111 },
    { 0, 1, 4, 0b1100 },
    { 0, 2, 5, 0b10010 },
    { 0, 3, 7, 0b0110101 },
    { 0, 4, 7, 0b0101111 },
    { 0, 5, 8, 0b01001100 },
    { 0, 6, 9, 0b001111100 },
    { 0, 7, 9, 0b001101100 },
    { 0, 8, 9, 0b001011001 },
    { 0, 9, 10, 0b0001111011 },
    { 0, 10, 10, 0b0001101100 },
    { 0, 11, 11, 0b00001110111 },
    { 0, 12, 11, 0b00001101011 },
    { 0, 13, 11, 0b00001010001 },
    { 0, 14, 12, 0b000001111010 },
    { 0, 15, 13, 0b0000000111111 },
    { 1, 0, 4, 0b1101 },
    { 1, 1, 3, 0b101 },
    { 1, 2, 5, 0b10000 },
    { 1, 3, 6, 0b011011 },
    { 1, 4, 7, 0b0101110 },
    { 1, 5, 7, 0b0100100 },
    { 1, 6, 8, 0b00111101 },
    { 1, 7, 8, 0b00110011 },
    { 1, 8, 8, 0b00101010 },
    { 1, 9, 9, 0b001000110 },
    { 1, 10, 9, 0b000110100 },
    { 1, 11, 10, 0b0001010011 },
    { 1, 12, 10, 0b0001000001 },
    { 1, 13, 10, 0b0000101001 },
    { 1, 14, 11, 0b00000111011 },
    { 1, 15, 11, 0b00000100100 },
    { 2, 0, 5, 0b10011 },
    { 2, 1, 5, 0b10001 },
    { 2, 2, 5, 0b01111 },
    { 2, 3, 6, 0b011000 },
    { 2, 4, 7, 0b0101001 },
    { 2, 5, 7, 0b0100010 },
    { 2, 6, 8, 0b00111011 },
    { 2, 7, 8, 0b00110000 },
    { 2, 8, 8, 0b00101000 },
    { 2, 9, 9, 0b001000000 },
    { 2, 10, 9, 0b000110010 },
    { 2, 11, 10, 0b0001001110 },
    { 2, 12, 10, 0b0000111110 },
    { 2, 13, 11, 0b00001010000 },
    { 2, 14, 11, 0b00000111000 },
    { 2, 15, 11, 0b00000100001 },
    { 3, 0, 6, 0b011101 },
    { 3, 1, 6, 0b011100 },
    { 3, 2, 6, 0b011001 },
    { 3, 3, 7, 0b0101011 },
    { 3, 4, 7, 0b0100111 },
    { 3, 5, 8, 0b00111111 },
    { 3, 6, 8, 0b00110111 },
    { 3, 7, 9, 0b001011101 },
    { 3, 8, 9, 0b001001100 },
    { 3, 9, 9, 0b000111011 },
    { 3, 10, 10, 0b0001011101 },
    { 3, 11, 10, 0b0001001000 },
    { 3, 12, 10, 0b0000110110 },
    { 3, 13, 11, 0b00001001011 },
    { 3, 14, 11, 0b00000110010 },
    { 3, 15, 11, 0b00000011101 },
    { 4, 0, 7, 0b0110100 },
    { 4, 1, 6, 0b010110 },
    { 4, 2, 7, 0b0101010 },
    { 4, 3, 7, 0b0101000 },
    { 4, 4, 8, 0b01000011 },
    { 4, 5, 8, 0b00111001 },
    { 4, 6, 9, 0b001011111 },
    { 4, 7, 9, 0b001001111 },
    { 4, 8, 9, 0b001001000 },
    { 4, 9, 9, 0b000111001 },
    { 4, 10, 10, 0b0001011001 },
    { 4, 11, 10, 0b0001000101 },
    { 4, 12, 10, 0b0000110001 },
    { 4, 13, 11, 0b00001000010 },
    { 4, 14, 11, 0b00000101110 },
    { 4, 15, 11, 0b00000011011 },
    { 5, 0, 8, 0b01001101 },
    { 5, 1, 7, 0b0100101 },
    { 5, 2, 7, 0b0100011 },
    { 5, 3, 8, 0b01000010 },
    { 5, 4, 8, 0b00111010 },
    { 5, 5, 8, 0b00110100 },
    { 5, 6, 9, 0b001011011 },
    { 5, 7, 9, 0b001001010 },
    { 5, 8, 9, 0b000111110 },
    { 5, 9, 9, 0b000110000 },
    { 5, 10, 10, 0b0001001111 },
    { 5, 11, 10, 0b0000111111 },
    { 5, 12, 11, 0b00001011010 },
    { 5, 13, 11, 0b00000111110 },
    { 5, 14, 11, 0b00000101000 },
    { 5, 15, 12, 0b000000100110 },
    { 6, 0, 9, 0b001111101 },
    { 6, 1, 7, 0b0100000 },
    { 6, 2, 8, 0b00111100 },
    { 6, 3, 8, 0b00111000 },
    { 6, 4, 8, 0b00110010 },
    { 6, 5, 9, 0b001011100 },
    { 6, 6, 9, 0b001001110 },
    { 6, 7, 9, 0b001000001 },
    { 6, 8, 9, 0b000110111 },
    { 6, 9, 10, 0b0001010111 },
    { 6, 10, 10, 0b0001000111 },
    { 6, 11, 10, 0b0000110011 },
    { 6, 12, 11, 0b00001001001 },
    { 6, 13, 11, 0b00000110011 },
    { 6, 14, 12, 0b000001000110 },
    { 6, 15, 12, 0b000000011110 },
    { 7, 0, 9, 0b001101101 },
    { 7, 1, 8, 0b00110101 },
    { 7, 2, 8, 0b00110001 },
    { 7, 3, 9, 0b001011110 },
    { 7, 4, 9, 0b001011000 },
    { 7, 5, 9, 0b001001011 },
    { 7, 6, 9, 0b001000010 },
    { 7, 7, 10, 0b0001111010 },
    { 7, 8, 10, 0b0001011011 },
    { 7, 9, 10, 0b0001001001 },
    { 7, 10, 10, 0b0000111000 },
    { 7, 11, 10, 0b0000101010 },
    { 7, 12, 11, 0b00001000000 },
    { 7, 13, 11, 0b00000101100 },
    { 7, 14, 11, 0b00000010101 },
    { 7, 15, 12, 0b000000011001 },
    { 8, 0, 9, 0b001011010 },
    { 8, 1, 8, 0b00101011 },
    { 8, 2, 8, 0b00101001 },
    { 8, 3, 9, 0b001001101 },
    { 8, 4, 9, 0b001001001 },
    { 8, 5, 9, 0b000111111 },
    { 8, 6, 9, 0b000111000 },
    { 8, 7, 10, 0b0001011100 },
    { 8, 8, 10, 0b0001001101 },
    { 8, 9, 10, 0b0001000010 },
    { 8, 10, 10, 0b0000101111 },
    { 8, 11, 11, 0b00001000011 },
    { 8, 12, 11, 0b00000110000 },
    { 8, 13, 12, 0b000000110101 },
    { 8, 14, 12, 0b000000100100 },
    { 8, 15, 12, 0b000000010100 },
    { 9, 0, 9, 0b001000111 },
    { 9, 1, 8, 0b00100010 },
    { 9, 2, 9, 0b001000011 },
    { 9, 3, 9, 0b000111100 },
    { 9, 4, 9, 0b000111010 },
    { 9, 5, 9, 0b000110001 },
    { 9, 6, 10, 0b0001011000 },
    { 9, 7, 10, 0b0001001100 },
    { 9, 8, 10, 0b0001000011 },
    { 9, 9, 11, 0b00001101010 },
    { 9, 10, 11, 0b00001000111 },
    { 9, 11, 11, 0b00000110110 },
    { 9, 12, 11, 0b00000100110 },
    { 9, 13, 12, 0b000000100111 },
    { 9, 14, 12, 0b000000010111 },
    { 9, 15, 12, 0b000000001111 },
    { 10, 0, 10, 0b0001101101 },
    { 10, 1, 9, 0b000110101 },
    { 10, 2, 9, 0b000110011 },
    { 10, 3, 9, 0b000101111 },
    { 10, 4, 10, 0b0001011010 },
    { 10, 5, 10, 0b0001010010 },
    { 10, 6, 10, 0b0000111010 },
    { 10, 7, 10, 0b0000111001 },
    { 10, 8, 10, 0b0000110000 },
    { 10, 9, 11, 0b00001001000 },
    { 10, 10, 11, 0b00000111001 },
    { 10, 11, 11, 0b00000101001 },
    { 10, 12, 11, 0b00000010111 },
    { 10, 13, 12, 0b000000011011 },
    { 10, 14, 13, 0b0000000111110 },
    { 10, 15, 12, 0b000000001001 },
    { 11, 0, 10, 0b0001010110 },
    { 11, 1, 9, 0b000101010 },
    { 11, 2, 9, 0b000101000 },
    { 11, 3, 9, 0b000100101 },
    { 11, 4, 10, 0b0001000110 },
    { 11, 5, 10, 0b0001000000 },
    { 11, 6, 10, 0b0000110100 },
    { 11, 7, 10, 0b0000101011 },
    { 11, 8, 11, 0b00001000110 },
    { 11, 9, 11, 0b00000110111 },
    { 11, 10, 11, 0b00000101010 },
    { 11, 11, 11, 0b00000011001 },
    { 11, 12, 12, 0b000000011101 },
    { 11, 13, 12, 0b000000010010 },
    { 11, 14, 12, 0b000000001011 },
    { 11, 15, 13, 0b0000000001011 },
    { 12, 0, 11, 0b00001110110 },
    { 12, 1, 10, 0b0001000100 },
    { 12, 2, 9, 0b000011110 },
    { 12, 3, 10, 0b0000110111 },
    { 12, 4, 10, 0b0000110010 },
    { 12, 5, 10, 0b0000101110 },
    { 12, 6, 11, 0b00001001010 },
    { 12, 7, 11, 0b00001000001 },
    { 12, 8, 11, 0b00000110001 },
    { 12, 9, 11, 0b00000100111 },
    { 12, 10, 11, 0b00000011000 },
    { 12, 11, 11, 0b00000010000 },
    { 12, 12, 12, 0b000000010110 },
    { 12, 13, 12, 0b000000001101 },
    { 12, 14, 13, 0b0000000001110 },
    { 12, 15, 13, 0b0000000000111 },
    { 13, 0, 11, 0b00001011011 },
    { 13, 1, 10, 0b0000101100 },
    { 13, 2, 10, 0b0000100111 },
    { 13, 3, 10, 0b0000100110 },
    { 13, 4, 10, 0b0000100010 },
    { 13, 5, 11, 0b00000111111 },
    { 13, 6, 11, 0b00000110100 },
    { 13, 7, 11, 0b00000101101 },
    { 13, 8, 11, 0b00000011111 },
    { 13, 9, 12, 0b000000110100 },
    { 13, 10, 12, 0b000000011100 },
    { 13, 11, 12, 0b000000010011 },
    { 13, 12, 12, 0b000000001110 },
    { 13, 13, 12, 0b000000001000 },
    { 13, 14, 13, 0b0000000001001 },
    { 13, 15, 13, 0b0000000000011 },
    { 14, 0, 12, 0b000001111011 },
    { 14, 1, 11, 0b00000111100 },
    { 14, 2, 11, 0b00000111010 },
    { 14, 3, 11, 0b00000110101 },
    { 14, 4, 11, 0b00000101111 },
    { 14, 5, 11, 0b00000101011 },
    { 14, 6, 11, 0b00000100000 },
    { 14, 7, 11, 0b00000010110 },
    { 14, 8, 12, 0b000000100101 },
    { 14, 9, 12, 0b000000011000 },
    { 14, 10, 12, 0b000000010001 },
    { 14, 11, 12, 0b000000001100 },
    { 14, 12, 13, 0b0000000001111 },
    { 14, 13, 13, 0b0000000001010 },
    { 14, 14, 12, 0b000000000010 },
    { 14, 15, 13, 0b0000000000001 },
    { 15, 0, 12, 0b000001000111 },
    { 15, 1, 11, 0b00000100101 },
    { 15, 2, 11, 0b00000100010 },
    { 15, 3, 11, 0b00000011110 },
    { 15, 4, 11, 0b00000011100 },
    { 15, 5, 11, 0b00000010100 },
    { 15, 6, 11, 0b00000010001 },
    { 15, 7, 12, 0b000000011010 },
    { 15, 8, 12, 0b000000010101 },
    { 15, 9, 12, 0b000000010000 },
    { 15, 10, 12, 0b000000001010 },
    { 15, 11, 12, 0b000000000110 },
    { 15, 12, 13, 0b0000000001000 },
    { 15, 13, 13, 0b0000000000110 },
    { 15, 14, 13, 0b0000000000010 },
    { 15, 15, 13, 0b0000000000000 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 256> Table16 { {
    { 0, 0, 1, 0b1 },
    { 0, 1, 4, 0b0101 },
    { 0, 2, 6, 0b001110 },
    { 0, 3, 8, 0b00101100 },
    { 0, 4, 9, 0b001001010 },
    { 0, 5, 9, 0b000111111 },
    { 0, 6, 10, 0b0001101110 },
    { 0, 7, 10, 0b0001011101 },
    { 0, 8, 11, 0b00010101100 },
    { 0, 9, 11, 0b00010010101 },
    { 0, 10, 11, 0b00010001010 },
    { 0, 11, 12, 0b000011110010 },
    { 0, 12, 12, 0b000011100001 },
    { 0, 13, 12, 0b000011000011 },
    { 0, 14, 13, 0b0000101111000 },
    { 0, 15, 9, 0b000010001 },
    { 1, 0, 3, 0b011 },
    { 1, 1, 4, 0b0100 },
    { 1, 2, 6, 0b001100 },
    { 1, 3, 7, 0b0010100 },
    { 1, 4, 8, 0b00100011 },
    { 1, 5, 9, 0b000111110 },
    { 1, 6, 9, 0b000110101 },
    { 1, 7, 9, 0b000101111 },
    { 1, 8, 10, 0b0001010011 },
    { 1, 9, 10, 0b0001001011 },
    { 1, 10, 10, 0b0001000100 },
    { 1, 11, 11, 0b00001110111 },
    { 1, 12, 12, 0b000011001001 },
    { 1, 13, 11, 0b00001101011 },
    { 1, 14, 12, 0b000011001111 },
    { 1, 15, 8, 0b00001001 },
    { 2, 0, 6, 0b001111 },
    { 2, 1, 6, 0b001101 },
    { 2, 2, 7, 0b0010111 },
    { 2, 3, 8, 0b00100110 },
    { 2, 4, 9, 0b001000011 },
    { 2, 5, 9, 0b000111010 },
    { 2, 6, 10, 0b0001100111 },
    { 2, 7, 10, 0b0001011010 },
    { 2, 8, 11, 0b00010100001 },
    { 2, 9, 10, 0b0001001000 },
    { 2, 10, 11, 0b00001111111 },
    { 2, 11, 11, 0b00001110101 },
    { 2, 12, 11, 0b00001101110 },
    { 2, 13, 12, 0b000011010001 },
    { 2, 14, 12, 0b000011001110 },
    { 2, 15, 9, 0b000010000 },
    { 3, 0, 8, 0b00101101 },
    { 3, 1, 7, 0b0010101 },
    { 3, 2, 8, 0b00100111 },
    { 3, 3, 9, 0b001000101 },
    { 3, 4, 9, 0b001000000 },
    { 3, 5, 10, 0b0001110010 },
    { 3, 6, 10, 0b0001100011 },
    { 3, 7, 10, 0b0001010111 },
    { 3, 8, 11, 0b00010011110 },
    { 3, 9, 11, 0b00010001100 },
    { 3, 10, 12, 0b000011111100 },
    { 3, 11, 12, 0b000011010100 },
    { 3, 12, 12, 0b000011000111 },
    { 3, 13, 13, 0b0000110000011 },
    { 3, 14, 13, 0b0000101101101 },
    { 3, 15, 10, 0b0000011010 },
    { 4, 0, 9, 0b001001011 },
    { 4, 1, 8, 0b00100100 },
    { 4, 2, 9, 0b001000100 },
    { 4, 3, 9, 0b001000001 },
    { 4, 4, 10, 0b0001110011 },
    { 4, 5, 10, 0b0001100101 },
    { 4, 6, 11, 0b00010110011 },
    { 4, 7, 11, 0b00010100100 },
    { 4, 8, 11, 0b00010011011 },
    { 4, 9, 12, 0b000100001000 },
    { 4, 10, 12, 0b000011110110 },
    { 4, 11, 12, 0b000011100010 },
    { 4, 12, 13, 0b0000110001011 },
    { 4, 13, 13, 0b0000101111110 },
    { 4, 14, 13, 0b0000101101010 },
    { 4, 15, 9, 0b000001001 },
    { 5, 0, 9, 0b001000010 },
    { 5, 1, 8, 0b00011110 },
    { 5, 2, 9, 0b000111011 },
    { 5, 3, 9, 0b000111000 },
    { 5, 4, 10, 0b0001100110 },
    { 5, 5, 11, 0b00010111001 },
    { 5, 6, 11, 0b00010101101 },
    { 5, 7, 12, 0b000100001001 },
    { 5, 8, 11, 0b00010001110 },
    { 5, 9, 12, 0b000011111101 },
    { 5, 10, 12, 0b000011101000 },
    { 5, 11, 13, 0b0000110010000 },
    { 5, 12, 13, 0b0000110000100 },
    { 5, 13, 13, 0b0000101111010 },
    { 5, 14, 14, 0b00000110111101 },
    { 5, 15, 10, 0b0000010000 },
    { 6, 0, 10, 0b0001101111 },
    { 6, 1, 9, 0b000110110 },
    { 6, 2, 9, 0b000110100 },
    { 6, 3, 10, 0b0001100100 },
    { 6, 4, 11, 0b00010111000 },
    { 6, 5, 11, 0b00010110010 },
    { 6, 6, 11, 0b00010100000 },
    { 6, 7, 11, 0b00010000101 },
    { 6, 8, 12, 0b000100000001 },
    { 6, 9, 12, 0b000011110100 },
    { 6, 10, 12, 0b000011100100 },
    { 6, 11, 12, 0b000011011001 },
    { 6, 12, 13, 0b0000110000001 },
    { 6, 13, 13, 0b0000101101110 },
    { 6, 14, 14, 0b00001011001011 },
    { 6, 15, 10, 0b0000001010 },
    { 7, 0, 10, 0b0001100010 },
    { 7, 1, 9, 0b000110000 },
    { 7, 2, 10, 0b0001011011 },
    { 7, 3, 10, 0b0001011000 },
    { 7, 4, 11, 0b00010100101 },
    { 7, 5, 11, 0b00010011101 },
    { 7, 6, 11, 0b00010010100 },
    { 7, 7, 12, 0b000100000101 },
    { 7, 8, 12, 0b000011111000 },
    { 7, 9, 13, 0b0000110010111 },
    { 7, 10, 13, 0b0000110001101 },
    { 7, 11, 13, 0b0000101110100 },
    { 7, 12, 13, 0b0000101111100 },
    { 7, 13, 15, 0b000001101111001 },
    { 7, 14, 15, 0b000001101110100 },
    { 7, 15, 10, 0b0000001000 },
    { 8, 0, 10, 0b0001010101 },
    { 8, 1, 10, 0b0001010100 },
    { 8, 2, 10, 0b0001010001 },
    { 8, 3, 11, 0b00010011111 },
    { 8, 4, 11, 0b00010011100 },
    { 8, 5, 11, 0b00010001111 },
    { 8, 6, 12, 0b000100000100 },
    { 8, 7, 12, 0b000011111001 },
    { 8, 8, 13, 0b0000110101011 },
    { 8, 9, 13, 0b0000110010001 },
    { 8, 10, 13, 0b0000110001000 },
    { 8, 11, 13, 0b0000101111111 },
    { 8, 12, 14, 0b00001011010111 },
    { 8, 13, 14, 0b00001011001001 },
    { 8, 14, 14, 0b00001011000100 },
    { 8, 15, 10, 0b0000000111 },
    { 9, 0, 11, 0b00010011010 },
    { 9, 1, 10, 0b0001001100 },
    { 9, 2, 10, 0b0001001001 },
    { 9, 3, 11, 0b00010001101 },
    { 9, 4, 11, 0b00010000011 },
    { 9, 5, 12, 0b000100000000 },
    { 9, 6, 12, 0b000011110101 },
    { 9, 7, 13, 0b0000110101010 },
    { 9, 8, 13, 0b0000110010110 },
    { 9, 9, 13, 0b0000110001010 },
    { 9, 10, 13, 0b0000110000000 },
    { 9, 11, 14, 0b00001011011111 },
    { 9, 12, 13, 0b0000101100111 },
    { 9, 13, 14, 0b00001011000110 },
    { 9, 14, 13, 0b0000101100000 },
    { 9, 15, 11, 0b00000001011 },
    { 10, 0, 11, 0b00010001011 },
    { 10, 1, 11, 0b00010000001 },
    { 10, 2, 10, 0b0001000011 },
    { 10, 3, 11, 0b00001111101 },
    { 10, 4, 12, 0b000011110111 },
    { 10, 5, 12, 0b000011101001 },
    { 10, 6, 12, 0b000011100101 },
    { 10, 7, 12, 0b000011011011 },
    { 10, 8, 13, 0b0000110001001 },
    { 10, 9, 14, 0b00001011100111 },
    { 10, 10, 14, 0b00001011100001 },
    { 10, 11, 14, 0b00001011010000 },
    { 10, 12, 15, 0b000001101110101 },
    { 10, 13, 15, 0b000001101110010 },
    { 10, 14, 14, 0b00000110110111 },
    { 10, 15, 10, 0b0000000100 },
    { 11, 0, 12, 0b000011110011 },
    { 11, 1, 11, 0b00001111000 },
    { 11, 2, 11, 0b00001110110 },
    { 11, 3, 11, 0b00001110011 },
    { 11, 4, 12, 0b000011100011 },
    { 11, 5, 12, 0b000011011111 },
    { 11, 6, 13, 0b0000110001100 },
    { 11, 7, 14, 0b00001011101010 },
    { 11, 8, 14, 0b00001011100110 },
    { 11, 9, 14, 0b00001011100000 },
    { 11, 10, 14, 0b00001011010001 },
    { 11, 11, 14, 0b00001011001000 },
    { 11, 12, 14, 0b00001011000010 },
    { 11, 13, 13, 0b0000011011111 },
    { 11, 14, 14, 0b00000110110100 },
    { 11, 15, 11, 0b00000000110 },
    { 12, 0, 12, 0b000011001010 },
    { 12, 1, 12, 0b000011100000 },
    { 12, 2, 12, 0b000011011110 },
    { 12, 3, 12, 0b000011011010 },
    { 12, 4, 12, 0b000011011000 },
    { 12, 5, 13, 0b0000110000101 },
    { 12, 6, 13, 0b0000110000010 },
    { 12, 7, 13, 0b0000101111101 },
    { 12, 8, 13, 0b0000101101100 },
    { 12, 9, 15, 0b000001101111000 },
    { 12, 10, 14, 0b00000110111011 },
    { 12, 11, 14, 0b00001011000011 },
    { 12, 12, 14, 0b00000110111000 },
    { 12, 13, 14, 0b00000110110101 },
    { 12, 14, 16, 0b0000011011000000 },
    { 12, 15, 11, 0b00000000100 },
    { 13, 0, 14, 0b00001011101011 },
    { 13, 1, 12, 0b000011010011 },
    { 13, 2, 12, 0b000011010010 },
    { 13, 3, 12, 0b000011010000 },
    { 13, 4, 13, 0b0000101110010 },
    { 13, 5, 13, 0b0000101111011 },
    { 13, 6, 14, 0b00001011011110 },
    { 13, 7, 14, 0b00001011010011 },
    { 13, 8, 14, 0b00001011001010 },
    { 13, 9, 16, 0b0000011011000111 },
    { 13, 10, 15, 0b000001101110011 },
    { 13, 11, 15, 0b000001101101101 },
    { 13, 12, 15, 0b000001101101100 },
    { 13, 13, 17, 0b00000110110000011 },
    { 13, 14, 15, 0b000001101100001 },
    { 13, 15, 11, 0b00000000010 },
    { 14, 0, 13, 0b0000101111001 },
    { 14, 1, 13, 0b0000101110001 },
    { 14, 2, 11, 0b00001100110 },
    { 14, 3, 12, 0b000010111011 },
    { 14, 4, 14, 0b00001011010110 },
    { 14, 5, 14, 0b00001011010010 },
    { 14, 6, 13, 0b0000101100110 },
    { 14, 7, 14, 0b00001011000111 },
    { 14, 8, 14, 0b00001011000101 },
    { 14, 9, 15, 0b000001101100010 },
    { 14, 10, 16, 0b0000011011000110 },
    { 14, 11, 15, 0b000001101100111 },
    { 14, 12, 17, 0b00000110110000010 },
    { 14, 13, 15, 0b000001101100110 },
    { 14, 14, 14, 0b00000110110010 },
    { 14, 15, 11, 0b00000000000 },
    { 15, 0, 9, 0b000001100 },
    { 15, 1, 8, 0b00001010 },
    { 15, 2, 8, 0b00000111 },
    { 15, 3, 9, 0b000001011 },
    { 15, 4, 9, 0b000001010 },
    { 15, 5, 10, 0b0000010001 },
    { 15, 6, 10, 0b0000001011 },
    { 15, 7, 10, 0b0000001001 },
    { 15, 8, 11, 0b00000001101 },
    { 15, 9, 11, 0b00000001100 },
    { 15, 10, 11, 0b00000001010 },
    { 15, 11, 11, 0b00000000111 },
    { 15, 12, 11, 0b00000000101 },
    { 15, 13, 11, 0b00000000011 },
    { 15, 14, 11, 0b00000000001 },
    { 15, 15, 8, 0b00000011 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 256> Table17 { Table16 };
constexpr HuffmanEntries<HuffmanEntryXY, 256> Table18 { Table16 };
constexpr HuffmanEntries<HuffmanEntryXY, 256> Table19 { Table16 };
constexpr HuffmanEntries<HuffmanEntryXY, 256> Table20 { Table16 };
constexpr HuffmanEntries<HuffmanEntryXY, 256> Table21 { Table16 };
constexpr HuffmanEntries<HuffmanEntryXY, 256> Table22 { Table16 };
constexpr HuffmanEntries<HuffmanEntryXY, 256> Table23 { Table16 };

constexpr HuffmanEntries<HuffmanEntryXY, 256> Table24 { {
    { 0, 0, 4, 0b1111 },
    { 0, 1, 4, 0b1101 },
    { 0, 2, 6, 0b101110 },
    { 0, 3, 7, 0b1010000 },
    { 0, 4, 8, 0b10010010 },
    { 0, 5, 9, 0b100000110 },
    { 0, 6, 9, 0b011111000 },
    { 0, 7, 10, 0b0110110010 },
    { 0, 8, 10, 0b0110101010 },
    { 0, 9, 11, 0b01010011101 },
    { 0, 10, 11, 0b01010001101 },
    { 0, 11, 11, 0b01010001001 },
    { 0, 12, 11, 0b01001101101 },
    { 0, 13, 11, 0b01000000101 },
    { 0, 14, 12, 0b010000001000 },
    { 0, 15, 9, 0b001011000 },
    { 1, 0, 4, 0b1110 },
    { 1, 1, 4, 0b1100 },
    { 1, 2, 5, 0b10101 },
    { 1, 3, 6, 0b100110 },
    { 1, 4, 7, 0b1000111 },
    { 1, 5, 8, 0b10000010 },
    { 1, 6, 8, 0b01111010 },
    { 1, 7, 9, 0b011011000 },
    { 1, 8, 9, 0b011010001 },
    { 1, 9, 9, 0b011000110 },
    { 1, 10, 10, 0b0101000111 },
    { 1, 11, 10, 0b0101011001 },
    { 1, 12, 10, 0b0100111111 },
    { 1, 13, 10, 0b0100101001 },
    { 1, 14, 10, 0b0100010111 },
    { 1, 15, 8, 0b00101010 },
    { 2, 0, 6, 0b101111 },
    { 2, 1, 5, 0b10110 },
    { 2, 2, 6, 0b101001 },
    { 2, 3, 7, 0b1001010 },
    { 2, 4, 7, 0b1000100 },
    { 2, 5, 8, 0b10000000 },
    { 2, 6, 8, 0b01111000 },
    { 2, 7, 9, 0b011011101 },
    { 2, 8, 9, 0b011001111 },
    { 2, 9, 9, 0b011000010 },
    { 2, 10, 9, 0b010110110 },
    { 2, 11, 10, 0b0101010100 },
    { 2, 12, 10, 0b0100111011 },
    { 2, 13, 10, 0b0100100111 },
    { 2, 14, 11, 0b01000011101 },
    { 2, 15, 7, 0b0010010 },
    { 3, 0, 7, 0b1010001 },
    { 3, 1, 6, 0b100111 },
    { 3, 2, 7, 0b1001011 },
    { 3, 3, 7, 0b1000110 },
    { 3, 4, 8, 0b10000110 },
    { 3, 5, 8, 0b01111101 },
    { 3, 6, 8, 0b01110100 },
    { 3, 7, 9, 0b011011100 },
    { 3, 8, 9, 0b011001100 },
    { 3, 9, 9, 0b010111110 },
    { 3, 10, 9, 0b010110010 },
    { 3, 11, 10, 0b0101000101 },
    { 3, 12, 10, 0b0100110111 },
    { 3, 13, 10, 0b0100100101 },
    { 3, 14, 10, 0b0100001111 },
    { 3, 15, 7, 0b0010000 },
    { 4, 0, 8, 0b10010011 },
    { 4, 1, 7, 0b1001000 },
    { 4, 2, 7, 0b1000101 },
    { 4, 3, 8, 0b10000111 },
    { 4, 4, 8, 0b01111111 },
    { 4, 5, 8, 0b01110110 },
    { 4, 6, 8, 0b01110000 },
    { 4, 7, 9, 0b011010010 },
    { 4, 8, 9, 0b011001000 },
    { 4, 9, 9, 0b010111100 },
    { 4, 10, 10, 0b0101100000 },
    { 4, 11, 10, 0b0101000011 },
    { 4, 12, 10, 0b0100110010 },
    { 4, 13, 10, 0b0100011101 },
    { 4, 14, 11, 0b01000011100 },
    { 4, 15, 7, 0b0001110 },
    { 5, 0, 9, 0b100000111 },
    { 5, 1, 7, 0b1000010 },
    { 5, 2, 8, 0b10000001 },
    { 5, 3, 8, 0b01111110 },
    { 5, 4, 8, 0b01110111 },
    { 5, 5, 8, 0b01110010 },
    { 5, 6, 9, 0b011010110 },
    { 5, 7, 9, 0b011001010 },
    { 5, 8, 9, 0b011000000 },
    { 5, 9, 9, 0b010110100 },
    { 5, 10, 10, 0b0101010101 },
    { 5, 11, 10, 0b0100111101 },
    { 5, 12, 10, 0b0100101101 },
    { 5, 13, 10, 0b0100011001 },
    { 5, 14, 10, 0b0100000110 },
    { 5, 15, 7, 0b0001100 },
    { 6, 0, 9, 0b011111001 },
    { 6, 1, 8, 0b01111011 },
    { 6, 2, 8, 0b01111001 },
    { 6, 3, 8, 0b01110101 },
    { 6, 4, 8, 0b01110001 },
    { 6, 5, 9, 0b011010111 },
    { 6, 6, 9, 0b011001110 },
    { 6, 7, 9, 0b011000011 },
    { 6, 8, 9, 0b010111001 },
    { 6, 9, 10, 0b0101011011 },
    { 6, 10, 10, 0b0101001010 },
    { 6, 11, 10, 0b0100110100 },
    { 6, 12, 10, 0b0100100011 },
    { 6, 13, 10, 0b0100010000 },
    { 6, 14, 11, 0b01000001000 },
    { 6, 15, 7, 0b0001010 },
    { 7, 0, 10, 0b0110110011 },
    { 7, 1, 8, 0b01110011 },
    { 7, 2, 8, 0b01101111 },
    { 7, 3, 8, 0b01101101 },
    { 7, 4, 9, 0b011010011 },
    { 7, 5, 9, 0b011001011 },
    { 7, 6, 9, 0b011000100 },
    { 7, 7, 9, 0b010111011 },
    { 7, 8, 10, 0b0101100001 },
    { 7, 9, 10, 0b0101001100 },
    { 7, 10, 10, 0b0100111001 },
    { 7, 11, 10, 0b0100101010 },
    { 7, 12, 10, 0b0100011011 },
    { 7, 13, 11, 0b01000010011 },
    { 7, 14, 11, 0b00101111101 },
    { 7, 15, 8, 0b00010001 },
    { 8, 0, 10, 0b0110101011 },
    { 8, 1, 9, 0b011010100 },
    { 8, 2, 9, 0b011010000 },
    { 8, 3, 9, 0b011001101 },
    { 8, 4, 9, 0b011001001 },
    { 8, 5, 9, 0b011000001 },
    { 8, 6, 9, 0b010111010 },
    { 8, 7, 9, 0b010110001 },
    { 8, 8, 9, 0b010101001 },
    { 8, 9, 10, 0b0101000000 },
    { 8, 10, 10, 0b0100101111 },
    { 8, 11, 10, 0b0100011110 },
    { 8, 12, 10, 0b0100001100 },
    { 8, 13, 11, 0b01000000010 },
    { 8, 14, 11, 0b00101111001 },
    { 8, 15, 8, 0b00010000 },
    { 9, 0, 10, 0b0101001111 },
    { 9, 1, 9, 0b011000111 },
    { 9, 2, 9, 0b011000101 },
    { 9, 3, 9, 0b010111111 },
    { 9, 4, 9, 0b010111101 },
    { 9, 5, 9, 0b010110101 },
    { 9, 6, 9, 0b010101110 },
    { 9, 7, 10, 0b0101001101 },
    { 9, 8, 10, 0b0101000001 },
    { 9, 9, 10, 0b0100110001 },
    { 9, 10, 10, 0b0100100001 },
    { 9, 11, 10, 0b0100010011 },
    { 9, 12, 11, 0b01000001001 },
    { 9, 13, 11, 0b00101111011 },
    { 9, 14, 11, 0b00101110011 },
    { 9, 15, 8, 0b00001011 },
    { 10, 0, 11, 0b01010011100 },
    { 10, 1, 9, 0b010111000 },
    { 10, 2, 9, 0b010110111 },
    { 10, 3, 9, 0b010110011 },
    { 10, 4, 9, 0b010101111 },
    { 10, 5, 10, 0b0101011000 },
    { 10, 6, 10, 0b0101001011 },
    { 10, 7, 10, 0b0100111010 },
    { 10, 8, 10, 0b0100110000 },
    { 10, 9, 10, 0b0100100010 },
    { 10, 10, 10, 0b0100010101 },
    { 10, 11, 11, 0b01000010010 },
    { 10, 12, 11, 0b00101111111 },
    { 10, 13, 11, 0b00101110101 },
    { 10, 14, 11, 0b00101101110 },
    { 10, 15, 8, 0b00001010 },
    { 11, 0, 11, 0b01010001100 },
    { 11, 1, 10, 0b0101011010 },
    { 11, 2, 9, 0b010101011 },
    { 11, 3, 9, 0b010101000 },
    { 11, 4, 9, 0b010100100 },
    { 11, 5, 10, 0b0100111110 },
    { 11, 6, 10, 0b0100110101 },
    { 11, 7, 10, 0b0100101011 },
    { 11, 8, 10, 0b0100011111 },
    { 11, 9, 10, 0b0100010100 },
    { 11, 10, 10, 0b0100000111 },
    { 11, 11, 11, 0b01000000001 },
    { 11, 12, 11, 0b00101110111 },
    { 11, 13, 11, 0b00101110000 },
    { 11, 14, 11, 0b00101101010 },
    { 11, 15, 8, 0b00000110 },
    { 12, 0, 11, 0b01010001000 },
    { 12, 1, 10, 0b0101000010 },
    { 12, 2, 10, 0b0100111100 },
    { 12, 3, 10, 0b0100111000 },
    { 12, 4, 10, 0b0100110011 },
    { 12, 5, 10, 0b0100101110 },
    { 12, 6, 10, 0b0100100100 },
    { 12, 7, 10, 0b0100011100 },
    { 12, 8, 10, 0b0100001101 },
    { 12, 9, 10, 0b0100000101 },
    { 12, 10, 11, 0b01000000000 },
    { 12, 11, 11, 0b00101111000 },
    { 12, 12, 11, 0b00101110010 },
    { 12, 13, 11, 0b00101101100 },
    { 12, 14, 11, 0b00101100111 },
    { 12, 15, 8, 0b00000100 },
    { 13, 0, 11, 0b01001101100 },
    { 13, 1, 10, 0b0100101100 },
    { 13, 2, 10, 0b0100101000 },
    { 13, 3, 10, 0b0100100110 },
    { 13, 4, 10, 0b0100100000 },
    { 13, 5, 10, 0b0100011010 },
    { 13, 6, 10, 0b0100010001 },
    { 13, 7, 10, 0b0100001010 },
    { 13, 8, 11, 0b01000000011 },
    { 13, 9, 11, 0b00101111100 },
    { 13, 10, 11, 0b00101110110 },
    { 13, 11, 11, 0b00101110001 },
    { 13, 12, 11, 0b00101101101 },
    { 13, 13, 11, 0b00101101001 },
    { 13, 14, 11, 0b00101100101 },
    { 13, 15, 8, 0b00000010 },
    { 14, 0, 12, 0b010000001001 },
    { 14, 1, 10, 0b0100011000 },
    { 14, 2, 10, 0b0100010110 },
    { 14, 3, 10, 0b0100010010 },
    { 14, 4, 10, 0b0100001011 },
    { 14, 5, 10, 0b0100001000 },
    { 14, 6, 10, 0b0100000011 },
    { 14, 7, 11, 0b00101111110 },
    { 14, 8, 11, 0b00101111010 },
    { 14, 9, 11, 0b00101110100 },
    { 14, 10, 11, 0b00101101111 },
    { 14, 11, 11, 0b00101101011 },
    { 14, 12, 11, 0b00101101000 },
    { 14, 13, 11, 0b00101100110 },
    { 14, 14, 11, 0b00101100100 },
    { 14, 15, 8, 0b00000000 },
    { 15, 0, 8, 0b00101011 },
    { 15, 1, 7, 0b0010100 },
    { 15, 2, 7, 0b0010011 },
    { 15, 3, 7, 0b0010001 },
    { 15, 4, 7, 0b0001111 },
    { 15, 5, 7, 0b0001101 },
    { 15, 6, 7, 0b0001011 },
    { 15, 7, 7, 0b0001001 },
    { 15, 8, 7, 0b0000111 },
    { 15, 9, 7, 0b0000110 },
    { 15, 10, 7, 0b0000100 },
    { 15, 11, 8, 0b00000111 },
    { 15, 12, 8, 0b00000101 },
    { 15, 13, 8, 0b00000011 },
    { 15, 14, 8, 0b00000001 },
    { 15, 15, 4, 0b0011 },
} };

constexpr HuffmanEntries<HuffmanEntryXY, 256> Table25 { Table24 };
constexpr HuffmanEntries<HuffmanEntryXY, 256> Table26 { Table24 };
constexpr HuffmanEntries<HuffmanEntryXY, 256> Table27 { Table24 };
constexpr HuffmanEntries<HuffmanEntryXY, 256> Table28 { Table24 };
constexpr HuffmanEntries<HuffmanEntryXY, 256> Table29 { Table24 };
constexpr HuffmanEntries<HuffmanEntryXY, 256> Table30 { Table24 };
constexpr HuffmanEntries<HuffmanEntryXY, 256> Table31 { Table24 };

constexpr auto TreeA = make_huffman_tree<TableA>();
constexpr auto TreeB = make_huffman_tree<TableB>();

constexpr auto Tree0 = make_huffman_tree<Table0>();
constexpr auto Tree1 = make_huffman_tree<Table1>();
constexpr auto Tree2 = make_huffman_tree<Table2>();
constexpr auto Tree3 = make_huffman_tree<Table3>();
constexpr auto Tree5 = make_huffman_tree<Table5>();
constexpr auto Tree6 = make_huffman_tree<Table6>();
constexpr auto Tree7 = make_huffman_tree<Table7>();
constexpr auto Tree8 = make_huffman_tree<Table8>();
constexpr auto Tree9 = make_huffman_tree<Table9>();
constexpr auto Tree10 = make_huffman_tree<Table10>();
constexpr auto Tree11 = make_huffman_tree<Table11>();
constexpr auto Tree12 = make_huffman_tree<Table12>();
constexpr auto Tree13 = make_huffman_tree<Table13>();
constexpr auto Tree15 = make_huffman_tree<Table15>();
constexpr auto Tree16 = make_huffman_tree<Table16>();
constexpr auto Tree17 = make_huffman_tree<Table17>();
constexpr auto Tree18 = make_huffman_tree<Table18>();
constexpr auto Tree19 = make_huffman_tree<Table19>();
constexpr auto Tree20 = make_huffman_tree<Table20>();
constexpr auto Tree21 = make_huffman_tree<Table21>();
constexpr auto Tree22 = make_huffman_tree<Table22>();
constexpr auto Tree23 = make_huffman_tree<Table23>();
constexpr auto Tree24 = make_huffman_tree<Table24>();
constexpr auto Tree25 = make_huffman_tree<Table25>();
constexpr auto Tree26 = make_huffman_tree<Table26>();
constexpr auto Tree27 = make_huffman_tree<Table27>();
constexpr auto Tree28 = make_huffman_tree<Table28>();
constexpr auto Tree29 = make_huffman_tree<Table29>();
constexpr auto Tree30 = make_huffman_tree<Table30>();
constexpr auto Tree31 = make_huffman_tree<Table31>();

struct HuffmanTreeXY {
    Span<HuffmanNode<HuffmanXY> const> nodes;
    int linbits;
};

Array<HuffmanTreeXY const, 32> HuffmanTreesXY { {
    { Tree0, 0 },
    { Tree1, 0 },
    { Tree2, 0 },
    { Tree3, 0 },
    { {}, 0 },
    { Tree5, 0 },
    { Tree6, 0 },
    { Tree7, 0 },
    { Tree8, 0 },
    { Tree9, 0 },
    { Tree10, 0 },
    { Tree11, 0 },
    { Tree12, 0 },
    { Tree13, 0 },
    { {}, 0 },
    { Tree15, 0 },
    { Tree16, 1 },
    { Tree17, 2 },
    { Tree18, 3 },
    { Tree19, 4 },
    { Tree20, 6 },
    { Tree21, 8 },
    { Tree22, 10 },
    { Tree23, 13 },
    { Tree24, 4 },
    { Tree25, 5 },
    { Tree26, 6 },
    { Tree27, 7 },
    { Tree28, 8 },
    { Tree29, 9 },
    { Tree30, 11 },
    { Tree31, 13 },
} };

}
