/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
/**
 * Helpers for converting FROM LanguageServer types language-server ts types
 */
import * as lsp from 'vscode-languageserver-protocol';
export var Range;
(function (Range) {
    Range.fromTextSpan = (span) => Range.fromLocations(span.start, span.end);
    Range.toTextSpan = (range) => ({
        start: Position.toLocation(range.start),
        end: Position.toLocation(range.end),
    });
    Range.fromLocations = (start, end) => lsp.Range.create(Math.max(0, start.line - 1), Math.max(start.offset - 1, 0), Math.max(0, end.line - 1), Math.max(0, end.offset - 1));
    Range.toFileRangeRequestArgs = (file, range) => ({
        file,
        startLine: range.start.line + 1,
        startOffset: range.start.character + 1,
        endLine: range.end.line + 1,
        endOffset: range.end.character + 1,
    });
    Range.toFormattingRequestArgs = (file, range) => ({
        file,
        line: range.start.line + 1,
        offset: range.start.character + 1,
        endLine: range.end.line + 1,
        endOffset: range.end.character + 1,
    });
    function intersection(one, other) {
        const start = Position.Max(other.start, one.start);
        const end = Position.Min(other.end, one.end);
        if (Position.isAfter(start, end)) {
            // this happens when there is no overlap:
            // |-----|
            //          |----|
            return undefined;
        }
        return lsp.Range.create(start, end);
    }
    Range.intersection = intersection;
})(Range = Range || (Range = {}));
export var Position;
(function (Position) {
    Position.fromLocation = (tslocation) => {
        // Clamping on the low side to 0 since Typescript returns 0, 0 when creating new file
        // even though position is supposed to be 1-based.
        return {
            line: Math.max(tslocation.line - 1, 0),
            character: Math.max(tslocation.offset - 1, 0),
        };
    };
    Position.toLocation = (position) => ({
        line: position.line + 1,
        offset: position.character + 1,
    });
    Position.toFileLocationRequestArgs = (file, position) => ({
        file,
        line: position.line + 1,
        offset: position.character + 1,
    });
    function Min(...positions) {
        if (!positions.length) {
            return undefined;
        }
        let result = positions.pop();
        for (const p of positions) {
            if (isBefore(p, result)) {
                result = p;
            }
        }
        return result;
    }
    Position.Min = Min;
    function isBefore(one, other) {
        if (one.line < other.line) {
            return true;
        }
        if (other.line < one.line) {
            return false;
        }
        return one.character < other.character;
    }
    Position.isBefore = isBefore;
    function Max(...positions) {
        if (!positions.length) {
            return undefined;
        }
        let result = positions.pop();
        for (const p of positions) {
            if (isAfter(p, result)) {
                result = p;
            }
        }
        return result;
    }
    Position.Max = Max;
    function isAfter(one, other) {
        return !isBeforeOrEqual(one, other);
    }
    Position.isAfter = isAfter;
    function isBeforeOrEqual(one, other) {
        if (one.line < other.line) {
            return true;
        }
        if (other.line < one.line) {
            return false;
        }
        return one.character <= other.character;
    }
    Position.isBeforeOrEqual = isBeforeOrEqual;
})(Position = Position || (Position = {}));
export var Location;
(function (Location) {
    Location.fromTextSpan = (resource, tsTextSpan) => lsp.Location.create(resource, Range.fromTextSpan(tsTextSpan));
})(Location = Location || (Location = {}));
//# sourceMappingURL=typeConverters.js.map