"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDefaultConfiguration = exports.getConfigFromEnvironmentVariables = exports.ConfigSchema = void 0;
const zod_1 = require("zod");
exports.ConfigSchema = zod_1.z
    .object({
    // Maximum number of files to analyze in the background. Set to 0 to disable background analysis.
    backgroundAnalysisMaxFiles: zod_1.z.number().int().min(0).default(500),
    // Glob pattern for finding and parsing shell script files in the workspace. Used by the background analysis features across files.
    globPattern: zod_1.z.string().trim().default('**/*@(.sh|.inc|.bash|.command)'),
    // Configure explainshell server endpoint in order to get hover documentation on flags and options.
    // And empty string will disable the feature.
    explainshellEndpoint: zod_1.z.string().trim().default(''),
    // Controls if Treesitter parsing errors will be highlighted as problems.
    highlightParsingErrors: zod_1.z.boolean().default(false),
    // Controls how symbols (e.g. variables and functions) are included and used for completion and documentation.
    // If false, then we only include symbols from sourced files (i.e. using non dynamic statements like 'source file.sh' or '. file.sh').
    // If true, then all symbols from the workspace are included.
    includeAllWorkspaceSymbols: zod_1.z.boolean().default(false),
    // Additional ShellCheck arguments. Note that we already add the following arguments: --shell, --format, --external-sources."
    shellcheckArguments: zod_1.z
        .preprocess((arg) => {
        let argsList = [];
        if (typeof arg === 'string') {
            argsList = arg.split(' ');
            //.map((s) => s.trim())
            //.filter((s) => s.length > 0)
        }
        else if (Array.isArray(arg)) {
            argsList = arg;
        }
        return argsList.map((s) => s.trim()).filter((s) => s.length > 0);
    }, zod_1.z.array(zod_1.z.string()))
        .default([]),
    // Controls the executable used for ShellCheck linting information. An empty string will disable linting.
    shellcheckPath: zod_1.z.string().trim().default('shellcheck'),
})
    .strict();
function getConfigFromEnvironmentVariables() {
    const rawConfig = {
        backgroundAnalysisMaxFiles: process.env.BACKGROUND_ANALYSIS_MAX_FILES,
        explainshellEndpoint: process.env.EXPLAINSHELL_ENDPOINT,
        globPattern: process.env.GLOB_PATTERN,
        highlightParsingErrors: toBoolean(process.env.HIGHLIGHT_PARSING_ERRORS),
        includeAllWorkspaceSymbols: toBoolean(process.env.INCLUDE_ALL_WORKSPACE_SYMBOLS),
        shellcheckArguments: process.env.SHELLCHECK_ARGUMENTS,
        shellcheckPath: process.env.SHELLCHECK_PATH,
    };
    const environmentVariablesUsed = Object.entries(rawConfig)
        .map(([key, value]) => (typeof value !== 'undefined' ? key : null))
        .filter((key) => key !== null);
    const config = exports.ConfigSchema.parse(rawConfig);
    return { config, environmentVariablesUsed };
}
exports.getConfigFromEnvironmentVariables = getConfigFromEnvironmentVariables;
function getDefaultConfiguration() {
    return exports.ConfigSchema.parse({});
}
exports.getDefaultConfiguration = getDefaultConfiguration;
const toBoolean = (s) => typeof s !== 'undefined' ? s === 'true' || s === '1' : undefined;
//# sourceMappingURL=config.js.map