"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DataSourceSavedObjectsClientWrapper = void 0;

var _server = require("../../../../../src/core/server");

var _common = require("../../common");

var _data_sources = require("../../common/data_sources");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * Describes the Credential Saved Objects Client Wrapper class,
 * which contains the factory used to create Saved Objects Client Wrapper instances
 */
class DataSourceSavedObjectsClientWrapper {
  constructor(cryptography, logger) {
    this.cryptography = cryptography;
    this.logger = logger;

    _defineProperty(this, "wrapperFactory", wrapperOptions => {
      const createWithCredentialsEncryption = async (type, attributes, options) => {
        if (_common.DATA_SOURCE_SAVED_OBJECT_TYPE !== type) {
          return await wrapperOptions.client.create(type, attributes, options);
        }

        const encryptedAttributes = await this.validateAndEncryptAttributes(attributes);
        return await wrapperOptions.client.create(type, encryptedAttributes, options);
      };

      const bulkCreateWithCredentialsEncryption = async (objects, options) => {
        objects = await Promise.all(objects.map(async object => {
          const {
            type,
            attributes
          } = object;

          if (_common.DATA_SOURCE_SAVED_OBJECT_TYPE !== type) {
            return object;
          }

          return { ...object,
            attributes: await this.validateAndEncryptAttributes(attributes)
          };
        }));
        return await wrapperOptions.client.bulkCreate(objects, options);
      };

      const updateWithCredentialsEncryption = async (type, id, attributes, options = {}) => {
        if (_common.DATA_SOURCE_SAVED_OBJECT_TYPE !== type) {
          return await wrapperOptions.client.update(type, id, attributes, options);
        }

        const encryptedAttributes = await this.validateAndUpdatePartialAttributes(wrapperOptions, id, attributes, options);
        return await wrapperOptions.client.update(type, id, encryptedAttributes, options);
      };

      const bulkUpdateWithCredentialsEncryption = async (objects, options) => {
        objects = await Promise.all(objects.map(async object => {
          const {
            id,
            type,
            attributes
          } = object;

          if (_common.DATA_SOURCE_SAVED_OBJECT_TYPE !== type) {
            return object;
          }

          const encryptedAttributes = await this.validateAndUpdatePartialAttributes(wrapperOptions, id, attributes, options);
          return { ...object,
            attributes: encryptedAttributes
          };
        }));
        return await wrapperOptions.client.bulkUpdate(objects, options);
      };

      return { ...wrapperOptions.client,
        create: createWithCredentialsEncryption,
        bulkCreate: bulkCreateWithCredentialsEncryption,
        checkConflicts: wrapperOptions.client.checkConflicts,
        delete: wrapperOptions.client.delete,
        find: wrapperOptions.client.find,
        bulkGet: wrapperOptions.client.bulkGet,
        get: wrapperOptions.client.get,
        update: updateWithCredentialsEncryption,
        bulkUpdate: bulkUpdateWithCredentialsEncryption,
        errors: wrapperOptions.client.errors,
        addToNamespaces: wrapperOptions.client.addToNamespaces,
        deleteFromNamespaces: wrapperOptions.client.deleteFromNamespaces
      };
    });
  }
  /**
   * Describes the factory used to create instances of Saved Objects Client Wrappers
   * for data source spcific operations such as credntials encryption
   */


  isValidUrl(endpoint) {
    try {
      const url = new URL(endpoint);
      return Boolean(url) && (url.protocol === 'http:' || url.protocol === 'https:');
    } catch (e) {
      return false;
    }
  }

  async validateAndEncryptAttributes(attributes) {
    this.validateAttributes(attributes);
    const {
      endpoint,
      auth
    } = attributes;

    switch (auth.type) {
      case _data_sources.AuthType.NoAuth:
        return { ...attributes,
          // Drop the credentials attribute for no_auth
          auth: {
            type: auth.type,
            credentials: undefined
          }
        };

      case _data_sources.AuthType.UsernamePasswordType:
        // Signing the data source with endpoint
        const encryptionContext = {
          endpoint
        };
        return { ...attributes,
          auth: await this.encryptCredentials(auth, encryptionContext)
        };

      default:
        throw _server.SavedObjectsErrorHelpers.createBadRequestError(`Invalid auth type: '${auth.type}'`);
    }
  }

  async validateAndUpdatePartialAttributes(wrapperOptions, id, attributes, options = {}) {
    const {
      auth,
      endpoint
    } = attributes;

    if (endpoint) {
      throw _server.SavedObjectsErrorHelpers.createBadRequestError(`Updating a dataSource endpoint is not supported`);
    }

    if (!auth) {
      return attributes;
    }

    const {
      type,
      credentials
    } = auth;

    switch (type) {
      case _data_sources.AuthType.NoAuth:
        return { ...attributes,
          // Drop the credentials attribute for no_auth
          auth: {
            type: auth.type,
            credentials: null
          }
        };

      case _data_sources.AuthType.UsernamePasswordType:
        if (credentials !== null && credentials !== void 0 && credentials.password) {
          // Fetch and validate existing signature
          const encryptionContext = await this.validateEncryptionContext(wrapperOptions, id, options);
          return { ...attributes,
            auth: await this.encryptCredentials(auth, encryptionContext)
          };
        } else {
          return attributes;
        }

      default:
        throw _server.SavedObjectsErrorHelpers.createBadRequestError(`Invalid credentials type: '${type}'`);
    }
  }

  validateAttributes(attributes) {
    var _title$trim;

    const {
      title,
      endpoint,
      auth
    } = attributes;

    if (!(title !== null && title !== void 0 && (_title$trim = title.trim) !== null && _title$trim !== void 0 && _title$trim.call(title).length)) {
      throw _server.SavedObjectsErrorHelpers.createBadRequestError('"title" attribute must be a non-empty string');
    }

    if (!this.isValidUrl(endpoint)) {
      throw _server.SavedObjectsErrorHelpers.createBadRequestError('"endpoint" attribute is not valid');
    }

    if (!auth) {
      throw _server.SavedObjectsErrorHelpers.createBadRequestError('"auth" attribute is required');
    }

    this.validateAuth(auth);
  }

  validateAuth(auth) {
    const {
      type,
      credentials
    } = auth;

    if (!type) {
      throw _server.SavedObjectsErrorHelpers.createBadRequestError('"auth.type" attribute is required');
    }

    switch (type) {
      case _data_sources.AuthType.NoAuth:
        break;

      case _data_sources.AuthType.UsernamePasswordType:
        if (!credentials) {
          throw _server.SavedObjectsErrorHelpers.createBadRequestError('"auth.credentials" attribute is required');
        }

        const {
          username,
          password
        } = credentials;

        if (!username) {
          throw _server.SavedObjectsErrorHelpers.createBadRequestError('"auth.credentials.username" attribute is required');
        }

        if (!password) {
          throw _server.SavedObjectsErrorHelpers.createBadRequestError('"auth.credentials.password" attribute is required');
        }

        break;

      default:
        throw _server.SavedObjectsErrorHelpers.createBadRequestError(`Invalid auth type: '${type}'`);
    }
  }

  async validateEncryptionContext(wrapperOptions, id, options = {}) {
    let attributes;

    try {
      // Fetch existing data source by id
      const savedObject = await wrapperOptions.client.get(_common.DATA_SOURCE_SAVED_OBJECT_TYPE, id, {
        namespace: options.namespace
      });
      attributes = savedObject.attributes;
    } catch (err) {
      const errMsg = `Failed to fetch existing data source for dataSourceId [${id}]`;
      this.logger.error(errMsg);
      this.logger.error(err);
      throw _server.SavedObjectsErrorHelpers.decorateBadRequestError(err, errMsg);
    }

    if (!attributes) {
      throw _server.SavedObjectsErrorHelpers.createBadRequestError('Update failed due to deprecated data source: "attributes" missing. Please delete and create another data source.');
    }

    const {
      endpoint,
      auth
    } = attributes;

    if (!endpoint) {
      throw _server.SavedObjectsErrorHelpers.createBadRequestError('Update failed due to deprecated data source: "endpoint" missing. Please delete and create another data source.');
    }

    if (!auth) {
      throw _server.SavedObjectsErrorHelpers.createBadRequestError('Update failed due to deprecated data source: "auth" missing. Please delete and create another data source.');
    }

    switch (auth.type) {
      case _data_sources.AuthType.NoAuth:
        // Signing the data source with exsiting endpoint
        return {
          endpoint
        };

      case _data_sources.AuthType.UsernamePasswordType:
        const {
          credentials
        } = auth;

        if (!credentials) {
          throw _server.SavedObjectsErrorHelpers.createBadRequestError('Update failed due to deprecated data source: "credentials" missing. Please delete and create another data source.');
        }

        const {
          username,
          password
        } = credentials;

        if (!username) {
          throw _server.SavedObjectsErrorHelpers.createBadRequestError('Update failed due to deprecated data source: "auth.credentials.username" missing. Please delete and create another data source.');
        }

        if (!password) {
          throw _server.SavedObjectsErrorHelpers.createBadRequestError('Update failed due to deprecated data source: "auth.credentials.username" missing. Please delete and create another data source.');
        }

        const {
          encryptionContext
        } = await this.cryptography.decodeAndDecrypt(password).catch(err => {
          const errMsg = `Failed to update existing data source for dataSourceId [${id}]: unable to decrypt "auth.credentials.password"`;
          this.logger.error(errMsg);
          this.logger.error(err);
          throw _server.SavedObjectsErrorHelpers.decorateBadRequestError(err, errMsg);
        });

        if (encryptionContext.endpoint !== endpoint) {
          throw _server.SavedObjectsErrorHelpers.createBadRequestError('Update failed due to deprecated data source: "endpoint" contaminated. Please delete and create another data source.');
        }

        return encryptionContext;

      default:
        throw _server.SavedObjectsErrorHelpers.createBadRequestError(`Invalid auth type: '${type}'`);
    }
  }

  async encryptCredentials(auth, encryptionContext) {
    const {
      credentials: {
        username,
        password
      }
    } = auth;
    return { ...auth,
      credentials: {
        username,
        password: await this.cryptography.encryptAndEncode(password, encryptionContext)
      }
    };
  }

}

exports.DataSourceSavedObjectsClientWrapper = DataSourceSavedObjectsClientWrapper;