/// <reference types="node" />
import { MatrixClient } from "../MatrixClient";
import { IMegolmEncrypted, IOlmEncrypted, IOlmSession, IToDeviceMessage, OTKCounts, Signatures } from "../models/Crypto";
import { EncryptedRoomEvent } from "../models/events/EncryptedRoomEvent";
import { RoomEvent } from "../models/events/RoomEvent";
import { EncryptedFile } from "../models/events/MessageEvent";
/**
 * Manages encryption for a MatrixClient. Get an instance from a MatrixClient directly
 * rather than creating one manually.
 * @category Encryption
 */
export declare class CryptoClient {
    private client;
    private ready;
    private deviceId;
    private pickleKey;
    private deviceEd25519;
    private deviceCurve25519;
    private maxOTKs;
    private roomTracker;
    private deviceTracker;
    constructor(client: MatrixClient);
    /**
     * The device ID for the MatrixClient.
     */
    get clientDeviceId(): string;
    /**
     * Whether or not the crypto client is ready to be used. If not ready, prepare() should be called.
     * @see prepare
     */
    get isReady(): boolean;
    private getOlmAccount;
    private storeAndFreeOlmAccount;
    /**
     * Prepares the crypto client for usage.
     * @param {string[]} roomIds The room IDs the MatrixClient is joined to.
     */
    prepare(roomIds: string[]): Promise<void>;
    /**
     * Checks if a room is encrypted.
     * @param {string} roomId The room ID to check.
     * @returns {Promise<boolean>} Resolves to true if encrypted, false otherwise.
     */
    isRoomEncrypted(roomId: string): Promise<boolean>;
    /**
     * Updates the One Time Key counts, potentially triggering an async upload of more
     * one time keys.
     * @param {OTKCounts} counts The current counts to work within.
     * @returns {Promise<void>} Resolves when complete.
     */
    updateCounts(counts: OTKCounts): Promise<void>;
    /**
     * Updates the client's fallback key.
     * @returns {Promise<void>} Resolves when complete.
     */
    updateFallbackKey(): Promise<void>;
    /**
     * Signs an object using the device keys.
     * @param {object} obj The object to sign.
     * @returns {Promise<Signatures>} The signatures for the object.
     */
    sign(obj: object): Promise<Signatures>;
    /**
     * Verifies a signature on an object.
     * @param {object} obj The signed object.
     * @param {string} key The key which has supposedly signed the object.
     * @param {string} signature The advertised signature.
     * @returns {Promise<boolean>} Resolves to true if a valid signature, false otherwise.
     */
    verifySignature(obj: object, key: string, signature: string): Promise<boolean>;
    /**
     * Flags multiple user's device lists as outdated, optionally queuing an immediate update.
     * @param {string} userIds The user IDs to flag the device lists of.
     * @param {boolean} resync True (default) to queue an immediate update, false otherwise.
     * @returns {Promise<void>} Resolves when the device lists have been flagged. Will also wait
     * for the resync if one was requested.
     */
    flagUsersDeviceListsOutdated(userIds: string[], resync?: boolean): Promise<void>;
    /**
     * Gets or creates Olm sessions for the given users and devices. Where sessions cannot be created,
     * the user/device will be excluded from the returned map.
     * @param {Record<string, string[]>} userDeviceMap Map of user IDs to device IDs
     * @param {boolean} force If true, force creation of a session for the referenced users.
     * @returns {Promise<Record<string, Record<string, IOlmSession>>>} Resolves to a map of user ID to device
     * ID to session. Users/devices which cannot have sessions made will not be included, thus the object
     * may be empty.
     */
    getOrCreateOlmSessions(userDeviceMap: Record<string, string[]>, force?: boolean): Promise<Record<string, Record<string, IOlmSession>>>;
    private encryptAndSendOlmMessage;
    /**
     * Encrypts the details of a room event, returning an encrypted payload to be sent in an
     * `m.room.encrypted` event to the room. If needed, this function will send decryption keys
     * to the appropriate devices in the room (this happens when the Megolm session rotates or
     * gets created).
     * @param {string} roomId The room ID to encrypt within. If the room is not encrypted, an
     * error is thrown.
     * @param {string} eventType The event type being encrypted.
     * @param {any} content The event content being encrypted.
     * @returns {Promise<IMegolmEncrypted>} Resolves to the encrypted content for an `m.room.encrypted` event.
     */
    encryptRoomEvent(roomId: string, eventType: string, content: any): Promise<IMegolmEncrypted>;
    /**
     * Decrypts a room event. Currently only supports Megolm-encrypted events (default for this SDK).
     * @param {EncryptedRoomEvent} event The encrypted event.
     * @param {string} roomId The room ID where the event was sent.
     * @returns {Promise<RoomEvent<unknown>>} Resolves to a decrypted room event, or rejects/throws with
     * an error if the event is undecryptable.
     */
    decryptRoomEvent(event: EncryptedRoomEvent, roomId: string): Promise<RoomEvent<unknown>>;
    /**
     * Handles an inbound to-device message, decrypting it if needed. This will not throw
     * under normal circumstances and should always resolve successfully.
     * @param {IToDeviceMessage<IOlmEncrypted>} message The message to process.
     * @returns {Promise<void>} Resolves when complete. Should never fail.
     */
    processInboundDeviceMessage(message: IToDeviceMessage<IOlmEncrypted>): Promise<void>;
    private handleInboundRoomKey;
    private storeInboundGroupSession;
    private establishNewOlmSession;
    /**
     * Encrypts a file for uploading in a room, returning the encrypted data and information
     * to include in a message event (except media URL) for sending.
     * @param {Buffer} file The file to encrypt.
     * @returns {{buffer: Buffer, file: Omit<EncryptedFile, "url">}} Resolves to the encrypted
     * contents and file information.
     */
    encryptMedia(file: Buffer): Promise<{
        buffer: Buffer;
        file: Omit<EncryptedFile, "url">;
    }>;
    /**
     * Decrypts a previously-uploaded encrypted file, validating the fields along the way.
     * @param {EncryptedFile} file The file to decrypt.
     * @returns {Promise<Buffer>} Resolves to the decrypted file contents.
     */
    decryptMedia(file: EncryptedFile): Promise<Buffer>;
}
