#!/bin/bash
#
# Simple shell script to update IP2Location database files
#
# For updating non-sample files, credentials are or token is required:
#  provided by file $HOME/.ip2location
#   credential-based authentication:
#    login=EMAIL
#    password=PASSWORD
#   token-based authentication:
#    token=TOKEN
#
# Project    : ipv6calc/IP2Location
# File       : IP2Location-update.sh
# Version    : $Id: 51d432e406fc84c7f196a92c9037e730e1172131 $
# Copyright  : 2012-2021 by Peter Bieringer <pb (at) bieringer.de>
# License    : GNU GPL version 2

# 20150416: remove softlink support, no longer needed
# 20210528: add support for 'token'

IP2LOCATION_CONFIG=${IP2LOCATION_CONFIG:-"$HOME/.ip2location"}

IP2LOCATION_DAT_DIR_DEFAULT="/usr/share/IP2Location"
[ -z "$IP2LOCATION_DAT_DIR" -a -n "$IPV6CALC_DB_IP2LOCATION_DIR" ] && IP2LOCATION_DAT_DIR="$IPV6CALC_DB_IP2LOCATION_DIR"
[ -z "$IP2LOCATION_DAT_DIR" ] && IP2LOCATION_DAT_DIR="$IP2LOCATION_DAT_DIR_DEFAULT"

IP2LOCATION_DAT_URL_BASE="https://www.ip2location.com/download?login=@LOGIN@&password=@PASSWORD@&token=@TOKEN@"
IP2LOCATION_DAT_URL_BASE_SAMPLE="https://www.ip2location.com/downloads"
IP2LOCATION_DAT_URL_BASE_LITE_FREE="https://download.ip2location.com/lite"

# Format: Proto:DB-Number

# free downloads
IP2LOCATION_DAT_FILES_LITE_FREE=${IP2LOCATION_DAT_FILES_LITE_FREE:-"IPv4:1 IPv6:1"}

# (free) subscription is required for LITE DB-Number > 1
IP2LOCATION_DAT_FILES_LITE=${IP2LOCATION_DAT_FILES_LITE:-"IPv4:11 IPv6:11"}

# list of normal (purchased subscription required)
IP2LOCATION_DAT_FILES="${IP2LOCATION_DAT_FILES}"

# list of recommended SAMPLE
IP2LOCATION_DAT_FILES_SAMPLE=${IP2LOCATION_DAT_FILES_SAMPLE:-"IPv4:20 IPv6:20 IPv4:24 IPv6:24"}


help() {
	cat <<END
Usage: $(basename "$0") [-D <dir>] [-n] [-s] [-t] [-v]
	-D <dir>	database destination directory (optional)
	-n		no action (dry-run)
	-s		skip already successfully downloaded files
	-t		enforce use of 'token' instead of using 'login' & 'password'
	-v		verbose

	database directory: $IP2LOCATION_DAT_DIR (default: $IP2LOCATION_DAT_DIR_DEFAULT)

	it honors externally defined environment value:
		prio 1: IP2LOCATION_DAT_DIR
		prio 2: IPV6CALC_DB_IP2LOCATION_DIR

 this script will download data from ip2location.com
  into given/set database directory

 Sample databases:
  IP2LOCATION_DAT_FILES_SAMPLE=$IP2LOCATION_DAT_FILES_SAMPLE

 Lite (free) databases:
  IP2LOCATION_DAT_FILES_LITE_FREE=$IP2LOCATION_DAT_FILES_LITE_FREE

 With authentication (requires login/password or token in $IP2LOCATION_CONFIG) and a valid subscription
  IP2LOCATION_DAT_FILES_LITE=$IP2LOCATION_DAT_FILES_LITE
  IP2LOCATION_DAT_FILES=$IP2LOCATION_DAT_FILES

 Credentials or token must be defined in \$HOME/.ip2location by
  login=IP2LOCATION-LOGIN       ${login:+found: ${login:0:2}...}
  password=IP2LOCATION-PASSWORD ${password:+found: ${password:0:2}...}
  token=IP2LOCATION-TOKEN       ${token:+found: ${token:0:2}...}

 In addition settings from above can be overwritten by setting related variables
END
}

# source credentials (must provide login= and password= or token=)
authenticated=false
has_token=false
has_credentials=false
verbose=false

if [ -f "$IP2LOCATION_CONFIG" ]; then
	source "$IP2LOCATION_CONFIG" || exit 1

	if [ -n "${token}" ]; then
		has_token=true
		authenticated=true
		echo "INFO  : found token in: $IP2LOCATION_CONFIG (for authenticated downloads)"
	fi

	if [ -n "${login}" -a -n "${password}" ]; then
		has_credentials=true
		authenticated=true
		echo "INFO  : found credentials in: $IP2LOCATION_CONFIG (for authenticated downloads)"
	fi

	if ! $authenticated; then
		echo "NOTICE: missing credentials/token in: $IP2LOCATION_CONFIG (skip authenticated downloads)"
	fi
fi

# option parsing
while getopts "vfsntD:h\?" opt; do
	case $opt in
	    D)
		IP2LOCATION_DAT_DIR=$OPTARG
		;;
	    n)
		NO_ACTION=1
		;;
	    s)
		SKIP_OK=1
		;;
	    t)
		has_credentials=false
		;;
	    v)
		verbose=true
		;;
	    *)
		help
		exit 1
		;;
	esac
done


# directory check
if [ ! -d "$IP2LOCATION_DAT_DIR" ]; then
	echo "ERROR : missing directory: $IP2LOCATION_DAT_DIR (check option -D or set IP2LOCATION_DAT_DIR)"
	exit 1
fi

if [ ! -w "$IP2LOCATION_DAT_DIR" ]; then
	echo "ERROR : missing write permissions on directory: $IP2LOCATION_DAT_DIR"
	exit 1
fi

# required binary check
if ! which unzip >/dev/null 2>&1; then
	echo "ERROR : missing binary: unzip"
	exit 1
fi

if ! $authenticated; then
	echo "NOTICE: IP2Location credentials/token not found, clear IP2LOCATION_DAT_FILES"
	IP2LOCATION_DAT_FILES=""

	echo "NOTICE: IP2Location credentials/token not found, clear IP2LOCATION_DAT_FILES_LITE"
	IP2LOCATION_DAT_FILES_LITE=""
fi

if $has_credentials; then
	# replace placeholders
	IP2LOCATION_DAT_URL_BASE=${IP2LOCATION_DAT_URL_BASE//@LOGIN@/$login}
	IP2LOCATION_DAT_URL_BASE=${IP2LOCATION_DAT_URL_BASE//@PASSWORD@/$password}
	# clear token parameter
	IP2LOCATION_DAT_URL_BASE=${IP2LOCATION_DAT_URL_BASE//@TOKEN@/}
fi

if $has_token; then
	# replace placeholders
	IP2LOCATION_DAT_URL_BASE=${IP2LOCATION_DAT_URL_BASE//@TOKEN@/$token}
	# clear credential parameters
	IP2LOCATION_DAT_URL_BASE=${IP2LOCATION_DAT_URL_BASE//@LOGIN@/}
	IP2LOCATION_DAT_URL_BASE=${IP2LOCATION_DAT_URL_BASE//@PASSWORD@/}
fi

# Download and unpack non-sample files
for list in IP2LOCATION_DAT_FILES_LITE IP2LOCATION_DAT_FILES IP2LOCATION_DAT_FILES_LITE_FREE IP2LOCATION_DAT_FILES_SAMPLE; do
	if [ -z "${!list}" ]; then
		echo "NOTICE: empty list, skip: $list"
		continue
	fi

	$verbose && echo "INFO  : download list $list: ${!list}"

	url_base="$IP2LOCATION_DAT_URL_BASE"

	case $list in
	    *_LITE|*_LITE_FREE)
		code_suffix="LITEBIN"
		file_prefix="IP2LOCATION-LITE"

		case $list in
		    *_LITE_FREE)
			url_base="$IP2LOCATION_DAT_URL_BASE_LITE_FREE"
			;;
		esac
		;;
	    *_SAMPLE)
		url_base="$IP2LOCATION_DAT_URL_BASE_SAMPLE"
		;;
	    *)
		code_suffix="BIN"
		file_prefix="IP2LOCATION"
		;;
	esac

	file_dest="$IP2LOCATION_DAT_DIR/$file"

	if [ "${!list}" = "skip" ]; then
		echo "NOTICE: skip list (by keyword 'skip'): $list"
		continue
	fi

	for item in ${!list}; do
		proto=${item/:*}
		number=${item/*:}

		case $list in
		    *_SAMPLE)
			case $proto in
			    IPv4)
				protoitem=""
				;;
			    IPv6)
				protoitem="6"
				;;
			    *)
				echo "ERROR : unsupported 'proto' in item: $item"
				exit 1
				;;
			esac

			file="sample${protoitem}.bin.db${number}.zip"
			;;
		    *)
			case $proto in
			    IPv4)
				code="DB${number}${code_suffix}"
				file="${file_prefix}-DB${number}.BIN.ZIP"
				;;
			    IPv6)
				code="DB${number}${code_suffix}IPV6"
				file="${file_prefix}-DB${number}.IPV6.BIN.ZIP"
				;;
			    *)
				echo "ERROR : unsupported 'proto' in item: $item"
				exit 1
				;;
			esac

			;;
		esac

		case $list in
		    *_SAMPLE|*_LITE_FREE)
			url="$url_base/$file"
			;;
		    *)
			url="$url_base&productcode=$code"
			;;
		esac

		file_dest="$IP2LOCATION_DAT_DIR/$file"

		if [ "$NO_ACTION" = "1" ]; then
			if $has_credentials; then
				echo "NOTICE: download skipped by option: ${url/password=*&/password=**PASS**&} ($file_dest)"
				continue
			fi
			if $has_token; then
				echo "NOTICE: download skipped by option: ${url/token=*&/token=**TOK**&} ($file_dest)"
				continue
			fi
		fi

		if [ "$SKIP_OK" = "1" ]; then
			if file "$file_dest" | grep -q "Zip archive data"; then
				echo "NOTICE: download skipped by option because file already existing and is a ZIP file: $code ($file_dest)"

				file_unzip=$(unzip -l "$file_dest" '*.BIN' | grep "\.BIN$" | awk '{ print $NF }' | head -1)

				if [ -n "$file_unzip" ]; then
					if [ ! -f "$IP2LOCATION_DAT_DIR/$file_unzip" ]; then
						unzip -o -d "$IP2LOCATION_DAT_DIR" "$file_dest" '*.BIN'
						if [ $? -ne 0 ]; then
							echo "ERROR : unzip of file not successful: $file_dest"
							continue
						else
							echo "INFO  : unzip of file successful: $file_dest"
						fi
					else
						echo "INFO  : skip unzip because unzipped file already existing: $code ($file_dest)"
					fi
				fi
				continue
			fi
		fi

		if $has_credentials; then
			$verbose && echo "INFO  : try to download URL: ${url/password=*&/password=**PASS**&} ($file_dest)"
		fi
		if $has_token; then
			$verbose && echo "INFO  : try to download URL: ${url/token=*&/token=**TOK**&} ($file_dest)"
		fi

		wget -q -O "$file_dest" "$url"
		result=$?
			
		if [ $result -ne 0 ]; then
			echo "ERROR : download of file not successful: $file_dest ($!)"
			continue
		fi
		$verbose && echo "INFO  : download of file successful: $file_dest"

		if [ ! -s "$file_dest" ]; then
			echo "ERROR : downloaded file has ZERO size: $file_dest"
			continue
		fi

		if ! file "$file_dest" | grep -q "Zip archive data"; then
			if file "$file_dest" | grep -q "ASCII text"; then
				echo "ERROR : downloaded file is not a ZIP archive: $file_dest ($(head -1 "$file_dest"))"
			else
				echo "ERROR : downloaded file is not a ZIP archive: $file_dest"
			fi
			continue
		fi

		unzip -o -d "$IP2LOCATION_DAT_DIR" "$file_dest" '*.BIN'
		if [ $? -ne 0 ]; then
			echo "ERROR : unzip of file not successful: $file_dest"
			continue
		fi
		$verbose && echo "INFO  : unzip of file successful: $file_dest (remove downloaded file now)"
		rm -f "$file_dest"

		echo "NOTICE: download/unzip successful: $item"
	done
done

# Adjust permissions
if [ "$NO_ACTION" != "1" ]; then
	chmod 644 $IP2LOCATION_DAT_DIR/*.BIN
fi
