"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.configureLegacyClient = void 0;

var _elasticsearch = require("elasticsearch");

var _lodash = require("lodash");

var _server = require("../../../../../src/core/server");

var _data_sources = require("../../common/data_sources");

var _client = require("../client");

var _client_config = require("./client_config");

var _error = require("../lib/error");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const configureLegacyClient = async ({
  dataSourceId,
  savedObjects,
  cryptography
}, callApiParams, openSearchClientPoolSetup, config, logger) => {
  try {
    const dataSource = await (0, _client.getDataSource)(dataSourceId, savedObjects);
    const rootClient = getRootClient(dataSource.attributes, config, openSearchClientPoolSetup);
    return await getQueryClient(rootClient, dataSource, cryptography, callApiParams);
  } catch (error) {
    logger.error(`Failed to get data source client for dataSourceId: [${dataSourceId}]`);
    logger.error(error); // Re-throw as DataSourceError

    throw (0, _error.createDataSourceError)(error);
  }
};
/**
 * With given auth info, wrap the rootClient and return
 *
 * @param rootClient root client for the connection with given data source endpoint.
 * @param dataSource data source saved object
 * @param cryptography cryptography service for password encryption / decryption
 * @returns child client.
 */


exports.configureLegacyClient = configureLegacyClient;

const getQueryClient = async (rootClient, dataSource, cryptography, {
  endpoint,
  clientParams,
  options
}) => {
  const authType = dataSource.attributes.auth.type;

  switch (authType) {
    case _data_sources.AuthType.NoAuth:
      return await callAPI.bind(null, rootClient)(endpoint, clientParams, options);

    case _data_sources.AuthType.UsernamePasswordType:
      const credential = await (0, _client.getCredential)(dataSource, cryptography);
      return getBasicAuthClient(rootClient, {
        endpoint,
        clientParams,
        options
      }, credential);

    default:
      throw Error(`${authType} is not a supported auth type for data source`);
  }
};
/**
 * Gets a root client object of the OpenSearch endpoint.
 * Will attempt to get from cache, if cache miss, create a new one and load into cache.
 *
 * @param dataSourceAttr data source saved objects attributes.
 * @param config data source config
 * @returns Legacy client for the given data source endpoint.
 */


const getRootClient = (dataSourceAttr, config, {
  getClientFromPool,
  addClientToPool
}) => {
  const endpoint = dataSourceAttr.endpoint;
  const cachedClient = getClientFromPool(endpoint);

  if (cachedClient) {
    return cachedClient;
  } else {
    const configOptions = (0, _client_config.parseClientOptions)(config, endpoint);
    const client = new _elasticsearch.Client(configOptions);
    addClientToPool(endpoint, client);
    return client;
  }
};
/**
 * Calls the OpenSearch API endpoint with the specified parameters.
 * @param client Raw legacy JS client instance to use.
 * @param endpoint Name of the API endpoint to call.
 * @param clientParams Parameters that will be directly passed to the
 * legacy JS client.
 * @param options Options that affect the way we call the API and process the result.
 * make wrap401Errors default to false, because we don't want login pop-up from browser
 */


const callAPI = async (client, endpoint, clientParams = {}, options = {
  wrap401Errors: false
}) => {
  const clientPath = endpoint.split('.');
  const api = (0, _lodash.get)(client, clientPath);

  if (!api) {
    throw new Error(`called with an invalid endpoint: ${endpoint}`);
  }

  const apiContext = clientPath.length === 1 ? client : (0, _lodash.get)(client, clientPath.slice(0, -1));

  try {
    return await new Promise((resolve, reject) => {
      const request = api.call(apiContext, clientParams);

      if (options.signal) {
        options.signal.addEventListener('abort', () => {
          request.abort();
          reject(new Error('Request was aborted'));
        });
      }

      return request.then(resolve, reject);
    });
  } catch (err) {
    if (!options.wrap401Errors || err.statusCode !== 401) {
      throw err;
    }

    throw _server.LegacyOpenSearchErrorHelpers.decorateNotAuthorizedError(err);
  }
};
/**
 * Get a legacy client that configured with basic auth
 *
 * @param rootClient Raw legacy client instance to use.
 * @param endpoint - String descriptor of the endpoint e.g. `cluster.getSettings` or `ping`.
 * @param clientParams - A dictionary of parameters that will be passed directly to the legacy JS client.
 * @param options - Options that affect the way we call the API and process the result.
 */


const getBasicAuthClient = async (rootClient, {
  endpoint,
  clientParams = {},
  options
}, {
  username,
  password
}) => {
  const headers = {
    authorization: 'Basic ' + Buffer.from(`${username}:${password}`).toString('base64')
  };
  clientParams.headers = Object.assign({}, clientParams.headers, headers);
  return await callAPI.bind(null, rootClient)(endpoint, clientParams, options);
};