import * as LSP from 'vscode-languageserver/node';
import { TextDocument } from 'vscode-languageserver-textdocument';
import * as Parser from 'web-tree-sitter';
/**
 * The Analyzer uses the Abstract Syntax Trees (ASTs) that are provided by
 * tree-sitter to find definitions, reference, etc.
 */
export default class Analyzer {
    private console;
    private includeAllWorkspaceSymbols;
    private parser;
    private uriToAnalyzedDocument;
    private workspaceFolder;
    constructor({ console, includeAllWorkspaceSymbols, parser, workspaceFolder, }: {
        console: LSP.RemoteConsole;
        includeAllWorkspaceSymbols?: boolean;
        parser: Parser;
        workspaceFolder: string | null;
    });
    /**
     * Initiates a background analysis of the files in the workspaceFolder to
     * enable features across files.
     *
     * NOTE that when the source aware feature is enabled files are also parsed
     * when they are found.
     */
    initiateBackgroundAnalysis({ backgroundAnalysisMaxFiles, globPattern, }: {
        backgroundAnalysisMaxFiles: number;
        globPattern: string;
    }): Promise<{
        filesParsed: number;
    }>;
    /**
     * Find all the locations where something has been defined.
     */
    findDefinition({ position, uri, word, }: {
        position?: {
            line: number;
            character: number;
        };
        uri: string;
        word: string;
    }): LSP.Location[];
    /**
     * Find all the symbols matching the query using fuzzy search.
     */
    search(query: string): LSP.SymbolInformation[];
    getExplainshellDocumentation({ params, endpoint, }: {
        params: LSP.TextDocumentPositionParams;
        endpoint: string;
    }): Promise<{
        helpHTML?: string;
    }>;
    /**
     * Find all the locations where something named name has been defined.
     */
    findReferences(name: string): LSP.Location[];
    /**
     * Find all occurrences of name in the given file.
     * It's currently not scope-aware.
     */
    findOccurrences(uri: string, query: string): LSP.Location[];
    /**
     * Find all symbol definitions in the given file.
     */
    findSymbolsForFile({ uri }: {
        uri: string;
    }): LSP.SymbolInformation[];
    /**
     * Find symbol completions for the given word.
     */
    findSymbolsMatchingWord({ exactMatch, uri, word, }: {
        exactMatch: boolean;
        uri: string;
        word: string;
    }): LSP.SymbolInformation[];
    /**
     * Analyze the given document, cache the tree-sitter AST, and iterate over the
     * tree to find declarations.
     *
     * Returns all, if any, syntax errors that occurred while parsing the file.
     *
     */
    analyze({ document, uri, }: {
        document: TextDocument;
        uri: string;
    }): LSP.Diagnostic[];
    findAllSourcedUris({ uri }: {
        uri: string;
    }): Set<string>;
    /**
     * Find the node at the given point.
     */
    private nodeAtPoint;
    /**
     * Find the full word at the given point.
     */
    wordAtPoint(uri: string, line: number, column: number): string | null;
    /**
     * Find the name of the command at the given point.
     */
    commandNameAtPoint(uri: string, line: number, column: number): string | null;
    /**
     * Find a block of comments above a line position
     */
    commentsAbove(uri: string, line: number): string | null;
    getAllVariableSymbols({ uri }: {
        uri: string;
    }): LSP.SymbolInformation[];
    setIncludeAllWorkspaceSymbols(includeAllWorkspaceSymbols: boolean): void;
    private getReachableUris;
    private getAllSymbols;
}
