unit tcbarcodes;

{$mode objfpc}{$H+}

interface

uses
  TypInfo, Classes, SysUtils, fpcunit, testutils, testregistry, fpbarcode;

type

  { TTestBarcodes }

  TTestBarcodes = class(TTestCase)
  private
    FText: String;
    FType: TBarcodeEncoding;
    class procedure AssertEquals(Msg: String; AExpected, AActual: TBarColor);
    class procedure AssertEquals(Msg: String; AExpected, AActual: TBarWidth);
    class procedure AssertEquals(Msg: String; AExpected, AActual: TBarHeight);
    procedure AssertParams(aType: TBarType; AColor: TBarColor; AWidth: TBarWidth; AHeight: TBarheight);
  Protected
    Procedure SetEncoding(aType: TBarcodeEncoding);
    Procedure SetText(aText : String);
    Function DoEncode : TBarTypeArray;
    Class Procedure AssertEquals(Msg : String; AExpected : Array of Byte; AActual : TBarTypeArray); overload;
    Property TheType : TBarcodeEncoding Read FType;
    Property TheText : String Read FText;
  published
    procedure TestEan8;
    procedure TestEan13;
    procedure TestCode39;
    procedure TestCode39Extended;
    procedure TestCode128A;
    procedure TestCode128B;
    procedure TestCode128C;
    procedure Testbe2of5industrial;
    procedure Testbe2of5interleaved;
    procedure Testbe2of5matrix;
    procedure TestCode93;
    procedure TestCode93Extended;
    procedure TestCodeCodabar;
    procedure TestCodeMSI;
    procedure TestCodePostNet;
    Procedure CheckParamsType;
    Procedure CheckNumerictypes;
  end;

implementation

procedure TTestBarcodes.SetEncoding(aType: TBarcodeEncoding);
begin
  FType:=aType;
end;

procedure TTestBarcodes.SetText(aText: String);
begin
  FText:=aText;
end;

function TTestBarcodes.DoEncode: TBarTypeArray;
begin
  Result:=StringToBarTypeArray(FText,FType);
  //Writeln(NewToOld(Result));
end;

class procedure TTestBarcodes.AssertEquals(Msg: String; AExpected: array of Byte; AActual: TBarTypeArray);

Var
  I : Integer;

begin
  AssertEquals(Msg+': length equal',Length(aExpected),Length(AActual));
  for I:=0 to Length(aExpected)-1 do
    AssertEquals(Msg+': Element at position ['+intTostr(I)+']',aExpected[i],AActual[i]);
end;

{
  The tests have been generated using the barcode unit distributed with lazarus.
  The string text in the comment is the string as reproduced with the MakeData function of the TBarCode class.
}

procedure TTestBarcodes.TestEan8;

begin
  SetEncoding(beEAN8);
  // 0 --> 5052605260526052605050507150715071507150505
  SetText('0');
  AssertEquals('Test for "0"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,6,1,4,0,6,1,4,0,4,0,4],DoEncode);
  // 1 --> 5052605260526052605050507150715061607150505
  SetText('1');
  AssertEquals('Test for "1"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,5,1,5,0,6,1,4,0,4,0,4],DoEncode);
  // 2 --> 5052605260526052605050507150715060617150505
  SetText('2');
  AssertEquals('Test for "2"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,5,0,5,1,6,1,4,0,4,0,4],DoEncode);
  // 3 --> 5052605260526052605050507150715053507150505
  SetText('3');
  AssertEquals('Test for "3"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,4,3,4,0,6,1,4,0,4,0,4],DoEncode);
  // 4 --> 5052605260526052605050507150715050717150505
  SetText('4');
  AssertEquals('Test for "4"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,4,0,6,1,6,1,4,0,4,0,4],DoEncode);
  // 5 --> 5052605260526052605050507150715051707150505
  SetText('5');
  AssertEquals('Test for "5"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,4,1,6,0,6,1,4,0,4,0,4],DoEncode);
  // 6 --> 5052605260526052605050507150715050537150505
  SetText('6');
  AssertEquals('Test for "6"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,4,0,4,3,6,1,4,0,4,0,4],DoEncode);
  // 7 --> 5052605260526052605050507150715052517150505
  SetText('7');
  AssertEquals('Test for "7"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,4,2,4,1,6,1,4,0,4,0,4],DoEncode);
  // 8 --> 5052605260526052605050507150715051527150505
  SetText('8');
  AssertEquals('Test for "8"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,4,1,4,2,6,1,4,0,4,0,4],DoEncode);
  // 9 --> 5052605260526052605050507150715070517150505
  SetText('9');
  AssertEquals('Test for "9"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,6,0,4,1,6,1,4,0,4,0,4],DoEncode);
  // 12345678 --> 5051615151608050526050505170505352515152505
  SetText('12345678');
  AssertEquals('Test for "12345678"',[4,0,4,1,5,1,4,1,4,1,5,0,7,0,4,0,4,2,5,0,4,0,4,0,4,1,6,0,4,0,4,3,4,2,4,1,4,1,4,2,4,0,4],DoEncode);
end;

procedure TTestBarcodes.TestEan13;
begin
  SetEncoding(beEAN13);
  // 0 --> 50526052605260526052605260505050715071507150715071507150505
  SetText('0');
  AssertEquals('Test for "0"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,6,1,4,0,6,1,4,0,6,1,4,0,6,1,4,0,4,0,4],DoEncode);
  // 1 --> 50526052605260526052605260505050715071507150715061607150505
  SetText('1');
  AssertEquals('Test for "1"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,6,1,4,0,6,1,4,0,5,1,5,0,6,1,4,0,4,0,4],DoEncode);
  // 2 --> 50526052605260526052605260505050715071507150715060617150505
  SetText('2');
  AssertEquals('Test for "2"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,6,1,4,0,6,1,4,0,5,0,5,1,6,1,4,0,4,0,4],DoEncode);
  // 3 --> 50526052605260526052605260505050715071507150715053507150505
  SetText('3');
  AssertEquals('Test for "3"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,6,1,4,0,6,1,4,0,4,3,4,0,6,1,4,0,4,0,4],DoEncode);
  // 4 --> 50526052605260526052605260505050715071507150715050717150505
  SetText('4');
  AssertEquals('Test for "4"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,6,1,4,0,6,1,4,0,4,0,6,1,6,1,4,0,4,0,4],DoEncode);
  // 5 --> 50526052605260526052605260505050715071507150715051707150505
  SetText('5');
  AssertEquals('Test for "5"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,6,1,4,0,6,1,4,0,4,1,6,0,6,1,4,0,4,0,4],DoEncode);
  // 6 --> 50526052605260526052605260505050715071507150715050537150505
  SetText('6');
  AssertEquals('Test for "6"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,6,1,4,0,6,1,4,0,4,0,4,3,6,1,4,0,4,0,4],DoEncode);
  // 7 --> 50526052605260526052605260505050715071507150715052517150505
  SetText('7');
  AssertEquals('Test for "7"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,6,1,4,0,6,1,4,0,4,2,4,1,6,1,4,0,4,0,4],DoEncode);
  // 8 --> 50526052605260526052605260505050715071507150715051527150505
  SetText('8');
  AssertEquals('Test for "8"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,6,1,4,0,6,1,4,0,4,1,4,2,6,1,4,0,4,0,4],DoEncode);
  // 9 --> 50526052605260526052605260505050715071507150715070517150505
  SetText('9');
  AssertEquals('Test for "9"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,2,5,0,4,0,4,0,4,0,6,1,4,0,6,1,4,0,6,1,4,0,6,1,4,0,6,0,4,1,6,1,4,0,4,0,4],DoEncode);
  // 12345678 --> 50526052605260516151516080505050507151705053525151527150505
  SetText('12345678');
  AssertEquals('Test for "12345678"',[4,0,4,2,5,0,4,2,5,0,4,2,5,0,4,1,5,1,4,1,4,1,5,0,7,0,4,0,4,0,4,0,4,0,6,1,4,1,6,0,4,0,4,3,4,2,4,1,4,1,4,2,6,1,4,0,4,0,4],DoEncode);
  // 1234567890123 --> 50515160805170506253505152505050515270517150616060615350505
  SetText('1234567890123');
  AssertEquals('Test for "1234567890123"',[4,0,4,1,4,1,5,0,7,0,4,1,6,0,4,0,5,2,4,3,4,0,4,1,4,2,4,0,4,0,4,0,4,1,4,2,6,0,4,1,6,1,4,0,5,1,5,0,5,0,5,1,4,3,4,0,4,0,4],DoEncode);
end;

procedure TTestBarcodes.TestCode39;
begin
  SetEncoding(be39);
  // 0 --> 51506060505051606050515060605
  SetText('0');
  AssertEquals('Test for "0"',[4,1,4,0,5,0,5,0,4,0,4,0,4,1,5,0,5,0,4,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 1 --> 51506060506051505060515060605
  SetText('1');
  AssertEquals('Test for "1"',[4,1,4,0,5,0,5,0,4,0,5,0,4,1,4,0,4,0,5,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 2 --> 51506060505061505060515060605
  SetText('2');
  AssertEquals('Test for "2"',[4,1,4,0,5,0,5,0,4,0,4,0,5,1,4,0,4,0,5,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 3 --> 51506060506061505050515060605
  SetText('3');
  AssertEquals('Test for "3"',[4,1,4,0,5,0,5,0,4,0,5,0,5,1,4,0,4,0,4,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 4 --> 51506060505051605060515060605
  SetText('4');
  AssertEquals('Test for "4"',[4,1,4,0,5,0,5,0,4,0,4,0,4,1,5,0,4,0,5,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 5 --> 51506060506051605050515060605
  SetText('5');
  AssertEquals('Test for "5"',[4,1,4,0,5,0,5,0,4,0,5,0,4,1,5,0,4,0,4,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 6 --> 51506060505061605050515060605
  SetText('6');
  AssertEquals('Test for "6"',[4,1,4,0,5,0,5,0,4,0,4,0,5,1,5,0,4,0,4,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 7 --> 51506060505051506060515060605
  SetText('7');
  AssertEquals('Test for "7"',[4,1,4,0,5,0,5,0,4,0,4,0,4,1,4,0,5,0,5,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 8 --> 51506060506051506050515060605
  SetText('8');
  AssertEquals('Test for "8"',[4,1,4,0,5,0,5,0,4,0,5,0,4,1,4,0,5,0,4,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 9 --> 51506060505061506050515060605
  SetText('9');
  AssertEquals('Test for "9"',[4,1,4,0,5,0,5,0,4,0,4,0,5,1,4,0,5,0,4,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 12345678 --> 515060605060515050605061505060606150505050516050606051605050506160505050515060606051506050515060605
  SetText('12345678');
  AssertEquals('Test for "12345678"',[4,1,4,0,5,0,5,0,4,0,5,0,4,1,4,0,4,0,5,0,4,0,5,1,4,0,4,0,5,0,5,0,5,1,4,0,4,0,4,0,4,0,4,1,5,0,4,0,5,0,5,0,4,1,5,0,4,0,4,0,4,0,5,1,5,0,4,0,4,0,4,0,4,1,4,0,5,0,5,0,5,0,4,1,4,0,5,0,4,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 123456789ABCDE --> 515060605060515050605061505060606150505050516050606051605050506160505050515060606051506050506150605060505150605060515060606051505050506150606050615050515060605
  SetText('123456789ABCDE');
  AssertEquals('Test for "123456789ABCDE"',[4,1,4,0,5,0,5,0,4,0,5,0,4,1,4,0,4,0,5,0,4,0,5,1,4,0,4,0,5,0,5,0,5,1,4,0,4,0,4,0,4,0,4,1,5,0,4,0,5,0,5,0,4,1,5,0,4,0,4,0,4,0,5,1,5,0,4,0,4,0,4,0,4,1,4,0,5,0,5,0,5,0,4,1,4,0,5,0,4,0,4,0,5,1,4,0,5,0,4,0,5,0,4,0,4,1,4,0,5,0,4,0,5,0,4,1,4,0,5,0,5,0,5,0,4,1,4,0,4,0,4,0,4,0,5,1,4,0,5,0,5,0,4,0,5,1,4,0,4,0,4,1,4,0,5,0,5,0,4],DoEncode);
end;

procedure TTestBarcodes.TestCode39Extended;

begin
  SetEncoding(be39extended);
  // 0 --> 51506060505051606050515060605
  SetText('0');
  AssertEquals('Test for "0"',[4,1,4,0,5,0,5,0,4,0,4,0,4,1,5,0,5,0,4,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 1 --> 51506060506051505060515060605
  SetText('1');
  AssertEquals('Test for "1"',[4,1,4,0,5,0,5,0,4,0,5,0,4,1,4,0,4,0,5,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 2 --> 51506060505061505060515060605
  SetText('2');
  AssertEquals('Test for "2"',[4,1,4,0,5,0,5,0,4,0,4,0,5,1,4,0,4,0,5,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 3 --> 51506060506061505050515060605
  SetText('3');
  AssertEquals('Test for "3"',[4,1,4,0,5,0,5,0,4,0,5,0,5,1,4,0,4,0,4,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 4 --> 51506060505051605060515060605
  SetText('4');
  AssertEquals('Test for "4"',[4,1,4,0,5,0,5,0,4,0,4,0,4,1,5,0,4,0,5,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 5 --> 51506060506051605050515060605
  SetText('5');
  AssertEquals('Test for "5"',[4,1,4,0,5,0,5,0,4,0,5,0,4,1,5,0,4,0,4,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 6 --> 51506060505061605050515060605
  SetText('6');
  AssertEquals('Test for "6"',[4,1,4,0,5,0,5,0,4,0,4,0,5,1,5,0,4,0,4,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 7 --> 51506060505051506060515060605
  SetText('7');
  AssertEquals('Test for "7"',[4,1,4,0,5,0,5,0,4,0,4,0,4,1,4,0,5,0,5,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 8 --> 51506060506051506050515060605
  SetText('8');
  AssertEquals('Test for "8"',[4,1,4,0,5,0,5,0,4,0,5,0,4,1,4,0,5,0,4,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 9 --> 51506060505061506050515060605
  SetText('9');
  AssertEquals('Test for "9"',[4,1,4,0,5,0,5,0,4,0,4,0,5,1,4,0,5,0,4,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 1234567800 --> 51506060506051505060506150506060615050505051605060605160505050616050505051506060605150605050516060505051606050515060605
  SetText('1234567800');
  AssertEquals('Test for "1234567800"',[4,1,4,0,5,0,5,0,4,0,5,0,4,1,4,0,4,0,5,0,4,0,5,1,4,0,4,0,5,0,5,0,5,1,4,0,4,0,4,0,4,0,4,1,5,0,4,0,5,0,5,0,4,1,5,0,4,0,4,0,4,0,5,1,5,0,4,0,4,0,4,0,4,1,4,0,5,0,5,0,5,0,4,1,4,0,5,0,4,0,4,0,4,1,5,0,5,0,4,0,4,0,4,1,5,0,5,0,4,0,4,1,4,0,5,0,5,0,4],DoEncode);
  // 123456789ABCDEwxyz --> 51506060506051505060506150506060615050505051605060605160505050616050505051506060605150605050615060506050515060506051506060605150505050615060605061505051505151506160505050515051515051506050605150515150615060505051505151505160605050515060605
  SetText('123456789ABCDEwxyz');
  AssertEquals('Test for "123456789ABCDEwxyz"',[4,1,4,0,5,0,5,0,4,0,5,0,4,1,4,0,4,0,5,0,4,0,5,1,4,0,4,0,5,0,5,0,5,1,4,0,4,0,4,0,4,0,4,1,5,0,4,0,5,0,5,0,4,1,5,0,4,0,4,0,4,0,5,1,5,0,4,0,4,0,4,0,4,1,4,0,5,0,5,0,5,0,4,1,4,0,5,0,4,0,4,0,5,1,4,0,5,0,4,0,5,0,4,0,4,1,4,0,5,0,4,0,5,0,4,1,4,0,5,0,5,0,5,0,4,1,4,0,4,0,4,0,4,0,5,1,4,0,5,0,5,0,4,0,5,1,4,0,4,0,4,1,4,0,4,1,4,1,4,0,5,1,5,0,4,0,4,0,4,0,4,1,4,0,4,1,4,1,4,0,4,1,4,0,5,0,4,0,5,0,4,1,4,0,4,1,4,1,4,0,5,1,4,0,5,0,4,0,4,0,4,1,4,0,4,1,4,1,4,0,4,1,5,0,5,0,4,0,4,0,4,1,4,0,5,0,5,0,4],DoEncode);
end;

Procedure TTestBarcodes.TestCode128A;


begin
  SetEncoding(be128A);
  // 0 --> 6053515170615170616270506
  SetText('0');
  AssertEquals('Test for "0"',[5,0,4,3,4,1,4,1,6,0,5,1,4,1,6,0,5,1,5,2,6,0,4,0,5],DoEncode);
  // 1 --> 6053515171605171606270506
  SetText('1');
  AssertEquals('Test for "1"',[5,0,4,3,4,1,4,1,6,1,5,0,4,1,6,1,5,0,5,2,6,0,4,0,5],DoEncode);
  // 2 --> 6053516171506171506270506
  SetText('2');
  AssertEquals('Test for "2"',[5,0,4,3,4,1,5,1,6,1,4,0,5,1,6,1,4,0,5,2,6,0,4,0,5],DoEncode);
  // 3 --> 6053516150716150716270506
  SetText('3');
  AssertEquals('Test for "3"',[5,0,4,3,4,1,5,1,4,0,6,1,5,1,4,0,6,1,5,2,6,0,4,0,5],DoEncode);
  // 4 --> 6053516151706151706270506
  SetText('4');
  AssertEquals('Test for "4"',[5,0,4,3,4,1,5,1,4,1,6,0,5,1,4,1,6,0,5,2,6,0,4,0,5],DoEncode);
  // 5 --> 6053516071516071516270506
  SetText('5');
  AssertEquals('Test for "5"',[5,0,4,3,4,1,5,0,6,1,4,1,5,0,6,1,4,1,5,2,6,0,4,0,5],DoEncode);
  // 6 --> 6053516170516170516270506
  SetText('6');
  AssertEquals('Test for "6"',[5,0,4,3,4,1,5,1,6,0,4,1,5,1,6,0,4,1,5,2,6,0,4,0,5],DoEncode);
  // 7 --> 6053517060707060706270506
  SetText('7');
  AssertEquals('Test for "7"',[5,0,4,3,4,1,6,0,5,0,6,0,6,0,5,0,6,0,5,2,6,0,4,0,5],DoEncode);
  // 8 --> 6053517051617051616270506
  SetText('8');
  AssertEquals('Test for "8"',[5,0,4,3,4,1,6,0,4,1,5,1,6,0,4,1,5,1,5,2,6,0,4,0,5],DoEncode);
  // 9 --> 6053517150617150616270506
  SetText('9');
  AssertEquals('Test for "9"',[5,0,4,3,4,1,6,1,4,0,5,1,6,1,4,0,5,1,5,2,6,0,4,0,5],DoEncode);
  // 1234567800 --> 6053515171606171506150716151706071516170517060707051615170615170617050626270506
  SetText('1234567800');
  AssertEquals('Test for "1234567800"',[5,0,4,3,4,1,4,1,6,1,5,0,5,1,6,1,4,0,5,1,4,0,6,1,5,1,4,1,6,0,5,0,6,1,4,1,5,1,6,0,4,1,6,0,5,0,6,0,6,0,4,1,5,1,4,1,6,0,5,1,4,1,6,0,5,1,6,0,4,0,5,2,5,2,6,0,4,0,5],DoEncode);
  // 123456789ABCD --> 6053515171606171506150716151706071516170517060707051617150615052625250625252605062526170516270506
  SetText('123456789ABCD');
  AssertEquals('Test for "123456789ABCD"',[5,0,4,3,4,1,4,1,6,1,5,0,5,1,6,1,4,0,5,1,4,0,6,1,5,1,4,1,6,0,5,0,6,1,4,1,5,1,6,0,4,1,6,0,5,0,6,0,6,0,4,1,5,1,6,1,4,0,5,1,4,0,4,2,5,2,4,2,4,0,5,2,4,2,4,2,5,0,4,0,5,2,4,2,5,1,6,0,4,1,5,2,6,0,4,0,5],DoEncode);
end;

procedure TTestBarcodes.TestCode128B;

begin
  SetEncoding(be128B);
  // 0 --> 6051535170615171606270506
  SetText('0');
  AssertEquals('Test for "0"',[5,0,4,1,4,3,4,1,6,0,5,1,4,1,6,1,5,0,5,2,6,0,4,0,5],DoEncode);
  // 1 --> 6051535171606171506270506
  SetText('1');
  AssertEquals('Test for "1"',[5,0,4,1,4,3,4,1,6,1,5,0,5,1,6,1,4,0,5,2,6,0,4,0,5],DoEncode);
  // 2 --> 6051536171506150716270506
  SetText('2');
  AssertEquals('Test for "2"',[5,0,4,1,4,3,5,1,6,1,4,0,5,1,4,0,6,1,5,2,6,0,4,0,5],DoEncode);
  // 3 --> 6051536150716151706270506
  SetText('3');
  AssertEquals('Test for "3"',[5,0,4,1,4,3,5,1,4,0,6,1,5,1,4,1,6,0,5,2,6,0,4,0,5],DoEncode);
  // 4 --> 6051536151706071516270506
  SetText('4');
  AssertEquals('Test for "4"',[5,0,4,1,4,3,5,1,4,1,6,0,5,0,6,1,4,1,5,2,6,0,4,0,5],DoEncode);
  // 5 --> 6051536071516170516270506
  SetText('5');
  AssertEquals('Test for "5"',[5,0,4,1,4,3,5,0,6,1,4,1,5,1,6,0,4,1,5,2,6,0,4,0,5],DoEncode);
  // 6 --> 6051536170517060706270506
  SetText('6');
  AssertEquals('Test for "6"',[5,0,4,1,4,3,5,1,6,0,4,1,6,0,5,0,6,0,5,2,6,0,4,0,5],DoEncode);
  // 7 --> 6051537060707051616270506
  SetText('7');
  AssertEquals('Test for "7"',[5,0,4,1,4,3,6,0,5,0,6,0,6,0,4,1,5,1,5,2,6,0,4,0,5],DoEncode);
  // 8 --> 6051537051617150616270506
  SetText('8');
  AssertEquals('Test for "8"',[5,0,4,1,4,3,6,0,4,1,5,1,6,1,4,0,5,1,5,2,6,0,4,0,5],DoEncode);
  // 9 --> 6051537150617151606270506
  SetText('9');
  AssertEquals('Test for "9"',[5,0,4,1,4,3,6,1,4,0,5,1,6,1,4,1,5,0,5,2,6,0,4,0,5],DoEncode);
  // 1234567800 --> 6051535171606171506150716151706071516170517060707051615170615170617052606270506
  SetText('1234567800');
  AssertEquals('Test for "1234567800"',[5,0,4,1,4,3,4,1,6,1,5,0,5,1,6,1,4,0,5,1,4,0,6,1,5,1,4,1,6,0,5,0,6,1,4,1,5,1,6,0,4,1,6,0,5,0,6,0,6,0,4,1,5,1,4,1,6,0,5,1,4,1,6,0,5,1,6,0,4,2,5,0,5,2,6,0,4,0,5],DoEncode);
  // 123456789ABCD --> 6051535171606171506150716151706071516170517060707051617150615052625250625252605062527060706270506
  SetText('123456789ABCD');
  AssertEquals('Test for "123456789ABCD"',[5,0,4,1,4,3,4,1,6,1,5,0,5,1,6,1,4,0,5,1,4,0,6,1,5,1,4,1,6,0,5,0,6,1,4,1,5,1,6,0,4,1,6,0,5,0,6,0,6,0,4,1,5,1,6,1,4,0,5,1,4,0,4,2,5,2,4,2,4,0,5,2,4,2,4,2,5,0,4,0,5,2,4,2,6,0,5,0,6,0,5,2,6,0,4,0,5],DoEncode);
end;


Procedure TTestBarcodes.TestCode128C;

begin
  SetEncoding(be128C);
  // 0 --> 6051716061616161606270506
  SetText('0');
  AssertEquals('Test for "0"',[5,0,4,1,6,1,5,0,5,1,5,1,5,1,5,1,5,0,5,2,6,0,4,0,5],DoEncode);
  // 1 --> 6051716160615151626270506
  SetText('1');
  AssertEquals('Test for "1"',[5,0,4,1,6,1,5,1,5,0,5,1,4,1,4,1,5,2,5,2,6,0,4,0,5],DoEncode);
  // 2 --> 6051716161605152616270506
  SetText('2');
  AssertEquals('Test for "2"',[5,0,4,1,6,1,5,1,5,1,5,0,4,1,4,2,5,1,5,2,6,0,4,0,5],DoEncode);
  // 3 --> 6051715151625251616270506
  SetText('3');
  AssertEquals('Test for "3"',[5,0,4,1,6,1,4,1,4,1,5,2,4,2,4,1,5,1,5,2,6,0,4,0,5],DoEncode);
  // 4 --> 6051715152615161526270506
  SetText('4');
  AssertEquals('Test for "4"',[5,0,4,1,6,1,4,1,4,2,5,1,4,1,5,1,4,2,5,2,6,0,4,0,5],DoEncode);
  // 5 --> 6051715251615162516270506
  SetText('5');
  AssertEquals('Test for "5"',[5,0,4,1,6,1,4,2,4,1,5,1,4,1,5,2,4,1,5,2,6,0,4,0,5],DoEncode);
  // 6 --> 6051715161525261516270506
  SetText('6');
  AssertEquals('Test for "6"',[5,0,4,1,6,1,4,1,5,1,4,2,4,2,5,1,4,1,5,2,6,0,4,0,5],DoEncode);
  // 7 --> 6051715162516151526270506
  SetText('7');
  AssertEquals('Test for "7"',[5,0,4,1,6,1,4,1,5,2,4,1,5,1,4,1,4,2,5,2,6,0,4,0,5],DoEncode);
  // 8 --> 6051715261516152516270506
  SetText('8');
  AssertEquals('Test for "8"',[5,0,4,1,6,1,4,2,5,1,4,1,5,1,4,2,4,1,5,2,6,0,4,0,5],DoEncode);
  // 9 --> 6051716151526251516270506
  SetText('9');
  AssertEquals('Test for "9"',[5,0,4,1,6,1,5,1,4,1,4,2,5,2,4,1,4,1,5,2,6,0,4,0,5],DoEncode);
  // 1234567800 --> 6051715061715250627250606350516061615270606270506
  SetText('1234567800');
  AssertEquals('Test for "1234567800"',[5,0,4,1,6,1,4,0,5,1,6,1,4,2,4,0,5,2,6,2,4,0,5,0,5,3,4,0,4,1,5,0,5,1,5,1,4,2,6,0,5,0,5,2,6,0,4,0,5],DoEncode);
end;


Procedure TTestBarcodes.Testbe2of5industrial;

begin
  SetEncoding(be2of5industrial);
  // 00 --> 60605050506060505050606050605060
  SetText('00');
  AssertEquals('Test for "00"',[5,0,5,0,4,0,4,0,4,0,5,0,5,0,4,0,4,0,4,0,5,0,5,0,4,0,5,0,4,0,5,0],DoEncode);
  // 11 --> 60605060505050606050505060605060
  SetText('11');
  AssertEquals('Test for "11"',[5,0,5,0,4,0,5,0,4,0,4,0,4,0,5,0,5,0,4,0,4,0,4,0,5,0,5,0,4,0,5,0],DoEncode);
  // 22 --> 60605050605050605060505060605060
  SetText('22');
  AssertEquals('Test for "22"',[5,0,5,0,4,0,4,0,5,0,4,0,4,0,5,0,4,0,5,0,4,0,4,0,5,0,5,0,4,0,5,0],DoEncode);
  // 33 --> 60605060605050506060505050605060
  SetText('33');
  AssertEquals('Test for "33"',[5,0,5,0,4,0,5,0,5,0,4,0,4,0,4,0,5,0,5,0,4,0,4,0,4,0,5,0,4,0,5,0],DoEncode);
  // 44 --> 60605050506050605050605060605060
  SetText('44');
  AssertEquals('Test for "44"',[5,0,5,0,4,0,4,0,4,0,5,0,4,0,5,0,4,0,4,0,5,0,4,0,5,0,5,0,4,0,5,0],DoEncode);
  // 55 --> 60605060506050506050605050605060
  SetText('55');
  AssertEquals('Test for "55"',[5,0,5,0,4,0,5,0,4,0,5,0,4,0,4,0,5,0,4,0,5,0,4,0,4,0,5,0,4,0,5,0],DoEncode);
  // 66 --> 60605050606050505060605050605060
  SetText('66');
  AssertEquals('Test for "66"',[5,0,5,0,4,0,4,0,5,0,5,0,4,0,4,0,4,0,5,0,5,0,4,0,4,0,5,0,4,0,5,0],DoEncode);
  // 77 --> 60605050505060605050506060605060
  SetText('77');
  AssertEquals('Test for "77"',[5,0,5,0,4,0,4,0,4,0,4,0,5,0,5,0,4,0,4,0,4,0,5,0,5,0,5,0,4,0,5,0],DoEncode);
  // 88 --> 60605060505060506050506050605060
  SetText('88');
  AssertEquals('Test for "88"',[5,0,5,0,4,0,5,0,4,0,4,0,5,0,4,0,5,0,4,0,4,0,5,0,4,0,5,0,4,0,5,0],DoEncode);
  // 99 --> 60605050605060505060506050605060
  SetText('99');
  AssertEquals('Test for "99"',[5,0,5,0,4,0,4,0,5,0,4,0,5,0,4,0,4,0,5,0,4,0,5,0,4,0,5,0,4,0,5,0],DoEncode);
  // 1234567800 --> 6060506050505060506050506060605050505050605060605060505050606050505050506060605050605050506060505050606050605060
  SetText('1234567800');
  AssertEquals('Test for "1234567800"',[5,0,5,0,4,0,5,0,4,0,4,0,4,0,5,0,4,0,5,0,4,0,4,0,5,0,5,0,5,0,4,0,4,0,4,0,4,0,4,0,5,0,4,0,5,0,5,0,4,0,5,0,4,0,4,0,4,0,5,0,5,0,4,0,4,0,4,0,4,0,4,0,5,0,5,0,5,0,4,0,4,0,5,0,4,0,4,0,4,0,5,0,5,0,4,0,4,0,4,0,5,0,5,0,4,0,5,0,4,0,5,0],DoEncode);
end;


Procedure TTestBarcodes.Testbe2of5interleaved;

begin
  SetEncoding(be2of5interleaved);
   // 00 --> 50505050616150605
   SetText('00');
   AssertEquals('Test for "00"',[4,0,4,0,4,0,4,0,5,1,5,1,4,0,5,0,4],DoEncode);
   // 11 --> 50506150505061605
   SetText('11');
   AssertEquals('Test for "11"',[4,0,4,0,5,1,4,0,4,0,4,0,5,1,5,0,4],DoEncode);
   // 22 --> 50505061505061605
   SetText('22');
   AssertEquals('Test for "22"',[4,0,4,0,4,0,5,1,4,0,4,0,5,1,5,0,4],DoEncode);
   // 33 --> 50506161505050605
   SetText('33');
   AssertEquals('Test for "33"',[4,0,4,0,5,1,5,1,4,0,4,0,4,0,5,0,4],DoEncode);
   // 44 --> 50505050615061605
   SetText('44');
   AssertEquals('Test for "44"',[4,0,4,0,4,0,4,0,5,1,4,0,5,1,5,0,4],DoEncode);
   // 55 --> 50506150615050605
   SetText('55');
   AssertEquals('Test for "55"',[4,0,4,0,5,1,4,0,5,1,4,0,4,0,5,0,4],DoEncode);
   // 66 --> 50505061615050605
   SetText('66');
   AssertEquals('Test for "66"',[4,0,4,0,4,0,5,1,5,1,4,0,4,0,5,0,4],DoEncode);
   // 77 --> 50505050506161605
   SetText('77');
   AssertEquals('Test for "77"',[4,0,4,0,4,0,4,0,4,0,5,1,5,1,5,0,4],DoEncode);
   // 88 --> 50506150506150605
   SetText('88');
   AssertEquals('Test for "88"',[4,0,4,0,5,1,4,0,4,0,5,1,4,0,5,0,4],DoEncode);
   // 99 --> 50505061506150605
   SetText('99');
   AssertEquals('Test for "99"',[4,0,4,0,4,0,5,1,4,0,5,1,4,0,5,0,4],DoEncode);
   // 1234567800 --> 505060515050616060515051605161505051505061605050616150605
   SetText('1234567800');
   AssertEquals('Test for "1234567800"',[4,0,4,0,5,0,4,1,4,0,4,0,5,1,5,0,5,0,4,1,4,0,4,1,5,0,4,1,5,1,4,0,4,0,4,1,4,0,4,0,5,1,5,0,4,0,4,0,5,1,5,1,4,0,5,0,4],DoEncode);
end;


Procedure TTestBarcodes.Testbe2of5matrix;

begin
  SetEncoding(be2of5matrix);
  // 00 --> 70505050615050615070505
  SetText('00');
  AssertEquals('Test for "00"',[6,0,4,0,4,0,4,0,5,1,4,0,4,0,5,1,4,0,6,0,4,0,4],DoEncode);
  // 11 --> 70505060506060506070505
  SetText('11');
  AssertEquals('Test for "11"',[6,0,4,0,4,0,5,0,4,0,5,0,5,0,4,0,5,0,6,0,4,0,4],DoEncode);
  // 22 --> 70505051506051506070505
  SetText('22');
  AssertEquals('Test for "22"',[6,0,4,0,4,0,4,1,4,0,5,0,4,1,4,0,5,0,6,0,4,0,4],DoEncode);
  // 33 --> 70505061505061505070505
  SetText('33');
  AssertEquals('Test for "33"',[6,0,4,0,4,0,5,1,4,0,4,0,5,1,4,0,4,0,6,0,4,0,4],DoEncode);
  // 44 --> 70505050606050606070505
  SetText('44');
  AssertEquals('Test for "44"',[6,0,4,0,4,0,4,0,5,0,5,0,4,0,5,0,5,0,6,0,4,0,4],DoEncode);
  // 55 --> 70505060605060605070505
  SetText('55');
  AssertEquals('Test for "55"',[6,0,4,0,4,0,5,0,5,0,4,0,5,0,5,0,4,0,6,0,4,0,4],DoEncode);
  // 66 --> 70505051605051605070505
  SetText('66');
  AssertEquals('Test for "66"',[6,0,4,0,4,0,4,1,5,0,4,0,4,1,5,0,4,0,6,0,4,0,4],DoEncode);
  // 77 --> 70505050516050516070505
  SetText('77');
  AssertEquals('Test for "77"',[6,0,4,0,4,0,4,0,4,1,5,0,4,0,4,1,5,0,6,0,4,0,4],DoEncode);
  // 88 --> 70505060515060515070505
  SetText('88');
  AssertEquals('Test for "88"',[6,0,4,0,4,0,5,0,4,1,4,0,5,0,4,1,4,0,6,0,4,0,4],DoEncode);
  // 99 --> 70505051515051515070505
  SetText('99');
  AssertEquals('Test for "99"',[6,0,4,0,4,0,4,1,4,1,4,0,4,1,4,1,4,0,6,0,4,0,4],DoEncode);
  // 1234567800 --> 70505060506051506061505050606060605051605050516060515050615050615070505
  SetText('1234567800');
  AssertEquals('Test for "1234567800"',[6,0,4,0,4,0,5,0,4,0,5,0,4,1,4,0,5,0,5,1,4,0,4,0,4,0,5,0,5,0,5,0,5,0,4,0,4,1,5,0,4,0,4,0,4,1,5,0,5,0,4,1,4,0,4,0,5,1,4,0,4,0,5,1,4,0,6,0,4,0,4],DoEncode);
end;


Procedure TTestBarcodes.TestCode93;

begin
  SetEncoding(be93);
  // 0 --> 5050805250515250515250515050805
  SetText('0');
  AssertEquals('Test for "0"',[4,0,4,0,7,0,4,2,4,0,4,1,4,2,4,0,4,1,4,2,4,0,4,1,4,0,4,0,7,0,4],DoEncode);
  // 1 --> 5050805051525051525053505050805
  SetText('1');
  AssertEquals('Test for "1"',[4,0,4,0,7,0,4,0,4,1,4,2,4,0,4,1,4,2,4,0,4,3,4,0,4,0,4,0,7,0,4],DoEncode);
  // 2 --> 5050805052515052515152505050805
  SetText('2');
  AssertEquals('Test for "2"',[4,0,4,0,7,0,4,0,4,2,4,1,4,0,4,2,4,1,4,1,4,2,4,0,4,0,4,0,7,0,4],DoEncode);
  // 3 --> 5050805053505053505350505050805
  SetText('3');
  AssertEquals('Test for "3"',[4,0,4,0,7,0,4,0,4,3,4,0,4,0,4,3,4,0,4,3,4,0,4,0,4,0,4,0,7,0,4],DoEncode);
  // 4 --> 5050805150525150526052505050805
  SetText('4');
  AssertEquals('Test for "4"',[4,0,4,0,7,0,4,1,4,0,4,2,4,1,4,0,4,2,5,0,4,2,4,0,4,0,4,0,7,0,4],DoEncode);
  // 5 --> 5050805151515151516250505050805
  SetText('5');
  AssertEquals('Test for "5"',[4,0,4,0,7,0,4,1,4,1,4,1,4,1,4,1,4,1,5,2,4,0,4,0,4,0,4,0,7,0,4],DoEncode);
  // 6 --> 5050805152505152505062505050805
  SetText('6');
  AssertEquals('Test for "6"',[4,0,4,0,7,0,4,1,4,2,4,0,4,1,4,2,4,0,4,0,5,2,4,0,4,0,4,0,7,0,4],DoEncode);
  // 7 --> 5050805050535050535050625050805
  SetText('7');
  AssertEquals('Test for "7"',[4,0,4,0,7,0,4,0,4,0,4,3,4,0,4,0,4,3,4,0,4,0,5,2,4,0,4,0,7,0,4],DoEncode);
  // 8 --> 5050805251505251505150615050805
  SetText('8');
  AssertEquals('Test for "8"',[4,0,4,0,7,0,4,2,4,1,4,0,4,2,4,1,4,0,4,1,4,0,5,1,4,0,4,0,7,0,4],DoEncode);
  // 9 --> 5050805350505350506061505050805
  SetText('9');
  AssertEquals('Test for "9"',[4,0,4,0,7,0,4,3,4,0,4,0,4,3,4,0,4,0,5,0,5,1,4,0,4,0,4,0,7,0,4],DoEncode);
  // 1234567800 --> 5050805051525052515053505150525151515152505050535251505250515250515150527060505050805
  SetText('1234567800');
  AssertEquals('Test for "1234567800"',[4,0,4,0,7,0,4,0,4,1,4,2,4,0,4,2,4,1,4,0,4,3,4,0,4,1,4,0,4,2,4,1,4,1,4,1,4,1,4,2,4,0,4,0,4,0,4,3,4,2,4,1,4,0,4,2,4,0,4,1,4,2,4,0,4,1,4,1,4,0,4,2,6,0,5,0,4,0,4,0,4,0,7,0,4],DoEncode);
  // 123456789ABCD --> 5050805051525052515053505150525151515152505050535251505350506050526051516052506150515060616151505050805
  SetText('123456789ABCD');
  AssertEquals('Test for "123456789ABCD"',[4,0,4,0,7,0,4,0,4,1,4,2,4,0,4,2,4,1,4,0,4,3,4,0,4,1,4,0,4,2,4,1,4,1,4,1,4,1,4,2,4,0,4,0,4,0,4,3,4,2,4,1,4,0,4,3,4,0,4,0,5,0,4,0,4,2,5,0,4,1,4,1,5,0,4,2,4,0,5,1,4,0,4,1,4,0,5,0,5,1,5,1,4,1,4,0,4,0,4,0,7,0,4],DoEncode);
end;


Procedure TTestBarcodes.TestCode93Extended;

begin
  SetEncoding(be93Extended);
  // 00 --> 5050805250515250515250515250515050805
  SetText('00');
  AssertEquals('Test for "00"',[4,0,4,0,7,0,4,2,4,0,4,1,4,2,4,0,4,1,4,2,4,0,4,1,4,2,4,0,4,1,4,0,4,0,7,0,4],DoEncode);
  // 11 --> 5050805051525051525053505251505050805
  SetText('11');
  AssertEquals('Test for "11"',[4,0,4,0,7,0,4,0,4,1,4,2,4,0,4,1,4,2,4,0,4,3,4,0,4,2,4,1,4,0,4,0,4,0,7,0,4],DoEncode);
  // 22 --> 5050805052515052515152505060525050805
  SetText('22');
  AssertEquals('Test for "22"',[4,0,4,0,7,0,4,0,4,2,4,1,4,0,4,2,4,1,4,1,4,2,4,0,4,0,5,0,4,2,4,0,4,0,7,0,4],DoEncode);
  // 33 --> 5050805053505053505350505150615050805
  SetText('33');
  AssertEquals('Test for "33"',[4,0,4,0,7,0,4,0,4,3,4,0,4,0,4,3,4,0,4,3,4,0,4,0,4,1,4,0,5,1,4,0,4,0,7,0,4],DoEncode);
  // 44 --> 5050805150525150526052505060615050805
  SetText('44');
  AssertEquals('Test for "44"',[4,0,4,0,7,0,4,1,4,0,4,2,4,1,4,0,4,2,5,0,4,2,4,0,4,0,5,0,5,1,4,0,4,0,7,0,4],DoEncode);
  // 55 --> 5050805151515151516250505060705050805
  SetText('55');
  AssertEquals('Test for "55"',[4,0,4,0,7,0,4,1,4,1,4,1,4,1,4,1,4,1,5,2,4,0,4,0,4,0,5,0,6,0,4,0,4,0,7,0,4],DoEncode);
  // 66 --> 5050805152505152505062505051525050805
  SetText('66');
  AssertEquals('Test for "66"',[4,0,4,0,7,0,4,1,4,2,4,0,4,1,4,2,4,0,4,0,5,2,4,0,4,0,4,1,4,2,4,0,4,0,7,0,4],DoEncode);
  // 77 --> 5050805050535050535050625350505050805
  SetText('77');
  AssertEquals('Test for "77"',[4,0,4,0,7,0,4,0,4,0,4,3,4,0,4,0,4,3,4,0,4,0,5,2,4,3,4,0,4,0,4,0,4,0,7,0,4],DoEncode);
  // 88 --> 5050805251505251505150615061515050805
  SetText('88');
  AssertEquals('Test for "88"',[4,0,4,0,7,0,4,2,4,1,4,0,4,2,4,1,4,0,4,1,4,0,5,1,4,0,5,1,4,1,4,0,4,0,7,0,4],DoEncode);
  // 99 --> 5050805350505350506061505250605050805
  SetText('99');
  AssertEquals('Test for "99"',[4,0,4,0,7,0,4,3,4,0,4,0,4,3,4,0,4,0,5,0,5,1,4,0,4,2,4,0,5,0,4,0,4,0,7,0,4],DoEncode);
  // 1234567800 --> 5050805051525052515053505150525151515152505050535251505250515250515150527060505050805
  SetText('1234567800');
  AssertEquals('Test for "1234567800"',[4,0,4,0,7,0,4,0,4,1,4,2,4,0,4,2,4,1,4,0,4,3,4,0,4,1,4,0,4,2,4,1,4,1,4,1,4,1,4,2,4,0,4,0,4,0,4,3,4,2,4,1,4,0,4,2,4,0,4,1,4,2,4,0,4,1,4,1,4,0,4,2,6,0,5,0,4,0,4,0,4,0,7,0,4],DoEncode);
  // 123456789ABCD --> 5050805051525052515053505150525151515152505050535251505350506050526051516052506150515060616151505050805
  SetText('123456789ABCD');
  AssertEquals('Test for "123456789ABCD"',[4,0,4,0,7,0,4,0,4,1,4,2,4,0,4,2,4,1,4,0,4,3,4,0,4,1,4,0,4,2,4,1,4,1,4,1,4,1,4,2,4,0,4,0,4,0,4,3,4,2,4,1,4,0,4,3,4,0,4,0,5,0,4,0,4,2,5,0,4,1,4,1,5,0,4,2,4,0,5,1,4,0,4,1,4,0,5,0,5,1,5,1,4,1,4,0,4,0,4,0,7,0,4],DoEncode);
end;


Procedure TTestBarcodes.TestCodeCodabar;

begin
  SetEncoding(beCodabar);
  // 0 --> 50615150505051605151506
  SetText('0');
  AssertEquals('Test for "0"',[4,0,5,1,4,1,4,0,4,0,4,0,4,1,5,0,4,1,4,1,4,0,5],DoEncode);
  // 1 --> 50615150505061505151506
  SetText('1');
  AssertEquals('Test for "1"',[4,0,5,1,4,1,4,0,4,0,4,0,5,1,4,0,4,1,4,1,4,0,5],DoEncode);
  // 2 --> 50615150505150605151506
  SetText('2');
  AssertEquals('Test for "2"',[4,0,5,1,4,1,4,0,4,0,4,1,4,0,5,0,4,1,4,1,4,0,5],DoEncode);
  // 3 --> 50615150615050505151506
  SetText('3');
  AssertEquals('Test for "3"',[4,0,5,1,4,1,4,0,5,1,4,0,4,0,4,0,4,1,4,1,4,0,5],DoEncode);
  // 4 --> 50615150506051505151506
  SetText('4');
  AssertEquals('Test for "4"',[4,0,5,1,4,1,4,0,4,0,5,0,4,1,4,0,4,1,4,1,4,0,5],DoEncode);
  // 5 --> 50615150605051505151506
  SetText('5');
  AssertEquals('Test for "5"',[4,0,5,1,4,1,4,0,5,0,4,0,4,1,4,0,4,1,4,1,4,0,5],DoEncode);
  // 6 --> 50615150515050605151506
  SetText('6');
  AssertEquals('Test for "6"',[4,0,5,1,4,1,4,0,4,1,4,0,4,0,5,0,4,1,4,1,4,0,5],DoEncode);
  // 7 --> 50615150515060505151506
  SetText('7');
  AssertEquals('Test for "7"',[4,0,5,1,4,1,4,0,4,1,4,0,5,0,4,0,4,1,4,1,4,0,5],DoEncode);
  // 8 --> 50615150516050505151506
  SetText('8');
  AssertEquals('Test for "8"',[4,0,5,1,4,1,4,0,4,1,5,0,4,0,4,0,4,1,4,1,4,0,5],DoEncode);
  // 9 --> 50615150605150505151506
  SetText('9');
  AssertEquals('Test for "9"',[4,0,5,1,4,1,4,0,5,0,4,1,4,0,4,0,4,1,4,1,4,0,5],DoEncode);
  // 1234567800 --> 50615150505061505051506061505050506051506050515051505060515060505160505050505160505051605151506
  SetText('1234567800');
  AssertEquals('Test for "1234567800"',[4,0,5,1,4,1,4,0,4,0,4,0,5,1,4,0,4,0,4,1,4,0,5,0,5,1,4,0,4,0,4,0,4,0,5,0,4,1,4,0,5,0,4,0,4,1,4,0,4,1,4,0,4,0,5,0,4,1,4,0,5,0,4,0,4,1,5,0,4,0,4,0,4,0,4,0,4,1,5,0,4,0,4,0,4,1,5,0,4,1,4,1,4,0,5],DoEncode);
  // 123456789ABCD --> 50615150505061505051506061505050506051506050515051505060515060505160505060515050506151505151506050515160505161505151506
  SetText('123456789ABCD');
  AssertEquals('Test for "123456789ABCD"',[4,0,5,1,4,1,4,0,4,0,4,0,5,1,4,0,4,0,4,1,4,0,5,0,5,1,4,0,4,0,4,0,4,0,5,0,4,1,4,0,5,0,4,0,4,1,4,0,4,1,4,0,4,0,5,0,4,1,4,0,5,0,4,0,4,1,5,0,4,0,4,0,5,0,4,1,4,0,4,0,4,0,5,1,4,1,4,0,4,1,4,1,4,0,5,0,4,0,4,1,4,1,5,0,4,0,4,1,5,1,4,0,4,1,4,1,4,0,5],DoEncode);
end;


Procedure TTestBarcodes.TestCodeMSI;

begin
  SetEncoding(beMSI);
  // 0 --> 605151515151516051515
  SetText('0');
  AssertEquals('Test for "0"',[5,0,4,1,4,1,4,1,4,1,4,1,4,1,5,0,4,1,4,1,4],DoEncode);
  // 1 --> 605151516051515160515
  SetText('1');
  AssertEquals('Test for "1"',[5,0,4,1,4,1,4,1,5,0,4,1,4,1,4,1,5,0,4,1,4],DoEncode);
  // 2 --> 605151605151515151515
  SetText('2');
  AssertEquals('Test for "2"',[5,0,4,1,4,1,5,0,4,1,4,1,4,1,4,1,4,1,4,1,4],DoEncode);
  // 3 --> 605151606060515160515
  SetText('3');
  AssertEquals('Test for "3"',[5,0,4,1,4,1,5,0,5,0,5,0,4,1,4,1,5,0,4,1,4],DoEncode);
  // 4 --> 605160515160515151515
  SetText('4');
  AssertEquals('Test for "4"',[5,0,4,1,5,0,4,1,4,1,5,0,4,1,4,1,4,1,4,1,4],DoEncode);
  // 5 --> 605160516051606060515
  SetText('5');
  AssertEquals('Test for "5"',[5,0,4,1,5,0,4,1,5,0,4,1,5,0,5,0,5,0,4,1,4],DoEncode);
  // 6 --> 605160605151606051515
  SetText('6');
  AssertEquals('Test for "6"',[5,0,4,1,5,0,5,0,4,1,4,1,5,0,5,0,4,1,4,1,4],DoEncode);
  // 7 --> 605160606051605160515
  SetText('7');
  AssertEquals('Test for "7"',[5,0,4,1,5,0,5,0,5,0,4,1,5,0,4,1,5,0,4,1,4],DoEncode);
  // 8 --> 606051515151605151515
  SetText('8');
  AssertEquals('Test for "8"',[5,0,5,0,4,1,4,1,4,1,4,1,5,0,4,1,4,1,4,1,4],DoEncode);
  // 9 --> 606051516051516060515
  SetText('9');
  AssertEquals('Test for "9"',[5,0,5,0,4,1,4,1,5,0,4,1,4,1,5,0,5,0,4,1,4],DoEncode);
  // 1234567800 --> 605151516051516051515160605160515151605160516060515160606060515151515151515151515160515151515
  SetText('1234567800');
  AssertEquals('Test for "1234567800"',[5,0,4,1,4,1,4,1,5,0,4,1,4,1,5,0,4,1,4,1,4,1,5,0,5,0,4,1,5,0,4,1,4,1,4,1,5,0,4,1,5,0,4,1,5,0,5,0,4,1,4,1,5,0,5,0,5,0,5,0,4,1,4,1,4,1,4,1,4,1,4,1,4,1,4,1,4,1,4,1,4,1,5,0,4,1,4,1,4,1,4,1,4],DoEncode);
end;


Procedure TTestBarcodes.TestCodePostNet;

begin
  SetEncoding(bePostNet);
  // 0 --> 515151A1A1A15
  SetText('0');
  AssertEquals('Test for "0"',[4,1,4,1,4,1,8,1,8,1,8,1,4],DoEncode);
  // 1 --> 51A1A1A151515
  SetText('1');
  AssertEquals('Test for "1"',[4,1,8,1,8,1,8,1,4,1,4,1,4],DoEncode);
  // 2 --> 51A1A151A1515
  SetText('2');
  AssertEquals('Test for "2"',[4,1,8,1,8,1,4,1,8,1,4,1,4],DoEncode);
  // 3 --> 51A1A15151A15
  SetText('3');
  AssertEquals('Test for "3"',[4,1,8,1,8,1,4,1,4,1,8,1,4],DoEncode);
  // 4 --> 51A151A1A1515
  SetText('4');
  AssertEquals('Test for "4"',[4,1,8,1,4,1,8,1,8,1,4,1,4],DoEncode);
  // 5 --> 51A151A151A15
  SetText('5');
  AssertEquals('Test for "5"',[4,1,8,1,4,1,8,1,4,1,8,1,4],DoEncode);
  // 6 --> 51A15151A1A15
  SetText('6');
  AssertEquals('Test for "6"',[4,1,8,1,4,1,4,1,8,1,8,1,4],DoEncode);
  // 7 --> 5151A1A1A1515
  SetText('7');
  AssertEquals('Test for "7"',[4,1,4,1,8,1,8,1,8,1,4,1,4],DoEncode);
  // 8 --> 5151A1A151A15
  SetText('8');
  AssertEquals('Test for "8"',[4,1,4,1,8,1,8,1,4,1,8,1,4],DoEncode);
  // 9 --> 5151A151A1A15
  SetText('9');
  AssertEquals('Test for "9"',[4,1,4,1,8,1,4,1,8,1,8,1,4],DoEncode);
  // 1234567800 --> 51A1A1A15151A1A151A151A1A15151A1A151A1A151A151A151A1A15151A1A151A1A1A15151A1A151A15151A1A1A15151A1A1A15
  SetText('1234567800');
  AssertEquals('Test for "1234567800"',[4,1,8,1,8,1,8,1,4,1,4,1,8,1,8,1,4,1,8,1,4,1,8,1,8,1,4,1,4,1,8,1,8,1,4,1,8,1,8,1,4,1,8,1,4,1,8,1,4,1,8,1,8,1,4,1,4,1,8,1,8,1,4,1,8,1,8,1,8,1,4,1,4,1,8,1,8,1,4,1,8,1,4,1,4,1,8,1,8,1,8,1,4,1,4,1,8,1,8,1,8,1,4],DoEncode);
end;
{
BarTypes : TBarTypeParams = (
);

}
Class procedure TTestBarcodes.AssertEquals(Msg : String; AExpected,AActual : TBarColor);
begin
  AssertEquals(Msg,GetEnumName(TypeInfo(TBarColor),Ord(AExpected)),
                   GetEnumName(TypeInfo(TBarColor),Ord(AActual)));
end;

Class procedure TTestBarcodes.AssertEquals(Msg : String; AExpected,AActual : TBarWidth);
begin
  AssertEquals(Msg,GetEnumName(TypeInfo(TBarWidth),Ord(AExpected)),
                   GetEnumName(TypeInfo(TBarWidth),Ord(AActual)));
end;
Class procedure TTestBarcodes.AssertEquals(Msg : String; AExpected,AActual : TBarHeight);
begin
  AssertEquals(Msg,GetEnumName(TypeInfo(TBarHeight),Ord(AExpected)),
                   GetEnumName(TypeInfo(TBarHeight),Ord(AActual)));
end;

// AWidth : TBarWidth; AHeight : TBarheight);
// AColor : TBarColor; AWidth : TBarWidth; AHeight : TBarheight

procedure TTestBarcodes.AssertParams(aType : TBarType; AColor : TBarColor; AWidth : TBarWidth; AHeight : TBarheight);

Var
  S : String;
  P : TBarParams;

begin
  S:=Format('Bar type %d: ',[aType]);
  P:=BarTypeToBarParams(aType);
  AssertEquals(S+'Color',aColor,P.c);
  AssertEquals(S+'Width',aWidth,P.w);
  AssertEquals(S+'Height',aHeight,P.h);
end;

procedure TTestBarcodes.CheckParamsType;

begin
  AssertParams(0, bcWhite, bw100,  bhFull);
  AssertParams(1, bcWhite, bwWeighted, bhFull);
  AssertParams(2, bcWhite, bw150, bhFull);
  AssertParams(3, bcWhite, bw200, bhFull);
  AssertParams(4, bcBlack, bw100,  bhFull);
  AssertParams(5, bcBlack, bwWeighted, bhFull);
  AssertParams(6, bcBlack, bw150, bhFull);
  AssertParams(7, bcBlack, bw200, bhFull);
  AssertParams(8, bcBlack, bw100,  bhTwoFifth);
  AssertParams(9, bcBlack, bwWeighted, bhTwoFifth);
  AssertParams(10, bcBlack, bw150, bhTwoFifth);
  AssertParams(11, bcBlack, bw200, bhTwoFifth);
end;

procedure TTestBarcodes.CheckNumerictypes;
begin
  AssertTrue('Numerical types',NumericalEncodings=[beEAN8,beEAN13,
    be2of5industrial,be2of5interleaved, be2of5matrix,
    bePostNet,
    beMSI,
    be128C]);
end;



initialization
  RegisterTest(TTestBarcodes);
end.

