#############################################################################
##
##  This file is part of GAP, a system for computational discrete algebra.
##  This file's authors include Thomas Breuer.
##
##  Copyright of GAP belongs to its developers, whose names are too numerous
##  to list here. Please refer to the COPYRIGHT file for details.
##
##  SPDX-License-Identifier: GPL-2.0-or-later
##
##  This file contains the declaration of operations for monoids.
##


#############################################################################
##
#P  IsMonoid( <D> )
##
##  <#GAPDoc Label="IsMonoid">
##  <ManSection>
##  <Filt Name="IsMonoid" Arg='D' Type='Synonym'/>
##
##  <Description>
##  A <E>monoid</E> is a magma-with-one (see&nbsp;<Ref Chap="Magmas"/>)
##  with associative multiplication.
##  </Description>
##  </ManSection>
##  <#/GAPDoc>
##
DeclareSynonymAttr( "IsMonoid", IsMagmaWithOne and IsAssociative );


#############################################################################
##
#F  Monoid( <gen1>, <gen2> ... )
#F  Monoid( <gens> )
#F  Monoid( <gens>, <id> )
##
##  <#GAPDoc Label="Monoid">
##  <ManSection>
##  <Heading>Monoid</Heading>
##  <Func Name="Monoid" Arg='gen1, gen2 ...'
##   Label="for various generators"/>
##  <Func Name="Monoid" Arg='gens[, id]' Label="for a list"/>
##
##  <Description>
##  In the first form, <Ref Func="Monoid" Label="for various generators"/>
##  returns the monoid generated by the arguments <A>gen1</A>, <A>gen2</A>,
##  <M>\ldots</M>,
##  that is, the closure of these elements under multiplication and taking
##  the 0-th power.
##  In the second form, <Ref Func="Monoid" Label="for a list"/> returns
##  the monoid generated by the elements in the homogeneous list <A>gens</A>;
##  a square matrix as only argument is treated as one generator,
##  not as a list of generators.
##  In the second form, the identity element <A>id</A> may be given as the
##  second argument.
##  <P/>
##  It is <E>not</E> checked whether the underlying multiplication is
##  associative, use <Ref Func="MagmaWithOne"/> and
##  <Ref Prop="IsAssociative"/>
##  if you want to check whether a magma-with-one is in fact a monoid.
##  </Description>
##  </ManSection>
##  <#/GAPDoc>
##
DeclareGlobalFunction( "Monoid" );


#############################################################################
##
#F  Submonoid( <M>, <gens> ) . . . . . . submonoid of <M> generated by <gens>
#F  SubmonoidNC( <M>, <gens> )
##
##  <#GAPDoc Label="Submonoid">
##  <ManSection>
##  <Func Name="Submonoid" Arg='M, gens'/>
##  <Func Name="SubmonoidNC" Arg='M, gens'/>
##
##  <Description>
##  are just synonyms of <Ref Func="SubmagmaWithOne"/>
##  and <Ref Func="SubmagmaWithOneNC"/>, respectively.
##  </Description>
##  </ManSection>
##  <#/GAPDoc>
##
DeclareSynonym( "Submonoid", SubmagmaWithOne );

DeclareSynonym( "SubmonoidNC", SubmagmaWithOneNC );


#############################################################################
##
#O  MonoidByGenerators( <gens>[, <one>] )  . . . . monoid generated by <gens>
##
##  <#GAPDoc Label="MonoidByGenerators">
##  <ManSection>
##  <Oper Name="MonoidByGenerators" Arg='gens[, one]'/>
##
##  <Description>
##  is the underlying operation of <Ref Func="Monoid" Label="for a list"/>.
##  </Description>
##  </ManSection>
##  <#/GAPDoc>
##
DeclareOperation( "MonoidByGenerators", [ IsCollection ] );


#############################################################################
##
#A  AsMonoid( <C> ) . . . . . . . . . . . . collection <C> regarded as monoid
##
##  <#GAPDoc Label="AsMonoid">
##  <ManSection>
##  <Oper Name="AsMonoid" Arg='C'/>
##
##  <Description>
##  If <A>C</A> is a collection whose elements form a monoid, 
##  then <Ref Oper="AsMonoid"/> returns this monoid.
##  Otherwise <K>fail</K> is returned.
##  </Description>
##  </ManSection>
##  <#/GAPDoc>
##
DeclareOperation( "AsMonoid", [IsCollection] );


#############################################################################
##
#O  AsSubmonoid( <D>, <C> )
##
##  <#GAPDoc Label="AsSubmonoid">
##  <ManSection>
##  <Oper Name="AsSubmonoid" Arg='D, C'/>
##
##  <Description>
##  Let <A>D</A> be a domain and <A>C</A> a collection.
##  If <A>C</A> is a subset of <A>D</A> that forms a monoid then
##  <Ref Oper="AsSubmonoid"/>
##  returns this monoid, with parent <A>D</A>.
##  Otherwise <K>fail</K> is returned.
##  </Description>
##  </ManSection>
##  <#/GAPDoc>
##
DeclareOperation( "AsSubmonoid", [ IsDomain, IsCollection ] );


#############################################################################
##
#A  GeneratorsOfMonoid( <M> )  . . . . . . .  monoid generators of monoid <M>
##
##  <#GAPDoc Label="GeneratorsOfMonoid">
##  <ManSection>
##  <Attr Name="GeneratorsOfMonoid" Arg='M'/>
##
##  <Description>
##  Monoid generators of a monoid <A>M</A> are the same as
##  magma-with-one generators
##  (see&nbsp;<Ref Attr="GeneratorsOfMagmaWithOne"/>).
##  </Description>
##  </ManSection>
##  <#/GAPDoc>
##
DeclareSynonymAttr( "GeneratorsOfMonoid", GeneratorsOfMagmaWithOne );


#############################################################################
##
#A  TrivialSubmonoid( <M> ) . . . . . . . . . trivial submonoid of monoid <M>
##
##  <#GAPDoc Label="TrivialSubmonoid">
##  <ManSection>
##  <Attr Name="TrivialSubmonoid" Arg='M'/>
##
##  <Description>
##  is just a synonym for <Ref Attr="TrivialSubmagmaWithOne"/>.
##  </Description>
##  </ManSection>
##  <#/GAPDoc>
##
DeclareSynonymAttr( "TrivialSubmonoid", TrivialSubmagmaWithOne );


#############################################################################
##
#F  FreeMonoid( [<wfilt>,]<rank>[, <name>] )
#F  FreeMonoid( [<wfilt>,][<name1>[, <name2>[, ...]]] )
#F  FreeMonoid( [<wfilt>,]<names> )
#F  FreeMonoid( [<wfilt>,]infinity[, <name>][, <init>] )
##
##  <#GAPDoc Label="FreeMonoid">
##  <ManSection>
##  <Heading>FreeMonoid</Heading>
##  <Func Name="FreeMonoid" Arg='[wfilt, ]rank[, name]'
##   Label="for given rank"/>
##  <Func Name="FreeMonoid" Arg='[wfilt, ][name1[, name2[, ...]]]'
##   Label="for various names"/>
##  <Func Name="FreeMonoid" Arg='[wfilt, ]names'
##   Label="for a list of names"/>
##  <Func Name="FreeMonoid" Arg='[wfilt, ]infinity[, name][, init]'
##   Label="for infinitely many generators"/>
##
##  <Description>
##  <C>FreeMonoid</C> returns a free monoid. The number of
##  generators, and the labels given to the generators, can be specified in
##  several different ways.
##  Warning: the labels of generators are only an aid for printing,
##  and do not necessarily distinguish generators;
##  see the examples at the end of
##  <Ref Func="FreeSemigroup" Label="for given rank"/>
##  for more information.
##  <List>
##    <Mark>
##      1: For a given rank, and an optional generator name prefix
##    </Mark>
##    <Item>
##      Called with a nonnegative integer <A>rank</A>,
##      <Ref Func="FreeMonoid" Label="for given rank"/> returns
##      a free monoid on <A>rank</A> generators.
##      The optional argument <A>name</A> must be a string;
##      its default value is <C>"m"</C>. <P/>
##
##      If <A>name</A> is not given but the <C>generatorNames</C> option is,
##      then this option is respected as described in
##      Section&nbsp;<Ref Sect="Generator Names"/>. <P/>
##
##      Otherwise, the generators of the returned free monoid are labelled
##      <A>name</A><C>1</C>, ..., <A>name</A><C>k</C>,
##      where <C>k</C> is the value of <A>rank</A>. <P/>
##    </Item>
##    <Mark>2: For given generator names</Mark>
##    <Item>
##      Called with various nonempty strings,
##      <Ref Func="FreeMonoid" Label="for various names"/> returns
##      a free monoid on as many generators as arguments, which are labelled
##      <A>name1</A>, <A>name2</A>, etc.
##    </Item>
##    <Mark>3: For a given list of generator names</Mark>
##    <Item>
##      Called with a finite list <A>names</A> of
##      nonempty strings,
##      <Ref Func="FreeMonoid" Label="for a list of names"/> returns
##      a free monoid on <C>Length(<A>names</A>)</C> generators, whose
##      <C>i</C>-th generator is labelled <A>names</A><C>[i]</C>.
##    </Item>
##    <Mark>
##      4: For the rank <K>infinity</K>,
##         an optional default generator name prefix,
##         and an optional finite list of generator names
##    </Mark>
##    <Item>
##      Called in the fourth form,
##      <Ref Func="FreeMonoid" Label="for infinitely many generators"/>
##      returns a free monoid on infinitely many generators.
##      The optional argument <A>name</A> must be a string; its default value is
##      <C>"m"</C>,
##      and the optional argument <A>init</A> must be a finite list of
##      nonempty strings; its default value is an empty list.
##      The generators are initially labelled according to the list <A>init</A>,
##      followed by
##      <A>name</A><C>i</C> for each <C>i</C> in the range from
##      <C>Length(<A>init</A>)+1</C> to <K>infinity</K>.
##    </Item>
##  </List>
##
##  If the optional first argument <A>wfilt</A> is given, then it must be either
##  <C>IsSyllableWordsFamily</C>, <C>IsLetterWordsFamily</C>,
##  <C>IsWLetterWordsFamily</C>, or <C>IsBLetterWordsFamily</C>.
##  This filter specifies the representation used for the elements of
##  the free monoid
##  (see&nbsp;<Ref Sect="Representations for Associative Words"/>).
##  If no such filter is given, a letter representation is used.
##  <P/>
##  For more on associative words see 
##  Chapter&nbsp;<Ref Chap="Associative Words"/>.
##
##  <Example><![CDATA[
##  gap> FreeMonoid(5);
##  <free monoid on the generators [ m1, m2, m3, m4, m5 ]>
##  gap> FreeMonoid(4, "gen");
##  <free monoid on the generators [ gen1, gen2, gen3, gen4 ]>
##  gap> FreeMonoid(3 : generatorNames := "turbo");
##  <free monoid on the generators [ turbo1, turbo2, turbo3 ]>
##  gap> FreeMonoid(2 : generatorNames := ["u", "v", "w"]);
##  <free monoid on the generators [ u, v ]>
##  gap> FreeMonoid(); FreeMonoid(0); FreeMonoid([]);
##  <free monoid of rank zero>
##  <free monoid of rank zero>
##  <free monoid of rank zero>
##  gap> FreeMonoid("a", "b", "c");
##  <free monoid on the generators [ a, b, c ]>
##  gap> FreeMonoid(["x", "y"]);
##  <free monoid on the generators [ x, y ]>
##  gap> FreeMonoid(infinity);
##  <free monoid with infinity generators>
##  gap> F := FreeMonoid(infinity, "g", ["a", "b"]);
##  <free monoid with infinity generators>
##  gap> GeneratorsOfMonoid(F){[1..4]};
##  [ a, b, g3, g4 ]
##  gap> GeneratorsOfMonoid(FreeMonoid(infinity, "gen")){[1..3]};
##  [ gen1, gen2, gen3 ]
##  gap> GeneratorsOfMonoid(FreeMonoid(infinity, [ "f", "g" ])){[1..3]};
##  [ f, g, m3 ]
##  gap> FreeMonoid(IsSyllableWordsFamily, 50);
##  <free monoid with 50 generators>
##  ]]></Example>
##  </Description>
##  </ManSection>
##  <#/GAPDoc>
##
DeclareGlobalFunction( "FreeMonoid" );

#############################################################################
##
#P  IsFinitelyGeneratedMonoid( <M> )
##
##  <#GAPDoc Label="IsFinitelyGeneratedMonoid">
##  <ManSection>
##  <Prop Name="IsFinitelyGeneratedMonoid" Arg='M'/>
##
##  <Description>
##  tests whether the monoid <A>M</A> can be generated by a finite number of
##  generators. (This property is mainly used to obtain finiteness
##  conditions.)
##  <P/>
##  Note that this is a pure existence statement. Even if a monoid is known
##  to be generated by a finite number of elements, it can be very hard or
##  even impossible to obtain such a generating set if it is not known.
##  </Description>
##  </ManSection>
##  <#/GAPDoc>
##
DeclareProperty( "IsFinitelyGeneratedMonoid", IsMonoid );
InstallTrueMethod( IsMonoid, IsFinitelyGeneratedMonoid );

# make IsFinitelyGeneratedMonoid equivalent to IsMonoid and IsFinitelyGeneratedMagma
InstallTrueMethod( IsFinitelyGeneratedMonoid, IsMonoid and IsFinitelyGeneratedMagma );
InstallTrueMethod( HasIsFinitelyGeneratedMonoid, IsMonoid and HasIsFinitelyGeneratedMagma );
InstallTrueMethod( IsFinitelyGeneratedMagma, IsFinitelyGeneratedMonoid );
InstallTrueMethod( HasIsFinitelyGeneratedMagma, HasIsFinitelyGeneratedMonoid );
