"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PRIVATE_TENANTS = exports.GLOBAL_TENANTS = void 0;
exports.isValidTenant = isValidTenant;
exports.resolveTenant = resolveTenant;

var _lodash = require("lodash");

var _common = require("../../common");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
const PRIVATE_TENANTS = [_common.PRIVATE_TENANT_SYMBOL, 'private'];
exports.PRIVATE_TENANTS = PRIVATE_TENANTS;
const GLOBAL_TENANTS = ['global', _common.GLOBAL_TENANT_SYMBOL];
/**
 * Resovles the tenant the user is using.
 *
 * @param request OpenSearchDashboards request.
 * @param config security plugin config.
 * @param cookie cookie extracted from the request. The cookie should have been parsed by AuthenticationHandler.
 * pass it as parameter instead of extracting again.
 * @param authInfo authentication info, the Elasticsearch authinfo API response.
 *
 * @returns user preferred tenant of the request.
 */

exports.GLOBAL_TENANTS = GLOBAL_TENANTS;

function resolveTenant(request, username, roles, availabeTenants, config, cookie) {
  var _request$url, _request$url$searchPa, _request$url2, _request$url2$searchP, _request$url3, _request$url3$searchP, _config$multitenancy, _config$multitenancy2, _config$multitenancy3;

  const DEFAULT_READONLY_ROLES = ['kibana_read_only'];
  let selectedTenant;
  const securityTenant_ = request === null || request === void 0 ? void 0 : (_request$url = request.url) === null || _request$url === void 0 ? void 0 : (_request$url$searchPa = _request$url.searchParams) === null || _request$url$searchPa === void 0 ? void 0 : _request$url$searchPa.get('securityTenant_');
  const securitytenant = request === null || request === void 0 ? void 0 : (_request$url2 = request.url) === null || _request$url2 === void 0 ? void 0 : (_request$url2$searchP = _request$url2.searchParams) === null || _request$url2$searchP === void 0 ? void 0 : _request$url2$searchP.get('securitytenant'); // eslint-disable-next-line @typescript-eslint/naming-convention

  const security_tenant = request === null || request === void 0 ? void 0 : (_request$url3 = request.url) === null || _request$url3 === void 0 ? void 0 : (_request$url3$searchP = _request$url3.searchParams) === null || _request$url3$searchP === void 0 ? void 0 : _request$url3$searchP.get('security_tenant');

  if (securityTenant_) {
    selectedTenant = securityTenant_;
  } else if (securitytenant) {
    selectedTenant = securitytenant;
  } else if (security_tenant) {
    selectedTenant = security_tenant;
  } else if (request.headers.securitytenant || request.headers.securityTenant_) {
    selectedTenant = request.headers.securitytenant ? request.headers.securitytenant : request.headers.securityTenant_;
  } else if (isValidTenant(cookie.tenant)) {
    selectedTenant = cookie.tenant;
  } else {
    selectedTenant = undefined;
  }

  const isReadonly = roles === null || roles === void 0 ? void 0 : roles.some(role => {
    var _config$readonly_mode;

    return ((_config$readonly_mode = config.readonly_mode) === null || _config$readonly_mode === void 0 ? void 0 : _config$readonly_mode.roles.includes(role)) || DEFAULT_READONLY_ROLES.includes(role);
  });
  const preferredTenants = (_config$multitenancy = config.multitenancy) === null || _config$multitenancy === void 0 ? void 0 : _config$multitenancy.tenants.preferred;
  const globalTenantEnabled = (_config$multitenancy2 = config.multitenancy) === null || _config$multitenancy2 === void 0 ? void 0 : _config$multitenancy2.tenants.enable_global;
  const privateTenantEnabled = ((_config$multitenancy3 = config.multitenancy) === null || _config$multitenancy3 === void 0 ? void 0 : _config$multitenancy3.tenants.enable_private) && !isReadonly;
  return resolve(username, selectedTenant, preferredTenants, availabeTenants, globalTenantEnabled, privateTenantEnabled);
}

function resolve(username, requestedTenant, preferredTenants, availableTenants, // is an object like { tenant_name_1: true, tenant_name_2: false, ... }
globalTenantEnabled, privateTenantEnabled) {
  const availableTenantsClone = (0, _lodash.cloneDeep)(availableTenants);
  delete availableTenantsClone[username];

  if (!globalTenantEnabled && !privateTenantEnabled && (0, _lodash.isEmpty)(availableTenantsClone)) {
    return undefined;
  }

  if (isValidTenant(requestedTenant)) {
    requestedTenant = requestedTenant;

    if (requestedTenant in availableTenants) {
      return requestedTenant;
    }

    if (privateTenantEnabled && username in availableTenants && PRIVATE_TENANTS.indexOf(requestedTenant) > -1) {
      return _common.PRIVATE_TENANT_SYMBOL;
    }

    if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(requestedTenant) > -1) {
      return _common.GLOBAL_TENANT_SYMBOL;
    }
  }

  if (preferredTenants && !(0, _lodash.isEmpty)(preferredTenants)) {
    for (const element of preferredTenants) {
      const tenant = element.toLowerCase();

      if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(tenant) > -1) {
        return _common.GLOBAL_TENANT_SYMBOL;
      }

      if (privateTenantEnabled && PRIVATE_TENANTS.indexOf(tenant) > -1 && username in availableTenants) {
        return _common.PRIVATE_TENANT_SYMBOL;
      }

      if (tenant in availableTenants) {
        return tenant;
      }
    }
  }

  if (globalTenantEnabled) {
    return _common.GLOBAL_TENANT_SYMBOL;
  }

  if (privateTenantEnabled) {
    return _common.PRIVATE_TENANT_SYMBOL;
  } // fall back to the first tenant in the available tenants


  return (0, _lodash.findKey)(availableTenantsClone, () => true);
}
/**
 * Return true if tenant parameter is a valid tenent.
 *
 * Note: empty string '' is valid, which means global tenant.
 *
 * @param tenant
 */


function isValidTenant(tenant) {
  return tenant !== undefined && tenant !== null;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInRlbmFudF9yZXNvbHZlci50cyJdLCJuYW1lcyI6WyJQUklWQVRFX1RFTkFOVFMiLCJQUklWQVRFX1RFTkFOVF9TWU1CT0wiLCJHTE9CQUxfVEVOQU5UUyIsIkdMT0JBTF9URU5BTlRfU1lNQk9MIiwicmVzb2x2ZVRlbmFudCIsInJlcXVlc3QiLCJ1c2VybmFtZSIsInJvbGVzIiwiYXZhaWxhYmVUZW5hbnRzIiwiY29uZmlnIiwiY29va2llIiwiREVGQVVMVF9SRUFET05MWV9ST0xFUyIsInNlbGVjdGVkVGVuYW50Iiwic2VjdXJpdHlUZW5hbnRfIiwidXJsIiwic2VhcmNoUGFyYW1zIiwiZ2V0Iiwic2VjdXJpdHl0ZW5hbnQiLCJzZWN1cml0eV90ZW5hbnQiLCJoZWFkZXJzIiwiaXNWYWxpZFRlbmFudCIsInRlbmFudCIsInVuZGVmaW5lZCIsImlzUmVhZG9ubHkiLCJzb21lIiwicm9sZSIsInJlYWRvbmx5X21vZGUiLCJpbmNsdWRlcyIsInByZWZlcnJlZFRlbmFudHMiLCJtdWx0aXRlbmFuY3kiLCJ0ZW5hbnRzIiwicHJlZmVycmVkIiwiZ2xvYmFsVGVuYW50RW5hYmxlZCIsImVuYWJsZV9nbG9iYWwiLCJwcml2YXRlVGVuYW50RW5hYmxlZCIsImVuYWJsZV9wcml2YXRlIiwicmVzb2x2ZSIsInJlcXVlc3RlZFRlbmFudCIsImF2YWlsYWJsZVRlbmFudHMiLCJhdmFpbGFibGVUZW5hbnRzQ2xvbmUiLCJpbmRleE9mIiwiZWxlbWVudCIsInRvTG93ZXJDYXNlIl0sIm1hcHBpbmdzIjoiOzs7Ozs7Ozs7QUFlQTs7QUFJQTs7QUFuQkE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQVFPLE1BQU1BLGVBQXlCLEdBQUcsQ0FBQ0MsNkJBQUQsRUFBd0IsU0FBeEIsQ0FBbEM7O0FBQ0EsTUFBTUMsY0FBd0IsR0FBRyxDQUFDLFFBQUQsRUFBV0MsNEJBQVgsQ0FBakM7QUFDUDtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOzs7O0FBQ08sU0FBU0MsYUFBVCxDQUNMQyxPQURLLEVBRUxDLFFBRkssRUFHTEMsS0FISyxFQUlMQyxlQUpLLEVBS0xDLE1BTEssRUFNTEMsTUFOSyxFQU9lO0FBQUE7O0FBQ3BCLFFBQU1DLHNCQUFzQixHQUFHLENBQUMsa0JBQUQsQ0FBL0I7QUFDQSxNQUFJQyxjQUFKO0FBQ0EsUUFBTUMsZUFBZSxHQUFHUixPQUFILGFBQUdBLE9BQUgsdUNBQUdBLE9BQU8sQ0FBRVMsR0FBWiwwRUFBRyxhQUFjQyxZQUFqQiwwREFBRyxzQkFBNEJDLEdBQTVCLENBQWdDLGlCQUFoQyxDQUF4QjtBQUNBLFFBQU1DLGNBQWMsR0FBR1osT0FBSCxhQUFHQSxPQUFILHdDQUFHQSxPQUFPLENBQUVTLEdBQVosMkVBQUcsY0FBY0MsWUFBakIsMERBQUcsc0JBQTRCQyxHQUE1QixDQUFnQyxnQkFBaEMsQ0FBdkIsQ0FKb0IsQ0FLcEI7O0FBQ0EsUUFBTUUsZUFBZSxHQUFHYixPQUFILGFBQUdBLE9BQUgsd0NBQUdBLE9BQU8sQ0FBRVMsR0FBWiwyRUFBRyxjQUFjQyxZQUFqQiwwREFBRyxzQkFBNEJDLEdBQTVCLENBQWdDLGlCQUFoQyxDQUF4Qjs7QUFDQSxNQUFJSCxlQUFKLEVBQXFCO0FBQ25CRCxJQUFBQSxjQUFjLEdBQUdDLGVBQWpCO0FBQ0QsR0FGRCxNQUVPLElBQUlJLGNBQUosRUFBb0I7QUFDekJMLElBQUFBLGNBQWMsR0FBR0ssY0FBakI7QUFDRCxHQUZNLE1BRUEsSUFBSUMsZUFBSixFQUFxQjtBQUMxQk4sSUFBQUEsY0FBYyxHQUFHTSxlQUFqQjtBQUNELEdBRk0sTUFFQSxJQUFJYixPQUFPLENBQUNjLE9BQVIsQ0FBZ0JGLGNBQWhCLElBQWtDWixPQUFPLENBQUNjLE9BQVIsQ0FBZ0JOLGVBQXRELEVBQXVFO0FBQzVFRCxJQUFBQSxjQUFjLEdBQUdQLE9BQU8sQ0FBQ2MsT0FBUixDQUFnQkYsY0FBaEIsR0FDWlosT0FBTyxDQUFDYyxPQUFSLENBQWdCRixjQURKLEdBRVpaLE9BQU8sQ0FBQ2MsT0FBUixDQUFnQk4sZUFGckI7QUFHRCxHQUpNLE1BSUEsSUFBSU8sYUFBYSxDQUFDVixNQUFNLENBQUNXLE1BQVIsQ0FBakIsRUFBa0M7QUFDdkNULElBQUFBLGNBQWMsR0FBR0YsTUFBTSxDQUFDVyxNQUF4QjtBQUNELEdBRk0sTUFFQTtBQUNMVCxJQUFBQSxjQUFjLEdBQUdVLFNBQWpCO0FBQ0Q7O0FBQ0QsUUFBTUMsVUFBVSxHQUFHaEIsS0FBSCxhQUFHQSxLQUFILHVCQUFHQSxLQUFLLENBQUVpQixJQUFQLENBQ2hCQyxJQUFEO0FBQUE7O0FBQUEsV0FBVSwwQkFBQWhCLE1BQU0sQ0FBQ2lCLGFBQVAsZ0ZBQXNCbkIsS0FBdEIsQ0FBNEJvQixRQUE1QixDQUFxQ0YsSUFBckMsTUFBOENkLHNCQUFzQixDQUFDZ0IsUUFBdkIsQ0FBZ0NGLElBQWhDLENBQXhEO0FBQUEsR0FEaUIsQ0FBbkI7QUFJQSxRQUFNRyxnQkFBZ0IsMkJBQUduQixNQUFNLENBQUNvQixZQUFWLHlEQUFHLHFCQUFxQkMsT0FBckIsQ0FBNkJDLFNBQXREO0FBQ0EsUUFBTUMsbUJBQW1CLDRCQUFHdkIsTUFBTSxDQUFDb0IsWUFBViwwREFBRyxzQkFBcUJDLE9BQXJCLENBQTZCRyxhQUF6RDtBQUNBLFFBQU1DLG9CQUFvQixHQUFHLDBCQUFBekIsTUFBTSxDQUFDb0IsWUFBUCxnRkFBcUJDLE9BQXJCLENBQTZCSyxjQUE3QixLQUErQyxDQUFDWixVQUE3RTtBQUVBLFNBQU9hLE9BQU8sQ0FDWjlCLFFBRFksRUFFWk0sY0FGWSxFQUdaZ0IsZ0JBSFksRUFJWnBCLGVBSlksRUFLWndCLG1CQUxZLEVBTVpFLG9CQU5ZLENBQWQ7QUFRRDs7QUFFRCxTQUFTRSxPQUFULENBQ0U5QixRQURGLEVBRUUrQixlQUZGLEVBR0VULGdCQUhGLEVBSUVVLGdCQUpGLEVBSXlCO0FBQ3ZCTixtQkFMRixFQU1FRSxvQkFORixFQU9zQjtBQUNwQixRQUFNSyxxQkFBcUIsR0FBRyx1QkFBVUQsZ0JBQVYsQ0FBOUI7QUFDQSxTQUFPQyxxQkFBcUIsQ0FBQ2pDLFFBQUQsQ0FBNUI7O0FBRUEsTUFBSSxDQUFDMEIsbUJBQUQsSUFBd0IsQ0FBQ0Usb0JBQXpCLElBQWlELHFCQUFRSyxxQkFBUixDQUFyRCxFQUFxRjtBQUNuRixXQUFPakIsU0FBUDtBQUNEOztBQUVELE1BQUlGLGFBQWEsQ0FBQ2lCLGVBQUQsQ0FBakIsRUFBb0M7QUFDbENBLElBQUFBLGVBQWUsR0FBR0EsZUFBbEI7O0FBQ0EsUUFBSUEsZUFBZSxJQUFJQyxnQkFBdkIsRUFBeUM7QUFDdkMsYUFBT0QsZUFBUDtBQUNEOztBQUVELFFBQ0VILG9CQUFvQixJQUNwQjVCLFFBQVEsSUFBSWdDLGdCQURaLElBRUF0QyxlQUFlLENBQUN3QyxPQUFoQixDQUF3QkgsZUFBeEIsSUFBMkMsQ0FBQyxDQUg5QyxFQUlFO0FBQ0EsYUFBT3BDLDZCQUFQO0FBQ0Q7O0FBRUQsUUFBSStCLG1CQUFtQixJQUFJOUIsY0FBYyxDQUFDc0MsT0FBZixDQUF1QkgsZUFBdkIsSUFBMEMsQ0FBQyxDQUF0RSxFQUF5RTtBQUN2RSxhQUFPbEMsNEJBQVA7QUFDRDtBQUNGOztBQUVELE1BQUl5QixnQkFBZ0IsSUFBSSxDQUFDLHFCQUFRQSxnQkFBUixDQUF6QixFQUFvRDtBQUNsRCxTQUFLLE1BQU1hLE9BQVgsSUFBc0JiLGdCQUF0QixFQUF3QztBQUN0QyxZQUFNUCxNQUFNLEdBQUdvQixPQUFPLENBQUNDLFdBQVIsRUFBZjs7QUFFQSxVQUFJVixtQkFBbUIsSUFBSTlCLGNBQWMsQ0FBQ3NDLE9BQWYsQ0FBdUJuQixNQUF2QixJQUFpQyxDQUFDLENBQTdELEVBQWdFO0FBQzlELGVBQU9sQiw0QkFBUDtBQUNEOztBQUVELFVBQ0UrQixvQkFBb0IsSUFDcEJsQyxlQUFlLENBQUN3QyxPQUFoQixDQUF3Qm5CLE1BQXhCLElBQWtDLENBQUMsQ0FEbkMsSUFFQWYsUUFBUSxJQUFJZ0MsZ0JBSGQsRUFJRTtBQUNBLGVBQU9yQyw2QkFBUDtBQUNEOztBQUVELFVBQUlvQixNQUFNLElBQUlpQixnQkFBZCxFQUFnQztBQUM5QixlQUFPakIsTUFBUDtBQUNEO0FBQ0Y7QUFDRjs7QUFFRCxNQUFJVyxtQkFBSixFQUF5QjtBQUN2QixXQUFPN0IsNEJBQVA7QUFDRDs7QUFFRCxNQUFJK0Isb0JBQUosRUFBMEI7QUFDeEIsV0FBT2pDLDZCQUFQO0FBQ0QsR0F2RG1CLENBeURwQjs7O0FBQ0EsU0FBTyxxQkFBUXNDLHFCQUFSLEVBQStCLE1BQU0sSUFBckMsQ0FBUDtBQUNEO0FBRUQ7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7OztBQUNPLFNBQVNuQixhQUFULENBQXVCQyxNQUF2QixFQUFtRTtBQUN4RSxTQUFPQSxNQUFNLEtBQUtDLFNBQVgsSUFBd0JELE1BQU0sS0FBSyxJQUExQztBQUNEIiwic291cmNlc0NvbnRlbnQiOlsiLypcbiAqICAgQ29weXJpZ2h0IE9wZW5TZWFyY2ggQ29udHJpYnV0b3JzXG4gKlxuICogICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLlxuICogICBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlIHdpdGggdGhlIExpY2Vuc2UuXG4gKiAgIEEgY29weSBvZiB0aGUgTGljZW5zZSBpcyBsb2NhdGVkIGF0XG4gKlxuICogICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogICBvciBpbiB0aGUgXCJsaWNlbnNlXCIgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWRcbiAqICAgb24gYW4gXCJBUyBJU1wiIEJBU0lTLCBXSVRIT1VUIFdBUlJBTlRJRVMgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZWl0aGVyXG4gKiAgIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nXG4gKiAgIHBlcm1pc3Npb25zIGFuZCBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG5pbXBvcnQgeyBpc0VtcHR5LCBmaW5kS2V5LCBjbG9uZURlZXAgfSBmcm9tICdsb2Rhc2gnO1xuaW1wb3J0IHsgT3BlblNlYXJjaERhc2hib2FyZHNSZXF1ZXN0IH0gZnJvbSAnLi4vLi4vLi4vLi4vc3JjL2NvcmUvc2VydmVyJztcbmltcG9ydCB7IFNlY3VyaXR5U2Vzc2lvbkNvb2tpZSB9IGZyb20gJy4uL3Nlc3Npb24vc2VjdXJpdHlfY29va2llJztcbmltcG9ydCB7IFNlY3VyaXR5UGx1Z2luQ29uZmlnVHlwZSB9IGZyb20gJy4uJztcbmltcG9ydCB7IEdMT0JBTF9URU5BTlRfU1lNQk9MLCBQUklWQVRFX1RFTkFOVF9TWU1CT0wgfSBmcm9tICcuLi8uLi9jb21tb24nO1xuXG5leHBvcnQgY29uc3QgUFJJVkFURV9URU5BTlRTOiBzdHJpbmdbXSA9IFtQUklWQVRFX1RFTkFOVF9TWU1CT0wsICdwcml2YXRlJ107XG5leHBvcnQgY29uc3QgR0xPQkFMX1RFTkFOVFM6IHN0cmluZ1tdID0gWydnbG9iYWwnLCBHTE9CQUxfVEVOQU5UX1NZTUJPTF07XG4vKipcbiAqIFJlc292bGVzIHRoZSB0ZW5hbnQgdGhlIHVzZXIgaXMgdXNpbmcuXG4gKlxuICogQHBhcmFtIHJlcXVlc3QgT3BlblNlYXJjaERhc2hib2FyZHMgcmVxdWVzdC5cbiAqIEBwYXJhbSBjb25maWcgc2VjdXJpdHkgcGx1Z2luIGNvbmZpZy5cbiAqIEBwYXJhbSBjb29raWUgY29va2llIGV4dHJhY3RlZCBmcm9tIHRoZSByZXF1ZXN0LiBUaGUgY29va2llIHNob3VsZCBoYXZlIGJlZW4gcGFyc2VkIGJ5IEF1dGhlbnRpY2F0aW9uSGFuZGxlci5cbiAqIHBhc3MgaXQgYXMgcGFyYW1ldGVyIGluc3RlYWQgb2YgZXh0cmFjdGluZyBhZ2Fpbi5cbiAqIEBwYXJhbSBhdXRoSW5mbyBhdXRoZW50aWNhdGlvbiBpbmZvLCB0aGUgRWxhc3RpY3NlYXJjaCBhdXRoaW5mbyBBUEkgcmVzcG9uc2UuXG4gKlxuICogQHJldHVybnMgdXNlciBwcmVmZXJyZWQgdGVuYW50IG9mIHRoZSByZXF1ZXN0LlxuICovXG5leHBvcnQgZnVuY3Rpb24gcmVzb2x2ZVRlbmFudChcbiAgcmVxdWVzdDogT3BlblNlYXJjaERhc2hib2FyZHNSZXF1ZXN0LFxuICB1c2VybmFtZTogc3RyaW5nLFxuICByb2xlczogc3RyaW5nW10gfCB1bmRlZmluZWQsXG4gIGF2YWlsYWJlVGVuYW50czogYW55LFxuICBjb25maWc6IFNlY3VyaXR5UGx1Z2luQ29uZmlnVHlwZSxcbiAgY29va2llOiBTZWN1cml0eVNlc3Npb25Db29raWVcbik6IHN0cmluZyB8IHVuZGVmaW5lZCB7XG4gIGNvbnN0IERFRkFVTFRfUkVBRE9OTFlfUk9MRVMgPSBbJ2tpYmFuYV9yZWFkX29ubHknXTtcbiAgbGV0IHNlbGVjdGVkVGVuYW50OiBzdHJpbmcgfCB1bmRlZmluZWQ7XG4gIGNvbnN0IHNlY3VyaXR5VGVuYW50XyA9IHJlcXVlc3Q/LnVybD8uc2VhcmNoUGFyYW1zPy5nZXQoJ3NlY3VyaXR5VGVuYW50XycpO1xuICBjb25zdCBzZWN1cml0eXRlbmFudCA9IHJlcXVlc3Q/LnVybD8uc2VhcmNoUGFyYW1zPy5nZXQoJ3NlY3VyaXR5dGVuYW50Jyk7XG4gIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBAdHlwZXNjcmlwdC1lc2xpbnQvbmFtaW5nLWNvbnZlbnRpb25cbiAgY29uc3Qgc2VjdXJpdHlfdGVuYW50ID0gcmVxdWVzdD8udXJsPy5zZWFyY2hQYXJhbXM/LmdldCgnc2VjdXJpdHlfdGVuYW50Jyk7XG4gIGlmIChzZWN1cml0eVRlbmFudF8pIHtcbiAgICBzZWxlY3RlZFRlbmFudCA9IHNlY3VyaXR5VGVuYW50XztcbiAgfSBlbHNlIGlmIChzZWN1cml0eXRlbmFudCkge1xuICAgIHNlbGVjdGVkVGVuYW50ID0gc2VjdXJpdHl0ZW5hbnQ7XG4gIH0gZWxzZSBpZiAoc2VjdXJpdHlfdGVuYW50KSB7XG4gICAgc2VsZWN0ZWRUZW5hbnQgPSBzZWN1cml0eV90ZW5hbnQ7XG4gIH0gZWxzZSBpZiAocmVxdWVzdC5oZWFkZXJzLnNlY3VyaXR5dGVuYW50IHx8IHJlcXVlc3QuaGVhZGVycy5zZWN1cml0eVRlbmFudF8pIHtcbiAgICBzZWxlY3RlZFRlbmFudCA9IHJlcXVlc3QuaGVhZGVycy5zZWN1cml0eXRlbmFudFxuICAgICAgPyAocmVxdWVzdC5oZWFkZXJzLnNlY3VyaXR5dGVuYW50IGFzIHN0cmluZylcbiAgICAgIDogKHJlcXVlc3QuaGVhZGVycy5zZWN1cml0eVRlbmFudF8gYXMgc3RyaW5nKTtcbiAgfSBlbHNlIGlmIChpc1ZhbGlkVGVuYW50KGNvb2tpZS50ZW5hbnQpKSB7XG4gICAgc2VsZWN0ZWRUZW5hbnQgPSBjb29raWUudGVuYW50O1xuICB9IGVsc2Uge1xuICAgIHNlbGVjdGVkVGVuYW50ID0gdW5kZWZpbmVkO1xuICB9XG4gIGNvbnN0IGlzUmVhZG9ubHkgPSByb2xlcz8uc29tZShcbiAgICAocm9sZSkgPT4gY29uZmlnLnJlYWRvbmx5X21vZGU/LnJvbGVzLmluY2x1ZGVzKHJvbGUpIHx8IERFRkFVTFRfUkVBRE9OTFlfUk9MRVMuaW5jbHVkZXMocm9sZSlcbiAgKTtcblxuICBjb25zdCBwcmVmZXJyZWRUZW5hbnRzID0gY29uZmlnLm11bHRpdGVuYW5jeT8udGVuYW50cy5wcmVmZXJyZWQ7XG4gIGNvbnN0IGdsb2JhbFRlbmFudEVuYWJsZWQgPSBjb25maWcubXVsdGl0ZW5hbmN5Py50ZW5hbnRzLmVuYWJsZV9nbG9iYWw7XG4gIGNvbnN0IHByaXZhdGVUZW5hbnRFbmFibGVkID0gY29uZmlnLm11bHRpdGVuYW5jeT8udGVuYW50cy5lbmFibGVfcHJpdmF0ZSAmJiAhaXNSZWFkb25seTtcblxuICByZXR1cm4gcmVzb2x2ZShcbiAgICB1c2VybmFtZSxcbiAgICBzZWxlY3RlZFRlbmFudCxcbiAgICBwcmVmZXJyZWRUZW5hbnRzLFxuICAgIGF2YWlsYWJlVGVuYW50cyxcbiAgICBnbG9iYWxUZW5hbnRFbmFibGVkLFxuICAgIHByaXZhdGVUZW5hbnRFbmFibGVkXG4gICk7XG59XG5cbmZ1bmN0aW9uIHJlc29sdmUoXG4gIHVzZXJuYW1lOiBzdHJpbmcsXG4gIHJlcXVlc3RlZFRlbmFudDogc3RyaW5nIHwgdW5kZWZpbmVkLFxuICBwcmVmZXJyZWRUZW5hbnRzOiBzdHJpbmdbXSB8IHVuZGVmaW5lZCxcbiAgYXZhaWxhYmxlVGVuYW50czogYW55LCAvLyBpcyBhbiBvYmplY3QgbGlrZSB7IHRlbmFudF9uYW1lXzE6IHRydWUsIHRlbmFudF9uYW1lXzI6IGZhbHNlLCAuLi4gfVxuICBnbG9iYWxUZW5hbnRFbmFibGVkOiBib29sZWFuLFxuICBwcml2YXRlVGVuYW50RW5hYmxlZDogYm9vbGVhblxuKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgY29uc3QgYXZhaWxhYmxlVGVuYW50c0Nsb25lID0gY2xvbmVEZWVwKGF2YWlsYWJsZVRlbmFudHMpO1xuICBkZWxldGUgYXZhaWxhYmxlVGVuYW50c0Nsb25lW3VzZXJuYW1lXTtcblxuICBpZiAoIWdsb2JhbFRlbmFudEVuYWJsZWQgJiYgIXByaXZhdGVUZW5hbnRFbmFibGVkICYmIGlzRW1wdHkoYXZhaWxhYmxlVGVuYW50c0Nsb25lKSkge1xuICAgIHJldHVybiB1bmRlZmluZWQ7XG4gIH1cblxuICBpZiAoaXNWYWxpZFRlbmFudChyZXF1ZXN0ZWRUZW5hbnQpKSB7XG4gICAgcmVxdWVzdGVkVGVuYW50ID0gcmVxdWVzdGVkVGVuYW50ITtcbiAgICBpZiAocmVxdWVzdGVkVGVuYW50IGluIGF2YWlsYWJsZVRlbmFudHMpIHtcbiAgICAgIHJldHVybiByZXF1ZXN0ZWRUZW5hbnQ7XG4gICAgfVxuXG4gICAgaWYgKFxuICAgICAgcHJpdmF0ZVRlbmFudEVuYWJsZWQgJiZcbiAgICAgIHVzZXJuYW1lIGluIGF2YWlsYWJsZVRlbmFudHMgJiZcbiAgICAgIFBSSVZBVEVfVEVOQU5UUy5pbmRleE9mKHJlcXVlc3RlZFRlbmFudCkgPiAtMVxuICAgICkge1xuICAgICAgcmV0dXJuIFBSSVZBVEVfVEVOQU5UX1NZTUJPTDtcbiAgICB9XG5cbiAgICBpZiAoZ2xvYmFsVGVuYW50RW5hYmxlZCAmJiBHTE9CQUxfVEVOQU5UUy5pbmRleE9mKHJlcXVlc3RlZFRlbmFudCkgPiAtMSkge1xuICAgICAgcmV0dXJuIEdMT0JBTF9URU5BTlRfU1lNQk9MO1xuICAgIH1cbiAgfVxuXG4gIGlmIChwcmVmZXJyZWRUZW5hbnRzICYmICFpc0VtcHR5KHByZWZlcnJlZFRlbmFudHMpKSB7XG4gICAgZm9yIChjb25zdCBlbGVtZW50IG9mIHByZWZlcnJlZFRlbmFudHMpIHtcbiAgICAgIGNvbnN0IHRlbmFudCA9IGVsZW1lbnQudG9Mb3dlckNhc2UoKTtcblxuICAgICAgaWYgKGdsb2JhbFRlbmFudEVuYWJsZWQgJiYgR0xPQkFMX1RFTkFOVFMuaW5kZXhPZih0ZW5hbnQpID4gLTEpIHtcbiAgICAgICAgcmV0dXJuIEdMT0JBTF9URU5BTlRfU1lNQk9MO1xuICAgICAgfVxuXG4gICAgICBpZiAoXG4gICAgICAgIHByaXZhdGVUZW5hbnRFbmFibGVkICYmXG4gICAgICAgIFBSSVZBVEVfVEVOQU5UUy5pbmRleE9mKHRlbmFudCkgPiAtMSAmJlxuICAgICAgICB1c2VybmFtZSBpbiBhdmFpbGFibGVUZW5hbnRzXG4gICAgICApIHtcbiAgICAgICAgcmV0dXJuIFBSSVZBVEVfVEVOQU5UX1NZTUJPTDtcbiAgICAgIH1cblxuICAgICAgaWYgKHRlbmFudCBpbiBhdmFpbGFibGVUZW5hbnRzKSB7XG4gICAgICAgIHJldHVybiB0ZW5hbnQ7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgaWYgKGdsb2JhbFRlbmFudEVuYWJsZWQpIHtcbiAgICByZXR1cm4gR0xPQkFMX1RFTkFOVF9TWU1CT0w7XG4gIH1cblxuICBpZiAocHJpdmF0ZVRlbmFudEVuYWJsZWQpIHtcbiAgICByZXR1cm4gUFJJVkFURV9URU5BTlRfU1lNQk9MO1xuICB9XG5cbiAgLy8gZmFsbCBiYWNrIHRvIHRoZSBmaXJzdCB0ZW5hbnQgaW4gdGhlIGF2YWlsYWJsZSB0ZW5hbnRzXG4gIHJldHVybiBmaW5kS2V5KGF2YWlsYWJsZVRlbmFudHNDbG9uZSwgKCkgPT4gdHJ1ZSk7XG59XG5cbi8qKlxuICogUmV0dXJuIHRydWUgaWYgdGVuYW50IHBhcmFtZXRlciBpcyBhIHZhbGlkIHRlbmVudC5cbiAqXG4gKiBOb3RlOiBlbXB0eSBzdHJpbmcgJycgaXMgdmFsaWQsIHdoaWNoIG1lYW5zIGdsb2JhbCB0ZW5hbnQuXG4gKlxuICogQHBhcmFtIHRlbmFudFxuICovXG5leHBvcnQgZnVuY3Rpb24gaXNWYWxpZFRlbmFudCh0ZW5hbnQ6IHN0cmluZyB8IHVuZGVmaW5lZCB8IG51bGwpOiBib29sZWFuIHtcbiAgcmV0dXJuIHRlbmFudCAhPT0gdW5kZWZpbmVkICYmIHRlbmFudCAhPT0gbnVsbDtcbn1cbiJdfQ==