/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.index.store;

import java.io.IOException;
import java.nio.file.NoSuchFileException;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.codecs.CodecUtil;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FilterDirectory;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.IndexOutput;
import org.opensearch.common.UUIDs;
import org.opensearch.index.store.RemoteDirectory;

public final class RemoteSegmentStoreDirectory
extends FilterDirectory {
    public static final String SEGMENT_NAME_UUID_SEPARATOR = "__";
    public static final MetadataFilenameUtils.MetadataFilenameComparator METADATA_FILENAME_COMPARATOR = new MetadataFilenameUtils.MetadataFilenameComparator();
    private final RemoteDirectory remoteDataDirectory;
    private final RemoteDirectory remoteMetadataDirectory;
    private String metadataFileUniqueSuffix;
    private Map<String, UploadedSegmentMetadata> segmentsUploadedToRemoteStore;
    private static final Logger logger = LogManager.getLogger(RemoteSegmentStoreDirectory.class);

    public RemoteSegmentStoreDirectory(RemoteDirectory remoteDataDirectory, RemoteDirectory remoteMetadataDirectory) throws IOException {
        super((Directory)remoteDataDirectory);
        this.remoteDataDirectory = remoteDataDirectory;
        this.remoteMetadataDirectory = remoteMetadataDirectory;
        this.init();
    }

    public void init() throws IOException {
        this.metadataFileUniqueSuffix = UUIDs.base64UUID();
        this.segmentsUploadedToRemoteStore = new ConcurrentHashMap<String, UploadedSegmentMetadata>(this.readLatestMetadataFile());
    }

    private Map<String, UploadedSegmentMetadata> readLatestMetadataFile() throws IOException {
        Map<String, UploadedSegmentMetadata> segmentMetadataMap = new HashMap<String, UploadedSegmentMetadata>();
        Collection<String> metadataFiles = this.remoteMetadataDirectory.listFilesByPrefix("metadata");
        Optional<String> latestMetadataFile = metadataFiles.stream().max(METADATA_FILENAME_COMPARATOR);
        if (latestMetadataFile.isPresent()) {
            logger.info("Reading latest Metadata file {}", (Object)latestMetadataFile.get());
            segmentMetadataMap = this.readMetadataFile(latestMetadataFile.get());
        } else {
            logger.info("No metadata file found, this can happen for new index with no data uploaded to remote segment store");
        }
        return segmentMetadataMap;
    }

    private Map<String, UploadedSegmentMetadata> readMetadataFile(String metadataFilename) throws IOException {
        try (IndexInput indexInput = this.remoteMetadataDirectory.openInput(metadataFilename, IOContext.DEFAULT);){
            Map segmentMetadata = indexInput.readMapOfStrings();
            Map<String, UploadedSegmentMetadata> map = segmentMetadata.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> UploadedSegmentMetadata.fromString((String)entry.getValue())));
            return map;
        }
    }

    public String[] listAll() throws IOException {
        return this.readLatestMetadataFile().keySet().toArray(new String[0]);
    }

    public void deleteFile(String name) throws IOException {
        String remoteFilename = this.getExistingRemoteFilename(name);
        if (remoteFilename != null) {
            this.remoteDataDirectory.deleteFile(remoteFilename);
            this.segmentsUploadedToRemoteStore.remove(name);
        }
    }

    public long fileLength(String name) throws IOException {
        String remoteFilename = this.getExistingRemoteFilename(name);
        if (remoteFilename != null) {
            return this.remoteDataDirectory.fileLength(remoteFilename);
        }
        throw new NoSuchFileException(name);
    }

    public IndexOutput createOutput(String name, IOContext context) throws IOException {
        return this.remoteDataDirectory.createOutput(this.getNewRemoteSegmentFilename(name), context);
    }

    public IndexInput openInput(String name, IOContext context) throws IOException {
        String remoteFilename = this.getExistingRemoteFilename(name);
        if (remoteFilename != null) {
            return this.remoteDataDirectory.openInput(remoteFilename, context);
        }
        throw new NoSuchFileException(name);
    }

    public void copyFrom(Directory from, String src, String dest, IOContext context) throws IOException {
        String remoteFilename = this.getNewRemoteSegmentFilename(dest);
        this.remoteDataDirectory.copyFrom(from, src, remoteFilename, context);
        String checksum = this.getChecksumOfLocalFile(from, src);
        UploadedSegmentMetadata segmentMetadata = new UploadedSegmentMetadata(src, remoteFilename, checksum);
        this.segmentsUploadedToRemoteStore.put(src, segmentMetadata);
    }

    public boolean containsFile(String localFilename, String checksum) {
        return this.segmentsUploadedToRemoteStore.containsKey(localFilename) && this.segmentsUploadedToRemoteStore.get((Object)localFilename).checksum.equals(checksum);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void uploadMetadata(Collection<String> segmentFiles, Directory storeDirectory, long primaryTerm, long generation) throws IOException {
        RemoteSegmentStoreDirectory remoteSegmentStoreDirectory = this;
        synchronized (remoteSegmentStoreDirectory) {
            String metadataFilename = MetadataFilenameUtils.getMetadataFilename(primaryTerm, generation, this.metadataFileUniqueSuffix);
            IndexOutput indexOutput = storeDirectory.createOutput(metadataFilename, IOContext.DEFAULT);
            HashMap<String, String> uploadedSegments = new HashMap<String, String>();
            for (String file : segmentFiles) {
                if (this.segmentsUploadedToRemoteStore.containsKey(file)) {
                    uploadedSegments.put(file, this.segmentsUploadedToRemoteStore.get(file).toString());
                    continue;
                }
                throw new NoSuchFileException(file);
            }
            indexOutput.writeMapOfStrings(uploadedSegments);
            indexOutput.close();
            storeDirectory.sync(Collections.singleton(metadataFilename));
            this.remoteMetadataDirectory.copyFrom(storeDirectory, metadataFilename, metadataFilename, IOContext.DEFAULT);
            storeDirectory.deleteFile(metadataFilename);
        }
    }

    private String getChecksumOfLocalFile(Directory directory, String file) throws IOException {
        try (IndexInput indexInput = directory.openInput(file, IOContext.DEFAULT);){
            String string = Long.toString(CodecUtil.retrieveChecksum((IndexInput)indexInput));
            return string;
        }
    }

    private String getExistingRemoteFilename(String localFilename) {
        if (this.segmentsUploadedToRemoteStore.containsKey(localFilename)) {
            return this.segmentsUploadedToRemoteStore.get((Object)localFilename).uploadedFilename;
        }
        return null;
    }

    private String getNewRemoteSegmentFilename(String localFilename) {
        return localFilename + SEGMENT_NAME_UUID_SEPARATOR + UUIDs.base64UUID();
    }

    private String getLocalSegmentFilename(String remoteFilename) {
        return remoteFilename.split(SEGMENT_NAME_UUID_SEPARATOR)[0];
    }

    public Map<String, UploadedSegmentMetadata> getSegmentsUploadedToRemoteStore() {
        return Collections.unmodifiableMap(this.segmentsUploadedToRemoteStore);
    }

    public void deleteStaleSegments(int lastNMetadataFilesToKeep) throws IOException {
        Collection<String> metadataFiles = this.remoteMetadataDirectory.listFilesByPrefix("metadata");
        List sortedMetadataFileList = metadataFiles.stream().sorted(METADATA_FILENAME_COMPARATOR).collect(Collectors.toList());
        if (sortedMetadataFileList.size() <= lastNMetadataFilesToKeep) {
            logger.info("Number of commits in remote segment store={}, lastNMetadataFilesToKeep={}", (Object)sortedMetadataFileList.size(), (Object)lastNMetadataFilesToKeep);
            return;
        }
        List latestNMetadataFiles = sortedMetadataFileList.subList(sortedMetadataFileList.size() - lastNMetadataFilesToKeep, sortedMetadataFileList.size());
        HashMap<String, UploadedSegmentMetadata> activeSegmentFilesMetadataMap = new HashMap<String, UploadedSegmentMetadata>();
        HashSet activeSegmentRemoteFilenames = new HashSet();
        for (String metadataFile : latestNMetadataFiles) {
            Map<String, UploadedSegmentMetadata> segmentMetadataMap = this.readMetadataFile(metadataFile);
            activeSegmentFilesMetadataMap.putAll(segmentMetadataMap);
            activeSegmentRemoteFilenames.addAll(segmentMetadataMap.values().stream().map(metadata -> metadata.uploadedFilename).collect(Collectors.toSet()));
        }
        for (String metadataFile : sortedMetadataFileList.subList(0, sortedMetadataFileList.size() - lastNMetadataFilesToKeep)) {
            Map<String, UploadedSegmentMetadata> staleSegmentFilesMetadataMap = this.readMetadataFile(metadataFile);
            Set staleSegmentRemoteFilenames = staleSegmentFilesMetadataMap.values().stream().map(metadata -> metadata.uploadedFilename).collect(Collectors.toSet());
            AtomicBoolean deletionSuccessful = new AtomicBoolean(true);
            staleSegmentRemoteFilenames.stream().filter(file -> !activeSegmentRemoteFilenames.contains(file)).forEach(file -> {
                try {
                    this.remoteDataDirectory.deleteFile((String)file);
                    if (!activeSegmentFilesMetadataMap.containsKey(this.getLocalSegmentFilename((String)file))) {
                        this.segmentsUploadedToRemoteStore.remove(this.getLocalSegmentFilename((String)file));
                    }
                }
                catch (NoSuchFileException e) {
                    logger.info("Segment file {} corresponding to metadata file {} does not exist in remote", file, (Object)metadataFile);
                }
                catch (IOException e) {
                    deletionSuccessful.set(false);
                    logger.info("Exception while deleting segment file {} corresponding to metadata file {}. Deletion will be re-tried", file, (Object)metadataFile);
                }
            });
            if (!deletionSuccessful.get()) continue;
            logger.info("Deleting stale metadata file {} from remote segment store", (Object)metadataFile);
            this.remoteMetadataDirectory.deleteFile(metadataFile);
        }
    }

    static class MetadataFilenameUtils {
        public static final String SEPARATOR = "__";
        public static final String METADATA_PREFIX = "metadata";

        MetadataFilenameUtils() {
        }

        static String getMetadataFilename(long primaryTerm, long generation, String uuid) {
            return String.join((CharSequence)"__", METADATA_PREFIX, Long.toString(primaryTerm), Long.toString(generation, 36), uuid);
        }

        static long getPrimaryTerm(String[] filenameTokens) {
            return Long.parseLong(filenameTokens[1]);
        }

        static long getGeneration(String[] filenameTokens) {
            return Long.parseLong(filenameTokens[2], 36);
        }

        static String getUuid(String[] filenameTokens) {
            return filenameTokens[3];
        }

        static class MetadataFilenameComparator
        implements Comparator<String> {
            MetadataFilenameComparator() {
            }

            @Override
            public int compare(String first, String second) {
                long secondGeneration;
                long secondPrimaryTerm;
                String[] secondTokens;
                String[] firstTokens = first.split("__");
                if (!firstTokens[0].equals((secondTokens = second.split("__"))[0])) {
                    return firstTokens[0].compareTo(secondTokens[0]);
                }
                long firstPrimaryTerm = MetadataFilenameUtils.getPrimaryTerm(firstTokens);
                if (firstPrimaryTerm != (secondPrimaryTerm = MetadataFilenameUtils.getPrimaryTerm(secondTokens))) {
                    return firstPrimaryTerm > secondPrimaryTerm ? 1 : -1;
                }
                long firstGeneration = MetadataFilenameUtils.getGeneration(firstTokens);
                if (firstGeneration != (secondGeneration = MetadataFilenameUtils.getGeneration(secondTokens))) {
                    return firstGeneration > secondGeneration ? 1 : -1;
                }
                return MetadataFilenameUtils.getUuid(firstTokens).compareTo(MetadataFilenameUtils.getUuid(secondTokens));
            }
        }
    }

    public static class UploadedSegmentMetadata {
        static final String SEPARATOR = "::";
        private final String originalFilename;
        private final String uploadedFilename;
        private final String checksum;

        UploadedSegmentMetadata(String originalFilename, String uploadedFilename, String checksum) {
            this.originalFilename = originalFilename;
            this.uploadedFilename = uploadedFilename;
            this.checksum = checksum;
        }

        public String toString() {
            return String.join((CharSequence)SEPARATOR, this.originalFilename, this.uploadedFilename, this.checksum);
        }

        public static UploadedSegmentMetadata fromString(String uploadedFilename) {
            String[] values = uploadedFilename.split(SEPARATOR);
            return new UploadedSegmentMetadata(values[0], values[1], values[2]);
        }
    }
}

