"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createVisualReport = void 0;

var _puppeteerCore = _interopRequireDefault(require("puppeteer-core"));

var _dompurify = _interopRequireDefault(require("dompurify"));

var _jsdom = require("jsdom");

var _constants = require("../constants");

var _helpers = require("../helpers");

var _fs = _interopRequireDefault(require("fs"));

var _lodash = _interopRequireDefault(require("lodash"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const createVisualReport = async (reportParams, queryUrl, logger, extraHeaders, timezone, validRequestProtocol = /^(data:image)/) => {
  const {
    core_params,
    report_name: reportName,
    report_source: reportSource
  } = reportParams;
  const coreParams = core_params;
  const {
    header,
    footer,
    window_height: windowHeight,
    window_width: windowWidth,
    report_format: reportFormat
  } = coreParams;
  const window = new _jsdom.JSDOM('').window;
  const DOMPurify = (0, _dompurify.default)(window);
  let keywordFilteredHeader = header ? _constants.converter.makeHtml(header) : _constants.DEFAULT_REPORT_HEADER;
  let keywordFilteredFooter = footer ? _constants.converter.makeHtml(footer) : '';
  keywordFilteredHeader = DOMPurify.sanitize(keywordFilteredHeader);
  keywordFilteredFooter = DOMPurify.sanitize(keywordFilteredFooter); // filter blocked keywords in header and footer

  if (keywordFilteredHeader !== '') {
    keywordFilteredHeader = (0, _constants.replaceBlockedKeywords)(keywordFilteredHeader);
  }

  if (keywordFilteredFooter !== '') {
    keywordFilteredFooter = (0, _constants.replaceBlockedKeywords)(keywordFilteredFooter);
  } // set up puppeteer


  const browser = await _puppeteerCore.default.launch({
    headless: true,

    /**
     * TODO: temp fix to disable sandbox when launching chromium on Linux instance
     * https://github.com/puppeteer/puppeteer/blob/main/docs/troubleshooting.md#setting-up-chrome-linux-sandbox
     */
    args: ['--no-sandbox', '--disable-setuid-sandbox', '--disable-gpu', '--no-zygote', '--font-render-hinting=none', '--js-flags="--jitless --no-opt"', '--disable-features=V8OptimizeJavascript'],
    executablePath: _constants.CHROMIUM_PATH,
    ignoreHTTPSErrors: true,
    env: {
      TZ: timezone || 'UTC'
    },
    pipe: true
  });
  const page = await browser.newPage();
  await page.setRequestInterception(true);
  let localStorageAvailable = true;
  page.on('request', req => {
    // disallow non-allowlisted connections. urls with valid protocols do not need ALLOWED_HOSTS check
    if (!validRequestProtocol.test(req.url()) && !_constants.ALLOWED_HOSTS.test(new URL(req.url()).hostname)) {
      if (req.isNavigationRequest() && req.redirectChain().length > 0) {
        localStorageAvailable = false;
        logger.error('Reporting does not allow redirections to outside of localhost, aborting. URL received: ' + req.url());
      } else {
        logger.warn('Disabled connection to non-allowlist domains: ' + req.url());
      }

      req.abort();
    } else {
      req.continue();
    }
  });
  page.setDefaultNavigationTimeout(0);
  page.setDefaultTimeout(300000); // use 300s timeout instead of default 30s
  // Set extra headers that are needed

  if (!_lodash.default.isEmpty(extraHeaders)) {
    await page.setExtraHTTPHeaders(extraHeaders);
  }

  logger.info(`original queryUrl ${queryUrl}`);
  await page.goto(queryUrl, {
    waitUntil: 'networkidle0'
  }); // should add to local storage after page.goto, then access the page again - browser must have an url to register local storage item on it

  try {
    await page.evaluate(
    /* istanbul ignore next */
    key => {
      try {
        if (localStorageAvailable && typeof localStorage !== 'undefined' && localStorage !== null) {
          localStorage.setItem(key, 'false');
        }
      } catch (err) {}
    }, _constants.SECURITY_CONSTANTS.TENANT_LOCAL_STORAGE_KEY);
  } catch (err) {
    logger.error(err);
  }

  await page.goto(queryUrl, {
    waitUntil: 'networkidle0'
  });
  logger.info(`page url ${page.url()}`);
  await page.setViewport({
    width: windowWidth,
    height: windowHeight
  });
  let buffer; // remove unwanted elements

  await page.evaluate(
  /* istanbul ignore next */
  (reportSource, REPORT_TYPE) => {
    // remove buttons
    document.querySelectorAll("[class^='euiButton']").forEach(e => e.remove()); // remove top navBar

    document.querySelectorAll("[class^='euiHeader']").forEach(e => e.remove()); // remove visualization editor

    if (reportSource === REPORT_TYPE.visualization) {
      var _document$querySelect, _document$querySelect2;

      (_document$querySelect = document.querySelector('[data-test-subj="splitPanelResizer"]')) === null || _document$querySelect === void 0 ? void 0 : _document$querySelect.remove();
      (_document$querySelect2 = document.querySelector('.visEditor__collapsibleSidebar')) === null || _document$querySelect2 === void 0 ? void 0 : _document$querySelect2.remove();
    }

    document.body.style.paddingTop = '0px';
  }, reportSource, _constants.REPORT_TYPE); // force wait for any resize to load after the above DOM modification

  await new Promise(resolve => setTimeout(resolve, 1000)); // crop content

  switch (reportSource) {
    case _constants.REPORT_TYPE.dashboard:
      await page.waitForSelector(_constants.SELECTOR.dashboard, {
        visible: true
      });
      break;

    case _constants.REPORT_TYPE.visualization:
      await page.waitForSelector(_constants.SELECTOR.visualization, {
        visible: true
      });
      break;

    case _constants.REPORT_TYPE.notebook:
      await page.waitForSelector(_constants.SELECTOR.notebook, {
        visible: true
      });
      break;

    default:
      throw Error(`report source can only be one of [Dashboard, Visualization]`);
  } // wait for dynamic page content to render


  await waitForDynamicContent(page);
  await addReportHeader(page, keywordFilteredHeader);
  await addReportFooter(page, keywordFilteredFooter);
  await addReportStyle(page); // this causes UT to fail in github CI but works locally

  try {
    const numDisallowedTags = await page.evaluate(() => document.getElementsByTagName('iframe').length + document.getElementsByTagName('embed').length + document.getElementsByTagName('object').length);

    if (numDisallowedTags > 0) {
      throw Error('Reporting does not support "iframe", "embed", or "object" tags, aborting');
    }
  } catch (error) {
    logger.error(error);
  } // create pdf or png accordingly


  if (reportFormat === _constants.FORMAT.pdf) {
    const scrollHeight = await page.evaluate(
    /* istanbul ignore next */
    () => document.documentElement.scrollHeight);
    buffer = await page.pdf({
      margin: undefined,
      width: windowWidth,
      height: scrollHeight + 'px',
      printBackground: true,
      pageRanges: '1'
    });
  } else if (reportFormat === _constants.FORMAT.png) {
    buffer = await page.screenshot({
      fullPage: true
    });
  }

  const curTime = new Date();
  const timeCreated = curTime.valueOf();
  const fileName = `${(0, _helpers.getFileName)(reportName, curTime)}.${reportFormat}`;
  await browser.close();
  return {
    timeCreated,
    dataUrl: buffer.toString('base64'),
    fileName
  };
};

exports.createVisualReport = createVisualReport;

const addReportStyle = async page => {
  const css = _fs.default.readFileSync(`${__dirname}/style.css`).toString();

  await page.evaluate(
  /* istanbul ignore next */
  style => {
    const styleElement = document.createElement('style');
    styleElement.innerHTML = style;
    document.getElementsByTagName('head')[0].appendChild(styleElement);
  }, css);
};

const addReportHeader = async (page, header) => {
  const headerHtml = _fs.default.readFileSync(`${__dirname}/header_template.html`).toString().replace('<!--CONTENT-->', header);

  await page.evaluate(
  /* istanbul ignore next */
  headerHtml => {
    var _content$parentNode;

    const content = document.body.firstChild;
    const headerContainer = document.createElement('div');
    headerContainer.className = 'reportWrapper';
    headerContainer.innerHTML = headerHtml;
    content === null || content === void 0 ? void 0 : (_content$parentNode = content.parentNode) === null || _content$parentNode === void 0 ? void 0 : _content$parentNode.insertBefore(headerContainer, content);
  }, headerHtml);
};

const addReportFooter = async (page, footer) => {
  const headerHtml = _fs.default.readFileSync(`${__dirname}/footer_template.html`).toString().replace('<!--CONTENT-->', footer);

  await page.evaluate(
  /* istanbul ignore next */
  headerHtml => {
    var _content$parentNode2;

    const content = document.body.firstChild;
    const headerContainer = document.createElement('div');
    headerContainer.className = 'reportWrapper';
    headerContainer.innerHTML = headerHtml;
    content === null || content === void 0 ? void 0 : (_content$parentNode2 = content.parentNode) === null || _content$parentNode2 === void 0 ? void 0 : _content$parentNode2.insertBefore(headerContainer, null);
  }, headerHtml);
}; // add waitForDynamicContent function


const waitForDynamicContent = async (page, timeout = 30000, interval = 1000, checks = 5) => {
  const maxChecks = timeout / interval;
  let passedChecks = 0;
  let previousLength = 0;
  let i = 0;

  while (i++ <= maxChecks) {
    let pageContent = await page.content();
    let currentLength = pageContent.length;
    previousLength === 0 || previousLength != currentLength ? passedChecks = 0 : passedChecks++;

    if (passedChecks >= checks) {
      break;
    }

    previousLength = currentLength;
    await new Promise(resolve => setTimeout(resolve, interval));
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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