/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.ad.caching;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.PriorityQueue;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.opensearch.action.ActionListener;
import org.opensearch.ad.MemoryTracker;
import org.opensearch.ad.caching.CacheBuffer;
import org.opensearch.ad.caching.DoorKeeper;
import org.opensearch.ad.caching.EntityCache;
import org.opensearch.ad.common.exception.AnomalyDetectionException;
import org.opensearch.ad.common.exception.LimitExceededException;
import org.opensearch.ad.ml.CheckpointDao;
import org.opensearch.ad.ml.EntityModel;
import org.opensearch.ad.ml.ModelManager;
import org.opensearch.ad.ml.ModelState;
import org.opensearch.ad.model.AnomalyDetector;
import org.opensearch.ad.model.Entity;
import org.opensearch.ad.model.ModelProfile;
import org.opensearch.ad.ratelimit.CheckpointMaintainWorker;
import org.opensearch.ad.ratelimit.CheckpointWriteWorker;
import org.opensearch.ad.settings.AnomalyDetectorSettings;
import org.opensearch.ad.util.DateUtils;
import org.opensearch.cluster.service.ClusterService;
import org.opensearch.common.Strings;
import org.opensearch.common.settings.Setting;
import org.opensearch.common.settings.Settings;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.threadpool.ThreadPool;

public class PriorityCache
implements EntityCache {
    private final Logger LOG = LogManager.getLogger(PriorityCache.class);
    private final Map<String, CacheBuffer> activeEnities;
    private final CheckpointDao checkpointDao;
    private volatile int dedicatedCacheSize;
    private Cache<String, ModelState<EntityModel>> inActiveEntities;
    private final MemoryTracker memoryTracker;
    private final ReentrantLock maintenanceLock;
    private final int numberOfTrees;
    private final Clock clock;
    private final Duration modelTtl;
    private Map<String, DoorKeeper> doorKeepers;
    private ThreadPool threadPool;
    private Random random;
    private CheckpointWriteWorker checkpointWriteQueue;
    private Instant lastInActiveEntityMaintenance;
    protected int maintenanceFreqConstant;
    private CheckpointMaintainWorker checkpointMaintainQueue;
    private int checkpointIntervalHrs;

    public PriorityCache(CheckpointDao checkpointDao, int dedicatedCacheSize, Setting<TimeValue> checkpointTtl, int maxInactiveStates, MemoryTracker memoryTracker, int numberOfTrees, Clock clock, ClusterService clusterService, Duration modelTtl, ThreadPool threadPool, CheckpointWriteWorker checkpointWriteQueue, int maintenanceFreqConstant, CheckpointMaintainWorker checkpointMaintainQueue, Settings settings, Setting<TimeValue> checkpointSavingFreq) {
        this.checkpointDao = checkpointDao;
        this.activeEnities = new ConcurrentHashMap<String, CacheBuffer>();
        this.dedicatedCacheSize = dedicatedCacheSize;
        clusterService.getClusterSettings().addSettingsUpdateConsumer(AnomalyDetectorSettings.DEDICATED_CACHE_SIZE, it -> {
            this.dedicatedCacheSize = it;
            this.setDedicatedCacheSizeListener();
            this.tryClearUpMemory();
        }, this::validateDedicatedCacheSize);
        clusterService.getClusterSettings().addSettingsUpdateConsumer(AnomalyDetectorSettings.MODEL_MAX_SIZE_PERCENTAGE, it -> this.tryClearUpMemory());
        this.memoryTracker = memoryTracker;
        this.maintenanceLock = new ReentrantLock();
        this.numberOfTrees = numberOfTrees;
        this.clock = clock;
        this.modelTtl = modelTtl;
        this.doorKeepers = new ConcurrentHashMap<String, DoorKeeper>();
        Duration inactiveEntityTtl = DateUtils.toDuration((TimeValue)checkpointTtl.get(settings));
        this.inActiveEntities = this.createInactiveCache(inactiveEntityTtl, maxInactiveStates);
        clusterService.getClusterSettings().addSettingsUpdateConsumer(checkpointTtl, it -> {
            this.inActiveEntities = this.createInactiveCache(DateUtils.toDuration(it), maxInactiveStates);
        });
        this.threadPool = threadPool;
        this.random = new Random(42L);
        this.checkpointWriteQueue = checkpointWriteQueue;
        this.lastInActiveEntityMaintenance = Instant.MIN;
        this.maintenanceFreqConstant = maintenanceFreqConstant;
        this.checkpointMaintainQueue = checkpointMaintainQueue;
        this.checkpointIntervalHrs = DateUtils.toDuration((TimeValue)checkpointSavingFreq.get(settings)).toHoursPart();
        clusterService.getClusterSettings().addSettingsUpdateConsumer(checkpointSavingFreq, it -> {
            this.checkpointIntervalHrs = DateUtils.toDuration(it).toHoursPart();
            this.setCheckpointFreqListener();
        });
    }

    @Override
    public ModelState<EntityModel> get(final String modelId, AnomalyDetector detector) {
        final String detectorId = detector.getDetectorId();
        CacheBuffer buffer = this.computeBufferIfAbsent(detector, detectorId);
        ModelState<EntityModel> modelState = buffer.get(modelId);
        if (!this.maintenanceLock.isLocked() && modelState == null) {
            DoorKeeper doorKeeper = this.doorKeepers.computeIfAbsent(detectorId, id -> new DoorKeeper(1000000L, 0.01, detector.getDetectionIntervalDuration().multipliedBy(60L), this.clock));
            if (!doorKeeper.mightContain(modelId) && !this.isActive(detectorId, modelId)) {
                doorKeeper.put(modelId);
                return null;
            }
            try {
                ModelState state = (ModelState)this.inActiveEntities.get((Object)modelId, (Callable)new Callable<ModelState<EntityModel>>(){

                    @Override
                    public ModelState<EntityModel> call() {
                        return new ModelState<Object>(null, modelId, detectorId, ModelManager.ModelType.ENTITY.getName(), PriorityCache.this.clock, 0.0f);
                    }
                });
                state.setModel(null);
                state.setPriority(buffer.getPriorityTracker().getUpdatedPriority(state.getPriority()));
                if (this.random.nextInt(this.maintenanceFreqConstant) == 1) {
                    this.tryClearUpMemory();
                }
            }
            catch (Exception e) {
                this.LOG.error((Message)new ParameterizedMessage("Fail to update priority of [{}]", (Object)modelId), (Throwable)e);
            }
        }
        return modelState;
    }

    private Optional<ModelState<EntityModel>> getStateFromInactiveEntiiyCache(String modelId) {
        if (modelId == null) {
            return Optional.empty();
        }
        return Optional.ofNullable((ModelState)this.inActiveEntities.getIfPresent((Object)modelId));
    }

    @Override
    public boolean hostIfPossible(AnomalyDetector detector, ModelState<EntityModel> toUpdate) {
        ModelState<EntityModel> removed;
        if (toUpdate == null) {
            return false;
        }
        String modelId = toUpdate.getModelId();
        String detectorId = toUpdate.getDetectorId();
        if (Strings.isEmpty((CharSequence)modelId) || Strings.isEmpty((CharSequence)detectorId)) {
            return false;
        }
        CacheBuffer buffer = this.computeBufferIfAbsent(detector, detectorId);
        Optional<ModelState<EntityModel>> state = this.getStateFromInactiveEntiiyCache(modelId);
        if (!state.isPresent()) {
            return false;
        }
        ModelState<EntityModel> modelState = state.get();
        float priority = modelState.getPriority();
        toUpdate.setLastUsedTime(this.clock.instant());
        toUpdate.setPriority(priority);
        if (buffer.dedicatedCacheAvailable() || this.memoryTracker.canAllocate(buffer.getMemoryConsumptionPerEntity())) {
            buffer.put(modelId, toUpdate);
            return true;
        }
        if (this.memoryTracker.canAllocate(buffer.getMemoryConsumptionPerEntity())) {
            buffer.put(modelId, toUpdate);
            return true;
        }
        if (buffer.canReplaceWithinDetector(priority) && (removed = buffer.replace(modelId, toUpdate)) != null) {
            this.addIntoInactiveCache(removed);
            return true;
        }
        float scaledPriority = buffer.getPriorityTracker().getScaledPriority(priority);
        Triple<CacheBuffer, String, Float> bufferToRemoveEntity = this.canReplaceInSharedCache(buffer, scaledPriority);
        CacheBuffer bufferToRemove = (CacheBuffer)bufferToRemoveEntity.getLeft();
        String entityModelId = (String)bufferToRemoveEntity.getMiddle();
        ModelState<EntityModel> removed2 = null;
        if (bufferToRemove != null && (removed2 = bufferToRemove.remove(entityModelId)) != null) {
            buffer.put(modelId, toUpdate);
            this.addIntoInactiveCache(removed2);
            return true;
        }
        return false;
    }

    private void addIntoInactiveCache(ModelState<EntityModel> removed) {
        if (removed == null) {
            return;
        }
        removed.setLastUsedTime(this.clock.instant());
        removed.setModel(null);
        this.inActiveEntities.put((Object)removed.getModelId(), removed);
    }

    private void addEntity(List<Entity> destination, Entity entity, String detectorId) {
        Optional<String> modelId;
        if (entity != null && (modelId = entity.getModelId(detectorId)).isPresent() && this.inActiveEntities.getIfPresent((Object)modelId.get()) != null) {
            destination.add(entity);
        }
    }

    @Override
    public Pair<List<Entity>, List<Entity>> selectUpdateCandidate(Collection<Entity> cacheMissEntities, String detectorId, AnomalyDetector detector) {
        ArrayList<Entity> hotEntities = new ArrayList<Entity>();
        ArrayList<Entity> coldEntities = new ArrayList<Entity>();
        CacheBuffer buffer = this.activeEnities.get(detectorId);
        if (buffer == null) {
            return Pair.of(hotEntities, coldEntities);
        }
        Iterator<Entity> cacheMissEntitiesIter = cacheMissEntities.iterator();
        while (cacheMissEntitiesIter.hasNext() && buffer.dedicatedCacheAvailable()) {
            this.addEntity(hotEntities, cacheMissEntitiesIter.next(), detectorId);
        }
        while (cacheMissEntitiesIter.hasNext() && this.memoryTracker.canAllocate(buffer.getMemoryConsumptionPerEntity())) {
            this.addEntity(hotEntities, cacheMissEntitiesIter.next(), detectorId);
        }
        ArrayList<Entity> otherBufferReplaceCandidates = new ArrayList<Entity>();
        while (cacheMissEntitiesIter.hasNext()) {
            Optional<ModelState<EntityModel>> state;
            Entity entity = cacheMissEntitiesIter.next();
            Optional<String> modelId = entity.getModelId(detectorId);
            if (!modelId.isPresent() || !(state = this.getStateFromInactiveEntiiyCache(modelId.get())).isPresent()) continue;
            ModelState<EntityModel> modelState = state.get();
            float priority = modelState.getPriority();
            if (buffer.canReplaceWithinDetector(priority)) {
                this.addEntity(hotEntities, entity, detectorId);
                continue;
            }
            otherBufferReplaceCandidates.add(entity);
        }
        CacheBuffer bufferToRemove = null;
        float minPriority = Float.MIN_VALUE;
        for (Entity entity : otherBufferReplaceCandidates) {
            Optional<ModelState<EntityModel>> inactiveState;
            Optional<String> modelId = entity.getModelId(detectorId);
            if (!modelId.isPresent() || !(inactiveState = this.getStateFromInactiveEntiiyCache(modelId.get())).isPresent()) continue;
            ModelState<EntityModel> state = inactiveState.get();
            float priority = state.getPriority();
            float scaledPriority = buffer.getPriorityTracker().getScaledPriority(priority);
            if (scaledPriority <= minPriority) {
                this.addEntity(coldEntities, entity, detectorId);
                continue;
            }
            if (minPriority == Float.MIN_VALUE) {
                Triple<CacheBuffer, String, Float> bufferToRemoveEntity = this.canReplaceInSharedCache(buffer, scaledPriority);
                bufferToRemove = (CacheBuffer)bufferToRemoveEntity.getLeft();
                minPriority = ((Float)bufferToRemoveEntity.getRight()).floatValue();
            }
            if (bufferToRemove != null) {
                this.addEntity(hotEntities, entity, detectorId);
                minPriority = Float.MIN_VALUE;
                continue;
            }
            this.addEntity(coldEntities, entity, detectorId);
        }
        return Pair.of(hotEntities, coldEntities);
    }

    private CacheBuffer computeBufferIfAbsent(AnomalyDetector detector, String detectorId) {
        CacheBuffer buffer = this.activeEnities.get(detectorId);
        if (buffer == null) {
            long requiredBytes = this.getRequiredMemory(detector, this.dedicatedCacheSize);
            if (this.memoryTracker.canAllocateReserved(requiredBytes)) {
                this.memoryTracker.consumeMemory(requiredBytes, true, MemoryTracker.Origin.HC_DETECTOR);
                long intervalSecs = detector.getDetectorIntervalInSeconds();
                buffer = new CacheBuffer(this.dedicatedCacheSize, intervalSecs, this.getRequiredMemory(detector, 1), this.memoryTracker, this.clock, this.modelTtl, detectorId, this.checkpointWriteQueue, this.checkpointMaintainQueue, this.checkpointIntervalHrs);
                this.activeEnities.put(detectorId, buffer);
                this.tryClearUpMemory();
            } else {
                throw new LimitExceededException(detectorId, "AD models memory usage exceeds our limit.");
            }
        }
        return buffer;
    }

    private long getRequiredMemory(AnomalyDetector detector, int numberOfEntity) {
        int dimension = detector.getEnabledFeatureIds().size() * detector.getShingleSize();
        return (long)numberOfEntity * this.memoryTracker.estimateTRCFModelSize(dimension, this.numberOfTrees, 0.0, detector.getShingleSize(), true);
    }

    private Triple<CacheBuffer, String, Float> canReplaceInSharedCache(CacheBuffer originBuffer, float candidatePriority) {
        CacheBuffer minPriorityBuffer = null;
        float minPriority = candidatePriority;
        String minPriorityEntityModelId = null;
        for (Map.Entry<String, CacheBuffer> entry : this.activeEnities.entrySet()) {
            float priority;
            Optional<Map.Entry<String, Float>> priorityEntry;
            CacheBuffer buffer = entry.getValue();
            if (buffer == originBuffer || !buffer.canRemove() || !(priorityEntry = buffer.getPriorityTracker().getMinimumScaledPriority()).isPresent() || !(candidatePriority > (priority = priorityEntry.get().getValue().floatValue())) || !(priority < minPriority)) continue;
            minPriority = priority;
            minPriorityBuffer = buffer;
            minPriorityEntityModelId = priorityEntry.get().getKey();
        }
        return Triple.of(minPriorityBuffer, minPriorityEntityModelId, (Object)Float.valueOf(minPriority));
    }

    private void tryClearUpMemory() {
        try {
            if (this.maintenanceLock.tryLock()) {
                this.threadPool.executor("ad-threadpool").execute(() -> this.clearMemory());
            } else {
                this.threadPool.schedule(() -> {
                    try {
                        this.tryClearUpMemory();
                    }
                    catch (Exception e) {
                        this.LOG.error("Fail to clear up memory taken by CacheBuffer.  Will retry during maintenance.");
                    }
                }, new TimeValue((long)this.random.nextInt(90), TimeUnit.SECONDS), "ad-threadpool");
            }
        }
        finally {
            if (this.maintenanceLock.isHeldByCurrentThread()) {
                this.maintenanceLock.unlock();
            }
        }
    }

    private void clearMemory() {
        this.recalculateUsedMemory();
        long memoryToShed = this.memoryTracker.memoryToShed();
        PriorityQueue<Triple> removalCandiates = null;
        if (memoryToShed > 0L) {
            removalCandiates = new PriorityQueue<Triple>((x, y) -> Float.compare(((Float)x.getLeft()).floatValue(), ((Float)y.getLeft()).floatValue()));
            for (Map.Entry<String, CacheBuffer> entry : this.activeEnities.entrySet()) {
                CacheBuffer buffer = entry.getValue();
                Optional<Map.Entry<String, Float>> priorityEntry = buffer.getPriorityTracker().getMinimumScaledPriority();
                if (!priorityEntry.isPresent()) continue;
                float priority = priorityEntry.get().getValue().floatValue();
                if (!buffer.canRemove()) continue;
                removalCandiates.add(Triple.of((Object)Float.valueOf(priority), (Object)buffer, (Object)priorityEntry.get().getKey()));
            }
        }
        while (memoryToShed > 0L) {
            if (!removalCandiates.isEmpty()) {
                Optional<Map.Entry<String, Float>> priorityEntry;
                Triple toRemove = (Triple)removalCandiates.poll();
                CacheBuffer minPriorityBuffer = (CacheBuffer)toRemove.getMiddle();
                String minPriorityEntityModelId = (String)toRemove.getRight();
                ModelState<EntityModel> removed = minPriorityBuffer.remove(minPriorityEntityModelId);
                memoryToShed -= minPriorityBuffer.getMemoryConsumptionPerEntity();
                this.addIntoInactiveCache(removed);
                if (minPriorityBuffer.canRemove() && (priorityEntry = minPriorityBuffer.getPriorityTracker().getMinimumScaledPriority()).isPresent()) {
                    removalCandiates.add(Triple.of((Object)priorityEntry.get().getValue(), (Object)minPriorityBuffer, (Object)priorityEntry.get().getKey()));
                }
            }
            if (!removalCandiates.isEmpty()) continue;
            break;
        }
    }

    private void recalculateUsedMemory() {
        long reserved = 0L;
        long shared = 0L;
        for (Map.Entry<String, CacheBuffer> entry : this.activeEnities.entrySet()) {
            CacheBuffer buffer = entry.getValue();
            reserved += buffer.getReservedBytes();
            shared += buffer.getBytesInSharedCache();
        }
        this.memoryTracker.syncMemoryState(MemoryTracker.Origin.HC_DETECTOR, reserved + shared, reserved);
    }

    @Override
    public void maintenance() {
        try {
            this.tryClearUpMemory();
            this.activeEnities.entrySet().stream().forEach(cacheBufferEntry -> {
                String detectorId = (String)cacheBufferEntry.getKey();
                CacheBuffer cacheBuffer = (CacheBuffer)cacheBufferEntry.getValue();
                if (cacheBuffer.expired(this.modelTtl)) {
                    this.activeEnities.remove(detectorId);
                    cacheBuffer.clear();
                } else {
                    List<ModelState<EntityModel>> removedStates = cacheBuffer.maintenance();
                    for (ModelState<EntityModel> state : removedStates) {
                        this.addIntoInactiveCache(state);
                    }
                }
            });
            this.maintainInactiveCache();
            this.doorKeepers.entrySet().stream().forEach(doorKeeperEntry -> {
                String detectorId = (String)doorKeeperEntry.getKey();
                DoorKeeper doorKeeper = (DoorKeeper)doorKeeperEntry.getValue();
                if (doorKeeper.expired(null)) {
                    this.doorKeepers.remove(detectorId);
                } else {
                    doorKeeper.maintenance();
                }
            });
        }
        catch (Exception e) {
            throw new AnomalyDetectionException("Fail to maintain cache", e);
        }
    }

    @Override
    public void clear(String detectorId) {
        if (Strings.isEmpty((CharSequence)detectorId)) {
            return;
        }
        CacheBuffer buffer = this.activeEnities.remove(detectorId);
        if (buffer != null) {
            buffer.clear();
        }
        this.checkpointDao.deleteModelCheckpointByDetectorId(detectorId);
        this.doorKeepers.remove(detectorId);
    }

    @Override
    public int getActiveEntities(String detectorId) {
        CacheBuffer cacheBuffer = this.activeEnities.get(detectorId);
        if (cacheBuffer != null) {
            return cacheBuffer.getActiveEntities();
        }
        return 0;
    }

    @Override
    public boolean isActive(String detectorId, String entityModelId) {
        CacheBuffer cacheBuffer = this.activeEnities.get(detectorId);
        if (cacheBuffer != null) {
            return cacheBuffer.isActive(entityModelId);
        }
        return false;
    }

    @Override
    public long getTotalUpdates(String detectorId) {
        return Optional.of(this.activeEnities).map(entities -> (CacheBuffer)entities.get(detectorId)).map(buffer -> buffer.getPriorityTracker().getHighestPriorityEntityId()).map(entityModelIdOptional -> (String)entityModelIdOptional.get()).map(entityModelId -> this.getTotalUpdates(detectorId, (String)entityModelId)).orElse(0L);
    }

    @Override
    public long getTotalUpdates(String detectorId, String entityModelId) {
        CacheBuffer cacheBuffer = this.activeEnities.get(detectorId);
        if (cacheBuffer != null) {
            Optional<EntityModel> modelOptional = cacheBuffer.getModel(entityModelId);
            long accumulatedShingles = modelOptional.flatMap(model -> model.getTrcf()).map(trcf -> trcf.getForest()).map(rcf -> rcf.getTotalUpdates()).orElseGet(() -> modelOptional.map(model -> model.getSamples()).map(samples -> samples.size()).map(Long::valueOf).orElse(0L));
            return accumulatedShingles;
        }
        return 0L;
    }

    @Override
    public int getTotalActiveEntities() {
        AtomicInteger total = new AtomicInteger();
        this.activeEnities.values().stream().forEach(cacheBuffer -> total.addAndGet(cacheBuffer.getActiveEntities()));
        return total.get();
    }

    @Override
    public List<ModelState<?>> getAllModels() {
        ArrayList states = new ArrayList();
        this.activeEnities.values().stream().forEach(cacheBuffer -> states.addAll(cacheBuffer.getAllModels()));
        return states;
    }

    @Override
    public Map<String, Long> getModelSize(String detectorId) {
        CacheBuffer cacheBuffer = this.activeEnities.get(detectorId);
        HashMap<String, Long> res = new HashMap<String, Long>();
        if (cacheBuffer != null) {
            long size = cacheBuffer.getMemoryConsumptionPerEntity();
            cacheBuffer.getAllModels().forEach(entry -> res.put(entry.getModelId(), size));
        }
        return res;
    }

    @Override
    public long getLastActiveMs(String detectorId, String entityModelId) {
        CacheBuffer cacheBuffer = this.activeEnities.get(detectorId);
        long lastUsedMs = -1L;
        if (cacheBuffer != null && (lastUsedMs = cacheBuffer.getLastUsedTime(entityModelId)) != -1L) {
            return lastUsedMs;
        }
        ModelState stateInActive = (ModelState)this.inActiveEntities.getIfPresent((Object)entityModelId);
        if (stateInActive != null) {
            lastUsedMs = stateInActive.getLastUsedTime().toEpochMilli();
        }
        return lastUsedMs;
    }

    @Override
    public void releaseMemoryForOpenCircuitBreaker() {
        this.maintainInactiveCache();
        this.tryClearUpMemory();
        this.activeEnities.values().stream().forEach(cacheBuffer -> {
            if (cacheBuffer.canRemove()) {
                ModelState<EntityModel> removed = cacheBuffer.remove();
                this.addIntoInactiveCache(removed);
            }
        });
    }

    private void maintainInactiveCache() {
        if (this.lastInActiveEntityMaintenance.plus(this.modelTtl).isAfter(this.clock.instant())) {
            return;
        }
        this.inActiveEntities.cleanUp();
        for (ModelState state : this.inActiveEntities.asMap().values()) {
            EntityModel model = (EntityModel)state.getModel();
            if (model == null || !model.getTrcf().isPresent()) continue;
            this.LOG.warn((Message)new ParameterizedMessage("Inactive entity's model is null: [{}]. Maybe there are bugs.", (Object)state.getModelId()));
            state.setModel(null);
        }
        this.lastInActiveEntityMaintenance = this.clock.instant();
    }

    private void setDedicatedCacheSizeListener() {
        this.activeEnities.values().stream().forEach(cacheBuffer -> cacheBuffer.setMinimumCapacity(this.dedicatedCacheSize));
    }

    private void setCheckpointFreqListener() {
        this.activeEnities.values().stream().forEach(cacheBuffer -> cacheBuffer.setCheckpointIntervalHrs(this.checkpointIntervalHrs));
    }

    @Override
    public List<ModelProfile> getAllModelProfile(String detectorId) {
        CacheBuffer cacheBuffer = this.activeEnities.get(detectorId);
        ArrayList<ModelProfile> res = new ArrayList<ModelProfile>();
        if (cacheBuffer != null) {
            long size = cacheBuffer.getMemoryConsumptionPerEntity();
            cacheBuffer.getAllModels().forEach(entry -> {
                EntityModel model = (EntityModel)entry.getModel();
                Entity entity = null;
                if (model != null && model.getEntity().isPresent()) {
                    entity = model.getEntity().get();
                }
                res.add(new ModelProfile(entry.getModelId(), entity, size));
            });
        }
        return res;
    }

    @Override
    public Optional<ModelProfile> getModelProfile(String detectorId, String entityModelId) {
        CacheBuffer cacheBuffer = this.activeEnities.get(detectorId);
        if (cacheBuffer != null && cacheBuffer.getModel(entityModelId).isPresent()) {
            EntityModel model = cacheBuffer.getModel(entityModelId).get();
            Entity entity = null;
            if (model != null && model.getEntity().isPresent()) {
                entity = model.getEntity().get();
            }
            return Optional.of(new ModelProfile(entityModelId, entity, cacheBuffer.getMemoryConsumptionPerEntity()));
        }
        return Optional.empty();
    }

    private void validateDedicatedCacheSize(Integer newDedicatedCacheSize) {
        if (this.dedicatedCacheSize < newDedicatedCacheSize) {
            int delta = newDedicatedCacheSize - this.dedicatedCacheSize;
            long totalIncreasedBytes = 0L;
            for (CacheBuffer cacheBuffer : this.activeEnities.values()) {
                totalIncreasedBytes += cacheBuffer.getMemoryConsumptionPerEntity() * (long)delta;
            }
            if (!this.memoryTracker.canAllocateReserved(totalIncreasedBytes)) {
                throw new IllegalArgumentException("We don't have enough memory for the required change");
            }
        }
    }

    @Override
    public Optional<ModelState<EntityModel>> getForMaintainance(String detectorId, String modelId) {
        CacheBuffer buffer = this.activeEnities.get(detectorId);
        if (buffer == null) {
            return Optional.empty();
        }
        return Optional.ofNullable(buffer.getWithoutUpdatePriority(modelId));
    }

    @Override
    public void removeEntityModel(String detectorId, String entityModelId) {
        CacheBuffer buffer = this.activeEnities.get(detectorId);
        if (buffer != null) {
            ModelState<EntityModel> removed = null;
            removed = buffer.remove(entityModelId, false);
            if (removed != null) {
                this.addIntoInactiveCache(removed);
            }
        }
        this.checkpointDao.deleteModelCheckpoint(entityModelId, (ActionListener<Void>)ActionListener.wrap(r -> this.LOG.debug((Message)new ParameterizedMessage("Succeeded in deleting checkpoint [{}].", (Object)entityModelId)), e -> this.LOG.error((Message)new ParameterizedMessage("Failed to delete checkpoint [{}].", (Object)entityModelId), (Throwable)e)));
    }

    private Cache<String, ModelState<EntityModel>> createInactiveCache(Duration inactiveEntityTtl, int maxInactiveStates) {
        return CacheBuilder.newBuilder().expireAfterAccess(inactiveEntityTtl.toHours(), TimeUnit.HOURS).maximumSize((long)maxInactiveStates).concurrencyLevel(1).build();
    }
}

