"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.removeBacktick = exports.preprocessQuery = exports.getIndexPatternFromRawQuery = exports.composeFinalQuery = exports.buildQuery = exports.buildPatternsQuery = void 0;

var _datemath = _interopRequireDefault(require("@elastic/datemath"));

var _lodash = require("lodash");

var _explorer = require("../../common/constants/explorer");

var _shared = require("../../common/constants/shared");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * @param literal - string literal that will be put inside single quotes in PPL command
 * @returns string with inner single quotes escaped
 */
const escapeQuotes = literal => {
  return literal.replaceAll("'", "''");
};

const getIndexPatternFromRawQuery = query => {
  const matches = query.match(_shared.PPL_INDEX_REGEX);

  if (matches) {
    return matches[2];
  }

  return '';
}; // insert time filter command and additional commands based on raw query


exports.getIndexPatternFromRawQuery = getIndexPatternFromRawQuery;

const preprocessQuery = ({
  rawQuery,
  startTime,
  endTime,
  timeField,
  isLiveQuery,
  selectedPatternField,
  patternRegex,
  filteredPattern
}) => {
  var _datemath$parse, _datemath$parse2;

  let finalQuery = '';
  if ((0, _lodash.isEmpty)(rawQuery)) return finalQuery; // convert to moment

  const start = (_datemath$parse = _datemath.default.parse(startTime)) === null || _datemath$parse === void 0 ? void 0 : _datemath$parse.utc().format(_explorer.DATE_PICKER_FORMAT);
  const end = (_datemath$parse2 = _datemath.default.parse(endTime, {
    roundUp: true
  })) === null || _datemath$parse2 === void 0 ? void 0 : _datemath$parse2.utc().format(_explorer.DATE_PICKER_FORMAT);
  const tokens = rawQuery.replaceAll(_shared.PPL_NEWLINE_REGEX, '').match(_shared.PPL_INDEX_INSERT_POINT_REGEX);
  if ((0, _lodash.isEmpty)(tokens)) return finalQuery;
  finalQuery = `${tokens[1]}=${tokens[2]} | where ${timeField} >= '${start}' and ${timeField} <= '${end}'${tokens[3]}`;

  if (isLiveQuery) {
    finalQuery = finalQuery + ` | sort - ${timeField}`;
  } // if a pattern is selected as filter, build it into finalQuery


  if (selectedPatternField && filteredPattern) finalQuery = buildPatternsQuery(finalQuery, selectedPatternField, patternRegex, filteredPattern);
  return finalQuery;
};

exports.preprocessQuery = preprocessQuery;

const buildPatternsQuery = (baseQuery, selectedPatternField, patternRegex, filteredPattern) => {
  let finalQuery = baseQuery;

  if (selectedPatternField) {
    finalQuery += ` | patterns `;

    if (patternRegex && patternRegex !== _explorer.PPL_DEFAULT_PATTERN_REGEX_FILETER) {
      finalQuery += `pattern='${escapeQuotes(patternRegex)}' `;
    }

    finalQuery += `\`${selectedPatternField}\` `;

    if (filteredPattern) {
      finalQuery += `| where patterns_field='${escapeQuotes(filteredPattern)}'`;
    }
  }

  return finalQuery;
};

exports.buildPatternsQuery = buildPatternsQuery;

const buildQuery = (baseQuery, currQuery) => {
  let fullQuery;

  if (baseQuery) {
    fullQuery = baseQuery;

    if (currQuery) {
      fullQuery += '| ' + currQuery;
    }
  } else {
    fullQuery = currQuery;
  }

  return fullQuery;
};

exports.buildQuery = buildQuery;

const composeFinalQuery = (curQuery, startingTime, endingTime, timeField, isLiveQuery, appBaseQuery, selectedPatternField, patternRegex, filteredPattern) => {
  const fullQuery = buildQuery(appBaseQuery, curQuery);
  if ((0, _lodash.isEmpty)(fullQuery)) return '';
  return preprocessQuery({
    rawQuery: fullQuery,
    startTime: startingTime,
    endTime: endingTime,
    timeField,
    isLiveQuery,
    selectedPatternField,
    patternRegex,
    filteredPattern
  });
};

exports.composeFinalQuery = composeFinalQuery;

const removeBacktick = stringContainsBacktick => {
  if (!stringContainsBacktick) return '';
  return stringContainsBacktick.replace(/`/g, '');
};

exports.removeBacktick = removeBacktick;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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