// ----------------------------------------------------------------------------
//
//  Copyright (C) 2005-2021 Fons Adriaensen <fons@linuxaudio.org>
//    
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// ----------------------------------------------------------------------------


#include "nffilt.h"


NFfiltbase::NFfiltbase (int degree, int nchan):
    _degree (degree),
    _nchan (nchan)
{
    _d = new float [_degree];
    _z = new float [_degree * _nchan];
}

NFfiltbase::~NFfiltbase (void)
{
    delete[] _d;
    delete[] _z;
}


// Add a first order section.
void NFfiltbase::init1 (int i, float a)
{
    float t;

    t = 1 / (1 + a);
    _d [i] = t * (2 * a);
    if (i) _g *= t;
    else   _g = t;
}

// Add a second order section.
void NFfiltbase::init2 (int i, float a, float b)
{
    float t;

    t = 1 / (1 + a + b);
    _d [i]   = t * (2 * a + 4 * b);
    _d [i+1] = t * (4 * b);
    if (i) _g *= t;
    else   _g = t;
} 


// ----------------------------------------------------------------------------


// Denormal protection.
#define EPS 1e-30f

// First order section.
#define PROC1(d0, z0) \
    x -= d0 * z0 + EPS; \
    z0 += x;

// Second order section.
#define PROC2(d0, d1, z0, z1) \
    x -= d0 * z0 + d1 * z1 + EPS; \
    z1 += z0; \
    z0 += x;

// ----------------------------------------------------------------------------


NFfilt1::NFfilt1 (int nchan):
    NFfiltbase (1, nchan)
{
}

void NFfilt1::init (float w)
{
    float r1;
    
    r1 = 0.5f * w;
    init1 (0, r1);         
    reset ();
}

void NFfilt1::process (int nsam, float *inp [], float *out [], float gain)
{
    int   i, n;
    float g, x;
    float *p, *q, *z;

    g = _g * gain;
    z = _z;
    for (i = 0; i < _nchan; i++)
    {
        p = inp [i];
        q = out [i];
        n = nsam;
        while (n--)
        {
            x = g * *p++;
            PROC1(_d [0], z [0]);
            *q++ = x;
        }
        z += 1;
    }
}


// ----------------------------------------------------------------------------


NFfilt2::NFfilt2 (int nchan):
    NFfiltbase (2, nchan)
{
}

void NFfilt2::init (float w)
{
    float r1, r2;
    
    r1 = 0.5f * w;
    r2 = r1 * r1;
    init2 (0, r1 * 3.0f, r2 * 3.0f);         
    reset ();
}

void NFfilt2::process (int nsam, float *inp [], float *out [], float gain)
{
    int   i, n;
    float g, x;
    float *p, *q, *z;

    g = _g * gain;
    z = _z;
    for (i = 0; i < _nchan; i++)
    {
        p = inp [i];
        q = out [i];
        n = nsam;
        while (n--)
        {
            x = g * *p++;
            PROC2(_d [0], _d [1], z [0], z [1]);
            *q++ = x;
        }
        z += 2;
    }
}


// ----------------------------------------------------------------------------


NFfilt3::NFfilt3 (int nchan):
    NFfiltbase (3, nchan)
{
}

void NFfilt3::init (float w)
{
    float r1, r2;
    
    r1 = 0.5f * w;
    r2 = r1 * r1;
    init2 (0, r1 * 3.6778f, r2 * 6.4594f);         
    init1 (2, r1 * 2.3222f);         
    reset ();
}

void NFfilt3::process (int nsam, float *inp [], float *out [], float gain)
{
    int   i, n;
    float g, x;
    float *p, *q, *z;
    
    g = _g * gain;
    z = _z;
    for (i = 0; i < _nchan; i++)
    {
        p = inp [i];
        q = out [i];
        n = nsam;
        while (n--)
        {
            x = g * *p++;
            PROC2(_d [0], _d [1], z [0], z [1]);
            PROC1(_d [2], z [2]);       
            *q++ = x;
        }
        z += 3;
    }
}


// ----------------------------------------------------------------------------


NFfilt4::NFfilt4 (int nchan):
    NFfiltbase (4, nchan)
{
}

void NFfilt4::init (float w)
{
    float r1, r2;
    
    r1 = 0.5f * w;
    r2 = r1 * r1;
    init2 (0, r1 * 4.2076f, r2 * 11.4878f);         
    init2 (2, r1 * 5.7924f, r2 *  9.1401f);         
    reset ();
}

void NFfilt4::process (int nsam, float *inp [], float *out [], float gain)
{
    int   i, n;
    float g, x;
    float *p, *q, *z;

    g = _g * gain;
    z = _z;
    for (i = 0; i < _nchan; i++)
    {
        p = inp [i];
        q = out [i];
        n = nsam;
        while (n--)
        {
            x = g * *p++;
            PROC2(_d [0], _d [1], z [0], z [1]);
            PROC2(_d [2], _d [3], z [2], z [3]);
            *q++ = x;
        }
        z += 4;
    }
}


// ----------------------------------------------------------------------------


NFfilt5::NFfilt5 (int nchan):
    NFfiltbase (5, nchan)
{
}

void NFfilt5::init (float w)
{
    float r1, r2;
    
    r1 = 0.5f * w;
    r2 = r1 * r1;
    init2 (0, r1 * 4.6493f, r2 * 18.1563f);         
    init2 (2, r1 * 6.7093f, r2 * 14.2725f);         
    init1 (4, r1 * 3.6467f);         
    reset ();
}

void NFfilt5::process (int nsam, float *inp [], float *out [], float gain)
{
    int   i, n;
    float g, x;
    float *p, *q, *z;
    
    g = _g * gain;
    z = _z;
    for (i = 0; i < _nchan; i++)
    {
        p = inp [i];
        q = out [i];
        n = nsam;
        while (n--)
        {
            x = g * *p++;
            PROC2(_d [0], _d [1], z [0], z [1]);
            PROC2(_d [2], _d [3], z [2], z [3]);
            PROC1(_d [4], z [4]);       
            *q++ = x;
        }
        z += 5;
    }
}


// ----------------------------------------------------------------------------


NFfilt6::NFfilt6 (int nchan):
    NFfiltbase (6, nchan)
{
}

void NFfilt6::init (float w)
{
    float r1, r2;
    
    r1 = 0.5f * w;
    r2 = r1 * r1;
    init2 (0, r1 * 5.0319f, r2 * 26.5140f);         
    init2 (2, r1 * 7.4614f, r2 * 20.8528f);         
    init2 (4, r1 * 8.4967f, r2 * 18.8021f);         
    reset ();
}

void NFfilt6::process (int nsam, float *inp [], float *out [], float gain)
{
    int   i, n;
    float g, x;
    float *p, *q, *z;

    g = _g * gain;
    z = _z;
    for (i = 0; i < _nchan; i++)
    {
        p = inp [i];
        q = out [i];
        n = nsam;
        while (n--)
        {
            x = g * *p++;
            PROC2(_d [0], _d [1], z [0], z [1]);
            PROC2(_d [2], _d [3], z [2], z [3]);
            PROC2(_d [4], _d [5], z [4], z [5]);
            *q++ = x;
        }
        z += 6;
    }
}


// ----------------------------------------------------------------------------


NFfilt7::NFfilt7 (int nchan):
    NFfiltbase (7, nchan)
{
}

void NFfilt7::init (float w)
{
    float r1, r2;
    
    r1 = 0.5f * w;
    r2 = r1 * r1;
    init2 (0, r1 * 5.3714f, r2 * 36.5968f);         
    init2 (2, r1 * 8.1403f, r2 * 28.9365f);         
    init2 (4, r1 * 9.5166f, r2 * 25.6664f);         
    init1 (6, r1 * 4.9718f);         
    reset ();
}

void NFfilt7::process (int nsam, float *inp [], float *out [], float gain)
{
    int   i, n;
    float g, x;
    float *p, *q, *z;
    
    g = _g * gain;
    z = _z;
    for (i = 0; i < _nchan; i++)
    {
        p = inp [i];
        q = out [i];
        n = nsam;
        while (n--)
        {
            x = g * *p++;
            PROC2(_d [0], _d [1], z [0], z [1]);
            PROC2(_d [2], _d [3], z [2], z [3]);
            PROC2(_d [4], _d [5], z [4], z [5]);
            PROC1(_d [6], z [6]);       
            *q++ = x;
        }
        z += 7;
    }
}


// ----------------------------------------------------------------------------



NFfilt8::NFfilt8 (int nchan):
    NFfiltbase (8, nchan)
{
}

void NFfilt8::init (float w)
{
    float r1, r2;
    
    r1 = 0.5f * w;
    r2 = r1 * r1;
    init2 (0, r1 *  5.6780f, r2 * 48.4320f);         
    init2 (2, r1 *  8.7366f, r2 * 38.5693f);         
    init2 (4, r1 * 10.4097f, r2 * 33.9347f);         
    init2 (6, r1 * 11.1758f, r2 * 31.9772f);         
    reset ();
}

void NFfilt8::process (int nsam, float *inp [], float *out [], float gain)
{
    int   i, n;
    float g, x;
    float *p, *q, *z;

    g = _g * gain;
    z = _z;
    for (i = 0; i < _nchan; i++)
    {
        p = inp [i];
        q = out [i];
        n = nsam;
        while (n--)
        {
            x = g * *p++;
            PROC2(_d [0], _d [1], z [0], z [1]);
            PROC2(_d [2], _d [3], z [2], z [3]);
            PROC2(_d [4], _d [5], z [4], z [5]);
            PROC2(_d [6], _d [7], z [6], z [7]);
            *q++ = x;
        }
        z += 8;
    }
}


// ----------------------------------------------------------------------------
