/*
 * Copyright (C) 2022 TypeFox and others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 */
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
import path from 'node:path';
import * as lsp from 'vscode-languageserver';
import { pathToUri } from '../protocol-translation.js';
import { tslib } from '../ts-protocol.js';
import { Range } from '../utils/typeConverters.js';
export function fromProtocolCallHierarchyItem(item, documents, workspaceRoot) {
    const useFileName = isSourceFileItem(item);
    const name = useFileName ? path.basename(item.file) : item.name;
    const detail = useFileName
        ? workspaceRoot ? path.relative(workspaceRoot, path.dirname(item.file)) : path.dirname(item.file)
        : item.containerName ?? '';
    const result = {
        kind: fromProtocolScriptElementKind(item.kind),
        name,
        detail,
        uri: pathToUri(item.file, documents),
        range: Range.fromTextSpan(item.span),
        selectionRange: Range.fromTextSpan(item.selectionSpan),
    };
    const kindModifiers = item.kindModifiers ? parseKindModifier(item.kindModifiers) : undefined;
    if (kindModifiers?.has(tslib.ScriptElementKindModifier.deprecatedModifier)) {
        result.tags = [lsp.SymbolTag.Deprecated];
    }
    return result;
}
export function fromProtocolCallHierarchyIncomingCall(item, documents, workspaceRoot) {
    return {
        from: fromProtocolCallHierarchyItem(item.from, documents, workspaceRoot),
        fromRanges: item.fromSpans.map(Range.fromTextSpan),
    };
}
export function fromProtocolCallHierarchyOutgoingCall(item, documents, workspaceRoot) {
    return {
        to: fromProtocolCallHierarchyItem(item.to, documents, workspaceRoot),
        fromRanges: item.fromSpans.map(Range.fromTextSpan),
    };
}
function isSourceFileItem(item) {
    return item.kind === tslib.ScriptElementKind.scriptElement || item.kind === tslib.ScriptElementKind.moduleElement && item.selectionSpan.start.line === 1 && item.selectionSpan.start.offset === 1;
}
function fromProtocolScriptElementKind(kind) {
    switch (kind) {
        case tslib.ScriptElementKind.moduleElement: return lsp.SymbolKind.Module;
        case tslib.ScriptElementKind.classElement: return lsp.SymbolKind.Class;
        case tslib.ScriptElementKind.enumElement: return lsp.SymbolKind.Enum;
        case tslib.ScriptElementKind.enumMemberElement: return lsp.SymbolKind.EnumMember;
        case tslib.ScriptElementKind.interfaceElement: return lsp.SymbolKind.Interface;
        case tslib.ScriptElementKind.indexSignatureElement: return lsp.SymbolKind.Method;
        case tslib.ScriptElementKind.callSignatureElement: return lsp.SymbolKind.Method;
        case tslib.ScriptElementKind.memberFunctionElement: return lsp.SymbolKind.Method;
        case tslib.ScriptElementKind.memberVariableElement: return lsp.SymbolKind.Property;
        case tslib.ScriptElementKind.memberGetAccessorElement: return lsp.SymbolKind.Property;
        case tslib.ScriptElementKind.memberSetAccessorElement: return lsp.SymbolKind.Property;
        case tslib.ScriptElementKind.variableElement: return lsp.SymbolKind.Variable;
        case tslib.ScriptElementKind.letElement: return lsp.SymbolKind.Variable;
        case tslib.ScriptElementKind.constElement: return lsp.SymbolKind.Variable;
        case tslib.ScriptElementKind.localVariableElement: return lsp.SymbolKind.Variable;
        case tslib.ScriptElementKind.alias: return lsp.SymbolKind.Variable;
        case tslib.ScriptElementKind.functionElement: return lsp.SymbolKind.Function;
        case tslib.ScriptElementKind.localFunctionElement: return lsp.SymbolKind.Function;
        case tslib.ScriptElementKind.constructSignatureElement: return lsp.SymbolKind.Constructor;
        case tslib.ScriptElementKind.constructorImplementationElement: return lsp.SymbolKind.Constructor;
        case tslib.ScriptElementKind.typeParameterElement: return lsp.SymbolKind.TypeParameter;
        case tslib.ScriptElementKind.string: return lsp.SymbolKind.String;
        default: return lsp.SymbolKind.Variable;
    }
}
function parseKindModifier(kindModifiers) {
    return new Set(kindModifiers.split(/,|\s+/g));
}
//# sourceMappingURL=call-hierarchy.js.map