"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenIdAuthRoutes = void 0;

var _configSchema = require("@osd/config-schema");

var _cryptiles = require("@hapi/cryptiles");

var _querystring = require("querystring");

var _helper = require("./helper");

var _next_url = require("../../../utils/next_url");

var _common = require("../../../../common");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class OpenIdAuthRoutes {
  constructor(router, config, sessionStorageFactory, openIdAuthConfig, securityClient, core, wreckClient) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.openIdAuthConfig = openIdAuthConfig;
    this.securityClient = securityClient;
    this.core = core;
    this.wreckClient = wreckClient;
  }

  redirectToLogin(request, response) {
    this.sessionStorageFactory.asScoped(request).clear();
    return response.redirected({
      headers: {
        location: `${this.core.http.basePath.serverBasePath}${_common.OPENID_AUTH_LOGIN}`
      }
    });
  }

  setupRoutes() {
    this.router.get({
      path: _common.OPENID_AUTH_LOGIN,
      validate: {
        query: _configSchema.schema.object({
          code: _configSchema.schema.maybe(_configSchema.schema.string()),
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: _next_url.validateNextUrl
          })),
          state: _configSchema.schema.maybe(_configSchema.schema.string()),
          refresh: _configSchema.schema.maybe(_configSchema.schema.string())
        }, {
          unknowns: 'allow'
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      var _this$config$openid2, _this$config$openid3;

      // implementation refers to https://github.com/hapijs/bell/blob/master/lib/oauth.js
      // Sign-in initialization
      if (!request.query.code) {
        var _this$config$openid;

        const nonce = (0, _cryptiles.randomString)(OpenIdAuthRoutes.NONCE_LENGTH);
        const query = {
          client_id: (_this$config$openid = this.config.openid) === null || _this$config$openid === void 0 ? void 0 : _this$config$openid.client_id,
          response_type: _common.AUTH_RESPONSE_TYPE,
          redirect_uri: `${(0, _helper.getBaseRedirectUrl)(this.config, this.core, request)}${_common.OPENID_AUTH_LOGIN}`,
          state: nonce,
          scope: this.openIdAuthConfig.scope
        };
        const queryString = (0, _querystring.stringify)(query);
        const location = `${this.openIdAuthConfig.authorizationEndpoint}?${queryString}`;
        const cookie = {
          oidc: {
            state: nonce,
            nextUrl: request.query.nextUrl || '/'
          },
          authType: _common.AuthType.OPEN_ID
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location
          }
        });
      } // Authentication callback
      // validate state first


      let cookie;

      try {
        var _cookie$oidc;

        cookie = await this.sessionStorageFactory.asScoped(request).get();

        if (!cookie || !((_cookie$oidc = cookie.oidc) !== null && _cookie$oidc !== void 0 && _cookie$oidc.state) || cookie.oidc.state !== request.query.state) {
          return this.redirectToLogin(request, response);
        }
      } catch (error) {
        return this.redirectToLogin(request, response);
      }

      const nextUrl = cookie.oidc.nextUrl;
      const clientId = (_this$config$openid2 = this.config.openid) === null || _this$config$openid2 === void 0 ? void 0 : _this$config$openid2.client_id;
      const clientSecret = (_this$config$openid3 = this.config.openid) === null || _this$config$openid3 === void 0 ? void 0 : _this$config$openid3.client_secret;
      const query = {
        grant_type: _common.AUTH_GRANT_TYPE,
        code: request.query.code,
        redirect_uri: `${(0, _helper.getBaseRedirectUrl)(this.config, this.core, request)}${_common.OPENID_AUTH_LOGIN}`,
        client_id: clientId,
        client_secret: clientSecret
      };

      try {
        var _this$config$openid4;

        const tokenResponse = await (0, _helper.callTokenEndpoint)(this.openIdAuthConfig.tokenEndpoint, query, this.wreckClient);
        const user = await this.securityClient.authenticateWithHeader(request, this.openIdAuthConfig.authHeaderName, `Bearer ${tokenResponse.idToken}`); // set to cookie

        const sessionStorage = {
          username: user.username,
          credentials: {
            authHeaderValue: `Bearer ${tokenResponse.idToken}`,
            expires_at: (0, _helper.getExpirationDate)(tokenResponse)
          },
          authType: _common.AuthType.OPEN_ID,
          expiryTime: Date.now() + this.config.session.ttl
        };

        if ((_this$config$openid4 = this.config.openid) !== null && _this$config$openid4 !== void 0 && _this$config$openid4.refresh_tokens && tokenResponse.refreshToken) {
          Object.assign(sessionStorage.credentials, {
            refresh_token: tokenResponse.refreshToken
          });
        }

        this.sessionStorageFactory.asScoped(request).set(sessionStorage);
        return response.redirected({
          headers: {
            location: nextUrl
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`OpenId authentication failed: ${error}`);

        if (error.toString().toLowerCase().includes('authentication exception')) {
          return response.unauthorized();
        } else {
          return this.redirectToLogin(request, response);
        }
      }
    });
    this.router.get({
      path: _common.OPENID_AUTH_LOGOUT,
      validate: false
    }, async (context, request, response) => {
      var _this$config$openid5;

      const cookie = await this.sessionStorageFactory.asScoped(request).get();
      this.sessionStorageFactory.asScoped(request).clear(); // authHeaderValue is the bearer header, e.g. "Bearer <auth_token>"

      const token = cookie === null || cookie === void 0 ? void 0 : cookie.credentials.authHeaderValue.split(' ')[1]; // get auth token

      const nextUrl = (0, _helper.getBaseRedirectUrl)(this.config, this.core, request);
      const logoutQueryParams = {
        post_logout_redirect_uri: `${nextUrl}`,
        id_token_hint: token
      };
      const endSessionUrl = (0, _helper.composeLogoutUrl)((_this$config$openid5 = this.config.openid) === null || _this$config$openid5 === void 0 ? void 0 : _this$config$openid5.logout_url, this.openIdAuthConfig.endSessionEndpoint, logoutQueryParams);
      return response.redirected({
        headers: {
          location: endSessionUrl
        }
      });
    });
  }

}

exports.OpenIdAuthRoutes = OpenIdAuthRoutes;

_defineProperty(OpenIdAuthRoutes, "NONCE_LENGTH", 22);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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