"use strict";
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* tslint:disable */
const chars = require("../ast/chars");
const interpolation_config_1 = require("../ast/interpolation_config");
const parse_util_1 = require("../ast/parse_util");
const ast_1 = require("./ast");
const lexer_1 = require("./lexer");
class SplitInterpolation {
    constructor(strings, expressions, offsets) {
        this.strings = strings;
        this.expressions = expressions;
        this.offsets = offsets;
    }
}
exports.SplitInterpolation = SplitInterpolation;
class TemplateBindingParseResult {
    constructor(templateBindings, warnings, errors) {
        this.templateBindings = templateBindings;
        this.warnings = warnings;
        this.errors = errors;
    }
}
exports.TemplateBindingParseResult = TemplateBindingParseResult;
function _createInterpolateRegExp(config) {
    const pattern = parse_util_1.escapeRegExp(config.start) + "([\\s\\S]*?)" + parse_util_1.escapeRegExp(config.end);
    return new RegExp(pattern, "g");
}
class Parser {
    constructor(_lexer) {
        this._lexer = _lexer;
        this.errors = [];
    }
    parseAction(input, location, interpolationConfig = interpolation_config_1.DEFAULT_INTERPOLATION_CONFIG) {
        this._checkNoInterpolation(input, location, interpolationConfig);
        const sourceToLex = this._stripComments(input);
        const tokens = this._lexer.tokenize(this._stripComments(input));
        const ast = new ParseAST(input, location, tokens, sourceToLex.length, true, this.errors, input.length - sourceToLex.length).parseChain();
        return new ast_1.ASTWithSource(ast, input, location, this.errors);
    }
    parseBinding(input, location, interpolationConfig = interpolation_config_1.DEFAULT_INTERPOLATION_CONFIG) {
        const ast = this._parseBindingAst(input, location, interpolationConfig);
        return new ast_1.ASTWithSource(ast, input, location, this.errors);
    }
    parseSimpleBinding(input, location, interpolationConfig = interpolation_config_1.DEFAULT_INTERPOLATION_CONFIG) {
        const ast = this._parseBindingAst(input, location, interpolationConfig);
        const errors = SimpleExpressionChecker.check(ast);
        if (errors.length > 0) {
            this._reportError(`Host binding expression cannot contain ${errors.join(" ")}`, input, location);
        }
        return new ast_1.ASTWithSource(ast, input, location, this.errors);
    }
    _reportError(message, input, errLocation, ctxLocation) {
        this.errors.push(new ast_1.ParserError(message, input, errLocation, ctxLocation));
    }
    _parseBindingAst(input, location, interpolationConfig) {
        // Quotes expressions use 3rd-party expression language. We don't want to use
        // our lexer or parser for that, so we check for that ahead of time.
        const quote = this._parseQuote(input, location);
        if (quote != null) {
            return quote;
        }
        this._checkNoInterpolation(input, location, interpolationConfig);
        const sourceToLex = this._stripComments(input);
        const tokens = this._lexer.tokenize(sourceToLex);
        return new ParseAST(input, location, tokens, sourceToLex.length, false, this.errors, input.length - sourceToLex.length).parseChain();
    }
    _parseQuote(input, location) {
        if (input === null) {
            return null;
        }
        const prefixSeparatorIndex = input.indexOf(":");
        if (prefixSeparatorIndex === -1) {
            return null;
        }
        const prefix = input.substring(0, prefixSeparatorIndex).trim();
        if (!lexer_1.isIdentifier(prefix)) {
            return null;
        }
        const uninterpretedExpression = input.substring(prefixSeparatorIndex + 1);
        return new ast_1.Quote(new ast_1.ParseSpan(0, input.length), prefix, uninterpretedExpression, location);
    }
    parseTemplateBindings(prefixToken, input, location) {
        const tokens = this._lexer.tokenize(input);
        if (prefixToken) {
            // Prefix the tokens with the tokens from prefixToken but have them take no space (0 index).
            const prefixTokens = this._lexer.tokenize(prefixToken).map(t => {
                t.index = 0;
                return t;
            });
            tokens.unshift(...prefixTokens);
        }
        return new ParseAST(input, location, tokens, input.length, false, this.errors, 0).parseTemplateBindings();
    }
    parseInterpolation(input, location, interpolationConfig = interpolation_config_1.DEFAULT_INTERPOLATION_CONFIG) {
        const split = this.splitInterpolation(input, location, interpolationConfig);
        if (split === null) {
            return null;
        }
        const expressions = [];
        for (let i = 0; i < split.expressions.length; ++i) {
            const expressionText = split.expressions[i];
            const sourceToLex = this._stripComments(expressionText);
            const tokens = this._lexer.tokenize(sourceToLex);
            const ast = new ParseAST(input, location, tokens, sourceToLex.length, false, this.errors, split.offsets[i] + (expressionText.length - sourceToLex.length)).parseChain();
            expressions.push(ast);
        }
        return new ast_1.ASTWithSource(new ast_1.Interpolation(new ast_1.ParseSpan(0, input === null ? 0 : input.length), split.strings, expressions), input, location, this.errors);
    }
    splitInterpolation(input, location, interpolationConfig = interpolation_config_1.DEFAULT_INTERPOLATION_CONFIG) {
        const regexp = _createInterpolateRegExp(interpolationConfig);
        const parts = input.split(regexp);
        if (parts.length <= 1) {
            return null;
        }
        const strings = [];
        const expressions = [];
        const offsets = [];
        let offset = 0;
        for (let i = 0; i < parts.length; i++) {
            const part = parts[i];
            if (i % 2 === 0) {
                // fixed string
                strings.push(part);
                offset += part.length;
            }
            else if (part.trim().length > 0) {
                offset += interpolationConfig.start.length;
                expressions.push(part);
                offsets.push(offset);
                offset += part.length + interpolationConfig.end.length;
            }
            else {
                this._reportError("Blank expressions are not allowed in interpolated strings", input, `at column ${this._findInterpolationErrorColumn(parts, i, interpolationConfig)} in`, location);
                expressions.push("$implict");
                offsets.push(offset);
            }
        }
        return new SplitInterpolation(strings, expressions, offsets);
    }
    wrapLiteralPrimitive(input, location) {
        return new ast_1.ASTWithSource(new ast_1.LiteralPrimitive(new ast_1.ParseSpan(0, input === null ? 0 : input.length), input), input, location, this.errors);
    }
    _stripComments(input) {
        const i = this._commentStart(input);
        return i != null ? input.substring(0, i).trim() : input;
    }
    _commentStart(input) {
        let outerQuote = null;
        for (let i = 0; i < input.length - 1; i++) {
            const char = input.charCodeAt(i);
            const nextChar = input.charCodeAt(i + 1);
            if (char === chars.$SLASH && nextChar === chars.$SLASH && outerQuote === null) {
                return i;
            }
            if (outerQuote === char) {
                outerQuote = null;
            }
            else if (outerQuote === null && lexer_1.isQuote(char)) {
                outerQuote = char;
            }
        }
        return null;
    }
    _checkNoInterpolation(input, location, interpolationConfig) {
        const regexp = _createInterpolateRegExp(interpolationConfig);
        const parts = input.split(regexp);
        if (parts.length > 1) {
            this._reportError(`Got interpolation (${interpolationConfig.start}${interpolationConfig.end}) where expression was expected`, input, `at column ${this._findInterpolationErrorColumn(parts, 1, interpolationConfig)} in`, location);
        }
    }
    _findInterpolationErrorColumn(parts, partInErrIdx, interpolationConfig) {
        let errLocation = "";
        for (let j = 0; j < partInErrIdx; j++) {
            errLocation += j % 2 === 0 ? parts[j] : `${interpolationConfig.start}${parts[j]}${interpolationConfig.end}`;
        }
        return errLocation.length;
    }
}
exports.Parser = Parser;
class ParseAST {
    constructor(input, location, tokens, inputLength, parseAction, errors, offset) {
        this.input = input;
        this.location = location;
        this.tokens = tokens;
        this.inputLength = inputLength;
        this.parseAction = parseAction;
        this.errors = errors;
        this.offset = offset;
        this.rparensExpected = 0;
        this.rbracketsExpected = 0;
        this.rbracesExpected = 0;
        this.index = 0;
    }
    peek(offset) {
        const i = this.index + offset;
        return i < this.tokens.length ? this.tokens[i] : lexer_1.EOF;
    }
    get next() {
        return this.peek(0);
    }
    get inputIndex() {
        return this.index < this.tokens.length ? this.next.index + this.offset : this.inputLength + this.offset;
    }
    span(start) {
        return new ast_1.ParseSpan(start, this.inputIndex);
    }
    advance() {
        this.index++;
    }
    optionalCharacter(code) {
        if (this.next.isCharacter(code)) {
            this.advance();
            return true;
        }
        else {
            return false;
        }
    }
    peekKeywordLet() {
        return this.next.isKeywordLet();
    }
    peekKeywordAs() {
        return this.next.isKeywordAs();
    }
    expectCharacter(code) {
        if (this.optionalCharacter(code)) {
            return;
        }
        this.error(`Missing expected ${String.fromCharCode(code)}`);
    }
    optionalOperator(op) {
        if (this.next.isOperator(op)) {
            this.advance();
            return true;
        }
        else {
            return false;
        }
    }
    expectOperator(operator) {
        if (this.optionalOperator(operator)) {
            return;
        }
        this.error(`Missing expected operator ${operator}`);
    }
    expectIdentifierOrKeyword() {
        const n = this.next;
        if (!n.isIdentifier() && !n.isKeyword()) {
            this.error(`Unexpected token ${n}, expected identifier or keyword`);
            return "";
        }
        this.advance();
        return n.toString();
    }
    expectIdentifierOrKeywordOrString() {
        const n = this.next;
        if (!n.isIdentifier() && !n.isKeyword() && !n.isString()) {
            this.error(`Unexpected token ${n}, expected identifier, keyword, or string`);
            return "";
        }
        this.advance();
        return n.toString();
    }
    parseChain() {
        const exprs = [];
        const start = this.inputIndex;
        while (this.index < this.tokens.length) {
            const expr = this.parsePipe();
            exprs.push(expr);
            if (this.optionalCharacter(chars.$SEMICOLON)) {
                if (!this.parseAction) {
                    this.error("Binding expression cannot contain chained expression");
                }
                while (this.optionalCharacter(chars.$SEMICOLON)) { } // read all semicolons
            }
            else if (this.index < this.tokens.length) {
                this.error(`Unexpected token '${this.next}'`);
            }
        }
        if (exprs.length === 0) {
            return new ast_1.EmptyExpr(this.span(start));
        }
        if (exprs.length === 1) {
            return exprs[0];
        }
        return new ast_1.Chain(this.span(start), exprs);
    }
    parsePipe() {
        let result = this.parseExpression();
        if (this.optionalOperator("|")) {
            if (this.parseAction) {
                this.error("Cannot have a pipe in an action expression");
            }
            do {
                const name = this.expectIdentifierOrKeyword();
                const args = [];
                while (this.optionalCharacter(chars.$COLON)) {
                    args.push(this.parseExpression());
                }
                result = new ast_1.BindingPipe(this.span(result.span.start), result, name, args);
            } while (this.optionalOperator("|"));
        }
        return result;
    }
    parseExpression() {
        return this.parseConditional();
    }
    parseConditional() {
        const start = this.inputIndex;
        const result = this.parseLogicalOr();
        if (this.optionalOperator("?")) {
            const yes = this.parsePipe();
            let no;
            if (!this.optionalCharacter(chars.$COLON)) {
                const end = this.inputIndex;
                const expression = this.input.substring(start, end);
                this.error(`Conditional expression ${expression} requires all 3 expressions`);
                no = new ast_1.EmptyExpr(this.span(start));
            }
            else {
                no = this.parsePipe();
            }
            return new ast_1.Conditional(this.span(start), result, yes, no);
        }
        else {
            return result;
        }
    }
    parseLogicalOr() {
        // '||'
        let result = this.parseLogicalAnd();
        while (this.optionalOperator("||")) {
            const right = this.parseLogicalAnd();
            result = new ast_1.Binary(this.span(result.span.start), "||", result, right);
        }
        return result;
    }
    parseLogicalAnd() {
        // '&&'
        let result = this.parseEquality();
        while (this.optionalOperator("&&")) {
            const right = this.parseEquality();
            result = new ast_1.Binary(this.span(result.span.start), "&&", result, right);
        }
        return result;
    }
    parseEquality() {
        // '==','!=','===','!=='
        let result = this.parseRelational();
        while (this.next.type === lexer_1.TokenType.Operator) {
            const operator = this.next.strValue;
            switch (operator) {
                case "==":
                case "===":
                case "!=":
                case "!==":
                    this.advance();
                    const right = this.parseRelational();
                    result = new ast_1.Binary(this.span(result.span.start), operator, result, right);
                    continue;
            }
            break;
        }
        return result;
    }
    parseRelational() {
        // '<', '>', '<=', '>='
        let result = this.parseAdditive();
        while (this.next.type === lexer_1.TokenType.Operator) {
            const operator = this.next.strValue;
            switch (operator) {
                case "<":
                case ">":
                case "<=":
                case ">=":
                    this.advance();
                    const right = this.parseAdditive();
                    result = new ast_1.Binary(this.span(result.span.start), operator, result, right);
                    continue;
            }
            break;
        }
        return result;
    }
    parseAdditive() {
        // '+', '-'
        let result = this.parseMultiplicative();
        while (this.next.type === lexer_1.TokenType.Operator) {
            const operator = this.next.strValue;
            switch (operator) {
                case "+":
                case "-":
                    this.advance();
                    const right = this.parseMultiplicative();
                    result = new ast_1.Binary(this.span(result.span.start), operator, result, right);
                    continue;
            }
            break;
        }
        return result;
    }
    parseMultiplicative() {
        // '*', '%', '/'
        let result = this.parsePrefix();
        while (this.next.type === lexer_1.TokenType.Operator) {
            const operator = this.next.strValue;
            switch (operator) {
                case "*":
                case "%":
                case "/":
                    this.advance();
                    const right = this.parsePrefix();
                    result = new ast_1.Binary(this.span(result.span.start), operator, result, right);
                    continue;
            }
            break;
        }
        return result;
    }
    parsePrefix() {
        if (this.next.type === lexer_1.TokenType.Operator) {
            const start = this.inputIndex;
            const operator = this.next.strValue;
            let result;
            switch (operator) {
                case "+":
                    this.advance();
                    return this.parsePrefix();
                case "-":
                    this.advance();
                    result = this.parsePrefix();
                    return new ast_1.Binary(this.span(start), operator, new ast_1.LiteralPrimitive(new ast_1.ParseSpan(start, start), 0), result);
                case "!":
                    this.advance();
                    result = this.parsePrefix();
                    return new ast_1.PrefixNot(this.span(start), result);
            }
        }
        return this.parseCallChain();
    }
    parseCallChain() {
        let result = this.parsePrimary();
        while (true) {
            if (this.optionalCharacter(chars.$PERIOD)) {
                result = this.parseAccessMemberOrMethodCall(result, false);
            }
            else if (this.optionalOperator("?.")) {
                result = this.parseAccessMemberOrMethodCall(result, true);
            }
            else if (this.optionalCharacter(chars.$LBRACKET)) {
                this.rbracketsExpected++;
                const key = this.parsePipe();
                this.rbracketsExpected--;
                this.expectCharacter(chars.$RBRACKET);
                if (this.optionalOperator("=")) {
                    const value = this.parseConditional();
                    result = new ast_1.KeyedWrite(this.span(result.span.start), result, key, value);
                }
                else {
                    result = new ast_1.KeyedRead(this.span(result.span.start), result, key);
                }
            }
            else if (this.optionalCharacter(chars.$LPAREN)) {
                this.rparensExpected++;
                const args = this.parseCallArguments();
                this.rparensExpected--;
                this.expectCharacter(chars.$RPAREN);
                result = new ast_1.FunctionCall(this.span(result.span.start), result, args);
            }
            else if (this.optionalOperator("!")) {
                result = new ast_1.NonNullAssert(this.span(result.span.start), result);
            }
            else {
                return result;
            }
        }
    }
    parsePrimary() {
        const start = this.inputIndex;
        if (this.optionalCharacter(chars.$LPAREN)) {
            this.rparensExpected++;
            const result = this.parsePipe();
            this.rparensExpected--;
            this.expectCharacter(chars.$RPAREN);
            return result;
        }
        else if (this.next.isKeywordNull()) {
            this.advance();
            return new ast_1.LiteralPrimitive(this.span(start), null);
        }
        else if (this.next.isKeywordUndefined()) {
            this.advance();
            return new ast_1.LiteralPrimitive(this.span(start), void 0);
        }
        else if (this.next.isKeywordTrue()) {
            this.advance();
            return new ast_1.LiteralPrimitive(this.span(start), true);
        }
        else if (this.next.isKeywordFalse()) {
            this.advance();
            return new ast_1.LiteralPrimitive(this.span(start), false);
        }
        else if (this.next.isKeywordThis()) {
            this.advance();
            return new ast_1.ImplicitReceiver(this.span(start));
        }
        else if (this.optionalCharacter(chars.$LBRACKET)) {
            this.rbracketsExpected++;
            const elements = this.parseExpressionList(chars.$RBRACKET);
            this.rbracketsExpected--;
            this.expectCharacter(chars.$RBRACKET);
            return new ast_1.LiteralArray(this.span(start), elements);
        }
        else if (this.next.isCharacter(chars.$LBRACE)) {
            return this.parseLiteralMap();
        }
        else if (this.next.isIdentifier()) {
            return this.parseAccessMemberOrMethodCall(new ast_1.ImplicitReceiver(this.span(start)), false);
        }
        else if (this.next.isNumber()) {
            const value = this.next.toNumber();
            this.advance();
            return new ast_1.LiteralPrimitive(this.span(start), value);
        }
        else if (this.next.isString()) {
            const literalValue = this.next.toString();
            this.advance();
            return new ast_1.LiteralPrimitive(this.span(start), literalValue);
        }
        else if (this.index >= this.tokens.length) {
            this.error(`Unexpected end of expression: ${this.input}`);
            return new ast_1.EmptyExpr(this.span(start));
        }
        else {
            this.error(`Unexpected token ${this.next}`);
            return new ast_1.EmptyExpr(this.span(start));
        }
    }
    parseExpressionList(terminator) {
        const result = [];
        if (!this.next.isCharacter(terminator)) {
            do {
                result.push(this.parsePipe());
            } while (this.optionalCharacter(chars.$COMMA));
        }
        return result;
    }
    parseLiteralMap() {
        const keys = [];
        const values = [];
        const start = this.inputIndex;
        this.expectCharacter(chars.$LBRACE);
        if (!this.optionalCharacter(chars.$RBRACE)) {
            this.rbracesExpected++;
            do {
                const quoted = this.next.isString();
                const key = this.expectIdentifierOrKeywordOrString();
                keys.push({ key, quoted });
                this.expectCharacter(chars.$COLON);
                values.push(this.parsePipe());
            } while (this.optionalCharacter(chars.$COMMA));
            this.rbracesExpected--;
            this.expectCharacter(chars.$RBRACE);
        }
        return new ast_1.LiteralMap(this.span(start), keys, values);
    }
    parseAccessMemberOrMethodCall(receiver, isSafe = false) {
        const start = receiver.span.start;
        const id = this.expectIdentifierOrKeyword();
        if (this.optionalCharacter(chars.$LPAREN)) {
            this.rparensExpected++;
            const args = this.parseCallArguments();
            this.expectCharacter(chars.$RPAREN);
            this.rparensExpected--;
            const span = this.span(start);
            return isSafe ? new ast_1.SafeMethodCall(span, receiver, id, args) : new ast_1.MethodCall(span, receiver, id, args);
        }
        else {
            if (isSafe) {
                if (this.optionalOperator("=")) {
                    this.error("The '?.' operator cannot be used in the assignment");
                    return new ast_1.EmptyExpr(this.span(start));
                }
                else {
                    return new ast_1.SafePropertyRead(this.span(start), receiver, id);
                }
            }
            else {
                if (this.optionalOperator("=")) {
                    if (!this.parseAction) {
                        this.error("Bindings cannot contain assignments");
                        return new ast_1.EmptyExpr(this.span(start));
                    }
                    const value = this.parseConditional();
                    return new ast_1.PropertyWrite(this.span(start), receiver, id, value);
                }
                else {
                    return new ast_1.PropertyRead(this.span(start), receiver, id);
                }
            }
        }
    }
    parseCallArguments() {
        if (this.next.isCharacter(chars.$RPAREN)) {
            return [];
        }
        const positionals = [];
        do {
            positionals.push(this.parsePipe());
        } while (this.optionalCharacter(chars.$COMMA));
        return positionals;
    }
    /**
     * An identifier, a keyword, a string with an optional `-` inbetween.
     */
    expectTemplateBindingKey() {
        let result = "";
        let operatorFound = false;
        do {
            result += this.expectIdentifierOrKeywordOrString();
            operatorFound = this.optionalOperator("-");
            if (operatorFound) {
                result += "-";
            }
        } while (operatorFound);
        return result.toString();
    }
    parseTemplateBindings() {
        const bindings = [];
        let prefix = null;
        const warnings = [];
        while (this.index < this.tokens.length) {
            const start = this.inputIndex;
            let keyIsVar = this.peekKeywordLet();
            if (keyIsVar) {
                this.advance();
            }
            const rawKey = this.expectTemplateBindingKey();
            let key = rawKey;
            if (!keyIsVar) {
                if (prefix === null) {
                    prefix = key;
                }
                else {
                    key = prefix + key[0].toUpperCase() + key.substring(1);
                }
            }
            this.optionalCharacter(chars.$COLON);
            let name = null;
            let expression = null;
            if (keyIsVar) {
                if (this.optionalOperator("=")) {
                    name = this.expectTemplateBindingKey();
                }
                else {
                    name = "$implicit";
                }
            }
            else if (this.peekKeywordAs()) {
                const letStart = this.inputIndex;
                this.advance(); // consume `as`
                name = rawKey;
                key = this.expectTemplateBindingKey(); // read local var name
                keyIsVar = true;
            }
            else if (this.next !== lexer_1.EOF && !this.peekKeywordLet()) {
                const st = this.inputIndex;
                const ast = this.parsePipe();
                const source = this.input.substring(st - this.offset, this.inputIndex - this.offset);
                expression = new ast_1.ASTWithSource(ast, source, this.location, this.errors);
            }
            bindings.push(new ast_1.TemplateBinding(this.span(start), key, keyIsVar, name, expression));
            if (this.peekKeywordAs() && !keyIsVar) {
                const letStart = this.inputIndex;
                this.advance(); // consume `as`
                const letName = this.expectTemplateBindingKey(); // read local var name
                bindings.push(new ast_1.TemplateBinding(this.span(letStart), letName, true, key, null));
            }
            if (!this.optionalCharacter(chars.$SEMICOLON)) {
                this.optionalCharacter(chars.$COMMA);
            }
        }
        return new TemplateBindingParseResult(bindings, warnings, this.errors);
    }
    error(message, index = null) {
        this.errors.push(new ast_1.ParserError(message, this.input, this.locationText(index), this.location));
        this.skip();
    }
    locationText(index = null) {
        if (index === null) {
            index = this.index;
        }
        return index < this.tokens.length ? `at column ${this.tokens[index].index + 1} in` : `at the end of the expression`;
    }
    // Error recovery should skip tokens until it encounters a recovery point. skip() treats
    // the end of input and a ';' as unconditionally a recovery point. It also treats ')',
    // '}' and ']' as conditional recovery points if one of calling productions is expecting
    // one of these symbols. This allows skip() to recover from errors such as '(a.) + 1' allowing
    // more of the AST to be retained (it doesn't skip any tokens as the ')' is retained because
    // of the '(' begins an '(' <expr> ')' production). The recovery points of grouping symbols
    // must be conditional as they must be skipped if none of the calling productions are not
    // expecting the closing token else we will never make progress in the case of an
    // extraneous group closing symbol (such as a stray ')'). This is not the case for ';' because
    // parseChain() is always the root production and it expects a ';'.
    // If a production expects one of these token it increments the corresponding nesting count,
    // and then decrements it just prior to checking if the token is in the input.
    skip() {
        let n = this.next;
        while (this.index < this.tokens.length &&
            !n.isCharacter(chars.$SEMICOLON) &&
            (this.rparensExpected <= 0 || !n.isCharacter(chars.$RPAREN)) &&
            (this.rbracesExpected <= 0 || !n.isCharacter(chars.$RBRACE)) &&
            (this.rbracketsExpected <= 0 || !n.isCharacter(chars.$RBRACKET))) {
            if (this.next.isError()) {
                this.errors.push(new ast_1.ParserError(this.next.toString(), this.input, this.locationText(), this.location));
            }
            this.advance();
            n = this.next;
        }
    }
}
exports.ParseAST = ParseAST;
class SimpleExpressionChecker {
    constructor() {
        this.errors = [];
    }
    static check(ast) {
        const s = new SimpleExpressionChecker();
        ast.visit(s);
        return s.errors;
    }
    visitImplicitReceiver(ast, context) { }
    visitInterpolation(ast, context) { }
    visitLiteralPrimitive(ast, context) { }
    visitPropertyRead(ast, context) { }
    visitPropertyWrite(ast, context) { }
    visitSafePropertyRead(ast, context) { }
    visitMethodCall(ast, context) { }
    visitSafeMethodCall(ast, context) { }
    visitFunctionCall(ast, context) { }
    visitLiteralArray(ast, context) {
        this.visitAll(ast.expressions);
    }
    visitLiteralMap(ast, context) {
        this.visitAll(ast.values);
    }
    visitBinary(ast, context) { }
    visitPrefixNot(ast, context) { }
    visitNonNullAssert(ast, context) { }
    visitConditional(ast, context) { }
    visitPipe(ast, context) {
        this.errors.push("pipes");
    }
    visitKeyedRead(ast, context) { }
    visitKeyedWrite(ast, context) { }
    visitAll(asts) {
        return asts.map(node => node.visit(this));
    }
    visitChain(ast, context) { }
    visitQuote(ast, context) { }
}
//# sourceMappingURL=parser.js.map