/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.cluster.decommission;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.opensearch.OpenSearchTimeoutException;
import org.opensearch.action.ActionListener;
import org.opensearch.action.admin.cluster.decommission.awareness.delete.DeleteDecommissionStateResponse;
import org.opensearch.action.admin.cluster.decommission.awareness.put.DecommissionRequest;
import org.opensearch.action.admin.cluster.decommission.awareness.put.DecommissionResponse;
import org.opensearch.cluster.ClusterState;
import org.opensearch.cluster.ClusterStateObserver;
import org.opensearch.cluster.ClusterStateUpdateTask;
import org.opensearch.cluster.NotClusterManagerException;
import org.opensearch.cluster.decommission.DecommissionAttribute;
import org.opensearch.cluster.decommission.DecommissionAttributeMetadata;
import org.opensearch.cluster.decommission.DecommissionController;
import org.opensearch.cluster.decommission.DecommissionStatus;
import org.opensearch.cluster.decommission.DecommissioningFailedException;
import org.opensearch.cluster.metadata.Metadata;
import org.opensearch.cluster.metadata.WeightedRoutingMetadata;
import org.opensearch.cluster.node.DiscoveryNode;
import org.opensearch.cluster.routing.WeightedRouting;
import org.opensearch.cluster.routing.allocation.AllocationService;
import org.opensearch.cluster.routing.allocation.decider.AwarenessAllocationDecider;
import org.opensearch.cluster.service.ClusterService;
import org.opensearch.common.Priority;
import org.opensearch.common.inject.Inject;
import org.opensearch.common.settings.ClusterSettings;
import org.opensearch.common.settings.Settings;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.threadpool.ThreadPool;
import org.opensearch.transport.TransportService;

public class DecommissionService {
    private static final Logger logger = LogManager.getLogger(DecommissionService.class);
    private final ClusterService clusterService;
    private final TransportService transportService;
    private final ThreadPool threadPool;
    private final DecommissionController decommissionController;
    private volatile List<String> awarenessAttributes;
    private volatile Map<String, List<String>> forcedAwarenessAttributes;

    @Inject
    public DecommissionService(Settings settings, ClusterSettings clusterSettings, ClusterService clusterService, TransportService transportService, ThreadPool threadPool, AllocationService allocationService) {
        this.clusterService = clusterService;
        this.transportService = transportService;
        this.threadPool = threadPool;
        this.decommissionController = new DecommissionController(clusterService, transportService, allocationService, threadPool);
        this.awarenessAttributes = AwarenessAllocationDecider.CLUSTER_ROUTING_ALLOCATION_AWARENESS_ATTRIBUTE_SETTING.get(settings);
        clusterSettings.addSettingsUpdateConsumer(AwarenessAllocationDecider.CLUSTER_ROUTING_ALLOCATION_AWARENESS_ATTRIBUTE_SETTING, this::setAwarenessAttributes);
        this.setForcedAwarenessAttributes(AwarenessAllocationDecider.CLUSTER_ROUTING_ALLOCATION_AWARENESS_FORCE_GROUP_SETTING.get(settings));
        clusterSettings.addSettingsUpdateConsumer(AwarenessAllocationDecider.CLUSTER_ROUTING_ALLOCATION_AWARENESS_FORCE_GROUP_SETTING, this::setForcedAwarenessAttributes);
    }

    private void setAwarenessAttributes(List<String> awarenessAttributes) {
        this.awarenessAttributes = awarenessAttributes;
    }

    private void setForcedAwarenessAttributes(Settings forceSettings) {
        HashMap<String, List<String>> forcedAwarenessAttributes = new HashMap<String, List<String>>();
        Map<String, Settings> forceGroups = forceSettings.getAsGroups();
        for (Map.Entry<String, Settings> entry : forceGroups.entrySet()) {
            List<String> aValues = entry.getValue().getAsList("values");
            if (aValues.size() <= 0) continue;
            forcedAwarenessAttributes.put(entry.getKey(), aValues);
        }
        this.forcedAwarenessAttributes = forcedAwarenessAttributes;
    }

    public void startDecommissionAction(final DecommissionRequest decommissionRequest, final ActionListener<DecommissionResponse> listener) {
        final DecommissionAttribute decommissionAttribute = decommissionRequest.getDecommissionAttribute();
        this.clusterService.submitStateUpdateTask("decommission [" + decommissionAttribute + "]", new ClusterStateUpdateTask(Priority.URGENT){

            @Override
            public ClusterState execute(ClusterState currentState) {
                DecommissionService.validateAwarenessAttribute(decommissionAttribute, DecommissionService.this.awarenessAttributes, DecommissionService.this.forcedAwarenessAttributes);
                DecommissionAttributeMetadata decommissionAttributeMetadata = currentState.metadata().decommissionAttributeMetadata();
                DecommissionService.ensureEligibleRequest(decommissionAttributeMetadata, decommissionAttribute);
                DecommissionService.ensureToBeDecommissionedAttributeWeighedAway(currentState, decommissionAttribute);
                decommissionAttributeMetadata = new DecommissionAttributeMetadata(decommissionAttribute);
                logger.info("registering decommission metadata [{}] to execute action", (Object)decommissionAttributeMetadata.toString());
                return ClusterState.builder(currentState).metadata(Metadata.builder(currentState.metadata()).decommissionAttributeMetadata(decommissionAttributeMetadata)).build();
            }

            @Override
            public void onFailure(String source, Exception e) {
                logger.error(() -> new ParameterizedMessage("failed to start decommission action for attribute [{}]", (Object)decommissionAttribute.toString()), (Throwable)e);
                listener.onFailure(e);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                DecommissionAttributeMetadata decommissionAttributeMetadata = newState.metadata().decommissionAttributeMetadata();
                assert (decommissionAttribute.equals(decommissionAttributeMetadata.decommissionAttribute()));
                logger.info("registered decommission metadata for attribute [{}] with status [{}]", (Object)decommissionAttributeMetadata.decommissionAttribute(), (Object)decommissionAttributeMetadata.status());
                DecommissionService.this.decommissionClusterManagerNodes(decommissionRequest, listener);
            }
        });
    }

    private synchronized void decommissionClusterManagerNodes(final DecommissionRequest decommissionRequest, final ActionListener<DecommissionResponse> listener) {
        final DecommissionAttribute decommissionAttribute = decommissionRequest.getDecommissionAttribute();
        ClusterState state = this.clusterService.getClusterApplierService().state();
        final Set<DiscoveryNode> clusterManagerNodesToBeDecommissioned = this.filterNodesWithDecommissionAttribute(state, decommissionAttribute, true);
        logger.info("resolved cluster manager eligible nodes [{}] that should be removed from Voting Configuration", (Object)clusterManagerNodesToBeDecommissioned.toString());
        final Set<String> nodeIdsToBeExcluded = clusterManagerNodesToBeDecommissioned.stream().map(DiscoveryNode::getId).collect(Collectors.toSet());
        final Predicate<ClusterState> allNodesRemovedAndAbdicated = clusterState -> {
            Set<String> votingConfigNodeIds = clusterState.getLastCommittedConfiguration().getNodeIds();
            return nodeIdsToBeExcluded.stream().noneMatch(votingConfigNodeIds::contains) && !nodeIdsToBeExcluded.contains(clusterState.nodes().getClusterManagerNodeId()) && clusterState.nodes().getClusterManagerNodeId() != null;
        };
        final ActionListener<Void> exclusionListener = new ActionListener<Void>(){

            @Override
            public void onResponse(Void unused) {
                if (DecommissionService.this.clusterService.getClusterApplierService().state().nodes().isLocalNodeElectedClusterManager()) {
                    if (DecommissionService.nodeHasDecommissionedAttribute(DecommissionService.this.clusterService.localNode(), decommissionAttribute)) {
                        String errorMsg = "unexpected state encountered [local node is to-be-decommissioned leader] while executing decommission request";
                        logger.error(errorMsg);
                        DecommissionService.this.clearVotingConfigExclusionAndUpdateStatus(false, false);
                        listener.onFailure(new IllegalStateException(errorMsg));
                    } else {
                        logger.info("will attempt to fail decommissioned nodes as local node is eligible to process the request");
                        listener.onResponse(new DecommissionResponse(true));
                        DecommissionService.this.drainNodesWithDecommissionedAttribute(decommissionRequest);
                    }
                } else {
                    logger.info("local node is not eligible to process the request, throwing NotClusterManagerException to attempt a retry on an eligible node");
                    listener.onFailure(new NotClusterManagerException("node [" + DecommissionService.this.transportService.getLocalNode().toString() + "] not eligible to execute decommission request. Will retry until timeout."));
                }
            }

            @Override
            public void onFailure(Exception e) {
                listener.onFailure(e);
                DecommissionService.this.clearVotingConfigExclusionAndUpdateStatus(false, false);
            }
        };
        if (allNodesRemovedAndAbdicated.test(state)) {
            exclusionListener.onResponse(null);
        } else {
            logger.debug("sending transport request to remove nodes [{}] from voting config", (Object)nodeIdsToBeExcluded.toString());
            this.decommissionController.excludeDecommissionedNodesFromVotingConfig(nodeIdsToBeExcluded, new ActionListener<Void>(){

                @Override
                public void onResponse(Void unused) {
                    logger.info("successfully removed decommissioned cluster manager eligible nodes [{}] from voting config ", (Object)clusterManagerNodesToBeDecommissioned.toString());
                    ClusterStateObserver abdicationObserver = new ClusterStateObserver(DecommissionService.this.clusterService, TimeValue.timeValueSeconds((long)60L), logger, DecommissionService.this.threadPool.getThreadContext());
                    ClusterStateObserver.Listener abdicationListener = new ClusterStateObserver.Listener(){

                        @Override
                        public void onNewClusterState(ClusterState state) {
                            logger.debug("to-be-decommissioned node is no more the active leader");
                            exclusionListener.onResponse(null);
                        }

                        @Override
                        public void onClusterServiceClose() {
                            String errorMsg = "cluster service closed while waiting for abdication of to-be-decommissioned leader";
                            logger.warn(errorMsg);
                            listener.onFailure(new DecommissioningFailedException(decommissionAttribute, errorMsg));
                        }

                        @Override
                        public void onTimeout(TimeValue timeout) {
                            logger.info("timed out while waiting for abdication of to-be-decommissioned leader");
                            DecommissionService.this.clearVotingConfigExclusionAndUpdateStatus(false, false);
                            listener.onFailure(new OpenSearchTimeoutException("timed out [{}] while waiting for abdication of to-be-decommissioned leader", timeout.toString()));
                        }
                    };
                    ClusterState currentState = DecommissionService.this.clusterService.getClusterApplierService().state();
                    if (allNodesRemovedAndAbdicated.test(currentState)) {
                        abdicationListener.onNewClusterState(currentState);
                    } else {
                        logger.debug("waiting to abdicate to-be-decommissioned leader");
                        abdicationObserver.waitForNextChange(abdicationListener, allNodesRemovedAndAbdicated);
                    }
                }

                @Override
                public void onFailure(Exception e) {
                    logger.error((Message)new ParameterizedMessage("failure in removing to-be-decommissioned cluster manager eligible nodes [{}] from voting config", (Object)nodeIdsToBeExcluded.toString()), (Throwable)e);
                    exclusionListener.onFailure(e);
                }
            });
        }
    }

    void drainNodesWithDecommissionedAttribute(final DecommissionRequest decommissionRequest) {
        ClusterState state = this.clusterService.getClusterApplierService().state();
        final Set<DiscoveryNode> decommissionedNodes = this.filterNodesWithDecommissionAttribute(state, decommissionRequest.getDecommissionAttribute(), false);
        if (decommissionRequest.isNoDelay()) {
            this.failDecommissionedNodes(decommissionedNodes, decommissionRequest.getDecommissionAttribute());
        } else {
            this.decommissionController.updateMetadataWithDecommissionStatus(DecommissionStatus.DRAINING, new ActionListener<DecommissionStatus>(){

                @Override
                public void onResponse(DecommissionStatus status) {
                    logger.info("updated the decommission status to [{}]", (Object)status);
                    DecommissionService.this.scheduleNodesDecommissionOnTimeout(decommissionedNodes, decommissionRequest.getDelayTimeout());
                }

                @Override
                public void onFailure(Exception e) {
                    logger.error(() -> new ParameterizedMessage("failed to update decommission status for attribute [{}] to [{}]", (Object)decommissionRequest.getDecommissionAttribute().toString(), (Object)DecommissionStatus.DRAINING), (Throwable)e);
                    DecommissionService.this.clearVotingConfigExclusionAndUpdateStatus(false, false);
                }
            });
        }
    }

    void scheduleNodesDecommissionOnTimeout(Set<DiscoveryNode> decommissionedNodes, TimeValue timeoutForNodeDraining) {
        ClusterState state = this.clusterService.getClusterApplierService().state();
        DecommissionAttributeMetadata decommissionAttributeMetadata = state.metadata().decommissionAttributeMetadata();
        if (decommissionAttributeMetadata == null) {
            return;
        }
        assert (decommissionAttributeMetadata.status().equals((Object)DecommissionStatus.DRAINING)) : "Unexpected status encountered while decommissioning nodes.";
        DecommissionAttribute decommissionAttribute = decommissionAttributeMetadata.decommissionAttribute();
        this.transportService.getThreadPool().schedule(() -> {
            this.decommissionController.getActiveRequestCountOnDecommissionedNodes(decommissionedNodes);
            this.failDecommissionedNodes(decommissionedNodes, decommissionAttribute);
        }, timeoutForNodeDraining, "generic");
    }

    private void failDecommissionedNodes(final Set<DiscoveryNode> decommissionedNodes, final DecommissionAttribute decommissionAttribute) {
        this.decommissionController.updateMetadataWithDecommissionStatus(DecommissionStatus.IN_PROGRESS, new ActionListener<DecommissionStatus>(){

            @Override
            public void onResponse(DecommissionStatus status) {
                logger.info("updated the decommission status to [{}]", (Object)status);
                DecommissionService.this.decommissionController.removeDecommissionedNodes(decommissionedNodes, "nodes-decommissioned", TimeValue.timeValueSeconds((long)120L), new ActionListener<Void>(){

                    @Override
                    public void onResponse(Void unused) {
                        DecommissionService.this.clearVotingConfigExclusionAndUpdateStatus(true, true);
                    }

                    @Override
                    public void onFailure(Exception e) {
                        DecommissionService.this.clearVotingConfigExclusionAndUpdateStatus(false, false);
                    }
                });
            }

            @Override
            public void onFailure(Exception e) {
                logger.error(() -> new ParameterizedMessage("failed to update decommission status for attribute [{}] to [{}]", (Object)decommissionAttribute.toString(), (Object)DecommissionStatus.IN_PROGRESS), (Throwable)e);
                DecommissionService.this.clearVotingConfigExclusionAndUpdateStatus(false, false);
            }
        });
    }

    private void clearVotingConfigExclusionAndUpdateStatus(final boolean decommissionSuccessful, boolean waitForRemoval) {
        this.decommissionController.clearVotingConfigExclusion(new ActionListener<Void>(){

            @Override
            public void onResponse(Void unused) {
                logger.info("successfully cleared voting config exclusion after completing decommission action, proceeding to update metadata");
                DecommissionStatus updateStatusWith = decommissionSuccessful ? DecommissionStatus.SUCCESSFUL : DecommissionStatus.FAILED;
                DecommissionService.this.decommissionController.updateMetadataWithDecommissionStatus(updateStatusWith, DecommissionService.this.statusUpdateListener());
            }

            @Override
            public void onFailure(Exception e) {
                logger.debug((Message)new ParameterizedMessage("failure in clearing voting config exclusion after processing decommission request", new Object[0]), (Throwable)e);
                DecommissionService.this.decommissionController.updateMetadataWithDecommissionStatus(DecommissionStatus.FAILED, DecommissionService.this.statusUpdateListener());
            }
        }, waitForRemoval);
    }

    private Set<DiscoveryNode> filterNodesWithDecommissionAttribute(ClusterState clusterState, DecommissionAttribute decommissionAttribute, boolean onlyClusterManagerNodes) {
        Iterator<DiscoveryNode> nodesIter;
        HashSet<DiscoveryNode> nodesWithDecommissionAttribute = new HashSet<DiscoveryNode>();
        Iterator<DiscoveryNode> iterator = nodesIter = onlyClusterManagerNodes ? clusterState.nodes().getClusterManagerNodes().valuesIt() : clusterState.nodes().getNodes().valuesIt();
        while (nodesIter.hasNext()) {
            DiscoveryNode node = nodesIter.next();
            if (!DecommissionService.nodeHasDecommissionedAttribute(node, decommissionAttribute)) continue;
            nodesWithDecommissionAttribute.add(node);
        }
        return nodesWithDecommissionAttribute;
    }

    private static void validateAwarenessAttribute(DecommissionAttribute decommissionAttribute, List<String> awarenessAttributes, Map<String, List<String>> forcedAwarenessAttributes) {
        Object msg = null;
        if (awarenessAttributes == null) {
            msg = "awareness attribute not set to the cluster.";
        } else if (forcedAwarenessAttributes == null) {
            msg = "forced awareness attribute not set to the cluster.";
        } else if (!awarenessAttributes.contains(decommissionAttribute.attributeName())) {
            msg = "invalid awareness attribute requested for decommissioning";
        } else if (!forcedAwarenessAttributes.containsKey(decommissionAttribute.attributeName())) {
            msg = "forced awareness attribute [" + forcedAwarenessAttributes.toString() + "] doesn't have the decommissioning attribute";
        } else if (!forcedAwarenessAttributes.get(decommissionAttribute.attributeName()).contains(decommissionAttribute.attributeValue())) {
            msg = "invalid awareness attribute value requested for decommissioning. Set forced awareness values before to decommission";
        }
        if (msg != null) {
            throw new DecommissioningFailedException(decommissionAttribute, (String)msg);
        }
    }

    private static void ensureToBeDecommissionedAttributeWeighedAway(ClusterState state, DecommissionAttribute decommissionAttribute) {
        WeightedRoutingMetadata weightedRoutingMetadata = state.metadata().weightedRoutingMetadata();
        if (weightedRoutingMetadata == null) {
            throw new DecommissioningFailedException(decommissionAttribute, "no weights are set to the attribute. Please set appropriate weights before triggering decommission action");
        }
        WeightedRouting weightedRouting = weightedRoutingMetadata.getWeightedRouting();
        if (!weightedRouting.attributeName().equals(decommissionAttribute.attributeName())) {
            throw new DecommissioningFailedException(decommissionAttribute, "no weights are specified to attribute [" + decommissionAttribute.attributeName() + "]");
        }
        Double attributeValueWeight = weightedRouting.weights().get(decommissionAttribute.attributeValue());
        if (attributeValueWeight == null || !attributeValueWeight.equals(0.0)) {
            throw new DecommissioningFailedException(decommissionAttribute, "weight for decommissioned attribute is expected to be [0.0] but found [" + attributeValueWeight + "]");
        }
    }

    private static void ensureEligibleRequest(DecommissionAttributeMetadata decommissionAttributeMetadata, DecommissionAttribute requestedDecommissionAttribute) {
        String msg;
        block10: {
            block11: {
                msg = null;
                if (decommissionAttributeMetadata == null) break block10;
                if (!decommissionAttributeMetadata.decommissionAttribute().equals(requestedDecommissionAttribute)) break block11;
                switch (decommissionAttributeMetadata.status()) {
                    case INIT: 
                    case FAILED: {
                        break block10;
                    }
                    case DRAINING: 
                    case IN_PROGRESS: 
                    case SUCCESSFUL: {
                        msg = "same request is already in status [" + decommissionAttributeMetadata.status() + "]";
                        break block10;
                    }
                    default: {
                        throw new IllegalStateException("unknown status [" + decommissionAttributeMetadata.status() + "] currently registered in metadata");
                    }
                }
            }
            switch (decommissionAttributeMetadata.status()) {
                case SUCCESSFUL: {
                    msg = "one awareness attribute [" + decommissionAttributeMetadata.decommissionAttribute().toString() + "] already successfully decommissioned, recommission before triggering another decommission";
                    break;
                }
                case INIT: 
                case DRAINING: 
                case IN_PROGRESS: {
                    msg = "there's an inflight decommission request for attribute [" + decommissionAttributeMetadata.decommissionAttribute().toString() + "] is in progress, cannot process this request";
                    break;
                }
                case FAILED: {
                    break;
                }
                default: {
                    throw new IllegalStateException("unknown status [" + decommissionAttributeMetadata.status() + "] currently registered in metadata");
                }
            }
        }
        if (msg != null) {
            throw new DecommissioningFailedException(requestedDecommissionAttribute, msg);
        }
    }

    private ActionListener<DecommissionStatus> statusUpdateListener() {
        return new ActionListener<DecommissionStatus>(){

            @Override
            public void onResponse(DecommissionStatus status) {
                logger.info("updated the decommission status to [{}]", (Object)status);
            }

            @Override
            public void onFailure(Exception e) {
                logger.error("unexpected failure occurred during decommission status update", (Throwable)e);
            }
        };
    }

    public void startRecommissionAction(final ActionListener<DeleteDecommissionStateResponse> listener) {
        this.decommissionController.clearVotingConfigExclusion(new ActionListener<Void>(){

            @Override
            public void onResponse(Void unused) {
                logger.info("successfully cleared voting config exclusion for deleting the decommission.");
                DecommissionService.this.deleteDecommissionState(listener);
            }

            @Override
            public void onFailure(Exception e) {
                logger.error("Failure in clearing voting config during delete_decommission request.", (Throwable)e);
                listener.onFailure(e);
            }
        }, false);
    }

    void deleteDecommissionState(final ActionListener<DeleteDecommissionStateResponse> listener) {
        this.clusterService.submitStateUpdateTask("delete_decommission_state", new ClusterStateUpdateTask(Priority.URGENT){

            @Override
            public ClusterState execute(ClusterState currentState) {
                logger.info("Deleting the decommission attribute from the cluster state");
                Metadata metadata = currentState.metadata();
                Metadata.Builder mdBuilder = Metadata.builder(metadata);
                mdBuilder.removeCustom("decommissionedAttribute");
                return ClusterState.builder(currentState).metadata(mdBuilder).build();
            }

            @Override
            public void onFailure(String source, Exception e) {
                logger.error(() -> new ParameterizedMessage("Failed to clear decommission attribute. [{}]", (Object)source), (Throwable)e);
                listener.onFailure(e);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                assert (newState.metadata().decommissionAttributeMetadata() == null);
                listener.onResponse(new DeleteDecommissionStateResponse(true));
            }
        });
    }

    public static boolean nodeHasDecommissionedAttribute(DiscoveryNode discoveryNode, DecommissionAttribute decommissionAttribute) {
        String nodeAttributeValue = discoveryNode.getAttributes().get(decommissionAttribute.attributeName());
        return nodeAttributeValue != null && nodeAttributeValue.equals(decommissionAttribute.attributeValue());
    }

    public static boolean nodeCommissioned(DiscoveryNode discoveryNode, Metadata metadata) {
        DecommissionAttributeMetadata decommissionAttributeMetadata = metadata.decommissionAttributeMetadata();
        if (decommissionAttributeMetadata != null) {
            DecommissionAttribute decommissionAttribute = decommissionAttributeMetadata.decommissionAttribute();
            DecommissionStatus status = decommissionAttributeMetadata.status();
            if (decommissionAttribute != null && status != null && DecommissionService.nodeHasDecommissionedAttribute(discoveryNode, decommissionAttribute) && (status.equals((Object)DecommissionStatus.IN_PROGRESS) || status.equals((Object)DecommissionStatus.SUCCESSFUL) || status.equals((Object)DecommissionStatus.DRAINING))) {
                return false;
            }
        }
        return true;
    }
}

