"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDataSource = exports.getCredential = exports.configureClient = void 0;

var _opensearch = require("@opensearch-project/opensearch");

var _common = require("../../common");

var _data_sources = require("../../common/data_sources");

var _error = require("../lib/error");

var _client_config = require("./client_config");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const configureClient = async ({
  dataSourceId,
  savedObjects,
  cryptography
}, openSearchClientPoolSetup, config, logger) => {
  try {
    const dataSource = await getDataSource(dataSourceId, savedObjects);
    const rootClient = getRootClient(dataSource.attributes, config, openSearchClientPoolSetup);
    return await getQueryClient(rootClient, dataSource, cryptography);
  } catch (error) {
    logger.error(`Failed to get data source client for dataSourceId: [${dataSourceId}]`);
    logger.error(error); // Re-throw as DataSourceError

    throw (0, _error.createDataSourceError)(error);
  }
};

exports.configureClient = configureClient;

const getDataSource = async (dataSourceId, savedObjects) => {
  const dataSource = await savedObjects.get(_common.DATA_SOURCE_SAVED_OBJECT_TYPE, dataSourceId);
  return dataSource;
};

exports.getDataSource = getDataSource;

const getCredential = async (dataSource, cryptography) => {
  const {
    endpoint
  } = dataSource.attributes;
  const {
    username,
    password
  } = dataSource.attributes.auth.credentials;
  const {
    decryptedText,
    encryptionContext
  } = await cryptography.decodeAndDecrypt(password).catch(err => {
    // Re-throw as DataSourceError
    throw (0, _error.createDataSourceError)(err);
  });

  if (encryptionContext.endpoint !== endpoint) {
    throw new Error('Data source "endpoint" contaminated. Please delete and create another data source.');
  }

  const credential = {
    username,
    password: decryptedText
  };
  return credential;
};
/**
 * Create a child client object with given auth info.
 *
 * @param rootClient root client for the connection with given data source endpoint.
 * @param dataSource data source saved object
 * @param cryptography cryptography service for password encryption / decryption
 * @returns child client.
 */


exports.getCredential = getCredential;

const getQueryClient = async (rootClient, dataSource, cryptography) => {
  const authType = dataSource.attributes.auth.type;

  switch (authType) {
    case _data_sources.AuthType.NoAuth:
      return rootClient.child();

    case _data_sources.AuthType.UsernamePasswordType:
      const credential = await getCredential(dataSource, cryptography);
      return getBasicAuthClient(rootClient, credential);

    default:
      throw Error(`${authType} is not a supported auth type for data source`);
  }
};
/**
 * Gets a root client object of the OpenSearch endpoint.
 * Will attempt to get from cache, if cache miss, create a new one and load into cache.
 *
 * @param dataSourceAttr data source saved objects attributes.
 * @param config data source config
 * @returns OpenSearch client for the given data source endpoint.
 */


const getRootClient = (dataSourceAttr, config, {
  getClientFromPool,
  addClientToPool
}) => {
  const endpoint = dataSourceAttr.endpoint;
  const cachedClient = getClientFromPool(endpoint);

  if (cachedClient) {
    return cachedClient;
  } else {
    const clientOptions = (0, _client_config.parseClientOptions)(config, endpoint);
    const client = new _opensearch.Client(clientOptions);
    addClientToPool(endpoint, client);
    return client;
  }
};

const getBasicAuthClient = (rootClient, credential) => {
  const {
    username,
    password
  } = credential;
  return rootClient.child({
    auth: {
      username,
      password
    },
    // Child client doesn't allow auth option, adding null auth header to bypass,
    // so logic in child() can rebuild the auth header based on the auth input.
    // See https://github.com/opensearch-project/OpenSearch-Dashboards/issues/2182 for details
    headers: {
      authorization: null
    }
  });
};