/*!
 * ${copyright}
 */
sap.ui.define([
	"sap/base/Log",
	"sap/ui/base/SyncPromise",
	"sap/ui/model/Binding",
	"sap/ui/model/ChangeReason",
	"sap/ui/model/Filter",
	"sap/ui/model/FilterOperator",
	"sap/ui/model/FilterProcessor",
	"sap/ui/model/FilterType",
	"sap/ui/model/ListBinding",
	"sap/ui/model/Sorter",
	"sap/ui/model/odata/OperationMode",
	"sap/ui/model/odata/v4/Context",
	"sap/ui/model/odata/v4/ODataListBinding",
	"sap/ui/model/odata/v4/ODataModel",
	"sap/ui/model/odata/v4/ODataParentBinding",
	"sap/ui/model/odata/v4/lib/_AggregationCache",
	"sap/ui/model/odata/v4/lib/_AggregationHelper",
	"sap/ui/model/odata/v4/lib/_Cache",
	"sap/ui/model/odata/v4/lib/_GroupLock",
	"sap/ui/model/odata/v4/lib/_Helper",
	"sap/ui/model/odata/v4/lib/_Parser"
], function (Log, SyncPromise, Binding, ChangeReason, Filter, FilterOperator, FilterProcessor,
		FilterType, ListBinding, Sorter, OperationMode, Context, ODataListBinding, ODataModel,
		asODataParentBinding, _AggregationCache, _AggregationHelper, _Cache, _GroupLock, _Helper,
		_Parser) {
	/*eslint no-sparse-arrays: 0 */
	"use strict";

	var aAllowedBindingParameters = ["$$aggregation", "$$canonicalPath", "$$getKeepAliveContext",
			"$$groupId", "$$operationMode", "$$ownRequest", "$$patchWithoutSideEffects",
			"$$sharedRequest", "$$updateGroupId"],
		sClassName = "sap.ui.model.odata.v4.ODataListBinding",
		oContextPrototype = Object.getPrototypeOf(Context.create(null, null, "/foo")),
		oParentBinding = {
			getRootBinding : function () {
				return {
					isSuspended : function () { return false; }
				};
			},
			getUpdateGroupId : function () { return "update"; }
		},
		rTransientPredicate = /^\(\$uid=.+\)$/;

	/**
	 * Creates the data for _Cache.read.
	 *
	 * @param {number} iLength
	 *   array length
	 * @param {number} [iStart=0]
	 *   start index
	 * @param {boolean} [bDrillDown]
	 *   simulate drill-down, i.e. resolve with unwrapped array
	 * @param {number} [iCount]
	 *   the value for "$count", remains unset if undefined
	 * @param {boolean} [bKeyPredicates]
	 *   add a property "@$ui5._/predicate" with a key predicate
	 * @returns {object}
	 *   the data
	 */
	function createData(iLength, iStart, bDrillDown, iCount, bKeyPredicates) {
		var oData = {value : []},
			i;

		if (iCount !== undefined) {
			oData.value.$count = iCount;
		}
		iStart = iStart || 0;
		for (i = 0; i < iLength; i += 1) {
			oData.value[i] = {
				Name : "Name " + (iStart + i),
				LOCATION : {
					COUNTRY : "COUNTRY " + (iStart + i)
				},
				NullValue : null
			};
			if (bKeyPredicates) {
				_Helper.setPrivateAnnotation(oData.value[i], "predicate",
					"('" + (iStart + i) + "')");
			}
		}
		return bDrillDown ? oData.value : oData;
	}

	//*********************************************************************************************
	QUnit.module("sap.ui.model.odata.v4.ODataListBinding", {
		beforeEach : function () {
			this.oLogMock = this.mock(Log);
			this.oLogMock.expects("error").never();
			this.oLogMock.expects("warning").never();

			// create ODataModel
			this.oModel = new ODataModel({
				serviceUrl : "/service/?sap-client=111",
				synchronizationMode : "None"
			});
			this.oModel.setSizeLimit(3);
			// ensure that the requestor does not trigger requests
			this.mock(this.oModel.oRequestor).expects("request").never();
			// avoid that the cache requests actual metadata for faked responses
			this.mock(this.oModel.oRequestor.oModelInterface).expects("fetchMetadata").atLeast(0)
				.returns(SyncPromise.resolve());

			// in case "request" is restored, this catches accidental requests
			this.mock(_Helper).expects("createError").never();
		},

		/**
		 * Calls <code>this.oModel.bindList</code> using the given arguments, but avoids creating
		 * the prerendering task to unlock the read group lock.
		 *
		 * @returns {sap.ui.model.odata.v4.ODataListBinding} The list binding
		 */
		bindList : function () {
			try {
				this.stub(this.oModel, "addPrerenderingTask");
				return this.oModel.bindList.apply(this.oModel, arguments);
			} finally {
				this.oModel.addPrerenderingTask.restore();
			}
		},

		/**
		 * Creates a sinon mock for a cache object with read and refresh methods.
		 * @returns {object}
		 *   a Sinon mock for the created cache object
		 */
		getCacheMock : function () {
			var oCache = {
					isDeletingInOtherGroup : function () {},
					read : function () {},
					requestSideEffects : function () {},
					toString : function () { return "/service/EMPLOYEES"; }
				};

			this.mock(_Cache).expects("create").returns(oCache);
			return this.mock(oCache);
		}
	});

	//*********************************************************************************************
	QUnit.test("mixin", function (assert) {
		var oBinding = this.bindList("EMPLOYEES"),
			oMixin = {},
			aOverriddenFunctions = ["adjustPredicate", "destroy", "getDependentBindings",
				"getGeneration", "hasPendingChangesForPath"];

		asODataParentBinding(oMixin);

		aOverriddenFunctions.forEach(function (sFunction) {
			assert.notStrictEqual(oBinding[sFunction], oMixin[sFunction], "overwrite " + sFunction);
		});
		Object.keys(oMixin).forEach(function (sKey) {
			if (!aOverriddenFunctions.includes(sKey)) {
				assert.strictEqual(oBinding[sKey], oMixin[sKey], sKey);
			}
		});
	});

	//*********************************************************************************************
	QUnit.test("bindingCreated", function () {
		var oBinding,
			oExpectation = this.mock(this.oModel).expects("bindingCreated")
				.withExactArgs(sinon.match.object);

		this.mock(ODataListBinding.prototype).expects("getGroupId").returns("myGroup");
		this.mock(ODataListBinding.prototype).expects("createReadGroupLock")
			.withExactArgs("myGroup", true);

		oBinding = this.bindList("/EMPLOYEES");

		sinon.assert.calledWithExactly(oExpectation, sinon.match.same(oBinding));
	});

	//*********************************************************************************************
	QUnit.test("constructor: lock when creating with base context", function () {
		var oContext = this.oModel.createBindingContext("/TEAMS('42')");

		this.mock(ODataListBinding.prototype).expects("getGroupId").returns("myGroup");
		this.mock(ODataListBinding.prototype).expects("createReadGroupLock")
			.withExactArgs("myGroup", true);

		// code under test
		this.bindList("TEAM_2_EMPLOYEES", oContext);
	});

	//*********************************************************************************************
	QUnit.test("be V8-friendly", function (assert) {
		var oParentBindingSpy = this.spy(asODataParentBinding, "call"),
			oBinding = this.bindList("/EMPLOYEES");

		assert.strictEqual(oBinding.iActiveContexts, 0);
		assert.ok(oBinding.hasOwnProperty("aApplicationFilters"));
		assert.ok(oBinding.hasOwnProperty("bFirstCreateAtEnd"));
		assert.ok(oBinding.hasOwnProperty("sChangeReason"));
		assert.ok(oBinding.hasOwnProperty("aContexts"));
		assert.strictEqual(oBinding.iCreatedContexts, 0);
		assert.ok(oBinding.hasOwnProperty("iCurrentBegin"));
		assert.ok(oBinding.hasOwnProperty("iCurrentEnd"));
		assert.strictEqual(oBinding.iDeletedContexts, 0);
		assert.ok(oBinding.hasOwnProperty("oDiff"));
		assert.ok(oBinding.hasOwnProperty("aFilters"));
		assert.ok(oBinding.hasOwnProperty("sGroupId"));
		assert.ok(oBinding.hasOwnProperty("bHasAnalyticalInfo"));
		assert.ok(oBinding.hasOwnProperty("oHeaderContext"));
		assert.ok(oBinding.hasOwnProperty("bLengthFinal"));
		assert.ok(oBinding.hasOwnProperty("iMaxLength"));
		assert.ok(oBinding.hasOwnProperty("sOperationMode"));
		assert.ok(oBinding.hasOwnProperty("mQueryOptions"));
		assert.ok(oBinding.hasOwnProperty("mParameters"));
		assert.ok(oBinding.hasOwnProperty("mPreviousContextsByPath"));
		assert.ok(oBinding.hasOwnProperty("aPreviousData"));
		assert.ok(oBinding.hasOwnProperty("bRefreshKeptElements"));
		assert.ok(oBinding.hasOwnProperty("sResumeAction"));
		assert.ok(oBinding.hasOwnProperty("aSorters"));
		assert.ok(oBinding.hasOwnProperty("sUpdateGroupId"));

		assert.ok(oParentBindingSpy.calledOnceWithExactly(sinon.match.same(oBinding)));
	});

	//*********************************************************************************************
[undefined, "AddVirtualContext"].forEach(function (sChangeReason) {
	var sTitle = "initialize: resolved, suspended; sChangeReason = " + sChangeReason;

	QUnit.test(sTitle, function (assert) {
		var oBinding = this.bindList("n/a"),
			oRootBinding = {isSuspended : function () {}};

		oBinding.sChangeReason = sChangeReason;
		this.mock(oBinding).expects("isResolved").withExactArgs().returns(true);
		this.mock(oBinding).expects("getRootBinding").withExactArgs().returns(oRootBinding);
		this.mock(oRootBinding).expects("isSuspended").withExactArgs().returns(true);
		this.mock(oBinding).expects("_fireChange").never();
		this.mock(oBinding).expects("_fireRefresh").never();

		// code under test
		oBinding.initialize();

		assert.strictEqual(oBinding.sChangeReason, sChangeReason);
		assert.strictEqual(oBinding.sResumeChangeReason,
			sChangeReason ? ChangeReason.Change : ChangeReason.Refresh);
	});
});

	//*********************************************************************************************
	QUnit.test("initialize: resolved, refresh", function (assert) {
		var oBinding = this.bindList("n/a"),
			oRootBinding = {isSuspended : function () {}};

		oBinding.sChangeReason = undefined;
		this.mock(oBinding).expects("isResolved").withExactArgs().returns(true);
		this.mock(oBinding).expects("getRootBinding").withExactArgs().returns(oRootBinding);
		this.mock(oRootBinding).expects("isSuspended").withExactArgs().returns(false);
		this.mock(oBinding).expects("_fireRefresh")
			.withExactArgs({reason : ChangeReason.Refresh});

		// code under test
		oBinding.initialize();

		assert.strictEqual(oBinding.sChangeReason, ChangeReason.Refresh);
	});

	//*********************************************************************************************
	QUnit.test("initialize: resolved, with change reason", function (assert) {
		var oBinding = this.bindList("n/a"),
			oRootBinding = {isSuspended : function () {}};

		oBinding.sChangeReason = "AddVirtualContext";
		this.mock(oBinding).expects("isResolved").withExactArgs().returns(true);
		this.mock(oBinding).expects("getRootBinding").withExactArgs().returns(oRootBinding);
		this.mock(oRootBinding).expects("isSuspended").withExactArgs().returns(false);
		this.mock(oBinding).expects("_fireChange").withExactArgs({
			detailedReason : "AddVirtualContext",
			reason : ChangeReason.Change
		});

		// code under test
		oBinding.initialize();

		assert.strictEqual(oBinding.sChangeReason, "AddVirtualContext");
	});

	//*********************************************************************************************
	QUnit.test("initialize: unresolved", function (assert) {
		var oBinding = this.bindList("/n/a"),
			sChangeReason = {};

		oBinding.sChangeReason = sChangeReason;
		this.mock(oBinding).expects("isResolved").withExactArgs().returns(false);
		this.mock(oBinding).expects("_fireRefresh").never();

		// code under test
		oBinding.initialize();

		assert.strictEqual(oBinding.sChangeReason, sChangeReason);
	});

	//*********************************************************************************************
	QUnit.test("constructor", function (assert) {
		var oBinding,
			oContext = {},
			oCreateMock,
			aFilters = [],
			vFilters = {},
			oHeaderContext = {},
			oHelperMock = this.mock(_Helper),
			oODataListBindingMock = this.mock(ODataListBinding.prototype),
			mParameters = {/*see clone below for actual content*/},
			mParametersClone = {
				$$groupId : "group",
				$$operationMode : OperationMode.Server,
				$$sharedRequest : "sharedRequest",
				$$updateGroupId : "update group"
			},
			aSorters = [],
			vSorters = {};

		oCreateMock = this.mock(Context).expects("createNewContext")
			.withExactArgs(sinon.match.same(this.oModel), sinon.match.object, "/EMPLOYEES")
			.returns(oHeaderContext);
		oHelperMock.expects("toArray").withExactArgs(sinon.match.same(vFilters)).returns(aFilters);
		oHelperMock.expects("toArray").withExactArgs(sinon.match.same(vSorters)).returns(aSorters);
		this.mock(_Helper).expects("clone").withExactArgs(sinon.match.same(mParameters))
			.returns(mParametersClone);
		oODataListBindingMock.expects("checkBindingParameters")
			.withExactArgs(sinon.match.same(mParametersClone), aAllowedBindingParameters);
		oODataListBindingMock.expects("applyParameters")
			.withExactArgs(sinon.match.same(mParametersClone));
		oODataListBindingMock.expects("setContext").withExactArgs(sinon.match.same(oContext));

		// code under test
		oBinding = new ODataListBinding(this.oModel, "/EMPLOYEES", oContext, vSorters, vFilters,
			mParameters);

		assert.strictEqual(oCreateMock.args[0][1], oBinding);

		assert.strictEqual(oBinding.aApplicationFilters, aFilters);
		assert.strictEqual(oBinding.sChangeReason, undefined);
		assert.strictEqual(oBinding.oDiff, undefined);
		assert.deepEqual(oBinding.aFilters, []);
		assert.strictEqual(oBinding.sGroupId, "group");
		assert.strictEqual(oBinding.bHasAnalyticalInfo, false);
		assert.deepEqual(oBinding.getHeaderContext(), oHeaderContext);
		assert.strictEqual(oBinding.sOperationMode, OperationMode.Server);
		assert.deepEqual(oBinding.mPreviousContextsByPath, {});
		assert.deepEqual(oBinding.aPreviousData, []);
		assert.strictEqual(oBinding.bRefreshKeptElements, false);
		assert.strictEqual(oBinding.bSharedRequest, "sharedRequest");
		assert.strictEqual(oBinding.aSorters, aSorters);
		assert.strictEqual(oBinding.sUpdateGroupId, "update group");
	});

	//*********************************************************************************************
	QUnit.test("constructor: $$sharedRequest from model", function (assert) {
		var oBinding,
			bSharedRequests = {/*false,true*/};

		this.oModel.bSharedRequests = bSharedRequests;

		// code under test
		oBinding = new ODataListBinding(this.oModel, "/EMPLOYEES");

		assert.strictEqual(oBinding.bSharedRequest, bSharedRequests);
	});

	//*********************************************************************************************
	[false, true].forEach(function (bAutoExpandSelect) {
		QUnit.test("c'tor: AddVirtualContext = " + bAutoExpandSelect, function (assert) {
			var oBinding;

			this.oModel.bAutoExpandSelect = bAutoExpandSelect;

			// code under test
			oBinding = this.bindList("/EMPLOYEES");

			assert.strictEqual(oBinding.sChangeReason,
				bAutoExpandSelect ? "AddVirtualContext" : undefined);
		});
	});

	//*********************************************************************************************
	QUnit.test("c'tor: no AddVirtualContext w/ $$aggregation", function (assert) {
		var oBinding,
			mClonedParameters = {},
			mParameters = {/*$$aggregation : {aggregate : {"n/a" : {}}}*/};

		this.oModel.bAutoExpandSelect = true;
		this.mock(_Helper).expects("clone").withExactArgs(sinon.match.same(mParameters))
			.returns(mClonedParameters);
		this.mock(_Helper).expects("isDataAggregation")
			.withExactArgs(sinon.match.same(mClonedParameters)).returns(true);
		// avoid 2nd call to _Helper.clone
		this.mock(ODataListBinding.prototype).expects("applyParameters");

		// code under test
		oBinding = this.bindList("/EMPLOYEES", null, [], [], mParameters);

		assert.strictEqual(oBinding.sChangeReason, undefined);
	});

	//*********************************************************************************************
	QUnit.test("c'tor: error case", function (assert) {
		assert.throws(function () {
			// code under test
			this.bindList("/EMPLOYEES", undefined, new Sorter("ID"));
		}, new Error("Unsupported operation mode: undefined"));
	});

	//*********************************************************************************************
	QUnit.test("setAggregation: pending changes", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		this.mock(oBinding).expects("hasPendingChanges").withExactArgs().returns(true);

		assert.throws(function () {
			// code under test
			oBinding.setAggregation({});
		}, new Error("Cannot set $$aggregation due to pending changes"));
	});

	//*********************************************************************************************
	QUnit.test("setAggregation: kept-alive context", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oContext0 = {isKeepAlive : function () {}},
			oContext1 = {isKeepAlive : function () {}};

		oBinding.aContexts = [oContext0, undefined, oContext1];

		this.mock(oContext0).expects("isKeepAlive").withExactArgs().returns(false);
		this.mock(oContext1).expects("isKeepAlive").withExactArgs().returns(true);

		assert.throws(function () {
			// code under test
			oBinding.setAggregation({});
		}, new Error("Cannot set $$aggregation due to a kept-alive context"));

		assert.notOk("$$aggregation" in oBinding.mQueryOptions);
	});

	//*********************************************************************************************
	QUnit.test("setAggregation: hidden kept-alive context", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oContext0 = {isKeepAlive : function () {}},
			oContext1 = {isKeepAlive : function () {}};

		oBinding.mPreviousContextsByPath = {foo : oContext0, bar : oContext1};

		this.mock(oContext0).expects("isKeepAlive").withExactArgs().returns(false);
		this.mock(oContext1).expects("isKeepAlive").withExactArgs().returns(true);

		assert.throws(function () {
			// code under test
			oBinding.setAggregation({});
		}, new Error("Cannot set $$aggregation due to a kept-alive context"));

		assert.notOk("$$aggregation" in oBinding.mQueryOptions);
	});

	//*********************************************************************************************
[null, {group : {dimension : {}}}].forEach(function (oAggregation, i) {
	QUnit.test("setAggregation, " + i, function () {
		var oBinding = this.bindList("/EMPLOYEES", undefined, undefined, undefined, {
				$$aggregation : {aggregate : {"n/a" : {}}},
				$$groupId : "foo",
				$filter : "bar",
				custom : "baz"
			});

		this.mock(oBinding).expects("resetKeepAlive").withExactArgs();
		// idea: #setAggregation(o) is like #changeParameters({$$aggregation : o})
		this.mock(oBinding).expects("applyParameters").withExactArgs({
				$$aggregation : oAggregation,
				$$groupId : "foo",
				$filter : "bar",
				custom : "baz"
			}, "");

		// code under test
		oBinding.setAggregation(oAggregation);
	});
});

	//*********************************************************************************************
	QUnit.test("setAggregation: undefined", function () {
		var oBinding = this.bindList("/EMPLOYEES", undefined, undefined, undefined, {
				$$aggregation : {aggregate : {"n/a" : {}}},
				$$groupId : "foo",
				$filter : "bar",
				custom : "baz"
			});

		this.mock(oBinding).expects("resetKeepAlive").never();
		// idea: #setAggregation(o) is like #changeParameters({$$aggregation : o})
		this.mock(oBinding).expects("applyParameters").withExactArgs({
				$$groupId : "foo",
				$filter : "bar",
				custom : "baz"
			}, "");

		// code under test
		oBinding.setAggregation();
	});

	//*********************************************************************************************
[undefined, {group : {dimension : {}}}].forEach(function (oAggregation) {
	QUnit.test("setAggregation: applyParameters fails", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES", undefined, undefined, undefined, {
				$$aggregation : {aggregate : {"n/a" : {}}},
				$$groupId : "foo",
				$filter : "bar",
				custom : "baz"
			}),
			oError = new Error("This call intentionally failed"),
			mExpectedParameters = {
				$$groupId : "foo",
				$filter : "bar",
				custom : "baz"
			},
			sOldValue = JSON.stringify(oBinding.mParameters.$$aggregation);

		// idea: #setAggregation(o) is like #changeParameters({$$aggregation : o})
		if (oAggregation) {
			mExpectedParameters.$$aggregation = oAggregation;
		}
		this.mock(oBinding).expects("applyParameters").withExactArgs(mExpectedParameters, "")
			.throws(oError);

		assert.throws(function () {
			// code under test
			oBinding.setAggregation(oAggregation);
		}, oError);

		assert.strictEqual(JSON.stringify(oBinding.mParameters.$$aggregation), sOldValue,
			"old value unchanged");
	});
});

	//*********************************************************************************************
	QUnit.test("applyParameters: simulate call from c'tor", function (assert) {
		var oAggregation = {},
			sApply = "A.P.P.L.E.",
			oBinding = this.bindList("/EMPLOYEES"),
			oExpectation,
			oModelMock = this.mock(this.oModel),
			mParameters = {
				$$aggregation : oAggregation,
				$filter : "bar"
			};

		assert.strictEqual(oBinding.mParameters.$$aggregation, undefined, "initial value");

		this.oModel.bAutoExpandSelect = "~autoExpandSelect~";
		oBinding.mCacheByResourcePath = {
			"/Products" : {}
		};
		oBinding.oHeaderContext = undefined; // not yet...
		this.mock(_AggregationHelper).expects("validateAggregation")
			.withExactArgs(sinon.match.same(oAggregation), "/EMPLOYEES",
				sinon.match.same(this.oModel.oInterface.fetchMetadata), "~autoExpandSelect~");
		this.mock(_AggregationHelper).expects("buildApply")
			.withExactArgs(sinon.match.same(oAggregation)).returns({$apply : sApply});
		oModelMock.expects("buildQueryOptions").withExactArgs(sinon.match.same(mParameters), true)
			.returns({$filter : "bar"});
		oExpectation = this.mock(oBinding).expects("removeCachesAndMessages").withExactArgs("");
		this.mock(oBinding).expects("fetchCache").callsFake(function () {
			// test if #removeCachesAndMessages is called before #fetchCache
			assert.ok(oExpectation.called);
		});
		this.mock(oBinding).expects("reset").withExactArgs(undefined);

		// code under test
		oBinding.applyParameters(mParameters);

		assert.deepEqual(oBinding.mQueryOptions, {
			$apply : sApply,
			$filter : "bar"
		}, "mQueryOptions");
		assert.deepEqual(oBinding.mParameters, mParameters);
		assert.strictEqual(oBinding.mParameters.$$aggregation, oAggregation, "$$aggregation");
	});

	//*********************************************************************************************
	QUnit.test("applyParameters: $$aggregation & $apply", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		assert.throws(function () {
			// code under test
			// Note: this is the same, no matter if both are supplied to c'tor or $apply is added
			// later via #changeParameters
			oBinding.applyParameters({$$aggregation : {}, $apply : ""});
		}, new Error("Cannot combine $$aggregation and $apply"));
		assert.notOk("$apply" in oBinding.mQueryOptions);
	});

	//*********************************************************************************************
	QUnit.test("applyParameters: $$getKeepAliveContext & $apply", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		assert.throws(function () {
			// code under test
			// Note: this is the same, no matter if both are supplied to c'tor or $apply is added
			// later via #changeParameters
			oBinding.applyParameters({$apply : "", $$getKeepAliveContext : true});
		}, new Error("Cannot combine $$getKeepAliveContext and $apply"));
		assert.notOk("$apply" in oBinding.mQueryOptions);
	});

	//*********************************************************************************************
	QUnit.test("applyParameters: validateAggregation fails", function (assert) {
		var oAggregation = {},
			oBinding = this.bindList("/EMPLOYEES"),
			oError = new Error("This call intentionally failed");

		oBinding.mParameters.$$aggregation = oAggregation;
		oBinding.mQueryOptions = {$apply : "A.P.P.L.E."};
		this.mock(_AggregationHelper).expects("validateAggregation")
			.withExactArgs({"n/a" : "unsupported content here"}, "/EMPLOYEES",
				sinon.match.same(this.oModel.oInterface.fetchMetadata), false)
			.throws(oError);
		this.mock(_AggregationHelper).expects("buildApply").never();

		assert.throws(function () {
			// code under test
			oBinding.applyParameters({$$aggregation : {"n/a" : "unsupported content here"}});
		}, oError);
		assert.strictEqual(oBinding.mParameters.$$aggregation, oAggregation, "unchanged");
		assert.deepEqual(oBinding.mQueryOptions, {$apply : "A.P.P.L.E."}, "unchanged");
	});

	//*********************************************************************************************
	QUnit.test("applyParameters: buildQueryOptions fails", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oError = new Error("This call intentionally failed"),
			mParameters = {"sap-*" : "not allowed"};

		this.mock(oBinding.oModel).expects("buildQueryOptions")
			.withExactArgs(sinon.match.same(mParameters), true)
			.throws(oError);

		assert.throws(function () {
			// code under test
			oBinding.applyParameters(mParameters);
		}, oError);
		assert.deepEqual(oBinding.mParameters, {}, "unchanged");
	});

	//*********************************************************************************************
[false, true].forEach(function (bSuspended) {
	var iCallCount = bSuspended ? 0 : 1;

	//*********************************************************************************************
[false, true].forEach(function (bAggregation) {
	var sTitle = "applyParameters: call from changeParameters, " + bSuspended
			+ ", w/ $$aggregation: " + bAggregation;

	QUnit.test(sTitle, function (assert) {
		var oAggregation = {},
			sApply = "A.P.P.L.E.",
			oBinding = this.bindList("TEAM_2_EMPLOYEES",
				Context.create(this.oModel, oParentBinding, "/TEAMS")),
			oModelMock = this.mock(this.oModel),
			mParameters = {
				$$operationMode : OperationMode.Server,
				$filter : "bar"
			};

		if (bAggregation) {
			mParameters.$$aggregation = oAggregation;
		}

		this.mock(_AggregationHelper).expects("validateAggregation").never();
		this.mock(_AggregationHelper).expects("buildApply").exactly(bAggregation ? 1 : 0)
			.withExactArgs(sinon.match.same(oAggregation)).returns({$apply : sApply});
		oModelMock.expects("buildQueryOptions")
			.withExactArgs(sinon.match.same(mParameters), true).returns({$filter : "bar"});
		this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs().returns(bSuspended);
		this.mock(oBinding).expects("setResumeChangeReason").exactly(bSuspended ? 1 : 0)
			.withExactArgs(ChangeReason.Change);
		this.mock(oBinding).expects("removeCachesAndMessages").exactly(iCallCount)
			.withExactArgs("");
		this.mock(oBinding).expects("fetchCache").exactly(iCallCount)
			.withExactArgs(sinon.match.same(oBinding.oContext));
		this.mock(oBinding).expects("reset").exactly(iCallCount).withExactArgs(ChangeReason.Change);
		this.mock(oBinding.oHeaderContext).expects("checkUpdate").exactly(iCallCount)
			.withExactArgs();

		// code under test
		oBinding.applyParameters(mParameters, ChangeReason.Change);

		assert.deepEqual(oBinding.mQueryOptions, bAggregation
			? {$apply : sApply, $filter : "bar"}
			: {$filter : "bar"});
		assert.deepEqual(oBinding.mParameters, mParameters);
		assert.strictEqual(oBinding.mParameters.$$aggregation,
			bAggregation ? oAggregation : undefined, "$$aggregation");
	});
});

	//*********************************************************************************************
	QUnit.test("applyParameters: with change in $apply, " + bSuspended, function (assert) {
		var oAggregation = {},
			sApply = "A.P.P.L.E.",
			oBinding = this.bindList("/EMPLOYEES"),
			mParameters = {
				$$aggregation : oAggregation,
				$filter : "bar"
			};

		oBinding.mQueryOptions.$apply = "old $apply";
		this.mock(_AggregationHelper).expects("validateAggregation")
			.withExactArgs(sinon.match.same(oAggregation), "/EMPLOYEES",
				sinon.match.same(this.oModel.oInterface.fetchMetadata), false);
		this.mock(_AggregationHelper).expects("buildApply")
			.withExactArgs(sinon.match.same(oAggregation)).returns({$apply : sApply});
		this.mock(this.oModel).expects("buildQueryOptions")
			.withExactArgs(sinon.match.same(mParameters), true).returns({$filter : "bar"});
		this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs().returns(bSuspended);
		this.mock(oBinding).expects("setResumeChangeReason").exactly(bSuspended ? 1 : 0)
			.withExactArgs(ChangeReason.Filter);
		this.mock(oBinding).expects("removeCachesAndMessages").exactly(iCallCount)
			.withExactArgs("");
		this.mock(oBinding).expects("fetchCache").exactly(iCallCount)
			.withExactArgs(sinon.match.same(oBinding.oContext));
		this.mock(oBinding).expects("reset").exactly(iCallCount).withExactArgs(ChangeReason.Filter);

		// code under test - simulate call from setAggregation
		oBinding.applyParameters(mParameters, "");

		assert.deepEqual(oBinding.mQueryOptions, {
			$apply : sApply,
			$filter : "bar"
		}, "mQueryOptions");
		assert.deepEqual(oBinding.mParameters, mParameters);
		assert.strictEqual(oBinding.mParameters.$$aggregation, oAggregation, "$$aggregation");
	});

	//*********************************************************************************************
	QUnit.test("applyParameters: $apply is dropped, " + bSuspended, function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			mParameters = {
				$filter : "bar"
			};

		oBinding.mQueryOptions.$apply = "old $apply";
		this.mock(_AggregationHelper).expects("validateAggregation").never();
		this.mock(_AggregationHelper).expects("buildApply").never();
		this.mock(this.oModel).expects("buildQueryOptions")
			.withExactArgs(sinon.match.same(mParameters), true).returns({$filter : "bar"});
		this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs().returns(bSuspended);
		this.mock(oBinding).expects("setResumeChangeReason").exactly(bSuspended ? 1 : 0)
			.withExactArgs(ChangeReason.Filter);
		this.mock(oBinding).expects("removeCachesAndMessages").exactly(iCallCount)
			.withExactArgs("");
		this.mock(oBinding).expects("fetchCache").exactly(iCallCount)
			.withExactArgs(sinon.match.same(oBinding.oContext));
		this.mock(oBinding).expects("reset").exactly(iCallCount).withExactArgs(ChangeReason.Filter);

		// code under test - simulate call from setAggregation
		oBinding.applyParameters(mParameters, "");

		assert.deepEqual(oBinding.mQueryOptions, {
			$filter : "bar"
		}, "mQueryOptions");
		assert.deepEqual(oBinding.mParameters, mParameters);
	});

	//*********************************************************************************************
	QUnit.test("applyParameters: $$aggregation not deepEqual, " + bSuspended, function (assert) {
		var oAggregation = {
				// aggregate : {GrossAmount : {subtotals : true}},
				// groupLevels : ["LifecycleStatus"]
			},
			sApply = "A.P.P.L.E.",
			oBinding = this.bindList("/EMPLOYEES"),
			mParameters = {
				$$aggregation : oAggregation,
				$filter : "bar"
			};

		oBinding.mQueryOptions.$apply = sApply; // no change in $apply
		oBinding.mParameters.$$aggregation = {
			// aggregate : {GrossAmount : {}},
			// groupLevels : ["LifecycleStatus"]
		};
		this.mock(_AggregationHelper).expects("validateAggregation")
			.withExactArgs(sinon.match.same(oAggregation), "/EMPLOYEES",
				sinon.match.same(this.oModel.oInterface.fetchMetadata), false);
		this.mock(_AggregationHelper).expects("buildApply")
			.withExactArgs(sinon.match.same(oAggregation)).returns({$apply : sApply});
		this.mock(this.oModel).expects("buildQueryOptions")
			.withExactArgs(sinon.match.same(mParameters), true).returns({$filter : "bar"});
		this.mock(_Helper).expects("deepEqual")
			.withExactArgs(sinon.match.same(oAggregation),
				sinon.match.same(oBinding.mParameters.$$aggregation))
			.returns(false);
		this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs().returns(bSuspended);
		this.mock(oBinding).expects("setResumeChangeReason").exactly(bSuspended ? 1 : 0)
			.withExactArgs(ChangeReason.Filter);
		this.mock(oBinding).expects("removeCachesAndMessages").exactly(iCallCount)
			.withExactArgs("");
		this.mock(oBinding).expects("fetchCache").exactly(iCallCount)
			.withExactArgs(sinon.match.same(oBinding.oContext));
		this.mock(oBinding).expects("reset").exactly(iCallCount).withExactArgs(ChangeReason.Filter);

		// code under test - simulate call from setAggregation
		oBinding.applyParameters(mParameters, "");

		assert.deepEqual(oBinding.mQueryOptions, {
			$apply : sApply,
			$filter : "bar"
		}, "mQueryOptions");
		assert.deepEqual(oBinding.mParameters, mParameters);
		assert.strictEqual(oBinding.mParameters.$$aggregation, oAggregation, "$$aggregation");
	});

	//*********************************************************************************************
	QUnit.test("applyParameters: from updateAnalyticalInfo, " + bSuspended, function (assert) {
		var oAggregation = {
				// aggregate : {GrossAmount : {subtotals : true}},
				// groupLevels : ["LifecycleStatus"]
			},
			sApply = "A.P.P.L.E.",
			oBinding = this.bindList("/EMPLOYEES"),
			mParameters = {
				$$aggregation : oAggregation
			};

		oBinding.bHasAnalyticalInfo = true;
		this.mock(_AggregationHelper).expects("validateAggregation")
			.withExactArgs(sinon.match.same(oAggregation), "/EMPLOYEES",
				sinon.match.same(this.oModel.oInterface.fetchMetadata), false);
		this.mock(_AggregationHelper).expects("buildApply")
			.withExactArgs(sinon.match.same(oAggregation)).returns({$apply : sApply});
		this.mock(this.oModel).expects("buildQueryOptions")
			.withExactArgs(sinon.match.same(mParameters), true).returns({});
		this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs().returns(bSuspended);
		this.mock(oBinding).expects("setResumeChangeReason").exactly(bSuspended ? 1 : 0)
			.withExactArgs(ChangeReason.Change);
		this.mock(oBinding).expects("removeCachesAndMessages").exactly(iCallCount)
			.withExactArgs("");
		this.mock(oBinding).expects("fetchCache").exactly(iCallCount)
			.withExactArgs(sinon.match.same(oBinding.oContext));
		this.mock(oBinding).expects("reset").exactly(iCallCount).withExactArgs(ChangeReason.Change);

		// code under test - simulate call from setAggregation
		oBinding.applyParameters(mParameters, "");

		assert.deepEqual(oBinding.mQueryOptions, {
			$apply : sApply
		}, "mQueryOptions");
		assert.deepEqual(oBinding.mParameters, mParameters);
		assert.strictEqual(oBinding.mParameters.$$aggregation, oAggregation, "$$aggregation");
	});
});

	//*********************************************************************************************
[{
	iDeepEqualCallCount : 1,
	oNewAggregation : {},
	oOldAggregation : {}
}, {
	iDeepEqualCallCount : 0,
	oNewAggregation : undefined,
	oOldAggregation : {}
}, {
	iDeepEqualCallCount : 0,
	oNewAggregation : {},
	oOldAggregation : undefined
}, {
	iDeepEqualCallCount : 0,
	oNewAggregation : undefined,
	oOldAggregation : undefined
}].forEach(function (oFixture, i) {
	QUnit.test("applyParameters: no change in $apply, " + i, function (assert) {
		var sApply = "A.P.P.L.E.",
			oBinding = this.bindList("/EMPLOYEES"),
			mParameters = {
				$$aggregation : oFixture.oNewAggregation,
				$filter : "bar"
			};

		oBinding.mParameters.$$aggregation = oFixture.oOldAggregation;
		oBinding.mQueryOptions.$apply = sApply;
		this.mock(_AggregationHelper).expects("validateAggregation")
			.withExactArgs(sinon.match.same(oFixture.oNewAggregation), "/EMPLOYEES",
				sinon.match.same(this.oModel.oInterface.fetchMetadata), false);
		this.mock(_AggregationHelper).expects("buildApply")
			.withExactArgs(sinon.match.same(oFixture.oNewAggregation)).returns({$apply : sApply});
		this.mock(this.oModel).expects("buildQueryOptions")
			.withExactArgs(sinon.match.same(mParameters), true).returns({$filter : "bar"});
		this.mock(_Helper).expects("deepEqual").exactly(oFixture.iDeepEqualCallCount)
			.withExactArgs(sinon.match.same(oFixture.oNewAggregation),
				sinon.match.same(oFixture.oOldAggregation))
			.returns(true);
		this.mock(oBinding).expects("isRootBindingSuspended").never();
		this.mock(oBinding).expects("setResumeChangeReason").never();
		this.mock(oBinding).expects("removeCachesAndMessages").never();
		this.mock(oBinding).expects("fetchCache").never();
		this.mock(oBinding).expects("reset").never();

		// code under test - simulate call from setAggregation
		oBinding.applyParameters(mParameters, "");

		assert.deepEqual(oBinding.mQueryOptions, {
			$apply : sApply,
			$filter : "bar"
		}, "mQueryOptions");
		assert.deepEqual(oBinding.mParameters, mParameters);
		assert.strictEqual(oBinding.mParameters.$$aggregation, oFixture.oNewAggregation);
	});
});

	//*********************************************************************************************
[undefined, false, true].forEach(function (bDrop) {
	[false, true].forEach(function (bKeepTransient) {
	QUnit.test("reset, bDrop=" + bDrop + ", bKeepTransient=" + bKeepTransient, function (assert) {
		var oBinding,
			oCreatedContext1 = { // "created persisted" from "inline creation row"
				getPath : function () { return "/EMPLOYEES('1')"; },
				isInactive : function () { return false; },
				isTransient : function () { return false; },
				iIndex : -1
			},
			oCreatedContext2 = { // ordinary "created persisted" => not kept!
				getPath : function () { return "/EMPLOYEES('2')"; },
				isInactive : function () { return undefined; },
				isTransient : function () { return false; },
				iIndex : -3
			},
			aPreviousContexts,
			oTransientContext1 = {
				getPath : function () { return "/EMPLOYEES($uid=id-1-23)"; },
				isInactive : function () { return undefined; },
				isTransient : function () { return true; },
				iIndex : -2
			},
			oTransientContext2 = {
				getPath : function () { return "/EMPLOYEES($uid=id-1-24)"; },
				isInactive : function () { return undefined; },
				isTransient : function () { return true; },
				iIndex : -4
			};

		// code under test: reset called from ODLB constructor
		oBinding = this.bindList("/EMPLOYEES");

		oBinding.createContexts(0, [{}, {}]);
		oBinding.createContexts(3, [{}]);
		aPreviousContexts = oBinding.aContexts.slice();
		oBinding.aContexts.unshift(oCreatedContext1);
		oBinding.aContexts.unshift(oTransientContext1);
		oBinding.aContexts.unshift(oCreatedContext2);
		oBinding.aContexts.unshift(oTransientContext2);
		// set members which should be reset to arbitrary values
		oBinding.iCurrentBegin = 10;
		oBinding.iCurrentEnd = 19;
		oBinding.bLengthFinal = true;
		oBinding.iMaxLength = 42;
		oBinding.iActiveContexts = 3; // let's assume one transient context is inactive
		oBinding.iCreatedContexts = 4;
		oBinding.bFirstCreateAtEnd = "~bFirstCreateAtEnd~";

		this.mock(oBinding).expects("getUpdateGroupId").withExactArgs()
			.returns(bKeepTransient ? "other" : "myGroup");
		this.mock(oBinding).expects("_fireRefresh").never();

		// code under test
		oBinding.reset(undefined, bDrop, "myGroup");

		assert.strictEqual(oBinding.iCurrentBegin, 0);
		assert.strictEqual(oBinding.iCurrentEnd, 0);
		assert.strictEqual(oBinding.isLengthFinal(), false);
		assert.strictEqual(oBinding.iMaxLength, Infinity);
		assert.strictEqual(oBinding.mPreviousContextsByPath["/EMPLOYEES/0"], aPreviousContexts[0]);
		assert.strictEqual(oBinding.mPreviousContextsByPath["/EMPLOYEES/1"], aPreviousContexts[1]);
		assert.strictEqual(oBinding.mPreviousContextsByPath["/EMPLOYEES/3"], aPreviousContexts[3]);

		if (bDrop === false) {
			assert.strictEqual(Object.keys(oBinding.mPreviousContextsByPath).length,
				bKeepTransient ? 4 : 6);
			assert.strictEqual(oBinding.mPreviousContextsByPath["/EMPLOYEES('2')"],
				oCreatedContext2);
			if (!bKeepTransient) {
				assert.strictEqual(oBinding.mPreviousContextsByPath["/EMPLOYEES($uid=id-1-23)"],
					oTransientContext1);
				assert.strictEqual(oBinding.mPreviousContextsByPath["/EMPLOYEES($uid=id-1-24)"],
					oTransientContext2);
			}
			assert.strictEqual(oCreatedContext1.iIndex, -1);
			assert.strictEqual(oCreatedContext2.iIndex, -3, "unchanged");
			assert.strictEqual(oTransientContext1.iIndex, -2);
			assert.strictEqual(oTransientContext2.iIndex, bKeepTransient ? -3 : -4);
			assert.deepEqual(oBinding.aContexts, bKeepTransient
				? [oTransientContext2, oTransientContext1, oCreatedContext1]
				: [oCreatedContext1]);
			assert.strictEqual(oBinding.iActiveContexts, bKeepTransient ? 2 : 0);
			assert.strictEqual(oBinding.iCreatedContexts, bKeepTransient ? 3 : 1);
			assert.strictEqual(oBinding.bFirstCreateAtEnd, "~bFirstCreateAtEnd~");
			return;
		}

		assert.strictEqual(Object.keys(oBinding.mPreviousContextsByPath).length, bDrop ? 7 : 5);
		assert.strictEqual(oBinding.mPreviousContextsByPath["/EMPLOYEES('1')"], oCreatedContext1);
		assert.strictEqual(oBinding.mPreviousContextsByPath["/EMPLOYEES('2')"], oCreatedContext2);
		if (bDrop) {
			assert.strictEqual(oBinding.mPreviousContextsByPath["/EMPLOYEES($uid=id-1-23)"],
				oTransientContext1);
			assert.strictEqual(oBinding.mPreviousContextsByPath["/EMPLOYEES($uid=id-1-24)"],
				oTransientContext2);
		} else {
			assert.strictEqual(oTransientContext1.iIndex, -1);
			assert.strictEqual(oTransientContext2.iIndex, -2);
		}
		assert.deepEqual(oBinding.aContexts, bDrop ? [] : [oTransientContext2, oTransientContext1]);
		assert.strictEqual(oBinding.iActiveContexts, bDrop ? 0 : 1);
		assert.strictEqual(oBinding.iCreatedContexts, bDrop ? 0 : 2);
		assert.strictEqual(oBinding.bFirstCreateAtEnd, bDrop ? undefined : "~bFirstCreateAtEnd~");
	});
	});
});

	//*********************************************************************************************
	QUnit.test("reset with change reason", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			sChangeReason = {};

		this.mock(oBinding).expects("getUpdateGroupId").never();
		this.mock(oBinding).expects("_fireRefresh")
			.withExactArgs({reason : sinon.match.same(sChangeReason)});

		// code under test
		oBinding.reset(sChangeReason);

		assert.strictEqual(oBinding.sChangeReason, sChangeReason);
	});

	//*********************************************************************************************
	QUnit.test("reset on initial binding with change reason 'Change'", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		this.mock(oBinding).expects("getUpdateGroupId").never();
		this.mock(oBinding).expects("_fireRefresh").never();

		// code under test
		oBinding.reset(ChangeReason.Change);

		assert.strictEqual(oBinding.sChangeReason, undefined);
	});

	//*********************************************************************************************
	QUnit.test("reset not initial binding with change reason 'Change'", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		oBinding.iCurrentEnd = 42;

		this.mock(oBinding).expects("getUpdateGroupId").never();
		this.mock(oBinding).expects("_fireRefresh").withExactArgs({reason : ChangeReason.Change});

		// code under test
		oBinding.reset(ChangeReason.Change);

		assert.strictEqual(oBinding.sChangeReason, ChangeReason.Change);
	});

	//*********************************************************************************************
	QUnit.test("bindList with OData query options", function (assert) {
		var oBaseContext = {getPath : function () { return "/"; }},
			oBinding,
			oCacheMock = this.mock(_Cache),
			oError = new Error("Unsupported ..."),
			oModelMock = this.mock(this.oModel),
			mParameters = {
				$apply : "filter(Amount gt 3)",
				$expand : "foo",
				$orderby : "bar",
				$search : '"foo bar" AND NOT foobar',
				$select : "bar",
				custom : "baz"
			},
			mQueryOptions = {$orderby : "bar"},
			oV4Context = {getBinding : function () {}};

		// absolute binding and binding with base context result in the same cache
		oModelMock.expects("buildQueryOptions").thrice()
			.withExactArgs(mParameters, true)
			.returns(mQueryOptions);
		this.mock(ODataListBinding.prototype).expects("getOrderby").twice()
			.withExactArgs(mQueryOptions.$orderby)
			.returns(mQueryOptions.$orderby);
		oCacheMock.expects("create")
			.withExactArgs(sinon.match.same(this.oModel.oRequestor), "EMPLOYEES",
				{$orderby : "bar", "sap-client" : "111"}, false, undefined, false)
			.returns({});
		this.mock(ODataListBinding.prototype).expects("restoreCreated").withExactArgs();
		this.spy(ODataListBinding.prototype, "reset");

		// code under test
		oBinding = this.bindList("/EMPLOYEES", oV4Context, undefined, undefined, mParameters);

		assert.ok(oBinding instanceof ODataListBinding);
		assert.strictEqual(oBinding.getModel(), this.oModel);
		assert.strictEqual(oBinding.getContext(), oV4Context);
		assert.strictEqual(oBinding.getPath(), "/EMPLOYEES");
		assert.deepEqual(oBinding.mParameters, mParameters);
		assert.strictEqual(oBinding.mQueryOptions, mQueryOptions);
		assert.ok(ODataListBinding.prototype.reset.calledWithExactly(undefined));
		assert.strictEqual(oBinding.hasOwnProperty("sChangeReason"), true);
		assert.strictEqual(oBinding.sChangeReason, undefined);
		assert.deepEqual(oBinding.oDiff, undefined);
		assert.deepEqual(oBinding.mPreviousContextsByPath, {});
		assert.deepEqual(oBinding.aPreviousData, []);

		oCacheMock.expects("create")
			.withExactArgs(sinon.match.same(this.oModel.oRequestor), "EMPLOYEES",
				{$orderby : "bar", "sap-client" : "111"}, false, "EMPLOYEES", false)
			.returns({});

		// code under test
		oBinding = this.bindList("EMPLOYEES", oBaseContext, undefined, undefined, mParameters);

		assert.ok(oBinding instanceof ODataListBinding);
		assert.strictEqual(oBinding.getModel(), this.oModel);
		assert.strictEqual(oBinding.getContext(), oBaseContext);
		assert.strictEqual(oBinding.getPath(), "EMPLOYEES");
		assert.deepEqual(oBinding.mParameters, mParameters);
		assert.strictEqual(oBinding.mQueryOptions, mQueryOptions);
		assert.ok(ODataListBinding.prototype.reset.calledWithExactly(undefined, true));
		assert.strictEqual(oBinding.hasOwnProperty("sChangeReason"), true);
		assert.strictEqual(oBinding.sChangeReason, undefined);
		assert.deepEqual(oBinding.oDiff, undefined);
		assert.deepEqual(oBinding.mPreviousContextsByPath, {});
		assert.deepEqual(oBinding.aPreviousData, []);

		// code under test
		oBinding = this.bindList("EMPLOYEE_2_TEAM", undefined, undefined, undefined, mParameters);

		assert.strictEqual(oBinding.oCachePromise.getResult(), null, "no cache");
		assert.strictEqual(oBinding.hasOwnProperty("sGroupId"), true);
		assert.strictEqual(oBinding.sGroupId, undefined);
		assert.deepEqual(oBinding.mParameters, mParameters);
		assert.strictEqual(oBinding.mQueryOptions, mQueryOptions);
		assert.ok(ODataListBinding.prototype.reset.calledWithExactly(undefined, true));
		assert.strictEqual(oBinding.hasOwnProperty("sChangeReason"), true);
		assert.strictEqual(oBinding.sChangeReason, undefined);

		//error for invalid parameters
		oModelMock.expects("buildQueryOptions").throws(oError);

		assert.throws(function () {
			// code under test
			this.bindList("/EMPLOYEES", null, undefined, undefined, mParameters);
		}, oError);
	});

	//*********************************************************************************************
	QUnit.test("bindList with sorters - error cases", function (assert) {
		assert.throws(function () {
			this.bindList("/EMPLOYEES", undefined, new Sorter("ID"), undefined,
				{$$operationMode : OperationMode.Client});
		}, new Error("Unsupported operation mode: Client"));
		assert.throws(function () {
			this.bindList("/EMPLOYEES", undefined, new Sorter("ID"), undefined,
				{$$operationMode : OperationMode.Auto});
		}, new Error("Unsupported operation mode: Auto"));
		assert.throws(function () {
			this.bindList("/EMPLOYEES", undefined, new Sorter("ID"));
		}, new Error("Unsupported operation mode: undefined"));
	});

	//*********************************************************************************************
	QUnit.test("bindList with filters", function (assert) {
		var oBinding,
			oFilter = new Filter("Name", FilterOperator.Contains, "foo"),
			aFilters = [oFilter],
			oHelperMock = this.mock(_Helper),
			mQueryParameters = {
				$$operationMode : OperationMode.Server,
				$filter : "bar"
			};

		oHelperMock.expects("toArray").withExactArgs(sinon.match.same(oFilter)).returns(aFilters);
		oHelperMock.expects("toArray").withExactArgs(undefined).returns([]);
		this.mock(ODataListBinding.prototype).expects("fetchFilter")
			.withExactArgs(undefined, mQueryParameters.$filter)
			.returns(SyncPromise.resolve());

		// code under test
		oBinding = this.bindList("/EMPLOYEES", undefined, undefined, oFilter, mQueryParameters);

		assert.strictEqual(oBinding.aApplicationFilters, aFilters);
	});

	//*********************************************************************************************
	QUnit.test("bindList with filters - error cases", function (assert) {
		assert.throws(function () {
			this.bindList("/EMPLOYEES", undefined, undefined, new Filter("ID", "eq", 42),
				{$$operationMode : OperationMode.Client});
		}, new Error("Unsupported operation mode: Client"));
		assert.throws(function () {
			this.bindList("/EMPLOYEES", undefined, undefined, new Filter("ID", "eq", 42),
				{$$operationMode : OperationMode.Auto});
		}, new Error("Unsupported operation mode: Auto"));
		assert.throws(function () {
			this.bindList("/EMPLOYEES", undefined, undefined, new Filter("ID", "eq", 42));
		}, new Error("Unsupported operation mode: undefined"));
	});

	//*********************************************************************************************
	QUnit.test("fetchData: w/ cache", function (assert) {
		var oBinding,
			oCache = {read : function () {}},
			oData = {},
			fnDataRequested = {/*function*/},
			oGroupLock = {},
			oPromise,
			that = this;

		this.mock(ODataListBinding.prototype).expects("fetchCache").callsFake(function () {
			this.oCachePromise = SyncPromise.resolve(Promise.resolve(oCache));
		});
		oBinding = this.bindList("/EMPLOYEES");
		this.mock(oCache).expects("read")
			.withExactArgs(1, 2, 3, sinon.match.same(oGroupLock),
				sinon.match.same(fnDataRequested))
			.returns(SyncPromise.resolve(Promise.resolve().then(function () {
				that.mock(oBinding).expects("assertSameCache")
					.withExactArgs(sinon.match.same(oCache));
				return oData;
			})));

		// code under test
		oPromise = oBinding.fetchData(1, 2, 3, oGroupLock, fnDataRequested);

		oBinding.sChangeReason = "sChangeReason";
		oBinding.bHasPathReductionToParent = true;
		this.oModel.bAutoExpandSelect = true;
		this.mock(oBinding).expects("checkSuspended").never();

		// code under test - must have no effect on absolute bindings
		oBinding.setContext({});

		assert.strictEqual(oBinding.sChangeReason, "sChangeReason");

		return oPromise.then(function (oResult) {
			assert.strictEqual(oResult, oData);
		});
	});

	//*********************************************************************************************
[false, true].forEach(function (bHasData) {
	QUnit.test("fetchData: w/o cache, data=" + bHasData, function (assert) {
		var oBinding = this.bindList("TEAM_2_EMPLOYEES"),
			oContext = Context.create({/*oModel*/}, oParentBinding, "/TEAMS('1')"),
			aData = [{id : 0}, {id : 1}, {id : 2}, {id : 3}, {id : 4}, {id : 5}],
			fnDataRequested = {/*function*/},
			oGroupLock = {unlock : function () {}};

		aData.$count = 42;
		this.mock(oBinding).expects("checkSuspended").withExactArgs(true);
		this.mock(oBinding).expects("fetchCache").callsFake(function () {
			this.oCache = undefined;
			this.oCachePromise = SyncPromise.resolve(Promise.resolve(null));
			this.sReducedPath = "/reduced/path";
		});
		this.mock(oBinding).expects("restoreCreated").withExactArgs();
		oBinding.setContext(oContext);
		this.mock(oGroupLock).expects("unlock").withExactArgs();
		this.mock(oContext).expects("fetchValue")
			.withExactArgs("/reduced/path")
			.returns(SyncPromise.resolve(Promise.resolve(bHasData ? aData : undefined)));

		// code under test
		return oBinding.fetchData(3, 2, 99, oGroupLock, fnDataRequested).then(function (oResult) {
			assert.deepEqual(oResult, {value : bHasData ? [{id : 3}, {id : 4}] : []});
			if (bHasData) {
				assert.strictEqual(oResult.value.$count, 42);
			}
		});
	});
});

	//*********************************************************************************************
	// This tests simulates the data access for a virtual context which may be removed from the
	// binding while fetchData still is waiting for the cache
[false, true].forEach(function (bHasCache) {
	QUnit.test("fetchData: context lost, cache=" + bHasCache, function (assert) {
		var oBinding = this.bindList("TEAM_2_EMPLOYEES"),
			oBindingMock = this.mock(oBinding),
			oContext = Context.create({/*oModel*/}, oParentBinding, "/TEAMS('1')"),
			oPromise;

		oBindingMock.expects("checkSuspended").withExactArgs(true);
		oBindingMock.expects("fetchCache").callsFake(function () {
			this.oCache = undefined;
			this.oCachePromise = SyncPromise.resolve(Promise.resolve(bHasCache ? {} : null));
			this.sReducedPath = "/reduced/path";
		});
		oBindingMock.expects("restoreCreated").withExactArgs();
		oBinding.setContext(oContext);
		this.mock(oContext).expects("fetchValue").never();

		// code under test
		oPromise = oBinding.fetchData(3, 2, 0);

		oBindingMock.expects("checkSuspended").withExactArgs(true);
		oBindingMock.expects("fetchCache").callsFake(function () {
			this.oCache = null;
			this.oCachePromise = SyncPromise.resolve(null);
			this.sReducedPath = undefined;
		});
		oBinding.setContext(null);

		return oPromise.then(function (oResult) {
			assert.deepEqual(oResult, undefined);
		});
	});
});

	//*********************************************************************************************
[false, true].forEach(function (bAsync) {
	[false, true].forEach(function (bGroupLock) {
	QUnit.test("fetchContexts: async=" + bAsync + ", groupLock=" + bGroupLock, function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			bChanged = {/*boolean*/},
			fnDataRequested = {/*function*/},
			oGroupLock = {},
			bPending = true,
			oPromise,
			oResult = {value : {}};

		this.mock(oBinding).expects("lockGroup").exactly(bGroupLock ? 0 : 1)
			.withExactArgs().returns(oGroupLock);
		this.mock(oBinding).expects("fetchData")
			.withExactArgs(1, 2, 3, sinon.match.same(oGroupLock),
				sinon.match.same(fnDataRequested))
			.returns(SyncPromise.resolve(oResult));
		this.mock(oBinding).expects("createContexts")
			.withExactArgs(1, sinon.match.same(oResult.value))
			.returns(SyncPromise.resolve(bChanged));

		// code under test
		oPromise = oBinding.fetchContexts(1, 2, 3, bGroupLock ? oGroupLock : undefined, bAsync,
				fnDataRequested)
			.then(function (bResult) {
				assert.strictEqual(bResult, bChanged);
				bPending = false;
			});
		this.mock(oBinding).expects("checkSuspended").never();
		oBinding.setContext({}); // must not change anything, the binding is absolute

		assert.strictEqual(bPending, bAsync);

		return oPromise;
	});
	});
});

	//*********************************************************************************************
[false, true].forEach(function (bFirstCreateAtEnd) {
	QUnit.test("fetchContexts: created, atEnd=" + bFirstCreateAtEnd, function () {
		var oBinding = this.bindList("/EMPLOYEES"),
			bChanged = {/*boolean*/},
			fnDataRequested = {/*function*/},
			oGroupLock = {},
			iReadStart = bFirstCreateAtEnd ? 3 : 1,
			oResult = {value : {}};

		oBinding.bFirstCreateAtEnd = bFirstCreateAtEnd;
		oBinding.iCreatedContexts = 2;
		this.mock(oBinding).expects("fetchData")
			.withExactArgs(iReadStart, 2, 3, sinon.match.same(oGroupLock),
				sinon.match.same(fnDataRequested))
			.returns(SyncPromise.resolve(oResult));
		this.mock(oBinding).expects("createContexts")
			.withExactArgs(iReadStart, sinon.match.same(oResult.value))
			.returns(SyncPromise.resolve(bChanged));

		// code under test
		return oBinding.fetchContexts(1, 2, 3, oGroupLock, false, fnDataRequested);
	});
});

	//*********************************************************************************************
	QUnit.test("fetchContexts: fetchData returns undefined", function (assert) {
		var oBinding = this.bindList("TEAM_2_EMPLOYEES",
				Context.create({/*oModel*/}, oParentBinding, "/TEAMS('1')")),
			fnDataRequested = {/*function*/},
			oGroupLock = {},
			oPromise;

		this.mock(oBinding).expects("fetchData")
			.withExactArgs(1, 2, 3, sinon.match.same(oGroupLock),
				sinon.match.same(fnDataRequested))
			.returns(SyncPromise.resolve(Promise.resolve(undefined)));
		this.mock(oBinding).expects("createContexts").never();

		// code under test
		oPromise = oBinding.fetchContexts(1, 2, 3, oGroupLock, false, fnDataRequested);

		return oPromise.then(function (bChanged) {
			assert.notOk(bChanged);
		});
	});

	//*********************************************************************************************
[false, true].forEach(function (bHasGroupLock) {
	QUnit.test("fetchContexts: read failure, groupLock=" + bHasGroupLock, function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			fnDataRequested = {/*function*/},
			oError = new Error(),
			oGroupLock = {unlock : function () {}};

		this.mock(oBinding).expects("lockGroup").exactly(bHasGroupLock ? 0 : 1)
			.withExactArgs().returns(oGroupLock);
		this.mock(oBinding).expects("fetchData")
			.withExactArgs(1, 2, 3, sinon.match.same(oGroupLock),
				sinon.match.same(fnDataRequested))
			.returns(SyncPromise.resolve(Promise.reject(oError)));
		this.mock(oBinding).expects("createContexts").never();
		this.mock(oGroupLock).expects("unlock").withExactArgs(true);

		// code under test
		return oBinding.fetchContexts(1, 2, 3, bHasGroupLock ? oGroupLock : undefined, false,
				fnDataRequested)
			.then(function () {
				assert.ok(false);
			}, function (oResult) {
				assert.strictEqual(oResult, oError);
			});
	});
});

	//*********************************************************************************************
	QUnit.test("fetchContexts: binding already destroyed", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oPromise;

		this.mock(oBinding).expects("fetchData")
			.withExactArgs(1, 2, 3, "~oGroupLock~", "~fnDataRequested~")
			.returns(SyncPromise.resolve({}));
		this.mock(oBinding).expects("createContexts").never();

		// code under test
		oPromise = oBinding.fetchContexts(1, 2, 3, "~oGroupLock~", true, "~fnDataRequested~");

		oBinding.destroy();

		return oPromise.then(function () {
			assert.ok(false, "Unexpected success");
		}, function (oError) {
			assert.strictEqual(oError.message, "Binding already destroyed");
			assert.strictEqual(oError.canceled, true);
		});
	});

	//*********************************************************************************************
[false, true].forEach(function (bChanged) {
	QUnit.test("requestContexts: changed=" + bChanged, function (assert) {
		var oBinding = this.bindList("n/a"),
			aContexts = [],
			oGroupLock = {},
			oPromise;

		this.mock(oBinding).expects("isResolved").withExactArgs().returns(true);
		this.mock(oBinding).expects("checkSuspended").withExactArgs();
		this.mock(_Helper).expects("checkGroupId").withExactArgs("groupId");
		this.mock(oBinding).expects("lockGroup").withExactArgs("groupId", true).returns(oGroupLock);
		this.mock(oBinding).expects("fetchContexts")
			.withExactArgs(1, 2, 0, sinon.match.same(oGroupLock))
			.returns(SyncPromise.resolve(Promise.resolve(bChanged)));
		this.mock(oBinding).expects("_fireChange").exactly(bChanged ? 1 : 0)
			.withExactArgs({reason : ChangeReason.Change});
		this.mock(oBinding).expects("getContextsInViewOrder")
			.withExactArgs(1, 2)
			.returns(aContexts);

		// code under test
		oPromise = oBinding.requestContexts(1, 2, "groupId").then(function (aResults) {
			assert.strictEqual(aResults, aContexts);
		});

		assert.ok(oPromise instanceof Promise);
		return oPromise;
	});
});

	//*********************************************************************************************
	QUnit.test("requestContexts: parameter defaults", function (assert) {
		var oBinding = this.bindList("n/a"),
			aContexts = [],
			oGroupLock = {};

		this.mock(oBinding).expects("isResolved").withExactArgs().returns(true);
		this.mock(oBinding).expects("checkSuspended").withExactArgs();
		this.mock(_Helper).expects("checkGroupId").withExactArgs(undefined);
		this.mock(oBinding).expects("lockGroup").withExactArgs(undefined, true).returns(oGroupLock);
		this.mock(oBinding).expects("fetchContexts")
			.withExactArgs(0, this.oModel.iSizeLimit, 0, sinon.match.same(oGroupLock))
			.returns(SyncPromise.resolve(Promise.resolve(false)));
		this.mock(oBinding).expects("_fireChange").never();
		this.mock(oBinding).expects("getContextsInViewOrder")
			.withExactArgs(0, this.oModel.iSizeLimit)
			.returns(aContexts);

		// code under test
		return oBinding.requestContexts().then(function (aResults) {
			assert.strictEqual(aResults, aContexts);
		});
	});

	//*********************************************************************************************
	QUnit.test("requestContexts: error in fetchContexts", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oError = new Error(),
			oGroupLock = {};

		this.mock(oBinding).expects("isResolved").withExactArgs().returns(true);
		this.mock(oBinding).expects("checkSuspended").withExactArgs();
		this.mock(_Helper).expects("checkGroupId").withExactArgs(undefined);
		this.mock(oBinding).expects("lockGroup").withExactArgs(undefined, true).returns(oGroupLock);
		this.mock(oBinding).expects("fetchContexts")
			.withExactArgs(1, 2, 0, sinon.match.same(oGroupLock))
			.returns(SyncPromise.resolve(Promise.reject(oError)));
		this.mock(oBinding).expects("_fireChange").never();
		this.mock(oBinding).expects("getContextsInViewOrder").never();
		this.mock(this.oModel).expects("reportError").withExactArgs(
			"Failed to get contexts for /service/EMPLOYEES with start index 1 and length 2",
			sClassName, sinon.match.same(oError));

		// code under test
		return oBinding.requestContexts(1, 2).then(function () {
			assert.ok(false);
		}, function (oResult) {
			assert.strictEqual(oResult, oError);
		});
	});

	//*********************************************************************************************
	QUnit.test("requestContexts: unresolved", function (assert) {
		var oBinding = this.bindList("unresolved");

		this.mock(oBinding).expects("isResolved").withExactArgs().returns(false);
		this.mock(oBinding).expects("fetchContexts").never();

		assert.throws(function () {
			oBinding.requestContexts();
		}, new Error("Unresolved binding: unresolved"));
	});

	//*********************************************************************************************
	QUnit.test("requestContexts: suspended", function (assert) {
		var oBinding = this.bindList("n/a"),
			oError = new Error();

		this.mock(oBinding).expects("isResolved").withExactArgs().returns(true);
		this.mock(oBinding).expects("checkSuspended").withExactArgs().throws(oError);
		this.mock(oBinding).expects("fetchContexts").never();

		assert.throws(function () {
			oBinding.requestContexts();
		}, oError);
	});

	//*********************************************************************************************
	QUnit.test("requestContexts: invalid group ID", function (assert) {
		var oBinding = this.bindList("n/a"),
			oError = new Error();

		this.mock(oBinding).expects("isResolved").withExactArgs().returns(true);
		this.mock(oBinding).expects("checkSuspended").withExactArgs();
		this.mock(_Helper).expects("checkGroupId").withExactArgs("$invalid").throws(oError);
		this.mock(oBinding).expects("fetchContexts").never();

		assert.throws(function () {
			oBinding.requestContexts(0, 10, "$invalid");
		}, oError);
	});

	//*********************************************************************************************
[false, true].forEach(function (bAsync) {
	[false, true].forEach(function (bChanged) {
		[undefined, true].forEach(function (bKeepCurrent) {
			var sTitle = "getContexts: async=" + bAsync + ", changed=" + bChanged
					+ ", bKeepCurrent=" + bKeepCurrent;

	QUnit.test(sTitle, function (assert) {
		var oBinding = this.bindList("n/a"),
			aContexts = [],
			oFetchContextsPromise = bAsync
				? SyncPromise.resolve(Promise.resolve(bChanged))
				: SyncPromise.resolve(bChanged),
			iMaximumPrefetchSize = bKeepCurrent ? 0 : 100,
			aResults;

		oBinding.oReadGroupLock = undefined;
		oBinding.iCurrentBegin = 2;
		oBinding.iCurrentEnd = 7;
		this.oLogMock.expects("debug")
			.withExactArgs(oBinding + "#getContexts(5, 10, " + iMaximumPrefetchSize + ")",
				undefined, sClassName);
		this.mock(oBinding).expects("checkSuspended").withExactArgs();
		this.mock(oBinding).expects("isResolved").withExactArgs().returns(true);
		this.mock(oBinding).expects("getDiff").never();
		this.mock(oBinding).expects("fetchContexts")
			.withExactArgs(5, 10, iMaximumPrefetchSize, undefined, false, sinon.match.func)
			.returns(oFetchContextsPromise);
		this.mock(oBinding).expects("resolveRefreshPromise")
			.withExactArgs(sinon.match.same(oFetchContextsPromise))
			.returns(oFetchContextsPromise);
		this.mock(oBinding).expects("getContextsInViewOrder")
			.withExactArgs(5, 10)
			.returns(aContexts);
		this.mock(oBinding).expects("_fireChange")
			.exactly(bAsync && bChanged ? 1 : 0)
			.withExactArgs({reason : ChangeReason.Change});

		// code under test
		aResults = oBinding.getContexts(5, 10, iMaximumPrefetchSize, bKeepCurrent);

		assert.strictEqual(aResults, aContexts);
		assert.strictEqual(oBinding.iCurrentBegin, bKeepCurrent ? 2 : 5);
		assert.strictEqual(oBinding.iCurrentEnd, bKeepCurrent ? 7 : 15);

		return oFetchContextsPromise;
	});
		});
	});
});

	//*********************************************************************************************
	QUnit.test("getContexts: unresolved", function (assert) {
		var oBinding = this.bindList("n/a"),
			aContexts;

		oBinding.aPreviousData = [{}];
		oBinding.bUseExtendedChangeDetection = true; // BCP: 2180095696
		this.mock(oBinding).expects("checkSuspended").withExactArgs();
		this.mock(oBinding).expects("isResolved").withExactArgs().returns(false);
		this.mock(oBinding).expects("fetchContexts").never();

		// code under test
		aContexts = oBinding.getContexts();

		assert.deepEqual(aContexts, []);
		assert.deepEqual(oBinding.aPreviousData, []);
	});

	//*********************************************************************************************
	QUnit.test("getContexts: dataRequested/dataReceived", function () {
		var oBinding = this.bindList("/EMPLOYEES"),
			oBindingMock = this.mock(oBinding),
			oFetchContextsCall,
			oFetchContextsPromise = SyncPromise.resolve(Promise.resolve()).then(function () {
				// expect this when fetchContexts is finished
				oBindingMock.expects("fireDataReceived")
					.withExactArgs({data : {}}, "~bPreventBubbling~");

				return false;
			});

		oFetchContextsCall = oBindingMock.expects("fetchContexts")
			.withExactArgs(0, 10, 100, sinon.match.object, false, sinon.match.func)
			.returns(oFetchContextsPromise);
		oBindingMock.expects("fireDataRequested").never(); // expect it later
		oBindingMock.expects("fireDataReceived").never(); // expect it later
		oBindingMock.expects("isRefreshWithoutBubbling").withExactArgs()
			.returns("~bPreventBubbling~");

		// code under test
		oBinding.getContexts(0, 10, 100);

		oBindingMock.expects("fireDataRequested").withExactArgs("~bPreventBubbling~");

		// code under test
		oFetchContextsCall.args[0][5]();

		return oFetchContextsPromise;
	});

	//*********************************************************************************************
	QUnit.test("getContexts: default values", function () {
		var oBinding = this.bindList("n/a"),
			oBindingMock = this.mock(oBinding);

		oBindingMock.expects("checkSuspended").withExactArgs();
		oBindingMock.expects("isResolved").withExactArgs().returns(true);
		oBindingMock.expects("fetchContexts")
			.withExactArgs(0, this.oModel.iSizeLimit, 0, undefined, false, sinon.match.func)
			.returns(SyncPromise.resolve());

		// code under test
		oBinding.getContexts();

		oBindingMock.expects("checkSuspended").withExactArgs();
		oBindingMock.expects("isResolved").withExactArgs().returns(true);
		oBindingMock.expects("fetchContexts")
			.withExactArgs(1, 2, 0, undefined, false, sinon.match.func)
			.returns(SyncPromise.resolve());

		// code under test
		oBinding.getContexts(1, 2, -42);
	});

	//*********************************************************************************************
	QUnit.test("getContexts: after refresh", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oFetchContextsPromise = SyncPromise.resolve(Promise.resolve(true)),
			sChangeReason = {/*ChangeReason*/};

		oBinding.sChangeReason = sChangeReason;
		this.mock(oBinding).expects("fetchContexts")
			.withExactArgs(0, 10, 100, sinon.match.object, /*bAsync=*/true, sinon.match.func)
			.returns(oFetchContextsPromise);
		this.mock(oBinding).expects("_fireChange")
			.withExactArgs({reason : sChangeReason});

		// code under test
		oBinding.getContexts(0, 10, 100);

		assert.strictEqual(oBinding.sChangeReason, undefined);

		return oFetchContextsPromise;
	});

	//*********************************************************************************************
	QUnit.test("getContexts: read group lock", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oFetchContextsPromise = SyncPromise.resolve(Promise.resolve(false)),
			oReadGroupLock = {/*GroupLock*/};

		oBinding.oReadGroupLock = oReadGroupLock;
		this.mock(oBinding).expects("fetchContexts")
			.withExactArgs(0, 10, 100, sinon.match.same(oReadGroupLock), false, sinon.match.func)
			.returns(oFetchContextsPromise);

		// code under test
		oBinding.getContexts(0, 10, 100);

		assert.strictEqual(oBinding.oReadGroupLock, undefined);

		return oFetchContextsPromise;
	});

	//*********************************************************************************************
[false, /*see strictEqual below*/"true"].forEach(function (bUseExtendedChangeDetection) {
	[/*destroyed early*/undefined, false, /*destroyed late*/0, true].forEach(function (bSuspend) {
		var sTitle = "getContexts: AddVirtualContext, suspend:" + bSuspend
				+ ", use extended change detection:" + bUseExtendedChangeDetection;

	QUnit.test(sTitle, function (assert) {
		var oContext = Context.create({/*oModel*/}, oParentBinding, "/TEAMS('1')"),
			oBinding = this.bindList("TEAM_2_EMPLOYEES", oContext),
			oBindingMock = this.mock(oBinding),
			aContexts,
			oModelMock = this.mock(this.oModel),
			oAddTask0,
			oAddTask1,
			oVirtualContext = {destroy : function () {}};

		oBinding.bUseExtendedChangeDetection = bUseExtendedChangeDetection;
		oBinding.sChangeReason = "AddVirtualContext";
		oBindingMock.expects("checkSuspended").withExactArgs();
		oBindingMock.expects("isResolved").withExactArgs().returns(true);
		oAddTask0 = oModelMock.expects("addPrerenderingTask").withExactArgs(sinon.match.func, true);
		this.mock(oBinding.oModel).expects("resolve")
			.withExactArgs(oBinding.sPath, sinon.match.same(oContext)).returns("/~");
		this.mock(Context).expects("create")
			.withExactArgs(sinon.match.same(oBinding.oModel), sinon.match.same(oBinding),
				"/~/" + Context.VIRTUAL, Context.VIRTUAL)
			.returns(oVirtualContext);
		oBindingMock.expects("fetchContexts").never();
		oBindingMock.expects("_fireChange").never();
		if (bSuspend !== false) {
			oBindingMock.expects("reset").never();
		}

		// code under test
		aContexts = oBinding.getContexts(0, 10, bUseExtendedChangeDetection ? undefined : 100);

		assert.strictEqual(oBinding.sChangeReason, undefined);
		assert.strictEqual(aContexts.length, 1);
		assert.strictEqual(aContexts[0], oVirtualContext);

		// prerendering task
		if (bSuspend === undefined) { // destroy early
			oBinding.destroy();
			this.mock(oVirtualContext).expects("destroy").withExactArgs();
		} else {
			oBindingMock.expects("isRootBindingSuspended").withExactArgs().returns(bSuspend);
			if (!bSuspend) {
				oBindingMock.expects("getContexts").on(oBinding)
					.withExactArgs(0, 10, bUseExtendedChangeDetection ? undefined : 100)
					.callsFake(function () {
						assert.strictEqual(this.bUseExtendedChangeDetection, false);
					});
			}
			oAddTask1 = oModelMock.expects("addPrerenderingTask").withExactArgs(sinon.match.func);
		}

		// code under test - call the 1st prerendering task
		oAddTask0.args[0][0]();

		assert.strictEqual(oBinding.bUseExtendedChangeDetection, bUseExtendedChangeDetection);

		if (oAddTask1) {
			if (bSuspend === 0) { // destroy late
				oBinding.destroy();
			} else {
				oBindingMock.expects("isRootBindingSuspended").withExactArgs().returns(bSuspend);
				if (!bSuspend) {
					oBindingMock.expects("_fireChange").withExactArgs({
							detailedReason : "RemoveVirtualContext",
							reason : ChangeReason.Change
						}).callsFake(function () {
							assert.strictEqual(oBinding.sChangeReason, "RemoveVirtualContext");
						});
					oBindingMock.expects("reset").withExactArgs(ChangeReason.Refresh);
				}
			}
			this.mock(oVirtualContext).expects("destroy").withExactArgs();

			// code under test - call the 2nd prerendering task
			oAddTask1.args[0][0]();
		}
	});
	});
});

	//*********************************************************************************************
	// Note: This happens for a list binding below another list binding during autoExpandSelect
	QUnit.test("getContexts: below a virtual context", function (assert) {
		var oContext = Context.create({/*oModel*/}, oParentBinding,
				"/TEAMS('1')/" + Context.VIRTUAL, Context.VIRTUAL),
			oBinding = this.bindList("TEAM_2_EMPLOYEES", oContext);

		this.mock(oBinding).expects("checkSuspended").withExactArgs();
		this.mock(oBinding).expects("isResolved").withExactArgs().returns(true);
		this.mock(oBinding).expects("fetchContexts").never();
		this.mock(oBinding).expects("_fireChange").never();

		// code under test
		assert.deepEqual(oBinding.getContexts(0, 10, 100), []);
	});

	//*********************************************************************************************
	QUnit.test("getContexts: RemoveVirtualContext", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oBindingMock = this.mock(oBinding),
			aContexts;

		oBinding.sChangeReason = "RemoveVirtualContext";
		oBindingMock.expects("checkSuspended").withExactArgs();
		oBindingMock.expects("isResolved").withExactArgs().returns(true);
		oBindingMock.expects("fetchContexts").never();
		oBindingMock.expects("_fireChange").never();

		// code under test
		aContexts = oBinding.getContexts(0, 10, 100);

		assert.strictEqual(oBinding.sChangeReason, undefined);
		assert.deepEqual(aContexts, []);
	});

	//*********************************************************************************************
[
	{bCanceled : true, bDataRequested : true},
	{bCanceled : false, bDataRequested : false},
	{bCanceled : false, bDataRequested : true},
	{bCanceled : false, bDataRequested : true, bDestroyed : true}
].forEach(function (oFixture) {
	var sTitle = "getContexts: error in fetchContexts, " + JSON.stringify(oFixture);

	QUnit.test(sTitle, function () {
		var oContext = Context.create({/*oModel*/}, oParentBinding, "/TEAMS('1')"),
			oBinding = this.bindList("TEAM_2_EMPLOYEES", oContext),
			oError = {canceled : oFixture.bCanceled},
			oFetchContextsCall,
			oFetchContextsPromise = SyncPromise.resolve(Promise.resolve().then(function () {
				if (oFixture.bDestroyed) {
					oBinding.destroy();
				}
				throw oError;
			}));

		this.mock(oBinding).expects("checkSuspended").withExactArgs();
		this.mock(oBinding).expects("isResolved").withExactArgs().returns(true);
		this.mock(oBinding).expects("getResolvedPath").withExactArgs().returns("/~");
		this.mock(oBinding).expects("isRefreshWithoutBubbling").withExactArgs()
			.returns("~bPreventBubbling~");
		oFetchContextsCall = this.mock(oBinding).expects("fetchContexts")
			.withExactArgs(0, 10, 100, undefined, false, sinon.match.func)
			.returns(oFetchContextsPromise);
		this.mock(oBinding.oModel).expects("reportError")
			.withExactArgs("Failed to get contexts for /service/~ with start index 0 and length 10",
				sClassName, sinon.match.same(oError));

		// code under test
		oBinding.getContexts(0, 10, 100);

		this.mock(oBinding).expects("fireDataReceived").exactly(oFixture.bDataRequested ? 1 : 0)
			.withExactArgs(oFixture.bCanceled ? {data : {}} : {error : sinon.match.same(oError)},
				"~bPreventBubbling~");

		// code under test - dataRequested/dataReceived
		if (oFixture.bDataRequested) {
			oFetchContextsCall.args[0][5]();
		}

		return oFetchContextsPromise.catch(function () { /* avoid "Uncaught (in promise)"*/ });
	});
});

	//*********************************************************************************************
	QUnit.test("getContexts: error in dataRequested", function () {
		var oContext = Context.create({/*oModel*/}, oParentBinding, "/TEAMS('1')"),
			oBinding = this.bindList("TEAM_2_EMPLOYEES", oContext),
			oBindingMock = this.mock(oBinding),
			oError = new Error(),
			oFetchContextsCall,
			oFetchContextsPromise = SyncPromise.resolve(Promise.resolve()).then(function () {
				// call fnDataRequested within the promise
				oFetchContextsCall.args[0][5]();

				return false;
			});

		oBindingMock.expects("checkSuspended").withExactArgs();
		oBindingMock.expects("isResolved").withExactArgs().returns(true);
		oFetchContextsCall = oBindingMock.expects("fetchContexts")
			.withExactArgs(0, 10, 100, undefined, false, sinon.match.func)
			.returns(oFetchContextsPromise);
		this.mock(oBinding.oModel).expects("resolve")
			.withExactArgs(oBinding.sPath, sinon.match.same(oContext)).returns("/~");
		oBindingMock.expects("fireDataRequested").withExactArgs(null).throws(oError);
		this.mock(oBinding.oModel).expects("reportError")
			.withExactArgs("Failed to get contexts for /service/~ with start index 0 and length 10",
				sClassName, sinon.match.same(oError));

		// code under test
		oBinding.getContexts(0, 10, 100);

		return oFetchContextsPromise.catch(function () { /* avoid "Uncaught (in promise)"*/ });
	});

	//*********************************************************************************************
	QUnit.test("getContexts: error in dataReceived", function () {
		var oContext = Context.create({/*oModel*/}, oParentBinding, "/TEAMS('1')"),
			oBinding = this.bindList("TEAM_2_EMPLOYEES", oContext),
			oBindingMock = this.mock(oBinding),
			oError = new Error(),
			oFetchContextsCall,
			oFetchContextsPromise = SyncPromise.resolve(Promise.resolve(false));

		oBindingMock.expects("checkSuspended").withExactArgs();
		oBindingMock.expects("isResolved").withExactArgs().returns(true);
		oFetchContextsCall = oBindingMock.expects("fetchContexts")
			.withExactArgs(0, 10, 100, undefined, false, sinon.match.func)
			.returns(oFetchContextsPromise);
		this.mock(oBinding.oModel).expects("resolve")
			.withExactArgs(oBinding.sPath, sinon.match.same(oContext)).returns("/~");
		oBindingMock.expects("fireDataReceived").withExactArgs({data : {}}, null)
			.throws(oError);
		this.mock(oBinding.oModel).expects("reportError")
			.withExactArgs("Failed to get contexts for /service/~ with start index 0 and length 10",
				sClassName, sinon.match.same(oError));

		// code under test
		oBinding.getContexts(0, 10, 100);

		// code under test - dataRequested/dataReceived
		oFetchContextsCall.args[0][5]();

		return oFetchContextsPromise;
	});

	//*********************************************************************************************
[
	{bChanged : true, aDiff : [{}]},
	{bChanged : false, aDiff : [{}]},
	{bChanged : false, aDiff : []}
].forEach(function (oFixture) {
	QUnit.test("getContexts: E.C.D, no diff yet, " + JSON.stringify(oFixture), function (assert) {
		var oBinding = this.bindList("TEAM_2_EMPLOYEES",
				Context.create({/*oModel*/}, oParentBinding, "/TEAMS('1')")),
			oBindingMock = this.mock(oBinding),
			sChangeReason = {/*string*/},
			aContexts,
			oFetchContextsPromise = SyncPromise.resolve(Promise.resolve()).then(function () {
				oBindingMock.expects("getDiff").withExactArgs(10).returns(oFixture.aDiff);
				return oFixture.bChanged;
			});

		oBinding.enableExtendedChangeDetection();
		oBinding.sChangeReason = sChangeReason;

		oBindingMock.expects("getDiff").never();
		oBindingMock.expects("checkSuspended").withExactArgs();
		oBindingMock.expects("isResolved").withExactArgs().returns(true);
		oBindingMock.expects("fetchContexts")
			.withExactArgs(0, 10, 0, undefined, true, sinon.match.func)
			.returns(oFetchContextsPromise);
		this.mock(oBinding).expects("_fireChange")
			.exactly(oFixture.bChanged || oFixture.aDiff.length ? 1 : 0)
			.withExactArgs({reason : sChangeReason});

		// code under test
		aContexts = oBinding.getContexts(0, 10);

		assert.strictEqual(aContexts.dataRequested, true);

		return oFetchContextsPromise.then(function () {
			if (oFixture.bChanged || oFixture.aDiff.length) {
				assert.deepEqual(oBinding.oDiff, {
					aDiff : oFixture.aDiff,
					iLength : 10
				});
				assert.strictEqual(oBinding.oDiff.aDiff, oFixture.aDiff);
			} else {
				assert.strictEqual(oBinding.oDiff, undefined);
			}
		});
	});
});

	//*********************************************************************************************
	QUnit.test("getContexts: E.C.D, with diff", function (assert) {
		var oBinding = this.bindList("TEAM_2_EMPLOYEES",
				Context.create({/*oModel*/}, oParentBinding, "/TEAMS('1')")),
			aContexts = [],
			aDiff = [],
			aResults;

		oBinding.enableExtendedChangeDetection();
		oBinding.oDiff = {
			aDiff : aDiff,
			iLength : 10
		};

		this.mock(oBinding).expects("checkSuspended").withExactArgs();
		this.mock(oBinding).expects("isResolved").withExactArgs().returns(true);
		this.mock(oBinding).expects("getDiff").never();
		this.mock(oBinding).expects("fetchContexts").never();
		this.mock(oBinding).expects("getContextsInViewOrder").withExactArgs(0, 10)
			.returns(aContexts);

		// code under test
		aResults = oBinding.getContexts(0, 10);

		assert.strictEqual(aResults, aContexts);
		assert.strictEqual(aContexts.dataRequested, false);
		assert.strictEqual(aContexts.diff, aDiff);
		assert.strictEqual(oBinding.oDiff, undefined);
	});

	//*********************************************************************************************
	QUnit.test("getContexts: E.C.D, with diff, length mismatch", function (assert) {
		var oBinding = this.bindList("TEAM_2_EMPLOYEES",
				Context.create({/*oModel*/}, oParentBinding, "/TEAMS('1')"));

		oBinding.enableExtendedChangeDetection();
		oBinding.oDiff = {
			aDiff : [],
			iLength : 10
		};

		this.mock(oBinding).expects("checkSuspended").withExactArgs();
		this.mock(oBinding).expects("isResolved").withExactArgs().returns(true);
		this.mock(oBinding).expects("getDiff").never();
		this.mock(oBinding).expects("fetchContexts").never();
		this.mock(oBinding).expects("getContextsInViewOrder").withExactArgs(0, 20);

		// code under test
		assert.throws(function () {
			oBinding.getContexts(0, 20);
		}, new Error("Extended change detection protocol violation: Expected "
			+ "getContexts(0,10), but got getContexts(0,20)"));
	});

	//*********************************************************************************************
	QUnit.test("getContextsInViewOrder: create at start", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			aContexts = [],
			aResults;

		this.mock(oBinding.aContexts).expects("slice").withExactArgs(2, 5).returns(aContexts);

		// code under test
		aResults = oBinding.getContextsInViewOrder(2, 3);

		assert.strictEqual(aResults, aContexts);
	});

	//*********************************************************************************************
	QUnit.test("getContextsInViewOrder: create at end", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oBindingMock = this.mock(oBinding),
			aResults;

		// assuming 3 created entities (with index 0, 1 and 2)
		// view order 3 4 5 2 1 0
		oBindingMock.expects("getLength").withExactArgs().returns(6);
		oBindingMock.expects("getModelIndex").withExactArgs(3).returns(0);
		oBindingMock.expects("getModelIndex").withExactArgs(4).returns(1);
		oBindingMock.expects("getModelIndex").withExactArgs(5).returns(2);

		oBinding.bFirstCreateAtEnd = true;
		oBinding.aContexts = [{}, {}, {}, {}, {}, {}];

		// code under test
		aResults = oBinding.getContextsInViewOrder(3, 10);

		assert.strictEqual(aResults.length, 3);
		assert.strictEqual(aResults[0], oBinding.aContexts[0]);
		assert.strictEqual(aResults[1], oBinding.aContexts[1]);
		assert.strictEqual(aResults[2], oBinding.aContexts[2]);

		oBindingMock.expects("getLength").withExactArgs().returns(6);
		oBindingMock.expects("getModelIndex").withExactArgs(1).returns(4);
		oBindingMock.expects("getModelIndex").withExactArgs(2).returns(3);

		// code under test
		aResults = oBinding.getContextsInViewOrder(1, 2);

		assert.strictEqual(aResults.length, 2);
		assert.strictEqual(aResults[0], oBinding.aContexts[4]);
		assert.strictEqual(aResults[1], oBinding.aContexts[3]);
	});

	//*********************************************************************************************
	QUnit.test("getLength", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		assert.strictEqual(oBinding.getLength(), 0);

		oBinding.aContexts = [{}, {}, {}, {}];
		assert.strictEqual(oBinding.getLength(), 14);

		oBinding.bLengthFinal = true;
		oBinding.iMaxLength = 20;
		assert.strictEqual(oBinding.getLength(), 20);

		oBinding.iCreatedContexts = 2;
		assert.strictEqual(oBinding.getLength(), 22);
	});

	//*********************************************************************************************
	["/", "foo/"].forEach(function (sPath) {
		QUnit.test("bindList: invalid path: " + sPath, function (assert) {
			assert.throws(function () {
				this.bindList(sPath);
			}, new Error("Invalid path: " + sPath));
		});
	});

	//*********************************************************************************************
	QUnit.test("bindList: empty path is valid for base context", function () {
		var oBaseContext = this.oModel.createBindingContext("/BusinessPartnerList");

		// code under test
		this.bindList("", oBaseContext);
	});

	//*********************************************************************************************
	QUnit.test("reset context for nested list binding with its own cache", function (assert) {
		var oBinding,
			oBindingMock = this.mock(ODataListBinding.prototype),
			oCache = {},
			oContext = Context.create(this.oModel, oParentBinding, "/TEAMS", 1);

		oBindingMock.expects("checkSuspended").withExactArgs(true);
		// fetchCache is called once from applyParameters before oBinding.oContext is set
		oBindingMock.expects("fetchCache").withExactArgs(undefined).callsFake(function () {
			this.oCache = null;
			this.oCachePromise = SyncPromise.resolve(null);
		});
		oBindingMock.expects("fetchCache").withExactArgs(sinon.match.same(oContext)).atLeast(1)
			.callsFake(function () {
				this.oCache = oCache;
				this.oCachePromise = SyncPromise.resolve(oCache);
			});
		oBindingMock.expects("restoreCreated").withExactArgs();
		oBinding = this.bindList("TEAM_2_EMPLOYEES", undefined, undefined, undefined,
			{$select : "ID"});

		// code under test
		oBinding.setContext(oContext);

		assert.strictEqual(oBinding.oCachePromise.getResult(), oCache);
	});

	//*********************************************************************************************
[ // The first test requests the virtual context, all others don't
	{aggregation : false, autoExpandSelect : true, backLink : true, newContext : true},
	{aggregation : false, autoExpandSelect : false, backLink : true, newContext : true},
	{aggregation : false, autoExpandSelect : true, backLink : false, newContext : true},
	{aggregation : false, autoExpandSelect : true, backLink : true, newContext : false},
	{aggregation : true, autoExpandSelect : true, backLink : true, newContext : true}
].forEach(function (oFixture, i) {
	QUnit.test("setContext: relative path, " + JSON.stringify(oFixture), function (assert) {
		var oBinding = this.bindList("Suppliers", Context.create(this.oModel, oParentBinding,
				"/foo")),
			oBindingMock = this.mock(oBinding),
			oBindingSetContextCall,
			oContext = oFixture.newContext
				? Context.create(this.oModel, oParentBinding, "/bar")
				: undefined,
			sExpectedChangeReason = i === 0 ? "AddVirtualContext" : "sChangeReason",
			oFetchCacheCall,
			oNewHeaderContext = Context.create(this.oModel, oBinding, "/bar/Suppliers"),
			oOldHeaderContext = oBinding.getHeaderContext(),
			oResetKeepAliveCall,
			oRestoreCreatedCall;

		this.oModel.bAutoExpandSelect = oFixture.autoExpandSelect;
		if (oFixture.aggregation) {
			oBinding.mParameters.$$aggregation = {};
		}
		oBinding.sChangeReason = "sChangeReason";
		oBinding.bHasPathReductionToParent = oFixture.backLink;

		// code under test - nothing must happen
		oBinding.setContext(oBinding.oContext);

		assert.strictEqual(oBinding.sChangeReason, "sChangeReason");
		assert.deepEqual(oBinding.mPreviousContextsByPath, {});

		oBindingMock.expects("checkSuspended").withExactArgs(true);
		oBindingMock.expects("reset").withExactArgs(undefined, true);
		oResetKeepAliveCall = oBindingMock.expects("resetKeepAlive").withExactArgs();
		oFetchCacheCall = oBindingMock.expects("fetchCache")
			.withExactArgs(sinon.match.same(oContext));
		oRestoreCreatedCall = oBindingMock.expects("restoreCreated").withExactArgs()
			.exactly(oFixture.newContext ? 1 : 0);
		this.mock(this.oModel).expects("resolve").exactly(oFixture.newContext ? 1 : 0)
			.withExactArgs(oBinding.sPath, sinon.match.same(oContext))
			.returns("/bar/Suppliers");
		this.mock(Context).expects("create").exactly(oFixture.newContext ? 1 : 0)
			.withExactArgs(sinon.match.same(this.oModel), sinon.match.same(oBinding),
				"/bar/Suppliers")
			.returns(oNewHeaderContext);
		oBindingSetContextCall = this.mock(Binding.prototype).expects("setContext").on(oBinding)
			.withExactArgs(sinon.match.same(oContext), {detailedReason : sExpectedChangeReason});

		// code under test
		oBinding.setContext(oContext);

		assert.ok(oFetchCacheCall.calledAfter(oResetKeepAliveCall));
		assert.strictEqual(oBinding.sChangeReason, sExpectedChangeReason);
		if (oFixture.newContext) {
			assert.deepEqual(oBinding.mPreviousContextsByPath, {
				"/foo/Suppliers" : oOldHeaderContext
			});
			assert.ok(oRestoreCreatedCall.calledAfter(oFetchCacheCall));
			assert.ok(oRestoreCreatedCall.calledBefore(oBindingSetContextCall));
		} else {
			assert.deepEqual(oBinding.mPreviousContextsByPath, {});
		}

		// mock needed because Binding.prototype.setContext is mocked!
		oBindingMock.expects("isResolved").withExactArgs().returns(true);
		assert.strictEqual(oBinding.getHeaderContext(),
			oFixture.newContext ? oNewHeaderContext : oOldHeaderContext);
	});
});

	//*********************************************************************************************
	QUnit.test("setContext: implicit suspend", function (assert) {
		var oBinding = this.bindList("Suppliers"),
			oContext = {
				getBinding : function () {}
			},
			oParentBinding = {
				getRootBinding : function () {}
			},
			oRootBinding = {
				isSuspended : function () {}
			};

		this.mock(oBinding).expects("checkSuspended").withExactArgs(true);
		this.mock(oBinding).expects("reset").withExactArgs(undefined, true);
		this.mock(oBinding).expects("resetKeepAlive").withExactArgs();
		this.mock(oBinding).expects("fetchCache").withExactArgs(sinon.match.same(oContext));
		this.mock(oBinding).expects("restoreCreated").withExactArgs();
		this.mock(this.oModel).expects("resolve")
			.withExactArgs(oBinding.sPath, sinon.match.same(oContext))
			.returns("/resolved/path");
		this.mock(Context).expects("create")
			.withExactArgs(sinon.match.same(this.oModel), sinon.match.same(oBinding),
				"/resolved/path")
			.returns("~headerContext~");
		this.mock(oContext).expects("getBinding").withExactArgs().returns(oParentBinding);
		this.mock(oParentBinding).expects("getRootBinding").withExactArgs().returns(oRootBinding);
		this.mock(oRootBinding).expects("isSuspended").withExactArgs().returns(true);
		this.mock(Binding.prototype).expects("setContext").never();
		this.mock(oBinding).expects("setResumeChangeReason").withExactArgs(ChangeReason.Context);

		// code under test
		oBinding.setContext(oContext);

		assert.strictEqual(oBinding.oContext, oContext);
	});

	//*********************************************************************************************
	QUnit.test("preserve headerContext when ManagedObject temporarily removes context",
		function (assert) {
		var oBinding = this.bindList("Suppliers"),
			oBindingMock = this.mock(oBinding),
			oContext = Context.create(this.oModel, oParentBinding, "/bar"),
			oHeaderContext = Context.create(this.oModel, oBinding, "/bar/Suppliers");

		oBindingMock.expects("checkSuspended").withExactArgs(true).thrice();
		this.mock(Context).expects("create")
			.withExactArgs(sinon.match.same(this.oModel), sinon.match.same(oBinding),
				"/bar/Suppliers")
			.returns(oHeaderContext);
		oBindingMock.expects("fetchCache").withExactArgs(null);
		oBindingMock.expects("fetchCache").twice().withExactArgs(sinon.match.same(oContext));
		oBinding.setContext(oContext);
		assert.strictEqual(oBinding.getHeaderContext(), oHeaderContext);
		this.mock(oBinding.getHeaderContext()).expects("destroy").never();

		// code under test
		oBinding.setContext(null);
		oBinding.setContext(oContext);

		assert.strictEqual(oBinding.getHeaderContext(), oHeaderContext);
	});

	//*********************************************************************************************
	QUnit.test("getCurrentContexts: iCurrentEnd limits", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		oBinding.iCreatedContexts = 1;
		oBinding.iCurrentBegin = 1;
		oBinding.iCurrentEnd = 2;
		oBinding.iMaxLength = 3;

		this.mock(oBinding).expects("getContextsInViewOrder").withExactArgs(1, 1)
			.returns(["~oContext~"]);

		// code under test
		assert.deepEqual(oBinding.getCurrentContexts(), ["~oContext~"]);
	});

	//*********************************************************************************************
	QUnit.test("getCurrentContexts: iMaxLength + iCreatedContexts limits", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		oBinding.iCreatedContexts = 2;
		oBinding.iCurrentBegin = 1;
		oBinding.iCurrentEnd = 7;
		oBinding.iMaxLength = 4;

		this.mock(oBinding).expects("getContextsInViewOrder").withExactArgs(1, 5)
			.returns(["~oContext~"]);

		// code under test
		assert.deepEqual(oBinding.getCurrentContexts(), ["~oContext~", undefined, undefined,
			undefined, undefined]);
	});

	//*********************************************************************************************
	QUnit.test("getCurrentContexts: special case Infinity", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		oBinding.iCreatedContexts = 1;
		oBinding.iCurrentBegin = 0;
		oBinding.iCurrentEnd = Infinity;
		oBinding.iMaxLength = Infinity;

		this.mock(oBinding).expects("getContextsInViewOrder").withExactArgs(0, Infinity)
			.returns(["~oContext~"]);

		// code under test (BCP: 2280015704)
		assert.deepEqual(oBinding.getCurrentContexts(), ["~oContext~"]);
	});

	//*********************************************************************************************
	QUnit.test("refreshInternal: relative binding with base context", function (assert) {
		var oBinding = this.bindList("TEAMS", this.oModel.createBindingContext("/"), undefined,
				undefined, {$$groupId : "group"});

		assert.strictEqual(oBinding.iCurrentEnd, 0);
		this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs().returns(false);
		this.mock(oBinding).expects("createReadGroupLock").withExactArgs("myGroup", true);
		this.mock(oBinding).expects("removeCachesAndMessages").withExactArgs("");
		this.mock(oBinding).expects("createRefreshPromise").never(); // iCurrentEnd === 0
		this.mock(oBinding).expects("reset")
			.withExactArgs(ChangeReason.Refresh, undefined, "myGroup");

		// code under test
		return oBinding.refreshInternal("", "myGroup");
	});

	//*********************************************************************************************
[
	{success : true},
	{success : true, refreshKeptElementsFails : true},
	{success : false}
].forEach(function (oFixture) {
	var sTitle = "refreshInternal: relative with own cache, success=" + oFixture.success
			+ ", refreshKeptElements fails = " + oFixture.refreshKeptElementsFails;

	QUnit.test(sTitle, function (assert) {
		var oBinding,
			oBindingMock = this.mock(ODataListBinding.prototype),
			oContext = Context.create(this.oModel, oParentBinding, "/TEAMS('1')"),
			oError = new Error(),
			oHeaderContextCheckUpdatePromise = SyncPromise.resolve(Promise.resolve({})),
			sPath = {/*TEAMS('1')*/},
			oRefreshKeptElementsPromise = oFixture.refreshKeptElementsFails
				? SyncPromise.reject(oError)
				: SyncPromise.resolve(),
			oRefreshResult;

		// fetchCache is called once from applyParameters before oBinding.oContext is set
		oBindingMock.expects("fetchCache").withExactArgs(undefined).callsFake(function () {
			this.oCache = null;
			this.oCachePromise = SyncPromise.resolve(null);
		});
		oBindingMock.expects("fetchCache").withExactArgs(sinon.match.same(oContext))
			.callsFake(function () {
				this.oCache = {
					getResourcePath : function () {
						return "TEAMS('1')/TEAM_2_EMPLOYEES";
					},
					// no #restore here, e.g. _AggregationCache
					setActive : function () {}
				};
				this.oCachePromise = SyncPromise.resolve(this.oCache);
			});
		oBindingMock.expects("restoreCreated").withExactArgs();
		oBinding = this.bindList("TEAM_2_EMPLOYEES", oContext, undefined, undefined,
			{$$groupId : "group"});
		oBindingMock.verify();
		oBinding.iCurrentEnd = 1;

		this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs().returns(false);
		this.mock(oBinding).expects("createReadGroupLock").withExactArgs("myGroup", false);
		this.mock(oBinding).expects("removeCachesAndMessages")
			.withExactArgs(sinon.match.same(sPath));
		this.mock(oBinding).expects("fetchCache")
			.withExactArgs(sinon.match.same(oContext), false, /*bKeepQueryOptions*/true, undefined);
		this.mock(oBinding).expects("refreshKeptElements").withExactArgs("myGroup")
			.returns(oRefreshKeptElementsPromise);
		this.mock(oBinding).expects("createRefreshPromise").withExactArgs(undefined).callThrough();
		this.mock(oBinding).expects("reset")
			.withExactArgs(ChangeReason.Refresh, undefined, "myGroup");
		this.mock(oBinding.oHeaderContext).expects("checkUpdateInternal")
			.exactly(oFixture.success && !oFixture.refreshKeptElementsFails ? 1 : 0)
			.withExactArgs()
			.returns(oHeaderContextCheckUpdatePromise);

		// code under test
		oRefreshResult = oBinding.refreshInternal(sPath, "myGroup");
		// simulate getContexts
		oBinding.resolveRefreshPromise(
			oFixture.success ? Promise.resolve() : Promise.reject(oError));

		return oRefreshResult.then(function (oResult) {
			assert.ok(oFixture.success);
			assert.notOk(oFixture.refreshKeptElementsFails);
			assert.strictEqual(oResult, oHeaderContextCheckUpdatePromise.getResult());
		}, function (oError0) {
			assert.strictEqual(oError0, oError);
			assert.ok(!oFixture.success || oFixture.refreshKeptElementsFails);
		});
	});
});

	//*********************************************************************************************
[false, true].forEach(function (bAsync) {
	[false, true].forEach(function (bKeepCacheOnError) {
		[false, true].forEach(function (bRelative) {
			[false, true].forEach(function (bRestore) {
			var sTitle = "refreshInternal: bAsync=" + bAsync
				+ ", bKeepCacheOnError=" + bKeepCacheOnError
				+ ", bRelative=" + bRelative + ", cache can be restored: " + bRestore;

			if (bRestore && !bKeepCacheOnError) {
				return;
			}

	QUnit.test(sTitle, function (assert) {
		var oContext = bRelative
				? Context.createNewContext(this.oModel, oParentBinding, "/TEAMS('42')")
				: undefined,
			oBinding = this.bindList(bRelative ? "TEAM_2_EMPLOYEES" : "/EMPLOYEES", oContext,
				null, null, {$$ownRequest : true}),
			oCache = oBinding.oCachePromise.getResult(),
			oCacheMock = this.mock(oCache),
			iNoOfCalls = bAsync ? 2 : 1,
			oDependentBinding = {
				getContext : function () {
					return {
						isKeepAlive : function () { return false; }
					};
				},
				refreshInternal : function () {}
			},
			oError = new Error(),
			aCreatedContexts,
			aPromises = [],
			oReadPromise = Promise.reject(oError),
			that = this,
			i;

		function getPath(i) {
			return "/EMPLOYEES/" + i;
		}

		oBinding.iActiveContexts = 40;
		oBinding.iCreatedContexts = 42;
		oBinding.iCurrentEnd = 1;
		oBinding.aContexts = [];
		oBinding.mPreviousContextsByPath = {
			"/EMPLOYEES/99" : 99 // not parked by #reset
		};
		for (i = 0; i < oBinding.iCreatedContexts; i += 1) {
			oBinding.aContexts[i] = { // dummy for a created context
				// for simplicity, ignore bRelative here
				getPath : getPath.bind(null, i)
			};
		}
		aCreatedContexts = oBinding.aContexts.slice();
		oBinding.aContexts.push("n/a"); // dummy for a non-created
		this.mock(oBinding).expects("isRootBindingSuspended").exactly(iNoOfCalls).returns(false);
		this.mock(oBinding).expects("refreshSuspended").never();
		oReadPromise.catch(function () {
			var iCallCount = bKeepCacheOnError ? 1 : 0,
				oResourcePathPromise
					= Promise.resolve(bRelative ? oCache.getResourcePath() : "n/a");

			that.mock(oBinding).expects("fetchResourcePath").exactly(iCallCount)
				.withExactArgs(sinon.match.same(oContext))
				.returns(SyncPromise.resolve(oResourcePathPromise));
			oResourcePathPromise.then(function () {
				oCacheMock.expects("restore").exactly(bRestore ? 1 : 0).withExactArgs(true);
				oCacheMock.expects("restore").withExactArgs(false); // free memory
				oCacheMock.expects("setActive").exactly(bRestore ? 0 : iCallCount)
					.withExactArgs(true);
				that.mock(oBinding).expects("_fireChange").exactly(iCallCount)
					.withExactArgs({reason : ChangeReason.Change})
					.callsFake(function () {
						if (bKeepCacheOnError) {
							assert.strictEqual(oBinding.oCache, oCache);
							assert.strictEqual(oBinding.iActiveContexts, 40);
							assert.strictEqual(oBinding.iCreatedContexts, 42);
							assert.strictEqual(oBinding.oCachePromise.getResult(), oCache);
							assert.strictEqual(oBinding.aContexts.length, 42);
							aCreatedContexts.forEach(function (oCreatedContext, i) {
								assert.strictEqual(oBinding.aContexts[i], oCreatedContext);
								assert.strictEqual(oCreatedContext.iIndex, i - 42);
							});
						} else {
							assert.notStrictEqual(oBinding.oCache, oCache);
							assert.strictEqual(oBinding.iActiveContexts, 0);
							assert.strictEqual(oBinding.iCreatedContexts, 0);
							assert.notStrictEqual(oBinding.oCachePromise.getResult(), oCache);
							assert.deepEqual(oBinding.aContexts, ["a", "b", "c"], "unchanged");
						}
						assert.deepEqual(oBinding.mPreviousContextsByPath, {
							"/EMPLOYEES/99" : 99
						});
						assert.strictEqual(oBinding.bRefreshKeptElements, false, "unchanged");
					});
			});
		});
		this.mock(oBinding).expects("fetchCache")
			.withExactArgs(sinon.match.same(oContext), false, true,
				bKeepCacheOnError ? "myGroup" : undefined)
			.callsFake(function () {
				if (!bRestore) { // simulate creation of new cache
					oBinding.oCache = {
						refreshKeptElements : function () {} // don't care
					};
					oBinding.oCachePromise = SyncPromise.resolve(oBinding.oCache);
				}
			});
		this.mock(oBinding).expects("reset").exactly(iNoOfCalls)
			.withExactArgs(ChangeReason.Refresh, bKeepCacheOnError ? false : undefined, "myGroup")
			.callsFake(function () {
				oBinding.iActiveContexts = 0;
				oBinding.iCreatedContexts = 0;
				oBinding.aContexts = ["a", "b", "c"];
				if (bKeepCacheOnError) {
					for (i = 0; i < oBinding.iCreatedContexts; i += 1) {
						oBinding.mPreviousContextsByPath[getPath(i)] = i;
					}
				}
				if (!bAsync) {
					// simulate #getContexts call sync to "Refresh" event
					oBinding.resolveRefreshPromise(oReadPromise);
				}
			});
		this.mock(oBinding).expects("getDependentBindings").exactly(iNoOfCalls).withExactArgs()
			.returns([oDependentBinding]);
		this.mock(oDependentBinding).expects("refreshInternal").exactly(iNoOfCalls)
			.withExactArgs("", "myGroup", false, bKeepCacheOnError)
			.resolves();

		aPromises.push(
			// code under test
			oBinding.refreshInternal("", "myGroup", false, bKeepCacheOnError).then(function () {
				assert.ok(false);
			}, function (oReturnedError) {
				assert.strictEqual(oReturnedError, oError);
			}));
		if (bAsync) { //TODO in the sync case, the wrong cache would be restored :-(
			aPromises.push(
				// code under test
				oBinding.refreshInternal("", "myGroup", false, bKeepCacheOnError).then(function () {
					assert.ok(false);
				}, function (oReturnedError) {
					assert.strictEqual(oReturnedError, oError);
				}));
			oBinding.oCachePromise.then(function () {
				// simulate #getContexts call async to "Refresh" event
				oBinding.resolveRefreshPromise(oReadPromise);
			});
		}

		return Promise.all(aPromises);
	});
			});
		});
	});
});

	//*********************************************************************************************
[false, true].forEach(function (bFetchResourcePathFails) {
	var sTitle = "refreshInternal: bAsync=false, bKeepCacheOnError=true, GET fails"
		+ ", parent context has changed in the meantime, fetchResourcePath fails="
		+ bFetchResourcePathFails;

	QUnit.test(sTitle, function (assert) {
		var oContext = Context.createNewContext(this.oModel, oParentBinding, "/TEAMS('42')"),
			oBinding = this.bindList("TEAM_2_EMPLOYEES", oContext, null, null,
				{$$ownRequest : true}),
			oError = new Error(),
			bIsRoot = "false,true",
			oNewCache = {refreshKeptElements : function () {}},
			oOldCache = oBinding.oCachePromise.getResult(),
			oRefreshPromise = Promise.reject(oError),
			oYetAnotherError = new Error(),
			that = this;

		oBinding.iCurrentEnd = 1;
		this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs().returns(false);
		this.mock(oBinding).expects("refreshSuspended").never();
		this.mock(oBinding).expects("isRoot").withExactArgs().returns(bIsRoot);
		this.mock(oBinding).expects("createReadGroupLock").withExactArgs("myGroup", bIsRoot);
		this.mock(oBinding).expects("removeCachesAndMessages").withExactArgs("path");
		this.mock(oBinding).expects("fetchCache")
			.withExactArgs(sinon.match.same(oContext), false, /*bKeepQueryOptions*/true, "myGroup")
			.callsFake(function () {
				oBinding.oCache = oNewCache;
				oBinding.oCachePromise = SyncPromise.resolve(oNewCache);
			});
		this.mock(oBinding).expects("createRefreshPromise").withExactArgs(true)
			.returns(oRefreshPromise);
		this.mock(oBinding).expects("reset").withExactArgs(ChangeReason.Refresh, false, "myGroup");
		this.mock(this.oModel).expects("getDependentBindings")
			.withExactArgs(sinon.match.same(oBinding)).returns([]);
		this.mock(oBinding.oHeaderContext).expects("checkUpdateInternal").never();
		oRefreshPromise.catch(function () {
			var oResourcePathPromise = Promise.resolve("n/a");

			that.mock(oBinding).expects("fetchResourcePath")
				.withExactArgs(sinon.match.same(oContext))
				.returns(bFetchResourcePathFails
					? SyncPromise.reject(oYetAnotherError)
					: SyncPromise.resolve(oResourcePathPromise));
			oResourcePathPromise.then(function () {
				that.mock(oOldCache).expects("setActive").never();
				that.mock(oOldCache).expects("restore").withExactArgs(false); // free memory
				that.mock(oBinding).expects("_fireChange").never();
			});
		});

		// code under test
		return oBinding.refreshInternal("path", "myGroup", /*_bCheckUpdate*/false, true)
			.then(function () {
				assert.ok(false);
			}, function (oReturnedError) {
				assert.strictEqual(oReturnedError,
					bFetchResourcePathFails ? oYetAnotherError : oError);
				assert.strictEqual(oBinding.oCache, oNewCache);
				assert.strictEqual(oBinding.oCachePromise.getResult(), oNewCache);
				assert.strictEqual(oBinding.bRefreshKeptElements, false, "unchanged");
			});
	});
});

	//*********************************************************************************************
	QUnit.test("refreshInternal: bKeepCacheOnError & canceled", function (assert) {
		var oContext = Context.createNewContext(this.oModel, oParentBinding, "/TEAMS('42')"),
			oBinding = this.bindList("TEAM_2_EMPLOYEES", oContext, null, null,
				{$$ownRequest : true}),
			oCache = oBinding.oCache,
			oError = new Error(),
			oNewCache = {refreshKeptElements : function () {}};

		oError.canceled = true;
		oBinding.iCurrentEnd = 1;
		this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs().returns(false);
		this.mock(oBinding).expects("refreshSuspended").never();
		this.mock(oBinding).expects("isRoot").withExactArgs().returns("bIsRoot");
		this.mock(oBinding).expects("createReadGroupLock").withExactArgs("myGroup", "bIsRoot");
		this.mock(oBinding).expects("removeCachesAndMessages").withExactArgs("path");
		this.mock(oBinding).expects("fetchCache")
			.withExactArgs(sinon.match.same(oContext), false, /*bKeepQueryOptions*/true, "myGroup")
			.callsFake(function () {
				oBinding.oCache = oNewCache;
				oBinding.oCachePromise = SyncPromise.resolve(oNewCache);
			});
		this.mock(oBinding).expects("createRefreshPromise").withExactArgs(true).rejects(oError);
		this.mock(oBinding).expects("fetchResourcePath").never();
		this.mock(oCache).expects("restore").withExactArgs(false); // free memory
		this.mock(oBinding).expects("reset").withExactArgs(ChangeReason.Refresh, false, "myGroup");
		this.mock(this.oModel).expects("getDependentBindings")
			.withExactArgs(sinon.match.same(oBinding)).returns([]);
		this.mock(oBinding.oHeaderContext).expects("checkUpdate").never();

		// code under test
		return oBinding.refreshInternal("path", "myGroup", /*_bCheckUpdate*/false, true)
			.then(function () {
				assert.ok(false);
			}, function (oReturnedError) {
				assert.strictEqual(oReturnedError, oError);
				assert.strictEqual(oBinding.oCache, oNewCache);
				assert.strictEqual(oBinding.oCachePromise.getResult(), oNewCache);
				assert.strictEqual(oBinding.bRefreshKeptElements, false, "unchanged");
			});
	});

	//*********************************************************************************************
	QUnit.test("refreshInternal: relative without own cache", function (assert) {
		var oBinding = this.bindList("TEAM_2_EMPLOYEES",
				Context.create(this.oModel, oParentBinding, "/TEAMS('1')"));

		this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs().returns(false);
		this.mock(oBinding).expects("createReadGroupLock").withExactArgs("myGroup", false);
		this.mock(oBinding).expects("removeCachesAndMessages").never();
		this.mock(oBinding).expects("fetchCache").never();
		this.mock(oBinding).expects("createRefreshPromise").never();
		this.mock(oBinding).expects("reset")
			.withExactArgs(ChangeReason.Refresh, /*bDrop*/true, "myGroup");

		// code under test (as called from #requestRefresh)
		assert.ok(oBinding.refreshInternal("", "myGroup", /*_bCheckUpdate*/true).isFulfilled());
	});

//*********************************************************************************************
[false, true].forEach(function (bSuspended) {
	[false, true].forEach(function (bShared) {
		var sTitle = "refreshInternal: dependent bindings, suspended=" + bSuspended
				+ ", shared=" + bShared;

		QUnit.test(sTitle, function (assert) {
			var oBinding = this.bindList("/EMPLOYEES", undefined, undefined, undefined,
					{$$groupId : "myGroup"}),
				oChild0 = {
					getContext : getNonKeptContext,
					refreshInternal : function () {}
				},
				oChild0Refreshed = false,
				oChild1 = {
					getContext : getKeptContext,
					hasPendingChanges : function () { return false; },
					refreshInternal : function () {}
				},
				oChild1Refreshed = false,
				oChild2 = {
					getContext : getNonKeptContext,
					refreshInternal : function () {}
				},
				oChild2Refreshed = false,
				oChild3 = {
					getContext : getNonKeptContext,
					refreshInternal : function () {}
				},
				oChild3RefreshedIfSuspended = false,
				oChild4 = {
					getContext : getKeptContext,
					hasPendingChanges : function () { return true; },
					refreshInternal : function () {}
				},
				oRefreshResult,
				sResourcePathPrefix = "foo";

			function getKeptContext() {
				return {
					isKeepAlive : function () { return true; }
				};
			}

			function getNonKeptContext() {
				return {
					isKeepAlive : function () { return false; }
				};
			}

			oBinding.bSharedRequest = bShared;
			this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs()
				.returns(bSuspended);
			this.mock(oBinding).expects("refreshSuspended").exactly(bSuspended && !bShared ? 1 : 0)
				.withExactArgs("myGroup");
			this.mock(oBinding).expects("createReadGroupLock").exactly(bSuspended ? 0 : 1)
				.withExactArgs("myGroup", true);
			this.mock(oBinding).expects("reset").exactly(bSuspended ? 0 : 1)
				.withExactArgs(ChangeReason.Refresh, undefined, "myGroup")
				.callsFake(function () {
					// BCP: 002075129400006474012021 reset may result in a destroyed child binding
					oChild3.bIsBeingDestroyed = true;
				});
			this.mock(oBinding).expects("getDependentBindings").withExactArgs()
				.returns([oChild0, oChild1, oChild2, oChild3, oChild4]);
			this.mock(oChild0).expects("refreshInternal")
				.withExactArgs(sResourcePathPrefix, "myGroup", false, undefined)
				.returns(new Promise(function (resolve) {
					setTimeout(function () {
						oChild0Refreshed = true;
						resolve();
					});
				}));
			this.mock(oChild1).expects("refreshInternal")
				.withExactArgs(sResourcePathPrefix, "myGroup", false, undefined)
				.returns(new Promise(function (resolve) {
					setTimeout(function () {
						oChild1Refreshed = true;
						resolve();
					});
				}));
			this.mock(oChild2).expects("refreshInternal")
				.withExactArgs(sResourcePathPrefix, "myGroup", false, undefined)
				.returns(new Promise(function (resolve) {
					setTimeout(function () {
						oChild2Refreshed = true;
						resolve();
					});
				}));
			this.mock(oChild3).expects("refreshInternal").exactly(bSuspended ? 1 : 0)
				.withExactArgs(sResourcePathPrefix, "myGroup", false, undefined)
				.returns(new Promise(function (resolve) {
					setTimeout(function () {
						oChild3RefreshedIfSuspended = true;
						resolve();
					});
				}));
			this.mock(oChild4).expects("refreshInternal").never();

			// code under test
			oRefreshResult = oBinding.refreshInternal(sResourcePathPrefix, "myGroup");
			if (bSuspended) {
				assert.strictEqual(oBinding.bRefreshKeptElements, !bShared);
				assert.strictEqual(oBinding.sResumeAction, bShared ? "resetCache" : undefined);
			} else {
				oBinding.resolveRefreshPromise(Promise.resolve()); // simulate getContexts
			}
			assert.ok(oRefreshResult.isPending());
			return oRefreshResult.then(function () {
				assert.strictEqual(oChild0Refreshed, true);
				assert.strictEqual(oChild1Refreshed, true);
				assert.strictEqual(oChild2Refreshed, true);
				assert.strictEqual(oChild3RefreshedIfSuspended, bSuspended);
			});
		});
	});
});

	//*********************************************************************************************
	QUnit.test("refreshInternal: shared cache", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES", null, null, null, {$$sharedRequest : true});

		this.mock(oBinding).expects("createReadGroupLock").withExactArgs("myGroup", true);
		this.mock(oBinding).expects("removeCachesAndMessages")
			.withExactArgs("~sResourcePathPrefix~");
		this.mock(oBinding).expects("createRefreshPromise").withExactArgs()
			.returns(Promise.reject("~oError~"));
		this.mock(oBinding.oCache).expects("reset").withExactArgs([]);
		this.mock(oBinding).expects("fetchCache").never();
		this.mock(oBinding).expects("refreshKeptElements").never();

		// code under test
		return oBinding.refreshInternal("~sResourcePathPrefix~", "myGroup").then(function () {
			assert.ok(false);
		}, function (oError) {
			assert.strictEqual(oError, "~oError~");
		});
	});

	//*********************************************************************************************
[false, true].forEach(function (bFail) {
	[undefined, 2].forEach(function (iIndex) {
		var sTitle = "refreshKeptElements: fail = " + bFail + ", index = " + iIndex;

	QUnit.test(sTitle, function (assert) {
		var oBinding = this.bindList("/TEAMS"),
			oError = new Error(),
			oKeptContext = {resetKeepAlive : function () {}},
			oNewCache = {refreshKeptElements : function () {}},
			oRefreshKeptElementsCall,
			that = this;

		oBinding.oCachePromise = SyncPromise.resolve(Promise.resolve(oNewCache));
		oBinding.aContexts = [,, "~aContexts[2]~"];
		oBinding.mPreviousContextsByPath = {
			"/resolved/path('42')" : oKeptContext
		};
		this.mock(oBinding).expects("lockGroup").withExactArgs("myGroup").returns("~groupLock~");
		oRefreshKeptElementsCall = this.mock(oNewCache).expects("refreshKeptElements")
			.withExactArgs("~groupLock~", sinon.match.func)
			.returns(bFail
				? SyncPromise.reject(oError)
				: SyncPromise.resolve("~result~"));
		this.mock(oBinding.getModel()).expects("reportError").exactly(bFail ? 1 : 0)
			.withExactArgs("Failed to refresh kept-alive elements", sClassName,
				sinon.match.same(oError));

		// code under test
		return oBinding.refreshKeptElements("myGroup").then(function (vResult) {
			var iCallCount = iIndex === undefined ? 1 : 0;

			assert.strictEqual(vResult, "~result~");
			assert.notOk(bFail);

			that.mock(oBinding).expects("getResolvedPath").exactly(iCallCount).withExactArgs()
				.returns("/resolved/path");
			that.mock(oKeptContext).expects("resetKeepAlive").exactly(iCallCount).withExactArgs();
			that.mock(oBinding).expects("removeCreated").exactly(1 - iCallCount)
				.withExactArgs("~aContexts[2]~");

			// code under test
			oRefreshKeptElementsCall.firstCall.args[1]("('42')", iIndex);
		}, function (oError0) {
			assert.strictEqual(oError0, oError);
			assert.ok(bFail);
		});
	});
	});
});

	//********************************************************************************************
[
	{bCached : false, oGroupLock : {}},
	{bCached : true, oGroupLock : _GroupLock.$cached}
].forEach(function (oFixture) {
	QUnit.test("fetchValue: absolute binding, bCached=" + oFixture.bCached, function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oListener = {},
			oPromise,
			oReadResult = {};

		this.mock(oBinding).expects("lockGroup").exactly(oFixture.bCached ? 0 : 1)
			.withExactArgs().returns(oFixture.oGroupLock);
		this.mock(oBinding).expects("getRelativePath")
			.withExactArgs("/EMPLOYEES/42/bar").returns("42/bar");
		this.mock(oBinding.oCachePromise.getResult()).expects("fetchValue")
			.withExactArgs(sinon.match.same(oFixture.oGroupLock), "42/bar", undefined,
				sinon.match.same(oListener))
			.returns(SyncPromise.resolve(oReadResult));

		// code under test
		oPromise = oBinding.fetchValue("/EMPLOYEES/42/bar", oListener, oFixture.bCached);

		assert.ok(oPromise.isFulfilled());
		return oPromise.then(function (oResult) {
			assert.strictEqual(oResult, oReadResult);
		});
	});
});

	//*********************************************************************************************
[false, true].forEach(function (bCached) {
	QUnit.test("fetchValue: relative binding, bCached = " + bCached, function (assert) {
		var oContext = Context.create(this.oModel, oParentBinding, "/foo"),
			oListener = {},
			sPath = "/foo/42/bar",
			oResult = {},
			oBinding = this.bindList("TEAM_2_EMPLOYEES", oContext);

		if (bCached) {
			// never resolved, must be ignored
			oBinding.oCachePromise = new SyncPromise(function () {});
		}
		this.mock(oContext).expects("fetchValue")
			.withExactArgs(sPath, sinon.match.same(oListener), sinon.match.same(bCached))
			.returns(SyncPromise.resolve(oResult));

		// code under test
		assert.strictEqual(oBinding.fetchValue(sPath, oListener, bCached).getResult(), oResult);
	});
});
	//TODO provide iStart, iLength parameter to fetchValue to support paging on nested list

	//*********************************************************************************************
	QUnit.test("fetchValue: relative binding, unresolved", function (assert) {
		this.bindList("TEAM_2_EMPLOYEES").fetchValue("bar", {}).then(function (oResult) {
			assert.strictEqual(oResult, undefined);
		});
	});

	//*********************************************************************************************
	QUnit.test("fetchValue: relative binding w/ cache, absolute path, mismatch", function (assert) {
		var oBinding,
			oBindingMock = this.mock(ODataListBinding.prototype),
			bCached = {/*false,true*/},
			oContext = Context.create(this.oModel, oParentBinding, "/SalesOrderList('1')"),
			oGroupLock = {unlock : function () {}},
			oListener = {},
			sPath = "/SalesOrderList('1')/ID",
			oResult = {};

		// fetchCache is called once from applyParameters before oBinding.oContext is set
		oBindingMock.expects("fetchCache").withExactArgs(undefined).callsFake(function () {
			this.oCache = null;
			this.oCachePromise = SyncPromise.resolve(null);
		});
		oBindingMock.expects("fetchCache").withExactArgs(sinon.match.same(oContext)).atLeast(1)
			.callsFake(function () {
				this.oCache = {};
				this.oCachePromise = SyncPromise.resolve(this.oCache);
			});
		oBindingMock.expects("restoreCreated").withExactArgs();
		oBinding = this.bindList("SO_2_SOITEM", oContext, undefined, undefined,
			{$$groupId : "group"});

		this.mock(oBinding).expects("getRelativePath").withExactArgs(sPath).returns(undefined);
		this.mock(oGroupLock).expects("unlock").never();
		this.mock(oContext).expects("fetchValue")
			.withExactArgs(sPath, sinon.match.same(oListener), sinon.match.same(bCached))
			.returns(oResult);

		// code under test
		assert.strictEqual(oBinding.fetchValue(sPath, oListener, bCached).getResult(), oResult);
	});

	//*********************************************************************************************
	QUnit.test("fetchValue: oCachePromise still pending", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oCache = oBinding.oCachePromise.getResult(),
			sPath = "/EMPLOYEES/42/bar",
			oReadResult = {};

		oBinding.oCache = undefined;
		oBinding.oCachePromise = SyncPromise.resolve(Promise.resolve(oCache));
		this.mock(oBinding).expects("getRelativePath").withExactArgs(sPath).returns("42/bar");
		this.mock(oCache).expects("fetchValue")
			.withExactArgs(sinon.match.same(_GroupLock.$cached), "42/bar", undefined, null)
			.returns(SyncPromise.resolve(oReadResult));

		// code under test
		return oBinding.fetchValue(sPath, null, true).then(function (oResult) {
			assert.strictEqual(oResult, oReadResult);
		});
	});

	//*********************************************************************************************
	QUnit.test("fetchValue: oCachePromise became pending again", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oCache = oBinding.oCachePromise.getResult(),
			sPath = "/EMPLOYEES/42/bar",
			oReadResult = {};

		oBinding.oCachePromise = new SyncPromise(function () {}); // never resolved, must be ignored
		this.mock(oBinding).expects("getRelativePath").withExactArgs(sPath).returns("42/bar");
		this.mock(oCache).expects("fetchValue")
			.withExactArgs(sinon.match.same(_GroupLock.$cached), "42/bar", undefined, null)
			.returns(SyncPromise.resolve(oReadResult));

		// code under test
		assert.strictEqual(oBinding.fetchValue(sPath, null, true).getResult(), oReadResult);
	});

	//*********************************************************************************************
	QUnit.test("fetchValue: !bCached, wait for oCachePromise again", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oCache = oBinding.oCachePromise.getResult(),
			oGroupLock = {},
			sPath = "/EMPLOYEES/42/bar",
			oReadResult = {};

		oBinding.oCache = {/*do not use!*/};
		oBinding.oCachePromise = SyncPromise.resolve(Promise.resolve(oCache));
		oBinding.oReadGroupLock = undefined; // not interested in the initial case
		this.mock(oBinding).expects("getRelativePath").withExactArgs(sPath).returns("42/bar");
		this.mock(oBinding).expects("lockGroup").withExactArgs().returns(oGroupLock);
		this.mock(oCache).expects("fetchValue")
			.withExactArgs(sinon.match.same(oGroupLock), "42/bar", undefined, undefined)
			.returns(SyncPromise.resolve(oReadResult));

		// code under test
		return oBinding.fetchValue(sPath).then(function (oResult) {
			assert.strictEqual(oResult, oReadResult);
		});
	});

	//*********************************************************************************************
	QUnit.test("forbidden", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		assert.throws(function () {
			oBinding.getDistinctValues();
		}, new Error("Unsupported operation: v4.ODataListBinding#getDistinctValues"));

		assert.throws(function () {
			oBinding.getContexts(0, 10, 100, true);
		}, new Error("Unsupported operation: v4.ODataListBinding#getContexts, must not use both"
				+ " iMaximumPrefetchSize and bKeepCurrent"));

		oBinding.enableExtendedChangeDetection();
		assert.throws(function () { //TODO implement?
			oBinding.getContexts(0, 42, 0);
		}, new Error("Unsupported operation: v4.ODataListBinding#getContexts, iMaximumPrefetchSize"
				+ " must not be set if extended change detection is enabled"));

		assert.throws(function () {
			oBinding.getContexts(42);
		}, new Error("Unsupported operation: v4.ODataListBinding#getContexts, iStart must be 0"
			+ " if extended change detection is enabled, but is 42"));

		assert.throws(function () {
			oBinding.getContexts(0, 10, undefined, true);
		}, new Error("Unsupported operation: v4.ODataListBinding#getContexts, must not use"
			+ " bKeepCurrent if extended change detection is enabled"));
	});
	//TODO errors on _fireFilter(mArguments) and below in Wiki

	//*********************************************************************************************
	QUnit.test("events", function (assert) {
		var oBinding,
			oBindingMock = this.mock(ListBinding.prototype),
			mEventParameters = {},
			oReturn = {};

		oBinding = this.bindList("/EMPLOYEES");

		[
			"AggregatedDataStateChange",
			"change",
			"createActivate",
			"createCompleted",
			"createSent",
			"dataReceived",
			"dataRequested",
			"DataStateChange",
			"patchCompleted",
			"patchSent",
			"refresh"
		].forEach(function (sEvent) {
			oBindingMock.expects("attachEvent")
				.withExactArgs(sEvent, sinon.match.same(mEventParameters)).returns(oReturn);

			assert.strictEqual(oBinding.attachEvent(sEvent, mEventParameters), oReturn);
		});

		["filter", "sort", "unsupportedEvent"].forEach(function (sEvent) {
			assert.throws(function () {
				oBinding.attachEvent(sEvent);
			}, new Error("Unsupported event '" + sEvent + "': v4.ODataListBinding#attachEvent"));
		});
	});

	//*********************************************************************************************
	[
		{
			mParameters : {$$operationMode : OperationMode.Server},
			queryOptions : {"sap-client" : "111"},
			vSorters : undefined
		}, {
			mParameters : {$$operationMode : OperationMode.Server},
			queryOptions : {$orderby : "foo", "sap-client" : "111"},
			vSorters : new Sorter("foo")
		}, {
			mParameters : {$$operationMode : OperationMode.Server, $orderby : "bar"},
			queryOptions : {$orderby : "foo,bar", "sap-client" : "111"},
			vSorters : [new Sorter("foo")]
		}, {
			oModel : new ODataModel({
				operationMode : OperationMode.Server,
				serviceUrl : "/service/?sap-client=111",
				synchronizationMode : "None"
			}),
			mParameters : {$orderby : "bar"},
			queryOptions : {$orderby : "foo,bar", "sap-client" : "111"},
			vSorters : [new Sorter("foo")]
		}
	].forEach(function (oFixture) {
		[false, true].forEach(function (bSuspended) {
			var sTitle = "bSuspended=" + bSuspended + ", vSorters = "
				+ JSON.stringify(oFixture.vSorters) + " and mParameters = "
				+ JSON.stringify(oFixture.mParameters);

			QUnit.test("sort: " + sTitle, function (assert) {
				var oBinding,
					oHelperMock = this.mock(_Helper),
					oModel = oFixture.oModel || this.oModel,
					oContext = Context.createNewContext(oModel, oParentBinding, "/TEAMS", 1),
					aSorters = [];

				oBinding = oModel.bindList("TEAM_2_EMPLOYEES", undefined, undefined, undefined,
					oFixture.mParameters);
				oBinding.setContext(oContext);

				this.mock(oBinding).expects("checkSuspended").never();
				this.mock(oBinding).expects("hasPendingChanges").returns(false);
				oHelperMock.expects("toArray")
					.withExactArgs(sinon.match.same(oFixture.vSorters))
					.returns(aSorters);
				oHelperMock.expects("deepEqual")
					.withExactArgs(sinon.match.same(aSorters), sinon.match.same(oBinding.aSorters))
					.returns(false);
				this.mock(oBinding).expects("isRootBindingSuspended").returns(bSuspended);
				this.mock(oBinding).expects("setResumeChangeReason").exactly(bSuspended ? 1 : 0)
					.withExactArgs(ChangeReason.Sort);
				this.mock(oBinding).expects("reset").exactly(bSuspended ? 0 : 1)
					.withExactArgs(ChangeReason.Sort);
				this.mock(oBinding).expects("removeCachesAndMessages").exactly(bSuspended ? 0 : 1)
					.withExactArgs("");
				this.mock(oBinding).expects("getGroupId").exactly(bSuspended ? 0 : 1)
					.withExactArgs().returns("group");
				this.mock(oBinding).expects("createReadGroupLock").exactly(bSuspended ? 0 : 1)
					.withExactArgs("group", true);
				this.mock(oBinding).expects("fetchCache").exactly(bSuspended ? 0 : 1)
					.withExactArgs(sinon.match.same(oContext))
					.callsFake(function () {
						this.oCache = {};
						this.oCachePromise = SyncPromise.resolve(this.oCache);
					});
				this.mock(oBinding.oHeaderContext).expects("checkUpdate")
					.exactly(bSuspended ? 0 : 1).withExactArgs();

				// code under test
				assert.strictEqual(oBinding.sort(oFixture.vSorters), oBinding, "chaining");

				assert.strictEqual(oBinding.aSorters, aSorters);
			});
		});
	});

	//*********************************************************************************************
	QUnit.test("sort: unresolved binding", function () {
		var oBinding = this.oModel.bindList("TEAM_2_EMPLOYEES", null, null, null,
				{$$operationMode : OperationMode.Server});

		oBinding.aSorters.push("~initial sorters~");

		// code under test
		oBinding.sort();
	});

	//*********************************************************************************************
	QUnit.test("sort: errors", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oContext;

		// code under test
		assert.throws(function () {
			oBinding.sort([]);
		}, new Error("Operation mode has to be sap.ui.model.odata.OperationMode.Server"));

		// code under test
		assert.throws(function () {
			oBinding.sort();
		}, new Error("Operation mode has to be sap.ui.model.odata.OperationMode.Server"));

		oBinding = this.bindList("/EMPLOYEES", null, null, null,
			{$$operationMode : OperationMode.Server});
		oBinding.aSorters.push("~initial sorters~");
		this.mock(oBinding).expects("hasPendingChanges").withExactArgs(true).returns(true);

		// code under test
		assert.throws(function () {
			oBinding.sort();
		}, new Error("Cannot sort due to pending changes"));

		this.mock(ODataListBinding.prototype).expects("fetchCache").atLeast(1)
			.callsFake(function () {
				this.oCache = {};
				this.oCachePromise = SyncPromise.resolve(this.oCache);
			});
		this.mock(ODataListBinding.prototype).expects("restoreCreated").atLeast(1).withExactArgs();
		oContext = Context.create(this.oModel, oParentBinding, "/TEAMS", 1);
		oBinding = this.bindList("TEAM_2_EMPLOYEES", oContext, undefined, undefined,
			{$$operationMode : OperationMode.Server});

		oBinding.aSorters.push("~initial sorters~");
		this.mock(oBinding).expects("hasPendingChanges").withExactArgs(true).returns(true);

		// code under test
		assert.throws(function () {
			oBinding.sort();
		}, new Error("Cannot sort due to pending changes"));
	});

	//*********************************************************************************************
	QUnit.test("sort: same sorters skips processing", function (assert) {
		var oBinding,
			oBindingMock = this.mock(ODataListBinding.prototype),
			oSorter = new Sorter("foo"),
			aSorters = [oSorter];

		oBinding = this.bindList("/EMPLOYEES", undefined, undefined, undefined, {
			$$operationMode : OperationMode.Server
		});

		oBinding.aSorters.push(oSorter);

		this.mock(_Helper).expects("toArray").withExactArgs(sinon.match.same(oSorter))
			.returns(aSorters);
		this.mock(_Helper).expects("deepEqual").withExactArgs(sinon.match.same(aSorters),
				sinon.match.same(oBinding.aSorters))
			.returns(true);

		oBindingMock.expects("hasPendingChanges").never();
		oBindingMock.expects("reset").never();

		// code under test
		assert.strictEqual(oBinding.sort(oSorter), oBinding);
	});

	//*********************************************************************************************
	[undefined, FilterType.Application, FilterType.Control].forEach(function (sFilterType) {
		[false, true].forEach(function (bSuspended) {
			var sTitle = "filter: FilterType=" + sFilterType + ", suspended=" + bSuspended;

			QUnit.test(sTitle, function (assert) {
				var oBinding,
					oBindingMock = this.mock(ODataListBinding.prototype),
					oFilter = new Filter("Name", FilterOperator.Contains, "foo"),
					aFilters = [oFilter],
					oHelperMock = this.mock(_Helper),
					sStaticFilter = "Age gt 18";

				oBindingMock.expects("checkSuspended").never();

				oBinding = this.bindList("/EMPLOYEES", undefined, undefined, undefined, {
					$filter : sStaticFilter,
					$$operationMode : OperationMode.Server
				});

				oHelperMock.expects("toArray").withExactArgs(sinon.match.same(oFilter))
					.returns(aFilters);
				oHelperMock.expects("deepEqual").exactly(sFilterType === FilterType.Control ? 1 : 0)
					.withExactArgs(sinon.match.same(aFilters), sinon.match.same(oBinding.aFilters))
					.returns(false);
				oHelperMock.expects("deepEqual").exactly(sFilterType === FilterType.Control ? 0 : 1)
					.withExactArgs(sinon.match.same(aFilters),
						sinon.match.same(oBinding.aApplicationFilters))
					.returns(false);
				oBindingMock.expects("hasPendingChanges").withExactArgs(true).returns(false);
				oBindingMock.expects("isRootBindingSuspended").withExactArgs().returns(bSuspended);
				oBindingMock.expects("getGroupId").exactly(bSuspended ? 0 : 1)
					.withExactArgs().returns("groupId");
				oBindingMock.expects("createReadGroupLock").exactly(bSuspended ? 0 : 1)
					.withExactArgs("groupId", true);
				oBindingMock.expects("removeCachesAndMessages").exactly(bSuspended ? 0 : 1)
					.withExactArgs("");
				oBindingMock.expects("fetchCache").exactly(bSuspended ? 0 : 1)
					.withExactArgs(sinon.match.same(oBinding.oContext));
				oBindingMock.expects("reset").exactly(bSuspended ? 0 : 1)
					.withExactArgs(ChangeReason.Filter);
				oBindingMock.expects("setResumeChangeReason").exactly(bSuspended ? 1 : 0)
					.withExactArgs(ChangeReason.Filter);
				this.mock(oBinding.oHeaderContext).expects("checkUpdate")
					.exactly(bSuspended ? 0 : 1).withExactArgs();

				// code under test
				assert.strictEqual(oBinding.filter(oFilter, sFilterType), oBinding, "chaining");

				if (sFilterType === FilterType.Control) {
					assert.strictEqual(oBinding.aFilters, aFilters);
					assert.deepEqual(oBinding.aApplicationFilters, []);
				} else {
					assert.strictEqual(oBinding.aApplicationFilters, aFilters);
					assert.deepEqual(oBinding.aFilters, []);
				}
			});
		});
	});

	//*********************************************************************************************
	[undefined, FilterType.Application, FilterType.Control].forEach(function (sFilterType) {
		var sTitle = "filter: same filters skips processing; FilterType=" + sFilterType;

		QUnit.test(sTitle, function (assert) {
			var oBinding,
				oBindingMock = this.mock(ODataListBinding.prototype),
				oHelperMock = this.mock(_Helper),
				aFilters = ["~filter~"];

			oBinding = this.bindList("/EMPLOYEES", undefined, undefined, undefined, {
				$$operationMode : OperationMode.Server
			});

			if (sFilterType === FilterType.Control) {
				oBinding.aFilters.push("~filter~");
			} else {
				oBinding.aApplicationFilters.push("~filter~");
			}

			oHelperMock.expects("toArray").withExactArgs(sinon.match.same("~filter~"))
				.returns(aFilters);
			oHelperMock.expects("deepEqual").withExactArgs(sinon.match.same(aFilters),
					sinon.match.same(sFilterType === FilterType.Control
						? oBinding.aFilters : oBinding.aApplicationFilters))
				.returns(true);

			oBindingMock.expects("hasPendingChanges").never();
			oBindingMock.expects("reset").never();

			// code under test
			assert.strictEqual(oBinding.filter("~filter~", sFilterType), oBinding, "chaining");
		});
	});

	//*********************************************************************************************
	QUnit.test("filter: BCP: 2280148151", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES", null, [/*vSorters*/], [/*vFilters*/], {
				$$operationMode : OperationMode.Server
			});

		oBinding.aFilters.push("~filter~");
		this.mock(oBinding).expects("getGroupId").withExactArgs().returns("groupId");
		this.mock(oBinding).expects("createReadGroupLock").withExactArgs("groupId", true);
		this.mock(oBinding).expects("removeCachesAndMessages").withExactArgs("");
		this.mock(oBinding).expects("fetchCache").withExactArgs(null);
		this.mock(oBinding).expects("reset").withExactArgs(ChangeReason.Filter);

		// code under test
		oBinding.filter([], FilterType.Control);

		assert.deepEqual(oBinding.aFilters, [], "control filters removed");
	});

	//*********************************************************************************************
	QUnit.test("filter: removes caches and messages", function () {
		var oBinding = this.bindList("/EMPLOYEES", undefined, undefined, undefined, {
			$$operationMode : OperationMode.Server
		});

		oBinding.aApplicationFilters.push("~filter~");

		this.mock(oBinding).expects("removeCachesAndMessages").withExactArgs("");
		this.mock(oBinding).expects("fetchCache").withExactArgs(undefined);

		// Code under test
		oBinding.filter(/*no filter*/);
	});

	//*********************************************************************************************
	QUnit.test("filter: unresolved binding", function () {
		var oBinding = this.oModel.bindList("TEAM_2_EMPLOYEES", null, null, null,
				{$$operationMode : OperationMode.Server});

		oBinding.aApplicationFilters.push("~filter~");

		// code under test
		oBinding.filter();
	});

	//*********************************************************************************************
	QUnit.test("filter: check errors", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		oBinding.aApplicationFilters.push("~filter~");

		// code under test
		assert.throws(function () {
			oBinding.filter();
		}, new Error("Operation mode has to be sap.ui.model.odata.OperationMode.Server"));

		oBinding = this.bindList("/EMPLOYEES", undefined, undefined, undefined,
			{$$operationMode : OperationMode.Server});

		this.mock(_Helper).expects("toArray").withExactArgs(undefined).returns([]);
		this.mock(_Helper).expects("deepEqual")
			.withExactArgs([], sinon.match.same(oBinding.aApplicationFilters))
			.returns(false);
		this.mock(oBinding).expects("hasPendingChanges").withExactArgs(true).returns(true);

		// code under test
		assert.throws(function () {
			oBinding.filter();
		}, new Error("Cannot filter due to pending changes"));
	});

	//*********************************************************************************************
	QUnit.test("destroy", function (assert) {
		var oBinding,
			oBindingContext = {destroy : function () {}},
			oBindingContextMock = this.mock(oBindingContext),
			oBindingMock = this.mock(ListBinding.prototype),
			oModelMock = this.mock(this.oModel),
			oParentBindingPrototypeMock = this.mock(asODataParentBinding.prototype),
			oTransientBindingContext = {destroy : function () {}},
			oTransientBindingContextMock = this.mock(oTransientBindingContext);

		oBinding = this.bindList("relative"); // unresolved
		this.mock(oBinding).expects("destroyPreviousContexts").withExactArgs();
		oModelMock.expects("bindingDestroyed").withExactArgs(sinon.match.same(oBinding));
		oBindingMock.expects("destroy").on(oBinding).withExactArgs();
		oParentBindingPrototypeMock.expects("destroy").on(oBinding).withExactArgs();
		oBinding.oDiff = [/*some diff*/];

		// code under test
		oBinding.destroy();

		assert.strictEqual(oBinding.aApplicationFilters, undefined);
		assert.strictEqual(oBinding.aContexts, undefined);
		assert.strictEqual(oBinding.oDiff, undefined);
		assert.strictEqual(oBinding.aFilters, undefined);
		//TODO does not work with ODataModel.integration "suspend/resume"
		// assert.strictEqual(oBinding.mParameters, undefined);
		assert.strictEqual(oBinding.mPreviousContextsByPath, undefined);
		assert.strictEqual(oBinding.aPreviousData, undefined);
		assert.strictEqual(oBinding.mQueryOptions, undefined);
		assert.strictEqual(oBinding.aSorters, undefined);

		assert.throws(function () {
			// code under test: must not destroy twice (fails somehow)
			oBinding.destroy();
		});

		oBinding = this.bindList("relative", Context.create(this.oModel, oParentBinding, "/foo"));
		oBinding.aContexts = [oBindingContext];
		oBinding.aContexts.unshift(oTransientBindingContext);
		oBindingContextMock.expects("destroy").withExactArgs();
		oTransientBindingContextMock.expects("destroy").withExactArgs();
		oModelMock.expects("bindingDestroyed").withExactArgs(sinon.match.same(oBinding));
		oBindingMock.expects("destroy").on(oBinding).withExactArgs();
		oParentBindingPrototypeMock.expects("destroy").on(oBinding).withExactArgs();
		this.mock(oBinding.getHeaderContext()).expects("destroy").withExactArgs();

		// code under test
		oBinding.destroy();

		assert.strictEqual(oBinding.oDiff, undefined);
		assert.strictEqual(oBinding.oHeaderContext, undefined);
	});

	//*********************************************************************************************
	QUnit.test("destroyPreviousContexts: all", function (assert) {
		var oBinding = this.bindList("relative"),
			oContext1 = {
				isTransient : function () {},
				destroy : function () {}
			},
			oContext2 = {
				isTransient : function () {},
				destroy : function () {}
			},
			oContext3 = {
				isTransient : function () {},
				destroy : function () {}
			};

		oBinding.mPreviousContextsByPath = {p1 : oContext1, p2 : oContext2, p3 : oContext3};
		this.mock(oContext1).expects("isTransient").withExactArgs().returns(false);
		this.mock(oContext1).expects("destroy").withExactArgs();
		this.mock(oContext2).expects("isTransient").withExactArgs().returns(false);
		this.mock(oContext2).expects("destroy").withExactArgs();
		this.mock(oContext3).expects("isTransient").withExactArgs().returns(false);
		this.mock(oContext3).expects("destroy").withExactArgs();

		// code under test
		oBinding.destroyPreviousContexts();

		assert.deepEqual(oBinding.mPreviousContextsByPath, {});
	});

	//*********************************************************************************************
	QUnit.test("destroyPreviousContexts: selection", function (assert) {
		var oBinding = this.bindList("relative"),
			oContext1 = { // no flag
				destroy : function () {},
				isDeleted : function () {},
				isKeepAlive : function () {},
				isTransient : function () {}
			},
			oContext2 = { // keepAlive
				iIndex : 2,
				isKeepAlive : function () {}
			},
			oContext3 = { // deleted
				isDeleted : function () {},
				isKeepAlive : function () {}
			},
			oContext4 = { // transient
				isDeleted : function () {},
				isKeepAlive : function () {},
				isTransient : function () {}
			},
			oContext5 = {};

		oBinding.mPreviousContextsByPath
			= {p1 : oContext1, p2 : oContext2, p3 : oContext3, p4 : oContext4, p5 : oContext5};
		this.mock(oContext1).expects("isKeepAlive").withExactArgs().returns(false);
		this.mock(oContext1).expects("isDeleted").withExactArgs().returns(false);
		this.mock(oContext1).expects("isTransient").withExactArgs().returns(false);
		this.mock(oContext1).expects("destroy").withExactArgs();
		this.mock(oContext2).expects("isKeepAlive").withExactArgs().returns(true);
		this.mock(oContext3).expects("isKeepAlive").withExactArgs().returns(false);
		this.mock(oContext3).expects("isDeleted").withExactArgs().returns(true);
		this.mock(oContext4).expects("isKeepAlive").withExactArgs().returns(false);
		this.mock(oContext4).expects("isDeleted").withExactArgs().returns(false);
		this.mock(oContext4).expects("isTransient").withExactArgs().returns(true);

		// code under test
		oBinding.destroyPreviousContexts(["p1", "p2", "p3", "p4", "p6"]);

		assert.deepEqual(oBinding.mPreviousContextsByPath,
			{p2 : oContext2, p3 : oContext3, p5 : oContext5});
		assert.strictEqual(oContext2.iIndex, undefined);
	});

	//*********************************************************************************************
	QUnit.test("destroyPreviousContexts: binding already destroyed", function (assert) {
		var oBinding = this.bindList("relative");

		oBinding.destroy();

		// code under test - simulate a pre-rendering task after the binding was destroyed
		oBinding.destroyPreviousContexts();

		assert.strictEqual(oBinding.mPreviousContextsByPath, undefined);
	});

	//*********************************************************************************************
	QUnit.test("removeCreated", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oContext0 = Context.create(this.oModel, oBinding, "/EMPLOYEES($uid=id-1-23)", -1,
				SyncPromise.resolve(Promise.resolve())),
			oContext1 = Context.create(this.oModel, oBinding, "/EMPLOYEES($uid=id-1-24)", -2,
				SyncPromise.resolve()), // let's assume this is created, persisted, kept-alive
			oContext2 = Context.create(this.oModel, oBinding, "/EMPLOYEES($uid=id-1-25)", -3,
				SyncPromise.resolve(Promise.resolve()), /*bInactive*/true),
			oContext3 = Context.create(this.oModel, oBinding, "/EMPLOYEES($uid=id-1-26)", -4,
				SyncPromise.resolve(Promise.resolve()));

		// simulate 4 created entities
		oBinding.aContexts.unshift(oContext3, oContext2, oContext1, oContext0);
		oBinding.iActiveContexts = 3;
		oBinding.iCreatedContexts = 4;
		assert.strictEqual(oBinding.getLength(), 14, "length");
		this.mock(oContext1).expects("isKeepAlive").withExactArgs().returns(true);
		this.mock(oContext2).expects("isKeepAlive").withExactArgs().returns(false);
		this.mock(oBinding).expects("destroyLater").withExactArgs(sinon.match.same(oContext2));

		// code under test
		oBinding.removeCreated(oContext1);
		oBinding.removeCreated(oContext2);

		assert.strictEqual(oBinding.getLength(), 12);
		assert.strictEqual(oBinding.iActiveContexts, 2);
		assert.strictEqual(oBinding.iCreatedContexts, 2);
		assert.strictEqual(oBinding.aContexts[0], oContext3);
		assert.strictEqual(oContext3.getIndex(), 0);
		assert.strictEqual(oBinding.aContexts[1], oContext0);
		assert.strictEqual(oContext0.getIndex(), 1);

		return Promise.all([
			oContext0.created(),
			oContext1.created(),
			oContext2.created(),
			oContext3.created()
		]);
	});

	//*********************************************************************************************
[0, 1].forEach(function (iCurrentEnd) {
	var sTitle = "destroyLater: iCurrentEnd=" + iCurrentEnd;

	QUnit.test(sTitle, function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oContext = Context.create(this.oModel, oBinding, "/EMPLOYEES($uid=id-1-24)", -2,
				SyncPromise.resolve());

		oBinding.iCurrentEnd = iCurrentEnd;
		this.mock(oContext).expects("destroy").exactly(iCurrentEnd ? 0 : 1)
			.withExactArgs();

		// code under test
		oBinding.destroyLater(oContext);

		assert.strictEqual(oBinding.mPreviousContextsByPath["/EMPLOYEES($uid=id-1-24)"],
			iCurrentEnd ? oContext : undefined);
	});
});

	//*********************************************************************************************
	[{
		sInit : "base", sTarget : undefined
	}, {
		sInit : "base", sTarget : "base"
	}, {
		sInit : "base", sTarget : "v4"
	}, {
		sInit : "v4", sTarget : "base"
	}, {
		sInit : undefined, sTarget : "base"
	}].forEach(function (oFixture) {
		QUnit.test("change context:" + oFixture.sInit + "->" + oFixture.sTarget, function (assert) {
			var oModel = this.oModel,
				oInitialContext = createContext(oFixture.sInit, "/EMPLOYEES(ID='1')"),
				oBinding,
				oTargetCache = oFixture.sTarget ? {} : undefined,
				oTargetContext = createContext(oFixture.sTarget, "/EMPLOYEES(ID='2')");

			function createContext(sType, sPath) {
				if (sType === "base") {
					return oModel.createBindingContext(sPath);
				}
				if (sType === "v4") {
					return Context.create(oModel, oParentBinding, sPath);
				}

				return undefined;
			}

			this.mock(ODataListBinding.prototype).expects("fetchCache").atLeast(1)
				.callsFake(function () {
					this.oCache = oTargetCache;
					this.oCachePromise = SyncPromise.resolve(oTargetCache);
				});
			this.mock(ODataListBinding.prototype).expects("restoreCreated").atLeast(1)
				.withExactArgs();
			oBinding = oModel.bindList("Equipments", oInitialContext);
			this.mock(oBinding).expects("checkSuspended").withExactArgs(true);

			// code under test
			oBinding.setContext(oTargetContext);

			assert.strictEqual(oBinding.oCachePromise.getResult(), oTargetCache);
		});
	});

	//*********************************************************************************************
	[false, true].forEach(function (bCreated) {
		[false, true].forEach(function (bUsePredicates) {
			var sTitle = "createContexts, bCreated = " + bCreated
					+ ", bUsePredicates = " + bUsePredicates;

			QUnit.test(sTitle, function (assert) {
				var oBinding = this.bindList("/EMPLOYEES", {/*oContext*/}),
					aContexts = [{}, {}, {}],
					oContextMock = this.mock(Context),
					i,
					sPath,
					aResults = [{}, {}, {}],
					iServerIndex,
					iStart = 2;

				if (bUsePredicates) {
					aResults.forEach(function (vValue, i) {
						_Helper.setPrivateAnnotation(vValue, "predicate", "('" + i + "')");
					});
				}
				if (bCreated) {
					oBinding.aContexts.unshift({/*created*/});
					oBinding.iCreatedContexts += 1;
				}
				this.mock(oBinding).expects("getResolvedPath").twice().withExactArgs()
					.returns("~resolved~");
				for (i = iStart; i < iStart + aResults.length; i += 1) {
					iServerIndex = bCreated ? i - 1 : i;
					sPath = "~resolved~" + (bUsePredicates
						? _Helper.getPrivateAnnotation(aResults[i - iStart], "predicate")
						: "/" + iServerIndex);
					oContextMock.expects("create")
						.withExactArgs(sinon.match.same(this.oModel), sinon.match.same(oBinding),
							sPath, iServerIndex)
						.returns(aContexts[i - iStart]);
				}

				// code under test
				assert.strictEqual(oBinding.createContexts(iStart, aResults), true);

				for (i = iStart; i < iStart + aResults.length; i += 1) {
					assert.strictEqual(oBinding.aContexts[i], aContexts[i - iStart]);
				}

				// code under test : no second change event
				assert.strictEqual(oBinding.createContexts(iStart, aResults), false);
			});
		});
	});

	//*********************************************************************************************
	[false, true].forEach(function (bCreated) {
		var sTitle = "createContexts, paging: less data than requested; w/ created: " + bCreated;

		QUnit.test(sTitle, function (assert) {
			var oBinding = this.bindList("/EMPLOYEES", {/*oContext*/}),
				iCreatedContexts = bCreated ? 2 : 0,
				i,
				aResults;

			function result(iLength, iCount) {
				// only active created contexts add to $count
				iCount = iCount && iCount + (bCreated ? 1 : 0);
				return createData(iLength, 0, true, iCount);
			}

			assert.strictEqual(oBinding.isLengthFinal(), false);
			assert.strictEqual(oBinding.getLength(), 0, "Initial estimated length is 0");
			assert.strictEqual(oBinding.iMaxLength, Infinity);

			if (bCreated) {
				// simulate an active (poss. persisted) and an inactive created entity
				oBinding.aContexts.unshift({});
				oBinding.aContexts.unshift({});
				oBinding.iActiveContexts = 1;
				oBinding.iCreatedContexts = 2;
			}

			// code under test: set length and length final flag
			// Note: short reads are handled by _Cache and set $count!
			assert.strictEqual(
				oBinding.createContexts(20 + iCreatedContexts, result(29, 20 + 29)),
				true);

			assert.strictEqual(oBinding.bLengthFinal, true,
				"some controls use bLengthFinal instead of isLengthFinal()");
			assert.strictEqual(oBinding.getLength(), 49 + iCreatedContexts);
			assert.strictEqual(oBinding.aContexts.length, 49 + iCreatedContexts);
			assert.strictEqual(oBinding.iMaxLength, 49);

			for (i = 37; i < 49; i += 1) {
				this.mock(oBinding.aContexts[i + iCreatedContexts]).expects("destroy")
					.withExactArgs();
			}
			// code under test: delete obsolete contexts
			assert.strictEqual(
				oBinding.createContexts(20 + iCreatedContexts, result(17, 20 + 17)),
				true);

			assert.strictEqual(oBinding.isLengthFinal(), true);
			assert.strictEqual(oBinding.getLength(), 37 + iCreatedContexts);
			assert.strictEqual(oBinding.aContexts.length, 37 + iCreatedContexts);
			assert.strictEqual(oBinding.iMaxLength, 37);

			// code under test
			assert.strictEqual(
				oBinding.createContexts(20 + iCreatedContexts, result(17)),
				false,
				"do not modify upper boundary if same data is read (no short read)");

			assert.strictEqual(oBinding.isLengthFinal(), true);
			assert.strictEqual(oBinding.getLength(), 37 + iCreatedContexts);
			assert.strictEqual(oBinding.aContexts.length, 37 + iCreatedContexts);
			assert.strictEqual(oBinding.iMaxLength, 37);

			// code under test: reset upper boundary
//TODO cannot happen with our _Cache; _Cache doesn't read more than final length elements
			assert.strictEqual(
				oBinding.createContexts(20 + iCreatedContexts, result(30)),
				true);

			assert.strictEqual(oBinding.isLengthFinal(), false);
			assert.strictEqual(oBinding.getLength(), 60 + iCreatedContexts);
			assert.strictEqual(oBinding.aContexts.length, 50 + iCreatedContexts);
			assert.strictEqual(oBinding.iMaxLength, Infinity);

			// code under test: no data for some other page is not a change
			assert.strictEqual(
				oBinding.createContexts(10000 + iCreatedContexts, result(0)),
				false);

			assert.strictEqual(oBinding.isLengthFinal(), false);
			assert.strictEqual(oBinding.getLength(), 60 + iCreatedContexts);
			assert.strictEqual(oBinding.aContexts.length, 50 + iCreatedContexts);
			assert.strictEqual(oBinding.iMaxLength, 10000);
//TODO iMaxLength must be set if iResultLength > 0 || iResultLength === 0 && oRange.start === 0;
// or oRange.start is just after the last known good;
//TODO it can only shrink if iResultLength === 0

			// code under test: no data for *next* page is a change (bLengthFinal changes)
			assert.strictEqual(
				oBinding.createContexts(50 + iCreatedContexts, result(0)),
				true);

			assert.strictEqual(oBinding.isLengthFinal(), true);
			assert.strictEqual(oBinding.getLength(), 50 + iCreatedContexts);
			assert.strictEqual(oBinding.aContexts.length, 50 + iCreatedContexts);
			assert.strictEqual(oBinding.iMaxLength, 50);

			// code under test
			assert.strictEqual(
				oBinding.createContexts(30 + iCreatedContexts, result(0)),
				true);

			assert.strictEqual(oBinding.isLengthFinal(), true);
			assert.strictEqual(oBinding.getLength(), 30 + iCreatedContexts);
			assert.strictEqual(oBinding.aContexts.length, 30 + iCreatedContexts);
			assert.strictEqual(oBinding.iMaxLength, 30);

			// code under test: preparation for following test for server-side paging: create a gap
			assert.strictEqual(
				oBinding.createContexts(100 + iCreatedContexts, result(20)),
				true);

			assert.strictEqual(oBinding.isLengthFinal(), false);
			assert.strictEqual(oBinding.getLength(), 120 + iCreatedContexts + 10);
			assert.strictEqual(oBinding.aContexts.length, 120 + iCreatedContexts);
			assert.strictEqual(oBinding.iMaxLength, Infinity);

			aResults = result(140);
			for (i = 50; i < 100; i += 1) {
				delete aResults[i];
			}

			// code under test: gap is not read completely
			assert.strictEqual(
				oBinding.createContexts(0 + iCreatedContexts, aResults),
				true);

			assert.strictEqual(oBinding.isLengthFinal(), false);
			assert.strictEqual(oBinding.getLength(), 140 + iCreatedContexts + 10);
			assert.strictEqual(oBinding.aContexts.length, 140 + iCreatedContexts);
			assert.strictEqual(oBinding.iMaxLength, Infinity);
		});
	});

	//*********************************************************************************************
	QUnit.test("createContexts, reuse previous contexts", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES", {/*oContext*/}),
			oContext1 = Context.create(this.oModel, oBinding, "/EMPLOYEES/1", 1),
			oContext2 = Context.create(this.oModel, oBinding, "/EMPLOYEES/2", 2),
			oContext3 = {},
			oContextMock = this.mock(Context);

		oBinding.mPreviousContextsByPath = {
			"/EMPLOYEES/0" : {},
			"/EMPLOYEES/1" : oContext1,
			"/EMPLOYEES/2" : oContext2
		};
		this.mock(oContext1).expects("checkUpdate").withExactArgs();
		this.mock(oContext2).expects("checkUpdate").withExactArgs();
		oContextMock.expects("create")
			.withExactArgs(sinon.match.same(this.oModel), sinon.match.same(oBinding),
				"/EMPLOYEES/3", 3)
			.returns(oContext3);
		this.mock(this.oModel).expects("addPrerenderingTask")
			.withExactArgs(sinon.match.func).callsArg(0);
		this.mock(oBinding).expects("destroyPreviousContexts").withExactArgs(["/EMPLOYEES/0"]);

		// code under test
		oBinding.createContexts(1, [{}, {}, {}]);

		assert.strictEqual(oBinding.aContexts[1], oContext1);
		assert.strictEqual(oBinding.aContexts[2], oContext2);
		assert.strictEqual(oBinding.aContexts[3], oContext3);
	});

	//*********************************************************************************************
	QUnit.test("createContexts w/ keyPredicates, reuse previous contexts", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES", {/*oContext*/}),
			oBindingMock = this.mock(oBinding),
			oContext0 = {},
			oContext1 = Context.create(this.oModel, oBinding, "/EMPLOYEES('B')", 99),
			oContext2 = Context.create(this.oModel, oBinding, "/EMPLOYEES('C')", 1),
			oContext3 = {},
			oContextMock = this.mock(Context),
			oTaskMock;

		// must be mocked here, so that later bind grabs the mock
		oBindingMock.expects("destroyPreviousContexts").never();
		assert.deepEqual(oBinding.aContexts, [], "binding is reset");
		oBinding.iCreatedContexts = 2; // reset might keep some
		oBinding.mPreviousContextsByPath = {
			"/EMPLOYEES('A')" : oContext0,
			"/EMPLOYEES('B')" : oContext1,
			"/EMPLOYEES('D')" : oContext3
		};
		this.mock(oContext1).expects("destroy").never();
		this.mock(oContext1).expects("checkUpdate").withExactArgs();
		oContextMock.expects("create")
			.withExactArgs(sinon.match.same(this.oModel), sinon.match.same(oBinding),
				"/EMPLOYEES('C')", 1)
			.returns(oContext2);
		oTaskMock = this.mock(this.oModel).expects("addPrerenderingTask")
			.withExactArgs(sinon.match.func);

		// code under test
		oBinding.createContexts(2, [{
			"@$ui5._" : {predicate : "('B')"}
		}, {
			"@$ui5._" : {predicate : "('C')"}
		}]);

		assert.strictEqual(oBinding.aContexts[2], oContext1);
		assert.strictEqual(oBinding.aContexts[3], oContext2);
		assert.strictEqual(oContext1.getModelIndex(), 2);
		assert.strictEqual(oContext2.getModelIndex(), 3);
		assert.strictEqual(oContext1.iIndex, 0);
		assert.strictEqual(oContext2.iIndex, 1);
		assert.strictEqual(Object.keys(oBinding.mPreviousContextsByPath).length, 2);
		assert.strictEqual(oBinding.mPreviousContextsByPath["/EMPLOYEES('A')"], oContext0);
		assert.strictEqual(oBinding.mPreviousContextsByPath["/EMPLOYEES('D')"], oContext3);

		oBindingMock.expects("destroyPreviousContexts")
			.withExactArgs(["/EMPLOYEES('A')", "/EMPLOYEES('D')"]);

		// code under test
		oTaskMock.firstCall.args[0]();
	});

	//*********************************************************************************************
	QUnit.test("createContexts, no prerendering task if no previous contexts", function () {
		var oBinding = this.bindList("/EMPLOYEES", {});

		this.mock(this.oModel).expects("addPrerenderingTask").never();

		// code under test
		oBinding.createContexts(1, 0);
	});

	//*********************************************************************************************
	QUnit.test("createContexts: shrink contexts", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES", {}),
			oContext = {
				destroy : function () {}
			},
			aResults = [];

		aResults.$count = 1;
		oBinding.aContexts = [, , oContext];

		this.mock(oContext).expects("destroy").withExactArgs();

		// code under test
		oBinding.createContexts(1, aResults);

		assert.deepEqual(oBinding.aContexts, []);
		assert.strictEqual(oBinding.bLengthFinal, true);
		assert.strictEqual(oBinding.iMaxLength, 1);
	});

	//*********************************************************************************************
	QUnit.test("createContexts: do not reuse a created context", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oCreatedContext = Context.create(this.oModel, oBinding, "/EMPLOYEES('1')", -1,
				SyncPromise.resolve()),
			oNewContext = {};

		oBinding.mPreviousContextsByPath = {
			"/EMPLOYEES('1')" : oCreatedContext
		};

		this.mock(Context).expects("create")
			.withExactArgs(sinon.match.same(this.oModel), sinon.match.same(oBinding),
				"/EMPLOYEES('1')", 0)
			.returns(oNewContext);
		this.mock(this.oModel).expects("addPrerenderingTask")
			.withExactArgs(sinon.match.func).callsArg(0);
		this.mock(oCreatedContext).expects("destroy").withExactArgs();

		oBinding.createContexts(0, [{
			"@$ui5._" : {predicate : "('1')"}
		}]);

		assert.strictEqual(oBinding.aContexts[0], oNewContext);
	});

	//*********************************************************************************************
	QUnit.test("createContexts: reuse a created context if kept alive", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oCreatedContext = Context.create(this.oModel, oBinding, "/EMPLOYEES('1')", -1,
				SyncPromise.resolve());

		oCreatedContext.setKeepAlive(true);
		oBinding.mPreviousContextsByPath = {
			"/EMPLOYEES('1')" : oCreatedContext
		};
		this.mock(Context).expects("create").never();
		this.mock(this.oModel).expects("addPrerenderingTask").never();
		this.mock(oCreatedContext).expects("destroy").never();
		this.mock(oCreatedContext).expects("checkUpdate").withExactArgs();

		// code under test
		oBinding.createContexts(0, [{
			"@$ui5._" : {predicate : "('1')"}
		}]);

		assert.strictEqual(oBinding.aContexts[0], oCreatedContext);
		assert.strictEqual(oCreatedContext.getModelIndex(), 0);
	});

	//*********************************************************************************************
[false, true].forEach(function (bSuccess) {
	[false, true].forEach(function (bCreated) { // the deleted context is created-persisted
		var sTitle = "_delete: success=" + bSuccess + ", created=" + bCreated;

		QUnit.test(sTitle, function (assert) {
			var oBinding = this.bindList("/EMPLOYEES"),
				oBindingMock = this.mock(oBinding),
				oContext1,
				oContext1Mock,
				sContext1Path,
				aData = createData(5, 0, true, undefined, true),
				aData2 = aData.slice(4, 5),
				oDeleteCall,
				fnResolve,
				fnReject,
				oDeleteFromCachePromise = new Promise(function (resolve, reject) {
					fnResolve = resolve;
					fnReject = reject;
				}),
				oETagEntity = {},
				sPath = "1",
				aPreviousContexts,
				oPromise,
				oTaskExpectation,
				that = this;

			oBinding.createContexts(0, aData.slice(0, 3));
			aData2.$count = 5; // non-empty short read adds $count
			oBinding.createContexts(4, aData2);
			// aContexts now is [0, 1, 2, undefined, 4]
			oBinding.iDeletedContexts = 3;
			oContext1 = oBinding.aContexts[1];
			oContext1Mock = this.mock(oContext1);
			sContext1Path = oContext1.getPath();
			if (bCreated) { // fake a created context: it needs a negative index
				oContext1.iIndex = -1;
				sPath = "-1";
			}
			oBindingMock.expects("destroyPreviousContexts").never();
			oContext1Mock.expects("resetKeepAlive").never();
			oDeleteCall = oBindingMock.expects("deleteFromCache")
				.withExactArgs("myGroup", "EMPLOYEES('1')", sPath, sinon.match.same(oETagEntity),
					sinon.match.func)
				.callsFake(function () {
					// Although delete works with existing cache data and the cache immediately
					// calls back, it is yet possibly asynchronous (oCachePromise, fetchValue).
					// So we add a created context here, and the index becomes 2, although we
					// started with index 1.
					oBinding.aContexts.unshift({
						getModelIndex : function () { return 0; } // called below, by ourselves
					}); // [-1, 0, 1, 2, undefined, 4]
					oBinding.iCreatedContexts = 1;
					oBinding.iActiveContexts = 1;
					aPreviousContexts = oBinding.aContexts.slice();
					assert.strictEqual(oBinding.getLength(), 6);

					arguments[4](2, -1); // now call the callback with the adjusted index

					// expectations for then
					oContext1Mock.expects("resetKeepAlive").exactly(bSuccess ? 1 : 0)
						.withExactArgs();
					oTaskExpectation = that.mock(that.oModel).expects("addPrerenderingTask")
						.exactly(bSuccess ? 1 : 0).withExactArgs(sinon.match.func);
					// expectations for catch
					oBindingMock.expects("_fireChange").exactly(bSuccess ? 0 : 1)
						.withExactArgs({reason : ChangeReason.Add});

					return oDeleteFromCachePromise;
				});
			oBinding.aContexts.forEach(function (oContext) {
				that.mock(oContext).expects("destroy").never();
			});
			oBindingMock.expects("_fireChange")
				.withExactArgs({reason : ChangeReason.Remove})
				.callsFake(function () {
					// aContexts : [-1, 0, 1, 2, undefined, 4] -> [-1, 0, 2, undefined, 4]
					assert.strictEqual(oBinding.getLength(), 5);
					assert.strictEqual(oBinding.aContexts.length, 5);
					assert.strictEqual(oBinding.iCreatedContexts, bCreated ? 0 : 1);
					assert.strictEqual(oBinding.iActiveContexts, bCreated ? 0 : 1);
					assert.strictEqual(oBinding.aContexts[0], aPreviousContexts[0]);
					assert.strictEqual(oBinding.aContexts[1], aPreviousContexts[1]);
					assert.strictEqual(oBinding.aContexts[2], aPreviousContexts[3]);
					assert.notOk(3 in oBinding.aContexts);
					assert.strictEqual(oBinding.aContexts[4], aPreviousContexts[5]);
					assert.strictEqual(
						oBinding.mPreviousContextsByPath[oContext1.getPath()],
						oContext1);
					assert.strictEqual(oContext1.iIndex, undefined);
					oBinding.aContexts.forEach(function (oContext, i) {
						assert.strictEqual(oContext.getModelIndex(), i);
					});

					// This assures that the change event must come before deleteFromCache finished
					if (bSuccess) {
						fnResolve();
					} else {
						oDeleteCall.args[0][4](2, 1); // call the callback for the re-insertion

						// aContexts : [-1, 0, 2, undefined, 4] -> [-1, 0, 1, 2, undefined, 4]
						assert.strictEqual(oBinding.getLength(), 6);
						assert.strictEqual(oBinding.aContexts.length, 6);
						assert.strictEqual(oBinding.iCreatedContexts, 1);
						assert.strictEqual(oBinding.iActiveContexts, 1);
						assert.strictEqual(oBinding.aContexts[0], aPreviousContexts[0]);
						assert.strictEqual(oBinding.aContexts[1], aPreviousContexts[1]);
						assert.strictEqual(oBinding.aContexts[2], aPreviousContexts[2]);
						assert.strictEqual(oBinding.aContexts[3], aPreviousContexts[3]);
						assert.notOk(4 in oBinding.aContexts);
						assert.strictEqual(oBinding.aContexts[5], aPreviousContexts[5]);
						assert.notOk(oContext1.getPath() in oBinding.mPreviousContextsByPath);
						oBinding.aContexts.forEach(function (oContext, i) {
							assert.strictEqual(oContext.getModelIndex(), i);
						});

						fnReject("~oError~");
					}
				});

			// code under test
			oPromise = oBinding._delete("myGroup", "EMPLOYEES('1')", oContext1, oETagEntity,
				"~bDoNotRequestCount~");

			assert.strictEqual(oBinding.iDeletedContexts, 4);

			return oPromise.then(function () {
				assert.ok(bSuccess);
				assert.strictEqual(oBinding.iDeletedContexts, 3);
				assert.strictEqual(oContext1.iIndex, Context.VIRTUAL);

				oBindingMock.expects("destroyPreviousContexts").withExactArgs([sContext1Path]);

				// code under test - run the prerendering task
				oTaskExpectation.args[0][0]();
			}, function (oError) {
				assert.notOk(bSuccess);
				assert.strictEqual(oError, "~oError~");
				assert.strictEqual(oBinding.iDeletedContexts, 3);
			});
		});
	});
});
	//TODO check the row of a pending update with higher index

	//*********************************************************************************************
[
	{lengthFinal : false},
	{lengthFinal : true, error : true},
	{lengthFinal : true, noGroup : true},
	{lengthFinal : true, noGroup : true, newMaxLength : 42},
	{lengthFinal : true, apiGroup : false, newMaxLength : 42},
	{lengthFinal : true, apiGroup : true, newMaxLength : 41}
].forEach(function (oFixture) {
	var sTitle = "_delete: kept-alive context not in the collection: " + JSON.stringify(oFixture);

	// we assume 42 entities matching the filter plus 2 created entities
	QUnit.test(sTitle, function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oBindingMock = this.mock(oBinding),
			oBindingResetCall,
			oCacheResetCall,
			aContexts = [{iIndex : -2}, {iIndex : -1}, {iIndex : 0}, {iIndex : 1}],
			oCountPromise = Promise.resolve(oFixture.newMaxLength + 1),
			oDeleteFromCacheExpectation,
			bFireChange = oFixture.newMaxLength === 41,
			oGroupLock = oFixture.noGroup
				? null
				: {
					getGroupId : function () {},
					getUnlockedCopy : function () {}
				},
			oHelperMock = this.mock(_Helper),
			oKeptAliveContext = {
				iIndex : undefined,
				created : function () { return undefined; },
				getPath : function () { return "~contextPath~"; },
				resetKeepAlive : function () {}
			},
			iOldMaxLength = oFixture.lengthFinal ? 42 : Infinity,
			oPromise,
			oTaskExpectation,
			that = this;

		// simulate an active and an inactive created entity
		oBinding.aContexts = aContexts;
		oBinding.iActiveContexts = 1;
		oBinding.iCreatedContexts = 2;
		oBinding.bLengthFinal = oFixture.lengthFinal;
		oBinding.iMaxLength = iOldMaxLength;
		oBinding.mPreviousContextsByPath = {
			"~contextPath~" : oKeptAliveContext
		};

		oBindingMock.expects("destroyPreviousContexts").never();
		oHelperMock.expects("getRelativePath")
			.withExactArgs("~contextPath~", "/EMPLOYEES").returns("~predicate~");
		oDeleteFromCacheExpectation = oBindingMock.expects("deleteFromCache")
			.withExactArgs(sinon.match.same(oGroupLock), "EMPLOYEES('1')", "~predicate~",
				"oETagEntity", sinon.match.func)
			.returns(oCountPromise.then(function () {
				if (oFixture.error) {
					that.mock(oBinding).expects("getKeepAlivePredicates").withExactArgs()
						.returns("~predicates~");
					oCacheResetCall = that.mock(oBinding.oCache).expects("reset")
						.withExactArgs("~predicates~");
					oBindingResetCall = oBindingMock.expects("reset")
						.withExactArgs(ChangeReason.Change);
					throw "~oError~";
				}
				that.mock(oKeptAliveContext).expects("resetKeepAlive").withExactArgs();
				oTaskExpectation = that.mock(that.oModel).expects("addPrerenderingTask")
					.withExactArgs(sinon.match.func);
			}));

		// code under test
		oPromise = oBinding._delete(oGroupLock, "EMPLOYEES('1')", oKeptAliveContext, "oETagEntity",
			oFixture.error || !oFixture.newMaxLength);

		if (oGroupLock) {
			this.mock(oGroupLock).expects("getGroupId").exactly(oFixture.newMaxLength ? 1 : 0)
				.withExactArgs().returns("group");
			this.mock(this.oModel).expects("isApiGroup").exactly(oFixture.newMaxLength ? 1 : 0)
				.withExactArgs("group").returns(oFixture.apiGroup);
			this.mock(oGroupLock).expects("getUnlockedCopy")
				.exactly(oFixture.apiGroup === false ? 1 : 0)
				.withExactArgs().returns("~groupLock~");
		}
		oBindingMock.expects("lockGroup")
			.exactly((!oGroupLock || oFixture.apiGroup) && oFixture.newMaxLength ? 1 : 0)
			.withExactArgs("$auto").returns("~groupLock~");
		this.mock(oBinding.oCache).expects("requestCount").exactly(oFixture.newMaxLength ? 1 : 0)
			.withExactArgs("~groupLock~")
			.returns(oCountPromise);
		oBindingMock.expects("_fireChange").exactly(bFireChange ? 1 : 0)
			.withExactArgs({reason : ChangeReason.Remove})
			.callsFake(function () {
				assert.strictEqual(oBinding.iMaxLength, 41);
			});

		// code under test - callback
		oDeleteFromCacheExpectation.args[0][4](undefined, -1);

		if (oFixture.error) {
			// code under test - callback for reinsertion
			oDeleteFromCacheExpectation.args[0][4](undefined, 1);
		}
		return oPromise.then(function () {
			assert.deepEqual(oBinding.aContexts, aContexts);
			assert.strictEqual(oBinding.iMaxLength, oFixture.newMaxLength || iOldMaxLength);

			oBindingMock.expects("destroyPreviousContexts").withExactArgs(["~contextPath~"]);

			// code under test - run the prerendering task
			oTaskExpectation.args[0][0]();
		}, function (oError) {
			assert.strictEqual(oError, "~oError~");

			assert.ok(oBindingResetCall.calledAfter(oCacheResetCall));
		});
	});
});
	//*********************************************************************************************
	QUnit.test("create: callbacks and eventing", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oBindingMock = this.mock(oBinding),
			oContext0,
			oContext1,
			oCreateInCacheExpectation,
			oCreateInCachePromise0 = Promise.resolve({}),
			oCreateInCachePromise1 = Promise.resolve({}),
			oCreatePathPromise = SyncPromise.resolve("~"),
			oError = {},
			oInitialData0 = {},
			oInitialData1 = {},
			oGroupLock0 = {},
			oGroupLock1 = {},
			oLockGroupExpectation,
			oPromise,
			that = this;

		oBindingMock.expects("getUpdateGroupId").withExactArgs().returns("~update~");
		oLockGroupExpectation = oBindingMock.expects("lockGroup")
			.withExactArgs("~update~", true, true, sinon.match.func)
			.returns(oGroupLock0);
		oBindingMock.expects("fetchResourcePath").withExactArgs().returns(oCreatePathPromise);
		oCreateInCacheExpectation = oBindingMock.expects("createInCache")
			.withExactArgs(sinon.match.same(oGroupLock0), sinon.match.same(oCreatePathPromise),
				"/EMPLOYEES", sinon.match(rTransientPredicate), sinon.match.same(oInitialData0),
				false, sinon.match.func, sinon.match.func)
			.returns(SyncPromise.resolve(oCreateInCachePromise0));
		oCreateInCachePromise0.then(function () {
			that.mock(oContext0).expects("refreshDependentBindings")
				.withExactArgs(sinon.match(/EMPLOYEES\(\$uid=.+\)/), "$auto", true);
		});
		this.mock(oContextPrototype).expects("fetchValue").twice().withExactArgs().resolves({});

		// code under test (create first entity, skip refresh)
		oContext0 = oBinding.create(oInitialData0, true);

		assert.strictEqual(oBinding.iCreatedContexts, 1);
		assert.strictEqual(oBinding.iActiveContexts, 1);
		assert.strictEqual(oBinding.aContexts[0], oContext0);
		assert.strictEqual(oContext0.getIndex(), 0);
		assert.strictEqual(oContext0.iIndex, -1);

		oBindingMock.expects("getUpdateGroupId").withExactArgs().returns("~update~");
		oBindingMock.expects("lockGroup")
			.withExactArgs("~update~", true, true, sinon.match.func)
			.returns(oGroupLock1);
		oBindingMock.expects("fetchResourcePath").withExactArgs().returns(oCreatePathPromise);
		oBindingMock.expects("createInCache")
			.withExactArgs(sinon.match.same(oGroupLock1), sinon.match.same(oCreatePathPromise),
				"/EMPLOYEES", sinon.match(rTransientPredicate), sinon.match.same(oInitialData1),
				false, sinon.match.func, sinon.match.func)
			.returns(SyncPromise.resolve(oCreateInCachePromise1));
		oCreateInCachePromise1.then(function () {
			that.mock(oContext1).expects("refreshDependentBindings")
				.withExactArgs(sinon.match(/EMPLOYEES\(\$uid=.+\)/), "$auto", true);
		});

		// code under test (create second entity, skip refresh)
		oContext1 = oBinding.create(oInitialData1, true);

		assert.strictEqual(oBinding.iCreatedContexts, 2);
		assert.strictEqual(oBinding.iActiveContexts, 2);
		assert.strictEqual(oBinding.aContexts[0], oContext1);
		assert.strictEqual(oContext1.getIndex(), 0);
		assert.strictEqual(oContext1.iIndex, -2);

		assert.strictEqual(oBinding.aContexts[1], oContext0);
		assert.strictEqual(oContext0.getIndex(), 1);
		assert.strictEqual(oContext0.iIndex, -1);

		oBindingMock.expects("fireEvent").on(oBinding)
			.withExactArgs("createSent", {context : sinon.match.same(oContext0)});

		// code under test
		oCreateInCacheExpectation.args[0][7](); // call fnSubmitCallback

		this.mock(oBinding.oModel).expects("reportError")
			.withExactArgs("POST on '~' failed; will be repeated automatically", sClassName,
				sinon.match.same(oError));
		oBindingMock.expects("fireEvent").on(oBinding)
			.withExactArgs("createCompleted",
				{context : sinon.match.same(oContext0), success : false});

		// code under test - call fnErrorCallback
		oCreateInCacheExpectation.args[0][6](oError);

		oBindingMock.expects("removeCreated").withExactArgs(sinon.match.same(oContext0));

		// code under test - call fnCancelCallback to simulate cancellation
		oPromise = oLockGroupExpectation.args[0][3]();

		// expect the event to be fired asynchronously
		oBindingMock.expects("_fireChange").withExactArgs({reason : ChangeReason.Remove});

		oBinding.aContexts = [];
		that.mock(oContext0).expects("destroy").withExactArgs();

		// code under test - call fnCancelCallback to simulate cancellation
		// artificial scenario - oContext0 is not in aContexts. this can only happen in relative
		// bindings where the parent context was changed via #setContext.
		assert.strictEqual(oLockGroupExpectation.args[0][3](), undefined);

		oBindingMock.expects("fireEvent").on(oBinding)
			.withExactArgs("createCompleted",
				{context : sinon.match.same(oContext0), success : true});
		oBindingMock.expects("fireEvent").on(oBinding)
			.withExactArgs("createCompleted",
				{context : sinon.match.same(oContext1), success : true});

		return SyncPromise.all([
			oPromise,
			oContext0.created(),
			oContext1.created()
		]);
	});

	//*********************************************************************************************
	[{
		sGroupId : "$auto",
		sTitle : "create: absolute"
	}, {
		sGroupId : "$auto",
		oInitialData : {},
		sTitle : "create: absolute, with initial data"
	}, {
		sGroupId : "deferred",
		bRelative : true,
		sTitle : "create: relative with base context"
	}, {
		sGroupId : "$direct",
		sTitle : "create: absolute with groupId=$direct"
	}, {
		sGroupId : "$auto",
		bInactive : true,
		sTitle : "create: inactive, with $auto groupId"
	}, {
		sGroupId : "deferred",
		bInactive : true,
		sTitle : "create: inactive, with deferred groupId"
	}].forEach(function (oFixture) {
		QUnit.test(oFixture.sTitle, function (assert) {
			var oBinding,
				oBindingContext = this.oModel.createBindingContext("/"),
				oBindingMock,
				iChangeFired = 0,
				oContextMock = this.mock(oContextPrototype),
				aContexts = [],
				iCreateNo = 0,
				oCreatePathPromise = {},
				aCreatePromises = [
					SyncPromise.resolve(Promise.resolve({})),
					SyncPromise.resolve(Promise.resolve({}))
				],
				oModelMock = this.mock(this.oModel),
				aRefreshSingleFinished = [false, false],
				aRefreshSinglePromises = [
					new Promise(function (resolve) {
						// ensure that it is finished after all Promises
						setTimeout(resolve.bind(null, {}), 0);
					}),
					new Promise(function (resolve) {
						// ensure that it is finished after all Promises
						setTimeout(resolve.bind(null, {}), 0);
					})
				],
				that = this;

			function checkCreatedContext() {
				var oCreatedContext = aContexts[iCreateNo];

				assert.strictEqual(oCreatedContext.getModel(), that.oModel);
				assert.strictEqual(oCreatedContext.getBinding(), oBinding);
				assert.ok(/^\/EMPLOYEES\(\$uid=.+\)$/, "path with uid");
				assert.strictEqual(oCreatedContext.getModelIndex(), 0);
				assert.strictEqual(oCreatedContext.isInactive(), oFixture.bInactive);
				assert.strictEqual(oCreatedContext.isTransient(), true);
				assert.strictEqual(oBinding.iMaxLength, 42, "transient contexts are not counted");
				assert.strictEqual(oBinding.aContexts[0], oCreatedContext, "Transient context");
				assert.strictEqual(iChangeFired, iCreateNo + 1, "Change event fired");
			}

			function expect() {
				var oGroupLock = {},
					iCurrentCreateNo = iCreateNo;

				oBindingMock.expects("checkSuspended").withExactArgs();
				oBindingMock.expects("getGroupId").returns(oFixture.sGroupId || "$auto");
				oModelMock.expects("isApiGroup").withExactArgs(oFixture.sGroupId || "$auto")
					.returns(oFixture.sGroupId === "deferred");
				oBindingMock.expects("getUpdateGroupId").withExactArgs().returns(oFixture.sGroupId);
				oBindingMock.expects("lockGroup")
					.withExactArgs(oFixture.bInactive
							? "$inactive." + oFixture.sGroupId
							: oFixture.sGroupId,
						true, true, sinon.match.func)
					.returns(oGroupLock);
				oBindingMock.expects("fetchResourcePath").withExactArgs()
					.returns(oCreatePathPromise);
				oBindingMock.expects("createInCache")
					.withExactArgs(sinon.match.same(oGroupLock),
						sinon.match.same(oCreatePathPromise), "/EMPLOYEES",
						sinon.match(rTransientPredicate), sinon.match.same(oFixture.oInitialData),
						false, sinon.match.func, sinon.match.func)
					.returns(aCreatePromises[iCurrentCreateNo]);
				oContextMock.expects("fetchValue").withExactArgs().resolves({});

				aCreatePromises[iCurrentCreateNo].then(function () {
					oBindingMock.expects("lockGroup")
						.withExactArgs(oFixture.sGroupId === "$direct" ? "$direct" : "$auto")
						.returns(oGroupLock);
					oBindingMock.expects("fireEvent")
						.withExactArgs("createCompleted", {
							context : sinon.match.same(aContexts[iCurrentCreateNo]),
							success : true
						});
					oBindingMock.expects("refreshSingle")
						.withExactArgs(sinon.match.same(aContexts[iCurrentCreateNo]),
							sinon.match.same(oGroupLock))
						.returns(aRefreshSinglePromises[iCurrentCreateNo]);
				});
				aRefreshSinglePromises[iCurrentCreateNo].then(function () {
					aRefreshSingleFinished[iCurrentCreateNo] = true;
				});
			}

			if (oFixture.bRelative) {
				oBinding = this.bindList("EMPLOYEES", oBindingContext);
			} else {
				oBinding = this.bindList("/EMPLOYEES");
			}
			oBindingMock = this.mock(oBinding);
			expect();
			oBinding.attachEvent("change", function (oEvent) {
				assert.strictEqual(oEvent.getParameter("reason"), ChangeReason.Add);
				assert.strictEqual(oBinding.iCreatedContexts, iCreateNo + 1);
				assert.ok(oBinding.aContexts[0].isTransient(), "transient context exists");
				iChangeFired += 1;
			});
			oBinding.iMaxLength = 42;
			oBindingMock.expects("refreshSingle").never();

			// code under test
			aContexts.push(oBinding.create(oFixture.oInitialData, false, false,
				oFixture.bInactive));

			checkCreatedContext();

			// code under test
			oBinding.hasPendingChanges();

			iCreateNo += 1;
			expect();

			// code under test: 2nd create
			aContexts.push(oBinding.create(oFixture.oInitialData, false, false,
				oFixture.bInactive));

			checkCreatedContext();
			assert.strictEqual(aContexts[0].getIndex(), 1);

			return Promise.all([aContexts[0].created(), aContexts[1].created()]).then(function () {
				assert.strictEqual(aContexts[0].isTransient(), false);
				assert.ok(aRefreshSingleFinished[0]);
				assert.strictEqual(aContexts[1].isTransient(), false);
				assert.ok(aRefreshSingleFinished[1]);
				assert.strictEqual(oBinding.iCreatedContexts, 2);
				assert.strictEqual(oBinding.iMaxLength, 42, "persisted contexts are not counted");
			});
		});
	});

	//*********************************************************************************************
	[false, true].forEach(function (bSkipRefresh) {
		QUnit.test("create: bSkipRefresh " + bSkipRefresh, function () {
			var oBinding = this.bindList("/EMPLOYEES"),
				oBindingMock = this.mock(oBinding),
				oContext,
				oCreatedEntity = {},
				oCreatePathPromise = {},
				oCreatePromise = SyncPromise.resolve(Promise.resolve(oCreatedEntity)),
				oGroupLock0 = {},
				oGroupLock1 = {},
				oInitialData = {},
				sPredicate = "(ID=42)",
				oRefreshedEntity = {},
				that = this;

			oBindingMock.expects("getUpdateGroupId").withExactArgs().returns("~update~");
			oBindingMock.expects("lockGroup")
				.withExactArgs("~update~", true, true, sinon.match.func)
				.returns(oGroupLock0);
			oBindingMock.expects("fetchResourcePath").withExactArgs().returns(oCreatePathPromise);
			oBindingMock.expects("createInCache")
				.withExactArgs(sinon.match.same(oGroupLock0), sinon.match.same(oCreatePathPromise),
					"/EMPLOYEES", sinon.match(rTransientPredicate), sinon.match.same(oInitialData),
					false, sinon.match.func, sinon.match.func)
				.returns(oCreatePromise);
			this.mock(oContextPrototype).expects("fetchValue").withExactArgs().resolves({});
			oCreatePromise.then(function () {
				that.mock(_Helper).expects("getPrivateAnnotation")
					.withExactArgs(sinon.match.same(oCreatedEntity), "predicate")
					.returns(sPredicate);
				oBindingMock.expects("fireEvent")
					.withExactArgs("createCompleted", {
						context : sinon.match.same(oContext),
						success : true
					});
				oBindingMock.expects("lockGroup").withExactArgs("$auto")
					.exactly(bSkipRefresh ? 0 : 1)
					.returns(oGroupLock1);
				oBindingMock.expects("refreshSingle")
					.withExactArgs(sinon.match(function (oContext0) {
						return oContext0 === oContext
							&& oContext0.getPath() === "/EMPLOYEES(ID=42)";
					}), sinon.match.same(oGroupLock1))
					.exactly(bSkipRefresh ? 0 : 1)
					.returns(SyncPromise.resolve(oRefreshedEntity));
			});

			// code under test
			oContext = oBinding.create(oInitialData, bSkipRefresh);

			return oContext.created();
		});
	});

	//*********************************************************************************************
	[{
		sPredicate : "('bar')"
	}, {
		oInitialData : {},
		sPredicate : "('bar')"
	}, {
		oInitialData : {},
		bGetPredicate : true
	}, {
		oInitialData : {"@$ui5.keepTransientPath" : true}
	}].forEach(function (oFixture) {
		var sTitle = "create: relative binding, initial data: "
				+ JSON.stringify(oFixture.oInitialData) + ", predicate: " + oFixture.sPredicate;

		QUnit.test(sTitle, function (assert) {
			var oBinding = this.bindList("TEAM_2_EMPLOYEES",
					Context.create(this.oModel, oParentBinding, "/TEAMS/1", 1)),
				oBindingMock = this.mock(oBinding),
				aCacheResult = [{}, {}, {"@$ui5._" : {predicate : "('foo')"}}, {}],
				oContext,
				oContext2 = Context.create(this.oModel, oParentBinding, "/TEAMS/2", 2),
				aContexts,
				oContextMock = this.mock(oContextPrototype),
				oCreatedEntity = {},
				oCreateGroupLock = {},
				oCreateInCachePromise = SyncPromise.resolve(Promise.resolve(oCreatedEntity)),
				oCreatePathPromise = {},
				oFetchDataGroupLock = {unlock : function () {}},
				oRefreshGroupLock = {},
				oRefreshPromise = oCreateInCachePromise.then(function () {
					return SyncPromise.resolve(Promise.resolve());
				}),
				that = this;

			oBinding.enableExtendedChangeDetection();
			oBindingMock.expects("fetchResourcePath")
				.withExactArgs()
				.returns(oCreatePathPromise);
			oBindingMock.expects("checkSuspended").withExactArgs().twice();
			oBindingMock.expects("getUpdateGroupId").withExactArgs().returns("~update~");
			oBindingMock.expects("lockGroup")
				.withExactArgs("~update~", true, true, sinon.match.func)
				.returns(oCreateGroupLock);
			oBindingMock.expects("createInCache")
				.withExactArgs(sinon.match.same(oCreateGroupLock),
					sinon.match.same(oCreatePathPromise), "/TEAMS/1/TEAM_2_EMPLOYEES",
					sinon.match(rTransientPredicate), sinon.match.same(oFixture.oInitialData),
					false, sinon.match.func, sinon.match.func)
				.returns(oCreateInCachePromise);
			oCreateInCachePromise.then(function () {
				that.mock(_Helper).expects("getPrivateAnnotation")
					.exactly(oFixture.sPredicate || oFixture.bGetPredicate ? 1 : 0)
					.withExactArgs(sinon.match.same(oCreatedEntity), "predicate")
					.returns(oFixture.sPredicate);
				oBindingMock.expects("adjustPredicate").exactly(oFixture.sPredicate ? 1 : 0)
					.withExactArgs(sinon.match(rTransientPredicate), oFixture.sPredicate,
						sinon.match.same(oContext));
				that.mock(that.oModel).expects("checkMessages").exactly(oFixture.sPredicate ? 1 : 0)
					.withExactArgs();
				oBindingMock.expects("getGroupId").withExactArgs().returns("$auto");
				oBindingMock.expects("lockGroup").withExactArgs("$auto").returns(oRefreshGroupLock);
				oBindingMock.expects("refreshSingle")
					.withExactArgs(sinon.match.same(oContext), sinon.match.same(oRefreshGroupLock))
					.returns(oRefreshPromise);
			});
			oContextMock.expects("fetchValue").withExactArgs().resolves({});

			// code under test
			oContext = oBinding.create(oFixture.oInitialData);

			aCacheResult.unshift({/*transient element*/});
			oBindingMock.expects("lockGroup").withExactArgs().returns(oFetchDataGroupLock);
			this.mock(oFetchDataGroupLock).expects("unlock").withExactArgs();
			oContextMock.expects("fetchValue")
				.withExactArgs("/TEAMS/1/TEAM_2_EMPLOYEES")
				.returns(SyncPromise.resolve(aCacheResult));

			// code under test - ensure that getContexts delivers the created context correctly
			aContexts = oBinding.getContexts(0, 4);

			assert.strictEqual(aContexts.length, 4);
			assert.strictEqual(aContexts[0], oContext);
			assert.strictEqual(aContexts[1].getPath(), "/TEAMS/1/TEAM_2_EMPLOYEES/0");
			assert.strictEqual(aContexts[2].getPath(), "/TEAMS/1/TEAM_2_EMPLOYEES/1");
			assert.strictEqual(aContexts[3].getPath(), "/TEAMS/1/TEAM_2_EMPLOYEES('foo')");
			assert.strictEqual(oBinding.aPreviousData.length, 4);
			assert.ok(
				/\/TEAMS\/1\/TEAM_2_EMPLOYEES\(\$uid=id-[-0-9]+\)/.test(oBinding.aPreviousData[0]),
				oBinding.aPreviousData[0]);
			assert.strictEqual(oBinding.aPreviousData[1], "/TEAMS/1/TEAM_2_EMPLOYEES/0");
			assert.strictEqual(oBinding.aPreviousData[2], "/TEAMS/1/TEAM_2_EMPLOYEES/1");
			assert.strictEqual(oBinding.aPreviousData[3], "/TEAMS/1/TEAM_2_EMPLOYEES('foo')");

			return oContext.created().then(function () {
				oBindingMock.expects("checkSuspended").withExactArgs(true);
				oBindingMock.expects("reset").withExactArgs(undefined, true);
				oBindingMock.expects("fetchCache").withExactArgs(sinon.match.same(oContext2));
				oBindingMock.expects("restoreCreated").withExactArgs();

				oBinding.setContext(oContext2);
			});
		});
	});

	//*********************************************************************************************
	QUnit.test("create: relative binding not yet resolved", function (assert) {
		var oBinding = this.bindList("TEAM_2_EMPLOYEES");

		// code under test
		assert.throws(function () {
			oBinding.create();
		}, new Error("Binding is unresolved: " + oBinding));
	});

	//*********************************************************************************************
	QUnit.test("create: bAtEnd & suspended", function (assert) {
		var oBinding = this.bindList("/TEAMS"),
			oError = new Error("suspended");

		this.mock(oBinding).expects("checkSuspended").withExactArgs().throws(oError);

		// code under test
		assert.throws(function () {
			oBinding.create();
		}, oError);

		assert.strictEqual(oBinding.bFirstCreateAtEnd, undefined);
	});

	//*********************************************************************************************
	QUnit.test("create: missing $$ownRequest", function (assert) {
		var oBinding = this.bindList("TEAM_2_EMPLOYEES", {
				getPath : function () { return "/TEAMS('1')"; }
			});

		this.mock(oBinding).expects("getUpdateGroupId").withExactArgs().returns("$auto");

		// code under test
		assert.throws(function () {
			oBinding.create({}, false, false, true);
		}, new Error("Missing $$ownRequest at " + oBinding));

		assert.strictEqual(oBinding.bFirstCreateAtEnd, undefined);
	});

	//*********************************************************************************************
	QUnit.test("create: failure", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oBindingMock = this.mock(oBinding),
			oContext,
			oCreatePathPromise = {},
			oError = new Error(),
			oCreatePromise = SyncPromise.resolve(Promise.reject(oError)),
			oGroupLock = {unlock : function () {}},
			oInitialData = {};

		oBindingMock.expects("getUpdateGroupId").withExactArgs().returns("~update~");
		oBindingMock.expects("lockGroup").withExactArgs("~update~", true, true, sinon.match.func)
			.returns(oGroupLock);
		oBindingMock.expects("fetchResourcePath").withExactArgs().returns(oCreatePathPromise);
		oBindingMock.expects("createInCache")
			.withExactArgs(sinon.match.same(oGroupLock), sinon.match.same(oCreatePathPromise),
				"/EMPLOYEES", sinon.match(rTransientPredicate), sinon.match.same(oInitialData),
				false, sinon.match.func, sinon.match.func)
			.returns(oCreatePromise);
		this.mock(oContextPrototype).expects("fetchValue").withExactArgs().resolves({});

		oBindingMock.expects("refreshSingle").never();
		this.mock(oGroupLock).expects("unlock").withExactArgs(true);

		// code under test
		oContext = oBinding.create(oInitialData);

		return oContext.created().then(function () {
			assert.ok(false);
		}, function (oError0) {
			assert.strictEqual(oError0, oError);
		});
	});

	//*********************************************************************************************
	[ // [first call, second call]
		[false, false],
		[true, true],
		[false, true],
		[true, false], // not allowed
		[undefined, true],
		[true, undefined] // not allowed
	].forEach(function (aAtEnd) {
		var sTitle = "create: position: " + (aAtEnd[1] ? "END" : "START") + "_OF_"
			+ (aAtEnd[0] ? "END" : "START");

		QUnit.test(sTitle, function (assert) {
			var oBinding = this.bindList("/EMPLOYEES"),
				oBindingMock = this.mock(oBinding),
				oContext0,
				oContext1,
				oContextMock = this.mock(Context),
				oCreatePathMatcher = sinon.match(function (oPromise) {
					return oPromise.getResult() === "EMPLOYEES";
				}),
				oElement0 = {},
				oElement1 = {},
				oGroupLock = {unlock : function () {}},
				oHelperMock = this.mock(_Helper),
				oNewContext0 = {
					created : function () {},
					fetchValue : function () {},
					getPath : function () {},
					refreshDependentBindings : function () {}
				},
				oNewContext1 = {
					created : function () {},
					fetchValue : function () {},
					getPath : function () {},
					refreshDependentBindings : function () {}
				},
				bNotAllowed = aAtEnd[0] && !aAtEnd[1];

			oBinding.aContexts.push("~oContext~");
			oBinding.bLengthFinal = true;
			oBinding.iMaxLength = 0;
			oBindingMock.expects("getUpdateGroupId").twice()
				.withExactArgs().returns("~update~");
			oBindingMock.expects("lockGroup").exactly(bNotAllowed ? 1 : 2)
				.withExactArgs("~update~", true, true, sinon.match.func)
				.returns(oGroupLock);
			oBindingMock.expects("createInCache")
				.withExactArgs(sinon.match.same(oGroupLock), oCreatePathMatcher, "/EMPLOYEES",
					sinon.match(rTransientPredicate), undefined,
					sinon.match(function (bAtEndOfCreated) {
						return bAtEndOfCreated === (oBinding.bFirstCreateAtEnd !== !!aAtEnd[0]);
					}), sinon.match.func, sinon.match.func)
				.returns(SyncPromise.resolve(Promise.resolve({})));
			oContextMock.expects("create")
				.withExactArgs(sinon.match.same(oBinding.oModel),
					sinon.match.same(oBinding), sinon.match.string, -oBinding.iCreatedContexts - 1,
					sinon.match.instanceOf(SyncPromise), undefined)
				.returns(oNewContext0);
			this.mock(oNewContext0).expects("fetchValue").withExactArgs().resolves(oElement0);
			oHelperMock.expects("setPrivateAnnotation")
				.withExactArgs(sinon.match.same(oElement0), "context",
					sinon.match.same(oNewContext0));
			oHelperMock.expects("setPrivateAnnotation")
				.withExactArgs(sinon.match.same(oElement0), "firstCreateAtEnd",
					sinon.match(function (bArgs) {
						return oBinding.bFirstCreateAtEnd === bArgs;
					}));
			this.mock(oNewContext0).expects("getPath").withExactArgs().returns("");
			this.mock(oNewContext0).expects("refreshDependentBindings")
				.withExactArgs("", sinon.match.string, true).returns("");

			// code under test
			oContext0 = oBinding.create(undefined, true, aAtEnd[0]);

			assert.strictEqual(oContext0, oNewContext0);
			assert.strictEqual(oBinding.bFirstCreateAtEnd, !!aAtEnd[0]);
			assert.strictEqual(oBinding.iActiveContexts, 1);

			if (bNotAllowed) {
				assert.throws(function () {
					// code under test
					oBinding.create(undefined, true, aAtEnd[1]);
				}, new Error("Cannot create at the start after creation at end"));

				assert.strictEqual(oBinding.iActiveContexts, 1); // unchanged
			} else {
				oBindingMock.expects("createInCache")
					.withExactArgs(sinon.match.same(oGroupLock),
						oCreatePathMatcher, "/EMPLOYEES",
						sinon.match(rTransientPredicate), undefined,
						sinon.match(function (bAtEndOfCreated) {
							return bAtEndOfCreated
								=== (oBinding.bFirstCreateAtEnd !== !!aAtEnd[1]);
						}), sinon.match.func, sinon.match.func)
					.returns(SyncPromise.resolve(Promise.resolve({})));
				oContextMock.expects("create")
					.withExactArgs(sinon.match.same(oBinding.oModel),
						sinon.match.same(oBinding), sinon.match.string,
						-oBinding.iCreatedContexts - 1, sinon.match.instanceOf(SyncPromise),
						undefined)
					.returns(oNewContext1);
				this.mock(oNewContext1).expects("fetchValue").withExactArgs().resolves(oElement1);
				oHelperMock.expects("setPrivateAnnotation")
					.withExactArgs(sinon.match.same(oElement1), "context",
						sinon.match.same(oNewContext1));
				oHelperMock.expects("setPrivateAnnotation")
					.withExactArgs(sinon.match.same(oElement1), "firstCreateAtEnd",
						sinon.match(function (bArgs) {
							return oBinding.bFirstCreateAtEnd === bArgs;
						}));
				this.mock(oNewContext1).expects("getPath").withExactArgs().returns("");
				this.mock(oNewContext1).expects("refreshDependentBindings")
					.withExactArgs("", sinon.match.string, true).returns("");

				// code under test
				oContext1 = oBinding.create(undefined, true, aAtEnd[1]);

				assert.strictEqual(oContext1, oNewContext1);

				assert.strictEqual(oBinding.iCreatedContexts, 2);

				if (aAtEnd[0] === aAtEnd[1]) { // START_OF_START, END_OF_END
					assert.deepEqual(oBinding.aContexts[0], oContext1);
					assert.deepEqual(oBinding.aContexts[1], oContext0);
				} else { // END_OF_START (fresh element is inserted at end of created, recalc index)
					assert.deepEqual(oBinding.aContexts[0], oContext0);
					assert.deepEqual(oBinding.aContexts[1], oContext1);
					assert.deepEqual(oBinding.aContexts[0].iIndex, -2);
					assert.deepEqual(oBinding.aContexts[1].iIndex, -1);
				}
				assert.deepEqual(oBinding.aContexts[2], "~oContext~");
			}

			return Promise.all([
				oContext0.created(),
				oContext1 ? oContext1.created() : undefined
			]);
		});
	});

	//*********************************************************************************************
	QUnit.test("create: fetchValue returns undefined", function () {
		var oBinding = this.bindList("/EMPLOYEES"),
			oBindingMock = this.mock(oBinding),
			oContext,
			oCreatePathMatcher = sinon.match(function (oPromise) {
				return oPromise.getResult() === "EMPLOYEES";
			}),
			oGroupLock = {unlock : function () {}},
			oNewContext = {
				created : function () {},
				fetchValue : function () {},
				getPath : function () { return ""; },
				refreshDependentBindings : function () {}
			};

		oBindingMock.expects("getUpdateGroupId")
			.withExactArgs().returns("~update~");
		oBindingMock.expects("lockGroup")
			.withExactArgs("~update~", true, true, sinon.match.func)
			.returns(oGroupLock);
		oBindingMock.expects("createInCache")
			.withExactArgs(sinon.match.same(oGroupLock), oCreatePathMatcher, "/EMPLOYEES",
				sinon.match(rTransientPredicate), undefined,
				false, sinon.match.func, sinon.match.func)
			.returns(SyncPromise.resolve(Promise.resolve({})));
		this.mock(Context).expects("create")
			.withExactArgs(sinon.match.same(oBinding.oModel),
				sinon.match.same(oBinding), sinon.match.string, -oBinding.iCreatedContexts - 1,
				sinon.match.instanceOf(SyncPromise), undefined)
			.returns(oNewContext);
		this.mock(oNewContext).expects("fetchValue").withExactArgs().resolves(undefined);
		this.mock(_Helper).expects("setPrivateAnnotation").never();

		// code under test
		oContext = oBinding.create(undefined, true, false);

		return oContext.created();
	});

	//*********************************************************************************************
	QUnit.test("create: bAtEnd without $count", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oContext,
			sError = "Must know the final length to create at the end. Consider setting $count";

		this.mock(oBinding).expects("checkSuspended").thrice().withExactArgs();

		assert.throws(function () {
			// code under test
			oBinding.create(undefined, true, true);
		}, new Error(sError));

		oBinding.createContexts(3, createData(3, 0, true)); // simulate a read

		assert.throws(function () {
			// code under test
			oBinding.create(undefined, true, true);
		}, new Error(sError));

		oBinding.createContexts(6, createData(1, 3, true, 1)); // simulate a short read
		this.mock(oBinding).expects("createInCache")
			.returns(SyncPromise.resolve(Promise.resolve({})));
		this.mock(oContextPrototype).expects("fetchValue").withExactArgs().resolves({});

		// code under test
		oContext = oBinding.create(undefined, true, true);

		oBinding = this.bindList("TEAM_2_EMPLOYEES",
			Context.create(this.oModel, oParentBinding, "/TEAMS('42')"));
		this.mock(oBinding).expects("checkSuspended").withExactArgs();

		assert.throws(function () {
			// code under test
			oBinding.create(undefined, true, true);
		}, new Error(sError));

		oBinding = this.bindList("TEAM_2_EMPLOYEES",
			this.oModel.createBindingContext("/TEAMS('42')"));
		this.mock(oBinding).expects("checkSuspended").withExactArgs();

		assert.throws(function () {
			// code under test
			oBinding.create(undefined, true, true);
		}, new Error(sError));

		return oContext.created();
	});

	//*********************************************************************************************
	QUnit.test("create: bAtEnd with $count, but before read", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES", null, null, null, {$count : true}),
			oContext0,
			oContext1,
			oContext2;

		this.mock(oBinding).expects("checkSuspended").thrice().withExactArgs();
		this.mock(oBinding).expects("createInCache").thrice()
			.returns(SyncPromise.resolve(Promise.resolve({})));
		this.mock(oContextPrototype).expects("fetchValue").thrice().withExactArgs().resolves({});

		// code under test
		oContext0 = oBinding.create(undefined, true, true);
		oContext1 = oBinding.create(undefined, true, true);
		oContext2 = oBinding.create(undefined, true, true);

		assert.strictEqual(oBinding.getModelIndex(0), 2);
		assert.strictEqual(oBinding.getModelIndex(1), 1);
		assert.strictEqual(oBinding.getModelIndex(2), 0);

		return Promise.all([
			oContext0.created(),
			oContext1.created(),
			oContext2.created()
		]);
	});

	//*********************************************************************************************
	QUnit.test("create and delete with bAtEnd varying", function () {
		var oBinding = this.bindList("/EMPLOYEES"),
			oBindingMock = this.mock(oBinding),
			oContext1,
			oContext2,
			oExpectation,
			oGroupLock = {getGroupId : function () { return "$auto"; }};

		oBinding.bLengthFinal = true;
		oBinding.iMaxLength = 0;
		oExpectation = oBindingMock.expects("lockGroup").atLeast(1).returns({});
		oBindingMock.expects("createInCache").returns(SyncPromise.resolve({}));
		this.mock(oContextPrototype).expects("fetchValue").atLeast(1).withExactArgs().resolves({});
		oBindingMock.expects("refreshSingle").atLeast(1).returns(SyncPromise.resolve());

		// code under test
		oBinding.create(undefined, false, /*bAtEnd*/true);

		// code under test - cancel the creation (via the group lock from the create)
		oExpectation.args[0][3]();

		oBindingMock.expects("createInCache").returns(SyncPromise.resolve({}));

		// code under test
		oContext1 = oBinding.create(undefined, false, /*bAtEnd*/false);

		oBindingMock.expects("createInCache").returns(SyncPromise.resolve({}));

		// code under test - create a second entity without bAtEnd
		oContext2 = oBinding.create(undefined);

		oBindingMock.expects("deleteFromCache")
			.callsArgWith(4, 0, -1) // the callback removing the context
			.returns(SyncPromise.resolve());

		// code under test
		oBinding._delete(oGroupLock, "~", oContext1);

		oBindingMock.expects("createInCache").returns(SyncPromise.resolve({}));

		// code under test
		oBinding.create(undefined, false, /*bAtEnd*/true);

		oBindingMock.expects("deleteFromCache")
			.callsArgWith(4, 0, -1) // the callback removing the context
			.returns(SyncPromise.resolve());

		// code under test
		oBinding._delete(oGroupLock, "~", oContext2);

		oBindingMock.expects("createInCache").returns(SyncPromise.resolve({}));

		// code under test
		oBinding.create(undefined);
	});

	//*********************************************************************************************
	QUnit.test("create at the start after creation at end, delete in between", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oBindingMock = this.mock(oBinding),
			oContext1,
			oContext2,
			oDeletePromise,
			oGroupLock = {getGroupId : function () { return "$auto"; }};

		oBinding.bLengthFinal = true;
		oBinding.iMaxLength = 0;
		this.mock(oContextPrototype).expects("fetchValue").atLeast(1).withExactArgs().resolves({});
		oBindingMock.expects("refreshSingle").atLeast(1).returns(SyncPromise.resolve());

		oBindingMock.expects("createInCache").twice().returns(SyncPromise.resolve({}));

		// code under test
		oContext1 = oBinding.create(undefined, false, /*bAtEnd*/true);
		oContext2 = oBinding.create(undefined, false, /*bAtEnd*/true);

		oBindingMock.expects("deleteFromCache")
			.withArgs(sinon.match.same(oGroupLock), "~1")
			.callsArgWith(4, 0, -1) // the callback removing the context
			.returns(SyncPromise.resolve(Promise.resolve()));
		oBindingMock.expects("deleteFromCache")
			.withArgs(null, "~2")
			.callsArgWith(4, 0, -1) // the callback removing the context
			.returns(SyncPromise.resolve(Promise)); // finish immediately

		// code under test
		oDeletePromise = oBinding._delete(oGroupLock, "~1", oContext1);
		oBinding._delete(null, "~2", oContext2);

		assert.throws(function () {
			// code under test - as long as a reinsertion is possible, bAtEnd must not be changed
			oBinding.create(undefined, false, /*bAtEnd*/false);
		}, new Error("Cannot create at the start after creation at end"));

		return oDeletePromise.then(function () {
			oBindingMock.expects("createInCache").returns(SyncPromise.resolve({}));

			// code under test
			oBinding.create(undefined, false, /*bAtEnd*/false);
		});
	});

	//*********************************************************************************************
	QUnit.test("delete transient entity", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oBindingMock = this.mock(oBinding),
			fnDeleteFromCache = oBinding.deleteFromCache,
			oContext;

		// initialize with 3 contexts and bLengthFinal===true
		oBinding.createContexts(0, createData(3, 0, true, 3));

		// remove request mock, all operations on client
		oBinding.oCachePromise.getResult().oRequestor.request.restore();

		oBindingMock.expects("_fireChange")
			.withExactArgs({reason : ChangeReason.Add});
		oBindingMock.expects("getUpdateGroupId").returns("update");

		oContext = oBinding.create();
		assert.strictEqual(oBinding.iCreatedContexts, 1);
		assert.strictEqual(oBinding.getLength(), 4);

		// avoid "Uncaught (in promise)"
		oContext.created().catch(function (oError) {
			assert.ok(oError.canceled, "create promise rejected with 'canceled'");
		});
		this.mock(oBinding).expects("removeCreated").withExactArgs(sinon.match.same(oContext))
			.callThrough();
		oBindingMock.expects("deleteFromCache").callsFake(function () {
			return fnDeleteFromCache.apply(this, arguments).then(function () {
				// the change must only be fired when deleteFromCache is finished
				// otherwise we run into trouble with extended change detection
				oBindingMock.expects("_fireChange")
					.withExactArgs({reason : ChangeReason.Remove})
					.callsFake(function () {
						assert.strictEqual(oBinding.iCreatedContexts, 0, "No transient context");
						assert.strictEqual(oBinding.getLength(), 3);
					});
			});
		});

		// code under test
		return oContext.delete("$direct");
	});

	//*********************************************************************************************
	QUnit.test("getEntryKey", function (assert) {
		var oContext = {
				getPath : function () {
					return "/some/path";
				}
			};

		// code under test
		// Note: not really an instance method
		assert.strictEqual(ODataListBinding.prototype.getEntryKey(oContext), "/some/path");
	});

	//*********************************************************************************************
	QUnit.test("getEntryData", function (assert) {
		var oContext = {
				getValue : function () {
					return "~oValue~";
				}
			};

		this.mock(_Helper).expects("publicClone")
			.withExactArgs("~oValue~", false, true).returns("~oClone~");

		// code under test
		// Note: not really an instance method
		assert.strictEqual(ODataListBinding.prototype.getEntryData(oContext), "~oClone~");
	});

	//*********************************************************************************************
	QUnit.test("getDiff", function (assert) {
		var oBinding = this.bindList("EMPLOYEE_2_EQUIPMENTS",
				Context.create(this.oModel, oParentBinding, "/EMPLOYEES/0")),
			oBindingMock = this.mock(oBinding),
			aContexts = [{}, {}],
			aDiff = [],
			aPreviousData = [];

		oBinding.aPreviousData = aPreviousData;
		oBindingMock.expects("getContextsInViewOrder")
			.withExactArgs(0, 50).returns(aContexts);
		oBindingMock.expects("getContextData").withExactArgs(sinon.match.same(aContexts[0]))
			.returns("~data~0");
		oBindingMock.expects("getContextData").withExactArgs(sinon.match.same(aContexts[1]))
			.returns("~data~1");
		oBindingMock.expects("diffData")
			.withExactArgs(sinon.match.same(aPreviousData), ["~data~0", "~data~1"])
			.returns(aDiff);

		// code under test
		assert.strictEqual(oBinding.getDiff(50), aDiff);

		assert.deepEqual(oBinding.aPreviousData, ["~data~0", "~data~1"]);
	});

	//*********************************************************************************************
	[
		{op : FilterOperator.BT, result : "SupplierName ge 'SAP' and SupplierName le 'XYZ'"},
		{op : FilterOperator.NB, result : "SupplierName lt 'SAP' or SupplierName gt 'XYZ'"},
		{op : FilterOperator.EQ, result : "SupplierName eq 'SAP'"},
		{op : FilterOperator.GE, result : "SupplierName ge 'SAP'"},
		{op : FilterOperator.GT, result : "SupplierName gt 'SAP'"},
		{op : FilterOperator.LE, result : "SupplierName le 'SAP'"},
		{op : FilterOperator.LT, result : "SupplierName lt 'SAP'"},
		{op : FilterOperator.NE, result : "SupplierName ne 'SAP'"},
		{op : FilterOperator.Contains, result : "contains(SupplierName,'SAP')"},
		{op : FilterOperator.NotContains, result : "not contains(SupplierName,'SAP')"},
		{op : FilterOperator.EndsWith, result : "endswith(SupplierName,'SAP')"},
		{op : FilterOperator.NotEndsWith, result : "not endswith(SupplierName,'SAP')"},
		{op : FilterOperator.StartsWith, result : "startswith(SupplierName,'SAP')"},
		{op : FilterOperator.NotStartsWith, result : "not startswith(SupplierName,'SAP')"},
		{caseSensitive : false, op : FilterOperator.BT,
			result : "tolower(SupplierName) ge tolower('SAP') and "
				+ "tolower(SupplierName) le tolower('XYZ')"},
		{caseSensitive : false, op : FilterOperator.NB,
			result : "tolower(SupplierName) lt tolower('SAP') or "
				+ "tolower(SupplierName) gt tolower('XYZ')"},
		{caseSensitive : false, op : FilterOperator.EQ,
			result : "tolower(SupplierName) eq tolower('SAP')"},
		{caseSensitive : false, op : FilterOperator.GE,
			result : "tolower(SupplierName) ge tolower('SAP')"},
		{caseSensitive : false, op : FilterOperator.GT,
			result : "tolower(SupplierName) gt tolower('SAP')"},
		{caseSensitive : false, op : FilterOperator.LE,
			result : "tolower(SupplierName) le tolower('SAP')"},
		{caseSensitive : false, op : FilterOperator.LT,
			result : "tolower(SupplierName) lt tolower('SAP')"},
		{caseSensitive : false, op : FilterOperator.NE,
			result : "tolower(SupplierName) ne tolower('SAP')"},
		{caseSensitive : false, op : FilterOperator.Contains,
			result : "contains(tolower(SupplierName),tolower('SAP'))"},
		{caseSensitive : false, op : FilterOperator.NotContains,
			result : "not contains(tolower(SupplierName),tolower('SAP'))"},
		{caseSensitive : false, op : FilterOperator.EndsWith,
			result : "endswith(tolower(SupplierName),tolower('SAP'))"},
		{caseSensitive : false, op : FilterOperator.NotEndsWith,
			result : "not endswith(tolower(SupplierName),tolower('SAP'))"},
		{caseSensitive : false, op : FilterOperator.StartsWith,
			result : "startswith(tolower(SupplierName),tolower('SAP'))"},
		{caseSensitive : false, op : FilterOperator.NotStartsWith,
			result : "not startswith(tolower(SupplierName),tolower('SAP'))"},
		{caseSensitive : true, op : FilterOperator.EQ, result : "SupplierName eq 'SAP'"},
		{caseSensitive : false, op : FilterOperator.EQ,
			result : "SupplierName eq 'SAP'",
			type : "Edm.Foo"}
	].forEach(function (oFixture) {
		QUnit.test("fetchFilter: " + oFixture.op + " --> " + oFixture.result, function (assert) {
			var oBinding = this.bindList("/SalesOrderList('4711')/SO_2_ITEMS"),
				oHelperMock = this.mock(_Helper),
				oMetaContext = {},
				oMetaModelMock = this.mock(this.oModel.oMetaModel),
				sType = oFixture.type || "Edm.String",
				oPropertyMetadata = {$Type : sType};

			this.mock(this.oModel).expects("resolve")
				.withExactArgs(oBinding.sPath, undefined).returns(oBinding.sPath);
			oMetaModelMock.expects("resolve")
				.withExactArgs("SupplierName", sinon.match.same(oMetaContext))
				.returns("/resolved/path");
			oMetaModelMock.expects("getMetaContext")
				.withExactArgs(oBinding.sPath).returns(oMetaContext);
			oMetaModelMock.expects("fetchObject")
				.withExactArgs("/resolved/path")
				.returns(SyncPromise.resolve(oPropertyMetadata));
			oHelperMock.expects("formatLiteral").withExactArgs("SAP", sType)
				.returns("'SAP'");
			if (oFixture.op === FilterOperator.BT || oFixture.op === FilterOperator.NB) {
				oHelperMock.expects("formatLiteral").withExactArgs("XYZ", sType)
					.returns("'XYZ'");
			}
			oBinding.aApplicationFilters = [new Filter({
				caseSensitive : oFixture.caseSensitive !== undefined
					? oFixture.caseSensitive
					: true,
				operator : oFixture.op,
				path : "SupplierName",
				value1 : "SAP",
				value2 : "XYZ"
			})];

			// code under test
			assert.deepEqual(oBinding.fetchFilter().getResult(), [oFixture.result, undefined]);
		});
	});

	//*********************************************************************************************
	[false, true].forEach(function (bRelative) {
		[false, true].forEach(function (bAnd) {
			QUnit.test("fetchFilter: dynamic '" + (bAnd ? "and" : "or") + "' and static filters, "
					+ (bRelative ? "relative" : "absolute") + " binding", function (assert) {
				var oBinding = this.bindList(bRelative ? "BP_2_SO" : "/SalesOrderList"),
					oContext = Context.create(this.oModel, {}, "/BusinessPartnerList"),
					oHelperMock = this.mock(_Helper),
					oMetaModelMock = this.mock(this.oModel.oMetaModel),
					sResolvedPath
						= bRelative ? "/BusinessPartnerList('42')/BP_2_SO" : "/SalesOrderList";

				this.mock(this.oModel).expects("resolve")
					.withExactArgs(oBinding.sPath, sinon.match.same(oContext))
					.returns(sResolvedPath);
				oMetaModelMock.expects("getMetaContext")
					.withExactArgs(sResolvedPath).returns("~");
				oMetaModelMock.expects("resolve")
					.withExactArgs("SO_2_BP/CompanyName", "~")
					.returns("/resolved/path1");
				oMetaModelMock.expects("fetchObject")
					.withExactArgs("/resolved/path1")
					.returns(SyncPromise.resolve({$Type : "Edm.String"}));
				oMetaModelMock.expects("resolve")
					.withExactArgs("GrossAmount", "~")
					.returns("/resolved/path2");
				oMetaModelMock.expects("fetchObject")
					.withExactArgs("/resolved/path2")
					.returns(SyncPromise.resolve({$Type : "Edm.Decimal"}));
				oHelperMock.expects("formatLiteral").withExactArgs("SAP", "Edm.String")
					.returns("'SAP'");
				oHelperMock.expects("formatLiteral").withExactArgs(12345, "Edm.Decimal")
					.returns(12345);
				oBinding.aApplicationFilters = [
					new Filter({
						filters : [
							new Filter("SO_2_BP/CompanyName", FilterOperator.EQ, "SAP"),
							new Filter("GrossAmount", FilterOperator.LE, 12345)
						],
						and : bAnd
					})
				];

				assert.deepEqual(
					oBinding.fetchFilter(oContext, "GrossAmount ge 1000").getResult(),
					[(bAnd
						? "SO_2_BP/CompanyName eq 'SAP' and GrossAmount le 12345"
						: "(SO_2_BP/CompanyName eq 'SAP' or GrossAmount le 12345)"
					) + " and (GrossAmount ge 1000)", undefined]
				);
			});
		});
	});

	//*********************************************************************************************
	QUnit.test("fetchFilter: static filter only", function (assert) {
		var oBinding = this.bindList("/SalesOrderList");

		assert.deepEqual(
			oBinding.fetchFilter(undefined, "GrossAmount ge 1000").getResult(),
			["GrossAmount ge 1000"]);
	});

	//*********************************************************************************************
	QUnit.test("fetchFilter: error invalid operator", function (assert) {
		var oBinding = this.bindList("/SalesOrderList"),
			oPropertyMetadata = {$Type : "Edm.String"};

		this.mock(this.oModel).expects("resolve")
			.withExactArgs(oBinding.sPath, undefined).returns(oBinding.sPath);
		this.mock(this.oModel.oMetaModel).expects("getMetaContext")
			.withExactArgs(oBinding.sPath).returns("~");
		this.mock(this.oModel.oMetaModel).expects("resolve")
			.withExactArgs("SO_2_BP/CompanyName", "~")
			.returns("/resolved/path");
		this.mock(this.oModel.oMetaModel).expects("fetchObject")
			.withExactArgs("/resolved/path")
			.returns(SyncPromise.resolve(oPropertyMetadata));
		this.mock(_Helper).expects("formatLiteral").withExactArgs("SAP", "Edm.String")
			.returns("'SAP'");
		oBinding.aApplicationFilters = [new Filter("SO_2_BP/CompanyName", "invalid", "SAP")];

		return oBinding.fetchFilter().then(function () {
			assert.ok(false);
		}, function (oError) {
			assert.strictEqual(oError.message, "Unsupported operator: invalid");
		});
	});

	//*********************************************************************************************
	QUnit.test("fetchFilter: error no metadata for filter path", function (assert) {
		var oBinding = this.bindList("/SalesOrderList"),
			sPath = "/SalesOrderList/SO_2_BP/CompanyName",
			oMetaContext = {
				getPath : function () { return sPath; }
			};

		this.mock(this.oModel.oMetaModel).expects("getMetaContext")
			.withExactArgs(oBinding.sPath).returns(oMetaContext);
		this.mock(this.oModel.oMetaModel).expects("resolve")
			.withExactArgs("SO_2_BP/CompanyName", sinon.match.same(oMetaContext))
			.returns("/resolved/path");
		this.mock(this.oModel.oMetaModel).expects("fetchObject")
			.withExactArgs("/resolved/path")
			.returns(SyncPromise.resolve());
		oBinding.aApplicationFilters = [new Filter("SO_2_BP/CompanyName", FilterOperator.EQ,
			"SAP")];

		return oBinding.fetchFilter().then(function () {
			assert.ok(false);
		}, function (oError) {
			assert.strictEqual(oError.message, "Type cannot be determined, no metadata for path: "
				+ "/resolved/path");
		});
	});

	//*********************************************************************************************
	[
		{filters : [], result : undefined},
		{filters : ["path0", "path1"], result : "path0 eq path0Value and path1 eq path1Value"},
		{ // "grouping": or conjunction for filters with same path
			filters : [{p : "path0", v : "foo"}, "path1", {p : "path0", v : "bar"}],
			result : "(path0 eq foo or path0 eq bar) and path1 eq path1Value"
		}
	].forEach(function (oFixture) {
		QUnit.test("fetchFilter: flat filter '" + oFixture.result + "'", function (assert) {
			var oBinding = this.bindList("/SalesOrderList"),
				aFilters = [],
				oHelperMock = this.mock(_Helper),
				oMetaModelMock = this.mock(this.oModel.oMetaModel),
				oPropertyMetadata = {$Type : "Edm.Type"};

			// call getMetaContext only if there are filters
			oMetaModelMock.expects("getMetaContext").exactly(oFixture.filters.length ? 1 : 0)
				.withExactArgs(oBinding.sPath).returns("~");
			oFixture.filters.forEach(function (vFilter) {
				var sPath,
					sValue;

				if (typeof vFilter === "string") { // single filter: path only
					sPath = vFilter; sValue = sPath + "Value";
				} else { // single filter: path and value
					sPath = vFilter.p; sValue = vFilter.v;
				}

				aFilters.push(new Filter(sPath, FilterOperator.EQ, sValue));
				oMetaModelMock.expects("resolve")
					.withExactArgs(sPath, "~")
					.returns("/resolved/path");
				oMetaModelMock.expects("fetchObject")
					.withExactArgs("/resolved/path")
					.returns(SyncPromise.resolve(oPropertyMetadata));
				oHelperMock.expects("formatLiteral").withExactArgs(sValue, "Edm.Type")
					.returns(sValue);
			});
			oBinding.aApplicationFilters = aFilters;

			return oBinding.fetchFilter().then(function (aFilterValues) {
				assert.strictEqual(aFilterValues[0], oFixture.result);
				assert.strictEqual(aFilterValues[1], undefined);
			});
		});
	});

	//*********************************************************************************************
	QUnit.test("fetchFilter: hierarchical filter", function (assert) {
		var oBinding = this.bindList("/Set"),
			oFilterPromise,
			aFilters = [
				new Filter("p0.0", FilterOperator.EQ, "v0.0"),
				new Filter({
					filters : [
						new Filter("p1.0", FilterOperator.EQ, "v1.0"),
						new Filter("p1.1", FilterOperator.EQ, "v1.1")
					]
				}),
				new Filter({
					filters : [
						new Filter("p2.0", FilterOperator.EQ, "v2.0"),
						new Filter("p2.1", FilterOperator.EQ, "v2.1"),
						new Filter("p2.2", FilterOperator.EQ, "v2.2")
					],
					and : true
				}),
				new Filter("p3.0", FilterOperator.EQ, "v3.0"),
				new Filter("p3.1", FilterOperator.NB, "v3.1", "v3.1")
			],
			oMetaModelMock = this.mock(this.oModel.oMetaModel),
			oPropertyMetadata = {$Type : "Edm.String"},
			oPromise = Promise.resolve(oPropertyMetadata);

		oMetaModelMock.expects("getMetaContext").withExactArgs(oBinding.sPath).returns("~");

		oMetaModelMock.expects("resolve").withExactArgs("p0.0", "~").returns("/resolved/p0.0");
		oMetaModelMock.expects("fetchObject").withExactArgs("/resolved/p0.0").returns(oPromise);

		oMetaModelMock.expects("resolve").withExactArgs("p1.0", "~").returns("/resolved/p1.0");
		oMetaModelMock.expects("fetchObject").withExactArgs("/resolved/p1.0").returns(oPromise);

		oMetaModelMock.expects("resolve").withExactArgs("p1.1", "~").returns("/resolved/p1.1");
		oMetaModelMock.expects("fetchObject").withExactArgs("/resolved/p1.1").returns(oPromise);

		oMetaModelMock.expects("resolve").withExactArgs("p2.0", "~").returns("/resolved/p2.0");
		oMetaModelMock.expects("fetchObject").withExactArgs("/resolved/p2.0").returns(oPromise);

		oMetaModelMock.expects("resolve").withExactArgs("p2.1", "~").returns("/resolved/p2.1");
		oMetaModelMock.expects("fetchObject").withExactArgs("/resolved/p2.1").returns(oPromise);

		oMetaModelMock.expects("resolve").withExactArgs("p2.2", "~").returns("/resolved/p2.2");
		oMetaModelMock.expects("fetchObject").withExactArgs("/resolved/p2.2").returns(oPromise);

		oMetaModelMock.expects("resolve").withExactArgs("p3.0", "~").returns("/resolved/p3.0");
		oMetaModelMock.expects("fetchObject").withExactArgs("/resolved/p3.0").returns(oPromise);

		oMetaModelMock.expects("resolve").withExactArgs("p3.1", "~").returns("/resolved/p3.1");
		oMetaModelMock.expects("fetchObject").withExactArgs("/resolved/p3.1").returns(oPromise);
		oBinding.aApplicationFilters = aFilters;

		oFilterPromise = oBinding.fetchFilter();

		assert.strictEqual(oFilterPromise.isFulfilled(), false);
		return oFilterPromise.then(function (sFilterValue) {
			assert.deepEqual(sFilterValue,
				["p0.0 eq 'v0.0'"
				+ " and (p1.0 eq 'v1.0' or p1.1 eq 'v1.1')"
				+ " and p2.0 eq 'v2.0' and p2.1 eq 'v2.1' and p2.2 eq 'v2.2'"
				+ " and p3.0 eq 'v3.0'"
				+ " and (p3.1 lt 'v3.1' or p3.1 gt 'v3.1')", undefined]
			);
		});
	});

	//*********************************************************************************************
	[FilterOperator.All, FilterOperator.Any].forEach(function (sFilterOperator) {
		[{
			description : "no nesting",
			expectedResult : "p0/" + sFilterOperator.toLowerCase() + "(v0:v0/p1 eq 'value1')",
			fetchObjects : {
				p0 : "Type0",
				"p0/p1" : "Edm.String"
			},
			filter : new Filter({
				condition : new Filter("v0/p1", FilterOperator.EQ, "value1"),
				operator : sFilterOperator,
				path : "p0",
				variable : "v0"
			})
		}, {
			description : "nested any/all filters",
			expectedResult : "p0/" + sFilterOperator.toLowerCase() + "(v0:"
				+ "v0/p1/" + sFilterOperator.toLowerCase() + "(v1:v1/p2 eq 'value2'))",
			fetchObjects : {
				p0 : "Type0",
				"p0/p1" : "Type1",
				"p0/p1/p2" : "Edm.String"
			},
			filter : new Filter({
				condition : new Filter({
					condition : new Filter("v1/p2", FilterOperator.EQ, "value2"),
					operator : sFilterOperator,
					path : "v0/p1",
					variable : "v1"
				}),
				operator : sFilterOperator,
				path : "p0",
				variable : "v0"
			})
		}, {
			description : "nested multi-filter",
			expectedResult : "p0/" + sFilterOperator.toLowerCase()
				+ "(v0:v0/p1 eq 'value1' and v0/p2 eq 'value2')",
			fetchObjects : {
				p0 : "Type0",
				"p0/p1" : "Edm.String",
				"p0/p2" : "Edm.String"
			},
			filter : new Filter({
				condition : new Filter({
					filters : [
						new Filter("v0/p1", FilterOperator.EQ, "value1"),
						new Filter("v0/p2", FilterOperator.EQ, "value2")
					],
					and : true
				}),
				operator : sFilterOperator,
				path : "p0",
				variable : "v0"
			})
		}, {
			description : "nested multi-filter containing an 'any' filter",
			expectedResult : "p0/" + sFilterOperator.toLowerCase()
			+ "(v0:v0/p1/any(v1:v1/p2 lt 'value1') or v0/p3 eq 'value2')",
			fetchObjects : {
				p0 : "Type0",
				"p0/p1" : "Type1",
				"p0/p1/p2" : "Edm.String",
				"p0/p3" : "Edm.String"
			},
			filter : new Filter({
				condition : new Filter({
					filters : [
						new Filter({
							condition : new Filter("v1/p2", FilterOperator.LT, "value1"),
							operator : FilterOperator.Any,
							path : "v0/p1",
							variable : "v1"
						}),
						new Filter("v0/p3", FilterOperator.EQ, "value2")
					]
				}),
				operator : sFilterOperator,
				path : "p0",
				variable : "v0"
			})
		}, {
			description : "multi filters using same lambda variable",
			expectedResult : "p0/" + sFilterOperator.toLowerCase()
				+ "(v0:v0/p1/any(v1:v1/p3 lt 'value1') or v0/p2/any(v1:v1/p4 gt \'value2\'))",
			fetchObjects : {
				p0 : "Type0",
				"p0/p1" : "Type1",
				"p0/p1/p3" : "Edm.String",
				"p0/p2" : "Type2",
				"p0/p2/p4" : "Edm.String"
			},
			filter : new Filter({
				condition : new Filter({
					filters : [
						new Filter({
							condition : new Filter("v1/p3", FilterOperator.LT, "value1"),
							operator : FilterOperator.Any,
							path : "v0/p1",
							variable : "v1"
						}),
						new Filter({
							condition : new Filter("v1/p4", FilterOperator.GT, "value2"),
							operator : FilterOperator.Any,
							path : "v0/p2",
							variable : "v1"
						})
					]
				}),
				operator : sFilterOperator,
				path : "p0",
				variable : "v0"
			})
		}, {
			description : "nested filter overwrites outer lambda variable",
			expectedResult : "p0/" + sFilterOperator.toLowerCase()
				+ "(v0:v0/p1/" + sFilterOperator.toLowerCase() + "(v0:v0/p2 lt 'value1'))",
			fetchObjects : {
				p0 : "Type0",
				"p0/p1" : "Type1",
				"p0/p1/p2" : "Edm.String"
			},
			filter : new Filter({
				condition : new Filter({
					condition : new Filter("v0/p2", FilterOperator.LT, "value1"),
					operator : sFilterOperator,
					path : "v0/p1",
					variable : "v0"
				}),
				operator : sFilterOperator,
				path : "p0",
				variable : "v0"
			})
		}].forEach(function (oFixture) {
			QUnit.test("fetchFilter: " + sFilterOperator + " - " + oFixture.description,
					function (assert) {
				var oBinding = this.bindList("/Set"),
					aFetchObjectKeys = Object.keys(oFixture.fetchObjects),
					oMetaModelMock = this.mock(this.oModel.oMetaModel);

				oBinding.aApplicationFilters = [oFixture.filter];
				oMetaModelMock.expects("getMetaContext")
					.withExactArgs(oBinding.sPath)
					.returns("~");

				aFetchObjectKeys.forEach(function (sObjectPath) {
					oMetaModelMock.expects("resolve")
						.withExactArgs(sObjectPath, "~")
						.returns("/resolved/path");
					oMetaModelMock.expects("fetchObject")
						.withExactArgs("/resolved/path")
						.returns(SyncPromise.resolve({
							$Type : oFixture.fetchObjects[sObjectPath]
						}));
				});

				// code under test
				return oBinding.fetchFilter().then(function (aFilterValues) {
					assert.deepEqual(aFilterValues, [oFixture.expectedResult, undefined]);
				});
			});
		});
	});

	//*********************************************************************************************
	QUnit.test("fetchFilter: any - without predicate", function (assert) {
		var oBinding = this.bindList("/Set"),
			oFilter = new Filter({
				operator : FilterOperator.Any,
				path : "p0"
			}),
			oMetaModelMock = this.mock(this.oModel.oMetaModel);

		oBinding.aApplicationFilters = [oFilter];
		oMetaModelMock.expects("getMetaContext").withExactArgs(oBinding.sPath).returns("~");
		oMetaModelMock.expects("resolve").withExactArgs("p0", "~").returns("/resolved/path");
		oMetaModelMock.expects("fetchObject").withExactArgs("/resolved/path")
			.returns(SyncPromise.resolve({
				$Type : "Type0"
			}));

		// code under test
		return oBinding.fetchFilter().then(function (aFilterValues) {
			assert.deepEqual(aFilterValues, ["p0/any()", undefined]);
		});
	});

	//*********************************************************************************************
	QUnit.test("fetchFilter: application and control filter", function (assert) {
		var oBinding = this.bindList("/Set"),
			oMetaModelMock = this.mock(this.oModel.oMetaModel),
			oPropertyMetadata = {$Type : "Edm.String"},
			oPromise = Promise.resolve(oPropertyMetadata);

		oMetaModelMock.expects("getMetaContext").withExactArgs(oBinding.sPath).returns("~");
		oMetaModelMock.expects("resolve").withExactArgs("p0.0", "~").returns("/resolved/p0.0");
		oMetaModelMock.expects("fetchObject").withExactArgs("/resolved/p0.0").returns(oPromise);

		oMetaModelMock.expects("resolve").withExactArgs("p1.0", "~").returns("/resolved/p1.0");
		oMetaModelMock.expects("fetchObject").withExactArgs("/resolved/p1.0").returns(oPromise);

		oBinding.aFilters = [new Filter("p0.0", FilterOperator.EQ, "v0.0")];
		oBinding.aApplicationFilters = [new Filter("p1.0", FilterOperator.EQ, "v1.0")];

		return oBinding.fetchFilter(undefined, "p2.0 eq 'v2.0'").then(function (sFilterValue) {
			assert.deepEqual(sFilterValue,
				["p0.0 eq 'v0.0' and p1.0 eq 'v1.0' and (p2.0 eq 'v2.0')", undefined]);
		});
	});

	//*********************************************************************************************
	QUnit.skip("fetchFilter: filter with encoded path", function (assert) {
		//TODO encode in the filter or not?
		var oBinding = this.bindList("/Set"),
			oMetaModelMock = this.mock(this.oModel.oMetaModel),
			oPropertyMetadata = {$Type : "Edm.Decimal"},
			oPromise = Promise.resolve(oPropertyMetadata);

		oMetaModelMock.expects("getMetaContext").withExactArgs(oBinding.sPath).returns("~");
		oMetaModelMock.expects("resolve").withExactArgs("AmountIn€", "~").returns("/resolved/path");
		oMetaModelMock.expects("fetchObject").withExactArgs("/resolved/path").returns(oPromise);
		oBinding.aApplicationFilters = [new Filter("AmountIn%E2%82%AC", FilterOperator.GT, "1000")];

		return oBinding.fetchFilter().then(function (aFilterValues) {
			assert.deepEqual(aFilterValues, ["AmountIn€ gt 1000", undefined]);
		});
	});

	//*********************************************************************************************
	// "a=b" -> new Filter("a", FilterOperator.EQ, "b")
	// {and : [a, b]} -> new Filter({filters: [a, b], and : true})
	// {or : [a, b]} -> new Filter({filters: [a, b]})
	[{
		filters : ["p1=v1"],
		result : "p1 eq 'v1'"
	}, {
		filters : ["p1=v1", "p1=v2"],
		result : "p1 eq 'v1' or p1 eq 'v2'"
	}, {
		filters : ["p1=v1", "p2=v2"],
		result : "p1 eq 'v1' and p2 eq 'v2'"
	}, {
		filters : ["p1=v1", "p2=v2", "p1=v3"],
		result : "(p1 eq 'v1' or p1 eq 'v3') and p2 eq 'v2'"
	}, {
		filters : [{or : ["p1=v1", "p1=v2"]}],
		result : "p1 eq 'v1' or p1 eq 'v2'"
	}, {
		filters : [{and : ["p1=v1", "p1=v2"]}],
		result : "p1 eq 'v1' and p1 eq 'v2'"
	}, {
		filters : [{or : ["p1=v1", "p1=v2", "p2=v3"]}],
		result : "p1 eq 'v1' or p1 eq 'v2' or p2 eq 'v3'"
	}, {
		filters : [{and : ["p1=v1", "p1=v2", "p2=v3"]}],
		result : "p1 eq 'v1' and p1 eq 'v2' and p2 eq 'v3'"
	}, {
		filters : ["p1=v1", {or : ["p1=v2", "p1=v3"]}],
		result : "p1 eq 'v1' and (p1 eq 'v2' or p1 eq 'v3')"
	}, {
		filters : ["p1=v1", {and : ["p1=v2", "p1=v3"]}],
		result : "p1 eq 'v1' and p1 eq 'v2' and p1 eq 'v3'"
	}, {
		filters : ["p1=v1", {or : ["p1=v2", "p2=v3"]}],
		result : "p1 eq 'v1' and (p1 eq 'v2' or p2 eq 'v3')"
	}, {
		filters : ["p1=v1", {and : ["p1=v2"]}],
		result : "p1 eq 'v1' and p1 eq 'v2'"
	}].forEach(function (oFixture, i) {
		QUnit.test("filter #" + i + ": " + JSON.stringify(oFixture.filters), function (assert) {
			var oBinding = this.bindList("/Set"),
				oMetaModelMock = this.mock(this.oModel.oMetaModel),
				oPropertyMetadata = {$Type : "Edm.String"},
				oPromise = Promise.resolve(oPropertyMetadata);

			function buildFilters(aNodes) {
				return aNodes.map(function (vNode) {
					var aParts;

					if (typeof vNode === "string") {
						aParts = vNode.split("=");
						return new Filter(aParts[0], FilterOperator.EQ, aParts[1]);
					}
					if (vNode.and) {
						return new Filter({filters : buildFilters(vNode.and), and : true});
					}
					return new Filter({filters : buildFilters(vNode.or)});
				});
			}

			oMetaModelMock.expects("fetchObject").atLeast(0).returns(oPromise);
			oBinding.aApplicationFilters = buildFilters(oFixture.filters);

			// code under test
			return oBinding.fetchFilter().then(function (aFilterValues) {
				assert.deepEqual(aFilterValues, [oFixture.result, undefined]);
			});
		});
	});

	//*********************************************************************************************
[{
	split : [new Filter("a", FilterOperator.GT, 42), undefined],
	result : ["a gt 42", undefined]
}, {
	split : [undefined, new Filter("b", FilterOperator.EQ, "before")],
	result : [undefined, "b eq 'before'"]
}, {
	split : [undefined, new Filter("b", FilterOperator.EQ, "before")],
	staticFilter : "c eq 47",
	result : ["c eq 47", "b eq 'before'"]
}, {
	split : [
		new Filter(
			[new Filter("a", FilterOperator.EQ, 1), new Filter("a", FilterOperator.EQ, 2)], false
		),
		new Filter("b", FilterOperator.EQ, "before")
	],
	staticFilter : "c eq 47",
	result : ["(a eq 1 or a eq 2) and (c eq 47)", "b eq 'before'"]
}, {
	split : [new Filter("a", FilterOperator.GT, 42), new Filter("b", FilterOperator.EQ, "before")],
	result : ["a gt 42", "b eq 'before'"]
}].forEach(function (oFixture, i) {
	QUnit.test("fetchFilter: list binding aggregates data " + i, function (assert) {
		var oAggregation = {},
			oBinding = this.bindList("Set"),
			oContext = {},
			oFilter = {/*any filter*/},
			oMetaModelMock = this.mock(this.oModel.oMetaModel);

		oBinding.mParameters.$$aggregation = oAggregation;

		this.mock(FilterProcessor).expects("combineFilters").returns(oFilter);
		this.mock(_AggregationHelper).expects("splitFilter")
			.withExactArgs(sinon.match.same(oFilter), sinon.match.same(oAggregation))
			.returns(oFixture.split);
		this.mock(this.oModel).expects("resolve").withExactArgs("Set", sinon.match.same(oContext))
			.returns("~");
		oMetaModelMock.expects("getMetaContext").withExactArgs("~").returns("oMetaContext");
		oMetaModelMock.expects("resolve").withExactArgs("a", "oMetaContext").atLeast(0)
			.returns("/resolved/a");
		oMetaModelMock.expects("fetchObject").withExactArgs("/resolved/a").atLeast(0)
			.returns(Promise.resolve({$Type : "Edm.Decimal"}));
		oMetaModelMock.expects("resolve").withExactArgs("b", "oMetaContext").atLeast(0)
			.returns("/resolved/b");
		oMetaModelMock.expects("fetchObject").withExactArgs("/resolved/b").atLeast(0)
			.returns(Promise.resolve({$Type : "Edm.String"}));

		// code under test
		return oBinding.fetchFilter(oContext, oFixture.staticFilter).then(function (aFilterValues) {
			assert.deepEqual(aFilterValues, oFixture.result);
		});
	});
});

	//*********************************************************************************************
	QUnit.test("getOrderby", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			sOrderby = "bar desc";

		assert.strictEqual(oBinding.getOrderby(), "", "empty sorters");
		assert.strictEqual(oBinding.getOrderby(sOrderby), sOrderby);

		oBinding.aSorters = [new Sorter("foo")];

		assert.strictEqual(oBinding.getOrderby(), "foo", "array of sorters");
		assert.strictEqual(oBinding.getOrderby(sOrderby), "foo," + sOrderby);

		oBinding.aSorters = [new Sorter("foo"), new Sorter("bar", true)];

		assert.strictEqual(oBinding.getOrderby(), "foo,bar desc");
		assert.strictEqual(oBinding.getOrderby(sOrderby), "foo,bar desc," + sOrderby);

		oBinding.aSorters = ["foo"];
		assert.throws(function () {
			oBinding.getOrderby();
		}, new Error("Unsupported sorter: foo - "
			+ "sap.ui.model.odata.v4.ODataListBinding: /EMPLOYEES"));
	});

	//*********************************************************************************************
	QUnit.test("doFetchQueryOptions", function (assert) {
		var oBinding = this.bindList("TEAM_2_EMPLOYEES"),
			oContext = {},
			mMergedQueryOptions = {},
			mResolvedQueryOptions = {$filter : "staticFilter", $orderby : "staticSorter"};

		this.mock(oBinding).expects("fetchResolvedQueryOptions")
			.withExactArgs(sinon.match.same(oContext))
			.returns(SyncPromise.resolve(mResolvedQueryOptions));
		this.mock(oBinding).expects("fetchFilter")
			.withExactArgs(sinon.match.same(oContext), "staticFilter")
			.returns(SyncPromise.resolve("resolvedFilter"));
		this.mock(oBinding).expects("getOrderby").withExactArgs("staticSorter")
			.returns("resolvedOrderby");
		this.mock(_Helper).expects("mergeQueryOptions")
			.withExactArgs(sinon.match.same(mResolvedQueryOptions), "resolvedOrderby",
				"resolvedFilter")
			.returns(mMergedQueryOptions);

		// code under test
		assert.strictEqual(oBinding.doFetchQueryOptions(oContext).getResult(), mMergedQueryOptions);
	});

	//*********************************************************************************************
	QUnit.test("doCreateCache w/ old cache", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oOldCache = {
				$deepResourcePath : "deep/resource/path",
				getResourcePath : function () {},
				reset : function () {},
				setQueryOptions : function () {}
			},
			aPredicates = ["('0')", "('2')"];

		this.mock(oOldCache).expects("getResourcePath").withExactArgs().returns("resource/path");
		this.mock(oBinding).expects("getKeepAlivePredicates").withExactArgs()
			.returns(aPredicates);
		this.mock(oOldCache).expects("reset")
			.withExactArgs(sinon.match.same(aPredicates), "myGroup");
		this.mock(oOldCache).expects("setQueryOptions").withExactArgs("~queryOptions~", true);
		this.mock(_AggregationCache).expects("create").never();

		assert.strictEqual(
			// code under test
			oBinding.doCreateCache("resource/path", "~queryOptions~", "~context~",
				"deep/resource/path", "myGroup", oOldCache),
			oOldCache);
	});

	//*********************************************************************************************
["iCreatedContexts", "iDeletedContexts"].forEach(function (sProperty) {
	QUnit.test("doCreateCache w/ old cache, " + sProperty, function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oOldCache = {
				$deepResourcePath : "deep/resource/path",
				getResourcePath : function () {},
				reset : function () {},
				setQueryOptions : function () {}
			};

		oBinding[sProperty] = 1;
		this.mock(oOldCache).expects("getResourcePath").withExactArgs().returns("resource/path");
		this.mock(oOldCache).expects("reset").withExactArgs([], "myGroup");
		this.mock(oOldCache).expects("setQueryOptions").withExactArgs("~queryOptions~", true);
		this.mock(_AggregationCache).expects("create").never();

		assert.strictEqual(
			// code under test
			oBinding.doCreateCache("resource/path", "~queryOptions~", "~context~",
				"deep/resource/path", "myGroup", oOldCache),
			oOldCache);
	});
});

	//*********************************************************************************************
[false, true].forEach(function (bWithOld) {
	[false, true].forEach(function (bFromModel) {
		[false, true].forEach(function (bShared) {
	var sTitle = (bWithOld
		? "doCreateCache w/ old cache, but w/o kept-alive elements"
		: "doCreateCache w/o old cache") + ", bFromModel=" + bFromModel + ", bShared=" + bShared;

	QUnit.test(sTitle, function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oCache = { // #setLateQueryOptions must not be called
				registerChangeListener : function () {}
			},
			oOldCache = {
				$deepResourcePath : "deep/resource/path",
				getResourcePath : function () {}
				// #getLateQueryOptions, #reset, #setQueryOptions must no be called
			};

		this.oModel.bAutoExpandSelect = "~autoExpandSelect~";
		oBinding.bSharedRequest = bShared;
		if (bWithOld) {
			this.mock(oOldCache).expects("getResourcePath").withExactArgs()
				.returns("resource/path");
			this.mock(oBinding).expects("getKeepAlivePredicates").withExactArgs().returns([]);
		}
		this.mock(oBinding).expects("inheritQueryOptions")
			.withExactArgs("~queryOptions~", "~context~").returns("~mergedQueryOptions~");
		this.mock(oBinding).expects("getCacheAndMoveKeepAliveContexts")
			.withExactArgs("resource/path", "~mergedQueryOptions~")
			.returns(bFromModel ? oCache : undefined);
		this.mock(oBinding).expects("isGrouped").exactly(bFromModel ? 0 : 1).withExactArgs()
			.returns("~isGrouped~");
		this.mock(_AggregationCache).expects("create").exactly(bFromModel ? 0 : 1)
			.withExactArgs(sinon.match.same(this.oModel.oRequestor), "resource/path",
				"deep/resource/path", sinon.match.same(oBinding.mParameters.$$aggregation),
				"~mergedQueryOptions~", "~autoExpandSelect~", bShared, "~isGrouped~")
			.returns(oCache);
		this.mock(oCache).expects("registerChangeListener").exactly(bShared ? 1 : 0)
			.withExactArgs("", sinon.match.same(oBinding));

		assert.strictEqual(
			// code under test
			oBinding.doCreateCache("resource/path", "~queryOptions~", "~context~",
				"deep/resource/path", undefined, bWithOld ? oOldCache : undefined),
			oCache);
	});
		});
	});
});

	//*********************************************************************************************
[false, true].forEach(function (bDeep) {
	var sTitle = "doCreateCache w/ old cache, but wrong " + (bDeep ? "deep " : "")
			+ "resource path";

	QUnit.test(sTitle, function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oCache = {
				registerChangeListener : function () {}
			},
			oOldCache = {
				$deepResourcePath : bDeep ? "W.R.O.N.G." : "deep/resource/path",
				getResourcePath : function () {}
			};

		this.oModel.bAutoExpandSelect = "~autoExpandSelect~";
		oBinding.bSharedRequest = "~sharedRequest~";
		this.mock(oOldCache).expects("getResourcePath").atMost(1).withExactArgs()
			.returns(bDeep ? "resource/path" : "W.R.O.N.G.");
		this.mock(oBinding).expects("getKeepAlivePredicates").never();
		this.mock(oBinding).expects("inheritQueryOptions")
			.withExactArgs("~queryOptions~", "~context~").returns("~mergedQueryOptions~");
		this.mock(oBinding).expects("isGrouped").withExactArgs().returns("~isGrouped~");
		this.mock(_AggregationCache).expects("create")
			.withExactArgs(sinon.match.same(this.oModel.oRequestor), "resource/path",
				"deep/resource/path", sinon.match.same(oBinding.mParameters.$$aggregation),
				"~mergedQueryOptions~", "~autoExpandSelect~", "~sharedRequest~", "~isGrouped~")
			.returns(oCache);

		assert.strictEqual(
			// code under test
			oBinding.doCreateCache("resource/path", "~queryOptions~", "~context~",
				"deep/resource/path", undefined, oOldCache),
			oCache);
	});
});

	//*********************************************************************************************
	QUnit.test("getQueryOptionsFromParameters", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		// code under test
		assert.strictEqual(oBinding.getQueryOptionsFromParameters(), oBinding.mQueryOptions);
	});

	//*********************************************************************************************
	QUnit.test("inheritQueryOptions - binding with parameters", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES", undefined, undefined, undefined,
				{$$operationMode : OperationMode.Server}),
			mQueryOptions = {};

		this.mock(oBinding).expects("getQueryOptionsForPath").never();

		// code under test
		assert.strictEqual(oBinding.inheritQueryOptions(mQueryOptions), mQueryOptions);
	});

	//*********************************************************************************************
	[{ // no filter or sort in inherited query options
		mDynamicQueryOptionsWithModelOptions : {
			$filter : "Age lt 60",
			$orderby : "Name asc"
		},
		mInheritedQueryOptions : {},
		mExpectedQueryOptions : {
			$filter : "Age lt 60",
			$orderby : "Name asc"
		}
	}, { // no filter or sort in dynamic query options
		mDynamicQueryOptionsWithModelOptions : {},
		mInheritedQueryOptions : {
			$filter : "Age lt 60",
			$orderby : "Name asc"
		},
		mExpectedQueryOptions : {}
	}, { // filter and sort in both dynamic and inherited query options
		mDynamicQueryOptionsWithModelOptions : {
			$filter : "Age lt 60",
			$orderby : "Name asc"
		},
		mInheritedQueryOptions : {
			$filter : "Age gt 20",
			$orderby : "Name desc"
		},
		mExpectedQueryOptions : {
			$filter : "(Age lt 60) and (Age gt 20)",
			$orderby : "Name asc,Name desc"
		}
	}].forEach(function (oFixture, i) {
		QUnit.test("inheritQueryOptions: Test " + i, function (assert) {
			var oBinding = this.bindList("TEAM_2_EMPLOYEES"),
				oContext = {},
				mQueryOptions = {};

			this.mock(oBinding).expects("getQueryOptionsForPath")
				.withExactArgs("", sinon.match.same(oContext))
				.returns(oFixture.mInheritedQueryOptions);
			this.mock(Object).expects("assign")
				.withExactArgs({}, sinon.match.same(oFixture.mInheritedQueryOptions),
					oFixture.mExpectedQueryOptions)
				.returns(mQueryOptions);

			// code under test
			assert.strictEqual(oBinding.inheritQueryOptions(
					oFixture.mDynamicQueryOptionsWithModelOptions, oContext),
				mQueryOptions);
		});
	});

	//*********************************************************************************************
	QUnit.test("getHeaderContext: created in c'tor", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oBindingMock = this.mock(oBinding);

		oBindingMock.expects("isResolved").withExactArgs().returns(true);

		// code under test
		assert.ok(oBinding.getHeaderContext());

		oBindingMock.expects("isResolved").withExactArgs().returns(false);

		// code under test
		assert.strictEqual(oBinding.getHeaderContext(), null);
	});
	//TODO How do dependent bindings learn of the changed context?

	//*********************************************************************************************
	QUnit.test("getHeaderContext: setContext", function (assert) {
		var oBinding = this.bindList("EMPLOYEES"),
			oContext = Context.create(this.oModel, oParentBinding, "/TEAMS", 0);

		assert.strictEqual(oBinding.getHeaderContext(), null);
		this.mock(oBinding).expects("checkSuspended").withExactArgs(true);

		// code under test
		oBinding.setContext(oContext);

		assert.deepEqual(oBinding.getHeaderContext(),
			Context.create(this.oModel, oBinding, "/TEAMS/EMPLOYEES"));
	});

	//*********************************************************************************************
	QUnit.test("BCP: 1770275040 Error occurs in table growing", function (assert) {
		var done = assert.async(),
			oBinding,
			bChangeFired = false,
			aContexts,
			oData = createData(50);

		oBinding = this.bindList("/EMPLOYEES");

		this.oModel.oRequestor.request.restore();
		this.mock(this.oModel.oRequestor).expects("request")
			// exact _GroupLock instance not of interest
			.withArgs("GET", "EMPLOYEES?sap-client=111&$skip=0&$top=50")
			.resolves(oData);

		oBinding.bUseExtendedChangeDetection = true;
		oBinding.attachEvent("change", function (oEvent) {
			assert.strictEqual(bChangeFired, false);
			bChangeFired = true;
			assert.strictEqual(oEvent.getParameter("reason"), ChangeReason.Change);
			setTimeout(function () {
				assert.strictEqual(oBinding.oDiff, undefined, "no 2nd change event, no diff!");
				done();
			}, 0);
		});

		aContexts = oBinding.getContexts(0, 50);
		assert.strictEqual(aContexts.length, 0);
		assert.strictEqual(aContexts.dataRequested, true);
		assert.deepEqual(aContexts.diff, []);

		// code under test
		aContexts = oBinding.getContexts(0, 50);
		assert.strictEqual(aContexts.length, 0);
		assert.strictEqual(aContexts.dataRequested, true);
		assert.deepEqual(aContexts.diff, []);
	});

	//*********************************************************************************************
	QUnit.test("updateAnalyticalInfo: invalid input", function (assert) {
		var aAggregation = [{
				grouped : false,
				name : "BothDimensionAndMeasure",
				total : false
			}],
			oBinding = this.bindList("/EMPLOYEES");

		this.mock(_AggregationHelper).expects("buildApply").never();
		this.mock(oBinding).expects("changeParameters").never();

		assert.throws(function () {
			// code under test
			oBinding.updateAnalyticalInfo(aAggregation);
		}, new Error("Both dimension and measure: BothDimensionAndMeasure"));
	});

	//*********************************************************************************************
	QUnit.test("updateAnalyticalInfo: inResult and visible; destroy twice", function (assert) {
		var aAggregation = [{
				grouped : false,
				inResult : true,
				name : "BillToParty"
			}, {
				name : "UnitProperty"
			}, {
				name : "GrossAmountInTransactionCurrency",
				total : false
			}, {
				grouped : false,
				name : "TransactionCurrency",
				visible : true
			}, {
				grouped : false,
				inResult : false,
				name : "IgnoreThisDimension",
				visible : false
			}],
			sAggregation = JSON.stringify(aAggregation),
			oBinding = this.bindList("/EMPLOYEES"),
			oTransformedAggregation = {
				aggregate : {
					GrossAmountInTransactionCurrency : {}
				},
				group : {
					BillToParty : {},
					TransactionCurrency : {},
					// Note: property which was neither dimension nor measure
					UnitProperty : {}
				},
				search : "covfefe"
			};

		oBinding.mParameters.$$aggregation = {search : "covfefe"};
		this.mock(oBinding).expects("setAggregation").withExactArgs(oTransformedAggregation);

		// code under test
		assert.strictEqual(oBinding.updateAnalyticalInfo(aAggregation), undefined);

		assert.strictEqual(JSON.stringify(aAggregation), sAggregation, "unchanged");

		this.mock(this.oModel).expects("bindingDestroyed")
			.withExactArgs(sinon.match.same(oBinding));
		this.mock(ListBinding.prototype).expects("destroy").on(oBinding).withExactArgs();

		// code under test
		oBinding.destroy();

		// code under test
		oBinding.destroy();
	});

	//*********************************************************************************************
	[{
		aAggregation : [{
			min : true,
			name : "GrossAmount",
			total : false
		}, {
			grouped : false,
			name : "Currency",
			visible : true
		}],
		oTransformedAggregation : {
			aggregate : {
				GrossAmount : {min : true}
			},
			group : {
				Currency : {}
			},
			search : undefined
		}
	}, {
		aAggregation : [{
			max : true,
			name : "GrossAmount",
			total : false
		}, {
			grouped : false,
			name : "Currency",
			visible : true
		}],
		oTransformedAggregation : {
			aggregate : {
				GrossAmount : {max : true}
			},
			group : {
				Currency : {}
			},
			search : undefined
		}
	}, {
		aAggregation : [{
			as : "AvgSalesAmount",
			max : true,
			min : true,
			name : "SalesAmount",
			total : false,
			with : "average"
		}],
		oTransformedAggregation : {
			aggregate : {
				AvgSalesAmount : {
					max : true,
					min : true,
					name : "SalesAmount",
					with : "average"
				}
			},
			group : {},
			search : undefined
		}
	}].forEach(function (oFixture, i) {
		[false, true].forEach(function (bHasMeasureRangePromiseAfterResume) {
			var sTitle = "updateAnalyticalInfo: min/max: " + i
					+ ", has measure range promise after resume: "
					+ bHasMeasureRangePromiseAfterResume;

			QUnit.test(sTitle, function (assert) {
				var sAggregation = JSON.stringify(oFixture.aAggregation),
					oBinding = this.bindList("/EMPLOYEES"),
					mMeasureRange = {},
					oNewCache = {getMeasureRangePromise : function () {}},
					oResult,
					oSetAggregationExpectation;

				oSetAggregationExpectation = this.mock(oBinding).expects("setAggregation")
					.withExactArgs(oFixture.oTransformedAggregation)
					.callsFake(function () {
						assert.strictEqual(oBinding.bHasAnalyticalInfo, true);
					});
				this.mock(oBinding).expects("getRootBindingResumePromise").withExactArgs()
					.callsFake(function () {
						assert.ok(oSetAggregationExpectation.called,
							"setAggregation called before");
						oBinding.oCache = oNewCache;
						oBinding.oCachePromise = SyncPromise.resolve(oNewCache);
						return SyncPromise.resolve();
					});
				this.mock(oNewCache).expects("getMeasureRangePromise").withExactArgs()
					.returns(bHasMeasureRangePromiseAfterResume
						? Promise.resolve(mMeasureRange)
						: undefined);

				// code under test
				oResult = oBinding.updateAnalyticalInfo(oFixture.aAggregation);

				assert.strictEqual(JSON.stringify(oFixture.aAggregation), sAggregation,
					"unchanged");
				assert.ok(oResult.measureRangePromise instanceof Promise);

				return oResult.measureRangePromise.then(function (mMeasureRange0) {
					assert.strictEqual(mMeasureRange0,
						bHasMeasureRangePromiseAfterResume ? mMeasureRange : undefined);
				});
			});
		});
	});

	//*********************************************************************************************
[false, true].forEach(function (bSameCache) {
	QUnit.test("refreshSingle: bSameCache=" + bSameCache, function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oCache = {
				refreshSingle : function () {}
			},
			bContextUpdated = false,
			oContext,
			bDependentsRefreshed = false,
			oGroupLock = {
				getGroupId : function () {},
				unlock : function () {}
			},
			oPromise,
			oRefreshDependentsPromise = new SyncPromise(function (resolve) {
				setTimeout(function () {
					bDependentsRefreshed = true;
					resolve();
				});
			}),
			oRefreshSingleExpectation,
			oRefreshSinglePromise = SyncPromise.resolve(Promise.resolve({})),
			oRootBinding = {
				assertSameCache : function () {},
				getGroupId : function () {}
			},
			that = this;

		// initialize with 3 contexts and bLengthFinal===true
		oBinding.createContexts(0, createData(3, 0, true, 3));

		oContext = oBinding.aContexts[2];
		oBinding.oCache = oCache;
		oBinding.oCachePromise = SyncPromise.resolve(oCache);

		this.mock(oBinding).expects("withCache")
			.withExactArgs(sinon.match.func)
			.callsArgWith(0, oCache, "path/in/cache", oRootBinding);
		this.mock(oContext).expects("getPath").withExactArgs().returns("/EMPLOYEES('2')");
		this.mock(oContext).expects("isKeepAlive").withExactArgs().returns("~keep~alive~");
		this.mock(oBinding.oHeaderContext).expects("getPath").withExactArgs().returns("/EMPLOYEES");
		this.mock(_Helper).expects("getRelativePath").withExactArgs("/EMPLOYEES('2')", "/EMPLOYEES")
			.returns("~key~predicate~");
		this.mock(oContext).expects("getModelIndex").withExactArgs().returns(42);
		oRefreshSingleExpectation = this.mock(oCache).expects("refreshSingle")
			.withExactArgs(sinon.match.same(oGroupLock), "path/in/cache", 42, "~key~predicate~",
				"~keep~alive~", sinon.match.func)
			.returns(oRefreshSinglePromise);
		this.mock(oGroupLock).expects("getGroupId").withExactArgs().returns("groupId");
		this.mock(oContext).expects("refreshDependentBindings")
			.withExactArgs("EMPLOYEES('2')", "groupId")
			.returns(oRefreshDependentsPromise);
		oRefreshSinglePromise.then(function () {
			var oCanceledError = new Error();

			// these must only be called when the cache's refreshSingle is finished
			that.mock(oBinding).expects("fireDataReceived").withExactArgs({data : {}});
			that.mock(oRootBinding).expects("assertSameCache")
				.withExactArgs(sinon.match.same(oCache))
				.callsFake(function () {
					if (!bSameCache) {
						oCanceledError.canceled = true;
						throw oCanceledError;
					}
				});
			that.mock(oContext).expects("checkUpdateInternal").exactly(bSameCache ? 1 : 0)
				.withExactArgs()
				.returns(new SyncPromise(function (resolve) {
					setTimeout(function () {
						bContextUpdated = true;
						resolve();
					});
				}));
			that.mock(oGroupLock).expects("unlock").exactly(bSameCache ? 0 : 1).withExactArgs(true);
			that.mock(oBinding.oModel).expects("reportError").exactly(bSameCache ? 0 : 1)
				.withExactArgs("Failed to refresh entity: " + oContext, sClassName,
					sinon.match.same(oCanceledError));
		});

		// code under test
		oPromise = oBinding.refreshSingle(oContext, oGroupLock);

		assert.strictEqual(oPromise.isFulfilled(), false);

		this.mock(oBinding).expects("fireDataRequested").withExactArgs();

		// code under test - callback fires data requested event
		oRefreshSingleExpectation.firstCall.args[5]();

		return oPromise.then(function () {
			assert.strictEqual(bContextUpdated, bSameCache);
			assert.strictEqual(bDependentsRefreshed, true);
		});
	});
});
	//TODO: within #refreshSingle
	// Eliminate checkUpdate and call refreshInternal with bCheckUpdate=true
	// Find a way to use _Helper.updateExisting in _Cache.refreshSingle to do the
	// notification for the changeListeners, currently it would fail because the lookup
	// for the changeListener fails because of different paths (index versus key predicate)

	//*********************************************************************************************
	[true, false].forEach(function (bOnRemoveCalled) {
		[true, false].forEach(function (bCreated) {
			var sTitle = "refreshSingle with allow remove: " + bOnRemoveCalled + ", created: "
				+ bCreated;

			QUnit.test(sTitle, function (assert) {
				var oBinding = this.bindList("/EMPLOYEES"),
					oCache = {
						refreshSingleWithRemove : function () {}
					},
					oCacheRequestPromise,
					oContext,
					oContextMock,
					bContextUpdated = false,
					bDependentsRefreshed = false,
					oExpectation,
					oGroupLock = {getGroupId : function () {}},
					iIndex = bCreated ? 1 : 3,
					oRefreshDependentsPromise = new SyncPromise(function (resolve) {
						setTimeout(function () {
							bDependentsRefreshed = true;
							resolve();
						});
					}),
					oRootBinding = {
						assertSameCache : function () {},
						getGroupId : function () {}
					},
					that = this;

				// initialize with 6 contexts, bLengthFinal===true and bKeyPredicates===true
				// [-2, -1, 0, 1, 2, undefined, 4, 5]
				oBinding.createContexts(0, createData(3, 0, true, 3, true));
				oBinding.createContexts(4, createData(2, 4, true, 6, true));
				assert.strictEqual(oBinding.iMaxLength, 6);
				// simulate create (but w/o #created promise, @see #doReplaceWith)
				oBinding.aContexts.unshift(
					Context.create(this.oModel, oBinding, "/EMPLOYEES($uid=id-1-24)", -2),
					Context.create(this.oModel, oBinding, "/EMPLOYEES($uid=id-1-23)", -1));
				oBinding.iCreatedContexts = 2;

				oContext = oBinding.aContexts[iIndex];
				oContextMock = this.mock(oContext);
				oBinding.oCache = oCache;
				oBinding.oCachePromise = SyncPromise.resolve(oCache);

				oCacheRequestPromise = SyncPromise.resolve(Promise.resolve().then(function () {
					// fnOnRemove Test
					if (bOnRemoveCalled) {
						oContextMock.expects("getModelIndex").withExactArgs().callThrough();
						that.mock(oBinding).expects("removeCreated").exactly(bCreated ? 1 : 0)
							.withExactArgs(sinon.match.same(oContext));
						oContextMock.expects("destroy").exactly(bCreated ? 0 : 1)
							.withExactArgs();
						that.mock(oBinding).expects("_fireChange")
							.withExactArgs({reason : ChangeReason.Remove});
						that.mock(that.oModel).expects("getDependentBindings").never();

						// code under test
						oExpectation.firstCall.args[6](false);

						if (!bCreated) {
							assert.strictEqual(oBinding.aContexts.length, 7);
							assert.notOk(4 in oBinding.aContexts);
							assert.strictEqual(oBinding.aContexts[0].iIndex, -2);
							assert.strictEqual(oBinding.aContexts[1].iIndex, -1);
							assert.strictEqual(oBinding.aContexts[2].iIndex, 0);
							assert.strictEqual(oBinding.aContexts[3].iIndex, 1);
							assert.strictEqual(oBinding.aContexts[5].iIndex, 3);
							assert.strictEqual(oBinding.aContexts[6].iIndex, 4);
							assert.strictEqual(oBinding.iCreatedContexts, 2);
							assert.strictEqual(oBinding.iMaxLength, 5);
						} // else removeCreated adjusted aContexts
					} else {
						that.mock(oGroupLock).expects("getGroupId").returns("resultingGroupId");
						oContextMock.expects("refreshDependentBindings")
							.withExactArgs("EMPLOYEES('2')", "resultingGroupId")
							.returns(oRefreshDependentsPromise);
					}
				}));

				oContextMock.expects("getPath").returns("/EMPLOYEES('2')");
				this.mock(oBinding).expects("withCache")
					.withExactArgs(sinon.match.func)
					.callsArgWith(0, oCache, "path/in/cache", oRootBinding);
				oContextMock.expects("getModelIndex").withExactArgs().returns(42);
				this.mock(oBinding.oHeaderContext).expects("getPath").withExactArgs()
					.returns("/EMPLOYEES");
				this.mock(_Helper).expects("getRelativePath")
					.withExactArgs("/EMPLOYEES('2')", "/EMPLOYEES")
					.returns("~key~predicate~");
				oExpectation = this.mock(oCache).expects("refreshSingleWithRemove")
					.withExactArgs(sinon.match.same(oGroupLock), "path/in/cache", 42,
						"~key~predicate~", false, sinon.match.func, sinon.match.func)
					.callsArg(5) //fireDataRequested
					.returns(oCacheRequestPromise);
				this.mock(oBinding).expects("fireDataRequested").withExactArgs();
				this.mock(oBinding).expects("fireDataReceived").withExactArgs({data : {}});
				oContextMock.expects("checkUpdateInternal").exactly(bOnRemoveCalled ? 0 : 1)
					.withExactArgs()
					.returns(new SyncPromise(function (resolve) {
						setTimeout(function () {
							bContextUpdated = true;
							resolve();
						});
					}));

				// code under test
				return oBinding.refreshSingle(oContext, oGroupLock, true).then(function () {
					assert.strictEqual(bContextUpdated, !bOnRemoveCalled);
					assert.strictEqual(bDependentsRefreshed, !bOnRemoveCalled);
				});
			});
		});
	});

	//*********************************************************************************************
[
	{index : undefined, stillAlive : false},
	/*{index : undefined, stillAlive : true*} combination is never called*/
	{index : 1, stillAlive : false},
	{index : 1, stillAlive : true}
].forEach(function (oFixture) {
	var sTitle = "refreshSingle with allow remove on a kept-alive context, index = "
		+ oFixture.index + ", stillAlive = " + oFixture.stillAlive;

	QUnit.test(sTitle, function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oCache = {
				refreshSingleWithRemove : function () {}
			},
			oCacheRequestPromise,
			oContext = {
				checkUpdateInternal : function () {},
				created : function () { return false; },
				destroy : function () {},
				getModelIndex : function () { return oFixture.index; },
				getPath : function () { return "~context~path~"; },
				isKeepAlive : function () { return true; },
				refreshDependentBindings : function () {}
			},
			oExpectation,
			oGroupLock = {getGroupId : function () {}},
			oRootBinding = {
				assertSameCache : function () {},
				getGroupId : function () {}
			},
			that = this;

		oBinding.oCache = oCache;
		oBinding.oCachePromise = SyncPromise.resolve(oCache);

		// simulate current state
		oBinding.aContexts = [{}];
		if (oFixture.index) {
			oBinding.aContexts[oFixture.index] = oContext;
		} else {
			oBinding.mPreviousContextsByPath = {"~context~path~" : oContext};
		}
		oBinding.iMaxLength = 42;

		oCacheRequestPromise = SyncPromise.resolve(Promise.resolve()).then(function () {
			// fnOnRemove Test
			that.mock(oContext).expects("destroy").exactly(oFixture.stillAlive ? 0 : 1)
				.withExactArgs();
			that.mock(oBinding).expects("_fireChange").exactly(oFixture.index ? 1 : 0)
				.withExactArgs({reason : ChangeReason.Remove});

			// code under test
			oExpectation.firstCall.args[6](oFixture.stillAlive);

			assert.strictEqual(oBinding.aContexts.length, 1);
			assert.notOk(1 in oBinding.aContexts);
			assert.strictEqual(oBinding.iMaxLength, oFixture.index ? 41 : 42);

			if (oFixture.stillAlive) {
				assert.strictEqual(oBinding.mPreviousContextsByPath["~context~path~"], oContext);
			} else {
				assert.notOk("~context~path~" in oBinding.mPreviousContextsByPath);
			}
		});

		this.mock(oBinding).expects("withCache")
			.withExactArgs(sinon.match.func)
			.callsArgWith(0, oCache, "path/in/cache", oRootBinding);
		this.mock(oBinding.oHeaderContext).expects("getPath").withExactArgs()
			.returns("~header~context~path~");
		this.mock(_Helper).expects("getRelativePath")
			.withExactArgs("~context~path~", "~header~context~path~")
			.returns("~key~predicate~");
		oExpectation = this.mock(oCache).expects("refreshSingleWithRemove")
			.withExactArgs(sinon.match.same(oGroupLock), "path/in/cache", oFixture.index,
				"~key~predicate~", true, sinon.match.func, sinon.match.func)
			.callsArg(5) //fireDataRequested
			.returns(oCacheRequestPromise);
		this.mock(oBinding).expects("fireDataRequested").withExactArgs();
		this.mock(oBinding).expects("fireDataReceived").withExactArgs({data : {}});

		this.mock(oContext).expects("checkUpdateInternal").exactly(oFixture.stillAlive ? 1 : 0)
			.withExactArgs().resolves();
		this.mock(oGroupLock).expects("getGroupId").exactly(oFixture.stillAlive ? 1 : 0)
			.withExactArgs().returns("groupId");
		this.mock(oContext).expects("refreshDependentBindings").exactly(oFixture.stillAlive ? 1 : 0)
			.withExactArgs("context~path~", "groupId").resolves();

		// code under test
		return oBinding.refreshSingle(oContext, oGroupLock, true);
	});
});

	//*********************************************************************************************
	QUnit.test("refreshSingle, no fireDataReceived if no fireDataRequested", function () {
		var oBinding = this.bindList("/EMPLOYEES"),
			oCache = {
				refreshSingle : function () {}
			},
			oContext,
			oGroupLock = {getGroupId : function () {}};

		// initialize with 3 contexts and bLengthFinal===true
		oBinding.createContexts(0, createData(3, 0, true, 3));

		oContext = oBinding.aContexts[2];
		oBinding.oCache = oCache;
		oBinding.oCachePromise = SyncPromise.resolve(oCache);

		this.mock(oBinding).expects("fireDataRequested").never();
		this.mock(oBinding).expects("fireDataReceived").never();

		this.mock(oContext).expects("getPath").withExactArgs().returns("/EMPLOYEES('2')");
		this.mock(oContext).expects("getModelIndex").withExactArgs().returns(42);
		this.mock(oContext).expects("isKeepAlive").withExactArgs().returns("~keep~alive~");
		this.mock(oBinding.oHeaderContext).expects("getPath").withExactArgs().returns("/EMPLOYEES");
		this.mock(_Helper).expects("getRelativePath")
			.withExactArgs("/EMPLOYEES('2')", "/EMPLOYEES")
			.returns("~key~predicate~");
		this.mock(oCache).expects("refreshSingle")
			.withExactArgs(sinon.match.same(oGroupLock), "", 42, "~key~predicate~", "~keep~alive~",
				sinon.match.func)
			.returns(SyncPromise.resolve({/*refreshed entity*/}));
		this.mock(oGroupLock).expects("getGroupId").withExactArgs().returns("groupId");

		// code under test
		oBinding.refreshSingle(oContext, oGroupLock);
	});

	//*********************************************************************************************
	[true, false].forEach(function (bDataRequested) {
		QUnit.test("refreshSingle, error handling: dataRequested already fired: " + bDataRequested,
				function (assert) {
			var oBinding = this.bindList("/EMPLOYEES"),
				oCache = {refreshSingle : function () {}},
				oContext = {
					getModelIndex : function () {},
					getPath : function () { return "/EMPLOYEES('1')"; },
					isKeepAlive : function () { return "~keep~alive~"; },
					refreshDependentBindings : function () {},
					toString : function () { return "Foo"; }
				},
				oError = new Error(),
				oExpectation,
				oGroupLock = {
					getGroupId : function () {},
					unlock : function () {}
				};

			oBinding.oCache = oCache;
			oBinding.oCachePromise = SyncPromise.resolve(oCache);

			this.mock(oBinding.oHeaderContext).expects("getPath").withExactArgs()
				.returns("/EMPLOYEES");
			this.mock(_Helper).expects("getRelativePath")
				.withExactArgs("/EMPLOYEES('1')", "/EMPLOYEES")
				.returns("~key~predicate~");
			this.mock(oContext).expects("getModelIndex").withExactArgs().returns(42);
			oExpectation = this.mock(oCache).expects("refreshSingle")
				.withExactArgs(sinon.match.same(oGroupLock), "", 42, "~key~predicate~",
					"~keep~alive~", sinon.match.func)
				.returns(Promise.reject(oError));
			if (bDataRequested) {
				oExpectation.callsArg(5);
			}
			this.mock(oGroupLock).expects("getGroupId").withExactArgs().returns("groupId");
			this.mock(oContext).expects("refreshDependentBindings")
				.withExactArgs("EMPLOYEES('1')", "groupId").resolves();
			this.mock(oBinding).expects("fireDataRequested")
				.exactly(bDataRequested ? 1 : 0)
				.withExactArgs();
			this.mock(oBinding).expects("fireDataReceived")
				.exactly(bDataRequested ? 1 : 0)
				.withExactArgs(bDataRequested ? {error : oError} : 0);
			this.mock(oGroupLock).expects("unlock").withExactArgs(true);
			this.mock(this.oModel).expects("reportError")
				.withExactArgs("Failed to refresh entity: Foo", sClassName,
					sinon.match.same(oError));

			// code under test
			return oBinding.refreshSingle(oContext, oGroupLock).then(function () {
				assert.ok(false);
			}, function (oError0) {
				assert.strictEqual(oError0, oError);
			});
		});
	});

	//*********************************************************************************************
	QUnit.test("refreshSingle: forbidden header context", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oHeaderContext = oBinding.getHeaderContext();

		assert.throws(function () {
			// code under test
			oBinding.refreshSingle(oHeaderContext);
		}, new Error("Unsupported header context: " + oHeaderContext));
	});

	//*********************************************************************************************
	[false, true].forEach(function (bInitial) {
		QUnit.test("resumeInternal: initial=" + bInitial, function (assert) {
			var sChangeReason = {/*Filter,Sort,Refresh,Change*/},
				oContext = Context.create(this.oModel, oParentBinding, "/TEAMS"),
				oBinding = this.bindList("TEAM_2_EMPLOYEES", oContext),
				oBindingMock = this.mock(oBinding),
				oDependent0 = {
					oContext : {
						isKeepAlive : function () {}
					},
					resumeInternal : function () {}
				},
				oDependent1 = {
					oContext : {
						isKeepAlive : function () {}
					},
					resumeInternal : function () {}
				},
				oFetchCacheExpectation,
				oFireExpectation,
				oGetDependentBindingsExpectation,
				oHeaderContextCheckUpdateExpectation,
				oResetExpectation;

			oBinding.bSharedRequest = true; // this must not have an influence
			oBinding.sChangeReason = bInitial ? "AddVirtualContext" : undefined;
			oBinding.sResumeChangeReason = sChangeReason;
			oBindingMock.expects("removeCachesAndMessages").withExactArgs("");
			oResetExpectation = oBindingMock.expects("reset").withExactArgs();
			oFetchCacheExpectation = oBindingMock.expects("fetchCache")
				.withExactArgs(sinon.match.same(oContext), true);
			oBindingMock.expects("refreshKeptElements").never();
			oGetDependentBindingsExpectation = oBindingMock.expects("getDependentBindings")
				.withExactArgs()
				.returns([oDependent0, oDependent1]);
			this.mock(oDependent0.oContext).expects("isKeepAlive").withExactArgs().returns(false);
			this.mock(oDependent0).expects("resumeInternal").withExactArgs(false, true);
			this.mock(oDependent1.oContext).expects("isKeepAlive").withExactArgs().returns(true);
			this.mock(oDependent1).expects("resumeInternal").withExactArgs(false, false);
			if (bInitial) {
				oFireExpectation = oBindingMock.expects("_fireChange")
					.withExactArgs({
						detailedReason : "AddVirtualContext",
						reason : sinon.match.same(sChangeReason)
					});
			} else {
				oFireExpectation = oBindingMock.expects("_fireRefresh")
					.withExactArgs({reason : sinon.match.same(sChangeReason)});
			}
			oHeaderContextCheckUpdateExpectation = this.mock(oBinding.oHeaderContext)
				.expects("checkUpdate").withExactArgs();

			// code under test
			oBinding.resumeInternal(true/*ignored*/);

			assert.strictEqual(oBinding.sResumeChangeReason, undefined);
			assert.ok(oResetExpectation.calledAfter(oGetDependentBindingsExpectation));
			assert.ok(oFetchCacheExpectation.calledAfter(oResetExpectation));
			assert.ok(oFireExpectation.calledAfter(oFetchCacheExpectation));
			assert.ok(oHeaderContextCheckUpdateExpectation.calledAfter(oFireExpectation));
		});
	});
	//TODO This is very similar to ODCB#resumeInternal; both should be refactored to
	//  ODParentBinding#resumeInternal. Differences
	// (a) bCheckUpdate parameter: dependent bindings of a list binding must not call checkUpdate on
	//     dependent bindings while context bindings have to; analogous to #refreshInternal.
	// (b) the "header context" of the list binding must update it's dependent bindings only after
	//     _fireChange leading to a new request, see ODLB#reset.
	// We need to have integration tests first for both differences.

	//*********************************************************************************************
[false, true].forEach(function (bAutoExpandSelect) {
	var sTitle = "resumeInternal: initial binding, bAutoExpandSelect = " + bAutoExpandSelect;

	QUnit.test(sTitle, function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oBindingMock = this.mock(oBinding);

		oBinding.sResumeChangeReason = "~sResumeChangeReason~";
		if (bAutoExpandSelect) {
			oBinding.sChangeReason = "AddVirtualContext";
			oBindingMock.expects("_fireChange").withExactArgs({
				detailedReason : "AddVirtualContext",
				reason : "~sResumeChangeReason~"
			});
		} else {
			oBindingMock.expects("_fireRefresh").withExactArgs({reason : "~sResumeChangeReason~"});
		}

		// code under test
		oBinding.resumeInternal();

		assert.strictEqual(oBinding.sResumeChangeReason, undefined);
	});
});

	//*********************************************************************************************
	QUnit.test("resumeInternal: no sResumeChangeReason", function () {
		var oBinding = this.bindList("/EMPLOYEES"),
			oDependent0 = {resumeInternal : function () {}},
			oDependent1 = {resumeInternal : function () {}};

		oBinding.sResumeChangeReason = undefined;

		this.mock(oBinding).expects("removeCachesAndMessages").never();
		this.mock(oBinding).expects("reset").never();
		this.mock(oBinding).expects("fetchCache").never();
		this.mock(oBinding).expects("refreshKeptElements").never();
		this.mock(oBinding).expects("getDependentBindings").withExactArgs()
			.returns([oDependent0, oDependent1]);
		this.mock(oDependent0).expects("resumeInternal").withExactArgs(true, false);
		this.mock(oDependent1).expects("resumeInternal").withExactArgs(true, false);
		this.mock(oBinding).expects("_fireRefresh").never();

		// code under test
		oBinding.resumeInternal(true/*ignored*/);
	});

	//*********************************************************************************************
[false, true].forEach(function (bRefreshKeptElements) {
	var sTitle = "resumeInternal: no sResumeChangeReason but parent has"
			+ "; bRefreshKeptElements=" + bRefreshKeptElements;

	QUnit.test(sTitle, function (assert) {
		var oContext = {},
			oBinding = this.bindList("/EMPLOYEES", oContext);

		oBinding.sResumeChangeReason = undefined;
		oBinding.bRefreshKeptElements = bRefreshKeptElements;
		this.mock(oBinding).expects("removeCachesAndMessages").withExactArgs("");
		this.mock(oBinding).expects("reset").withExactArgs();
		this.mock(oBinding).expects("getGroupId").exactly(bRefreshKeptElements ? 1 : 0)
			.returns("myGroup");
		this.mock(oBinding).expects("refreshKeptElements").exactly(bRefreshKeptElements ? 1 : 0)
			.withExactArgs("myGroup");
		this.mock(oBinding).expects("fetchCache")
			.withExactArgs(sinon.match.same(oContext), false);
		this.mock(oBinding).expects("_fireRefresh").never();

		// code under test
		oBinding.resumeInternal(true/*ignored*/, true);

		assert.strictEqual(oBinding.bRefreshKeptElements, false);
	});
});

	//*********************************************************************************************
	QUnit.test("resumeInternal: suspend in change event of resume", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		oBinding.sResumeChangeReason = ChangeReason.Filter;
		this.mock(oBinding).expects("_fireRefresh").withExactArgs({reason : ChangeReason.Filter})
			.callsFake(function () {
				// simulate a suspend and a sort
				oBinding.sResumeChangeReason = ChangeReason.Sort;
			});

		// code under test
		oBinding.resumeInternal(true/*ignored*/);

		assert.strictEqual(oBinding.sResumeChangeReason, ChangeReason.Sort);
	});

	//*********************************************************************************************
	QUnit.test("resumeInternal: shared cache, after refresh", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES", null, null, null, {$$sharedRequest : true});

		oBinding.sResumeAction = "resetCache";
		this.mock(oBinding).expects("getDependentBindings").withExactArgs()
			.returns(["do", "not", "use"]);
		this.mock(oBinding).expects("removeCachesAndMessages").withExactArgs("");
		this.mock(oBinding.oCache).expects("reset").withExactArgs([]);
		this.mock(oBinding).expects("onChange").never();
		this.mock(oBinding).expects("fetchCache").never();
		this.mock(oBinding).expects("refreshKeptElements").never();

		// code under test
		oBinding.resumeInternal(true/*ignored*/);

		assert.strictEqual(oBinding.sResumeChangeReason, undefined);
	});

	//*********************************************************************************************
	QUnit.test("resumeInternal: shared cache, after onChange", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES", null, null, null, {$$sharedRequest : true});

		oBinding.sResumeAction = "onChange";
		this.mock(oBinding).expects("getDependentBindings").withExactArgs()
			.returns(["do", "not", "use"]);
		this.mock(oBinding).expects("removeCachesAndMessages").withExactArgs("");
		this.mock(oBinding.oCache).expects("reset").never();
		this.mock(oBinding).expects("onChange").withExactArgs();
		this.mock(oBinding).expects("fetchCache").never();
		this.mock(oBinding).expects("refreshKeptElements").never();

		// code under test
		oBinding.resumeInternal(true/*ignored*/);

		assert.strictEqual(oBinding.sResumeChangeReason, undefined);
	});

	//*********************************************************************************************
	QUnit.test("getDependentBindings", function (assert) {
		var oActiveBinding = {
				oContext : {
					getPath : function () { return "/FOO('1')/active"; },
					isKeepAlive : function () { return false; }
				}
			},
			oBinding = this.oModel.bindList("/FOO"),
			oInactiveBinding = {
				oContext : {
					getPath : function () { return "/FOO('1')/inactive"; },
					isKeepAlive : function () { return false; }
				}
			},
			oKeptBinding = {
				oContext : {
					getPath : function () { return "/kept"; },
					isKeepAlive : function () { return true; }
				}
			},
			aDependentBindings = [oActiveBinding, oInactiveBinding, oKeptBinding];

		// simulate inactive binding
		oBinding.mPreviousContextsByPath["/FOO('1')/inactive"] = {};
		// simulate binding form a kept-alive context
		oBinding.mPreviousContextsByPath["kept"] = {};

		this.mock(this.oModel).expects("getDependentBindings")
			.withExactArgs(sinon.match.same(oBinding))
			.returns(aDependentBindings);

		// code under test
		assert.deepEqual(oBinding.getDependentBindings(), [oActiveBinding, oKeptBinding]);
	});

	//*********************************************************************************************
	[true, false].forEach(function (bWithStaticFilter) {
		QUnit.test("getFilterInfo with static filter: " + bWithStaticFilter, function (assert) {
			var aApplicationFilter = [new Filter("AmountIn%E2%82%AC", FilterOperator.GT, "1000")],
				oAST = {},
				oBinding = this.bindList("/Set"),
				oCombinedFilter = {
					getAST : function () {}
				},
				aControlFilter = [new Filter("AmountIn%E2%82%AC", FilterOperator.GT, "1000")],
				oExpectedFilterInfo = {
					left : {},
					op : "&&",
					right : {
						expression : "someFilterExpression",
						syntax : "OData 4.0",
						type : "Custom"
					},
					type : "Logical"
				},
				bIncludeOrigin = {/*true or false*/},
				oResultAST;

			oBinding.aApplicationFilters = aApplicationFilter;
			oBinding.aFilters = aControlFilter;
			if (bWithStaticFilter) {
				oBinding.mQueryOptions.$filter = "someFilterExpression";
			}
			this.mock(FilterProcessor).expects("combineFilters")
				.withExactArgs(sinon.match.same(aControlFilter),
					sinon.match.same(aApplicationFilter))
				.returns(oCombinedFilter);
			this.mock(oCombinedFilter).expects("getAST")
				.withExactArgs(sinon.match.same(bIncludeOrigin))
				.returns(oAST);

			// code under test
			oResultAST = oBinding.getFilterInfo(bIncludeOrigin);

			if (bWithStaticFilter) {
				assert.deepEqual(oResultAST, oExpectedFilterInfo);
			} else {
				assert.strictEqual(oResultAST, oAST);
			}
		});
	});

	//*********************************************************************************************
	QUnit.test("getFilterInfo: no filters", function (assert) {
		var aApplicationFilter = [],
			oBinding = this.bindList("/Set"),
			aControlFilter = [],
			bIncludeOrigin = {/*true or false*/};

		oBinding.aApplicationFilters = aApplicationFilter;
		oBinding.aFilters = aControlFilter;
		this.mock(FilterProcessor).expects("combineFilters")
			.withExactArgs(sinon.match.same(aControlFilter), sinon.match.same(aApplicationFilter))
			.returns(undefined);

		// code under test
		assert.strictEqual(oBinding.getFilterInfo(bIncludeOrigin), null);
	});

	//*********************************************************************************************
	QUnit.test("getFilterInfo: with only static filter", function (assert) {
		var aApplicationFilter = [],
			oBinding = this.bindList("/Set"),
			aControlFilter = [],
			sODataVersion = "foo",
			oExpectedFilterInfo = {
				expression : "someFilterExpression",
				syntax : "OData " + sODataVersion,
				type : "Custom"
			},
			bIncludeOrigin = {/*true or false*/};

		oBinding.aApplicationFilters = aApplicationFilter;
		oBinding.aFilters = aControlFilter;
		oBinding.mQueryOptions.$filter = "someFilterExpression";
		this.mock(FilterProcessor).expects("combineFilters")
			.withExactArgs(sinon.match.same(aControlFilter), sinon.match.same(aApplicationFilter))
			.returns(undefined);
		this.mock(this.oModel).expects("getODataVersion")
			.returns(sODataVersion);

		// code under test
		assert.deepEqual(oBinding.getFilterInfo(bIncludeOrigin),
			oExpectedFilterInfo);
	});

	//*********************************************************************************************
[false, true].forEach(function (bHeader) {
	QUnit.test("requestSideEffects: refresh needed, refresh fails, " + bHeader, function (assert) {
		var oCacheMock = this.getCacheMock(), // must be called before creating the binding
			oBinding = this.bindList("/Set"),
			oContext = bHeader ? oBinding.getHeaderContext() : undefined,
			oError = new Error(),
			sGroupId = "group";

		oBinding.iCurrentEnd = 42;
		oCacheMock.expects("isDeletingInOtherGroup").withExactArgs(sGroupId).returns(false);

		this.mock(oBinding).expects("lockGroup").never();
		oCacheMock.expects("requestSideEffects").never();
		this.mock(oBinding).expects("refreshInternal").withExactArgs("", sGroupId, false, true)
			.rejects(oError);

		// code under test
		return oBinding.requestSideEffects(sGroupId, ["n/a", ""], oContext).then(function () {
			assert.ok(false);
		}, function (oError0) {
			assert.strictEqual(oError0, oError);
		});
	});
});

	//*********************************************************************************************
	QUnit.test("requestSideEffects: refreshSingle needed, refreshSingle fails", function (assert) {
		var oCacheMock = this.getCacheMock(), // must be called before creating the binding
			oContext = {},
			oBinding = this.bindList("/Set"),
			oError = new Error(),
			sGroupId = "group",
			oGroupLock = {};

		oCacheMock.expects("isDeletingInOtherGroup").never();
		this.mock(oBinding).expects("lockGroup").withExactArgs(sGroupId).returns(oGroupLock);
		oCacheMock.expects("requestSideEffects").never();
		this.mock(oBinding).expects("refreshSingle")
			.withExactArgs(sinon.match.same(oContext), sinon.match.same(oGroupLock), false)
			.rejects(oError);

		// code under test
		return oBinding.requestSideEffects(sGroupId, ["n/a", ""], oContext).then(function () {
			assert.ok(false);
		}, function (oError0) {
			assert.strictEqual(oError0, oError);
		});
	});

	//*********************************************************************************************
[false, true].forEach(function (bHeader) {
	QUnit.test("requestSideEffects: deleting in other group, " + bHeader, function (assert) {
		var oCacheMock = this.getCacheMock(), // must be called before creating the binding
			oBinding = this.bindList("/Set"),
			oContext = bHeader ? oBinding.getHeaderContext() : undefined;

		oCacheMock.expects("isDeletingInOtherGroup").withExactArgs("group").returns(true);
		oCacheMock.expects("requestSideEffects").never();
		this.mock(oBinding).expects("refreshSingle").never();

		// code under test
		assert.throws(function () {
			oBinding.requestSideEffects("group", ["n/a", ""], oContext);
		}, new Error("Must not request side effects when there is a pending delete in a different "
			+ "batch group"));
	});
});

	//*********************************************************************************************
	QUnit.test("requestSideEffects: call refreshInternal for relative binding", function (assert) {
		var oBinding = this.bindList("relative", this.oModel.createBindingContext("/")),
			oContext = oBinding.getHeaderContext(),
			oResult = {};

		oBinding.iCurrentEnd = 42;
		this.mock(oBinding).expects("refreshSingle").never();
		this.mock(oBinding).expects("refreshInternal").withExactArgs("", "group", false, true)
			.resolves(oResult);

		// code under test
		return oBinding.requestSideEffects("group", [""], oContext).then(function (oResult0) {
			assert.strictEqual(oResult0, oResult);
		});
	});

	//*********************************************************************************************
	QUnit.test("requestSideEffects: call refreshSingle for relative binding", function (assert) {
		var oBinding = this.bindList("relative", this.oModel.createBindingContext("/")),
			oContext = Context.create(this.oModel, {}, "/EMPLOYEES('42')"),
			oGroupLock = {},
			oResult = {};

		this.mock(oBinding).expects("lockGroup").withExactArgs("group").returns(oGroupLock);
		this.mock(oBinding).expects("refreshSingle")
			.withExactArgs(sinon.match.same(oContext), sinon.match.same(oGroupLock), false)
			.resolves(oResult);
		this.mock(oBinding).expects("refreshInternal").never();

		// code under test
		return oBinding.requestSideEffects("group", [""], oContext).then(function (oResult0) {
			assert.strictEqual(oResult0, oResult);
		});
	});

	//*********************************************************************************************
[false, true].forEach(function (bHeader) {
	[false, true].forEach(function (bRecursionRejects) {
		[false, true].forEach(function (bHasCache) {
			var sTitle = "requestSideEffects: efficient request possible, header=" + bHeader
					+ ", reject=" + bRecursionRejects + ", has cache=" + bHasCache;

	QUnit.test(sTitle, function (assert) {
		var oCacheMock = this.getCacheMock(), // must be called before creating the binding
			oBinding = this.bindList("/Set"),
			oCanceledError = new Error(),
			oContext = bHeader
				? oBinding.getHeaderContext()
				: {getPath : function () { return "/Set('foo')"; }},
			oError = new Error(),
			sGroupId = "group",
			oGroupLock = {},
			oModelMock = this.mock(this.oModel),
			oPreviousContext6,
			oPreviousContext7,
			oPreviousContext8,
			aPaths = ["A"],
			oPromise = SyncPromise.resolve(),
			oResult,
			that = this;

		function expectVisitAndRefresh(aPromises) {
			that.mock(oBinding).expects("visitSideEffects").withExactArgs(sGroupId,
					sinon.match.same(aPaths), bHeader ? undefined : sinon.match.same(oContext),
					aPromises)
				.callsFake(function (_sGroupId, _aPaths, _oContext, aPromises) {
					aPromises.push(Promise.resolve());
					aPromises.push(Promise.reject(oCanceledError));
					if (bRecursionRejects) {
						aPromises.push(Promise.reject(oError));
					}
				});
			that.mock(oBinding).expects("refreshDependentListBindingsWithoutCache")
				.exactly(bRecursionRejects ? 0 : 1).withExactArgs().resolves("~");
		}

		oCanceledError.canceled = true;
		oBinding.createContexts(3, createData(9, 3, true, 9, true));
		oBinding.iCurrentBegin = 3;
		oBinding.iCurrentEnd = 6;
		oBinding.mPreviousContextsByPath["('6')"] = oPreviousContext6 = oBinding.aContexts[6];
		oBinding.mPreviousContextsByPath["('7')"] = oPreviousContext7 = oBinding.aContexts[7];
		oBinding.mPreviousContextsByPath["('8')"] = oPreviousContext8 = oBinding.aContexts[8];
		oBinding.aContexts.length = 5; // less than iCurrentEnd, this can happen due to a delete

		oCacheMock.expects("isDeletingInOtherGroup").exactly(bHeader && bHasCache ? 1 : 0)
			.withExactArgs(sGroupId).returns(false);
		this.mock(oBinding).expects("lockGroup").exactly(bHasCache ? 1 : 0)
			.withExactArgs(sGroupId).returns(oGroupLock);
		this.mock(oPreviousContext6).expects("isKeepAlive").exactly(bHeader ? 1 : 0)
			.returns(true);
		this.mock(oPreviousContext7).expects("isKeepAlive").exactly(bHeader ? 1 : 0)
			.returns(false);
		this.mock(oPreviousContext8).expects("isKeepAlive").exactly(bHeader ? 1 : 0)
			.returns(true);
		oCacheMock.expects("requestSideEffects").exactly(bHasCache ? 1 : 0)
			.withExactArgs(sinon.match.same(oGroupLock), sinon.match.same(aPaths),
				bHeader ? ["('3')", "('4')", "('6')", "('8')"] : ["('foo')"], !bHeader)
			.callsFake(function (_oGroupLock, _aPaths) {
				expectVisitAndRefresh([oPromise]);

				return oPromise;
			});
		if (!bHasCache) {
			oBinding.oCache = undefined; // not yet there
			expectVisitAndRefresh([]);
		}
		oModelMock.expects("reportError")
			.withExactArgs("Failed to request side effects", sClassName,
				sinon.match.same(oCanceledError));
		oModelMock.expects("reportError").exactly(bRecursionRejects ? 1 : 0)
			.withExactArgs("Failed to request side effects", sClassName, sinon.match.same(oError));
		this.mock(oBinding).expects("refreshInternal").never();

		// code under test
		oResult = oBinding.requestSideEffects(sGroupId, aPaths, oContext);

		assert.ok(oResult.isPending(), "instanceof SyncPromise");

		return oResult.then(function (vValue) {
				assert.notOk(bRecursionRejects);
				assert.strictEqual(vValue, "~",
					"refreshDependentListBindingsWithoutCache finished");
			}, function (oError0) {
				assert.ok(bRecursionRejects);
				assert.strictEqual(oError0, oError);
			});
	});
		});
	});
});

	//*********************************************************************************************
	QUnit.test("requestSideEffects: transient context", function (assert) {
		var oCacheMock = this.getCacheMock(), // must be called before creating the binding
			oBinding = this.bindList("/Set"),
			sGroupId = "group",
			oGroupLock = {},
			aPaths = ["A"],
			oPromise = SyncPromise.resolve(),
			that = this;

		oBinding.createContexts(0, createData(1, 0, true, 1, true));
		oBinding.aContexts.unshift({isTransient : function () {}});
		oBinding.iCreatedContexts = 1;
		oBinding.iCurrentBegin = 0;
		oBinding.iCurrentEnd = 2;

		oCacheMock.expects("isDeletingInOtherGroup").withExactArgs(sGroupId).returns(false);
		this.mock(oBinding).expects("lockGroup").withExactArgs(sGroupId).returns(oGroupLock);
		this.mock(oBinding.aContexts[0]).expects("isTransient").withExactArgs().returns(true);
		this.mock(oBinding.aContexts[1]).expects("isTransient").withExactArgs().returns(false);
		oCacheMock.expects("requestSideEffects")
			.withExactArgs(sinon.match.same(oGroupLock), sinon.match.same(aPaths), ["('0')"],
				undefined)
			.callsFake(function (_oGroupLock, aPaths) {
				that.mock(oBinding).expects("visitSideEffects").withExactArgs(sGroupId,
						sinon.match.same(aPaths), undefined, [oPromise]);
				that.mock(oBinding).expects("refreshDependentListBindingsWithoutCache")
					.withExactArgs().resolves("~");

				return oPromise;
			});
		this.mock(oBinding).expects("refreshInternal").never();

		// code under test
		return oBinding.requestSideEffects(sGroupId, aPaths).then(function (vValue) {
			assert.strictEqual(vValue, "~",
				"refreshDependentListBindingsWithoutCache finished");
		});
	});

	//*********************************************************************************************
	QUnit.test("requestSideEffects: fallback to refresh", function (assert) {
		var oCacheMock = this.getCacheMock(), // must be called before creating the binding
			oBinding = this.bindList("/Set"),
			oError = new Error(),
			sGroupId = "group",
			aPaths = ["A"];

		oBinding.createContexts(3, createData(8, 3, true)); // no key predicates
		oBinding.iCurrentBegin = 3;
		oBinding.iCurrentEnd = 8;
		oCacheMock.expects("isDeletingInOtherGroup").withExactArgs(sGroupId).returns(false);
		this.mock(oBinding).expects("lockGroup").never();
		oCacheMock.expects("requestSideEffects").never();
		this.mock(oBinding).expects("refreshInternal").withExactArgs("", sGroupId, false, true)
			.rejects(oError);

		// code under test
		return oBinding.requestSideEffects(sGroupId, aPaths).then(function () {
				assert.ok(false);
			}, function (oError0) {
				assert.strictEqual(oError0, oError);
			});
	});
	// Note: although a list binding's oCachePromise may become pending again due to late properties
	// being added, there is no need to wait for them to arrive. We can just request the current
	// side effects now and the late property will fetch its own value later on.

	//*********************************************************************************************
	QUnit.test("requestSideEffects: no data read => no refresh", function (assert) {
		var oCacheMock = this.getCacheMock(), // must be called before creating the binding
			oBinding = this.bindList("/Set");

		oCacheMock.expects("isDeletingInOtherGroup").withExactArgs("group").returns(false);
		this.mock(oBinding).expects("lockGroup").never();
		oCacheMock.expects("requestSideEffects").never();
		this.mock(oBinding).expects("refreshInternal").never();

		assert.strictEqual(
			// code under test
			oBinding.requestSideEffects("group", ["n/a", ""]),
			SyncPromise.resolve()
		);
	});

	//*********************************************************************************************
[false, true].forEach(function (bRefresh) {
	[false, true].forEach(function (bHeaderContext) {
		var sTitle = "requestSideEffects: $$aggregation, refresh=" + bRefresh + ", headerContext="
				+ bHeaderContext;

		QUnit.test(sTitle, function (assert) {
		var oBinding = this.bindList("/Set", undefined, undefined, undefined, {
				$$aggregation : {}
			}),
			oContext = bHeaderContext ? oBinding.getHeaderContext() : undefined,
			aFilters = [],
			aPaths = [],
			oPromise,
			oRefreshPromise = {};

		this.mock(oBinding.oCache).expects("requestSideEffects").never();
		this.mock(oBinding.aFilters).expects("concat").withExactArgs(oBinding.aApplicationFilters)
			.returns(aFilters);
		this.mock(_AggregationHelper).expects("isAffected")
			.withExactArgs(sinon.match.same(oBinding.mParameters.$$aggregation),
				sinon.match.same(aFilters), sinon.match.same(aPaths))
			.returns(bRefresh);
		this.mock(oBinding).expects("refreshInternal").exactly(bRefresh ? 1 : 0)
			.withExactArgs("", "group", false, true)
			.returns(oRefreshPromise);

		// code under test
		oPromise = oBinding.requestSideEffects("group", aPaths, oContext);

		if (bRefresh) {
			assert.strictEqual(oPromise, oRefreshPromise);
		} else {
			assert.strictEqual(oPromise, SyncPromise.resolve());
		}
	});
	});
});

	//*********************************************************************************************
	QUnit.test("requestSideEffects with $$aggregation and row context", function (assert) {
		var oBinding = this.bindList("/Set", undefined, undefined, undefined, {
				$$aggregation : {}
			});

		this.mock(oBinding.oCache).expects("requestSideEffects").never();
		this.mock(oBinding).expects("refreshInternal").never();
		this.mock(_AggregationHelper).expects("isAffected").never();

		assert.throws(function () {
			// code under test
			oBinding.requestSideEffects("group", [/*aPaths*/], {/*oContext*/});
		}, new Error(
			"Must not request side effects for a context of a binding with $$aggregation"));
	});

	//*********************************************************************************************
	QUnit.test("getQueryOptions: with system query options", function (assert) {
		var oBinding = this.bindList("/Set");

		assert.throws(function () {
			// code under test
			oBinding.getQueryOptions(/*bWithSystemQueryOptions*/true);
		}, new Error("Unsupported parameter value: bWithSystemQueryOptions: true"));
	});

	//*********************************************************************************************
	QUnit.test("getQueryOptions: without system query options", function (assert) {
		var oBinding = this.bindList("/Set", undefined, undefined, undefined, {
				$select : "a,b,c",
				custom : "query option"
			});

		// code under test
		assert.deepEqual(oBinding.getQueryOptions(/*bWithSystemQueryOptions*/),
				{custom : "query option"});
	});

	//*********************************************************************************************
	QUnit.test("getModelIndex", function (assert) {
		var oBinding = this.bindList("/Set"),
			oBindingMock = this.mock(oBinding);

		oBindingMock.expects("getLength").atLeast(0).withExactArgs().returns(10);

		// code under test
		assert.strictEqual(oBinding.getModelIndex(5), 5);
		assert.strictEqual(oBinding.getModelIndex(42), 42);

		oBinding.bLengthFinal = true;
		oBinding.iCreatedContexts = 1;

		// code under test
		assert.strictEqual(oBinding.getModelIndex(5), 5);
		assert.strictEqual(oBinding.getModelIndex(42), 42);

		oBinding.bFirstCreateAtEnd = true;

		// code under test
		assert.strictEqual(oBinding.getModelIndex(2), 3);
		assert.strictEqual(oBinding.getModelIndex(5), 6);
		assert.strictEqual(oBinding.getModelIndex(9), 0);

		oBinding.iCreatedContexts = 2;

		// code under test
		assert.strictEqual(oBinding.getModelIndex(2), 4);
		assert.strictEqual(oBinding.getModelIndex(5), 7);
		assert.strictEqual(oBinding.getModelIndex(8), 1);
		assert.strictEqual(oBinding.getModelIndex(9), 0);

		oBinding.iCreatedContexts = 0;

		// code under test
		assert.strictEqual(oBinding.getModelIndex(0), 0);
	});

	//*********************************************************************************************
	QUnit.test("attachCreateActivate/detachCreateActivate", function (assert) {
		var oBinding = this.bindList("/Set");

		this.mock(oBinding).expects("attachEvent")
			.withExactArgs("createActivate", "~function~", "~listener~")
			.returns(oBinding);

		// code under test
		assert.strictEqual(oBinding.attachCreateActivate("~function~", "~listener~"), oBinding);

		this.mock(oBinding).expects("detachEvent")
			.withExactArgs("createActivate", "~function~", "~listener~")
			.returns(oBinding);

		// code under test
		assert.strictEqual(oBinding.detachCreateActivate("~function~", "~listener~"), oBinding);
	});

	//*********************************************************************************************
	QUnit.test("attachCreateCompleted/detachCreateCompleted", function (assert) {
		var oBinding = this.bindList("/Set"),
			oBindingMock = this.mock(oBinding);

		oBindingMock.expects("attachEvent")
			.withExactArgs("createCompleted", "~function~", "~listener~")
			.returns(oBinding);

		// code under test
		assert.strictEqual(oBinding.attachCreateCompleted("~function~", "~listener~"), oBinding);

		oBindingMock.expects("detachEvent")
			.withExactArgs("createCompleted", "~function~", "~listener~")
			.returns(oBinding);

		// code under test
		assert.strictEqual(oBinding.detachCreateCompleted("~function~", "~listener~"), oBinding);
	});

	//*********************************************************************************************
	QUnit.test("attachCreateSent/detachCreateSent", function (assert) {
		var oBinding = this.bindList("/Set"),
			oBindingMock = this.mock(oBinding);

		oBindingMock.expects("attachEvent")
			.withExactArgs("createSent", "~function~", "~listener~")
			.returns(oBinding);

		// code under test
		assert.strictEqual(oBinding.attachCreateSent("~function~", "~listener~"), oBinding);

		oBindingMock.expects("detachEvent")
			.withExactArgs("createSent", "~function~", "~listener~")
			.returns(oBinding);

		// code under test
		assert.strictEqual(oBinding.detachCreateSent("~function~", "~listener~"), oBinding);
	});

	//*********************************************************************************************
[false, true].forEach(function (bHasPath, i) {
	QUnit.test("adjustPredicate: single context #" + i, function (assert) {
		var oBinding = this.bindList("/SalesOrderList"),
			oContext = {adjustPredicate : function () {}},
			oExpectation;

		oBinding.aPreviousData = bHasPath
			? ["foo", "/SalesOrderList($uid=1)", "bar"]
			: ["foo", "bar"];
		oExpectation = this.mock(oContext).expects("adjustPredicate")
			.withExactArgs("($uid=1)", "('42')", sinon.match.func);

		// code under test
		oBinding.adjustPredicate("($uid=1)", "('42')", oContext);
		oExpectation.args[0][2]("/SalesOrderList($uid=1)", "/SalesOrderList('42')");

		assert.deepEqual(oBinding.aPreviousData, bHasPath
			? ["foo", "/SalesOrderList('42')", "bar"]
			: ["foo", "bar"]);
		assert.notOk(oBinding.aPreviousData.hasOwnProperty("-1"));
	});
});

	//*********************************************************************************************
[undefined, {}].forEach(function (mCacheQueryOptions, i) {
	QUnit.test("adjustPredicate # " + i, function (assert) {
		var oBinding = this.bindList("SO_2_SOITEM",
				Context.create({/*oModel*/}, oParentBinding, "/SalesOrderList($uid=1)")),
			oContext1 = {adjustPredicate : function () {}},
			oContext2 = {adjustPredicate : function () {}},
			oExpectation1,
			oExpectation2;

		oBinding.aPreviousData = ["/SalesOrderList($uid=1)/SO_2_SOITEM($uid=2)"];
		oBinding.aContexts = [,, oContext1, oContext2]; // sparse array
		oBinding.mCacheQueryOptions = mCacheQueryOptions;
		this.mock(asODataParentBinding.prototype).expects("adjustPredicate").on(oBinding)
			.withExactArgs("($uid=1)", "('42')");
		this.mock(oBinding).expects("fetchCache").exactly(mCacheQueryOptions ? 1 : 0)
			.withExactArgs(sinon.match.same(oBinding.oContext), true);
		this.mock(oBinding.oHeaderContext).expects("adjustPredicate")
			.withExactArgs("($uid=1)", "('42')");
		oExpectation1 = this.mock(oContext1).expects("adjustPredicate")
			.withExactArgs("($uid=1)", "('42')", sinon.match.func);
		oExpectation2 = this.mock(oContext2).expects("adjustPredicate")
			.withExactArgs("($uid=1)", "('42')", sinon.match.func);

		// code under test
		oBinding.adjustPredicate("($uid=1)", "('42')");
		oExpectation1.args[0][2]("/SalesOrderList($uid=1)/SO_2_SOITEM($uid=2)",
			"/SalesOrderList('42')/SO_2_SOITEM($uid=2)");
		oExpectation2.args[0][2]("/SalesOrderList($uid=1)/SO_2_SOITEM($uid=3)",
			"/SalesOrderList('42')/SO_2_SOITEM($uid=3)");

		assert.deepEqual(oBinding.aPreviousData, ["/SalesOrderList('42')/SO_2_SOITEM($uid=2)"]);
		assert.notOk(oBinding.aPreviousData.hasOwnProperty("-1"));
	});
});

	//*********************************************************************************************
[{}, null].forEach(function (oCache, i) {
	QUnit.test("hasPendingChangesForPath: calls super, #" + i, function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			sPath = {/*string*/},
			bResult = {/*boolean*/};

		oBinding.oCache = oCache;

		this.mock(asODataParentBinding.prototype).expects("hasPendingChangesForPath")
			.on(oBinding)
			.withExactArgs(sinon.match.same(sPath))
			.returns(bResult);

		// code under test
		assert.strictEqual(oBinding.hasPendingChangesForPath(sPath), bResult);
	});
});

	//*********************************************************************************************
	QUnit.test("hasPendingChangesForPath: iActiveContexts", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		oBinding.oCache = undefined;
		oBinding.iActiveContexts = 0;
		oBinding.iCreatedContexts = 2;
		this.mock(asODataParentBinding.prototype).expects("hasPendingChangesForPath").never();

		// code under test
		assert.notOk(oBinding.hasPendingChangesForPath());

		oBinding.iActiveContexts = 1;

		// code under test
		assert.ok(oBinding.hasPendingChangesForPath());
	});

	//*********************************************************************************************
	QUnit.test("doSetProperty: returns undefined", function (assert) {
		// code under test
		assert.strictEqual(this.bindList("/EMPLOYEES").doSetProperty(), undefined);
	});

	//*********************************************************************************************
	QUnit.test("fetchDownloadUrl", function (assert) {
		var oBinding = this.bindList("n/a"),
			oCache = {
				getDownloadUrl : function () {}
			},
			oExpectation,
			oPromise = {};

		this.mock(oBinding).expects("isResolved").returns(true);
		oExpectation = this.mock(oBinding).expects("withCache").returns(oPromise);

		// code under test
		assert.strictEqual(oBinding.fetchDownloadUrl(), oPromise);

		this.mock(oCache).expects("getDownloadUrl")
			.withExactArgs("~path~", sinon.match.same(this.oModel.mUriParameters))
			.returns("~url~");

		// code under test - callback function
		assert.strictEqual(oExpectation.args[0][0](oCache, "~path~"), "~url~");
	});

	//*********************************************************************************************
	QUnit.test("fetchDownloadUrl: unresolved", function (assert) {
		var oBinding = this.bindList("n/a");

		this.mock(oBinding).expects("isResolved").returns(false);

		assert.throws(function () {
			oBinding.fetchDownloadUrl();
		}, new Error("Binding is unresolved"));
	});

	//*********************************************************************************************
[false, true].forEach(function (bSuccess) {
	QUnit.test("requestDownloadUrl: success=" + bSuccess, function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oError = new Error(),
			oPromise;

		this.mock(oBinding).expects("fetchDownloadUrl").withExactArgs()
			.returns(SyncPromise.resolve(
				bSuccess ? Promise.resolve("/service/resource?query") : Promise.reject(oError)
			));

		oPromise = oBinding.requestDownloadUrl();

		assert.ok(oPromise instanceof Promise);

		return oPromise.then(function (sResult) {
			assert.ok(bSuccess);
			assert.strictEqual(sResult, "/service/resource?query");
		}, function (oResult) {
			assert.notOk(bSuccess);
			assert.strictEqual(oResult, oError);
		});
	});
});

	//*********************************************************************************************
	QUnit.test("getDownloadUrl: success", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		this.mock(oBinding).expects("fetchDownloadUrl").withExactArgs()
			.returns(SyncPromise.resolve("/service/resource?query"));

		assert.strictEqual(oBinding.getDownloadUrl(), "/service/resource?query");
	});

	//*********************************************************************************************
	QUnit.test("getDownloadUrl: result pending", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oPromise = SyncPromise.resolve(Promise.resolve("/service/resource?query"));

		this.mock(oBinding).expects("fetchDownloadUrl").withExactArgs()
			.returns(oPromise);

		assert.throws(function () {
			// code under test
			oBinding.getDownloadUrl();
		}, new Error("Result pending"));

		return oPromise;
	});

	//*********************************************************************************************
	QUnit.test("getDownloadUrl: error", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oError = new Error("Failure");

		this.mock(oBinding).expects("fetchDownloadUrl").withExactArgs()
			.returns(SyncPromise.reject(oError));

		assert.throws(function () {
			// code under test
			oBinding.getDownloadUrl();
		}, oError);
	});

	//*********************************************************************************************
	QUnit.test("checkKeepAlive", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		// code under test
		oBinding.checkKeepAlive({/*oContext*/});

		assert.throws(function () {
			// code under test
			oBinding.checkKeepAlive(oBinding.getHeaderContext());
		}, new Error("Unsupported header context " + oBinding.getHeaderContext()));
	});

	//*********************************************************************************************
	QUnit.test("checkKeepAlive: $$aggregation", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES", undefined, undefined, undefined,
				{$$aggregation : {}});

		assert.throws(function () {
			// code under test
			oBinding.checkKeepAlive({/*oContext*/});
		}, new Error("Unsupported $$aggregation at " + oBinding));
	});

	//*********************************************************************************************
	QUnit.test("checkKeepAlive: $$sharedRequest", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES", undefined, undefined, undefined,
				{$$sharedRequest : true});

		assert.throws(function () {
			// code under test
			oBinding.checkKeepAlive({/*oContext*/});
		}, new Error("Unsupported $$sharedRequest at " + oBinding));
	});

	//*********************************************************************************************
	QUnit.test("checkKeepAlive: sharedRequests from model", function (assert) {
		var oModel = new ODataModel({
				serviceUrl : "/service/?sap-client=111",
				synchronizationMode : "None",
				sharedRequests : true
			}),
			oBinding = oModel.bindList("/EMPLOYEES");

		assert.throws(function () {
			// code under test
			oBinding.checkKeepAlive({/*oContext*/});
		}, new Error("Unsupported $$sharedRequest at " + oBinding));
	});

	//*********************************************************************************************
	QUnit.test("checkKeepAlive: relative", function (assert) {
		var oBinding,
			oParentContext = Context.createNewContext({/*oModel*/}, oParentBinding, "/TEAMS('1')");

		oBinding = this.bindList("TEAM_2_EMPLOYEES", oParentContext);

		assert.throws(function () {
			// code under test
			oBinding.checkKeepAlive({/*oContext*/});
		}, new Error("Missing $$ownRequest at " + oBinding));

		oBinding = this.bindList("TEAM_2_EMPLOYEES", oParentContext, undefined, undefined,
			{$$ownRequest : true});

		// code under test
		oBinding.checkKeepAlive({/*oContext*/});
	});

	//*********************************************************************************************
[false, true].forEach(function (bSuccess) {
	[false, true].forEach(function (bDataRequested) {
		[0, 3].forEach(function (iCount) { // 0 means collapse before expand has finished
			var sTitle = "expand: success=" + bSuccess + ", data requested=" + bDataRequested
					+ ", count=" + iCount;

	if (!bSuccess && !iCount) { // ignore useless combination
		return;
	}

	QUnit.test(sTitle, function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oContext = {
				getModelIndex : function () {},
				getPath : function () {},
				toString : function () { return "~context~"; }
			},
			oChangeCall,
			aContextsBefore,
			oDataReceivedCall,
			oError = new Error(),
			oExpectation,
			oGroupLock = {},
			oPromise,
			that = this;

		oBinding.oCache = { // simulate an aggregation cache
			expand : function () {}
		};
		oBinding.createContexts(0, createData(2, 0, true, 5));
		oBinding.createContexts(3, createData(2, 3, true, 5));
		aContextsBefore = oBinding.aContexts.slice();

		this.mock(oBinding).expects("checkSuspended").withExactArgs();
		this.mock(oBinding).expects("lockGroup").withExactArgs().returns(oGroupLock);
		this.mock(oContext).expects("getPath").withExactArgs().returns("~contextpath~");
		this.mock(oBinding.oHeaderContext).expects("getPath").withExactArgs()
			.returns("~bindingpath~");
		this.mock(_Helper).expects("getRelativePath")
			.withExactArgs("~contextpath~", "~bindingpath~").returns("~cachepath~");

		oExpectation = this.mock(oBinding.oCache).expects("expand")
			.withExactArgs(sinon.match.same(oGroupLock), "~cachepath~", sinon.match.func)
			.returns(Promise.resolve().then(function () {
				if (bSuccess) {
					that.mock(oContext).expects("getModelIndex").exactly(iCount ? 1 : 0)
						.withExactArgs().returns(1);
					oChangeCall = that.mock(oBinding).expects("_fireChange").exactly(iCount ? 1 : 0)
						.withExactArgs({reason : ChangeReason.Change});
					oDataReceivedCall = that.mock(oBinding).expects("fireDataReceived")
						.exactly(bDataRequested ? 1 : 0).withExactArgs({});

					return iCount;
				}
				that.mock(oBinding).expects("fireDataReceived").exactly(bDataRequested ? 1 : 0)
					.withExactArgs({error : sinon.match.same(oError)});

				throw oError;
			}));

		// code under test
		oPromise = oBinding.expand(oContext).then(function () {
			assert.ok(bSuccess);
			assert.strictEqual(oBinding.getLength(), 5 + iCount);
			assert.strictEqual(oBinding.aContexts.length, 5 + iCount);
			assert.strictEqual(oBinding.aContexts[0], aContextsBefore[0], "0");
			assert.strictEqual(oBinding.aContexts[0].iIndex, 0);
			assert.strictEqual(oBinding.aContexts[1], aContextsBefore[1], "1");
			assert.strictEqual(oBinding.aContexts[1].iIndex, 1);
			assert.notOk(2 in oBinding.aContexts, "2");
			if (iCount) {
				assert.notOk(3 in oBinding.aContexts, "3");
				assert.notOk(4 in oBinding.aContexts, "4");
				assert.notOk(5 in oBinding.aContexts, "5");
				assert.strictEqual(oBinding.aContexts[6], aContextsBefore[3], "6");
				assert.strictEqual(oBinding.aContexts[6].iIndex, 6);
				assert.strictEqual(oBinding.aContexts[7], aContextsBefore[4], "7");
				assert.strictEqual(oBinding.aContexts[7].iIndex, 7);
			} else {
				assert.strictEqual(oBinding.aContexts[3], aContextsBefore[3], "3");
				assert.strictEqual(oBinding.aContexts[3].iIndex, 3);
				assert.strictEqual(oBinding.aContexts[4], aContextsBefore[4], "4");
				assert.strictEqual(oBinding.aContexts[4].iIndex, 4);
			}

			if (bDataRequested && iCount) {
				sinon.assert.callOrder(oChangeCall, oDataReceivedCall);
			}
		}, function (oResult) {
			assert.notOk(bSuccess);
			assert.strictEqual(oResult, oError);
		});

		that.mock(oBinding).expects("fireDataRequested").exactly(bDataRequested ? 1 : 0)
			.withExactArgs();
		if (bDataRequested) {
			oExpectation.args[0][2]();
		}

		return oPromise;
	});
	// TODO aContexts may be sparse
		});
	});
});

	//*********************************************************************************************
[0, 3].forEach(function (iCount) {
	QUnit.test("collapse: iCount = " + iCount, function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oCollapseExpectation,
			oContext = {
				getModelIndex : function () {},
				getPath : function () {}
			},
			aContextsBefore,
			oFireChangeExpectation,
			i;

		// create a context dummy object with index i
		function createContextDummy(i) {
			return {
				iIndex : i,
				getPath : function () {
					return "/EMPLOYEES/" + i;
				}
			};
		}

		oBinding.oCache = { // simulate an aggregation cache
			collapse : function () {}
		};
		for (i = 0; i < 8; i += 1) {
			// with gap at 6
			oBinding.aContexts.push(i === 6 ? undefined : createContextDummy(i));
		}
		oBinding.iMaxLength = 8;
		aContextsBefore = oBinding.aContexts.slice();
		assert.deepEqual(oBinding.mPreviousContextsByPath, {});
		this.mock(oContext).expects("getModelIndex").withExactArgs().returns(1);
		this.mock(oContext).expects("getPath").withExactArgs().returns("~contextpath~");
		this.mock(oBinding.oHeaderContext).expects("getPath").withExactArgs()
			.returns("~bindingpath~");
		this.mock(_Helper).expects("getRelativePath")
			.withExactArgs("~contextpath~", "~bindingpath~").returns("~cachepath~");
		oCollapseExpectation = this.mock(oBinding.oCache).expects("collapse")
			.withExactArgs("~cachepath~").returns(iCount);
		oFireChangeExpectation = this.mock(oBinding).expects("_fireChange").exactly(iCount ? 1 : 0)
			.withExactArgs({reason : ChangeReason.Change});

		// code under test
		oBinding.collapse(oContext);

		if (iCount) {
			sinon.assert.callOrder(oCollapseExpectation, oFireChangeExpectation);
			assert.strictEqual(oBinding.aContexts[0], aContextsBefore[0], "0");
			assert.strictEqual(oBinding.aContexts[1], aContextsBefore[1], "1");
			assert.strictEqual(oBinding.aContexts[2], aContextsBefore[5], "2");
			assert.strictEqual(oBinding.aContexts[4], aContextsBefore[7], "4");
			assert.strictEqual(oBinding.aContexts.length, 5);
			assert.strictEqual(oBinding.iMaxLength, 5);
			oBinding.aContexts.forEach(function (oContext, iIndex) {
				if (iIndex !== 3) { // 6 - iCount
					assert.strictEqual(oContext.iIndex, iIndex);
				}
			});
			assert.deepEqual(oBinding.mPreviousContextsByPath, {
				"/EMPLOYEES/2" : aContextsBefore[2],
				"/EMPLOYEES/3" : aContextsBefore[3],
				"/EMPLOYEES/4" : aContextsBefore[4]
			});
		} else {
			assert.strictEqual(oBinding.iMaxLength, 8);
			assert.strictEqual(oBinding.aContexts.length, 8);
			oBinding.aContexts.forEach(function (oContext, iIndex) {
				if (iIndex !== 6) {
					assert.strictEqual(oContext.iIndex, iIndex);
					assert.strictEqual(oContext.getPath(), "/EMPLOYEES/" + iIndex);
				}
			});
			assert.deepEqual(oBinding.mPreviousContextsByPath, {});
		}
	});
});

	//*********************************************************************************************
	QUnit.test("resetKeepAlive", function () {
		var oBinding = this.bindList("/EMPLOYEES"),
			oContext1 = {
				isKeepAlive : function () {}
			},
			oContext2 = {
				isKeepAlive : function () {},
				resetKeepAlive : function () {}
			},
			oContext3 = {
				isKeepAlive : function () {}
			},
			oContext4 = {
				isKeepAlive : function () {},
				resetKeepAlive : function () {}
			};

		oBinding.aContexts = [oContext1, oContext2];
		oBinding.mPreviousContextsByPath = {
			foo : oContext3,
			bar : oContext4
		};
		this.mock(oContext1).expects("isKeepAlive").withExactArgs().returns(false);
		this.mock(oContext2).expects("isKeepAlive").withExactArgs().returns(true);
		this.mock(oContext2).expects("resetKeepAlive").withExactArgs();
		this.mock(oContext3).expects("isKeepAlive").withExactArgs().returns(false);
		this.mock(oContext4).expects("isKeepAlive").withExactArgs().returns(true);
		this.mock(oContext4).expects("resetKeepAlive").withExactArgs();

		// code under test
		oBinding.resetKeepAlive();
	});

	//*********************************************************************************************
	QUnit.test("_checkDataStateMessages", function () {
		var oBinding = this.bindList("/EMPLOYEES"),
			oDataState = {
				setModelMessages : function () {}
			};

		this.mock(this.oModel).expects("getMessagesByPath")
			.withExactArgs("/resolved/path", true)
			.returns("aMessages");
		this.mock(oDataState).expects("setModelMessages").withExactArgs("aMessages");

		// code under test
		oBinding._checkDataStateMessages(oDataState, "/resolved/path");

		// code under test - no resolved path
		oBinding._checkDataStateMessages(oDataState);
	});

	//*********************************************************************************************
	QUnit.test("requestFilterForMessages, (with unresolved binding)", function (assert) {
		var oBinding = this.bindList("TEAM_2_EMPLOYEES");

		// code under test
		return oBinding.requestFilterForMessages().then(function (oFilter) {
			assert.strictEqual(oFilter, null);
		});
	});

	//*********************************************************************************************
[{
	messages : [], predicates : []
}, {
	messages : [{
		getTargets : function () { return ["/TEAMS('1')/foo"]; }
	}, {
		getTargets : function () { return ["/TEAMS('1')/bar"]; }
	}, {
		getTargets : function () { return ["/TEAMS"]; }
	}],
	predicates : ["('1')"]
}, {
	messages : [{
		getTargets : function () { return ["/TEAMS('1')/foo", "/TEAMS('1')/bar"]; }
	}, {
		getTargets : function () { return ["/TEAMS('2')/bar"]; }
	}, {
		getTargets : function () { return ["/TEAMS($uid='xyz')"]; }
	}],
	predicates : ["('1')", "('2')"]
}, {
	callbackReturns : true,
	messages : [{
		getTargets : function () { return ["/TEAMS('1')/foo"]; }
	}, {
		getTargets : function () { return ["/TEAMS('2')/bar"]; }
	}],
	predicates : ["('1')", "('2')"]
}, {
	callbackReturns : false,
	messages : [{
		getTargets : function () { return ["/TEAMS('1')/foo"]; }
	}, {
		getTargets : function () { return ["/TEAMS('2')/bar"]; }
	}],
	predicates : []
}].forEach(function (oFixture) {
	var sTitle = "requestFilterForMessages; messages: " + oFixture.messages.length
		+ " predicates: " + oFixture.predicates
		+ " callbackReturns: " + oFixture.callbackReturns;

	QUnit.test(sTitle, function (assert) {
		var fnCallback,
			oBinding = this.bindList("/TEAMS"),
			aFilters = [],
			oListBindingMock = this.mock(ODataListBinding),
			that = this;

		aFilters = oFixture.predicates.map(function () {
			return new Filter("does", "NOT", "matter");
		});

		this.mock(oBinding.oHeaderContext).expects("getPath").withExactArgs()
			.returns("/TEAMS");
		this.mock(_Helper).expects("getMetaPath").withExactArgs("/TEAMS").returns("~meta~path~");
		this.mock(this.oModel.oMetaModel).expects("requestObject").withExactArgs("~meta~path~/")
			.callsFake(function () {
				that.mock(that.oModel).expects("getMessagesByPath")
					.withExactArgs("/TEAMS", true).returns(oFixture.messages);
				if (oFixture.predicates.length === 0) {
					oListBindingMock.expects("getFilterForPredicate").never();
				} else {
					oFixture.predicates.map(function (sPredicate, i) {
						oListBindingMock.expects("getFilterForPredicate")
							.withExactArgs(sPredicate, "~entity~type~",
								sinon.match.same(that.oModel.oMetaModel), "~meta~path~")
							.returns(aFilters[i]);
					});
				}
				return Promise.resolve("~entity~type~");
			});

		if (oFixture.callbackReturns !== undefined) {
			fnCallback = sinon.spy(function () { return oFixture.callbackReturns; });
		}

		// code under test
		return oBinding.requestFilterForMessages(fnCallback).then(function (oFilter) {
			if (oFixture.predicates.length === 0 || oFixture.callbackReturns === false) {
				assert.strictEqual(oFilter, null);

				return;
			}
			if (oFixture.predicates.length === 1) {
				assert.strictEqual(oFilter, aFilters[0]);

				return;
			}
			assert.strictEqual(oFilter.aFilters.length, oFixture.predicates.length);
			assert.notOk(oFilter.bAnd);
			oFixture.predicates.forEach(function (_sPredicate, i) {
				assert.strictEqual(oFilter.aFilters[i], aFilters[i]);
			});
			if (fnCallback) {
				oFixture.messages.forEach(function (oMessage) {
					assert.ok(fnCallback.calledWithExactly(oMessage));
				});
			}
		});
	});
});

	//*********************************************************************************************
	QUnit.test("getGeneration", function (assert) {
		var oBinding = this.bindList("/TEAMS"),
			oContext = Context.createNewContext(this.oModel, oParentBinding, "/TEAMS('42')");

		this.mock(oBinding.oHeaderContext).expects("getGeneration").withExactArgs(true)
			.returns(42);

		// code under test
		assert.strictEqual(oBinding.getGeneration(), 42);

		oBinding = this.bindList("TEAM_2_EMPLOYEES", oContext);
		this.mock(asODataParentBinding.prototype).expects("getGeneration").on(oBinding)
			.withExactArgs().returns(34);

		// code under test
		assert.strictEqual(oBinding.getGeneration(), 34);
	});

	//*********************************************************************************************
	QUnit.test("getFilterForPredicate (one key property)", function (assert) {
		var oEntityType = {
				$Key : ["key"]
			},
			oFilter,
			oMetaModel = {
				getObject : function () {}
			};

		this.mock(_Parser).expects("parseKeyPredicate").withExactArgs("('value')")
			.returns({"" : "'value'"});
		this.mock(oMetaModel).expects("getObject").withExactArgs("~meta~path~/key/$Type")
			.returns("type");
		this.mock(window).expects("decodeURIComponent").withExactArgs("'value'")
			.returns("decoded value");
		this.mock(_Helper).expects("parseLiteral").withExactArgs("decoded value", "type", "key")
			.returns("parsed value");

		// code under test
		oFilter = ODataListBinding.getFilterForPredicate("('value')", oEntityType, oMetaModel,
			"~meta~path~");

		assert.ok(oFilter instanceof Filter);
		assert.deepEqual(oFilter, new Filter("key", FilterOperator.EQ, "parsed value"));
	});

	//*********************************************************************************************
	QUnit.test("getFilterForPredicate (more key properties, aliases)", function (assert) {
		var oEntityType = {
				$Key : ["key1", "key2", {alias : "key3/p"}]
			},
			oFilter,
			oHelperMock = this.mock(_Helper),
			oMetaModel = {
				getObject : function () {}
			},
			oMetaModelMock = this.mock(oMetaModel);

		this.mock(_Parser).expects("parseKeyPredicate")
			.withExactArgs("(key1='42',key2=43,alias='44')")
			.returns({key1 : "'42'", key2 : "43", alias : "'44'"});
		oMetaModelMock.expects("getObject").withExactArgs("~meta~path~/key1/$Type")
			.returns("type1");
		oMetaModelMock.expects("getObject").withExactArgs("~meta~path~/key2/$Type")
			.returns("type2");
		oMetaModelMock.expects("getObject").withExactArgs("~meta~path~/key3/p/$Type")
			.returns("type3");
		oHelperMock.expects("parseLiteral").withExactArgs("'42'", "type1", "key1")
			.returns("42");
		oHelperMock.expects("parseLiteral").withExactArgs("43", "type2", "key2")
			.returns(43);
		oHelperMock.expects("parseLiteral").withExactArgs("'44'", "type3", "key3/p")
			.returns("44");

		// code under test
		oFilter = ODataListBinding.getFilterForPredicate("(key1='42',key2=43,alias='44')",
			oEntityType, oMetaModel, "~meta~path~");

		assert.ok(oFilter instanceof Filter);
		assert.deepEqual(oFilter, new Filter({
			and : true,
			filters : [
				new Filter("key1", FilterOperator.EQ, "42"),
				new Filter("key2", FilterOperator.EQ, 43),
				new Filter("key3/p", FilterOperator.EQ, "44")
			]
		}));
	});

	//*********************************************************************************************
	QUnit.test("getCount", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oBindingMock = this.mock(oBinding);

		oBindingMock.expects("getHeaderContext").withExactArgs().returns(oBinding.oHeaderContext);
		this.mock(oBinding.oHeaderContext).expects("getProperty").withExactArgs("$count")
			.returns(42);

		// code under test
		assert.strictEqual(oBinding.getCount(), 42);

		oBindingMock.expects("getHeaderContext").withExactArgs().returns(null);

		// code under test
		assert.strictEqual(oBinding.getCount(), undefined);
	});

	//*********************************************************************************************
[undefined, 0, 42].forEach(function (iIndex) {
	var bKeepAlive = iIndex === 42,
		sTitle = "doReplaceWith: existing Context, bKeepAlive = " + bKeepAlive + ", index = "
			+ iIndex;

	QUnit.test(sTitle, function (assert) {
		var oAddKeptElementExpectation,
			oBinding = this.bindList("/EMPLOYEES"),
			oDoReplaceWithExpectation,
			oElement = {},
			oExistingContext = {}, // no #setKeepAlive
			oOldContext = {
				iIndex : iIndex,
				getModelIndex : function () { return iIndex; },
				getPath : function () {},
				isKeepAlive : function () {}
			},
			sPredicate = "('1')";

		oBinding.mPreviousContextsByPath["~header~context~path~('1')"] = oExistingContext;
		this.mock(oOldContext).expects("getPath").exactly(bKeepAlive ? 1 : 0).withExactArgs()
			.returns("~old~context~path~");
		this.mock(oOldContext).expects("isKeepAlive").withExactArgs().returns(bKeepAlive);
		this.mock(oBinding.oHeaderContext).expects("getPath").withExactArgs()
			.returns("~header~context~path~");
		this.mock(Context).expects("create").never();
		oAddKeptElementExpectation = this.mock(oBinding.oCache).expects("addKeptElement")
			.exactly(iIndex === undefined ? 1 : 0)
			.withExactArgs(sinon.match.same(oElement));
		oDoReplaceWithExpectation = this.mock(oBinding.oCache).expects("doReplaceWith")
			.exactly(iIndex === undefined ? 0 : 1)
			.withExactArgs(iIndex, sinon.match.same(oElement));
		this.mock(oBinding).expects("destroyLater").exactly(bKeepAlive ? 0 : 1)
			.withExactArgs(sinon.match.same(oOldContext));
		this.mock(oBinding).expects("_fireChange").withExactArgs({reason : ChangeReason.Change})
			.callsFake(function () {
				if (iIndex === undefined) {
					assert.ok(oAddKeptElementExpectation.calledOnce);
				} else {
					assert.ok(oDoReplaceWithExpectation.calledOnce);
				}
				assert.strictEqual(oExistingContext.iIndex, iIndex);
				assert.strictEqual(oOldContext.iIndex, undefined);
				assert.strictEqual(oBinding.aContexts.indexOf(oExistingContext),
					iIndex === undefined ? -1 : iIndex);

				assert.strictEqual(Object.keys(oBinding.mPreviousContextsByPath).length,
					iIndex === undefined || bKeepAlive ? 1 : 0);
				assert.strictEqual(oBinding.mPreviousContextsByPath["~old~context~path~"],
					bKeepAlive ? oOldContext : undefined);
				assert.strictEqual(oBinding.mPreviousContextsByPath["~header~context~path~('1')"],
					iIndex === undefined ? oExistingContext : undefined);
			});

		assert.strictEqual(
			// code under test
			oBinding.doReplaceWith(oOldContext, oElement, sPredicate),
			oExistingContext
		);
	});
});

	//*********************************************************************************************
	QUnit.test("doReplaceWith: unexpected index", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oOldContext = {
				getModelIndex : function () {}, // result does not matter
				isKeepAlive : function () {} // result does not matter
			};

		oBinding.mPreviousContextsByPath["/EMPLOYEES('1')"] = {
			iIndex : 0,
			toString : function () { return "~toString~"; }
		}; // no #setKeepAlive
		this.mock(Context).expects("create").never();
		this.mock(oBinding.oCache).expects("doReplaceWith").never();
		this.mock(oBinding).expects("_fireChange").never();

		assert.throws(function () {
			// code under test
			oBinding.doReplaceWith(oOldContext, {}, "('1')");
		}, new Error("Unexpected index: ~toString~"));
	});

	//*********************************************************************************************
[undefined, -1, 0, 42].forEach(function (iIndex) {
	[false, true].forEach(function (bHasOnBeforeDestroy) {
		var bKeepAlive = iIndex === 42,
			sTitle = "doReplaceWith: new Context, bKeepAlive = " + bKeepAlive
				+ ", bHasOnBeforeDestroy = " + bHasOnBeforeDestroy + ", index = " + iIndex;

		if (!bKeepAlive && bHasOnBeforeDestroy) {
			return;
		}

	QUnit.test(sTitle, function (assert) {
		var oAddKeptElementExpectation,
			oBinding = this.bindList("/EMPLOYEES"),
			oDoReplaceWithExpectation,
			oElement = {},
			iModelIndex = iIndex < 0 ? 17 : iIndex,
			oNewContext = {
				setKeepAlive : function () {}
			},
			oOldContext = {
				iIndex : iIndex,
				fnOnBeforeDestroy : bHasOnBeforeDestroy ? sinon.spy() : undefined,
				getModelIndex : function () {},
				getPath : function () {},
				isKeepAlive : function () {}
			},
			sPredicate = "('1')",
			oSetKeepAliveExpectation;

		this.mock(oOldContext).expects("getModelIndex").withExactArgs().returns(iModelIndex);
		this.mock(oOldContext).expects("getPath").exactly(bKeepAlive ? 1 : 0).withExactArgs()
			.returns("~old~context~path~");
		this.mock(oOldContext).expects("isKeepAlive").withExactArgs().returns(bKeepAlive);
		this.mock(oBinding.oHeaderContext).expects("getPath").withExactArgs()
			.returns("~header~context~path~");
		this.mock(Context).expects("create")
			.withExactArgs(sinon.match.same(oBinding.oModel), sinon.match.same(oBinding),
				"~header~context~path~('1')", iIndex)
			.returns(oNewContext);
		oAddKeptElementExpectation = this.mock(oBinding.oCache).expects("addKeptElement")
			.exactly(iIndex === undefined ? 1 : 0)
			.withExactArgs(sinon.match.same(oElement));
		oDoReplaceWithExpectation = this.mock(oBinding.oCache).expects("doReplaceWith")
			.exactly(iIndex === undefined ? 0 : 1)
			.withExactArgs(iModelIndex, sinon.match.same(oElement));
		oSetKeepAliveExpectation = this.mock(oNewContext).expects("setKeepAlive")
			.exactly(bKeepAlive ? 1 : 0)
			.withExactArgs(true, bHasOnBeforeDestroy ? sinon.match.func : undefined);
		this.mock(oBinding).expects("destroyLater").exactly(bKeepAlive ? 0 : 1)
			.withExactArgs(sinon.match.same(oOldContext));
		this.mock(oBinding).expects("_fireChange").withExactArgs({reason : ChangeReason.Change})
			.callsFake(function () {
				if (iIndex === undefined) {
					assert.ok(oAddKeptElementExpectation.calledOnce);
				} else {
					assert.ok(oDoReplaceWithExpectation.calledOnce);
				}
				if (bKeepAlive) {
					assert.ok(oSetKeepAliveExpectation.calledOnce);
					assert.ok(oSetKeepAliveExpectation.calledAfter(oDoReplaceWithExpectation));
				}
				assert.strictEqual(oOldContext.iIndex, undefined);
				assert.strictEqual(oBinding.aContexts.indexOf(oNewContext),
					iIndex === undefined ? -1 : iModelIndex);
				assert.strictEqual(Object.keys(oBinding.mPreviousContextsByPath).length,
					iIndex === undefined || bKeepAlive ? 1 : 0);
				assert.strictEqual(oBinding.mPreviousContextsByPath["~old~context~path~"],
					bKeepAlive ? oOldContext : undefined);
				assert.strictEqual(oBinding.mPreviousContextsByPath["~header~context~path~('1')"],
					iIndex === undefined ? oNewContext : undefined);
			});

		assert.strictEqual(
			// code under test
			oBinding.doReplaceWith(oOldContext, oElement, sPredicate),
			oNewContext
		);

		if (bHasOnBeforeDestroy) {
			assert.notOk(oOldContext.fnOnBeforeDestroy.called);

			// code under test
			oSetKeepAliveExpectation.args[0][1]();

			assert.ok(
				oOldContext.fnOnBeforeDestroy.calledOnceWithExactly(sinon.match.same(oNewContext)));
		}
	});
	});
});

	//*********************************************************************************************
	QUnit.test("doReplaceWith: copy the copy", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oCacheMock = this.mock(oBinding.oCache),
			oContextMock = this.mock(Context),
			oElement1 = {},
			oElement2 = {},
			oNewContext1 = {
				iIndex : 42,
				getModelIndex : function () { return 42; },
				getPath : function () {
					return "/EMPLOYEES('1')";
				},
				setKeepAlive : function () {}
			},
			oNewContext2 = {
				setKeepAlive : function () {}
			},
			oOldContext = {
				iIndex : 42,
				fnOnBeforeDestroy : sinon.spy(),
				getModelIndex : function () { return 42; },
				getPath : function () {
					return "/EMPLOYEES('0')";
				},
				isKeepAlive : function () {
					return true;
				}
			},
			sPredicate1 = "('1')",
			sPredicate2 = "('2')",
			oSetKeepAliveExpectation1,
			oSetKeepAliveExpectation2;

		oContextMock.expects("create")
			.withExactArgs(sinon.match.same(oBinding.oModel), sinon.match.same(oBinding),
				"/EMPLOYEES('1')", 42)
			.returns(oNewContext1);
		oCacheMock.expects("doReplaceWith").withExactArgs(42, sinon.match.same(oElement1));
		oSetKeepAliveExpectation1 = this.mock(oNewContext1).expects("setKeepAlive")
			.withExactArgs(true, sinon.match.func)
			.callsFake(function (_bKeepAlive, fnOnBeforeDestroy1) {
				this.isKeepAlive = function () {
					return true;
				};
				this.fnOnBeforeDestroy = fnOnBeforeDestroy1;
			});
		// ignore call to #_fireChange (no listeners)

		assert.strictEqual(
			// code under test
			oBinding.doReplaceWith(oOldContext, oElement1, sPredicate1),
			oNewContext1
		);

		assert.notOk(oOldContext.fnOnBeforeDestroy.called);

		// code under test
		oSetKeepAliveExpectation1.args[0][1]();

		assert.ok(
			oOldContext.fnOnBeforeDestroy.calledOnceWithExactly(sinon.match.same(oNewContext1)));

		oContextMock.expects("create")
			.withExactArgs(sinon.match.same(oBinding.oModel), sinon.match.same(oBinding),
				"/EMPLOYEES('2')", 42)
			.returns(oNewContext2);
		oCacheMock.expects("doReplaceWith").withExactArgs(42, sinon.match.same(oElement2));
		oSetKeepAliveExpectation2 = this.mock(oNewContext2).expects("setKeepAlive")
			.withExactArgs(true, sinon.match.func);

		assert.strictEqual(
			// code under test
			oBinding.doReplaceWith(oNewContext1, oElement2, sPredicate2),
			oNewContext2
		);

		// code under test
		oSetKeepAliveExpectation2.args[0][1]();

		assert.ok(oOldContext.fnOnBeforeDestroy.calledTwice);
		assert.ok(oOldContext.fnOnBeforeDestroy.secondCall.calledWithExactly(
			sinon.match.same(oNewContext2)));
		assert.notStrictEqual(oOldContext.fnOnBeforeDestroy.args[1][0], oNewContext1);
		assert.strictEqual(oOldContext.fnOnBeforeDestroy.args[1][0], oNewContext2);
	});

	//*********************************************************************************************
	QUnit.test("doReplaceWith: same instance", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oOldContext = {
				getModelIndex : function () {}, // result does not matter
				isKeepAlive : function () {} // result does not matter
			};

		oBinding.mPreviousContextsByPath["/EMPLOYEES('1')"] = oOldContext;
		this.mock(Context).expects("create").never();
		this.mock(oBinding.oCache).expects("doReplaceWith").never();
		this.mock(oBinding).expects("_fireChange").never();

		// code under test
		assert.strictEqual(oBinding.doReplaceWith(oOldContext, {}, "('1')"), oOldContext);
	});

	//*********************************************************************************************
	QUnit.test("fireCreateActivate", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		oBinding.iActiveContexts = 40;

		this.mock(oBinding).expects("fireEvent").withExactArgs("createActivate")
			.callsFake(function () {
				assert.strictEqual(oBinding.iActiveContexts, 41);
			});

		// code under test
		oBinding.fireCreateActivate();

		assert.strictEqual(oBinding.iActiveContexts, 41);
	});

	//*********************************************************************************************
[false, true].forEach(function (bFireChange) {
	QUnit.test("getAllCurrentContexts: bFireChange = " + bFireChange, function (assert) {
		var aAllCurrentContexts,
			oBinding = this.bindList("/EMPLOYEES"),
			oCache = {
				getAllElements : function () {}
			},
			oKeptContext0 = {isKeepAlive : function () {}},
			oKeptContext1 = {isKeepAlive : function () {}},
			oNotKeptContext = {isKeepAlive : function () {}}; // BCP 2270081950:
			// there is a point in time when contexts with keepAlive=false are present in
			// mPreviousContextsByPath which need be filtered out.

		oBinding.mPreviousContextsByPath = {
			"~sPath1~" : oKeptContext0,
			"~sPath2~" : oKeptContext1,
			"~sPath3~" : oNotKeptContext
		};

		this.mock(oBinding).expects("withCache").withExactArgs(sinon.match.func, "", true)
			.callsArgWith(0, oCache, "path/to/cache");
		this.mock(oCache).expects("getAllElements").withExactArgs("path/to/cache")
			.returns("~aElements~");
		this.mock(oBinding).expects("createContexts").withExactArgs(0, "~aElements~")
			.callsFake(function () {
				oBinding.aContexts = ["~oContext0~", undefined, "~oContext2~"];
				return bFireChange;
			});
		this.mock(oBinding).expects("_fireChange").withExactArgs({reason : ChangeReason.Change})
			.exactly(bFireChange ? 1 : 0);

		this.mock(oKeptContext0).expects("isKeepAlive").withExactArgs().returns(true);
		this.mock(oKeptContext1).expects("isKeepAlive").withExactArgs().returns(true);
		this.mock(oNotKeptContext).expects("isKeepAlive").withExactArgs().returns(false);

		// code under test
		aAllCurrentContexts = oBinding.getAllCurrentContexts();

		assert.deepEqual(aAllCurrentContexts,
			["~oContext0~", "~oContext2~", oKeptContext0, oKeptContext1]);
		assert.strictEqual(aAllCurrentContexts[2], oKeptContext0);
		assert.strictEqual(aAllCurrentContexts[3], oKeptContext1);
	});
});

	//*********************************************************************************************
	QUnit.test("getAllCurrentContexts: no cache yet", function (assert) {
		var oBinding = this.bindList("relativePath");

		this.mock(oBinding).expects("withCache").withExactArgs(sinon.match.func, "", true);
		this.mock(oBinding).expects("createContexts").withExactArgs(0, []).returns(false);
		this.mock(oBinding).expects("_fireChange").never();

		// code under test
		assert.deepEqual(oBinding.getAllCurrentContexts(), []);
	});

	//*********************************************************************************************
	QUnit.test("getKeepAliveContext: existing context", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oContext;

		oBinding.createContexts(3, createData(3, 3, true, 6, true)); // simulate a read
		oContext = oBinding.aContexts[4];
		oContext.fnOnBeforeDestroy = "~fnOnBeforeDestroy~";
		this.mock(oBinding).expects("checkSuspended").withExactArgs();
		this.mock(oContext).expects("setKeepAlive")
			.withExactArgs(true, "~fnOnBeforeDestroy~", "~bRequestMessages~");

		assert.strictEqual(
			// code under test
			oBinding.getKeepAliveContext("/EMPLOYEES('4')", "~bRequestMessages~"),
			oContext);
	});

	//*********************************************************************************************
	QUnit.test("getKeepAliveContext: kept-alive context not in the list", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			sPath = "/EMPLOYEES('4')",
			oContext = Context.create(this.oModel, oBinding, sPath);

		oBinding.mPreviousContextsByPath[sPath] = oContext;
		oContext.fnOnBeforeDestroy = "~fnOnBeforeDestroy~";
		this.mock(oBinding).expects("checkSuspended").withExactArgs();
		this.mock(oContext).expects("setKeepAlive")
			.withExactArgs(true, "~fnOnBeforeDestroy~", "~bRequestMessages~");

		assert.strictEqual(
			// code under test
			oBinding.getKeepAliveContext(sPath, "~bRequestMessages~"),
			oContext);
	});

	//*********************************************************************************************
[false, true].forEach(function (bAsync) {
	[undefined, "group"].forEach(function (sGroupId) {
		var sTitle = "getKeepAliveContext: create context, async=" + bAsync + ", group=" + sGroupId;

	// The test always fails in requestProperty to check that the reporter is attached correctly
	QUnit.test(sTitle, function (assert) {
		var done = assert.async(),
			oParentContext = this.oModel.createBindingContext("/"),
			oBinding = this.bindList("EMPLOYEES", oParentContext),
			oCache = {
				createEmptyElement : function () {}
			},
			oContext = {
				requestProperty : function () {},
				setKeepAlive : function () {}
			},
			oError = new Error(),
			bHasEmptyElement = false,
			sPath = "/EMPLOYEES('4')",
			oSetKeepAliveExpectation,
			oType = {
				$Key : ["a", {b : "c/d"}, "e", {f : "g/h"}]
			};

		oBinding.oCachePromise = bAsync ? Promise.resolve(oCache) : SyncPromise.resolve(oCache);
		this.mock(oBinding).expects("checkSuspended").withExactArgs();
		this.mock(_Helper).expects("checkGroupId").withExactArgs(sGroupId);
		this.mock(oBinding).expects("getResolvedPath").withExactArgs().returns("/EMPLOYEES");
		this.mock(_Helper).expects("getPredicateIndex").withExactArgs(sPath).returns(10);
		this.mock(Context).expects("create")
			.withExactArgs(sinon.match.same(this.oModel), sinon.match.same(oBinding), sPath)
			.returns(oContext);
		oSetKeepAliveExpectation = this.mock(oContext).expects("setKeepAlive")
			.withExactArgs(true, undefined, "~bRequestMessages~");
		this.mock(_Helper).expects("getMetaPath").withExactArgs("/EMPLOYEES").returns("/meta/path");
		this.mock(this.oModel.getMetaModel()).expects("requestObject")
			.withExactArgs("/meta/path/").resolves(oType);
		this.mock(oCache).expects("createEmptyElement").withExactArgs("('4')")
			.callsFake(function () {
				bHasEmptyElement = true;
				return "~oElement~";
			});
		this.mock(_Helper).expects("setPrivateAnnotation").exactly(sGroupId ? 1 : 0)
			.withExactArgs("~oElement~", "groupId", sGroupId);
		this.mock(oContext).expects("requestProperty").withExactArgs(["a", "c/d", "e", "g/h"])
			.callsFake(function () {
				assert.ok(oSetKeepAliveExpectation.called);
				assert.strictEqual(bHasEmptyElement, true);
				return Promise.reject(oError);
			});
		this.mock(this.oModel).expects("getReporter").withExactArgs().returns(function (oError0) {
			assert.strictEqual(oError0, oError);
			done();
		});

		assert.strictEqual(
			// code under test
			oBinding.getKeepAliveContext(sPath, "~bRequestMessages~", sGroupId),
			oContext);
		assert.strictEqual(oBinding.mPreviousContextsByPath[sPath], oContext);
		assert.ok(oSetKeepAliveExpectation.called);
	});
	});
});

	//*********************************************************************************************
	QUnit.test("getKeepAliveContext: unresolved", function (assert) {
		var oBinding = this.bindList("EMPLOYEES");

		assert.throws(function () {
			// code under test
			oBinding.getKeepAliveContext("/EMPLOYEES('1')");
		}, new Error("Binding is unresolved: " + oBinding));
	});

	//*********************************************************************************************
	QUnit.test("getKeepAliveContext: missing path", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		assert.throws(function () {
			// code under test
			oBinding.getKeepAliveContext();
		}, new Error("Not a list context path to an entity: undefined"));
	});

	//*********************************************************************************************
	QUnit.test("getKeepAliveContext: not a valid context path", function (assert) {
		var oParentContext = this.oModel.createBindingContext("/"),
			oBinding = this.bindList("EMPLOYEES", oParentContext),
			sPath = "/TEAMS('1')";

		this.mock(oBinding).expects("getResolvedPath").withExactArgs().returns("/EMPLOYEES");
		this.mock(_Helper).expects("getPredicateIndex").withExactArgs(sPath).returns(6);

		assert.throws(function () {
			// code under test
			oBinding.getKeepAliveContext(sPath);
		}, new Error(oBinding + ": Not a valid context path: " + sPath));
	});

	//*********************************************************************************************
	QUnit.test("getCacheAndMoveKeepAliveContexts", function (assert) {
		var oBinding = this.bindList("/path", undefined, undefined, undefined,
				{$$getKeepAliveContext : true}),
			oCache = {
				setQueryOptions : function () {}
			},
			oContext1 = {},
			oContext2 = {},
			oTemporaryBinding = {
				destroy : function () {},
				oCache : oCache,
				mLateQueryOptions : "~mLateQueryOptions~",
				mParameters : {},
				mPreviousContextsByPath : {
					"/path(1)" : oContext1,
					"/path(2)" : oContext2
				}
			};

		this.mock(this.oModel).expects("releaseKeepAliveBinding").withExactArgs("/path")
			.returns(oTemporaryBinding);
		this.mock(_Helper).expects("clone").withExactArgs("~mQueryOptions~")
			.returns("~mQueryOptionsClone~");
		this.mock(_Helper).expects("aggregateExpandSelect")
			.withExactArgs("~mQueryOptionsClone~", "~mLateQueryOptions~");
		this.mock(oCache).expects("setQueryOptions").withExactArgs("~mQueryOptions~");
		this.mock(oTemporaryBinding).expects("destroy").withExactArgs().callsFake(function () {
			assert.deepEqual(oTemporaryBinding.mPreviousContextsByPath, {});
			assert.strictEqual(oTemporaryBinding.oCache, null);
			assert.strictEqual(oTemporaryBinding.oCachePromise.getResult(), null);
		});

		// code under test
		assert.strictEqual(oBinding.getCacheAndMoveKeepAliveContexts("path", "~mQueryOptions~"),
			oCache);

		assert.strictEqual(oBinding.mLateQueryOptions, "~mQueryOptionsClone~");
		assert.strictEqual(Object.keys(oBinding.mPreviousContextsByPath).length, 2);
		assert.strictEqual(oBinding.mPreviousContextsByPath["/path(1)"], oContext1);
		assert.strictEqual(oBinding.mPreviousContextsByPath["/path(2)"], oContext2);
		assert.strictEqual(oContext1.oBinding, oBinding);
		assert.strictEqual(oContext2.oBinding, oBinding);
	});

	//*********************************************************************************************
	QUnit.test("getCacheAndMoveKeepAliveContexts: no binding", function (assert) {
		var oBinding = this.bindList("/path", undefined, undefined, undefined,
				{$$getKeepAliveContext : true});

		oBinding.mLateQueryOptions = "~mLateQueryOptions~";
		this.mock(this.oModel).expects("releaseKeepAliveBinding").withExactArgs("/path")
			.returns(undefined);

		// code under test
		assert.strictEqual(oBinding.getCacheAndMoveKeepAliveContexts("path"), undefined);

		assert.strictEqual(oBinding.mLateQueryOptions, "~mLateQueryOptions~");
		assert.strictEqual(Object.keys(oBinding.mPreviousContextsByPath).length, 0);
	});

	//*********************************************************************************************
	QUnit.test("getCacheAndMoveKeepAliveContexts: unmarked", function (assert) {
		var oBinding = this.bindList("/path");

		oBinding.mLateQueryOptions = "~mLateQueryOptions~";
		this.mock(this.oModel).expects("releaseKeepAliveBinding").never();

		// code under test
		assert.strictEqual(oBinding.getCacheAndMoveKeepAliveContexts("path"), undefined);

		assert.strictEqual(oBinding.mLateQueryOptions, "~mLateQueryOptions~");
		assert.strictEqual(Object.keys(oBinding.mPreviousContextsByPath).length, 0);
	});

	//*********************************************************************************************
["foo", "bar", "$$patchWithoutSideEffects", "$$updateGroupId"].forEach(function (sParameter) {
	QUnit.test("getCacheAndMoveKeepAliveContexts: mismatch in" + sParameter, function (assert) {
		var oBinding = this.bindList("/path"),
			oTemporaryBinding = {};

		this.mock(this.oModel).expects("releaseKeepAliveBinding").twice().withExactArgs("/path")
			.returns(oTemporaryBinding);

		assert.throws(function () {
			oBinding.mParameters = {$$getKeepAliveContext : true, $count : true};
			oTemporaryBinding.mParameters = {};
			oTemporaryBinding.mParameters[sParameter] = "~";

			// code under test
			oBinding.getCacheAndMoveKeepAliveContexts("path");
		}, new Error(oBinding + ": parameter does not match getKeepAliveContext: " + sParameter));

		assert.throws(function () {
			oBinding.mParameters = {$$getKeepAliveContext : true, $count : true};
			oBinding.mParameters[sParameter] = "~";
			oTemporaryBinding.mParameters = {};

			// code under test
			oBinding.getCacheAndMoveKeepAliveContexts("path");
		}, new Error(oBinding + ": parameter does not match getKeepAliveContext: " + sParameter));
	});
});

	//*********************************************************************************************
	QUnit.test("isKeepAliveBindingFor: no $$getKeepAliveContext", function (assert) {
		var oBinding = this.bindList("/path");

		this.mock(oBinding).expects("getResolvedPath").never();
		this.mock(oBinding).expects("isRootBindingSuspended").never();

		// code under test
		assert.notOk(oBinding.isKeepAliveBindingFor("/path"));
	});

	//*********************************************************************************************
	QUnit.test("isKeepAliveBindingFor: wrong path", function (assert) {
		var oBinding = this.bindList("/other/path", undefined, undefined, undefined,
			{$$getKeepAliveContext : true});

		oBinding.aContexts = [{}];
		oBinding.mPreviousContextsByPath["/other/path(1)"] = {};
		this.mock(oBinding).expects("getResolvedPath").withExactArgs().returns("/other/path");
		this.mock(oBinding).expects("isRootBindingSuspended").never();

		// code under test
		assert.notOk(oBinding.isKeepAliveBindingFor("/path"));
	});

	//*********************************************************************************************
	QUnit.test("isKeepAliveBindingFor: not suspended", function (assert) {
		var oBinding = this.bindList("/path", undefined, undefined, undefined,
			{$$getKeepAliveContext : true});

		oBinding.aContexts = [{}];
		oBinding.mPreviousContextsByPath["/path(1)"] = {};
		this.mock(oBinding).expects("getResolvedPath").withExactArgs().returns("/path");
		this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs().returns(false);

		// code under test
		assert.ok(oBinding.isKeepAliveBindingFor("/path"));
	});

	//*********************************************************************************************
	QUnit.test("isKeepAliveBindingFor: suspended, no contexts", function (assert) {
		var oBinding = this.bindList("/path", undefined, undefined, undefined,
			{$$getKeepAliveContext : true});

		this.mock(oBinding).expects("getResolvedPath").withExactArgs().returns("/path");
		this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs().returns(true);

		// code under test
		assert.notOk(oBinding.isKeepAliveBindingFor("/path"));
	});

	//*********************************************************************************************
	QUnit.test("isKeepAliveBindingFor: suspended, context in aContexts", function (assert) {
		var oBinding = this.bindList("/path", undefined, undefined, undefined,
			{$$getKeepAliveContext : true});

		oBinding.aContexts = [{}];
		this.mock(oBinding).expects("getResolvedPath").withExactArgs().returns("/path");
		this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs().returns(true);

		// code under test
		assert.ok(oBinding.isKeepAliveBindingFor("/path"));
	});

	//*********************************************************************************************
	QUnit.test("isKeepAliveBindingFor: suspended, kept-alive context", function (assert) {
		var oBinding = this.bindList("/path", undefined, undefined, undefined,
			{$$getKeepAliveContext : true});

		oBinding.mPreviousContextsByPath["/path(1)"] = {};
		this.mock(oBinding).expects("getResolvedPath").withExactArgs().returns("/path");
		this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs().returns(true);

		// code under test
		assert.ok(oBinding.isKeepAliveBindingFor("/path"));
	});

	//*********************************************************************************************
	QUnit.test("isFirstCreateAtEnd", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		// code under test
		assert.strictEqual(oBinding.isFirstCreateAtEnd(), undefined);

		oBinding.bFirstCreateAtEnd = "~foo~";

		// code under test
		assert.strictEqual(oBinding.isFirstCreateAtEnd(), "~foo~");
	});

	//*********************************************************************************************
	QUnit.test("restoreCreated", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES"),
			oCache = {
				getCreatedElements : function () { }
			},
			oElement0 = {},
			oElement1 = {"@$ui5.context.isInactive" : false},
			oElement2 = {"@$ui5.context.isInactive" : true},
			oHelperMock = this.mock(_Helper);

		this.mock(oBinding).expects("withCache").withExactArgs(sinon.match.func)
			.callsArgWith(0, oCache, "path/in/cache").returns(SyncPromise.resolve());
		this.mock(this.oModel).expects("getReporter").withExactArgs();
		this.mock(oCache).expects("getCreatedElements").withExactArgs("path/in/cache")
			.returns([oElement0, oElement1, oElement2]);

		oHelperMock.expects("getPrivateAnnotation")
			.withExactArgs(oElement0, "context").returns("~context0~");
		oHelperMock.expects("getPrivateAnnotation")
			.withExactArgs(oElement1, "context").returns("~context1~");
		oHelperMock.expects("getPrivateAnnotation")
			.withExactArgs(oElement2, "context").returns("~context1~");
		oHelperMock.expects("getPrivateAnnotation")
			.withExactArgs(oElement0, "firstCreateAtEnd").returns(false);
		oHelperMock.expects("getPrivateAnnotation")
			.withExactArgs(oElement1, "firstCreateAtEnd").returns(false);
		oHelperMock.expects("getPrivateAnnotation")
			.withExactArgs(oElement2, "firstCreateAtEnd").returns(false);

		// code under test
		oBinding.restoreCreated();

		assert.strictEqual(oBinding.aContexts.length, 3);
		assert.strictEqual(oBinding.bFirstCreateAtEnd, false);
		assert.strictEqual(oBinding.iCreatedContexts, 3);
		assert.strictEqual(oBinding.iActiveContexts, 2);
	});

	//*********************************************************************************************
[false, true].forEach(function (bMatch) {
	QUnit.test("findContextForCanonicalPath: aContexts, match=" + bMatch, function (assert) {
		var oBinding = this.bindList("/TEAM('1')/TEAM_2_EMPLOYEES"),
			oContext1 = {
				fetchCanonicalPath : function () {}
			},
			oContext2 = {
				fetchCanonicalPath : function () {}
			},
			oContext3 = {
				fetchCanonicalPath : function () {}
			};

		oBinding.aContexts = [oContext1, undefined, oContext2, oContext3];
		this.mock(oContext1).expects("fetchCanonicalPath").withExactArgs()
			.returns(SyncPromise.resolve("/EMPLOYEES('1')"));
		this.mock(oContext2).expects("fetchCanonicalPath").withExactArgs()
			.returns(SyncPromise.resolve("/EMPLOYEES('2')"));
		this.mock(oContext3).expects("fetchCanonicalPath").exactly(bMatch ? 0 : 1).withExactArgs()
			.returns(SyncPromise.resolve("/EMPLOYEES('3')"));

		assert.strictEqual(
			// code under test
			oBinding.findContextForCanonicalPath(bMatch ? "/EMPLOYEES('2')" : "/EMPLOYEES('99')"),
			bMatch ? oContext2 : undefined);
	});
});

	//*********************************************************************************************
[false, true].forEach(function (bMatch) {
	var sTitle = "findContextForCanonicalPath: mPreviousContextsByPath, match=" + bMatch;

	QUnit.test(sTitle, function (assert) {
		var oBinding = this.bindList("/TEAM('1')/TEAM_2_EMPLOYEES"),
			oContext1 = {
				fetchCanonicalPath : function () {},
				isKeepAlive : function () {}
			},
			oContext2 = {
				fetchCanonicalPath : function () {},
				isKeepAlive : function () {}
			},
			oContext3 = {
				fetchCanonicalPath : function () {},
				isKeepAlive : function () {}
			},
			oContext4 = {
				isKeepAlive : function () {}
			};

		oBinding.mPreviousContextsByPath = {
			"/TEAM('1')/TEAM_2_EMPLOYEES('1')" : oContext1,
			"/TEAM('2')/TEAM_2_EMPLOYEES('2')" : oContext4,
			"/TEAM('1')/TEAM_2_EMPLOYEES('2')" : oContext2
		};
		oBinding.aContexts = [oContext3];
		this.mock(oContext1).expects("isKeepAlive").withExactArgs().returns(true);
		this.mock(oContext1).expects("fetchCanonicalPath").withExactArgs()
			.returns(SyncPromise.resolve("/EMPLOYEES('1')"));
		this.mock(oContext4).expects("isKeepAlive").withExactArgs().returns(false);
		this.mock(oContext2).expects("isKeepAlive").withExactArgs().returns(true);
		this.mock(oContext2).expects("fetchCanonicalPath").withExactArgs()
			.returns(SyncPromise.resolve("/EMPLOYEES('2')"));
		this.mock(oContext3).expects("fetchCanonicalPath").exactly(bMatch ? 0 : 1).withExactArgs()
			.returns(SyncPromise.resolve("/EMPLOYEES('3')"));

		assert.strictEqual(
			// code under test
			oBinding.findContextForCanonicalPath(bMatch ? "/EMPLOYEES('2')" : "/EMPLOYEES('99')"),
			bMatch ? oContext2 : undefined);
	});
});

	//*********************************************************************************************
	QUnit.test("findContextForCanonicalPath: fetchCanonicalPath fails", function (assert) {
		var oBinding = this.bindList("/TEAM('1')/TEAM_2_EMPLOYEES"),
			oContext1 = {
				fetchCanonicalPath : function () {}
			};

		oBinding.aContexts = [oContext1];
		this.mock(oContext1).expects("fetchCanonicalPath").withExactArgs()
			.returns(SyncPromise.reject(new Error()));

		assert.strictEqual(
			// code under test
			oBinding.findContextForCanonicalPath("/EMPLOYEES('2')"),
			undefined);
	});

	//*********************************************************************************************
[false, true].forEach(function (bRefreshFails) {
	var sTitle = "onChange: " + (bRefreshFails ? ": refresh" : "checkUpdate") + " fails";

	QUnit.test(sTitle, function (assert) {
		var done = assert.async(),
			oBinding = this.bindList("/EMPLOYEES"),
			oDependent1 = {
				refreshInternal : function () {}
			},
			oDependent2 = {
				refreshInternal : function () {}
			},
			oDependentsExpectation,
			oRejectedPromise = Promise.reject("~oError~"),
			oResetExpectation;

		this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs().returns(false);
		this.mock(oBinding).expects("refreshSuspended").never();
		oDependentsExpectation = this.mock(oBinding).expects("getDependentBindings")
			.withExactArgs().returns([oDependent1, oDependent2]);
		oResetExpectation = this.mock(oBinding).expects("reset")
			.withExactArgs(ChangeReason.Refresh);
		this.mock(oDependent1).expects("refreshInternal").withExactArgs("").resolves();
		this.mock(oDependent2).expects("refreshInternal").withExactArgs("")
			.returns(bRefreshFails ? oRejectedPromise : Promise.resolve());
		this.mock(oBinding.oHeaderContext).expects("checkUpdateInternal")
			.exactly(bRefreshFails ? 0 : 1).withExactArgs().returns(oRejectedPromise);
		this.mock(this.oModel).expects("getReporter").withExactArgs().returns(function (oError) {
			assert.strictEqual(oError, "~oError~");
			done();
		});

		// code under test
		oBinding.onChange();

		assert.ok(oDependentsExpectation.calledBefore(oResetExpectation));

		oRejectedPromise.catch(function () {});
	});
});

	//*********************************************************************************************
	QUnit.test("onChange: refreshing", function () {
		var oBinding = this.bindList("/EMPLOYEES");

		oBinding.oRefreshPromise = "~oRefreshPromise~";
		this.mock(oBinding).expects("isRootBindingSuspended").never();
		this.mock(oBinding).expects("refreshSuspended").never();
		this.mock(oBinding).expects("getDependentBindings").never();
		this.mock(oBinding).expects("reset").never();
		this.mock(oBinding.oHeaderContext).expects("checkUpdateInternal").never();

		// code under test
		oBinding.onChange();
	});

	//*********************************************************************************************
	QUnit.test("onChange: suspended", function (assert) {
		var oBinding = this.bindList("/EMPLOYEES");

		this.mock(oBinding).expects("isRootBindingSuspended").withExactArgs().returns(true);
		this.mock(oBinding).expects("getDependentBindings").never();
		this.mock(oBinding).expects("reset").never();
		this.mock(oBinding.oHeaderContext).expects("checkUpdateInternal").never();

		// code under test
		oBinding.onChange();

		assert.strictEqual(oBinding.sResumeAction, "onChange");
	});

	//*********************************************************************************************
	QUnit.test("getKeepAlivePredicates", function (assert) {
		var oBinding = this.bindList("/n/a"), // absolute, but path is irrelevant
			oContext0 = {
				getPath : function () {},
				isKeepAlive : function () {}
			},
			oContext1 = {
				isKeepAlive : function () {}
			},
			oContext2 = {
				getPath : function () {},
				isKeepAlive : function () {}
			},
			oContext3 = {
				isKeepAlive : function () {}
			},
			oContext4 = {
				getPath : function () {},
				isKeepAlive : function () {}
			};

		oBinding.mPreviousContextsByPath = {
			a : oContext0,
			b : oContext1,
			c : oContext2
		};
		oBinding.aContexts = [oContext3, oContext4];
		this.mock(oBinding.getHeaderContext()).expects("getPath").withExactArgs()
			.returns("/binding/path");
		this.mock(oContext0).expects("isKeepAlive").withExactArgs().returns(true);
		this.mock(oContext1).expects("isKeepAlive").withExactArgs().returns(false);
		this.mock(oContext2).expects("isKeepAlive").withExactArgs().returns(true);
		this.mock(oContext3).expects("isKeepAlive").withExactArgs().returns(false);
		this.mock(oContext4).expects("isKeepAlive").withExactArgs().returns(true);
		this.mock(oContext0).expects("getPath").withExactArgs().returns("/binding/path('0')");
		this.mock(oContext2).expects("getPath").withExactArgs().returns("/binding/path('2')");
		this.mock(oContext4).expects("getPath").withExactArgs().returns("/binding/path('4')");

		assert.deepEqual(
			oBinding.getKeepAlivePredicates(), // code under test
			["('0')", "('2')", "('4')"]
		);
	});
});

//TODO integration: 2 entity sets with same $expand, but different $select
//TODO extended change detection:
//     Wir sollten auch dafür sorgen, dass die Antwort auf diesen "change"-Event dann keinen Diff
//     enthält. So macht es v2, und das haben wir letzte Woche erst richtig verstanden.
