# frozen_string_literal: true

# Released under the MIT License.
# Copyright, 2022, by Samuel Williams.

require 'bake/modernize'

LICENSE = <<~LICENSE
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
LICENSE

# @parameter root [String] The root directory to scan for changes.
def license
	update(root: Dir.pwd)
end

def update(root:)
	authorship = Bake::Modernize::License::Authorship.new
	authorship.extract(root)
	
	buffer = StringIO.new
	buffer.puts "# MIT License"
	buffer.puts
	
	authorship.copyrights.each do |copyright|
		buffer.puts "#{copyright.statement}  "
	end
	
	buffer.puts
	buffer.puts LICENSE
	
	File.open("license.md", "w") do |file|
		file.write(buffer.string)
	end
	
	authorship.paths.each do |path, modifications|
		next unless File.exist?(path)
		
		case path
		when /\.rb$/
			self.update_source_file_authors(authorship, path, modifications)
		end
	end
end

private

def update_source_file_authors(authorship, path, modifications)
	copyrights = authorship.copyrights_for_modifications(modifications)
	
	input = File.readlines(path)
	output = []
	
	if input.first =~ /^\#!/
		output.push input.shift
	end
	
	if input.first =~ /^\#.*?\:/
		output.push input.shift
		if input.first.chomp.empty?
			input.shift
		end
		output << "\n"
	end
	
	# Remove any existing license:
	while input.first =~ /^#.*$/
		input.shift
	end
	
	# Remove any empty lines:
	while input.first&.chomp&.empty?
		input.shift
	end
	
	output << "# Released under the MIT License.\n"
	
	copyrights.each do |copyright|
		output << "# #{copyright.statement}\n"
	end
	
	output << "\n"
	
	output.concat(input)
	
	File.open(path, "w") do |file|
		file.puts(output)
	end
end
