/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.action.bulk;

import java.util.Iterator;
import java.util.NoSuchElementException;
import org.opensearch.common.Randomness;
import org.opensearch.common.unit.TimeValue;

public abstract class BackoffPolicy
implements Iterable<TimeValue> {
    private static final BackoffPolicy NO_BACKOFF = new NoBackoff();

    public static BackoffPolicy noBackoff() {
        return NO_BACKOFF;
    }

    public static BackoffPolicy constantBackoff(TimeValue delay, int maxNumberOfRetries) {
        return new ConstantBackoff(BackoffPolicy.checkDelay(delay), maxNumberOfRetries);
    }

    public static BackoffPolicy exponentialBackoff() {
        return BackoffPolicy.exponentialBackoff(TimeValue.timeValueMillis((long)50L), 8);
    }

    public static BackoffPolicy exponentialBackoff(TimeValue initialDelay, int maxNumberOfRetries) {
        return new ExponentialBackoff((int)BackoffPolicy.checkDelay(initialDelay).millis(), maxNumberOfRetries);
    }

    public static BackoffPolicy exponentialEqualJitterBackoff(int baseDelay, int maxDelayForRetry) {
        return new ExponentialEqualJitterBackoff(baseDelay, maxDelayForRetry);
    }

    public static BackoffPolicy exponentialFullJitterBackoff(long baseDelay) {
        return new ExponentialFullJitterBackoff(baseDelay);
    }

    public static BackoffPolicy wrap(BackoffPolicy delegate, Runnable onBackoff) {
        return new WrappedBackoffPolicy(delegate, onBackoff);
    }

    private static TimeValue checkDelay(TimeValue delay) {
        if (delay.millis() > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("delay must be <= 2147483647 ms");
        }
        return delay;
    }

    private static final class WrappedBackoffIterator
    implements Iterator<TimeValue> {
        private final Iterator<TimeValue> delegate;
        private final Runnable onBackoff;

        WrappedBackoffIterator(Iterator<TimeValue> delegate, Runnable onBackoff) {
            this.delegate = delegate;
            this.onBackoff = onBackoff;
        }

        @Override
        public boolean hasNext() {
            return this.delegate.hasNext();
        }

        @Override
        public TimeValue next() {
            if (!this.delegate.hasNext()) {
                throw new NoSuchElementException();
            }
            this.onBackoff.run();
            return this.delegate.next();
        }
    }

    private static final class WrappedBackoffPolicy
    extends BackoffPolicy {
        private final BackoffPolicy delegate;
        private final Runnable onBackoff;

        WrappedBackoffPolicy(BackoffPolicy delegate, Runnable onBackoff) {
            this.delegate = delegate;
            this.onBackoff = onBackoff;
        }

        @Override
        public Iterator<TimeValue> iterator() {
            return new WrappedBackoffIterator(this.delegate.iterator(), this.onBackoff);
        }
    }

    private static final class ConstantBackoffIterator
    implements Iterator<TimeValue> {
        private final TimeValue delay;
        private final int numberOfElements;
        private int curr;

        ConstantBackoffIterator(TimeValue delay, int numberOfElements) {
            this.delay = delay;
            this.numberOfElements = numberOfElements;
        }

        @Override
        public boolean hasNext() {
            return this.curr < this.numberOfElements;
        }

        @Override
        public TimeValue next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            ++this.curr;
            return this.delay;
        }
    }

    private static final class ConstantBackoff
    extends BackoffPolicy {
        private final TimeValue delay;
        private final int numberOfElements;

        ConstantBackoff(TimeValue delay, int numberOfElements) {
            assert (numberOfElements >= 0);
            this.delay = delay;
            this.numberOfElements = numberOfElements;
        }

        @Override
        public Iterator<TimeValue> iterator() {
            return new ConstantBackoffIterator(this.delay, this.numberOfElements);
        }
    }

    private static class ExponentialFullJitterBackoffIterator
    implements Iterator<TimeValue> {
        private long currentDelay;

        private ExponentialFullJitterBackoffIterator(long baseDelay) {
            this.currentDelay = baseDelay;
        }

        @Override
        public boolean hasNext() {
            return true;
        }

        @Override
        public TimeValue next() {
            TimeValue delayToReturn = TimeValue.timeValueMillis((long)(Randomness.get().nextInt(Math.toIntExact(this.currentDelay)) + 1));
            this.currentDelay = Math.min(2L * this.currentDelay, Integer.MAX_VALUE);
            return delayToReturn;
        }
    }

    private static class ExponentialFullJitterBackoff
    extends BackoffPolicy {
        private final long baseDelay;

        private ExponentialFullJitterBackoff(long baseDelay) {
            this.baseDelay = baseDelay;
        }

        @Override
        public Iterator<TimeValue> iterator() {
            return new ExponentialFullJitterBackoffIterator(this.baseDelay);
        }
    }

    private static class ExponentialEqualJitterBackoffIterator
    implements Iterator<TimeValue> {
        private final int RETRIES_TILL_JITTER_INCREASE = 30;
        private final int maxDelayForRetry;
        private final int baseDelay;
        private int retriesAttempted;

        private ExponentialEqualJitterBackoffIterator(int baseDelay, int maxDelayForRetry) {
            this.baseDelay = baseDelay;
            this.maxDelayForRetry = maxDelayForRetry;
        }

        @Override
        public boolean hasNext() {
            return true;
        }

        @Override
        public TimeValue next() {
            int retries = Math.min(this.retriesAttempted, 30);
            int exponentialDelay = (int)Math.min((1L << retries) * (long)this.baseDelay, (long)this.maxDelayForRetry);
            ++this.retriesAttempted;
            return TimeValue.timeValueMillis((long)(exponentialDelay / 2 + Randomness.get().nextInt(exponentialDelay / 2 + 1)));
        }
    }

    private static class ExponentialEqualJitterBackoff
    extends BackoffPolicy {
        private final int maxDelayForRetry;
        private final int baseDelay;

        private ExponentialEqualJitterBackoff(int baseDelay, int maxDelayForRetry) {
            this.maxDelayForRetry = maxDelayForRetry;
            this.baseDelay = baseDelay;
        }

        @Override
        public Iterator<TimeValue> iterator() {
            return new ExponentialEqualJitterBackoffIterator(this.baseDelay, this.maxDelayForRetry);
        }
    }

    private static class ExponentialBackoffIterator
    implements Iterator<TimeValue> {
        private final int numberOfElements;
        private final int start;
        private int currentlyConsumed;

        private ExponentialBackoffIterator(int start, int numberOfElements) {
            this.start = start;
            this.numberOfElements = numberOfElements;
        }

        @Override
        public boolean hasNext() {
            return this.currentlyConsumed < this.numberOfElements;
        }

        @Override
        public TimeValue next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException("Only up to " + this.numberOfElements + " elements");
            }
            int result = this.start + 10 * ((int)Math.exp(0.8 * (double)this.currentlyConsumed) - 1);
            ++this.currentlyConsumed;
            return TimeValue.timeValueMillis((long)result);
        }
    }

    private static class ExponentialBackoff
    extends BackoffPolicy {
        private final int start;
        private final int numberOfElements;

        private ExponentialBackoff(int start, int numberOfElements) {
            assert (start >= 0);
            assert (numberOfElements >= 0);
            this.start = start;
            this.numberOfElements = numberOfElements;
        }

        @Override
        public Iterator<TimeValue> iterator() {
            return new ExponentialBackoffIterator(this.start, this.numberOfElements);
        }
    }

    private static class NoBackoff
    extends BackoffPolicy {
        private NoBackoff() {
        }

        @Override
        public Iterator<TimeValue> iterator() {
            return new Iterator<TimeValue>(){

                @Override
                public boolean hasNext() {
                    return false;
                }

                @Override
                public TimeValue next() {
                    throw new NoSuchElementException("No backoff");
                }
            };
        }
    }
}

