"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MultipleAuthentication = void 0;

var _authentication_type = require("../authentication_type");

var _common = require("../../../../common");

var _next_url = require("../../../utils/next_url");

var _routes = require("./routes");

var _types = require("../../types");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class MultipleAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    super(config, sessionStorageFactory, router, esClient, coreSetup, logger);

    _defineProperty(this, "authTypes", void 0);

    _defineProperty(this, "authHandlers", void 0);

    this.authTypes = this.config.auth.type;
    this.authHandlers = new Map();
  }

  async init() {
    const routes = new _routes.MultiAuthRoutes(this.router, this.sessionStorageFactory);
    routes.setupRoutes();

    for (let i = 0; i < this.authTypes.length; i++) {
      switch (this.authTypes[i].toLowerCase()) {
        case _common.AuthType.BASIC:
          {
            const BasicAuth = new _types.BasicAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await BasicAuth.init();
            this.authHandlers.set(_common.AuthType.BASIC, BasicAuth);
            break;
          }

        case _common.AuthType.OPEN_ID:
          {
            const OidcAuth = new _types.OpenIdAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await OidcAuth.init();
            this.authHandlers.set(_common.AuthType.OPEN_ID, OidcAuth);
            break;
          }

        case _common.AuthType.SAML:
          {
            const SamlAuth = new _types.SamlAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await SamlAuth.init();
            this.authHandlers.set(_common.AuthType.SAML, SamlAuth);
            break;
          }

        default:
          {
            throw new Error(`Unsupported authentication type: ${this.authTypes[i]}`);
          }
      }
    }
  } // override functions inherited from AuthenticationType


  requestIncludesAuthInfo(request) {
    for (const key of this.authHandlers.keys()) {
      if (this.authHandlers.get(key).requestIncludesAuthInfo(request)) {
        return true;
      }
    }

    return false;
  }

  async getAdditionalAuthHeader(request) {
    var _cookie$authType;

    // To Do: refactor this method to improve the effiency to get cookie, get cookie from input parameter
    const cookie = await this.sessionStorageFactory.asScoped(request).get();
    const reqAuthType = cookie === null || cookie === void 0 ? void 0 : (_cookie$authType = cookie.authType) === null || _cookie$authType === void 0 ? void 0 : _cookie$authType.toLowerCase();

    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return this.authHandlers.get(reqAuthType).getAdditionalAuthHeader(request);
    } else {
      return {};
    }
  }

  getCookie(request, authInfo) {
    return {};
  }

  async isValidCookie(cookie) {
    var _cookie$authType2;

    const reqAuthType = cookie === null || cookie === void 0 ? void 0 : (_cookie$authType2 = cookie.authType) === null || _cookie$authType2 === void 0 ? void 0 : _cookie$authType2.toLowerCase();

    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return this.authHandlers.get(reqAuthType).isValidCookie(cookie);
    } else {
      return false;
    }
  }

  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      const nextUrlParam = (0, _next_url.composeNextUrlQueryParam)(request, this.coreSetup.http.basePath.serverBasePath);

      if (this.config.auth.anonymous_auth_enabled) {
        const redirectLocation = `${this.coreSetup.http.basePath.serverBasePath}${_common.ANONYMOUS_AUTH_LOGIN}?${nextUrlParam}`;
        return response.redirected({
          headers: {
            location: `${redirectLocation}`
          }
        });
      }

      return response.redirected({
        headers: {
          location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}?${nextUrlParam}`
        }
      });
    } else {
      return response.unauthorized();
    }
  }

  buildAuthHeaderFromCookie(cookie) {
    var _cookie$authType3;

    const reqAuthType = cookie === null || cookie === void 0 ? void 0 : (_cookie$authType3 = cookie.authType) === null || _cookie$authType3 === void 0 ? void 0 : _cookie$authType3.toLowerCase();

    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return this.authHandlers.get(reqAuthType).buildAuthHeaderFromCookie(cookie);
    } else {
      return {};
    }
  }

}

exports.MultipleAuthentication = MultipleAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIm11bHRpX2F1dGgudHMiXSwibmFtZXMiOlsiTXVsdGlwbGVBdXRoZW50aWNhdGlvbiIsIkF1dGhlbnRpY2F0aW9uVHlwZSIsImNvbnN0cnVjdG9yIiwiY29uZmlnIiwic2Vzc2lvblN0b3JhZ2VGYWN0b3J5Iiwicm91dGVyIiwiZXNDbGllbnQiLCJjb3JlU2V0dXAiLCJsb2dnZXIiLCJhdXRoVHlwZXMiLCJhdXRoIiwidHlwZSIsImF1dGhIYW5kbGVycyIsIk1hcCIsImluaXQiLCJyb3V0ZXMiLCJNdWx0aUF1dGhSb3V0ZXMiLCJzZXR1cFJvdXRlcyIsImkiLCJsZW5ndGgiLCJ0b0xvd2VyQ2FzZSIsIkF1dGhUeXBlIiwiQkFTSUMiLCJCYXNpY0F1dGgiLCJCYXNpY0F1dGhlbnRpY2F0aW9uIiwic2V0IiwiT1BFTl9JRCIsIk9pZGNBdXRoIiwiT3BlbklkQXV0aGVudGljYXRpb24iLCJTQU1MIiwiU2FtbEF1dGgiLCJTYW1sQXV0aGVudGljYXRpb24iLCJFcnJvciIsInJlcXVlc3RJbmNsdWRlc0F1dGhJbmZvIiwicmVxdWVzdCIsImtleSIsImtleXMiLCJnZXQiLCJnZXRBZGRpdGlvbmFsQXV0aEhlYWRlciIsImNvb2tpZSIsImFzU2NvcGVkIiwicmVxQXV0aFR5cGUiLCJhdXRoVHlwZSIsImhhcyIsImdldENvb2tpZSIsImF1dGhJbmZvIiwiaXNWYWxpZENvb2tpZSIsImhhbmRsZVVuYXV0aGVkUmVxdWVzdCIsInJlc3BvbnNlIiwidG9vbGtpdCIsImlzUGFnZVJlcXVlc3QiLCJuZXh0VXJsUGFyYW0iLCJodHRwIiwiYmFzZVBhdGgiLCJzZXJ2ZXJCYXNlUGF0aCIsImFub255bW91c19hdXRoX2VuYWJsZWQiLCJyZWRpcmVjdExvY2F0aW9uIiwiQU5PTllNT1VTX0FVVEhfTE9HSU4iLCJyZWRpcmVjdGVkIiwiaGVhZGVycyIsImxvY2F0aW9uIiwiTE9HSU5fUEFHRV9VUkkiLCJ1bmF1dGhvcml6ZWQiLCJidWlsZEF1dGhIZWFkZXJGcm9tQ29va2llIl0sIm1hcHBpbmdzIjoiOzs7Ozs7O0FBMEJBOztBQUNBOztBQUNBOztBQUNBOztBQUVBOzs7O0FBRU8sTUFBTUEsc0JBQU4sU0FBcUNDLHVDQUFyQyxDQUF3RDtBQUk3REMsRUFBQUEsV0FBVyxDQUNUQyxNQURTLEVBRVRDLHFCQUZTLEVBR1RDLE1BSFMsRUFJVEMsUUFKUyxFQUtUQyxTQUxTLEVBTVRDLE1BTlMsRUFPVDtBQUNBLFVBQU1MLE1BQU4sRUFBY0MscUJBQWQsRUFBcUNDLE1BQXJDLEVBQTZDQyxRQUE3QyxFQUF1REMsU0FBdkQsRUFBa0VDLE1BQWxFOztBQURBOztBQUFBOztBQUVBLFNBQUtDLFNBQUwsR0FBaUIsS0FBS04sTUFBTCxDQUFZTyxJQUFaLENBQWlCQyxJQUFsQztBQUNBLFNBQUtDLFlBQUwsR0FBb0IsSUFBSUMsR0FBSixFQUFwQjtBQUNEOztBQUVnQixRQUFKQyxJQUFJLEdBQUc7QUFDbEIsVUFBTUMsTUFBTSxHQUFHLElBQUlDLHVCQUFKLENBQW9CLEtBQUtYLE1BQXpCLEVBQWlDLEtBQUtELHFCQUF0QyxDQUFmO0FBQ0FXLElBQUFBLE1BQU0sQ0FBQ0UsV0FBUDs7QUFFQSxTQUFLLElBQUlDLENBQUMsR0FBRyxDQUFiLEVBQWdCQSxDQUFDLEdBQUcsS0FBS1QsU0FBTCxDQUFlVSxNQUFuQyxFQUEyQ0QsQ0FBQyxFQUE1QyxFQUFnRDtBQUM5QyxjQUFRLEtBQUtULFNBQUwsQ0FBZVMsQ0FBZixFQUFrQkUsV0FBbEIsRUFBUjtBQUNFLGFBQUtDLGlCQUFTQyxLQUFkO0FBQXFCO0FBQ25CLGtCQUFNQyxTQUFTLEdBQUcsSUFBSUMsMEJBQUosQ0FDaEIsS0FBS3JCLE1BRFcsRUFFaEIsS0FBS0MscUJBRlcsRUFHaEIsS0FBS0MsTUFIVyxFQUloQixLQUFLQyxRQUpXLEVBS2hCLEtBQUtDLFNBTFcsRUFNaEIsS0FBS0MsTUFOVyxDQUFsQjtBQVFBLGtCQUFNZSxTQUFTLENBQUNULElBQVYsRUFBTjtBQUNBLGlCQUFLRixZQUFMLENBQWtCYSxHQUFsQixDQUFzQkosaUJBQVNDLEtBQS9CLEVBQXNDQyxTQUF0QztBQUNBO0FBQ0Q7O0FBQ0QsYUFBS0YsaUJBQVNLLE9BQWQ7QUFBdUI7QUFDckIsa0JBQU1DLFFBQVEsR0FBRyxJQUFJQywyQkFBSixDQUNmLEtBQUt6QixNQURVLEVBRWYsS0FBS0MscUJBRlUsRUFHZixLQUFLQyxNQUhVLEVBSWYsS0FBS0MsUUFKVSxFQUtmLEtBQUtDLFNBTFUsRUFNZixLQUFLQyxNQU5VLENBQWpCO0FBUUEsa0JBQU1tQixRQUFRLENBQUNiLElBQVQsRUFBTjtBQUNBLGlCQUFLRixZQUFMLENBQWtCYSxHQUFsQixDQUFzQkosaUJBQVNLLE9BQS9CLEVBQXdDQyxRQUF4QztBQUNBO0FBQ0Q7O0FBQ0QsYUFBS04saUJBQVNRLElBQWQ7QUFBb0I7QUFDbEIsa0JBQU1DLFFBQVEsR0FBRyxJQUFJQyx5QkFBSixDQUNmLEtBQUs1QixNQURVLEVBRWYsS0FBS0MscUJBRlUsRUFHZixLQUFLQyxNQUhVLEVBSWYsS0FBS0MsUUFKVSxFQUtmLEtBQUtDLFNBTFUsRUFNZixLQUFLQyxNQU5VLENBQWpCO0FBUUEsa0JBQU1zQixRQUFRLENBQUNoQixJQUFULEVBQU47QUFDQSxpQkFBS0YsWUFBTCxDQUFrQmEsR0FBbEIsQ0FBc0JKLGlCQUFTUSxJQUEvQixFQUFxQ0MsUUFBckM7QUFDQTtBQUNEOztBQUNEO0FBQVM7QUFDUCxrQkFBTSxJQUFJRSxLQUFKLENBQVcsb0NBQW1DLEtBQUt2QixTQUFMLENBQWVTLENBQWYsQ0FBa0IsRUFBaEUsQ0FBTjtBQUNEO0FBMUNIO0FBNENEO0FBQ0YsR0FuRTRELENBcUU3RDs7O0FBQ0FlLEVBQUFBLHVCQUF1QixDQUNyQkMsT0FEcUIsRUFFWjtBQUNULFNBQUssTUFBTUMsR0FBWCxJQUFrQixLQUFLdkIsWUFBTCxDQUFrQndCLElBQWxCLEVBQWxCLEVBQTRDO0FBQzFDLFVBQUksS0FBS3hCLFlBQUwsQ0FBa0J5QixHQUFsQixDQUFzQkYsR0FBdEIsRUFBNEJGLHVCQUE1QixDQUFvREMsT0FBcEQsQ0FBSixFQUFrRTtBQUNoRSxlQUFPLElBQVA7QUFDRDtBQUNGOztBQUNELFdBQU8sS0FBUDtBQUNEOztBQUU0QixRQUF2QkksdUJBQXVCLENBQzNCSixPQUQyQixFQUViO0FBQUE7O0FBQ2Q7QUFDQSxVQUFNSyxNQUFNLEdBQUcsTUFBTSxLQUFLbkMscUJBQUwsQ0FBMkJvQyxRQUEzQixDQUFvQ04sT0FBcEMsRUFBNkNHLEdBQTdDLEVBQXJCO0FBQ0EsVUFBTUksV0FBVyxHQUFHRixNQUFILGFBQUdBLE1BQUgsMkNBQUdBLE1BQU0sQ0FBRUcsUUFBWCxxREFBRyxpQkFBa0J0QixXQUFsQixFQUFwQjs7QUFFQSxRQUFJcUIsV0FBVyxJQUFJLEtBQUs3QixZQUFMLENBQWtCK0IsR0FBbEIsQ0FBc0JGLFdBQXRCLENBQW5CLEVBQXVEO0FBQ3JELGFBQU8sS0FBSzdCLFlBQUwsQ0FBa0J5QixHQUFsQixDQUFzQkksV0FBdEIsRUFBb0NILHVCQUFwQyxDQUE0REosT0FBNUQsQ0FBUDtBQUNELEtBRkQsTUFFTztBQUNMLGFBQU8sRUFBUDtBQUNEO0FBQ0Y7O0FBRURVLEVBQUFBLFNBQVMsQ0FBQ1YsT0FBRCxFQUF1Q1csUUFBdkMsRUFBNkU7QUFDcEYsV0FBTyxFQUFQO0FBQ0Q7O0FBRWtCLFFBQWJDLGFBQWEsQ0FBQ1AsTUFBRCxFQUFrRDtBQUFBOztBQUNuRSxVQUFNRSxXQUFXLEdBQUdGLE1BQUgsYUFBR0EsTUFBSCw0Q0FBR0EsTUFBTSxDQUFFRyxRQUFYLHNEQUFHLGtCQUFrQnRCLFdBQWxCLEVBQXBCOztBQUNBLFFBQUlxQixXQUFXLElBQUksS0FBSzdCLFlBQUwsQ0FBa0IrQixHQUFsQixDQUFzQkYsV0FBdEIsQ0FBbkIsRUFBdUQ7QUFDckQsYUFBTyxLQUFLN0IsWUFBTCxDQUFrQnlCLEdBQWxCLENBQXNCSSxXQUF0QixFQUFvQ0ssYUFBcEMsQ0FBa0RQLE1BQWxELENBQVA7QUFDRCxLQUZELE1BRU87QUFDTCxhQUFPLEtBQVA7QUFDRDtBQUNGOztBQUVEUSxFQUFBQSxxQkFBcUIsQ0FDbkJiLE9BRG1CLEVBRW5CYyxRQUZtQixFQUduQkMsT0FIbUIsRUFJVztBQUM5QixRQUFJLEtBQUtDLGFBQUwsQ0FBbUJoQixPQUFuQixDQUFKLEVBQWlDO0FBQy9CLFlBQU1pQixZQUFZLEdBQUcsd0NBQ25CakIsT0FEbUIsRUFFbkIsS0FBSzNCLFNBQUwsQ0FBZTZDLElBQWYsQ0FBb0JDLFFBQXBCLENBQTZCQyxjQUZWLENBQXJCOztBQUtBLFVBQUksS0FBS25ELE1BQUwsQ0FBWU8sSUFBWixDQUFpQjZDLHNCQUFyQixFQUE2QztBQUMzQyxjQUFNQyxnQkFBZ0IsR0FBSSxHQUFFLEtBQUtqRCxTQUFMLENBQWU2QyxJQUFmLENBQW9CQyxRQUFwQixDQUE2QkMsY0FBZSxHQUFFRyw0QkFBcUIsSUFBR04sWUFBYSxFQUEvRztBQUNBLGVBQU9ILFFBQVEsQ0FBQ1UsVUFBVCxDQUFvQjtBQUN6QkMsVUFBQUEsT0FBTyxFQUFFO0FBQ1BDLFlBQUFBLFFBQVEsRUFBRyxHQUFFSixnQkFBaUI7QUFEdkI7QUFEZ0IsU0FBcEIsQ0FBUDtBQUtEOztBQUNELGFBQU9SLFFBQVEsQ0FBQ1UsVUFBVCxDQUFvQjtBQUN6QkMsUUFBQUEsT0FBTyxFQUFFO0FBQ1BDLFVBQUFBLFFBQVEsRUFBRyxHQUFFLEtBQUtyRCxTQUFMLENBQWU2QyxJQUFmLENBQW9CQyxRQUFwQixDQUE2QkMsY0FBZSxHQUFFTyxzQkFBZSxJQUFHVixZQUFhO0FBRG5GO0FBRGdCLE9BQXBCLENBQVA7QUFLRCxLQW5CRCxNQW1CTztBQUNMLGFBQU9ILFFBQVEsQ0FBQ2MsWUFBVCxFQUFQO0FBQ0Q7QUFDRjs7QUFFREMsRUFBQUEseUJBQXlCLENBQUN4QixNQUFELEVBQXFDO0FBQUE7O0FBQzVELFVBQU1FLFdBQVcsR0FBR0YsTUFBSCxhQUFHQSxNQUFILDRDQUFHQSxNQUFNLENBQUVHLFFBQVgsc0RBQUcsa0JBQWtCdEIsV0FBbEIsRUFBcEI7O0FBRUEsUUFBSXFCLFdBQVcsSUFBSSxLQUFLN0IsWUFBTCxDQUFrQitCLEdBQWxCLENBQXNCRixXQUF0QixDQUFuQixFQUF1RDtBQUNyRCxhQUFPLEtBQUs3QixZQUFMLENBQWtCeUIsR0FBbEIsQ0FBc0JJLFdBQXRCLEVBQW9Dc0IseUJBQXBDLENBQThEeEIsTUFBOUQsQ0FBUDtBQUNELEtBRkQsTUFFTztBQUNMLGFBQU8sRUFBUDtBQUNEO0FBQ0Y7O0FBako0RCIsInNvdXJjZXNDb250ZW50IjpbIi8qXG4gKiAgIENvcHlyaWdodCBPcGVuU2VhcmNoIENvbnRyaWJ1dG9yc1xuICpcbiAqICAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS5cbiAqICAgWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZSB3aXRoIHRoZSBMaWNlbnNlLlxuICogICBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICAgb3IgaW4gdGhlIFwibGljZW5zZVwiIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkXG4gKiAgIG9uIGFuIFwiQVMgSVNcIiBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGVpdGhlclxuICogICBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZ1xuICogICBwZXJtaXNzaW9ucyBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cbmltcG9ydCB7XG4gIENvcmVTZXR1cCxcbiAgU2Vzc2lvblN0b3JhZ2VGYWN0b3J5LFxuICBJUm91dGVyLFxuICBJTGVnYWN5Q2x1c3RlckNsaWVudCxcbiAgT3BlblNlYXJjaERhc2hib2FyZHNSZXF1ZXN0LFxuICBMb2dnZXIsXG4gIExpZmVjeWNsZVJlc3BvbnNlRmFjdG9yeSxcbiAgQXV0aFRvb2xraXQsXG59IGZyb20gJy4uLy4uLy4uLy4uL29wZW5zZWFyY2gtZGFzaGJvYXJkcy9zZXJ2ZXInO1xuaW1wb3J0IHsgT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZSB9IGZyb20gJy4uLy4uLy4uLy4uLy4uLy4uL3NyYy9jb3JlL3NlcnZlci9odHRwL3JvdXRlcic7XG5pbXBvcnQgeyBTZWN1cml0eVBsdWdpbkNvbmZpZ1R5cGUgfSBmcm9tICcuLi8uLi8uLic7XG5pbXBvcnQgeyBBdXRoZW50aWNhdGlvblR5cGUsIElBdXRoZW50aWNhdGlvblR5cGUgfSBmcm9tICcuLi9hdXRoZW50aWNhdGlvbl90eXBlJztcbmltcG9ydCB7IEFOT05ZTU9VU19BVVRIX0xPR0lOLCBBdXRoVHlwZSwgTE9HSU5fUEFHRV9VUkkgfSBmcm9tICcuLi8uLi8uLi8uLi9jb21tb24nO1xuaW1wb3J0IHsgY29tcG9zZU5leHRVcmxRdWVyeVBhcmFtIH0gZnJvbSAnLi4vLi4vLi4vdXRpbHMvbmV4dF91cmwnO1xuaW1wb3J0IHsgTXVsdGlBdXRoUm91dGVzIH0gZnJvbSAnLi9yb3V0ZXMnO1xuaW1wb3J0IHsgU2VjdXJpdHlTZXNzaW9uQ29va2llIH0gZnJvbSAnLi4vLi4vLi4vc2Vzc2lvbi9zZWN1cml0eV9jb29raWUnO1xuaW1wb3J0IHsgQmFzaWNBdXRoZW50aWNhdGlvbiwgT3BlbklkQXV0aGVudGljYXRpb24sIFNhbWxBdXRoZW50aWNhdGlvbiB9IGZyb20gJy4uLy4uL3R5cGVzJztcblxuZXhwb3J0IGNsYXNzIE11bHRpcGxlQXV0aGVudGljYXRpb24gZXh0ZW5kcyBBdXRoZW50aWNhdGlvblR5cGUge1xuICBwcml2YXRlIGF1dGhUeXBlczogc3RyaW5nIHwgc3RyaW5nW107XG4gIHByaXZhdGUgYXV0aEhhbmRsZXJzOiBNYXA8c3RyaW5nLCBBdXRoZW50aWNhdGlvblR5cGU+O1xuXG4gIGNvbnN0cnVjdG9yKFxuICAgIGNvbmZpZzogU2VjdXJpdHlQbHVnaW5Db25maWdUeXBlLFxuICAgIHNlc3Npb25TdG9yYWdlRmFjdG9yeTogU2Vzc2lvblN0b3JhZ2VGYWN0b3J5PFNlY3VyaXR5U2Vzc2lvbkNvb2tpZT4sXG4gICAgcm91dGVyOiBJUm91dGVyLFxuICAgIGVzQ2xpZW50OiBJTGVnYWN5Q2x1c3RlckNsaWVudCxcbiAgICBjb3JlU2V0dXA6IENvcmVTZXR1cCxcbiAgICBsb2dnZXI6IExvZ2dlclxuICApIHtcbiAgICBzdXBlcihjb25maWcsIHNlc3Npb25TdG9yYWdlRmFjdG9yeSwgcm91dGVyLCBlc0NsaWVudCwgY29yZVNldHVwLCBsb2dnZXIpO1xuICAgIHRoaXMuYXV0aFR5cGVzID0gdGhpcy5jb25maWcuYXV0aC50eXBlO1xuICAgIHRoaXMuYXV0aEhhbmRsZXJzID0gbmV3IE1hcDxzdHJpbmcsIEF1dGhlbnRpY2F0aW9uVHlwZT4oKTtcbiAgfVxuXG4gIHB1YmxpYyBhc3luYyBpbml0KCkge1xuICAgIGNvbnN0IHJvdXRlcyA9IG5ldyBNdWx0aUF1dGhSb3V0ZXModGhpcy5yb3V0ZXIsIHRoaXMuc2Vzc2lvblN0b3JhZ2VGYWN0b3J5KTtcbiAgICByb3V0ZXMuc2V0dXBSb3V0ZXMoKTtcblxuICAgIGZvciAobGV0IGkgPSAwOyBpIDwgdGhpcy5hdXRoVHlwZXMubGVuZ3RoOyBpKyspIHtcbiAgICAgIHN3aXRjaCAodGhpcy5hdXRoVHlwZXNbaV0udG9Mb3dlckNhc2UoKSkge1xuICAgICAgICBjYXNlIEF1dGhUeXBlLkJBU0lDOiB7XG4gICAgICAgICAgY29uc3QgQmFzaWNBdXRoID0gbmV3IEJhc2ljQXV0aGVudGljYXRpb24oXG4gICAgICAgICAgICB0aGlzLmNvbmZpZyxcbiAgICAgICAgICAgIHRoaXMuc2Vzc2lvblN0b3JhZ2VGYWN0b3J5LFxuICAgICAgICAgICAgdGhpcy5yb3V0ZXIsXG4gICAgICAgICAgICB0aGlzLmVzQ2xpZW50LFxuICAgICAgICAgICAgdGhpcy5jb3JlU2V0dXAsXG4gICAgICAgICAgICB0aGlzLmxvZ2dlclxuICAgICAgICAgICk7XG4gICAgICAgICAgYXdhaXQgQmFzaWNBdXRoLmluaXQoKTtcbiAgICAgICAgICB0aGlzLmF1dGhIYW5kbGVycy5zZXQoQXV0aFR5cGUuQkFTSUMsIEJhc2ljQXV0aCk7XG4gICAgICAgICAgYnJlYWs7XG4gICAgICAgIH1cbiAgICAgICAgY2FzZSBBdXRoVHlwZS5PUEVOX0lEOiB7XG4gICAgICAgICAgY29uc3QgT2lkY0F1dGggPSBuZXcgT3BlbklkQXV0aGVudGljYXRpb24oXG4gICAgICAgICAgICB0aGlzLmNvbmZpZyxcbiAgICAgICAgICAgIHRoaXMuc2Vzc2lvblN0b3JhZ2VGYWN0b3J5LFxuICAgICAgICAgICAgdGhpcy5yb3V0ZXIsXG4gICAgICAgICAgICB0aGlzLmVzQ2xpZW50LFxuICAgICAgICAgICAgdGhpcy5jb3JlU2V0dXAsXG4gICAgICAgICAgICB0aGlzLmxvZ2dlclxuICAgICAgICAgICk7XG4gICAgICAgICAgYXdhaXQgT2lkY0F1dGguaW5pdCgpO1xuICAgICAgICAgIHRoaXMuYXV0aEhhbmRsZXJzLnNldChBdXRoVHlwZS5PUEVOX0lELCBPaWRjQXV0aCk7XG4gICAgICAgICAgYnJlYWs7XG4gICAgICAgIH1cbiAgICAgICAgY2FzZSBBdXRoVHlwZS5TQU1MOiB7XG4gICAgICAgICAgY29uc3QgU2FtbEF1dGggPSBuZXcgU2FtbEF1dGhlbnRpY2F0aW9uKFxuICAgICAgICAgICAgdGhpcy5jb25maWcsXG4gICAgICAgICAgICB0aGlzLnNlc3Npb25TdG9yYWdlRmFjdG9yeSxcbiAgICAgICAgICAgIHRoaXMucm91dGVyLFxuICAgICAgICAgICAgdGhpcy5lc0NsaWVudCxcbiAgICAgICAgICAgIHRoaXMuY29yZVNldHVwLFxuICAgICAgICAgICAgdGhpcy5sb2dnZXJcbiAgICAgICAgICApO1xuICAgICAgICAgIGF3YWl0IFNhbWxBdXRoLmluaXQoKTtcbiAgICAgICAgICB0aGlzLmF1dGhIYW5kbGVycy5zZXQoQXV0aFR5cGUuU0FNTCwgU2FtbEF1dGgpO1xuICAgICAgICAgIGJyZWFrO1xuICAgICAgICB9XG4gICAgICAgIGRlZmF1bHQ6IHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFVuc3VwcG9ydGVkIGF1dGhlbnRpY2F0aW9uIHR5cGU6ICR7dGhpcy5hdXRoVHlwZXNbaV19YCk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICAvLyBvdmVycmlkZSBmdW5jdGlvbnMgaW5oZXJpdGVkIGZyb20gQXV0aGVudGljYXRpb25UeXBlXG4gIHJlcXVlc3RJbmNsdWRlc0F1dGhJbmZvKFxuICAgIHJlcXVlc3Q6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdDx1bmtub3duLCB1bmtub3duLCB1bmtub3duLCBhbnk+XG4gICk6IGJvb2xlYW4ge1xuICAgIGZvciAoY29uc3Qga2V5IG9mIHRoaXMuYXV0aEhhbmRsZXJzLmtleXMoKSkge1xuICAgICAgaWYgKHRoaXMuYXV0aEhhbmRsZXJzLmdldChrZXkpIS5yZXF1ZXN0SW5jbHVkZXNBdXRoSW5mbyhyZXF1ZXN0KSkge1xuICAgICAgICByZXR1cm4gdHJ1ZTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG5cbiAgYXN5bmMgZ2V0QWRkaXRpb25hbEF1dGhIZWFkZXIoXG4gICAgcmVxdWVzdDogT3BlblNlYXJjaERhc2hib2FyZHNSZXF1ZXN0PHVua25vd24sIHVua25vd24sIHVua25vd24sIGFueT5cbiAgKTogUHJvbWlzZTxhbnk+IHtcbiAgICAvLyBUbyBEbzogcmVmYWN0b3IgdGhpcyBtZXRob2QgdG8gaW1wcm92ZSB0aGUgZWZmaWVuY3kgdG8gZ2V0IGNvb2tpZSwgZ2V0IGNvb2tpZSBmcm9tIGlucHV0IHBhcmFtZXRlclxuICAgIGNvbnN0IGNvb2tpZSA9IGF3YWl0IHRoaXMuc2Vzc2lvblN0b3JhZ2VGYWN0b3J5LmFzU2NvcGVkKHJlcXVlc3QpLmdldCgpO1xuICAgIGNvbnN0IHJlcUF1dGhUeXBlID0gY29va2llPy5hdXRoVHlwZT8udG9Mb3dlckNhc2UoKTtcblxuICAgIGlmIChyZXFBdXRoVHlwZSAmJiB0aGlzLmF1dGhIYW5kbGVycy5oYXMocmVxQXV0aFR5cGUpKSB7XG4gICAgICByZXR1cm4gdGhpcy5hdXRoSGFuZGxlcnMuZ2V0KHJlcUF1dGhUeXBlKSEuZ2V0QWRkaXRpb25hbEF1dGhIZWFkZXIocmVxdWVzdCk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiB7fTtcbiAgICB9XG4gIH1cblxuICBnZXRDb29raWUocmVxdWVzdDogT3BlblNlYXJjaERhc2hib2FyZHNSZXF1ZXN0LCBhdXRoSW5mbzogYW55KTogU2VjdXJpdHlTZXNzaW9uQ29va2llIHtcbiAgICByZXR1cm4ge307XG4gIH1cblxuICBhc3luYyBpc1ZhbGlkQ29va2llKGNvb2tpZTogU2VjdXJpdHlTZXNzaW9uQ29va2llKTogUHJvbWlzZTxib29sZWFuPiB7XG4gICAgY29uc3QgcmVxQXV0aFR5cGUgPSBjb29raWU/LmF1dGhUeXBlPy50b0xvd2VyQ2FzZSgpO1xuICAgIGlmIChyZXFBdXRoVHlwZSAmJiB0aGlzLmF1dGhIYW5kbGVycy5oYXMocmVxQXV0aFR5cGUpKSB7XG4gICAgICByZXR1cm4gdGhpcy5hdXRoSGFuZGxlcnMuZ2V0KHJlcUF1dGhUeXBlKSEuaXNWYWxpZENvb2tpZShjb29raWUpO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXR1cm4gZmFsc2U7XG4gICAgfVxuICB9XG5cbiAgaGFuZGxlVW5hdXRoZWRSZXF1ZXN0KFxuICAgIHJlcXVlc3Q6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdCxcbiAgICByZXNwb25zZTogTGlmZWN5Y2xlUmVzcG9uc2VGYWN0b3J5LFxuICAgIHRvb2xraXQ6IEF1dGhUb29sa2l0XG4gICk6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2Uge1xuICAgIGlmICh0aGlzLmlzUGFnZVJlcXVlc3QocmVxdWVzdCkpIHtcbiAgICAgIGNvbnN0IG5leHRVcmxQYXJhbSA9IGNvbXBvc2VOZXh0VXJsUXVlcnlQYXJhbShcbiAgICAgICAgcmVxdWVzdCxcbiAgICAgICAgdGhpcy5jb3JlU2V0dXAuaHR0cC5iYXNlUGF0aC5zZXJ2ZXJCYXNlUGF0aFxuICAgICAgKTtcblxuICAgICAgaWYgKHRoaXMuY29uZmlnLmF1dGguYW5vbnltb3VzX2F1dGhfZW5hYmxlZCkge1xuICAgICAgICBjb25zdCByZWRpcmVjdExvY2F0aW9uID0gYCR7dGhpcy5jb3JlU2V0dXAuaHR0cC5iYXNlUGF0aC5zZXJ2ZXJCYXNlUGF0aH0ke0FOT05ZTU9VU19BVVRIX0xPR0lOfT8ke25leHRVcmxQYXJhbX1gO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2UucmVkaXJlY3RlZCh7XG4gICAgICAgICAgaGVhZGVyczoge1xuICAgICAgICAgICAgbG9jYXRpb246IGAke3JlZGlyZWN0TG9jYXRpb259YCxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICAgIHJldHVybiByZXNwb25zZS5yZWRpcmVjdGVkKHtcbiAgICAgICAgaGVhZGVyczoge1xuICAgICAgICAgIGxvY2F0aW9uOiBgJHt0aGlzLmNvcmVTZXR1cC5odHRwLmJhc2VQYXRoLnNlcnZlckJhc2VQYXRofSR7TE9HSU5fUEFHRV9VUkl9PyR7bmV4dFVybFBhcmFtfWAsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIHJlc3BvbnNlLnVuYXV0aG9yaXplZCgpO1xuICAgIH1cbiAgfVxuXG4gIGJ1aWxkQXV0aEhlYWRlckZyb21Db29raWUoY29va2llOiBTZWN1cml0eVNlc3Npb25Db29raWUpOiBhbnkge1xuICAgIGNvbnN0IHJlcUF1dGhUeXBlID0gY29va2llPy5hdXRoVHlwZT8udG9Mb3dlckNhc2UoKTtcblxuICAgIGlmIChyZXFBdXRoVHlwZSAmJiB0aGlzLmF1dGhIYW5kbGVycy5oYXMocmVxQXV0aFR5cGUpKSB7XG4gICAgICByZXR1cm4gdGhpcy5hdXRoSGFuZGxlcnMuZ2V0KHJlcUF1dGhUeXBlKSEuYnVpbGRBdXRoSGVhZGVyRnJvbUNvb2tpZShjb29raWUpO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXR1cm4ge307XG4gICAgfVxuICB9XG59XG4iXX0=