/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.repositories.s3;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.List;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.opensearch.core.internal.io.IOUtils;
import org.opensearch.repositories.s3.AmazonS3Reference;
import org.opensearch.repositories.s3.S3BlobStore;
import org.opensearch.repositories.s3.SocketAccess;

class S3RetryingInputStream
extends InputStream {
    private static final Logger logger = LogManager.getLogger(S3RetryingInputStream.class);
    static final int MAX_SUPPRESSED_EXCEPTIONS = 10;
    private final S3BlobStore blobStore;
    private final String blobKey;
    private final long start;
    private final long end;
    private final int maxAttempts;
    private final List<IOException> failures;
    private S3ObjectInputStream currentStream;
    private long currentStreamLastOffset;
    private int attempt = 1;
    private long currentOffset;
    private boolean closed;
    private boolean eof;

    S3RetryingInputStream(S3BlobStore blobStore, String blobKey) throws IOException {
        this(blobStore, blobKey, 0L, 0x7FFFFFFFFFFFFFFEL);
    }

    S3RetryingInputStream(S3BlobStore blobStore, String blobKey, long start, long end) throws IOException {
        if (start < 0L) {
            throw new IllegalArgumentException("start must be non-negative");
        }
        if (end < start || end == Long.MAX_VALUE) {
            throw new IllegalArgumentException("end must be >= start and not Long.MAX_VALUE");
        }
        this.blobStore = blobStore;
        this.blobKey = blobKey;
        this.maxAttempts = blobStore.getMaxRetries() + 1;
        this.failures = new ArrayList<IOException>(10);
        this.start = start;
        this.end = end;
        this.openStream();
    }

    private void openStream() throws IOException {
        try (AmazonS3Reference clientReference = this.blobStore.clientReference();){
            GetObjectRequest getObjectRequest = new GetObjectRequest(this.blobStore.bucket(), this.blobKey);
            getObjectRequest.setRequestMetricCollector(this.blobStore.getMetricCollector);
            if (this.currentOffset > 0L || this.start > 0L || this.end < 0x7FFFFFFFFFFFFFFEL) {
                assert (this.start + this.currentOffset <= this.end) : "requesting beyond end, start = " + this.start + " offset=" + this.currentOffset + " end=" + this.end;
                getObjectRequest.setRange(Math.addExact(this.start, this.currentOffset), this.end);
            }
            S3Object s3Object = SocketAccess.doPrivileged(() -> ((AmazonS3)clientReference.get()).getObject(getObjectRequest));
            this.currentStreamLastOffset = Math.addExact(Math.addExact(this.start, this.currentOffset), this.getStreamLength(s3Object));
            this.currentStream = s3Object.getObjectContent();
        }
        catch (AmazonClientException e) {
            if (e instanceof AmazonS3Exception && 404 == ((AmazonS3Exception)e).getStatusCode()) {
                throw this.addSuppressedExceptions(new NoSuchFileException("Blob object [" + this.blobKey + "] not found: " + e.getMessage()));
            }
            throw this.addSuppressedExceptions(e);
        }
    }

    private long getStreamLength(S3Object object) {
        ObjectMetadata metadata = object.getObjectMetadata();
        try {
            Long[] range = metadata.getContentRange();
            if (range != null) {
                assert (range[1] >= range[0]) : range[1] + " vs " + range[0];
                assert (range[0] == this.start + this.currentOffset) : "Content-Range start value [" + range[0] + "] exceeds start [" + this.start + "] + current offset [" + this.currentOffset + "]";
                assert (range[1] == this.end) : "Content-Range end value [" + range[1] + "] exceeds end [" + this.end + "]";
                return range[1] - range[0] + 1L;
            }
            return metadata.getContentLength();
        }
        catch (Exception e) {
            assert (false) : e;
            return 0x7FFFFFFFFFFFFFFEL;
        }
    }

    @Override
    public int read() throws IOException {
        this.ensureOpen();
        while (true) {
            try {
                int result = this.currentStream.read();
                if (result == -1) {
                    this.eof = true;
                    return -1;
                }
                ++this.currentOffset;
                return result;
            }
            catch (IOException e) {
                this.reopenStreamOrFail(e);
                continue;
            }
            break;
        }
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        this.ensureOpen();
        while (true) {
            try {
                int bytesRead = this.currentStream.read(b, off, len);
                if (bytesRead == -1) {
                    this.eof = true;
                    return -1;
                }
                this.currentOffset += (long)bytesRead;
                return bytesRead;
            }
            catch (IOException e) {
                this.reopenStreamOrFail(e);
                continue;
            }
            break;
        }
    }

    private void ensureOpen() {
        if (this.closed) {
            assert (false) : "using S3RetryingInputStream after close";
            throw new IllegalStateException("using S3RetryingInputStream after close");
        }
    }

    private void reopenStreamOrFail(IOException e) throws IOException {
        if (this.attempt >= this.maxAttempts) {
            logger.debug((Message)new ParameterizedMessage("failed reading [{}/{}] at offset [{}], attempt [{}] of [{}], giving up", new Object[]{this.blobStore.bucket(), this.blobKey, this.start + this.currentOffset, this.attempt, this.maxAttempts}), (Throwable)e);
            throw this.addSuppressedExceptions(e);
        }
        logger.debug((Message)new ParameterizedMessage("failed reading [{}/{}] at offset [{}], attempt [{}] of [{}], retrying", new Object[]{this.blobStore.bucket(), this.blobKey, this.start + this.currentOffset, this.attempt, this.maxAttempts}), (Throwable)e);
        ++this.attempt;
        if (this.failures.size() < 10) {
            this.failures.add(e);
        }
        this.maybeAbort(this.currentStream);
        IOUtils.closeWhileHandlingException((Closeable)this.currentStream);
        this.openStream();
    }

    @Override
    public void close() throws IOException {
        this.maybeAbort(this.currentStream);
        try {
            this.currentStream.close();
        }
        finally {
            this.closed = true;
        }
    }

    private void maybeAbort(S3ObjectInputStream stream) {
        if (this.isEof()) {
            return;
        }
        try {
            if (this.start + this.currentOffset < this.currentStreamLastOffset) {
                stream.abort();
            }
        }
        catch (Exception e) {
            logger.warn("Failed to abort stream before closing", (Throwable)e);
        }
    }

    @Override
    public long skip(long n) {
        throw new UnsupportedOperationException("S3RetryingInputStream does not support seeking");
    }

    @Override
    public void reset() {
        throw new UnsupportedOperationException("S3RetryingInputStream does not support seeking");
    }

    private <T extends Exception> T addSuppressedExceptions(T e) {
        for (IOException failure : this.failures) {
            e.addSuppressed(failure);
        }
        return e;
    }

    boolean isEof() {
        return this.eof || this.start + this.currentOffset == this.currentStreamLastOffset;
    }

    boolean isAborted() {
        if (this.currentStream == null || this.currentStream.getHttpRequest() == null) {
            return false;
        }
        return this.currentStream.getHttpRequest().isAborted();
    }
}

