"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenSearchClientPool = void 0;

var _lruCache = _interopRequireDefault(require("lru-cache"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * OpenSearch client pool for data source.
 *
 * This client pool uses an LRU cache to manage OpenSearch Js client objects.
 * It reuse TPC connections for each OpenSearch endpoint.
 */
class OpenSearchClientPool {
  // LRU cache
  //   key: data source endpoint
  //   value: OpenSearch client object | Legacy client object
  constructor(logger) {
    this.logger = logger;

    _defineProperty(this, "cache", void 0);

    _defineProperty(this, "isClosed", false);
  }

  async setup(config) {
    const logger = this.logger;
    const {
      size
    } = config.clientPool;
    this.cache = new _lruCache.default({
      max: size,
      maxAge: 15 * 60 * 1000,

      // by default, TCP connection times out in 15 minutes
      async dispose(endpoint, client) {
        try {
          await client.close();
        } catch (error) {
          // log and do nothing since we are anyways evicting the client object from cache
          logger.warn(`Error closing OpenSearch client when removing from client pool: ${error.message}`);
        }
      }

    });
    this.logger.info(`Created data source client pool of size ${size}`);

    const getClientFromPool = endpoint => {
      return this.cache.get(endpoint);
    };

    const addClientToPool = (endpoint, client) => {
      this.cache.set(endpoint, client);
    };

    return {
      getClientFromPool,
      addClientToPool
    };
  }

  start() {} // close all clients in the pool


  async stop() {
    if (this.isClosed) {
      return;
    }

    await Promise.all(this.cache.values().map(client => client.close()));
    this.isClosed = true;
  }

}

exports.OpenSearchClientPool = OpenSearchClientPool;