package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.36.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.36.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````7!@``#0```"\D0````0`!30`(``)
M`"@`'``;``$``'"LC```K(P``*R,``!H````:`````0````$````!@```#0`
M```T````-````"`!```@`0``!`````0````#````5`$``%0!``!4`0``&0``
M`!D````$`````0````$````````````````````8C0``&(T```4```````$`
M`0```+"-``"PC0$`L(T!`)@"``"L`@``!@```````0`"````N(T``+B-`0"X
MC0$`&`$``!@!```&````!`````0```!P`0``<`$``'`!``!$````1`````0`
M```$````4>5T9```````````````````````````!@```!````!2Y71DL(T`
M`+"-`0"PC0$`4`(``%`"```$`````0```"]L:6(O;&0M;&EN=7@M87)M:&8N
M<V\N,P`````$````%`````,```!'3E4`=Q`L'NEYMB>N6/?=<&@U:U0JJ#`$
M````$`````$```!'3E4```````,````"`````````"X```!-````"`````@`
M```!D!`$)4`$@"B```6@AE"$@`$```LH!(PX@!)!B@``"$T`````````3P``
M`%$`````````4@````````!3````5`````````!5````5@```%<```!8````
M`````%H``````````````%L`````````7`````````````````````````!=
M````7@```%\```!@`````````&$``````````````&,`````````9````&4`
M``````````````````!F````9P````````!H```````````````NQ9NV:W^:
M?`PZEPL5C"7,0_K7AZ/@^TE#1=7LB0=PT]EQ6!R_VE<;`:\TZ'ZIG)O/WD<M
MP^4_+-5DU'PS8MOMX:<""EGOEWEOQ=\?:PE#UG94/)0A_^0.N^.2?*U+X\"S
M`8MO:QJ>4.-'Q@%FK\QF^Z%L8```````````````````````````N!(`````
M```#``L```````"0`0```````P`7`-8"`````````````!(````_`0``````
M```````2````70(`````````````$@```(\"`````````````!(```!$`@``
M```````````2````@```````````````$@```+```````````````!(````*
M`P`````````````2````?`$`````````````$@````$``````````````"``
M``"Q`0`````````````2````IP``````````````$@```%,`````````````
M`!(````#`@`````````````2````NP``````````````$@```'\"````````
M`````!(````0```````````````@````'P$`````````````$@```!("````
M`````````!(```!$`0`````````````2````K`$`````````````$@```,\"
M`````````````!(````L```````````````@````B`(`````````````$@``
M`(\!`````````````!(````,`0`````````````6````EP``````````````
M$@```&\``````````````!(```!K`0`````````````2````V```````````
M````$@```#8"`````````````!(```#U`0`````````````2````>`(`````
M````````$@```.\#`````````````!(```#)```````````````2````W@(`
M````````````$0```.X!`````````````!(```#X`@`````````````1````
M*0(`````````````$@```.<``````````````!(````@`@`````````````2
M````]0``````````````$@```#T"`````````````!(```!C````````````
M```2````EP(`````````````$@```%4!`````````````!(```"^`@``````
M```````2````V0$`````````````$@```/P!`````````````!(````P`@``
M```````````2````AP(`````````````$@```+\!`````````````!(```#F
M`@`````````````2````W0(`````````````(0```"H!`````````````!(`
M``"X`0`````````````2````20$`````````````$@```+4"````````````
M`!(```#<`@`````````````1````<0(`````````````$0```$8`````````
M`````!$```!3`@`````````````2````8P(`````````````$@````H"````
M`````````!(```",```````````````2````2P(`````````````$@```%P!
M`````````````!(```#'`0`````````````2````(0,`````````````$@``
M`!D"`````````````!(```#G`0`````````````2````L`(`````````````
M$@```)\"`````````````!(```"=`0`````````````B````;P,``!0D```$
M````$@`-`-0!``"H%0``M`(``!(`#0`5!````)`!```````@`!<`800``!Q'
M``!4````$@`-`"X$``!P1P``&`$``!(`#0"L!```B"@``$0%```2``T`?`0`
M`$B0`0``````$``7`*($``!$D`$`!````!$`%P!5!```2)`!```````0`!@`
M200``%R0`0``````$``8`$H$``!<D`$``````!``&`#=`P``;"X``"0````2
M``T`;P0```0P``#,!@``$@`-`#D$``"L(@``5````!(`#0"-`P``T#8``-0"
M```2``T`&00``%P8````````$@`-`+$#````(P``%`$``!(`#0!!!```7)`!
M```````0`!@`^`,``*0Y``!``0``$@`-`!,$````D`$``````!``%P`@!```
M2)`!```````0`!@`E00``!@D``"0`@``$@`-`&H$``!<D`$``````!``&``$
M!```$$H```0````1``\`HP,``,PM``"@````$@`-`-,#``"(2```;`$``!(`
M#0"``P``J"8``#`````2``T`QP,``-@F``"P`0``$@`-`(,$``"0+@``=`$`
M`!(`#0``7U]G;6]N7W-T87)T7U\`7TE435]D97)E9VES=&5R5$U#;&]N951A
M8FQE`%])5$U?<F5G:7-T97)434-L;VYE5&%B;&4`4$Q?9&]?=6YD=6UP`%!E
M<FQ?9W9?9F5T8VAP=@!097)L7VUG7W-E=`!097)L7V%T9F]R:U]L;V-K`%!E
M<FQ?9V5T7W-V`%!E<FQ?;F5W6%,`8F]O=%]$>6YA3&]A9&5R`'!E<FQ?<G5N
M`'!E<FQ?<&%R<V4`4&5R;%]S=E]S971P=@!097)L7W-V7W-E='!V;@!P97)L
M7V-O;G-T<G5C=`!P97)L7V1E<W1R=6-T`%!E<FQ?8W)O86M?;65M;W)Y7W=R
M87``4$Q?8W5R<F5N=%]C;VYT97AT`'!E<FQ?86QL;V,`4&5R;%]C<F]A:U]N
M;V-O;G1E>'0`<&5R;%]F<F5E`%!E<FQ?<V%F97-Y<VUA;&QO8P!097)L7W-Y
M<U]I;FET,P!097)L7VYE=UA37V9L86=S`%!E<FQ?871F;W)K7W5N;&]C:P!0
M97)L7W-Y<U]T97)M`%]?8WAA7V9I;F%L:7IE`')E860`<W1R8VAR`&=E='!I
M9`!M96UM;W9E`%]?;&EB8U]S=&%R=%]M86EN`%]?9G!R:6YT9E]C:&L`<W1R
M8VUP`&]P96XV-`!U;FQI;FL`<W1R=&]K`&UE;7-E=`!S=')N8VUP`&=E='5I
M9`!M96UC:'(`8VQO<V5D:7(`<W1R<W1R`')M9&ER`'-T<FQE;@!S=')D=7``
M;65M8VUP`')E86QL;V,`<F5A9&1I<C8T`&%B;W)T`%]?<W!R:6YT9E]C:&L`
M<W1D97)R`&UE;6-P>0!S=')R8VAR`&QS=&%T-C0`;W!E;F1I<@!L<V5E:S8T
M`%]?97)R;F]?;&]C871I;VX`97AI=`!R96%D;&EN:P!?7W-T86-K7V-H:U]F
M86EL`&%C8V5S<P!M:V1I<@!?7V5N=FER;VX`7U]R96=I<W1E<E]A=&9O<FL`
M7U]S=&%C:U]C:&M?9W5A<F0`7U]A96%B:5]U;G=I;F1?8W!P7W!R,`!?7V%E
M86)I7W5N=VEN9%]C<'!?<'(Q`&QI8G!E<FPN<V\`;&EB8RYS;RXV`&QD+6QI
M;G5X+6%R;6AF+G-O+C,`;&EB9V-C7W,N<V\N,0!P87)?8W5R<F5N=%]E>&5C
M`'!A<E]B87-E;F%M90!84U]);G1E<FYA;'-?4$%27T)/3U0`<&%R7V5N=E]C
M;&5A;@!P87)?8W5R<F5N=%]E>&5C7W!R;V,`<&%R7V1I<FYA;64`<VAA7V9I
M;F%L`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`'!A<E]C;&5A;G5P`%])
M3U]S=&1I;E]U<V5D`%]?9&%T85]S=&%R=`!?7V)S<U]S=&%R=%]?`'-H85]U
M<&1A=&4`>'-?:6YI=`!?7V5N9%]?`%]?8G-S7V5N9%]?`%]?8G-S7W-T87)T
M`'-H85]I;FET`%]E;F0`<&%R7VUK=&UP9&ER`%]E9&%T80!P87)?<V5T=7!?
M;&EB<&%T:`!P87)?9FEN9'!R;V<`;7E?<&%R7W!L`'!A<E]I;FET7V5N=@!'
M3$E"0U\R+C0`1T-#7S,N-0!'3$E"0U\R+C,T`$=,24)#7S(N,S,`+W5S<B]L
M:6(O<&5R;#4O-2XS-B]C;W)E7W!E<FPO0T]210`````````"``$``@`"``(`
M`0`!``,``0`!``(``0`!``(``0`"``$``0`"``(``@`"``$`!``!``$``0`!
M``$``0`"``(``@`"``$``@`"``4``@`!``(``0`"``$``@`"``(``@`"``(`
M!``"``(``@`!``(``0`"``(``@`!``(``@`"``$``@`!``8``P`"``(``@`"
M``(``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M`0`!``$``0`!``$``0`!``$``0!-`P``$````"`````4:6D-```%`+D$````
M`````0`!`&$#```0````(````%4F>0L```,`PP0````````!``,`0P,``!``
M````````M)&6!@``!@#+!```$````+.1E@8```0`U@0``!`````4:6D-```"
M`+D$````````L(T!`!<```"TC0$`%P```/R/`0`7````!)`!`!<````4D`$`
M%P```!B0`0`7````()`!`!<````DD`$`%P```"B0`0`7````+)`!`!<````P
MD`$`%P```#B0`0`7````/)`!`!<```!$D`$`%P```,R/`0`5"P``T(\!`!4,
M``#4CP$`%1,``-B/`0`5&0``W(\!`!,<``#@CP$`%1T``.2/`0`5'@``Z(\!
M`!4F``#LCP$`%2@``/"/`0`5/@``](\!`!4_``#XCP$`%4P``-R.`0`6`P``
MX(X!`!8$``#DC@$`%@4``.B.`0`6!@``[(X!`!8'``#PC@$`%@@``/2.`0`6
M"0``^(X!`!8,``#\C@$`%@T```"/`0`6#@``!(\!`!8/```(CP$`%A````R/
M`0`6$0``$(\!`!82```4CP$`%A0``!B/`0`6%0``'(\!`!86```@CP$`%A<`
M`"2/`0`6&```*(\!`!8:```LCP$`%AL``#"/`0`6'P``-(\!`!8@```XCP$`
M%B$``#R/`0`6(@``0(\!`!8C``!$CP$`%B0``$B/`0`6)0``3(\!`!8G``!0
MCP$`%BD``%2/`0`6*@``6(\!`!8K``!<CP$`%BP``&"/`0`6+0``9(\!`!8N
M``!HCP$`%B\``&R/`0`6,```<(\!`!8Q``!TCP$`%C(``'B/`0`6,P``?(\!
M`!8T``"`CP$`%C4``(2/`0`6-@``B(\!`!8W``",CP$`%CD``)"/`0`6.@``
ME(\!`!8[``"8CP$`%CP``)R/`0`60```H(\!`!9!``"DCP$`%D(``*B/`0`6
M0P``K(\!`!9$``"PCP$`%D4``+2/`0`61@``N(\!`!9(``"\CP$`%DD``,"/
M`0`62@``Q(\!`!9+``#(CP$`%DP```A`+>EW`0#K"("]Z`3@+>4$X)_E#N"/
MX`CPON7\>P$``,:/XA?*C.+\^[SE`,:/XA?*C.+T^[SE`,:/XA?*C.+L^[SE
M`,:/XA?*C.+D^[SE`,:/XA?*C.+<^[SE`,:/XA?*C.+4^[SE`,:/XA?*C.+,
M^[SE`,:/XA?*C.+$^[SE`,:/XA?*C.*\^[SE`,:/XA?*C.*T^[SE`,:/XA?*
MC.*L^[SE`,:/XA?*C.*D^[SE`,:/XA?*C.*<^[SE`,:/XA?*C.*4^[SE`,:/
MXA?*C.*,^[SE`,:/XA?*C.*$^[SE`,:/XA?*C.)\^[SE`,:/XA?*C.)T^[SE
M`,:/XA?*C.)L^[SE`,:/XA?*C.)D^[SE`,:/XA?*C.)<^[SE`,:/XA?*C.)4
M^[SE`,:/XA?*C.),^[SE`,:/XA?*C.)$^[SE`,:/XA?*C.(\^[SE`,:/XA?*
MC.(T^[SE`,:/XA?*C.(L^[SE`,:/XA?*C.(D^[SE`,:/XA?*C.(<^[SE`,:/
MXA?*C.(4^[SE`,:/XA?*C.(,^[SE`,:/XA?*C.($^[SE`,:/XA?*C.+\^KSE
M`,:/XA?*C.+T^KSE`,:/XA?*C.+L^KSE`,:/XA?*C.+D^KSE`,:/XA?*C.+<
M^KSE`,:/XA?*C.+4^KSE`,:/XA?*C.+,^KSE`,:/XA?*C.+$^KSE`,:/XA?*
MC.*\^KSE`,:/XA?*C.*T^KSE`,:/XA?*C.*L^KSE`,:/XA?*C.*D^KSE`,:/
MXA?*C.*<^KSE`,:/XA?*C.*4^KSE`,:/XA?*C.*,^KSE`,:/XA?*C.*$^KSE
M`,:/XA?*C.)\^KSE`,:/XA?*C.)T^KSE`,:/XA?*C.)L^KSE`,:/XA?*C.)D
M^KSE`,:/XA?*C.)<^KSE`,:/XA?*C.)4^KSE`,:/XA?*C.),^KSE`,:/XA?*
MC.)$^KSE`,:/XA?*C.(\^KSE`,:/XA?*C.(T^KSE`,:/XA?*C.(L^KSE`,:/
MXA?*C.(D^KSE<$`MZ0'*H.,,P$W@R`^,Y4S"G^4HT$WB2#*?Y0S`C^!$0I_E
M'`"-Y1P`C>(8$(WE&!"-XA0@C>44((WB!$"/X`,PG.<`,)/E)#"-Y0`PH./5
M___K%"*?Y10RG^4"()3G#""-Y0(0H.$#,)3G$#"-Y0,`H.'T#`#K^#&?Y0,P
ME.<`0-/E``!4XU$```IP+QWNY#&?Y1Q0G>4$`(7B`U"%X@,PG^<'`7#C`R"2
MYYXSTN4#,(/CGC/"Y6,``(H``:#AI/__Z[0AG^480)WE`#"@X0(@C^"H$9_E
M`1"/X`0`@N4`()3E`""#Y9@AG^4<`)WE`B"/X`P@@^6,(9_E`0!0XP`0D>4"
M((_@"!"#Y00@@^5.``#:!!"$X@S`@^(`083@!."1Y`3@K.4!`%3A^___&@,`
M@.)489_E`!"@XP`0C>4%(*#A!F"/X``1@^=`$9_E#`"6Y0$0C^`&___K`$!0
MXB@```HL49_E(&"-X@50C^`,`)7E1___ZQP!G^4&$*#A``"/X*<``.L``%#C
M(0``"@@QG^4#,(_@#`"3Y>;^_^L>___K^""?Y;@PG^4"((_@`S"2YP`@D^4D
M,)WE`B`SX``PH.,E```:!`"@X2C0C>)P@+WH__[_Z\@@G^4``%#C`B"/X`P`
M@N4:```*$?__Z[0PG^5P+QWN`S"?YP,PDN='1\/EH?__Z@P`EN7B_O_K`$"@
MX=+__^H(`)7E``!0XP$```IT"`#KV/__ZGP`G^4&$*#A``"/X'H``.L``%#C
M"`"%Y='__PKU___J%___ZP0`H..V___J`0"@XU[__^LA___K!'D!`!P!``#D
M>`$`_````!0!```D`0``C'D!`,QY`0"\>0$``#<``.@V``!8>0$`_`H``#!Y
M`0`X-```"'D!`'1W`0"\>`$`-'@!`"`S````L*#C`."@XP00G>0-(*#A!"`M
MY00`+>4<H)_E&#"/X@.@BN``,*#C!#`MY0P`G^4``)KG,O__ZY7^_^LX=@$`
M+`$``!0PG^44()_E`S"/X`(@D^<``%+C'O\O`9O^_^H@=@$```$``"P`G^4L
M,)_E``"/X"@@G^4#,(_@``!3X0(@C^`>_R\!&#"?Y0,PDN<``%/C'O\O`1/_
M+^%T=P$`;'<!`.QU`0`$`0``.`"?Y3@PG^4``(_@-""?Y0,PC^``,$/@`B"/
MX*,?H.%#$8'@P1"PX1[_+P$8,)_E`S"2YP``4^,>_R\!$_\OX3!W`0`H=P$`
MJ'4!``@!``!8,)_E6""?Y0,PC^`"((_@`##3Y0``4^,>_R\1$$`MZ0'*H.,,
MP$W@^`^,Y30PG^4#,)+G``!3XP(```HH,)_E`P"?Y_[^_^O'___K'#"?Y0$@
MH.,#,(_@`"##Y1"`O>C@=@$`9'4!`"@!``!D=@$`E'8!`,W__^KP1RWI`<J@
MXPS`3>#@#XSER#"?Y0"`4.(#,(_@*P``"KP@G^4",)/G`)"3Y0``6>,H```*
M`%#8Y0%PH.$``%7C/0!5$P%0H!,`4*`#$0``&@!`F>4``%3C#```"@E@H.$$
M`*#A!2"@X0@0H.&[_O_K``!0XP(``!H%,-3G/0!3XPL```H$0+;E``!4X_/_
M_QH$`*#A\(>]Z`A0H.$!,/7E``!3XST`4Q/[__\:"%!%X.;__^H)8$;@`5"%
MX@5`A.!&8:#A!`"@X0!@A^7PA[WH"$"@X>W__^H)0*#AZ___ZN!T`0`8`0``
M$$`MZ0'*H.,,P$W@B`^,Y8P@G^5PT$WBB#"?Y0T0H.$"((_@`$"@X0,PDN<`
M,)/E;#"-Y0`PH.-S_O_K``!0XP``H!,$```:$#"=Y0\Z`^(*"E/C`0E3$PH`
M``I(()_E0#"?Y0(@C^`#,)+G`""3Y6PPG>4"(#/@`#"@XP<``!IPT(WB$("]
MZ`(0H.,$`*#A&/[_ZQ`/;^&@`J#A[O__ZDW^_^OP<P$`'`$``*QS`0#P0"WI
M`<J@XPS`3>!P#XSE7"&?Y7S03>)8,9_E`%!0X@(@C^`#,)+G`#"3Y70PC>4`
M,*#C`@``"@`PU>4``%/C"@``&C`AG^4H,9_E`B"/X`,PDN<`()/E=#"=Y0(@
M,^``,*#C00``&GS0C>+P@+WHQ_W_ZP!`4.+Q__\*!`"@X4K^_^L``%#C*@``
M"A,PT.438(#B+@!3XP(``!H!,-;E``!3X_3__PH3,-#E+@!3XR4```H%`*#A
M\?W_ZP!PH.$&`*#A[OW_ZP``A^`"`(#B$O[_ZZ0PG^4`<*#A`1"@XP,PC^``
M(.#C!&"-Y0!0C>4Q_O_K"!"-X@<`H.'4_?_K`0!PXP<`H.$#```*&#"=Y0\Z
M`^(!"5/C$@``"MO]_^L'`*#AP?W_ZP0`H.$?_O_K``!0X]3__QH$`*#AZ/W_
MZP4`H.$!_O_KO?__Z@$PUN4N`%/CUO__&@(PUN4``%/CQ?__"M+__^JD___K
MZ___ZNW]_^M`<P$`'`$``!1S`0!4+@``\$\MZ0#`H.,!Z*#C#,!-X`[@3.`!
MRDSB``",Y0X`7.'[__\:0``.Y0'83>(<T$WB4,&?Y0(0H.,!Z(WB#,"/X!3@
MCN(`$(WE`""@XS@1G^4`,*#C`)"@X0$0G.<`$)'E`!".Y0`0H./(_?_K`0!0
MX@"`P>+(+Z#A`C`@X`(P4^!S,/_F`C`CX`(P4^`"(,+@`V!0X`*`R.```%CC
M-0``NNAPG^4(H*#C`:!`XP"PH./_OT_C!W"/X`Q0C>((,*#A`!"@XP8@H.$)
M`*#A`!"-Y:[]_^L*(*#A!1"@X0D`H.%T_?_K!P!0XQT``)H(`$#B`$"5X`,`
M`#H9``#J`4!$X@4`5.$6```Z"""@XP<0H.$$`*#A0/W_ZP``4./V__\:!4!$
MX`8`E.#$'ZC@8""?Y0'(C>)0,)_E%,",X@(@C^`#,)+G`""3Y0`PG.4"(#/@
M`#"@XPH``!H!V(WB'-"-XO"/O>@!:%;B`(#(X@$`>.,+`%8!T/__&@``X.,`
M$*#AZ?__ZH?]_^N@<0$`'`$``&!R`0"@<`$`\$$MZ0'*H.,,P$W@``",Y2``
M#.4!<*#A`!&?Y0`QG^4!VDWB`1"/X`C03>(`8*#A`0J-X@0`@.($@(WB`S"1
MY_\O`.,($*#A`#"3Y0`P@.4`,*#CS`"?Y0``C^")_?_K`$!0X@<``-H(4(WB
M`3!$X@/`A>`$(%SE``!2XQ<``!H``%/C%```&IPPG^5P#QWN!R"@X080H.$#
M,)_G`P"0YP_]_^N$()_E`1J-XG`PG^4$$('B`B"/X`,PDN<`()/E`#"1Y0(@
M,^``,*#C$@``&@':C>((T(WB\(&]Z`-`H.$$(*#A+Q"@XP@`H.%]_?_K``!0
MX^/__PHT`)_E<,\=[@0PH.$((*#A!A"@X0``G^<``)SG'_W_Z^'__^H[_?_K
M('`!`!P!``#(*P``N'`!`)!O`0!$<`$`\$<MZ0'*H.,,P$W@V`^,Y10BG^4(
MT$WB$#*?Y0%`H.$"((_@`%"@X02"G^4#,)+G"("/X``PD^4$,(WE`#"@XP`P
MT>4]`%/C`4"!`@0`H.'V_/_K#1"@X0!@H.$%`*#A:_[_ZP"04.)````*[_S_
MZP``5N$W``":N#&?Y0!PG>4#@)CG!W&@X0`PU>4``%/C/0!3$P$PH!,`,*`#
M!0``"@4PH.$!(//E``!2XST`4A/[__\:!3!#X`(`AN(`8)CE`P"`X`#]_^L`
M`%#C!P"&YQ4```H`,)CE!0"@X0<0D^<!,-#D`##!Y0``4^,]`%,3!@``"@$@
M@>(!,-#D`A"@X0$PPN0]`%/C``!3$_G__QH!(*#A/3"@XP`PP>4!,-3D`3#B
MY0``4^/[__\:""&?Y?@PG^4"((_@`S"2YP`@D^4$,)WE`B`SX``PH.,U```:
M"-"-XO"'O>@!D$GB`3#4Y`$PZ>4``%/C^___&N[__^K`,)_E`X"8YP!PF.4`
M,)?E``!3XR0```H',*#A!""SY0D0H.$!D(GB``!2X_K__QH#$('B`1&@X9`P
MG^4#,(_@`""3Y0``4N,/```:`0"@X0$@H.,`((/EO_S_ZP"@4.+5__\*!Q"@
MX0EQH.$'(*#AR_S_ZP"@B.4`D(WE`9")X@`PH.,),8KGHO__Z@<`H.'A_/_K
M`*"@X0``4.,``(CEQ?__"@EQH.'R___J`Y"@X0@0H./?___JJOS_Z^QN`0`<
M`0``W&X!`!@!``#8;0$`W&X!`!!`+>D!RJ#C#,!-X.@/C.5P$)_E<$\=[FP@
MG^40T$WB:`"?Y0(@C^!D,)_E`1"?YP,PC^``()+G`0"4YU00G^4,((WE`1"/
MX+O\_^M($)_E2#"?Y0#`H.-$()_E`S"/X`$0G^<"((_@`0"4YS00G^4$P(WE
M`1"/X``PC>54_/_K$-"-XA"`O>BH;0$`I&P!`!`!``"X:0``<"@``'1M`0`T
M*P``9!0``%PH```X`)_E"-!-XC0@G^5PSQWN,!"?Y0(@C^`L,)_E``"?YP,P
MC^`!()+G(!"?Y0``G.<!$(_@!""-Y0C0C>*3_/_J#&T!``AL`0`0`0``'&D`
M`-@G``!P0"WI`<J@XPS`3>```(SE`<I,X@``C.4H``SEW""?Y0+:3>+8,)_E
M&-!-X@(@C^`"&HWB%!"!X@Q`C>(!:HWB`S"2YQ!@AN(`,)/E`#"!Y0`PH.-F
M_/_KJ!"?Y0$@`>.D,)_E`1"/X/``S>$#,(_@`1"@XP0`H.%I_/_K!`"@X0$J
MH.,&$*#A7_S_ZP!`4.(8``"Z`0"$XCK\_^L`4%#B%```"@0@H.$&$*#A%/S_
MZP`PH.,$,,7G4""?Y0(:C>(\,)_E%!"!X@(@C^`#,)+G`""3Y0`PD>4"(#/@
M`#"@XP,``!H%`*#A`MJ-XAC0C>)P@+WH)OS_ZP!0H./M___JG&L!`!P!``"0
M)P``?"<```!K`0"Y___J\$\MZ0'*H.,,P$W@``",Y:@`#.5$(I_E0MU-XD`R
MG^4$T$WB`B"/X`!@H.$T`I_E0<V-XA"`C>(\P(SB`S"2YP%`H.$``(_@"!"@
MX0`PD^4`,(SE`#"@XU;]_^LO$*#C`%"@X08`H.&L^__K``!0XQ(```KP`9_E
M!A"@X0``C^#'_O_K!@"@X>`AG^5!'8WBS#&?Y3P0@>("((_@`S"2YP`@D^4`
M,)'E`B`SX``PH.-I```:0MV-X@30C>+PC[WH!`"@X>#[_^ND$9_E`1"/X._[
M_^L`0%#BY?__"I21G^6`H(WB`##4Y01P2N()D(_@``!3XSL```H``%7C!```
M"@00H.$%`*#A%/S_ZP``4.,K```*!`"@X:W[_^L!,$#B`S"$X`"PH.$#`%3A
M`#"$,``0H#,!`(0R`P``.@@``.H#`%#A`!##Y0(```H!('/E+P!2X_G__PH$
M`*#AG/O_ZP"PH.$&`*#AF?O_ZPL`@.`!`(#B`0I0X[W__RKX`)_E"3"@X0$J
MH.,!$*#C``"/X`!`C>5!`(WI!P"@X=K[_^L($*#A!P"@X7W[_^L``%#C`P``
M&F`P&N4/.@/B`@E3XQ4```JT$)_E``"@XP$0C^"O^__K`$!0XJ7__PH`,-3E
M``!3X\/__QH``%7C%@``"@`PU>4N`%/C`P``"H!`G^4!L*#C!$"/X-7__^H!
M,-7E``!3X^K__PKW___J`1"@XP<`H.%;^__K``!0X^3__QI0`)_E!Q"@X0``
MC^!8_O_K!P"@X8#[_^N.___J.$"?Y0&PH.,$0(_@P?__ZH;[_^N,:@$`'`$`
M`*`F``!T)@``&&H!`#0F```8)@``@"4``#0E``#<)```N"0``(PD```00"WI
M`<J@XPS`3>#X#XSE+Q"@XP!`H.$K^__K``!0XP%`@!($`*#A$$"]Z&+[_^KP
M0"WI`<J@XPS`3>```(SE(``,Y7@AG^4!VDWB=#&?Y0S03>("((_@`1J-X@00
M@>(`4%#B`S"2YP`PD^4`,('E`#"@XS$```H`,-7E``!3XRX```HN^__K`2"`
MX@!`H.$!"E+C.@``B@1PC>(%$*#A!P"@X2S[_^L!,$3B`S"'X`,`5^$)```J
M!`"'X`4@C>(`$*#C`@``Z@(`4.$`$,#E`@``"@$P<.4O`%/C^?__"B\0H.,'
M`*#A]_K_ZP``4.,E```*!P!0X0`PH(,`,,"%!P"@X2S[_^N\()_E`1J-XK`P
MG^4$$('B`B"/X`,PDN<`()/E`#"1Y0(@,^``,*#C(@``&@':C>(,T(WB\("]
MZ(@@G^4!&HWB>#"?Y000@>("((_@`S"2YP`@D^4`,)'E`B`SX``PH.,4```:
M8`"?Y0``C^`!VHWB#-"-XO!`O>@-^__J``"@X][__^I$()_E`1J-XBPPG^4$
M$('B`B"/X`,PDN<`()/E`#"1Y0(@,^``,*#C(`"?!0``CP#L__\*"?O_Z\QG
M`0`<`0``"&<!`-!F`0#@(@``A&8!`)@B``!P0"WI`<J@XPS`3>#H#XSEQ"2?
MY0C03>+`-)_E#4"@X0(@C^"X9)_EN%2?Y09@C^`#,)+G!5"/X``PD^4$,(WE
M`#"@XP00H.$&`*#A/?S_ZP``4.,.```*C#2?Y0``G>4#,)7G`!"3Y0`Q@>`$
M()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J5&2?Y09@
MC^`$$*#A!@"@X2?\_^L``%#C#@``"C0TG^4``)WE`S"5YP`0D^4`,8'@!""3
MY0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___Z@!DG^4&8(_@
M!!"@X08`H.$1_/_K``!0XPX```K<,Y_E``"=Y0,PE><`$)/E`#&!X`0@D^4`
M(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^JL8Y_E!F"/X`00
MH.$&`*#A^_O_ZP``4.,.```*A#.?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E`"&!
MYP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J6&.?Y09@C^`$$*#A
M!@"@X>7[_^L``%#C#@``"BPSG^4``)WE`S"5YP`0D^4`,8'@!""3Y0`A@><`
M`%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___Z@1CG^4&8(_@!!"@X08`
MH.'/^__K``!0XPX```K4,I_E``"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2
MX_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^JP8I_E!F"/X`00H.$&`*#A
MN?O_ZP``4.,.```*?#*?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q
M__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J7&*?Y09@C^`$$*#A!@"@X:/[
M_^L``%#C#@``"B0RG^4``)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__
M"@0P@^($()/E!""#Y```4N/[__\:Z___Z@ABG^4&8(_@!!"@X08`H.&-^__K
M``!0XPX```K,,9_E``"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$
M,(/B!""3Y00@@^0``%+C^___&NO__^JT89_E!F"/X`00H.$&`*#A=_O_ZP``
M4.,.```*=#&?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#
MX@0@D^4$((/D``!2X_O__QKK___J8&&?Y09@C^`$$*#A!@"@X6'[_^L``%#C
M#@``"APQG^4``)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($
M()/E!""#Y```4N/[__\:Z___Z@P!G^4$$*#A``"/X$S[_^L`$%#B`@``"O@`
MG^4``(_@POS_Z_``G^4$$*#A``"/X$/[_^L`$%#B`@``"MP`G^4``(_@N?S_
MZ]0`G^4$$*#A``"/X#K[_^L`$%#B$@``"L``G^4``(_@L/S_Z[@@G^5D,)_E
M`B"/X`,PDN<`()/E!#"=Y0(@,^``,*#C$```&I@0G^68`)_E`1"/X```C^`(
MT(WB<$"]Z*#\_^J$`)_E!!"@X0``C^`A^__K`!!0XNG__PIP`)_E``"/X)?\
M_^OE___JR_G_ZR!F`0`<`0``="(```QF`0`8`0``%"(``,0A``!X(0``*"$`
M`.`@```D(```/"```/`?``"D'P``T!X``/P>``#0'@``[!X``.P>``#H'@``
M#!X``,AA`0#$'@``*!X``)0>```8'@``!.`MY0'*H.,,P$W@\`^,Y7@@G^4,
MT$WB=#"?Y0T0H.$"((_@;`"?Y0``C^`#,)+G`#"3Y00PC>4`,*#C\/K_ZP``
M4.,%```*``#0Y0``4.,P`%`3`0"@`P``H!,!`"#B-""?Y2@PG^4"((_@`S"2
MYP`@D^4$,)WE`B`SX``PH.,!```:#-"-X@3PG>2-^?_KW&`!`!P!``!X'0``
ME&`!`!!`+>D!RJ#C#,!-X/@/C.5"^?_K8OG_ZP``4.,``)`5$("]Z/!`+>D!
MRJ#C#,!-X-`/C.4X(9_E'-!-XC0QG^40$(WB`B"/X"QAG^4`4*#A!F"/X`,P
MDN<&`*#A`#"3Y10PC>4`,*#CO?K_ZP!`4.("```*`##4Y0``4^,.```:^""?
MY>PPG^4"((_@`S"2YP`@D^44,)WE`B`SX``PH.,R```:V`"?Y040H.$``(_@
M'-"-XO!`O>@D_/_J!1"@X4#Y_^L``%#C"@``"K0@G^6@,)_E`B"/X`,PDN<`
M()/E%#"=Y0(@,^``,*#C'P``&AS0C>+P@+WH!0"@X1_Y_^L`<*#A!`"@X1SY
M_^L``(?@`@"`XD#Y_^ML,)_E;""?Y0$0H.,#,(_@"$"-Y0(@C^``4(WE!""-
MY0`@X.,`0*#A7/G_ZT@@G^4H,)_E`B"/X`,PDN<`()/E%#"=Y0(@,^``,*#C
M!!"@`08`H`'/__\*+/G_ZQA@`0`<`0``,!T``-1?`0#4'```B%\!`(P;``"`
M&P``$%\!`/!/+>D!RJ#C#,!-X.@.C.4T)I_E]-!-XC`VG^6(8(WB`B"/X"C&
MG^4`H*#AH$"-X@S`C^`$X*#A`S"2YQAPC.*44(WB#+:?Y0`PD^7L,(WE`#"@
MXPPPC.(PP(SB(("-X@NPC^`'`)/H!P"&Z`\`M^@/`*[H`P"7Z`,`CN@'`)SH
M!P"%Z-`%G^4($*#A``"/X$WZ_^L`<%#B`@``"@`PU^4``%/C%P$`&K;X_^O6
M^/_K``!0XWH```H`<)#E``!7XW<```H'`*#AQOC_ZX``H.$!`(#BZOC_ZP`P
MU^4`4*#A'`"-Y0``4^,+```*;)6?Y0F0C^`%`*#A`#"-Y0$0H.,),*#A`"#@
MXP)0A>(#^?_K`3#WY0``4^/U__\:0`6?Y0``C^`"``#J!`"TY0``4.,-```*
M"!"@X2/Z_^L`4%#B^/__"@`PU>4``%/C]?__"E?Z_^L``%#C\O__"@4`H.&^
M^/_K`$!0X@T``!H$0);D``!4XPH```H`,-3E``!3X\\```H$`*#A2?K_ZP``
M4./U__\*!`"@X;#X_^L`0%#B\?__"@0`H.&.^/_K'%"=Y0!@H.$%`*#ABOC_
MZP!@AN`!:X;B!&"&X@8`H.&L^/_KD#2?Y0!PH.$!$*#C`S"/X`@PC>6`-)_E
M`"#@XPQ0C>4#,(_@!#"-Y7`TG^4`0(WE`S"/X,7X_^L'':#C!P"@X2_X_^L!
M`'#C`P``&MKX_^L`$)#E$0!1XZL``!H($*#A!P"@X:'X_^L`4%#B+0``"BP4
MG^4`,)KE*"2?Y0$`F^<"((_@`'"-Y0$0H.,``)#EC?C_ZP!@H.,,))_EW#.?
MY0(@C^`#,)+G`""3Y>PPG>4"(#/@`#"@X^X``!H&`*#A]-"-XO"/O>AP^/_K
M`'!0XHD``!H$`)7D``!0XPD```H($*#AQ/G_ZP`P4.+X__\*`##3Y0``4^/R
M__\:!`"5Y```4./U__\:H'.?Y0T`H.-G^/_K4S"@XP!0H.$'<(_@'`"-Y7S_
M_^HP,)WE#SH#X@$)4^/-__\:.$"=Y1WX_^L``%3AR?__&C`PG>53,.CG!PU3
MX\7__QH&`*#A5/C_ZP!@H.%(`Y_E"!"@X0``C^"A^?_K`!!0XGP```H``)KE
M+_S_ZP"04.)X```*"0"@X2#X_^L`0*#A`P!0XPP``)H0$Y_E!`!`X@``B>`!
M$(_@>_C_ZP``4.,%```:!`!4XYT```H$0(G@!3!4Y2\`4^.9```*X`*?Y0@0
MH.$``(_@A?G_ZP``4.,#```*`##0Y3``4^,``%,3!```&@`0H.,)`*#A$OC_
MZP!`4.)9```:J!*?Y:@"G^4!$(_@I**?Y0``C^"@DI_E[OK_ZYR"G^4Z^/_K
MF#*?Y0@`C>4!$*#C`S"/X`PPC>6(,I_E!@"@X0!PC>4`(.#C`S"/X`0PC>5T
M,I_E"J"/X`F0C^`(@(_@`S"/X#7X_^L.``#J3OC_ZP`PD.41`%/C$```&B/X
M_^L!4(7B`,"@X0HPH.$$(*#A`1"@XP8`H.$0D(WE#%"-Y8`1C>@E^/_K!QV@
MXP8`H.&/]__K`$"@X0$`<./J__\*!P"@X;KW_^L``I_E!A"@X0``C^"_^O_K
M!@"@X7+^_^MH___J`T"@X37__^K%]__K@`"@X0$`@.+]_O_JD"&?Y0`PFN4"
M`)OG!!"-Y0$0H.,`<(WEN"&?Y0``D.4"((_@Y??_ZU;__^I?_O_KI"&?Y3@Q
MG^4"((_@`S"2YP`@D^7L,)WE`B`SX``PH.-%```:!P"@X?30C>+P3[WHQ_?_
MZ@"0FN6$___JYOG_ZP"`H.$`D%'BH?__N@H@4.(`H*#C`##)X@0`H.&XL(WB
M`*"-Y<#W_^L&(*#C"Q"@X00`H.&&]__KN""=Y4,\H.-!,T3C`P!2X9'__QJ\
M*]WA2#4$XP,`4N&-__\:,B!8X@0`H.$`,,GB`*"-Y:[W_^L$`*#AP$"-XB@@
MH.,$$*#A<_?_Z^0PG^4!$*#C!@"@X0,PC^`,,(WEU#"?Y0`@X.,`<(WE`S"/
MX!@`C>G$,)_EZ*#-Y0,PC^#%]__KI/__Z@1`FN4``%3C8O__"@0`H.%Q]__K
M`P!0XU[__YJ8$)_E!`!`X@``A.`!$(_@S??_ZP``4.,$D*`!5O__ZI#W_^ND
M7@$`'`$``,Q?`0!D7@$`;!H``!`;``"4&@``.!H``#`9```(&@``(`$```@:
M``!47`$`#!D``'`9```X&0``O!<``/P7``!X%P``L!@``%09``#D%@``?!D`
M`/@6``"4&```2!8``"`7``"P60$`L!<``#`5``#$%@``B!8``/!`+>D!RJ#C
M#,!-X.`/C.5<$I_E#-!-XE@BG^5P3QWN`1"/X%`RG^50`I_E#6"@X0(@D><`
M`(_@!A"@X0`@DN4$((WE`""@XP,PG^<#4)3G9##5Y64PQ>6H^/_K``!0XP(`
M``H`,-#E,0!3XV8```I0_/_K"#*?Y0,PG^<#4)3G`!*?Y04`H.$#,*#C`2"@
MXP$0C^#Y]O_K`%!0X@P```KD,9_E#""5Y0,PG^<`$)+EV"&?Y0,`E.<"((_@
M]?;_ZPPPE>4`$)/E"#"1Y0$%$^-9```:N!&?Y0,PH.,!(*#C`1"?YP$`E.>H
M$9_E`1"/X.+V_^L`4%#B"P``"I@!G^4&$*#A``"/X'SX_^L`(%#B00``"H0Q
MG^4,$)7E`S"?YP`0D>4#`)3GVO;_ZW`QG^4!(*#C;!&?Y6QQG^4!$(_@`S"?
MYP=PC^`#4)3G!0"@X;OV_^M4(9_E`!"@X04`H.$"((_@R_;_ZT0QG^5$`9_E
M`""@XP80H.$``(_@`S"?YP,PE.=E(,/E7/C_ZP!`H.$``%#C"`"'Y1(```H'
M':#C!`"@X9?V_^L!((#B`#"@X1$`4.,!`%(3)0``BO@@G^6P,)_E`B"/X`,P
MDN<`()/E!#"=Y0(@,^``,*#C&P``&@S0C>+P@+WH!`"7Y=#]_^L`0*#A``!0
MXP@`A^7F__\:N`"?Y0``C^`"]__K`3#0Y0``4^.9__\*E/__ZJ`PG^4,()7E
M`S"/X```DN4$,)/E`!"3Y5[Y_^N[___JA#"?Y0,PG^<#`)3GU/;_ZZ#__^K;
M]O_K<!"?Y00@H.%L`)_E`1"/X```C^`$$)'E`!"1Y>CV_^O85P$`'`$``+Q8
M`0!,%```C%@!`)P5``!<6`$`>!4``"18`0!,%0``-!,``.A7`0#$5P$``!4`
M`#!8`0#T%```B%<!`+02```T5@$`6!0``%!7`0#$5@$`%%<!`!P4```P0"WI
M`<J@XPS`3>#H#XSE$"&?Y0S03>(,,9_E`%"@X0(@C^`#,)+G`#"3Y00PC>4`
M,*#C/OO_ZR\0H.,`0*#A8O;_ZP``4.,!0(`2!`"@X9KV_^L`0*#AT`"?Y0T0
MH.$``(_@\/?_ZP`P4.(*```*`##3Y3``4^,``%,3`3"@`P`PH!,!,"/B``!5
MXP`PH`,!,`,2``!3XPH``!J0()_EA#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@
M`#"@XQ@``!H,T(WB,("]Z``PU>4``%/C\?__"EP0G^4$(*#C!`"@X0$0C^"H
M]O_K``!0X^K__QI$()_E,#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@XP,`
M`!H%`*#A#-"-XC!`O>@G^/_J<?;_ZP15`0`<`0``:!$``(!4`0"4$0``+%0!
M``````#P3RWI`<J@XPS`3>"`#HSE$$R?Y5??3>(,[)_E$#"-X@1`C^`#$*#A
M'""`XES`@.(.X)3G`.">Y53AC>4`X*#CC0IB].``\/,"`%SAW0I!]/K__QH&
M2]WM"#O=[1$>C>(*&]WM````ZK(18O((*]/M-""#X@`+T^T(,(/B`0!3X8]7
M8O2P`4+SM0%`\[0!0/.S06/RL3%A\J`88/(P`.'SL`%A\@X+P^WN__\:($"0
MZ9DY!^.".D7C#$"0Y1#`G>4`8)#E!!`NX!!PD.4#P(S@!1`!X&61H.$'P(S@
M!!`AX!0@G>7FS8S@`<",X`D0+N`#((+@!A`!X&:!H.$$((+@&*"=Y0X0(>`"
M$('@""`IX`.@BN`,(`+@#J"*X`D@(N`*((+@'*"=Y>P=@>!LP:#A`Z"*X`F@
MBN`,D"C@`9`)X.$M@N`(D"G@81&@X0J0B>`@H)WEXIV)X`.@BN`(H(K@`8`L
MX`*`".!B(:#A#(`HX`*P(>`*@(C@)*"=Y0FP"^#IC8C@`Z"*X`R@BN!IP:#A
M`9`KX`RP(N`*D(G@**"=Y0BP"^#HG8G@`Z"*X`&@BN!H$:#A`H`KX`&P+.`*
M@(C@+*"=Y0FP"^#IC8C@`Z"*X`*@BN!I(:#A#)`KX`*P(>`*D(G@,*"=Y0BP
M"^#HG8G@`Z"*X`R@BN!HP:#A`8`KX`RP(N`*@(C@-*"=Y0FP"^#IC8C@`Z"*
MX`&@BN!I$:#A`I`KX`&P+.`*D(G@.*"=Y0BP"^#HG8G@`Z"*X`*@BN!H(:#A
M#(`KX`*P(>`*@(C@/*"=Y0FP"^#IC8C@`Z"*X`R@BN!IP:#A`9`KX`RP(N`*
MD(G@0*"=Y0BP"^#HG8G@`Z"*X`&@BN!H$:#A`H`KX`&P+.`*@(C@1*"=Y0FP
M"^#IC8C@`Z"*X`*@BN!I(:#A#)`KX`*P(>`*D(G@2*"=Y0BP"^#HG8G@`Z"*
MX`R@BN!HP:#A`8`KX`RP(N`*@(C@3*"=Y0FP"^#IC8C@`Z"*X`&@BN!I$:#A
M`I`KX`&P+.`*D(G@4*"=Y0BP"^#HG8G@`Z"*X`*@BN!H(:#A#(`KX`J`B.!4
MH)WEZ8V(X`.@BN`,P(K@`J`AX`F@"N!ID:#A`:`JX`R@BN!8P)WEZ*V*X`/`
MC.`!P(S@"1`BX`@0`>!H@:#A`A`AX`P0@>!<P)WEZAV!X`,PC.`(P"G@"L`,
MX`(P@^`)P"S@8""=Y0/`C."A.P[CV3Y&XVJAH.$#((+@X<V,X`F0@N!D()WE
M`R""X`@@@N`*@"C@`8`HX`B`B>!AD:#A:!"=Y>R-B.`#$('@"A"!X`F@*N`,
MH"K@;,&@X0H@@N#HK8+@;""=Y0,@@N`)((+@#)`IX`B0*>!H@:#A"1"!X.J=
M@>!P$)WE`Q"!X`P0@>`(P"S@"L`LX&JAH.$,((+@Z<V"X'0@G>4#((+@""""
MX`J`*.`)@"C@:9&@X0@0@>#LC8'@>!"=Y0,0@>`*$('@":`JX`R@*N!LP:#A
M"B""X.BM@N!\()WE`R""X`D@@N`,D"G@")`IX&B!H.$)$('@ZIV!X(`0G>4#
M$('@#!"!X`C`+.`*P"S@:J&@X0P@@N#IS8+@A""=Y0,@@N`(((+@"H`HX`F`
M*.!ID:#A"!"!X.R-@>"($)WE`Q"!X`H0@>`)H"K@#*`JX&S!H.$*((+@Z*V"
MX(P@G>4#((+@:K&@X0D@@N`,D"G@")`IX&B!H.$)$('@ZIV!X)`0G>4#$('@
M#!"!X`C`+.`*P"S@#"""X.FM@N"4()WE`R""X`@@@N`+@"C@"<`HX&F1H.$,
M$('@F,"=Y>J-@>`)$"O@`\",X`H0(>`+P(S@G+"=Y6JAH.$!((+@`["+X.@=
M@N`)L(O@"I`IX`@@*>!H@:#A`L",X*`@G>7AG8S@`R""X`H@@N`(H"K@`<`J
MX*2@G>5A$:#A#,"+X`.@BN#IS8S@"*"*X`&`*.`)@"C@:9&@X0@@@N"PL)WE
M[(V"X*@@G>4#((+@`2""X`D0(>`,$"'@;,&@X0&@BN`,$"G@Z*V*X`@0(>`!
M((+@:(&@X>H=@N"L()WE`R""X`@P+.`*,"/@"2""X`,@@N#</`OC&S](XVJ1
MH.$#L(O@":"(X0RPB^"TP)WEX2V"X`&@"N`#P(S@81&@X0C`C.`)@`C@"*"*
MX0&`B>$+H(K@`H`(X`&P">#BK8K@"X"(X;RPG>4,@(C@N,"=Y6(AH.$#L(O@
M`\",X`&PB^`)D(S@`L"!X0K`#.`"$`'@`<",X>J-B.`)P(S@P)"=Y6JAH.'H
MS8S@"A""X0.0B>`"D(G@"!`!X`H@`N!H@:#A`A"!X0@@"N`+L('@"!"*X0P0
M`>#LO8O@`A"!X6S!H.$)$('@Q)"=Y0P@B.'K'8'@`Y")X`L@`N`*D(G@R*"=
MY6NQH.$#H(K@"*"*X`R`".`(((+A88&@X0F0@N`+((SAX9V)X`$@`N#,$)WE
M`Q"!X`P0@>`+P`S@#,""X0@@B^$*P(S@"2`"X`B@"^#IS8S@"B""X="@G>5I
MD:#A`1""X`.@BN`)((CA"Z"*X-2PG>4,(`+@[!V!X`.PB^!LP:#A"+"+X`F`
M".`(((+A88&@X0J@@N`,((GAX:V*X`$@`N#8$)WE`Q"!X`D0@>`,D`G@"9""
MX0@@C.$+D(G@"B`"X`BP#.#JG8G@"R""X>"PG>4!$(+@W""=Y6JAH.$#L(O@
M`R""X`BPB^`,P(+@"B"(X0D@`N`*@`C@Z1V!X`B`@N%ID:#A#("(X`D@BN%A
MP:#AX8V(X`$@`N#D$)WE`Q"!X`H0@>`)H`K@"J""X0P@B>$+H(K@""`"X`RP
M">#HK8K@"R""X>RPG>4!$(+@Z""=Y6B!H.$#L(O@`R""X`RPB^`)D(+@""",
MX0H@`N`(P`S@ZAV!X`P@@N%JH:#A"9""X`H@B.%AP:#AX9V)X`$@`N#P$)WE
M`Q"!X`@0@>`*@`C@"(""X0P@BN$+@(C@"2`"X`RP"N#IC8C@"R""X?BPG>4!
M((+@]!"=Y6F1H.$#L(O@`Q"!X`RPB^`*H('@"1",X0@0`>`)P`S@#,"!X>@M
M@N`*P(S@_*"=Y6B!H.'BS8S@"!")X0,PBN`),(/@`A`!X`B0">!B(:#A`J"(
MX0D0@>$+$('@#*`*X`*P".#L'8'@;)&@X0N@BN$`P9WE`Z"*X-8Q#.-B.DSC
M`\",X.&MBN`(@(S@!,&=Y6&QH.$#P(S@`L",X`D@(N`!("+@"!&=Y0*`B.`+
M("G@ZHV(X`H@(N`#$('@:J&@X0D0@>`"P(S@"I`KX.C-C.`(D"G@#"&=Y0D0
M@>!H@:#A[)V!X!`1G>4#((+@`Q"!X`L@@N`*$('@"*`JX`R@*N!LP:#A"B""
MX.FM@N`4(9WE`R""X`@@@N`,@"C@"8`HX&F1H.$($('@ZHV!X!@1G>4#$('@
M#!"!X`G`+.`*P"S@:J&@X0P@@N#HS8+@'"&=Y0,@@N`)((+@"I`IX`B0*>!H
M@:#A"1"!X.R=@>`@$9WE`Q"!X`H0@>`(H"K@#*`JX&S!H.$*((+@Z:V"X"0A
MG>4#((+@""""X`R`*.`)@"C@:9&@X0@0@>#JC8'@*!&=Y0,0@>`,$('@"<`L
MX`K`+.!JH:#A#"""X.C-@N`L(9WE`R""X`D@@N`*D"G@")`IX&B!H.$)$('@
M[)V!X#`1G>4#$('@"A"!X`B@*N`,H"K@;,&@X0H@@N#IK8+@-"&=Y0,@@N`(
M((+@#(`HX`F`*.!ID:#A"!"!X.J-@>`X$9WE`Q"!X`P0@>`)P"S@"L`LX&JA
MH.$,((+@Z,V"X#PAG>4#((+@"2""X`J0*>`(D"G@:(&@X0D0@>!`D9WE[!V!
MX`.0B>`*D(G@"*`JX`R@*N!LP:#A"B""X$2AG>7A+8+@`Z"*X`B@BN`,@"C@
M`8`HX&$1H.$(D(G@2(&=Y>*=B>`#@(C@#("(X`'`+.`"P"S@8B&@X0S`BN`"
M<(?@Z<V,X!!P@.5LX8[@"."-Y4SAG>4#,([@!C"#X`$P@^`"$"'@"1`AX&F1
MH.$!$(C@"2`BX.P=@>`,("+@`C"#X`E`A.#A/8/@#$"-Y0`PC>4!4(7@/""?
MY010C>7?"FWT`B"/X"@PG^6/"D#T`S"2YP`@D^54,9WE`B`SX``PH.,!```:
M5]^-XO"/O>A@\__KP%,!`!P!``#H1P$`$$`MZ0'*H.,,P$W@^`^,Y6``H.-4
M\__K!PO?[0@KW^T).]_M`""@XP0+P.T8((#E7""`Y=\J0/00@+WH\.'2PP``
M```!(T5GB:O-[_[<NIAV5#(0\$<MZ0'*H.,,P$W@X`^,Y10PD.4"8*#A@B&@
MX0!0H.$",)/@&""0Y10P@.4!0*#A7`"0Y0$@@B*F+H+@'("%X@``4.,8((7E
M(```&C\`5N,N``#:0)!&XBECH.$!<(;B!W.$X`00H.$(,*#A0$"$X@#@D>40
M$('B#,`1Y1`P@^((`!'E!"`1Y00`4>$0X`/E#,`#Y0@``^4$(`/E\___&@%`
MH.$%`*#ALOS_ZP<`5.'K__\:!F-)X`8@H.$'$*#A"`"@X?+R_^M<8(7E\(>]
MZ$!P8.(``(C@!@!7X09PH*$'(*#AZO+_ZUPPE>4#,(?@7#"%Y4``4^/PA[T8
M!0"@X0=@1N`'0(3@FOS_Z\[__^H$<*#AY___ZG!`+>D!RJ#C`HLM[0S`3>#H
M#XSE`4"@X100D>4`4*#A'&"$XC$_O^84$)3E&`"4Y=$1Y><P+[_F&"M#[`$@
M@>)_,.#C.`!2XP(`AN`!,,;G00``VD`@8N(`$*#CG/+_ZP0`H.%]_/_K+#"$
MXE``P/(/"D;T#PI#]#PPA.(/"D/T3#"$X@\'0_14,(3B!`"@X0^'`_1Q_/_K
M`S#4Y0`PQ>4$`*#ALC#4X0$PQ>4`,)3E(S2@X0(PQ>4`,)3E`S#%Y0<PU.4$
M,,7EMC#4X04PQ>4$,)3E(S2@X08PQ>4$,)3E!S#%Y0LPU.4(,,7ENC#4X0DP
MQ>4(,)3E(S2@X0HPQ>4(,)3E"S#%Y0\PU.4,,,7EOC#4X0TPQ>4,,)3E(S2@
MX0XPQ>4,,)3E#S#%Y1,PU.40,,7ELC'4X1$PQ>40,)3E(S2@X1(PQ>40,)3E
M$S#%Y0*+O>QP0+WH;/+_ZC@@8N(`$*#C6O+_Z\;__^H(,)_E`S"/X``PD^6U
M\O_J!$8!``A`+>D(@+WH`0`"`#T```!`````"P```$`````0````0````"8`
M``!`````-0```$`````#````0`````8```!`````%````$`````.````0```
M``T```!`````)0```$`````2````0````"@```!``````@```$`````8````
M0`````D```!`````"@```$```````````````"5S+R5S````+W!R;V,O<V5L
M9B]E>&4``$1Y;F%,;V%D97(Z.F)O;W1?1'EN84QO861E<@!);G1E<FYA;',Z
M.E!!4CHZ0D]/5``````O<')O8R\E:2\E<P!E>&4`+@```%!!4E]414U0````
M`%!!4E]04D]'3D%-10`````Z````)7,E<R5S```O````4$523#5,24(`````
M4$523$Q)0@!015),-4]05`````!015),24\``%!!4E])3DE424%,25I%1`!0
M05)?4U!!5TY%1`!005)?0TQ%04X```!005)?1$5"54<```!005)?0T%#2$4`
M``!005)?1TQ/0D%,7T1%0E5'`````%!!4E]'3$]"04Q?5$U01$E2````4$%2
M7U1-4$1)4@``4$%27T=,3T)!3%]414U0`%!!4E]'3$]"04Q?0TQ%04X`````
M,0```$Q$7TQ)0E)!4EE?4$%42`!365-414T``"4P,G@`````)7,E<R5S)7,`
M````<&%R+0`````E<SH@8W)E871I;VX@;V8@<')I=F%T92!S=6)D:7)E8W1O
M<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*```E<SH@<')I=F%T92!S=6)D:7)E
M8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E(&ET(&%N9"!R971R
M>2!Y;W5R(&]P97)A=&EO;BD*``!0051(`````'!A<FP`````+G!A<@`````E
M<R5S8V%C:&4M)7,E<P``)7,E<W1E;7`M)74E<P```"5S)7-T96UP+25U+25U
M)7,`````&````'!E<FP`````,````%!!4CHZ4&%C:V5R.CI615)324].````
M`#$N,#4W````56YA8FQE('1O(&-R96%T92!C86-H92!D:7)E8W1O<GD`````
M)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B9&ER96-T;W)Y("5S
M(&9A:6QE9"`H97)R;F\])6DI"@`````M90``+2T``%5315(`````55-%4DY!
M344`````5$5-4$1)4@!435``+W1M<`````!E=F%L("=E>&5C("]U<W(O8FEN
M+W!E<FP@("U3("0P("1[,2LB)$`B?2<*:68@,#L@(R!N;W0@<G5N;FEN9R!U
M;F1E<B!S;VUE('-H96QL"G!A8VMA9V4@7U]P87)?<&P["FUY("@D4$%27TU!
M1TE#+"`D1DE,15]O9F9S971?<VEZ92P@)&-A8VAE7VYA;65?<VEZ92D["D)%
M1TE.('L*)%!!4E]-04=)0R`](")<;E!!4BYP;5QN(CL*)$9)3$5?;V9F<V5T
M7W-I>F4@/2`T.R`@(",@<&%C:R@B3B(I"B1C86-H95]N86UE7W-I>F4@/2`T
M,#L*)%!+6DE07TU!1TE#(#T@(E!+7#`P,UPP,#0B.PI]"G-U8B!F:6YD7W!A
M<E]M86=I8PI["FUY("@D9F@I(#T@0%\["FUY("1C:'5N:U]S:7IE(#T@-C0@
M*B`Q,#(T.PIM>2`D8G5F.PIM>2`D<VEZ92`]("US("1F:#L*;7D@)'!O<R`]
M("@D<VEZ92TQ*2`M("@D<VEZ92TQ*2`E("1C:'5N:U]S:7IE.R`@("`@(",@
M3D]413H@)'!O<R!I<R!A(&UU;'1I<&QE(&]F("1C:'5N:U]S:7IE"G=H:6QE
M("@D<&]S(#X](#`I('L*<V5E:R`D9F@L("1P;W,L(#`["G)E860@)&9H+"`D
M8G5F+"`D8VAU;FM?<VEZ92`K(&QE;F=T:"@D4$%27TU!1TE#*3L*:68@*"AM
M>2`D:2`](')I;F1E>"@D8G5F+"`D4$%27TU!1TE#*2D@/CT@,"D@>PIR971U
M<FX@)'!O<R`K("1I.PI]"B1P;W,@+3T@)&-H=6YK7W-I>F4["GT*<F5T=7)N
M("TQ.PI]"FUY("@D<&%R7W1E;7`L("1P<F]G;F%M92P@0'1M<&9I;&4L("5-
M;V1U;&5#86-H92D["D5.1"![(&EF("@D14Y6>U!!4E]#3$5!3GTI('L*<F5Q
M=6ER92!&:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIR97%U
M:7)E($9I;&4Z.E-P96,["FUY("1T;W!D:7(@/2!&:6QE.CI"87-E;F%M93HZ
M9&ER;F%M92@D<&%R7W1E;7`I.PIO=71S*'%Q6U)E;6]V:6YG(&9I;&5S(&EN
M("(D<&%R7W1E;7`B72D["D9I;&4Z.D9I;F0Z.F9I;F1D97!T:"AS=6(@>R`H
M("UD("D@/R!R;61I<B`Z('5N;&EN:R!]+"`D<&%R7W1E;7`I.PIR;61I<B`D
M<&%R7W1E;7`["FEF("@M9"`D<&%R7W1E;7`@)B8@)%Y/(&YE("=-4U=I;C,R
M)RD@>PIM>2`D=&UP(#T@;F5W($9I;&4Z.E1E;7`H"E1%35!,051%(#T^("=T
M;7!86%A86"<L"D1)4B`]/B!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D=&]P
M9&ER*2P*4U5&1DE8(#T^("<N8VUD)RP*54Y,24Y+(#T^(#`L"BD["FUY("1F
M:6QE;F%M92`]("1T;7`M/F9I;&5N86UE.PIP<FEN="`D=&UP(#P\(BXN+B([
M"G@],3L@=VAI;&4@6R!<)'@@+6QT(#$P(%T[(&1O"G)M("UR9B`G)'!A<E]T
M96UP)PII9B!;(%PA("UD("<D<&%R7W1E;7`G(%T[('1H96X*8G)E86L*9FD*
M<VQE97`@,0IX/6!E>'!R(%PD>"`K(#%@"F1O;F4*<FT@)R1F:6QE;F%M92<*
M+BXN"F-L;W-E("1T;7`["F-H;6]D(#`W,#`L("1F:6QE;F%M93L*;7D@)&-M
M9"`]("(D9FEL96YA;64@/B]D978O;G5L;"`R/B8Q("8B.PIS>7-T96TH)&-M
M9"D["F]U=',H<7%;4W!A=VYE9"!B86-K9W)O=6YD('!R;V-E<W,@=&\@<&5R
M9F]R;2!C;&5A;G5P.B`D9FEL96YA;65=*3L*?0I]('T*0D5'24X@>PI);G1E
M<FYA;',Z.E!!4CHZ0D]/5"@I(&EF(&1E9FEN960@)DEN=&5R;F%L<SHZ4$%2
M.CI"3T]4.PIE=F%L('L*7W!A<E]I;FET7V5N=B@I.PIM>2`D<75I970@/2`A
M)$5.5GM005)?1$5"54=].PIM>2`E<WES(#T@*`IP871H7W-E<"`@("`]/B`H
M)%Y/(#U^("]>35-7:6XO(#\@)SLG(#H@)SHG*2P*7V5X92`@("`@("`@/3X@
M*"1>3R`]?B`O7B@_.DU35VEN?$]3,GQC>6=W:6XI+R`_("<N97AE)R`Z("<G
M*2P*7V1E;&EM("`@("`@/3X@*"1>3R`]?B`O7DU35VEN?$]3,B\@/R`G7%PG
M(#H@)R\G*2P**3L*7W-E=%]P<F]G;F%M92@I.PIO=71S*'%Q6UPD<')O9VYA
M;64@/2`B)'!R;V=N86UE(ETI.PI?<V5T7W!A<E]T96UP*"D["F]U=',H<7%;
M7"1%3E9[4$%27U1%35!](#T@(B1%3E9[4$%27U1%35!](ETI.PIM>2`H)'-T
M87)T7W!O<RP@)'-T87)T7V]F7T9)3$5?<V5C=&EO;BD["DU!1TE#.B!["FQO
M8V%L("1324=[7U]705).7U]](#T@<W5B('M].PIU;FQE<W,@*&]P96X@7T9(
M+"`G/#IR87<G+"`D<')O9VYA;64I('L*;W5T<RAQ<5M#86XG="!R96%D(&9R
M;VT@9FEL92`B)'!R;V=N86UE(ETI.R`@(R!D;VXG="!U<V4@)"$@:&5R92!A
M<R!I="!R97%U:7)E<R!%<G)N;RYP;0IL87-T($U!1TE#.PI]"FUY("1M86=I
M8U]P;W,@/2!F:6YD7W!A<E]M86=I8R@J7T9(*3L*:68@*"1M86=I8U]P;W,@
M/"`P*2!["F]U=',H<7%;0V%N)W0@9FEN9"!M86=I8R!S=')I;F<@(B1005)?
M34%'24,B(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0SL*?0IO
M=71S*")&;W5N9"!005(@;6%G:6,@870@<&]S:71I;VX@)&UA9VEC7W!O<R(I
M.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M("1&24Q%7V]F9G-E=%]S:7IE("T@
M;&5N9W1H*")<,$-!0TA%(BDL(#`["G)E860@7T9(+"`D8G5F+"!L96YG=&@H
M(EPP0T%#2$4B*3L*:68@*"1B=68@;F4@(EPP0T%#2$4B("8F("1B=68@;F4@
M(EPP0TQ%04XB*2!["F]U=',H(DYO(&-A8VAE(&UA<FME<B!F;W5N9"(I.PIL
M87-T($U!1TE#.PI]"F]U=',H<7%;0V%C:&4@;6%R:V5R("(D8G5F(B!F;W5N
M9%TI.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M("1&24Q%7V]F9G-E=%]S:7IE
M+"`P.PIR96%D(%]&2"P@)&)U9BP@)$9)3$5?;V9F<V5T7W-I>F4["FUY("1O
M9F9S970@/2!U;G!A8VLH(DXB+"`D8G5F*3L*;W5T<R@B3V9F<V5T(&9R;VT@
M<W1A<G0@;V8@1DE,17,@:7,@)&]F9G-E="(I.PIS965K(%]&2"P@)&UA9VEC
M7W!O<R`M("1&24Q%7V]F9G-E=%]S:7IE("T@)&]F9G-E="P@,#L*)'-T87)T
M7V]F7T9)3$5?<V5C=&EO;B`]('1E;&P@7T9(.PIM>2`E<F5Q=6ER95]L:7-T
M.PIR96%D(%]&2"P@)&)U9BP@-#L@("`@("`@("`@("`@("`@("`@("`@("`@
M("`C(')E860@=&AE(&9I<G-T(")&24Q%(@IW:&EL92`H)&)U9B!E<2`B1DE,
M12(I('L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A
M8VLH(DXB+"`D8G5F*3L*;7D@)&9U;&QN86UE(#T@)&)U9CL*;W5T<RAQ<5M5
M;G!A8VMI;F<@1DE,12`B)&9U;&QN86UE(BXN+ETI.PIM>2`D8W)C(#T@*"`D
M9G5L;&YA;64@/7X@<WQ>*%MA+69<9%U[.'TI+WQ\("D@/R`D,2`Z('5N9&5F
M.PIM>2`H)&)A<V5N86UE+"`D97AT*2`]("@D8G5F(#U^(&U\*#\Z+BHO*3\H
M+BHI*%PN+BHI?"D["G)E860@7T9(+"`D8G5F+"`T.PIR96%D(%]&2"P@)&)U
M9BP@=6YP86-K*").(BP@)&)U9BD["FEF("AD969I;F5D*"1E>'0I(&%N9"`D
M97AT("%^("]<+B@_.G!M?'!L?&EX?&%L*20O:2D@>PIM>2`D9FEL96YA;64@
M/2!?<V%V95]A<R@B)&-R8R1E>'0B+"`D8G5F+"`P-S4U*3L*)%!!4CHZ2&5A
M=GDZ.D9U;&Q#86-H97LD9G5L;&YA;65](#T@)&9I;&5N86UE.PHD4$%2.CI(
M96%V>3HZ1G5L;$-A8VAE>R1F:6QE;F%M97T@/2`D9G5L;&YA;64["GT*96QS
M:68@*"`D9G5L;&YA;64@/7X@;7Q>+S]S:&QI8B]\(&%N9"!D969I;F5D("1%
M3E9[4$%27U1%35!]("D@>PIM>2`D9FEL96YA;64@/2!?<V%V95]A<R@B)&)A
M<V5N86UE)&5X="(L("1B=68L(#`W-34I.PIO=71S*")32$Q)0CH@)&9I;&5N
M86UE7&XB*3L*?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M97T@/2![
M"F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L;&YA;64L
M"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E860@7T9(+"`D
M8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF+"`D;6]D
M=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M;V1U;&4[
M"FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR971U<FX@=6YL
M97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?3L*)$E.
M0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII9B`H)$5.
M5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W*2D@>PIM
M>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#86XG="!C
M<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD9F@M/G!R
M:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T=7)N("1F
M:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I;F9O+3Y[
M8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@/2`D9FEL
M96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@;W(@9&EE
M('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N("1F:#L*
M?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD(&UO9'5L
M92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R.PIR97%U
M:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@
M17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R
M*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA;64@/2`H
M<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H)$E.0WLD
M9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII9B`H)&9I
M;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA;64["GT*
M96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^("]S:71E
M8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI<W1[)&9I
M;&5N86UE?3L*?0II9B`H)&)U9B!N92`D4$M:25!?34%'24,I('L*;W5T<RAQ
M<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I;&4@(B1P
M<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]("AT96QL
M(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II<`I]"FUY
M($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@)&-A8VAE
M7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI=&EZ
M92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*:68@*"$D<W1A
M<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@)B8@<VAI
M9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P87(*:2`@
M(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N7W!A<@IV
M("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=(&5Q("<M
M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%55-%?2`]
M('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E:&%V:6]U
M<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=66S!=(#U^
M("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF("@D,2!E
M<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@*"0Q(&5Q
M("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G02<I('L*
M=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G3R<I('L*
M)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L92`]("=S
M:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G86QL)SL*
M?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS:68@*"0Q
M(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE('%Q6T-A
M;B=T(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D,2!E<2`G
M5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD["FEF("AM
M>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG4$%27U1%
M35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[(E!!4CHZ
M1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I<W0Z.B1C
M;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`24Y#+"!`
M861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)3SHZ1FEL
M93L*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$[
M"GT*;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E9FEN
M960@)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@
M<7%;0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE<W,H)&9H
M+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH("1Z
M:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R8VAI=F4Z
M.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A
M<F-H:79E("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R(#T@9&\@>PII9B`H)'II
M<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY;6PG*2D@
M>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>7%,N*B\O
M;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I].PII9B`H
M9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D<&%R(&]R
M(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=.PIM>2`D
M8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I<R!N;W0@
M82!P87(@9FEL95T@=6YL97-S("1B=68@97$@)%!+6DE07TU!1TE#.PIC;&]S
M92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/.CI&:6QE
M+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL93HZ
M3U]21%=2*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI(&]R(&1I
M92!Q<5M#86XG="!C<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H+3YB:6YM
M;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H9&5F:6YE
M9"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*2!["G)E860@7T9(+"`D;&]A9&5R
M+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N.PI](&5L<V4@>PIL;V-A;"`D+R`]
M('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!4E]615)"
M051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2!["G)E<75I
M<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ4&]D4W1R
M:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D;&]A9&5R
M*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O=70B.B`D
M(5T["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*4$%2.CI(
M96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR97%U:7)E
M7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@?0IG<F5P
M('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I9SHZ0V]N
M9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF:6=[<')I
M=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R96%C:"!`
M:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,;V%D97(Z
M.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI8E]E>'0@
M/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!86%@@;&EB
M7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY("1K97D@
M*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D["FEF("AD
M969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&ME>7TI
M*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS92!["F9O
M<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<421D:7)<
M12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q*3L*;&%S
M=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD?"D@>PII
M9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE+"`D;F%M
M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D,2(I('L*
M*"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0["GT*?0I]
M"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D=W)I='1E
M;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I;&4@/7X@
M+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C:VEN9R!&
M24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65](#H@)&9I
M;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C;VYT96YT
M(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PIO
M<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A;B=T(')E
M860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S92`D=&@[
M"E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT+"`B/&5M
M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@86YD("1N
M86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z.E!A=&-H
M0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D;F%M92D[
M"GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H)&YA;64I
M("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z.F-O;7!U
M=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE;F=T:"@D
M8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@
M96UB961D960@1DE,12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7<FET=&5N
M(&%S("(D;F%M92)=*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R:71E5&]&
M:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO<B!D:64@
M<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]"FEF("@D
M;65T85]P87)[8VQE86Y]*2!["B1F:"T^<')I;G0H(EPP0TQ%04XB*3L*?0IE
M;'-E('L*:68@*"%D969I;F5D("1C86-H95]N86UE*2!["FUY("1C='@@/2!$
M:6=E<W0Z.E-(02T^;F5W*#$I.PIS965K("1F:"P@,"P@,#L*)&-T>"T^861D
M9FEL92@D9F@I.PIS965K("1F:"P@,"P@,CL*)&-A8VAE7VYA;64@/2`D8W1X
M+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE7VYA
M;64L(#`L("1C86-H95]N86UE7W-I>F4["B1C86-H95]N86UE("X](")<,"(@
M>"`H)&-A8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M92D["B1F
M:"T^<')I;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%(BD["GT*;7D@)&]F9G-E
M="`]("1F:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP<FEN="AP
M86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I.PHD9F@M/F-L;W-E(&]R
M(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R86EL97(@;V8@(B1O=70B.B`D(5T[
M"F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL87-T('5N;&5S<R!D969I
M;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER92!005([
M"E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U:7)E($9I
M;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F:"`]($E/
M.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@("`@("`@("`@("`C($%R
M8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F:"T^9F1O
M<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH*2!F86EL
M960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@7T9(*3L*
M;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH)'II<"T^<F5A9$9R;VU&
M:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI::7`Z.D%:
M7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E("(D
M<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H-C0@*B`Q
M,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ3&EB0V%C
M:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U=',H(D5X=')A8W1I;F<@>FEP+BXN
M(BD["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO=6QD(&)E
M('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M96UB97(@*"`D>FEP
M+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M/FES1&ER96-T;W)Y.PIM
M>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X="!U;FQE
M<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O;F9I9SHZ
M0V]N9FEG>W9E<G-I;VY]+RD_"B@_.B1#;VYF:6<Z.D-O;F9I9WMA<F-H;F%M
M97TO*3\**%M>+UTK*0HD"GUX.PIM>2`D97AT<F%C=%]N86UE(#T@)#$["FUY
M("1D97-T7VYA;64@/2!&:6QE.CI3<&5C+3YC871F:6QE*"1%3E9[4$%27U1%
M35!]+"`D97AT<F%C=%]N86UE*3L*:68@*"UF("1D97-T7VYA;64@)B8@+7,@
M7R`]/2`D;65M8F5R+3YU;F-O;7!R97-S9613:7IE*"DI('L*;W5T<RAQ<5M3
M:VEP<&EN9R`B)&UE;6)E<E]N86UE(B!S:6YC92!I="!A;')E861Y(&5X:7-T
M<R!A="`B)&1E<W1?;F%M92)=*3L*?2!E;'-E('L*;W5T<RAQ<5M%>'1R86-T
M:6YG("(D;65M8F5R7VYA;64B('1O("(D9&5S=%]N86UE(BXN+ETI.PHH)&UE
M;6)E<BT^97AT<F%C=%1O1FEL94YA;65D*"1D97-T7VYA;64I(#T]($%R8VAI
M=F4Z.EII<#HZ05I?3TLH*2D*;W(@9&EE('%Q6T5R<F]R(&5X=')A8W1I;F<@
M>FEP(&UE;6)E<B!T;R`B)&1E<W1?;F%M92)=.PIC:&UO9"@P-34U+"`D9&5S
M=%]N86UE*2!I9B`D7D\@97$@(FAP=7@B.PI]"GT*?0I]"G5N;&5S<R`H)%!!
M4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2D@>PID:64@/#P@(BXB('5N;&5S<R!`
M05)'5CL*57-A9V4Z("0P(%L@+4%L:6(N<&%R(%T@6R`M261I<B!=(%L@+4UM
M;V1U;&4@72!;('-R8RYP87(@72!;('!R;V=R86TN<&P@70HD,"!;("U"?"UB
M(%T@6RU/;W5T9FEL95T@<W)C+G!A<@HN"B1%3E9[4$%27U!23T=.04U%?2`]
M("1P<F]G;F%M92`]("0P(#T@<VAI9G0H0$%21U8I.PI]"G-U8B!#<F5A=&50
M871H('L*;7D@*"1N86UE*2`]($!?.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE
M.PIM>2`H)&)A<V5N86UE+"`D<&%T:"P@)&5X="D@/2!&:6QE.CI"87-E;F%M
M93HZ9FEL97!A<G-E*"1N86UE+"`H)UPN+BHG*2D["G)E<75I<F4@1FEL93HZ
M4&%T:#L*1FEL93HZ4&%T:#HZ;6MP871H*"1P871H*2!U;FQE<W,H+64@)'!A
M=&@I.R`C(&UK<&%T:"!D:65S('=I=&@@97)R;W(*?0IS=6(@<F5Q=6ER95]M
M;V1U;&5S('L*<F5Q=6ER92!L:6(["G)E<75I<F4@1'EN84QO861E<CL*<F5Q
M=6ER92!I;G1E9V5R.PIR97%U:7)E('-T<FEC=#L*<F5Q=6ER92!W87)N:6YG
M<SL*<F5Q=6ER92!V87)S.PIR97%U:7)E($-A<G`["G)E<75I<F4@0V%R<#HZ
M2&5A=GD["G)E<75I<F4@17)R;F\["G)E<75I<F4@17AP;W)T97(Z.DAE879Y
M.PIR97%U:7)E($5X<&]R=&5R.PIR97%U:7)E($9C;G1L.PIR97%U:7)E($9I
M;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ4W!E8SL*<F5Q=6ER92!84TQO861E
M<CL*<F5Q=6ER92!#;VYF:6<["G)E<75I<F4@24\Z.DAA;F1L93L*<F5Q=6ER
M92!)3SHZ1FEL93L*<F5Q=6ER92!#;VUP<F5S<SHZ6FQI8CL*<F5Q=6ER92!!
M<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["G)E<75I<F4@4$%2
M.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@4$%2.CI$:7-T.PIR97%U
M:7)E(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*<F5Q=6ER92!005(Z.D9I;'1E
M<CHZ4&%T8VA#;VYT96YT.PIR97%U:7)E(&%T=')I8G5T97,["F5V86P@>R!R
M97%U:7)E($-W9"!].PIE=F%L('L@<F5Q=6ER92!7:6XS,B!].PIE=F%L('L@
M<F5Q=6ER92!38V%L87(Z.E5T:6P@?3L*979A;"![(')E<75I<F4@07)C:&EV
M93HZ56YZ:7`Z.D)U<G-T('T["F5V86P@>R!R97%U:7)E(%1I93HZ2&%S:#HZ
M3F%M961#87!T=7)E('T["F5V86P@>R!R97%U:7)E(%!E<FQ)3SL@<F5Q=6ER
M92!097)L24\Z.G-C86QA<B!].PIE=F%L('L@<F5Q=6ER92!U=&8X('T["GT*
M<W5B(%]S971?<&%R7W1E;7`@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0
M?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO*2!["B1P87)?=&5M<"`]
M("0Q.PIR971U<FX["GT*9F]R96%C:"!M>2`D<&%T:"`H"BAM87`@)$5.5GLD
M7WTL('%W*"!005)?5$U01$E2(%1-4$1)4B!414U01$E2(%1%35`@5$U0("DI
M+`IQ=R@@0SI<7%1%35`@+W1M<"`N("D**2!["FYE>'0@=6YL97-S(&1E9FEN
M960@)'!A=&@@86YD("UD("1P871H(&%N9"`M=R`D<&%T:#L*;7D@)'5S97)N
M86UE.PIM>2`D<'=U:60["F5V86P@>R@D<'=U:60I(#T@9V5T<'=U:60H)#XI
M(&EF(&1E9FEN960@)#X[?3L*:68@*"!D969I;F5D*"97:6XS,CHZ3&]G:6Y.
M86UE*2`I('L*)'5S97)N86UE(#T@)E=I;C,R.CI,;V=I;DYA;64["GT*96QS
M:68@*&1E9FEN960@)'!W=6ED*2!["B1U<V5R;F%M92`]("1P=W5I9#L*?0IE
M;'-E('L*)'5S97)N86UE(#T@)$5.5GM54T523D%-17T@?'P@)$5.5GM54T52
M?2!\?"`G4UE35$5-)SL*?0IM>2`D<W1M<&1I<B`]("(D<&%T:"1S>7-[7V1E
M;&EM?7!A<BTB+G5N<&%C:R@B2"HB+"`D=7-E<FYA;64I.PIM:V1I<B`D<W1M
M<&1I<BP@,#<U-3L*;7D@)&-A8VAE7V1I<CL*:68@*"1%3E9[4$%27T-,14%.
M?2D@>PHD8V%C:&5?9&ER(#T@(G1E;7`M)"0B.PI]"F5L<V4@>PIO<&5N(&UY
M("1F:"P@(CPZ<F%W(BP@)'!R;V=N86UE(&]R(&1I92!Q<5M#86XG="!R96%D
M("(D<')O9VYA;64B.B`D(5T["FEF("@H;7D@)&UA9VEC7W!O<R`](&9I;F1?
M<&%R7VUA9VEC*"1F:"DI(#X](#`I('L*<V5E:R`D9F@L("1M86=I8U]P;W,*
M+2`D1DE,15]O9F9S971?<VEZ90HM(&QE;F=T:"@B7#!#04-(12(I+"`P.PIM
M>2`D8G5F.PIR96%D("1F:"P@)&)U9BP@;&5N9W1H*")<,$-!0TA%(BD["FEF
M("@D8G5F(&5Q(")<,$-!0TA%(BD@>PIS965K("1F:"P@)&UA9VEC7W!O<PHM
M("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H*")<,$-!0TA%(BD*+2`D8V%C
M:&5?;F%M95]S:7IE+"`P.PIR96%D("1F:"P@)&)U9BP@)&-A8VAE7VYA;65?
M<VEZ93L*)&)U9B`]?B!S+UPP+R]G.PHD8V%C:&5?9&ER(#T@(F-A8VAE+21B
M=68B.PI]"GT*8VQO<V4@)&9H.PI]"FEF("@A)&-A8VAE7V1I<BD@>PHD8V%C
M:&5?9&ER(#T@(G1E;7`M)"0B.PHD14Y6>U!!4E]#3$5!3GT@/2`Q.PI]"B1S
M=&UP9&ER("X]("(D<WES>U]D96QI;7TD8V%C:&5?9&ER(CL*;6MD:7(@)'-T
M;7!D:7(L(#`W-34["B1%3E9[4$%27U1%35!](#T@)'-T;7!D:7(["FQA<W0[
M"GT*)'!A<E]T96UP(#T@)#$@:68@)$5.5GM005)?5$5-4'T@86YD("1%3E9[
M4$%27U1%35!](#U^("\H+BLI+SL*?0IS=6(@7W-A=F5?87,@>PIM>2`H)&YA
M;64L("1C;VYT96YT<RP@)&UO9&4I(#T@0%\["FUY("1F=6QL;F%M92`]("(D
M<&%R7W1E;7`O)&YA;64B.PIU;FQE<W,@*"UE("1F=6QL;F%M92D@>PIM>2`D
M=&5M<&YA;64@/2`B)&9U;&QN86UE+B0D(CL*;7D@)&9H.PHH;W!E;B`D9F@L
M("<^.G)A=R<L("1T96UP;F%M92D*)B8@*'!R:6YT("1F:"`D8V]N=&5N=',I
M"B8F("AC;&]S92`D9F@I"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG("(D=&5M
M<&YA;64B.B`D(5T["F-H;6]D("1M;V1E+"`D=&5M<&YA;64@:68@9&5F:6YE
M9"`D;6]D93L*<F5N86UE*"1T96UP;F%M92P@)&9U;&QN86UE*2!O<B!U;FQI
M;FLH)'1E;7!N86UE*3L*?0IR971U<FX@)&9U;&QN86UE.PI]"G-U8B!?<V5T
M7W!R;V=N86UE('L*:68@*&1E9FEN960@)$5.5GM005)?4%)/1TY!345](&%N
M9"`D14Y6>U!!4E]04D]'3D%-17T@/7X@+R@N*RDO*2!["B1P<F]G;F%M92`]
M("0Q.PI]"B1P<F]G;F%M92!\?#T@)#`["FEF("@D14Y6>U!!4E]414U0?2!A
M;F0@:6YD97@H)'!R;V=N86UE+"`D14Y6>U!!4E]414U0?2D@/CT@,"D@>PHD
M<')O9VYA;64@/2!S=6)S='(H)'!R;V=N86UE+"!R:6YD97@H)'!R;V=N86UE
M+"`D<WES>U]D96QI;7TI("L@,2D["GT*:68@*"$D14Y6>U!!4E]04D]'3D%-
M17T@;W(@:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#X](#`I('L*
M:68@*&]P96X@;7D@)&9H+"`G/"<L("1P<F]G;F%M92D@>PIR971U<FX@:68@
M+7,@)&9H.PI]"FEF("@M<R`B)'!R;V=N86UE)'-Y<WM?97AE?2(I('L*)'!R
M;V=N86UE("X]("1S>7-[7V5X97T["G)E='5R;CL*?0I]"F9O<F5A8V@@;7D@
M)&1I<B`H<W!L:70@+UQ1)'-Y<WMP871H7W-E<'U<12\L("1%3E9[4$%42'TI
M('L*;F5X="!I9B!E>&ES=',@)$5.5GM005)?5$5-4'T@86YD("1D:7(@97$@
M)$5.5GM005)?5$5-4'T["B1D:7(@/7X@<R]<421S>7-[7V1E;&EM?5Q%)"\O
M.PHH*"1P<F]G;F%M92`]("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y
M<WM?97AE?2(I+"!L87-T*0II9B`M<R`B)&1I<B1S>7-[7V1E;&EM?21P<F]G
M;F%M921S>7-[7V5X97TB.PHH*"1P<F]G;F%M92`]("(D9&ER)'-Y<WM?9&5L
M:6U])'!R;V=N86UE(BDL(&QA<W0I"FEF("US("(D9&ER)'-Y<WM?9&5L:6U]
M)'!R;V=N86UE(CL*?0I]"G-U8B!?9FEX7W!R;V=N86UE('L*)#`@/2`D<')O
M9VYA;64@?'P]("1%3E9[4$%27U!23T=.04U%?3L*:68@*&EN9&5X*"1P<F]G
M;F%M92P@)'-Y<WM?9&5L:6U]*2`\(#`I('L*)'!R;V=N86UE(#T@(BXD<WES
M>U]D96QI;7TD<')O9VYA;64B.PI]"FUY("1P=V0@/2`H9&5F:6YE9"`F0W=D
M.CIG971C=V0I(#\@0W=D.CIG971C=V0H*0HZ("@H9&5F:6YE9"`F5VEN,S(Z
M.D=E=$-W9"D@/R!7:6XS,CHZ1V5T0W=D*"D@.B!@<'=D8"D["F-H;VUP*"1P
M=V0I.PHD<')O9VYA;64@/7X@<R]>*#\]7"Y<+C]<421S>7-[7V1E;&EM?5Q%
M*2\D<'=D)'-Y<WM?9&5L:6U]+SL*)$5.5GM005)?4%)/1TY!345](#T@)'!R
M;V=N86UE.PI]"G-U8B!?<&%R7VEN:71?96YV('L*:68@*"`D14Y6>U!!4E])
M3DE424%,25I%1'TK*R`]/2`Q("D@>PIR971U<FX["GT@96QS92!["B1%3E9[
M4$%27TE.251)04Q)6D5$?2`](#(["GT*9F]R("AQ=R@@4U!!5TY%1"!414U0
M($-,14%.($1%0E5'($-!0TA%(%!23T=.04U%("D@*2!["F1E;&5T92`D14Y6
M>R)005)?)%\B?3L*?0IF;W(@*'%W+R!435!$25(@5$5-4"!#3$5!3B!$14)5
M1R`O*2!["B1%3E9[(E!!4E\D7R)](#T@)$5.5GLB4$%27T=,3T)!3%\D7R)]
M(&EF(&5X:7-T<R`D14Y6>R)005)?1TQ/0D%,7R1?(GT["GT*:68@*"1%3E9[
M4$%27U1%35!]*2!["F1E;&5T92`D14Y6>U!!4E]#3$5!3GT["GT*?0IS=6(@
M;W5T<R!["G)E='5R;B!I9B`D<75I970["FEF("@D;&]G9F@I('L*<')I;G0@
M)&QO9V9H(")`7UQN(CL*?0IE;'-E('L*<')I;G0@(D!?7&XB.PI]"GT*<W5B
M(&EN:71?:6YC('L*<F5Q=6ER92!#;VYF:6<["G!U<V@@0$E.0RP@9W)E<"!D
M969I;F5D+"!M87`@)$-O;F9I9SHZ0V]N9FEG>R1??2P@<7<H"F%R8VAL:6)E
M>'`@<')I=FQI8F5X<"!S:71E87)C:&5X<"!S:71E;&EB97AP"G9E;F1O<F%R
M8VAE>'`@=F5N9&]R;&EB97AP"BD["GT*<&%C:V%G92!M86EN.PIR97%U:7)E
M(%!!4CL*=6YS:&EF="!`24Y#+"!<)E!!4CHZ9FEN9%]P87(["E!!4BT^:6UP
M;W)T*$!P87)?87)G<RD["F1I92!Q<5MP87(N<&PZ($-A;B=T(&]P96X@<&5R
M;"!S8W)I<'0@(B1P<F]G;F%M92(Z($YO('-U8V@@9FEL92!O<B!D:7)E8W1O
M<GE<;ET*=6YL97-S("UE("1P<F]G;F%M93L*9&\@)'!R;V=N86UE.PI#3U)%
M.CIE>&ET*"0Q*2!I9B`H)$`@/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID
M:64@)$`@:68@)$`["GT["B0Z.E]?15)23U(@/2`D0"!I9B`D0#L*?0I#3U)%
M.CIE>&ET*"0Q*2!I9B`H)#HZ7U]%4E)/4B`]?B]>7U1+7T58251?7"@H7&0K
M*5PI+RD["F1I92`D.CI?7T524D]2(&EF("0Z.E]?15)23U(["C$["E]?14Y$
M7U\*`'!E<FQX<VDN8P````!#04-(10``/0```$`````+````0````!````!`
M````)@```$`````U````0`````,```!`````!@```$`````4````0`````X`
M``!`````#0```$`````E````0````!(```!`````*````$`````"````0```
M`!@```!`````"0```$`````*````0```````````````AK(!@;"PKW\`````
M@;(!@;"PK`<`````_(C_?["J"8"HB_]_`0```*R._W^PJQZ`*)#_?\S__W^X
MD?]_T/__?^B2_W\!````))7_?["H`X#(E?]_`0```("A_W^PL*B`G*'_?P$`
M```(H_]_L*\\@,RI_W^PJP*`F*S_?P$`````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``#,&0``6!D```$````X`P```0```$,#```!````30,```$```!A`P``#P``
M`.$$```,````N!(```T````(2@``&0```+"-`0`;````!````!H```"TC0$`
M'`````0```#U_O]OM`$```4```"P"0``!@```!`#```*````!`4```L````0
M````%0`````````#````T(X!``(```#@`0``%````!$````7````V!```!$`
M```($```$@```-`````3````"````!@`````````^___;P$```C^__]OB`\`
M`/___V\#````\/__;[0.``#Z__]O#@``````````````````````````````
M``````````````````````````"XC0$```````````#$$@``Q!(``,02``#$
M$@``Q!(``,02``#$$@``Q!(``,02``#$$@``Q!(``,02``#$$@``Q!(``,02
M``#$$@``Q!(``,02``#$$@``Q!(``,02``#$$@``Q!(``,02``#$$@``Q!(`
M`,02``#$$@``Q!(``,02``#$$@``Q!(``,02``#$$@``Q!(``,02``#$$@``
MQ!(``,02``#$$@``Q!(``,02``#$$@``Q!(``,02``#$$@``Q!(``,02``#$
M$@``Q!(``,02``#$$@``Q!(``,02``#$$@``Q!(``,02``#$$@``Q!(``,02
M````````````````````````````````````````````````````````````
M``````"H%0````````20`0`*4$%2+G!M"@````#$30```$L```````#`2P``
MQ$L``+A-```(2P``P$T```````"D30``K$T```````#,30``1T-#.B`H1TY5
M*2`Q,BXQ+C``1T-#.B`H8W)O<W-T;V]L+4Y'(#$N,C4N,"XR.%]B.#AD,S,X
M*2`Q,BXQ+C``030```!A96%B:0`!*@````4W+4$`!@H'00@!"0(*`PP!$@03
M`10!%0$7`Q@!&0$:`AP!(@$`+G-H<W1R=&%B`"YI;G1E<G``+FYO=&4N9VYU
M+F)U:6QD+6ED`"YN;W1E+D%"22UT86<`+F=N=2YH87-H`"YD>6YS>6T`+F1Y
M;G-T<@`N9VYU+G9E<G-I;VX`+F=N=2YV97)S:6]N7W(`+G)E;"YD>6X`+G)E
M;"YP;'0`+FEN:70`+G1E>'0`+F9I;FD`+G)O9&%T80`N05)-+F5X=&%B`"Y!
M4DTN97AI9'@`+F5H7V9R86UE`"YI;FET7V%R<F%Y`"YF:6YI7V%R<F%Y`"YD
M>6YA;6EC`"YG;W0`+F1A=&$`+F)S<P`N8V]M;65N=``N05)-+F%T=')I8G5T
M97,```````````````````````````````````````````````````````L`
M```!`````@```%0!``!4`0``&0```````````````0`````````3````!P``
M``(```!P`0``<`$``"0```````````````0`````````)@````<````"````
ME`$``)0!```@```````````````$`````````#0```#V__]O`@```+0!``"T
M`0``7`$```4`````````!`````0````^````"P````(````0`P``$`,``*`&
M```&`````P````0````0````1@````,````"````L`D``+`)```$!0``````
M```````!`````````$X```#___]O`@```+0.``"T#@``U`````4`````````
M`@````(```!;````_O__;P(```"(#P``B`\``(`````&`````P````0`````
M````:@````D````"````"!````@0``#0````!0`````````$````"````',`
M```)````0@```-@0``#8$```X`$```4````6````!`````@```!\`````0``
M``8```"X$@``N!(```P```````````````0`````````=P````$````&````
MQ!(``,02``#D`@`````````````$````!````((````!````!@```*@5``"H
M%0``8#0`````````````"`````````"(`````0````8````(2@``"$H```@`
M``````````````0`````````C@````$````"````$$H``!!*``"$0@``````
M```````$`````````)8````!`````@```)2,``"4C```&```````````````
M!`````````"A`````0``<((```"LC```K(P``&@````-``````````0`````
M````K`````$````"````%(T``!2-```$```````````````$`````````+8`
M```.`````P```+"-`0"PC0``!```````````````!`````0```#"````#P``
M``,```"TC0$`M(T```0```````````````0````$````S@````8````#````
MN(T!`+B-```8`0``!@`````````$````"````-<````!`````P```-".`0#0
MC@``,`$`````````````!`````0```#<`````0````,`````D`$``)```$@`
M``````````````0`````````X@````@````#````2)`!`$B0```4````````
M```````$`````````.<````!````,`````````!(D```/P``````````````
M`0````$```#P`````P``<```````````AY```#4```````````````$`````
K`````0````,``````````````+R0`````0`````````````!````````````
