'use strict';

function _interopDefault (ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }

var fs = require('fs-extra');
var path = require('path');
var Ast = require('ts-simple-ast');
var Ast__default = _interopDefault(Ast);
var LiveServer = require('live-server');
var _ = require('lodash');
var i18next = _interopDefault(require('i18next'));
var Handlebars = require('handlebars');
var semver = require('semver');
var JSON5 = require('json5');

/*! *****************************************************************************
Copyright (c) Microsoft Corporation. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License"); you may not use
this file except in compliance with the License. You may obtain a copy of the
License at http://www.apache.org/licenses/LICENSE-2.0

THIS CODE IS PROVIDED ON AN *AS IS* BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION ANY IMPLIED
WARRANTIES OR CONDITIONS OF TITLE, FITNESS FOR A PARTICULAR PURPOSE,
MERCHANTABLITY OR NON-INFRINGEMENT.

See the Apache Version 2.0 License for specific language governing permissions
and limitations under the License.
***************************************************************************** */
/* global Reflect, Promise */

var extendStatics = function(d, b) {
    extendStatics = Object.setPrototypeOf ||
        ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
        function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
    return extendStatics(d, b);
};

function __extends(d, b) {
    extendStatics(d, b);
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
}

var __assign = function() {
    __assign = Object.assign || function __assign(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p)) t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};

var log = require('fancy-log');
var c = require('chalk');
var LEVEL;
(function (LEVEL) {
    LEVEL[LEVEL["INFO"] = 0] = "INFO";
    LEVEL[LEVEL["DEBUG"] = 1] = "DEBUG";
    LEVEL[LEVEL["ERROR"] = 2] = "ERROR";
    LEVEL[LEVEL["WARN"] = 3] = "WARN";
})(LEVEL || (LEVEL = {}));
var Logger = /** @class */ (function () {
    function Logger() {
        this.logger = log;
        this.silent = true;
    }
    Logger.prototype.info = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        if (!this.silent) {
            return;
        }
        this.logger(this.format.apply(this, [LEVEL.INFO].concat(args)));
    };
    Logger.prototype.error = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        if (!this.silent) {
            return;
        }
        this.logger(this.format.apply(this, [LEVEL.ERROR].concat(args)));
    };
    Logger.prototype.warn = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        if (!this.silent) {
            return;
        }
        this.logger(this.format.apply(this, [LEVEL.WARN].concat(args)));
    };
    Logger.prototype.debug = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        if (!this.silent) {
            return;
        }
        this.logger(this.format.apply(this, [LEVEL.DEBUG].concat(args)));
    };
    Logger.prototype.format = function (level) {
        var args = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            args[_i - 1] = arguments[_i];
        }
        var pad = function (s, l, z) {
            if (z === void 0) { z = ''; }
            return s + Array(Math.max(0, l - s.length + 1)).join(z);
        };
        var msg = args.join(' ');
        if (args.length > 1) {
            msg = pad(args.shift(), 15, ' ') + ": " + args.join(' ');
        }
        switch (level) {
            case LEVEL.INFO:
                msg = c.green(msg);
                break;
            case LEVEL.DEBUG:
                msg = c.cyan(msg);
                break;
            case LEVEL.WARN:
                msg = c.yellow(msg);
                break;
            case LEVEL.ERROR:
                msg = c.red(msg);
                break;
        }
        return [msg].join('');
    };
    return Logger;
}());
var logger = new Logger();

var COMPODOC_DEFAULTS = {
    title: 'Application documentation',
    additionalEntryName: 'Additional documentation',
    additionalEntryPath: 'additional-documentation',
    folder: './documentation/',
    hostname: '127.0.0.1',
    port: 8080,
    theme: 'gitbook',
    exportFormat: 'html',
    exportFormatsSupported: ['html', 'json', 'pdf'],
    base: '/',
    defaultCoverageThreshold: 70,
    defaultCoverageMinimumPerFile: 0,
    coverageTestThresholdFail: true,
    toggleMenuItems: ['all'],
    navTabConfig: [],
    disableSourceCode: false,
    disableDomTree: false,
    disableTemplateTab: false,
    disableStyleTab: false,
    disableGraph: false,
    disableMainGraph: false,
    disableCoverage: false,
    disablePrivate: false,
    disableProtected: false,
    disableInternal: false,
    disableLifeCycleHooks: false,
    disableRoutesGraph: false,
    disableDependencies: false,
    PAGE_TYPES: {
        ROOT: 'root',
        INTERNAL: 'internal'
    },
    gaSite: 'auto',
    coverageTestShowOnlyFailed: false,
    language: 'en-US',
    maxSearchResults: 15
};

var Configuration = /** @class */ (function () {
    function Configuration() {
        this._pages = [];
        this._mainData = {
            output: COMPODOC_DEFAULTS.folder,
            theme: COMPODOC_DEFAULTS.theme,
            extTheme: '',
            serve: false,
            hostname: COMPODOC_DEFAULTS.hostname,
            host: '',
            port: COMPODOC_DEFAULTS.port,
            open: false,
            assetsFolder: '',
            documentationMainName: COMPODOC_DEFAULTS.title,
            documentationMainDescription: '',
            base: COMPODOC_DEFAULTS.base,
            hideGenerator: false,
            hasFilesToCoverage: false,
            modules: [],
            readme: false,
            changelog: '',
            contributing: '',
            license: '',
            todo: '',
            markdowns: [],
            additionalPages: [],
            pipes: [],
            classes: [],
            interfaces: [],
            components: [],
            controllers: [],
            directives: [],
            injectables: [],
            interceptors: [],
            guards: [],
            miscellaneous: [],
            routes: [],
            tsconfig: '',
            toggleMenuItems: COMPODOC_DEFAULTS.toggleMenuItems,
            navTabConfig: [],
            templates: '',
            includes: '',
            includesName: COMPODOC_DEFAULTS.additionalEntryName,
            includesFolder: COMPODOC_DEFAULTS.additionalEntryPath,
            disableSourceCode: COMPODOC_DEFAULTS.disableSourceCode,
            disableDomTree: COMPODOC_DEFAULTS.disableDomTree,
            disableTemplateTab: COMPODOC_DEFAULTS.disableTemplateTab,
            disableStyleTab: COMPODOC_DEFAULTS.disableStyleTab,
            disableGraph: COMPODOC_DEFAULTS.disableGraph,
            disableMainGraph: COMPODOC_DEFAULTS.disableMainGraph,
            disableCoverage: COMPODOC_DEFAULTS.disableCoverage,
            disablePrivate: COMPODOC_DEFAULTS.disablePrivate,
            disableInternal: COMPODOC_DEFAULTS.disableInternal,
            disableProtected: COMPODOC_DEFAULTS.disableProtected,
            disableLifeCycleHooks: COMPODOC_DEFAULTS.disableLifeCycleHooks,
            disableRoutesGraph: COMPODOC_DEFAULTS.disableRoutesGraph,
            disableSearch: false,
            disableDependencies: COMPODOC_DEFAULTS.disableDependencies,
            watch: false,
            mainGraph: '',
            coverageTest: false,
            coverageTestThreshold: COMPODOC_DEFAULTS.defaultCoverageThreshold,
            coverageTestThresholdFail: COMPODOC_DEFAULTS.coverageTestThresholdFail,
            coverageTestPerFile: false,
            coverageMinimumPerFile: COMPODOC_DEFAULTS.defaultCoverageMinimumPerFile,
            unitTestCoverage: '',
            unitTestData: undefined,
            coverageTestShowOnlyFailed: COMPODOC_DEFAULTS.coverageTestShowOnlyFailed,
            routesLength: 0,
            angularVersion: '',
            exportFormat: COMPODOC_DEFAULTS.exportFormat,
            coverageData: {},
            customFavicon: '',
            customLogo: '',
            packageDependencies: [],
            packagePeerDependencies: [],
            gaID: '',
            gaSite: '',
            angularProject: false,
            angularJSProject: false,
            language: COMPODOC_DEFAULTS.language,
            maxSearchResults: 15
        };
    }
    Configuration.getInstance = function () {
        if (!Configuration.instance) {
            Configuration.instance = new Configuration();
        }
        return Configuration.instance;
    };
    Configuration.prototype.addPage = function (page) {
        var indexPage = _.findIndex(this._pages, { name: page.name });
        if (indexPage === -1) {
            this._pages.push(page);
        }
    };
    Configuration.prototype.hasPage = function (name) {
        var indexPage = _.findIndex(this._pages, { name: name });
        return indexPage !== -1;
    };
    Configuration.prototype.addAdditionalPage = function (page) {
        this._mainData.additionalPages.push(page);
    };
    Configuration.prototype.getAdditionalPageById = function (id) {
        return this._mainData.additionalPages.find(function (page) { return page.id === id; });
    };
    Configuration.prototype.resetPages = function () {
        this._pages = [];
    };
    Configuration.prototype.resetAdditionalPages = function () {
        this._mainData.additionalPages = [];
    };
    Configuration.prototype.resetRootMarkdownPages = function () {
        var indexPage = _.findIndex(this._pages, { name: 'index' });
        this._pages.splice(indexPage, 1);
        indexPage = _.findIndex(this._pages, { name: 'changelog' });
        this._pages.splice(indexPage, 1);
        indexPage = _.findIndex(this._pages, { name: 'contributing' });
        this._pages.splice(indexPage, 1);
        indexPage = _.findIndex(this._pages, { name: 'license' });
        this._pages.splice(indexPage, 1);
        indexPage = _.findIndex(this._pages, { name: 'todo' });
        this._pages.splice(indexPage, 1);
        this._mainData.markdowns = [];
    };
    Object.defineProperty(Configuration.prototype, "pages", {
        get: function () {
            return this._pages;
        },
        set: function (pages) {
            this._pages = [];
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(Configuration.prototype, "markDownPages", {
        get: function () {
            return this._pages.filter(function (page) { return page.markdown; });
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(Configuration.prototype, "mainData", {
        get: function () {
            return this._mainData;
        },
        set: function (data) {
            Object.assign(this._mainData, data);
        },
        enumerable: true,
        configurable: true
    });
    return Configuration;
}());
var Configuration$1 = Configuration.getInstance();

var AngularAPIs = require('../src/data/api-list.json');
var AngularApiUtil = /** @class */ (function () {
    function AngularApiUtil() {
    }
    AngularApiUtil.getInstance = function () {
        if (!AngularApiUtil.instance) {
            AngularApiUtil.instance = new AngularApiUtil();
        }
        return AngularApiUtil.instance;
    };
    AngularApiUtil.prototype.findApi = function (type) {
        var foundedApi;
        _.forEach(AngularAPIs, function (mainApi) {
            _.forEach(mainApi.items, function (api) {
                if (api.title === type) {
                    foundedApi = api;
                }
            });
        });
        return {
            source: 'external',
            data: foundedApi
        };
    };
    return AngularApiUtil;
}());
var AngularApiUtil$1 = AngularApiUtil.getInstance();

function extractLeadingText(string, completeTag) {
    var tagIndex = string.indexOf(completeTag);
    var leadingText = undefined;
    var leadingTextRegExp = /\[(.+?)\]/g;
    var leadingTextInfo = leadingTextRegExp.exec(string);
    // did we find leading text, and if so, does it immediately precede the tag?
    while (leadingTextInfo && leadingTextInfo.length) {
        if (leadingTextInfo.index + leadingTextInfo[0].length === tagIndex) {
            string = string.replace(leadingTextInfo[0], '');
            leadingText = leadingTextInfo[1];
            break;
        }
        leadingTextInfo = leadingTextRegExp.exec(string);
    }
    return {
        leadingText: leadingText,
        string: string
    };
}
function splitLinkText(text) {
    var linkText;
    var target;
    var splitIndex;
    // if a pipe is not present, we split on the first space
    splitIndex = text.indexOf('|');
    if (splitIndex === -1) {
        splitIndex = text.search(/\s/);
    }
    if (splitIndex !== -1) {
        linkText = text.substr(splitIndex + 1);
        // Normalize subsequent newlines to a single space.
        linkText = linkText.replace(/\n+/, ' ');
        target = text.substr(0, splitIndex);
    }
    return {
        linkText: linkText,
        target: target || text
    };
}
var LinkParser = (function () {
    var processTheLink = function (string, tagInfo, leadingText) {
        var leading = extractLeadingText(string, tagInfo.completeTag), linkText, split, target, stringtoReplace;
        linkText = leadingText ? leadingText : leading.leadingText || '';
        split = splitLinkText(tagInfo.text);
        target = split.target;
        if (leading.leadingText !== undefined) {
            stringtoReplace = '[' + leading.leadingText + ']' + tagInfo.completeTag;
        }
        else if (typeof split.linkText !== 'undefined') {
            stringtoReplace = tagInfo.completeTag;
            linkText = split.linkText;
        }
        return string.replace(stringtoReplace, '[' + linkText + '](' + target + ')');
    };
    /**
     * Convert
     * {@link http://www.google.com|Google} or {@link https://github.com GitHub} or [Github]{@link https://github.com} to [Github](https://github.com)
     */
    var replaceLinkTag = function (str) {
        if (typeof str === 'undefined') {
            return {
                newString: ''
            };
        }
        // new RegExp('\\[((?:.|\n)+?)]\\{@link\\s+((?:.|\n)+?)\\}', 'i').exec('ee [TO DO]{@link Todo} fo') -> "[TO DO]{@link Todo}", "TO DO", "Todo"
        // new RegExp('\\{@link\\s+((?:.|\n)+?)\\}', 'i').exec('ee [TODO]{@link Todo} fo') -> "{@link Todo}", "Todo"
        var tagRegExpLight = new RegExp('\\{@link\\s+((?:.|\n)+?)\\}', 'i'), tagRegExpFull = new RegExp('\\{@link\\s+((?:.|\n)+?)\\}', 'i'), tagRegExp, matches, previousString;
        tagRegExp = str.indexOf(']{') !== -1 ? tagRegExpFull : tagRegExpLight;
        function replaceMatch(replacer, tag, match, text, linkText) {
            var matchedTag = {
                completeTag: match,
                tag: tag,
                text: text
            };
            if (linkText) {
                return replacer(str, matchedTag, linkText);
            }
            else {
                return replacer(str, matchedTag);
            }
        }
        do {
            matches = tagRegExp.exec(str);
            if (matches) {
                previousString = str;
                if (matches.length === 2) {
                    str = replaceMatch(processTheLink, 'link', matches[0], matches[1]);
                }
                if (matches.length === 3) {
                    str = replaceMatch(processTheLink, 'link', matches[0], matches[2], matches[1]);
                }
            }
        } while (matches && previousString !== str);
        return {
            newString: str
        };
    };
    var _resolveLinks = function (str) {
        return replaceLinkTag(str).newString;
    };
    return {
        resolveLinks: _resolveLinks
    };
})();

var AngularLifecycleHooks;
(function (AngularLifecycleHooks) {
    AngularLifecycleHooks[AngularLifecycleHooks["ngOnChanges"] = 0] = "ngOnChanges";
    AngularLifecycleHooks[AngularLifecycleHooks["ngOnInit"] = 1] = "ngOnInit";
    AngularLifecycleHooks[AngularLifecycleHooks["ngDoCheck"] = 2] = "ngDoCheck";
    AngularLifecycleHooks[AngularLifecycleHooks["ngAfterContentInit"] = 3] = "ngAfterContentInit";
    AngularLifecycleHooks[AngularLifecycleHooks["ngAfterContentChecked"] = 4] = "ngAfterContentChecked";
    AngularLifecycleHooks[AngularLifecycleHooks["ngAfterViewInit"] = 5] = "ngAfterViewInit";
    AngularLifecycleHooks[AngularLifecycleHooks["ngAfterViewChecked"] = 6] = "ngAfterViewChecked";
    AngularLifecycleHooks[AngularLifecycleHooks["ngOnDestroy"] = 7] = "ngOnDestroy";
})(AngularLifecycleHooks || (AngularLifecycleHooks = {}));

function kindToType(kind) {
    var _type = '';
    switch (kind) {
        case Ast.SyntaxKind.StringKeyword:
        case Ast.SyntaxKind.StringLiteral:
            _type = 'string';
            break;
        case Ast.SyntaxKind.NumberKeyword:
        case Ast.SyntaxKind.NumericLiteral:
            _type = 'number';
            break;
        case Ast.SyntaxKind.ArrayType:
        case Ast.SyntaxKind.ArrayLiteralExpression:
            _type = '[]';
            break;
        case Ast.SyntaxKind.VoidKeyword:
            _type = 'void';
            break;
        case Ast.SyntaxKind.FunctionType:
            _type = 'function';
            break;
        case Ast.SyntaxKind.TypeLiteral:
            _type = 'literal type';
            break;
        case Ast.SyntaxKind.BooleanKeyword:
            _type = 'boolean';
            break;
        case Ast.SyntaxKind.AnyKeyword:
            _type = 'any';
            break;
        case Ast.SyntaxKind.NullKeyword:
            _type = 'null';
            break;
        case Ast.SyntaxKind.SymbolKeyword:
            _type = 'symbol';
            break;
        case Ast.SyntaxKind.NeverKeyword:
            _type = 'never';
            break;
        case Ast.SyntaxKind.UndefinedKeyword:
            _type = 'undefined';
            break;
        case Ast.SyntaxKind.ObjectKeyword:
        case Ast.SyntaxKind.ObjectLiteralExpression:
            _type = 'object';
            break;
    }
    return _type;
}

var getCurrentDirectory = Ast.ts.sys.getCurrentDirectory;
var useCaseSensitiveFileNames = Ast.ts.sys.useCaseSensitiveFileNames;
var newLine = Ast.ts.sys.newLine;
var marked = require('marked');
function getNewLine() {
    return newLine;
}
function cleanNameWithoutSpaceAndToLowerCase(name) {
    return name.toLowerCase().replace(/ /g, '-');
}
function getCanonicalFileName(fileName) {
    return useCaseSensitiveFileNames ? fileName : fileName.toLowerCase();
}
var formatDiagnosticsHost = {
    getCurrentDirectory: getCurrentDirectory,
    getCanonicalFileName: getCanonicalFileName,
    getNewLine: getNewLine
};
function markedtags(tags) {
    var mtags = tags;
    _.forEach(mtags, function (tag) {
        tag.comment = marked(LinkParser.resolveLinks(tag.comment));
    });
    return mtags;
}
function mergeTagsAndArgs(args, jsdoctags) {
    var margs = _.cloneDeep(args);
    _.forEach(margs, function (arg) {
        arg.tagName = {
            text: 'param'
        };
        if (jsdoctags) {
            _.forEach(jsdoctags, function (jsdoctag) {
                if (jsdoctag.name && jsdoctag.name.text === arg.name) {
                    arg.tagName = jsdoctag.tagName;
                    arg.name = jsdoctag.name;
                    arg.comment = jsdoctag.comment;
                    arg.typeExpression = jsdoctag.typeExpression;
                }
            });
        }
    });
    // Add example & returns & private
    if (jsdoctags) {
        _.forEach(jsdoctags, function (jsdoctag) {
            if (jsdoctag.tagName &&
                (jsdoctag.tagName.text === 'example' || jsdoctag.tagName.text === 'private')) {
                margs.push({
                    tagName: jsdoctag.tagName,
                    comment: jsdoctag.comment
                });
            }
            if (jsdoctag.tagName &&
                (jsdoctag.tagName.text === 'returns' || jsdoctag.tagName.text === 'return')) {
                var ret = {
                    tagName: jsdoctag.tagName,
                    comment: jsdoctag.comment
                };
                if (jsdoctag.typeExpression && jsdoctag.typeExpression.type) {
                    ret.returnType = kindToType(jsdoctag.typeExpression.type.kind);
                }
                margs.push(ret);
            }
        });
    }
    return margs;
}
function readConfig(configFile) {
    var result = Ast.ts.readConfigFile(configFile, Ast.ts.sys.readFile);
    if (result.error) {
        var message = Ast.ts.formatDiagnostics([result.error], formatDiagnosticsHost);
        throw new Error(message);
    }
    return result.config;
}
function stripBom(source) {
    if (source.charCodeAt(0) === 0xfeff) {
        return source.slice(1);
    }
    return source;
}
function hasBom(source) {
    return source.charCodeAt(0) === 0xfeff;
}
function handlePath(files, cwd) {
    var _files = files;
    var i = 0;
    var len = files.length;
    for (i; i < len; i++) {
        if (files[i].indexOf(cwd) === -1) {
            files[i] = path.resolve(cwd + path.sep + files[i]);
        }
    }
    return _files;
}
function cleanLifecycleHooksFromMethods(methods) {
    var result = [];
    if (typeof methods !== 'undefined') {
        var i = 0;
        var len = methods.length;
        for (i; i < len; i++) {
            if (!(methods[i].name in AngularLifecycleHooks)) {
                result.push(methods[i]);
            }
        }
    }
    return result;
}
function cleanSourcesForWatch(list) {
    return list.filter(function (element) {
        if (fs.existsSync(process.cwd() + path.sep + element)) {
            return element;
        }
    });
}
function getNamesCompareFn(name) {
    /**
     * Copyright https://github.com/ng-bootstrap/ng-bootstrap
     */
    name = name || 'name';
    var t = function (a, b) {
        if (a[name]) {
            return a[name].localeCompare(b[name]);
        }
        else {
            return 0;
        }
    };
    return t;
}
function isIgnore(member) {
    if (member.jsDoc) {
        for (var _i = 0, _a = member.jsDoc; _i < _a.length; _i++) {
            var doc = _a[_i];
            if (doc.tags) {
                for (var _b = 0, _c = doc.tags; _b < _c.length; _b++) {
                    var tag = _c[_b];
                    if (tag.tagName.text.indexOf('ignore') > -1) {
                        return true;
                    }
                }
            }
        }
    }
    return false;
}
// https://tc39.github.io/ecma262/#sec-array.prototype.includes
if (!Array.prototype.includes) {
    Object.defineProperty(Array.prototype, 'includes', {
        value: function (searchElement, fromIndex) {
            if (this == null) {
                throw new TypeError('"this" is null or not defined');
            }
            // 1. Let O be ? ToObject(this value).
            var o = Object(this);
            // 2. Let len be ? ToLength(? Get(O, "length")).
            var len = o.length >>> 0;
            // 3. If len is 0, return false.
            if (len === 0) {
                return false;
            }
            // 4. Let n be ? ToInteger(fromIndex).
            //    (If fromIndex is undefined, this step produces the value 0.)
            var n = fromIndex | 0;
            // 5. If n ≥ 0, then
            //  a. Let k be n.
            // 6. Else n < 0,
            //  a. Let k be len + n.
            //  b. If k < 0, let k be 0.
            var k = Math.max(n >= 0 ? n : len - Math.abs(n), 0);
            function sameValueZero(x, y) {
                return (x === y ||
                    (typeof x === 'number' && typeof y === 'number' && isNaN(x) && isNaN(y)));
            }
            // 7. Repeat, while k < len
            while (k < len) {
                // a. Let elementK be the result of ? Get(O, ! ToString(k)).
                // b. If SameValueZero(searchElement, elementK) is true, return true.
                if (sameValueZero(o[k], searchElement)) {
                    return true;
                }
                // c. Increase k by 1.
                k++;
            }
            // 8. Return false
            return false;
        }
    });
}
function findMainSourceFolder(files) {
    var mainFolder = '';
    var mainFolderCount = 0;
    var rawFolders = files.map(function (filepath) {
        var shortPath = filepath.replace(process.cwd() + path.sep, '');
        return path.dirname(shortPath);
    });
    var folders = {};
    rawFolders = _.uniq(rawFolders);
    for (var i = 0; i < rawFolders.length; i++) {
        var sep = rawFolders[i].split(path.sep);
        sep.forEach(function (folder) {
            if (folders[folder]) {
                folders[folder] += 1;
            }
            else {
                folders[folder] = 1;
            }
        });
    }
    for (var f in folders) {
        if (folders[f] > mainFolderCount) {
            mainFolderCount = folders[f];
            mainFolder = f;
        }
    }
    return mainFolder;
}
// Create a compilerHost object to allow the compiler to read and write files
function compilerHost(transpileOptions) {
    var inputFileName = transpileOptions.fileName || (transpileOptions.jsx ? 'module.tsx' : 'module.ts');
    var toReturn = {
        getSourceFile: function (fileName) {
            if (fileName.lastIndexOf('.ts') !== -1 || fileName.lastIndexOf('.js') !== -1) {
                if (fileName === 'lib.d.ts') {
                    return undefined;
                }
                if (fileName.substr(-5) === '.d.ts') {
                    return undefined;
                }
                if (path.isAbsolute(fileName) === false) {
                    fileName = path.join(transpileOptions.tsconfigDirectory, fileName);
                }
                if (!fs.existsSync(fileName)) {
                    return undefined;
                }
                var libSource = '';
                try {
                    libSource = fs.readFileSync(fileName).toString();
                    if (hasBom(libSource)) {
                        libSource = stripBom(libSource);
                    }
                }
                catch (e) {
                    logger.debug(e, fileName);
                }
                return Ast.ts.createSourceFile(fileName, libSource, transpileOptions.target, false);
            }
            return undefined;
        },
        writeFile: function (name, text) { },
        getDefaultLibFileName: function () { return 'lib.d.ts'; },
        useCaseSensitiveFileNames: function () { return false; },
        getCanonicalFileName: function (fileName) { return fileName; },
        getCurrentDirectory: function () { return ''; },
        getNewLine: function () { return '\n'; },
        fileExists: function (fileName) { return fileName === inputFileName; },
        readFile: function () { return ''; },
        directoryExists: function () { return true; },
        getDirectories: function () { return []; }
    };
    return toReturn;
}
function detectIndent(str, count) {
    var stripIndent = function (stripedString) {
        var match = stripedString.match(/^[ \t]*(?=\S)/gm);
        if (!match) {
            return stripedString;
        }
        // TODO: use spread operator when targeting Node.js 6
        var indent = Math.min.apply(Math, match.map(function (x) { return x.length; })); // eslint-disable-line
        var re = new RegExp("^[ \\t]{" + indent + "}", 'gm');
        return indent > 0 ? stripedString.replace(re, '') : stripedString;
    };
    var repeating = function (n, repeatString) {
        repeatString = repeatString === undefined ? ' ' : repeatString;
        if (typeof repeatString !== 'string') {
            throw new TypeError("Expected `input` to be a `string`, got `" + typeof repeatString + "`");
        }
        if (n < 0) {
            throw new TypeError("Expected `count` to be a positive finite number, got `" + n + "`");
        }
        var ret = '';
        do {
            if (n & 1) {
                ret += repeatString;
            }
            repeatString += repeatString;
        } while ((n >>= 1));
        return ret;
    };
    var indentString = function (indentedString, indentCount) {
        var indent = ' ';
        indentCount = indentCount === undefined ? 1 : indentCount;
        if (typeof indentedString !== 'string') {
            throw new TypeError("Expected `input` to be a `string`, got `" + typeof indentedString + "`");
        }
        if (typeof indentCount !== 'number') {
            throw new TypeError("Expected `count` to be a `number`, got `" + typeof indentCount + "`");
        }
        if (typeof indent !== 'string') {
            throw new TypeError("Expected `indent` to be a `string`, got `" + typeof indent + "`");
        }
        if (indentCount === 0) {
            return indentedString;
        }
        indent = indentCount > 1 ? repeating(indentCount, indent) : indent;
        return indentedString.replace(/^(?!\s*$)/gm, indent);
    };
    return indentString(stripIndent(str), count || 0);
}

var traverse = require('traverse');
var DependenciesEngine = /** @class */ (function () {
    function DependenciesEngine() {
        this.miscellaneous = {
            variables: [],
            functions: [],
            typealiases: [],
            enumerations: [],
            groupedVariables: [],
            groupedFunctions: [],
            groupedEnumerations: [],
            groupedTypeAliases: []
        };
    }
    DependenciesEngine.getInstance = function () {
        if (!DependenciesEngine.instance) {
            DependenciesEngine.instance = new DependenciesEngine();
        }
        return DependenciesEngine.instance;
    };
    DependenciesEngine.prototype.updateModulesDeclarationsExportsTypes = function () {
        var _this = this;
        var mergeTypes = function (entry) {
            var directive = _this.findInCompodocDependencies(entry.name, _this.directives, entry.file);
            if (typeof directive.data !== 'undefined') {
                entry.type = 'directive';
                entry.id = directive.data.id;
            }
            var component = _this.findInCompodocDependencies(entry.name, _this.components, entry.file);
            if (typeof component.data !== 'undefined') {
                entry.type = 'component';
                entry.id = component.data.id;
            }
            var pipe = _this.findInCompodocDependencies(entry.name, _this.pipes, entry.file);
            if (typeof pipe.data !== 'undefined') {
                entry.type = 'pipe';
                entry.id = pipe.data.id;
            }
        };
        this.modules.forEach(function (module) {
            module.declarations.forEach(function (declaration) {
                mergeTypes(declaration);
            });
            module.exports.forEach(function (expt) {
                mergeTypes(expt);
            });
            module.entryComponents.forEach(function (ent) {
                mergeTypes(ent);
            });
        });
    };
    DependenciesEngine.prototype.init = function (data) {
        traverse(data).forEach(function (node) {
            if (node) {
                if (node.parent) {
                    delete node.parent;
                }
                if (node.initializer) {
                    delete node.initializer;
                }
            }
        });
        this.rawData = data;
        this.modules = _.sortBy(this.rawData.modules, [function (el) { return el.name.toLowerCase(); }]);
        this.rawModulesForOverview = _.sortBy(data.modulesForGraph, [function (el) { return el.name.toLowerCase(); }]);
        this.rawModules = _.sortBy(data.modulesForGraph, [function (el) { return el.name.toLowerCase(); }]);
        this.components = _.sortBy(this.rawData.components, [function (el) { return el.name.toLowerCase(); }]);
        this.controllers = _.sortBy(this.rawData.controllers, [function (el) { return el.name.toLowerCase(); }]);
        this.directives = _.sortBy(this.rawData.directives, [function (el) { return el.name.toLowerCase(); }]);
        this.injectables = _.sortBy(this.rawData.injectables, [function (el) { return el.name.toLowerCase(); }]);
        this.interceptors = _.sortBy(this.rawData.interceptors, [function (el) { return el.name.toLowerCase(); }]);
        this.guards = _.sortBy(this.rawData.guards, [function (el) { return el.name.toLowerCase(); }]);
        this.interfaces = _.sortBy(this.rawData.interfaces, [function (el) { return el.name.toLowerCase(); }]);
        this.pipes = _.sortBy(this.rawData.pipes, [function (el) { return el.name.toLowerCase(); }]);
        this.classes = _.sortBy(this.rawData.classes, [function (el) { return el.name.toLowerCase(); }]);
        this.miscellaneous = this.rawData.miscellaneous;
        this.prepareMiscellaneous();
        this.updateModulesDeclarationsExportsTypes();
        this.routes = this.rawData.routesTree;
        this.manageDuplicatesName();
        this.cleanRawModulesNames();
    };
    DependenciesEngine.prototype.cleanRawModulesNames = function () {
        this.rawModulesForOverview = this.rawModulesForOverview.map(function (module) {
            module.name = module.name.replace('$', '');
            return module;
        });
    };
    DependenciesEngine.prototype.findInCompodocDependencies = function (name, data, file) {
        var _result = {
            source: 'internal',
            data: undefined
        };
        var nameFoundCounter = 0;
        if (data && data.length > 0) {
            for (var i = 0; i < data.length; i++) {
                if (typeof name !== 'undefined') {
                    if (typeof file !== 'undefined') {
                        if (name.indexOf(data[i].name) !== -1 &&
                            file.replace(/\\/g, '/').indexOf(data[i].file) !== -1) {
                            nameFoundCounter += 1;
                            _result.data = data[i];
                        }
                    }
                    else {
                        if (name.indexOf(data[i].name) !== -1) {
                            nameFoundCounter += 1;
                            _result.data = data[i];
                        }
                    }
                }
            }
            // Prevent wrong matching like MultiSelectOptionDirective with SelectOptionDirective, or QueryParamGroupService with QueryParamGroup
            if (nameFoundCounter > 1) {
                var found = false;
                for (var i = 0; i < data.length; i++) {
                    if (typeof name !== 'undefined') {
                        if (typeof file !== 'undefined') {
                            if (name === data[i].name &&
                                file.replace(/\\/g, '/').indexOf(data[i].file) !== -1) {
                                found = true;
                                _result.data = data[i];
                            }
                        }
                        else {
                            if (name === data[i].name) {
                                found = true;
                                _result.data = data[i];
                            }
                        }
                    }
                }
                if (!found) {
                    _result = {
                        source: 'internal',
                        data: undefined
                    };
                }
            }
        }
        return _result;
    };
    DependenciesEngine.prototype.manageDuplicatesName = function () {
        var processDuplicates = function (element, index, array) {
            var elementsWithSameName = _.filter(array, { name: element.name });
            if (elementsWithSameName.length > 1) {
                // First element is the reference for duplicates
                for (var i = 1; i < elementsWithSameName.length; i++) {
                    var elementToEdit = elementsWithSameName[i];
                    if (typeof elementToEdit.isDuplicate === 'undefined') {
                        elementToEdit.isDuplicate = true;
                        elementToEdit.duplicateId = i;
                        elementToEdit.duplicateName =
                            elementToEdit.name + '-' + elementToEdit.duplicateId;
                        elementToEdit.id = elementToEdit.id + '-' + elementToEdit.duplicateId;
                    }
                }
            }
            return element;
        };
        this.classes = this.classes.map(processDuplicates);
        this.interfaces = this.interfaces.map(processDuplicates);
        this.injectables = this.injectables.map(processDuplicates);
        this.pipes = this.pipes.map(processDuplicates);
        this.interceptors = this.interceptors.map(processDuplicates);
        this.guards = this.guards.map(processDuplicates);
        this.modules = this.modules.map(processDuplicates);
        this.components = this.components.map(processDuplicates);
        this.controllers = this.controllers.map(processDuplicates);
        this.directives = this.directives.map(processDuplicates);
    };
    DependenciesEngine.prototype.find = function (name) {
        var _this = this;
        var searchFunctions = [
            function () { return _this.findInCompodocDependencies(name, _this.injectables); },
            function () { return _this.findInCompodocDependencies(name, _this.interceptors); },
            function () { return _this.findInCompodocDependencies(name, _this.guards); },
            function () { return _this.findInCompodocDependencies(name, _this.interfaces); },
            function () { return _this.findInCompodocDependencies(name, _this.classes); },
            function () { return _this.findInCompodocDependencies(name, _this.components); },
            function () { return _this.findInCompodocDependencies(name, _this.controllers); },
            function () { return _this.findInCompodocDependencies(name, _this.miscellaneous.variables); },
            function () { return _this.findInCompodocDependencies(name, _this.miscellaneous.functions); },
            function () { return _this.findInCompodocDependencies(name, _this.miscellaneous.typealiases); },
            function () { return _this.findInCompodocDependencies(name, _this.miscellaneous.enumerations); },
            function () { return AngularApiUtil$1.findApi(name); }
        ];
        for (var _i = 0, searchFunctions_1 = searchFunctions; _i < searchFunctions_1.length; _i++) {
            var searchFunction = searchFunctions_1[_i];
            var result = searchFunction();
            if (result.data) {
                return result;
            }
        }
        return undefined;
    };
    DependenciesEngine.prototype.update = function (updatedData) {
        var _this = this;
        if (updatedData.modules.length > 0) {
            _.forEach(updatedData.modules, function (module) {
                var _index = _.findIndex(_this.modules, { name: module.name });
                _this.modules[_index] = module;
            });
        }
        if (updatedData.components.length > 0) {
            _.forEach(updatedData.components, function (component) {
                var _index = _.findIndex(_this.components, { name: component.name });
                _this.components[_index] = component;
            });
        }
        if (updatedData.controllers.length > 0) {
            _.forEach(updatedData.controllers, function (controller) {
                var _index = _.findIndex(_this.controllers, { name: controller.name });
                _this.controllers[_index] = controller;
            });
        }
        if (updatedData.directives.length > 0) {
            _.forEach(updatedData.directives, function (directive) {
                var _index = _.findIndex(_this.directives, { name: directive.name });
                _this.directives[_index] = directive;
            });
        }
        if (updatedData.injectables.length > 0) {
            _.forEach(updatedData.injectables, function (injectable) {
                var _index = _.findIndex(_this.injectables, { name: injectable.name });
                _this.injectables[_index] = injectable;
            });
        }
        if (updatedData.interceptors.length > 0) {
            _.forEach(updatedData.interceptors, function (interceptor) {
                var _index = _.findIndex(_this.interceptors, { name: interceptor.name });
                _this.interceptors[_index] = interceptor;
            });
        }
        if (updatedData.guards.length > 0) {
            _.forEach(updatedData.guards, function (guard) {
                var _index = _.findIndex(_this.guards, { name: guard.name });
                _this.guards[_index] = guard;
            });
        }
        if (updatedData.interfaces.length > 0) {
            _.forEach(updatedData.interfaces, function (int) {
                var _index = _.findIndex(_this.interfaces, { name: int.name });
                _this.interfaces[_index] = int;
            });
        }
        if (updatedData.pipes.length > 0) {
            _.forEach(updatedData.pipes, function (pipe) {
                var _index = _.findIndex(_this.pipes, { name: pipe.name });
                _this.pipes[_index] = pipe;
            });
        }
        if (updatedData.classes.length > 0) {
            _.forEach(updatedData.classes, function (classe) {
                var _index = _.findIndex(_this.classes, { name: classe.name });
                _this.classes[_index] = classe;
            });
        }
        /**
         * Miscellaneous update
         */
        if (updatedData.miscellaneous.variables.length > 0) {
            _.forEach(updatedData.miscellaneous.variables, function (variable) {
                var _index = _.findIndex(_this.miscellaneous.variables, {
                    name: variable.name,
                    file: variable.file
                });
                _this.miscellaneous.variables[_index] = variable;
            });
        }
        if (updatedData.miscellaneous.functions.length > 0) {
            _.forEach(updatedData.miscellaneous.functions, function (func) {
                var _index = _.findIndex(_this.miscellaneous.functions, {
                    name: func.name,
                    file: func.file
                });
                _this.miscellaneous.functions[_index] = func;
            });
        }
        if (updatedData.miscellaneous.typealiases.length > 0) {
            _.forEach(updatedData.miscellaneous.typealiases, function (typealias) {
                var _index = _.findIndex(_this.miscellaneous.typealiases, {
                    name: typealias.name,
                    file: typealias.file
                });
                _this.miscellaneous.typealiases[_index] = typealias;
            });
        }
        if (updatedData.miscellaneous.enumerations.length > 0) {
            _.forEach(updatedData.miscellaneous.enumerations, function (enumeration) {
                var _index = _.findIndex(_this.miscellaneous.enumerations, {
                    name: enumeration.name,
                    file: enumeration.file
                });
                _this.miscellaneous.enumerations[_index] = enumeration;
            });
        }
        this.prepareMiscellaneous();
    };
    DependenciesEngine.prototype.findInCompodoc = function (name) {
        var mergedData = _.concat([], this.modules, this.components, this.controllers, this.directives, this.injectables, this.interceptors, this.guards, this.interfaces, this.pipes, this.classes, this.miscellaneous.enumerations, this.miscellaneous.typealiases, this.miscellaneous.variables, this.miscellaneous.functions);
        var result = _.find(mergedData, { name: name });
        return result || false;
    };
    DependenciesEngine.prototype.prepareMiscellaneous = function () {
        this.miscellaneous.variables.sort(getNamesCompareFn());
        this.miscellaneous.functions.sort(getNamesCompareFn());
        this.miscellaneous.enumerations.sort(getNamesCompareFn());
        this.miscellaneous.typealiases.sort(getNamesCompareFn());
        // group each subgoup by file
        this.miscellaneous.groupedVariables = _.groupBy(this.miscellaneous.variables, 'file');
        this.miscellaneous.groupedFunctions = _.groupBy(this.miscellaneous.functions, 'file');
        this.miscellaneous.groupedEnumerations = _.groupBy(this.miscellaneous.enumerations, 'file');
        this.miscellaneous.groupedTypeAliases = _.groupBy(this.miscellaneous.typealiases, 'file');
    };
    DependenciesEngine.prototype.getModule = function (name) {
        return _.find(this.modules, ['name', name]);
    };
    DependenciesEngine.prototype.getRawModule = function (name) {
        return _.find(this.rawModules, ['name', name]);
    };
    DependenciesEngine.prototype.getModules = function () {
        return this.modules;
    };
    DependenciesEngine.prototype.getComponents = function () {
        return this.components;
    };
    DependenciesEngine.prototype.getControllers = function () {
        return this.controllers;
    };
    DependenciesEngine.prototype.getDirectives = function () {
        return this.directives;
    };
    DependenciesEngine.prototype.getInjectables = function () {
        return this.injectables;
    };
    DependenciesEngine.prototype.getInterceptors = function () {
        return this.interceptors;
    };
    DependenciesEngine.prototype.getGuards = function () {
        return this.guards;
    };
    DependenciesEngine.prototype.getInterfaces = function () {
        return this.interfaces;
    };
    DependenciesEngine.prototype.getRoutes = function () {
        return this.routes;
    };
    DependenciesEngine.prototype.getPipes = function () {
        return this.pipes;
    };
    DependenciesEngine.prototype.getClasses = function () {
        return this.classes;
    };
    DependenciesEngine.prototype.getMiscellaneous = function () {
        return this.miscellaneous;
    };
    return DependenciesEngine;
}());
var DependenciesEngine$1 = DependenciesEngine.getInstance();

var FileEngine = /** @class */ (function () {
    function FileEngine() {
    }
    FileEngine.getInstance = function () {
        if (!FileEngine.instance) {
            FileEngine.instance = new FileEngine();
        }
        return FileEngine.instance;
    };
    FileEngine.prototype.get = function (filepath) {
        return new Promise(function (resolve, reject) {
            fs.readFile(path.resolve(filepath), 'utf8', function (err, data) {
                if (err) {
                    reject('Error during ' + filepath + ' read');
                }
                else {
                    resolve(data);
                }
            });
        });
    };
    FileEngine.prototype.write = function (filepath, contents) {
        return new Promise(function (resolve, reject) {
            fs.outputFile(path.resolve(filepath), contents, function (err) {
                if (err) {
                    reject(err);
                }
                else {
                    resolve();
                }
            });
        });
    };
    FileEngine.prototype.writeSync = function (filepath, contents) {
        fs.outputFileSync(filepath, contents);
    };
    FileEngine.prototype.getSync = function (filepath) {
        return fs.readFileSync(path.resolve(filepath), 'utf8');
    };
    /**
     * @param file The file to check
     */
    FileEngine.prototype.existsSync = function (file) {
        return fs.existsSync(file);
    };
    return FileEngine;
}());
var FileEngine$1 = FileEngine.getInstance();

var traverse$1 = require('traverse');
var ExportJsonEngine = /** @class */ (function () {
    function ExportJsonEngine() {
    }
    ExportJsonEngine.getInstance = function () {
        if (!ExportJsonEngine.instance) {
            ExportJsonEngine.instance = new ExportJsonEngine();
        }
        return ExportJsonEngine.instance;
    };
    ExportJsonEngine.prototype["export"] = function (outputFolder, data) {
        var exportData = {};
        traverse$1(data).forEach(function (node) {
            if (node) {
                if (node.parent) {
                    delete node.parent;
                }
                if (node.initializer) {
                    delete node.initializer;
                }
                if (Configuration$1.mainData.disableSourceCode) {
                    delete node.sourceCode;
                }
            }
        });
        exportData.pipes = data.pipes;
        exportData.interfaces = data.interfaces;
        exportData.injectables = data.injectables;
        exportData.classes = data.classes;
        exportData.directives = data.directives;
        exportData.components = data.components;
        exportData.modules = this.processModules();
        exportData.miscellaneous = data.miscellaneous;
        if (!Configuration$1.mainData.disableRoutesGraph) {
            exportData.routes = data.routes;
        }
        if (!Configuration$1.mainData.disableCoverage) {
            exportData.coverage = data.coverageData;
        }
        return FileEngine$1.write(outputFolder + path.sep + '/documentation.json', JSON.stringify(exportData, undefined, 4))["catch"](function (err) {
            logger.error('Error during export file generation ', err);
            return Promise.reject(err);
        });
    };
    ExportJsonEngine.prototype.processModules = function () {
        var modules = DependenciesEngine$1.getModules();
        var _resultedModules = [];
        for (var moduleNr = 0; moduleNr < modules.length; moduleNr++) {
            var moduleElement = {
                name: modules[moduleNr].name,
                children: [
                    {
                        type: 'providers',
                        elements: []
                    },
                    {
                        type: 'declarations',
                        elements: []
                    },
                    {
                        type: 'imports',
                        elements: []
                    },
                    {
                        type: 'exports',
                        elements: []
                    },
                    {
                        type: 'bootstrap',
                        elements: []
                    },
                    {
                        type: 'classes',
                        elements: []
                    }
                ]
            };
            for (var k = 0; k < modules[moduleNr].providers.length; k++) {
                var providerElement = {
                    name: modules[moduleNr].providers[k].name
                };
                moduleElement.children[0].elements.push(providerElement);
            }
            for (var k = 0; k < modules[moduleNr].declarations.length; k++) {
                var declarationElement = {
                    name: modules[moduleNr].declarations[k].name
                };
                moduleElement.children[1].elements.push(declarationElement);
            }
            for (var k = 0; k < modules[moduleNr].imports.length; k++) {
                var importElement = {
                    name: modules[moduleNr].imports[k].name
                };
                moduleElement.children[2].elements.push(importElement);
            }
            for (var k = 0; k < modules[moduleNr].exports.length; k++) {
                var exportElement = {
                    name: modules[moduleNr].exports[k].name
                };
                moduleElement.children[3].elements.push(exportElement);
            }
            for (var k = 0; k < modules[moduleNr].bootstrap.length; k++) {
                var bootstrapElement = {
                    name: modules[moduleNr].bootstrap[k].name
                };
                moduleElement.children[4].elements.push(bootstrapElement);
            }
            _resultedModules.push(moduleElement);
        }
        return _resultedModules;
    };
    return ExportJsonEngine;
}());
var ExportJsonEngine$1 = ExportJsonEngine.getInstance();

var TRANSLATION_DE_DE = {
    accessors: 'Accessors',
    arguments: 'Arguments',
    bootstrap: 'Bootstrap',
    branches: 'Branches',
    browse: 'Browse',
    classe: 'Klasse',
    classes: 'Klassen',
    component: 'Component',
    components: 'Components',
    constructor: 'Konstruktor',
    controllers: 'Controllers',
    controller: 'Controller',
    'coverage-page-title': 'Dokumentation Abdeckung',
    declarations: 'Declarations',
    decorators: 'Decorators',
    'default-value': 'Default value',
    'defined-in': 'Definiert in',
    dependencies: 'Abhängigkeiten',
    description: 'Beschreibung',
    directive: 'Directive',
    directives: 'Directives',
    entrycomponents: 'EntryComponents',
    enumerations: 'Enumerations',
    enums: 'Enums',
    example: 'Beispiel',
    exports: 'Exports',
    "extends": 'Extends',
    file: 'Datei',
    functions: 'Funktionen',
    'generated-using': 'Dokumentation generiert mit',
    'getting-started': "Los geht's",
    guard: 'Guard',
    guards: 'Guards',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'Html Element',
    'html-element-with-directive': 'Html Element mit directive',
    identifier: 'Identifier',
    implements: 'Implements',
    imports: 'Importe',
    index: 'Index',
    indexable: 'Indexable',
    'inherited-from': 'Inherited from',
    injectable: 'Injectable',
    injectables: 'Injectables',
    inputs: 'Inputs',
    interceptors: 'Interceptors',
    interface: 'Interface',
    interfaces: 'Interfaces',
    legend: 'Legend',
    license: 'Lizenz',
    lines: 'Lines',
    metadata: 'Metadata',
    methods: 'Methoden',
    miscellaneous: 'Verschiedenes',
    module: 'Module',
    modules: 'Modules',
    name: 'Name',
    no: 'Nein',
    'no-graph': 'Kein Graph verfügbar.',
    'no-iframe': 'Dein Browser unterstützt keine iframes.',
    'no-result-matching': 'Kein passendes Ergebnis',
    'no-svg': 'Dein Browser unterstützt kein SVG',
    optional: 'Optional',
    outputs: 'Ausgaben',
    overview: 'Übersicht',
    parameters: 'Parameter',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Prefix',
    properties: 'Properties',
    providers: 'Providers',
    pure: 'Pure',
    readme: 'README',
    reset: 'Zurücksetzen',
    'results-matching': 'übereinstimmende Ergebnisse',
    returns: 'Returns',
    route: 'Route',
    routes: 'Routes',
    schemas: 'Schemas',
    'search-placeholder': 'Eingeben zur Suche',
    selector: 'Selector',
    signature: 'Unterschrift',
    statements: 'Statements',
    type: 'Type',
    'type-aliases': 'Type aliases',
    'type-parameters': 'Type parameters',
    types: 'Types',
    'unamed-property': 'Unamed property',
    'unit-test-coverage': 'Unit test coverage',
    value: 'Value',
    variables: 'Variablen',
    yes: 'Ja',
    zoomin: 'Vergrößern',
    zoomout: 'Verkleinern'
};

var TRANSLATION_EN_US = {
    accessors: 'Accessors',
    arguments: 'Arguments',
    bootstrap: 'Bootstrap',
    branches: 'Branches',
    browse: 'Browse',
    classe: 'Class',
    classes: 'Classes',
    component: 'Component',
    components: 'Components',
    constructor: 'Constructor',
    controllers: 'Controllers',
    controller: 'Controller',
    'coverage-page-title': 'Documentation coverage',
    declarations: 'Declarations',
    decorators: 'Decorators',
    'default-value': 'Default value',
    'defined-in': 'Defined in',
    dependencies: 'Dependencies',
    description: 'Description',
    directive: 'Directive',
    directives: 'Directives',
    entrycomponents: 'EntryComponents',
    enumerations: 'Enumerations',
    enums: 'Enums',
    example: 'Example',
    exports: 'Exports',
    "extends": 'Extends',
    file: 'File',
    functions: 'Functions',
    'generated-using': 'Documentation generated using',
    'getting-started': 'Getting started',
    guard: 'Guard',
    guards: 'Guards',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'Html element',
    'html-element-with-directive': 'Html element with directive',
    identifier: 'Identifier',
    implements: 'Implements',
    imports: 'Imports',
    index: 'Index',
    indexable: 'Indexable',
    'inherited-from': 'Inherited from',
    injectable: 'Injectable',
    injectables: 'Injectables',
    inputs: 'Inputs',
    interceptors: 'Interceptors',
    interface: 'Interface',
    interfaces: 'Interfaces',
    legend: 'Legend',
    license: 'License',
    lines: 'Lines',
    metadata: 'Metadata',
    methods: 'Methods',
    miscellaneous: 'Miscellaneous',
    module: 'Module',
    modules: 'Modules',
    name: 'Name',
    no: 'No',
    'no-graph': 'No graph available.',
    'no-iframe': 'Your browser does not support iframes.',
    'no-result-matching': 'No results matching',
    'no-svg': 'Your browser does not support SVG',
    optional: 'Optional',
    outputs: 'Outputs',
    overview: 'Overview',
    parameters: 'Parameters',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Prefix',
    properties: 'Properties',
    providers: 'Providers',
    pure: 'Pure',
    readme: 'README',
    reset: 'Reset',
    'results-matching': 'results matching',
    returns: 'Returns',
    route: 'Route',
    routes: 'Routes',
    schemas: 'Schemas',
    'search-placeholder': 'Type to search',
    selector: 'Selector',
    signature: 'Signature',
    statements: 'Statements',
    type: 'Type',
    'type-aliases': 'Type aliases',
    'type-parameters': 'Type parameters',
    types: 'Types',
    'unamed-property': 'Unamed property',
    'unit-test-coverage': 'Unit test coverage',
    value: 'Value',
    variables: 'Variables',
    yes: 'Yes',
    zoomin: 'Zoom in',
    zoomout: 'Zoom out'
};

var TRANSLATION_ES_ES = {
    accessors: 'Accesorios',
    arguments: 'Argumentos',
    bootstrap: 'Arranque',
    branches: 'Ramas',
    browse: 'Navegar',
    classe: 'Clase',
    classes: 'Clases',
    component: 'Componente',
    components: 'Componentes',
    constructor: 'Constructor',
    controllers: 'Controladores',
    controller: 'Controlador',
    'coverage-page-title': 'Cobertura de la documentación',
    declarations: 'Declaraciones',
    decorators: 'Decoradores',
    'default-value': 'Valor por defecto',
    'defined-in': 'Definido en',
    dependencies: 'Dependencias',
    description: 'Descripción',
    directive: 'Directiva',
    directives: 'Directivas',
    entrycomponents: 'Componentes de entrada',
    enumerations: 'Enumeraciones',
    enums: 'Enums',
    example: 'Ejemplo',
    exports: 'Exporta',
    "extends": 'Extiende',
    file: 'Fichero',
    functions: 'Funciones',
    'generated-using': 'Documentación generada utilizando',
    'getting-started': 'Comenzando',
    guard: 'Guardia',
    guards: 'Guardias',
    hostbindings: 'Fijaciones de Host',
    hostlisteners: 'Escuchadores de Host',
    'html-element': 'Elemento Html',
    'html-element-with-directive': 'Elemento Html con directiva',
    identifier: 'Identificador',
    implements: 'Implementa',
    imports: 'Importa',
    index: 'Índice',
    indexable: 'Indexable',
    'inherited-from': 'Heredado desde',
    injectable: 'Inyectable',
    injectables: 'Inyectables',
    inputs: 'Entradas',
    interceptors: 'Interceptores',
    interface: 'Interfaz',
    interfaces: 'Interfaces',
    legend: 'Leyenda',
    license: 'Licencia',
    lines: 'Líneas',
    metadata: 'Meta datos',
    methods: 'Métodos',
    miscellaneous: 'Miscelánea',
    module: 'Módulo',
    modules: 'Módulos',
    name: 'Nombre',
    no: 'No',
    'no-graph': 'No hay gráfica disponible.',
    'no-iframe': 'Tu navegador no soporta iframes.',
    'no-result-matching': 'No hay resultados que coincidan',
    'no-svg': 'Tu navegador no soporta SVG',
    optional: 'Opcional',
    outputs: 'Salidas',
    overview: 'Descripción general',
    parameters: 'Parámetros',
    'peer-dependencies': 'Dependencias entre pares',
    pipe: 'Tubería',
    pipes: 'Tuberías',
    prefix: 'Prefijo',
    properties: 'Propiedades',
    providers: 'Proveedores',
    pure: 'Puro',
    readme: 'Léeme',
    reset: 'Restablecer',
    'results-matching': 'comparación de resultados',
    returns: 'Devuelve',
    route: 'Ruta',
    routes: 'Rutas',
    schemas: 'Esquemas',
    'search-placeholder': 'Escribe para buscar',
    selector: 'Selector',
    signature: 'Firma',
    statements: 'Declaraciones',
    type: 'Tipo',
    'type-aliases': 'Alias de tipo',
    'type-parameters': 'Parámetros de tipo',
    types: 'Tipos',
    'unamed-property': 'Propiedad sin nombre',
    'unit-test-coverage': 'Cobertura de las pruebas unitarias',
    value: 'Valor',
    variables: 'Variables',
    yes: 'Si',
    zoomin: 'Ampliar',
    zoomout: 'Alejar'
};

var TRANSLATION_FR_FR = {
    accessors: 'Accesseurs',
    arguments: 'Arguments',
    bootstrap: 'Bootstrap',
    branches: 'Branches',
    browse: 'Parcourir',
    classe: 'Class',
    classes: 'Classes',
    component: 'Composant',
    components: 'Composants',
    constructor: 'Constructeur',
    controllers: 'Contrôleurs',
    controller: 'Contrôleur',
    'coverage-page-title': 'Couverture de documentation',
    declarations: 'Déclarations',
    decorators: 'Décorateurs',
    'default-value': 'Valeur par défaut',
    'defined-in': 'Défini dans',
    dependencies: 'Dépendances',
    description: 'Description',
    directive: 'Directive',
    directives: 'Directives',
    entrycomponents: "Composants d'entrée",
    enumerations: 'Enumérations',
    enums: 'Enumérations',
    example: 'Example',
    exports: 'Exports',
    "extends": 'Etend',
    file: 'Fichier',
    functions: 'Fonctions',
    'generated-using': 'Documentation générée avec',
    'getting-started': 'Démarrage',
    guard: 'Garde',
    guards: 'Gardes',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'Elément Html',
    'html-element-with-directive': 'Elément Html avec une directive',
    identifier: 'Identifiant',
    implements: 'Implémente',
    imports: 'Imports',
    index: 'Index',
    indexable: 'Indexable',
    'inherited-from': 'Hérité de',
    injectable: 'Injectable',
    injectables: 'Injectables',
    inputs: 'Entrées',
    interceptors: 'Intercepteurs',
    interface: 'Interface',
    interfaces: 'Interfaces',
    legend: 'Légende',
    license: 'License',
    lines: 'Lignes',
    metadata: 'Métadonnées',
    methods: 'Méthodes',
    miscellaneous: 'Divers',
    module: 'Module',
    modules: 'Modules',
    name: 'Nom',
    no: 'Non',
    'no-graph': 'Aucun graphique disponible.',
    'no-iframe': 'Votre navigateur ne supporte pas les iframes.',
    'no-result-matching': 'Aucun résultat matchant',
    'no-svg': 'Votre navigateur ne supporte pas le SVG',
    optional: 'Optionnel',
    outputs: 'Sorties',
    overview: "Vue d'ensemble",
    parameters: 'Paramètres',
    'peer-dependencies': 'Dépendances de pair',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Préfixe',
    properties: 'Propriétés',
    providers: 'Providers',
    pure: 'Pure',
    readme: 'README',
    reset: 'Remise à zéro',
    'results-matching': 'résultats matchant',
    returns: 'Renvoie',
    route: 'Route',
    routes: 'Routes',
    schemas: 'Schémas',
    'search-placeholder': 'Saisissez un texte',
    selector: 'Sélecteur',
    signature: 'Signature',
    statements: 'Déclarations',
    'table-of-contents': 'Table des matières',
    type: 'Type',
    'type-aliases': 'Alias de type',
    'type-parameters': 'Paramètres de type',
    types: 'Types',
    'unamed-property': 'Propriété non nommée',
    'unit-test-coverage': 'Couverture de test unitaire',
    value: 'Valeur',
    variables: 'Variables',
    yes: 'Oui',
    zoomin: 'Zoom avant',
    zoomout: 'Zoom arrière'
};

var TRANSLATION_HU_HU = {
    accessors: 'Getter/setter metódusok',
    arguments: 'Argumentumok',
    bootstrap: 'Betöltés',
    branches: 'Branchek',
    browse: 'Böngészés',
    classe: 'Osztály',
    classes: 'Osztályok',
    component: 'Komponens',
    components: 'Komponensek',
    constructor: 'Konstruktor',
    controllers: 'Kontrollerek',
    controller: 'Kontroller',
    'coverage-page-title': 'Dokumentáció lefedettség',
    declarations: 'Deklarációk',
    decorators: 'Dekorátorok',
    'default-value': 'Alapértelmezett érték',
    'defined-in': 'Definíció helye:',
    dependencies: 'Függőségek',
    description: 'Leírás',
    directive: 'Direktíva',
    directives: 'Direktívák',
    entrycomponents: 'Entry komponensek',
    enumerations: 'Enumerációk',
    enums: 'Enumok',
    example: 'Példa',
    exports: 'Exportok',
    "extends": 'Ősosztály',
    file: 'File',
    functions: 'Függvények',
    'generated-using': 'A dokumentációt generálta:',
    'getting-started': 'Bevezető',
    guard: 'Guard',
    guards: 'Guardok',
    hostbindings: 'HostBindingok',
    hostlisteners: 'HostListenerek',
    'html-element': 'Html elem',
    'html-element-with-directive': 'Html elem direktívával',
    identifier: 'Azonosító',
    implements: 'Implementált interfészek',
    imports: 'Importok',
    index: 'Tartalomjegyzék',
    indexable: 'Indexelhető',
    'inherited-from': 'Örökölve innen:',
    injectable: 'Injektálható',
    injectables: 'Injektálhatók',
    inputs: 'Bemenetek',
    interceptors: 'Interceptorok',
    interface: 'Interfész',
    interfaces: 'Interfészek',
    legend: 'Jelmagyarázat',
    license: 'Licenc',
    lines: 'Sorok',
    metadata: 'Metaadatok',
    methods: 'Metódusok',
    miscellaneous: 'Egyéb',
    module: 'Modul',
    modules: 'Modulok',
    name: 'Név',
    no: 'Nem',
    'no-graph': 'Grafikon nem elérhető.',
    'no-iframe': 'A böngészője nem támogatja az iframe-eket.',
    'no-result-matching': 'Nincs találat',
    'no-svg': 'A böngészője nem támogatja az SVG formátumot.',
    optional: 'Opcionális',
    outputs: 'Kimenetek',
    overview: 'Áttekintés',
    parameters: 'Paraméterek',
    'peer-dependencies': 'Peer függőségek',
    pipe: 'Pipe',
    pipes: 'Pipe-ok',
    prefix: 'Előtag',
    properties: 'Tagváltozók',
    providers: 'Providerek',
    pure: 'Pure',
    readme: 'README',
    reset: 'Visszaállít',
    'results-matching': 'találat',
    returns: 'Visszatérési érték',
    route: 'Útvonal',
    routes: 'Útvonalak',
    schemas: 'Sémák',
    'search-placeholder': 'Keresendő kifejezés',
    selector: 'Szelektor',
    signature: 'Aláírás',
    statements: 'Utasítások',
    type: 'Típus',
    'type-aliases': 'Típus álnév',
    'type-parameters': 'Típus paraméterek',
    types: 'Típusok',
    'unamed-property': 'Névtelen property',
    'unit-test-coverage': 'Unit teszt lefedettség',
    value: 'Érték',
    variables: 'Változók',
    yes: 'Igen',
    zoomin: 'Nagyítás',
    zoomout: 'Kicsinyítés'
};

var TRANSLATION_IT_IT = {
    accessors: 'Accessori',
    arguments: 'Argomenti',
    bootstrap: 'Bootstrap',
    branches: 'Rami',
    browse: 'Cerca',
    classe: 'Classe',
    classes: 'Classi',
    component: 'Componente',
    components: 'Componenti',
    constructor: 'Costruttore',
    controllers: 'Controllers',
    controller: 'Controller',
    'coverage-page-title': 'Copertura codice',
    declarations: 'Dichiarazioni',
    decorators: 'Decorators',
    'default-value': 'Valore predefinito',
    'defined-in': 'Definito in',
    dependencies: 'Dependencies',
    description: 'Descrizione',
    directive: 'Direttiva',
    directives: 'Direttive',
    entrycomponents: 'EntryComponents',
    enumerations: 'Enumerations',
    enums: 'Enums',
    example: 'Esempio',
    exports: 'Exports',
    "extends": 'Extends',
    file: 'File',
    functions: 'Funzioni',
    'generated-using': 'Documentazione generata usando',
    'getting-started': 'Iniziamo',
    guard: 'Guardia',
    guards: 'Guardie',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'Elemento Html',
    'html-element-with-directive': 'Elemento html con direttive',
    identifier: 'Identificatore',
    implements: 'Implementa',
    imports: 'Importa',
    index: 'Indice',
    indexable: 'Indicizzabile',
    'inherited-from': 'ereditato da',
    injectable: 'Injectable',
    injectables: 'Injectables',
    inputs: 'Input',
    interceptors: 'Interceptors',
    interface: 'Interfaccia',
    interfaces: 'Interfacce',
    legend: 'Legenda',
    license: 'Licenza',
    lines: 'Linee',
    metadata: 'Metadati',
    methods: 'Metodi',
    miscellaneous: 'Varie',
    module: 'Modulo',
    modules: 'Moduli',
    name: 'Nome',
    no: 'No',
    'no-graph': 'Grafico non disponibile.',
    'no-iframe': 'Il tuo browser non supporta iframe.',
    'no-result-matching': 'Nessun risultato corrispondente',
    'no-svg': 'Il tuo browser non supporta SVG',
    optional: 'Opzionale',
    outputs: 'Output',
    overview: 'Sommario',
    parameters: 'Parametri',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Prefisso',
    properties: 'Proprietà',
    providers: 'Providers',
    pure: 'Pure',
    readme: 'README',
    reset: 'Reset',
    'results-matching': 'corrispondenza',
    returns: 'Returns',
    route: 'Route',
    routes: 'Routes',
    schemas: 'Schemas',
    'search-placeholder': 'Digita per avviare la ricerca',
    selector: 'Selector',
    signature: 'Signature',
    statements: 'Statements',
    type: 'Tipo',
    'type-aliases': 'Type aliases',
    'type-parameters': 'Type parameters',
    types: 'Tipi',
    'unamed-property': 'Proprietà senza nome',
    'unit-test-coverage': 'Copertura unit test',
    value: 'Valori',
    variables: 'Variabili',
    yes: 'Si',
    zoomin: 'Ingrandisci',
    zoomout: 'Rimpocciolisci'
};

var TRANSLATION_JA_JP = {
    accessors: 'アクセサ',
    arguments: '引数',
    bootstrap: 'ブートストラップ',
    branches: 'ブランチ',
    browse: 'ブラウズ',
    classe: 'クラス',
    classes: 'クラス',
    component: 'コンポーネント',
    components: 'コンポーネント',
    constructor: 'コンストラクタ',
    controllers: 'コントローラー',
    controller: 'コントローラー',
    'coverage-page-title': 'カバレッジ',
    declarations: '宣言',
    decorators: 'デコレーター',
    'default-value': '初期値',
    'defined-in': 'Defined in',
    dependencies: '依存関係',
    description: '説明',
    directive: 'ディレクティブ',
    directives: 'ディレクティブ',
    entrycomponents: 'エントリーコンポーネント',
    enumerations: '列挙型',
    enums: 'Enums',
    example: '例',
    exports: 'エクスポート',
    "extends": '継承',
    file: 'ファイル',
    functions: '関数',
    'generated-using': 'このドキュメントは以下を使用して生成されています',
    'getting-started': 'はじめに',
    guard: 'ガード',
    guards: 'ガード',
    hostbindings: 'ホストバインディング',
    hostlisteners: 'ホストリスナー',
    'html-element': 'Html要素',
    'html-element-with-directive': 'ディレクティブHtml要素',
    identifier: '識別子',
    implements: '実装',
    imports: 'インポート',
    index: '索引',
    indexable: 'インデクサブル',
    'inherited-from': 'Inherited from',
    injectable: 'インジェクタブル',
    injectables: 'インジェクタブル',
    inputs: '入力',
    interceptors: 'インターセプター',
    interface: 'インターフェイス',
    interfaces: 'インターフェイス',
    legend: '凡例',
    license: 'ライセンス',
    lines: '行数',
    metadata: 'メタデータ',
    methods: 'メソッド',
    miscellaneous: 'その他',
    module: 'モジュール',
    modules: 'モジュール',
    name: '名前',
    no: 'いいえ',
    'no-graph': '使用できるグラフがありません',
    'no-iframe': 'ブラウザがiframeを対応していません',
    'no-result-matching': '見つかりませんでした',
    'no-svg': 'ブラウザがSVGに対応してません',
    optional: 'オプション',
    outputs: '出力',
    overview: '概要',
    parameters: 'パラメータ',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'パイプ',
    pipes: 'パイプ',
    prefix: '接頭辞',
    properties: 'プロパティ',
    providers: 'プロバイダー',
    pure: 'Pure',
    readme: 'README',
    reset: 'リセット',
    'results-matching': '件の結果が一致しました',
    returns: '戻り値',
    route: 'ルート',
    routes: 'ルート',
    schemas: 'スキーマ',
    'search-placeholder': '入力して検索',
    selector: 'セレクタ',
    signature: 'シグネチャ',
    statements: '文',
    type: '型',
    'type-aliases': 'タイプエイリアス',
    'type-parameters': '型パラメーター',
    types: '型',
    'unamed-property': '匿名プロパティ',
    'unit-test-coverage': 'ユニットテストカバレッジ',
    value: '値',
    variables: '変数',
    yes: 'はい',
    zoomin: '拡大',
    zoomout: '縮小'
};

var TRANSLATION_NL_NL = {
    accessors: 'Accessors',
    arguments: 'Argumenten',
    bootstrap: 'Bootstrap',
    branches: 'Branches',
    browse: 'Browse',
    classe: 'Klasse',
    classes: 'Klassen',
    component: 'Component',
    components: 'Componenten',
    constructor: 'Constructor',
    controllers: 'Controllers',
    controller: 'Controller',
    'coverage-page-title': 'Documentatie coverage',
    declarations: 'Declaraties',
    decorators: 'Decorators',
    'default-value': 'Default waarde',
    'defined-in': 'Gedefinieerd in',
    dependencies: 'Dependencies',
    description: 'Omschrijving',
    directive: 'Directive',
    directives: 'Directives',
    entrycomponents: 'EntryComponents',
    enumerations: 'Enumerations',
    enums: 'Enums',
    example: 'Voorbeeld',
    exports: 'Exports',
    "extends": 'Extends',
    file: 'Bestand',
    functions: 'Functies',
    'generated-using': 'Documentatie gegenereed met',
    'getting-started': 'Aan de slag',
    guard: 'Guard',
    guards: 'Guards',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'Html element',
    'html-element-with-directive': 'Html element met directive',
    identifier: 'Identifier',
    implements: 'Implementeert',
    imports: 'Imports',
    index: 'Index',
    indexable: 'Indexeerbaar',
    'inherited-from': 'Inherited van',
    injectable: 'Injectable',
    injectables: 'Injectables',
    inputs: 'Inputs',
    interceptors: 'Interceptors',
    interface: 'Interface',
    interfaces: 'Interfaces',
    legend: 'Legenda',
    license: 'Licentie',
    lines: 'Regels',
    metadata: 'Metadata',
    methods: 'Methods',
    miscellaneous: 'Diversen',
    module: 'Module',
    modules: 'Modules',
    name: 'Naam',
    no: 'Nee',
    'no-graph': 'Geen diagram beschikbaar.',
    'no-iframe': 'Uw browser ondersteund geen iframes.',
    'no-result-matching': 'Geen overeenkomende resultaten',
    'no-svg': 'Uw browser ondersteund geen SVG',
    optional: 'Optioneel',
    outputs: 'Outputs',
    overview: 'Overzicht',
    parameters: 'Parameters',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Voorvoegsel',
    properties: 'Properties',
    providers: 'Providers',
    pure: 'Puur',
    readme: 'README',
    reset: 'Reset',
    'results-matching': 'overeenkomende resultaten',
    returns: 'Returns',
    route: 'Route',
    routes: 'Routes',
    schemas: "Schema's",
    'search-placeholder': 'Type om te zoeken',
    selector: 'Selector',
    signature: 'Handtekening',
    statements: 'Statements',
    type: 'Type',
    'type-aliases': 'Type aliassen',
    'type-parameters': 'Type parameters',
    types: 'Types',
    'unamed-property': 'Naamloze property',
    'unit-test-coverage': 'Unit test coverage',
    value: 'Waarde',
    variables: 'Variabelen',
    yes: 'Ja',
    zoomin: 'Zoom in',
    zoomout: 'Zoom uit'
};

var TRANSLATION_PT_BR = {
    accessors: 'Acessores',
    arguments: 'Argumentos',
    bootstrap: 'Bootstrap',
    branches: 'Branches',
    browse: 'Navegar',
    classe: 'Classe',
    classes: 'Classes',
    component: 'Componente',
    components: 'Componentes',
    constructor: 'Construtor',
    controllers: 'Controladores',
    controller: 'Controlador',
    'coverage-page-title': 'Cobertura da documentação',
    declarations: 'Declarações',
    decorators: 'Decoradores',
    'default-value': 'Valor padrão',
    'defined-in': 'Definido em',
    dependencies: 'Dependências',
    description: 'Descrição',
    directive: 'Diretiva',
    directives: 'Diretivas',
    entrycomponents: 'EntryComponents',
    enumerations: 'Enumerações',
    enums: 'Enums',
    example: 'Exemplo',
    exports: 'Exports',
    "extends": 'Extende',
    file: 'Arquivo',
    functions: 'Funções',
    'generated-using': 'Documentação gerada usando',
    'getting-started': 'Começando',
    guard: 'Guarda',
    guards: 'Guardas',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'Elemento HTML',
    'html-element-with-directive': 'Elemento HTML com diretiva',
    identifier: 'Identificador',
    implements: 'Implementa',
    imports: 'Imports',
    index: 'Index',
    indexable: 'Indexável',
    'inherited-from': 'Herdado de',
    injectable: 'Injetável',
    injectables: 'Injetáveis',
    inputs: 'Inputs',
    interceptors: 'Interceptors',
    interface: 'Interface',
    interfaces: 'Interfaces',
    legend: 'Legend',
    license: 'Licença',
    lines: 'Linhas',
    metadata: 'Metadata',
    methods: 'Métodos',
    miscellaneous: 'Miscelânea',
    module: 'Módulo',
    modules: 'Módulos',
    name: 'Nome',
    no: 'Não',
    'no-graph': 'Sem gráfico disponível.',
    'no-iframe': 'Seu browser não tem suporte a iframes.',
    'no-result-matching': 'Nenhum resultado correspondente',
    'no-svg': 'Seu browser não tem suporte a SVG',
    optional: 'Opcional',
    outputs: 'Outputs',
    overview: 'Visão geral',
    parameters: 'Parâmetros',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Prefixo',
    properties: 'Propriedades',
    providers: 'Providers',
    pure: 'Puro',
    readme: 'README',
    reset: 'Resetar',
    'results-matching': 'resultados correspondentes',
    returns: 'Retorna',
    route: 'Rota',
    routes: 'Rotas',
    schemas: 'Esquemas',
    'search-placeholder': 'Digite para pesquisar',
    selector: 'Seletor',
    signature: 'Assinatura',
    statements: 'Statements',
    type: 'Tipo',
    'type-aliases': 'Aliases de tipo',
    'type-parameters': 'Parâmetros de tipo',
    types: 'Tipos',
    'unamed-property': 'Propriedade não-nomeada',
    'unit-test-coverage': 'Cobertua de teste unitário',
    value: 'Valor',
    variables: 'Variáveis',
    yes: 'Sim',
    zoomin: 'Zoom in',
    zoomout: 'Zoom out'
};

var TRANSLATION_SK_SK = {
    accessors: 'Modifikátory prístupu',
    arguments: 'Argumenty',
    bootstrap: 'Bootstrap',
    branches: 'Vetvy',
    browse: 'Prezerať',
    classe: 'Trieda',
    classes: 'Triedy',
    component: 'Komponent',
    components: 'Komponenty',
    constructor: 'Konštruktor',
    controllers: 'Controllers',
    controller: 'Controller',
    'coverage-page-title': 'Pokrytie dokumentáciou',
    declarations: 'Deklarácie',
    decorators: 'Dekorátory',
    'default-value': 'Predvolená hodnota',
    'defined-in': 'Definované v',
    dependencies: 'Závislosti',
    description: 'Popis',
    directive: 'Direktíva',
    directives: 'Direktívy',
    entrycomponents: 'EntryComponents',
    enumerations: 'Enumerátory',
    enums: 'Enumerátory',
    example: 'Príklad',
    exports: 'Exporty',
    "extends": 'Rozširuje',
    file: 'Súbor',
    functions: 'Funkcie',
    'generated-using': 'Dokumentácia vytvorená pomocou',
    'getting-started': 'Začíname',
    guard: 'Guard',
    guards: 'Guards',
    hostbindings: 'HostBindings',
    hostlisteners: 'HostListeners',
    'html-element': 'HTML element',
    'html-element-with-directive': 'HTML element s direktívou',
    identifier: 'Identifikátor',
    implements: 'Implementuje',
    imports: 'Importuje',
    index: 'Index',
    indexable: 'Indexovateľný',
    'inherited-from': 'Zdedené od',
    injectable: 'Injectable',
    injectables: 'Injectables',
    inputs: 'Vstupy',
    interceptors: 'Interceptors',
    interface: 'Rozhranie',
    interfaces: 'Rozhrania',
    legend: 'Legenda',
    license: 'Licencia',
    lines: 'Riadky',
    metadata: 'Metadáta',
    methods: 'Metódy',
    miscellaneous: 'Rôzne',
    module: 'Modul',
    modules: 'Moduly',
    name: 'Názov',
    no: 'Nie',
    'no-graph': 'Nie je k dispozícii žiadny graf.',
    'no-iframe': 'Váš prehliadač nepodporuje iframe',
    'no-result-matching': 'Nenájdené žiadne výsledky pre',
    'no-svg': 'Váš prehliadač nepodporuje SVG',
    optional: 'Voliteľný',
    outputs: 'Výstupy',
    overview: 'Prehľad',
    parameters: 'Parametre',
    'peer-dependencies': 'Peer dependencies',
    pipe: 'Pipe',
    pipes: 'Pipes',
    prefix: 'Prefix',
    properties: 'Vlastnosti',
    providers: 'Providers',
    pure: 'Pure',
    readme: 'README',
    reset: 'Resetovať',
    'results-matching': 'výsledkov pre',
    returns: 'Návratová hodnota',
    route: 'Route',
    routes: 'Routes',
    schemas: 'Schémy',
    'search-placeholder': 'Zadajte hľadaný text',
    selector: 'Selektor',
    signature: 'Podpis',
    statements: 'Statements',
    type: 'Typ',
    'type-aliases': 'Type aliases',
    'type-parameters': 'Type parameters',
    types: 'Typy',
    'unamed-property': 'Nepomenovaný atribút',
    'unit-test-coverage': 'Pokrytie unit testami',
    value: 'Hodnota',
    variables: 'Premenné',
    yes: 'Áno',
    zoomin: 'Priblížiť',
    zoomout: 'Oddialiť'
};

var TRANSLATION_ZH_CN = {
    accessors: '存取器',
    arguments: 'Arguments',
    bootstrap: '根组件',
    branches: '分支',
    browse: '查看',
    classe: '类',
    classes: '类列表',
    component: '组件',
    components: '组件列表',
    constructor: '构造方法',
    controllers: 'Controllers',
    controller: 'Controller',
    'coverage-page-title': '文档概览',
    declarations: '可声明对象列表',
    decorators: '装饰器列表',
    'default-value': '缺省值',
    'defined-in': '被定义在',
    dependencies: '依赖项',
    description: '描述',
    directive: '指令',
    directives: '指令列表',
    entrycomponents: '入口组件列表',
    enumerations: '列举',
    enums: '枚举列表',
    example: '例子',
    exports: '导出',
    "extends": '继承',
    file: '文件',
    functions: '函数',
    'generated-using': '文档生成使用',
    'getting-started': '入门指南',
    guard: '路由守卫',
    guards: '路由守卫列表',
    hostbindings: '宿主绑定',
    hostlisteners: '宿主监听',
    'html-element': 'Html 元素',
    'html-element-with-directive': '带指令的Html元素',
    identifier: '标识符',
    implements: '实现',
    imports: '引入',
    index: '索引',
    indexable: 'Indexable',
    'inherited-from': '继承自',
    injectable: '可注入的',
    injectables: '可注入的',
    inputs: '输入属性',
    interceptors: '拦截器',
    interface: '接口',
    interfaces: '接口',
    legend: '图例',
    license: '许可协议',
    lines: 'Lines',
    metadata: '元数据',
    methods: '方法',
    miscellaneous: '其他',
    module: '模块',
    modules: '模块列表',
    name: '名称',
    no: '否',
    'no-graph': '无数据显示',
    'no-iframe': '你的浏览器不支持iframes',
    'no-result-matching': '无匹配的结果',
    'no-svg': '你的浏览器不支持SVG',
    optional: '可选的',
    outputs: '输出属性',
    overview: '概述',
    parameters: '参数列表',
    'peer-dependencies': '同级依赖',
    pipe: '管道',
    pipes: '管道列表',
    prefix: '字首',
    properties: '属性列表',
    providers: '提供商列表',
    pure: 'Pure',
    readme: '手册',
    reset: '重置',
    'results-matching': '匹配的结果',
    returns: '返回',
    route: '路由',
    routes: '路由列表',
    schemas: '模式',
    'search-placeholder': '请输入查询关键字',
    selector: '选择器',
    signature: '签名',
    statements: '注释',
    type: '类型',
    'type-aliases': '类型别名',
    'type-parameters': '类型参数',
    types: '类型',
    'unamed-property': '未命名属性',
    'unit-test-coverage': '单元测试概览',
    value: '值',
    variables: '变量',
    yes: '是',
    zoomin: '放大',
    zoomout: '缩小'
};

var I18nEngine = /** @class */ (function () {
    function I18nEngine() {
        this.availablesLanguages = {
            'de-DE': 'de-DE',
            'en-US': 'en-US',
            'es-ES': 'es-ES',
            'fr-FR': 'fr-FR',
            'hu-HU': 'hu-HU',
            'it-IT': 'it-IT',
            'ja-JP': 'ja-JP',
            'nl-NL': 'nl-NL',
            'pt-BR': 'pt-BR',
            'sk-SK': 'sk-SK',
            'zh-CN': 'zh-CN'
        };
        this.fallbackLanguage = 'en-US';
    }
    I18nEngine.getInstance = function () {
        if (!I18nEngine.instance) {
            I18nEngine.instance = new I18nEngine();
        }
        return I18nEngine.instance;
    };
    I18nEngine.prototype.init = function (language) {
        i18next.init({
            lng: language,
            fallbackLng: this.fallbackLanguage
        });
        i18next.addResources('de-DE', 'translation', TRANSLATION_DE_DE);
        i18next.addResources('en-US', 'translation', TRANSLATION_EN_US);
        i18next.addResources('es-ES', 'translation', TRANSLATION_ES_ES);
        i18next.addResources('fr-FR', 'translation', TRANSLATION_FR_FR);
        i18next.addResources('hu-HU', 'translation', TRANSLATION_HU_HU);
        i18next.addResources('it-IT', 'translation', TRANSLATION_IT_IT);
        i18next.addResources('ja-JP', 'translation', TRANSLATION_JA_JP);
        i18next.addResources('nl-NL', 'translation', TRANSLATION_NL_NL);
        i18next.addResources('pt-BR', 'translation', TRANSLATION_PT_BR);
        i18next.addResources('sk-SK', 'translation', TRANSLATION_SK_SK);
        i18next.addResources('zh-CN', 'translation', TRANSLATION_ZH_CN);
    };
    I18nEngine.prototype.translate = function (key) {
        return i18next.t(key);
    };
    I18nEngine.prototype.supportLanguage = function (language) {
        return typeof this.availablesLanguages[language] !== 'undefined';
    };
    return I18nEngine;
}());
var I18nEngine$1 = I18nEngine.getInstance();

var decache = require('decache');
var MarkdownToPDFEngine = /** @class */ (function () {
    function MarkdownToPDFEngine() {
        var _this = this;
        this.convertedTokens = [];
        decache('marked');
        this.markedInstance = require('marked');
        var renderer = new this.markedInstance.Renderer();
        renderer.strong = function (text) {
            // console.log('MarkdownToPDFEngine strong: ', text);
            return { text: text, bold: true };
        };
        renderer.em = function (text) {
            // console.log('MarkdownToPDFEngine em: ', text);
            _this.convertedTokens.push({ text: text, italics: true });
            return text;
        };
        renderer.paragraph = function (text) {
            // console.log('MarkdownToPDFEngine paragraph: ', text);
            return text;
        };
        // TODO Add custom parser... -> https://github.com/markedjs/marked/issues/504
        this.markedInstance.setOptions({
            renderer: renderer,
            gfm: true,
            breaks: false
        });
    }
    MarkdownToPDFEngine.getInstance = function () {
        if (!MarkdownToPDFEngine.instance) {
            MarkdownToPDFEngine.instance = new MarkdownToPDFEngine();
        }
        return MarkdownToPDFEngine.instance;
    };
    MarkdownToPDFEngine.prototype.convert = function (stringToConvert) {
        this.convertedTokens = [];
        // console.log('MarkdownToPDFEngine convert: ', stringToConvert);
        var tokens = this.markedInstance.lexer(stringToConvert);
        // console.log(tokens);
        var pdfmakeData = this.markedInstance.Parser.parse(tokens);
        // console.log(this.convertedTokens);
        var result = {
            text: this.convertedTokens
        };
        return result;
    };
    return MarkdownToPDFEngine;
}());
var MarkdownToPdfEngine = MarkdownToPDFEngine.getInstance();

var PdfPrinter = require('pdfmake');
var ExportPdfEngine = /** @class */ (function () {
    function ExportPdfEngine() {
    }
    ExportPdfEngine.getInstance = function () {
        if (!ExportPdfEngine.instance) {
            ExportPdfEngine.instance = new ExportPdfEngine();
        }
        return ExportPdfEngine.instance;
    };
    ExportPdfEngine.prototype["export"] = function (outputFolder) {
        var fonts = {
            Roboto: {
                normal: path.join(__dirname, '../src/resources/fonts/roboto-v15-latin-regular.ttf'),
                bold: path.join(__dirname, '../src/resources/fonts/roboto-v15-latin-700.ttf'),
                italics: path.join(__dirname, '../src/resources/fonts/roboto-v15-latin-italic.ttf')
            }
        };
        var printer = new PdfPrinter(fonts);
        var docDefinition = {
            info: {
                title: Configuration$1.mainData.documentationMainName
            },
            content: [],
            styles: {
                header: {
                    fontSize: 18,
                    bold: true,
                    color: '#008cff',
                    margin: [0, 0, 0, 15]
                },
                subheader: {
                    fontSize: 15,
                    bold: true
                }
            }
        };
        docDefinition.content.push({
            text: Configuration$1.mainData.documentationMainName,
            alignment: 'center',
            bold: true,
            fontSize: 22,
            margin: [10, 350, 10, 270]
        });
        if (!Configuration$1.mainData.hideGenerator) {
            docDefinition.content.push({
                text: I18nEngine$1.translate('generated-using'),
                alignment: 'center'
            });
            docDefinition.content.push({
                image: "data:image/jpeg;base64,/9j/4AAQSkZJRgABAgAAZABkAAD/7AARRHVja3kAAQAEAAAAUgAA/+4AJkFkb2JlAGTAAAAAAQMAFQQDBgoNAAAEqAAAB+0AAAr7AAAPLf/bAIQAAgEBAQIBAgICAgMCAgIDBAMCAgMEBAMDBAMDBAUEBQUFBQQFBQYHBwcGBQkJCQkJCQwMDAwMDAwMDAwMDAwMDAECAgIEBAQIBQUIDAkICQwODg4ODg4ODAwMDAwODgwMDAwMDA4MDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwM/8IAEQgARgBGAwERAAIRAQMRAf/EAOoAAAIDAQEBAQAAAAAAAAAAAAAIBAYHBQIBCQEBAAIDAQEBAAAAAAAAAAAAAAQHAgUGAwgBEAABBAIBAgUDBQAAAAAAAAAEAQIDBQAGByATEBESFBVAMSIhMzUWCBEAAQIDAwcGCwcFAAAAAAAAAQIDERMEABIFITFBUSIyFBAgYUIzFXGBkaHBUmIjcyQG8LFykkM0NdGC4lMWEgABAwMCAwgDAAAAAAAAAAABABECEDESIEHwIQMwQFBRYYGhIrEyExMBAAEDAwMEAwEBAQAAAAAAAREAITFBUWEQcYEg8JGhscHh0UDx/9oADAMBAAIRAxEAAAF/gADn4SvT8nZxgAONlChfvhZsNorWW8per7lk+uou+6bdycPTjZQsBlV/u8XvbBjsER5b6NYHoqX7vT1tjvT8o0Ff2Pkfzx119ujk7juPajxujVWsbqdm1fneTl5Lnts6N2VYtjTNrZXQnX6tf/HqzIsP3G27a7um8sg9rguss5tN7RfZ9IHL0MuNF9fz1k3i5+6pTWPXU5TB7fSZnITco4AFX8NxZ/fT/SISwAS0gFYOmaUdM0o04AA5ZMJAAB//2gAIAQEAAQUC6LO0ACCr7OuKg6Ly8CAC1rZhLEXP9Ib20KPgJYrPb9n5JqK2zELGnGy8vAgApZbm6ude14OvDzkW0Wy2LgnUQ67UuQOP4bKHS90PpT/l6z4zaYTpTtLBqYarORb34/UQiorBwgsMAmc0yaywirl2H+l2tUMWMi2VZZVlmMUN/oLe6+O84DqISrvORuRhqgbSdJtdgtf6/TfDZa1QxY1RUDAibuIyz2PhbUEp9Bzb+Kam0uK+vDFD6LvTNXscRE8sBNFJC8bGHf8A23w26kWVlX8j1mrwQ8iSh6bBsaKMJfQ6tqX8z0XXwvxAntfa9H//2gAIAQIAAQUC6FXyxF6UTFTw9v3EaK5jhwXPa5qouImfbFXwfP8AnG9ytCN9GFCpKnbd6ijyoCgrBhDMld5NhAf3k8BrLtLI+L12FewhiLMDMCcwhk7VXB2frksvlkcfqz0p5ZYV7CGVlY0ZnfXG/bHwIqonS6NF+o//2gAIAQMAAQUC6IYXSOlhcxeiedI0GJSRMr7BBpLG7hmgIsGxua5FTJ50jRVfM8cdI0wARrILaOJk5wPcwQtYl7rfS4OKaJgPY8Kkbuzyt7WPerlxdbkLZHBP6YZlYv4ytliVi0EscWbAYnaypqVnWxsWCs9y/uZDMrFmmWRWU7GsO/dwC7fCyWVz3dA580X1H//aAAgBAgIGPwLtuadMdXIrmmNk4usd1lK/wyyjV64ysoy4ZYyXDELKK5aWpjJeu/gX/9oACAEDAgY/AtGMbppBtLmuZi6aF0ycUc6B05e6I6YZOLpjZZbLFNQBHqHZOaGcLj5Uum3t6vVijKe6EYm9Mpfr+VjG+wX9H+2gCJYojyphfyWUr6frLvH/2gAIAQEBBj8C5i6iqcDTKIXln2jAZrTKZ9D6PWQoKHm5peePw29KjaKdh5Has6ukdHJQYYE358X6iBhBKdlHlMfJYvtLWluibK3k5U5VbKQYfbJZqmKS+qPzN39NJ9PRZDzKw404IoWMxHIXnj8NvSo29Zat1PUQi0tvacV2rulR5KupKb9OpctgkRQUN5B5c9lVKGg27iS5ivho2Uek+OxqaYBFcgeAOAaD06jZVPUJUaUqg+wd5CtYtxk5PC3b87q3bLTV5/04bt32bDh9pw/uFHevf05Kt4GDriZTP43MnmGW1PhrSDOqVIazbKY5z4rNMtiDbKQhseykQHIzD+TPaBGaV7fTqtid3+M93MKs0yajc9Nri8hG4vSDbUoflUm0xv8AuTpBtS4UtSkBlE5xcNiY5kAjrAHns/iUQtukRdaUMomu/wCMeSQxBzEHBspzhsHrK9AsuoqFq4a9GqqjvKV6qen7rd38Ong7lyRoh9tPJcXkI3F6QbHSuEXXPBatxCY4w9VuEqyxF0ZEhSTqAtStKCZ9VGpqFJFyKnd3J+ADkaqws06ifnAnrpGrUbN09O2GmWhBCBzY1dG24v8A27q/zJgeVmoYXMZqEJcZXrQsRBy9HM+pi2zj/wD05fq+CeQX+7uCvAolZZMZW5DavWZp6BGPsfTj2L4ciFUqrbrEtqpnxVmKjNS1EpiTkjmsmsL2MoLtHjbWILfcqShtqlAFGdvIgw3Tps67h4+oDgb3dhxebxPHuRvGqNLf95AxEbniyWIxNvGR9O8XW92NL4w1wR8vw82V8xc7W7e6I9WzDBZxNGJt4VRN4ElsVMlFQlgCDsv3YId7SZ1bVPGIr+859TFa+K4Ph5ypV0/t4S7sBnz6Y82o7xk8BLPF8TdkStN+/sw8Nm5F2TdEm5C5chkuwyQhzf/aAAgBAQMBPyH0Z6rshBiLloU785aMmHj0yIFsHin7afFrBl/IXRNcl2LM5uPCspYSGzxlViaRVDmyy+9XCj8qZlHSRAtg8U/bQcDHCQT9B996PhYLn8oaHTjXpAVJGHyqwgFzqB+CnsQEd/wNsPWiAh2E3CamtfI6+T+Zm2aICD2Gh4e2pg2NkPa7bP8AZ6eROPVnMnhUaumRcBZcC9cUadEXwdEpxDaNiz/7I4iolcRS4WHd8I5inB5R+0blecA9+KNJfD+E1OvYmuVYnJdqEiXCLiSJsJdzogf/AN/qPZGQrTnKrqOVq/4H4Nrn7zN9U3z0cHlH7RuUmzNYNLoDYqLZqOCUMIjipniCRAwYQKb9ATriTGz2MT5yX0OEfD9rq+lzhofnDy0AAQFgMR0NoKogfGg3XoBFCmLOrSEA4Jiuf7Iem5CFda5btRSkCbLpWkJFOob7ZhLsb0PIrbHgSB2bK6VAD0ESMrhDNlySpBYj+SHOlhWNx6f9Tm3jucK/TvW+qI9P/9oACAECAwE/IfQYloXHpkqPotQYipJhS7HbmsCE6SVYVP0BWT7okuaR/wDnrx3O9aR2U/8AG17Tj7nmvkUaj7116TDWc2nz73oQR0Mri+uf5VwL3Z7HPFagNHUfeStip7HH47189zUffzVp0U97p0xzNI5cV2DpqA0dR95Kihd5f5xTNcmnMukyoAg9OTP+j//aAAgBAwMBPyH0DBlVJkuben2Uat6zqdJUikbR23aT5CWS5rO3Fmime/FGkkenso1yL6Co6Z1ehWCxIb3b3OMVGEGe+v8AlAP/AH15/jatQ7q2K31Gu477+9ukj4LvY/3HmmDLSe/HmmOZZ+egRxgnXt/cadr0chZexYN96kBSe/bpgAkGIxaNb9/xV+Semx/Y+OmCh9g/bQUC3scvH5rvhM+/cW6SArjOlIAgvFxeRzS0DKLEDHHRfD9D/lIklenKQbZPh/6P/9oADAMBAAIRAxEAABAADIAAEUBAJBiqBCnAtSQhDBOICAAYiAACASAAAAAf/9oACAEBAwE/EPQ0Vyd+zkJMsfVQdesoiZOQKE19N+ymQOSG4DsB4EgAwvAsJBsGLNnSazPqxJiQhRhiFO2A8prqSQwEZTsrkOxaFkS+XDcilaaaLwEdEwjcbN+l+ymQOSG4DsB4F/Kla5RW91dyt0FGgAJADzdNy3Kq0sJdcSncyDhuqxoE84dVAiw0pb87bALJYAEJ9Bpb/keEZqjTAbw18fXx6szNrNhhV6us10zH5arOVaVBMIYBYLxGIvfoZ4QxCMiWhFqpAFxlFE44KjS7N/jwA6LhRFMkL4jC26U6KoZhIe8+rtHkdS+gAAtMm68neEfic3VPiVFnI7JUfkQsWvwbOEotp8IuIFrwAb7T5HaQLKDEDJ0pB4EgJY9VlM5aE/7WX4QoJ1AMt6ADuUk2e6Z7u3eXS+gAAtMm68neEY8wnlRYKUKYC7QMVsGMcUsbFCZjPn3pgoloNiegCWw7JJLWTiiXDQO+4mD3VWVCqKqqqvpvqGZduIlRWwhxR9jg0AWADoFwg3hQAKgg3uei7zCjSUWEiLmyhHvBcRMAGYuUaGBARTnIrq0yGnW4b7j3EsA4ThUXVFvgEZRh2oooc22B33E/RFmA1gvOm0wSpWIk30fuUn+a9pm1ex9gmi0oiLen/9oACAECAwE/EPRLGAouUJx6XcFI+OgIUh3nvsUZEyhGz++blChhGWr9HO9OAUQj0dwUFBHL0iSDoNhva5T4leJ2oEkr2Tjc8l8gkLOAbPGzpWsbo3T71xF8U4cCxfdW3FvhcwGi6xFl8J/WA8hW7DY80II3WRZTvbnVvQgMHRqFXdn+G/kXsy6bQCRdSsWYfi00nCBc/KNx7CbIJH3dF7r7WjlbCjCXwnhP6cDyENA5JN/ijI05bv8AOhnI/qnqcnetIwiOicIFz8o3HsJsgi5rV3Y0Gg01cuxOxCbTZOyVEXEk3Z++hKcb8/2jQQHpx0v3/wBH/9oACAEDAwE/EPQxWZAReBXMGBrgtIV95OS3pev2NVsft0qU2Pg5NzoKaaY7CshILRJZzenPUxhDlq0XI3pyk5w0ft1jbxQJgSJ0ev2NVsft0rf6wewB996yLXcX/Nj99HC6AGpSTq0aUV84AmHNAtokgQSNqfgAPB2ednw2w/Bv5Fuc7mvetP2zsj3pmbZqVSVf7XvxhLYoTbKu++wfD7ahpLfcw8KKCBR5b2Mdyt5pHJcndUv30NYRlYBpcSGHYwmL0hHLUsZrL7D5vFaH9TRPeGtwD8r/AH89q+CjolTLsFq1gDVhOhC9OPXyoS2xbdQ7uhoFt3Cj6PgwXxjTAMDABgaGVY1TXfkZ/ERaMaEW6aH9TRPeGg5ESgd9+WjJ4iVmuygLeLFDCQSYTELCxLOLbdEMCDe5tHfNizNphs4lsq6/zQCwWLekmOYPyAdwmlVl/wCf/9k=",
                width: 70,
                alignment: 'center',
                pageBreak: 'after'
            });
        }
        docDefinition.content.push({
            toc: {
                title: {
                    text: I18nEngine$1.translate('table-of-contents'),
                    bold: true,
                    alignment: 'center',
                    fontSize: 18,
                    margin: [10, 10, 10, 50]
                },
                numberStyle: {
                    bold: true
                }
            },
            pageBreak: 'after'
        });
        // Add README page if available
        docDefinition.content.push(this.generateMarkdownContent());
        // Add CHANGELOG page if available
        // Add CONTRIBUTING page if available
        // Add LICENSE page if available
        // Add TODO page if available
        // Add Dependencies page if available
        // Add Additional pages if available
        docDefinition.content.push(this.generateModulesContent());
        docDefinition.content.push(this.generateComponentsContent());
        // Classes
        // Injectables
        // Interceptors
        // Guards
        // Interfaces
        // Pipes
        // Miscellaneous
        // Routes
        // Coverage - docDefinition.content.push(...this.coverageEngine.calculateTable());
        var pdfDoc = printer.createPdfKitDocument(docDefinition);
        return new Promise(function (resolve, reject) {
            fs.ensureFile(outputFolder + path.sep + 'documentation.pdf', function (err) {
                if (err) {
                    reject("Error during pdf generation: " + err);
                }
                else {
                    pdfDoc.pipe(fs.createWriteStream(outputFolder + path.sep + 'documentation.pdf'));
                    pdfDoc.end();
                    resolve();
                }
            });
        });
    };
    ExportPdfEngine.prototype.firstCharacterUpperCase = function (sentence) {
        return sentence.charAt(0).toUpperCase() + sentence.slice(1);
    };
    ExportPdfEngine.prototype.generateMarkdownContent = function () {
        var _this = this;
        var pages = Configuration$1.markDownPages;
        var data = [];
        pages.forEach(function (page) {
            data.push({
                text: "" + _this.firstCharacterUpperCase(page.name),
                tocItem: true,
                style: 'header'
            });
            var convertedMarkdownObject = MarkdownToPdfEngine.convert(page.data);
            convertedMarkdownObject.margin = [0, 10];
            data.push(convertedMarkdownObject);
        });
        this.insertPageReturn(data);
        return data;
    };
    ExportPdfEngine.prototype.insertPageReturn = function (data) {
        data.push({
            text: " ",
            margin: [0, 0, 0, 20],
            pageBreak: 'after'
        });
    };
    ExportPdfEngine.prototype.generateModulesContent = function () {
        var data = [];
        data.push({
            text: 'Modules',
            tocItem: true,
            style: 'header'
        });
        _.forEach(Configuration$1.mainData.modules, function (module) {
            data.push({
                text: "" + module.name,
                style: 'subheader',
                margin: [0, 15, 0, 15]
            });
            data.push({
                text: [
                    {
                        text: "Filename : ",
                        bold: true
                    },
                    {
                        text: module.file
                    }
                ],
                margin: [0, 10]
            });
            if (module.rawdescription != '') {
                data.push({
                    text: "Description :",
                    bold: true,
                    margin: [0, 10]
                });
                data.push({
                    text: "" + module.rawdescription,
                    margin: [0, 5]
                });
            }
            if (module.declarations.length > 0) {
                data.push({
                    text: "Declarations :",
                    bold: true,
                    margin: [0, 10]
                });
                var list_1 = { ul: [] };
                _.forEach(module.declarations, function (declaration) {
                    list_1.ul.push({
                        text: "" + declaration.name
                    });
                });
                data.push(list_1);
            }
            if (module.providers.length > 0) {
                data.push({
                    text: "Providers :",
                    bold: true,
                    margin: [0, 10]
                });
                var list_2 = { ul: [] };
                _.forEach(module.providers, function (provider) {
                    list_2.ul.push({
                        text: "" + provider.name
                    });
                });
                data.push(list_2);
            }
            if (module.imports.length > 0) {
                data.push({
                    text: "Imports :",
                    bold: true,
                    margin: [0, 10]
                });
                var list_3 = { ul: [] };
                _.forEach(module.imports, function (importRef) {
                    list_3.ul.push({
                        text: "" + importRef.name
                    });
                });
                data.push(list_3);
            }
            if (module.exports.length > 0) {
                data.push({
                    text: "Exports :",
                    bold: true,
                    margin: [0, 10]
                });
                var list_4 = { ul: [] };
                _.forEach(module.exports, function (exportRef) {
                    list_4.ul.push({
                        text: "" + exportRef.name
                    });
                });
                data.push(list_4);
            }
            data.push({
                text: " ",
                margin: [0, 0, 0, 20]
            });
        });
        this.insertPageReturn(data);
        return data;
    };
    ExportPdfEngine.prototype.generateComponentsContent = function () {
        var data = [];
        data.push({
            text: 'Components',
            tocItem: true,
            style: 'header'
        });
        _.forEach(Configuration$1.mainData.components, function (component) {
            data.push({
                text: "" + component.name,
                style: 'subheader',
                margin: [0, 15, 0, 15]
            });
            data.push({
                text: [
                    {
                        text: "Filename : ",
                        bold: true
                    },
                    {
                        text: component.file
                    }
                ],
                margin: [0, 10]
            });
            if (component.rawdescription != '') {
                data.push({
                    text: "Description :",
                    bold: true,
                    margin: [0, 10]
                });
                data.push({
                    text: "" + component.rawdescription,
                    margin: [0, 5]
                });
            }
            data.push({
                text: " ",
                margin: [0, 0, 0, 20]
            });
        });
        this.insertPageReturn(data);
        return data;
    };
    return ExportPdfEngine;
}());
var ExportPdfEngine$1 = ExportPdfEngine.getInstance();

var ExportEngine = /** @class */ (function () {
    function ExportEngine() {
    }
    ExportEngine.getInstance = function () {
        if (!ExportEngine.instance) {
            ExportEngine.instance = new ExportEngine();
        }
        return ExportEngine.instance;
    };
    ExportEngine.prototype["export"] = function (outputFolder, data) {
        switch (Configuration$1.mainData.exportFormat) {
            case 'json':
                return ExportJsonEngine$1["export"](outputFolder, data);
            case 'pdf':
                return ExportPdfEngine$1["export"](outputFolder);
        }
    };
    return ExportEngine;
}());
var ExportEngine$1 = ExportEngine.getInstance();

var BreakCommaHelper = /** @class */ (function () {
    function BreakCommaHelper(bars) {
        this.bars = bars;
    }
    BreakCommaHelper.prototype.helperFunc = function (context, text) {
        text = this.bars.Utils.escapeExpression(text);
        text = text.replace(/,/g, ',<br>');
        return new Handlebars.SafeString(text);
    };
    return BreakCommaHelper;
}());

var BreakLinesHelper = /** @class */ (function () {
    function BreakLinesHelper(bars) {
        this.bars = bars;
    }
    BreakLinesHelper.prototype.helperFunc = function (context, text) {
        text = this.bars.Utils.escapeExpression(text);
        text = text.replace(/(\r\n|\n|\r)/gm, '<br>');
        text = text.replace(/ /gm, '&nbsp;');
        text = text.replace(/	/gm, '&nbsp;&nbsp;&nbsp;&nbsp;');
        return new Handlebars.SafeString(text);
    };
    return BreakLinesHelper;
}());

var CleanParagraphHelper = /** @class */ (function () {
    function CleanParagraphHelper() {
    }
    CleanParagraphHelper.prototype.helperFunc = function (context, text) {
        text = text.replace(/<p>/gm, '');
        text = text.replace(/<\/p>/gm, '');
        return new Handlebars.SafeString(text);
    };
    return CleanParagraphHelper;
}());

var CompareHelper = /** @class */ (function () {
    function CompareHelper() {
    }
    CompareHelper.prototype.helperFunc = function (context, a, operator, b, options) {
        if (arguments.length < 4) {
            throw new Error('handlebars Helper {{compare}} expects 4 arguments');
        }
        var result;
        switch (operator) {
            case 'indexof':
                result = b.indexOf(a) !== -1;
                break;
            case '===':
                result = a === b;
                break;
            case '!==':
                result = a !== b;
                break;
            case '>':
                result = a > b;
                break;
            default: {
                throw new Error('helper {{compare}}: invalid operator: `' + operator + '`');
            }
        }
        if (result === false) {
            return options.inverse(context);
        }
        return options.fn(context);
    };
    return CompareHelper;
}());

var DebugHelper = /** @class */ (function () {
    function DebugHelper() {
    }
    DebugHelper.prototype.helperFunc = function (context, optionalValue) {
        console.log('Current Context');
        console.log('====================');
        console.log(context);
        if (optionalValue) {
            console.log('OptionalValue');
            console.log('====================');
            console.log(optionalValue);
        }
    };
    return DebugHelper;
}());

var ElementAloneHelper = /** @class */ (function () {
    function ElementAloneHelper() {
    }
    ElementAloneHelper.prototype.helperFunc = function (context, elements, elementType, options) {
        var alones = [];
        var modules = DependenciesEngine$1.modules;
        elements.forEach(function (element) {
            var foundInOneModule = false;
            modules.forEach(function (module) {
                module.declarations.forEach(function (declaration) {
                    if (declaration.id === element.id) {
                        foundInOneModule = true;
                    }
                    if (declaration.file === element.file) {
                        foundInOneModule = true;
                    }
                });
                module.controllers.forEach(function (controller) {
                    if (controller.id === element.id) {
                        foundInOneModule = true;
                    }
                    if (controller.file === element.file) {
                        foundInOneModule = true;
                    }
                });
                module.providers.forEach(function (provider) {
                    if (provider.id === element.id) {
                        foundInOneModule = true;
                    }
                    if (provider.file === element.file) {
                        foundInOneModule = true;
                    }
                });
            });
            if (!foundInOneModule) {
                alones.push(element);
            }
        });
        if (alones.length > 0) {
            switch (elementType) {
                case 'component':
                    context.components = alones;
                    break;
                case 'directive':
                    context.directives = alones;
                    break;
                case 'controller':
                    context.controllers = alones;
                    break;
                case 'injectable':
                    context.injectables = alones;
                    break;
                case 'pipe':
                    context.pipes = alones;
                    break;
            }
            return options.fn(context);
        }
    };
    return ElementAloneHelper;
}());

var EscapeSimpleQuoteHelper = /** @class */ (function () {
    function EscapeSimpleQuoteHelper() {
    }
    EscapeSimpleQuoteHelper.prototype.helperFunc = function (context, text) {
        if (!text) {
            return;
        }
        text = text.replace(/'/g, "\\'");
        text = text.replace(/(\r\n|\n|\r)/gm, '');
        return text;
    };
    return EscapeSimpleQuoteHelper;
}());

var FilterAngular2ModulesHelper = /** @class */ (function () {
    function FilterAngular2ModulesHelper() {
    }
    FilterAngular2ModulesHelper.prototype.helperFunc = function (context, text, options) {
        var NG2_MODULES = [
            'BrowserModule',
            'FormsModule',
            'HttpModule',
            'RouterModule'
        ];
        var len = NG2_MODULES.length;
        var i = 0;
        var result = false;
        for (i; i < len; i++) {
            if (text.indexOf(NG2_MODULES[i]) > -1) {
                result = true;
            }
        }
        if (result) {
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return FilterAngular2ModulesHelper;
}());

var AngularVersionUtil = /** @class */ (function () {
    function AngularVersionUtil() {
    }
    AngularVersionUtil.getInstance = function () {
        if (!AngularVersionUtil.instance) {
            AngularVersionUtil.instance = new AngularVersionUtil();
        }
        return AngularVersionUtil.instance;
    };
    AngularVersionUtil.prototype.cleanVersion = function (version) {
        return version
            .replace('~', '')
            .replace('^', '')
            .replace('=', '')
            .replace('<', '')
            .replace('>', '');
    };
    AngularVersionUtil.prototype.getAngularVersionOfProject = function (packageData) {
        var _result = '';
        if (packageData.dependencies) {
            var angularCore = packageData.dependencies[AngularVersionUtil.CorePackage];
            if (angularCore) {
                _result = this.cleanVersion(angularCore);
            }
        }
        return _result;
    };
    AngularVersionUtil.prototype.isAngularVersionArchived = function (version) {
        var result;
        try {
            result = semver.compare(version, '2.4.10') <= 0;
        }
        catch (e) { }
        return result;
    };
    AngularVersionUtil.prototype.prefixOfficialDoc = function (version) {
        return this.isAngularVersionArchived(version) ? 'v2.' : '';
    };
    AngularVersionUtil.prototype.getApiLink = function (api, angularVersion) {
        var angularDocPrefix = this.prefixOfficialDoc(angularVersion);
        return "https://" + angularDocPrefix + "angular.io/" + api.path;
    };
    AngularVersionUtil.CorePackage = '@angular/core';
    return AngularVersionUtil;
}());
var AngularVersionUtil$1 = AngularVersionUtil.getInstance();

var BasicTypes;
(function (BasicTypes) {
    BasicTypes[BasicTypes["number"] = 0] = "number";
    BasicTypes[BasicTypes["boolean"] = 1] = "boolean";
    BasicTypes[BasicTypes["string"] = 2] = "string";
    BasicTypes[BasicTypes["object"] = 3] = "object";
    BasicTypes[BasicTypes["date"] = 4] = "date";
    BasicTypes[BasicTypes["function"] = 5] = "function";
})(BasicTypes || (BasicTypes = {}));
var BasicTypeScriptTypes;
(function (BasicTypeScriptTypes) {
    BasicTypeScriptTypes[BasicTypeScriptTypes["any"] = 0] = "any";
    BasicTypeScriptTypes[BasicTypeScriptTypes["void"] = 1] = "void";
})(BasicTypeScriptTypes || (BasicTypeScriptTypes = {}));
var BasicTypeUtil = /** @class */ (function () {
    function BasicTypeUtil() {
    }
    BasicTypeUtil.getInstance = function () {
        if (!BasicTypeUtil.instance) {
            BasicTypeUtil.instance = new BasicTypeUtil();
        }
        return BasicTypeUtil.instance;
    };
    /**
     * Checks if a given types is a basic javascript type
     * https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects
     * @param type The type to check
     */
    BasicTypeUtil.prototype.isJavascriptType = function (type) {
        if (typeof type !== 'undefined' && type.toLowerCase) {
            return type.toLowerCase() in BasicTypes;
        }
        else {
            return false;
        }
    };
    /**
     * Checks if a given type is a typescript type (That is not a javascript type)
     * https://www.typescriptlang.org/docs/handbook/basic-types.html
     * @param type The type to check
     */
    BasicTypeUtil.prototype.isTypeScriptType = function (type) {
        if (typeof type !== 'undefined' && type.toLowerCase) {
            return type.toLowerCase() in BasicTypeScriptTypes;
        }
        else {
            return false;
        }
    };
    /**
     * Check if the type is a typescript or javascript type
     * @param type The type to check
     */
    BasicTypeUtil.prototype.isKnownType = function (type) {
        return this.isJavascriptType(type) || this.isTypeScriptType(type);
    };
    /**
     * Returns a official documentation link to either the javascript or typescript type
     * @param type The type to check
     * @returns The documentation link or undefined if type not found
     */
    BasicTypeUtil.prototype.getTypeUrl = function (type) {
        if (this.isJavascriptType(type)) {
            return "https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/" + type;
        }
        if (this.isTypeScriptType(type)) {
            return "https://www.typescriptlang.org/docs/handbook/basic-types.html";
        }
        return undefined;
    };
    return BasicTypeUtil;
}());
var BasicTypeUtil$1 = BasicTypeUtil.getInstance();

var FunctionSignatureHelper = /** @class */ (function () {
    function FunctionSignatureHelper() {
    }
    FunctionSignatureHelper.prototype.handleFunction = function (arg) {
        var _this = this;
        if (arg["function"].length === 0) {
            return "" + arg.name + this.getOptionalString(arg) + ": () => void";
        }
        var argums = arg["function"].map(function (argu) {
            var _result = DependenciesEngine$1.find(argu.type);
            if (_result) {
                if (_result.source === 'internal') {
                    var path = _result.data.type;
                    if (_result.data.type === 'class') {
                        path = 'classe';
                    }
                    return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"../" + path + "s/" + _result.data.name + ".html\">" + argu.type + "</a>";
                }
                else {
                    var path = AngularVersionUtil$1.getApiLink(_result.data, Configuration$1.mainData.angularVersion);
                    return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"" + path + "\" target=\"_blank\">" + argu.type + "</a>";
                }
            }
            else if (BasicTypeUtil$1.isKnownType(argu.type)) {
                var path = BasicTypeUtil$1.getTypeUrl(argu.type);
                return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"" + path + "\" target=\"_blank\">" + argu.type + "</a>";
            }
            else {
                if (argu.name && argu.type) {
                    return "" + argu.name + _this.getOptionalString(arg) + ": " + argu.type;
                }
                else {
                    if (argu.name) {
                        return "" + argu.name.text;
                    }
                    else {
                        return '';
                    }
                }
            }
        });
        return "" + arg.name + this.getOptionalString(arg) + ": (" + argums + ") => void";
    };
    FunctionSignatureHelper.prototype.getOptionalString = function (arg) {
        return arg.optional ? '?' : '';
    };
    FunctionSignatureHelper.prototype.helperFunc = function (context, method) {
        var _this = this;
        var args = [];
        if (method.args) {
            args = method.args
                .map(function (arg) {
                var _result = DependenciesEngine$1.find(arg.type);
                if (_result) {
                    if (_result.source === 'internal') {
                        var path = _result.data.type;
                        if (_result.data.type === 'class') {
                            path = 'classe';
                        }
                        return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"../" + path + "s/" + _result.data.name + ".html\">" + arg.type + "</a>";
                    }
                    else {
                        var path = AngularVersionUtil$1.getApiLink(_result.data, Configuration$1.mainData.angularVersion);
                        return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"" + path + "\" target=\"_blank\">" + arg.type + "</a>";
                    }
                }
                else if (arg.dotDotDotToken) {
                    return "..." + arg.name + ": " + arg.type;
                }
                else if (arg["function"]) {
                    return _this.handleFunction(arg);
                }
                else if (BasicTypeUtil$1.isKnownType(arg.type)) {
                    var path = BasicTypeUtil$1.getTypeUrl(arg.type);
                    return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"" + path + "\" target=\"_blank\">" + arg.type + "</a>";
                }
                else {
                    if (arg.type) {
                        return "" + arg.name + _this.getOptionalString(arg) + ": " + arg.type;
                    }
                    else {
                        return "" + arg.name + _this.getOptionalString(arg);
                    }
                }
            })
                .join(', ');
        }
        if (method.name) {
            return method.name + "(" + args + ")";
        }
        else {
            return "(" + args + ")";
        }
    };
    return FunctionSignatureHelper;
}());

var HasOwnHelper = /** @class */ (function () {
    function HasOwnHelper() {
    }
    HasOwnHelper.prototype.helperFunc = function (context, entity, key, options) {
        if (Object.hasOwnProperty.call(entity, key)) {
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return HasOwnHelper;
}());

var I18nHelper = /** @class */ (function () {
    function I18nHelper() {
    }
    I18nHelper.prototype.helperFunc = function (context, i18n_key, options) {
        var result = I18nEngine$1.translate(i18n_key);
        return result;
    };
    return I18nHelper;
}());

var IfStringHelper = /** @class */ (function () {
    function IfStringHelper() {
    }
    IfStringHelper.prototype.helperFunc = function (context, a, options) {
        if (typeof a === 'string') {
            return options.fn(context);
        }
        return options.inverse(context);
    };
    return IfStringHelper;
}());

var IndexableSignatureHelper = /** @class */ (function () {
    function IndexableSignatureHelper() {
    }
    IndexableSignatureHelper.prototype.helperFunc = function (context, method) {
        var args = method.args.map(function (arg) { return arg.name + ": " + arg.type; }).join(', ');
        if (method.name) {
            return method.name + "[" + args + "]";
        }
        else {
            return "[" + args + "]";
        }
    };
    return IndexableSignatureHelper;
}());

var IsInitialTabHelper = /** @class */ (function () {
    function IsInitialTabHelper() {
    }
    IsInitialTabHelper.prototype.helperFunc = function (context, tabs, tabId, options) {
        return tabs[0].id === tabId ? options.fn(context) : options.inverse(context);
    };
    return IsInitialTabHelper;
}());

var IsNotToggleHelper = /** @class */ (function () {
    function IsNotToggleHelper() {
    }
    IsNotToggleHelper.prototype.helperFunc = function (context, type, options) {
        var result = Configuration$1.mainData.toggleMenuItems.indexOf(type);
        if (Configuration$1.mainData.toggleMenuItems.indexOf('all') !== -1) {
            return options.inverse(context);
        }
        else if (result !== -1) {
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return IsNotToggleHelper;
}());

var IsTabEnabledHelper = /** @class */ (function () {
    function IsTabEnabledHelper() {
    }
    IsTabEnabledHelper.prototype.helperFunc = function (context, tabs, tabId, options) {
        var isTabEnabled = -1 !== _.findIndex(tabs, { id: tabId });
        return isTabEnabled ? options.fn(context) : options.inverse(context);
    };
    return IsTabEnabledHelper;
}());

var JsdocCodeExampleHelper = /** @class */ (function () {
    function JsdocCodeExampleHelper() {
    }
    JsdocCodeExampleHelper.prototype.cleanTag = function (comment) {
        if (comment.charAt(0) === '*') {
            comment = comment.substring(1, comment.length);
        }
        if (comment.charAt(0) === ' ') {
            comment = comment.substring(1, comment.length);
        }
        if (comment.indexOf('<p>') === 0) {
            comment = comment.substring(3, comment.length);
        }
        if (comment.substr(-1) === '\n') {
            comment = comment.substring(0, comment.length - 1);
        }
        if (comment.substr(-4) === '</p>') {
            comment = comment.substring(0, comment.length - 4);
        }
        return comment;
    };
    JsdocCodeExampleHelper.prototype.getHtmlEntities = function (str) {
        return String(str)
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;');
    };
    JsdocCodeExampleHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        var i = 0;
        var len = jsdocTags.length;
        var tags = [];
        var type = 'html';
        if (options.hash.type) {
            type = options.hash.type;
        }
        for (i; i < len; i++) {
            if (jsdocTags[i].tagName) {
                if (jsdocTags[i].tagName.text === 'example') {
                    var tag = {};
                    if (jsdocTags[i].comment) {
                        if (jsdocTags[i].comment.indexOf('<caption>') !== -1) {
                            tag.comment = jsdocTags[i].comment
                                .replace(/<caption>/g, '<b><i>')
                                .replace(/\/caption>/g, '/b></i>');
                        }
                        else {
                            tag.comment =
                                "<pre class=\"line-numbers\"><code class=\"language-" + type + "\">" +
                                    this.getHtmlEntities(this.cleanTag(jsdocTags[i].comment)) +
                                    "</code></pre>";
                        }
                        tags.push(tag);
                    }
                }
            }
        }
        if (tags.length > 0) {
            context.tags = tags;
            return options.fn(context);
        }
    };
    return JsdocCodeExampleHelper;
}());

var JsdocDefaultHelper = /** @class */ (function () {
    function JsdocDefaultHelper() {
    }
    JsdocDefaultHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        if (jsdocTags) {
            var i = 0;
            var len = jsdocTags.length;
            var tag = {};
            var defaultValue = false;
            for (i; i < len; i++) {
                if (jsdocTags[i].tagName) {
                    if (jsdocTags[i].tagName.text === 'default') {
                        defaultValue = true;
                        if (jsdocTags[i].typeExpression && jsdocTags[i].typeExpression.type.name) {
                            tag.type = jsdocTags[i].typeExpression.type.name.text;
                        }
                        if (jsdocTags[i].comment) {
                            tag.comment = jsdocTags[i].comment;
                        }
                        if (jsdocTags[i].name) {
                            tag.name = jsdocTags[i].name.text;
                        }
                    }
                }
            }
            if (defaultValue) {
                context.tag = tag;
                return options.fn(context);
            }
        }
    };
    return JsdocDefaultHelper;
}());

var JsdocExampleHelper = /** @class */ (function () {
    function JsdocExampleHelper() {
    }
    JsdocExampleHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        var i = 0;
        var len = jsdocTags.length;
        var tags = [];
        for (i; i < len; i++) {
            if (jsdocTags[i].tagName) {
                if (jsdocTags[i].tagName.text === 'example') {
                    var tag = {};
                    if (jsdocTags[i].comment) {
                        tag.comment = jsdocTags[i].comment
                            .replace(/<caption>/g, '<b><i>')
                            .replace(/\/caption>/g, '/b></i>');
                    }
                    tags.push(tag);
                }
            }
        }
        if (tags.length > 0) {
            context.tags = tags;
            return options.fn(context);
        }
    };
    return JsdocExampleHelper;
}());

var JsdocParamsValidHelper = /** @class */ (function () {
    function JsdocParamsValidHelper() {
    }
    JsdocParamsValidHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        var i = 0;
        var len = jsdocTags.length;
        var valid = false;
        for (i; i < len; i++) {
            if (jsdocTags[i].tagName) {
                if (jsdocTags[i].tagName.text === 'param') {
                    valid = true;
                }
            }
        }
        if (valid) {
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return JsdocParamsValidHelper;
}());

var JsdocParamsHelper = /** @class */ (function () {
    function JsdocParamsHelper() {
    }
    JsdocParamsHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        var i = 0;
        var len = jsdocTags.length;
        var tags = [];
        for (i; i < len; i++) {
            if (jsdocTags[i].tagName) {
                if (jsdocTags[i].tagName.text === 'param') {
                    var tag = {};
                    if (jsdocTags[i].typeExpression && jsdocTags[i].typeExpression.type.kind) {
                        tag.type = kindToType(jsdocTags[i].typeExpression.type.kind);
                    }
                    if (jsdocTags[i].typeExpression && jsdocTags[i].typeExpression.type.name) {
                        tag.type = jsdocTags[i].typeExpression.type.name.text;
                    }
                    else {
                        tag.type = jsdocTags[i].type;
                    }
                    if (jsdocTags[i].comment) {
                        tag.comment = jsdocTags[i].comment;
                    }
                    if (jsdocTags[i].defaultValue) {
                        tag.defaultValue = jsdocTags[i].defaultValue;
                    }
                    if (jsdocTags[i].name) {
                        if (jsdocTags[i].name.text) {
                            tag.name = jsdocTags[i].name.text;
                        }
                        else {
                            tag.name = jsdocTags[i].name;
                        }
                    }
                    if (jsdocTags[i].optional) {
                        tag.optional = true;
                    }
                    tags.push(tag);
                }
            }
        }
        if (tags.length >= 1) {
            context.tags = tags;
            return options.fn(context);
        }
    };
    return JsdocParamsHelper;
}());

var JsdocReturnsCommentHelper = /** @class */ (function () {
    function JsdocReturnsCommentHelper() {
    }
    JsdocReturnsCommentHelper.prototype.helperFunc = function (context, jsdocTags, options) {
        var i = 0;
        var len = jsdocTags.length;
        var result;
        for (i; i < len; i++) {
            if (jsdocTags[i].tagName) {
                if (jsdocTags[i].tagName.text === 'returns' ||
                    jsdocTags[i].tagName.text === 'return') {
                    result = jsdocTags[i].comment;
                    break;
                }
            }
        }
        return result;
    };
    return JsdocReturnsCommentHelper;
}());

var LinkTypeHelper = /** @class */ (function () {
    function LinkTypeHelper() {
    }
    LinkTypeHelper.prototype.helperFunc = function (context, name, options) {
        var _result = DependenciesEngine$1.find(name);
        var angularDocPrefix = AngularVersionUtil$1.prefixOfficialDoc(Configuration$1.mainData.angularVersion);
        if (_result) {
            context.type = {
                raw: name
            };
            if (_result.source === 'internal') {
                if (_result.data.type === 'class') {
                    _result.data.type = 'classe';
                }
                context.type.href = '../' + _result.data.type + 's/' + _result.data.name + '.html';
                if (_result.data.type === 'miscellaneous' ||
                    (_result.data.ctype && _result.data.ctype === 'miscellaneous')) {
                    var mainpage = '';
                    switch (_result.data.subtype) {
                        case 'enum':
                            mainpage = 'enumerations';
                            break;
                        case 'function':
                            mainpage = 'functions';
                            break;
                        case 'typealias':
                            mainpage = 'typealiases';
                            break;
                        case 'variable':
                            mainpage = 'variables';
                    }
                    context.type.href =
                        '../' + _result.data.ctype + '/' + mainpage + '.html#' + _result.data.name;
                }
                context.type.target = '_self';
            }
            else {
                context.type.href = "https://" + angularDocPrefix + "angular.io/" + _result.data.path;
                context.type.target = '_blank';
            }
            return options.fn(context);
        }
        else if (BasicTypeUtil$1.isKnownType(name)) {
            context.type = {
                raw: name
            };
            context.type.target = '_blank';
            context.type.href = BasicTypeUtil$1.getTypeUrl(name);
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return LinkTypeHelper;
}());

var ModifIconHelper = /** @class */ (function () {
    function ModifIconHelper() {
    }
    ModifIconHelper.prototype.helperFunc = function (context, kind) {
        var _kindText = '';
        switch (kind) {
            case Ast.SyntaxKind.PrivateKeyword:
                _kindText = 'lock'; // private
                break;
            case Ast.SyntaxKind.ProtectedKeyword:
                _kindText = 'lock'; // protected
                break;
            case Ast.SyntaxKind.StaticKeyword:
                _kindText = 'reset'; // static
                break;
            case Ast.SyntaxKind.ExportKeyword:
                _kindText = 'export'; // export
                break;
            default:
                _kindText = 'reset';
                break;
        }
        return _kindText;
    };
    return ModifIconHelper;
}());

var ModifKindHelper = /** @class */ (function () {
    function ModifKindHelper() {
    }
    /**
     * Transform SyntaxKind into string
     * @param  {any}           context Handlebars context
     * @param  {SyntaxKind[]} kind  SyntaxKind concatenated
     * @return {string}                Parsed string
     */
    ModifKindHelper.prototype.helperFunc = function (context, kind) {
        var _kindText = '';
        switch (kind) {
            case Ast.SyntaxKind.PrivateKeyword:
                _kindText = 'Private';
                break;
            case Ast.SyntaxKind.ReadonlyKeyword:
                _kindText = 'Readonly';
                break;
            case Ast.SyntaxKind.ProtectedKeyword:
                _kindText = 'Protected';
                break;
            case Ast.SyntaxKind.PublicKeyword:
                _kindText = 'Public';
                break;
            case Ast.SyntaxKind.StaticKeyword:
                _kindText = 'Static';
                break;
            case Ast.SyntaxKind.AsyncKeyword:
                _kindText = 'Async';
                break;
            case Ast.SyntaxKind.AbstractKeyword:
                _kindText = 'Abstract';
                break;
        }
        return new Handlebars.SafeString(_kindText);
    };
    return ModifKindHelper;
}());

var ObjectLengthHelper = /** @class */ (function () {
    function ObjectLengthHelper() {
    }
    ObjectLengthHelper.prototype.helperFunc = function (context, obj, operator, length) {
        var len = arguments.length - 1;
        var options = arguments[len];
        if (typeof obj !== 'object') {
            return options.inverse(context);
        }
        var size = 0, key;
        for (key in obj) {
            if (obj.hasOwnProperty(key)) {
                size++;
            }
        }
        var result;
        switch (operator) {
            case '===':
                result = size === length;
                break;
            case '!==':
                result = size !== length;
                break;
            case '>':
                result = size > length;
                break;
            default: {
                throw new Error('helper {{objectLength}}: invalid operator: `' + operator + '`');
            }
        }
        if (result === false) {
            return options.inverse(context);
        }
        return options.fn(context);
    };
    return ObjectLengthHelper;
}());

var ObjectHelper = /** @class */ (function () {
    function ObjectHelper() {
    }
    ObjectHelper.prototype.helperFunc = function (context, text) {
        text = JSON.stringify(text);
        text = text.replace(/{"/, '{<br>&nbsp;&nbsp;&nbsp;&nbsp;"');
        text = text.replace(/,"/, ',<br>&nbsp;&nbsp;&nbsp;&nbsp;"');
        text = text.replace(/}$/, '<br>}');
        return new Handlebars.SafeString(text);
    };
    return ObjectHelper;
}());

var OneParameterHasHelper = /** @class */ (function () {
    function OneParameterHasHelper() {
    }
    OneParameterHasHelper.prototype.helperFunc = function (context, tags, typeToCheck) {
        var result = false;
        var len = arguments.length - 1;
        var options = arguments[len];
        var i = 0, leng = tags.length;
        for (i; i < leng; i++) {
            if (typeof tags[i][typeToCheck] !== 'undefined' && tags[i][typeToCheck] !== '') {
                result = true;
            }
        }
        if (result) {
            return options.fn(context);
        }
        else {
            return options.inverse(context);
        }
    };
    return OneParameterHasHelper;
}());

var OrLengthHelper = /** @class */ (function () {
    function OrLengthHelper() {
    }
    OrLengthHelper.prototype.helperFunc = function (context /* any, any, ..., options */) {
        var len = arguments.length - 1;
        var options = arguments[len];
        // We start at 1 because of options
        for (var i = 1; i < len; i++) {
            if (typeof arguments[i] !== 'undefined') {
                if (Object.keys(arguments[i]).length > 0) {
                    return options.fn(context);
                }
            }
        }
        return options.inverse(context);
    };
    return OrLengthHelper;
}());

var OrHelper = /** @class */ (function () {
    function OrHelper() {
    }
    OrHelper.prototype.helperFunc = function (context /* any, any, ..., options */) {
        var len = arguments.length - 1;
        var options = arguments[len];
        // We start at 1 because of options
        for (var i = 1; i < len; i++) {
            if (arguments[i]) {
                return options.fn(context);
            }
        }
        return options.inverse(context);
    };
    return OrHelper;
}());

var ParseDescriptionHelper = /** @class */ (function () {
    function ParseDescriptionHelper() {
    }
    ParseDescriptionHelper.prototype.helperFunc = function (context, description, depth) {
        var tagRegExpLight = new RegExp('\\{@link\\s+((?:.|\n)+?)\\}', 'i');
        var tagRegExpFull = new RegExp('\\{@link\\s+((?:.|\n)+?)\\}', 'i');
        var tagRegExp;
        var matches;
        var previousString;
        tagRegExp = description.indexOf(']{') !== -1 ? tagRegExpFull : tagRegExpLight;
        var processTheLink = function (originalDescription, matchedTag, leadingText) {
            var leading = extractLeadingText(originalDescription, matchedTag.completeTag);
            var split;
            var resultInCompodoc;
            var newLink;
            var rootPath;
            var stringtoReplace;
            var anchor = '';
            var label;
            var pageName;
            split = splitLinkText(matchedTag.text);
            if (typeof split.linkText !== 'undefined') {
                resultInCompodoc = DependenciesEngine$1.findInCompodoc(split.target);
            }
            else {
                var info = matchedTag.text;
                if (matchedTag.text.indexOf('#') !== -1) {
                    anchor = matchedTag.text.substr(matchedTag.text.indexOf('#'), matchedTag.text.length);
                    info = matchedTag.text.substr(0, matchedTag.text.indexOf('#'));
                }
                resultInCompodoc = DependenciesEngine$1.findInCompodoc(info);
            }
            if (resultInCompodoc) {
                label = resultInCompodoc.name;
                pageName = resultInCompodoc.name;
                if (leadingText) {
                    stringtoReplace = '[' + leadingText + ']' + matchedTag.completeTag;
                }
                else if (leading.leadingText !== undefined) {
                    stringtoReplace = '[' + leading.leadingText + ']' + matchedTag.completeTag;
                }
                else if (typeof split.linkText !== 'undefined') {
                    stringtoReplace = matchedTag.completeTag;
                }
                else {
                    stringtoReplace = matchedTag.completeTag;
                }
                if (resultInCompodoc.type === 'class') {
                    resultInCompodoc.type = 'classe';
                }
                else if (resultInCompodoc.type === 'miscellaneous' ||
                    (resultInCompodoc.ctype && resultInCompodoc.ctype === 'miscellaneous')) {
                    resultInCompodoc.type = 'miscellaneou'; // Not a typo, it is for matching other single types : component, module etc
                    label = resultInCompodoc.name;
                    anchor = '#' + resultInCompodoc.name;
                    if (resultInCompodoc.subtype === 'enum') {
                        pageName = 'enumerations';
                    }
                    else if (resultInCompodoc.subtype === 'function') {
                        pageName = 'functions';
                    }
                    else if (resultInCompodoc.subtype === 'typealias') {
                        pageName = 'typealiases';
                    }
                    else if (resultInCompodoc.subtype === 'variable') {
                        pageName = 'variables';
                    }
                }
                rootPath = '';
                switch (depth) {
                    case 0:
                        rootPath = './';
                        break;
                    case 1:
                    case 2:
                    case 3:
                    case 4:
                    case 5:
                        rootPath = '../'.repeat(depth);
                        break;
                }
                if (leading.leadingText !== undefined) {
                    label = leading.leadingText;
                }
                if (typeof split.linkText !== 'undefined') {
                    label = split.linkText;
                }
                newLink = "<a href=\"" + rootPath + resultInCompodoc.type + "s/" + pageName + ".html" + anchor + "\">" + label + "</a>";
                return originalDescription.replace(stringtoReplace, newLink);
            }
            else if (!resultInCompodoc && typeof split.linkText !== 'undefined') {
                newLink = "<a href=\"" + split.target + "\">" + split.linkText + "</a>";
                if (leadingText) {
                    stringtoReplace = '[' + leadingText + ']' + matchedTag.completeTag;
                }
                else if (leading.leadingText !== undefined) {
                    stringtoReplace = '[' + leading.leadingText + ']' + matchedTag.completeTag;
                }
                else if (typeof split.linkText !== 'undefined') {
                    stringtoReplace = matchedTag.completeTag;
                }
                else {
                    stringtoReplace = matchedTag.completeTag;
                }
                return originalDescription.replace(stringtoReplace, newLink);
            }
            else if (!resultInCompodoc && leading && typeof leading.leadingText !== 'undefined') {
                newLink = "<a href=\"" + split.target + "\">" + leading.leadingText + "</a>";
                if (leadingText) {
                    stringtoReplace = '[' + leadingText + ']' + matchedTag.completeTag;
                }
                else if (leading.leadingText !== undefined) {
                    stringtoReplace = '[' + leading.leadingText + ']' + matchedTag.completeTag;
                }
                else if (typeof split.linkText !== 'undefined') {
                    stringtoReplace = matchedTag.completeTag;
                }
                else {
                    stringtoReplace = matchedTag.completeTag;
                }
                return originalDescription.replace(stringtoReplace, newLink);
            }
            else if (!resultInCompodoc && typeof split.linkText === 'undefined') {
                newLink = "<a href=\"" + split.target + "\">" + split.target + "</a>";
                if (leadingText) {
                    stringtoReplace = '[' + leadingText + ']' + matchedTag.completeTag;
                }
                else if (leading.leadingText !== undefined) {
                    stringtoReplace = '[' + leading.leadingText + ']' + matchedTag.completeTag;
                }
                else {
                    stringtoReplace = matchedTag.completeTag;
                }
                return originalDescription.replace(stringtoReplace, newLink);
            }
            else {
                return originalDescription;
            }
        };
        function replaceMatch(replacer, tag, match, text, linkText) {
            var matchedTag = {
                completeTag: match,
                tag: tag,
                text: text
            };
            if (linkText) {
                return replacer(description, matchedTag, linkText);
            }
            else {
                return replacer(description, matchedTag);
            }
        }
        // Clean description for marked a tag parsed too early
        if (description.indexOf('href=') !== -1) {
            var insideMarkedATagResults = description.match(/<a [^>]+>([^<]+)<\/a>/g);
            if (insideMarkedATagResults && insideMarkedATagResults.length > 0) {
                for (var i = 0; i < insideMarkedATagResults.length; i++) {
                    var markedATagRegExp = new RegExp('<a [^>]+>([^<]+)</a>', 'gm');
                    var parsedATag = markedATagRegExp.exec(description);
                    if (parsedATag && parsedATag.length === 2) {
                        var insideMarkedATag = parsedATag[1];
                        description = description.replace("{@link <a href=\"" + encodeURI(insideMarkedATag) + "\">" + insideMarkedATag + "</a>", "{@link " + insideMarkedATag);
                    }
                }
            }
        }
        do {
            matches = tagRegExp.exec(description);
            // Did we have {@link ?
            if (matches) {
                previousString = description;
                if (matches.length === 2) {
                    description = replaceMatch(processTheLink, 'link', matches[0], matches[1]);
                }
                if (matches.length === 3) {
                    description = replaceMatch(processTheLink, 'link', matches[0], matches[2], matches[1]);
                }
            }
        } while (matches && previousString !== description);
        return description;
    };
    return ParseDescriptionHelper;
}());

var RelativeURLHelper = /** @class */ (function () {
    function RelativeURLHelper() {
    }
    RelativeURLHelper.prototype.helperFunc = function (context, currentDepth, options) {
        switch (currentDepth) {
            case 0:
                return './';
            case 1:
            case 2:
            case 3:
            case 4:
            case 5:
                return '../'.repeat(currentDepth);
        }
        return '';
    };
    return RelativeURLHelper;
}());

var ShortURLHelper = /** @class */ (function () {
    function ShortURLHelper() {
    }
    ShortURLHelper.prototype.helperFunc = function (context, url, options) {
        var newUrl = url;
        var firstIndexOfSlash = newUrl.indexOf('/');
        var lastIndexOfSlash = newUrl.lastIndexOf('/');
        if (firstIndexOfSlash !== -1 || lastIndexOfSlash !== -1) {
            newUrl =
                newUrl.substr(0, firstIndexOfSlash + 1) +
                    '...' +
                    newUrl.substr(lastIndexOfSlash, newUrl.length);
        }
        return newUrl;
    };
    return ShortURLHelper;
}());

var StripURLHelper = /** @class */ (function () {
    function StripURLHelper() {
    }
    StripURLHelper.prototype.helperFunc = function (context, prefix, url, options) {
        return prefix + url.split("/").pop();
    };
    return StripURLHelper;
}());

var HtmlEngineHelpers = /** @class */ (function () {
    function HtmlEngineHelpers() {
    }
    HtmlEngineHelpers.prototype.registerHelpers = function (bars) {
        this.registerHelper(bars, 'compare', new CompareHelper());
        this.registerHelper(bars, 'or', new OrHelper());
        this.registerHelper(bars, 'functionSignature', new FunctionSignatureHelper());
        this.registerHelper(bars, 'isNotToggle', new IsNotToggleHelper());
        this.registerHelper(bars, 'isInitialTab', new IsInitialTabHelper());
        this.registerHelper(bars, 'isTabEnabled', new IsTabEnabledHelper());
        this.registerHelper(bars, 'ifString', new IfStringHelper());
        this.registerHelper(bars, 'orLength', new OrLengthHelper());
        this.registerHelper(bars, 'filterAngular2Modules', new FilterAngular2ModulesHelper());
        this.registerHelper(bars, 'debug', new DebugHelper());
        this.registerHelper(bars, 'breaklines', new BreakLinesHelper(bars));
        this.registerHelper(bars, 'clean-paragraph', new CleanParagraphHelper());
        this.registerHelper(bars, 'escapeSimpleQuote', new EscapeSimpleQuoteHelper());
        this.registerHelper(bars, 'breakComma', new BreakCommaHelper(bars));
        this.registerHelper(bars, 'modifKind', new ModifKindHelper());
        this.registerHelper(bars, 'modifIcon', new ModifIconHelper());
        this.registerHelper(bars, 'relativeURL', new RelativeURLHelper());
        this.registerHelper(bars, 'jsdoc-returns-comment', new JsdocReturnsCommentHelper());
        this.registerHelper(bars, 'jsdoc-code-example', new JsdocCodeExampleHelper());
        this.registerHelper(bars, 'jsdoc-example', new JsdocExampleHelper());
        this.registerHelper(bars, 'jsdoc-params', new JsdocParamsHelper());
        this.registerHelper(bars, 'jsdoc-params-valid', new JsdocParamsValidHelper());
        this.registerHelper(bars, 'jsdoc-default', new JsdocDefaultHelper());
        this.registerHelper(bars, 'linkType', new LinkTypeHelper());
        this.registerHelper(bars, 'indexableSignature', new IndexableSignatureHelper());
        this.registerHelper(bars, 'object', new ObjectHelper());
        this.registerHelper(bars, 'objectLength', new ObjectLengthHelper());
        this.registerHelper(bars, 'parseDescription', new ParseDescriptionHelper());
        this.registerHelper(bars, 'one-parameter-has', new OneParameterHasHelper());
        this.registerHelper(bars, 'element-alone', new ElementAloneHelper());
        this.registerHelper(bars, 'hasOwn', new HasOwnHelper());
        this.registerHelper(bars, 'short-url', new ShortURLHelper());
        this.registerHelper(bars, 'strip-url', new StripURLHelper());
        this.registerHelper(bars, 't', new I18nHelper());
    };
    HtmlEngineHelpers.prototype.registerHelper = function (bars, key, helper) {
        Handlebars.registerHelper(key, function () {
            // tslint:disable-next-line:no-invalid-this
            return helper.helperFunc.apply(helper, [this].concat(_.slice(arguments)));
        });
    };
    return HtmlEngineHelpers;
}());

var HtmlEngine = /** @class */ (function () {
    function HtmlEngine() {
        this.cache = {};
        var helper = new HtmlEngineHelpers();
        helper.registerHelpers(Handlebars);
    }
    HtmlEngine.getInstance = function () {
        if (!HtmlEngine.instance) {
            HtmlEngine.instance = new HtmlEngine();
        }
        return HtmlEngine.instance;
    };
    HtmlEngine.prototype.init = function (templatePath) {
        var _this = this;
        var partials = [
            'overview',
            'markdown',
            'modules',
            'module',
            'components',
            'component',
            'controller',
            'component-detail',
            'directives',
            'directive',
            'injectables',
            'injectable',
            'interceptor',
            'guard',
            'pipes',
            'pipe',
            'classes',
            'class',
            'interface',
            'routes',
            'index',
            'index-misc',
            'search-results',
            'search-input',
            'link-type',
            'block-method',
            'block-enum',
            'block-property',
            'block-index',
            'block-constructor',
            'block-typealias',
            'block-accessors',
            'block-input',
            'block-output',
            'coverage-report',
            'unit-test-report',
            'miscellaneous-functions',
            'miscellaneous-variables',
            'miscellaneous-typealiases',
            'miscellaneous-enumerations',
            'additional-page',
            'package-dependencies'
        ];
        if (templatePath) {
            if (FileEngine$1.existsSync(path.resolve(process.cwd() + path.sep + templatePath)) ===
                false) {
                logger.warn('Template path specificed but does not exist...using default templates');
            }
        }
        return Promise.all(partials.map(function (partial) {
            var partialPath = _this.determineTemplatePath(templatePath, 'partials/' + partial + '.hbs');
            return FileEngine$1.get(partialPath).then(function (data) {
                return Handlebars.registerPartial(partial, data);
            });
        }))
            .then(function () {
            var pagePath = _this.determineTemplatePath(templatePath, 'page.hbs');
            return FileEngine$1.get(pagePath).then(function (data) {
                _this.cache.page = data;
                _this.compiledPage = Handlebars.compile(_this.cache.page, {
                    preventIndent: true,
                    strict: true
                });
            });
        })
            .then(function () {
            var menuPath = _this.determineTemplatePath(templatePath, 'partials/menu.hbs');
            return FileEngine$1.get(menuPath).then(function (menuTemplate) {
                _this.precompiledMenu = Handlebars.compile(menuTemplate, {
                    preventIndent: true,
                    strict: true
                });
            });
        });
    };
    HtmlEngine.prototype.renderMenu = function (templatePath, data) {
        var menuPath = this.determineTemplatePath(templatePath, 'partials/menu.hbs');
        return FileEngine$1.get(menuPath).then(function (menuTemplate) {
            data.menu = 'normal';
            return Handlebars.compile(menuTemplate, {
                preventIndent: true,
                strict: true
            })(__assign({}, data));
        });
    };
    HtmlEngine.prototype.render = function (mainData, page) {
        var o = mainData;
        Object.assign(o, page);
        // let mem = process.memoryUsage();
        // console.log(`heapTotal: ${mem.heapTotal} | heapUsed: ${mem.heapUsed}`);
        return this.compiledPage({
            data: o
        });
    };
    HtmlEngine.prototype.determineTemplatePath = function (templatePath, filePath) {
        var outPath = path.resolve(__dirname + '/../src/templates/' + filePath);
        if (templatePath) {
            var testPath = path.resolve(process.cwd() + path.sep + templatePath + path.sep + filePath);
            outPath = FileEngine$1.existsSync(testPath) ? testPath : outPath;
        }
        return outPath;
    };
    HtmlEngine.prototype.generateCoverageBadge = function (outputFolder, label, coverageData) {
        return FileEngine$1.get(path.resolve(__dirname + '/../src/templates/partials/coverage-badge.hbs')).then(function (data) {
            var template = Handlebars.compile(data);
            coverageData.label = label;
            var result = template({
                data: coverageData
            });
            var testOutputDir = outputFolder.match(process.cwd());
            if (testOutputDir && testOutputDir.length > 0) {
                outputFolder = outputFolder.replace(process.cwd() + path.sep, '');
            }
            return FileEngine$1.write(outputFolder + path.sep + '/images/coverage-badge-' + label + '.svg', result)["catch"](function (err) {
                logger.error('Error during coverage badge ' + label + ' file generation ', err);
                return Promise.reject(err);
            });
        }, function (err) { return Promise.reject('Error during coverage badge generation'); });
    };
    return HtmlEngine;
}());
var HtmlEngine$1 = HtmlEngine.getInstance();

var decache$1 = require('decache');
var MarkdownEngine = /** @class */ (function () {
    function MarkdownEngine() {
        var _this = this;
        /**
         * List of markdown files without .md extension
         */
        this.markdownFiles = ['README', 'CHANGELOG', 'LICENSE', 'CONTRIBUTING', 'TODO'];
        decache$1('marked');
        this.markedInstance = require('marked');
        var renderer = new this.markedInstance.Renderer();
        renderer.code = function (code, language) {
            var highlighted = code;
            if (!language) {
                language = 'none';
            }
            highlighted = _this.escape(code);
            return "<div><pre class=\"line-numbers\"><code class=\"language-" + language + "\">" + highlighted + "</code></pre></div>";
        };
        renderer.table = function (header, body) {
            return ('<table class="table table-bordered compodoc-table">\n' +
                '<thead>\n' +
                header +
                '</thead>\n' +
                '<tbody>\n' +
                body +
                '</tbody>\n' +
                '</table>\n');
        };
        renderer.image = function (href, title, text) {
            var out = '<img src="' + href + '" alt="' + text + '" class="img-responsive"';
            if (title) {
                out += ' title="' + title + '"';
            }
            out += '>';
            return out;
        };
        this.markedInstance.setOptions({
            renderer: renderer,
            gfm: true,
            breaks: false
        });
    }
    MarkdownEngine.getInstance = function () {
        if (!MarkdownEngine.instance) {
            MarkdownEngine.instance = new MarkdownEngine();
        }
        return MarkdownEngine.instance;
    };
    MarkdownEngine.prototype.getTraditionalMarkdown = function (filepath) {
        var _this = this;
        return FileEngine$1.get(process.cwd() + path.sep + filepath + '.md')["catch"](function (err) { return FileEngine$1.get(process.cwd() + path.sep + filepath).then(); })
            .then(function (data) {
            var returnedData = {
                markdown: _this.markedInstance(data),
                rawData: data
            };
            return returnedData;
        });
    };
    MarkdownEngine.prototype.getTraditionalMarkdownSync = function (filepath) {
        return this.markedInstance(FileEngine$1.getSync(process.cwd() + path.sep + filepath));
    };
    MarkdownEngine.prototype.getReadmeFile = function () {
        var _this = this;
        return FileEngine$1.get(process.cwd() + path.sep + 'README.md').then(function (data) {
            return _this.markedInstance(data);
        });
    };
    MarkdownEngine.prototype.readNeighbourReadmeFile = function (file) {
        var dirname = path.dirname(file);
        var readmeFile = dirname + path.sep + path.basename(file, '.ts') + '.md';
        return fs.readFileSync(readmeFile, 'utf8');
    };
    MarkdownEngine.prototype.hasNeighbourReadmeFile = function (file) {
        var dirname = path.dirname(file);
        var readmeFile = dirname + path.sep + path.basename(file, '.ts') + '.md';
        return FileEngine$1.existsSync(readmeFile);
    };
    MarkdownEngine.prototype.componentReadmeFile = function (file) {
        var dirname = path.dirname(file);
        var readmeFile = dirname + path.sep + 'README.md';
        var readmeAlternativeFile = dirname + path.sep + path.basename(file, '.ts') + '.md';
        var finalPath = '';
        if (FileEngine$1.existsSync(readmeFile)) {
            finalPath = readmeFile;
        }
        else {
            finalPath = readmeAlternativeFile;
        }
        return finalPath;
    };
    /**
     * Checks if any of the markdown files is exists with or without endings
     */
    MarkdownEngine.prototype.hasRootMarkdowns = function () {
        return this.addEndings(this.markdownFiles).some(function (x) {
            return FileEngine$1.existsSync(process.cwd() + path.sep + x);
        });
    };
    MarkdownEngine.prototype.listRootMarkdowns = function () {
        var foundFiles = this.markdownFiles.filter(function (x) {
            return FileEngine$1.existsSync(process.cwd() + path.sep + x + '.md') ||
                FileEngine$1.existsSync(process.cwd() + path.sep + x);
        });
        return this.addEndings(foundFiles);
    };
    MarkdownEngine.prototype.escape = function (html) {
        return html
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
            .replace(/@/g, '&#64;');
    };
    /**
     * ['README'] => ['README', 'README.md']
     */
    MarkdownEngine.prototype.addEndings = function (files) {
        return _.flatMap(files, function (x) { return [x, x + '.md']; });
    };
    return MarkdownEngine;
}());
var MarkdownEngine$1 = MarkdownEngine.getInstance();

var ngdT = require('@compodoc/ngd-transformer');
var NgdEngine = /** @class */ (function () {
    function NgdEngine() {
    }
    NgdEngine.getInstance = function () {
        if (!NgdEngine.instance) {
            NgdEngine.instance = new NgdEngine();
        }
        return NgdEngine.instance;
    };
    NgdEngine.prototype.init = function (outputpath) {
        this.engine = new ngdT.DotEngine({
            output: outputpath,
            displayLegend: true,
            outputFormats: 'svg',
            silent: true
        });
    };
    NgdEngine.prototype.renderGraph = function (filepath, outputpath, type, name) {
        this.engine.updateOutput(outputpath);
        if (type === 'f') {
            return this.engine.generateGraph([DependenciesEngine$1.getRawModule(name)]);
        }
        else {
            return this.engine.generateGraph(DependenciesEngine$1.rawModulesForOverview);
        }
    };
    NgdEngine.prototype.readGraph = function (filepath, name) {
        return FileEngine$1.get(filepath)["catch"](function (err) {
            return Promise.reject('Error during graph read ' + name);
        });
    };
    return NgdEngine;
}());
var NgdEngine$1 = NgdEngine.getInstance();

var COMPODOC_CONSTANTS = {
    navTabDefinitions: [
        {
            id: 'info',
            href: '#info',
            'data-link': 'info',
            label: 'Info',
            depTypes: ['all']
        },
        {
            id: 'readme',
            href: '#readme',
            'data-link': 'readme',
            label: 'README',
            depTypes: ['all']
        },
        {
            id: 'source',
            href: '#source',
            'data-link': 'source',
            label: 'Source',
            depTypes: ['all']
        },
        {
            id: 'templateData',
            href: '#templateData',
            'data-link': 'template',
            label: 'Template',
            depTypes: ['component']
        },
        {
            id: 'styleData',
            href: '#styleData',
            'data-link': 'style',
            label: 'Styles',
            depTypes: ['component']
        },
        {
            id: 'tree',
            href: '#tree',
            'data-link': 'dom-tree',
            label: 'DOM Tree',
            depTypes: ['component']
        },
        {
            id: 'example',
            href: '#example',
            'data-link': 'example',
            label: 'Examples',
            depTypes: ['component', 'directive', 'injectable', 'pipe']
        }
    ]
};
/**
 * Max length for the string of a file during Lunr search engine indexing.
 * Prevent stack size exceeded
 */
var MAX_SIZE_FILE_SEARCH_INDEX = 50000;
/**
 * Max length for the string of a file during cheerio parsing.
 * Prevent stack size exceeded
 */
var MAX_SIZE_FILE_CHEERIO_PARSING = 400000000;

var lunr = require('lunr');
var cheerio = require('cheerio');
var Entities = require('html-entities').AllHtmlEntities;
var Html = new Entities();
var SearchEngine = /** @class */ (function () {
    function SearchEngine() {
        this.searchDocuments = [];
        this.documentsStore = {};
        this.amountOfMemory = 0;
    }
    SearchEngine.getInstance = function () {
        if (!SearchEngine.instance) {
            SearchEngine.instance = new SearchEngine();
        }
        return SearchEngine.instance;
    };
    SearchEngine.prototype.indexPage = function (page) {
        var text;
        this.amountOfMemory += page.rawData.length;
        if (this.amountOfMemory < MAX_SIZE_FILE_CHEERIO_PARSING) {
            var indexStartContent = page.rawData.indexOf('<!-- START CONTENT -->');
            var indexEndContent = page.rawData.indexOf('<!-- END CONTENT -->');
            var $ = cheerio.load(page.rawData.substring(indexStartContent + 1, indexEndContent));
            text = $('.content').html();
            text = Html.decode(text);
            text = text.replace(/(<([^>]+)>)/gi, '');
            page.url = page.url.replace(Configuration$1.mainData.output, '');
            var doc = {
                url: page.url,
                title: page.infos.context + ' - ' + page.infos.name,
                body: text
            };
            if (!this.documentsStore.hasOwnProperty(doc.url) &&
                doc.body.length < MAX_SIZE_FILE_SEARCH_INDEX) {
                this.documentsStore[doc.url] = doc;
                this.searchDocuments.push(doc);
            }
        }
    };
    SearchEngine.prototype.generateSearchIndexJson = function (outputFolder) {
        var _this = this;
        var that = this;
        var searchIndex = lunr(function () {
            /* tslint:disable:no-invalid-this */
            this.ref('url');
            this.field('title');
            this.field('body');
            this.pipeline.remove(lunr.stemmer);
            var i = 0;
            var len = that.searchDocuments.length;
            for (i; i < len; i++) {
                this.add(that.searchDocuments[i]);
            }
        });
        return FileEngine$1.get(__dirname + '/../src/templates/partials/search-index.hbs').then(function (data) {
            var template = Handlebars.compile(data);
            var result = template({
                index: JSON.stringify(searchIndex),
                store: JSON.stringify(_this.documentsStore)
            });
            var testOutputDir = outputFolder.match(process.cwd());
            if (testOutputDir && testOutputDir.length > 0) {
                outputFolder = outputFolder.replace(process.cwd() + path.sep, '');
            }
            return FileEngine$1.write(outputFolder + path.sep + '/js/search/search_index.js', result)["catch"](function (err) {
                logger.error('Error during search index file generation ', err);
                return Promise.reject(err);
            });
        }, function (err) { return Promise.reject('Error during search index generation'); });
    };
    return SearchEngine;
}());
var SearchEngine$1 = SearchEngine.getInstance();

var $ = require('cheerio');
var ComponentsTreeEngine = /** @class */ (function () {
    function ComponentsTreeEngine() {
        this.components = [];
        this.componentsForTree = [];
    }
    ComponentsTreeEngine.getInstance = function () {
        if (!ComponentsTreeEngine.instance) {
            ComponentsTreeEngine.instance = new ComponentsTreeEngine();
        }
        return ComponentsTreeEngine.instance;
    };
    ComponentsTreeEngine.prototype.addComponent = function (component) {
        this.components.push(component);
    };
    ComponentsTreeEngine.prototype.readTemplates = function () {
        var _this = this;
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = _this.componentsForTree.length;
            var loop = function () {
                if (i <= len - 1) {
                    if (_this.componentsForTree[i].templateUrl) {
                        var filePath = process.cwd() +
                            path.sep +
                            path.dirname(_this.componentsForTree[i].file) +
                            path.sep +
                            _this.componentsForTree[i].templateUrl;
                        FileEngine$1.get(filePath).then(function (templateData) {
                            _this.componentsForTree[i].templateData = templateData;
                            i++;
                            loop();
                        }, function (e) {
                            logger.error(e);
                            reject();
                        });
                    }
                    else {
                        _this.componentsForTree[i].templateData = _this.componentsForTree[i].template;
                        i++;
                        loop();
                    }
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    ComponentsTreeEngine.prototype.findChildrenAndParents = function () {
        var _this = this;
        return new Promise(function (resolve, reject) {
            _.forEach(_this.componentsForTree, function (component) {
                var $component = $(component.templateData);
                _.forEach(_this.componentsForTree, function (componentToFind) {
                    if ($component.find(componentToFind.selector).length > 0) {
                        console.log(componentToFind.name + ' found in ' + component.name);
                        component.children.push(componentToFind.name);
                    }
                });
            });
            resolve();
        });
    };
    ComponentsTreeEngine.prototype.createTreesForComponents = function () {
        var _this = this;
        return new Promise(function (resolve, reject) {
            _.forEach(_this.components, function (component) {
                var _component = {
                    name: component.name,
                    file: component.file,
                    selector: component.selector,
                    children: [],
                    template: '',
                    templateUrl: ''
                };
                if (typeof component.template !== 'undefined') {
                    _component.template = component.template;
                }
                if (component.templateUrl.length > 0) {
                    _component.templateUrl = component.templateUrl[0];
                }
                _this.componentsForTree.push(_component);
            });
            _this.readTemplates().then(function () {
                _this.findChildrenAndParents().then(function () {
                    console.log('this.componentsForTree: ', _this.componentsForTree);
                    resolve();
                }, function (e) {
                    logger.error(e);
                    reject();
                });
            }, function (e) {
                logger.error(e);
            });
        });
    };
    return ComponentsTreeEngine;
}());
var ComponentsTreeEngine$1 = ComponentsTreeEngine.getInstance();

var JsdocParserUtil = /** @class */ (function () {
    function JsdocParserUtil() {
    }
    JsdocParserUtil.prototype.isVariableLike = function (node) {
        if (node) {
            switch (node.kind) {
                case Ast.SyntaxKind.BindingElement:
                case Ast.SyntaxKind.EnumMember:
                case Ast.SyntaxKind.Parameter:
                case Ast.SyntaxKind.PropertyAssignment:
                case Ast.SyntaxKind.PropertyDeclaration:
                case Ast.SyntaxKind.PropertySignature:
                case Ast.SyntaxKind.ShorthandPropertyAssignment:
                case Ast.SyntaxKind.VariableDeclaration:
                    return true;
            }
        }
        return false;
    };
    JsdocParserUtil.prototype.getMainCommentOfNode = function (node) {
        var description = '';
        if (node.jsDoc) {
            if (node.jsDoc.length > 0) {
                if (typeof node.jsDoc[0].comment !== 'undefined') {
                    description = node.jsDoc[0].comment;
                }
            }
        }
        return description;
    };
    JsdocParserUtil.prototype.getJSDocTags = function (node, kind) {
        var docs = this.getJSDocs(node);
        if (docs) {
            var result = [];
            for (var _i = 0, docs_1 = docs; _i < docs_1.length; _i++) {
                var doc = docs_1[_i];
                if (Ast.ts.isJSDocParameterTag(doc)) {
                    if (doc.kind === kind) {
                        result.push(doc);
                    }
                }
                else if (Ast.ts.isJSDoc(doc)) {
                    result.push.apply(result, _.filter(doc.tags, function (tag) { return tag.kind === kind; }));
                }
                else {
                    throw new Error('Unexpected type');
                }
            }
            return result;
        }
    };
    JsdocParserUtil.prototype.getJSDocs = function (node) {
        // TODO: jsDocCache is internal, see if there's a way around it
        var cache = node.jsDocCache;
        if (!cache) {
            cache = this.getJSDocsWorker(node, []).filter(function (x) { return x; });
            node.jsDocCache = cache;
        }
        return cache;
    };
    // Try to recognize this pattern when node is initializer
    // of variable declaration and JSDoc comments are on containing variable statement.
    // /**
    //   * @param {number} name
    //   * @returns {number}
    //   */
    // var x = function(name) { return name.length; }
    JsdocParserUtil.prototype.getJSDocsWorker = function (node, cache) {
        var parent = node.parent;
        var isInitializerOfVariableDeclarationInStatement = this.isVariableLike(parent) &&
            parent.initializer === node &&
            Ast.ts.isVariableStatement(parent.parent.parent);
        var isVariableOfVariableDeclarationStatement = this.isVariableLike(node) && Ast.ts.isVariableStatement(parent.parent);
        var variableStatementNode = isInitializerOfVariableDeclarationInStatement
            ? parent.parent.parent
            : isVariableOfVariableDeclarationStatement
                ? parent.parent
                : undefined;
        if (variableStatementNode) {
            cache = this.getJSDocsWorker(variableStatementNode, cache);
        }
        // Also recognize when the node is the RHS of an assignment expression
        var isSourceOfAssignmentExpressionStatement = parent &&
            parent.parent &&
            Ast.ts.isBinaryExpression(parent) &&
            parent.operatorToken.kind === Ast.SyntaxKind.EqualsToken &&
            Ast.ts.isExpressionStatement(parent.parent);
        if (isSourceOfAssignmentExpressionStatement) {
            cache = this.getJSDocsWorker(parent.parent, cache);
        }
        var isModuleDeclaration = Ast.ts.isModuleDeclaration(node) && parent && Ast.ts.isModuleDeclaration(parent);
        var isPropertyAssignmentExpression = parent && Ast.ts.isPropertyAssignment(parent);
        if (isModuleDeclaration || isPropertyAssignmentExpression) {
            cache = this.getJSDocsWorker(parent, cache);
        }
        // Pull parameter comments from declaring function as well
        if (Ast.ts.isParameter(node)) {
            cache = _.concat(cache, this.getJSDocParameterTags(node));
        }
        if (this.isVariableLike(node) && node.initializer) {
            cache = _.concat(cache, node.initializer.jsDoc);
        }
        cache = _.concat(cache, node.jsDoc);
        return cache;
    };
    JsdocParserUtil.prototype.getJSDocParameterTags = function (param) {
        var func = param.parent;
        var tags = this.getJSDocTags(func, Ast.SyntaxKind.JSDocParameterTag);
        if (!param.name) {
            // this is an anonymous jsdoc param from a `function(type1, type2): type3` specification
            var i = func.parameters.indexOf(param);
            var paramTags = _.filter(tags, function (tag) { return Ast.ts.isJSDocParameterTag(tag); });
            if (paramTags && 0 <= i && i < paramTags.length) {
                return [paramTags[i]];
            }
        }
        else if (Ast.ts.isIdentifier(param.name)) {
            var name_1 = param.name.text;
            return _.filter(tags, function (tag) {
                if (Ast.ts && Ast.ts.isJSDocParameterTag(tag)) {
                    var t = tag;
                    if (typeof t.parameterName !== 'undefined') {
                        return t.parameterName.text === name_1;
                    }
                    else if (typeof t.name !== 'undefined') {
                        if (typeof t.name.escapedText !== 'undefined') {
                            return t.name.escapedText === name_1;
                        }
                    }
                }
            });
        }
        else {
            // TODO: it's a destructured parameter, so it should look up an "object type" series of multiple lines
            // But multi-line object types aren't supported yet either
            return undefined;
        }
    };
    return JsdocParserUtil;
}());

var ast = new Ast__default();
var ImportsUtil = /** @class */ (function () {
    function ImportsUtil() {
    }
    ImportsUtil.getInstance = function () {
        if (!ImportsUtil.instance) {
            ImportsUtil.instance = new ImportsUtil();
        }
        return ImportsUtil.instance;
    };
    /**
     * Find for a sourceFile a variable value in a local enum
     * @param srcFile
     * @param variableName
     * @param variableValue
     */
    ImportsUtil.prototype.findInEnums = function (srcFile, variableName, variableValue) {
        var res = '';
        srcFile.getEnum(function (e) {
            if (e.getName() === variableName) {
                e.getMember(function (m) {
                    if (m.getName() === variableValue) {
                        res = m.getValue();
                    }
                });
            }
        });
        return res;
    };
    /**
     * Find for a sourceFile a variable value in a local static class
     * @param srcFile
     * @param variableName
     * @param variableValue
     */
    ImportsUtil.prototype.findInClasses = function (srcFile, variableName, variableValue) {
        var res = '';
        srcFile.getClass(function (c) {
            var staticProperty = c.getStaticProperty(variableValue);
            if (staticProperty) {
                if (staticProperty.getInitializer()) {
                    res = staticProperty.getInitializer().getText();
                }
            }
        });
        return res;
    };
    /**
     * Find a value in a local variable declaration like an object
     * @param variableDeclaration
     * @param variablesAttributes
     */
    ImportsUtil.prototype.findInObjectVariableDeclaration = function (variableDeclaration, variablesAttributes) {
        var variableKind = variableDeclaration.getKind();
        if (variableKind && variableKind === Ast.SyntaxKind.VariableDeclaration) {
            var initializer = variableDeclaration.getInitializer();
            if (initializer) {
                var initializerKind = initializer.getKind();
                if (initializerKind && initializerKind === Ast.SyntaxKind.ObjectLiteralExpression) {
                    var compilerNode = initializer.compilerNode, finalValue_1 = '';
                    // Find thestring from AVAR.BVAR.thestring inside properties
                    var depth_1 = 0;
                    var loopProperties_1 = function (properties) {
                        properties.forEach(function (prop) {
                            if (prop.name) {
                                if (variablesAttributes[depth_1 + 1]) {
                                    if (prop.name.getText() === variablesAttributes[depth_1 + 1]) {
                                        if (prop.initializer) {
                                            if (prop.initializer.properties) {
                                                depth_1 += 1;
                                                loopProperties_1(prop.initializer.properties);
                                            }
                                            else {
                                                finalValue_1 = prop.initializer.text;
                                            }
                                        }
                                        else {
                                            finalValue_1 = prop.initializer.text;
                                        }
                                    }
                                }
                            }
                        });
                    };
                    loopProperties_1(compilerNode.properties);
                    return finalValue_1;
                }
            }
        }
    };
    /**
     * Find in imports something like myvar
     * @param  {string} inputVariableName              like myvar
     * @return {[type]}                                myvar value
     */
    ImportsUtil.prototype.findValueInImportOrLocalVariables = function (inputVariableName, sourceFile) {
        var metadataVariableName = inputVariableName, searchedImport, aliasOriginalName = '', foundWithAlias = false;
        var file = typeof ast.getSourceFile(sourceFile.fileName) !== 'undefined'
            ? ast.getSourceFile(sourceFile.fileName)
            : ast.addExistingSourceFileIfExists(sourceFile.fileName); // tslint:disable-line
        var imports = file.getImportDeclarations();
        /**
         * Loop through all imports, and find one matching inputVariableName
         */
        imports.forEach(function (i) {
            var namedImports = i.getNamedImports(), namedImportsLength = namedImports.length, j = 0;
            if (namedImportsLength > 0) {
                for (j; j < namedImportsLength; j++) {
                    var importName = namedImports[j].getNameNode().getText(), importAlias = void 0;
                    if (namedImports[j].getAliasIdentifier()) {
                        importAlias = namedImports[j].getAliasIdentifier().getText();
                    }
                    if (importName === metadataVariableName) {
                        searchedImport = i;
                        break;
                    }
                    if (importAlias === metadataVariableName) {
                        foundWithAlias = true;
                        aliasOriginalName = importName;
                        searchedImport = i;
                        break;
                    }
                }
            }
        });
        function hasFoundValues(variableDeclaration) {
            var variableKind = variableDeclaration.getKind();
            if (variableKind && variableKind === Ast.SyntaxKind.VariableDeclaration) {
                var initializer = variableDeclaration.getInitializer();
                if (initializer) {
                    var initializerKind = initializer.getKind();
                    if (initializerKind && initializerKind === Ast.SyntaxKind.ObjectLiteralExpression) {
                        var compilerNode = initializer.compilerNode;
                        return compilerNode.properties;
                    }
                }
            }
        }
        if (typeof searchedImport !== 'undefined') {
            var importPathReference = searchedImport.getModuleSpecifierSourceFile();
            var importPath = void 0;
            if (typeof importPathReference !== 'undefined') {
                importPath = importPathReference.compilerNode.fileName;
                var sourceFileImport = typeof ast.getSourceFile(importPath) !== 'undefined'
                    ? ast.getSourceFile(importPath)
                    : ast.addExistingSourceFileIfExists(importPath); // tslint:disable-line
                if (sourceFileImport) {
                    var variableName = foundWithAlias ? aliasOriginalName : metadataVariableName;
                    var variableDeclaration = sourceFileImport.getVariableDeclaration(variableName);
                    if (variableDeclaration) {
                        return hasFoundValues(variableDeclaration);
                    }
                    else {
                        // Try with exports
                        var exportDeclarations = sourceFileImport.getExportDeclarations();
                        if (exportDeclarations && exportDeclarations.length > 0) {
                            var i = 0, len = exportDeclarations.length;
                            for (i; i < len; i++) {
                                var exportDeclaration = exportDeclarations[i];
                                var sourceFileExportedReference = exportDeclaration.getModuleSpecifierSourceFile();
                                if (sourceFileExportedReference) {
                                    var sourceFileExportedReferencePath = sourceFileExportedReference.getFilePath();
                                    var sourceFileExported = typeof ast.getSourceFile(sourceFileExportedReferencePath) !== 'undefined'
                                        ? ast.getSourceFile(sourceFileExportedReferencePath)
                                        : ast.addExistingSourceFileIfExists(sourceFileExportedReferencePath);
                                    if (sourceFileExported) {
                                        variableDeclaration = sourceFileExported.getVariableDeclaration(variableName);
                                        if (variableDeclaration) {
                                            return hasFoundValues(variableDeclaration);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        else {
            // Find in local variables of the file
            var variableDeclaration = file.getVariableDeclaration(metadataVariableName);
            if (variableDeclaration) {
                var variableKind = variableDeclaration.getKind();
                if (variableKind && variableKind === Ast.SyntaxKind.VariableDeclaration) {
                    var initializer = variableDeclaration.getInitializer();
                    if (initializer) {
                        var initializerKind = initializer.getKind();
                        if (initializerKind &&
                            initializerKind === Ast.SyntaxKind.ObjectLiteralExpression) {
                            var compilerNode = initializer.compilerNode;
                            return compilerNode.properties;
                        }
                        else if (initializerKind) {
                            return variableDeclaration.compilerNode;
                        }
                    }
                }
            }
        }
        return [];
    };
    ImportsUtil.prototype.getFileNameOfImport = function (variableName, sourceFile) {
        var file = typeof ast.getSourceFile(sourceFile.fileName) !== 'undefined'
            ? ast.getSourceFile(sourceFile.fileName)
            : ast.addExistingSourceFile(sourceFile.fileName); // tslint:disable-line
        var imports = file.getImportDeclarations();
        var searchedImport, finalPath = '';
        imports.forEach(function (i) {
            var namedImports = i.getNamedImports(), namedImportsLength = namedImports.length, j = 0;
            if (namedImportsLength > 0) {
                for (j; j < namedImportsLength; j++) {
                    var importName = namedImports[j].getNameNode().getText(), importAlias = void 0;
                    if (namedImports[j].getAliasIdentifier()) {
                        importAlias = namedImports[j].getAliasIdentifier().getText();
                    }
                    if (importName === variableName) {
                        searchedImport = i;
                        break;
                    }
                    if (importAlias === variableName) {
                        searchedImport = i;
                        break;
                    }
                }
            }
        });
        if (typeof searchedImport !== 'undefined') {
            var importPath = path.resolve(path.dirname(sourceFile.fileName) +
                '/' +
                searchedImport.getModuleSpecifierValue() +
                '.ts');
            var cleaner = (process.cwd() + path.sep).replace(/\\/g, '/');
            finalPath = importPath.replace(cleaner, '');
        }
        return finalPath;
    };
    /**
     * Find the file path of imported variable
     * @param  {string} inputVariableName  like thestring
     * @return {[type]}                    thestring destination path
     */
    ImportsUtil.prototype.findFilePathOfImportedVariable = function (inputVariableName, sourceFilePath) {
        var searchedImport, finalPath = '';
        var file = typeof ast.getSourceFile(sourceFilePath) !== 'undefined'
            ? ast.getSourceFile(sourceFilePath)
            : ast.addExistingSourceFile(sourceFilePath); // tslint:disable-line
        var imports = file.getImportDeclarations();
        /**
         * Loop through all imports, and find one matching inputVariableName
         */
        imports.forEach(function (i) {
            var namedImports = i.getNamedImports(), namedImportsLength = namedImports.length, j = 0;
            if (namedImportsLength > 0) {
                for (j; j < namedImportsLength; j++) {
                    var importName = namedImports[j].getNameNode().getText(), importAlias = void 0;
                    if (namedImports[j].getAliasIdentifier()) {
                        importAlias = namedImports[j].getAliasIdentifier().getText();
                    }
                    if (importName === inputVariableName) {
                        searchedImport = i;
                        break;
                    }
                    if (importAlias === inputVariableName) {
                        searchedImport = i;
                        break;
                    }
                }
            }
        });
        if (typeof searchedImport !== 'undefined') {
            finalPath = path.resolve(path.dirname(sourceFilePath) +
                '/' +
                searchedImport.getModuleSpecifierValue() +
                '.ts');
        }
        return finalPath;
    };
    /**
     * Find in imports something like VAR.AVAR.BVAR.thestring
     * @param  {string} inputVariableName                   like VAR.AVAR.BVAR.thestring
     * @return {[type]}                                thestring value
     */
    ImportsUtil.prototype.findPropertyValueInImportOrLocalVariables = function (inputVariableName, sourceFile) {
        var variablesAttributes = inputVariableName.split('.'), metadataVariableName = variablesAttributes[0], searchedImport, aliasOriginalName = '', foundWithAlias = false;
        var file = typeof ast.getSourceFile(sourceFile.fileName) !== 'undefined'
            ? ast.getSourceFile(sourceFile.fileName)
            : ast.addExistingSourceFile(sourceFile.fileName); // tslint:disable-line
        var imports = file.getImportDeclarations();
        /**
         * Loop through all imports, and find one matching inputVariableName
         */
        imports.forEach(function (i) {
            var namedImports = i.getNamedImports(), namedImportsLength = namedImports.length, j = 0;
            if (namedImportsLength > 0) {
                for (j; j < namedImportsLength; j++) {
                    var importName = namedImports[j].getNameNode().getText(), importAlias = void 0;
                    if (namedImports[j].getAliasIdentifier()) {
                        importAlias = namedImports[j].getAliasIdentifier().getText();
                    }
                    if (importName === metadataVariableName) {
                        searchedImport = i;
                        break;
                    }
                    if (importAlias === metadataVariableName) {
                        foundWithAlias = true;
                        aliasOriginalName = importName;
                        searchedImport = i;
                        break;
                    }
                }
            }
        });
        var fileToSearchIn, variableDeclaration;
        if (typeof searchedImport !== 'undefined') {
            var importPath = path.resolve(path.dirname(sourceFile.fileName) +
                '/' +
                searchedImport.getModuleSpecifierValue() +
                '.ts');
            var sourceFileImport = typeof ast.getSourceFile(importPath) !== 'undefined'
                ? ast.getSourceFile(importPath)
                : ast.addExistingSourceFile(importPath); // tslint:disable-line
            if (sourceFileImport) {
                fileToSearchIn = sourceFileImport;
                var variableName = foundWithAlias ? aliasOriginalName : metadataVariableName;
                variableDeclaration = fileToSearchIn.getVariableDeclaration(variableName);
            }
        }
        else {
            fileToSearchIn = file;
            // Find in local variables of the file
            variableDeclaration = fileToSearchIn.getVariableDeclaration(metadataVariableName);
        }
        if (variableDeclaration) {
            return this.findInObjectVariableDeclaration(variableDeclaration, variablesAttributes);
        }
        // Try find it in enums
        if (variablesAttributes.length > 0) {
            if (typeof fileToSearchIn !== 'undefined') {
                var val = this.findInEnums(fileToSearchIn, metadataVariableName, variablesAttributes[1]);
                if (val !== '') {
                    return val;
                }
                val = this.findInClasses(fileToSearchIn, metadataVariableName, variablesAttributes[1]);
                if (val !== '') {
                    return val;
                }
            }
        }
    };
    return ImportsUtil;
}());
var ImportsUtil$1 = ImportsUtil.getInstance();

var traverse$2 = require('traverse');
var ast$1 = new Ast__default();
var RouterParserUtil = /** @class */ (function () {
    function RouterParserUtil() {
        this.routes = [];
        this.incompleteRoutes = [];
        this.modules = [];
        this.modulesWithRoutes = [];
        this.transformAngular8ImportSyntax = /(['"]loadChildren['"]:)\(\)=>"import\((\\'|'|")([^'"]+?)(\\'|'|")\)\.then\(\w+?=>\S+?\.([^)]+?)\)(\\'|'|")/g;
    }
    RouterParserUtil.getInstance = function () {
        if (!RouterParserUtil.instance) {
            RouterParserUtil.instance = new RouterParserUtil();
        }
        return RouterParserUtil.instance;
    };
    RouterParserUtil.prototype.addRoute = function (route) {
        this.routes.push(route);
        this.routes = _.sortBy(_.uniqWith(this.routes, _.isEqual), ['name']);
    };
    RouterParserUtil.prototype.addIncompleteRoute = function (route) {
        this.incompleteRoutes.push(route);
        this.incompleteRoutes = _.sortBy(_.uniqWith(this.incompleteRoutes, _.isEqual), ['name']);
    };
    RouterParserUtil.prototype.addModuleWithRoutes = function (moduleName, moduleImports, filename) {
        this.modulesWithRoutes.push({
            name: moduleName,
            importsNode: moduleImports,
            filename: filename
        });
        this.modulesWithRoutes = _.sortBy(_.uniqWith(this.modulesWithRoutes, _.isEqual), ['name']);
    };
    RouterParserUtil.prototype.addModule = function (moduleName, moduleImports) {
        this.modules.push({
            name: moduleName,
            importsNode: moduleImports
        });
        this.modules = _.sortBy(_.uniqWith(this.modules, _.isEqual), ['name']);
    };
    RouterParserUtil.prototype.cleanRawRouteParsed = function (route) {
        var routesWithoutSpaces = route.replace(/ /gm, '');
        var testTrailingComma = routesWithoutSpaces.indexOf('},]');
        if (testTrailingComma !== -1) {
            routesWithoutSpaces = routesWithoutSpaces.replace('},]', '}]');
        }
        routesWithoutSpaces = routesWithoutSpaces.replace(this.transformAngular8ImportSyntax, '$1"$3#$5"');
        return JSON5.parse(routesWithoutSpaces);
    };
    RouterParserUtil.prototype.cleanRawRoute = function (route) {
        var routesWithoutSpaces = route.replace(/ /gm, '');
        var testTrailingComma = routesWithoutSpaces.indexOf('},]');
        if (testTrailingComma !== -1) {
            routesWithoutSpaces = routesWithoutSpaces.replace('},]', '}]');
        }
        routesWithoutSpaces = routesWithoutSpaces.replace(this.transformAngular8ImportSyntax, '$1"$3#$5"');
        return routesWithoutSpaces;
    };
    RouterParserUtil.prototype.setRootModule = function (module) {
        this.rootModule = module;
    };
    RouterParserUtil.prototype.hasRouterModuleInImports = function (imports) {
        for (var i = 0; i < imports.length; i++) {
            if (imports[i].name.indexOf('RouterModule.forChild') !== -1 ||
                imports[i].name.indexOf('RouterModule.forRoot') !== -1 ||
                imports[i].name.indexOf('RouterModule') !== -1) {
                return true;
            }
        }
        return false;
    };
    RouterParserUtil.prototype.fixIncompleteRoutes = function (miscellaneousVariables) {
        var matchingVariables = [];
        // For each incompleteRoute, scan if one misc variable is in code
        // if ok, try recreating complete route
        for (var i = 0; i < this.incompleteRoutes.length; i++) {
            for (var j = 0; j < miscellaneousVariables.length; j++) {
                if (this.incompleteRoutes[i].data.indexOf(miscellaneousVariables[j].name) !== -1) {
                    console.log('found one misc var inside incompleteRoute');
                    console.log(miscellaneousVariables[j].name);
                    matchingVariables.push(miscellaneousVariables[j]);
                }
            }
            // Clean incompleteRoute
            this.incompleteRoutes[i].data = this.incompleteRoutes[i].data.replace('[', '');
            this.incompleteRoutes[i].data = this.incompleteRoutes[i].data.replace(']', '');
        }
    };
    RouterParserUtil.prototype.linkModulesAndRoutes = function () {
        var _this = this;
        var i = 0;
        var len = this.modulesWithRoutes.length;
        for (i; i < len; i++) {
            _.forEach(this.modulesWithRoutes[i].importsNode, function (node) {
                var initializer = node.initializer;
                if (initializer) {
                    if (initializer.elements) {
                        _.forEach(initializer.elements, function (element) {
                            // find element with arguments
                            if (element.arguments) {
                                _.forEach(element.arguments, function (argument) {
                                    _.forEach(_this.routes, function (route) {
                                        if (argument.text &&
                                            route.name === argument.text &&
                                            route.filename === _this.modulesWithRoutes[i].filename) {
                                            route.module = _this.modulesWithRoutes[i].name;
                                        }
                                        else if (argument.text &&
                                            route.name === argument.text &&
                                            route.filename !== _this.modulesWithRoutes[i].filename) {
                                            var argumentImportPath = ImportsUtil$1.findFilePathOfImportedVariable(argument.text, _this.modulesWithRoutes[i].filename);
                                            argumentImportPath = argumentImportPath
                                                .replace(process.cwd() + path.sep, '')
                                                .replace(/\\/g, '/');
                                            if (argument.text &&
                                                route.name === argument.text &&
                                                route.filename === argumentImportPath) {
                                                route.module = _this.modulesWithRoutes[i].name;
                                            }
                                        }
                                    });
                                });
                            }
                        });
                    }
                }
                /**
                 * direct support of for example
                 * export const HomeRoutingModule: ModuleWithProviders = RouterModule.forChild(HOME_ROUTES);
                 */
                if (Ast.ts.isCallExpression(node)) {
                    if (node.arguments) {
                        _.forEach(node.arguments, function (argument) {
                            _.forEach(_this.routes, function (route) {
                                if (argument.text &&
                                    route.name === argument.text &&
                                    route.filename === _this.modulesWithRoutes[i].filename) {
                                    route.module = _this.modulesWithRoutes[i].name;
                                }
                            });
                        });
                    }
                }
            });
        }
    };
    RouterParserUtil.prototype.foundRouteWithModuleName = function (moduleName) {
        return _.find(this.routes, { module: moduleName });
    };
    RouterParserUtil.prototype.foundLazyModuleWithPath = function (modulePath) {
        // path is like app/customers/customers.module#CustomersModule
        var split = modulePath.split('#');
        var lazyModulePath = split[0];
        var lazyModuleName = split[1];
        return lazyModuleName;
    };
    RouterParserUtil.prototype.constructRoutesTree = function () {
        var _this = this;
        // routes[] contains routes with module link
        // modulesTree contains modules tree
        // make a final routes tree with that
        traverse$2(this.modulesTree).forEach(function (node) {
            if (node) {
                if (node.parent) {
                    delete node.parent;
                }
                if (node.initializer) {
                    delete node.initializer;
                }
                if (node.importsNode) {
                    delete node.importsNode;
                }
            }
        });
        this.cleanModulesTree = _.cloneDeep(this.modulesTree);
        var routesTree = {
            name: '<root>',
            kind: 'module',
            className: this.rootModule,
            children: []
        };
        var loopModulesParser = function (node) {
            if (node.children && node.children.length > 0) {
                // If module has child modules
                for (var i in node.children) {
                    var route = _this.foundRouteWithModuleName(node.children[i].name);
                    if (route && route.data) {
                        try {
                            route.children = JSON5.parse(route.data);
                        }
                        catch (e) {
                            logger.error('Error during generation of routes JSON file, maybe a trailing comma or an external variable inside one route.');
                        }
                        delete route.data;
                        route.kind = 'module';
                        routesTree.children.push(route);
                    }
                    if (node.children[i].children) {
                        loopModulesParser(node.children[i]);
                    }
                }
            }
            else {
                // else routes are directly inside the module
                var rawRoutes = _this.foundRouteWithModuleName(node.name);
                if (rawRoutes) {
                    var routes = JSON5.parse(rawRoutes.data);
                    if (routes) {
                        var i = 0;
                        var len = routes.length;
                        var routeAddedOnce = false;
                        for (i; i < len; i++) {
                            var route = routes[i];
                            if (routes[i].component) {
                                routeAddedOnce = true;
                                routesTree.children.push({
                                    kind: 'component',
                                    component: routes[i].component,
                                    path: routes[i].path
                                });
                            }
                        }
                        if (!routeAddedOnce) {
                            routesTree.children = routesTree.children.concat(routes);
                        }
                    }
                }
            }
        };
        var startModule = _.find(this.cleanModulesTree, { name: this.rootModule });
        if (startModule) {
            loopModulesParser(startModule);
            // Loop twice for routes with lazy loading
            // loopModulesParser(routesTree);
        }
        var cleanedRoutesTree = undefined;
        var cleanRoutesTree = function (route) {
            for (var i in route.children) {
                var routes = route.children[i].routes;
            }
            return route;
        };
        cleanedRoutesTree = cleanRoutesTree(routesTree);
        // Try updating routes with lazy loading
        var loopInsideModule = function (mod, _rawModule) {
            if (mod.children) {
                for (var z in mod.children) {
                    var route = _this.foundRouteWithModuleName(mod.children[z].name);
                    if (typeof route !== 'undefined') {
                        if (route.data) {
                            route.children = JSON5.parse(route.data);
                            delete route.data;
                            route.kind = 'module';
                            _rawModule.children.push(route);
                        }
                    }
                }
            }
            else {
                var route = _this.foundRouteWithModuleName(mod.name);
                if (typeof route !== 'undefined') {
                    if (route.data) {
                        route.children = JSON5.parse(route.data);
                        delete route.data;
                        route.kind = 'module';
                        _rawModule.children.push(route);
                    }
                }
            }
        };
        var loopRoutesParser = function (route) {
            if (route.children) {
                for (var i in route.children) {
                    if (route.children[i].loadChildren) {
                        var child = _this.foundLazyModuleWithPath(route.children[i].loadChildren);
                        var module = _.find(_this.cleanModulesTree, {
                            name: child
                        });
                        if (module) {
                            var _rawModule = {};
                            _rawModule.kind = 'module';
                            _rawModule.children = [];
                            _rawModule.module = module.name;
                            loopInsideModule(module, _rawModule);
                            route.children[i].children = [];
                            route.children[i].children.push(_rawModule);
                        }
                    }
                    loopRoutesParser(route.children[i]);
                }
            }
        };
        loopRoutesParser(cleanedRoutesTree);
        return cleanedRoutesTree;
    };
    RouterParserUtil.prototype.constructModulesTree = function () {
        var _this = this;
        var getNestedChildren = function (arr, parent) {
            var out = [];
            for (var i in arr) {
                if (arr[i].parent === parent) {
                    var children = getNestedChildren(arr, arr[i].name);
                    if (children.length) {
                        arr[i].children = children;
                    }
                    out.push(arr[i]);
                }
            }
            return out;
        };
        // Scan each module and add parent property
        _.forEach(this.modules, function (firstLoopModule) {
            _.forEach(firstLoopModule.importsNode, function (importNode) {
                _.forEach(_this.modules, function (module) {
                    if (module.name === importNode.name) {
                        module.parent = firstLoopModule.name;
                    }
                });
            });
        });
        this.modulesTree = getNestedChildren(this.modules);
    };
    RouterParserUtil.prototype.generateRoutesIndex = function (outputFolder, routes) {
        return FileEngine$1.get(__dirname + '/../src/templates/partials/routes-index.hbs').then(function (data) {
            var template = Handlebars.compile(data);
            var result = template({
                routes: JSON.stringify(routes)
            });
            var testOutputDir = outputFolder.match(process.cwd());
            if (testOutputDir && testOutputDir.length > 0) {
                outputFolder = outputFolder.replace(process.cwd() + path.sep, '');
            }
            return FileEngine$1.write(outputFolder + path.sep + '/js/routes/routes_index.js', result);
        }, function (err) { return Promise.reject('Error during routes index generation'); });
    };
    RouterParserUtil.prototype.routesLength = function () {
        var _n = 0;
        var routesParser = function (route) {
            if (typeof route.path !== 'undefined') {
                _n += 1;
            }
            if (route.children) {
                for (var j in route.children) {
                    routesParser(route.children[j]);
                }
            }
        };
        for (var i in this.routes) {
            routesParser(this.routes[i]);
        }
        return _n;
    };
    RouterParserUtil.prototype.printRoutes = function () {
        console.log('');
        console.log('printRoutes: ');
        console.log(this.routes);
    };
    RouterParserUtil.prototype.printModulesRoutes = function () {
        console.log('');
        console.log('printModulesRoutes: ');
        console.log(this.modulesWithRoutes);
    };
    RouterParserUtil.prototype.isVariableRoutes = function (node) {
        var result = false;
        if (node.declarationList && node.declarationList.declarations) {
            var i = 0;
            var len = node.declarationList.declarations.length;
            for (i; i < len; i++) {
                if (node.declarationList.declarations[i].type) {
                    if (node.declarationList.declarations[i].type.typeName &&
                        node.declarationList.declarations[i].type.typeName.text === 'Routes') {
                        result = true;
                    }
                }
            }
        }
        return result;
    };
    RouterParserUtil.prototype.cleanFileIdentifiers = function (sourceFile) {
        var _this = this;
        var file = sourceFile;
        var identifiers = file.getDescendantsOfKind(Ast.SyntaxKind.Identifier).filter(function (p) {
            return (Ast.TypeGuards.isArrayLiteralExpression(p.getParentOrThrow()) ||
                Ast.TypeGuards.isPropertyAssignment(p.getParentOrThrow()));
        });
        var identifiersInRoutesVariableStatement = [];
        var _loop_1 = function (identifier) {
            // Loop through their parents nodes, and if one is a variableStatement and === 'routes'
            var foundParentVariableStatement = false;
            var parent = identifier.getParentWhile(function (n) {
                if (n.getKind() === Ast.SyntaxKind.VariableStatement) {
                    if (_this.isVariableRoutes(n.compilerNode)) {
                        foundParentVariableStatement = true;
                    }
                }
                return true;
            });
            if (foundParentVariableStatement) {
                identifiersInRoutesVariableStatement.push(identifier);
            }
        };
        for (var _i = 0, identifiers_1 = identifiers; _i < identifiers_1.length; _i++) {
            var identifier = identifiers_1[_i];
            _loop_1(identifier);
        }
        // inline the property access expressions
        for (var _a = 0, identifiersInRoutesVariableStatement_1 = identifiersInRoutesVariableStatement; _a < identifiersInRoutesVariableStatement_1.length; _a++) {
            var identifier = identifiersInRoutesVariableStatement_1[_a];
            var identifierDeclaration = identifier
                .getSymbolOrThrow()
                .getValueDeclarationOrThrow();
            if (!Ast.TypeGuards.isPropertyAssignment(identifierDeclaration) &&
                Ast.TypeGuards.isVariableDeclaration(identifierDeclaration) &&
                (Ast.TypeGuards.isPropertyAssignment(identifierDeclaration) &&
                    !Ast.TypeGuards.isVariableDeclaration(identifierDeclaration))) {
                throw new Error("Not implemented referenced declaration kind: " + identifierDeclaration.getKindName());
            }
            if (Ast.TypeGuards.isVariableDeclaration(identifierDeclaration)) {
                identifier.replaceWithText(identifierDeclaration.getInitializerOrThrow().getText());
            }
        }
        return file;
    };
    RouterParserUtil.prototype.cleanFileSpreads = function (sourceFile) {
        var _this = this;
        var file = sourceFile;
        var spreadElements = file
            .getDescendantsOfKind(Ast.SyntaxKind.SpreadElement)
            .filter(function (p) { return Ast.TypeGuards.isArrayLiteralExpression(p.getParentOrThrow()); });
        var spreadElementsInRoutesVariableStatement = [];
        var _loop_2 = function (spreadElement) {
            // Loop through their parents nodes, and if one is a variableStatement and === 'routes'
            var foundParentVariableStatement = false;
            var parent = spreadElement.getParentWhile(function (n) {
                if (n.getKind() === Ast.SyntaxKind.VariableStatement) {
                    if (_this.isVariableRoutes(n.compilerNode)) {
                        foundParentVariableStatement = true;
                    }
                }
                return true;
            });
            if (foundParentVariableStatement) {
                spreadElementsInRoutesVariableStatement.push(spreadElement);
            }
        };
        for (var _i = 0, spreadElements_1 = spreadElements; _i < spreadElements_1.length; _i++) {
            var spreadElement = spreadElements_1[_i];
            _loop_2(spreadElement);
        }
        var _loop_3 = function (spreadElement) {
            var spreadElementIdentifier = spreadElement.getExpression().getText(), searchedImport, aliasOriginalName = '', foundWithAliasInImports = false, foundWithAlias = false;
            // Try to find it in imports
            var imports = file.getImportDeclarations();
            imports.forEach(function (i) {
                var namedImports = i.getNamedImports(), namedImportsLength = namedImports.length, j = 0;
                if (namedImportsLength > 0) {
                    for (j; j < namedImportsLength; j++) {
                        var importName = namedImports[j].getNameNode().getText(), importAlias = void 0;
                        if (namedImports[j].getAliasIdentifier()) {
                            importAlias = namedImports[j].getAliasIdentifier().getText();
                        }
                        if (importName === spreadElementIdentifier) {
                            foundWithAliasInImports = true;
                            searchedImport = i;
                            break;
                        }
                        if (importAlias === spreadElementIdentifier) {
                            foundWithAliasInImports = true;
                            foundWithAlias = true;
                            aliasOriginalName = importName;
                            searchedImport = i;
                            break;
                        }
                    }
                }
            });
            var referencedDeclaration = void 0;
            if (foundWithAliasInImports) {
                if (typeof searchedImport !== 'undefined') {
                    var importPath = path.resolve(path.dirname(file.getFilePath()) +
                        '/' +
                        searchedImport.getModuleSpecifierValue() +
                        '.ts');
                    var sourceFileImport = typeof ast$1.getSourceFile(importPath) !== 'undefined'
                        ? ast$1.getSourceFile(importPath)
                        : ast$1.addExistingSourceFile(importPath);
                    if (sourceFileImport) {
                        var variableName = foundWithAlias
                            ? aliasOriginalName
                            : spreadElementIdentifier;
                        referencedDeclaration = sourceFileImport.getVariableDeclaration(variableName);
                    }
                }
            }
            else {
                // if not, try directly in file
                referencedDeclaration = spreadElement
                    .getExpression()
                    .getSymbolOrThrow()
                    .getValueDeclarationOrThrow();
            }
            if (!Ast.TypeGuards.isVariableDeclaration(referencedDeclaration)) {
                throw new Error("Not implemented referenced declaration kind: " + referencedDeclaration.getKindName());
            }
            var referencedArray = referencedDeclaration.getInitializerIfKindOrThrow(Ast.SyntaxKind.ArrayLiteralExpression);
            var spreadElementArray = spreadElement.getParentIfKindOrThrow(Ast.SyntaxKind.ArrayLiteralExpression);
            var insertIndex = spreadElementArray.getElements().indexOf(spreadElement);
            spreadElementArray.removeElement(spreadElement);
            spreadElementArray.insertElements(insertIndex, referencedArray.getElements().map(function (e) { return e.getText(); }));
        };
        // inline the ArrayLiteralExpression SpreadElements
        for (var _a = 0, spreadElementsInRoutesVariableStatement_1 = spreadElementsInRoutesVariableStatement; _a < spreadElementsInRoutesVariableStatement_1.length; _a++) {
            var spreadElement = spreadElementsInRoutesVariableStatement_1[_a];
            _loop_3(spreadElement);
        }
        return file;
    };
    RouterParserUtil.prototype.cleanFileDynamics = function (sourceFile) {
        var _this = this;
        var file = sourceFile;
        var propertyAccessExpressions = file
            .getDescendantsOfKind(Ast.SyntaxKind.PropertyAccessExpression)
            .filter(function (p) { return !Ast.TypeGuards.isPropertyAccessExpression(p.getParentOrThrow()); });
        var propertyAccessExpressionsInRoutesVariableStatement = [];
        var _loop_4 = function (propertyAccessExpression) {
            // Loop through their parents nodes, and if one is a variableStatement and === 'routes'
            var foundParentVariableStatement = false;
            var parent = propertyAccessExpression.getParentWhile(function (n) {
                if (n.getKind() === Ast.SyntaxKind.VariableStatement) {
                    if (_this.isVariableRoutes(n.compilerNode)) {
                        foundParentVariableStatement = true;
                    }
                }
                return true;
            });
            if (foundParentVariableStatement) {
                propertyAccessExpressionsInRoutesVariableStatement.push(propertyAccessExpression);
            }
        };
        for (var _i = 0, propertyAccessExpressions_1 = propertyAccessExpressions; _i < propertyAccessExpressions_1.length; _i++) {
            var propertyAccessExpression = propertyAccessExpressions_1[_i];
            _loop_4(propertyAccessExpression);
        }
        // inline the property access expressions
        for (var _a = 0, propertyAccessExpressionsInRoutesVariableStatement_1 = propertyAccessExpressionsInRoutesVariableStatement; _a < propertyAccessExpressionsInRoutesVariableStatement_1.length; _a++) {
            var propertyAccessExpression = propertyAccessExpressionsInRoutesVariableStatement_1[_a];
            var referencedDeclaration = propertyAccessExpression
                .getNameNode()
                .getSymbolOrThrow()
                .getValueDeclarationOrThrow();
            if (!Ast.TypeGuards.isPropertyAssignment(referencedDeclaration) &&
                Ast.TypeGuards.isEnumMember(referencedDeclaration) &&
                (Ast.TypeGuards.isPropertyAssignment(referencedDeclaration) &&
                    !Ast.TypeGuards.isEnumMember(referencedDeclaration))) {
                throw new Error("Not implemented referenced declaration kind: " + referencedDeclaration.getKindName());
            }
            if (typeof referencedDeclaration.getInitializerOrThrow !== 'undefined') {
                propertyAccessExpression.replaceWithText(referencedDeclaration.getInitializerOrThrow().getText());
            }
        }
        return file;
    };
    /**
     * replace callexpressions with string : utils.doWork() -> 'utils.doWork()' doWork() -> 'doWork()'
     * @param sourceFile ts.SourceFile
     */
    RouterParserUtil.prototype.cleanCallExpressions = function (sourceFile) {
        var file = sourceFile;
        var variableStatements = sourceFile.getVariableDeclaration(function (v) {
            var result = false;
            if (typeof v.compilerNode.type !== 'undefined') {
                result = v.compilerNode.type.typeName.text === 'Routes';
            }
            return result;
        });
        var initializer = variableStatements.getInitializer();
        var _loop_5 = function (callExpr) {
            if (callExpr.wasForgotten()) {
                return "continue";
            }
            callExpr.replaceWithText(function (writer) { return writer.quote(callExpr.getText()); });
        };
        for (var _i = 0, _a = initializer.getDescendantsOfKind(Ast.SyntaxKind.CallExpression); _i < _a.length; _i++) {
            var callExpr = _a[_i];
            _loop_5(callExpr);
        }
        return file;
    };
    /**
     * Clean routes definition with imported data, for example path, children, or dynamic stuff inside data
     *
     * const MY_ROUTES: Routes = [
     *     {
     *         path: 'home',
     *         component: HomeComponent
     *     },
     *     {
     *         path: PATHS.home,
     *         component: HomeComponent
     *     }
     * ];
     *
     * The initializer is an array (ArrayLiteralExpression - 177 ), it has elements, objects (ObjectLiteralExpression - 178)
     * with properties (PropertyAssignment - 261)
     *
     * For each know property (https://angular.io/api/router/Routes#description), we try to see if we have what we want
     *
     * Ex: path and pathMatch want a string, component a component reference.
     *
     * It is an imperative approach, not a generic way, parsing all the tree
     * and find something like this which willl break JSON.stringify : MYIMPORT.path
     *
     * @param  {ts.Node} initializer The node of routes definition
     * @return {ts.Node}             The edited node
     */
    RouterParserUtil.prototype.cleanRoutesDefinitionWithImport = function (initializer, node, sourceFile) {
        initializer.elements.forEach(function (element) {
            element.properties.forEach(function (property) {
                var propertyName = property.name.getText(), propertyInitializer = property.initializer;
                switch (propertyName) {
                    case 'path':
                    case 'redirectTo':
                    case 'outlet':
                    case 'pathMatch':
                        if (propertyInitializer) {
                            if (propertyInitializer.kind !== Ast.SyntaxKind.StringLiteral) {
                                // Identifier(71) won't break parsing, but it will be better to retrive them
                                // PropertyAccessExpression(179) ex: MYIMPORT.path will break it, find it in import
                                if (propertyInitializer.kind === Ast.SyntaxKind.PropertyAccessExpression) {
                                    var lastObjectLiteralAttributeName = propertyInitializer.name.getText(), firstObjectLiteralAttributeName = void 0;
                                    if (propertyInitializer.expression) {
                                        firstObjectLiteralAttributeName = propertyInitializer.expression.getText();
                                        var result = ImportsUtil$1.findPropertyValueInImportOrLocalVariables(firstObjectLiteralAttributeName +
                                            '.' +
                                            lastObjectLiteralAttributeName, sourceFile); // tslint:disable-line
                                        if (result !== '') {
                                            propertyInitializer.kind = 9;
                                            propertyInitializer.text = result;
                                        }
                                    }
                                }
                            }
                        }
                        break;
                }
            });
        });
        return initializer;
    };
    return RouterParserUtil;
}());
var RouterParserUtil$1 = RouterParserUtil.getInstance();

function isModuleWithProviders(node) {
    var result = false;
    if (node.declarationList) {
        if (node.declarationList.declarations && node.declarationList.declarations.length > 0) {
            var i = 0, declarations = node.declarationList.declarations, len = node.declarationList.declarations.length;
            for (i; i < len; i++) {
                var declaration = node.declarationList.declarations[i];
                if (declaration.type) {
                    var type = declaration.type;
                    if (type.typeName) {
                        var text = type.typeName.getText();
                        if (text === 'ModuleWithProviders') {
                            result = true;
                        }
                    }
                }
            }
        }
    }
    return result;
}

function getModuleWithProviders(node) {
    var result;
    if (node.declarationList) {
        if (node.declarationList.declarations && node.declarationList.declarations.length > 0) {
            var i = 0, len = node.declarationList.declarations.length;
            for (i; i < len; i++) {
                var declaration = node.declarationList.declarations[i];
                if (declaration.type) {
                    var type = declaration.type;
                    if (type.typeName) {
                        var text = type.typeName.getText();
                        if (text === 'ModuleWithProviders') {
                            result = declaration.initializer;
                        }
                    }
                }
            }
        }
    }
    return result;
}

function StringifyObjectLiteralExpression(ole) {
    var returnedString = '{';
    if (ole.properties && ole.properties.length > 0) {
        ole.properties.forEach(function (property, index) {
            if (property.name) {
                returnedString += property.name.text + ': ';
            }
            if (property.initializer) {
                if (property.initializer.kind === Ast.SyntaxKind.StringLiteral) {
                    returnedString += "'" + property.initializer.text + "'";
                }
                else if (property.initializer.kind === Ast.SyntaxKind.TrueKeyword) {
                    returnedString += "true";
                }
                else if (property.initializer.kind === Ast.SyntaxKind.FalseKeyword) {
                    returnedString += "false";
                }
                else {
                    returnedString += property.initializer.text;
                }
            }
            if (index < ole.properties.length - 1) {
                returnedString += ', ';
            }
        });
    }
    returnedString += '}';
    return returnedString;
}

var crypto = require('crypto');
var marked$1 = require('marked');
var ClassHelper = /** @class */ (function () {
    function ClassHelper(typeChecker) {
        this.typeChecker = typeChecker;
        this.jsdocParserUtil = new JsdocParserUtil();
    }
    /**
     * HELPERS
     */
    ClassHelper.prototype.stringifyDefaultValue = function (node) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        if (node.getText()) {
            return node.getText();
        }
        else if (node.kind === Ast.SyntaxKind.FalseKeyword) {
            return 'false';
        }
        else if (node.kind === Ast.SyntaxKind.TrueKeyword) {
            return 'true';
        }
    };
    ClassHelper.prototype.getDecoratorOfType = function (node, decoratorType) {
        var decorators = node.decorators || [];
        for (var i = 0; i < decorators.length; i++) {
            if (decorators[i].expression.expression) {
                if (decorators[i].expression.expression.text === decoratorType) {
                    return decorators[i];
                }
            }
        }
        return undefined;
    };
    ClassHelper.prototype.formatDecorators = function (decorators) {
        var _this = this;
        var _decorators = [];
        _.forEach(decorators, function (decorator) {
            if (decorator.expression) {
                if (decorator.expression.text) {
                    _decorators.push({ name: decorator.expression.text });
                }
                if (decorator.expression.expression) {
                    var info = { name: decorator.expression.expression.text };
                    if (decorator.expression.arguments) {
                        info.stringifiedArguments = _this.stringifyArguments(decorator.expression.arguments);
                    }
                    _decorators.push(info);
                }
            }
        });
        return _decorators;
    };
    ClassHelper.prototype.handleFunction = function (arg) {
        var _this = this;
        if (arg["function"].length === 0) {
            return "" + arg.name + this.getOptionalString(arg) + ": () => void";
        }
        var argums = arg["function"].map(function (argu) {
            var _result = DependenciesEngine$1.find(argu.type);
            if (_result) {
                if (_result.source === 'internal') {
                    var path_1 = _result.data.type;
                    if (_result.data.type === 'class') {
                        path_1 = 'classe';
                    }
                    return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"../" + path_1 + "s/" + _result.data.name + ".html\">" + argu.type + "</a>";
                }
                else {
                    var path_2 = AngularVersionUtil$1.getApiLink(_result.data, Configuration$1.mainData.angularVersion);
                    return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"" + path_2 + "\" target=\"_blank\">" + argu.type + "</a>";
                }
            }
            else if (BasicTypeUtil$1.isKnownType(argu.type)) {
                var path_3 = BasicTypeUtil$1.getTypeUrl(argu.type);
                return "" + argu.name + _this.getOptionalString(arg) + ": <a href=\"" + path_3 + "\" target=\"_blank\">" + argu.type + "</a>";
            }
            else {
                if (argu.name && argu.type) {
                    return "" + argu.name + _this.getOptionalString(arg) + ": " + argu.type;
                }
                else {
                    if (argu.name) {
                        return "" + argu.name.text;
                    }
                    else {
                        return '';
                    }
                }
            }
        });
        return "" + arg.name + this.getOptionalString(arg) + ": (" + argums + ") => void";
    };
    ClassHelper.prototype.getOptionalString = function (arg) {
        return arg.optional ? '?' : '';
    };
    ClassHelper.prototype.stringifyArguments = function (args) {
        var _this = this;
        var stringifyArgs = [];
        stringifyArgs = args
            .map(function (arg) {
            var _result = DependenciesEngine$1.find(arg.type);
            if (_result) {
                if (_result.source === 'internal') {
                    var path_4 = _result.data.type;
                    if (_result.data.type === 'class') {
                        path_4 = 'classe';
                    }
                    return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"../" + path_4 + "s/" + _result.data.name + ".html\">" + arg.type + "</a>";
                }
                else {
                    var path_5 = AngularVersionUtil$1.getApiLink(_result.data, Configuration$1.mainData.angularVersion);
                    return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"" + path_5 + "\" target=\"_blank\">" + arg.type + "</a>";
                }
            }
            else if (arg.dotDotDotToken) {
                return "..." + arg.name + ": " + arg.type;
            }
            else if (arg["function"]) {
                return _this.handleFunction(arg);
            }
            else if (arg.expression && arg.name) {
                return arg.expression.text + '.' + arg.name.text;
            }
            else if (arg.expression && arg.kind === Ast.SyntaxKind.NewExpression) {
                return 'new ' + arg.expression.text + '()';
            }
            else if (arg.kind && arg.kind === Ast.SyntaxKind.StringLiteral) {
                return "'" + arg.text + "'";
            }
            else if (arg.kind && arg.kind === Ast.SyntaxKind.ObjectLiteralExpression) {
                return StringifyObjectLiteralExpression(arg);
            }
            else if (BasicTypeUtil$1.isKnownType(arg.type)) {
                var path_6 = BasicTypeUtil$1.getTypeUrl(arg.type);
                return "" + arg.name + _this.getOptionalString(arg) + ": <a href=\"" + path_6 + "\" target=\"_blank\">" + arg.type + "</a>";
            }
            else {
                if (arg.type) {
                    var finalStringifiedArgument = '';
                    var separator = ':';
                    if (arg.name) {
                        finalStringifiedArgument += arg.name;
                    }
                    if (arg.kind === Ast.SyntaxKind.AsExpression &&
                        arg.expression &&
                        arg.expression.text) {
                        finalStringifiedArgument += arg.expression.text;
                        separator = ' as';
                    }
                    if (arg.optional) {
                        finalStringifiedArgument += _this.getOptionalString(arg);
                    }
                    if (arg.type) {
                        finalStringifiedArgument += separator + ' ' + _this.visitType(arg.type);
                    }
                    return finalStringifiedArgument;
                }
                else if (arg.text) {
                    return "" + arg.text;
                }
                else {
                    return "" + arg.name + _this.getOptionalString(arg);
                }
            }
        })
            .join(', ');
        return stringifyArgs;
    };
    ClassHelper.prototype.getPosition = function (node, sourceFile) {
        var position;
        if (node.name && node.name.end) {
            position = Ast.ts.getLineAndCharacterOfPosition(sourceFile, node.name.end);
        }
        else {
            position = Ast.ts.getLineAndCharacterOfPosition(sourceFile, node.pos);
        }
        return position;
    };
    ClassHelper.prototype.addAccessor = function (accessors, nodeAccessor, sourceFile) {
        var nodeName = '';
        if (nodeAccessor.name) {
            nodeName = nodeAccessor.name.text;
            var jsdoctags = this.jsdocParserUtil.getJSDocs(nodeAccessor);
            if (!accessors[nodeName]) {
                accessors[nodeName] = {
                    name: nodeName,
                    setSignature: undefined,
                    getSignature: undefined
                };
            }
            if (nodeAccessor.kind === Ast.SyntaxKind.SetAccessor) {
                var setSignature = {
                    name: nodeName,
                    type: 'void',
                    args: nodeAccessor.parameters.map(function (param) {
                        return {
                            name: param.name.text,
                            type: param.type ? kindToType(param.type.kind) : ''
                        };
                    }),
                    returnType: nodeAccessor.type ? this.visitType(nodeAccessor.type) : 'void',
                    line: this.getPosition(nodeAccessor, sourceFile).line + 1
                };
                if (nodeAccessor.jsDoc && nodeAccessor.jsDoc.length >= 1) {
                    var comment = nodeAccessor.jsDoc[0].comment;
                    if (typeof comment !== 'undefined') {
                        setSignature.description = marked$1(comment);
                    }
                }
                if (jsdoctags && jsdoctags.length >= 1) {
                    if (jsdoctags[0].tags) {
                        setSignature.jsdoctags = markedtags(jsdoctags[0].tags);
                    }
                }
                if (setSignature.jsdoctags && setSignature.jsdoctags.length > 0) {
                    setSignature.jsdoctags = mergeTagsAndArgs(setSignature.args, setSignature.jsdoctags);
                }
                else if (setSignature.args && setSignature.args.length > 0) {
                    setSignature.jsdoctags = mergeTagsAndArgs(setSignature.args);
                }
                accessors[nodeName].setSignature = setSignature;
            }
            if (nodeAccessor.kind === Ast.SyntaxKind.GetAccessor) {
                var getSignature = {
                    name: nodeName,
                    type: nodeAccessor.type ? kindToType(nodeAccessor.type.kind) : '',
                    returnType: nodeAccessor.type ? this.visitType(nodeAccessor.type) : '',
                    line: this.getPosition(nodeAccessor, sourceFile).line + 1
                };
                if (nodeAccessor.jsDoc && nodeAccessor.jsDoc.length >= 1) {
                    var comment = nodeAccessor.jsDoc[0].comment;
                    if (typeof comment !== 'undefined') {
                        getSignature.description = marked$1(comment);
                    }
                }
                if (jsdoctags && jsdoctags.length >= 1) {
                    if (jsdoctags[0].tags) {
                        getSignature.jsdoctags = markedtags(jsdoctags[0].tags);
                    }
                }
                accessors[nodeName].getSignature = getSignature;
            }
        }
    };
    ClassHelper.prototype.isDirectiveDecorator = function (decorator) {
        if (decorator.expression.expression) {
            var decoratorIdentifierText = decorator.expression.expression.text;
            return (decoratorIdentifierText === 'Directive' || decoratorIdentifierText === 'Component');
        }
        else {
            return false;
        }
    };
    ClassHelper.prototype.isServiceDecorator = function (decorator) {
        return decorator.expression.expression
            ? decorator.expression.expression.text === 'Injectable'
            : false;
    };
    ClassHelper.prototype.isPrivate = function (member) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        if (member.modifiers) {
            var isPrivate = member.modifiers.some(function (modifier) { return modifier.kind === Ast.SyntaxKind.PrivateKeyword; });
            if (isPrivate) {
                return true;
            }
        }
        return this.isHiddenMember(member);
    };
    ClassHelper.prototype.isProtected = function (member) {
        if (member.modifiers) {
            var isProtected = member.modifiers.some(function (modifier) { return modifier.kind === Ast.SyntaxKind.ProtectedKeyword; });
            if (isProtected) {
                return true;
            }
        }
        return this.isHiddenMember(member);
    };
    ClassHelper.prototype.isInternal = function (member) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var internalTags = ['internal'];
        if (member.jsDoc) {
            for (var _i = 0, _a = member.jsDoc; _i < _a.length; _i++) {
                var doc = _a[_i];
                if (doc.tags) {
                    for (var _b = 0, _c = doc.tags; _b < _c.length; _b++) {
                        var tag = _c[_b];
                        if (internalTags.indexOf(tag.tagName.text) > -1) {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    };
    ClassHelper.prototype.isPublic = function (member) {
        if (member.modifiers) {
            var isPublic = member.modifiers.some(function (modifier) { return modifier.kind === Ast.SyntaxKind.PublicKeyword; });
            if (isPublic) {
                return true;
            }
        }
        return this.isHiddenMember(member);
    };
    ClassHelper.prototype.isHiddenMember = function (member) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var internalTags = ['hidden'];
        if (member.jsDoc) {
            for (var _i = 0, _a = member.jsDoc; _i < _a.length; _i++) {
                var doc = _a[_i];
                if (doc.tags) {
                    for (var _b = 0, _c = doc.tags; _b < _c.length; _b++) {
                        var tag = _c[_b];
                        if (internalTags.indexOf(tag.tagName.text) > -1) {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    };
    ClassHelper.prototype.isPipeDecorator = function (decorator) {
        return decorator.expression.expression
            ? decorator.expression.expression.text === 'Pipe'
            : false;
    };
    ClassHelper.prototype.isModuleDecorator = function (decorator) {
        return decorator.expression.expression
            ? decorator.expression.expression.text === 'NgModule'
            : false;
    };
    /**
     * VISITERS
     */
    ClassHelper.prototype.visitClassDeclaration = function (fileName, classDeclaration, sourceFile) {
        var symbol = this.typeChecker.getSymbolAtLocation(classDeclaration.name);
        var rawdescription = '';
        var description = '';
        if (symbol) {
            rawdescription = this.jsdocParserUtil.getMainCommentOfNode(classDeclaration);
            description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(classDeclaration));
            if (symbol.valueDeclaration && isIgnore(symbol.valueDeclaration)) {
                return [{ ignore: true }];
            }
            if (symbol.declarations && symbol.declarations.length > 0) {
                if (isIgnore(symbol.declarations[0])) {
                    return [{ ignore: true }];
                }
            }
        }
        var className = classDeclaration.name.text;
        var members;
        var implementsElements = [];
        var extendsElement;
        var jsdoctags = [];
        if (typeof Ast.ts.getClassImplementsHeritageClauseElements !== 'undefined') {
            var implementedTypes = Ast.ts.getClassImplementsHeritageClauseElements(classDeclaration);
            if (implementedTypes) {
                var i = 0;
                var len = implementedTypes.length;
                for (i; i < len; i++) {
                    if (implementedTypes[i].expression) {
                        implementsElements.push(implementedTypes[i].expression.text);
                    }
                }
            }
        }
        if (typeof Ast.ts.getClassExtendsHeritageClauseElement !== 'undefined') {
            var extendsTypes = Ast.ts.getClassExtendsHeritageClauseElement(classDeclaration);
            if (extendsTypes) {
                if (extendsTypes.expression) {
                    extendsElement = extendsTypes.expression.text;
                }
            }
        }
        if (symbol) {
            if (symbol.valueDeclaration) {
                jsdoctags = this.jsdocParserUtil.getJSDocs(symbol.valueDeclaration);
            }
        }
        members = this.visitMembers(classDeclaration.members, sourceFile);
        if (classDeclaration.decorators) {
            for (var i = 0; i < classDeclaration.decorators.length; i++) {
                if (this.isDirectiveDecorator(classDeclaration.decorators[i])) {
                    return {
                        description: description,
                        rawdescription: rawdescription,
                        inputs: members.inputs,
                        outputs: members.outputs,
                        hostBindings: members.hostBindings,
                        hostListeners: members.hostListeners,
                        properties: members.properties,
                        methods: members.methods,
                        indexSignatures: members.indexSignatures,
                        kind: members.kind,
                        constructor: members.constructor,
                        jsdoctags: jsdoctags,
                        "extends": extendsElement,
                        implements: implementsElements,
                        accessors: members.accessors
                    };
                }
                else if (this.isServiceDecorator(classDeclaration.decorators[i])) {
                    return [
                        {
                            fileName: fileName,
                            className: className,
                            description: description,
                            rawdescription: rawdescription,
                            methods: members.methods,
                            indexSignatures: members.indexSignatures,
                            properties: members.properties,
                            kind: members.kind,
                            constructor: members.constructor,
                            jsdoctags: jsdoctags,
                            "extends": extendsElement,
                            implements: implementsElements,
                            accessors: members.accessors
                        }
                    ];
                }
                else if (this.isPipeDecorator(classDeclaration.decorators[i])) {
                    return [
                        {
                            fileName: fileName,
                            className: className,
                            description: description,
                            rawdescription: rawdescription,
                            jsdoctags: jsdoctags,
                            properties: members.properties,
                            methods: members.methods
                        }
                    ];
                }
                else if (this.isModuleDecorator(classDeclaration.decorators[i])) {
                    return [
                        {
                            fileName: fileName,
                            className: className,
                            description: description,
                            rawdescription: rawdescription,
                            jsdoctags: jsdoctags,
                            methods: members.methods
                        }
                    ];
                }
                else {
                    return [
                        {
                            description: description,
                            rawdescription: rawdescription,
                            methods: members.methods,
                            indexSignatures: members.indexSignatures,
                            properties: members.properties,
                            kind: members.kind,
                            constructor: members.constructor,
                            jsdoctags: jsdoctags,
                            "extends": extendsElement,
                            implements: implementsElements,
                            accessors: members.accessors
                        }
                    ];
                }
            }
        }
        else if (description) {
            return [
                {
                    description: description,
                    rawdescription: rawdescription,
                    inputs: members.inputs,
                    outputs: members.outputs,
                    hostBindings: members.hostBindings,
                    hostListeners: members.hostListeners,
                    methods: members.methods,
                    indexSignatures: members.indexSignatures,
                    properties: members.properties,
                    kind: members.kind,
                    constructor: members.constructor,
                    jsdoctags: jsdoctags,
                    "extends": extendsElement,
                    implements: implementsElements,
                    accessors: members.accessors
                }
            ];
        }
        else {
            return [
                {
                    methods: members.methods,
                    inputs: members.inputs,
                    outputs: members.outputs,
                    hostBindings: members.hostBindings,
                    hostListeners: members.hostListeners,
                    indexSignatures: members.indexSignatures,
                    properties: members.properties,
                    kind: members.kind,
                    constructor: members.constructor,
                    jsdoctags: jsdoctags,
                    "extends": extendsElement,
                    implements: implementsElements,
                    accessors: members.accessors
                }
            ];
        }
        return [];
    };
    ClassHelper.prototype.visitMembers = function (members, sourceFile) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var inputs = [];
        var outputs = [];
        var hostBindings = [];
        var hostListeners = [];
        var methods = [];
        var properties = [];
        var indexSignatures = [];
        var kind;
        var inputDecorator;
        var hostBinding;
        var hostListener;
        var constructor;
        var outDecorator;
        var accessors = {};
        var result = {};
        for (var i = 0; i < members.length; i++) {
            // Allows typescript guess type when using ts.is*
            var member = members[i];
            inputDecorator = this.getDecoratorOfType(member, 'Input');
            outDecorator = this.getDecoratorOfType(member, 'Output');
            hostBinding = this.getDecoratorOfType(member, 'HostBinding');
            hostListener = this.getDecoratorOfType(member, 'HostListener');
            kind = member.kind;
            if (isIgnore(member)) {
                continue;
            }
            if (inputDecorator) {
                inputs.push(this.visitInputAndHostBinding(member, inputDecorator, sourceFile));
                if (Ast.ts.isSetAccessorDeclaration(member)) {
                    this.addAccessor(accessors, members[i], sourceFile);
                }
            }
            else if (outDecorator) {
                outputs.push(this.visitOutput(member, outDecorator, sourceFile));
            }
            else if (hostBinding) {
                hostBindings.push(this.visitInputAndHostBinding(member, hostBinding, sourceFile));
            }
            else if (hostListener) {
                hostListeners.push(this.visitHostListener(member, hostListener, sourceFile));
            }
            else if (!this.isHiddenMember(member)) {
                if (!(this.isPrivate(member) && Configuration$1.mainData.disablePrivate)) {
                    if (!(this.isInternal(member) && Configuration$1.mainData.disableInternal)) {
                        if (!(this.isProtected(member) && Configuration$1.mainData.disableProtected)) {
                            if (Ast.ts.isMethodDeclaration(member) || Ast.ts.isMethodSignature(member)) {
                                methods.push(this.visitMethodDeclaration(member, sourceFile));
                            }
                            else if (Ast.ts.isPropertyDeclaration(member) ||
                                Ast.ts.isPropertySignature(member)) {
                                properties.push(this.visitProperty(member, sourceFile));
                            }
                            else if (Ast.ts.isCallSignatureDeclaration(member)) {
                                properties.push(this.visitCallDeclaration(member, sourceFile));
                            }
                            else if (Ast.ts.isGetAccessorDeclaration(member) ||
                                Ast.ts.isSetAccessorDeclaration(member)) {
                                this.addAccessor(accessors, members[i], sourceFile);
                            }
                            else if (Ast.ts.isIndexSignatureDeclaration(member)) {
                                indexSignatures.push(this.visitIndexDeclaration(member, sourceFile));
                            }
                            else if (Ast.ts.isConstructorDeclaration(member)) {
                                var _constructorProperties = this.visitConstructorProperties(member, sourceFile);
                                var j = 0;
                                var len = _constructorProperties.length;
                                for (j; j < len; j++) {
                                    properties.push(_constructorProperties[j]);
                                }
                                constructor = this.visitConstructorDeclaration(member, sourceFile);
                            }
                        }
                    }
                }
            }
        }
        inputs.sort(getNamesCompareFn());
        outputs.sort(getNamesCompareFn());
        hostBindings.sort(getNamesCompareFn());
        hostListeners.sort(getNamesCompareFn());
        properties.sort(getNamesCompareFn());
        methods.sort(getNamesCompareFn());
        indexSignatures.sort(getNamesCompareFn());
        result = {
            inputs: inputs,
            outputs: outputs,
            hostBindings: hostBindings,
            hostListeners: hostListeners,
            methods: methods,
            properties: properties,
            indexSignatures: indexSignatures,
            kind: kind,
            constructor: constructor
        };
        if (Object.keys(accessors).length) {
            result['accessors'] = accessors;
        }
        return result;
    };
    ClassHelper.prototype.visitTypeName = function (typeName) {
        if (typeName.text) {
            return typeName.text;
        }
        return this.visitTypeName(typeName.left) + "." + this.visitTypeName(typeName.right);
    };
    ClassHelper.prototype.visitType = function (node) {
        var _return = 'void';
        if (!node) {
            return _return;
        }
        if (node.typeName) {
            _return = this.visitTypeName(node.typeName);
        }
        else if (node.type) {
            if (node.type.kind) {
                _return = kindToType(node.type.kind);
            }
            if (node.type.typeName) {
                _return = this.visitTypeName(node.type.typeName);
            }
            if (node.type.typeArguments) {
                _return += '<';
                var typeArguments = [];
                for (var _i = 0, _a = node.type.typeArguments; _i < _a.length; _i++) {
                    var argument = _a[_i];
                    typeArguments.push(this.visitType(argument));
                }
                _return += typeArguments.join(' | ');
                _return += '>';
            }
            if (node.type.elementType) {
                var _firstPart = this.visitType(node.type.elementType);
                _return = _firstPart + kindToType(node.type.kind);
                if (node.type.elementType.kind === Ast.SyntaxKind.ParenthesizedType) {
                    _return = '(' + _firstPart + ')' + kindToType(node.type.kind);
                }
            }
            if (node.type.types && Ast.ts.isUnionTypeNode(node.type)) {
                _return = '';
                var i = 0;
                var len = node.type.types.length;
                for (i; i < len; i++) {
                    var type = node.type.types[i];
                    if (type.elementType) {
                        var _firstPart = this.visitType(type.elementType);
                        if (type.elementType.kind === Ast.SyntaxKind.ParenthesizedType) {
                            _return += '(' + _firstPart + ')' + kindToType(type.kind);
                        }
                        else {
                            _return += _firstPart + kindToType(type.kind);
                        }
                    }
                    else {
                        _return += kindToType(type.kind);
                        if (Ast.ts.isLiteralTypeNode(type) && type.literal) {
                            _return += '"' + type.literal.text + '"';
                        }
                        if (type.typeName) {
                            _return += this.visitTypeName(type.typeName);
                        }
                        if (type.typeArguments) {
                            _return += '<';
                            var typeArguments = [];
                            for (var _b = 0, _c = type.typeArguments; _b < _c.length; _b++) {
                                var argument = _c[_b];
                                typeArguments.push(this.visitType(argument));
                            }
                            _return += typeArguments.join(' | ');
                            _return += '>';
                        }
                    }
                    if (i < len - 1) {
                        _return += ' | ';
                    }
                }
            }
            if (node.type.elementTypes) {
                var elementTypes = node.type.elementTypes;
                var i = 0;
                var len = elementTypes.length;
                if (len > 0) {
                    _return = '[';
                    for (i; i < len; i++) {
                        var type = elementTypes[i];
                        _return += kindToType(type.kind);
                        if (Ast.ts.isLiteralTypeNode(type) && type.literal) {
                            _return += '"' + type.literal.text + '"';
                        }
                        if (type.typeName) {
                            _return += this.visitTypeName(type.typeName);
                        }
                        if (i < len - 1) {
                            _return += ', ';
                        }
                    }
                    _return += ']';
                }
            }
        }
        else if (node.elementType) {
            _return = kindToType(node.elementType.kind) + kindToType(node.kind);
            if (node.elementType.typeName) {
                _return = this.visitTypeName(node.elementType.typeName) + kindToType(node.kind);
            }
        }
        else if (node.types && Ast.ts.isUnionTypeNode(node)) {
            _return = '';
            var i = 0;
            var len = node.types.length;
            for (i; i < len; i++) {
                var type = node.types[i];
                _return += kindToType(type.kind);
                if (Ast.ts.isLiteralTypeNode(type) && type.literal) {
                    _return += '"' + type.literal.text + '"';
                }
                if (type.typeName) {
                    _return += this.visitTypeName(type.typeName);
                }
                if (i < len - 1) {
                    _return += ' | ';
                }
            }
        }
        else if (node.dotDotDotToken) {
            _return = 'any[]';
        }
        else {
            _return = kindToType(node.kind);
            if (_return === '' &&
                node.initializer &&
                node.initializer.kind &&
                (node.kind === Ast.SyntaxKind.PropertyDeclaration || node.kind === Ast.SyntaxKind.Parameter)) {
                _return = kindToType(node.initializer.kind);
            }
            if (node.kind === Ast.SyntaxKind.TypeParameter) {
                _return = node.name.text;
            }
            if (node.kind === Ast.SyntaxKind.LiteralType) {
                _return = node.literal.text;
            }
        }
        if (node.typeArguments && node.typeArguments.length > 0) {
            _return += '<';
            var i = 0, len = node.typeArguments.length;
            for (i; i < len; i++) {
                var argument = node.typeArguments[i];
                _return += this.visitType(argument);
                if (i >= 0 && i < len - 1) {
                    _return += ', ';
                }
            }
            _return += '>';
        }
        return _return;
    };
    ClassHelper.prototype.visitCallDeclaration = function (method, sourceFile) {
        var _this = this;
        var sourceCode = sourceFile.getText();
        var hash = crypto
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var result = {
            id: 'call-declaration-' + hash,
            args: method.parameters ? method.parameters.map(function (prop) { return _this.visitArgument(prop); }) : [],
            returnType: this.visitType(method.type),
            line: this.getPosition(method, sourceFile).line + 1
        };
        if (method.jsDoc) {
            result.description = marked$1(marked$1(this.jsdocParserUtil.getMainCommentOfNode(method)));
        }
        var jsdoctags = this.jsdocParserUtil.getJSDocs(method);
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
            }
        }
        return result;
    };
    ClassHelper.prototype.visitIndexDeclaration = function (method, sourceFile) {
        var _this = this;
        var sourceCode = sourceFile.getText();
        var hash = crypto
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var result = {
            id: 'index-declaration-' + hash,
            args: method.parameters ? method.parameters.map(function (prop) { return _this.visitArgument(prop); }) : [],
            returnType: this.visitType(method.type),
            line: this.getPosition(method, sourceFile).line + 1
        };
        if (method.jsDoc) {
            result.description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(method));
        }
        return result;
    };
    ClassHelper.prototype.visitConstructorDeclaration = function (method, sourceFile) {
        var _this = this;
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var result = {
            name: 'constructor',
            description: '',
            args: method.parameters ? method.parameters.map(function (prop) { return _this.visitArgument(prop); }) : [],
            line: this.getPosition(method, sourceFile).line + 1
        };
        var jsdoctags = this.jsdocParserUtil.getJSDocs(method);
        if (method.jsDoc) {
            result.description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(method));
        }
        if (method.modifiers) {
            if (method.modifiers.length > 0) {
                var kinds = method.modifiers.map(function (modifier) {
                    return modifier.kind;
                });
                if (_.indexOf(kinds, Ast.SyntaxKind.PublicKeyword) !== -1 &&
                    _.indexOf(kinds, Ast.SyntaxKind.StaticKeyword) !== -1) {
                    kinds = kinds.filter(function (kind) { return kind !== Ast.SyntaxKind.PublicKeyword; });
                }
                result.modifierKind = kinds;
            }
        }
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
            }
        }
        if (result.jsdoctags && result.jsdoctags.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args, result.jsdoctags);
        }
        else if (result.args.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args);
        }
        return result;
    };
    ClassHelper.prototype.visitProperty = function (property, sourceFile) {
        var result = {
            name: property.name.text,
            defaultValue: property.initializer
                ? this.stringifyDefaultValue(property.initializer)
                : undefined,
            type: this.visitType(property),
            optional: typeof property.questionToken !== 'undefined',
            description: '',
            line: this.getPosition(property, sourceFile).line + 1
        };
        var jsdoctags;
        if (property.initializer && property.initializer.kind === Ast.SyntaxKind.ArrowFunction) {
            result.defaultValue = '() => {...}';
        }
        if (typeof result.name === 'undefined' && typeof property.name.expression !== 'undefined') {
            result.name = property.name.expression.text;
        }
        if (property.jsDoc) {
            jsdoctags = this.jsdocParserUtil.getJSDocs(property);
            result.description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(property));
        }
        if (property.decorators) {
            result.decorators = this.formatDecorators(property.decorators);
        }
        if (property.modifiers) {
            if (property.modifiers.length > 0) {
                var kinds = property.modifiers.map(function (modifier) {
                    return modifier.kind;
                });
                if (_.indexOf(kinds, Ast.SyntaxKind.PublicKeyword) !== -1 &&
                    _.indexOf(kinds, Ast.SyntaxKind.StaticKeyword) !== -1) {
                    kinds = kinds.filter(function (kind) { return kind !== Ast.SyntaxKind.PublicKeyword; });
                }
                result.modifierKind = kinds;
            }
        }
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
            }
        }
        return result;
    };
    ClassHelper.prototype.visitConstructorProperties = function (constr, sourceFile) {
        if (constr.parameters) {
            var _parameters_1 = [];
            var i = 0;
            var len = constr.parameters.length;
            for (i; i < len; i++) {
                if (this.isPublic(constr.parameters[i])) {
                    _parameters_1.push(this.visitProperty(constr.parameters[i], sourceFile));
                }
            }
            /**
             * Merge JSDoc tags description from constructor with parameters
             */
            if (constr.jsDoc) {
                if (constr.jsDoc.length > 0) {
                    var constrTags = constr.jsDoc[0].tags;
                    if (constrTags && constrTags.length > 0) {
                        constrTags.forEach(function (tag) {
                            _parameters_1.forEach(function (param) {
                                if (tag.tagName &&
                                    tag.tagName.escapedText &&
                                    tag.tagName.escapedText === 'param') {
                                    if (tag.name &&
                                        tag.name.escapedText &&
                                        tag.name.escapedText === param.name) {
                                        param.description = tag.comment;
                                    }
                                }
                            });
                        });
                    }
                }
            }
            return _parameters_1;
        }
        else {
            return [];
        }
    };
    ClassHelper.prototype.visitInputAndHostBinding = function (property, inDecorator, sourceFile) {
        var inArgs = inDecorator.expression.arguments;
        var _return = {};
        _return.name = inArgs.length > 0 ? inArgs[0].text : property.name.text;
        _return.defaultValue = property.initializer
            ? this.stringifyDefaultValue(property.initializer)
            : undefined;
        if (!_return.description) {
            if (property.jsDoc) {
                if (property.jsDoc.length > 0) {
                    if (typeof property.jsDoc[0].comment !== 'undefined') {
                        _return.description = marked$1(property.jsDoc[0].comment);
                    }
                }
            }
        }
        _return.line = this.getPosition(property, sourceFile).line + 1;
        if (property.type) {
            _return.type = this.visitType(property);
        }
        else {
            // handle NewExpression
            if (property.initializer) {
                if (Ast.ts.isNewExpression(property.initializer)) {
                    if (property.initializer.expression) {
                        _return.type = property.initializer.expression.text;
                    }
                }
            }
        }
        if (property.kind === Ast.SyntaxKind.SetAccessor) {
            // For setter accessor, find type in first parameter
            if (property.parameters && property.parameters.length === 1) {
                if (property.parameters[0].type) {
                    _return.type = kindToType(property.parameters[0].type.kind);
                }
            }
        }
        return _return;
    };
    ClassHelper.prototype.visitMethodDeclaration = function (method, sourceFile) {
        var _this = this;
        var result = {
            name: method.name.text,
            args: method.parameters ? method.parameters.map(function (prop) { return _this.visitArgument(prop); }) : [],
            optional: typeof method.questionToken !== 'undefined',
            returnType: this.visitType(method.type),
            typeParameters: [],
            line: this.getPosition(method, sourceFile).line + 1
        };
        var jsdoctags = this.jsdocParserUtil.getJSDocs(method);
        if (typeof method.type === 'undefined') {
            // Try to get inferred type
            if (method.symbol) {
                var symbol = method.symbol;
                if (symbol.valueDeclaration) {
                    var symbolType = this.typeChecker.getTypeOfSymbolAtLocation(symbol, symbol.valueDeclaration);
                    if (symbolType) {
                        try {
                            var signature = this.typeChecker.getSignatureFromDeclaration(method);
                            var returnType = signature.getReturnType();
                            result.returnType = this.typeChecker.typeToString(returnType);
                            // tslint:disable-next-line:no-empty
                        }
                        catch (error) { }
                    }
                }
            }
        }
        if (method.typeParameters && method.typeParameters.length > 0) {
            result.typeParameters = method.typeParameters.map(function (typeParameter) {
                return _this.visitType(typeParameter);
            });
        }
        if (method.jsDoc) {
            result.description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(method));
        }
        if (method.decorators) {
            result.decorators = this.formatDecorators(method.decorators);
        }
        if (method.modifiers) {
            if (method.modifiers.length > 0) {
                var kinds = method.modifiers.map(function (modifier) {
                    return modifier.kind;
                });
                if (_.indexOf(kinds, Ast.SyntaxKind.PublicKeyword) !== -1 &&
                    _.indexOf(kinds, Ast.SyntaxKind.StaticKeyword) !== -1) {
                    kinds = kinds.filter(function (kind) { return kind !== Ast.SyntaxKind.PublicKeyword; });
                }
                result.modifierKind = kinds;
            }
        }
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
            }
        }
        if (result.jsdoctags && result.jsdoctags.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args, result.jsdoctags);
        }
        else if (result.args.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args);
        }
        return result;
    };
    ClassHelper.prototype.visitOutput = function (property, outDecorator, sourceFile) {
        var inArgs = outDecorator.expression.arguments;
        var _return = {
            name: inArgs.length > 0 ? inArgs[0].text : property.name.text,
            defaultValue: property.initializer
                ? this.stringifyDefaultValue(property.initializer)
                : undefined
        };
        if (property.jsDoc) {
            _return.description = marked$1(marked$1(this.jsdocParserUtil.getMainCommentOfNode(property)));
        }
        if (!_return.description) {
            if (property.jsDoc && property.jsDoc.length > 0) {
                if (typeof property.jsDoc[0].comment !== 'undefined') {
                    _return.description = marked$1(property.jsDoc[0].comment);
                }
            }
        }
        _return.line = this.getPosition(property, sourceFile).line + 1;
        if (property.type) {
            _return.type = this.visitType(property);
        }
        else {
            // handle NewExpression
            if (property.initializer) {
                if (Ast.ts.isNewExpression(property.initializer)) {
                    if (property.initializer.expression) {
                        _return.type = property.initializer.expression.text;
                    }
                }
            }
        }
        return _return;
    };
    ClassHelper.prototype.visitArgument = function (arg) {
        var _this = this;
        var _result = { name: arg.name.text, type: this.visitType(arg) };
        if (arg.dotDotDotToken) {
            _result.dotDotDotToken = true;
        }
        if (arg.questionToken) {
            _result.optional = true;
        }
        if (arg.type) {
            if (arg.type.kind) {
                if (Ast.ts.isFunctionTypeNode(arg.type)) {
                    _result["function"] = arg.type.parameters
                        ? arg.type.parameters.map(function (prop) { return _this.visitArgument(prop); })
                        : [];
                }
            }
        }
        if (arg.initializer) {
            _result.defaultValue = this.stringifyDefaultValue(arg.initializer);
        }
        return _result;
    };
    ClassHelper.prototype.visitHostListener = function (property, hostListenerDecorator, sourceFile) {
        var _this = this;
        var inArgs = hostListenerDecorator.expression.arguments;
        var _return = {};
        _return.name = inArgs.length > 0 ? inArgs[0].text : property.name.text;
        _return.args = property.parameters
            ? property.parameters.map(function (prop) { return _this.visitArgument(prop); })
            : [];
        _return.argsDecorator =
            inArgs.length > 1
                ? inArgs[1].elements.map(function (prop) {
                    return prop.text;
                })
                : [];
        if (property.jsDoc) {
            _return.description = marked$1(this.jsdocParserUtil.getMainCommentOfNode(property));
        }
        if (!_return.description) {
            if (property.jsDoc) {
                if (property.jsDoc.length > 0) {
                    if (typeof property.jsDoc[0].comment !== 'undefined') {
                        _return.description = marked$1(property.jsDoc[0].comment);
                    }
                }
            }
        }
        _return.line = this.getPosition(property, sourceFile).line + 1;
        return _return;
    };
    return ClassHelper;
}());

var TsPrinterUtil = /** @class */ (function () {
    function TsPrinterUtil() {
        this.printer = Ast.ts.createPrinter({
            newLine: Ast.ts.NewLineKind.LineFeed
        });
    }
    TsPrinterUtil.prototype.print = function (node) {
        return this.printer.printNode(Ast.ts.EmitHint.Unspecified, node, Ast.ts.createSourceFile('', '', Ast.ts.ScriptTarget.Latest));
    };
    return TsPrinterUtil;
}());

var SymbolHelper = /** @class */ (function () {
    function SymbolHelper() {
        this.unknown = '???';
    }
    SymbolHelper.prototype.parseDeepIndentifier = function (name, srcFile) {
        var result = {
            name: '',
            type: ''
        };
        if (typeof name === 'undefined') {
            return result;
        }
        var nsModule = name.split('.');
        var type = this.getType(name);
        if (nsModule.length > 1) {
            result.ns = nsModule[0];
            result.name = name;
            result.type = type;
            return result;
        }
        if (typeof srcFile !== 'undefined') {
            result.file = ImportsUtil$1.getFileNameOfImport(name, srcFile);
        }
        result.name = name;
        result.type = type;
        return result;
    };
    SymbolHelper.prototype.getType = function (name) {
        var type;
        if (name.toLowerCase().indexOf('component') !== -1) {
            type = 'component';
        }
        else if (name.toLowerCase().indexOf('pipe') !== -1) {
            type = 'pipe';
        }
        else if (name.toLowerCase().indexOf('controller') !== -1) {
            type = 'controller';
        }
        else if (name.toLowerCase().indexOf('module') !== -1) {
            type = 'module';
        }
        else if (name.toLowerCase().indexOf('directive') !== -1) {
            type = 'directive';
        }
        return type;
    };
    /**
     * Output
     * RouterModule.forRoot 179
     */
    SymbolHelper.prototype.buildIdentifierName = function (node, name) {
        if (Ast.ts.isIdentifier(node) && !Ast.ts.isPropertyAccessExpression(node)) {
            return node.text + "." + name;
        }
        name = name ? "." + name : '';
        var nodeName = this.unknown;
        if (node.name) {
            nodeName = node.name.text;
        }
        else if (node.text) {
            nodeName = node.text;
        }
        else if (node.expression) {
            if (node.expression.text) {
                nodeName = node.expression.text;
            }
            else if (node.expression.elements) {
                if (Ast.ts.isArrayLiteralExpression(node.expression)) {
                    nodeName = node.expression.elements.map(function (el) { return el.text; }).join(', ');
                    nodeName = "[" + nodeName + "]";
                }
            }
        }
        if (Ast.ts.isSpreadElement(node)) {
            return "..." + nodeName;
        }
        return "" + this.buildIdentifierName(node.expression, nodeName) + name;
    };
    /**
     * parse expressions such as:
     * { provide: APP_BASE_HREF, useValue: '/' }
     * { provide: 'Date', useFactory: (d1, d2) => new Date(), deps: ['d1', 'd2'] }
     */
    SymbolHelper.prototype.parseProviderConfiguration = function (node) {
        if (node.kind && node.kind === Ast.SyntaxKind.ObjectLiteralExpression) {
            // Search for provide: HTTP_INTERCEPTORS
            // and if true, return type: 'interceptor' + name
            var interceptorName_1, hasInterceptor_1;
            if (node.properties) {
                if (node.properties.length > 0) {
                    _.forEach(node.properties, function (property) {
                        if (property.kind && property.kind === Ast.SyntaxKind.PropertyAssignment) {
                            if (property.name.text === 'provide') {
                                if (property.initializer.text === 'HTTP_INTERCEPTORS') {
                                    hasInterceptor_1 = true;
                                }
                            }
                            if (property.name.text === 'useClass' ||
                                property.name.text === 'useExisting') {
                                interceptorName_1 = property.initializer.text;
                            }
                        }
                    });
                }
            }
            if (hasInterceptor_1) {
                return interceptorName_1;
            }
            else {
                return new TsPrinterUtil().print(node);
            }
        }
        else {
            return new TsPrinterUtil().print(node);
        }
    };
    /**
     * Kind
     *  181 CallExpression => "RouterModule.forRoot(args)"
     *   71 Identifier     => "RouterModule" "TodoStore"
     *    9 StringLiteral  => "./app.component.css" "./tab.scss"
     */
    SymbolHelper.prototype.parseSymbolElements = function (node) {
        // parse expressions such as: AngularFireModule.initializeApp(firebaseConfig)
        // if (ts.isCallExpression(node) && ts.isPropertyAccessExpression(node.expression)) {
        if ((Ast.ts.isCallExpression(node) && Ast.ts.isPropertyAccessExpression(node.expression)) ||
            (Ast.ts.isNewExpression(node) && Ast.ts.isElementAccessExpression(node.expression))) {
            var className = this.buildIdentifierName(node.expression);
            // function arguments could be really complex. There are so
            // many use cases that we can't handle. Just print "args" to indicate
            // that we have arguments.
            var functionArgs = node.arguments.length > 0 ? 'args' : '';
            var text = className + "(" + functionArgs + ")";
            return text;
        }
        else if (Ast.ts.isPropertyAccessExpression(node)) {
            // parse expressions such as: Shared.Module
            return this.buildIdentifierName(node);
        }
        else if (Ast.ts.isIdentifier(node)) {
            // parse expressions such as: MyComponent
            if (node.text) {
                return node.text;
            }
            if (node.escapedText) {
                return node.escapedText;
            }
        }
        else if (Ast.ts.isSpreadElement(node)) {
            // parse expressions such as: ...MYARRAY
            // Resolve MYARRAY in imports or local file variables after full scan, just return the name of the variable
            if (node.expression && node.expression.text) {
                return node.expression.text;
            }
        }
        return node.text ? node.text : this.parseProviderConfiguration(node);
    };
    /**
     * Kind
     *  177 ArrayLiteralExpression
     *  122 BooleanKeyword
     *    9 StringLiteral
     */
    SymbolHelper.prototype.parseSymbols = function (node, srcFile) {
        var _this = this;
        var localNode = node;
        if (Ast.ts.isShorthandPropertyAssignment(localNode)) {
            localNode = ImportsUtil$1.findValueInImportOrLocalVariables(node.name.text, srcFile);
        }
        if (Ast.ts.isArrayLiteralExpression(localNode.initializer)) {
            return localNode.initializer.elements.map(function (x) { return _this.parseSymbolElements(x); });
        }
        else if (Ast.ts.isStringLiteral(localNode.initializer) ||
            Ast.ts.isTemplateLiteral(localNode.initializer) ||
            (Ast.ts.isPropertyAssignment(localNode) && localNode.initializer.text)) {
            return [localNode.initializer.text];
        }
        else if (localNode.initializer.kind &&
            (localNode.initializer.kind === Ast.SyntaxKind.TrueKeyword ||
                localNode.initializer.kind === Ast.SyntaxKind.FalseKeyword)) {
            return [localNode.initializer.kind === Ast.SyntaxKind.TrueKeyword ? true : false];
        }
        else if (Ast.ts.isPropertyAccessExpression(localNode.initializer)) {
            var identifier = this.parseSymbolElements(localNode.initializer);
            return [identifier];
        }
        else if (localNode.initializer &&
            localNode.initializer.elements &&
            localNode.initializer.elements.length > 0) {
            // Node replaced by ts-simple-ast & kind = 265
            return localNode.initializer.elements.map(function (x) { return _this.parseSymbolElements(x); });
        }
    };
    SymbolHelper.prototype.getSymbolDeps = function (props, type, srcFile, multiLine) {
        var _this = this;
        if (props.length === 0) {
            return [];
        }
        var i = 0, len = props.length, filteredProps = [];
        for (i; i < len; i++) {
            if (props[i].name && props[i].name.text === type) {
                filteredProps.push(props[i]);
            }
        }
        return filteredProps.map(function (x) { return _this.parseSymbols(x, srcFile); }).pop() || [];
    };
    SymbolHelper.prototype.getSymbolDepsRaw = function (props, type, multiLine) {
        return props.filter(function (node) { return node.name.text === type; });
    };
    return SymbolHelper;
}());

var ComponentHelper = /** @class */ (function () {
    function ComponentHelper(classHelper, symbolHelper) {
        if (symbolHelper === void 0) { symbolHelper = new SymbolHelper(); }
        this.classHelper = classHelper;
        this.symbolHelper = symbolHelper;
    }
    ComponentHelper.prototype.getComponentChangeDetection = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'changeDetection', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentEncapsulation = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'encapsulation', srcFile);
    };
    ComponentHelper.prototype.getComponentPure = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'pure', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentName = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'name', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentExportAs = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'exportAs', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentHost = function (props) {
        return this.getSymbolDepsObject(props, 'host');
    };
    ComponentHelper.prototype.getComponentTag = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'tag', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentInputsMetadata = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'inputs', srcFile);
    };
    ComponentHelper.prototype.getComponentTemplate = function (props, srcFile) {
        var t = this.symbolHelper.getSymbolDeps(props, 'template', srcFile, true).pop();
        if (t) {
            t = detectIndent(t, 0);
            t = t.replace(/\n/, '');
            t = t.replace(/ +$/gm, '');
        }
        return t;
    };
    ComponentHelper.prototype.getComponentStyleUrls = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'styleUrls', srcFile);
    };
    ComponentHelper.prototype.getComponentStyleUrl = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'styleUrl', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentShadow = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'shadow', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentScoped = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'scoped', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentAssetsDir = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'assetsDir', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentAssetsDirs = function (props, srcFile) {
        return this.sanitizeUrls(this.symbolHelper.getSymbolDeps(props, 'assetsDir', srcFile));
    };
    ComponentHelper.prototype.getComponentStyles = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'styles', srcFile);
    };
    ComponentHelper.prototype.getComponentModuleId = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'moduleId', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentOutputs = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'outputs', srcFile);
    };
    ComponentHelper.prototype.getComponentProviders = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'providers', srcFile)
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name); });
    };
    ComponentHelper.prototype.getComponentEntryComponents = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'entryComponents', srcFile)
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name); });
    };
    ComponentHelper.prototype.getComponentViewProviders = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'viewProviders', srcFile)
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name); });
    };
    ComponentHelper.prototype.getComponentTemplateUrl = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'templateUrl', srcFile);
    };
    ComponentHelper.prototype.getComponentExampleUrls = function (text) {
        var exampleUrlsMatches = text.match(/<example-url>(.*?)<\/example-url>/g);
        var exampleUrls = undefined;
        if (exampleUrlsMatches && exampleUrlsMatches.length) {
            exampleUrls = exampleUrlsMatches.map(function (val) {
                return val.replace(/<\/?example-url>/g, '');
            });
        }
        return exampleUrls;
    };
    ComponentHelper.prototype.getComponentPreserveWhitespaces = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'preserveWhitespaces', srcFile).pop();
    };
    ComponentHelper.prototype.getComponentSelector = function (props, srcFile) {
        return this.symbolHelper.getSymbolDeps(props, 'selector', srcFile).pop();
    };
    ComponentHelper.prototype.parseProperties = function (node) {
        var obj = new Map();
        var properties = node.initializer.properties || [];
        properties.forEach(function (prop) {
            obj.set(prop.name.text, prop.initializer.text);
        });
        return obj;
    };
    ComponentHelper.prototype.getSymbolDepsObject = function (props, type, multiLine) {
        var _this = this;
        var i = 0, len = props.length, filteredProps = [];
        for (i; i < len; i++) {
            if (props[i].name && props[i].name.text === type) {
                filteredProps.push(props[i]);
            }
        }
        return filteredProps.map(function (x) { return _this.parseProperties(x); }).pop();
    };
    ComponentHelper.prototype.getComponentIO = function (filename, sourceFile, node, fileBody) {
        var _this = this;
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var reducedSource = fileBody ? fileBody.statements : sourceFile.statements;
        var res = reducedSource.reduce(function (directive, statement) {
            if (Ast.ts.isClassDeclaration(statement)) {
                if (statement.pos === node.pos && statement.end === node.end) {
                    return directive.concat(_this.classHelper.visitClassDeclaration(filename, statement, sourceFile));
                }
            }
            return directive;
        }, []);
        return res[0] || {};
    };
    ComponentHelper.prototype.sanitizeUrls = function (urls) {
        return urls.map(function (url) { return url.replace('./', ''); });
    };
    return ComponentHelper;
}());
var ComponentCache = /** @class */ (function () {
    function ComponentCache() {
        this.cache = new Map();
    }
    ComponentCache.prototype.get = function (key) {
        return this.cache.get(key);
    };
    ComponentCache.prototype.set = function (key, value) {
        this.cache.set(key, value);
    };
    return ComponentCache;
}());

var FrameworkDependencies = /** @class */ (function () {
    function FrameworkDependencies(files, options) {
        this.files = files;
        var transpileOptions = {
            target: Ast.ts.ScriptTarget.ES5,
            module: Ast.ts.ModuleKind.CommonJS,
            tsconfigDirectory: options.tsconfigDirectory,
            allowJs: true
        };
        this.program = Ast.ts.createProgram(this.files, transpileOptions, compilerHost(transpileOptions));
        this.typeChecker = this.program.getTypeChecker();
        this.classHelper = new ClassHelper(this.typeChecker);
        this.componentHelper = new ComponentHelper(this.classHelper);
    }
    return FrameworkDependencies;
}());

var ExtendsMerger = /** @class */ (function () {
    function ExtendsMerger() {
    }
    ExtendsMerger.getInstance = function () {
        if (!ExtendsMerger.instance) {
            ExtendsMerger.instance = new ExtendsMerger();
        }
        return ExtendsMerger.instance;
    };
    ExtendsMerger.prototype.merge = function (deps) {
        var _this = this;
        this.components = deps.components;
        this.classes = deps.classes;
        this.injectables = deps.injectables;
        this.components.forEach(function (component) {
            var ext;
            if (typeof component["extends"] !== 'undefined') {
                ext = _this.findInDependencies(component["extends"]);
                if (ext) {
                    var recursiveScanWithInheritance_1 = function (cls) {
                        // From class to component
                        if (typeof cls.methods !== 'undefined' && cls.methods.length > 0) {
                            var newMethods = _.cloneDeep(cls.methods);
                            newMethods = _this.markInheritance(newMethods, cls);
                            if (typeof component.methodsClass !== 'undefined') {
                                component.methodsClass = component.methodsClass.concat(newMethods);
                            }
                        }
                        if (typeof cls.properties !== 'undefined' && cls.properties.length > 0) {
                            var newProperties = _.cloneDeep(cls.properties);
                            newProperties = _this.markInheritance(newProperties, cls);
                            if (typeof component.propertiesClass !== 'undefined') {
                                component.propertiesClass = component.propertiesClass.concat(newProperties);
                            }
                        }
                        // From component to component
                        if (typeof cls.inputsClass !== 'undefined' && cls.inputsClass.length > 0) {
                            var newInputs = _.cloneDeep(cls.inputsClass);
                            newInputs = _this.markInheritance(newInputs, cls);
                            if (typeof component.inputsClass !== 'undefined') {
                                component.inputsClass = component.inputsClass.concat(newInputs);
                            }
                        }
                        if (typeof cls.outputsClass !== 'undefined' &&
                            cls.outputsClass.length > 0) {
                            var newOutputs = _.cloneDeep(cls.outputsClass);
                            newOutputs = _this.markInheritance(newOutputs, cls);
                            if (typeof component.outputsClass !== 'undefined') {
                                component.outputsClass = component.outputsClass.concat(newOutputs);
                            }
                        }
                        if (typeof cls.methodsClass !== 'undefined' &&
                            cls.methodsClass.length > 0) {
                            var newMethods = _.cloneDeep(cls.methodsClass);
                            newMethods = _this.markInheritance(newMethods, cls);
                            if (typeof component.methodsClass !== 'undefined') {
                                component.methodsClass = component.methodsClass.concat(newMethods);
                            }
                        }
                        if (typeof cls.propertiesClass !== 'undefined' &&
                            cls.propertiesClass.length > 0) {
                            var newProperties = _.cloneDeep(cls.propertiesClass);
                            newProperties = _this.markInheritance(newProperties, cls);
                            if (typeof component.propertiesClass !== 'undefined') {
                                component.propertiesClass = component.propertiesClass.concat(newProperties);
                            }
                        }
                        if (typeof cls.hostBindings !== 'undefined' &&
                            cls.hostBindings.length > 0) {
                            var newHostBindings = _.cloneDeep(cls.hostBindings);
                            newHostBindings = _this.markInheritance(newHostBindings, cls);
                            if (typeof component.hostBindings !== 'undefined') {
                                component.hostBindings = component.hostBindings.concat(newHostBindings);
                            }
                        }
                        if (typeof cls.hostListeners !== 'undefined' &&
                            cls.hostListeners.length > 0) {
                            var newHostListeners = _.cloneDeep(cls.hostListeners);
                            newHostListeners = _this.markInheritance(newHostListeners, cls);
                            if (typeof component.hostListeners !== 'undefined') {
                                component.hostListeners = component.hostListeners.concat(newHostListeners);
                            }
                        }
                        if (Configuration$1.mainData.disableLifeCycleHooks) {
                            component.methodsClass = cleanLifecycleHooksFromMethods(component.methodsClass);
                        }
                        if (cls["extends"]) {
                            recursiveScanWithInheritance_1(_this.findInDependencies(cls["extends"]));
                        }
                    };
                    // From class to class
                    recursiveScanWithInheritance_1(ext);
                }
            }
        });
        var mergeExtendedClasses = function (el) {
            var ext;
            if (typeof el["extends"] !== 'undefined') {
                ext = _this.findInDependencies(el["extends"]);
                if (ext) {
                    var recursiveScanWithInheritance_2 = function (cls) {
                        if (typeof cls.methods !== 'undefined' && cls.methods.length > 0) {
                            var newMethods = _.cloneDeep(cls.methods);
                            newMethods = _this.markInheritance(newMethods, cls);
                            if (typeof el.methods !== 'undefined') {
                                el.methods = el.methods.concat(newMethods);
                            }
                        }
                        if (typeof cls.properties !== 'undefined' && cls.properties.length > 0) {
                            var newProperties = _.cloneDeep(cls.properties);
                            newProperties = _this.markInheritance(newProperties, cls);
                            if (typeof el.properties !== 'undefined') {
                                el.properties = el.properties.concat(newProperties);
                            }
                        }
                        if (cls["extends"]) {
                            recursiveScanWithInheritance_2(_this.findInDependencies(cls["extends"]));
                        }
                    };
                    // From elss to elss
                    recursiveScanWithInheritance_2(ext);
                }
            }
        };
        this.classes.forEach(mergeExtendedClasses);
        this.injectables.forEach(mergeExtendedClasses);
        return deps;
    };
    ExtendsMerger.prototype.markInheritance = function (data, originalource) {
        return data.map(function (el) {
            var newElement = el;
            newElement.inheritance = {
                file: originalource.name
            };
            return newElement;
        });
    };
    ExtendsMerger.prototype.findInDependencies = function (name) {
        var mergedData = _.concat([], this.components, this.classes, this.injectables);
        var result = _.find(mergedData, { name: name });
        return result || false;
    };
    return ExtendsMerger;
}());
var ExtendsMerger$1 = ExtendsMerger.getInstance();

var CodeGenerator = /** @class */ (function () {
    function CodeGenerator() {
    }
    CodeGenerator.prototype.generate = function (node) {
        return this.visitAndRecognize(node, []).join('');
    };
    CodeGenerator.prototype.visitAndRecognize = function (node, code, depth) {
        var _this = this;
        if (depth === void 0) { depth = 0; }
        this.recognize(node, code);
        node.getChildren().forEach(function (c) { return _this.visitAndRecognize(c, code, depth + 1); });
        return code;
    };
    CodeGenerator.prototype.recognize = function (node, code) {
        var _this = this;
        var conversion = TsKindConversion.find(function (x) { return x.kinds.some(function (z) { return z === node.kind; }); });
        if (conversion) {
            var result = conversion.output(node);
            result.forEach(function (text) { return _this.gen(text, code); });
        }
    };
    CodeGenerator.prototype.gen = function (token, code) {
        if (!token) {
            return;
        }
        if (token === '\n') {
            code.push('');
        }
        else {
            code.push(token);
        }
    };
    return CodeGenerator;
}());
var TsKindsToText = /** @class */ (function () {
    function TsKindsToText(output, kinds) {
        this.output = output;
        this.kinds = kinds;
    }
    return TsKindsToText;
}());
var TsKindConversion = [
    new TsKindsToText(function (node) { return ['"', node.text, '"']; }, [
        Ast.SyntaxKind.FirstLiteralToken,
        Ast.SyntaxKind.Identifier
    ]),
    new TsKindsToText(function (node) { return ['"', node.text, '"']; }, [Ast.SyntaxKind.StringLiteral]),
    new TsKindsToText(function (node) { return []; }, [Ast.SyntaxKind.ArrayLiteralExpression]),
    new TsKindsToText(function (node) { return ['import', ' ']; }, [Ast.SyntaxKind.ImportKeyword]),
    new TsKindsToText(function (node) { return ['from', ' ']; }, [Ast.SyntaxKind.FromKeyword]),
    new TsKindsToText(function (node) { return ['\n', 'export', ' ']; }, [Ast.SyntaxKind.ExportKeyword]),
    new TsKindsToText(function (node) { return ['class', ' ']; }, [Ast.SyntaxKind.ClassKeyword]),
    new TsKindsToText(function (node) { return ['this']; }, [Ast.SyntaxKind.ThisKeyword]),
    new TsKindsToText(function (node) { return ['constructor']; }, [Ast.SyntaxKind.ConstructorKeyword]),
    new TsKindsToText(function (node) { return ['false']; }, [Ast.SyntaxKind.FalseKeyword]),
    new TsKindsToText(function (node) { return ['true']; }, [Ast.SyntaxKind.TrueKeyword]),
    new TsKindsToText(function (node) { return ['null']; }, [Ast.SyntaxKind.NullKeyword]),
    new TsKindsToText(function (node) { return []; }, [Ast.SyntaxKind.AtToken]),
    new TsKindsToText(function (node) { return ['+']; }, [Ast.SyntaxKind.PlusToken]),
    new TsKindsToText(function (node) { return [' => ']; }, [Ast.SyntaxKind.EqualsGreaterThanToken]),
    new TsKindsToText(function (node) { return ['(']; }, [Ast.SyntaxKind.OpenParenToken]),
    new TsKindsToText(function (node) { return ['{', ' ']; }, [
        Ast.SyntaxKind.ImportClause,
        Ast.SyntaxKind.ObjectLiteralExpression
    ]),
    new TsKindsToText(function (node) { return ['{', '\n']; }, [Ast.SyntaxKind.Block]),
    new TsKindsToText(function (node) { return ['}']; }, [Ast.SyntaxKind.CloseBraceToken]),
    new TsKindsToText(function (node) { return [')']; }, [Ast.SyntaxKind.CloseParenToken]),
    new TsKindsToText(function (node) { return ['[']; }, [Ast.SyntaxKind.OpenBracketToken]),
    new TsKindsToText(function (node) { return [']']; }, [Ast.SyntaxKind.CloseBracketToken]),
    new TsKindsToText(function (node) { return [';', '\n']; }, [Ast.SyntaxKind.SemicolonToken]),
    new TsKindsToText(function (node) { return [',', ' ']; }, [Ast.SyntaxKind.CommaToken]),
    new TsKindsToText(function (node) { return [' ', ':', ' ']; }, [Ast.SyntaxKind.ColonToken]),
    new TsKindsToText(function (node) { return ['.']; }, [Ast.SyntaxKind.DotToken]),
    new TsKindsToText(function (node) { return []; }, [Ast.SyntaxKind.DoStatement]),
    new TsKindsToText(function (node) { return []; }, [Ast.SyntaxKind.Decorator]),
    new TsKindsToText(function (node) { return [' = ']; }, [Ast.SyntaxKind.FirstAssignment]),
    new TsKindsToText(function (node) { return [' ']; }, [Ast.SyntaxKind.FirstPunctuation]),
    new TsKindsToText(function (node) { return ['private', ' ']; }, [Ast.SyntaxKind.PrivateKeyword]),
    new TsKindsToText(function (node) { return ['public', ' ']; }, [Ast.SyntaxKind.PublicKeyword])
];

var crypto$1 = require('crypto');
var ComponentDepFactory = /** @class */ (function () {
    function ComponentDepFactory(helper) {
        this.helper = helper;
    }
    ComponentDepFactory.prototype.create = function (file, srcFile, name, props, IO) {
        // console.log(util.inspect(props, { showHidden: true, depth: 10 }));
        var sourceCode = srcFile.getText();
        var hash = crypto$1
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var componentDep = {
            name: name,
            id: 'component-' + name + '-' + hash,
            file: file,
            // animations?: string[]; // TODO
            changeDetection: this.helper.getComponentChangeDetection(props, srcFile),
            encapsulation: this.helper.getComponentEncapsulation(props, srcFile),
            entryComponents: this.helper.getComponentEntryComponents(props, srcFile),
            exportAs: this.helper.getComponentExportAs(props, srcFile),
            host: this.helper.getComponentHost(props),
            inputs: this.helper.getComponentInputsMetadata(props, srcFile),
            // interpolation?: string; // TODO waiting doc infos
            moduleId: this.helper.getComponentModuleId(props, srcFile),
            outputs: this.helper.getComponentOutputs(props, srcFile),
            providers: this.helper.getComponentProviders(props, srcFile),
            // queries?: Deps[]; // TODO
            selector: this.helper.getComponentSelector(props, srcFile),
            styleUrls: this.helper.getComponentStyleUrls(props, srcFile),
            styles: this.helper.getComponentStyles(props, srcFile),
            template: this.helper.getComponentTemplate(props, srcFile),
            templateUrl: this.helper.getComponentTemplateUrl(props, srcFile),
            viewProviders: this.helper.getComponentViewProviders(props, srcFile),
            inputsClass: IO.inputs,
            outputsClass: IO.outputs,
            propertiesClass: IO.properties,
            methodsClass: IO.methods,
            hostBindings: IO.hostBindings,
            hostListeners: IO.hostListeners,
            description: IO.description,
            rawdescription: IO.rawdescription,
            type: 'component',
            sourceCode: srcFile.getText(),
            exampleUrls: this.helper.getComponentExampleUrls(srcFile.getText()),
            tag: this.helper.getComponentTag(props, srcFile),
            styleUrl: this.helper.getComponentStyleUrl(props, srcFile),
            shadow: this.helper.getComponentShadow(props, srcFile),
            scoped: this.helper.getComponentScoped(props, srcFile),
            assetsDir: this.helper.getComponentAssetsDir(props, srcFile),
            assetsDirs: this.helper.getComponentAssetsDirs(props, srcFile),
            styleUrlsData: '',
            stylesData: ''
        };
        if (typeof this.helper.getComponentPreserveWhitespaces(props, srcFile) !== 'undefined') {
            componentDep.preserveWhitespaces = this.helper.getComponentPreserveWhitespaces(props, srcFile);
        }
        if (Configuration$1.mainData.disableLifeCycleHooks) {
            componentDep.methodsClass = cleanLifecycleHooksFromMethods(componentDep.methodsClass);
        }
        if (IO.jsdoctags && IO.jsdoctags.length > 0) {
            componentDep.jsdoctags = IO.jsdoctags[0].tags;
        }
        if (IO.constructor) {
            componentDep.constructorObj = IO.constructor;
        }
        if (IO["extends"]) {
            componentDep["extends"] = IO["extends"];
        }
        if (IO.implements && IO.implements.length > 0) {
            componentDep.implements = IO.implements;
        }
        if (IO.accessors) {
            componentDep.accessors = IO.accessors;
        }
        return componentDep;
    };
    return ComponentDepFactory;
}());

var crypto$2 = require('crypto');
var ControllerDepFactory = /** @class */ (function () {
    function ControllerDepFactory() {
    }
    ControllerDepFactory.prototype.create = function (file, srcFile, name, properties, IO) {
        var sourceCode = srcFile.getText();
        var hash = crypto$2
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var infos = {
            name: name,
            id: 'controller-' + name + '-' + hash,
            file: file,
            methodsClass: IO.methods,
            type: 'controller',
            description: IO.description,
            sourceCode: srcFile.text
        };
        if (properties && properties.length === 1) {
            if (properties[0].text) {
                infos.prefix = properties[0].text;
            }
        }
        return infos;
    };
    return ControllerDepFactory;
}());

var crypto$3 = require('crypto');
var DirectiveDepFactory = /** @class */ (function () {
    function DirectiveDepFactory(helper) {
        this.helper = helper;
    }
    DirectiveDepFactory.prototype.create = function (file, srcFile, name, props, IO) {
        var sourceCode = srcFile.getText();
        var hash = crypto$3
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var directiveDeps = {
            name: name,
            id: 'directive-' + name + '-' + hash,
            file: file,
            type: 'directive',
            description: IO.description,
            sourceCode: srcFile.getText(),
            selector: this.helper.getComponentSelector(props),
            providers: this.helper.getComponentProviders(props),
            inputsClass: IO.inputs,
            outputsClass: IO.outputs,
            hostBindings: IO.hostBindings,
            hostListeners: IO.hostListeners,
            propertiesClass: IO.properties,
            methodsClass: IO.methods,
            exampleUrls: this.helper.getComponentExampleUrls(srcFile.getText())
        };
        if (Configuration$1.mainData.disableLifeCycleHooks) {
            directiveDeps.methodsClass = cleanLifecycleHooksFromMethods(directiveDeps.methodsClass);
        }
        if (IO.jsdoctags && IO.jsdoctags.length > 0) {
            directiveDeps.jsdoctags = IO.jsdoctags[0].tags;
        }
        if (IO.implements && IO.implements.length > 0) {
            directiveDeps.implements = IO.implements;
        }
        if (IO.constructor) {
            directiveDeps.constructorObj = IO.constructor;
        }
        if (IO.accessors) {
            directiveDeps.accessors = IO.accessors;
        }
        return directiveDeps;
    };
    return DirectiveDepFactory;
}());

var JsDocHelper = /** @class */ (function () {
    function JsDocHelper() {
    }
    JsDocHelper.prototype.hasJSDocInternalTag = function (filename, sourceFile, node) {
        if (typeof sourceFile.statements !== 'undefined') {
            return this.checkStatements(sourceFile.statements, node);
        }
        return false;
    };
    JsDocHelper.prototype.checkStatements = function (statements, node) {
        var _this = this;
        return statements.some(function (x) { return _this.checkStatement(x, node); });
    };
    JsDocHelper.prototype.checkStatement = function (statement, node) {
        if (statement.pos === node.pos && statement.end === node.end) {
            if (node.jsDoc && node.jsDoc.length > 0) {
                return this.checkJsDocs(node.jsDoc);
            }
        }
        return false;
    };
    JsDocHelper.prototype.checkJsDocs = function (jsDocs) {
        var _this = this;
        return jsDocs
            .filter(function (x) { return x.tags && x.tags.length > 0; })
            .some(function (x) { return _this.checkJsDocTags(x.tags); });
    };
    JsDocHelper.prototype.checkJsDocTags = function (tags) {
        return tags.some(function (x) { return x.tagName && x.tagName.text === 'internal'; });
    };
    return JsDocHelper;
}());

var ModuleHelper = /** @class */ (function () {
    function ModuleHelper(cache, symbolHelper) {
        if (symbolHelper === void 0) { symbolHelper = new SymbolHelper(); }
        this.cache = cache;
        this.symbolHelper = symbolHelper;
    }
    ModuleHelper.prototype.getModuleProviders = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'providers', srcFile)
            .map(function (providerName) { return _this.symbolHelper.parseDeepIndentifier(providerName, srcFile); });
    };
    ModuleHelper.prototype.getModuleControllers = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'controllers', srcFile)
            .map(function (providerName) { return _this.symbolHelper.parseDeepIndentifier(providerName, srcFile); });
    };
    ModuleHelper.prototype.getModuleDeclarations = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper.getSymbolDeps(props, 'declarations', srcFile).map(function (name) {
            var component = _this.cache.get(name);
            if (component) {
                return component;
            }
            return _this.symbolHelper.parseDeepIndentifier(name, srcFile);
        });
    };
    ModuleHelper.prototype.getModuleEntryComponents = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper.getSymbolDeps(props, 'entryComponents', srcFile).map(function (name) {
            var component = _this.cache.get(name);
            if (component) {
                return component;
            }
            return _this.symbolHelper.parseDeepIndentifier(name, srcFile);
        });
    };
    ModuleHelper.prototype.cleanImportForRootForChild = function (name) {
        var nsModule = name.split('.');
        if (nsModule.length > 0) {
            name = nsModule[0];
        }
        return name;
    };
    ModuleHelper.prototype.getModuleImports = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'imports', srcFile)
            .map(function (name) { return _this.cleanImportForRootForChild(name); })
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name); });
    };
    ModuleHelper.prototype.getModuleExports = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'exports', srcFile)
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name, srcFile); });
    };
    ModuleHelper.prototype.getModuleImportsRaw = function (props, srcFile) {
        return this.symbolHelper.getSymbolDepsRaw(props, 'imports');
    };
    ModuleHelper.prototype.getModuleId = function (props, srcFile) {
        var _id = this.symbolHelper.getSymbolDeps(props, 'id', srcFile), id;
        if (_id.length === 1) {
            id = _id[0];
        }
        return id;
    };
    ModuleHelper.prototype.getModuleSchemas = function (props, srcFile) {
        var schemas = this.symbolHelper.getSymbolDeps(props, 'schemas', srcFile);
        return schemas;
    };
    ModuleHelper.prototype.getModuleBootstrap = function (props, srcFile) {
        var _this = this;
        return this.symbolHelper
            .getSymbolDeps(props, 'bootstrap', srcFile)
            .map(function (name) { return _this.symbolHelper.parseDeepIndentifier(name, srcFile); });
    };
    return ModuleHelper;
}());

var crypto$4 = require('crypto');
var ModuleDepFactory = /** @class */ (function () {
    function ModuleDepFactory(moduleHelper) {
        this.moduleHelper = moduleHelper;
    }
    ModuleDepFactory.prototype.create = function (file, srcFile, name, properties, IO) {
        var sourceCode = srcFile.getText();
        var hash = crypto$4
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        return {
            name: name,
            id: 'module-' + name + '-' + hash,
            file: file,
            ngid: this.moduleHelper.getModuleId(properties, srcFile),
            providers: this.moduleHelper.getModuleProviders(properties, srcFile),
            declarations: this.moduleHelper.getModuleDeclarations(properties, srcFile),
            controllers: this.moduleHelper.getModuleControllers(properties, srcFile),
            entryComponents: this.moduleHelper.getModuleEntryComponents(properties, srcFile),
            imports: this.moduleHelper.getModuleImports(properties, srcFile),
            exports: this.moduleHelper.getModuleExports(properties, srcFile),
            schemas: this.moduleHelper.getModuleSchemas(properties, srcFile),
            bootstrap: this.moduleHelper.getModuleBootstrap(properties, srcFile),
            type: 'module',
            rawdescription: IO.rawdescription,
            methods: IO.methods,
            description: IO.description,
            sourceCode: srcFile.text
        };
    };
    return ModuleDepFactory;
}());

var crypto$5 = require('crypto');
var marked$2 = require('marked');
var ast$2 = new Ast__default();
// TypeScript reference : https://github.com/Microsoft/TypeScript/blob/master/lib/typescript.d.ts
var AngularDependencies = /** @class */ (function (_super) {
    __extends(AngularDependencies, _super);
    function AngularDependencies(files, options) {
        var _this = _super.call(this, files, options) || this;
        _this.cache = new ComponentCache();
        _this.moduleHelper = new ModuleHelper(_this.cache);
        _this.jsDocHelper = new JsDocHelper();
        _this.symbolHelper = new SymbolHelper();
        _this.jsdocParserUtil = new JsdocParserUtil();
        return _this;
    }
    AngularDependencies.prototype.getDependencies = function () {
        var _this = this;
        var deps = {
            modules: [],
            modulesForGraph: [],
            components: [],
            controllers: [],
            injectables: [],
            interceptors: [],
            guards: [],
            pipes: [],
            directives: [],
            routes: [],
            classes: [],
            interfaces: [],
            miscellaneous: {
                variables: [],
                functions: [],
                typealiases: [],
                enumerations: []
            },
            routesTree: undefined
        };
        var sourceFiles = this.program.getSourceFiles() || [];
        sourceFiles.map(function (file) {
            var filePath = file.fileName;
            if (path.extname(filePath) === '.ts' || path.extname(filePath) === '.tsx') {
                if (!Configuration$1.mainData.angularJSProject && path.extname(filePath) === '.js') {
                    logger.info('parsing', filePath);
                    _this.getSourceFileDecorators(file, deps);
                }
                else {
                    if (filePath.lastIndexOf('.d.ts') === -1 &&
                        filePath.lastIndexOf('spec.ts') === -1) {
                        logger.info('parsing', filePath);
                        _this.getSourceFileDecorators(file, deps);
                    }
                }
            }
            return deps;
        });
        // End of file scanning
        // Try merging inside the same file declarated variables & modules with imports | exports | declarations | providers
        if (deps.miscellaneous.variables.length > 0) {
            deps.miscellaneous.variables.forEach(function (_variable) {
                var newVar = [];
                (function (_var, _newVar) {
                    // getType pr reconstruire....
                    if (_var.initializer) {
                        if (_var.initializer.elements) {
                            if (_var.initializer.elements.length > 0) {
                                _var.initializer.elements.forEach(function (element) {
                                    if (element.text) {
                                        newVar.push({
                                            name: element.text,
                                            type: _this.symbolHelper.getType(element.text)
                                        });
                                    }
                                });
                            }
                        }
                    }
                })(_variable);
                var onLink = function (mod) {
                    var process = function (initialArray, _var) {
                        var indexToClean = 0;
                        var found = false;
                        var findVariableInArray = function (el, index, theArray) {
                            if (el.name === _var.name) {
                                indexToClean = index;
                                found = true;
                            }
                        };
                        initialArray.forEach(findVariableInArray);
                        // Clean indexes to replace
                        if (found) {
                            initialArray.splice(indexToClean, 1);
                            // Add variable
                            newVar.forEach(function (newEle) {
                                if (typeof _.find(initialArray, { name: newEle.name }) ===
                                    'undefined') {
                                    initialArray.push(newEle);
                                }
                            });
                        }
                    };
                    process(mod.imports, _variable);
                    process(mod.exports, _variable);
                    process(mod.controllers, _variable);
                    process(mod.declarations, _variable);
                    process(mod.providers, _variable);
                };
                deps.modules.forEach(onLink);
                deps.modulesForGraph.forEach(onLink);
            });
        }
        /**
         * If one thing extends another, merge them, only for internal sources
         * - classes
         * - components
         * - injectables
         * for
         * - inputs
         * - outputs
         * - properties
         * - methods
         */
        deps = ExtendsMerger$1.merge(deps);
        // RouterParserUtil.printModulesRoutes();
        // RouterParserUtil.printRoutes();
        if (!Configuration$1.mainData.disableRoutesGraph) {
            RouterParserUtil$1.linkModulesAndRoutes();
            RouterParserUtil$1.constructModulesTree();
            deps.routesTree = RouterParserUtil$1.constructRoutesTree();
        }
        return deps;
    };
    AngularDependencies.prototype.processClass = function (node, file, srcFile, outputSymbols, fileBody) {
        var name = this.getSymboleName(node);
        var IO = this.getClassIO(file, srcFile, node, fileBody);
        var sourceCode = srcFile.getText();
        var hash = crypto$5
            .createHash('md5')
            .update(sourceCode)
            .digest('hex');
        var deps = {
            name: name,
            id: 'class-' + name + '-' + hash,
            file: file,
            type: 'class',
            sourceCode: srcFile.getText()
        };
        var excludeFromClassArray = false;
        if (IO.constructor) {
            deps.constructorObj = IO.constructor;
        }
        if (IO.properties) {
            deps.properties = IO.properties;
        }
        if (IO.description) {
            deps.description = IO.description;
        }
        if (IO.rawdescription) {
            deps.rawdescription = IO.rawdescription;
        }
        if (IO.methods) {
            deps.methods = IO.methods;
        }
        if (IO.indexSignatures) {
            deps.indexSignatures = IO.indexSignatures;
        }
        if (IO["extends"]) {
            deps["extends"] = IO["extends"];
        }
        if (IO.jsdoctags && IO.jsdoctags.length > 0) {
            deps.jsdoctags = IO.jsdoctags[0].tags;
        }
        if (IO.accessors) {
            deps.accessors = IO.accessors;
        }
        if (IO.inputs) {
            deps.inputsClass = IO.inputs;
        }
        if (IO.outputs) {
            deps.outputsClass = IO.outputs;
        }
        if (IO.hostBindings) {
            deps.hostBindings = IO.hostBindings;
        }
        if (IO.hostListeners) {
            deps.hostListeners = IO.hostListeners;
        }
        if (Configuration$1.mainData.disableLifeCycleHooks) {
            deps.methods = cleanLifecycleHooksFromMethods(deps.methods);
        }
        if (IO.implements && IO.implements.length > 0) {
            deps.implements = IO.implements;
            if (this.isGuard(IO.implements)) {
                // We don't want the Guard to show up in the Classes menu
                excludeFromClassArray = true;
                deps.type = 'guard';
                outputSymbols.guards.push(deps);
            }
        }
        if (typeof IO.ignore === 'undefined') {
            this.debug(deps);
            if (!excludeFromClassArray) {
                outputSymbols.classes.push(deps);
            }
        }
        else {
            this.ignore(deps);
        }
    };
    AngularDependencies.prototype.getSourceFileDecorators = function (initialSrcFile, outputSymbols) {
        var _this = this;
        var cleaner = (process.cwd() + path.sep).replace(/\\/g, '/');
        var fileName = initialSrcFile.fileName.replace(cleaner, '');
        var scannedFile = initialSrcFile;
        // Search in file for variable statement as routes definitions
        var astFile = typeof ast$2.getSourceFile(initialSrcFile.fileName) !== 'undefined'
            ? ast$2.getSourceFile(initialSrcFile.fileName)
            : ast$2.addExistingSourceFile(initialSrcFile.fileName);
        var variableRoutesStatements = astFile.getVariableStatements();
        var hasRoutesStatements = false;
        if (variableRoutesStatements.length > 0) {
            // Clean file for spread and dynamics inside routes definitions
            variableRoutesStatements.forEach(function (s) {
                var variableDeclarations = s.getDeclarations();
                var len = variableDeclarations.length;
                var i = 0;
                for (i; i < len; i++) {
                    if (variableDeclarations[i].compilerNode.type) {
                        if (variableDeclarations[i].compilerNode.type.typeName &&
                            variableDeclarations[i].compilerNode.type.typeName.text === 'Routes') {
                            hasRoutesStatements = true;
                        }
                    }
                }
            });
        }
        if (hasRoutesStatements && !Configuration$1.mainData.disableRoutesGraph) {
            // Clean file for spread and dynamics inside routes definitions
            logger.info('Analysing routes definitions and clean them if necessary');
            // scannedFile = RouterParserUtil.cleanFileIdentifiers(astFile).compilerNode;
            RouterParserUtil$1.cleanFileSpreads(astFile);
            scannedFile = RouterParserUtil$1.cleanCallExpressions(astFile).compilerNode;
            scannedFile = RouterParserUtil$1.cleanFileDynamics(astFile).compilerNode;
            scannedFile.kind = Ast.SyntaxKind.SourceFile;
        }
        Ast.ts.forEachChild(scannedFile, function (initialNode) {
            if (_this.jsDocHelper.hasJSDocInternalTag(fileName, scannedFile, initialNode) &&
                Configuration$1.mainData.disableInternal) {
                return;
            }
            var parseNode = function (file, srcFile, node, fileBody) {
                var sourceCode = srcFile.getText();
                var hash = crypto$5
                    .createHash('md5')
                    .update(sourceCode)
                    .digest('hex');
                if (node.decorators) {
                    var classWithCustomDecorator_1 = false;
                    var visitDecorator = function (visitedDecorator, index) {
                        var deps;
                        var name = _this.getSymboleName(node);
                        var props = _this.findProperties(visitedDecorator, srcFile);
                        var IO = _this.componentHelper.getComponentIO(file, srcFile, node, fileBody);
                        if (_this.isModule(visitedDecorator)) {
                            var moduleDep = new ModuleDepFactory(_this.moduleHelper).create(file, srcFile, name, props, IO);
                            if (RouterParserUtil$1.hasRouterModuleInImports(moduleDep.imports)) {
                                RouterParserUtil$1.addModuleWithRoutes(name, _this.moduleHelper.getModuleImportsRaw(props, srcFile), file);
                            }
                            deps = moduleDep;
                            if (typeof IO.ignore === 'undefined') {
                                RouterParserUtil$1.addModule(name, moduleDep.imports);
                                outputSymbols.modules.push(moduleDep);
                                outputSymbols.modulesForGraph.push(moduleDep);
                            }
                        }
                        else if (_this.isComponent(visitedDecorator)) {
                            if (props.length === 0) {
                                return;
                            }
                            var componentDep = new ComponentDepFactory(_this.componentHelper).create(file, srcFile, name, props, IO);
                            deps = componentDep;
                            if (typeof IO.ignore === 'undefined') {
                                ComponentsTreeEngine$1.addComponent(componentDep);
                                outputSymbols.components.push(componentDep);
                            }
                        }
                        else if (_this.isController(visitedDecorator)) {
                            var controllerDep = new ControllerDepFactory().create(file, srcFile, name, props, IO);
                            deps = controllerDep;
                            if (typeof IO.ignore === 'undefined') {
                                outputSymbols.controllers.push(controllerDep);
                            }
                        }
                        else if (_this.isInjectable(visitedDecorator)) {
                            var injectableDeps = {
                                name: name,
                                id: 'injectable-' + name + '-' + hash,
                                file: file,
                                properties: IO.properties,
                                methods: IO.methods,
                                description: IO.description,
                                sourceCode: srcFile.getText(),
                                exampleUrls: _this.componentHelper.getComponentExampleUrls(srcFile.getText())
                            };
                            if (IO.constructor) {
                                injectableDeps.constructorObj = IO.constructor;
                            }
                            if (IO.jsdoctags && IO.jsdoctags.length > 0) {
                                injectableDeps.jsdoctags = IO.jsdoctags[0].tags;
                            }
                            if (IO.accessors) {
                                injectableDeps.accessors = IO.accessors;
                            }
                            if (IO["extends"]) {
                                injectableDeps["extends"] = IO["extends"];
                            }
                            deps = injectableDeps;
                            if (typeof IO.ignore === 'undefined') {
                                if (_.includes(IO.implements, 'HttpInterceptor')) {
                                    injectableDeps.type = 'interceptor';
                                    outputSymbols.interceptors.push(injectableDeps);
                                }
                                else if (_this.isGuard(IO.implements)) {
                                    injectableDeps.type = 'guard';
                                    outputSymbols.guards.push(injectableDeps);
                                }
                                else {
                                    injectableDeps.type = 'injectable';
                                    _this.addNewEntityInStore(injectableDeps, outputSymbols.injectables);
                                }
                            }
                        }
                        else if (_this.isPipe(visitedDecorator)) {
                            var pipeDeps = {
                                name: name,
                                id: 'pipe-' + name + '-' + hash,
                                file: file,
                                type: 'pipe',
                                description: IO.description,
                                properties: IO.properties,
                                methods: IO.methods,
                                pure: _this.componentHelper.getComponentPure(props, srcFile),
                                ngname: _this.componentHelper.getComponentName(props, srcFile),
                                sourceCode: srcFile.getText(),
                                exampleUrls: _this.componentHelper.getComponentExampleUrls(srcFile.getText())
                            };
                            if (IO.jsdoctags && IO.jsdoctags.length > 0) {
                                pipeDeps.jsdoctags = IO.jsdoctags[0].tags;
                            }
                            deps = pipeDeps;
                            if (typeof IO.ignore === 'undefined') {
                                outputSymbols.pipes.push(pipeDeps);
                            }
                        }
                        else if (_this.isDirective(visitedDecorator)) {
                            if (props.length === 0) {
                                return;
                            }
                            var directiveDeps = new DirectiveDepFactory(_this.componentHelper).create(file, srcFile, name, props, IO);
                            deps = directiveDeps;
                            if (typeof IO.ignore === 'undefined') {
                                outputSymbols.directives.push(directiveDeps);
                            }
                        }
                        else {
                            var hasMultipleDecoratorsWithInternalOne = _this.hasInternalDecorator(node.decorators);
                            // Just a class
                            if (!classWithCustomDecorator_1 &&
                                !hasMultipleDecoratorsWithInternalOne) {
                                classWithCustomDecorator_1 = true;
                                _this.processClass(node, file, srcFile, outputSymbols, fileBody);
                            }
                        }
                        _this.cache.set(name, deps);
                        if (typeof IO.ignore === 'undefined') {
                            _this.debug(deps);
                        }
                        else {
                            _this.ignore(deps);
                        }
                    };
                    var filterByDecorators = function (filteredNode) {
                        if (filteredNode.expression && filteredNode.expression.expression) {
                            var _test = /(NgModule|Component|Injectable|Pipe|Directive)/.test(filteredNode.expression.expression.text);
                            if (!_test && Ast.ts.isClassDeclaration(node)) {
                                _test = true;
                            }
                            return _test;
                        }
                        if (Ast.ts.isClassDeclaration(node)) {
                            return true;
                        }
                        return false;
                    };
                    node.decorators.filter(filterByDecorators).forEach(visitDecorator);
                }
                else if (node.symbol) {
                    if (node.symbol.flags === Ast.ts.SymbolFlags.Class) {
                        _this.processClass(node, file, srcFile, outputSymbols, fileBody);
                    }
                    else if (node.symbol.flags === Ast.ts.SymbolFlags.Interface) {
                        var name = _this.getSymboleName(node);
                        var IO = _this.getInterfaceIO(file, srcFile, node, fileBody);
                        var interfaceDeps = {
                            name: name,
                            id: 'interface-' + name + '-' + hash,
                            file: file,
                            type: 'interface',
                            sourceCode: srcFile.getText()
                        };
                        if (IO.properties) {
                            interfaceDeps.properties = IO.properties;
                        }
                        if (IO.indexSignatures) {
                            interfaceDeps.indexSignatures = IO.indexSignatures;
                        }
                        if (IO.kind) {
                            interfaceDeps.kind = IO.kind;
                        }
                        if (IO.description) {
                            interfaceDeps.description = IO.description;
                        }
                        if (IO.methods) {
                            interfaceDeps.methods = IO.methods;
                        }
                        if (IO["extends"]) {
                            interfaceDeps["extends"] = IO["extends"];
                        }
                        if (typeof IO.ignore === 'undefined') {
                            _this.debug(interfaceDeps);
                            outputSymbols.interfaces.push(interfaceDeps);
                        }
                        else {
                            _this.ignore(interfaceDeps);
                        }
                    }
                    else if (Ast.ts.isFunctionDeclaration(node)) {
                        var infos = _this.visitFunctionDeclaration(node);
                        // let tags = this.visitFunctionDeclarationJSDocTags(node);
                        var name = infos.name;
                        var functionDep = {
                            name: name,
                            file: file,
                            ctype: 'miscellaneous',
                            subtype: 'function',
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node)
                        };
                        if (infos.args) {
                            functionDep.args = infos.args;
                        }
                        if (infos.returnType) {
                            functionDep.returnType = infos.returnType;
                        }
                        if (infos.jsdoctags && infos.jsdoctags.length > 0) {
                            functionDep.jsdoctags = infos.jsdoctags;
                        }
                        if (typeof infos.ignore === 'undefined') {
                            if (!(_this.hasPrivateJSDocTag(functionDep.jsdoctags) &&
                                Configuration$1.mainData.disablePrivate)) {
                                outputSymbols.miscellaneous.functions.push(functionDep);
                            }
                        }
                    }
                    else if (Ast.ts.isEnumDeclaration(node)) {
                        var infos = _this.visitEnumDeclaration(node);
                        var name = node.name.text;
                        var enumDeps = {
                            name: name,
                            childs: infos,
                            ctype: 'miscellaneous',
                            subtype: 'enum',
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node),
                            file: file
                        };
                        if (!isIgnore(node)) {
                            outputSymbols.miscellaneous.enumerations.push(enumDeps);
                        }
                    }
                    else if (Ast.ts.isTypeAliasDeclaration(node)) {
                        var infos = _this.visitTypeDeclaration(node);
                        var name = infos.name;
                        var typeAliasDeps = {
                            name: name,
                            ctype: 'miscellaneous',
                            subtype: 'typealias',
                            rawtype: _this.classHelper.visitType(node),
                            file: file,
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node)
                        };
                        if (node.type) {
                            typeAliasDeps.kind = node.type.kind;
                            if (typeAliasDeps.rawtype === '') {
                                typeAliasDeps.rawtype = kindToType(node.type.kind);
                            }
                        }
                        if (!isIgnore(node)) {
                            outputSymbols.miscellaneous.typealiases.push(typeAliasDeps);
                        }
                    }
                    else if (Ast.ts.isModuleDeclaration(node)) {
                        if (node.body) {
                            if (node.body.statements && node.body.statements.length > 0) {
                                node.body.statements.forEach(function (statement) {
                                    return parseNode(file, srcFile, statement, node.body);
                                });
                            }
                        }
                    }
                }
                else {
                    var IO = _this.getRouteIO(file, srcFile, node);
                    if (IO.routes) {
                        var newRoutes = void 0;
                        try {
                            newRoutes = RouterParserUtil$1.cleanRawRouteParsed(IO.routes);
                        }
                        catch (e) {
                            // tslint:disable-next-line:max-line-length
                            logger.error('Routes parsing error, maybe a trailing comma or an external variable, trying to fix that later after sources scanning.');
                            newRoutes = IO.routes.replace(/ /gm, '');
                            RouterParserUtil$1.addIncompleteRoute({
                                data: newRoutes,
                                file: file
                            });
                            return true;
                        }
                        outputSymbols.routes = outputSymbols.routes.concat(newRoutes);
                    }
                    if (Ast.ts.isClassDeclaration(node)) {
                        _this.processClass(node, file, srcFile, outputSymbols, fileBody);
                    }
                    if (Ast.ts.isExpressionStatement(node) || Ast.ts.isIfStatement(node)) {
                        var bootstrapModuleReference = 'bootstrapModule';
                        // Find the root module with bootstrapModule call
                        // 1. find a simple call : platformBrowserDynamic().bootstrapModule(AppModule);
                        // 2. or inside a call :
                        // () => {
                        //     platformBrowserDynamic().bootstrapModule(AppModule);
                        // });
                        // 3. with a catch : platformBrowserDynamic().bootstrapModule(AppModule).catch(error => console.error(error));
                        // 4. with parameters : platformBrowserDynamic().bootstrapModule(AppModule, {}).catch(error => console.error(error));
                        // Find recusively in expression nodes one with name 'bootstrapModule'
                        var rootModule_1;
                        var resultNode = void 0;
                        if (srcFile.text.indexOf(bootstrapModuleReference) !== -1) {
                            if (node.expression) {
                                resultNode = _this.findExpressionByNameInExpressions(node.expression, 'bootstrapModule');
                            }
                            if (typeof node.thenStatement !== 'undefined') {
                                if (node.thenStatement.statements &&
                                    node.thenStatement.statements.length > 0) {
                                    var firstStatement = node.thenStatement.statements[0];
                                    resultNode = _this.findExpressionByNameInExpressions(firstStatement.expression, 'bootstrapModule');
                                }
                            }
                            if (!resultNode) {
                                if (node.expression &&
                                    node.expression.arguments &&
                                    node.expression.arguments.length > 0) {
                                    resultNode = _this.findExpressionByNameInExpressionArguments(node.expression.arguments, 'bootstrapModule');
                                }
                            }
                            if (resultNode) {
                                if (resultNode.arguments.length > 0) {
                                    _.forEach(resultNode.arguments, function (argument) {
                                        if (argument.text) {
                                            rootModule_1 = argument.text;
                                        }
                                    });
                                }
                                if (rootModule_1) {
                                    RouterParserUtil$1.setRootModule(rootModule_1);
                                }
                            }
                        }
                    }
                    if (Ast.ts.isVariableStatement(node) && !RouterParserUtil$1.isVariableRoutes(node)) {
                        var infos = _this.visitVariableDeclaration(node);
                        var name = infos.name;
                        var deps = {
                            name: name,
                            ctype: 'miscellaneous',
                            subtype: 'variable',
                            file: file
                        };
                        deps.type = infos.type ? infos.type : '';
                        if (infos.defaultValue) {
                            deps.defaultValue = infos.defaultValue;
                        }
                        if (infos.initializer) {
                            deps.initializer = infos.initializer;
                        }
                        if (node.jsDoc && node.jsDoc.length > 0 && node.jsDoc[0].comment) {
                            deps.description = marked$2(node.jsDoc[0].comment);
                        }
                        if (isModuleWithProviders(node)) {
                            var routingInitializer = getModuleWithProviders(node);
                            RouterParserUtil$1.addModuleWithRoutes(name, [routingInitializer], file);
                            RouterParserUtil$1.addModule(name, [routingInitializer]);
                        }
                        if (!isIgnore(node)) {
                            outputSymbols.miscellaneous.variables.push(deps);
                        }
                    }
                    if (Ast.ts.isTypeAliasDeclaration(node)) {
                        var infos = _this.visitTypeDeclaration(node);
                        var name = infos.name;
                        var deps = {
                            name: name,
                            ctype: 'miscellaneous',
                            subtype: 'typealias',
                            rawtype: _this.classHelper.visitType(node),
                            file: file,
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node)
                        };
                        if (node.type) {
                            deps.kind = node.type.kind;
                        }
                        if (!isIgnore(node)) {
                            outputSymbols.miscellaneous.typealiases.push(deps);
                        }
                    }
                    if (Ast.ts.isFunctionDeclaration(node)) {
                        var infos = _this.visitFunctionDeclaration(node);
                        var name = infos.name;
                        var functionDep = {
                            name: name,
                            ctype: 'miscellaneous',
                            subtype: 'function',
                            file: file,
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node)
                        };
                        if (infos.args) {
                            functionDep.args = infos.args;
                        }
                        if (infos.returnType) {
                            functionDep.returnType = infos.returnType;
                        }
                        if (infos.jsdoctags && infos.jsdoctags.length > 0) {
                            functionDep.jsdoctags = infos.jsdoctags;
                        }
                        if (typeof infos.ignore === 'undefined') {
                            if (!(_this.hasPrivateJSDocTag(functionDep.jsdoctags) &&
                                Configuration$1.mainData.disablePrivate)) {
                                outputSymbols.miscellaneous.functions.push(functionDep);
                            }
                        }
                    }
                    if (Ast.ts.isEnumDeclaration(node)) {
                        var infos = _this.visitEnumDeclaration(node);
                        var name = node.name.text;
                        var enumDeps = {
                            name: name,
                            childs: infos,
                            ctype: 'miscellaneous',
                            subtype: 'enum',
                            description: _this.visitEnumTypeAliasFunctionDeclarationDescription(node),
                            file: file
                        };
                        if (!isIgnore(node)) {
                            outputSymbols.miscellaneous.enumerations.push(enumDeps);
                        }
                    }
                }
            };
            parseNode(fileName, scannedFile, initialNode);
        });
    };
    /**
     * Function to in a specific store an entity, and check before is there is not the same one
     * in that store : same name, id and file
     * @param entity Entity to store
     * @param store Store
     */
    AngularDependencies.prototype.addNewEntityInStore = function (entity, store) {
        var findSameEntityInStore = _.filter(store, {
            name: entity.name,
            id: entity.id,
            file: entity.file
        });
        if (findSameEntityInStore.length === 0) {
            store.push(entity);
        }
    };
    AngularDependencies.prototype.debug = function (deps) {
        if (deps) {
            logger.debug('found', "" + deps.name);
        }
        else {
            return;
        }
        ['imports', 'exports', 'declarations', 'providers', 'bootstrap'].forEach(function (symbols) {
            if (deps[symbols] && deps[symbols].length > 0) {
                logger.debug('', "- " + symbols + ":");
                deps[symbols]
                    .map(function (i) { return i.name; })
                    .forEach(function (d) {
                    logger.debug('', "\t- " + d);
                });
            }
        });
    };
    AngularDependencies.prototype.ignore = function (deps) {
        if (deps) {
            logger.warn('ignore', "" + deps.name);
        }
        else {
            return;
        }
    };
    AngularDependencies.prototype.findExpressionByNameInExpressions = function (entryNode, name) {
        var result;
        var loop = function (node, z) {
            if (node) {
                if (node.expression && !node.expression.name) {
                    loop(node.expression, z);
                }
                if (node.expression && node.expression.name) {
                    if (node.expression.name.text === z) {
                        result = node;
                    }
                    else {
                        loop(node.expression, z);
                    }
                }
            }
        };
        loop(entryNode, name);
        return result;
    };
    AngularDependencies.prototype.findExpressionByNameInExpressionArguments = function (arg, name) {
        var result;
        var that = this;
        var i = 0;
        var len = arg.length;
        var loop = function (node, z) {
            if (node.body) {
                if (node.body.statements && node.body.statements.length > 0) {
                    var j = 0;
                    var leng = node.body.statements.length;
                    for (j; j < leng; j++) {
                        result = that.findExpressionByNameInExpressions(node.body.statements[j], z);
                    }
                }
            }
        };
        for (i; i < len; i++) {
            loop(arg[i], name);
        }
        return result;
    };
    AngularDependencies.prototype.parseDecorators = function (decorators, type) {
        var result = false;
        if (decorators.length > 1) {
            _.forEach(decorators, function (decorator) {
                if (decorator.expression.expression) {
                    if (decorator.expression.expression.text === type) {
                        result = true;
                    }
                }
            });
        }
        else {
            if (decorators[0].expression.expression) {
                if (decorators[0].expression.expression.text === type) {
                    result = true;
                }
            }
        }
        return result;
    };
    AngularDependencies.prototype.parseDecorator = function (decorator, type) {
        var result = false;
        if (decorator.expression.expression) {
            if (decorator.expression.expression.text === type) {
                result = true;
            }
        }
        return result;
    };
    AngularDependencies.prototype.isController = function (metadata) {
        return this.parseDecorator(metadata, 'Controller');
    };
    AngularDependencies.prototype.isComponent = function (metadata) {
        return this.parseDecorator(metadata, 'Component');
    };
    AngularDependencies.prototype.isPipe = function (metadata) {
        return this.parseDecorator(metadata, 'Pipe');
    };
    AngularDependencies.prototype.isDirective = function (metadata) {
        return this.parseDecorator(metadata, 'Directive');
    };
    AngularDependencies.prototype.isInjectable = function (metadata) {
        return this.parseDecorator(metadata, 'Injectable');
    };
    AngularDependencies.prototype.isModule = function (metadata) {
        return this.parseDecorator(metadata, 'NgModule') || this.parseDecorator(metadata, 'Module');
    };
    AngularDependencies.prototype.hasInternalDecorator = function (metadatas) {
        return (this.parseDecorators(metadatas, 'Controller') ||
            this.parseDecorators(metadatas, 'Component') ||
            this.parseDecorators(metadatas, 'Pipe') ||
            this.parseDecorators(metadatas, 'Directive') ||
            this.parseDecorators(metadatas, 'Injectable') ||
            this.parseDecorators(metadatas, 'NgModule') ||
            this.parseDecorators(metadatas, 'Module'));
    };
    AngularDependencies.prototype.isGuard = function (ioImplements) {
        return (_.includes(ioImplements, 'CanActivate') ||
            _.includes(ioImplements, 'CanActivateChild') ||
            _.includes(ioImplements, 'CanDeactivate') ||
            _.includes(ioImplements, 'Resolve') ||
            _.includes(ioImplements, 'CanLoad'));
    };
    AngularDependencies.prototype.getSymboleName = function (node) {
        return node.name.text;
    };
    AngularDependencies.prototype.findProperties = function (visitedNode, sourceFile) {
        if (visitedNode.expression &&
            visitedNode.expression.arguments &&
            visitedNode.expression.arguments.length > 0) {
            var pop = visitedNode.expression.arguments[0];
            if (pop && pop.properties && pop.properties.length >= 0) {
                return pop.properties;
            }
            else if (pop && pop.kind && pop.kind === Ast.SyntaxKind.StringLiteral) {
                return [pop];
            }
            else {
                logger.warn('Empty metadatas, trying to find it with imports.');
                return ImportsUtil$1.findValueInImportOrLocalVariables(pop.text, sourceFile);
            }
        }
        return [];
    };
    AngularDependencies.prototype.isAngularLifecycleHook = function (methodName) {
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var ANGULAR_LIFECYCLE_METHODS = [
            'ngOnInit',
            'ngOnChanges',
            'ngDoCheck',
            'ngOnDestroy',
            'ngAfterContentInit',
            'ngAfterContentChecked',
            'ngAfterViewInit',
            'ngAfterViewChecked',
            'writeValue',
            'registerOnChange',
            'registerOnTouched',
            'setDisabledState'
        ];
        return ANGULAR_LIFECYCLE_METHODS.indexOf(methodName) >= 0;
    };
    AngularDependencies.prototype.visitTypeDeclaration = function (node) {
        var result = {
            name: node.name.text,
            kind: node.kind
        };
        var jsdoctags = this.jsdocParserUtil.getJSDocs(node);
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
            }
        }
        return result;
    };
    AngularDependencies.prototype.visitArgument = function (arg) {
        var result = {
            name: arg.name.text,
            type: this.classHelper.visitType(arg)
        };
        if (arg.dotDotDotToken) {
            result.dotDotDotToken = true;
        }
        if (arg.questionToken) {
            result.optional = true;
        }
        if (arg.type) {
            result.type = this.mapType(arg.type.kind);
            if (arg.type.kind === 157) {
                // try replace TypeReference with typeName
                if (arg.type.typeName) {
                    result.type = arg.type.typeName.text;
                }
            }
        }
        return result;
    };
    AngularDependencies.prototype.mapType = function (type) {
        switch (type) {
            case 95:
                return 'null';
            case 119:
                return 'any';
            case 122:
                return 'boolean';
            case 130:
                return 'never';
            case 133:
                return 'number';
            case 136:
                return 'string';
            case 139:
                return 'undefined';
            case 159:
                return 'typeReference';
        }
    };
    AngularDependencies.prototype.hasPrivateJSDocTag = function (tags) {
        var result = false;
        if (tags) {
            tags.forEach(function (tag) {
                if (tag.tagName && tag.tagName.text && tag.tagName.text === 'private') {
                    result = true;
                }
            });
        }
        return result;
    };
    AngularDependencies.prototype.visitFunctionDeclaration = function (method) {
        var _this = this;
        var methodName = method.name ? method.name.text : 'Unnamed function';
        var result = {
            name: methodName,
            args: method.parameters ? method.parameters.map(function (prop) { return _this.visitArgument(prop); }) : []
        };
        var jsdoctags = this.jsdocParserUtil.getJSDocs(method);
        if (typeof method.type !== 'undefined') {
            result.returnType = this.classHelper.visitType(method.type);
        }
        if (method.modifiers) {
            if (method.modifiers.length > 0) {
                var kinds = method.modifiers
                    .map(function (modifier) {
                    return modifier.kind;
                })
                    .reverse();
                if (_.indexOf(kinds, Ast.SyntaxKind.PublicKeyword) !== -1 &&
                    _.indexOf(kinds, Ast.SyntaxKind.StaticKeyword) !== -1) {
                    kinds = kinds.filter(function (kind) { return kind !== Ast.SyntaxKind.PublicKeyword; });
                }
            }
        }
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result.jsdoctags = markedtags(jsdoctags[0].tags);
                _.forEach(jsdoctags[0].tags, function (tag) {
                    if (tag.tagName) {
                        if (tag.tagName.text) {
                            if (tag.tagName.text.indexOf('ignore') > -1) {
                                result.ignore = true;
                            }
                        }
                    }
                });
            }
        }
        if (result.jsdoctags && result.jsdoctags.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args, result.jsdoctags);
        }
        else if (result.args.length > 0) {
            result.jsdoctags = mergeTagsAndArgs(result.args);
        }
        return result;
    };
    AngularDependencies.prototype.visitVariableDeclaration = function (node) {
        if (node.declarationList.declarations) {
            var i = 0;
            var len = node.declarationList.declarations.length;
            for (i; i < len; i++) {
                var result = {
                    name: node.declarationList.declarations[i].name.text,
                    defaultValue: node.declarationList.declarations[i].initializer
                        ? this.classHelper.stringifyDefaultValue(node.declarationList.declarations[i].initializer)
                        : undefined
                };
                if (node.declarationList.declarations[i].initializer) {
                    result.initializer = node.declarationList.declarations[i].initializer;
                }
                if (node.declarationList.declarations[i].type) {
                    result.type = this.classHelper.visitType(node.declarationList.declarations[i].type);
                }
                if (typeof result.type === 'undefined' && result.initializer) {
                    result.type = kindToType(result.initializer.kind);
                }
                return result;
            }
        }
    };
    AngularDependencies.prototype.visitFunctionDeclarationJSDocTags = function (node) {
        var jsdoctags = this.jsdocParserUtil.getJSDocs(node);
        var result;
        if (jsdoctags && jsdoctags.length >= 1) {
            if (jsdoctags[0].tags) {
                result = markedtags(jsdoctags[0].tags);
            }
        }
        return result;
    };
    AngularDependencies.prototype.visitEnumTypeAliasFunctionDeclarationDescription = function (node) {
        var description = '';
        if (node.jsDoc) {
            if (node.jsDoc.length > 0) {
                if (typeof node.jsDoc[0].comment !== 'undefined') {
                    description = marked$2(node.jsDoc[0].comment);
                }
            }
        }
        return description;
    };
    AngularDependencies.prototype.visitEnumDeclaration = function (node) {
        var result = [];
        if (node.members) {
            var i = 0;
            var len = node.members.length;
            for (i; i < len; i++) {
                var member = {
                    name: node.members[i].name.text
                };
                if (node.members[i].initializer) {
                    member.value = node.members[i].initializer.text;
                }
                result.push(member);
            }
        }
        return result;
    };
    AngularDependencies.prototype.visitEnumDeclarationForRoutes = function (fileName, node) {
        if (node.declarationList.declarations) {
            var i = 0;
            var len = node.declarationList.declarations.length;
            for (i; i < len; i++) {
                var routesInitializer = node.declarationList.declarations[i].initializer;
                var data = new CodeGenerator().generate(routesInitializer);
                RouterParserUtil$1.addRoute({
                    name: node.declarationList.declarations[i].name.text,
                    data: RouterParserUtil$1.cleanRawRoute(data),
                    filename: fileName
                });
                return [
                    {
                        routes: data
                    }
                ];
            }
        }
        return [];
    };
    AngularDependencies.prototype.getRouteIO = function (filename, sourceFile, node) {
        var _this = this;
        var res;
        if (sourceFile.statements) {
            res = sourceFile.statements.reduce(function (directive, statement) {
                if (RouterParserUtil$1.isVariableRoutes(statement)) {
                    if (statement.pos === node.pos && statement.end === node.end) {
                        return directive.concat(_this.visitEnumDeclarationForRoutes(filename, statement));
                    }
                }
                return directive;
            }, []);
            return res[0] || {};
        }
        else {
            return {};
        }
    };
    AngularDependencies.prototype.getClassIO = function (filename, sourceFile, node, fileBody) {
        var _this = this;
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var reducedSource = fileBody ? fileBody.statements : sourceFile.statements;
        var res = reducedSource.reduce(function (directive, statement) {
            if (Ast.ts.isClassDeclaration(statement)) {
                if (statement.pos === node.pos && statement.end === node.end) {
                    return directive.concat(_this.classHelper.visitClassDeclaration(filename, statement, sourceFile));
                }
            }
            return directive;
        }, []);
        return res[0] || {};
    };
    AngularDependencies.prototype.getInterfaceIO = function (filename, sourceFile, node, fileBody) {
        var _this = this;
        /**
         * Copyright https://github.com/ng-bootstrap/ng-bootstrap
         */
        var reducedSource = fileBody ? fileBody.statements : sourceFile.statements;
        var res = reducedSource.reduce(function (directive, statement) {
            if (Ast.ts.isInterfaceDeclaration(statement)) {
                if (statement.pos === node.pos && statement.end === node.end) {
                    return directive.concat(_this.classHelper.visitClassDeclaration(filename, statement, sourceFile));
                }
            }
            return directive;
        }, []);
        return res[0] || {};
    };
    return AngularDependencies;
}(FrameworkDependencies));

var AngularJSDependencies = /** @class */ (function (_super) {
    __extends(AngularJSDependencies, _super);
    function AngularJSDependencies(files, options) {
        var _this = _super.call(this, files, options) || this;
        _this.cache = new ComponentCache();
        _this.moduleHelper = new ModuleHelper(_this.cache);
        _this.jsDocHelper = new JsDocHelper();
        _this.symbolHelper = new SymbolHelper();
        return _this;
    }
    AngularJSDependencies.prototype.getDependencies = function () {
        var deps = {
            modules: [],
            modulesForGraph: [],
            components: [],
            injectables: [],
            interceptors: [],
            pipes: [],
            directives: [],
            routes: [],
            classes: [],
            interfaces: [],
            miscellaneous: {
                variables: [],
                functions: [],
                typealiases: [],
                enumerations: []
            },
            routesTree: undefined
        };
        return deps;
    };
    return AngularJSDependencies;
}(FrameworkDependencies));

function promiseSequential(promises) {
    if (!Array.isArray(promises)) {
        throw new Error('First argument need to be an array of Promises');
    }
    return new Promise(function (resolve, reject) {
        var count = 0;
        var results = [];
        var iterateeFunc = function (previousPromise, currentPromise) {
            return previousPromise
                .then(function (result) {
                if (count++ !== 0) {
                    results = results.concat(result);
                }
                return currentPromise(result, results, count);
            })["catch"](function (err) {
                return reject(err);
            });
        };
        promises = promises.concat(function () { return Promise.resolve(); });
        promises.reduce(iterateeFunc, Promise.resolve(false)).then(function (res) {
            resolve(results);
        });
    });
}

var chokidar = require('chokidar');
var marked$3 = require('marked');
var traverse$3 = require('traverse');
var crypto$6 = require('crypto');
var cwd = process.cwd();
var startTime = new Date();
var generationPromiseResolve;
var generationPromiseReject;
var generationPromise = new Promise(function (resolve, reject) {
    generationPromiseResolve = resolve;
    generationPromiseReject = reject;
});
var Application = /** @class */ (function () {
    /**
     * Create a new compodoc application instance.
     *
     * @param options An object containing the options that should be used.
     */
    function Application(options) {
        var _this = this;
        /**
         * Files changed during watch scanning
         */
        this.watchChangedFiles = [];
        /**
         * Boolean for watching status
         * @type {boolean}
         */
        this.isWatching = false;
        /**
         * Store package.json data
         */
        this.packageJsonData = {};
        this.preparePipes = function (somePipes) {
            logger.info('Prepare pipes');
            Configuration$1.mainData.pipes = somePipes ? somePipes : DependenciesEngine$1.getPipes();
            return new Promise(function (resolve, reject) {
                var i = 0;
                var len = Configuration$1.mainData.pipes.length;
                var loop = function () {
                    if (i < len) {
                        var pipe = Configuration$1.mainData.pipes[i];
                        if (MarkdownEngine$1.hasNeighbourReadmeFile(pipe.file)) {
                            logger.info(" " + pipe.name + " has a README file, include it");
                            var readme = MarkdownEngine$1.readNeighbourReadmeFile(pipe.file);
                            pipe.readme = marked$3(readme);
                        }
                        var page = {
                            path: 'pipes',
                            name: pipe.name,
                            id: pipe.id,
                            navTabs: _this.getNavTabs(pipe),
                            context: 'pipe',
                            pipe: pipe,
                            depth: 1,
                            pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                        };
                        if (pipe.isDuplicate) {
                            page.name += '-' + pipe.duplicateId;
                        }
                        Configuration$1.addPage(page);
                        i++;
                        loop();
                    }
                    else {
                        resolve();
                    }
                };
                loop();
            });
        };
        this.prepareClasses = function (someClasses) {
            logger.info('Prepare classes');
            Configuration$1.mainData.classes = someClasses
                ? someClasses
                : DependenciesEngine$1.getClasses();
            return new Promise(function (resolve, reject) {
                var i = 0;
                var len = Configuration$1.mainData.classes.length;
                var loop = function () {
                    if (i < len) {
                        var classe = Configuration$1.mainData.classes[i];
                        if (MarkdownEngine$1.hasNeighbourReadmeFile(classe.file)) {
                            logger.info(" " + classe.name + " has a README file, include it");
                            var readme = MarkdownEngine$1.readNeighbourReadmeFile(classe.file);
                            classe.readme = marked$3(readme);
                        }
                        var page = {
                            path: 'classes',
                            name: classe.name,
                            id: classe.id,
                            navTabs: _this.getNavTabs(classe),
                            context: 'class',
                            "class": classe,
                            depth: 1,
                            pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                        };
                        if (classe.isDuplicate) {
                            page.name += '-' + classe.duplicateId;
                        }
                        Configuration$1.addPage(page);
                        i++;
                        loop();
                    }
                    else {
                        resolve();
                    }
                };
                loop();
            });
        };
        for (var option in options) {
            if (typeof Configuration$1.mainData[option] !== 'undefined') {
                Configuration$1.mainData[option] = options[option];
            }
            // For documentationMainName, process it outside the loop, for handling conflict with pages name
            if (option === 'name') {
                Configuration$1.mainData.documentationMainName = options[option];
            }
            // For documentationMainName, process it outside the loop, for handling conflict with pages name
            if (option === 'silent') {
                logger.silent = false;
            }
        }
    }
    /**
     * Start compodoc process
     */
    Application.prototype.generate = function () {
        var _this = this;
        process.on('unhandledRejection', this.unhandledRejectionListener);
        process.on('uncaughtException', this.uncaughtExceptionListener);
        I18nEngine$1.init(Configuration$1.mainData.language);
        if (Configuration$1.mainData.output.charAt(Configuration$1.mainData.output.length - 1) !== '/') {
            Configuration$1.mainData.output += '/';
        }
        if (Configuration$1.mainData.exportFormat !== COMPODOC_DEFAULTS.exportFormat) {
            this.processPackageJson();
        }
        else {
            HtmlEngine$1.init(Configuration$1.mainData.templates).then(function () { return _this.processPackageJson(); });
        }
        return generationPromise;
    };
    Application.prototype.endCallback = function () {
        process.removeListener('unhandledRejection', this.unhandledRejectionListener);
        process.removeListener('uncaughtException', this.uncaughtExceptionListener);
    };
    Application.prototype.unhandledRejectionListener = function (err, p) {
        console.log('Unhandled Rejection at:', p, 'reason:', err);
        logger.error('Sorry, but there was a problem during parsing or generation of the documentation. Please fill an issue on github. (https://github.com/compodoc/compodoc/issues/new)'); // tslint:disable-line
        process.exit(1);
    };
    Application.prototype.uncaughtExceptionListener = function (err) {
        logger.error(err);
        logger.error('Sorry, but there was a problem during parsing or generation of the documentation. Please fill an issue on github. (https://github.com/compodoc/compodoc/issues/new)'); // tslint:disable-line
        process.exit(1);
    };
    /**
     * Start compodoc documentation coverage
     */
    Application.prototype.testCoverage = function () {
        this.getDependenciesData();
    };
    /**
     * Store files for initial processing
     * @param  {Array<string>} files Files found during source folder and tsconfig scan
     */
    Application.prototype.setFiles = function (files) {
        this.files = files;
    };
    /**
     * Store files for watch processing
     * @param  {Array<string>} files Files found during source folder and tsconfig scan
     */
    Application.prototype.setUpdatedFiles = function (files) {
        this.updatedFiles = files;
    };
    /**
     * Return a boolean indicating presence of one TypeScript file in updatedFiles list
     * @return {boolean} Result of scan
     */
    Application.prototype.hasWatchedFilesTSFiles = function () {
        var result = false;
        _.forEach(this.updatedFiles, function (file) {
            if (path.extname(file) === '.ts') {
                result = true;
            }
        });
        return result;
    };
    /**
     * Return a boolean indicating presence of one root markdown files in updatedFiles list
     * @return {boolean} Result of scan
     */
    Application.prototype.hasWatchedFilesRootMarkdownFiles = function () {
        var result = false;
        _.forEach(this.updatedFiles, function (file) {
            if (path.extname(file) === '.md' && path.dirname(file) === cwd) {
                result = true;
            }
        });
        return result;
    };
    /**
     * Clear files for watch processing
     */
    Application.prototype.clearUpdatedFiles = function () {
        this.updatedFiles = [];
        this.watchChangedFiles = [];
    };
    Application.prototype.processPackageJson = function () {
        var _this = this;
        logger.info('Searching package.json file');
        FileEngine$1.get(cwd + path.sep + 'package.json').then(function (packageData) {
            var parsedData = JSON.parse(packageData);
            _this.packageJsonData = parsedData;
            if (typeof parsedData.name !== 'undefined' &&
                Configuration$1.mainData.documentationMainName === COMPODOC_DEFAULTS.title) {
                Configuration$1.mainData.documentationMainName =
                    parsedData.name + ' documentation';
            }
            if (typeof parsedData.description !== 'undefined') {
                Configuration$1.mainData.documentationMainDescription = parsedData.description;
            }
            Configuration$1.mainData.angularVersion = AngularVersionUtil$1.getAngularVersionOfProject(parsedData);
            logger.info('package.json file found');
            if (!Configuration$1.mainData.disableDependencies) {
                if (typeof parsedData.dependencies !== 'undefined') {
                    _this.processPackageDependencies(parsedData.dependencies);
                }
                if (typeof parsedData.peerDependencies !== 'undefined') {
                    _this.processPackagePeerDependencies(parsedData.peerDependencies);
                }
            }
            _this.processMarkdowns().then(function () {
                _this.getDependenciesData();
            }, function (errorMessage) {
                logger.error(errorMessage);
            });
        }, function (errorMessage) {
            logger.error(errorMessage);
            logger.error('Continuing without package.json file');
            _this.processMarkdowns().then(function () {
                _this.getDependenciesData();
            }, function (errorMessage1) {
                logger.error(errorMessage1);
            });
        });
    };
    Application.prototype.processPackagePeerDependencies = function (dependencies) {
        logger.info('Processing package.json peerDependencies');
        Configuration$1.mainData.packagePeerDependencies = dependencies;
        if (!Configuration$1.hasPage('dependencies')) {
            Configuration$1.addPage({
                name: 'dependencies',
                id: 'packageDependencies',
                context: 'package-dependencies',
                depth: 0,
                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
            });
        }
    };
    Application.prototype.processPackageDependencies = function (dependencies) {
        logger.info('Processing package.json dependencies');
        Configuration$1.mainData.packageDependencies = dependencies;
        Configuration$1.addPage({
            name: 'dependencies',
            id: 'packageDependencies',
            context: 'package-dependencies',
            depth: 0,
            pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
        });
    };
    Application.prototype.processMarkdowns = function () {
        logger.info('Searching README.md, CHANGELOG.md, CONTRIBUTING.md, LICENSE.md, TODO.md files');
        return new Promise(function (resolve, reject) {
            var i = 0;
            var markdowns = ['readme', 'changelog', 'contributing', 'license', 'todo'];
            var numberOfMarkdowns = 5;
            var loop = function () {
                if (i < numberOfMarkdowns) {
                    MarkdownEngine$1.getTraditionalMarkdown(markdowns[i].toUpperCase()).then(function (readmeData) {
                        Configuration$1.addPage({
                            name: markdowns[i] === 'readme' ? 'index' : markdowns[i],
                            context: 'getting-started',
                            id: 'getting-started',
                            markdown: readmeData.markdown,
                            data: readmeData.rawData,
                            depth: 0,
                            pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
                        });
                        if (markdowns[i] === 'readme') {
                            Configuration$1.mainData.readme = true;
                            Configuration$1.addPage({
                                name: 'overview',
                                id: 'overview',
                                context: 'overview',
                                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
                            });
                        }
                        else {
                            Configuration$1.mainData.markdowns.push({
                                name: markdowns[i],
                                uppername: markdowns[i].toUpperCase(),
                                depth: 0,
                                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
                            });
                        }
                        logger.info(markdowns[i].toUpperCase() + ".md file found");
                        i++;
                        loop();
                    }, function (errorMessage) {
                        logger.warn(errorMessage);
                        logger.warn("Continuing without " + markdowns[i].toUpperCase() + ".md file");
                        if (markdowns[i] === 'readme') {
                            Configuration$1.addPage({
                                name: 'index',
                                id: 'index',
                                context: 'overview',
                                depth: 0,
                                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
                            });
                        }
                        i++;
                        loop();
                    });
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.rebuildRootMarkdowns = function () {
        var _this = this;
        logger.info('Regenerating README.md, CHANGELOG.md, CONTRIBUTING.md, LICENSE.md, TODO.md pages');
        var actions = [];
        Configuration$1.resetRootMarkdownPages();
        actions.push(function () {
            return _this.processMarkdowns();
        });
        promiseSequential(actions)
            .then(function (res) {
            _this.processPages();
            _this.clearUpdatedFiles();
        })["catch"](function (errorMessage) {
            logger.error(errorMessage);
        });
    };
    /**
     * Get dependency data for small group of updated files during watch process
     */
    Application.prototype.getMicroDependenciesData = function () {
        logger.info('Get diff dependencies data');
        var dependenciesClass = AngularDependencies;
        Configuration$1.mainData.angularProject = true;
        if (this.detectAngularJSProjects()) {
            logger.info('AngularJS project detected');
            Configuration$1.mainData.angularProject = false;
            Configuration$1.mainData.angularJSProject = true;
            dependenciesClass = AngularJSDependencies;
        }
        var crawler = new dependenciesClass(this.updatedFiles, {
            tsconfigDirectory: path.dirname(Configuration$1.mainData.tsconfig)
        }, Configuration$1, RouterParserUtil$1);
        var dependenciesData = crawler.getDependencies();
        DependenciesEngine$1.update(dependenciesData);
        this.prepareJustAFewThings(dependenciesData);
    };
    /**
     * Rebuild external documentation during watch process
     */
    Application.prototype.rebuildExternalDocumentation = function () {
        var _this = this;
        logger.info('Rebuild external documentation');
        var actions = [];
        Configuration$1.resetAdditionalPages();
        if (Configuration$1.mainData.includes !== '') {
            actions.push(function () {
                return _this.prepareExternalIncludes();
            });
        }
        promiseSequential(actions)
            .then(function (res) {
            _this.processPages();
            _this.clearUpdatedFiles();
        })["catch"](function (errorMessage) {
            logger.error(errorMessage);
        });
    };
    Application.prototype.detectAngularJSProjects = function () {
        if (typeof this.packageJsonData.dependencies !== 'undefined') {
            if (typeof this.packageJsonData.dependencies.angular !== 'undefined') ;
            else {
                var countJSFiles_1 = 0;
                this.files.forEach(function (file) {
                    if (path.extname(file) === '.js') {
                        countJSFiles_1 += 1;
                    }
                });
                var percentOfJSFiles = (countJSFiles_1 * 100) / this.files.length;
            }
        }
        return false;
    };
    Application.prototype.getDependenciesData = function () {
        logger.info('Get dependencies data');
        /**
         * AngularJS detection strategy :
         * - if in package.json
         * - if 75% of scanned files are *.js files
         */
        var dependenciesClass = AngularDependencies;
        Configuration$1.mainData.angularProject = true;
        if (this.detectAngularJSProjects()) {
            logger.info('AngularJS project detected');
            Configuration$1.mainData.angularProject = false;
            Configuration$1.mainData.angularJSProject = true;
            dependenciesClass = AngularJSDependencies;
        }
        var crawler = new dependenciesClass(this.files, {
            tsconfigDirectory: path.dirname(Configuration$1.mainData.tsconfig)
        }, Configuration$1, RouterParserUtil$1);
        var dependenciesData = crawler.getDependencies();
        DependenciesEngine$1.init(dependenciesData);
        Configuration$1.mainData.routesLength = RouterParserUtil$1.routesLength();
        this.printStatistics();
        this.prepareEverything();
    };
    Application.prototype.prepareJustAFewThings = function (diffCrawledData) {
        var _this = this;
        var actions = [];
        Configuration$1.resetPages();
        if (!Configuration$1.mainData.disableRoutesGraph) {
            actions.push(function () { return _this.prepareRoutes(); });
        }
        if (diffCrawledData.components.length > 0) {
            actions.push(function () { return _this.prepareComponents(); });
        }
        if (diffCrawledData.controllers.length > 0) {
            actions.push(function () { return _this.prepareControllers(); });
        }
        if (diffCrawledData.modules.length > 0) {
            actions.push(function () { return _this.prepareModules(); });
        }
        if (diffCrawledData.directives.length > 0) {
            actions.push(function () { return _this.prepareDirectives(); });
        }
        if (diffCrawledData.injectables.length > 0) {
            actions.push(function () { return _this.prepareInjectables(); });
        }
        if (diffCrawledData.interceptors.length > 0) {
            actions.push(function () { return _this.prepareInterceptors(); });
        }
        if (diffCrawledData.guards.length > 0) {
            actions.push(function () { return _this.prepareGuards(); });
        }
        if (diffCrawledData.pipes.length > 0) {
            actions.push(function () { return _this.preparePipes(); });
        }
        if (diffCrawledData.classes.length > 0) {
            actions.push(function () { return _this.prepareClasses(); });
        }
        if (diffCrawledData.interfaces.length > 0) {
            actions.push(function () { return _this.prepareInterfaces(); });
        }
        if (diffCrawledData.miscellaneous.variables.length > 0 ||
            diffCrawledData.miscellaneous.functions.length > 0 ||
            diffCrawledData.miscellaneous.typealiases.length > 0 ||
            diffCrawledData.miscellaneous.enumerations.length > 0) {
            actions.push(function () { return _this.prepareMiscellaneous(); });
        }
        if (!Configuration$1.mainData.disableCoverage) {
            actions.push(function () { return _this.prepareCoverage(); });
        }
        promiseSequential(actions)
            .then(function (res) {
            _this.processGraphs();
            _this.clearUpdatedFiles();
        })["catch"](function (errorMessage) {
            logger.error(errorMessage);
        });
    };
    Application.prototype.printStatistics = function () {
        logger.info('-------------------');
        logger.info('Project statistics ');
        if (DependenciesEngine$1.modules.length > 0) {
            logger.info("- files      : " + this.files.length);
        }
        if (DependenciesEngine$1.modules.length > 0) {
            logger.info("- module     : " + DependenciesEngine$1.modules.length);
        }
        if (DependenciesEngine$1.components.length > 0) {
            logger.info("- component  : " + DependenciesEngine$1.components.length);
        }
        if (DependenciesEngine$1.controllers.length > 0) {
            logger.info("- controller : " + DependenciesEngine$1.controllers.length);
        }
        if (DependenciesEngine$1.directives.length > 0) {
            logger.info("- directive  : " + DependenciesEngine$1.directives.length);
        }
        if (DependenciesEngine$1.injectables.length > 0) {
            logger.info("- injectable : " + DependenciesEngine$1.injectables.length);
        }
        if (DependenciesEngine$1.interceptors.length > 0) {
            logger.info("- injector   : " + DependenciesEngine$1.interceptors.length);
        }
        if (DependenciesEngine$1.guards.length > 0) {
            logger.info("- guard      : " + DependenciesEngine$1.guards.length);
        }
        if (DependenciesEngine$1.pipes.length > 0) {
            logger.info("- pipe       : " + DependenciesEngine$1.pipes.length);
        }
        if (DependenciesEngine$1.classes.length > 0) {
            logger.info("- class      : " + DependenciesEngine$1.classes.length);
        }
        if (DependenciesEngine$1.interfaces.length > 0) {
            logger.info("- interface  : " + DependenciesEngine$1.interfaces.length);
        }
        if (Configuration$1.mainData.routesLength > 0) {
            logger.info("- route      : " + Configuration$1.mainData.routesLength);
        }
        logger.info('-------------------');
    };
    Application.prototype.prepareEverything = function () {
        var _this = this;
        var actions = [];
        actions.push(function () {
            return _this.prepareComponents();
        });
        actions.push(function () {
            return _this.prepareModules();
        });
        if (DependenciesEngine$1.directives.length > 0) {
            actions.push(function () {
                return _this.prepareDirectives();
            });
        }
        if (DependenciesEngine$1.controllers.length > 0) {
            actions.push(function () {
                return _this.prepareControllers();
            });
        }
        if (DependenciesEngine$1.injectables.length > 0) {
            actions.push(function () {
                return _this.prepareInjectables();
            });
        }
        if (DependenciesEngine$1.interceptors.length > 0) {
            actions.push(function () {
                return _this.prepareInterceptors();
            });
        }
        if (DependenciesEngine$1.guards.length > 0) {
            actions.push(function () {
                return _this.prepareGuards();
            });
        }
        if (DependenciesEngine$1.routes &&
            DependenciesEngine$1.routes.children.length > 0 &&
            !Configuration$1.mainData.disableRoutesGraph) {
            actions.push(function () {
                return _this.prepareRoutes();
            });
        }
        if (DependenciesEngine$1.pipes.length > 0) {
            actions.push(function () {
                return _this.preparePipes();
            });
        }
        if (DependenciesEngine$1.classes.length > 0) {
            actions.push(function () {
                return _this.prepareClasses();
            });
        }
        if (DependenciesEngine$1.interfaces.length > 0) {
            actions.push(function () {
                return _this.prepareInterfaces();
            });
        }
        if (DependenciesEngine$1.miscellaneous.variables.length > 0 ||
            DependenciesEngine$1.miscellaneous.functions.length > 0 ||
            DependenciesEngine$1.miscellaneous.typealiases.length > 0 ||
            DependenciesEngine$1.miscellaneous.enumerations.length > 0) {
            actions.push(function () {
                return _this.prepareMiscellaneous();
            });
        }
        if (!Configuration$1.mainData.disableCoverage) {
            actions.push(function () {
                return _this.prepareCoverage();
            });
        }
        if (Configuration$1.mainData.unitTestCoverage !== '') {
            actions.push(function () {
                return _this.prepareUnitTestCoverage();
            });
        }
        if (Configuration$1.mainData.includes !== '') {
            actions.push(function () {
                return _this.prepareExternalIncludes();
            });
        }
        promiseSequential(actions)
            .then(function (res) {
            if (Configuration$1.mainData.exportFormat !== COMPODOC_DEFAULTS.exportFormat) {
                if (COMPODOC_DEFAULTS.exportFormatsSupported.indexOf(Configuration$1.mainData.exportFormat) > -1) {
                    logger.info("Generating documentation in export format " + Configuration$1.mainData.exportFormat);
                    ExportEngine$1["export"](Configuration$1.mainData.output, Configuration$1.mainData).then(function () {
                        generationPromiseResolve();
                        _this.endCallback();
                        logger.info('Documentation generated in ' +
                            Configuration$1.mainData.output +
                            ' in ' +
                            _this.getElapsedTime() +
                            ' seconds');
                    });
                }
                else {
                    logger.warn("Exported format not supported");
                }
            }
            else {
                _this.processGraphs();
            }
        })["catch"](function (errorMessage) {
            logger.error(errorMessage);
        });
    };
    Application.prototype.getIncludedPathForFile = function (file) {
        return path.join(Configuration$1.mainData.includes, file);
    };
    Application.prototype.prepareExternalIncludes = function () {
        var _this = this;
        logger.info('Adding external markdown files');
        // Scan include folder for files detailed in summary.json
        // For each file, add to Configuration.mainData.additionalPages
        // Each file will be converted to html page, inside COMPODOC_DEFAULTS.additionalEntryPath
        return new Promise(function (resolve, reject) {
            FileEngine$1.get(_this.getIncludedPathForFile('summary.json')).then(function (summaryData) {
                logger.info('Additional documentation: summary.json file found');
                var parsedSummaryData = JSON.parse(summaryData);
                var that = _this;
                var lastLevelOnePage = undefined;
                traverse$3(parsedSummaryData).forEach(function () {
                    // tslint:disable-next-line:no-invalid-this
                    if (this.notRoot && typeof this.node === 'object') {
                        // tslint:disable-next-line:no-invalid-this
                        var rawPath = this.path;
                        // tslint:disable-next-line:no-invalid-this
                        var additionalNode = this.node;
                        var file = additionalNode.file;
                        var title = additionalNode.title;
                        var finalPath_1 = Configuration$1.mainData.includesFolder;
                        var finalDepth = rawPath.filter(function (el) {
                            return !isNaN(parseInt(el, 10));
                        });
                        if (typeof file !== 'undefined' && typeof title !== 'undefined') {
                            var url = cleanNameWithoutSpaceAndToLowerCase(title);
                            /**
                             * Id created with title + file path hash, seems to be hypothetically unique here
                             */
                            var id = crypto$6
                                .createHash('md5')
                                .update(title + file)
                                .digest('hex');
                            // tslint:disable-next-line:no-invalid-this
                            this.node.id = id;
                            var lastElementRootTree_1 = undefined;
                            finalDepth.forEach(function (el) {
                                var elementTree = typeof lastElementRootTree_1 === 'undefined'
                                    ? parsedSummaryData
                                    : lastElementRootTree_1;
                                if (typeof elementTree.children !== 'undefined') {
                                    elementTree = elementTree.children[el];
                                }
                                else {
                                    elementTree = elementTree[el];
                                }
                                finalPath_1 +=
                                    '/' +
                                        cleanNameWithoutSpaceAndToLowerCase(elementTree.title);
                                lastElementRootTree_1 = elementTree;
                            });
                            finalPath_1 = finalPath_1.replace('/' + url, '');
                            var markdownFile = MarkdownEngine$1.getTraditionalMarkdownSync(that.getIncludedPathForFile(file));
                            if (finalDepth.length > 5) {
                                logger.error('Only 5 levels of depth are supported');
                            }
                            else {
                                var _page = {
                                    name: title,
                                    id: id,
                                    filename: url,
                                    context: 'additional-page',
                                    path: finalPath_1,
                                    additionalPage: markdownFile,
                                    depth: finalDepth.length,
                                    childrenLength: additionalNode.children
                                        ? additionalNode.children.length
                                        : 0,
                                    children: [],
                                    lastChild: false,
                                    pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                                };
                                if (finalDepth.length === 1) {
                                    lastLevelOnePage = _page;
                                }
                                if (finalDepth.length > 1) {
                                    // store all child pages of the last root level 1 page inside it
                                    lastLevelOnePage.children.push(_page);
                                }
                                else {
                                    Configuration$1.addAdditionalPage(_page);
                                }
                            }
                        }
                    }
                });
                resolve();
            }, function (errorMessage) {
                logger.error(errorMessage);
                reject('Error during Additional documentation generation');
            });
        });
    };
    Application.prototype.prepareModules = function (someModules) {
        var _this = this;
        logger.info('Prepare modules');
        var i = 0;
        var _modules = someModules ? someModules : DependenciesEngine$1.getModules();
        return new Promise(function (resolve, reject) {
            Configuration$1.mainData.modules = _modules.map(function (ngModule) {
                ngModule.compodocLinks = {
                    components: [],
                    controllers: [],
                    directives: [],
                    injectables: [],
                    pipes: []
                };
                ['declarations', 'bootstrap', 'imports', 'exports', 'controllers'].forEach(function (metadataType) {
                    ngModule[metadataType] = ngModule[metadataType].filter(function (metaDataItem) {
                        switch (metaDataItem.type) {
                            case 'directive':
                                return DependenciesEngine$1.getDirectives().some(function (directive) {
                                    var selectedDirective;
                                    if (typeof metaDataItem.id !== 'undefined') {
                                        selectedDirective =
                                            directive.id === metaDataItem.id;
                                    }
                                    else {
                                        selectedDirective =
                                            directive.name === metaDataItem.name;
                                    }
                                    if (selectedDirective &&
                                        !ngModule.compodocLinks.directives.includes(directive)) {
                                        ngModule.compodocLinks.directives.push(directive);
                                    }
                                    return selectedDirective;
                                });
                            case 'component':
                                return DependenciesEngine$1.getComponents().some(function (component) {
                                    var selectedComponent;
                                    if (typeof metaDataItem.id !== 'undefined') {
                                        selectedComponent =
                                            component.id === metaDataItem.id;
                                    }
                                    else {
                                        selectedComponent =
                                            component.name === metaDataItem.name;
                                    }
                                    if (selectedComponent &&
                                        !ngModule.compodocLinks.components.includes(component)) {
                                        ngModule.compodocLinks.components.push(component);
                                    }
                                    return selectedComponent;
                                });
                            case 'controller':
                                return DependenciesEngine$1.getControllers().some(function (controller) {
                                    var selectedController;
                                    if (typeof metaDataItem.id !== 'undefined') {
                                        selectedController =
                                            controller.id === metaDataItem.id;
                                    }
                                    else {
                                        selectedController =
                                            controller.name === metaDataItem.name;
                                    }
                                    if (selectedController &&
                                        !ngModule.compodocLinks.controllers.includes(controller)) {
                                        ngModule.compodocLinks.controllers.push(controller);
                                    }
                                    return selectedController;
                                });
                            case 'module':
                                return DependenciesEngine$1.getModules().some(function (module) { return module.name === metaDataItem.name; });
                            case 'pipe':
                                return DependenciesEngine$1.getPipes().some(function (pipe) {
                                    var selectedPipe;
                                    if (typeof metaDataItem.id !== 'undefined') {
                                        selectedPipe = pipe.id === metaDataItem.id;
                                    }
                                    else {
                                        selectedPipe = pipe.name === metaDataItem.name;
                                    }
                                    if (selectedPipe &&
                                        !ngModule.compodocLinks.pipes.includes(pipe)) {
                                        ngModule.compodocLinks.pipes.push(pipe);
                                    }
                                    return selectedPipe;
                                });
                            default:
                                return true;
                        }
                    });
                });
                ngModule.providers = ngModule.providers.filter(function (provider) {
                    return (DependenciesEngine$1.getInjectables().some(function (injectable) {
                        var selectedInjectable = injectable.name === provider.name;
                        if (selectedInjectable &&
                            !ngModule.compodocLinks.injectables.includes(injectable)) {
                            ngModule.compodocLinks.injectables.push(injectable);
                        }
                        return selectedInjectable;
                    }) ||
                        DependenciesEngine$1.getInterceptors().some(function (interceptor) { return interceptor.name === provider.name; }));
                });
                // Try fixing type undefined for each providers
                _.forEach(ngModule.providers, function (provider) {
                    if (DependenciesEngine$1.getInjectables().find(function (injectable) { return injectable.name === provider.name; })) {
                        provider.type = 'injectable';
                    }
                    if (DependenciesEngine$1.getInterceptors().find(function (interceptor) { return interceptor.name === provider.name; })) {
                        provider.type = 'interceptor';
                    }
                });
                // Order things
                ngModule.compodocLinks.components = _.sortBy(ngModule.compodocLinks.components, [
                    'name'
                ]);
                ngModule.compodocLinks.controllers = _.sortBy(ngModule.compodocLinks.controllers, [
                    'name'
                ]);
                ngModule.compodocLinks.directives = _.sortBy(ngModule.compodocLinks.directives, [
                    'name'
                ]);
                ngModule.compodocLinks.injectables = _.sortBy(ngModule.compodocLinks.injectables, [
                    'name'
                ]);
                ngModule.compodocLinks.pipes = _.sortBy(ngModule.compodocLinks.pipes, ['name']);
                ngModule.declarations = _.sortBy(ngModule.declarations, ['name']);
                ngModule.entryComponents = _.sortBy(ngModule.entryComponents, ['name']);
                ngModule.providers = _.sortBy(ngModule.providers, ['name']);
                ngModule.imports = _.sortBy(ngModule.imports, ['name']);
                ngModule.exports = _.sortBy(ngModule.exports, ['name']);
                return ngModule;
            });
            Configuration$1.addPage({
                name: 'modules',
                id: 'modules',
                context: 'modules',
                depth: 0,
                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
            });
            var len = Configuration$1.mainData.modules.length;
            var loop = function () {
                if (i < len) {
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(Configuration$1.mainData.modules[i].file)) {
                        logger.info(" " + Configuration$1.mainData.modules[i].name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(Configuration$1.mainData.modules[i].file);
                        Configuration$1.mainData.modules[i].readme = marked$3(readme);
                    }
                    Configuration$1.addPage({
                        path: 'modules',
                        name: Configuration$1.mainData.modules[i].name,
                        id: Configuration$1.mainData.modules[i].id,
                        navTabs: _this.getNavTabs(Configuration$1.mainData.modules[i]),
                        context: 'module',
                        module: Configuration$1.mainData.modules[i],
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    });
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareInterfaces = function (someInterfaces) {
        var _this = this;
        logger.info('Prepare interfaces');
        Configuration$1.mainData.interfaces = someInterfaces
            ? someInterfaces
            : DependenciesEngine$1.getInterfaces();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.interfaces.length;
            var loop = function () {
                if (i < len) {
                    var interf = Configuration$1.mainData.interfaces[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(interf.file)) {
                        logger.info(" " + interf.name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(interf.file);
                        interf.readme = marked$3(readme);
                    }
                    var page = {
                        path: 'interfaces',
                        name: interf.name,
                        id: interf.id,
                        navTabs: _this.getNavTabs(interf),
                        context: 'interface',
                        interface: interf,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (interf.isDuplicate) {
                        page.name += '-' + interf.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareMiscellaneous = function (someMisc) {
        logger.info('Prepare miscellaneous');
        Configuration$1.mainData.miscellaneous = someMisc
            ? someMisc
            : DependenciesEngine$1.getMiscellaneous();
        return new Promise(function (resolve, reject) {
            if (Configuration$1.mainData.miscellaneous.functions.length > 0) {
                Configuration$1.addPage({
                    path: 'miscellaneous',
                    name: 'functions',
                    id: 'miscellaneous-functions',
                    context: 'miscellaneous-functions',
                    depth: 1,
                    pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                });
            }
            if (Configuration$1.mainData.miscellaneous.variables.length > 0) {
                Configuration$1.addPage({
                    path: 'miscellaneous',
                    name: 'variables',
                    id: 'miscellaneous-variables',
                    context: 'miscellaneous-variables',
                    depth: 1,
                    pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                });
            }
            if (Configuration$1.mainData.miscellaneous.typealiases.length > 0) {
                Configuration$1.addPage({
                    path: 'miscellaneous',
                    name: 'typealiases',
                    id: 'miscellaneous-typealiases',
                    context: 'miscellaneous-typealiases',
                    depth: 1,
                    pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                });
            }
            if (Configuration$1.mainData.miscellaneous.enumerations.length > 0) {
                Configuration$1.addPage({
                    path: 'miscellaneous',
                    name: 'enumerations',
                    id: 'miscellaneous-enumerations',
                    context: 'miscellaneous-enumerations',
                    depth: 1,
                    pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                });
            }
            resolve();
        });
    };
    Application.prototype.handleTemplateurl = function (component) {
        var dirname = path.dirname(component.file);
        var templatePath = path.resolve(dirname + path.sep + component.templateUrl);
        if (!FileEngine$1.existsSync(templatePath)) {
            var err = "Cannot read template for " + component.name;
            logger.error(err);
            return new Promise(function (resolve, reject) { });
        }
        return FileEngine$1.get(templatePath).then(function (data) { return (component.templateData = data); }, function (err) {
            logger.error(err);
            return Promise.reject('');
        });
    };
    Application.prototype.handleStyles = function (component) {
        var styles = component.styles;
        component.stylesData = '';
        return new Promise(function (resolveStyles, rejectStyles) {
            styles.forEach(function (style) {
                component.stylesData = component.stylesData + style + '\n';
            });
            resolveStyles();
        });
    };
    Application.prototype.handleStyleurls = function (component) {
        var dirname = path.dirname(component.file);
        var styleDataPromise = component.styleUrls.map(function (styleUrl) {
            var stylePath = path.resolve(dirname + path.sep + styleUrl);
            if (!FileEngine$1.existsSync(stylePath)) {
                var err = "Cannot read style url " + stylePath + " for " + component.name;
                logger.error(err);
                return new Promise(function (resolve, reject) { });
            }
            return new Promise(function (resolve, reject) {
                FileEngine$1.get(stylePath).then(function (data) {
                    resolve({
                        data: data,
                        styleUrl: styleUrl
                    });
                });
            });
        });
        return Promise.all(styleDataPromise).then(function (data) { return (component.styleUrlsData = data); }, function (err) {
            logger.error(err);
            return Promise.reject('');
        });
    };
    Application.prototype.getNavTabs = function (dependency) {
        var navTabConfig = Configuration$1.mainData.navTabConfig;
        navTabConfig =
            navTabConfig.length === 0
                ? _.cloneDeep(COMPODOC_CONSTANTS.navTabDefinitions)
                : navTabConfig;
        var matchDepType = function (depType) {
            return depType === 'all' || depType === dependency.type;
        };
        var navTabs = [];
        _.forEach(navTabConfig, function (customTab) {
            var navTab = _.find(COMPODOC_CONSTANTS.navTabDefinitions, { id: customTab.id });
            if (!navTab) {
                throw new Error("Invalid tab ID '" + customTab.id + "' specified in tab configuration");
            }
            navTab.label = customTab.label;
            // is tab applicable to target dependency?
            if (-1 === _.findIndex(navTab.depTypes, matchDepType)) {
                return;
            }
            // global config
            if (customTab.id === 'tree' && Configuration$1.mainData.disableDomTree) {
                return;
            }
            if (customTab.id === 'source' && Configuration$1.mainData.disableSourceCode) {
                return;
            }
            if (customTab.id === 'templateData' && Configuration$1.mainData.disableTemplateTab) {
                return;
            }
            if (customTab.id === 'styleData' && Configuration$1.mainData.disableStyleTab) {
                return;
            }
            // per dependency config
            if (customTab.id === 'readme' && !dependency.readme) {
                return;
            }
            if (customTab.id === 'example' && !dependency.exampleUrls) {
                return;
            }
            if (customTab.id === 'templateData' &&
                (!dependency.templateUrl || dependency.templateUrl.length === 0)) {
                return;
            }
            if (customTab.id === 'styleData' &&
                ((!dependency.styleUrls || dependency.styleUrls.length === 0) &&
                    (!dependency.styles || dependency.styles.length === 0))) {
                return;
            }
            navTabs.push(navTab);
        });
        if (navTabs.length === 0) {
            throw new Error("No valid navigation tabs have been defined for dependency type '" + dependency.type + "'. Specify at least one config for the 'info' or 'source' tab in --navTabConfig.");
        }
        return navTabs;
    };
    Application.prototype.prepareControllers = function (someControllers) {
        var _this = this;
        logger.info('Prepare controllers');
        Configuration$1.mainData.controllers = someControllers
            ? someControllers
            : DependenciesEngine$1.getControllers();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.controllers.length;
            var loop = function () {
                if (i < len) {
                    var controller = Configuration$1.mainData.controllers[i];
                    var page = {
                        path: 'controllers',
                        name: controller.name,
                        id: controller.id,
                        navTabs: _this.getNavTabs(controller),
                        context: 'controller',
                        controller: controller,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (controller.isDuplicate) {
                        page.name += '-' + controller.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareComponents = function (someComponents) {
        var _this = this;
        logger.info('Prepare components');
        Configuration$1.mainData.components = someComponents
            ? someComponents
            : DependenciesEngine$1.getComponents();
        return new Promise(function (mainPrepareComponentResolve, mainPrepareComponentReject) {
            var i = 0;
            var len = Configuration$1.mainData.components.length;
            var loop = function () {
                if (i <= len - 1) {
                    var component_1 = Configuration$1.mainData.components[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(component_1.file)) {
                        logger.info(" " + component_1.name + " has a README file, include it");
                        var readmeFile = MarkdownEngine$1.readNeighbourReadmeFile(component_1.file);
                        component_1.readme = marked$3(readmeFile);
                    }
                    var page = {
                        path: 'components',
                        name: component_1.name,
                        id: component_1.id,
                        navTabs: _this.getNavTabs(component_1),
                        context: 'component',
                        component: component_1,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (component_1.isDuplicate) {
                        page.name += '-' + component_1.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    var componentTemplateUrlPromise = new Promise(function (componentTemplateUrlResolve, componentTemplateUrlReject) {
                        if (component_1.templateUrl.length > 0) {
                            logger.info(" " + component_1.name + " has a templateUrl, include it");
                            _this.handleTemplateurl(component_1).then(function () {
                                componentTemplateUrlResolve();
                            }, function (e) {
                                logger.error(e);
                                componentTemplateUrlReject();
                            });
                        }
                        else {
                            componentTemplateUrlResolve();
                        }
                    });
                    var componentStyleUrlsPromise = new Promise(function (componentStyleUrlsResolve, componentStyleUrlsReject) {
                        if (component_1.styleUrls.length > 0) {
                            logger.info(" " + component_1.name + " has styleUrls, include them");
                            _this.handleStyleurls(component_1).then(function () {
                                componentStyleUrlsResolve();
                            }, function (e) {
                                logger.error(e);
                                componentStyleUrlsReject();
                            });
                        }
                        else {
                            componentStyleUrlsResolve();
                        }
                    });
                    var componentStylesPromise = new Promise(function (componentStylesResolve, componentStylesReject) {
                        if (component_1.styles.length > 0) {
                            logger.info(" " + component_1.name + " has styles, include them");
                            _this.handleStyles(component_1).then(function () {
                                componentStylesResolve();
                            }, function (e) {
                                logger.error(e);
                                componentStylesReject();
                            });
                        }
                        else {
                            componentStylesResolve();
                        }
                    });
                    Promise.all([
                        componentTemplateUrlPromise,
                        componentStyleUrlsPromise,
                        componentStylesPromise
                    ]).then(function () {
                        i++;
                        loop();
                    });
                }
                else {
                    mainPrepareComponentResolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareDirectives = function (someDirectives) {
        var _this = this;
        logger.info('Prepare directives');
        Configuration$1.mainData.directives = someDirectives
            ? someDirectives
            : DependenciesEngine$1.getDirectives();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.directives.length;
            var loop = function () {
                if (i < len) {
                    var directive = Configuration$1.mainData.directives[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(directive.file)) {
                        logger.info(" " + directive.name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(directive.file);
                        directive.readme = marked$3(readme);
                    }
                    var page = {
                        path: 'directives',
                        name: directive.name,
                        id: directive.id,
                        navTabs: _this.getNavTabs(directive),
                        context: 'directive',
                        directive: directive,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (directive.isDuplicate) {
                        page.name += '-' + directive.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareInjectables = function (someInjectables) {
        var _this = this;
        logger.info('Prepare injectables');
        Configuration$1.mainData.injectables = someInjectables
            ? someInjectables
            : DependenciesEngine$1.getInjectables();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.injectables.length;
            var loop = function () {
                if (i < len) {
                    var injec = Configuration$1.mainData.injectables[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(injec.file)) {
                        logger.info(" " + injec.name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(injec.file);
                        injec.readme = marked$3(readme);
                    }
                    var page = {
                        path: 'injectables',
                        name: injec.name,
                        id: injec.id,
                        navTabs: _this.getNavTabs(injec),
                        context: 'injectable',
                        injectable: injec,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (injec.isDuplicate) {
                        page.name += '-' + injec.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareInterceptors = function (someInterceptors) {
        var _this = this;
        logger.info('Prepare interceptors');
        Configuration$1.mainData.interceptors = someInterceptors
            ? someInterceptors
            : DependenciesEngine$1.getInterceptors();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.interceptors.length;
            var loop = function () {
                if (i < len) {
                    var interceptor = Configuration$1.mainData.interceptors[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(interceptor.file)) {
                        logger.info(" " + interceptor.name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(interceptor.file);
                        interceptor.readme = marked$3(readme);
                    }
                    var page = {
                        path: 'interceptors',
                        name: interceptor.name,
                        id: interceptor.id,
                        navTabs: _this.getNavTabs(interceptor),
                        context: 'interceptor',
                        injectable: interceptor,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (interceptor.isDuplicate) {
                        page.name += '-' + interceptor.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareGuards = function (someGuards) {
        var _this = this;
        logger.info('Prepare guards');
        Configuration$1.mainData.guards = someGuards ? someGuards : DependenciesEngine$1.getGuards();
        return new Promise(function (resolve, reject) {
            var i = 0;
            var len = Configuration$1.mainData.guards.length;
            var loop = function () {
                if (i < len) {
                    var guard = Configuration$1.mainData.guards[i];
                    if (MarkdownEngine$1.hasNeighbourReadmeFile(guard.file)) {
                        logger.info(" " + guard.name + " has a README file, include it");
                        var readme = MarkdownEngine$1.readNeighbourReadmeFile(guard.file);
                        guard.readme = marked$3(readme);
                    }
                    var page = {
                        path: 'guards',
                        name: guard.name,
                        id: guard.id,
                        navTabs: _this.getNavTabs(guard),
                        context: 'guard',
                        injectable: guard,
                        depth: 1,
                        pageType: COMPODOC_DEFAULTS.PAGE_TYPES.INTERNAL
                    };
                    if (guard.isDuplicate) {
                        page.name += '-' + guard.duplicateId;
                    }
                    Configuration$1.addPage(page);
                    i++;
                    loop();
                }
                else {
                    resolve();
                }
            };
            loop();
        });
    };
    Application.prototype.prepareRoutes = function () {
        logger.info('Process routes');
        Configuration$1.mainData.routes = DependenciesEngine$1.getRoutes();
        return new Promise(function (resolve, reject) {
            Configuration$1.addPage({
                name: 'routes',
                id: 'routes',
                context: 'routes',
                depth: 0,
                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
            });
            if (Configuration$1.mainData.exportFormat === COMPODOC_DEFAULTS.exportFormat) {
                RouterParserUtil$1.generateRoutesIndex(Configuration$1.mainData.output, Configuration$1.mainData.routes).then(function () {
                    logger.info(' Routes index generated');
                    resolve();
                }, function (e) {
                    logger.error(e);
                    reject();
                });
            }
            else {
                resolve();
            }
        });
    };
    Application.prototype.prepareCoverage = function () {
        logger.info('Process documentation coverage report');
        return new Promise(function (resolve, reject) {
            /*
             * loop with components, directives, controllers, classes, injectables, interfaces, pipes, guards, misc functions variables
             */
            var files = [];
            var totalProjectStatementDocumented = 0;
            var getStatus = function (percent) {
                var status;
                if (percent <= 25) {
                    status = 'low';
                }
                else if (percent > 25 && percent <= 50) {
                    status = 'medium';
                }
                else if (percent > 50 && percent <= 75) {
                    status = 'good';
                }
                else {
                    status = 'very-good';
                }
                return status;
            };
            var processComponentsAndDirectivesAndControllers = function (list) {
                _.forEach(list, function (el) {
                    var element = Object.assign({}, el);
                    if (!element.propertiesClass) {
                        element.propertiesClass = [];
                    }
                    if (!element.methodsClass) {
                        element.methodsClass = [];
                    }
                    if (!element.hostBindings) {
                        element.hostBindings = [];
                    }
                    if (!element.hostListeners) {
                        element.hostListeners = [];
                    }
                    if (!element.inputsClass) {
                        element.inputsClass = [];
                    }
                    if (!element.outputsClass) {
                        element.outputsClass = [];
                    }
                    var cl = {
                        filePath: element.file,
                        type: element.type,
                        linktype: element.type,
                        name: element.name
                    };
                    var totalStatementDocumented = 0;
                    var totalStatements = element.propertiesClass.length +
                        element.methodsClass.length +
                        element.inputsClass.length +
                        element.hostBindings.length +
                        element.hostListeners.length +
                        element.outputsClass.length +
                        1; // +1 for element decorator comment
                    if (element.constructorObj) {
                        totalStatements += 1;
                        if (element.constructorObj &&
                            element.constructorObj.description &&
                            element.constructorObj.description !== '') {
                            totalStatementDocumented += 1;
                        }
                    }
                    if (element.description && element.description !== '') {
                        totalStatementDocumented += 1;
                    }
                    _.forEach(element.propertiesClass, function (property) {
                        if (property.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (property.description &&
                            property.description !== '' &&
                            property.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.methodsClass, function (method) {
                        if (method.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (method.description &&
                            method.description !== '' &&
                            method.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.hostBindings, function (property) {
                        if (property.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (property.description &&
                            property.description !== '' &&
                            property.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.hostListeners, function (method) {
                        if (method.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (method.description &&
                            method.description !== '' &&
                            method.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.inputsClass, function (input) {
                        if (input.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (input.description &&
                            input.description !== '' &&
                            input.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.outputsClass, function (output) {
                        if (output.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (output.description &&
                            output.description !== '' &&
                            output.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    cl.coveragePercent = Math.floor((totalStatementDocumented / totalStatements) * 100);
                    if (totalStatements === 0) {
                        cl.coveragePercent = 0;
                    }
                    cl.coverageCount = totalStatementDocumented + '/' + totalStatements;
                    cl.status = getStatus(cl.coveragePercent);
                    totalProjectStatementDocumented += cl.coveragePercent;
                    files.push(cl);
                });
            };
            var processCoveragePerFile = function () {
                logger.info('Process documentation coverage per file');
                logger.info('-------------------');
                var overFiles = files.filter(function (f) {
                    var overTest = f.coveragePercent >= Configuration$1.mainData.coverageMinimumPerFile;
                    if (overTest && !Configuration$1.mainData.coverageTestShowOnlyFailed) {
                        logger.info(f.coveragePercent + " % for file " + f.filePath + " - " + f.name + " - over minimum per file");
                    }
                    return overTest;
                });
                var underFiles = files.filter(function (f) {
                    var underTest = f.coveragePercent < Configuration$1.mainData.coverageMinimumPerFile;
                    if (underTest) {
                        logger.error(f.coveragePercent + " % for file " + f.filePath + " - " + f.name + " - under minimum per file");
                    }
                    return underTest;
                });
                logger.info('-------------------');
                return {
                    overFiles: overFiles,
                    underFiles: underFiles
                };
            };
            var processFunctionsAndVariables = function (id, type) {
                _.forEach(id, function (el) {
                    var cl = {
                        filePath: el.file,
                        type: type,
                        linktype: el.type,
                        linksubtype: el.subtype,
                        name: el.name
                    };
                    if (type === 'variable') {
                        cl.linktype = 'miscellaneous';
                    }
                    var totalStatementDocumented = 0;
                    var totalStatements = 1;
                    if (el.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                        // Doesn't handle private for coverage
                        totalStatements -= 1;
                    }
                    if (el.description &&
                        el.description !== '' &&
                        el.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                        totalStatementDocumented += 1;
                    }
                    cl.coveragePercent = Math.floor((totalStatementDocumented / totalStatements) * 100);
                    cl.coverageCount = totalStatementDocumented + '/' + totalStatements;
                    cl.status = getStatus(cl.coveragePercent);
                    totalProjectStatementDocumented += cl.coveragePercent;
                    files.push(cl);
                });
            };
            var processClasses = function (list, type, linktype) {
                _.forEach(list, function (cl) {
                    var element = Object.assign({}, cl);
                    if (!element.properties) {
                        element.properties = [];
                    }
                    if (!element.methods) {
                        element.methods = [];
                    }
                    var cla = {
                        filePath: element.file,
                        type: type,
                        linktype: linktype,
                        name: element.name
                    };
                    var totalStatementDocumented = 0;
                    var totalStatements = element.properties.length + element.methods.length + 1; // +1 for element itself
                    if (element.constructorObj) {
                        totalStatements += 1;
                        if (element.constructorObj &&
                            element.constructorObj.description &&
                            element.constructorObj.description !== '') {
                            totalStatementDocumented += 1;
                        }
                    }
                    if (element.description && element.description !== '') {
                        totalStatementDocumented += 1;
                    }
                    _.forEach(element.properties, function (property) {
                        if (property.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (property.description &&
                            property.description !== '' &&
                            property.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    _.forEach(element.methods, function (method) {
                        if (method.modifierKind === Ast.SyntaxKind.PrivateKeyword) {
                            // Doesn't handle private for coverage
                            totalStatements -= 1;
                        }
                        if (method.description &&
                            method.description !== '' &&
                            method.modifierKind !== Ast.SyntaxKind.PrivateKeyword) {
                            totalStatementDocumented += 1;
                        }
                    });
                    cla.coveragePercent = Math.floor((totalStatementDocumented / totalStatements) * 100);
                    if (totalStatements === 0) {
                        cla.coveragePercent = 0;
                    }
                    cla.coverageCount = totalStatementDocumented + '/' + totalStatements;
                    cla.status = getStatus(cla.coveragePercent);
                    totalProjectStatementDocumented += cla.coveragePercent;
                    files.push(cla);
                });
            };
            processComponentsAndDirectivesAndControllers(Configuration$1.mainData.components);
            processComponentsAndDirectivesAndControllers(Configuration$1.mainData.directives);
            processComponentsAndDirectivesAndControllers(Configuration$1.mainData.controllers);
            processClasses(Configuration$1.mainData.classes, 'class', 'classe');
            processClasses(Configuration$1.mainData.injectables, 'injectable', 'injectable');
            processClasses(Configuration$1.mainData.interfaces, 'interface', 'interface');
            processClasses(Configuration$1.mainData.guards, 'guard', 'guard');
            processClasses(Configuration$1.mainData.interceptors, 'interceptor', 'interceptor');
            _.forEach(Configuration$1.mainData.pipes, function (pipe) {
                var cl = {
                    filePath: pipe.file,
                    type: pipe.type,
                    linktype: pipe.type,
                    name: pipe.name
                };
                var totalStatementDocumented = 0;
                var totalStatements = 1;
                if (pipe.description && pipe.description !== '') {
                    totalStatementDocumented += 1;
                }
                cl.coveragePercent = Math.floor((totalStatementDocumented / totalStatements) * 100);
                cl.coverageCount = totalStatementDocumented + '/' + totalStatements;
                cl.status = getStatus(cl.coveragePercent);
                totalProjectStatementDocumented += cl.coveragePercent;
                files.push(cl);
            });
            processFunctionsAndVariables(Configuration$1.mainData.miscellaneous.functions, 'function');
            processFunctionsAndVariables(Configuration$1.mainData.miscellaneous.variables, 'variable');
            files = _.sortBy(files, ['filePath']);
            var coverageData = {
                count: files.length > 0
                    ? Math.floor(totalProjectStatementDocumented / files.length)
                    : 0,
                status: '',
                files: files
            };
            coverageData.status = getStatus(coverageData.count);
            Configuration$1.addPage({
                name: 'coverage',
                id: 'coverage',
                context: 'coverage',
                files: files,
                data: coverageData,
                depth: 0,
                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
            });
            coverageData.files = files;
            Configuration$1.mainData.coverageData = coverageData;
            if (Configuration$1.mainData.exportFormat === COMPODOC_DEFAULTS.exportFormat) {
                HtmlEngine$1.generateCoverageBadge(Configuration$1.mainData.output, 'documentation', coverageData);
            }
            files = _.sortBy(files, ['coveragePercent']);
            var coverageTestPerFileResults;
            if (Configuration$1.mainData.coverageTest &&
                !Configuration$1.mainData.coverageTestPerFile) {
                // Global coverage test and not per file
                if (coverageData.count >= Configuration$1.mainData.coverageTestThreshold) {
                    logger.info("Documentation coverage (" + coverageData.count + "%) is over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)");
                    generationPromiseResolve();
                    process.exit(0);
                }
                else {
                    var message = "Documentation coverage (" + coverageData.count + "%) is not over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)";
                    generationPromiseReject();
                    if (Configuration$1.mainData.coverageTestThresholdFail) {
                        logger.error(message);
                        process.exit(1);
                    }
                    else {
                        logger.warn(message);
                        process.exit(0);
                    }
                }
            }
            else if (!Configuration$1.mainData.coverageTest &&
                Configuration$1.mainData.coverageTestPerFile) {
                coverageTestPerFileResults = processCoveragePerFile();
                // Per file coverage test and not global
                if (coverageTestPerFileResults.underFiles.length > 0) {
                    var message = "Documentation coverage per file is not over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)";
                    generationPromiseReject();
                    if (Configuration$1.mainData.coverageTestThresholdFail) {
                        logger.error(message);
                        process.exit(1);
                    }
                    else {
                        logger.warn(message);
                        process.exit(0);
                    }
                }
                else {
                    logger.info("Documentation coverage per file is over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)");
                    generationPromiseResolve();
                    process.exit(0);
                }
            }
            else if (Configuration$1.mainData.coverageTest &&
                Configuration$1.mainData.coverageTestPerFile) {
                // Per file coverage test and global
                coverageTestPerFileResults = processCoveragePerFile();
                if (coverageData.count >= Configuration$1.mainData.coverageTestThreshold &&
                    coverageTestPerFileResults.underFiles.length === 0) {
                    logger.info("Documentation coverage (" + coverageData.count + "%) is over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)");
                    logger.info("Documentation coverage per file is over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)");
                    generationPromiseResolve();
                    process.exit(0);
                }
                else if (coverageData.count >= Configuration$1.mainData.coverageTestThreshold &&
                    coverageTestPerFileResults.underFiles.length > 0) {
                    logger.info("Documentation coverage (" + coverageData.count + "%) is over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)");
                    var message = "Documentation coverage per file is not over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)";
                    generationPromiseReject();
                    if (Configuration$1.mainData.coverageTestThresholdFail) {
                        logger.error(message);
                        process.exit(1);
                    }
                    else {
                        logger.warn(message);
                        process.exit(0);
                    }
                }
                else if (coverageData.count < Configuration$1.mainData.coverageTestThreshold &&
                    coverageTestPerFileResults.underFiles.length > 0) {
                    var messageGlobal = "Documentation coverage (" + coverageData.count + "%) is not over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)", messagePerFile = "Documentation coverage per file is not over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)";
                    generationPromiseReject();
                    if (Configuration$1.mainData.coverageTestThresholdFail) {
                        logger.error(messageGlobal);
                        logger.error(messagePerFile);
                        process.exit(1);
                    }
                    else {
                        logger.warn(messageGlobal);
                        logger.warn(messagePerFile);
                        process.exit(0);
                    }
                }
                else {
                    var message = "Documentation coverage (" + coverageData.count + "%) is not over threshold (" + Configuration$1.mainData.coverageTestThreshold + "%)", messagePerFile = "Documentation coverage per file is over threshold (" + Configuration$1.mainData.coverageMinimumPerFile + "%)";
                    generationPromiseReject();
                    if (Configuration$1.mainData.coverageTestThresholdFail) {
                        logger.error(message);
                        logger.info(messagePerFile);
                        process.exit(1);
                    }
                    else {
                        logger.warn(message);
                        logger.info(messagePerFile);
                        process.exit(0);
                    }
                }
            }
            else {
                resolve();
            }
        });
    };
    Application.prototype.prepareUnitTestCoverage = function () {
        logger.info('Process unit test coverage report');
        return new Promise(function (resolve, reject) {
            var covDat, covFileNames;
            var coverageData = Configuration$1.mainData.coverageData;
            if (!coverageData.files) {
                logger.warn('Missing documentation coverage data');
            }
            else {
                covDat = {};
                covFileNames = _.map(coverageData.files, function (el) {
                    var fileName = el.filePath;
                    covDat[fileName] = {
                        type: el.type,
                        linktype: el.linktype,
                        linksubtype: el.linksubtype,
                        name: el.name
                    };
                    return fileName;
                });
            }
            // read coverage summary file and data
            var unitTestSummary = {};
            var fileDat = FileEngine$1.getSync(Configuration$1.mainData.unitTestCoverage);
            if (fileDat) {
                unitTestSummary = JSON.parse(fileDat);
            }
            else {
                return Promise.reject('Error reading unit test coverage file');
            }
            var getCovStatus = function (percent, totalLines) {
                var status;
                if (totalLines === 0) {
                    status = 'uncovered';
                }
                else if (percent <= 25) {
                    status = 'low';
                }
                else if (percent > 25 && percent <= 50) {
                    status = 'medium';
                }
                else if (percent > 50 && percent <= 75) {
                    status = 'good';
                }
                else {
                    status = 'very-good';
                }
                return status;
            };
            var getCoverageData = function (data, fileName) {
                var out = {};
                if (fileName !== 'total') {
                    if (covDat === undefined) {
                        // need a name to include in output but this isn't visible
                        out = { name: fileName, filePath: fileName };
                    }
                    else {
                        var findMatch = _.filter(covFileNames, function (el) {
                            return el.includes(fileName) || fileName.includes(el);
                        });
                        if (findMatch.length > 0) {
                            out = _.clone(covDat[findMatch[0]]);
                            out['filePath'] = fileName;
                        }
                    }
                }
                var keysToGet = ['statements', 'branches', 'functions', 'lines'];
                _.forEach(keysToGet, function (key) {
                    if (data[key]) {
                        var t = data[key];
                        out[key] = {
                            coveragePercent: Math.round(t.pct),
                            coverageCount: '' + t.covered + '/' + t.total,
                            status: getCovStatus(t.pct, t.total)
                        };
                    }
                });
                return out;
            };
            var unitTestData = {};
            var files = [];
            for (var file in unitTestSummary) {
                var dat = getCoverageData(unitTestSummary[file], file);
                if (file === 'total') {
                    unitTestData['total'] = dat;
                }
                else {
                    files.push(dat);
                }
            }
            unitTestData['files'] = files;
            unitTestData['idColumn'] = covDat !== undefined; // should we include the id column
            Configuration$1.mainData.unitTestData = unitTestData;
            Configuration$1.addPage({
                name: 'unit-test',
                id: 'unit-test',
                context: 'unit-test',
                files: files,
                data: unitTestData,
                depth: 0,
                pageType: COMPODOC_DEFAULTS.PAGE_TYPES.ROOT
            });
            if (Configuration$1.mainData.exportFormat === COMPODOC_DEFAULTS.exportFormat) {
                var keysToGet = ['statements', 'branches', 'functions', 'lines'];
                _.forEach(keysToGet, function (key) {
                    if (unitTestData['total'][key]) {
                        HtmlEngine$1.generateCoverageBadge(Configuration$1.mainData.output, key, {
                            count: unitTestData['total'][key]['coveragePercent'],
                            status: unitTestData['total'][key]['status']
                        });
                    }
                });
            }
            resolve();
        });
    };
    Application.prototype.processPage = function (page) {
        logger.info('Process page', page.name);
        var htmlData = HtmlEngine$1.render(Configuration$1.mainData, page);
        var finalPath = Configuration$1.mainData.output;
        if (Configuration$1.mainData.output.lastIndexOf('/') === -1) {
            finalPath += '/';
        }
        if (page.path) {
            finalPath += page.path + '/';
        }
        if (page.filename) {
            finalPath += page.filename + '.html';
        }
        else {
            finalPath += page.name + '.html';
        }
        if (!Configuration$1.mainData.disableSearch) {
            SearchEngine$1.indexPage({
                infos: page,
                rawData: htmlData,
                url: finalPath
            });
        }
        FileEngine$1.writeSync(finalPath, htmlData);
        return Promise.resolve();
    };
    Application.prototype.processPages = function () {
        var _this = this;
        var pages = _.sortBy(Configuration$1.pages, ['name']);
        logger.info('Process pages');
        Promise.all(pages.map(function (page) { return _this.processPage(page); }))
            .then(function () {
            var callbacksAfterGenerateSearchIndexJson = function () {
                if (Configuration$1.mainData.additionalPages.length > 0) {
                    _this.processAdditionalPages();
                }
                else {
                    if (Configuration$1.mainData.assetsFolder !== '') {
                        _this.processAssetsFolder();
                    }
                    _this.processResources();
                }
            };
            if (!Configuration$1.mainData.disableSearch) {
                SearchEngine$1.generateSearchIndexJson(Configuration$1.mainData.output).then(function () {
                    callbacksAfterGenerateSearchIndexJson();
                }, function (e) {
                    logger.error(e);
                });
            }
            else {
                callbacksAfterGenerateSearchIndexJson();
            }
        })
            .then(function () {
            return _this.processMenu(Configuration$1.mainData);
        })["catch"](function (e) {
            logger.error(e);
        });
    };
    Application.prototype.processMenu = function (mainData) {
        logger.info('Process menu...');
        return HtmlEngine$1.renderMenu(Configuration$1.mainData.templates, mainData).then(function (htmlData) {
            var finalPath = mainData.output + "/js/menu-wc.js";
            return FileEngine$1.write(finalPath, htmlData)["catch"](function (err) {
                logger.error('Error during ' + finalPath + ' page generation');
                return Promise.reject('');
            });
        });
    };
    Application.prototype.processAdditionalPages = function () {
        var _this = this;
        logger.info('Process additional pages');
        var pages = Configuration$1.mainData.additionalPages;
        Promise.all(pages.map(function (page) {
            if (page.children.length > 0) {
                return Promise.all([
                    _this.processPage(page)
                ].concat(page.children.map(function (childPage) { return _this.processPage(childPage); })));
            }
            else {
                return _this.processPage(page);
            }
        }))
            .then(function () {
            SearchEngine$1.generateSearchIndexJson(Configuration$1.mainData.output).then(function () {
                if (Configuration$1.mainData.assetsFolder !== '') {
                    _this.processAssetsFolder();
                }
                _this.processResources();
            });
        })["catch"](function (e) {
            logger.error(e);
            return Promise.reject(e);
        });
    };
    Application.prototype.processAssetsFolder = function () {
        logger.info('Copy assets folder');
        if (!FileEngine$1.existsSync(Configuration$1.mainData.assetsFolder)) {
            logger.error("Provided assets folder " + Configuration$1.mainData.assetsFolder + " did not exist");
        }
        else {
            var finalOutput = Configuration$1.mainData.output;
            var testOutputDir = Configuration$1.mainData.output.match(cwd);
            if (testOutputDir && testOutputDir.length > 0) {
                finalOutput = Configuration$1.mainData.output.replace(cwd + path.sep, '');
            }
            var destination = path.join(finalOutput, path.basename(Configuration$1.mainData.assetsFolder));
            fs.copy(path.resolve(Configuration$1.mainData.assetsFolder), path.resolve(destination), function (err) {
                if (err) {
                    logger.error('Error during resources copy ', err);
                }
            });
        }
    };
    Application.prototype.processResources = function () {
        var _this = this;
        logger.info('Copy main resources');
        var onComplete = function () {
            logger.info('Documentation generated in ' +
                Configuration$1.mainData.output +
                ' in ' +
                _this.getElapsedTime() +
                ' seconds using ' +
                Configuration$1.mainData.theme +
                ' theme');
            if (Configuration$1.mainData.serve) {
                logger.info("Serving documentation from " + Configuration$1.mainData.output + " at http://" + Configuration$1.mainData.hostname + ":" + Configuration$1.mainData.port);
                _this.runWebServer(Configuration$1.mainData.output);
            }
            else {
                generationPromiseResolve();
                _this.endCallback();
            }
        };
        var finalOutput = Configuration$1.mainData.output;
        var testOutputDir = Configuration$1.mainData.output.match(cwd);
        if (testOutputDir && testOutputDir.length > 0) {
            finalOutput = Configuration$1.mainData.output.replace(cwd + path.sep, '');
        }
        fs.copy(path.resolve(__dirname + '/../src/resources/'), path.resolve(finalOutput), function (errorCopy) {
            if (errorCopy) {
                logger.error('Error during resources copy ', errorCopy);
            }
            else {
                var extThemePromise = new Promise(function (extThemeResolve, extThemeReject) {
                    if (Configuration$1.mainData.extTheme) {
                        fs.copy(path.resolve(cwd + path.sep + Configuration$1.mainData.extTheme), path.resolve(finalOutput + '/styles/'), function (errorCopyTheme) {
                            if (errorCopyTheme) {
                                logger.error('Error during external styling theme copy ', errorCopyTheme);
                                extThemeReject();
                            }
                            else {
                                logger.info('External styling theme copy succeeded');
                                extThemeResolve();
                            }
                        });
                    }
                    else {
                        extThemeResolve();
                    }
                });
                var customFaviconPromise = new Promise(function (customFaviconResolve, customFaviconReject) {
                    if (Configuration$1.mainData.customFavicon !== '') {
                        logger.info("Custom favicon supplied");
                        fs.copy(path.resolve(cwd + path.sep + Configuration$1.mainData.customFavicon), path.resolve(finalOutput + '/images/favicon.ico'), function (errorCopyFavicon) {
                            // tslint:disable-line
                            if (errorCopyFavicon) {
                                logger.error('Error during resources copy of favicon', errorCopyFavicon);
                                customFaviconReject();
                            }
                            else {
                                logger.info('External custom favicon copy succeeded');
                                customFaviconResolve();
                            }
                        });
                    }
                    else {
                        customFaviconResolve();
                    }
                });
                var customLogoPromise = new Promise(function (customLogoResolve, customLogoReject) {
                    if (Configuration$1.mainData.customLogo !== '') {
                        logger.info("Custom logo supplied");
                        fs.copy(path.resolve(cwd + path.sep + Configuration$1.mainData.customLogo), path.resolve(finalOutput +
                            '/images/' +
                            Configuration$1.mainData.customLogo.split('/').pop()), function (errorCopyLogo) {
                            // tslint:disable-line
                            if (errorCopyLogo) {
                                logger.error('Error during resources copy of logo', errorCopyLogo);
                                customLogoReject();
                            }
                            else {
                                logger.info('External custom logo copy succeeded');
                                customLogoResolve();
                            }
                        });
                    }
                    else {
                        customLogoResolve();
                    }
                });
                Promise.all([extThemePromise, customFaviconPromise, customLogoPromise]).then(function () {
                    onComplete();
                });
            }
        });
    };
    /**
     * Calculates the elapsed time since the program was started.
     *
     * @returns {number}
     */
    Application.prototype.getElapsedTime = function () {
        return (new Date().valueOf() - startTime.valueOf()) / 1000;
    };
    Application.prototype.processGraphs = function () {
        var _this = this;
        if (Configuration$1.mainData.disableGraph) {
            logger.info('Graph generation disabled');
            this.processPages();
        }
        else {
            logger.info('Process main graph');
            var modules_1 = Configuration$1.mainData.modules;
            var i_1 = 0;
            var len_1 = modules_1.length;
            var loop_1 = function () {
                if (i_1 <= len_1 - 1) {
                    logger.info('Process module graph ', modules_1[i_1].name);
                    var finalPath_2 = Configuration$1.mainData.output;
                    if (Configuration$1.mainData.output.lastIndexOf('/') === -1) {
                        finalPath_2 += '/';
                    }
                    finalPath_2 += 'modules/' + modules_1[i_1].name;
                    var _rawModule = DependenciesEngine$1.getRawModule(modules_1[i_1].name);
                    if (_rawModule.declarations.length > 0 ||
                        _rawModule.bootstrap.length > 0 ||
                        _rawModule.imports.length > 0 ||
                        _rawModule.exports.length > 0 ||
                        _rawModule.providers.length > 0) {
                        NgdEngine$1.renderGraph(modules_1[i_1].file, finalPath_2, 'f', modules_1[i_1].name).then(function () {
                            NgdEngine$1.readGraph(path.resolve(finalPath_2 + path.sep + 'dependencies.svg'), modules_1[i_1].name).then(function (data) {
                                modules_1[i_1].graph = data;
                                i_1++;
                                loop_1();
                            }, function (err) {
                                logger.error('Error during graph read: ', err);
                            });
                        }, function (errorMessage) {
                            logger.error(errorMessage);
                        });
                    }
                    else {
                        i_1++;
                        loop_1();
                    }
                }
                else {
                    _this.processPages();
                }
            };
            var finalMainGraphPath_1 = Configuration$1.mainData.output;
            if (finalMainGraphPath_1.lastIndexOf('/') === -1) {
                finalMainGraphPath_1 += '/';
            }
            finalMainGraphPath_1 += 'graph';
            NgdEngine$1.init(path.resolve(finalMainGraphPath_1));
            NgdEngine$1.renderGraph(Configuration$1.mainData.tsconfig, path.resolve(finalMainGraphPath_1), 'p').then(function () {
                NgdEngine$1.readGraph(path.resolve(finalMainGraphPath_1 + path.sep + 'dependencies.svg'), 'Main graph').then(function (data) {
                    Configuration$1.mainData.mainGraph = data;
                    loop_1();
                }, function (err) {
                    logger.error('Error during main graph reading : ', err);
                    Configuration$1.mainData.disableMainGraph = true;
                    loop_1();
                });
            }, function (err) {
                logger.error('Ooops error during main graph generation, moving on next part with main graph disabled : ', err);
                Configuration$1.mainData.disableMainGraph = true;
                loop_1();
            });
        }
    };
    Application.prototype.runWebServer = function (folder) {
        if (!this.isWatching) {
            var liveServerConfiguration = {
                root: folder,
                open: Configuration$1.mainData.open,
                quiet: true,
                logLevel: 0,
                wait: 1000,
                port: Configuration$1.mainData.port
            };
            if (Configuration$1.mainData.host !== '') {
                liveServerConfiguration.host = Configuration$1.mainData.host;
            }
            LiveServer.start(liveServerConfiguration);
        }
        if (Configuration$1.mainData.watch && !this.isWatching) {
            if (typeof this.files === 'undefined') {
                logger.error('No sources files available, please use -p flag');
                generationPromiseReject();
                process.exit(1);
            }
            else {
                this.runWatch();
            }
        }
        else if (Configuration$1.mainData.watch && this.isWatching) {
            var srcFolder = findMainSourceFolder(this.files);
            logger.info("Already watching sources in " + srcFolder + " folder");
        }
    };
    Application.prototype.runWatch = function () {
        var _this = this;
        var sources = [findMainSourceFolder(this.files)];
        var watcherReady = false;
        this.isWatching = true;
        logger.info("Watching sources in " + findMainSourceFolder(this.files) + " folder");
        if (MarkdownEngine$1.hasRootMarkdowns()) {
            sources = sources.concat(MarkdownEngine$1.listRootMarkdowns());
        }
        if (Configuration$1.mainData.includes !== '') {
            sources = sources.concat(Configuration$1.mainData.includes);
        }
        // Check all elements of sources list exist
        sources = cleanSourcesForWatch(sources);
        var watcher = chokidar.watch(sources, {
            awaitWriteFinish: true,
            ignoreInitial: true,
            ignored: /(spec|\.d)\.ts/
        });
        var timerAddAndRemoveRef;
        var timerChangeRef;
        var runnerAddAndRemove = function () {
            startTime = new Date();
            _this.generate();
        };
        var waiterAddAndRemove = function () {
            clearTimeout(timerAddAndRemoveRef);
            timerAddAndRemoveRef = setTimeout(runnerAddAndRemove, 1000);
        };
        var runnerChange = function () {
            startTime = new Date();
            _this.setUpdatedFiles(_this.watchChangedFiles);
            if (_this.hasWatchedFilesTSFiles()) {
                _this.getMicroDependenciesData();
            }
            else if (_this.hasWatchedFilesRootMarkdownFiles()) {
                _this.rebuildRootMarkdowns();
            }
            else {
                _this.rebuildExternalDocumentation();
            }
        };
        var waiterChange = function () {
            clearTimeout(timerChangeRef);
            timerChangeRef = setTimeout(runnerChange, 1000);
        };
        watcher.on('ready', function () {
            if (!watcherReady) {
                watcherReady = true;
                watcher
                    .on('add', function (file) {
                    logger.debug("File " + file + " has been added");
                    // Test extension, if ts
                    // rescan everything
                    if (path.extname(file) === '.ts') {
                        waiterAddAndRemove();
                    }
                })
                    .on('change', function (file) {
                    logger.debug("File " + file + " has been changed");
                    // Test extension, if ts
                    // rescan only file
                    if (path.extname(file) === '.ts' ||
                        path.extname(file) === '.md' ||
                        path.extname(file) === '.json') {
                        _this.watchChangedFiles.push(path.join(cwd + path.sep + file));
                        waiterChange();
                    }
                })
                    .on('unlink', function (file) {
                    logger.debug("File " + file + " has been removed");
                    // Test extension, if ts
                    // rescan everything
                    if (path.extname(file) === '.ts') {
                        waiterAddAndRemove();
                    }
                });
            }
        });
    };
    Object.defineProperty(Application.prototype, "application", {
        /**
         * Return the application / root component instance.
         */
        get: function () {
            return this;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(Application.prototype, "isCLI", {
        get: function () {
            return false;
        },
        enumerable: true,
        configurable: true
    });
    return Application;
}());

exports.AngularVersionUtil = AngularVersionUtil$1;
exports.Application = Application;
exports.COMPODOC_DEFAULTS = COMPODOC_DEFAULTS;
exports.Configuration = Configuration$1;
exports.FileEngine = FileEngine$1;
exports.I18nEngine = I18nEngine$1;
exports.__extends = __extends;
exports.handlePath = handlePath;
exports.logger = logger;
exports.readConfig = readConfig;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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