local api = vim.api
local ts = vim.treesitter

local new_lang_api = ts.language.register ~= nil

local filetype_to_parsername = {}

if new_lang_api then
  filetype_to_parsername = setmetatable({}, {
    __newindex = function(_, k, v)
      require("nvim-treesitter.utils").notify(
        "filetype_to_parsername is deprecated, please use 'vim.treesitter.language.register'",
        vim.log.levels.WARN
      )
      ts.language.register(v, k)
    end,
  })
end

local function register_lang(lang, ft)
  if new_lang_api then
    ts.language.register(lang, ft)
    return
  end
  filetype_to_parsername[ft] = lang
end

for ft, lang in pairs {
  javascriptreact = "javascript",
  ecma = "javascript",
  jsx = "javascript",
  PKGBUILD = "bash",
  html_tags = "html",
  ["typescript.tsx"] = "tsx",
  ["html.handlebars"] = "glimmer",
  systemverilog = "verilog",
  cls = "latex",
  sty = "latex",
  OpenFOAM = "foam",
  pandoc = "markdown",
  rmd = "markdown",
  quarto = "markdown",
  cs = "c_sharp",
  tape = "vhs",
  dosini = "ini",
  confini = "ini",
} do
  register_lang(lang, ft)
end

---@class InstallInfo
---@field url string
---@field branch string|nil
---@field revision string|nil
---@field files string[]
---@field generate_requires_npm boolean|nil
---@field requires_generate_from_grammar boolean|nil
---@field location string|nil

---@class ParserInfo
---@field install_info InstallInfo
---@field filetype string
---@field maintainers string[]
---@field experimental boolean|nil
---@field readme_name string|nil

---@type ParserInfo[]
local list = setmetatable({}, {
  __newindex = function(table, parsername, parserconfig)
    rawset(table, parsername, parserconfig)
    register_lang(parsername, parserconfig.filetype or parsername)
  end,
})

list.ada = {
  install_info = {
    url = "https://github.com/briot/tree-sitter-ada",
    branch = "master",
    files = { "src/parser.c" },
  },
  filetype = "ada",
  maintainers = { "@briot" },
}

list.agda = {
  install_info = {
    url = "https://github.com/AusCyberman/tree-sitter-agda",
    branch = "master",
    files = { "src/parser.c", "src/scanner.cc" },
    generate_requires_npm = true,
  },
  filetype = "agda",
  maintainers = { "@Decodetalkers" },
}

list.arduino = {
  install_info = {
    url = "https://github.com/ObserverOfTime/tree-sitter-arduino",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  maintainers = { "@ObserverOfTime" },
}

list.astro = {
  install_info = {
    url = "https://github.com/virchau13/tree-sitter-astro",
    branch = "master",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  maintainers = { "@virchau13" },
}

list.awk = {
  install_info = {
    url = "https://github.com/Beaglefoot/tree-sitter-awk",
    files = { "src/parser.c", "src/scanner.c" },
  },
}

list.bash = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-bash",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  filetype = "sh",
  maintainers = { "@TravonteD" },
}

list.beancount = {
  install_info = {
    url = "https://github.com/polarmutex/tree-sitter-beancount",
    files = { "src/parser.c", "src/scanner.cc" },
    branch = "master",
  },
  maintainers = { "@polarmutex" },
}

list.bibtex = {
  install_info = {
    url = "https://github.com/latex-lsp/tree-sitter-bibtex",
    files = { "src/parser.c" },
  },
  filetype = "bib",
  maintainers = { "@theHamsta", "@clason" },
}

list.blueprint = {
  install_info = {
    url = "https://gitlab.com/gabmus/tree-sitter-blueprint.git",
    files = { "src/parser.c" },
  },
  maintainers = { "@gabmus" },
  experimental = true,
}

list.c = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-c",
    files = { "src/parser.c" },
  },
  maintainers = { "@vigoux" },
}

list.c_sharp = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-c-sharp",
    files = { "src/parser.c", "src/scanner.c" },
  },
  filetype = "cs",
  maintainers = { "@Luxed" },
}

list.capnp = {
  install_info = {
    url = "https://github.com/amaanq/tree-sitter-capnp",
    files = { "src/parser.c" },
  },
  maintainers = { "@amaanq" },
}

list.chatito = {
  install_info = {
    url = "https://github.com/ObserverOfTime/tree-sitter-chatito",
    files = { "src/parser.c" },
  },
  maintainers = { "@ObserverOfTime" },
}

list.clojure = {
  install_info = {
    url = "https://github.com/sogaiu/tree-sitter-clojure",
    files = { "src/parser.c" },
  },
  maintainers = { "@sogaiu" },
}

list.cmake = {
  install_info = {
    url = "https://github.com/uyha/tree-sitter-cmake",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  maintainers = { "@uyha" },
}

list.comment = {
  install_info = {
    url = "https://github.com/stsewd/tree-sitter-comment",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@stsewd" },
}

list.commonlisp = {
  install_info = {
    url = "https://github.com/theHamsta/tree-sitter-commonlisp",
    files = { "src/parser.c" },
    generate_requires_npm = true,
  },
  filetype = "lisp",
  maintainers = { "@theHamsta" },
}

list.cooklang = {
  install_info = {
    url = "https://github.com/addcninblue/tree-sitter-cooklang",
    branch = "master",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  maintainers = { "@addcninblue" },
}

list.cpon = {
  install_info = {
    url = "https://github.com/amaanq/tree-sitter-cpon",
    files = { "src/parser.c" },
  },
  maintainers = { "@amaanq" },
}

list.cpp = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-cpp",
    files = { "src/parser.c", "src/scanner.cc" },
    generate_requires_npm = true,
  },
  maintainers = { "@theHamsta" },
}

list.css = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-css",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@TravonteD" },
}

list.cuda = {
  install_info = {
    url = "https://github.com/theHamsta/tree-sitter-cuda",
    files = { "src/parser.c", "src/scanner.cc" },
    generate_requires_npm = true,
  },
  maintainers = { "@theHamsta" },
}

list.d = {
  install_info = {
    url = "https://github.com/CyberShadow/tree-sitter-d",
    files = { "src/parser.c", "src/scanner.cc" },
    requires_generate_from_grammar = true,
  },
  maintainers = { "@nawordar" },
  -- Generating grammar takes ~60s
  experimental = true,
}

list.dart = {
  install_info = {
    url = "https://github.com/UserNobody14/tree-sitter-dart",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@akinsho" },
}

list.devicetree = {
  install_info = {
    url = "https://github.com/joelspadin/tree-sitter-devicetree",
    files = { "src/parser.c" },
    branch = "main",
    requires_generate_from_grammar = true,
  },
  filetype = "dts",
  maintainers = { "@jedrzejboczar" },
}

list.dhall = {
  install_info = {
    url = "https://github.com/jbellerb/tree-sitter-dhall",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@amaanq" },
}

list.diff = {
  install_info = {
    url = "https://github.com/the-mikedavis/tree-sitter-diff",
    branch = "main",
    files = { "src/parser.c" },
  },
  maintainers = { "@gbprod" },
  filetype = "gitdiff",
}

list.dockerfile = {
  install_info = {
    url = "https://github.com/camdencheek/tree-sitter-dockerfile",
    branch = "main",
    files = { "src/parser.c" },
  },
  maintainers = { "@camdencheek" },
}

list.dot = {
  install_info = {
    url = "https://github.com/rydesun/tree-sitter-dot",
    branch = "main",
    files = { "src/parser.c" },
  },
  maintainers = { "@rydesun" },
}

list.ebnf = {
  install_info = {
    url = "https://github.com/RubixDev/ebnf.git",
    files = { "src/parser.c" },
    location = "crates/tree-sitter-ebnf",
    branch = "main",
  },
  maintainers = { "@RubixDev" },
  experimental = true,
}

list.eex = {
  install_info = {
    url = "https://github.com/connorlay/tree-sitter-eex",
    files = { "src/parser.c" },
    branch = "main",
  },
  filetype = "eex",
  maintainers = { "@connorlay" },
}

list.elixir = {
  install_info = {
    url = "https://github.com/elixir-lang/tree-sitter-elixir",
    files = { "src/parser.c", "src/scanner.cc" },
    branch = "main",
  },
  maintainers = { "@connorlay" },
}

list.elm = {
  install_info = {
    url = "https://github.com/elm-tooling/tree-sitter-elm",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  maintainers = { "@zweimach" },
}

list.elsa = {
  install_info = {
    url = "https://github.com/glapa-grossklag/tree-sitter-elsa",
    files = { "src/parser.c" },
  },
  maintainers = { "@glapa-grossklag", "@amaanq" },
}

list.elvish = {
  install_info = {
    url = "https://github.com/ckafi/tree-sitter-elvish",
    branch = "main",
    files = { "src/parser.c" },
  },
  maintainers = { "@ckafi" },
}

list.embedded_template = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-embedded-template",
    files = { "src/parser.c" },
  },
  filetype = "eruby",
}

list.erlang = {
  install_info = {
    url = "https://github.com/WhatsApp/tree-sitter-erlang",
    files = { "src/parser.c" },
    branch = "main",
  },
  maintainers = { "@filmor" },
}

list.fennel = {
  install_info = {
    url = "https://github.com/travonted/tree-sitter-fennel",
    files = { "src/parser.c" },
  },
  maintainers = { "@TravonteD" },
}

list.fish = {
  install_info = {
    url = "https://github.com/ram02z/tree-sitter-fish",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@ram02z" },
}

list.foam = {
  install_info = {
    url = "https://github.com/FoamScience/tree-sitter-foam",
    branch = "master",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@FoamScience" },
  filetype = "foam",
  -- Queries might change over time on the grammar's side
  -- Otherwise everything runs fine
  experimental = true,
}

list.fortran = {
  install_info = {
    url = "https://github.com/stadelmanma/tree-sitter-fortran",
    files = { "src/parser.c", "src/scanner.cc" },
  },
}

list.fsh = {
  install_info = {
    url = "https://github.com/mgramigna/tree-sitter-fsh",
    files = { "src/parser.c" },
    branch = "main",
  },
  maintainers = { "@mgramigna" },
}

list.func = {
  install_info = {
    url = "https://github.com/amaanq/tree-sitter-func",
    files = { "src/parser.c" },
  },
  maintainers = { "@amaanq" },
}

list.fusion = {
  install_info = {
    url = "https://gitlab.com/jirgn/tree-sitter-fusion.git",
    files = { "src/parser.c", "src/scanner.c" },
    branch = "main",
  },
  maintainers = { "@jirgn" },
}

list.gdscript = {
  install_info = {
    url = "https://github.com/PrestonKnopp/tree-sitter-gdscript",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  readme_name = "Godot (gdscript)",
}

list.git_rebase = {
  install_info = {
    url = "https://github.com/the-mikedavis/tree-sitter-git-rebase",
    files = { "src/parser.c" },
    branch = "main",
  },
  filetype = "gitrebase",
  maintainers = { "@gbprod" },
}

list.gitattributes = {
  install_info = {
    url = "https://github.com/ObserverOfTime/tree-sitter-gitattributes",
    files = { "src/parser.c" },
  },
  maintainers = { "@ObserverOfTime" },
}

list.gitcommit = {
  install_info = {
    url = "https://github.com/gbprod/tree-sitter-gitcommit",
    files = { "src/parser.c", "src/scanner.c" },
    branch = "main",
  },
  maintainers = { "@gbprod" },
}

list.gitignore = {
  install_info = {
    url = "https://github.com/shunsambongi/tree-sitter-gitignore",
    files = { "src/parser.c" },
    branch = "main",
    requires_generate_from_grammar = true,
  },
  maintainers = { "@theHamsta" },
}

list.gleam = {
  install_info = {
    url = "https://github.com/J3RN/tree-sitter-gleam",
    files = { "src/parser.c" },
    branch = "main",
  },
  maintainers = { "@connorlay" },
}

list.glimmer = {
  install_info = {
    url = "https://github.com/alexlafroscia/tree-sitter-glimmer",
    files = { "src/parser.c", "src/scanner.c" },
    branch = "main",
  },
  readme_name = "Glimmer and Ember",
  maintainers = { "@NullVoxPopuli" },
  filetype = "handlebars",
}

list.glsl = {
  install_info = {
    url = "https://github.com/theHamsta/tree-sitter-glsl",
    files = { "src/parser.c" },
    generate_requires_npm = true,
  },
  maintainers = { "@theHamsta" },
}

list.go = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-go",
    files = { "src/parser.c" },
  },
  maintainers = { "@theHamsta", "@WinWisely268" },
}

list.godot_resource = {
  install_info = {
    url = "https://github.com/PrestonKnopp/tree-sitter-godot-resource",
    files = { "src/parser.c", "src/scanner.c" },
    requires_generate_from_grammar = true,
  },
  filetype = "gdresource",
  readme_name = "Godot Resources (gdresource)",
  maintainers = { "@pierpo" },
}

list.gomod = {
  install_info = {
    url = "https://github.com/camdencheek/tree-sitter-go-mod",
    branch = "main",
    files = { "src/parser.c" },
  },
  maintainers = { "@camdencheek" },
  filetype = "gomod",
}

list.gosum = {
  install_info = {
    url = "https://github.com/amaanq/tree-sitter-go-sum",
    files = { "src/parser.c" },
  },
  maintainers = { "@amaanq" },
  filetype = "gosum",
}

list.gowork = {
  install_info = {
    url = "https://github.com/omertuc/tree-sitter-go-work",
    branch = "main",
    files = { "src/parser.c" },
  },
  maintainers = { "@omertuc" },
  filetype = "gowork",
}

list.graphql = {
  install_info = {
    url = "https://github.com/bkegley/tree-sitter-graphql",
    files = { "src/parser.c" },
  },
  maintainers = { "@bkegley" },
}

list.hack = {
  install_info = {
    url = "https://github.com/slackhq/tree-sitter-hack",
    branch = "main",
    files = { "src/parser.c", "src/scanner.cc" },
  },
}

list.haskell = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-haskell",
    files = { "src/parser.c", "src/scanner.c" },
  },
}

list.hcl = {
  install_info = {
    url = "https://github.com/MichaHoffmann/tree-sitter-hcl",
    files = { "src/parser.c", "src/scanner.cc" },
    branch = "main",
  },
  maintainers = { "@MichaHoffmann" },
  filetype = "hcl",
}

list.heex = {
  install_info = {
    url = "https://github.com/connorlay/tree-sitter-heex",
    files = { "src/parser.c" },
    branch = "main",
  },
  filetype = "heex",
  maintainers = { "@connorlay" },
}

list.help = {
  install_info = {
    url = "https://github.com/neovim/tree-sitter-vimdoc",
    files = { "src/parser.c" },
  },
  filetype = "help",
  maintainers = { "@vigoux" },
}

list.hjson = {
  install_info = {
    url = "https://github.com/winston0410/tree-sitter-hjson",
    files = { "src/parser.c" },
    generate_requires_npm = true,
  },
  maintainers = { "@winston0410" },
}

list.hlsl = {
  install_info = {
    url = "https://github.com/theHamsta/tree-sitter-hlsl",
    files = { "src/parser.c", "src/scanner.cc" },
    generate_requires_npm = true,
  },
  maintainers = { "@theHamsta" },
}

list.hocon = {
  install_info = {
    url = "https://github.com/antosha417/tree-sitter-hocon",
    files = { "src/parser.c" },
    generate_requires_npm = true,
  },
  maintainers = { "@antosha417" },
}

list.html = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-html",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  maintainers = { "@TravonteD" },
}

list.htmldjango = {
  install_info = {
    url = "https://github.com/interdependence/tree-sitter-htmldjango",
    files = { "src/parser.c" },
    branch = "main",
  },
  maintainers = { "@ObserverOfTime" },
  experimental = true,
}

list.http = {
  install_info = {
    url = "https://github.com/rest-nvim/tree-sitter-http",
    branch = "main",
    files = { "src/parser.c" },
    generate_requires_npm = true,
  },
  maintainers = { "@NTBBloodbath" },
}

list.ini = {
  install_info = {
    url = "https://github.com/justinmk/tree-sitter-ini",
    files = { "src/parser.c" },
  },
  maintainers = { "@theHamsta" },
  experimental = true,
}

list.java = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-java",
    files = { "src/parser.c" },
  },
  maintainers = { "@p00f" },
}

list.javascript = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-javascript",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@steelsojka" },
}

list.jq = {
  install_info = {
    url = "https://github.com/flurie/tree-sitter-jq",
    files = { "src/parser.c" },
  },
  maintainers = { "@ObserverOfTime" },
}

list.jsdoc = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-jsdoc",
    files = { "src/parser.c" },
  },
  maintainers = { "@steelsojka" },
}

list.json = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-json",
    files = { "src/parser.c" },
  },
  maintainers = { "@steelsojka" },
}

list.json5 = {
  install_info = {
    url = "https://github.com/Joakker/tree-sitter-json5",
    files = { "src/parser.c" },
  },
  filetype = "json5",
  maintainers = { "@Joakker" },
}

list.jsonc = {
  install_info = {
    url = "https://gitlab.com/WhyNotHugo/tree-sitter-jsonc.git",
    files = { "src/parser.c" },
    generate_requires_npm = true,
  },
  readme_name = "JSON with comments",
  maintainers = { "@WhyNotHugo" },
}

list.jsonnet = {
  install_info = {
    url = "https://github.com/sourcegraph/tree-sitter-jsonnet",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@nawordar" },
}

list.julia = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-julia",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@theHamsta" },
}

list.kdl = {
  install_info = {
    url = "https://github.com/amaanq/tree-sitter-kdl",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@amaanq" },
}

list.kotlin = {
  install_info = {
    url = "https://github.com/fwcd/tree-sitter-kotlin",
    branch = "main",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@SalBakraa" },
}

list.lalrpop = {
  install_info = {
    url = "https://github.com/traxys/tree-sitter-lalrpop",
    branch = "master",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@traxys" },
}

list.latex = {
  install_info = {
    url = "https://github.com/latex-lsp/tree-sitter-latex",
    files = { "src/parser.c", "src/scanner.c" },
  },
  filetype = "tex",
  maintainers = { "@theHamsta", "@clason" },
}

list.ledger = {
  install_info = {
    url = "https://github.com/cbarrete/tree-sitter-ledger",
    files = { "src/parser.c" },
  },
  maintainers = { "@cbarrete" },
}

list.llvm = {
  install_info = {
    url = "https://github.com/benwilliamgraham/tree-sitter-llvm",
    branch = "main",
    files = { "src/parser.c" },
  },
  maintainers = { "@benwilliamgraham" },
}

list.lua = {
  install_info = {
    url = "https://github.com/MunifTanjim/tree-sitter-lua",
    branch = "main",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@muniftanjim" },
}

list.m68k = {
  install_info = {
    url = "https://github.com/grahambates/tree-sitter-m68k",
    files = { "src/parser.c" },
  },
  maintainers = { "@grahambates" },
  filetype = "asm68k",
}

list.make = {
  install_info = {
    url = "https://github.com/alemuller/tree-sitter-make",
    branch = "main",
    files = { "src/parser.c" },
  },
  maintainers = { "@lewis6991" },
}

list.markdown = {
  install_info = {
    url = "https://github.com/MDeiml/tree-sitter-markdown",
    location = "tree-sitter-markdown",
    files = { "src/parser.c", "src/scanner.cc" },
    branch = "split_parser",
    readme_name = "markdown (basic highlighting)",
  },
  maintainers = { "@MDeiml" },
  experimental = true,
}

list.markdown_inline = {
  install_info = {
    url = "https://github.com/MDeiml/tree-sitter-markdown",
    location = "tree-sitter-markdown-inline",
    files = { "src/parser.c", "src/scanner.cc" },
    branch = "split_parser",
    readme_name = "markdown_inline (needs to be installed for full markdown highlighting)",
  },
  maintainers = { "@MDeiml" },
  experimental = true,
}

list.menhir = {
  install_info = {
    url = "https://github.com/Kerl13/tree-sitter-menhir",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  maintainers = { "@Kerl13" },
  filetype = "menhir",
}

list.mermaid = {
  install_info = {
    url = "https://github.com/monaqa/tree-sitter-mermaid",
    files = { "src/parser.c" },
  },
  experimental = true,
}

list.meson = {
  install_info = {
    url = "https://github.com/Decodetalkers/tree-sitter-meson",
    branch = "master",
    files = { "src/parser.c" },
  },
  filetype = "meson",
  maintainers = { "@Decodetalkers" },
}

list.nickel = {
  install_info = {
    url = "https://github.com/nickel-lang/tree-sitter-nickel",
    files = { "src/parser.c", "src/scanner.cc" },
    branch = "main",
  },
}

list.ninja = {
  install_info = {
    url = "https://github.com/alemuller/tree-sitter-ninja",
    files = { "src/parser.c" },
    branch = "main",
  },
  maintainers = { "@alemuller" },
}

list.nix = {
  install_info = {
    url = "https://github.com/cstrahan/tree-sitter-nix",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@leo60228" },
}

list.norg = {
  install_info = {
    url = "https://github.com/nvim-neorg/tree-sitter-norg",
    branch = "main",
    files = { "src/parser.c", "src/scanner.cc" },
    use_makefile = true,
    cxx_standard = "c++14",
  },
  maintainers = { "@JoeyGrajciar", "@vhyrro" },
}

list.ocaml = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-ocaml",
    files = { "src/parser.c", "src/scanner.cc" },
    location = "ocaml",
  },
  maintainers = { "@undu" },
}

list.ocaml_interface = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-ocaml",
    files = { "src/parser.c", "src/scanner.cc" },
    location = "interface",
  },
  maintainers = { "@undu" },
  filetype = "ocamlinterface",
}

list.ocamllex = {
  install_info = {
    url = "https://github.com/atom-ocaml/tree-sitter-ocamllex",
    files = { "src/parser.c", "src/scanner.cc" },
    requires_generate_from_grammar = true,
  },
  maintainers = { "@undu" },
}

list.org = {
  install_info = {
    url = "https://github.com/milisims/tree-sitter-org",
    branch = "main",
    files = { "src/parser.c", "src/scanner.cc" },
  },
}

list.pascal = {
  install_info = {
    url = "https://github.com/Isopod/tree-sitter-pascal.git",
    files = { "src/parser.c" },
  },
  maintainers = { "@Isopod" },
}

list.perl = {
  install_info = {
    url = "https://github.com/ganezdragon/tree-sitter-perl",
    files = { "src/parser.c", "src/scanner.cc" },
    generate_requires_npm = true,
  },
  maintainers = { "@lcrownover" },
}

list.php = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-php",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  maintainers = { "@tk-shirasaka" },
}

-- Parsers for injections
list.phpdoc = {
  install_info = {
    url = "https://github.com/claytonrcarter/tree-sitter-phpdoc",
    files = { "src/parser.c", "src/scanner.c" },
    generate_requires_npm = true,
  },
  maintainers = { "@mikehaertl" },
  experimental = true,
}

list.pioasm = {
  install_info = {
    url = "https://github.com/leo60228/tree-sitter-pioasm",
    branch = "main",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@leo60228" },
}

list.poe_filter = {
  install_info = {
    url = "https://github.com/ObserverOfTime/tree-sitter-poe-filter",
    files = { "src/parser.c" },
  },
  maintainers = { "@ObserverOfTime" },
  experimental = true,
  filetype = "poefilter",
  readme_name = "Path of Exile item filter",
}

list.prisma = {
  install_info = {
    url = "https://github.com/victorhqc/tree-sitter-prisma",
    branch = "master",
    files = { "src/parser.c" },
  },
  maintainers = { "@elianiva" },
}

list.proto = {
  install_info = {
    url = "https://github.com/mitchellh/tree-sitter-proto",
    branch = "main",
    files = { "src/parser.c" },
  },
  maintainers = { "@fsouza" },
  filetype = "proto",
}

list.pug = {
  install_info = {
    url = "https://github.com/zealot128/tree-sitter-pug",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  maintainers = { "@zealot128" },
  filetype = "pug",
  experimental = true,
}

list.python = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-python",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  maintainers = { "@stsewd", "@theHamsta" },
}

list.ql = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-ql",
    files = { "src/parser.c" },
  },
  maintainers = { "@pwntester" },
}

list.qmljs = {
  install_info = {
    url = "https://github.com/yuja/tree-sitter-qmljs",
    branch = "master",
    files = { "src/parser.c", "src/scanner.c" },
  },
  filetype = "qml",
  maintainers = { "@Decodetalkers" },
}

list.query = {
  install_info = {
    url = "https://github.com/nvim-treesitter/tree-sitter-query",
    files = { "src/parser.c" },
  },
  readme_name = "Tree-sitter query language",
  maintainers = { "@steelsojka" },
}

list.r = {
  install_info = {
    url = "https://github.com/r-lib/tree-sitter-r",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  maintainers = { "@echasnovski" },
}

list.racket = {
  install_info = {
    url = "https://github.com/6cdh/tree-sitter-racket",
    branch = "main",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  maintainers = { "@6cdh" },
}

list.rasi = {
  install_info = {
    url = "https://github.com/Fymyte/tree-sitter-rasi",
    branch = "main",
    files = { "src/parser.c" },
  },
  maintainers = { "@Fymyte" },
}

list.regex = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-regex",
    files = { "src/parser.c" },
  },
  maintainers = { "@theHamsta" },
}

list.rego = {
  install_info = {
    url = "https://github.com/FallenAngel97/tree-sitter-rego",
    files = { "src/parser.c" },
  },
  maintainers = { "@FallenAngel97" },
  filetype = "rego",
}

list.rnoweb = {
  install_info = {
    url = "https://github.com/bamonroe/tree-sitter-rnoweb",
    files = { "src/parser.c", "src/scanner.c" },
  },
  filetype = "rnoweb",
  maintainers = { "@bamonroe" },
}

list.ron = {
  install_info = {
    url = "https://github.com/amaanq/tree-sitter-ron",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@amaanq" },
}

list.rst = {
  install_info = {
    url = "https://github.com/stsewd/tree-sitter-rst",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@stsewd" },
}

list.ruby = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-ruby",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  maintainers = { "@TravonteD" },
}

list.rust = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-rust",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@vigoux" },
}

list.scala = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-scala",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@stevanmilic" },
}

list.scheme = {
  install_info = {
    url = "https://github.com/6cdh/tree-sitter-scheme",
    branch = "main",
    files = { "src/parser.c" },
  },
  maintainers = { "@6cdh" },
}

list.scss = {
  install_info = {
    url = "https://github.com/serenadeai/tree-sitter-scss",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@elianiva" },
}

list.slint = {
  install_info = {
    url = "https://github.com/jrmoulton/tree-sitter-slint",
    files = { "src/parser.c" },
    branch = "main",
  },
  maintainers = { "@jrmoulton" },
  experimental = true,
}

list.smali = {
  install_info = {
    url = "https://git.sr.ht/~yotam/tree-sitter-smali",
    files = { "src/parser.c" },
  },
  maintainers = { "@amaanq" },
}

list.smithy = {
  install_info = {
    url = "https://github.com/indoorvivants/tree-sitter-smithy",
    files = { "src/parser.c" },
  },
  maintainers = { "@amaanq", "@keynmol" },
}

list.solidity = {
  install_info = {
    url = "https://github.com/YongJieYongJie/tree-sitter-solidity",
    branch = "with-generated-c-code",
    files = { "src/parser.c" },
  },
  maintainers = { "@YongJieYongJie" },
}

list.sparql = {
  install_info = {
    url = "https://github.com/BonaBeavis/tree-sitter-sparql",
    files = { "src/parser.c" },
    branch = "main",
  },
  maintainers = { "@BonaBeavis" },
}

list.sql = {
  install_info = {
    url = "https://github.com/derekstride/tree-sitter-sql",
    files = { "src/parser.c" },
    requires_generate_from_grammar = true,
  },
  maintainers = { "@derekstride" },
}

list.starlark = {
  install_info = {
    url = "https://github.com/amaanq/tree-sitter-starlark",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  filetype = "bzl",
  maintainers = { "@amaanq" },
}

list.supercollider = {
  install_info = {
    url = "https://github.com/madskjeldgaard/tree-sitter-supercollider",
    files = { "src/parser.c", "src/scanner.c" },
    branch = "main",
  },
  maintainers = { "@madskjeldgaard" },
  filetype = "supercollider",
}

list.surface = {
  install_info = {
    url = "https://github.com/connorlay/tree-sitter-surface",
    files = { "src/parser.c" },
    branch = "main",
  },
  filetype = "sface",
  maintainers = { "@connorlay" },
}

list.svelte = {
  install_info = {
    url = "https://github.com/Himujjal/tree-sitter-svelte",
    files = { "src/parser.c", "src/scanner.c" },
    branch = "master",
  },
  maintainers = { "@elianiva" },
}

list.swift = {
  install_info = {
    url = "https://github.com/alex-pinkus/tree-sitter-swift",
    branch = "main",
    files = { "src/parser.c", "src/scanner.c" },
    requires_generate_from_grammar = true,
  },
  maintainers = { "@alex-pinkus" },
}

list.sxhkdrc = {
  install_info = {
    url = "https://github.com/RaafatTurki/tree-sitter-sxhkdrc",
    files = { "src/parser.c" },
    branch = "master",
    generate_requires_npm = false,
    requires_generate_from_grammar = false,
    filetype = "sxhkdrc",
  },
  maintainers = { "@RaafatTurki" },
}

list.t32 = {
  install_info = {
    url = "https://codeberg.org/xasc/tree-sitter-t32",
    files = { "src/parser.c", "src/scanner.c" },
    branch = "main",
    revision = "675cd7de4eb80f2faa1e6d3bc9ee195fa9ef9790",
    generate_requires_npm = false,
    requires_generate_from_grammar = false,
  },
  filetype = "t32",
  maintainers = { "@xasc" },
}

list.teal = {
  install_info = {
    url = "https://github.com/euclidianAce/tree-sitter-teal",
    files = { "src/parser.c", "src/scanner.c" },
    requires_generate_from_grammar = true,
  },
  maintainers = { "@euclidianAce" },
}

list.terraform = {
  install_info = {
    url = "https://github.com/MichaHoffmann/tree-sitter-hcl",
    files = { "src/parser.c", "src/scanner.cc" },
    branch = "main",
    location = "dialects/terraform",
  },
  maintainers = { "@MichaHoffmann" },
  filetype = "terraform",
}

list.thrift = {
  install_info = {
    url = "https://github.com/duskmoon314/tree-sitter-thrift",
    files = { "src/parser.c" },
    branch = "main",
  },
  maintainers = { "@amaanq", "@duskmoon314" },
}

list.tiger = {
  install_info = {
    url = "https://github.com/ambroisie/tree-sitter-tiger",
    files = { "src/parser.c", "src/scanner.c" },
    branch = "main",
    generate_requires_npm = false,
    requires_generate_from_grammar = false,
  },
  filetype = "tiger",
  maintainers = { "@ambroisie" },
}

list.tlaplus = {
  install_info = {
    url = "https://github.com/tlaplus-community/tree-sitter-tlaplus",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  maintainers = { "@ahelwer", "@susliko" },
  filetype = "tla",
}

list.todotxt = {
  install_info = {
    url = "https://github.com/arnarg/tree-sitter-todotxt.git",
    files = { "src/parser.c" },
    branch = "main",
  },
  filetype = "todotxt",
  maintainers = { "@arnarg" },
  experimental = true,
}

list.toml = {
  install_info = {
    url = "https://github.com/ikatyang/tree-sitter-toml",
    files = { "src/parser.c", "src/scanner.c" },
    generate_requires_npm = true,
  },
  maintainers = { "@tk-shirasaka" },
}

list.tsx = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-typescript",
    files = { "src/parser.c", "src/scanner.c" },
    location = "tsx",
    generate_requires_npm = true,
  },
  filetype = "typescriptreact",
  maintainers = { "@steelsojka" },
}

list.turtle = {
  install_info = {
    url = "https://github.com/BonaBeavis/tree-sitter-turtle",
    files = { "src/parser.c" },
    branch = "main",
  },
  maintainers = { "@BonaBeavis" },
}

list.twig = {
  install_info = {
    url = "https://github.com/gbprod/tree-sitter-twig",
    branch = "main",
    files = { "src/parser.c" },
  },
  maintainers = { "@gbprod" },
  filetype = "twig",
}

list.typescript = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-typescript",
    files = { "src/parser.c", "src/scanner.c" },
    location = "typescript",
    generate_requires_npm = true,
  },
  maintainers = { "@steelsojka" },
}

list.v = {
  install_info = {
    url = "https://github.com/vlang/vls",
    files = { "src/parser.c", "src/scanner.c" },
    location = "tree_sitter_v",
    generate_requires_npm = false,
    requires_generate_from_grammar = false,
  },
  filetype = "vlang",
  maintainers = { "@kkharji" },
}

list.vala = {
  install_info = {
    url = "https://github.com/vala-lang/tree-sitter-vala",
    branch = "master",
    files = { "src/parser.c" },
  },
  maintainers = { "@Prince781" },
}

list.verilog = {
  install_info = {
    url = "https://github.com/tree-sitter/tree-sitter-verilog",
    files = { "src/parser.c" },
    generate_requires_npm = true,
  },
  maintainers = { "@zegervdv" },
}

list.vhs = {
  install_info = {
    url = "https://github.com/charmbracelet/tree-sitter-vhs",
    branch = "main",
    files = { "src/parser.c" },
  },
  maintainers = { "@caarlos0" },
  filetype = "tape",
}

list.vim = {
  install_info = {
    url = "https://github.com/vigoux/tree-sitter-viml",
    files = { "src/parser.c", "src/scanner.c" },
  },
  filetype = "vim",
  maintainers = { "@vigoux" },
}

list.vue = {
  install_info = {
    url = "https://github.com/ikatyang/tree-sitter-vue",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  maintainers = { "@WhyNotHugo" },
}

list.wgsl = {
  install_info = {
    url = "https://github.com/szebniok/tree-sitter-wgsl",
    files = { "src/parser.c", "src/scanner.c" },
  },
  maintainers = { "@szebniok" },
  filetype = "wgsl",
}

list.wgsl_bevy = {
  install_info = {
    url = "https://github.com/theHamsta/tree-sitter-wgsl-bevy",
    files = { "src/parser.c", "src/scanner.c" },
    generate_requires_npm = true,
  },
  maintainers = { "@theHamsta" },
}

list.yaml = {
  install_info = {
    url = "https://github.com/ikatyang/tree-sitter-yaml",
    files = { "src/parser.c", "src/scanner.cc" },
  },
  maintainers = { "@stsewd" },
}

list.yang = {
  install_info = {
    url = "https://github.com/Hubro/tree-sitter-yang",
    files = { "src/parser.c" },
  },
  maintainers = { "@Hubro" },
  filetype = "yang",
}

list.zig = {
  install_info = {
    url = "https://github.com/maxxnino/tree-sitter-zig",
    files = { "src/parser.c" },
    branch = "main",
  },
  filetype = "zig",
  maintainers = { "@maxxnino" },
}

local M = {
  list = list,
  filetype_to_parsername = filetype_to_parsername,
}

local function get_lang(ft)
  if new_lang_api then
    return ts.language.get_lang(ft)
  end
  return filetype_to_parsername[ft]
end

function M.ft_to_lang(ft)
  local result = get_lang(ft)
  if result then
    return result
  else
    ft = vim.split(ft, ".", { plain = true })[1]
    return get_lang(ft) or ft
  end
end

-- Get a list of all available parsers
---@return string[]
function M.available_parsers()
  if vim.fn.executable "tree-sitter" == 1 and vim.fn.executable "node" == 1 then
    return vim.tbl_keys(M.list)
  else
    return vim.tbl_filter(function(p) ---@param p string
      return not M.list[p].install_info.requires_generate_from_grammar
    end, vim.tbl_keys(M.list))
  end
end

function M.get_parser_configs()
  return M.list
end

local parser_files

function M.reset_cache()
  parser_files = setmetatable({}, {
    __index = function(tbl, key)
      rawset(tbl, key, api.nvim_get_runtime_file("parser/" .. key .. ".*", false))
      return rawget(tbl, key)
    end,
  })
end

M.reset_cache()

function M.has_parser(lang)
  lang = lang or M.get_buf_lang(api.nvim_get_current_buf())

  if not lang or #lang == 0 then
    return false
  end
  -- HACK: nvim internal API
  if vim._ts_has_language(lang) then
    return true
  end
  return #parser_files[lang] > 0
end

function M.get_parser(bufnr, lang)
  bufnr = bufnr or api.nvim_get_current_buf()
  lang = lang or M.get_buf_lang(bufnr)

  if M.has_parser(lang) then
    return ts.get_parser(bufnr, lang)
  end
end

-- @deprecated This is only kept for legacy purposes.
--             All root nodes should be accounted for.
function M.get_tree_root(bufnr)
  bufnr = bufnr or api.nvim_get_current_buf()
  return M.get_parser(bufnr):parse()[1]:root()
end

-- Gets the language of a given buffer
---@param bufnr number? or current buffer
---@return string
function M.get_buf_lang(bufnr)
  bufnr = bufnr or api.nvim_get_current_buf()
  return M.ft_to_lang(api.nvim_buf_get_option(bufnr, "ft"))
end

return M
