/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.cluster.decommission;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.opensearch.OpenSearchTimeoutException;
import org.opensearch.action.ActionListener;
import org.opensearch.action.admin.cluster.configuration.TransportAddVotingConfigExclusionsAction;
import org.opensearch.action.admin.cluster.configuration.VotingConfigExclusionsHelper;
import org.opensearch.action.admin.cluster.decommission.awareness.delete.DeleteDecommissionStateResponse;
import org.opensearch.action.admin.cluster.decommission.awareness.put.DecommissionRequest;
import org.opensearch.action.admin.cluster.decommission.awareness.put.DecommissionResponse;
import org.opensearch.cluster.ClusterState;
import org.opensearch.cluster.ClusterStateObserver;
import org.opensearch.cluster.ClusterStateUpdateTask;
import org.opensearch.cluster.NotClusterManagerException;
import org.opensearch.cluster.coordination.CoordinationMetadata;
import org.opensearch.cluster.decommission.DecommissionAttribute;
import org.opensearch.cluster.decommission.DecommissionAttributeMetadata;
import org.opensearch.cluster.decommission.DecommissionController;
import org.opensearch.cluster.decommission.DecommissionHelper;
import org.opensearch.cluster.decommission.DecommissionStatus;
import org.opensearch.cluster.decommission.DecommissioningFailedException;
import org.opensearch.cluster.metadata.WeightedRoutingMetadata;
import org.opensearch.cluster.node.DiscoveryNode;
import org.opensearch.cluster.routing.WeightedRouting;
import org.opensearch.cluster.routing.allocation.AllocationService;
import org.opensearch.cluster.routing.allocation.decider.AwarenessAllocationDecider;
import org.opensearch.cluster.service.ClusterService;
import org.opensearch.common.Priority;
import org.opensearch.common.UUIDs;
import org.opensearch.common.inject.Inject;
import org.opensearch.common.settings.ClusterSettings;
import org.opensearch.common.settings.Settings;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.threadpool.ThreadPool;
import org.opensearch.transport.TransportService;

public class DecommissionService {
    private static final Logger logger = LogManager.getLogger(DecommissionService.class);
    private final ClusterService clusterService;
    private final TransportService transportService;
    private final ThreadPool threadPool;
    private final DecommissionController decommissionController;
    private volatile List<String> awarenessAttributes;
    private volatile Map<String, List<String>> forcedAwarenessAttributes;
    private volatile int maxVotingConfigExclusions;

    @Inject
    public DecommissionService(Settings settings, ClusterSettings clusterSettings, ClusterService clusterService, TransportService transportService, ThreadPool threadPool, AllocationService allocationService) {
        this.clusterService = clusterService;
        this.transportService = transportService;
        this.threadPool = threadPool;
        this.decommissionController = new DecommissionController(clusterService, transportService, allocationService, threadPool);
        this.awarenessAttributes = AwarenessAllocationDecider.CLUSTER_ROUTING_ALLOCATION_AWARENESS_ATTRIBUTE_SETTING.get(settings);
        clusterSettings.addSettingsUpdateConsumer(AwarenessAllocationDecider.CLUSTER_ROUTING_ALLOCATION_AWARENESS_ATTRIBUTE_SETTING, this::setAwarenessAttributes);
        this.setForcedAwarenessAttributes(AwarenessAllocationDecider.CLUSTER_ROUTING_ALLOCATION_AWARENESS_FORCE_GROUP_SETTING.get(settings));
        clusterSettings.addSettingsUpdateConsumer(AwarenessAllocationDecider.CLUSTER_ROUTING_ALLOCATION_AWARENESS_FORCE_GROUP_SETTING, this::setForcedAwarenessAttributes);
        this.maxVotingConfigExclusions = TransportAddVotingConfigExclusionsAction.MAXIMUM_VOTING_CONFIG_EXCLUSIONS_SETTING.get(settings);
        clusterSettings.addSettingsUpdateConsumer(TransportAddVotingConfigExclusionsAction.MAXIMUM_VOTING_CONFIG_EXCLUSIONS_SETTING, this::setMaxVotingConfigExclusions);
    }

    private void setAwarenessAttributes(List<String> awarenessAttributes) {
        this.awarenessAttributes = awarenessAttributes;
    }

    private void setForcedAwarenessAttributes(Settings forceSettings) {
        HashMap<String, List<String>> forcedAwarenessAttributes = new HashMap<String, List<String>>();
        Map<String, Settings> forceGroups = forceSettings.getAsGroups();
        for (Map.Entry<String, Settings> entry : forceGroups.entrySet()) {
            List<String> aValues = entry.getValue().getAsList("values");
            if (aValues.size() <= 0) continue;
            forcedAwarenessAttributes.put(entry.getKey(), aValues);
        }
        this.forcedAwarenessAttributes = forcedAwarenessAttributes;
    }

    private void setMaxVotingConfigExclusions(int maxVotingConfigExclusions) {
        this.maxVotingConfigExclusions = maxVotingConfigExclusions;
    }

    public void startDecommissionAction(final DecommissionRequest decommissionRequest, final ActionListener<DecommissionResponse> listener) {
        final DecommissionAttribute decommissionAttribute = decommissionRequest.getDecommissionAttribute();
        this.clusterService.submitStateUpdateTask("decommission [" + decommissionAttribute + "]", new ClusterStateUpdateTask(Priority.URGENT){
            private Set<String> nodeIdsToBeExcluded;

            @Override
            public ClusterState execute(ClusterState currentState) {
                DecommissionService.validateAwarenessAttribute(decommissionAttribute, DecommissionService.this.awarenessAttributes, DecommissionService.this.forcedAwarenessAttributes);
                if (decommissionRequest.requestID() == null) {
                    decommissionRequest.setRequestID(UUIDs.randomBase64UUID());
                }
                DecommissionAttributeMetadata decommissionAttributeMetadata = currentState.metadata().decommissionAttributeMetadata();
                DecommissionService.ensureEligibleRequest(decommissionAttributeMetadata, decommissionRequest);
                DecommissionService.ensureToBeDecommissionedAttributeWeighedAway(currentState, decommissionAttribute);
                ClusterState newState = DecommissionHelper.registerDecommissionAttributeInClusterState(currentState, decommissionAttribute, decommissionRequest.requestID());
                this.nodeIdsToBeExcluded = DecommissionHelper.filterNodesWithDecommissionAttribute(currentState, decommissionAttribute, true).stream().map(DiscoveryNode::getId).collect(Collectors.toSet());
                logger.info("resolved cluster manager eligible nodes [{}] that should be added to voting config exclusion", (Object)this.nodeIdsToBeExcluded.toString());
                newState = DecommissionHelper.addVotingConfigExclusionsForNodesToBeDecommissioned(newState, this.nodeIdsToBeExcluded, TimeValue.timeValueSeconds((long)120L), DecommissionService.this.maxVotingConfigExclusions);
                logger.debug("registering decommission metadata [{}] to execute action", (Object)newState.metadata().decommissionAttributeMetadata().toString());
                return newState;
            }

            @Override
            public void onFailure(String source, Exception e) {
                logger.error(() -> new ParameterizedMessage("failed to start decommission action for attribute [{}]", (Object)decommissionAttribute.toString()), (Throwable)e);
                listener.onFailure(e);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                DecommissionAttributeMetadata decommissionAttributeMetadata = newState.metadata().decommissionAttributeMetadata();
                assert (decommissionAttribute.equals(decommissionAttributeMetadata.decommissionAttribute()));
                assert (decommissionAttributeMetadata.status().equals((Object)DecommissionStatus.INIT));
                assert (decommissionAttributeMetadata.requestID().equals(decommissionRequest.requestID()));
                assert (newState.getVotingConfigExclusions().stream().map(CoordinationMetadata.VotingConfigExclusion::getNodeId).collect(Collectors.toSet()).containsAll(this.nodeIdsToBeExcluded));
                logger.debug("registered decommission metadata for attribute [{}] with status [{}]", (Object)decommissionAttributeMetadata.decommissionAttribute(), (Object)decommissionAttributeMetadata.status());
                ClusterStateObserver observer = new ClusterStateObserver(DecommissionService.this.clusterService, TimeValue.timeValueSeconds((long)120L), logger, DecommissionService.this.threadPool.getThreadContext());
                Predicate<ClusterState> allNodesRemovedAndAbdicated = clusterState -> {
                    Set<String> votingConfigNodeIds = clusterState.getLastCommittedConfiguration().getNodeIds();
                    return this.nodeIdsToBeExcluded.stream().noneMatch(votingConfigNodeIds::contains) && clusterState.nodes().getClusterManagerNodeId() != null && !this.nodeIdsToBeExcluded.contains(clusterState.nodes().getClusterManagerNodeId());
                };
                ClusterStateObserver.Listener clusterStateListener = new ClusterStateObserver.Listener(){

                    @Override
                    public void onNewClusterState(ClusterState state) {
                        logger.info("successfully removed decommissioned cluster manager eligible nodes [{}] from voting config ", (Object)nodeIdsToBeExcluded.toString());
                        if (state.nodes().isLocalNodeElectedClusterManager()) {
                            if (DecommissionHelper.nodeHasDecommissionedAttribute(DecommissionService.this.clusterService.localNode(), decommissionAttribute)) {
                                String errorMsg = "unexpected state encountered [local node is to-be-decommissioned leader] while executing decommission request";
                                logger.error(errorMsg);
                                DecommissionService.this.decommissionController.updateMetadataWithDecommissionStatus(DecommissionStatus.FAILED, DecommissionService.this.statusUpdateListener());
                                listener.onFailure(new IllegalStateException(errorMsg));
                            } else {
                                logger.info("will proceed to drain decommissioned nodes as local node is eligible to process the request");
                                listener.onResponse(new DecommissionResponse(true));
                                DecommissionService.this.drainNodesWithDecommissionedAttribute(decommissionRequest);
                            }
                        } else {
                            logger.info("local node is not eligible to process the request, throwing NotClusterManagerException to attempt a retry on an eligible node");
                            listener.onFailure(new NotClusterManagerException("node [" + DecommissionService.this.transportService.getLocalNode().toString() + "] not eligible to execute decommission request. Will retry until timeout."));
                        }
                    }

                    @Override
                    public void onClusterServiceClose() {
                        String errorMsg = "cluster service closed while waiting for abdication of to-be-decommissioned leader";
                        logger.error(errorMsg);
                        listener.onFailure(new DecommissioningFailedException(decommissionAttribute, errorMsg));
                    }

                    @Override
                    public void onTimeout(TimeValue timeout) {
                        String errorMsg = "timed out [" + timeout.toString() + "] while removing to-be-decommissioned cluster manager eligible nodes [" + nodeIdsToBeExcluded.toString() + "] from voting config";
                        logger.error(errorMsg);
                        listener.onFailure(new OpenSearchTimeoutException(errorMsg, new Object[0]));
                        DecommissionService.this.decommissionController.updateMetadataWithDecommissionStatus(DecommissionStatus.FAILED, DecommissionService.this.statusUpdateListener());
                    }
                };
                if (allNodesRemovedAndAbdicated.test(newState)) {
                    clusterStateListener.onNewClusterState(newState);
                } else {
                    logger.debug("waiting to abdicate to-be-decommissioned leader");
                    observer.waitForNextChange(clusterStateListener, allNodesRemovedAndAbdicated);
                }
            }
        });
    }

    void drainNodesWithDecommissionedAttribute(final DecommissionRequest decommissionRequest) {
        ClusterState state = this.clusterService.getClusterApplierService().state();
        assert (state.metadata().decommissionAttributeMetadata().requestID().equals(decommissionRequest.requestID()));
        final Set<DiscoveryNode> decommissionedNodes = DecommissionHelper.filterNodesWithDecommissionAttribute(state, decommissionRequest.getDecommissionAttribute(), false);
        if (decommissionRequest.isNoDelay()) {
            this.failDecommissionedNodes(decommissionedNodes, decommissionRequest.getDecommissionAttribute());
        } else {
            this.decommissionController.updateMetadataWithDecommissionStatus(DecommissionStatus.DRAINING, new ActionListener<DecommissionStatus>(){

                @Override
                public void onResponse(DecommissionStatus status) {
                    logger.info("updated the decommission status to [{}]", (Object)status);
                    DecommissionService.this.scheduleNodesDecommissionOnTimeout(decommissionedNodes, decommissionRequest.getDelayTimeout());
                }

                @Override
                public void onFailure(Exception e) {
                    logger.error(() -> new ParameterizedMessage("failed to update decommission status for attribute [{}] to [{}]", (Object)decommissionRequest.getDecommissionAttribute().toString(), (Object)DecommissionStatus.DRAINING), (Throwable)e);
                    DecommissionService.this.decommissionController.updateMetadataWithDecommissionStatus(DecommissionStatus.FAILED, DecommissionService.this.statusUpdateListener());
                }
            });
        }
    }

    void scheduleNodesDecommissionOnTimeout(Set<DiscoveryNode> decommissionedNodes, TimeValue timeoutForNodeDraining) {
        ClusterState state = this.clusterService.getClusterApplierService().state();
        DecommissionAttributeMetadata decommissionAttributeMetadata = state.metadata().decommissionAttributeMetadata();
        if (decommissionAttributeMetadata == null) {
            return;
        }
        assert (decommissionAttributeMetadata.status().equals((Object)DecommissionStatus.DRAINING)) : "Unexpected status encountered while decommissioning nodes.";
        DecommissionAttribute decommissionAttribute = decommissionAttributeMetadata.decommissionAttribute();
        this.transportService.getThreadPool().schedule(() -> {
            this.decommissionController.getActiveRequestCountOnDecommissionedNodes(decommissionedNodes);
            this.failDecommissionedNodes(decommissionedNodes, decommissionAttribute);
        }, timeoutForNodeDraining, "generic");
    }

    private void failDecommissionedNodes(final Set<DiscoveryNode> decommissionedNodes, final DecommissionAttribute decommissionAttribute) {
        this.decommissionController.updateMetadataWithDecommissionStatus(DecommissionStatus.IN_PROGRESS, new ActionListener<DecommissionStatus>(){

            @Override
            public void onResponse(DecommissionStatus status) {
                logger.info("updated the decommission status to [{}]", (Object)status);
                DecommissionService.this.decommissionController.removeDecommissionedNodes(decommissionedNodes, "nodes-decommissioned", TimeValue.timeValueSeconds((long)120L), new ActionListener<Void>(){

                    @Override
                    public void onResponse(Void unused) {
                        DecommissionService.this.decommissionController.updateMetadataWithDecommissionStatus(DecommissionStatus.SUCCESSFUL, DecommissionService.this.statusUpdateListener());
                    }

                    @Override
                    public void onFailure(Exception e) {
                        DecommissionService.this.decommissionController.updateMetadataWithDecommissionStatus(DecommissionStatus.FAILED, DecommissionService.this.statusUpdateListener());
                    }
                });
            }

            @Override
            public void onFailure(Exception e) {
                logger.error(() -> new ParameterizedMessage("failed to update decommission status for attribute [{}] to [{}]", (Object)decommissionAttribute.toString(), (Object)DecommissionStatus.IN_PROGRESS), (Throwable)e);
                DecommissionService.this.decommissionController.updateMetadataWithDecommissionStatus(DecommissionStatus.FAILED, DecommissionService.this.statusUpdateListener());
            }
        });
    }

    private static void validateAwarenessAttribute(DecommissionAttribute decommissionAttribute, List<String> awarenessAttributes, Map<String, List<String>> forcedAwarenessAttributes) {
        Object msg = null;
        if (awarenessAttributes == null) {
            msg = "awareness attribute not set to the cluster.";
        } else if (forcedAwarenessAttributes == null) {
            msg = "forced awareness attribute not set to the cluster.";
        } else if (!awarenessAttributes.contains(decommissionAttribute.attributeName())) {
            msg = "invalid awareness attribute requested for decommissioning";
        } else if (!forcedAwarenessAttributes.containsKey(decommissionAttribute.attributeName())) {
            msg = "forced awareness attribute [" + forcedAwarenessAttributes.toString() + "] doesn't have the decommissioning attribute";
        }
        if (msg != null) {
            throw new DecommissioningFailedException(decommissionAttribute, (String)msg);
        }
    }

    private static void ensureToBeDecommissionedAttributeWeighedAway(ClusterState state, DecommissionAttribute decommissionAttribute) {
        WeightedRoutingMetadata weightedRoutingMetadata = state.metadata().weightedRoutingMetadata();
        if (weightedRoutingMetadata == null) {
            throw new DecommissioningFailedException(decommissionAttribute, "no weights are set to the attribute. Please set appropriate weights before triggering decommission action");
        }
        WeightedRouting weightedRouting = weightedRoutingMetadata.getWeightedRouting();
        if (!weightedRouting.attributeName().equals(decommissionAttribute.attributeName())) {
            throw new DecommissioningFailedException(decommissionAttribute, "no weights are specified to attribute [" + decommissionAttribute.attributeName() + "]");
        }
        Double attributeValueWeight = weightedRouting.weights().get(decommissionAttribute.attributeValue());
        if (attributeValueWeight != null && !attributeValueWeight.equals(0.0)) {
            throw new DecommissioningFailedException(decommissionAttribute, "weight for decommissioned attribute is expected to be [0.0] but found [" + attributeValueWeight + "]");
        }
    }

    private static void ensureEligibleRequest(DecommissionAttributeMetadata decommissionAttributeMetadata, DecommissionRequest decommissionRequest) {
        block11: {
            DecommissionAttribute requestedDecommissionAttribute;
            block12: {
                requestedDecommissionAttribute = decommissionRequest.getDecommissionAttribute();
                if (decommissionAttributeMetadata == null) break block11;
                if (!decommissionAttributeMetadata.decommissionAttribute().equals(requestedDecommissionAttribute)) break block12;
                switch (decommissionAttributeMetadata.status()) {
                    case INIT: {
                        if (!decommissionRequest.requestID().equals(decommissionAttributeMetadata.requestID())) {
                            throw new DecommissioningFailedException(requestedDecommissionAttribute, "same request is already in status [INIT]");
                        }
                        break block11;
                    }
                    case FAILED: {
                        break block11;
                    }
                    case DRAINING: 
                    case IN_PROGRESS: 
                    case SUCCESSFUL: {
                        String msg = "same request is already in status [" + decommissionAttributeMetadata.status() + "]";
                        throw new DecommissioningFailedException(requestedDecommissionAttribute, msg);
                    }
                    default: {
                        throw new IllegalStateException("unknown status [" + decommissionAttributeMetadata.status() + "] currently registered in metadata");
                    }
                }
            }
            switch (decommissionAttributeMetadata.status()) {
                case SUCCESSFUL: {
                    String msg = "one awareness attribute [" + decommissionAttributeMetadata.decommissionAttribute().toString() + "] already successfully decommissioned, recommission before triggering another decommission";
                    throw new DecommissioningFailedException(requestedDecommissionAttribute, msg);
                }
                case INIT: 
                case DRAINING: 
                case IN_PROGRESS: {
                    String msg = "there's an inflight decommission request for attribute [" + decommissionAttributeMetadata.decommissionAttribute().toString() + "] is in progress, cannot process this request";
                    throw new DecommissioningFailedException(requestedDecommissionAttribute, msg);
                }
                case FAILED: {
                    break;
                }
                default: {
                    throw new IllegalStateException("unknown status [" + decommissionAttributeMetadata.status() + "] currently registered in metadata");
                }
            }
        }
    }

    private ActionListener<DecommissionStatus> statusUpdateListener() {
        return new ActionListener<DecommissionStatus>(){

            @Override
            public void onResponse(DecommissionStatus status) {
                logger.info("updated the decommission status to [{}]", (Object)status);
            }

            @Override
            public void onFailure(Exception e) {
                logger.error("unexpected failure occurred during decommission status update", (Throwable)e);
            }
        };
    }

    public void startRecommissionAction(final ActionListener<DeleteDecommissionStateResponse> listener) {
        this.clusterService.submitStateUpdateTask("delete-decommission-state", new ClusterStateUpdateTask(Priority.URGENT){

            @Override
            public ClusterState execute(ClusterState currentState) {
                ClusterState newState = VotingConfigExclusionsHelper.clearExclusionsAndGetState(currentState);
                logger.info("Deleting the decommission attribute from the cluster state");
                newState = DecommissionHelper.deleteDecommissionAttributeInClusterState(newState);
                return newState;
            }

            @Override
            public void onFailure(String source, Exception e) {
                logger.error(() -> new ParameterizedMessage("failure during recommission action [{}]", (Object)source), (Throwable)e);
                listener.onFailure(e);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                logger.info("successfully cleared voting config exclusion and decommissioned attribute");
                assert (newState.metadata().decommissionAttributeMetadata() == null);
                assert (newState.coordinationMetadata().getVotingConfigExclusions().isEmpty());
                listener.onResponse(new DeleteDecommissionStateResponse(true));
            }
        });
    }
}

