from __future__ import annotations

import os
import platform
import sys

from pathlib import Path
from typing import TYPE_CHECKING

from installer import install
from installer.destinations import SchemeDictionaryDestination
from installer.sources import WheelFile

from poetry.__version__ import __version__
from poetry.utils._compat import WINDOWS


if TYPE_CHECKING:
    from typing import BinaryIO

    from installer.records import RecordEntry
    from installer.scripts import LauncherKind
    from installer.utils import Scheme

    from poetry.utils.env import Env


class WheelDestination(SchemeDictionaryDestination):
    """ """

    def write_to_fs(
        self,
        scheme: Scheme,
        path: Path | str,
        stream: BinaryIO,
        is_executable: bool,
    ) -> RecordEntry:
        from installer.records import Hash
        from installer.records import RecordEntry
        from installer.utils import copyfileobj_with_hashing
        from installer.utils import make_file_executable

        target_path = Path(self.scheme_dict[scheme]) / path
        if target_path.exists():
            # Contrary to the base library we don't raise an error
            # here since it can break namespace packages (like Poetry's)
            pass

        parent_folder = target_path.parent
        if not parent_folder.exists():
            # Due to the parallel installation it can happen
            # that two threads try to create the directory.
            os.makedirs(parent_folder, exist_ok=True)

        with open(target_path, "wb") as f:
            hash_, size = copyfileobj_with_hashing(stream, f, self.hash_algorithm)

        if is_executable:
            make_file_executable(target_path)

        return RecordEntry(str(path), Hash(self.hash_algorithm, hash_), size)

    def for_source(self, source: WheelFile) -> WheelDestination:
        scheme_dict = self.scheme_dict.copy()

        scheme_dict["headers"] = str(Path(scheme_dict["headers"]) / source.distribution)

        return self.__class__(
            scheme_dict,
            interpreter=self.interpreter,
            script_kind=self.script_kind,
            bytecode_optimization_levels=self.bytecode_optimization_levels,
        )


class WheelInstaller:
    def __init__(self, env: Env) -> None:
        self._env = env

        script_kind: LauncherKind
        if not WINDOWS:
            script_kind = "posix"
        else:
            if platform.uname()[4].startswith("arm"):
                script_kind = "win-arm64" if sys.maxsize > 2**32 else "win-arm"
            else:
                script_kind = "win-amd64" if sys.maxsize > 2**32 else "win-ia32"

        schemes = self._env.paths
        schemes["headers"] = schemes["include"]

        self._destination = WheelDestination(
            schemes, interpreter=self._env.python, script_kind=script_kind
        )

    def enable_bytecode_compilation(self, enable: bool = True) -> None:
        self._destination.bytecode_optimization_levels = (1,) if enable else ()

    def install(self, wheel: Path) -> None:
        with WheelFile.open(Path(wheel.as_posix())) as source:
            install(
                source=source,
                destination=self._destination.for_source(source),
                # Additional metadata that is generated by the installation tool.
                additional_metadata={
                    "INSTALLER": f"Poetry {__version__}".encode(),
                },
            )
