/*! @header
 *  This header defines functions for constructing, extending, and truncating
 *  simd vector types.
 *
 *  For each vector type `simd_typeN` supported by <simd/simd.h>, the following
 *  constructors are provided:
 *
 *  ~~~
 *  simd_typeN simd_make_typeN(type other);
 *  simd_typeN simd_make_typeN(simd_typeM other);
 *  ~~~
 *  For the scalar-input version, or if M < N, these functions zero-extend
 *  `other` to produce a wider vector. If M == N, `other` is passed through
 *  unmodified. If `M > N`, `other` is truncated to form the result.
 *
 *  ~~~
 *  simd_typeN simd_make_typeN_undef(type other);
 *  simd_typeN simd_make_typeN_undef(simd_typeM other);
 *  ~~~
 *  These functions are only available for M < N and for scalar inputs. They 
 *  extend `other` to produce a wider vector where the contents of the newly-
 *  formed lanes are undefined.
 *
 *  In addition, if N is 2, 3, or 4, the following constructors are available:
 *  ~~~
 *  simd_make_typeN(parts ...)
 *  ~~~
 *  where parts is a list of scalars and smaller vectors such that the sum of
 *  the number of lanes in the arguments is equal to N. For example, a
 *  `simd_float3` can be constructed from three `floats`, or a `float` and a
 *  `simd_float2` in any order:
 *  ~~~
 *  simd_float2 ab = { 1, 2 };
 *  simd_float3 vector = simd_make_float3(ab, 3);
 *  ~~~
 *
 *  In C++ the above functions are templated in the simd:: namespace.
 *
 *      C++ Function                            Equivalent C Function
 *      -------------------------------------------------------------------
 *      simd::make<simd::typeN>(x ...)          simd_make_typeN(x ...)
 *      simd::make_undef<simd::typeN>(x ...)    simd_make_typeN_undef(x ...)
 *
 *
 *  In addition, templated Vector<ScalarType, count> struct is available for
 *  templated code based on the scalar type.
 *
 *      template <typename ScalarType, size_t count> struct simd::Vector {
 *        //  static const size_t count
 *        //  typedef scalar_t
 *        //  typedef type
 *        //  typedef packed_t
 *      };
 *
 *  Lookup the equivalent Vector struct according to typeN:
 *      template <typename typeN> struct simd::get_traits
 *      {
 *      //    using type = Vector<ScalarType, count>;
 *      };
 *
 *  This is commonly used to get the type traits of typeN, so a helper type,
 *  namely traits, is available to query the type traits easily.
 *      simd::traits<typeN>::count
 *      simd::traits<typeN>::scalar_t
 *
 *  @copyright 2014-2016 Apple, Inc. All rights reserved.
 *  @unsorted                                                                 */

#ifndef SIMD_VECTOR_CONSTRUCTORS
#define SIMD_VECTOR_CONSTRUCTORS

#include <simd/vector_types.h>
#if SIMD_COMPILER_HAS_REQUIRED_FEATURES

#ifdef __cplusplus
extern "C" {
#endif

/*! @abstract Concatenates `x` and `y` to form a vector of two 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char2 simd_make_char2(char x, char y) {
  simd_char2 result;
  result.x = x;
  result.y = y;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of two 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char2 simd_make_char2(char other) {
  simd_char2 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of two 8-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_char2 simd_make_char2_undef(char other) {
  simd_char2 result;
  result.x = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_char2 simd_make_char2(simd_char2 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of two 8-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_char2 simd_make_char2(simd_char3 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 8-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_char2 simd_make_char2(simd_char4 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 8-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_char2 simd_make_char2(simd_char8 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 8-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_char2 simd_make_char2(simd_char16 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 8-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_char2 simd_make_char2(simd_char32 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 8-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_char2 simd_make_char2(simd_char64 other) {
  return other.xy;
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char3 simd_make_char3(char x, char y, char z) {
  simd_char3 result;
  result.x = x;
  result.y = y;
  result.z = z;
  return result;
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char3 simd_make_char3(char x, simd_char2 yz) {
  simd_char3 result;
  result.x = x;
  result.yz = yz;
  return result;
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char3 simd_make_char3(simd_char2 xy, char z) {
  simd_char3 result;
  result.xy = xy;
  result.z = z;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of three 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char3 simd_make_char3(char other) {
  simd_char3 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 8-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_char3 simd_make_char3_undef(char other) {
  simd_char3 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of three 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char3 simd_make_char3(simd_char2 other) {
  simd_char3 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 8-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_char3 simd_make_char3_undef(simd_char2 other) {
  simd_char3 result;
  result.xy = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_char3 simd_make_char3(simd_char3 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of three 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char3 simd_make_char3(simd_char4 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char3 simd_make_char3(simd_char8 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char3 simd_make_char3(simd_char16 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char3 simd_make_char3(simd_char32 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char3 simd_make_char3(simd_char64 other) {
  return other.xyz;
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  8-bit signed (twos-complement) integers.                                  */
static inline SIMD_CFUNC simd_char4 simd_make_char4(char x, char y, char z, char w) {
  simd_char4 result;
  result.x = x;
  result.y = y;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char4 simd_make_char4(char x, char y, simd_char2 zw) {
  simd_char4 result;
  result.x = x;
  result.y = y;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char4 simd_make_char4(char x, simd_char2 yz, char w) {
  simd_char4 result;
  result.x = x;
  result.yz = yz;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char4 simd_make_char4(simd_char2 xy, char z, char w) {
  simd_char4 result;
  result.xy = xy;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char4 simd_make_char4(char x, simd_char3 yzw) {
  simd_char4 result;
  result.x = x;
  result.yzw = yzw;
  return result;
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char4 simd_make_char4(simd_char2 xy, simd_char2 zw) {
  simd_char4 result;
  result.xy = xy;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char4 simd_make_char4(simd_char3 xyz, char w) {
  simd_char4 result;
  result.xyz = xyz;
  result.w = w;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of four 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char4 simd_make_char4(char other) {
  simd_char4 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 8-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_char4 simd_make_char4_undef(char other) {
  simd_char4 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char4 simd_make_char4(simd_char2 other) {
  simd_char4 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 8-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_char4 simd_make_char4_undef(simd_char2 other) {
  simd_char4 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char4 simd_make_char4(simd_char3 other) {
  simd_char4 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 8-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_char4 simd_make_char4_undef(simd_char3 other) {
  simd_char4 result;
  result.xyz = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_char4 simd_make_char4(simd_char4 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of four 8-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_char4 simd_make_char4(simd_char8 other) {
  return other.xyzw;
}

/*! @abstract Truncates `other` to form a vector of four 8-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_char4 simd_make_char4(simd_char16 other) {
  return other.xyzw;
}

/*! @abstract Truncates `other` to form a vector of four 8-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_char4 simd_make_char4(simd_char32 other) {
  return other.xyzw;
}

/*! @abstract Truncates `other` to form a vector of four 8-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_char4 simd_make_char4(simd_char64 other) {
  return other.xyzw;
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char8 simd_make_char8(simd_char4 lo, simd_char4 hi) {
  simd_char8 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char8 simd_make_char8(char other) {
  simd_char8 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 8-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_char8 simd_make_char8_undef(char other) {
  simd_char8 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char8 simd_make_char8(simd_char2 other) {
  simd_char8 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 8-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_char8 simd_make_char8_undef(simd_char2 other) {
  simd_char8 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char8 simd_make_char8(simd_char3 other) {
  simd_char8 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 8-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_char8 simd_make_char8_undef(simd_char3 other) {
  simd_char8 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char8 simd_make_char8(simd_char4 other) {
  simd_char8 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 8-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_char8 simd_make_char8_undef(simd_char4 other) {
  simd_char8 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_char8 simd_make_char8(simd_char8 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of eight 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char8 simd_make_char8(simd_char16 other) {
  return simd_make_char8(other.lo);
}

/*! @abstract Truncates `other` to form a vector of eight 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char8 simd_make_char8(simd_char32 other) {
  return simd_make_char8(other.lo);
}

/*! @abstract Truncates `other` to form a vector of eight 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char8 simd_make_char8(simd_char64 other) {
  return simd_make_char8(other.lo);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixteen 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char16 simd_make_char16(simd_char8 lo, simd_char8 hi) {
  simd_char16 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char16 simd_make_char16(char other) {
  simd_char16 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char16 simd_make_char16_undef(char other) {
  simd_char16 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char16 simd_make_char16(simd_char2 other) {
  simd_char16 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char16 simd_make_char16_undef(simd_char2 other) {
  simd_char16 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char16 simd_make_char16(simd_char3 other) {
  simd_char16 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char16 simd_make_char16_undef(simd_char3 other) {
  simd_char16 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char16 simd_make_char16(simd_char4 other) {
  simd_char16 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char16 simd_make_char16_undef(simd_char4 other) {
  simd_char16 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char16 simd_make_char16(simd_char8 other) {
  simd_char16 result = 0;
  result.lo = simd_make_char8(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char16 simd_make_char16_undef(simd_char8 other) {
  simd_char16 result;
  result.lo = simd_make_char8(other);
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_char16 simd_make_char16(simd_char16 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of sixteen 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char16 simd_make_char16(simd_char32 other) {
  return simd_make_char16(other.lo);
}

/*! @abstract Truncates `other` to form a vector of sixteen 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char16 simd_make_char16(simd_char64 other) {
  return simd_make_char16(other.lo);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of thirty-two
 *  8-bit signed (twos-complement) integers.                                  */
static inline SIMD_CFUNC simd_char32 simd_make_char32(simd_char16 lo, simd_char16 hi) {
  simd_char32 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char32 simd_make_char32(char other) {
  simd_char32 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char32 simd_make_char32_undef(char other) {
  simd_char32 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char32 simd_make_char32(simd_char2 other) {
  simd_char32 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char32 simd_make_char32_undef(simd_char2 other) {
  simd_char32 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char32 simd_make_char32(simd_char3 other) {
  simd_char32 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char32 simd_make_char32_undef(simd_char3 other) {
  simd_char32 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char32 simd_make_char32(simd_char4 other) {
  simd_char32 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char32 simd_make_char32_undef(simd_char4 other) {
  simd_char32 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char32 simd_make_char32(simd_char8 other) {
  simd_char32 result = 0;
  result.lo = simd_make_char16(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char32 simd_make_char32_undef(simd_char8 other) {
  simd_char32 result;
  result.lo = simd_make_char16(other);
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char32 simd_make_char32(simd_char16 other) {
  simd_char32 result = 0;
  result.lo = simd_make_char16(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char32 simd_make_char32_undef(simd_char16 other) {
  simd_char32 result;
  result.lo = simd_make_char16(other);
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_char32 simd_make_char32(simd_char32 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of thirty-two 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_char32 simd_make_char32(simd_char64 other) {
  return simd_make_char32(other.lo);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixty-four
 *  8-bit signed (twos-complement) integers.                                  */
static inline SIMD_CFUNC simd_char64 simd_make_char64(simd_char32 lo, simd_char32 hi) {
  simd_char64 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixty-four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char64 simd_make_char64(char other) {
  simd_char64 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixty-four 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char64 simd_make_char64_undef(char other) {
  simd_char64 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixty-four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char64 simd_make_char64(simd_char2 other) {
  simd_char64 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixty-four 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char64 simd_make_char64_undef(simd_char2 other) {
  simd_char64 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixty-four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char64 simd_make_char64(simd_char3 other) {
  simd_char64 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixty-four 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char64 simd_make_char64_undef(simd_char3 other) {
  simd_char64 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixty-four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char64 simd_make_char64(simd_char4 other) {
  simd_char64 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixty-four 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char64 simd_make_char64_undef(simd_char4 other) {
  simd_char64 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixty-four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char64 simd_make_char64(simd_char8 other) {
  simd_char64 result = 0;
  result.lo = simd_make_char32(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixty-four 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char64 simd_make_char64_undef(simd_char8 other) {
  simd_char64 result;
  result.lo = simd_make_char32(other);
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixty-four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char64 simd_make_char64(simd_char16 other) {
  simd_char64 result = 0;
  result.lo = simd_make_char32(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixty-four 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char64 simd_make_char64_undef(simd_char16 other) {
  simd_char64 result;
  result.lo = simd_make_char32(other);
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixty-four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_char64 simd_make_char64(simd_char32 other) {
  simd_char64 result = 0;
  result.lo = simd_make_char32(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixty-four 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_char64 simd_make_char64_undef(simd_char32 other) {
  simd_char64 result;
  result.lo = simd_make_char32(other);
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_char64 simd_make_char64(simd_char64 other) {
  return other;
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar2 simd_make_uchar2(unsigned char x, unsigned char y) {
  simd_uchar2 result;
  result.x = x;
  result.y = y;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of two 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar2 simd_make_uchar2(unsigned char other) {
  simd_uchar2 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of two 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar2 simd_make_uchar2_undef(unsigned char other) {
  simd_uchar2 result;
  result.x = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_uchar2 simd_make_uchar2(simd_uchar2 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of two 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar2 simd_make_uchar2(simd_uchar3 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar2 simd_make_uchar2(simd_uchar4 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar2 simd_make_uchar2(simd_uchar8 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar2 simd_make_uchar2(simd_uchar16 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar2 simd_make_uchar2(simd_uchar32 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar2 simd_make_uchar2(simd_uchar64 other) {
  return other.xy;
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar3 simd_make_uchar3(unsigned char x, unsigned char y, unsigned char z) {
  simd_uchar3 result;
  result.x = x;
  result.y = y;
  result.z = z;
  return result;
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar3 simd_make_uchar3(unsigned char x, simd_uchar2 yz) {
  simd_uchar3 result;
  result.x = x;
  result.yz = yz;
  return result;
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar3 simd_make_uchar3(simd_uchar2 xy, unsigned char z) {
  simd_uchar3 result;
  result.xy = xy;
  result.z = z;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of three 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar3 simd_make_uchar3(unsigned char other) {
  simd_uchar3 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar3 simd_make_uchar3_undef(unsigned char other) {
  simd_uchar3 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of three 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar3 simd_make_uchar3(simd_uchar2 other) {
  simd_uchar3 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar3 simd_make_uchar3_undef(simd_uchar2 other) {
  simd_uchar3 result;
  result.xy = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_uchar3 simd_make_uchar3(simd_uchar3 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of three 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar3 simd_make_uchar3(simd_uchar4 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar3 simd_make_uchar3(simd_uchar8 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar3 simd_make_uchar3(simd_uchar16 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar3 simd_make_uchar3(simd_uchar32 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar3 simd_make_uchar3(simd_uchar64 other) {
  return other.xyz;
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  8-bit unsigned integers.                                                  */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4(unsigned char x, unsigned char y, unsigned char z, unsigned char w) {
  simd_uchar4 result;
  result.x = x;
  result.y = y;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4(unsigned char x, unsigned char y, simd_uchar2 zw) {
  simd_uchar4 result;
  result.x = x;
  result.y = y;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4(unsigned char x, simd_uchar2 yz, unsigned char w) {
  simd_uchar4 result;
  result.x = x;
  result.yz = yz;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4(simd_uchar2 xy, unsigned char z, unsigned char w) {
  simd_uchar4 result;
  result.xy = xy;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4(unsigned char x, simd_uchar3 yzw) {
  simd_uchar4 result;
  result.x = x;
  result.yzw = yzw;
  return result;
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4(simd_uchar2 xy, simd_uchar2 zw) {
  simd_uchar4 result;
  result.xy = xy;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4(simd_uchar3 xyz, unsigned char w) {
  simd_uchar4 result;
  result.xyz = xyz;
  result.w = w;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of four 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4(unsigned char other) {
  simd_uchar4 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4_undef(unsigned char other) {
  simd_uchar4 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4(simd_uchar2 other) {
  simd_uchar4 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4_undef(simd_uchar2 other) {
  simd_uchar4 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4(simd_uchar3 other) {
  simd_uchar4 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4_undef(simd_uchar3 other) {
  simd_uchar4 result;
  result.xyz = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4(simd_uchar4 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of four 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4(simd_uchar8 other) {
  return other.xyzw;
}

/*! @abstract Truncates `other` to form a vector of four 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4(simd_uchar16 other) {
  return other.xyzw;
}

/*! @abstract Truncates `other` to form a vector of four 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4(simd_uchar32 other) {
  return other.xyzw;
}

/*! @abstract Truncates `other` to form a vector of four 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar4 simd_make_uchar4(simd_uchar64 other) {
  return other.xyzw;
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar8 simd_make_uchar8(simd_uchar4 lo, simd_uchar4 hi) {
  simd_uchar8 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar8 simd_make_uchar8(unsigned char other) {
  simd_uchar8 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar8 simd_make_uchar8_undef(unsigned char other) {
  simd_uchar8 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar8 simd_make_uchar8(simd_uchar2 other) {
  simd_uchar8 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar8 simd_make_uchar8_undef(simd_uchar2 other) {
  simd_uchar8 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar8 simd_make_uchar8(simd_uchar3 other) {
  simd_uchar8 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar8 simd_make_uchar8_undef(simd_uchar3 other) {
  simd_uchar8 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar8 simd_make_uchar8(simd_uchar4 other) {
  simd_uchar8 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar8 simd_make_uchar8_undef(simd_uchar4 other) {
  simd_uchar8 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_uchar8 simd_make_uchar8(simd_uchar8 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of eight 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar8 simd_make_uchar8(simd_uchar16 other) {
  return simd_make_uchar8(other.lo);
}

/*! @abstract Truncates `other` to form a vector of eight 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar8 simd_make_uchar8(simd_uchar32 other) {
  return simd_make_uchar8(other.lo);
}

/*! @abstract Truncates `other` to form a vector of eight 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar8 simd_make_uchar8(simd_uchar64 other) {
  return simd_make_uchar8(other.lo);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixteen 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar16 simd_make_uchar16(simd_uchar8 lo, simd_uchar8 hi) {
  simd_uchar16 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar16 simd_make_uchar16(unsigned char other) {
  simd_uchar16 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar16 simd_make_uchar16_undef(unsigned char other) {
  simd_uchar16 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar16 simd_make_uchar16(simd_uchar2 other) {
  simd_uchar16 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar16 simd_make_uchar16_undef(simd_uchar2 other) {
  simd_uchar16 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar16 simd_make_uchar16(simd_uchar3 other) {
  simd_uchar16 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar16 simd_make_uchar16_undef(simd_uchar3 other) {
  simd_uchar16 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar16 simd_make_uchar16(simd_uchar4 other) {
  simd_uchar16 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar16 simd_make_uchar16_undef(simd_uchar4 other) {
  simd_uchar16 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar16 simd_make_uchar16(simd_uchar8 other) {
  simd_uchar16 result = 0;
  result.lo = simd_make_uchar8(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar16 simd_make_uchar16_undef(simd_uchar8 other) {
  simd_uchar16 result;
  result.lo = simd_make_uchar8(other);
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_uchar16 simd_make_uchar16(simd_uchar16 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of sixteen 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar16 simd_make_uchar16(simd_uchar32 other) {
  return simd_make_uchar16(other.lo);
}

/*! @abstract Truncates `other` to form a vector of sixteen 8-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uchar16 simd_make_uchar16(simd_uchar64 other) {
  return simd_make_uchar16(other.lo);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of thirty-two
 *  8-bit unsigned integers.                                                  */
static inline SIMD_CFUNC simd_uchar32 simd_make_uchar32(simd_uchar16 lo, simd_uchar16 hi) {
  simd_uchar32 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar32 simd_make_uchar32(unsigned char other) {
  simd_uchar32 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar32 simd_make_uchar32_undef(unsigned char other) {
  simd_uchar32 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar32 simd_make_uchar32(simd_uchar2 other) {
  simd_uchar32 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar32 simd_make_uchar32_undef(simd_uchar2 other) {
  simd_uchar32 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar32 simd_make_uchar32(simd_uchar3 other) {
  simd_uchar32 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar32 simd_make_uchar32_undef(simd_uchar3 other) {
  simd_uchar32 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar32 simd_make_uchar32(simd_uchar4 other) {
  simd_uchar32 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar32 simd_make_uchar32_undef(simd_uchar4 other) {
  simd_uchar32 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar32 simd_make_uchar32(simd_uchar8 other) {
  simd_uchar32 result = 0;
  result.lo = simd_make_uchar16(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar32 simd_make_uchar32_undef(simd_uchar8 other) {
  simd_uchar32 result;
  result.lo = simd_make_uchar16(other);
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar32 simd_make_uchar32(simd_uchar16 other) {
  simd_uchar32 result = 0;
  result.lo = simd_make_uchar16(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar32 simd_make_uchar32_undef(simd_uchar16 other) {
  simd_uchar32 result;
  result.lo = simd_make_uchar16(other);
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_uchar32 simd_make_uchar32(simd_uchar32 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of thirty-two 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar32 simd_make_uchar32(simd_uchar64 other) {
  return simd_make_uchar32(other.lo);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixty-four
 *  8-bit unsigned integers.                                                  */
static inline SIMD_CFUNC simd_uchar64 simd_make_uchar64(simd_uchar32 lo, simd_uchar32 hi) {
  simd_uchar64 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixty-four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar64 simd_make_uchar64(unsigned char other) {
  simd_uchar64 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixty-four 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar64 simd_make_uchar64_undef(unsigned char other) {
  simd_uchar64 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixty-four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar64 simd_make_uchar64(simd_uchar2 other) {
  simd_uchar64 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixty-four 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar64 simd_make_uchar64_undef(simd_uchar2 other) {
  simd_uchar64 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixty-four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar64 simd_make_uchar64(simd_uchar3 other) {
  simd_uchar64 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixty-four 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar64 simd_make_uchar64_undef(simd_uchar3 other) {
  simd_uchar64 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixty-four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar64 simd_make_uchar64(simd_uchar4 other) {
  simd_uchar64 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixty-four 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar64 simd_make_uchar64_undef(simd_uchar4 other) {
  simd_uchar64 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixty-four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar64 simd_make_uchar64(simd_uchar8 other) {
  simd_uchar64 result = 0;
  result.lo = simd_make_uchar32(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixty-four 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar64 simd_make_uchar64_undef(simd_uchar8 other) {
  simd_uchar64 result;
  result.lo = simd_make_uchar32(other);
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixty-four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar64 simd_make_uchar64(simd_uchar16 other) {
  simd_uchar64 result = 0;
  result.lo = simd_make_uchar32(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixty-four 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar64 simd_make_uchar64_undef(simd_uchar16 other) {
  simd_uchar64 result;
  result.lo = simd_make_uchar32(other);
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixty-four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uchar64 simd_make_uchar64(simd_uchar32 other) {
  simd_uchar64 result = 0;
  result.lo = simd_make_uchar32(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixty-four 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uchar64 simd_make_uchar64_undef(simd_uchar32 other) {
  simd_uchar64 result;
  result.lo = simd_make_uchar32(other);
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_uchar64 simd_make_uchar64(simd_uchar64 other) {
  return other;
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short2 simd_make_short2(short x, short y) {
  simd_short2 result;
  result.x = x;
  result.y = y;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of two 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short2 simd_make_short2(short other) {
  simd_short2 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of two 16-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_short2 simd_make_short2_undef(short other) {
  simd_short2 result;
  result.x = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_short2 simd_make_short2(simd_short2 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of two 16-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_short2 simd_make_short2(simd_short3 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 16-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_short2 simd_make_short2(simd_short4 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 16-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_short2 simd_make_short2(simd_short8 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 16-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_short2 simd_make_short2(simd_short16 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 16-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_short2 simd_make_short2(simd_short32 other) {
  return other.xy;
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short3 simd_make_short3(short x, short y, short z) {
  simd_short3 result;
  result.x = x;
  result.y = y;
  result.z = z;
  return result;
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short3 simd_make_short3(short x, simd_short2 yz) {
  simd_short3 result;
  result.x = x;
  result.yz = yz;
  return result;
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short3 simd_make_short3(simd_short2 xy, short z) {
  simd_short3 result;
  result.xy = xy;
  result.z = z;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of three 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short3 simd_make_short3(short other) {
  simd_short3 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 16-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_short3 simd_make_short3_undef(short other) {
  simd_short3 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of three 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short3 simd_make_short3(simd_short2 other) {
  simd_short3 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 16-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_short3 simd_make_short3_undef(simd_short2 other) {
  simd_short3 result;
  result.xy = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_short3 simd_make_short3(simd_short3 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of three 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short3 simd_make_short3(simd_short4 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short3 simd_make_short3(simd_short8 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short3 simd_make_short3(simd_short16 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short3 simd_make_short3(simd_short32 other) {
  return other.xyz;
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  16-bit signed (twos-complement) integers.                                 */
static inline SIMD_CFUNC simd_short4 simd_make_short4(short x, short y, short z, short w) {
  simd_short4 result;
  result.x = x;
  result.y = y;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short4 simd_make_short4(short x, short y, simd_short2 zw) {
  simd_short4 result;
  result.x = x;
  result.y = y;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short4 simd_make_short4(short x, simd_short2 yz, short w) {
  simd_short4 result;
  result.x = x;
  result.yz = yz;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short4 simd_make_short4(simd_short2 xy, short z, short w) {
  simd_short4 result;
  result.xy = xy;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short4 simd_make_short4(short x, simd_short3 yzw) {
  simd_short4 result;
  result.x = x;
  result.yzw = yzw;
  return result;
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short4 simd_make_short4(simd_short2 xy, simd_short2 zw) {
  simd_short4 result;
  result.xy = xy;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short4 simd_make_short4(simd_short3 xyz, short w) {
  simd_short4 result;
  result.xyz = xyz;
  result.w = w;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of four 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short4 simd_make_short4(short other) {
  simd_short4 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 16-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_short4 simd_make_short4_undef(short other) {
  simd_short4 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short4 simd_make_short4(simd_short2 other) {
  simd_short4 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 16-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_short4 simd_make_short4_undef(simd_short2 other) {
  simd_short4 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short4 simd_make_short4(simd_short3 other) {
  simd_short4 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 16-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_short4 simd_make_short4_undef(simd_short3 other) {
  simd_short4 result;
  result.xyz = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_short4 simd_make_short4(simd_short4 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of four 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short4 simd_make_short4(simd_short8 other) {
  return other.xyzw;
}

/*! @abstract Truncates `other` to form a vector of four 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short4 simd_make_short4(simd_short16 other) {
  return other.xyzw;
}

/*! @abstract Truncates `other` to form a vector of four 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short4 simd_make_short4(simd_short32 other) {
  return other.xyzw;
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short8 simd_make_short8(simd_short4 lo, simd_short4 hi) {
  simd_short8 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short8 simd_make_short8(short other) {
  simd_short8 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 16-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_short8 simd_make_short8_undef(short other) {
  simd_short8 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short8 simd_make_short8(simd_short2 other) {
  simd_short8 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 16-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_short8 simd_make_short8_undef(simd_short2 other) {
  simd_short8 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short8 simd_make_short8(simd_short3 other) {
  simd_short8 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 16-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_short8 simd_make_short8_undef(simd_short3 other) {
  simd_short8 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short8 simd_make_short8(simd_short4 other) {
  simd_short8 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 16-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_short8 simd_make_short8_undef(simd_short4 other) {
  simd_short8 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_short8 simd_make_short8(simd_short8 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of eight 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short8 simd_make_short8(simd_short16 other) {
  return simd_make_short8(other.lo);
}

/*! @abstract Truncates `other` to form a vector of eight 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short8 simd_make_short8(simd_short32 other) {
  return simd_make_short8(other.lo);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixteen 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short16 simd_make_short16(simd_short8 lo, simd_short8 hi) {
  simd_short16 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short16 simd_make_short16(short other) {
  simd_short16 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 16-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_short16 simd_make_short16_undef(short other) {
  simd_short16 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short16 simd_make_short16(simd_short2 other) {
  simd_short16 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 16-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_short16 simd_make_short16_undef(simd_short2 other) {
  simd_short16 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short16 simd_make_short16(simd_short3 other) {
  simd_short16 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 16-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_short16 simd_make_short16_undef(simd_short3 other) {
  simd_short16 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short16 simd_make_short16(simd_short4 other) {
  simd_short16 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 16-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_short16 simd_make_short16_undef(simd_short4 other) {
  simd_short16 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short16 simd_make_short16(simd_short8 other) {
  simd_short16 result = 0;
  result.lo = simd_make_short8(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 16-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_short16 simd_make_short16_undef(simd_short8 other) {
  simd_short16 result;
  result.lo = simd_make_short8(other);
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_short16 simd_make_short16(simd_short16 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of sixteen 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_short16 simd_make_short16(simd_short32 other) {
  return simd_make_short16(other.lo);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of thirty-two
 *  16-bit signed (twos-complement) integers.                                 */
static inline SIMD_CFUNC simd_short32 simd_make_short32(simd_short16 lo, simd_short16 hi) {
  simd_short32 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short32 simd_make_short32(short other) {
  simd_short32 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 16-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_short32 simd_make_short32_undef(short other) {
  simd_short32 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short32 simd_make_short32(simd_short2 other) {
  simd_short32 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 16-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_short32 simd_make_short32_undef(simd_short2 other) {
  simd_short32 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short32 simd_make_short32(simd_short3 other) {
  simd_short32 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 16-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_short32 simd_make_short32_undef(simd_short3 other) {
  simd_short32 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short32 simd_make_short32(simd_short4 other) {
  simd_short32 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 16-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_short32 simd_make_short32_undef(simd_short4 other) {
  simd_short32 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short32 simd_make_short32(simd_short8 other) {
  simd_short32 result = 0;
  result.lo = simd_make_short16(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 16-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_short32 simd_make_short32_undef(simd_short8 other) {
  simd_short32 result;
  result.lo = simd_make_short16(other);
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_short32 simd_make_short32(simd_short16 other) {
  simd_short32 result = 0;
  result.lo = simd_make_short16(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 16-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_short32 simd_make_short32_undef(simd_short16 other) {
  simd_short32 result;
  result.lo = simd_make_short16(other);
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_short32 simd_make_short32(simd_short32 other) {
  return other;
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort2 simd_make_ushort2(unsigned short x, unsigned short y) {
  simd_ushort2 result;
  result.x = x;
  result.y = y;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of two 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort2 simd_make_ushort2(unsigned short other) {
  simd_ushort2 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of two 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort2 simd_make_ushort2_undef(unsigned short other) {
  simd_ushort2 result;
  result.x = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_ushort2 simd_make_ushort2(simd_ushort2 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of two 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort2 simd_make_ushort2(simd_ushort3 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort2 simd_make_ushort2(simd_ushort4 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort2 simd_make_ushort2(simd_ushort8 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort2 simd_make_ushort2(simd_ushort16 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort2 simd_make_ushort2(simd_ushort32 other) {
  return other.xy;
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort3 simd_make_ushort3(unsigned short x, unsigned short y, unsigned short z) {
  simd_ushort3 result;
  result.x = x;
  result.y = y;
  result.z = z;
  return result;
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort3 simd_make_ushort3(unsigned short x, simd_ushort2 yz) {
  simd_ushort3 result;
  result.x = x;
  result.yz = yz;
  return result;
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort3 simd_make_ushort3(simd_ushort2 xy, unsigned short z) {
  simd_ushort3 result;
  result.xy = xy;
  result.z = z;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of three 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort3 simd_make_ushort3(unsigned short other) {
  simd_ushort3 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort3 simd_make_ushort3_undef(unsigned short other) {
  simd_ushort3 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of three 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort3 simd_make_ushort3(simd_ushort2 other) {
  simd_ushort3 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort3 simd_make_ushort3_undef(simd_ushort2 other) {
  simd_ushort3 result;
  result.xy = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_ushort3 simd_make_ushort3(simd_ushort3 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of three 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort3 simd_make_ushort3(simd_ushort4 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort3 simd_make_ushort3(simd_ushort8 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort3 simd_make_ushort3(simd_ushort16 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort3 simd_make_ushort3(simd_ushort32 other) {
  return other.xyz;
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  16-bit unsigned integers.                                                 */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4(unsigned short x, unsigned short y, unsigned short z, unsigned short w) {
  simd_ushort4 result;
  result.x = x;
  result.y = y;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4(unsigned short x, unsigned short y, simd_ushort2 zw) {
  simd_ushort4 result;
  result.x = x;
  result.y = y;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4(unsigned short x, simd_ushort2 yz, unsigned short w) {
  simd_ushort4 result;
  result.x = x;
  result.yz = yz;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4(simd_ushort2 xy, unsigned short z, unsigned short w) {
  simd_ushort4 result;
  result.xy = xy;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4(unsigned short x, simd_ushort3 yzw) {
  simd_ushort4 result;
  result.x = x;
  result.yzw = yzw;
  return result;
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4(simd_ushort2 xy, simd_ushort2 zw) {
  simd_ushort4 result;
  result.xy = xy;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4(simd_ushort3 xyz, unsigned short w) {
  simd_ushort4 result;
  result.xyz = xyz;
  result.w = w;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of four 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4(unsigned short other) {
  simd_ushort4 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4_undef(unsigned short other) {
  simd_ushort4 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4(simd_ushort2 other) {
  simd_ushort4 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4_undef(simd_ushort2 other) {
  simd_ushort4 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4(simd_ushort3 other) {
  simd_ushort4 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4_undef(simd_ushort3 other) {
  simd_ushort4 result;
  result.xyz = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4(simd_ushort4 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of four 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4(simd_ushort8 other) {
  return other.xyzw;
}

/*! @abstract Truncates `other` to form a vector of four 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4(simd_ushort16 other) {
  return other.xyzw;
}

/*! @abstract Truncates `other` to form a vector of four 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort4 simd_make_ushort4(simd_ushort32 other) {
  return other.xyzw;
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort8 simd_make_ushort8(simd_ushort4 lo, simd_ushort4 hi) {
  simd_ushort8 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort8 simd_make_ushort8(unsigned short other) {
  simd_ushort8 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort8 simd_make_ushort8_undef(unsigned short other) {
  simd_ushort8 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort8 simd_make_ushort8(simd_ushort2 other) {
  simd_ushort8 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort8 simd_make_ushort8_undef(simd_ushort2 other) {
  simd_ushort8 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort8 simd_make_ushort8(simd_ushort3 other) {
  simd_ushort8 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort8 simd_make_ushort8_undef(simd_ushort3 other) {
  simd_ushort8 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort8 simd_make_ushort8(simd_ushort4 other) {
  simd_ushort8 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort8 simd_make_ushort8_undef(simd_ushort4 other) {
  simd_ushort8 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_ushort8 simd_make_ushort8(simd_ushort8 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of eight 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort8 simd_make_ushort8(simd_ushort16 other) {
  return simd_make_ushort8(other.lo);
}

/*! @abstract Truncates `other` to form a vector of eight 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort8 simd_make_ushort8(simd_ushort32 other) {
  return simd_make_ushort8(other.lo);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixteen 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort16 simd_make_ushort16(simd_ushort8 lo, simd_ushort8 hi) {
  simd_ushort16 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort16 simd_make_ushort16(unsigned short other) {
  simd_ushort16 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort16 simd_make_ushort16_undef(unsigned short other) {
  simd_ushort16 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort16 simd_make_ushort16(simd_ushort2 other) {
  simd_ushort16 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort16 simd_make_ushort16_undef(simd_ushort2 other) {
  simd_ushort16 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort16 simd_make_ushort16(simd_ushort3 other) {
  simd_ushort16 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort16 simd_make_ushort16_undef(simd_ushort3 other) {
  simd_ushort16 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort16 simd_make_ushort16(simd_ushort4 other) {
  simd_ushort16 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort16 simd_make_ushort16_undef(simd_ushort4 other) {
  simd_ushort16 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort16 simd_make_ushort16(simd_ushort8 other) {
  simd_ushort16 result = 0;
  result.lo = simd_make_ushort8(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort16 simd_make_ushort16_undef(simd_ushort8 other) {
  simd_ushort16 result;
  result.lo = simd_make_ushort8(other);
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_ushort16 simd_make_ushort16(simd_ushort16 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of sixteen 16-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ushort16 simd_make_ushort16(simd_ushort32 other) {
  return simd_make_ushort16(other.lo);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of thirty-two
 *  16-bit unsigned integers.                                                 */
static inline SIMD_CFUNC simd_ushort32 simd_make_ushort32(simd_ushort16 lo, simd_ushort16 hi) {
  simd_ushort32 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort32 simd_make_ushort32(unsigned short other) {
  simd_ushort32 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort32 simd_make_ushort32_undef(unsigned short other) {
  simd_ushort32 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort32 simd_make_ushort32(simd_ushort2 other) {
  simd_ushort32 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort32 simd_make_ushort32_undef(simd_ushort2 other) {
  simd_ushort32 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort32 simd_make_ushort32(simd_ushort3 other) {
  simd_ushort32 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort32 simd_make_ushort32_undef(simd_ushort3 other) {
  simd_ushort32 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort32 simd_make_ushort32(simd_ushort4 other) {
  simd_ushort32 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort32 simd_make_ushort32_undef(simd_ushort4 other) {
  simd_ushort32 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort32 simd_make_ushort32(simd_ushort8 other) {
  simd_ushort32 result = 0;
  result.lo = simd_make_ushort16(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort32 simd_make_ushort32_undef(simd_ushort8 other) {
  simd_ushort32 result;
  result.lo = simd_make_ushort16(other);
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of thirty-two 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ushort32 simd_make_ushort32(simd_ushort16 other) {
  simd_ushort32 result = 0;
  result.lo = simd_make_ushort16(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ushort32 simd_make_ushort32_undef(simd_ushort16 other) {
  simd_ushort32 result;
  result.lo = simd_make_ushort16(other);
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_ushort32 simd_make_ushort32(simd_ushort32 other) {
  return other;
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int2 simd_make_int2(int x, int y) {
  simd_int2 result;
  result.x = x;
  result.y = y;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of two 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int2 simd_make_int2(int other) {
  simd_int2 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of two 32-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_int2 simd_make_int2_undef(int other) {
  simd_int2 result;
  result.x = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_int2 simd_make_int2(simd_int2 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of two 32-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_int2 simd_make_int2(simd_int3 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 32-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_int2 simd_make_int2(simd_int4 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 32-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_int2 simd_make_int2(simd_int8 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 32-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_int2 simd_make_int2(simd_int16 other) {
  return other.xy;
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_int3 simd_make_int3(int x, int y, int z) {
  simd_int3 result;
  result.x = x;
  result.y = y;
  result.z = z;
  return result;
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_int3 simd_make_int3(int x, simd_int2 yz) {
  simd_int3 result;
  result.x = x;
  result.yz = yz;
  return result;
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_int3 simd_make_int3(simd_int2 xy, int z) {
  simd_int3 result;
  result.xy = xy;
  result.z = z;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of three 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int3 simd_make_int3(int other) {
  simd_int3 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 32-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_int3 simd_make_int3_undef(int other) {
  simd_int3 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of three 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int3 simd_make_int3(simd_int2 other) {
  simd_int3 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 32-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_int3 simd_make_int3_undef(simd_int2 other) {
  simd_int3 result;
  result.xy = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_int3 simd_make_int3(simd_int3 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of three 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int3 simd_make_int3(simd_int4 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int3 simd_make_int3(simd_int8 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int3 simd_make_int3(simd_int16 other) {
  return other.xyz;
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  32-bit signed (twos-complement) integers.                                 */
static inline SIMD_CFUNC simd_int4 simd_make_int4(int x, int y, int z, int w) {
  simd_int4 result;
  result.x = x;
  result.y = y;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_int4 simd_make_int4(int x, int y, simd_int2 zw) {
  simd_int4 result;
  result.x = x;
  result.y = y;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_int4 simd_make_int4(int x, simd_int2 yz, int w) {
  simd_int4 result;
  result.x = x;
  result.yz = yz;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_int4 simd_make_int4(simd_int2 xy, int z, int w) {
  simd_int4 result;
  result.xy = xy;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_int4 simd_make_int4(int x, simd_int3 yzw) {
  simd_int4 result;
  result.x = x;
  result.yzw = yzw;
  return result;
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_int4 simd_make_int4(simd_int2 xy, simd_int2 zw) {
  simd_int4 result;
  result.xy = xy;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_int4 simd_make_int4(simd_int3 xyz, int w) {
  simd_int4 result;
  result.xyz = xyz;
  result.w = w;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of four 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int4 simd_make_int4(int other) {
  simd_int4 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 32-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_int4 simd_make_int4_undef(int other) {
  simd_int4 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int4 simd_make_int4(simd_int2 other) {
  simd_int4 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 32-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_int4 simd_make_int4_undef(simd_int2 other) {
  simd_int4 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int4 simd_make_int4(simd_int3 other) {
  simd_int4 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 32-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_int4 simd_make_int4_undef(simd_int3 other) {
  simd_int4 result;
  result.xyz = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_int4 simd_make_int4(simd_int4 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of four 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int4 simd_make_int4(simd_int8 other) {
  return other.xyzw;
}

/*! @abstract Truncates `other` to form a vector of four 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int4 simd_make_int4(simd_int16 other) {
  return other.xyzw;
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_int8 simd_make_int8(simd_int4 lo, simd_int4 hi) {
  simd_int8 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int8 simd_make_int8(int other) {
  simd_int8 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 32-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_int8 simd_make_int8_undef(int other) {
  simd_int8 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int8 simd_make_int8(simd_int2 other) {
  simd_int8 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 32-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_int8 simd_make_int8_undef(simd_int2 other) {
  simd_int8 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int8 simd_make_int8(simd_int3 other) {
  simd_int8 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 32-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_int8 simd_make_int8_undef(simd_int3 other) {
  simd_int8 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int8 simd_make_int8(simd_int4 other) {
  simd_int8 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 32-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_int8 simd_make_int8_undef(simd_int4 other) {
  simd_int8 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_int8 simd_make_int8(simd_int8 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of eight 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int8 simd_make_int8(simd_int16 other) {
  return simd_make_int8(other.lo);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixteen 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_int16 simd_make_int16(simd_int8 lo, simd_int8 hi) {
  simd_int16 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int16 simd_make_int16(int other) {
  simd_int16 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_int16 simd_make_int16_undef(int other) {
  simd_int16 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int16 simd_make_int16(simd_int2 other) {
  simd_int16 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_int16 simd_make_int16_undef(simd_int2 other) {
  simd_int16 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int16 simd_make_int16(simd_int3 other) {
  simd_int16 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_int16 simd_make_int16_undef(simd_int3 other) {
  simd_int16 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int16 simd_make_int16(simd_int4 other) {
  simd_int16 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_int16 simd_make_int16_undef(simd_int4 other) {
  simd_int16 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_int16 simd_make_int16(simd_int8 other) {
  simd_int16 result = 0;
  result.lo = simd_make_int8(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
static inline SIMD_CFUNC simd_int16 simd_make_int16_undef(simd_int8 other) {
  simd_int16 result;
  result.lo = simd_make_int8(other);
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_int16 simd_make_int16(simd_int16 other) {
  return other;
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint2 simd_make_uint2(unsigned int x, unsigned int y) {
  simd_uint2 result;
  result.x = x;
  result.y = y;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of two 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint2 simd_make_uint2(unsigned int other) {
  simd_uint2 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of two 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uint2 simd_make_uint2_undef(unsigned int other) {
  simd_uint2 result;
  result.x = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_uint2 simd_make_uint2(simd_uint2 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of two 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint2 simd_make_uint2(simd_uint3 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint2 simd_make_uint2(simd_uint4 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint2 simd_make_uint2(simd_uint8 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint2 simd_make_uint2(simd_uint16 other) {
  return other.xy;
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint3 simd_make_uint3(unsigned int x, unsigned int y, unsigned int z) {
  simd_uint3 result;
  result.x = x;
  result.y = y;
  result.z = z;
  return result;
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint3 simd_make_uint3(unsigned int x, simd_uint2 yz) {
  simd_uint3 result;
  result.x = x;
  result.yz = yz;
  return result;
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint3 simd_make_uint3(simd_uint2 xy, unsigned int z) {
  simd_uint3 result;
  result.xy = xy;
  result.z = z;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of three 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint3 simd_make_uint3(unsigned int other) {
  simd_uint3 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uint3 simd_make_uint3_undef(unsigned int other) {
  simd_uint3 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of three 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint3 simd_make_uint3(simd_uint2 other) {
  simd_uint3 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uint3 simd_make_uint3_undef(simd_uint2 other) {
  simd_uint3 result;
  result.xy = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_uint3 simd_make_uint3(simd_uint3 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of three 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint3 simd_make_uint3(simd_uint4 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint3 simd_make_uint3(simd_uint8 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint3 simd_make_uint3(simd_uint16 other) {
  return other.xyz;
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  32-bit unsigned integers.                                                 */
static inline SIMD_CFUNC simd_uint4 simd_make_uint4(unsigned int x, unsigned int y, unsigned int z, unsigned int w) {
  simd_uint4 result;
  result.x = x;
  result.y = y;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint4 simd_make_uint4(unsigned int x, unsigned int y, simd_uint2 zw) {
  simd_uint4 result;
  result.x = x;
  result.y = y;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint4 simd_make_uint4(unsigned int x, simd_uint2 yz, unsigned int w) {
  simd_uint4 result;
  result.x = x;
  result.yz = yz;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint4 simd_make_uint4(simd_uint2 xy, unsigned int z, unsigned int w) {
  simd_uint4 result;
  result.xy = xy;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint4 simd_make_uint4(unsigned int x, simd_uint3 yzw) {
  simd_uint4 result;
  result.x = x;
  result.yzw = yzw;
  return result;
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint4 simd_make_uint4(simd_uint2 xy, simd_uint2 zw) {
  simd_uint4 result;
  result.xy = xy;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint4 simd_make_uint4(simd_uint3 xyz, unsigned int w) {
  simd_uint4 result;
  result.xyz = xyz;
  result.w = w;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of four 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint4 simd_make_uint4(unsigned int other) {
  simd_uint4 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uint4 simd_make_uint4_undef(unsigned int other) {
  simd_uint4 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint4 simd_make_uint4(simd_uint2 other) {
  simd_uint4 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uint4 simd_make_uint4_undef(simd_uint2 other) {
  simd_uint4 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint4 simd_make_uint4(simd_uint3 other) {
  simd_uint4 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uint4 simd_make_uint4_undef(simd_uint3 other) {
  simd_uint4 result;
  result.xyz = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_uint4 simd_make_uint4(simd_uint4 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of four 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint4 simd_make_uint4(simd_uint8 other) {
  return other.xyzw;
}

/*! @abstract Truncates `other` to form a vector of four 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint4 simd_make_uint4(simd_uint16 other) {
  return other.xyzw;
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint8 simd_make_uint8(simd_uint4 lo, simd_uint4 hi) {
  simd_uint8 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint8 simd_make_uint8(unsigned int other) {
  simd_uint8 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uint8 simd_make_uint8_undef(unsigned int other) {
  simd_uint8 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint8 simd_make_uint8(simd_uint2 other) {
  simd_uint8 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uint8 simd_make_uint8_undef(simd_uint2 other) {
  simd_uint8 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint8 simd_make_uint8(simd_uint3 other) {
  simd_uint8 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uint8 simd_make_uint8_undef(simd_uint3 other) {
  simd_uint8 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint8 simd_make_uint8(simd_uint4 other) {
  simd_uint8 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uint8 simd_make_uint8_undef(simd_uint4 other) {
  simd_uint8 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_uint8 simd_make_uint8(simd_uint8 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of eight 32-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_uint8 simd_make_uint8(simd_uint16 other) {
  return simd_make_uint8(other.lo);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixteen 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint16 simd_make_uint16(simd_uint8 lo, simd_uint8 hi) {
  simd_uint16 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint16 simd_make_uint16(unsigned int other) {
  simd_uint16 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uint16 simd_make_uint16_undef(unsigned int other) {
  simd_uint16 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint16 simd_make_uint16(simd_uint2 other) {
  simd_uint16 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uint16 simd_make_uint16_undef(simd_uint2 other) {
  simd_uint16 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint16 simd_make_uint16(simd_uint3 other) {
  simd_uint16 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uint16 simd_make_uint16_undef(simd_uint3 other) {
  simd_uint16 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint16 simd_make_uint16(simd_uint4 other) {
  simd_uint16 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uint16 simd_make_uint16_undef(simd_uint4 other) {
  simd_uint16 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_uint16 simd_make_uint16(simd_uint8 other) {
  simd_uint16 result = 0;
  result.lo = simd_make_uint8(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_uint16 simd_make_uint16_undef(simd_uint8 other) {
  simd_uint16 result;
  result.lo = simd_make_uint8(other);
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_uint16 simd_make_uint16(simd_uint16 other) {
  return other;
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float2 simd_make_float2(float x, float y) {
  simd_float2 result;
  result.x = x;
  result.y = y;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of two 32-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_float2 simd_make_float2(float other) {
  simd_float2 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of two 32-bit floating-point
 *  numbers. The contents of the newly-created vector lanes are unspecified.  */
static inline SIMD_CFUNC simd_float2 simd_make_float2_undef(float other) {
  simd_float2 result;
  result.x = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_float2 simd_make_float2(simd_float2 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of two 32-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_float2 simd_make_float2(simd_float3 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 32-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_float2 simd_make_float2(simd_float4 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 32-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_float2 simd_make_float2(simd_float8 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 32-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_float2 simd_make_float2(simd_float16 other) {
  return other.xy;
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float3 simd_make_float3(float x, float y, float z) {
  simd_float3 result;
  result.x = x;
  result.y = y;
  result.z = z;
  return result;
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float3 simd_make_float3(float x, simd_float2 yz) {
  simd_float3 result;
  result.x = x;
  result.yz = yz;
  return result;
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float3 simd_make_float3(simd_float2 xy, float z) {
  simd_float3 result;
  result.xy = xy;
  result.z = z;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of three 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float3 simd_make_float3(float other) {
  simd_float3 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 32-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_float3 simd_make_float3_undef(float other) {
  simd_float3 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of three 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float3 simd_make_float3(simd_float2 other) {
  simd_float3 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 32-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_float3 simd_make_float3_undef(simd_float2 other) {
  simd_float3 result;
  result.xy = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_float3 simd_make_float3(simd_float3 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of three 32-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_float3 simd_make_float3(simd_float4 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 32-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_float3 simd_make_float3(simd_float8 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 32-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_float3 simd_make_float3(simd_float16 other) {
  return other.xyz;
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  32-bit floating-point numbers.                                            */
static inline SIMD_CFUNC simd_float4 simd_make_float4(float x, float y, float z, float w) {
  simd_float4 result;
  result.x = x;
  result.y = y;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float4 simd_make_float4(float x, float y, simd_float2 zw) {
  simd_float4 result;
  result.x = x;
  result.y = y;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float4 simd_make_float4(float x, simd_float2 yz, float w) {
  simd_float4 result;
  result.x = x;
  result.yz = yz;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float4 simd_make_float4(simd_float2 xy, float z, float w) {
  simd_float4 result;
  result.xy = xy;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float4 simd_make_float4(float x, simd_float3 yzw) {
  simd_float4 result;
  result.x = x;
  result.yzw = yzw;
  return result;
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float4 simd_make_float4(simd_float2 xy, simd_float2 zw) {
  simd_float4 result;
  result.xy = xy;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float4 simd_make_float4(simd_float3 xyz, float w) {
  simd_float4 result;
  result.xyz = xyz;
  result.w = w;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of four 32-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_float4 simd_make_float4(float other) {
  simd_float4 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 32-bit floating-point
 *  numbers. The contents of the newly-created vector lanes are unspecified.  */
static inline SIMD_CFUNC simd_float4 simd_make_float4_undef(float other) {
  simd_float4 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 32-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_float4 simd_make_float4(simd_float2 other) {
  simd_float4 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 32-bit floating-point
 *  numbers. The contents of the newly-created vector lanes are unspecified.  */
static inline SIMD_CFUNC simd_float4 simd_make_float4_undef(simd_float2 other) {
  simd_float4 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 32-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_float4 simd_make_float4(simd_float3 other) {
  simd_float4 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 32-bit floating-point
 *  numbers. The contents of the newly-created vector lanes are unspecified.  */
static inline SIMD_CFUNC simd_float4 simd_make_float4_undef(simd_float3 other) {
  simd_float4 result;
  result.xyz = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_float4 simd_make_float4(simd_float4 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of four 32-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_float4 simd_make_float4(simd_float8 other) {
  return other.xyzw;
}

/*! @abstract Truncates `other` to form a vector of four 32-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_float4 simd_make_float4(simd_float16 other) {
  return other.xyzw;
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float8 simd_make_float8(simd_float4 lo, simd_float4 hi) {
  simd_float8 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float8 simd_make_float8(float other) {
  simd_float8 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 32-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_float8 simd_make_float8_undef(float other) {
  simd_float8 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float8 simd_make_float8(simd_float2 other) {
  simd_float8 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 32-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_float8 simd_make_float8_undef(simd_float2 other) {
  simd_float8 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float8 simd_make_float8(simd_float3 other) {
  simd_float8 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 32-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_float8 simd_make_float8_undef(simd_float3 other) {
  simd_float8 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float8 simd_make_float8(simd_float4 other) {
  simd_float8 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 32-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_float8 simd_make_float8_undef(simd_float4 other) {
  simd_float8 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_float8 simd_make_float8(simd_float8 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of eight 32-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_float8 simd_make_float8(simd_float16 other) {
  return simd_make_float8(other.lo);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixteen 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float16 simd_make_float16(simd_float8 lo, simd_float8 hi) {
  simd_float16 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float16 simd_make_float16(float other) {
  simd_float16 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_float16 simd_make_float16_undef(float other) {
  simd_float16 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float16 simd_make_float16(simd_float2 other) {
  simd_float16 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_float16 simd_make_float16_undef(simd_float2 other) {
  simd_float16 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float16 simd_make_float16(simd_float3 other) {
  simd_float16 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_float16 simd_make_float16_undef(simd_float3 other) {
  simd_float16 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float16 simd_make_float16(simd_float4 other) {
  simd_float16 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_float16 simd_make_float16_undef(simd_float4 other) {
  simd_float16 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of sixteen 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_float16 simd_make_float16(simd_float8 other) {
  simd_float16 result = 0;
  result.lo = simd_make_float8(other);
  return result;
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_float16 simd_make_float16_undef(simd_float8 other) {
  simd_float16 result;
  result.lo = simd_make_float8(other);
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_float16 simd_make_float16(simd_float16 other) {
  return other;
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 64-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_long2 simd_make_long2(simd_long1 x, simd_long1 y) {
  simd_long2 result;
  result.x = x;
  result.y = y;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of two 64-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_long2 simd_make_long2(simd_long1 other) {
  simd_long2 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of two 64-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_long2 simd_make_long2_undef(simd_long1 other) {
  simd_long2 result;
  result.x = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_long2 simd_make_long2(simd_long2 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of two 64-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_long2 simd_make_long2(simd_long3 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 64-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_long2 simd_make_long2(simd_long4 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 64-bit signed (twos-
 *  complement) integers.                                                     */
static inline SIMD_CFUNC simd_long2 simd_make_long2(simd_long8 other) {
  return other.xy;
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_long3 simd_make_long3(simd_long1 x, simd_long1 y, simd_long1 z) {
  simd_long3 result;
  result.x = x;
  result.y = y;
  result.z = z;
  return result;
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_long3 simd_make_long3(simd_long1 x, simd_long2 yz) {
  simd_long3 result;
  result.x = x;
  result.yz = yz;
  return result;
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_long3 simd_make_long3(simd_long2 xy, simd_long1 z) {
  simd_long3 result;
  result.xy = xy;
  result.z = z;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of three 64-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_long3 simd_make_long3(simd_long1 other) {
  simd_long3 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 64-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_long3 simd_make_long3_undef(simd_long1 other) {
  simd_long3 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of three 64-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_long3 simd_make_long3(simd_long2 other) {
  simd_long3 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 64-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_long3 simd_make_long3_undef(simd_long2 other) {
  simd_long3 result;
  result.xy = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_long3 simd_make_long3(simd_long3 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of three 64-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_long3 simd_make_long3(simd_long4 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 64-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_long3 simd_make_long3(simd_long8 other) {
  return other.xyz;
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  64-bit signed (twos-complement) integers.                                 */
static inline SIMD_CFUNC simd_long4 simd_make_long4(simd_long1 x, simd_long1 y, simd_long1 z, simd_long1 w) {
  simd_long4 result;
  result.x = x;
  result.y = y;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_long4 simd_make_long4(simd_long1 x, simd_long1 y, simd_long2 zw) {
  simd_long4 result;
  result.x = x;
  result.y = y;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_long4 simd_make_long4(simd_long1 x, simd_long2 yz, simd_long1 w) {
  simd_long4 result;
  result.x = x;
  result.yz = yz;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_long4 simd_make_long4(simd_long2 xy, simd_long1 z, simd_long1 w) {
  simd_long4 result;
  result.xy = xy;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_long4 simd_make_long4(simd_long1 x, simd_long3 yzw) {
  simd_long4 result;
  result.x = x;
  result.yzw = yzw;
  return result;
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_long4 simd_make_long4(simd_long2 xy, simd_long2 zw) {
  simd_long4 result;
  result.xy = xy;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_long4 simd_make_long4(simd_long3 xyz, simd_long1 w) {
  simd_long4 result;
  result.xyz = xyz;
  result.w = w;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of four 64-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_long4 simd_make_long4(simd_long1 other) {
  simd_long4 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 64-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_long4 simd_make_long4_undef(simd_long1 other) {
  simd_long4 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 64-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_long4 simd_make_long4(simd_long2 other) {
  simd_long4 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 64-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_long4 simd_make_long4_undef(simd_long2 other) {
  simd_long4 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 64-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_long4 simd_make_long4(simd_long3 other) {
  simd_long4 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 64-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_long4 simd_make_long4_undef(simd_long3 other) {
  simd_long4 result;
  result.xyz = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_long4 simd_make_long4(simd_long4 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of four 64-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_long4 simd_make_long4(simd_long8 other) {
  return other.xyzw;
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CFUNC simd_long8 simd_make_long8(simd_long4 lo, simd_long4 hi) {
  simd_long8 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 64-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_long8 simd_make_long8(simd_long1 other) {
  simd_long8 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 64-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_long8 simd_make_long8_undef(simd_long1 other) {
  simd_long8 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 64-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_long8 simd_make_long8(simd_long2 other) {
  simd_long8 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 64-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_long8 simd_make_long8_undef(simd_long2 other) {
  simd_long8 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 64-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_long8 simd_make_long8(simd_long3 other) {
  simd_long8 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 64-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_long8 simd_make_long8_undef(simd_long3 other) {
  simd_long8 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 64-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CFUNC simd_long8 simd_make_long8(simd_long4 other) {
  simd_long8 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 64-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_long8 simd_make_long8_undef(simd_long4 other) {
  simd_long8 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_long8 simd_make_long8(simd_long8 other) {
  return other;
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ulong2 simd_make_ulong2(simd_ulong1 x, simd_ulong1 y) {
  simd_ulong2 result;
  result.x = x;
  result.y = y;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of two 64-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ulong2 simd_make_ulong2(simd_ulong1 other) {
  simd_ulong2 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of two 64-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ulong2 simd_make_ulong2_undef(simd_ulong1 other) {
  simd_ulong2 result;
  result.x = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_ulong2 simd_make_ulong2(simd_ulong2 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of two 64-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ulong2 simd_make_ulong2(simd_ulong3 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 64-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ulong2 simd_make_ulong2(simd_ulong4 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 64-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ulong2 simd_make_ulong2(simd_ulong8 other) {
  return other.xy;
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ulong3 simd_make_ulong3(simd_ulong1 x, simd_ulong1 y, simd_ulong1 z) {
  simd_ulong3 result;
  result.x = x;
  result.y = y;
  result.z = z;
  return result;
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ulong3 simd_make_ulong3(simd_ulong1 x, simd_ulong2 yz) {
  simd_ulong3 result;
  result.x = x;
  result.yz = yz;
  return result;
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ulong3 simd_make_ulong3(simd_ulong2 xy, simd_ulong1 z) {
  simd_ulong3 result;
  result.xy = xy;
  result.z = z;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of three 64-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ulong3 simd_make_ulong3(simd_ulong1 other) {
  simd_ulong3 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 64-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ulong3 simd_make_ulong3_undef(simd_ulong1 other) {
  simd_ulong3 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of three 64-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ulong3 simd_make_ulong3(simd_ulong2 other) {
  simd_ulong3 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 64-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ulong3 simd_make_ulong3_undef(simd_ulong2 other) {
  simd_ulong3 result;
  result.xy = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_ulong3 simd_make_ulong3(simd_ulong3 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of three 64-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ulong3 simd_make_ulong3(simd_ulong4 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 64-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ulong3 simd_make_ulong3(simd_ulong8 other) {
  return other.xyz;
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  64-bit unsigned integers.                                                 */
static inline SIMD_CFUNC simd_ulong4 simd_make_ulong4(simd_ulong1 x, simd_ulong1 y, simd_ulong1 z, simd_ulong1 w) {
  simd_ulong4 result;
  result.x = x;
  result.y = y;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ulong4 simd_make_ulong4(simd_ulong1 x, simd_ulong1 y, simd_ulong2 zw) {
  simd_ulong4 result;
  result.x = x;
  result.y = y;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ulong4 simd_make_ulong4(simd_ulong1 x, simd_ulong2 yz, simd_ulong1 w) {
  simd_ulong4 result;
  result.x = x;
  result.yz = yz;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ulong4 simd_make_ulong4(simd_ulong2 xy, simd_ulong1 z, simd_ulong1 w) {
  simd_ulong4 result;
  result.xy = xy;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ulong4 simd_make_ulong4(simd_ulong1 x, simd_ulong3 yzw) {
  simd_ulong4 result;
  result.x = x;
  result.yzw = yzw;
  return result;
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ulong4 simd_make_ulong4(simd_ulong2 xy, simd_ulong2 zw) {
  simd_ulong4 result;
  result.xy = xy;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ulong4 simd_make_ulong4(simd_ulong3 xyz, simd_ulong1 w) {
  simd_ulong4 result;
  result.xyz = xyz;
  result.w = w;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of four 64-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ulong4 simd_make_ulong4(simd_ulong1 other) {
  simd_ulong4 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 64-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ulong4 simd_make_ulong4_undef(simd_ulong1 other) {
  simd_ulong4 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 64-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ulong4 simd_make_ulong4(simd_ulong2 other) {
  simd_ulong4 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 64-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ulong4 simd_make_ulong4_undef(simd_ulong2 other) {
  simd_ulong4 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 64-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ulong4 simd_make_ulong4(simd_ulong3 other) {
  simd_ulong4 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 64-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ulong4 simd_make_ulong4_undef(simd_ulong3 other) {
  simd_ulong4 result;
  result.xyz = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_ulong4 simd_make_ulong4(simd_ulong4 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of four 64-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ulong4 simd_make_ulong4(simd_ulong8 other) {
  return other.xyzw;
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CFUNC simd_ulong8 simd_make_ulong8(simd_ulong4 lo, simd_ulong4 hi) {
  simd_ulong8 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 64-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ulong8 simd_make_ulong8(simd_ulong1 other) {
  simd_ulong8 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 64-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ulong8 simd_make_ulong8_undef(simd_ulong1 other) {
  simd_ulong8 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 64-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ulong8 simd_make_ulong8(simd_ulong2 other) {
  simd_ulong8 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 64-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ulong8 simd_make_ulong8_undef(simd_ulong2 other) {
  simd_ulong8 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 64-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ulong8 simd_make_ulong8(simd_ulong3 other) {
  simd_ulong8 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 64-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ulong8 simd_make_ulong8_undef(simd_ulong3 other) {
  simd_ulong8 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 64-bit unsigned
 *  integers.                                                                 */
static inline SIMD_CFUNC simd_ulong8 simd_make_ulong8(simd_ulong4 other) {
  simd_ulong8 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 64-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_ulong8 simd_make_ulong8_undef(simd_ulong4 other) {
  simd_ulong8 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_ulong8 simd_make_ulong8(simd_ulong8 other) {
  return other;
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double2 simd_make_double2(double x, double y) {
  simd_double2 result;
  result.x = x;
  result.y = y;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of two 64-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_double2 simd_make_double2(double other) {
  simd_double2 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of two 64-bit floating-point
 *  numbers. The contents of the newly-created vector lanes are unspecified.  */
static inline SIMD_CFUNC simd_double2 simd_make_double2_undef(double other) {
  simd_double2 result;
  result.x = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_double2 simd_make_double2(simd_double2 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of two 64-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_double2 simd_make_double2(simd_double3 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 64-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_double2 simd_make_double2(simd_double4 other) {
  return other.xy;
}

/*! @abstract Truncates `other` to form a vector of two 64-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_double2 simd_make_double2(simd_double8 other) {
  return other.xy;
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double3 simd_make_double3(double x, double y, double z) {
  simd_double3 result;
  result.x = x;
  result.y = y;
  result.z = z;
  return result;
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double3 simd_make_double3(double x, simd_double2 yz) {
  simd_double3 result;
  result.x = x;
  result.yz = yz;
  return result;
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double3 simd_make_double3(simd_double2 xy, double z) {
  simd_double3 result;
  result.xy = xy;
  result.z = z;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of three 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double3 simd_make_double3(double other) {
  simd_double3 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 64-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_double3 simd_make_double3_undef(double other) {
  simd_double3 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of three 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double3 simd_make_double3(simd_double2 other) {
  simd_double3 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of three 64-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_double3 simd_make_double3_undef(simd_double2 other) {
  simd_double3 result;
  result.xy = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_double3 simd_make_double3(simd_double3 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of three 64-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_double3 simd_make_double3(simd_double4 other) {
  return other.xyz;
}

/*! @abstract Truncates `other` to form a vector of three 64-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_double3 simd_make_double3(simd_double8 other) {
  return other.xyz;
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  64-bit floating-point numbers.                                            */
static inline SIMD_CFUNC simd_double4 simd_make_double4(double x, double y, double z, double w) {
  simd_double4 result;
  result.x = x;
  result.y = y;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double4 simd_make_double4(double x, double y, simd_double2 zw) {
  simd_double4 result;
  result.x = x;
  result.y = y;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double4 simd_make_double4(double x, simd_double2 yz, double w) {
  simd_double4 result;
  result.x = x;
  result.yz = yz;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double4 simd_make_double4(simd_double2 xy, double z, double w) {
  simd_double4 result;
  result.xy = xy;
  result.z = z;
  result.w = w;
  return result;
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double4 simd_make_double4(double x, simd_double3 yzw) {
  simd_double4 result;
  result.x = x;
  result.yzw = yzw;
  return result;
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double4 simd_make_double4(simd_double2 xy, simd_double2 zw) {
  simd_double4 result;
  result.xy = xy;
  result.zw = zw;
  return result;
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double4 simd_make_double4(simd_double3 xyz, double w) {
  simd_double4 result;
  result.xyz = xyz;
  result.w = w;
  return result;
}
  
/*! @abstract Zero-extends `other` to form a vector of four 64-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_double4 simd_make_double4(double other) {
  simd_double4 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 64-bit floating-point
 *  numbers. The contents of the newly-created vector lanes are unspecified.  */
static inline SIMD_CFUNC simd_double4 simd_make_double4_undef(double other) {
  simd_double4 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 64-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_double4 simd_make_double4(simd_double2 other) {
  simd_double4 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 64-bit floating-point
 *  numbers. The contents of the newly-created vector lanes are unspecified.  */
static inline SIMD_CFUNC simd_double4 simd_make_double4_undef(simd_double2 other) {
  simd_double4 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of four 64-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_double4 simd_make_double4(simd_double3 other) {
  simd_double4 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of four 64-bit floating-point
 *  numbers. The contents of the newly-created vector lanes are unspecified.  */
static inline SIMD_CFUNC simd_double4 simd_make_double4_undef(simd_double3 other) {
  simd_double4 result;
  result.xyz = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_double4 simd_make_double4(simd_double4 other) {
  return other;
}

/*! @abstract Truncates `other` to form a vector of four 64-bit floating-
 *  point numbers.                                                            */
static inline SIMD_CFUNC simd_double4 simd_make_double4(simd_double8 other) {
  return other.xyzw;
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double8 simd_make_double8(simd_double4 lo, simd_double4 hi) {
  simd_double8 result;
  result.lo = lo;
  result.hi = hi;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double8 simd_make_double8(double other) {
  simd_double8 result = 0;
  result.x = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 64-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_double8 simd_make_double8_undef(double other) {
  simd_double8 result;
  result.x = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double8 simd_make_double8(simd_double2 other) {
  simd_double8 result = 0;
  result.xy = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 64-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_double8 simd_make_double8_undef(simd_double2 other) {
  simd_double8 result;
  result.xy = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double8 simd_make_double8(simd_double3 other) {
  simd_double8 result = 0;
  result.xyz = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 64-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_double8 simd_make_double8_undef(simd_double3 other) {
  simd_double8 result;
  result.xyz = other;
  return result;
}

/*! @abstract Zero-extends `other` to form a vector of eight 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CFUNC simd_double8 simd_make_double8(simd_double4 other) {
  simd_double8 result = 0;
  result.xyzw = other;
  return result;
}
  
/*! @abstract Extends `other` to form a vector of eight 64-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
static inline SIMD_CFUNC simd_double8 simd_make_double8_undef(simd_double4 other) {
  simd_double8 result;
  result.xyzw = other;
  return result;
}

/*! @abstract Returns `other` unmodified. This function is a convenience for
 *  templated and autogenerated code.                                         */
static inline SIMD_CFUNC simd_double8 simd_make_double8(simd_double8 other) {
  return other;
}

#ifdef __cplusplus
} /* extern "C" */

#include <tuple>
#include <simd/packed.h>

namespace simd {
/*! @abstract Concatenates `x` and `y` to form a vector of two 8-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CPPFUNC char2 make_char2(char x, char y) {
 return ::simd_make_char2(x, y);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of two
 *  8-bit signed (twos-complement) integers.                                  */
template <typename typeN> static SIMD_CPPFUNC char2 make_char2(typeN other) {
  return ::simd_make_char2(other);
}
  
/*! @abstract Extends `other` to form a vector of two 8-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC char2 make_char2_undef(typeN other) {
  return ::simd_make_char2_undef(other);
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC char3 make_char3(char x, char y, char z) {
 return ::simd_make_char3(x, y, z);
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC char3 make_char3(char x, char2 yz) {
 return ::simd_make_char3(x, yz);
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC char3 make_char3(char2 xy, char z) {
 return ::simd_make_char3(xy, z);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of three
 *  8-bit signed (twos-complement) integers.                                  */
template <typename typeN> static SIMD_CPPFUNC char3 make_char3(typeN other) {
  return ::simd_make_char3(other);
}
  
/*! @abstract Extends `other` to form a vector of three 8-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC char3 make_char3_undef(typeN other) {
  return ::simd_make_char3_undef(other);
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  8-bit signed (twos-complement) integers.                                  */
static inline SIMD_CPPFUNC char4 make_char4(char x, char y, char z, char w) {
 return ::simd_make_char4(x, y, z, w);
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC char4 make_char4(char x, char y, char2 zw) {
 return ::simd_make_char4(x, y, zw);
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC char4 make_char4(char x, char2 yz, char w) {
 return ::simd_make_char4(x, yz, w);
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC char4 make_char4(char2 xy, char z, char w) {
 return ::simd_make_char4(xy, z, w);
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC char4 make_char4(char x, char3 yzw) {
 return ::simd_make_char4(x, yzw);
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC char4 make_char4(char2 xy, char2 zw) {
 return ::simd_make_char4(xy, zw);
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC char4 make_char4(char3 xyz, char w) {
 return ::simd_make_char4(xyz, w);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of four
 *  8-bit signed (twos-complement) integers.                                  */
template <typename typeN> static SIMD_CPPFUNC char4 make_char4(typeN other) {
  return ::simd_make_char4(other);
}
  
/*! @abstract Extends `other` to form a vector of four 8-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC char4 make_char4_undef(typeN other) {
  return ::simd_make_char4_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC char8 make_char8(char4 lo, char4 hi) {
 return ::simd_make_char8(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of eight
 *  8-bit signed (twos-complement) integers.                                  */
template <typename typeN> static SIMD_CPPFUNC char8 make_char8(typeN other) {
  return ::simd_make_char8(other);
}
  
/*! @abstract Extends `other` to form a vector of eight 8-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC char8 make_char8_undef(typeN other) {
  return ::simd_make_char8_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixteen 8-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC char16 make_char16(char8 lo, char8 hi) {
 return ::simd_make_char16(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of sixteen
 *  8-bit signed (twos-complement) integers.                                  */
template <typename typeN> static SIMD_CPPFUNC char16 make_char16(typeN other) {
  return ::simd_make_char16(other);
}
  
/*! @abstract Extends `other` to form a vector of sixteen 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
template <typename typeN> static SIMD_CPPFUNC char16 make_char16_undef(typeN other) {
  return ::simd_make_char16_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of thirty-two
 *  8-bit signed (twos-complement) integers.                                  */
static inline SIMD_CPPFUNC char32 make_char32(char16 lo, char16 hi) {
 return ::simd_make_char32(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of thirty-
 *  two 8-bit signed (twos-complement) integers.                              */
template <typename typeN> static SIMD_CPPFUNC char32 make_char32(typeN other) {
  return ::simd_make_char32(other);
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
template <typename typeN> static SIMD_CPPFUNC char32 make_char32_undef(typeN other) {
  return ::simd_make_char32_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixty-four
 *  8-bit signed (twos-complement) integers.                                  */
static inline SIMD_CPPFUNC char64 make_char64(char32 lo, char32 hi) {
 return ::simd_make_char64(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of sixty-
 *  four 8-bit signed (twos-complement) integers.                             */
template <typename typeN> static SIMD_CPPFUNC char64 make_char64(typeN other) {
  return ::simd_make_char64(other);
}
  
/*! @abstract Extends `other` to form a vector of sixty-four 8-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
template <typename typeN> static SIMD_CPPFUNC char64 make_char64_undef(typeN other) {
  return ::simd_make_char64_undef(other);
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uchar2 make_uchar2(unsigned char x, unsigned char y) {
 return ::simd_make_uchar2(x, y);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of two
 *  8-bit unsigned integers.                                                  */
template <typename typeN> static SIMD_CPPFUNC uchar2 make_uchar2(typeN other) {
  return ::simd_make_uchar2(other);
}
  
/*! @abstract Extends `other` to form a vector of two 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC uchar2 make_uchar2_undef(typeN other) {
  return ::simd_make_uchar2_undef(other);
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uchar3 make_uchar3(unsigned char x, unsigned char y, unsigned char z) {
 return ::simd_make_uchar3(x, y, z);
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uchar3 make_uchar3(unsigned char x, uchar2 yz) {
 return ::simd_make_uchar3(x, yz);
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uchar3 make_uchar3(uchar2 xy, unsigned char z) {
 return ::simd_make_uchar3(xy, z);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of three
 *  8-bit unsigned integers.                                                  */
template <typename typeN> static SIMD_CPPFUNC uchar3 make_uchar3(typeN other) {
  return ::simd_make_uchar3(other);
}
  
/*! @abstract Extends `other` to form a vector of three 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC uchar3 make_uchar3_undef(typeN other) {
  return ::simd_make_uchar3_undef(other);
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  8-bit unsigned integers.                                                  */
static inline SIMD_CPPFUNC uchar4 make_uchar4(unsigned char x, unsigned char y, unsigned char z, unsigned char w) {
 return ::simd_make_uchar4(x, y, z, w);
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uchar4 make_uchar4(unsigned char x, unsigned char y, uchar2 zw) {
 return ::simd_make_uchar4(x, y, zw);
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uchar4 make_uchar4(unsigned char x, uchar2 yz, unsigned char w) {
 return ::simd_make_uchar4(x, yz, w);
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uchar4 make_uchar4(uchar2 xy, unsigned char z, unsigned char w) {
 return ::simd_make_uchar4(xy, z, w);
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uchar4 make_uchar4(unsigned char x, uchar3 yzw) {
 return ::simd_make_uchar4(x, yzw);
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uchar4 make_uchar4(uchar2 xy, uchar2 zw) {
 return ::simd_make_uchar4(xy, zw);
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uchar4 make_uchar4(uchar3 xyz, unsigned char w) {
 return ::simd_make_uchar4(xyz, w);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of four
 *  8-bit unsigned integers.                                                  */
template <typename typeN> static SIMD_CPPFUNC uchar4 make_uchar4(typeN other) {
  return ::simd_make_uchar4(other);
}
  
/*! @abstract Extends `other` to form a vector of four 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC uchar4 make_uchar4_undef(typeN other) {
  return ::simd_make_uchar4_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uchar8 make_uchar8(uchar4 lo, uchar4 hi) {
 return ::simd_make_uchar8(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of eight
 *  8-bit unsigned integers.                                                  */
template <typename typeN> static SIMD_CPPFUNC uchar8 make_uchar8(typeN other) {
  return ::simd_make_uchar8(other);
}
  
/*! @abstract Extends `other` to form a vector of eight 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC uchar8 make_uchar8_undef(typeN other) {
  return ::simd_make_uchar8_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixteen 8-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uchar16 make_uchar16(uchar8 lo, uchar8 hi) {
 return ::simd_make_uchar16(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of sixteen
 *  8-bit unsigned integers.                                                  */
template <typename typeN> static SIMD_CPPFUNC uchar16 make_uchar16(typeN other) {
  return ::simd_make_uchar16(other);
}
  
/*! @abstract Extends `other` to form a vector of sixteen 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC uchar16 make_uchar16_undef(typeN other) {
  return ::simd_make_uchar16_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of thirty-two
 *  8-bit unsigned integers.                                                  */
static inline SIMD_CPPFUNC uchar32 make_uchar32(uchar16 lo, uchar16 hi) {
 return ::simd_make_uchar32(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of thirty-
 *  two 8-bit unsigned integers.                                              */
template <typename typeN> static SIMD_CPPFUNC uchar32 make_uchar32(typeN other) {
  return ::simd_make_uchar32(other);
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC uchar32 make_uchar32_undef(typeN other) {
  return ::simd_make_uchar32_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixty-four
 *  8-bit unsigned integers.                                                  */
static inline SIMD_CPPFUNC uchar64 make_uchar64(uchar32 lo, uchar32 hi) {
 return ::simd_make_uchar64(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of sixty-
 *  four 8-bit unsigned integers.                                             */
template <typename typeN> static SIMD_CPPFUNC uchar64 make_uchar64(typeN other) {
  return ::simd_make_uchar64(other);
}
  
/*! @abstract Extends `other` to form a vector of sixty-four 8-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC uchar64 make_uchar64_undef(typeN other) {
  return ::simd_make_uchar64_undef(other);
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 16-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CPPFUNC short2 make_short2(short x, short y) {
 return ::simd_make_short2(x, y);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of two
 *  16-bit signed (twos-complement) integers.                                 */
template <typename typeN> static SIMD_CPPFUNC short2 make_short2(typeN other) {
  return ::simd_make_short2(other);
}
  
/*! @abstract Extends `other` to form a vector of two 16-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC short2 make_short2_undef(typeN other) {
  return ::simd_make_short2_undef(other);
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC short3 make_short3(short x, short y, short z) {
 return ::simd_make_short3(x, y, z);
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC short3 make_short3(short x, short2 yz) {
 return ::simd_make_short3(x, yz);
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC short3 make_short3(short2 xy, short z) {
 return ::simd_make_short3(xy, z);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of three
 *  16-bit signed (twos-complement) integers.                                 */
template <typename typeN> static SIMD_CPPFUNC short3 make_short3(typeN other) {
  return ::simd_make_short3(other);
}
  
/*! @abstract Extends `other` to form a vector of three 16-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC short3 make_short3_undef(typeN other) {
  return ::simd_make_short3_undef(other);
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  16-bit signed (twos-complement) integers.                                 */
static inline SIMD_CPPFUNC short4 make_short4(short x, short y, short z, short w) {
 return ::simd_make_short4(x, y, z, w);
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC short4 make_short4(short x, short y, short2 zw) {
 return ::simd_make_short4(x, y, zw);
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC short4 make_short4(short x, short2 yz, short w) {
 return ::simd_make_short4(x, yz, w);
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC short4 make_short4(short2 xy, short z, short w) {
 return ::simd_make_short4(xy, z, w);
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC short4 make_short4(short x, short3 yzw) {
 return ::simd_make_short4(x, yzw);
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC short4 make_short4(short2 xy, short2 zw) {
 return ::simd_make_short4(xy, zw);
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC short4 make_short4(short3 xyz, short w) {
 return ::simd_make_short4(xyz, w);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of four
 *  16-bit signed (twos-complement) integers.                                 */
template <typename typeN> static SIMD_CPPFUNC short4 make_short4(typeN other) {
  return ::simd_make_short4(other);
}
  
/*! @abstract Extends `other` to form a vector of four 16-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC short4 make_short4_undef(typeN other) {
  return ::simd_make_short4_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC short8 make_short8(short4 lo, short4 hi) {
 return ::simd_make_short8(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of eight
 *  16-bit signed (twos-complement) integers.                                 */
template <typename typeN> static SIMD_CPPFUNC short8 make_short8(typeN other) {
  return ::simd_make_short8(other);
}
  
/*! @abstract Extends `other` to form a vector of eight 16-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC short8 make_short8_undef(typeN other) {
  return ::simd_make_short8_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixteen 16-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC short16 make_short16(short8 lo, short8 hi) {
 return ::simd_make_short16(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of sixteen
 *  16-bit signed (twos-complement) integers.                                 */
template <typename typeN> static SIMD_CPPFUNC short16 make_short16(typeN other) {
  return ::simd_make_short16(other);
}
  
/*! @abstract Extends `other` to form a vector of sixteen 16-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
template <typename typeN> static SIMD_CPPFUNC short16 make_short16_undef(typeN other) {
  return ::simd_make_short16_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of thirty-two
 *  16-bit signed (twos-complement) integers.                                 */
static inline SIMD_CPPFUNC short32 make_short32(short16 lo, short16 hi) {
 return ::simd_make_short32(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of thirty-
 *  two 16-bit signed (twos-complement) integers.                             */
template <typename typeN> static SIMD_CPPFUNC short32 make_short32(typeN other) {
  return ::simd_make_short32(other);
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 16-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
template <typename typeN> static SIMD_CPPFUNC short32 make_short32_undef(typeN other) {
  return ::simd_make_short32_undef(other);
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ushort2 make_ushort2(unsigned short x, unsigned short y) {
 return ::simd_make_ushort2(x, y);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of two
 *  16-bit unsigned integers.                                                 */
template <typename typeN> static SIMD_CPPFUNC ushort2 make_ushort2(typeN other) {
  return ::simd_make_ushort2(other);
}
  
/*! @abstract Extends `other` to form a vector of two 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC ushort2 make_ushort2_undef(typeN other) {
  return ::simd_make_ushort2_undef(other);
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ushort3 make_ushort3(unsigned short x, unsigned short y, unsigned short z) {
 return ::simd_make_ushort3(x, y, z);
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ushort3 make_ushort3(unsigned short x, ushort2 yz) {
 return ::simd_make_ushort3(x, yz);
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ushort3 make_ushort3(ushort2 xy, unsigned short z) {
 return ::simd_make_ushort3(xy, z);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of three
 *  16-bit unsigned integers.                                                 */
template <typename typeN> static SIMD_CPPFUNC ushort3 make_ushort3(typeN other) {
  return ::simd_make_ushort3(other);
}
  
/*! @abstract Extends `other` to form a vector of three 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC ushort3 make_ushort3_undef(typeN other) {
  return ::simd_make_ushort3_undef(other);
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  16-bit unsigned integers.                                                 */
static inline SIMD_CPPFUNC ushort4 make_ushort4(unsigned short x, unsigned short y, unsigned short z, unsigned short w) {
 return ::simd_make_ushort4(x, y, z, w);
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ushort4 make_ushort4(unsigned short x, unsigned short y, ushort2 zw) {
 return ::simd_make_ushort4(x, y, zw);
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ushort4 make_ushort4(unsigned short x, ushort2 yz, unsigned short w) {
 return ::simd_make_ushort4(x, yz, w);
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ushort4 make_ushort4(ushort2 xy, unsigned short z, unsigned short w) {
 return ::simd_make_ushort4(xy, z, w);
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ushort4 make_ushort4(unsigned short x, ushort3 yzw) {
 return ::simd_make_ushort4(x, yzw);
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ushort4 make_ushort4(ushort2 xy, ushort2 zw) {
 return ::simd_make_ushort4(xy, zw);
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ushort4 make_ushort4(ushort3 xyz, unsigned short w) {
 return ::simd_make_ushort4(xyz, w);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of four
 *  16-bit unsigned integers.                                                 */
template <typename typeN> static SIMD_CPPFUNC ushort4 make_ushort4(typeN other) {
  return ::simd_make_ushort4(other);
}
  
/*! @abstract Extends `other` to form a vector of four 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC ushort4 make_ushort4_undef(typeN other) {
  return ::simd_make_ushort4_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ushort8 make_ushort8(ushort4 lo, ushort4 hi) {
 return ::simd_make_ushort8(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of eight
 *  16-bit unsigned integers.                                                 */
template <typename typeN> static SIMD_CPPFUNC ushort8 make_ushort8(typeN other) {
  return ::simd_make_ushort8(other);
}
  
/*! @abstract Extends `other` to form a vector of eight 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC ushort8 make_ushort8_undef(typeN other) {
  return ::simd_make_ushort8_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixteen 16-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ushort16 make_ushort16(ushort8 lo, ushort8 hi) {
 return ::simd_make_ushort16(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of sixteen
 *  16-bit unsigned integers.                                                 */
template <typename typeN> static SIMD_CPPFUNC ushort16 make_ushort16(typeN other) {
  return ::simd_make_ushort16(other);
}
  
/*! @abstract Extends `other` to form a vector of sixteen 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC ushort16 make_ushort16_undef(typeN other) {
  return ::simd_make_ushort16_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of thirty-two
 *  16-bit unsigned integers.                                                 */
static inline SIMD_CPPFUNC ushort32 make_ushort32(ushort16 lo, ushort16 hi) {
 return ::simd_make_ushort32(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of thirty-
 *  two 16-bit unsigned integers.                                             */
template <typename typeN> static SIMD_CPPFUNC ushort32 make_ushort32(typeN other) {
  return ::simd_make_ushort32(other);
}
  
/*! @abstract Extends `other` to form a vector of thirty-two 16-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC ushort32 make_ushort32_undef(typeN other) {
  return ::simd_make_ushort32_undef(other);
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 32-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CPPFUNC int2 make_int2(int x, int y) {
 return ::simd_make_int2(x, y);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of two
 *  32-bit signed (twos-complement) integers.                                 */
template <typename typeN> static SIMD_CPPFUNC int2 make_int2(typeN other) {
  return ::simd_make_int2(other);
}
  
/*! @abstract Extends `other` to form a vector of two 32-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC int2 make_int2_undef(typeN other) {
  return ::simd_make_int2_undef(other);
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC int3 make_int3(int x, int y, int z) {
 return ::simd_make_int3(x, y, z);
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC int3 make_int3(int x, int2 yz) {
 return ::simd_make_int3(x, yz);
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC int3 make_int3(int2 xy, int z) {
 return ::simd_make_int3(xy, z);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of three
 *  32-bit signed (twos-complement) integers.                                 */
template <typename typeN> static SIMD_CPPFUNC int3 make_int3(typeN other) {
  return ::simd_make_int3(other);
}
  
/*! @abstract Extends `other` to form a vector of three 32-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC int3 make_int3_undef(typeN other) {
  return ::simd_make_int3_undef(other);
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  32-bit signed (twos-complement) integers.                                 */
static inline SIMD_CPPFUNC int4 make_int4(int x, int y, int z, int w) {
 return ::simd_make_int4(x, y, z, w);
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC int4 make_int4(int x, int y, int2 zw) {
 return ::simd_make_int4(x, y, zw);
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC int4 make_int4(int x, int2 yz, int w) {
 return ::simd_make_int4(x, yz, w);
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC int4 make_int4(int2 xy, int z, int w) {
 return ::simd_make_int4(xy, z, w);
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC int4 make_int4(int x, int3 yzw) {
 return ::simd_make_int4(x, yzw);
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC int4 make_int4(int2 xy, int2 zw) {
 return ::simd_make_int4(xy, zw);
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC int4 make_int4(int3 xyz, int w) {
 return ::simd_make_int4(xyz, w);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of four
 *  32-bit signed (twos-complement) integers.                                 */
template <typename typeN> static SIMD_CPPFUNC int4 make_int4(typeN other) {
  return ::simd_make_int4(other);
}
  
/*! @abstract Extends `other` to form a vector of four 32-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC int4 make_int4_undef(typeN other) {
  return ::simd_make_int4_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC int8 make_int8(int4 lo, int4 hi) {
 return ::simd_make_int8(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of eight
 *  32-bit signed (twos-complement) integers.                                 */
template <typename typeN> static SIMD_CPPFUNC int8 make_int8(typeN other) {
  return ::simd_make_int8(other);
}
  
/*! @abstract Extends `other` to form a vector of eight 32-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC int8 make_int8_undef(typeN other) {
  return ::simd_make_int8_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixteen 32-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC int16 make_int16(int8 lo, int8 hi) {
 return ::simd_make_int16(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of sixteen
 *  32-bit signed (twos-complement) integers.                                 */
template <typename typeN> static SIMD_CPPFUNC int16 make_int16(typeN other) {
  return ::simd_make_int16(other);
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit signed
 *  (twos-complement) integers. The contents of the newly-created vector
 *  lanes are unspecified.                                                    */
template <typename typeN> static SIMD_CPPFUNC int16 make_int16_undef(typeN other) {
  return ::simd_make_int16_undef(other);
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uint2 make_uint2(unsigned int x, unsigned int y) {
 return ::simd_make_uint2(x, y);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of two
 *  32-bit unsigned integers.                                                 */
template <typename typeN> static SIMD_CPPFUNC uint2 make_uint2(typeN other) {
  return ::simd_make_uint2(other);
}
  
/*! @abstract Extends `other` to form a vector of two 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC uint2 make_uint2_undef(typeN other) {
  return ::simd_make_uint2_undef(other);
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uint3 make_uint3(unsigned int x, unsigned int y, unsigned int z) {
 return ::simd_make_uint3(x, y, z);
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uint3 make_uint3(unsigned int x, uint2 yz) {
 return ::simd_make_uint3(x, yz);
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uint3 make_uint3(uint2 xy, unsigned int z) {
 return ::simd_make_uint3(xy, z);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of three
 *  32-bit unsigned integers.                                                 */
template <typename typeN> static SIMD_CPPFUNC uint3 make_uint3(typeN other) {
  return ::simd_make_uint3(other);
}
  
/*! @abstract Extends `other` to form a vector of three 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC uint3 make_uint3_undef(typeN other) {
  return ::simd_make_uint3_undef(other);
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  32-bit unsigned integers.                                                 */
static inline SIMD_CPPFUNC uint4 make_uint4(unsigned int x, unsigned int y, unsigned int z, unsigned int w) {
 return ::simd_make_uint4(x, y, z, w);
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uint4 make_uint4(unsigned int x, unsigned int y, uint2 zw) {
 return ::simd_make_uint4(x, y, zw);
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uint4 make_uint4(unsigned int x, uint2 yz, unsigned int w) {
 return ::simd_make_uint4(x, yz, w);
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uint4 make_uint4(uint2 xy, unsigned int z, unsigned int w) {
 return ::simd_make_uint4(xy, z, w);
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uint4 make_uint4(unsigned int x, uint3 yzw) {
 return ::simd_make_uint4(x, yzw);
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uint4 make_uint4(uint2 xy, uint2 zw) {
 return ::simd_make_uint4(xy, zw);
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uint4 make_uint4(uint3 xyz, unsigned int w) {
 return ::simd_make_uint4(xyz, w);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of four
 *  32-bit unsigned integers.                                                 */
template <typename typeN> static SIMD_CPPFUNC uint4 make_uint4(typeN other) {
  return ::simd_make_uint4(other);
}
  
/*! @abstract Extends `other` to form a vector of four 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC uint4 make_uint4_undef(typeN other) {
  return ::simd_make_uint4_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uint8 make_uint8(uint4 lo, uint4 hi) {
 return ::simd_make_uint8(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of eight
 *  32-bit unsigned integers.                                                 */
template <typename typeN> static SIMD_CPPFUNC uint8 make_uint8(typeN other) {
  return ::simd_make_uint8(other);
}
  
/*! @abstract Extends `other` to form a vector of eight 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC uint8 make_uint8_undef(typeN other) {
  return ::simd_make_uint8_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixteen 32-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC uint16 make_uint16(uint8 lo, uint8 hi) {
 return ::simd_make_uint16(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of sixteen
 *  32-bit unsigned integers.                                                 */
template <typename typeN> static SIMD_CPPFUNC uint16 make_uint16(typeN other) {
  return ::simd_make_uint16(other);
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC uint16 make_uint16_undef(typeN other) {
  return ::simd_make_uint16_undef(other);
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC float2 make_float2(float x, float y) {
 return ::simd_make_float2(x, y);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of two
 *  32-bit floating-point numbers.                                            */
template <typename typeN> static SIMD_CPPFUNC float2 make_float2(typeN other) {
  return ::simd_make_float2(other);
}
  
/*! @abstract Extends `other` to form a vector of two 32-bit floating-point
 *  numbers. The contents of the newly-created vector lanes are unspecified.  */
template <typename typeN> static SIMD_CPPFUNC float2 make_float2_undef(typeN other) {
  return ::simd_make_float2_undef(other);
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC float3 make_float3(float x, float y, float z) {
 return ::simd_make_float3(x, y, z);
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC float3 make_float3(float x, float2 yz) {
 return ::simd_make_float3(x, yz);
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC float3 make_float3(float2 xy, float z) {
 return ::simd_make_float3(xy, z);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of three
 *  32-bit floating-point numbers.                                            */
template <typename typeN> static SIMD_CPPFUNC float3 make_float3(typeN other) {
  return ::simd_make_float3(other);
}
  
/*! @abstract Extends `other` to form a vector of three 32-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC float3 make_float3_undef(typeN other) {
  return ::simd_make_float3_undef(other);
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  32-bit floating-point numbers.                                            */
static inline SIMD_CPPFUNC float4 make_float4(float x, float y, float z, float w) {
 return ::simd_make_float4(x, y, z, w);
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC float4 make_float4(float x, float y, float2 zw) {
 return ::simd_make_float4(x, y, zw);
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC float4 make_float4(float x, float2 yz, float w) {
 return ::simd_make_float4(x, yz, w);
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC float4 make_float4(float2 xy, float z, float w) {
 return ::simd_make_float4(xy, z, w);
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC float4 make_float4(float x, float3 yzw) {
 return ::simd_make_float4(x, yzw);
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC float4 make_float4(float2 xy, float2 zw) {
 return ::simd_make_float4(xy, zw);
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC float4 make_float4(float3 xyz, float w) {
 return ::simd_make_float4(xyz, w);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of four
 *  32-bit floating-point numbers.                                            */
template <typename typeN> static SIMD_CPPFUNC float4 make_float4(typeN other) {
  return ::simd_make_float4(other);
}
  
/*! @abstract Extends `other` to form a vector of four 32-bit floating-point
 *  numbers. The contents of the newly-created vector lanes are unspecified.  */
template <typename typeN> static SIMD_CPPFUNC float4 make_float4_undef(typeN other) {
  return ::simd_make_float4_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC float8 make_float8(float4 lo, float4 hi) {
 return ::simd_make_float8(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of eight
 *  32-bit floating-point numbers.                                            */
template <typename typeN> static SIMD_CPPFUNC float8 make_float8(typeN other) {
  return ::simd_make_float8(other);
}
  
/*! @abstract Extends `other` to form a vector of eight 32-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC float8 make_float8_undef(typeN other) {
  return ::simd_make_float8_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of sixteen 32-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC float16 make_float16(float8 lo, float8 hi) {
 return ::simd_make_float16(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of sixteen
 *  32-bit floating-point numbers.                                            */
template <typename typeN> static SIMD_CPPFUNC float16 make_float16(typeN other) {
  return ::simd_make_float16(other);
}
  
/*! @abstract Extends `other` to form a vector of sixteen 32-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC float16 make_float16_undef(typeN other) {
  return ::simd_make_float16_undef(other);
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 64-bit signed
 *  (twos-complement) integers.                                               */
static inline SIMD_CPPFUNC long2 make_long2(long1 x, long1 y) {
 return ::simd_make_long2(x, y);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of two
 *  64-bit signed (twos-complement) integers.                                 */
template <typename typeN> static SIMD_CPPFUNC long2 make_long2(typeN other) {
  return ::simd_make_long2(other);
}
  
/*! @abstract Extends `other` to form a vector of two 64-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC long2 make_long2_undef(typeN other) {
  return ::simd_make_long2_undef(other);
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC long3 make_long3(long1 x, long1 y, long1 z) {
 return ::simd_make_long3(x, y, z);
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC long3 make_long3(long1 x, long2 yz) {
 return ::simd_make_long3(x, yz);
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC long3 make_long3(long2 xy, long1 z) {
 return ::simd_make_long3(xy, z);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of three
 *  64-bit signed (twos-complement) integers.                                 */
template <typename typeN> static SIMD_CPPFUNC long3 make_long3(typeN other) {
  return ::simd_make_long3(other);
}
  
/*! @abstract Extends `other` to form a vector of three 64-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC long3 make_long3_undef(typeN other) {
  return ::simd_make_long3_undef(other);
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  64-bit signed (twos-complement) integers.                                 */
static inline SIMD_CPPFUNC long4 make_long4(long1 x, long1 y, long1 z, long1 w) {
 return ::simd_make_long4(x, y, z, w);
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC long4 make_long4(long1 x, long1 y, long2 zw) {
 return ::simd_make_long4(x, y, zw);
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC long4 make_long4(long1 x, long2 yz, long1 w) {
 return ::simd_make_long4(x, yz, w);
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC long4 make_long4(long2 xy, long1 z, long1 w) {
 return ::simd_make_long4(xy, z, w);
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC long4 make_long4(long1 x, long3 yzw) {
 return ::simd_make_long4(x, yzw);
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC long4 make_long4(long2 xy, long2 zw) {
 return ::simd_make_long4(xy, zw);
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC long4 make_long4(long3 xyz, long1 w) {
 return ::simd_make_long4(xyz, w);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of four
 *  64-bit signed (twos-complement) integers.                                 */
template <typename typeN> static SIMD_CPPFUNC long4 make_long4(typeN other) {
  return ::simd_make_long4(other);
}
  
/*! @abstract Extends `other` to form a vector of four 64-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC long4 make_long4_undef(typeN other) {
  return ::simd_make_long4_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 64-bit
 *  signed (twos-complement) integers.                                        */
static inline SIMD_CPPFUNC long8 make_long8(long4 lo, long4 hi) {
 return ::simd_make_long8(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of eight
 *  64-bit signed (twos-complement) integers.                                 */
template <typename typeN> static SIMD_CPPFUNC long8 make_long8(typeN other) {
  return ::simd_make_long8(other);
}
  
/*! @abstract Extends `other` to form a vector of eight 64-bit signed (twos-
 *  complement) integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC long8 make_long8_undef(typeN other) {
  return ::simd_make_long8_undef(other);
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ulong2 make_ulong2(ulong1 x, ulong1 y) {
 return ::simd_make_ulong2(x, y);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of two
 *  64-bit unsigned integers.                                                 */
template <typename typeN> static SIMD_CPPFUNC ulong2 make_ulong2(typeN other) {
  return ::simd_make_ulong2(other);
}
  
/*! @abstract Extends `other` to form a vector of two 64-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC ulong2 make_ulong2_undef(typeN other) {
  return ::simd_make_ulong2_undef(other);
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ulong3 make_ulong3(ulong1 x, ulong1 y, ulong1 z) {
 return ::simd_make_ulong3(x, y, z);
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ulong3 make_ulong3(ulong1 x, ulong2 yz) {
 return ::simd_make_ulong3(x, yz);
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ulong3 make_ulong3(ulong2 xy, ulong1 z) {
 return ::simd_make_ulong3(xy, z);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of three
 *  64-bit unsigned integers.                                                 */
template <typename typeN> static SIMD_CPPFUNC ulong3 make_ulong3(typeN other) {
  return ::simd_make_ulong3(other);
}
  
/*! @abstract Extends `other` to form a vector of three 64-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC ulong3 make_ulong3_undef(typeN other) {
  return ::simd_make_ulong3_undef(other);
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  64-bit unsigned integers.                                                 */
static inline SIMD_CPPFUNC ulong4 make_ulong4(ulong1 x, ulong1 y, ulong1 z, ulong1 w) {
 return ::simd_make_ulong4(x, y, z, w);
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ulong4 make_ulong4(ulong1 x, ulong1 y, ulong2 zw) {
 return ::simd_make_ulong4(x, y, zw);
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ulong4 make_ulong4(ulong1 x, ulong2 yz, ulong1 w) {
 return ::simd_make_ulong4(x, yz, w);
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ulong4 make_ulong4(ulong2 xy, ulong1 z, ulong1 w) {
 return ::simd_make_ulong4(xy, z, w);
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ulong4 make_ulong4(ulong1 x, ulong3 yzw) {
 return ::simd_make_ulong4(x, yzw);
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ulong4 make_ulong4(ulong2 xy, ulong2 zw) {
 return ::simd_make_ulong4(xy, zw);
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ulong4 make_ulong4(ulong3 xyz, ulong1 w) {
 return ::simd_make_ulong4(xyz, w);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of four
 *  64-bit unsigned integers.                                                 */
template <typename typeN> static SIMD_CPPFUNC ulong4 make_ulong4(typeN other) {
  return ::simd_make_ulong4(other);
}
  
/*! @abstract Extends `other` to form a vector of four 64-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC ulong4 make_ulong4_undef(typeN other) {
  return ::simd_make_ulong4_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 64-bit
 *  unsigned integers.                                                        */
static inline SIMD_CPPFUNC ulong8 make_ulong8(ulong4 lo, ulong4 hi) {
 return ::simd_make_ulong8(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of eight
 *  64-bit unsigned integers.                                                 */
template <typename typeN> static SIMD_CPPFUNC ulong8 make_ulong8(typeN other) {
  return ::simd_make_ulong8(other);
}
  
/*! @abstract Extends `other` to form a vector of eight 64-bit unsigned
 *  integers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC ulong8 make_ulong8_undef(typeN other) {
  return ::simd_make_ulong8_undef(other);
}

/*! @abstract Concatenates `x` and `y` to form a vector of two 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC double2 make_double2(double x, double y) {
 return ::simd_make_double2(x, y);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of two
 *  64-bit floating-point numbers.                                            */
template <typename typeN> static SIMD_CPPFUNC double2 make_double2(typeN other) {
  return ::simd_make_double2(other);
}
  
/*! @abstract Extends `other` to form a vector of two 64-bit floating-point
 *  numbers. The contents of the newly-created vector lanes are unspecified.  */
template <typename typeN> static SIMD_CPPFUNC double2 make_double2_undef(typeN other) {
  return ::simd_make_double2_undef(other);
}

/*! @abstract Concatenates `x`, `y` and `z` to form a vector of three 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC double3 make_double3(double x, double y, double z) {
 return ::simd_make_double3(x, y, z);
}
  
/*! @abstract Concatenates `x` and `yz` to form a vector of three 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC double3 make_double3(double x, double2 yz) {
 return ::simd_make_double3(x, yz);
}
  
/*! @abstract Concatenates `xy` and `z` to form a vector of three 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC double3 make_double3(double2 xy, double z) {
 return ::simd_make_double3(xy, z);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of three
 *  64-bit floating-point numbers.                                            */
template <typename typeN> static SIMD_CPPFUNC double3 make_double3(typeN other) {
  return ::simd_make_double3(other);
}
  
/*! @abstract Extends `other` to form a vector of three 64-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC double3 make_double3_undef(typeN other) {
  return ::simd_make_double3_undef(other);
}

/*! @abstract Concatenates `x`, `y`, `z` and `w` to form a vector of four
 *  64-bit floating-point numbers.                                            */
static inline SIMD_CPPFUNC double4 make_double4(double x, double y, double z, double w) {
 return ::simd_make_double4(x, y, z, w);
}
  
/*! @abstract Concatenates `x`, `y` and `zw` to form a vector of four 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC double4 make_double4(double x, double y, double2 zw) {
 return ::simd_make_double4(x, y, zw);
}
  
/*! @abstract Concatenates `x`, `yz` and `w` to form a vector of four 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC double4 make_double4(double x, double2 yz, double w) {
 return ::simd_make_double4(x, yz, w);
}
  
/*! @abstract Concatenates `xy`, `z` and `w` to form a vector of four 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC double4 make_double4(double2 xy, double z, double w) {
 return ::simd_make_double4(xy, z, w);
}
  
/*! @abstract Concatenates `x` and `yzw` to form a vector of four 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC double4 make_double4(double x, double3 yzw) {
 return ::simd_make_double4(x, yzw);
}
  
/*! @abstract Concatenates `xy` and `zw` to form a vector of four 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC double4 make_double4(double2 xy, double2 zw) {
 return ::simd_make_double4(xy, zw);
}
  
/*! @abstract Concatenates `xyz` and `w` to form a vector of four 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC double4 make_double4(double3 xyz, double w) {
 return ::simd_make_double4(xyz, w);
}
  
/*! @abstract Truncates or zero-extends `other` to form a vector of four
 *  64-bit floating-point numbers.                                            */
template <typename typeN> static SIMD_CPPFUNC double4 make_double4(typeN other) {
  return ::simd_make_double4(other);
}
  
/*! @abstract Extends `other` to form a vector of four 64-bit floating-point
 *  numbers. The contents of the newly-created vector lanes are unspecified.  */
template <typename typeN> static SIMD_CPPFUNC double4 make_double4_undef(typeN other) {
  return ::simd_make_double4_undef(other);
}

/*! @abstract Concatenates `lo` and `hi` to form a vector of eight 64-bit
 *  floating-point numbers.                                                   */
static inline SIMD_CPPFUNC double8 make_double8(double4 lo, double4 hi) {
 return ::simd_make_double8(lo, hi);
}

/*! @abstract Truncates or zero-extends `other` to form a vector of eight
 *  64-bit floating-point numbers.                                            */
template <typename typeN> static SIMD_CPPFUNC double8 make_double8(typeN other) {
  return ::simd_make_double8(other);
}
  
/*! @abstract Extends `other` to form a vector of eight 64-bit floating-
 *  point numbers. The contents of the newly-created vector lanes are
 *  unspecified.                                                              */
template <typename typeN> static SIMD_CPPFUNC double8 make_double8_undef(typeN other) {
  return ::simd_make_double8_undef(other);
}

/*! @struct Vector
 *  @abstract Templated Vector struct based on scalar type and number of
 *  elements
 *  @field count Number of elements in the vector
 *  @field scalar_t The scalar type of each element
 *  @field type The inferred simd::typeN type
 *  @field packed_t The inferred simd::packed::typeN type
 *  @field mask_t The return type of comparison operations                    */
template <typename ScalarType, size_t count> struct Vector {
  //  static const size_t count
  //  typedef scalar_t
  //  typedef type
  //  typedef packed_t
  //  typedef mask_t
};
/*! @abstract Helper type to access the simd type easily.                     */
template <typename ScalarType, size_t count>
using Vector_t = typename Vector<ScalarType, count>::type;

/*! @abstract Look up the equivalent Vector struct according to the simd
 *  type.                                                                     */
template <typename typeN> struct get_traits
{
//    using type = Vector<ScalarType, count>;
};
/*! @abstract Helper type to access the Vector struct easily.
 *  @discussion This is commonly used to query the type traits of a simd
 *  type.
 *  For example, simd::traits<simd::float4>::count is 4.                      */
template<typename typeN>
using traits = typename get_traits<typeN>::type;

template<> struct Vector<char1, 1> {
  static const size_t count = 1;
  typedef char1 scalar_t;
  typedef char1 type;
  typedef char1 mask_t;
};

template <> struct get_traits<char1>
{
    using type = Vector<char1, 1>;
};

template<> struct Vector<char1, 2> {
  static const size_t count = 2;
  typedef char1 scalar_t;
  typedef char2 type;
  typedef packed::char2 packed_t;
  typedef char2 mask_t;
};

template <> struct get_traits<char2>
{
    using type = Vector<char1, 2>;
};

template<> struct Vector<char1, 3> {
  static const size_t count = 3;
  typedef char1 scalar_t;
  typedef char3 type;
  typedef char3 mask_t;
};

template <> struct get_traits<char3>
{
    using type = Vector<char1, 3>;
};

template<> struct Vector<char1, 4> {
  static const size_t count = 4;
  typedef char1 scalar_t;
  typedef char4 type;
  typedef packed::char4 packed_t;
  typedef char4 mask_t;
};

template <> struct get_traits<char4>
{
    using type = Vector<char1, 4>;
};

template<> struct Vector<char1, 8> {
  static const size_t count = 8;
  typedef char1 scalar_t;
  typedef char8 type;
  typedef packed::char8 packed_t;
  typedef char8 mask_t;
};

template <> struct get_traits<char8>
{
    using type = Vector<char1, 8>;
};

template<> struct Vector<char1, 16> {
  static const size_t count = 16;
  typedef char1 scalar_t;
  typedef char16 type;
  typedef packed::char16 packed_t;
  typedef char16 mask_t;
};

template <> struct get_traits<char16>
{
    using type = Vector<char1, 16>;
};

template<> struct Vector<char1, 32> {
  static const size_t count = 32;
  typedef char1 scalar_t;
  typedef char32 type;
  typedef packed::char32 packed_t;
  typedef char32 mask_t;
};

template <> struct get_traits<char32>
{
    using type = Vector<char1, 32>;
};

template<> struct Vector<char1, 64> {
  static const size_t count = 64;
  typedef char1 scalar_t;
  typedef char64 type;
  typedef packed::char64 packed_t;
  typedef char64 mask_t;
};

template <> struct get_traits<char64>
{
    using type = Vector<char1, 64>;
};

template<> struct Vector<uchar1, 1> {
  static const size_t count = 1;
  typedef uchar1 scalar_t;
  typedef uchar1 type;
  typedef char1 mask_t;
};

template <> struct get_traits<uchar1>
{
    using type = Vector<uchar1, 1>;
};

template<> struct Vector<uchar1, 2> {
  static const size_t count = 2;
  typedef uchar1 scalar_t;
  typedef uchar2 type;
  typedef packed::uchar2 packed_t;
  typedef char2 mask_t;
};

template <> struct get_traits<uchar2>
{
    using type = Vector<uchar1, 2>;
};

template<> struct Vector<uchar1, 3> {
  static const size_t count = 3;
  typedef uchar1 scalar_t;
  typedef uchar3 type;
  typedef char3 mask_t;
};

template <> struct get_traits<uchar3>
{
    using type = Vector<uchar1, 3>;
};

template<> struct Vector<uchar1, 4> {
  static const size_t count = 4;
  typedef uchar1 scalar_t;
  typedef uchar4 type;
  typedef packed::uchar4 packed_t;
  typedef char4 mask_t;
};

template <> struct get_traits<uchar4>
{
    using type = Vector<uchar1, 4>;
};

template<> struct Vector<uchar1, 8> {
  static const size_t count = 8;
  typedef uchar1 scalar_t;
  typedef uchar8 type;
  typedef packed::uchar8 packed_t;
  typedef char8 mask_t;
};

template <> struct get_traits<uchar8>
{
    using type = Vector<uchar1, 8>;
};

template<> struct Vector<uchar1, 16> {
  static const size_t count = 16;
  typedef uchar1 scalar_t;
  typedef uchar16 type;
  typedef packed::uchar16 packed_t;
  typedef char16 mask_t;
};

template <> struct get_traits<uchar16>
{
    using type = Vector<uchar1, 16>;
};

template<> struct Vector<uchar1, 32> {
  static const size_t count = 32;
  typedef uchar1 scalar_t;
  typedef uchar32 type;
  typedef packed::uchar32 packed_t;
  typedef char32 mask_t;
};

template <> struct get_traits<uchar32>
{
    using type = Vector<uchar1, 32>;
};

template<> struct Vector<uchar1, 64> {
  static const size_t count = 64;
  typedef uchar1 scalar_t;
  typedef uchar64 type;
  typedef packed::uchar64 packed_t;
  typedef char64 mask_t;
};

template <> struct get_traits<uchar64>
{
    using type = Vector<uchar1, 64>;
};

template<> struct Vector<short1, 1> {
  static const size_t count = 1;
  typedef short1 scalar_t;
  typedef short1 type;
  typedef short1 mask_t;
};

template <> struct get_traits<short1>
{
    using type = Vector<short1, 1>;
};

template<> struct Vector<short1, 2> {
  static const size_t count = 2;
  typedef short1 scalar_t;
  typedef short2 type;
  typedef packed::short2 packed_t;
  typedef short2 mask_t;
};

template <> struct get_traits<short2>
{
    using type = Vector<short1, 2>;
};

template<> struct Vector<short1, 3> {
  static const size_t count = 3;
  typedef short1 scalar_t;
  typedef short3 type;
  typedef short3 mask_t;
};

template <> struct get_traits<short3>
{
    using type = Vector<short1, 3>;
};

template<> struct Vector<short1, 4> {
  static const size_t count = 4;
  typedef short1 scalar_t;
  typedef short4 type;
  typedef packed::short4 packed_t;
  typedef short4 mask_t;
};

template <> struct get_traits<short4>
{
    using type = Vector<short1, 4>;
};

template<> struct Vector<short1, 8> {
  static const size_t count = 8;
  typedef short1 scalar_t;
  typedef short8 type;
  typedef packed::short8 packed_t;
  typedef short8 mask_t;
};

template <> struct get_traits<short8>
{
    using type = Vector<short1, 8>;
};

template<> struct Vector<short1, 16> {
  static const size_t count = 16;
  typedef short1 scalar_t;
  typedef short16 type;
  typedef packed::short16 packed_t;
  typedef short16 mask_t;
};

template <> struct get_traits<short16>
{
    using type = Vector<short1, 16>;
};

template<> struct Vector<short1, 32> {
  static const size_t count = 32;
  typedef short1 scalar_t;
  typedef short32 type;
  typedef packed::short32 packed_t;
  typedef short32 mask_t;
};

template <> struct get_traits<short32>
{
    using type = Vector<short1, 32>;
};

template<> struct Vector<ushort1, 1> {
  static const size_t count = 1;
  typedef ushort1 scalar_t;
  typedef ushort1 type;
  typedef short1 mask_t;
};

template <> struct get_traits<ushort1>
{
    using type = Vector<ushort1, 1>;
};

template<> struct Vector<ushort1, 2> {
  static const size_t count = 2;
  typedef ushort1 scalar_t;
  typedef ushort2 type;
  typedef packed::ushort2 packed_t;
  typedef short2 mask_t;
};

template <> struct get_traits<ushort2>
{
    using type = Vector<ushort1, 2>;
};

template<> struct Vector<ushort1, 3> {
  static const size_t count = 3;
  typedef ushort1 scalar_t;
  typedef ushort3 type;
  typedef short3 mask_t;
};

template <> struct get_traits<ushort3>
{
    using type = Vector<ushort1, 3>;
};

template<> struct Vector<ushort1, 4> {
  static const size_t count = 4;
  typedef ushort1 scalar_t;
  typedef ushort4 type;
  typedef packed::ushort4 packed_t;
  typedef short4 mask_t;
};

template <> struct get_traits<ushort4>
{
    using type = Vector<ushort1, 4>;
};

template<> struct Vector<ushort1, 8> {
  static const size_t count = 8;
  typedef ushort1 scalar_t;
  typedef ushort8 type;
  typedef packed::ushort8 packed_t;
  typedef short8 mask_t;
};

template <> struct get_traits<ushort8>
{
    using type = Vector<ushort1, 8>;
};

template<> struct Vector<ushort1, 16> {
  static const size_t count = 16;
  typedef ushort1 scalar_t;
  typedef ushort16 type;
  typedef packed::ushort16 packed_t;
  typedef short16 mask_t;
};

template <> struct get_traits<ushort16>
{
    using type = Vector<ushort1, 16>;
};

template<> struct Vector<ushort1, 32> {
  static const size_t count = 32;
  typedef ushort1 scalar_t;
  typedef ushort32 type;
  typedef packed::ushort32 packed_t;
  typedef short32 mask_t;
};

template <> struct get_traits<ushort32>
{
    using type = Vector<ushort1, 32>;
};

template<> struct Vector<int1, 1> {
  static const size_t count = 1;
  typedef int1 scalar_t;
  typedef int1 type;
  typedef int1 mask_t;
};

template <> struct get_traits<int1>
{
    using type = Vector<int1, 1>;
};

template<> struct Vector<int1, 2> {
  static const size_t count = 2;
  typedef int1 scalar_t;
  typedef int2 type;
  typedef packed::int2 packed_t;
  typedef int2 mask_t;
};

template <> struct get_traits<int2>
{
    using type = Vector<int1, 2>;
};

template<> struct Vector<int1, 3> {
  static const size_t count = 3;
  typedef int1 scalar_t;
  typedef int3 type;
  typedef int3 mask_t;
};

template <> struct get_traits<int3>
{
    using type = Vector<int1, 3>;
};

template<> struct Vector<int1, 4> {
  static const size_t count = 4;
  typedef int1 scalar_t;
  typedef int4 type;
  typedef packed::int4 packed_t;
  typedef int4 mask_t;
};

template <> struct get_traits<int4>
{
    using type = Vector<int1, 4>;
};

template<> struct Vector<int1, 8> {
  static const size_t count = 8;
  typedef int1 scalar_t;
  typedef int8 type;
  typedef packed::int8 packed_t;
  typedef int8 mask_t;
};

template <> struct get_traits<int8>
{
    using type = Vector<int1, 8>;
};

template<> struct Vector<int1, 16> {
  static const size_t count = 16;
  typedef int1 scalar_t;
  typedef int16 type;
  typedef packed::int16 packed_t;
  typedef int16 mask_t;
};

template <> struct get_traits<int16>
{
    using type = Vector<int1, 16>;
};

template<> struct Vector<uint1, 1> {
  static const size_t count = 1;
  typedef uint1 scalar_t;
  typedef uint1 type;
  typedef int1 mask_t;
};

template <> struct get_traits<uint1>
{
    using type = Vector<uint1, 1>;
};

template<> struct Vector<uint1, 2> {
  static const size_t count = 2;
  typedef uint1 scalar_t;
  typedef uint2 type;
  typedef packed::uint2 packed_t;
  typedef int2 mask_t;
};

template <> struct get_traits<uint2>
{
    using type = Vector<uint1, 2>;
};

template<> struct Vector<uint1, 3> {
  static const size_t count = 3;
  typedef uint1 scalar_t;
  typedef uint3 type;
  typedef int3 mask_t;
};

template <> struct get_traits<uint3>
{
    using type = Vector<uint1, 3>;
};

template<> struct Vector<uint1, 4> {
  static const size_t count = 4;
  typedef uint1 scalar_t;
  typedef uint4 type;
  typedef packed::uint4 packed_t;
  typedef int4 mask_t;
};

template <> struct get_traits<uint4>
{
    using type = Vector<uint1, 4>;
};

template<> struct Vector<uint1, 8> {
  static const size_t count = 8;
  typedef uint1 scalar_t;
  typedef uint8 type;
  typedef packed::uint8 packed_t;
  typedef int8 mask_t;
};

template <> struct get_traits<uint8>
{
    using type = Vector<uint1, 8>;
};

template<> struct Vector<uint1, 16> {
  static const size_t count = 16;
  typedef uint1 scalar_t;
  typedef uint16 type;
  typedef packed::uint16 packed_t;
  typedef int16 mask_t;
};

template <> struct get_traits<uint16>
{
    using type = Vector<uint1, 16>;
};

template<> struct Vector<float1, 1> {
  static const size_t count = 1;
  typedef float1 scalar_t;
  typedef float1 type;
  typedef int1 mask_t;
};

template <> struct get_traits<float1>
{
    using type = Vector<float1, 1>;
};

template<> struct Vector<float1, 2> {
  static const size_t count = 2;
  typedef float1 scalar_t;
  typedef float2 type;
  typedef packed::float2 packed_t;
  typedef int2 mask_t;
};

template <> struct get_traits<float2>
{
    using type = Vector<float1, 2>;
};

template<> struct Vector<float1, 3> {
  static const size_t count = 3;
  typedef float1 scalar_t;
  typedef float3 type;
  typedef int3 mask_t;
};

template <> struct get_traits<float3>
{
    using type = Vector<float1, 3>;
};

template<> struct Vector<float1, 4> {
  static const size_t count = 4;
  typedef float1 scalar_t;
  typedef float4 type;
  typedef packed::float4 packed_t;
  typedef int4 mask_t;
};

template <> struct get_traits<float4>
{
    using type = Vector<float1, 4>;
};

template<> struct Vector<float1, 8> {
  static const size_t count = 8;
  typedef float1 scalar_t;
  typedef float8 type;
  typedef packed::float8 packed_t;
  typedef int8 mask_t;
};

template <> struct get_traits<float8>
{
    using type = Vector<float1, 8>;
};

template<> struct Vector<float1, 16> {
  static const size_t count = 16;
  typedef float1 scalar_t;
  typedef float16 type;
  typedef packed::float16 packed_t;
  typedef int16 mask_t;
};

template <> struct get_traits<float16>
{
    using type = Vector<float1, 16>;
};

template<> struct Vector<long1, 1> {
  static const size_t count = 1;
  typedef long1 scalar_t;
  typedef long1 type;
  typedef long1 mask_t;
};

template <> struct get_traits<long1>
{
    using type = Vector<long1, 1>;
};

template<> struct Vector<long1, 2> {
  static const size_t count = 2;
  typedef long1 scalar_t;
  typedef long2 type;
  typedef packed::long2 packed_t;
  typedef long2 mask_t;
};

template <> struct get_traits<long2>
{
    using type = Vector<long1, 2>;
};

template<> struct Vector<long1, 3> {
  static const size_t count = 3;
  typedef long1 scalar_t;
  typedef long3 type;
  typedef long3 mask_t;
};

template <> struct get_traits<long3>
{
    using type = Vector<long1, 3>;
};

template<> struct Vector<long1, 4> {
  static const size_t count = 4;
  typedef long1 scalar_t;
  typedef long4 type;
  typedef packed::long4 packed_t;
  typedef long4 mask_t;
};

template <> struct get_traits<long4>
{
    using type = Vector<long1, 4>;
};

template<> struct Vector<long1, 8> {
  static const size_t count = 8;
  typedef long1 scalar_t;
  typedef long8 type;
  typedef packed::long8 packed_t;
  typedef long8 mask_t;
};

template <> struct get_traits<long8>
{
    using type = Vector<long1, 8>;
};

template<> struct Vector<ulong1, 1> {
  static const size_t count = 1;
  typedef ulong1 scalar_t;
  typedef ulong1 type;
  typedef long1 mask_t;
};

template <> struct get_traits<ulong1>
{
    using type = Vector<ulong1, 1>;
};

template<> struct Vector<ulong1, 2> {
  static const size_t count = 2;
  typedef ulong1 scalar_t;
  typedef ulong2 type;
  typedef packed::ulong2 packed_t;
  typedef long2 mask_t;
};

template <> struct get_traits<ulong2>
{
    using type = Vector<ulong1, 2>;
};

template<> struct Vector<ulong1, 3> {
  static const size_t count = 3;
  typedef ulong1 scalar_t;
  typedef ulong3 type;
  typedef long3 mask_t;
};

template <> struct get_traits<ulong3>
{
    using type = Vector<ulong1, 3>;
};

template<> struct Vector<ulong1, 4> {
  static const size_t count = 4;
  typedef ulong1 scalar_t;
  typedef ulong4 type;
  typedef packed::ulong4 packed_t;
  typedef long4 mask_t;
};

template <> struct get_traits<ulong4>
{
    using type = Vector<ulong1, 4>;
};

template<> struct Vector<ulong1, 8> {
  static const size_t count = 8;
  typedef ulong1 scalar_t;
  typedef ulong8 type;
  typedef packed::ulong8 packed_t;
  typedef long8 mask_t;
};

template <> struct get_traits<ulong8>
{
    using type = Vector<ulong1, 8>;
};

template<> struct Vector<double1, 1> {
  static const size_t count = 1;
  typedef double1 scalar_t;
  typedef double1 type;
  typedef long1 mask_t;
};

template <> struct get_traits<double1>
{
    using type = Vector<double1, 1>;
};

template<> struct Vector<double1, 2> {
  static const size_t count = 2;
  typedef double1 scalar_t;
  typedef double2 type;
  typedef packed::double2 packed_t;
  typedef long2 mask_t;
};

template <> struct get_traits<double2>
{
    using type = Vector<double1, 2>;
};

template<> struct Vector<double1, 3> {
  static const size_t count = 3;
  typedef double1 scalar_t;
  typedef double3 type;
  typedef long3 mask_t;
};

template <> struct get_traits<double3>
{
    using type = Vector<double1, 3>;
};

template<> struct Vector<double1, 4> {
  static const size_t count = 4;
  typedef double1 scalar_t;
  typedef double4 type;
  typedef packed::double4 packed_t;
  typedef long4 mask_t;
};

template <> struct get_traits<double4>
{
    using type = Vector<double1, 4>;
};

template<> struct Vector<double1, 8> {
  static const size_t count = 8;
  typedef double1 scalar_t;
  typedef double8 type;
  typedef packed::double8 packed_t;
  typedef long8 mask_t;
};

template <> struct get_traits<double8>
{
    using type = Vector<double1, 8>;
};

#if __has_feature(cxx_constexpr)
/*! @abstract Templated make function based on return type and argument
 *  type.                                                                     */
template<typename typeN, typename... Args>
static constexpr typeN make(Args... args)
{
    if constexpr (traits<typeN>::count == 1)
    {
        using FirstArgType = typename std::tuple_element<0, std::tuple<Args...>>::type;
        if constexpr (std::is_same<FirstArgType, typename traits<FirstArgType>::scalar_t>::value)
            return typeN(std::get<0>(std::make_tuple(args...)));
        else
            return typeN(std::get<0>(std::make_tuple(args...))[0]);
    }
    else if constexpr (std::is_same<typeN, char2>::value)
        return make_char2(args...);
    else if constexpr (std::is_same<typeN, char3>::value)
        return make_char3(args...);
    else if constexpr (std::is_same<typeN, char4>::value)
        return make_char4(args...);
    else if constexpr (std::is_same<typeN, char8>::value)
        return make_char8(args...);
    else if constexpr (std::is_same<typeN, char16>::value)
        return make_char16(args...);
    else if constexpr (std::is_same<typeN, char32>::value)
        return make_char32(args...);
    else if constexpr (std::is_same<typeN, char64>::value)
        return make_char64(args...);
    else if constexpr (std::is_same<typeN, uchar2>::value)
        return make_uchar2(args...);
    else if constexpr (std::is_same<typeN, uchar3>::value)
        return make_uchar3(args...);
    else if constexpr (std::is_same<typeN, uchar4>::value)
        return make_uchar4(args...);
    else if constexpr (std::is_same<typeN, uchar8>::value)
        return make_uchar8(args...);
    else if constexpr (std::is_same<typeN, uchar16>::value)
        return make_uchar16(args...);
    else if constexpr (std::is_same<typeN, uchar32>::value)
        return make_uchar32(args...);
    else if constexpr (std::is_same<typeN, uchar64>::value)
        return make_uchar64(args...);
    else if constexpr (std::is_same<typeN, short2>::value)
        return make_short2(args...);
    else if constexpr (std::is_same<typeN, short3>::value)
        return make_short3(args...);
    else if constexpr (std::is_same<typeN, short4>::value)
        return make_short4(args...);
    else if constexpr (std::is_same<typeN, short8>::value)
        return make_short8(args...);
    else if constexpr (std::is_same<typeN, short16>::value)
        return make_short16(args...);
    else if constexpr (std::is_same<typeN, short32>::value)
        return make_short32(args...);
    else if constexpr (std::is_same<typeN, ushort2>::value)
        return make_ushort2(args...);
    else if constexpr (std::is_same<typeN, ushort3>::value)
        return make_ushort3(args...);
    else if constexpr (std::is_same<typeN, ushort4>::value)
        return make_ushort4(args...);
    else if constexpr (std::is_same<typeN, ushort8>::value)
        return make_ushort8(args...);
    else if constexpr (std::is_same<typeN, ushort16>::value)
        return make_ushort16(args...);
    else if constexpr (std::is_same<typeN, ushort32>::value)
        return make_ushort32(args...);
    else if constexpr (std::is_same<typeN, int2>::value)
        return make_int2(args...);
    else if constexpr (std::is_same<typeN, int3>::value)
        return make_int3(args...);
    else if constexpr (std::is_same<typeN, int4>::value)
        return make_int4(args...);
    else if constexpr (std::is_same<typeN, int8>::value)
        return make_int8(args...);
    else if constexpr (std::is_same<typeN, int16>::value)
        return make_int16(args...);
    else if constexpr (std::is_same<typeN, uint2>::value)
        return make_uint2(args...);
    else if constexpr (std::is_same<typeN, uint3>::value)
        return make_uint3(args...);
    else if constexpr (std::is_same<typeN, uint4>::value)
        return make_uint4(args...);
    else if constexpr (std::is_same<typeN, uint8>::value)
        return make_uint8(args...);
    else if constexpr (std::is_same<typeN, uint16>::value)
        return make_uint16(args...);
    else if constexpr (std::is_same<typeN, float2>::value)
        return make_float2(args...);
    else if constexpr (std::is_same<typeN, float3>::value)
        return make_float3(args...);
    else if constexpr (std::is_same<typeN, float4>::value)
        return make_float4(args...);
    else if constexpr (std::is_same<typeN, float8>::value)
        return make_float8(args...);
    else if constexpr (std::is_same<typeN, float16>::value)
        return make_float16(args...);
    else if constexpr (std::is_same<typeN, long2>::value)
        return make_long2(args...);
    else if constexpr (std::is_same<typeN, long3>::value)
        return make_long3(args...);
    else if constexpr (std::is_same<typeN, long4>::value)
        return make_long4(args...);
    else if constexpr (std::is_same<typeN, long8>::value)
        return make_long8(args...);
    else if constexpr (std::is_same<typeN, ulong2>::value)
        return make_ulong2(args...);
    else if constexpr (std::is_same<typeN, ulong3>::value)
        return make_ulong3(args...);
    else if constexpr (std::is_same<typeN, ulong4>::value)
        return make_ulong4(args...);
    else if constexpr (std::is_same<typeN, ulong8>::value)
        return make_ulong8(args...);
    else if constexpr (std::is_same<typeN, double2>::value)
        return make_double2(args...);
    else if constexpr (std::is_same<typeN, double3>::value)
        return make_double3(args...);
    else if constexpr (std::is_same<typeN, double4>::value)
        return make_double4(args...);
    else if constexpr (std::is_same<typeN, double8>::value)
        return make_double8(args...);
}

/*! @abstract Templated make_undef function based on return type and
 *  argument type.                                                            */
template<typename typeN, typename... Args>
static constexpr typeN make_undef(Args... args)
{
    if constexpr (traits<typeN>::count == 1)
    {
        using FirstArgType = typename std::tuple_element<0, std::tuple<Args...>>::type;
        if constexpr (std::is_same<FirstArgType, typename traits<FirstArgType>::scalar_t>::value)
            return typeN(std::get<0>(std::make_tuple(args...)));
        else
            return typeN(std::get<0>(std::make_tuple(args...))[0]);
    }
    else if constexpr (std::is_same<typeN, char2>::value)
        return make_char2_undef(args...);
    else if constexpr (std::is_same<typeN, char3>::value)
        return make_char3_undef(args...);
    else if constexpr (std::is_same<typeN, char4>::value)
        return make_char4_undef(args...);
    else if constexpr (std::is_same<typeN, char8>::value)
        return make_char8_undef(args...);
    else if constexpr (std::is_same<typeN, char16>::value)
        return make_char16_undef(args...);
    else if constexpr (std::is_same<typeN, char32>::value)
        return make_char32_undef(args...);
    else if constexpr (std::is_same<typeN, char64>::value)
        return make_char64_undef(args...);
    else if constexpr (std::is_same<typeN, uchar2>::value)
        return make_uchar2_undef(args...);
    else if constexpr (std::is_same<typeN, uchar3>::value)
        return make_uchar3_undef(args...);
    else if constexpr (std::is_same<typeN, uchar4>::value)
        return make_uchar4_undef(args...);
    else if constexpr (std::is_same<typeN, uchar8>::value)
        return make_uchar8_undef(args...);
    else if constexpr (std::is_same<typeN, uchar16>::value)
        return make_uchar16_undef(args...);
    else if constexpr (std::is_same<typeN, uchar32>::value)
        return make_uchar32_undef(args...);
    else if constexpr (std::is_same<typeN, uchar64>::value)
        return make_uchar64_undef(args...);
    else if constexpr (std::is_same<typeN, short2>::value)
        return make_short2_undef(args...);
    else if constexpr (std::is_same<typeN, short3>::value)
        return make_short3_undef(args...);
    else if constexpr (std::is_same<typeN, short4>::value)
        return make_short4_undef(args...);
    else if constexpr (std::is_same<typeN, short8>::value)
        return make_short8_undef(args...);
    else if constexpr (std::is_same<typeN, short16>::value)
        return make_short16_undef(args...);
    else if constexpr (std::is_same<typeN, short32>::value)
        return make_short32_undef(args...);
    else if constexpr (std::is_same<typeN, ushort2>::value)
        return make_ushort2_undef(args...);
    else if constexpr (std::is_same<typeN, ushort3>::value)
        return make_ushort3_undef(args...);
    else if constexpr (std::is_same<typeN, ushort4>::value)
        return make_ushort4_undef(args...);
    else if constexpr (std::is_same<typeN, ushort8>::value)
        return make_ushort8_undef(args...);
    else if constexpr (std::is_same<typeN, ushort16>::value)
        return make_ushort16_undef(args...);
    else if constexpr (std::is_same<typeN, ushort32>::value)
        return make_ushort32_undef(args...);
    else if constexpr (std::is_same<typeN, int2>::value)
        return make_int2_undef(args...);
    else if constexpr (std::is_same<typeN, int3>::value)
        return make_int3_undef(args...);
    else if constexpr (std::is_same<typeN, int4>::value)
        return make_int4_undef(args...);
    else if constexpr (std::is_same<typeN, int8>::value)
        return make_int8_undef(args...);
    else if constexpr (std::is_same<typeN, int16>::value)
        return make_int16_undef(args...);
    else if constexpr (std::is_same<typeN, uint2>::value)
        return make_uint2_undef(args...);
    else if constexpr (std::is_same<typeN, uint3>::value)
        return make_uint3_undef(args...);
    else if constexpr (std::is_same<typeN, uint4>::value)
        return make_uint4_undef(args...);
    else if constexpr (std::is_same<typeN, uint8>::value)
        return make_uint8_undef(args...);
    else if constexpr (std::is_same<typeN, uint16>::value)
        return make_uint16_undef(args...);
    else if constexpr (std::is_same<typeN, float2>::value)
        return make_float2_undef(args...);
    else if constexpr (std::is_same<typeN, float3>::value)
        return make_float3_undef(args...);
    else if constexpr (std::is_same<typeN, float4>::value)
        return make_float4_undef(args...);
    else if constexpr (std::is_same<typeN, float8>::value)
        return make_float8_undef(args...);
    else if constexpr (std::is_same<typeN, float16>::value)
        return make_float16_undef(args...);
    else if constexpr (std::is_same<typeN, long2>::value)
        return make_long2_undef(args...);
    else if constexpr (std::is_same<typeN, long3>::value)
        return make_long3_undef(args...);
    else if constexpr (std::is_same<typeN, long4>::value)
        return make_long4_undef(args...);
    else if constexpr (std::is_same<typeN, long8>::value)
        return make_long8_undef(args...);
    else if constexpr (std::is_same<typeN, ulong2>::value)
        return make_ulong2_undef(args...);
    else if constexpr (std::is_same<typeN, ulong3>::value)
        return make_ulong3_undef(args...);
    else if constexpr (std::is_same<typeN, ulong4>::value)
        return make_ulong4_undef(args...);
    else if constexpr (std::is_same<typeN, ulong8>::value)
        return make_ulong8_undef(args...);
    else if constexpr (std::is_same<typeN, double2>::value)
        return make_double2_undef(args...);
    else if constexpr (std::is_same<typeN, double3>::value)
        return make_double3_undef(args...);
    else if constexpr (std::is_same<typeN, double4>::value)
        return make_double4_undef(args...);
    else if constexpr (std::is_same<typeN, double8>::value)
        return make_double8_undef(args...);
}
#endif /* __has_feature(cxx_constexpr) */
} /* namespace simd */
#endif /* __cplusplus */
#endif /* SIMD_COMPILER_HAS_REQUIRED_FEATURES */
#endif /* SIMD_VECTOR_CONSTRUCTORS */