"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProcessedDict = void 0;
/*
Copyright 2019 The Matrix.org Foundation C.I.C.
Copyright 2020 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
const CLEANUP_TIME_MS = 1000 * 60 * 10; // 10min
class ProcessedDict {
    constructor() {
        this.processed = {};
        this.timeoutObj = null;
    }
    getClaimer(domain, hash) {
        if (!this.processed[domain] || !this.processed[domain][hash]) {
            return null;
        }
        return this.processed[domain][hash].nick;
    }
    claim(domain, hash, nick, cmd) {
        if (!this.processed[domain]) {
            this.processed[domain] = {};
        }
        this.processed[domain][hash] = {
            nick: nick,
            // we don't ever want to purge NAMES events
            ts: cmd === "names" ? null : Date.now()
        };
    }
    stopCleaner() {
        if (this.timeoutObj) {
            clearTimeout(this.timeoutObj);
        }
    }
    startCleaner(parentLog) {
        const expiredList = {};
        this.timeoutObj = setTimeout(() => {
            const now = Date.now();
            // loop the processed list looking for entries older than CLEANUP_TIME_MS
            Object.keys(this.processed).forEach((domain) => {
                const entries = this.processed[domain];
                if (!entries) {
                    return;
                }
                Object.keys(entries).forEach((hash) => {
                    const entry = entries[hash];
                    if (entry.ts && (entry.ts + CLEANUP_TIME_MS) < now) {
                        if (!expiredList[domain]) {
                            expiredList[domain] = [];
                        }
                        expiredList[domain].push(hash);
                    }
                });
            });
            // purge the entries
            Object.keys(expiredList).forEach((domain) => {
                const hashes = expiredList[domain];
                parentLog.debug("Cleaning up %s entries from %s", hashes.length, domain);
                hashes.forEach((hash) => {
                    delete this.processed[domain][hash];
                });
            });
            this.startCleaner(parentLog);
        }, CLEANUP_TIME_MS);
    }
}
exports.ProcessedDict = ProcessedDict;
//# sourceMappingURL=ProcessedDict.js.map