"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeActionsProviderImpl = exports.SORT_IMPORT_CODE_ACTION_KIND = void 0;
const typescript_1 = __importDefault(require("typescript"));
const vscode_languageserver_1 = require("vscode-languageserver");
const documents_1 = require("../../../lib/documents");
const utils_1 = require("../../../utils");
const DocumentSnapshot_1 = require("../DocumentSnapshot");
const utils_2 = require("../utils");
const utils_3 = require("./utils");
/**
 * TODO change this to protocol constant if it's part of the protocol
 */
exports.SORT_IMPORT_CODE_ACTION_KIND = 'source.sortImports';
class CodeActionsProviderImpl {
    constructor(lsAndTsDocResolver, completionProvider, configManager) {
        this.lsAndTsDocResolver = lsAndTsDocResolver;
        this.completionProvider = completionProvider;
        this.configManager = configManager;
    }
    async getCodeActions(document, range, context, cancellationToken) {
        var _a, _b, _c;
        if (((_a = context.only) === null || _a === void 0 ? void 0 : _a[0]) === vscode_languageserver_1.CodeActionKind.SourceOrganizeImports) {
            return await this.organizeImports(document, cancellationToken);
        }
        if (((_b = context.only) === null || _b === void 0 ? void 0 : _b[0]) === exports.SORT_IMPORT_CODE_ACTION_KIND) {
            return await this.organizeImports(document, cancellationToken, 
            /**skipDestructiveCodeActions */ true);
        }
        // for source action command (all source.xxx)
        // vscode would show different source code action kinds to choose from
        if (((_c = context.only) === null || _c === void 0 ? void 0 : _c[0]) === vscode_languageserver_1.CodeActionKind.Source) {
            return [
                ...(await this.organizeImports(document, cancellationToken)),
                ...(await this.organizeImports(document, cancellationToken, 
                /**skipDestructiveCodeActions */ true))
            ];
        }
        if (context.diagnostics.length &&
            (!context.only || context.only.includes(vscode_languageserver_1.CodeActionKind.QuickFix))) {
            return await this.applyQuickfix(document, range, context, cancellationToken);
        }
        if (!context.only || context.only.includes(vscode_languageserver_1.CodeActionKind.Refactor)) {
            return await this.getApplicableRefactors(document, range, cancellationToken);
        }
        return [];
    }
    async organizeImports(document, cancellationToken, skipDestructiveCodeActions = false) {
        if (!document.scriptInfo && !document.moduleScriptInfo) {
            return [];
        }
        const { lang, tsDoc, userPreferences } = await this.getLSAndTSDoc(document);
        if ((cancellationToken === null || cancellationToken === void 0 ? void 0 : cancellationToken.isCancellationRequested) || tsDoc.parserError) {
            // If there's a parser error, we fall back to only the script contents,
            // so organize imports likely throws out a lot of seemingly unused imports
            // because they are only used in the template. Therefore do nothing in this case.
            return [];
        }
        const changes = lang.organizeImports({
            fileName: tsDoc.filePath,
            type: 'file',
            skipDestructiveCodeActions
        }, {
            ...(await this.configManager.getFormatCodeSettingsForFile(document, tsDoc.scriptKind)),
            // handle it on our own
            baseIndentSize: undefined
        }, userPreferences);
        const documentChanges = await Promise.all(changes.map(async (change) => {
            // Organize Imports will only affect the current file, so no need to check the file path
            return vscode_languageserver_1.TextDocumentEdit.create(vscode_languageserver_1.OptionalVersionedTextDocumentIdentifier.create(document.url, null), change.textChanges.map((edit) => {
                const range = this.checkRemoveImportCodeActionRange(edit, tsDoc, (0, documents_1.mapRangeToOriginal)(tsDoc, (0, utils_2.convertRange)(tsDoc, edit.span)));
                return this.fixIndentationOfImports(vscode_languageserver_1.TextEdit.replace(range, edit.newText), document);
            }));
        }));
        return [
            vscode_languageserver_1.CodeAction.create(skipDestructiveCodeActions ? 'Sort Imports' : 'Organize Imports', { documentChanges }, skipDestructiveCodeActions
                ? exports.SORT_IMPORT_CODE_ACTION_KIND
                : vscode_languageserver_1.CodeActionKind.SourceOrganizeImports)
        ];
    }
    fixIndentationOfImports(edit, document) {
        // "Organize Imports" will have edits that delete a group of imports by return empty edits
        // and one edit which contains all the organized imports of the group. Fix indentation
        // of that one by prepending all lines with the indentation of the first line.
        const { newText, range } = edit;
        if (!newText || range.start.character === 0) {
            return edit;
        }
        const line = (0, documents_1.getLineAtPosition)(range.start, document.getText());
        const leadingChars = line.substring(0, range.start.character);
        if (leadingChars.trim() !== '') {
            return edit;
        }
        const fixedNewText = (0, utils_1.modifyLines)(edit.newText, (line, idx) => idx === 0 || !line ? line : leadingChars + line);
        if (range.end.character > 0) {
            const endLine = (0, documents_1.getLineAtPosition)(range.end, document.getText());
            const isIndent = !endLine.substring(0, range.end.character).trim();
            if (isIndent) {
                const trimmedEndLine = endLine.trim();
                // imports that would be removed by the next delete edit
                if (trimmedEndLine && !trimmedEndLine.startsWith('import')) {
                    range.end.character = 0;
                }
            }
        }
        return vscode_languageserver_1.TextEdit.replace(range, fixedNewText);
    }
    checkRemoveImportCodeActionRange(edit, snapshot, range) {
        // Handle svelte2tsx wrong import mapping:
        // The character after the last import maps to the start of the script
        // TODO find a way to fix this in svelte2tsx and then remove this
        if ((range.end.line === 0 && range.end.character === 1) ||
            range.end.line < range.start.line) {
            edit.span.length -= 1;
            range = (0, documents_1.mapRangeToOriginal)(snapshot, (0, utils_2.convertRange)(snapshot, edit.span));
            if (!(snapshot instanceof DocumentSnapshot_1.SvelteDocumentSnapshot)) {
                range.end.character += 1;
                return range;
            }
            const line = (0, documents_1.getLineAtPosition)(range.end, snapshot.getOriginalText());
            // remove-import code action will removes the
            // line break generated by svelte2tsx,
            // but when there's no line break in the source
            // move back to next character would remove the next character
            if ([';', '"', "'"].includes(line[range.end.character])) {
                range.end.character += 1;
            }
            if ((0, documents_1.isAtEndOfLine)(line, range.end.character)) {
                range.end.line += 1;
                range.end.character = 0;
            }
        }
        return range;
    }
    async applyQuickfix(document, range, context, cancellationToken) {
        const { lang, tsDoc, userPreferences } = await this.getLSAndTSDoc(document);
        if (cancellationToken === null || cancellationToken === void 0 ? void 0 : cancellationToken.isCancellationRequested) {
            return [];
        }
        const start = tsDoc.offsetAt(tsDoc.getGeneratedPosition(range.start));
        const end = tsDoc.offsetAt(tsDoc.getGeneratedPosition(range.end));
        const errorCodes = context.diagnostics.map((diag) => Number(diag.code));
        const cannotFoundNameDiagnostic = context.diagnostics.filter((diagnostic) => diagnostic.code === 2304); // "Cannot find name '...'."
        const formatCodeSettings = await this.configManager.getFormatCodeSettingsForFile(document, tsDoc.scriptKind);
        const formatCodeBasis = (0, utils_3.getFormatCodeBasis)(formatCodeSettings);
        let codeFixes = cannotFoundNameDiagnostic.length
            ? this.getComponentImportQuickFix(start, end, lang, tsDoc, userPreferences, cannotFoundNameDiagnostic, formatCodeSettings)
            : undefined;
        codeFixes =
            // either-or situation
            codeFixes ||
                lang
                    .getCodeFixesAtPosition(tsDoc.filePath, start, end, errorCodes, formatCodeSettings, userPreferences)
                    .concat(await this.getSvelteQuickFixes(lang, document, cannotFoundNameDiagnostic, tsDoc, formatCodeBasis, userPreferences));
        const snapshots = new utils_3.SnapshotMap(this.lsAndTsDocResolver);
        snapshots.set(tsDoc.filePath, tsDoc);
        const codeActionsPromises = codeFixes.map(async (fix) => {
            const documentChangesPromises = fix.changes.map(async (change) => {
                const snapshot = await snapshots.retrieve(change.fileName);
                return vscode_languageserver_1.TextDocumentEdit.create(vscode_languageserver_1.OptionalVersionedTextDocumentIdentifier.create((0, utils_1.pathToUrl)(change.fileName), null), change.textChanges
                    .map((edit) => {
                    if (fix.fixName === 'import' &&
                        snapshot instanceof DocumentSnapshot_1.SvelteDocumentSnapshot) {
                        return this.completionProvider.codeActionChangeToTextEdit(document, snapshot, edit, true, range.start);
                    }
                    if ((0, utils_3.isTextSpanInGeneratedCode)(snapshot.getFullText(), edit.span)) {
                        return undefined;
                    }
                    let originalRange = (0, documents_1.mapRangeToOriginal)(snapshot, (0, utils_2.convertRange)(snapshot, edit.span));
                    if (fix.fixName === 'unusedIdentifier') {
                        originalRange = this.checkRemoveImportCodeActionRange(edit, snapshot, originalRange);
                    }
                    if (fix.fixName === 'fixMissingFunctionDeclaration') {
                        originalRange = this.checkEndOfFileCodeInsert(originalRange, range, document);
                        // ts doesn't add base indent to the first line
                        if (formatCodeSettings.baseIndentSize) {
                            const emptyLine = formatCodeBasis.newLine.repeat(2);
                            edit.newText =
                                emptyLine +
                                    formatCodeBasis.baseIndent +
                                    edit.newText.trimLeft();
                        }
                    }
                    if (fix.fixName === 'disableJsDiagnostics') {
                        if (edit.newText.includes('ts-nocheck')) {
                            return this.checkTsNoCheckCodeInsert(document, edit);
                        }
                        return this.checkDisableJsDiagnosticsCodeInsert(originalRange, document, edit);
                    }
                    if (fix.fixName === 'inferFromUsage') {
                        originalRange = this.checkAddJsDocCodeActionRange(snapshot, originalRange, document);
                    }
                    if (fix.fixName === 'fixConvertConstToLet') {
                        const offset = document.offsetAt(originalRange.start);
                        const constOffset = document.getText().indexOf('const', offset);
                        if (constOffset < 0) {
                            return undefined;
                        }
                        const beforeConst = document.getText().slice(0, constOffset);
                        if (beforeConst[beforeConst.length - 1] === '@' &&
                            beforeConst
                                .slice(0, beforeConst.length - 1)
                                .trimEnd()
                                .endsWith('{')) {
                            return undefined;
                        }
                    }
                    if (originalRange.start.line < 0 || originalRange.end.line < 0) {
                        return undefined;
                    }
                    return vscode_languageserver_1.TextEdit.replace(originalRange, edit.newText);
                })
                    .filter(utils_1.isNotNullOrUndefined));
            });
            const documentChanges = await Promise.all(documentChangesPromises);
            return vscode_languageserver_1.CodeAction.create(fix.description, {
                documentChanges
            }, vscode_languageserver_1.CodeActionKind.QuickFix);
        });
        const codeActions = await Promise.all(codeActionsPromises);
        // filter out empty code action
        return codeActions.filter((codeAction) => {
            var _a, _b;
            return (_b = (_a = codeAction.edit) === null || _a === void 0 ? void 0 : _a.documentChanges) === null || _b === void 0 ? void 0 : _b.every((change) => change.edits.length > 0);
        });
    }
    /**
     * import quick fix requires the symbol name to be the same as where it's defined.
     * But we have suffix on component default export to prevent conflict with
     * a local variable. So we use auto-import completion as a workaround here.
     */
    getComponentImportQuickFix(start, end, lang, tsDoc, userPreferences, diagnostics, formatCodeSetting) {
        var _a;
        const sourceFile = (_a = lang.getProgram()) === null || _a === void 0 ? void 0 : _a.getSourceFile(tsDoc.filePath);
        if (!sourceFile) {
            return;
        }
        const node = (0, utils_3.findContainingNode)(sourceFile, {
            start,
            length: end - start
        }, (node) => typescript_1.default.isCallExpression(node.parent) &&
            typescript_1.default.isIdentifier(node.parent.expression) &&
            node.parent.expression.text === '__sveltets_2_ensureComponent' &&
            typescript_1.default.isIdentifier(node));
        if (!node) {
            return;
        }
        const tagName = typescript_1.default.isIdentifier(node) ? node : node.tagName;
        const tagNameEnd = tagName.getEnd();
        const tagNameEndOriginalPosition = tsDoc.offsetAt(tsDoc.getOriginalPosition(tsDoc.positionAt(tagNameEnd)));
        const name = tagName.getText();
        if (!(0, utils_1.possiblyComponent)(name)) {
            return;
        }
        const hasDiagnosticForTag = diagnostics.some(({ range }) => tsDoc.offsetAt(range.start) <= tagNameEndOriginalPosition &&
            tagNameEndOriginalPosition <= tsDoc.offsetAt(range.end));
        if (!hasDiagnosticForTag) {
            return;
        }
        const completion = lang.getCompletionsAtPosition(tsDoc.filePath, tagNameEnd, userPreferences, formatCodeSetting);
        if (!completion) {
            return;
        }
        const suffixedName = (0, utils_2.toGeneratedSvelteComponentName)(name);
        const errorPreventingUserPreferences = this.completionProvider.fixUserPreferencesForSvelteComponentImport(userPreferences);
        const toFix = (c) => {
            var _a, _b, _c;
            return (_c = (_b = (_a = lang
                .getCompletionEntryDetails(tsDoc.filePath, end, c.name, formatCodeSetting, c.source, errorPreventingUserPreferences, c.data)) === null || _a === void 0 ? void 0 : _a.codeActions) === null || _b === void 0 ? void 0 : _b.map((a) => ({
                ...a,
                description: (0, utils_2.changeSvelteComponentName)(a.description),
                fixName: 'import'
            }))) !== null && _c !== void 0 ? _c : [];
        };
        return (0, utils_1.flatten)(completion.entries.filter((c) => c.name === name || c.name === suffixedName).map(toFix));
    }
    async getSvelteQuickFixes(lang, document, diagnostics, tsDoc, formatCodeBasis, userPreferences) {
        const program = lang.getProgram();
        const sourceFile = program === null || program === void 0 ? void 0 : program.getSourceFile(tsDoc.filePath);
        if (!program || !sourceFile) {
            return [];
        }
        const typeChecker = program.getTypeChecker();
        const results = [];
        const quote = (0, utils_3.getQuotePreference)(sourceFile, userPreferences);
        for (const diagnostic of diagnostics) {
            const start = tsDoc.offsetAt(tsDoc.getGeneratedPosition(diagnostic.range.start));
            const end = tsDoc.offsetAt(tsDoc.getGeneratedPosition(diagnostic.range.end));
            const identifier = (0, utils_3.findContainingNode)(sourceFile, { start, length: end - start }, typescript_1.default.isIdentifier);
            if (!identifier) {
                continue;
            }
            const isQuickFixTargetTargetStore = (identifier === null || identifier === void 0 ? void 0 : identifier.escapedText.toString().startsWith('$')) && diagnostic.code === 2304;
            const isQuickFixTargetEventHandler = this.isQuickFixForEventHandler(document, diagnostic);
            if (isQuickFixTargetTargetStore) {
                results.push(...(await this.getSvelteStoreQuickFixes(identifier, lang, document, tsDoc, userPreferences)));
            }
            if (isQuickFixTargetEventHandler) {
                results.push(...this.getEventHandlerQuickFixes(identifier, tsDoc, typeChecker, quote, formatCodeBasis));
            }
        }
        return results;
    }
    async getSvelteStoreQuickFixes(identifier, lang, document, tsDoc, userPreferences) {
        const storeIdentifier = identifier.escapedText.toString().substring(1);
        const formatCodeSettings = await this.configManager.getFormatCodeSettingsForFile(document, tsDoc.scriptKind);
        const completion = lang.getCompletionsAtPosition(tsDoc.filePath, 0, userPreferences, formatCodeSettings);
        if (!completion) {
            return [];
        }
        const toFix = (c) => {
            var _a, _b, _c;
            return (_c = (_b = (_a = lang
                .getCompletionEntryDetails(tsDoc.filePath, 0, c.name, formatCodeSettings, c.source, userPreferences, c.data)) === null || _a === void 0 ? void 0 : _a.codeActions) === null || _b === void 0 ? void 0 : _b.map((a) => ({
                ...a,
                changes: a.changes.map((change) => {
                    return {
                        ...change,
                        textChanges: change.textChanges.map((textChange) => {
                            // For some reason, TS sometimes adds the `type` modifier. Remove it.
                            return {
                                ...textChange,
                                newText: textChange.newText.replace(' type ', ' ')
                            };
                        })
                    };
                }),
                fixName: 'import'
            }))) !== null && _c !== void 0 ? _c : [];
        };
        return (0, utils_1.flatten)(completion.entries.filter((c) => c.name === storeIdentifier).map(toFix));
    }
    /**
     * Workaround for TypeScript doesn't provide a quick fix if the signature is typed as union type, like `(() => void) | null`
     * We can remove this once TypeScript doesn't have this limitation.
     */
    getEventHandlerQuickFixes(identifier, tsDoc, typeChecker, quote, formatCodeBasis) {
        const type = identifier && typeChecker.getContextualType(identifier);
        // if it's not union typescript should be able to do it. no need to enhance
        if (!type || !type.isUnion()) {
            return [];
        }
        const nonNullable = type.getNonNullableType();
        if (!(nonNullable.flags & typescript_1.default.TypeFlags.Object &&
            nonNullable.objectFlags & typescript_1.default.ObjectFlags.Anonymous)) {
            return [];
        }
        const signature = typeChecker.getSignaturesOfType(nonNullable, typescript_1.default.SignatureKind.Call)[0];
        const parameters = signature.parameters.map((p) => {
            var _a, _b;
            const declaration = (_a = p.valueDeclaration) !== null && _a !== void 0 ? _a : (_b = p.declarations) === null || _b === void 0 ? void 0 : _b[0];
            const typeString = declaration
                ? typeChecker.typeToString(typeChecker.getTypeOfSymbolAtLocation(p, declaration))
                : '';
            return { name: p.name, typeString };
        });
        const returnType = typeChecker.typeToString(signature.getReturnType());
        const useJsDoc = tsDoc.scriptKind === typescript_1.default.ScriptKind.JS || tsDoc.scriptKind === typescript_1.default.ScriptKind.JSX;
        const parametersText = (useJsDoc
            ? parameters.map((p) => p.name)
            : parameters.map((p) => p.name + (p.typeString ? ': ' + p.typeString : ''))).join(', ');
        const jsDoc = useJsDoc
            ? ['/**', ...parameters.map((p) => ` * @param {${p.typeString}} ${p.name}`), ' */']
            : [];
        const newText = [
            ...jsDoc,
            `function ${identifier.text}(${parametersText})${useJsDoc ? '' : ': ' + returnType} {`,
            formatCodeBasis.indent +
                `throw new Error(${quote}Function not implemented.${quote})` +
                formatCodeBasis.semi,
            '}'
        ]
            .map((line) => formatCodeBasis.baseIndent + line + formatCodeBasis.newLine)
            .join('');
        return [
            {
                description: `Add missing function declaration '${identifier.text}'`,
                fixName: 'fixMissingFunctionDeclaration',
                changes: [
                    {
                        fileName: tsDoc.filePath,
                        textChanges: [
                            {
                                newText,
                                span: { start: 0, length: 0 }
                            }
                        ]
                    }
                ]
            }
        ];
    }
    isQuickFixForEventHandler(document, diagnostic) {
        const htmlNode = document.html.findNodeAt(document.offsetAt(diagnostic.range.start));
        if (!htmlNode.attributes ||
            !Object.keys(htmlNode.attributes).some((attr) => attr.startsWith('on:'))) {
            return false;
        }
        return true;
    }
    async getApplicableRefactors(document, range, cancellationToken) {
        if (!(0, documents_1.isRangeInTag)(range, document.scriptInfo) &&
            !(0, documents_1.isRangeInTag)(range, document.moduleScriptInfo)) {
            return [];
        }
        // Don't allow refactorings when there is likely a store subscription.
        // Reason: Extracting that would lead to svelte2tsx' transformed store representation
        // showing up, which will confuse the user. In the long run, we maybe have to
        // setup a separate ts language service which only knows of the original script.
        const textInRange = document
            .getText()
            .substring(document.offsetAt(range.start), document.offsetAt(range.end));
        if (textInRange.includes('$')) {
            return [];
        }
        const { lang, tsDoc, userPreferences } = await this.getLSAndTSDoc(document);
        if (cancellationToken === null || cancellationToken === void 0 ? void 0 : cancellationToken.isCancellationRequested) {
            return [];
        }
        const textRange = {
            pos: tsDoc.offsetAt(tsDoc.getGeneratedPosition(range.start)),
            end: tsDoc.offsetAt(tsDoc.getGeneratedPosition(range.end))
        };
        const applicableRefactors = lang.getApplicableRefactors(document.getFilePath() || '', textRange, userPreferences);
        return (this.applicableRefactorsToCodeActions(applicableRefactors, document, range, textRange)
            // Only allow refactorings from which we know they work
            .filter((refactor) => {
            var _a, _b, _c;
            return ((_a = refactor.command) === null || _a === void 0 ? void 0 : _a.command.includes('function_scope')) ||
                ((_b = refactor.command) === null || _b === void 0 ? void 0 : _b.command.includes('constant_scope')) ||
                ((_c = refactor.command) === null || _c === void 0 ? void 0 : _c.command) === 'Infer function return type';
        })
            // The language server also proposes extraction into const/function in module scope,
            // which is outside of the render function, which is svelte2tsx-specific and unmapped,
            // so it would both not work and confuse the user ("What is this render? Never declared that").
            // So filter out the module scope proposal and rename the render-title
            .filter((refactor) => !refactor.title.includes('module scope'))
            .map((refactor) => ({
            ...refactor,
            title: refactor.title
                .replace("Extract to inner function in function 'render'", 'Extract to function')
                .replace("Extract to constant in function 'render'", 'Extract to constant')
        })));
    }
    applicableRefactorsToCodeActions(applicableRefactors, document, originalRange, textRange) {
        return (0, utils_1.flatten)(applicableRefactors.map((applicableRefactor) => {
            if (applicableRefactor.inlineable === false) {
                return [
                    vscode_languageserver_1.CodeAction.create(applicableRefactor.description, {
                        title: applicableRefactor.description,
                        command: applicableRefactor.name,
                        arguments: [
                            document.uri,
                            {
                                type: 'refactor',
                                textRange,
                                originalRange,
                                refactorName: 'Extract Symbol'
                            }
                        ]
                    })
                ];
            }
            return applicableRefactor.actions.map((action) => {
                return vscode_languageserver_1.CodeAction.create(action.description, {
                    title: action.description,
                    command: action.name,
                    arguments: [
                        document.uri,
                        {
                            type: 'refactor',
                            textRange,
                            originalRange,
                            refactorName: applicableRefactor.name
                        }
                    ]
                });
            });
        }));
    }
    async executeCommand(document, command, args) {
        var _a;
        if (!(((_a = args === null || args === void 0 ? void 0 : args[1]) === null || _a === void 0 ? void 0 : _a.type) === 'refactor')) {
            return null;
        }
        const { lang, tsDoc, userPreferences } = await this.getLSAndTSDoc(document);
        const path = document.getFilePath() || '';
        const { refactorName, originalRange, textRange } = args[1];
        const edits = lang.getEditsForRefactor(path, {}, textRange, refactorName, command, userPreferences);
        if (!edits || edits.edits.length === 0) {
            return null;
        }
        const documentChanges = edits === null || edits === void 0 ? void 0 : edits.edits.map((edit) => vscode_languageserver_1.TextDocumentEdit.create(vscode_languageserver_1.OptionalVersionedTextDocumentIdentifier.create(document.uri, null), edit.textChanges.map((edit) => {
            const range = (0, documents_1.mapRangeToOriginal)(tsDoc, (0, utils_2.convertRange)(tsDoc, edit.span));
            return vscode_languageserver_1.TextEdit.replace(this.checkEndOfFileCodeInsert(range, originalRange, document), edit.newText);
        })));
        return { documentChanges };
    }
    /**
     * Some refactorings place the new code at the end of svelte2tsx' render function,
     *  which is unmapped. In this case, add it to the end of the script tag ourselves.
     */
    checkEndOfFileCodeInsert(resultRange, targetRange, document) {
        if (resultRange.start.line < 0 || resultRange.end.line < 0) {
            if ((0, documents_1.isRangeInTag)(targetRange, document.moduleScriptInfo)) {
                return vscode_languageserver_1.Range.create(document.moduleScriptInfo.endPos, document.moduleScriptInfo.endPos);
            }
            if (document.scriptInfo) {
                return vscode_languageserver_1.Range.create(document.scriptInfo.endPos, document.scriptInfo.endPos);
            }
        }
        return resultRange;
    }
    checkTsNoCheckCodeInsert(document, edit) {
        var _a;
        const scriptInfo = (_a = document.moduleScriptInfo) !== null && _a !== void 0 ? _a : document.scriptInfo;
        if (!scriptInfo) {
            return undefined;
        }
        const newText = typescript_1.default.sys.newLine + edit.newText;
        return vscode_languageserver_1.TextEdit.insert(scriptInfo.startPos, newText);
    }
    checkDisableJsDiagnosticsCodeInsert(originalRange, document, edit) {
        var _a;
        const inModuleScript = (0, documents_1.isInTag)(originalRange.start, document.moduleScriptInfo);
        if (!(0, documents_1.isInTag)(originalRange.start, document.scriptInfo) && !inModuleScript) {
            return null;
        }
        const position = inModuleScript
            ? originalRange.start
            : (_a = this.fixPropsCodeActionRange(originalRange.start, document)) !== null && _a !== void 0 ? _a : originalRange.start;
        // fix the length of trailing indent
        const linesOfNewText = edit.newText.split('\n');
        if (/^[ \t]*$/.test(linesOfNewText[linesOfNewText.length - 1])) {
            const line = (0, documents_1.getLineAtPosition)(originalRange.start, document.getText());
            const indent = (0, utils_1.getIndent)(line);
            linesOfNewText[linesOfNewText.length - 1] = indent;
        }
        return vscode_languageserver_1.TextEdit.insert(position, linesOfNewText.join('\n'));
    }
    /**
     * svelte2tsx removes export in instance script
     */
    fixPropsCodeActionRange(start, document) {
        const documentText = document.getText();
        const offset = document.offsetAt(start);
        const exportKeywordOffset = documentText.lastIndexOf('export', offset);
        // export                 let a;
        if (exportKeywordOffset < 0 ||
            documentText.slice(exportKeywordOffset + 'export'.length, offset).trim()) {
            return;
        }
        const charBeforeExport = documentText[exportKeywordOffset - 1];
        if ((charBeforeExport !== undefined && !charBeforeExport.trim()) ||
            charBeforeExport === ';') {
            return document.positionAt(exportKeywordOffset);
        }
    }
    checkAddJsDocCodeActionRange(snapshot, originalRange, document) {
        if (snapshot.scriptKind !== typescript_1.default.ScriptKind.JS &&
            snapshot.scriptKind !== typescript_1.default.ScriptKind.JSX &&
            !(0, documents_1.isInTag)(originalRange.start, document.scriptInfo)) {
            return originalRange;
        }
        const position = this.fixPropsCodeActionRange(originalRange.start, document);
        if (position) {
            return {
                start: position,
                end: position
            };
        }
        return originalRange;
    }
    async getLSAndTSDoc(document) {
        return this.lsAndTsDocResolver.getLSAndTSDoc(document);
    }
}
exports.CodeActionsProviderImpl = CodeActionsProviderImpl;
//# sourceMappingURL=CodeActionsProvider.js.map