"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.removeBacktick = exports.preprocessQuery = exports.getIndexPatternFromRawQuery = exports.composeFinalQuery = exports.buildQuery = exports.buildPatternsQuery = void 0;

var _datemath = _interopRequireDefault(require("@elastic/datemath"));

var _lodash = require("lodash");

var _explorer = require("../../common/constants/explorer");

var _shared = require("../../common/constants/shared");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * @param literal - string literal that will be put inside single quotes in PPL command
 * @returns string with inner single quotes escaped
 */
const escapeQuotes = literal => {
  return literal.replaceAll("'", "''");
};

const getIndexPatternFromRawQuery = query => {
  const matches = query.match(_shared.PPL_INDEX_REGEX);

  if (matches) {
    return matches[2];
  }

  return '';
}; // insert time filter command and additional commands based on raw query


exports.getIndexPatternFromRawQuery = getIndexPatternFromRawQuery;

const preprocessQuery = ({
  rawQuery,
  startTime,
  endTime,
  timeField,
  isLiveQuery,
  selectedPatternField,
  patternRegex,
  filteredPattern
}) => {
  var _datemath$parse, _datemath$parse2;

  let finalQuery = '';
  if ((0, _lodash.isEmpty)(rawQuery)) return finalQuery; // convert to moment

  const start = (_datemath$parse = _datemath.default.parse(startTime)) === null || _datemath$parse === void 0 ? void 0 : _datemath$parse.utc().format(_explorer.DATE_PICKER_FORMAT);
  const end = (_datemath$parse2 = _datemath.default.parse(endTime, {
    roundUp: true
  })) === null || _datemath$parse2 === void 0 ? void 0 : _datemath$parse2.utc().format(_explorer.DATE_PICKER_FORMAT);
  const tokens = rawQuery.replaceAll(_shared.PPL_NEWLINE_REGEX, '').match(_shared.PPL_INDEX_INSERT_POINT_REGEX);
  if ((0, _lodash.isEmpty)(tokens)) return finalQuery;
  finalQuery = `${tokens[1]}=${tokens[2]} | where ${timeField} >= '${start}' and ${timeField} <= '${end}'${tokens[3]}`;

  if (isLiveQuery) {
    finalQuery = finalQuery + ` | sort - ${timeField}`;
  } // if a pattern is selected as filter, build it into finalQuery


  if (selectedPatternField && filteredPattern) finalQuery = buildPatternsQuery(finalQuery, selectedPatternField, patternRegex, filteredPattern);
  return finalQuery;
};

exports.preprocessQuery = preprocessQuery;

const buildPatternsQuery = (baseQuery, selectedPatternField, patternRegex, filteredPattern) => {
  let finalQuery = baseQuery;

  if (selectedPatternField) {
    finalQuery += ` | patterns `;

    if (patternRegex && patternRegex !== _explorer.PPL_DEFAULT_PATTERN_REGEX_FILETER) {
      finalQuery += `pattern='${escapeQuotes(patternRegex)}' `;
    }

    finalQuery += `\`${selectedPatternField}\` `;

    if (filteredPattern) {
      finalQuery += `| where patterns_field='${escapeQuotes(filteredPattern)}'`;
    }
  }

  return finalQuery;
};

exports.buildPatternsQuery = buildPatternsQuery;

const buildQuery = (baseQuery, currQuery) => {
  let fullQuery;

  if (baseQuery) {
    fullQuery = baseQuery;

    if (currQuery) {
      fullQuery += '| ' + currQuery;
    }
  } else {
    fullQuery = currQuery;
  }

  return fullQuery;
};

exports.buildQuery = buildQuery;

const composeFinalQuery = (curQuery, startingTime, endingTime, timeField, isLiveQuery, appBaseQuery, selectedPatternField, patternRegex, filteredPattern) => {
  const fullQuery = curQuery.includes(appBaseQuery) ? curQuery : buildQuery(appBaseQuery, curQuery);
  if ((0, _lodash.isEmpty)(fullQuery)) return '';
  return preprocessQuery({
    rawQuery: fullQuery,
    startTime: startingTime,
    endTime: endingTime,
    timeField,
    isLiveQuery,
    selectedPatternField,
    patternRegex,
    filteredPattern
  });
};

exports.composeFinalQuery = composeFinalQuery;

const removeBacktick = stringContainsBacktick => {
  if (!stringContainsBacktick) return '';
  return stringContainsBacktick.replace(/`/g, '');
};

exports.removeBacktick = removeBacktick;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInF1ZXJ5X3V0aWxzLnRzIl0sIm5hbWVzIjpbImVzY2FwZVF1b3RlcyIsImxpdGVyYWwiLCJyZXBsYWNlQWxsIiwiZ2V0SW5kZXhQYXR0ZXJuRnJvbVJhd1F1ZXJ5IiwicXVlcnkiLCJtYXRjaGVzIiwibWF0Y2giLCJQUExfSU5ERVhfUkVHRVgiLCJwcmVwcm9jZXNzUXVlcnkiLCJyYXdRdWVyeSIsInN0YXJ0VGltZSIsImVuZFRpbWUiLCJ0aW1lRmllbGQiLCJpc0xpdmVRdWVyeSIsInNlbGVjdGVkUGF0dGVybkZpZWxkIiwicGF0dGVyblJlZ2V4IiwiZmlsdGVyZWRQYXR0ZXJuIiwiZmluYWxRdWVyeSIsInN0YXJ0IiwiZGF0ZW1hdGgiLCJwYXJzZSIsInV0YyIsImZvcm1hdCIsIkRBVEVfUElDS0VSX0ZPUk1BVCIsImVuZCIsInJvdW5kVXAiLCJ0b2tlbnMiLCJQUExfTkVXTElORV9SRUdFWCIsIlBQTF9JTkRFWF9JTlNFUlRfUE9JTlRfUkVHRVgiLCJidWlsZFBhdHRlcm5zUXVlcnkiLCJiYXNlUXVlcnkiLCJQUExfREVGQVVMVF9QQVRURVJOX1JFR0VYX0ZJTEVURVIiLCJidWlsZFF1ZXJ5IiwiY3VyclF1ZXJ5IiwiZnVsbFF1ZXJ5IiwiY29tcG9zZUZpbmFsUXVlcnkiLCJjdXJRdWVyeSIsInN0YXJ0aW5nVGltZSIsImVuZGluZ1RpbWUiLCJhcHBCYXNlUXVlcnkiLCJpbmNsdWRlcyIsInJlbW92ZUJhY2t0aWNrIiwic3RyaW5nQ29udGFpbnNCYWNrdGljayIsInJlcGxhY2UiXSwibWFwcGluZ3MiOiI7Ozs7Ozs7QUFLQTs7QUFDQTs7QUFDQTs7QUFJQTs7OztBQVhBO0FBQ0E7QUFDQTtBQUNBOztBQWVBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsTUFBTUEsWUFBWSxHQUFJQyxPQUFELElBQXFCO0FBQ3hDLFNBQU9BLE9BQU8sQ0FBQ0MsVUFBUixDQUFtQixHQUFuQixFQUF3QixJQUF4QixDQUFQO0FBQ0QsQ0FGRDs7QUFJTyxNQUFNQywyQkFBMkIsR0FBSUMsS0FBRCxJQUEyQjtBQUNwRSxRQUFNQyxPQUFPLEdBQUdELEtBQUssQ0FBQ0UsS0FBTixDQUFZQyx1QkFBWixDQUFoQjs7QUFDQSxNQUFJRixPQUFKLEVBQWE7QUFDWCxXQUFPQSxPQUFPLENBQUMsQ0FBRCxDQUFkO0FBQ0Q7O0FBQ0QsU0FBTyxFQUFQO0FBQ0QsQ0FOTSxDLENBUVA7Ozs7O0FBQ08sTUFBTUcsZUFBZSxHQUFHLENBQUM7QUFDOUJDLEVBQUFBLFFBRDhCO0FBRTlCQyxFQUFBQSxTQUY4QjtBQUc5QkMsRUFBQUEsT0FIOEI7QUFJOUJDLEVBQUFBLFNBSjhCO0FBSzlCQyxFQUFBQSxXQUw4QjtBQU05QkMsRUFBQUEsb0JBTjhCO0FBTzlCQyxFQUFBQSxZQVA4QjtBQVE5QkMsRUFBQUE7QUFSOEIsQ0FBRCxLQWtCekI7QUFBQTs7QUFDSixNQUFJQyxVQUFVLEdBQUcsRUFBakI7QUFFQSxNQUFJLHFCQUFRUixRQUFSLENBQUosRUFBdUIsT0FBT1EsVUFBUCxDQUhuQixDQUtKOztBQUNBLFFBQU1DLEtBQUssc0JBQUdDLGtCQUFTQyxLQUFULENBQWVWLFNBQWYsQ0FBSCxvREFBRyxnQkFBMkJXLEdBQTNCLEdBQWlDQyxNQUFqQyxDQUF3Q0MsNEJBQXhDLENBQWQ7QUFDQSxRQUFNQyxHQUFHLHVCQUFHTCxrQkFBU0MsS0FBVCxDQUFlVCxPQUFmLEVBQXdCO0FBQUVjLElBQUFBLE9BQU8sRUFBRTtBQUFYLEdBQXhCLENBQUgscURBQUcsaUJBQTRDSixHQUE1QyxHQUFrREMsTUFBbEQsQ0FBeURDLDRCQUF6RCxDQUFaO0FBQ0EsUUFBTUcsTUFBTSxHQUFHakIsUUFBUSxDQUFDUCxVQUFULENBQW9CeUIseUJBQXBCLEVBQXVDLEVBQXZDLEVBQTJDckIsS0FBM0MsQ0FBaURzQixvQ0FBakQsQ0FBZjtBQUVBLE1BQUkscUJBQVFGLE1BQVIsQ0FBSixFQUFxQixPQUFPVCxVQUFQO0FBRXJCQSxFQUFBQSxVQUFVLEdBQUksR0FBRVMsTUFBTSxDQUFFLENBQUYsQ0FBSyxJQUN6QkEsTUFBTSxDQUFFLENBQUYsQ0FDUCxZQUFXZCxTQUFVLFFBQU9NLEtBQU0sU0FBUU4sU0FBVSxRQUFPWSxHQUFJLElBQUdFLE1BQU0sQ0FBRSxDQUFGLENBQUssRUFGOUU7O0FBSUEsTUFBSWIsV0FBSixFQUFpQjtBQUNmSSxJQUFBQSxVQUFVLEdBQUdBLFVBQVUsR0FBSSxhQUFZTCxTQUFVLEVBQWpEO0FBQ0QsR0FsQkcsQ0FvQko7OztBQUNBLE1BQUlFLG9CQUFvQixJQUFJRSxlQUE1QixFQUNFQyxVQUFVLEdBQUdZLGtCQUFrQixDQUM3QlosVUFENkIsRUFFN0JILG9CQUY2QixFQUc3QkMsWUFINkIsRUFJN0JDLGVBSjZCLENBQS9CO0FBT0YsU0FBT0MsVUFBUDtBQUNELENBaERNOzs7O0FBa0RBLE1BQU1ZLGtCQUFrQixHQUFHLENBQ2hDQyxTQURnQyxFQUVoQ2hCLG9CQUZnQyxFQUdoQ0MsWUFIZ0MsRUFJaENDLGVBSmdDLEtBSzdCO0FBQ0gsTUFBSUMsVUFBVSxHQUFHYSxTQUFqQjs7QUFDQSxNQUFJaEIsb0JBQUosRUFBMEI7QUFDeEJHLElBQUFBLFVBQVUsSUFBSyxjQUFmOztBQUNBLFFBQUlGLFlBQVksSUFBSUEsWUFBWSxLQUFLZ0IsMkNBQXJDLEVBQXdFO0FBQ3RFZCxNQUFBQSxVQUFVLElBQUssWUFBV2pCLFlBQVksQ0FBQ2UsWUFBRCxDQUFlLElBQXJEO0FBQ0Q7O0FBQ0RFLElBQUFBLFVBQVUsSUFBSyxLQUFJSCxvQkFBcUIsS0FBeEM7O0FBQ0EsUUFBSUUsZUFBSixFQUFxQjtBQUNuQkMsTUFBQUEsVUFBVSxJQUFLLDJCQUEwQmpCLFlBQVksQ0FBQ2dCLGVBQUQsQ0FBa0IsR0FBdkU7QUFDRDtBQUNGOztBQUNELFNBQU9DLFVBQVA7QUFDRCxDQWxCTTs7OztBQW9CQSxNQUFNZSxVQUFVLEdBQUcsQ0FBQ0YsU0FBRCxFQUFvQkcsU0FBcEIsS0FBMEM7QUFDbEUsTUFBSUMsU0FBSjs7QUFDQSxNQUFJSixTQUFKLEVBQWU7QUFDYkksSUFBQUEsU0FBUyxHQUFHSixTQUFaOztBQUNBLFFBQUlHLFNBQUosRUFBZTtBQUNiQyxNQUFBQSxTQUFTLElBQUksT0FBT0QsU0FBcEI7QUFDRDtBQUNGLEdBTEQsTUFLTztBQUNMQyxJQUFBQSxTQUFTLEdBQUdELFNBQVo7QUFDRDs7QUFDRCxTQUFPQyxTQUFQO0FBQ0QsQ0FYTTs7OztBQWFBLE1BQU1DLGlCQUFpQixHQUFHLENBQy9CQyxRQUQrQixFQUUvQkMsWUFGK0IsRUFHL0JDLFVBSCtCLEVBSS9CMUIsU0FKK0IsRUFLL0JDLFdBTCtCLEVBTS9CMEIsWUFOK0IsRUFPL0J6QixvQkFQK0IsRUFRL0JDLFlBUitCLEVBUy9CQyxlQVQrQixLQVU1QjtBQUNILFFBQU1rQixTQUFTLEdBQUdFLFFBQVEsQ0FBQ0ksUUFBVCxDQUFrQkQsWUFBbEIsSUFBa0NILFFBQWxDLEdBQTZDSixVQUFVLENBQUNPLFlBQUQsRUFBZUgsUUFBZixDQUF6RTtBQUNBLE1BQUkscUJBQVFGLFNBQVIsQ0FBSixFQUF3QixPQUFPLEVBQVA7QUFDeEIsU0FBTzFCLGVBQWUsQ0FBQztBQUNyQkMsSUFBQUEsUUFBUSxFQUFFeUIsU0FEVztBQUVyQnhCLElBQUFBLFNBQVMsRUFBRTJCLFlBRlU7QUFHckIxQixJQUFBQSxPQUFPLEVBQUUyQixVQUhZO0FBSXJCMUIsSUFBQUEsU0FKcUI7QUFLckJDLElBQUFBLFdBTHFCO0FBTXJCQyxJQUFBQSxvQkFOcUI7QUFPckJDLElBQUFBLFlBUHFCO0FBUXJCQyxJQUFBQTtBQVJxQixHQUFELENBQXRCO0FBVUQsQ0F2Qk07Ozs7QUF5QkEsTUFBTXlCLGNBQWMsR0FBSUMsc0JBQUQsSUFBb0M7QUFDaEUsTUFBSSxDQUFDQSxzQkFBTCxFQUE2QixPQUFPLEVBQVA7QUFDN0IsU0FBT0Esc0JBQXNCLENBQUNDLE9BQXZCLENBQStCLElBQS9CLEVBQXFDLEVBQXJDLENBQVA7QUFDRCxDQUhNIiwic291cmNlc0NvbnRlbnQiOlsiLypcbiAqIENvcHlyaWdodCBPcGVuU2VhcmNoIENvbnRyaWJ1dG9yc1xuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqL1xuXG5pbXBvcnQgZGF0ZW1hdGggZnJvbSAnQGVsYXN0aWMvZGF0ZW1hdGgnO1xuaW1wb3J0IHsgaXNFbXB0eSB9IGZyb20gJ2xvZGFzaCc7XG5pbXBvcnQge1xuICBEQVRFX1BJQ0tFUl9GT1JNQVQsXG4gIFBQTF9ERUZBVUxUX1BBVFRFUk5fUkVHRVhfRklMRVRFUixcbn0gZnJvbSAnLi4vLi4vY29tbW9uL2NvbnN0YW50cy9leHBsb3Jlcic7XG5pbXBvcnQge1xuICBQUExfSU5ERVhfSU5TRVJUX1BPSU5UX1JFR0VYLFxuICBQUExfSU5ERVhfUkVHRVgsXG4gIFBQTF9ORVdMSU5FX1JFR0VYLFxuICBQUExfU1RBVFNfUkVHRVgsXG59IGZyb20gJy4uLy4uL2NvbW1vbi9jb25zdGFudHMvc2hhcmVkJztcblxuLyoqXG4gKiBAcGFyYW0gbGl0ZXJhbCAtIHN0cmluZyBsaXRlcmFsIHRoYXQgd2lsbCBiZSBwdXQgaW5zaWRlIHNpbmdsZSBxdW90ZXMgaW4gUFBMIGNvbW1hbmRcbiAqIEByZXR1cm5zIHN0cmluZyB3aXRoIGlubmVyIHNpbmdsZSBxdW90ZXMgZXNjYXBlZFxuICovXG5jb25zdCBlc2NhcGVRdW90ZXMgPSAobGl0ZXJhbDogc3RyaW5nKSA9PiB7XG4gIHJldHVybiBsaXRlcmFsLnJlcGxhY2VBbGwoXCInXCIsIFwiJydcIik7XG59O1xuXG5leHBvcnQgY29uc3QgZ2V0SW5kZXhQYXR0ZXJuRnJvbVJhd1F1ZXJ5ID0gKHF1ZXJ5OiBzdHJpbmcpOiBzdHJpbmcgPT4ge1xuICBjb25zdCBtYXRjaGVzID0gcXVlcnkubWF0Y2goUFBMX0lOREVYX1JFR0VYKTtcbiAgaWYgKG1hdGNoZXMpIHtcbiAgICByZXR1cm4gbWF0Y2hlc1syXTtcbiAgfVxuICByZXR1cm4gJyc7XG59O1xuXG4vLyBpbnNlcnQgdGltZSBmaWx0ZXIgY29tbWFuZCBhbmQgYWRkaXRpb25hbCBjb21tYW5kcyBiYXNlZCBvbiByYXcgcXVlcnlcbmV4cG9ydCBjb25zdCBwcmVwcm9jZXNzUXVlcnkgPSAoe1xuICByYXdRdWVyeSxcbiAgc3RhcnRUaW1lLFxuICBlbmRUaW1lLFxuICB0aW1lRmllbGQsXG4gIGlzTGl2ZVF1ZXJ5LFxuICBzZWxlY3RlZFBhdHRlcm5GaWVsZCxcbiAgcGF0dGVyblJlZ2V4LFxuICBmaWx0ZXJlZFBhdHRlcm4sXG59OiB7XG4gIHJhd1F1ZXJ5OiBzdHJpbmc7XG4gIHN0YXJ0VGltZTogc3RyaW5nO1xuICBlbmRUaW1lOiBzdHJpbmc7XG4gIHRpbWVGaWVsZD86IHN0cmluZztcbiAgaXNMaXZlUXVlcnk6IGJvb2xlYW47XG4gIHNlbGVjdGVkUGF0dGVybkZpZWxkPzogc3RyaW5nO1xuICBwYXR0ZXJuUmVnZXg/OiBzdHJpbmc7XG4gIGZpbHRlcmVkUGF0dGVybj86IHN0cmluZztcbn0pID0+IHtcbiAgbGV0IGZpbmFsUXVlcnkgPSAnJztcblxuICBpZiAoaXNFbXB0eShyYXdRdWVyeSkpIHJldHVybiBmaW5hbFF1ZXJ5O1xuXG4gIC8vIGNvbnZlcnQgdG8gbW9tZW50XG4gIGNvbnN0IHN0YXJ0ID0gZGF0ZW1hdGgucGFyc2Uoc3RhcnRUaW1lKT8udXRjKCkuZm9ybWF0KERBVEVfUElDS0VSX0ZPUk1BVCk7XG4gIGNvbnN0IGVuZCA9IGRhdGVtYXRoLnBhcnNlKGVuZFRpbWUsIHsgcm91bmRVcDogdHJ1ZSB9KT8udXRjKCkuZm9ybWF0KERBVEVfUElDS0VSX0ZPUk1BVCk7XG4gIGNvbnN0IHRva2VucyA9IHJhd1F1ZXJ5LnJlcGxhY2VBbGwoUFBMX05FV0xJTkVfUkVHRVgsICcnKS5tYXRjaChQUExfSU5ERVhfSU5TRVJUX1BPSU5UX1JFR0VYKTtcblxuICBpZiAoaXNFbXB0eSh0b2tlbnMpKSByZXR1cm4gZmluYWxRdWVyeTtcblxuICBmaW5hbFF1ZXJ5ID0gYCR7dG9rZW5zIVsxXX09JHtcbiAgICB0b2tlbnMhWzJdXG4gIH0gfCB3aGVyZSAke3RpbWVGaWVsZH0gPj0gJyR7c3RhcnR9JyBhbmQgJHt0aW1lRmllbGR9IDw9ICcke2VuZH0nJHt0b2tlbnMhWzNdfWA7XG5cbiAgaWYgKGlzTGl2ZVF1ZXJ5KSB7XG4gICAgZmluYWxRdWVyeSA9IGZpbmFsUXVlcnkgKyBgIHwgc29ydCAtICR7dGltZUZpZWxkfWA7XG4gIH1cblxuICAvLyBpZiBhIHBhdHRlcm4gaXMgc2VsZWN0ZWQgYXMgZmlsdGVyLCBidWlsZCBpdCBpbnRvIGZpbmFsUXVlcnlcbiAgaWYgKHNlbGVjdGVkUGF0dGVybkZpZWxkICYmIGZpbHRlcmVkUGF0dGVybilcbiAgICBmaW5hbFF1ZXJ5ID0gYnVpbGRQYXR0ZXJuc1F1ZXJ5KFxuICAgICAgZmluYWxRdWVyeSxcbiAgICAgIHNlbGVjdGVkUGF0dGVybkZpZWxkLFxuICAgICAgcGF0dGVyblJlZ2V4LFxuICAgICAgZmlsdGVyZWRQYXR0ZXJuXG4gICAgKTtcblxuICByZXR1cm4gZmluYWxRdWVyeTtcbn07XG5cbmV4cG9ydCBjb25zdCBidWlsZFBhdHRlcm5zUXVlcnkgPSAoXG4gIGJhc2VRdWVyeTogc3RyaW5nLFxuICBzZWxlY3RlZFBhdHRlcm5GaWVsZD86IHN0cmluZyxcbiAgcGF0dGVyblJlZ2V4Pzogc3RyaW5nLFxuICBmaWx0ZXJlZFBhdHRlcm4/OiBzdHJpbmdcbikgPT4ge1xuICBsZXQgZmluYWxRdWVyeSA9IGJhc2VRdWVyeTtcbiAgaWYgKHNlbGVjdGVkUGF0dGVybkZpZWxkKSB7XG4gICAgZmluYWxRdWVyeSArPSBgIHwgcGF0dGVybnMgYDtcbiAgICBpZiAocGF0dGVyblJlZ2V4ICYmIHBhdHRlcm5SZWdleCAhPT0gUFBMX0RFRkFVTFRfUEFUVEVSTl9SRUdFWF9GSUxFVEVSKSB7XG4gICAgICBmaW5hbFF1ZXJ5ICs9IGBwYXR0ZXJuPScke2VzY2FwZVF1b3RlcyhwYXR0ZXJuUmVnZXgpfScgYDtcbiAgICB9XG4gICAgZmluYWxRdWVyeSArPSBgXFxgJHtzZWxlY3RlZFBhdHRlcm5GaWVsZH1cXGAgYDtcbiAgICBpZiAoZmlsdGVyZWRQYXR0ZXJuKSB7XG4gICAgICBmaW5hbFF1ZXJ5ICs9IGB8IHdoZXJlIHBhdHRlcm5zX2ZpZWxkPScke2VzY2FwZVF1b3RlcyhmaWx0ZXJlZFBhdHRlcm4pfSdgO1xuICAgIH1cbiAgfVxuICByZXR1cm4gZmluYWxRdWVyeTtcbn07XG5cbmV4cG9ydCBjb25zdCBidWlsZFF1ZXJ5ID0gKGJhc2VRdWVyeTogc3RyaW5nLCBjdXJyUXVlcnk6IHN0cmluZykgPT4ge1xuICBsZXQgZnVsbFF1ZXJ5OiBzdHJpbmc7XG4gIGlmIChiYXNlUXVlcnkpIHtcbiAgICBmdWxsUXVlcnkgPSBiYXNlUXVlcnk7XG4gICAgaWYgKGN1cnJRdWVyeSkge1xuICAgICAgZnVsbFF1ZXJ5ICs9ICd8ICcgKyBjdXJyUXVlcnk7XG4gICAgfVxuICB9IGVsc2Uge1xuICAgIGZ1bGxRdWVyeSA9IGN1cnJRdWVyeTtcbiAgfVxuICByZXR1cm4gZnVsbFF1ZXJ5O1xufTtcblxuZXhwb3J0IGNvbnN0IGNvbXBvc2VGaW5hbFF1ZXJ5ID0gKFxuICBjdXJRdWVyeTogc3RyaW5nLFxuICBzdGFydGluZ1RpbWU6IHN0cmluZyxcbiAgZW5kaW5nVGltZTogc3RyaW5nLFxuICB0aW1lRmllbGQ6IHN0cmluZyxcbiAgaXNMaXZlUXVlcnk6IGJvb2xlYW4sXG4gIGFwcEJhc2VRdWVyeTogc3RyaW5nLFxuICBzZWxlY3RlZFBhdHRlcm5GaWVsZD86IHN0cmluZyxcbiAgcGF0dGVyblJlZ2V4Pzogc3RyaW5nLFxuICBmaWx0ZXJlZFBhdHRlcm4/OiBzdHJpbmdcbikgPT4ge1xuICBjb25zdCBmdWxsUXVlcnkgPSBjdXJRdWVyeS5pbmNsdWRlcyhhcHBCYXNlUXVlcnkpID8gY3VyUXVlcnkgOiBidWlsZFF1ZXJ5KGFwcEJhc2VRdWVyeSwgY3VyUXVlcnkpO1xuICBpZiAoaXNFbXB0eShmdWxsUXVlcnkpKSByZXR1cm4gJyc7XG4gIHJldHVybiBwcmVwcm9jZXNzUXVlcnkoe1xuICAgIHJhd1F1ZXJ5OiBmdWxsUXVlcnksXG4gICAgc3RhcnRUaW1lOiBzdGFydGluZ1RpbWUsXG4gICAgZW5kVGltZTogZW5kaW5nVGltZSxcbiAgICB0aW1lRmllbGQsXG4gICAgaXNMaXZlUXVlcnksXG4gICAgc2VsZWN0ZWRQYXR0ZXJuRmllbGQsXG4gICAgcGF0dGVyblJlZ2V4LFxuICAgIGZpbHRlcmVkUGF0dGVybixcbiAgfSk7XG59O1xuXG5leHBvcnQgY29uc3QgcmVtb3ZlQmFja3RpY2sgPSAoc3RyaW5nQ29udGFpbnNCYWNrdGljazogc3RyaW5nKSA9PiB7XG4gIGlmICghc3RyaW5nQ29udGFpbnNCYWNrdGljaykgcmV0dXJuICcnO1xuICByZXR1cm4gc3RyaW5nQ29udGFpbnNCYWNrdGljay5yZXBsYWNlKC9gL2csICcnKTtcbn07XG4iXX0=