"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SecurityClient = void 0;

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class SecurityClient {
  constructor(esClient) {
    this.esClient = esClient;
  }

  async authenticate(request, credentials) {
    const authHeader = Buffer.from(`${credentials.username}:${credentials.password}`).toString('base64');

    try {
      const esResponse = await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        headers: {
          authorization: `Basic ${authHeader}`
        }
      });
      return {
        username: credentials.username,
        roles: esResponse.roles,
        backendRoles: esResponse.backend_roles,
        tenants: esResponse.tenants,
        selectedTenant: esResponse.user_requested_tenant,
        credentials,
        proxyCredentials: credentials
      };
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async authenticateWithHeader(request, headerName, headerValue, whitelistedHeadersAndValues = {}, additionalAuthHeaders = {}) {
    try {
      const credentials = {
        headerName,
        headerValue
      };
      const headers = {};

      if (headerValue) {
        headers[headerName] = headerValue;
      } // cannot get config elasticsearch.requestHeadersWhitelist from kibana.yml file in new platfrom
      // meanwhile, do we really need to save all headers in cookie?


      const esResponse = await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        headers
      });
      return {
        username: esResponse.user_name,
        roles: esResponse.roles,
        backendRoles: esResponse.backend_roles,
        tenants: esResponse.teanats,
        selectedTenant: esResponse.user_requested_tenant,
        credentials
      };
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async authenticateWithHeaders(request, additionalAuthHeaders = {}) {
    try {
      const esResponse = await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        headers: additionalAuthHeaders
      });
      return {
        username: esResponse.user_name,
        roles: esResponse.roles,
        backendRoles: esResponse.backend_roles,
        tenants: esResponse.tenants,
        selectedTenant: esResponse.user_requested_tenant
      };
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async authinfo(request, headers = {}) {
    try {
      return await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        headers
      });
    } catch (error) {
      throw new Error(error.message);
    }
  } // Multi-tenancy APIs


  async getMultitenancyInfo(request) {
    try {
      return await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.multitenancyinfo');
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async getTenantInfoWithInternalUser() {
    try {
      return this.esClient.callAsInternalUser('opensearch_security.tenantinfo');
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async getTenantInfo(request) {
    try {
      return await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.tenantinfo');
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async getSamlHeader(request) {
    try {
      // response is expected to be an error
      await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo');
    } catch (error) {
      // the error looks like
      // wwwAuthenticateDirective:
      //   '
      //     X-Security-IdP realm="Open Distro Security"
      //     location="https://<your-auth-domain.com>/api/saml2/v1/sso?SAMLRequest=<some-encoded-string>"
      //     requestId="<request_id>"
      //   '
      if (!error.wwwAuthenticateDirective) {
        throw error;
      }

      try {
        const locationRegExp = /location="(.*?)"/;
        const requestIdRegExp = /requestId="(.*?)"/;
        const locationExecArray = locationRegExp.exec(error.wwwAuthenticateDirective);
        const requestExecArray = requestIdRegExp.exec(error.wwwAuthenticateDirective);

        if (locationExecArray && requestExecArray) {
          return {
            location: locationExecArray[1],
            requestId: requestExecArray[1]
          };
        }

        throw Error('failed parsing SAML config');
      } catch (parsingError) {
        console.log(parsingError);
        throw new Error(parsingError);
      }
    }

    throw new Error(`Invalid SAML configuration.`);
  }

  async authToken(requestId, samlResponse, acsEndpoint = undefined) {
    const body = {
      RequestId: requestId,
      SAMLResponse: samlResponse,
      acsEndpoint
    };

    try {
      return await this.esClient.asScoped().callAsCurrentUser('opensearch_security.authtoken', {
        body
      });
    } catch (error) {
      console.log(error);
      throw new Error('failed to get token');
    }
  }

}

exports.SecurityClient = SecurityClient;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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