"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DiagnosticsProviderImpl = void 0;
const typescript_1 = __importDefault(require("typescript"));
const vscode_languageserver_1 = require("vscode-languageserver");
const documents_1 = require("../../../lib/documents");
const utils_1 = require("../utils");
const utils_2 = require("./utils");
const utils_3 = require("../../../utils");
const svelte_ast_utils_1 = require("../svelte-ast-utils");
var DiagnosticCode;
(function (DiagnosticCode) {
    DiagnosticCode[DiagnosticCode["MODIFIERS_CANNOT_APPEAR_HERE"] = 1184] = "MODIFIERS_CANNOT_APPEAR_HERE";
    DiagnosticCode[DiagnosticCode["USED_BEFORE_ASSIGNED"] = 2454] = "USED_BEFORE_ASSIGNED";
    DiagnosticCode[DiagnosticCode["JSX_ELEMENT_DOES_NOT_SUPPORT_ATTRIBUTES"] = 2607] = "JSX_ELEMENT_DOES_NOT_SUPPORT_ATTRIBUTES";
    DiagnosticCode[DiagnosticCode["CANNOT_BE_USED_AS_JSX_COMPONENT"] = 2786] = "CANNOT_BE_USED_AS_JSX_COMPONENT";
    DiagnosticCode[DiagnosticCode["NOOP_IN_COMMAS"] = 2695] = "NOOP_IN_COMMAS";
    DiagnosticCode[DiagnosticCode["NEVER_READ"] = 6133] = "NEVER_READ";
    DiagnosticCode[DiagnosticCode["ALL_IMPORTS_UNUSED"] = 6192] = "ALL_IMPORTS_UNUSED";
    DiagnosticCode[DiagnosticCode["UNUSED_LABEL"] = 7028] = "UNUSED_LABEL";
    DiagnosticCode[DiagnosticCode["DUPLICATED_JSX_ATTRIBUTES"] = 17001] = "DUPLICATED_JSX_ATTRIBUTES";
    DiagnosticCode[DiagnosticCode["DUPLICATE_IDENTIFIER"] = 2300] = "DUPLICATE_IDENTIFIER";
    DiagnosticCode[DiagnosticCode["MULTIPLE_PROPS_SAME_NAME"] = 1117] = "MULTIPLE_PROPS_SAME_NAME";
    DiagnosticCode[DiagnosticCode["TYPE_X_NOT_ASSIGNABLE_TO_TYPE_Y"] = 2345] = "TYPE_X_NOT_ASSIGNABLE_TO_TYPE_Y";
    DiagnosticCode[DiagnosticCode["MISSING_PROPS"] = 2739] = "MISSING_PROPS";
    DiagnosticCode[DiagnosticCode["MISSING_PROP"] = 2741] = "MISSING_PROP";
    DiagnosticCode[DiagnosticCode["NO_OVERLOAD_MATCHES_CALL"] = 2769] = "NO_OVERLOAD_MATCHES_CALL"; // "No overload matches this call"
})(DiagnosticCode || (DiagnosticCode = {}));
class DiagnosticsProviderImpl {
    constructor(lsAndTsDocResolver, configManager) {
        this.lsAndTsDocResolver = lsAndTsDocResolver;
        this.configManager = configManager;
    }
    async getDiagnostics(document, cancellationToken) {
        const { lang, tsDoc } = await this.getLSAndTSDoc(document);
        if (['coffee', 'coffeescript'].includes(document.getLanguageAttribute('script')) ||
            (cancellationToken === null || cancellationToken === void 0 ? void 0 : cancellationToken.isCancellationRequested)) {
            return [];
        }
        const isTypescript = tsDoc.scriptKind === typescript_1.default.ScriptKind.TSX || tsDoc.scriptKind === typescript_1.default.ScriptKind.TS;
        // Document preprocessing failed, show parser error instead
        if (tsDoc.parserError) {
            return [
                {
                    range: tsDoc.parserError.range,
                    severity: vscode_languageserver_1.DiagnosticSeverity.Error,
                    source: isTypescript ? 'ts' : 'js',
                    message: tsDoc.parserError.message,
                    code: tsDoc.parserError.code
                }
            ];
        }
        let diagnostics = [
            ...lang.getSyntacticDiagnostics(tsDoc.filePath),
            ...lang.getSuggestionDiagnostics(tsDoc.filePath),
            ...lang.getSemanticDiagnostics(tsDoc.filePath)
        ];
        const additionalStoreDiagnostics = [];
        const notGenerated = isNotGenerated(tsDoc.getFullText());
        for (const diagnostic of diagnostics) {
            if ((diagnostic.code === DiagnosticCode.NO_OVERLOAD_MATCHES_CALL ||
                diagnostic.code === DiagnosticCode.TYPE_X_NOT_ASSIGNABLE_TO_TYPE_Y) &&
                !notGenerated(diagnostic)) {
                if ((0, utils_2.isStoreVariableIn$storeDeclaration)(tsDoc.getFullText(), diagnostic.start)) {
                    const storeName = tsDoc
                        .getFullText()
                        .substring(diagnostic.start, diagnostic.start + diagnostic.length);
                    const storeUsages = lang.findReferences(tsDoc.filePath, (0, utils_2.get$storeOffsetOf$storeDeclaration)(tsDoc.getFullText(), diagnostic.start))[0].references;
                    for (const storeUsage of storeUsages) {
                        additionalStoreDiagnostics.push({
                            ...diagnostic,
                            messageText: `Cannot use '${storeName}' as a store. '${storeName}' needs to be an object with a subscribe method on it.\n\n${typescript_1.default.flattenDiagnosticMessageText(diagnostic.messageText, '\n')}`,
                            start: storeUsage.textSpan.start,
                            length: storeUsage.textSpan.length
                        });
                    }
                }
            }
        }
        diagnostics.push(...additionalStoreDiagnostics);
        diagnostics = diagnostics.filter(notGenerated).filter((0, utils_3.not)(isUnusedReactiveStatementLabel));
        diagnostics = resolveNoopsInReactiveStatements(lang, diagnostics);
        return diagnostics
            .map((diagnostic) => ({
            range: (0, utils_1.convertRange)(tsDoc, diagnostic),
            severity: (0, utils_1.mapSeverity)(diagnostic.category),
            source: isTypescript ? 'ts' : 'js',
            message: typescript_1.default.flattenDiagnosticMessageText(diagnostic.messageText, '\n'),
            code: diagnostic.code,
            tags: (0, utils_1.getDiagnosticTag)(diagnostic)
        }))
            .map(mapRange(tsDoc, document, lang))
            .filter(hasNoNegativeLines)
            .filter(isNoFalsePositive(document, tsDoc))
            .map(enhanceIfNecessary)
            .map(swapDiagRangeStartEndIfNecessary);
    }
    async getLSAndTSDoc(document) {
        return this.lsAndTsDocResolver.getLSAndTSDoc(document);
    }
}
exports.DiagnosticsProviderImpl = DiagnosticsProviderImpl;
function mapRange(snapshot, document, lang) {
    const get$$PropsDefWithCache = (0, utils_3.memoize)(() => get$$PropsDef(lang, snapshot));
    const get$$PropsAliasInfoWithCache = (0, utils_3.memoize)(() => get$$PropsAliasForInfo(get$$PropsDefWithCache, lang, document));
    return (diagnostic) => {
        var _a, _b;
        let range = (0, documents_1.mapRangeToOriginal)(snapshot, diagnostic.range);
        if (range.start.line < 0) {
            range =
                (_a = movePropsErrorRangeBackIfNecessary(diagnostic, snapshot, get$$PropsDefWithCache, get$$PropsAliasInfoWithCache)) !== null && _a !== void 0 ? _a : range;
        }
        if ([DiagnosticCode.MISSING_PROP, DiagnosticCode.MISSING_PROPS].includes(diagnostic.code) &&
            !(0, utils_1.hasNonZeroRange)({ range })) {
            const node = (0, documents_1.getNodeIfIsInStartTag)(document.html, document.offsetAt(range.start));
            if (node) {
                // This is a "some prop missing" error on a component -> remap
                range.start = document.positionAt(node.start + 1);
                range.end = document.positionAt(node.start + 1 + (((_b = node.tag) === null || _b === void 0 ? void 0 : _b.length) || 1));
            }
        }
        return { ...diagnostic, range };
    };
}
function findDiagnosticNode(diagnostic) {
    const { file, start, length } = diagnostic;
    if (!file || !start || !length) {
        return;
    }
    const span = { start, length };
    return (0, utils_2.findNodeAtSpan)(file, span);
}
function copyDiagnosticAndChangeNode(diagnostic) {
    return (node) => ({
        ...diagnostic,
        start: node.getStart(),
        length: node.getWidth()
    });
}
/**
 * In some rare cases mapping of diagnostics does not work and produces negative lines.
 * We filter out these diagnostics with negative lines because else the LSP
 * apparently has a hickup and does not show any diagnostics at all.
 */
function hasNoNegativeLines(diagnostic) {
    return diagnostic.range.start.line >= 0 && diagnostic.range.end.line >= 0;
}
function isNoFalsePositive(document, tsDoc) {
    const text = document.getText();
    const usesPug = document.getLanguageAttribute('template') === 'pug';
    return (diagnostic) => {
        if ([DiagnosticCode.MULTIPLE_PROPS_SAME_NAME, DiagnosticCode.DUPLICATE_IDENTIFIER].includes(diagnostic.code)) {
            const node = tsDoc.svelteNodeAt(diagnostic.range.start);
            if ((0, svelte_ast_utils_1.isAttributeName)(node, 'Element') || (0, svelte_ast_utils_1.isEventHandler)(node, 'Element')) {
                return false;
            }
        }
        return (isNoUsedBeforeAssigned(diagnostic, text, tsDoc) &&
            (!usesPug || isNoPugFalsePositive(diagnostic, document)));
    };
}
/**
 * All diagnostics inside the template tag and the unused import/variable diagnostics
 * are marked as false positive.
 */
function isNoPugFalsePositive(diagnostic, document) {
    return (!(0, documents_1.isRangeInTag)(diagnostic.range, document.templateInfo) &&
        diagnostic.code !== DiagnosticCode.NEVER_READ &&
        diagnostic.code !== DiagnosticCode.ALL_IMPORTS_UNUSED);
}
/**
 * Variable used before being assigned, can happen when  you do `export let x`
 * without assigning a value in strict mode. Should not throw an error here
 * but on the component-user-side ("you did not set a required prop").
 */
function isNoUsedBeforeAssigned(diagnostic, text, tsDoc) {
    if (diagnostic.code !== DiagnosticCode.USED_BEFORE_ASSIGNED) {
        return true;
    }
    return !tsDoc.hasProp((0, documents_1.getTextInRange)(diagnostic.range, text));
}
/**
 * Some diagnostics have JSX-specific nomenclature. Enhance them for more clarity.
 */
function enhanceIfNecessary(diagnostic) {
    if (diagnostic.code === DiagnosticCode.TYPE_X_NOT_ASSIGNABLE_TO_TYPE_Y &&
        diagnostic.message.includes('ConstructorOfATypedSvelteComponent')) {
        return {
            ...diagnostic,
            message: diagnostic.message +
                '\n\nPossible causes:\n' +
                '- You use the instance type of a component where you should use the constructor type\n' +
                '- Type definitions are missing for this Svelte Component. ' +
                'If you are using Svelte 3.31+, use SvelteComponentTyped to add a definition:\n' +
                '  import type { SvelteComponentTyped } from "svelte";\n' +
                '  class ComponentName extends SvelteComponentTyped<{propertyName: string;}> {}'
        };
    }
    if (diagnostic.code === DiagnosticCode.MODIFIERS_CANNOT_APPEAR_HERE) {
        return {
            ...diagnostic,
            message: diagnostic.message +
                '\nIf this is a declare statement, move it into <script context="module">..</script>'
        };
    }
    return diagnostic;
}
/**
 * Due to source mapping, some ranges may be swapped: Start is end. Swap back in this case.
 */
function swapDiagRangeStartEndIfNecessary(diag) {
    diag.range = (0, utils_3.swapRangeStartEndIfNecessary)(diag.range);
    return diag;
}
/**
 * Checks if diagnostic is not within a section that should be completely ignored
 * because it's purely generated.
 */
function isNotGenerated(text) {
    return (diagnostic) => {
        if (diagnostic.start === undefined || diagnostic.length === undefined) {
            return true;
        }
        return !(0, utils_2.isInGeneratedCode)(text, diagnostic.start, diagnostic.start + diagnostic.length);
    };
}
function isUnusedReactiveStatementLabel(diagnostic) {
    if (diagnostic.code !== DiagnosticCode.UNUSED_LABEL) {
        return false;
    }
    const diagNode = findDiagnosticNode(diagnostic);
    if (!diagNode) {
        return false;
    }
    // TS warning targets the identifier
    if (!typescript_1.default.isIdentifier(diagNode)) {
        return false;
    }
    if (!diagNode.parent) {
        return false;
    }
    return (0, utils_2.isReactiveStatement)(diagNode.parent);
}
/**
 * Checks if diagnostics should be ignored because they report an unused expression* in
 * a reactive statement, and those actually have side effects in Svelte (hinting deps).
 *
 *     $: x, update()
 *
 * Only `let` (i.e. reactive) variables are ignored. For the others, new diagnostics are
 * emitted, centered on the (non reactive) identifiers in the initial warning.
 */
function resolveNoopsInReactiveStatements(lang, diagnostics) {
    const isLet = (file) => (node) => {
        const defs = lang.getDefinitionAtPosition(file.fileName, node.getStart());
        return !!defs && defs.some((def) => def.fileName === file.fileName && def.kind === 'let');
    };
    const expandRemainingNoopWarnings = (diagnostic) => {
        const { code, file } = diagnostic;
        // guard: missing info
        if (!file) {
            return;
        }
        // guard: not target error
        const isNoopDiag = code === DiagnosticCode.NOOP_IN_COMMAS;
        if (!isNoopDiag) {
            return;
        }
        const diagNode = findDiagnosticNode(diagnostic);
        if (!diagNode) {
            return;
        }
        if (!(0, utils_2.isInReactiveStatement)(diagNode)) {
            return;
        }
        return (
        // for all identifiers in diagnostic node
        (0, utils_2.gatherIdentifiers)(diagNode)
            // ignore `let` (i.e. reactive) variables
            .filter((0, utils_3.not)(isLet(file)))
            // and create targeted diagnostics just for the remaining ids
            .map(copyDiagnosticAndChangeNode(diagnostic)));
    };
    const expandedDiagnostics = (0, utils_3.flatten)((0, utils_3.passMap)(diagnostics, expandRemainingNoopWarnings));
    return expandedDiagnostics.length === diagnostics.length
        ? expandedDiagnostics
        : // This can generate duplicate diagnostics
            expandedDiagnostics.filter(dedupDiagnostics());
}
function dedupDiagnostics() {
    const hashDiagnostic = (diag) => [diag.start, diag.length, diag.category, diag.source, diag.code]
        .map((x) => JSON.stringify(x))
        .join(':');
    const known = new Set();
    return (diag) => {
        const key = hashDiagnostic(diag);
        if (known.has(key)) {
            return false;
        }
        else {
            known.add(key);
            return true;
        }
    };
}
function get$$PropsAliasForInfo(get$$PropsDefWithCache, lang, document) {
    var _a, _b, _c, _d;
    if (!/type\s+\$\$Props[\s\n]+=/.test(document.getText())) {
        return;
    }
    const propsDef = get$$PropsDefWithCache();
    if (!propsDef || !typescript_1.default.isTypeAliasDeclaration(propsDef)) {
        return;
    }
    const type = (_b = (_a = lang.getProgram()) === null || _a === void 0 ? void 0 : _a.getTypeChecker()) === null || _b === void 0 ? void 0 : _b.getTypeAtLocation(propsDef.name);
    if (!type) {
        return;
    }
    // TS says symbol is always defined but it's not
    const rootSymbolName = (_d = ((_c = type.aliasSymbol) !== null && _c !== void 0 ? _c : type.symbol)) === null || _d === void 0 ? void 0 : _d.name;
    if (!rootSymbolName) {
        return;
    }
    return [rootSymbolName, propsDef];
}
function get$$PropsDef(lang, snapshot) {
    var _a;
    const program = lang.getProgram();
    const sourceFile = program === null || program === void 0 ? void 0 : program.getSourceFile(snapshot.filePath);
    if (!program || !sourceFile) {
        return undefined;
    }
    const renderFunction = sourceFile.statements.find((statement) => { var _a; return typescript_1.default.isFunctionDeclaration(statement) && ((_a = statement.name) === null || _a === void 0 ? void 0 : _a.getText()) === 'render'; });
    return (_a = renderFunction === null || renderFunction === void 0 ? void 0 : renderFunction.body) === null || _a === void 0 ? void 0 : _a.statements.find((node) => (typescript_1.default.isTypeAliasDeclaration(node) || typescript_1.default.isInterfaceDeclaration(node)) &&
        node.name.getText() === '$$Props');
}
function movePropsErrorRangeBackIfNecessary(diagnostic, snapshot, get$$PropsDefWithCache, get$$PropsAliasForWithCache) {
    const possibly$$PropsError = (0, utils_2.isAfterSvelte2TsxPropsReturn)(snapshot.getFullText(), snapshot.offsetAt(diagnostic.range.start));
    if (!possibly$$PropsError) {
        return;
    }
    if (diagnostic.message.includes('$$Props')) {
        const propsDef = get$$PropsDefWithCache();
        const generatedPropsStart = propsDef === null || propsDef === void 0 ? void 0 : propsDef.name.getStart();
        const propsStart = generatedPropsStart != null &&
            snapshot.getOriginalPosition(snapshot.positionAt(generatedPropsStart));
        if (propsStart) {
            return {
                start: propsStart,
                end: { ...propsStart, character: propsStart.character + '$$Props'.length }
            };
        }
        return;
    }
    const aliasForInfo = get$$PropsAliasForWithCache();
    if (!aliasForInfo) {
        return;
    }
    const [aliasFor, propsDef] = aliasForInfo;
    if (diagnostic.message.includes(aliasFor)) {
        return (0, documents_1.mapRangeToOriginal)(snapshot, {
            start: snapshot.positionAt(propsDef.name.getStart()),
            end: snapshot.positionAt(propsDef.name.getEnd())
        });
    }
}
//# sourceMappingURL=DiagnosticsProvider.js.map