"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = __importDefault(require("assert"));
const path_1 = __importDefault(require("path"));
const typescript_1 = __importDefault(require("typescript"));
const documents_1 = require("../../../src/lib/documents");
const service_1 = require("../../../src/plugins/typescript/service");
const SnapshotManager_1 = require("../../../src/plugins/typescript/SnapshotManager");
const utils_1 = require("../../../src/utils");
const test_utils_1 = require("./test-utils");
describe('service', () => {
    const testDir = path_1.default.join(__dirname, 'testfiles');
    function setup() {
        const virtualSystem = (0, test_utils_1.createVirtualTsSystem)(testDir);
        const lsDocumentContext = {
            ambientTypesSource: 'svelte2tsx',
            createDocument(fileName, content) {
                return new documents_1.Document((0, utils_1.pathToUrl)(fileName), content);
            },
            extendedConfigCache: new Map(),
            globalSnapshotsManager: new SnapshotManager_1.GlobalSnapshotsManager(virtualSystem),
            transformOnTemplateError: true,
            tsSystem: virtualSystem,
            watchTsConfig: false,
            notifyExceedSizeLimit: undefined,
            onProjectReloaded: undefined
        };
        const rootUris = [(0, utils_1.pathToUrl)(testDir)];
        return { virtualSystem, lsDocumentContext, rootUris };
    }
    it('can find tsconfig and override with default config', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { virtualSystem, lsDocumentContext, rootUris } = setup();
        virtualSystem.writeFile(path_1.default.join(dirPath, 'tsconfig.json'), JSON.stringify({
            compilerOptions: {
                checkJs: true,
                strict: true
            }
        }));
        const ls = await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, lsDocumentContext);
        // ts internal
        delete ls.compilerOptions.configFilePath;
        assert_1.default.deepStrictEqual(ls.compilerOptions, {
            allowJs: true,
            allowNonTsExtensions: true,
            checkJs: true,
            strict: true,
            declaration: false,
            module: typescript_1.default.ModuleKind.ESNext,
            moduleResolution: typescript_1.default.ModuleResolutionKind.NodeJs,
            noEmit: true,
            skipLibCheck: true,
            target: typescript_1.default.ScriptTarget.ESNext
        });
    });
    function createReloadTester(docContext, testAfterReload) {
        let _resolve;
        const reloadPromise = new Promise((resolve) => {
            _resolve = resolve;
        });
        return {
            docContextWithReload: {
                ...docContext,
                async onProjectReloaded() {
                    try {
                        await testAfterReload();
                    }
                    finally {
                        _resolve();
                    }
                }
            },
            reloadPromise
        };
    }
    it('can watch tsconfig', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { virtualSystem, lsDocumentContext, rootUris } = setup();
        const tsconfigPath = path_1.default.join(dirPath, 'tsconfig.json');
        virtualSystem.writeFile(tsconfigPath, JSON.stringify({
            compilerOptions: {
                strict: false
            }
        }));
        const { reloadPromise, docContextWithReload } = createReloadTester({ ...lsDocumentContext, watchTsConfig: true }, testAfterReload);
        await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, docContextWithReload);
        virtualSystem.writeFile(tsconfigPath, JSON.stringify({
            compilerOptions: {
                strict: true
            }
        }));
        await reloadPromise;
        async function testAfterReload() {
            const newLs = await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, {
                ...lsDocumentContext,
                watchTsConfig: true
            });
            assert_1.default.strictEqual(newLs.compilerOptions.strict, true, 'expected to reload compilerOptions');
        }
    });
    it('can watch extended tsconfig', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { virtualSystem, lsDocumentContext, rootUris } = setup();
        const tsconfigPath = path_1.default.join(dirPath, 'tsconfig.json');
        const extend = './.svelte-kit/tsconfig.json';
        const extendedConfigPathFull = path_1.default.resolve(tsconfigPath, extend);
        virtualSystem.writeFile(tsconfigPath, JSON.stringify({
            extends: extend
        }));
        const { reloadPromise, docContextWithReload } = createReloadTester({ ...lsDocumentContext, watchTsConfig: true }, testAfterReload);
        await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, docContextWithReload);
        virtualSystem.writeFile(extendedConfigPathFull, JSON.stringify({
            compilerOptions: {
                strict: true
            }
        }));
        await reloadPromise;
        async function testAfterReload() {
            const newLs = await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, {
                ...lsDocumentContext,
                watchTsConfig: true
            });
            assert_1.default.strictEqual(newLs.compilerOptions.strict, true, 'expected to reload compilerOptions');
        }
    });
});
//# sourceMappingURL=service.test.js.map