package Lire::OutputFormats::LaTeX;

use strict;

use base qw/Lire::OutputFormat/;

use Locale::TextDomain 'lire';

use Lire::Config;
use Lire::ReportParser::LaTeXWriter;
use Lire::ReportParser::ReportBuilder;
use Lire::Utils qw/tempdir/;

eval "use MIME::Entity";

=pod

=head1 NAME

Lire::OutputFormat::LaTeX - LaTeX output format.

=head1 SYNOPSIS

  my $format = Lire::PluginManager->get_plugin( 'output_format', 'latex' );
  $format->format_report( $xml_file, $latex_file, $cfg );

=head1 DESCRIPTION

Format a XML report into LaTeX with charts.

=cut

sub name {
    return 'latex';
}

sub title {
    return __( 'LaTeX' );
}

sub description {
    return '<para>' . __( 'LaTeX report with charts.' ) . '</para>';
}

sub format_report {
    my ( $self, $report_file, $output_file, $cfg ) = @_;

    my $parser = new Lire::ReportParser::ReportBuilder();
    my $report = $parser->parsefile( $report_file );
    my $writer =
      new Lire::ReportParser::LaTeXWriter();
    $writer->write_report( $report, $output_file, $self->name(),
                           Lire::Config->get( 'lr_latex_include' ) );

    return;
}

sub mime_filename {
    return "report.tex";
}

sub mime_report {
    my ( $self, $report_file, $cfg ) = @_;

    my $tmpdir = tempdir( __PACKAGE__ . '_XXXXXX', 'CLEANUP' => 1 );
    my $report = $tmpdir . "/" . $self->mime_filename();
    $self->format_report( $report_file, $report, $cfg );
    opendir my $dh, $tmpdir
      or die "opendir '$tmpdir' failed: $!";
    my @charts = grep { /\.eps$/ } readdir $dh;
    closedir $dh;
    my $entity = MIME::Entity->build( 'Type' => $self->content_type(),
                                      'Charset' => $self->content_type_charset(),
                                      'Encoding' => $self->content_transfer_encoding(),
                                      'Path' => $report );
    if ( @charts ) {
        my $container = MIME::Entity->build( 'Type' => 'multipart/mixed' );
        $container->add_part( $entity );

        foreach my $chart ( @charts ) {
            $container->attach( 'Type' => 'application/postscript',
                                'Path' => "$tmpdir/$chart" );
        }
        return $container;
    }

    return $entity;
}

sub content_type {
    return "text/x-tex";
}

sub content_type_charset {
    return 'utf-8';
}

sub content_transfer_encoding {
    return "quoted-printable";
}

1;

__END__

=pod

=head1 SEE ALSO

 Lire::PluginManager(3pm) Lire::OutputFormat(3pm)

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: LaTeX.pm,v 1.3 2006/07/23 13:16:31 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html.

=cut

