"use strict";
const path_1 = require("path");
const language_service_1 = require("./language-service");
const logger_1 = require("./logger");
const module_loader_1 = require("./module-loader");
const svelte_snapshots_1 = require("./svelte-snapshots");
const config_manager_1 = require("./config-manager");
const project_svelte_files_1 = require("./project-svelte-files");
const utils_1 = require("./utils");
function init(modules) {
    const configManager = new config_manager_1.ConfigManager();
    let resolvedSvelteTsxFiles;
    function create(info) {
        var _a, _b, _c, _d;
        const logger = new logger_1.Logger(info.project.projectService.logger);
        if (!isSvelteProject(info.project.getCompilerOptions())) {
            logger.log('Detected that this is not a Svelte project, abort patching TypeScript');
            return info.languageService;
        }
        if ((0, language_service_1.isPatched)(info.languageService)) {
            logger.log('Already patched. Checking tsconfig updates.');
            (_a = project_svelte_files_1.ProjectSvelteFilesManager.getInstance(info.project.getProjectName())) === null || _a === void 0 ? void 0 : _a.updateProjectConfig(info.languageServiceHost);
            return info.languageService;
        }
        configManager.updateConfigFromPluginConfig(info.config);
        if (configManager.getConfig().enable) {
            logger.log('Starting Svelte plugin');
        }
        else {
            logger.log('Svelte plugin disabled');
            logger.log(info.config);
        }
        // This call the ConfiguredProject.getParsedCommandLine
        // where it'll try to load the cached version of the parsedCommandLine
        const parsedCommandLine = (_c = (_b = info.languageServiceHost).getParsedCommandLine) === null || _c === void 0 ? void 0 : _c.call(_b, (0, utils_1.getConfigPathForProject)(info.project));
        // For some reason it's no longer enough to patch this at the projectService level, so we do it here, too
        // TODO investigate if we can use the script snapshot for all Svelte files, too, enabling Svelte file
        // updates getting picked up without a file save - move this logic into the snapshot manager then?
        const getScriptSnapshot = info.languageServiceHost.getScriptSnapshot.bind(info.languageServiceHost);
        info.languageServiceHost.getScriptSnapshot = (fileName) => {
            const normalizedPath = fileName.replace(/\\/g, '/');
            if (normalizedPath.endsWith('node_modules/svelte/types/runtime/ambient.d.ts')) {
                return modules.typescript.ScriptSnapshot.fromString('');
            }
            else if (normalizedPath.endsWith('svelte2tsx/svelte-jsx.d.ts')) {
                // Remove the dom lib reference to not load these ambient types in case
                // the user has a tsconfig.json with different lib settings like in
                // https://github.com/sveltejs/language-tools/issues/1733
                const snapshot = getScriptSnapshot(fileName);
                if (snapshot) {
                    const originalText = snapshot.getText(0, snapshot.getLength());
                    const toReplace = '/// <reference lib="dom" />';
                    return modules.typescript.ScriptSnapshot.fromString(originalText.replace(toReplace, ' '.repeat(toReplace.length)));
                }
                return snapshot;
            }
            else if (normalizedPath.endsWith('svelte2tsx/svelte-shims.d.ts')) {
                const snapshot = getScriptSnapshot(fileName);
                if (snapshot) {
                    let originalText = snapshot.getText(0, snapshot.getLength());
                    if (!originalText.includes('// -- start svelte-ls-remove --')) {
                        return snapshot; // uses an older version of svelte2tsx or is already patched
                    }
                    const startIdx = originalText.indexOf('// -- start svelte-ls-remove --');
                    const endIdx = originalText.indexOf('// -- end svelte-ls-remove --');
                    originalText =
                        originalText.substring(0, startIdx) +
                            ' '.repeat(endIdx - startIdx) +
                            originalText.substring(endIdx);
                    return modules.typescript.ScriptSnapshot.fromString(originalText);
                }
                return snapshot;
            }
            return getScriptSnapshot(fileName);
        };
        const svelteOptions = ((_d = parsedCommandLine === null || parsedCommandLine === void 0 ? void 0 : parsedCommandLine.raw) === null || _d === void 0 ? void 0 : _d.svelteOptions) || { namespace: 'svelteHTML' };
        logger.log('svelteOptions:', svelteOptions);
        logger.debug(parsedCommandLine === null || parsedCommandLine === void 0 ? void 0 : parsedCommandLine.wildcardDirectories);
        const snapshotManager = new svelte_snapshots_1.SvelteSnapshotManager(modules.typescript, info.project.projectService, svelteOptions, logger, configManager);
        const projectSvelteFilesManager = parsedCommandLine
            ? new project_svelte_files_1.ProjectSvelteFilesManager(modules.typescript, info.project, info.serverHost, snapshotManager, parsedCommandLine, configManager)
            : undefined;
        (0, module_loader_1.patchModuleLoader)(logger, snapshotManager, modules.typescript, info.languageServiceHost, info.project, configManager);
        configManager.onConfigurationChanged(() => {
            // enabling/disabling the plugin means TS has to recompute stuff
            info.languageService.cleanupSemanticCache();
            info.project.markAsDirty();
            // updateGraph checks for new root files
            // if there's no tsconfig there isn't root files to check
            if (projectSvelteFilesManager) {
                info.project.updateGraph();
            }
        });
        return decorateLanguageServiceDispose((0, language_service_1.decorateLanguageService)(info.languageService, snapshotManager, logger, configManager, info, modules.typescript), projectSvelteFilesManager !== null && projectSvelteFilesManager !== void 0 ? projectSvelteFilesManager : {
            dispose() { }
        });
    }
    function getExternalFiles(project) {
        var _a, _b;
        if (!isSvelteProject(project.getCompilerOptions()) || !configManager.getConfig().enable) {
            return [];
        }
        // Needed so the ambient definitions are known inside the tsx files
        const svelteTsxFiles = resolveSvelteTsxFiles();
        if (!project.getCompilerOptions().configFilePath) {
            svelteTsxFiles.forEach((file) => {
                openSvelteTsxFileForInferredProject(project, file);
            });
        }
        // let ts know project svelte files to do its optimization
        return svelteTsxFiles.concat((_b = (_a = project_svelte_files_1.ProjectSvelteFilesManager.getInstance(project.getProjectName())) === null || _a === void 0 ? void 0 : _a.getFiles()) !== null && _b !== void 0 ? _b : []);
    }
    function resolveSvelteTsxFiles() {
        if (resolvedSvelteTsxFiles) {
            return resolvedSvelteTsxFiles;
        }
        const svelteTsPath = (0, path_1.dirname)(require.resolve('svelte2tsx'));
        const svelteTsxFiles = [
            './svelte-shims.d.ts',
            './svelte-jsx.d.ts',
            './svelte-native-jsx.d.ts'
        ].map((f) => modules.typescript.sys.resolvePath((0, path_1.resolve)(svelteTsPath, f)));
        resolvedSvelteTsxFiles = svelteTsxFiles;
        return svelteTsxFiles;
    }
    function isSvelteProject(compilerOptions) {
        // Add more checks like "no Svelte file found" or "no config file found"?
        return (0, utils_1.hasNodeModule)(compilerOptions, 'svelte');
    }
    function onConfigurationChanged(config) {
        configManager.updateConfigFromPluginConfig(config);
    }
    function decorateLanguageServiceDispose(languageService, disposable) {
        const dispose = languageService.dispose;
        languageService.dispose = () => {
            disposable.dispose();
            dispose();
        };
        return languageService;
    }
    /**
     * TypeScript doesn't load the external files in projects without a config file. So we load it by ourselves.
     * TypeScript also seems to expect files added to the root to be opened by the client in this situation.
     */
    function openSvelteTsxFileForInferredProject(project, file) {
        const normalizedPath = modules.typescript.server.toNormalizedPath(file);
        if (project.containsFile(normalizedPath)) {
            return;
        }
        const scriptInfo = project.projectService.getOrCreateScriptInfoForNormalizedPath(normalizedPath, 
        /*openedByClient*/ true, project.readFile(file));
        if (!scriptInfo) {
            return;
        }
        if (!project.projectService.openFiles.has(scriptInfo.path)) {
            project.projectService.openFiles.set(scriptInfo.path, undefined);
        }
        if (project.projectRootPath) {
            // Only add the file to the project if it has a projectRootPath, because else
            // a ts.Assert error will be thrown when multiple inferred projects are tried
            // to be merged.
            project.addRoot(scriptInfo);
        }
    }
    return { create, getExternalFiles, onConfigurationChanged };
}
module.exports = init;
//# sourceMappingURL=index.js.map