"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PRIVATE_TENANTS = exports.GLOBAL_TENANTS = void 0;
exports.isValidTenant = isValidTenant;
exports.resolve = resolve;
exports.resolveTenant = resolveTenant;

var _lodash = require("lodash");

var _common = require("../../common");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
const PRIVATE_TENANTS = [_common.PRIVATE_TENANT_SYMBOL, 'private'];
exports.PRIVATE_TENANTS = PRIVATE_TENANTS;
const GLOBAL_TENANTS = ['global', _common.GLOBAL_TENANT_SYMBOL];
/**
 * Resovles the tenant the user is using.
 *
 * @param request OpenSearchDashboards request.
 * @param config security plugin config.
 * @param cookie cookie extracted from the request. The cookie should have been parsed by AuthenticationHandler.
 * pass it as parameter instead of extracting again.
 * @param authInfo authentication info, the Elasticsearch authinfo API response.
 *
 * @returns user preferred tenant of the request.
 */

exports.GLOBAL_TENANTS = GLOBAL_TENANTS;

function resolveTenant(request, username, roles, availabeTenants, config, cookie) {
  var _request$url, _request$url$searchPa, _request$url2, _request$url2$searchP, _request$url3, _request$url3$searchP, _config$multitenancy, _config$multitenancy2, _config$multitenancy3;

  const DEFAULT_READONLY_ROLES = ['kibana_read_only'];
  let selectedTenant;
  const securityTenant_ = request === null || request === void 0 ? void 0 : (_request$url = request.url) === null || _request$url === void 0 ? void 0 : (_request$url$searchPa = _request$url.searchParams) === null || _request$url$searchPa === void 0 ? void 0 : _request$url$searchPa.get('securityTenant_');
  const securitytenant = request === null || request === void 0 ? void 0 : (_request$url2 = request.url) === null || _request$url2 === void 0 ? void 0 : (_request$url2$searchP = _request$url2.searchParams) === null || _request$url2$searchP === void 0 ? void 0 : _request$url2$searchP.get('securitytenant'); // eslint-disable-next-line @typescript-eslint/naming-convention

  const security_tenant = request === null || request === void 0 ? void 0 : (_request$url3 = request.url) === null || _request$url3 === void 0 ? void 0 : (_request$url3$searchP = _request$url3.searchParams) === null || _request$url3$searchP === void 0 ? void 0 : _request$url3$searchP.get('security_tenant');

  if (securityTenant_) {
    selectedTenant = securityTenant_;
  } else if (securitytenant) {
    selectedTenant = securitytenant;
  } else if (security_tenant) {
    selectedTenant = security_tenant;
  } else if (request.headers.securitytenant || request.headers.securityTenant_) {
    selectedTenant = request.headers.securitytenant ? request.headers.securitytenant : request.headers.securityTenant_;
  } else if (isValidTenant(cookie.tenant)) {
    selectedTenant = cookie.tenant;
  } else {
    selectedTenant = undefined;
  }

  const isReadonly = roles === null || roles === void 0 ? void 0 : roles.some(role => {
    var _config$readonly_mode;

    return ((_config$readonly_mode = config.readonly_mode) === null || _config$readonly_mode === void 0 ? void 0 : _config$readonly_mode.roles.includes(role)) || DEFAULT_READONLY_ROLES.includes(role);
  });
  const preferredTenants = (_config$multitenancy = config.multitenancy) === null || _config$multitenancy === void 0 ? void 0 : _config$multitenancy.tenants.preferred;
  const globalTenantEnabled = (_config$multitenancy2 = config.multitenancy) === null || _config$multitenancy2 === void 0 ? void 0 : _config$multitenancy2.tenants.enable_global;
  const privateTenantEnabled = ((_config$multitenancy3 = config.multitenancy) === null || _config$multitenancy3 === void 0 ? void 0 : _config$multitenancy3.tenants.enable_private) && !isReadonly;
  return resolve(username, selectedTenant, preferredTenants, availabeTenants, globalTenantEnabled, privateTenantEnabled);
}

function resolve(username, requestedTenant, preferredTenants, availableTenants, // is an object like { tenant_name_1: true, tenant_name_2: false, ... }
globalTenantEnabled, privateTenantEnabled) {
  const availableTenantsClone = (0, _lodash.cloneDeep)(availableTenants);
  delete availableTenantsClone[username];

  if (!globalTenantEnabled && !privateTenantEnabled && (0, _lodash.isEmpty)(availableTenantsClone)) {
    return undefined;
  }

  if (isValidTenant(requestedTenant)) {
    requestedTenant = requestedTenant;

    if (requestedTenant in availableTenants) {
      return requestedTenant;
    }

    if (privateTenantEnabled && username in availableTenants && PRIVATE_TENANTS.indexOf(requestedTenant) > -1) {
      return _common.PRIVATE_TENANT_SYMBOL;
    }

    if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(requestedTenant) > -1) {
      return _common.GLOBAL_TENANT_SYMBOL;
    }
  }

  if (preferredTenants && !(0, _lodash.isEmpty)(preferredTenants)) {
    for (const element of preferredTenants) {
      const tenant = element.toLowerCase();

      if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(tenant) > -1) {
        return _common.GLOBAL_TENANT_SYMBOL;
      }

      if (privateTenantEnabled && PRIVATE_TENANTS.indexOf(tenant) > -1 && username in availableTenants) {
        return _common.PRIVATE_TENANT_SYMBOL;
      }

      if (tenant in availableTenants) {
        return tenant;
      }
    }
  }

  if (globalTenantEnabled) {
    return _common.GLOBAL_TENANT_SYMBOL;
  }

  if (privateTenantEnabled) {
    return _common.PRIVATE_TENANT_SYMBOL;
  }
  /**
   * Fall back to the first tenant in the available tenants
   * Under the condition of enabling multitenancy, if the user has disabled both 'Global' and 'Private' tenants:
   * it will remove the default global tenant key for custom tenant.
   */


  if (Object.keys(availableTenantsClone).length > 1 && availableTenantsClone.hasOwnProperty(_common.globalTenantName)) {
    delete availableTenantsClone[_common.globalTenantName];
  }

  return (0, _lodash.findKey)(availableTenantsClone, () => true);
}
/**
 * Return true if tenant parameter is a valid tenent.
 *
 * Note: empty string '' is valid, which means global tenant.
 *
 * @param tenant
 */


function isValidTenant(tenant) {
  return tenant !== undefined && tenant !== null;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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