"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectSvelteFilesManager = void 0;
const utils_1 = require("./utils");
class ProjectSvelteFilesManager {
    static getInstance(projectName) {
        return this.instances.get(projectName);
    }
    constructor(typescript, project, serverHost, snapshotManager, parsedCommandLine, configManager) {
        this.typescript = typescript;
        this.project = project;
        this.serverHost = serverHost;
        this.snapshotManager = snapshotManager;
        this.parsedCommandLine = parsedCommandLine;
        this.files = new Set();
        this.directoryWatchers = new Set();
        if (configManager.getConfig().enable) {
            this.setupWatchers();
            this.updateProjectSvelteFiles();
        }
        configManager.onConfigurationChanged(this.onConfigChanged.bind(this));
        ProjectSvelteFilesManager.instances.set(project.getProjectName(), this);
    }
    updateProjectConfig(serviceHost) {
        var _a;
        const parsedCommandLine = (_a = serviceHost.getParsedCommandLine) === null || _a === void 0 ? void 0 : _a.call(serviceHost, (0, utils_1.getConfigPathForProject)(this.project));
        if (!parsedCommandLine) {
            return;
        }
        this.disposeWatchersAndFiles();
        this.parsedCommandLine = parsedCommandLine;
        this.setupWatchers();
        this.updateProjectSvelteFiles();
    }
    getFiles() {
        return Array.from(this.files);
    }
    /**
     * Create directory watcher for include and exclude
     * The watcher in tsserver doesn't support svelte file
     * It won't add new created svelte file to root
     */
    setupWatchers() {
        for (const directory in this.parsedCommandLine.wildcardDirectories) {
            if (!Object.prototype.hasOwnProperty.call(this.parsedCommandLine.wildcardDirectories, directory)) {
                continue;
            }
            const watchDirectoryFlags = this.parsedCommandLine.wildcardDirectories[directory];
            const watcher = this.serverHost.watchDirectory(directory, this.watcherCallback.bind(this), watchDirectoryFlags === this.typescript.WatchDirectoryFlags.Recursive, this.parsedCommandLine.watchOptions);
            this.directoryWatchers.add(watcher);
        }
    }
    watcherCallback(fileName) {
        if (!(0, utils_1.isSvelteFilePath)(fileName)) {
            return;
        }
        // We can't just add the file to the project directly, because
        // - the casing of fileName is different
        // - we don't know whether the file was added or deleted
        this.updateProjectSvelteFiles();
    }
    updateProjectSvelteFiles() {
        const fileNamesAfter = this.readProjectSvelteFilesFromFs();
        const removedFiles = new Set(...this.files);
        const newFiles = fileNamesAfter.filter((fileName) => {
            const has = this.files.has(fileName);
            if (has) {
                removedFiles.delete(fileName);
            }
            return !has;
        });
        for (const newFile of newFiles) {
            this.addFileToProject(newFile);
            this.files.add(newFile);
        }
        for (const removedFile of removedFiles) {
            this.removeFileFromProject(removedFile, false);
            this.files.delete(removedFile);
        }
    }
    addFileToProject(newFile) {
        this.snapshotManager.create(newFile);
        const snapshot = this.project.projectService.getScriptInfo(newFile);
        if (snapshot) {
            this.project.addRoot(snapshot);
        }
    }
    readProjectSvelteFilesFromFs() {
        const fileSpec = this.parsedCommandLine.raw;
        const { include, exclude } = fileSpec;
        if ((include === null || include === void 0 ? void 0 : include.length) === 0) {
            return [];
        }
        return this.typescript.sys
            .readDirectory(this.project.getCurrentDirectory() || process.cwd(), ['.svelte'], exclude, include)
            .map(this.typescript.server.toNormalizedPath);
    }
    onConfigChanged(config) {
        this.disposeWatchersAndFiles();
        if (config.enable) {
            this.setupWatchers();
            this.updateProjectSvelteFiles();
        }
    }
    removeFileFromProject(file, exists = true) {
        const info = this.project.getScriptInfo(file);
        if (info) {
            this.project.removeFile(info, exists, true);
        }
    }
    disposeWatchersAndFiles() {
        this.directoryWatchers.forEach((watcher) => watcher.close());
        this.directoryWatchers.clear();
        this.files.forEach((file) => this.removeFileFromProject(file));
        this.files.clear();
    }
    dispose() {
        this.disposeWatchersAndFiles();
        ProjectSvelteFilesManager.instances.delete(this.project.getProjectName());
    }
}
ProjectSvelteFilesManager.instances = new Map();
exports.ProjectSvelteFilesManager = ProjectSvelteFilesManager;
//# sourceMappingURL=project-svelte-files.js.map