"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SveltePlugin = void 0;
const path_1 = require("path");
const vscode_languageserver_1 = require("vscode-languageserver");
const importPackage_1 = require("../../importPackage");
const logger_1 = require("../../logger");
const utils_1 = require("../../utils");
const getCodeActions_1 = require("./features/getCodeActions");
const getCompletions_1 = require("./features/getCompletions");
const getDiagnostics_1 = require("./features/getDiagnostics");
const getHoverInfo_1 = require("./features/getHoverInfo");
const getSelectionRanges_1 = require("./features/getSelectionRanges");
const SvelteDocument_1 = require("./SvelteDocument");
class SveltePlugin {
    constructor(configManager) {
        this.configManager = configManager;
        this.__name = 'svelte';
        this.docManager = new Map();
    }
    async getDiagnostics(document) {
        if (!this.featureEnabled('diagnostics') || !this.configManager.getIsTrusted()) {
            return [];
        }
        return (0, getDiagnostics_1.getDiagnostics)(document, await this.getSvelteDoc(document), this.configManager.getConfig().svelte.compilerWarnings);
    }
    async getCompiledResult(document) {
        try {
            const svelteDoc = await this.getSvelteDoc(document);
            return svelteDoc.getCompiledWith({ generate: 'dom' });
        }
        catch (error) {
            return null;
        }
    }
    async formatDocument(document, options) {
        var _a, _b, _c, _d;
        if (!this.featureEnabled('format')) {
            return [];
        }
        const filePath = document.getFilePath();
        const prettier = (0, importPackage_1.importPrettier)(filePath);
        // Try resolving the config through prettier and fall back to possible editor config
        const config = this.configManager.getMergedPrettierConfig(await prettier.resolveConfig(filePath, { editorconfig: true }), 
        // Be defensive here because IDEs other than VSCode might not have these settings
        options && {
            tabWidth: options.tabSize,
            useTabs: !options.insertSpaces
        });
        // If user has prettier-plugin-svelte 1.x, then remove `options` from the sort
        // order or else it will throw a config error (`options` was not present back then).
        if ((config === null || config === void 0 ? void 0 : config.svelteSortOrder) &&
            ((_a = (0, importPackage_1.getPackageInfo)('prettier-plugin-svelte', filePath)) === null || _a === void 0 ? void 0 : _a.version.major) < 2) {
            config.svelteSortOrder = config.svelteSortOrder
                .replace('-options', '')
                .replace('options-', '');
        }
        // Take .prettierignore into account
        const fileInfo = await prettier.getFileInfo(filePath, {
            ignorePath: (_c = (_b = this.configManager.getPrettierConfig()) === null || _b === void 0 ? void 0 : _b.ignorePath) !== null && _c !== void 0 ? _c : '.prettierignore',
            // Sapper places stuff within src/node_modules, we want to format that, too
            withNodeModules: true
        });
        if (fileInfo.ignored) {
            logger_1.Logger.debug('File is ignored, formatting skipped');
            return [];
        }
        const formattedCode = prettier.format(document.getText(), {
            ...config,
            plugins: Array.from(new Set([
                ...((_d = config.plugins) !== null && _d !== void 0 ? _d : [])
                    .map(resolvePlugin)
                    .filter(utils_1.isNotNullOrUndefined),
                ...getSveltePlugin()
            ])),
            parser: 'svelte'
        });
        return document.getText() === formattedCode
            ? []
            : [
                vscode_languageserver_1.TextEdit.replace(vscode_languageserver_1.Range.create(document.positionAt(0), document.positionAt(document.getTextLength())), formattedCode)
            ];
        function getSveltePlugin() {
            // Only provide our version of the svelte plugin if the user doesn't have one in
            // the workspace already. If we did it, Prettier would - for some reason - use
            // the workspace version for parsing and the extension version for printing,
            // which could crash if the contract of the parser output changed.
            const hasPluginLoadedAlready = prettier
                .getSupportInfo()
                .languages.some((l) => l.name === 'svelte');
            return hasPluginLoadedAlready ? [] : [require.resolve('prettier-plugin-svelte')];
        }
        function resolvePlugin(plugin) {
            // https://github.com/prettier/prettier-vscode/blob/160b0e92d88fa19003dce2745d5ab8c67e886a04/src/ModuleResolver.ts#L373
            if (typeof plugin != 'string' || (0, path_1.isAbsolute)(plugin) || plugin.startsWith('.')) {
                return plugin;
            }
            try {
                return require.resolve(plugin, {
                    paths: [filePath]
                });
            }
            catch (error) {
                logger_1.Logger.error(`failed to resolve plugin ${plugin} with error:\n`, error);
            }
        }
    }
    async getCompletions(document, position, _, cancellationToken) {
        if (!this.featureEnabled('completions')) {
            return null;
        }
        const svelteDoc = await this.getSvelteDoc(document);
        if (cancellationToken === null || cancellationToken === void 0 ? void 0 : cancellationToken.isCancellationRequested) {
            return null;
        }
        return (0, getCompletions_1.getCompletions)(document, svelteDoc, position);
    }
    async doHover(document, position) {
        if (!this.featureEnabled('hover')) {
            return null;
        }
        return (0, getHoverInfo_1.getHoverInfo)(document, await this.getSvelteDoc(document), position);
    }
    async getCodeActions(document, range, context, cancellationToken) {
        if (!this.featureEnabled('codeActions')) {
            return [];
        }
        const svelteDoc = await this.getSvelteDoc(document);
        if (cancellationToken === null || cancellationToken === void 0 ? void 0 : cancellationToken.isCancellationRequested) {
            return [];
        }
        try {
            return (0, getCodeActions_1.getCodeActions)(svelteDoc, range, context);
        }
        catch (error) {
            return [];
        }
    }
    async executeCommand(document, command, args) {
        if (!this.featureEnabled('codeActions')) {
            return null;
        }
        const svelteDoc = await this.getSvelteDoc(document);
        try {
            return (0, getCodeActions_1.executeCommand)(svelteDoc, command, args);
        }
        catch (error) {
            return null;
        }
    }
    async getSelectionRange(document, position) {
        if (!this.featureEnabled('selectionRange')) {
            return null;
        }
        const svelteDoc = await this.getSvelteDoc(document);
        return (0, getSelectionRanges_1.getSelectionRange)(svelteDoc, position);
    }
    featureEnabled(feature) {
        return (this.configManager.enabled('svelte.enable') &&
            this.configManager.enabled(`svelte.${feature}.enable`));
    }
    async getSvelteDoc(document) {
        let svelteDoc = this.docManager.get(document);
        if (!svelteDoc || svelteDoc.version !== document.version) {
            svelteDoc = new SvelteDocument_1.SvelteDocument(document);
            this.docManager.set(document, svelteDoc);
        }
        return svelteDoc;
    }
}
exports.SveltePlugin = SveltePlugin;
//# sourceMappingURL=SveltePlugin.js.map