"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _lodash = require("lodash");

var _helpers = require("./utils/helpers");

var _adHelpers = require("./utils/adHelpers");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const MAX_DETECTOR_COUNT = 1000;

class DestinationsService {
  constructor(esDriver) {
    _defineProperty(this, "getDetector", async (context, req, res) => {
      const {
        detectorId
      } = req.params;
      const {
        callAsCurrentUser
      } = this.esDriver.asScoped(req);

      try {
        const resp = await callAsCurrentUser('alertingAD.getDetector', {
          detectorId
        });
        const {
          anomaly_detector,
          _seq_no: seqNo,
          _primary_term: primaryTerm,
          _version: version
        } = resp;
        return res.ok({
          body: {
            ok: true,
            detector: anomaly_detector,
            version,
            seqNo,
            primaryTerm
          }
        });
      } catch (err) {
        console.error('Alerting - AnomalyDetectorService - getDetector:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "getDetectors", async (context, req, res) => {
      const searchRequest = {
        query: {
          bool: {}
        },
        size: MAX_DETECTOR_COUNT
      };
      const {
        callAsCurrentUser
      } = this.esDriver.asScoped(req);

      try {
        const resp = await callAsCurrentUser('alertingAD.searchDetectors', {
          body: searchRequest
        });
        const totalDetectors = resp.hits.total.value;
        const detectors = resp.hits.hits.map(hit => {
          const {
            _source: detector,
            _id: id,
            _version: version,
            _seq_no: seqNo,
            _primary_term: primaryTerm
          } = hit;
          return {
            id,
            ...detector,
            version,
            seqNo,
            primaryTerm
          };
        });
        return res.ok({
          body: {
            ok: true,
            detectors: (0, _helpers.mapKeysDeep)(detectors, _helpers.toCamel),
            totalDetectors
          }
        });
      } catch (err) {
        console.error('Alerting - AnomalyDetectorService - searchDetectors:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "getDetectorResults", async (context, req, res) => {
      try {
        const {
          startTime = 0,
          endTime = 20,
          preview = 'false'
        } = req.query;
        const {
          detectorId
        } = req.params;
        const {
          callAsCurrentUser
        } = this.esDriver.asScoped(req);

        if (preview == 'true') {
          const requestBody = {
            period_start: startTime,
            period_end: endTime
          };
          const previewResponse = await callAsCurrentUser('alertingAD.previewDetector', {
            detectorId,
            body: requestBody
          });
          const transformedKeys = (0, _helpers.mapKeysDeep)(previewResponse, _helpers.toCamel);
          return res.ok({
            body: {
              ok: true,
              response: {
                anomalyResult: (0, _adHelpers.anomalyResultMapper)(transformedKeys.anomalyResult),
                detector: transformedKeys.anomalyDetector
              }
            }
          });
        } else {
          //Get results
          const requestBody = {
            size: 10000,
            sort: {
              data_start_time: 'asc'
            },
            query: {
              bool: {
                filter: [{
                  term: {
                    detector_id: detectorId
                  }
                }, {
                  range: {
                    data_start_time: {
                      gte: startTime,
                      lte: endTime
                    }
                  }
                }]
              }
            }
          };
          const detectorResponse = await callAsCurrentUser('alertingAD.getDetector', {
            detectorId
          });
          const anomaliesResponse = await callAsCurrentUser('alertingAD.searchResults', {
            body: requestBody
          });
          const transformedKeys = (0, _lodash.get)(anomaliesResponse, 'hits.hits', []).map(result => (0, _helpers.mapKeysDeep)(result._source, _helpers.toCamel));
          return res.ok({
            body: {
              ok: true,
              response: {
                detector: (0, _helpers.mapKeysDeep)((0, _lodash.get)(detectorResponse, 'anomaly_detector', {}), _helpers.toCamel),
                anomalyResult: (0, _adHelpers.anomalyResultMapper)(transformedKeys)
              }
            }
          });
        }
      } catch (err) {
        console.log('Alerting - AnomalyDetectorService - getDetectorResults', err);
        return res.ok({
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    this.esDriver = esDriver;
  }

}

exports.default = DestinationsService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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